package decimal

import (
	"sync"
)

type lazyDecimal struct {
	once sync.Once
	s    string
	x    *Big
}

func (l *lazyDecimal) get() *Big {
	l.once.Do(func() {
		l.x, _ = new(Big).SetString(l.s)
	})
	return l.x
}

func newLazyDecimal(s string) *lazyDecimal {
	return &lazyDecimal{s: s}
}

const (
	// constPrec is the precision for mathematical constants like
	// e, pi, etc.
	constPrec             = 300
	defaultExtraPrecision = 3
)

var (
	_E     = newLazyDecimal("2.71828182845904523536028747135266249775724709369995957496696762772407663035354759457138217852516642742746639193200305992181741359662904357290033429526059563073813232862794349076323382988075319525101901157383418793070215408914993488416750924476146066808226480016847741185374234544243710753907774499207")
	_Ln10  = newLazyDecimal("2.30258509299404568401799145468436420760110148862877297603332790096757260967735248023599720508959829834196778404228624863340952546508280675666628736909878168948290720832555468084379989482623319852839350530896537773262884616336622228769821988674654366747440424327436515504893431493939147961940440022211")
	_Pi    = newLazyDecimal("3.14159265358979323846264338327950288419716939937510582097494459230781640628620899862803482534211706798214808651328230664709384460955058223172535940812848111745028410270193852110555964462294895493038196442881097566593344612847564823378678316527120190914564856692346034861045432664821339360726024914127")
	_Pi2   = newLazyDecimal("1.57079632679489661923132169163975144209858469968755291048747229615390820314310449931401741267105853399107404325664115332354692230477529111586267970406424055872514205135096926055277982231147447746519098221440548783296672306423782411689339158263560095457282428346173017430522716332410669680363012457064")
	_Sqrt3 = newLazyDecimal("1.73205080756887729352744634150587236694280525381038062805580697945193301690880003708114618675724857567562614141540670302996994509499895247881165551209437364852809323190230558206797482010108467492326501531234326690332288665067225466892183797122704713166036786158801904998653737985938946765034750657605")

	approx1 = newLazyDecimal("0.259")
	approx2 = newLazyDecimal("0.819")
	approx3 = newLazyDecimal("0.0819")
	approx4 = newLazyDecimal("2.59")
	ptFive  = newLazyDecimal("0.5")

	_10005             = newLazyDecimal("10005")
	_426880            = newLazyDecimal("426880")
	_13591409          = newLazyDecimal("13591409")
	_545140134         = newLazyDecimal("545140134")
	_10939058860032000 = newLazyDecimal("10939058860032000")

	negFour   = newLazyDecimal("-4")
	negOne    = newLazyDecimal("-1")
	one       = newLazyDecimal("1")
	onePtFour = newLazyDecimal("1.4")
	two       = newLazyDecimal("2")
	three     = newLazyDecimal("3")
	four      = newLazyDecimal("4")
	five      = newLazyDecimal("5")
	six       = newLazyDecimal("6")
	eight     = newLazyDecimal("8")
	ten       = newLazyDecimal("10")
	eleven    = newLazyDecimal("11")
	sixteen   = newLazyDecimal("16")
	eighteen  = newLazyDecimal("18")
	thirtyTwo = newLazyDecimal("32")
	eightyOne = newLazyDecimal("81")
)
