package io

import (
	"encoding/json"
	"io"
	"strings"

	"git.sr.ht/~charles/rq/util"
)

func init() {
	registerOutputHandler("raw", func() OutputHandler { return &RawOutputHandler{} })
}

// Declare conformance with OutputHandler interface.
var _ OutputHandler = &RawOutputHandler{}

// RawOutputHandler handles serializing JSON data.
type RawOutputHandler struct {
	initialized bool
	fs          string
	fl          string
	fr          string
	rs          string
	rl          string
	rr          string
}

func (r *RawOutputHandler) init() {
	if r.initialized {
		return
	}

	r.initialized = true
	r.fs = "\t"
	r.rs = "\n"
}

// Name implements OutputHandler.Name().
func (r *RawOutputHandler) Name() string {
	return "raw"
}

// SetOption implements OutputHandler.SetOption().
func (r *RawOutputHandler) SetOption(name string, value string) error {
	r.init()

	if (name == "raw.fl") || (name == "raw.fs") || (name == "raw.fr") || (name == "raw.rl") || (name == "raw.rs") || (name == "raw.rr") {
		u, err := util.Unescape(value)
		if err == nil {
			value = u
		}
	}

	switch name {
	case "raw.fs":
		r.fs = value
	case "raw.fl":
		r.fl = value
	case "raw.fr":
		r.fr = value
	case "raw.rs":
		r.rs = value
	case "raw.rl":
		r.rl = value
	case "raw.rr":
		r.rr = value
	}

	return nil
}

// Format implements OutputHandler.Format()
func (r *RawOutputHandler) Format(writer io.Writer, data interface{}) error {
	r.init()

	var err error
	var jsonBytes []byte

	if s, ok := data.(string); ok {
		_, err := writer.Write([]byte(s))
		return err
	} else if strList, ok := data.([]string); ok {
		newData := make([]interface{}, len(strList))
		for i, v := range strList {
			newData[i] = v
		}

		data = newData

	} else if intList, ok := data.([]int); ok {
		newData := make([]interface{}, len(intList))
		for i, v := range intList {
			newData[i] = v
		}

		data = newData
	} else if floatList, ok := data.([]float64); ok {
		newData := make([]interface{}, len(floatList))
		for i, v := range floatList {
			newData[i] = v
		}

		data = newData
	} else if boolList, ok := data.([]bool); ok {
		newData := make([]interface{}, len(boolList))
		for i, v := range boolList {
			newData[i] = v
		}

		data = newData
	}

	// If this is a list of only primitive types, we can print it out as a
	// list of strings.
	if list, ok := data.([]interface{}); ok {
		allPrimitive := true
		strs := make([]string, len(list))
		for i, elem := range list {
			if util.IsPrimitive(elem); ok {
				strs[i] = r.rl + util.ValueToString(elem) + r.rr
			} else {
				allPrimitive = false
				break
			}
		}

		if allPrimitive {
			_, err := writer.Write([]byte(strings.Join(strs, r.rs)))
			return err
		}
	}

	// If we couldn't treat it as a list of primitives, we may be able to
	// represent it as a table.
	if tab, err := util.Tabularize(data, false); err == nil {

		for i, rec := range tab {
			_, err := writer.Write([]byte(r.rl))
			if err != nil {
				return err
			}

			for j, field := range rec {
				tab[i][j] = r.fl + field + r.fr
			}

			_, err = writer.Write([]byte(strings.Join(tab[i], r.fs)))
			if err != nil {
				return err
			}

			_, err = writer.Write([]byte(r.rr))
			if err != nil {
				return err
			}

			if i <= (len(tab) - 1) {
				_, err := writer.Write([]byte(r.rs))
				if err != nil {
					return err
				}
			}

		}

		return nil
	}

	// fallback in case we weren't able to handle the data properly before
	jsonBytes, err = json.Marshal(data)
	if err != nil {
		return err
	}

	_, err = writer.Write(jsonBytes)
	if err != nil {
		return err
	}

	return nil
}
