// Copyright 2019 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "device/fido/cros/authenticator.h"

#include <memory>
#include <string>
#include <utility>
#include <vector>

#include "base/bind.h"
#include "base/containers/span.h"
#include "base/strings/string_number_conversions.h"
#include "base/strings/utf_string_conversions.h"
#include "chromeos/dbus/u2f/u2f_client.h"
#include "chromeos/dbus/u2f/u2f_interface.pb.h"
#include "components/cbor/reader.h"
#include "components/device_event_log/device_event_log.h"
#include "device/fido/attestation_statement_formats.h"
#include "device/fido/authenticator_data.h"
#include "device/fido/fido_parsing_utils.h"
#include "device/fido/fido_transport_protocol.h"
#include "device/fido/opaque_attestation_statement.h"
#include "third_party/cros_system_api/dbus/u2f/dbus-constants.h"

namespace device {

ChromeOSAuthenticator::ChromeOSAuthenticator(
    base::RepeatingCallback<std::string()> generate_request_id_callback,
    ChromeOSAuthenticator::Config config)
    : generate_request_id_callback_(std::move(generate_request_id_callback)),
      config_(config),
      weak_factory_(this) {}

ChromeOSAuthenticator::~ChromeOSAuthenticator() {}

FidoAuthenticator::Type ChromeOSAuthenticator::GetType() const {
  return Type::kChromeOS;
}

std::string ChromeOSAuthenticator::GetId() const {
  return "ChromeOSAuthenticator";
}

namespace {

AuthenticatorSupportedOptions ChromeOSAuthenticatorOptions() {
  AuthenticatorSupportedOptions options;
  options.is_platform_device = true;
  // TODO(yichengli): change supports_resident_key to true once it's supported.
  options.supports_resident_key = false;
  // Even if the user has no fingerprints enrolled, we will have password
  // as fallback.
  options.user_verification_availability = AuthenticatorSupportedOptions::
      UserVerificationAvailability::kSupportedAndConfigured;
  options.supports_user_presence = true;
  return options;
}

}  // namespace

const absl::optional<AuthenticatorSupportedOptions>&
ChromeOSAuthenticator::Options() const {
  static const absl::optional<AuthenticatorSupportedOptions> options =
      ChromeOSAuthenticatorOptions();
  return options;
}

absl::optional<FidoTransportProtocol>
ChromeOSAuthenticator::AuthenticatorTransport() const {
  return FidoTransportProtocol::kInternal;
}

void ChromeOSAuthenticator::InitializeAuthenticator(
    base::OnceClosure callback) {
  u2f::GetAlgorithmsRequest request;
  chromeos::U2FClient::Get()->GetAlgorithms(
      request, base::BindOnce(&ChromeOSAuthenticator::OnGetAlgorithmsResponse,
                              weak_factory_.GetWeakPtr(), std::move(callback)));
}

void ChromeOSAuthenticator::OnGetAlgorithmsResponse(
    base::OnceClosure callback,
    absl::optional<u2f::GetAlgorithmsResponse> response) {
  if (response && response->status() ==
                      u2f::GetAlgorithmsResponse_GetAlgorithmsStatus_SUCCESS) {
    supported_algorithms_ = std::vector<int32_t>();
    for (int i = 0; i < response->algorithm_size(); i++) {
      supported_algorithms_->push_back(response->algorithm(i));
    }
  } else {
    // Keep `supported_algorithms_` as nullopt if fetching supported algorithms
    // from u2fd failed, since the caller of `GetAlgorithms` method might want
    // to provide defaults.
    supported_algorithms_ = absl::nullopt;
  }

  std::move(callback).Run();
}

absl::optional<base::span<const int32_t>>
ChromeOSAuthenticator::GetAlgorithms() {
  if (supported_algorithms_) {
    return base::span<const int32_t>(*supported_algorithms_);
  }
  return absl::nullopt;
}

void ChromeOSAuthenticator::MakeCredential(
    CtapMakeCredentialRequest request,
    MakeCredentialOptions request_options,
    MakeCredentialCallback callback) {
  u2f::MakeCredentialRequest req;
  // Only allow skipping user verification if user presence checks via power
  // button press have been configured. This is only the case when running with
  // the "DeviceSecondFactorAuthentication" enterprise policy.
  req.set_verification_type(
      (request.user_verification == UserVerificationRequirement::kDiscouraged &&
       config_.power_button_enabled)
          ? u2f::VERIFICATION_USER_PRESENCE
          : u2f::VERIFICATION_USER_VERIFICATION);
  req.set_rp_id(request.rp.id);
  req.set_client_data_hash(std::string(request.client_data_hash.begin(),
                                       request.client_data_hash.end()));

  // The ChromeOS platform authenticator supports attestation only for
  // credentials created through the legacy, enterprise-policy-controlled power
  // button authenticator. It has two modes, regular U2F attestation and and
  // individually identifying mode called G2F that needs to be explicitly
  // configured in the enterprise policy.
  switch (request.attestation_preference) {
    case AttestationConveyancePreference::kNone:
    case AttestationConveyancePreference::kIndirect:
      req.set_attestation_conveyance_preference(
          u2f::MakeCredentialRequest_AttestationConveyancePreference_NONE);
      break;
    case AttestationConveyancePreference::kDirect:
      req.set_attestation_conveyance_preference(
          u2f::MakeCredentialRequest_AttestationConveyancePreference_U2F);
      break;
    case AttestationConveyancePreference::kEnterpriseIfRPListedOnAuthenticator:
      // There is no separate mechanism for allowing individual RPs to use
      // individual G2F attestation. (Same as with regular U2F authenticators.)
      req.set_attestation_conveyance_preference(
          u2f::MakeCredentialRequest_AttestationConveyancePreference_U2F);
      break;
    case AttestationConveyancePreference::kEnterpriseApprovedByBrowser:
      req.set_attestation_conveyance_preference(
          u2f::MakeCredentialRequest_AttestationConveyancePreference_G2F);
      break;
  }

  req.set_user_id(std::string(request.user.id.begin(), request.user.id.end()));
  if (request.user.display_name.has_value())
    req.set_user_display_name(request.user.display_name.value());
  req.set_resident_credential(request.resident_key_required);
  DCHECK(generate_request_id_callback_);
  DCHECK(current_request_id_.empty());
  current_request_id_ = generate_request_id_callback_.Run();
  req.set_request_id_str(current_request_id_);

  for (const PublicKeyCredentialDescriptor& descriptor : request.exclude_list) {
    req.add_excluded_credential_id(
        std::string(descriptor.id.begin(), descriptor.id.end()));
  }
  if (request.app_id_exclude) {
    req.set_app_id_exclude(*request.app_id_exclude);
  }

  chromeos::U2FClient::Get()->MakeCredential(
      req, base::BindOnce(&ChromeOSAuthenticator::OnMakeCredentialResponse,
                          weak_factory_.GetWeakPtr(), std::move(request),
                          std::move(callback)));
}

void ChromeOSAuthenticator::OnMakeCredentialResponse(
    CtapMakeCredentialRequest request,
    MakeCredentialCallback callback,
    absl::optional<u2f::MakeCredentialResponse> response) {
  if (!response) {
    FIDO_LOG(ERROR) << "MakeCredential dbus call failed";
    std::move(callback).Run(CtapDeviceResponseCode::kCtap2ErrOther,
                            absl::nullopt);
    return;
  }

  FIDO_LOG(DEBUG) << "Make credential status: " << response->status();
  if (response->status() !=
      u2f::MakeCredentialResponse_MakeCredentialStatus_SUCCESS) {
    std::move(callback).Run(CtapDeviceResponseCode::kCtap2ErrOperationDenied,
                            absl::nullopt);
    return;
  }

  absl::optional<AuthenticatorData> authenticator_data =
      AuthenticatorData::DecodeAuthenticatorData(
          base::as_bytes(base::make_span(response->authenticator_data())));
  if (!authenticator_data) {
    FIDO_LOG(ERROR) << "Authenticator data corrupted.";
    std::move(callback).Run(CtapDeviceResponseCode::kCtap2ErrOther,
                            absl::nullopt);
    return;
  }

  absl::optional<cbor::Value> statement_map = cbor::Reader::Read(
      base::as_bytes(base::make_span(response->attestation_statement())));
  if (!statement_map ||
      statement_map.value().type() != cbor::Value::Type::MAP) {
    FIDO_LOG(ERROR) << "Attestation statement is not a CBOR map.";
    std::move(callback).Run(CtapDeviceResponseCode::kCtap2ErrOther,
                            absl::nullopt);
    return;
  }
  auto statement = std::make_unique<OpaqueAttestationStatement>(
      response->attestation_format(), std::move(*statement_map));

  std::move(callback).Run(CtapDeviceResponseCode::kSuccess,
                          AuthenticatorMakeCredentialResponse(
                              FidoTransportProtocol::kInternal,
                              AttestationObject(std::move(*authenticator_data),
                                                std::move(statement))));
}

void ChromeOSAuthenticator::GetAssertion(CtapGetAssertionRequest request,
                                         CtapGetAssertionOptions options,
                                         GetAssertionCallback callback) {
  u2f::GetAssertionRequest req;
  // Only allow skipping user verification if user presence checks via power
  // button press have been configured. This is only the case when running with
  // the "DeviceSecondFactorAuthentication" enterprise policy.
  req.set_verification_type(
      (request.user_verification == UserVerificationRequirement::kDiscouraged &&
       config_.power_button_enabled)
          ? u2f::VERIFICATION_USER_PRESENCE
          : u2f::VERIFICATION_USER_VERIFICATION);
  req.set_rp_id(request.rp_id);
  if (request.app_id) {
    req.set_app_id(*request.app_id);
  }
  req.set_client_data_hash(std::string(request.client_data_hash.begin(),
                                       request.client_data_hash.end()));
  DCHECK(generate_request_id_callback_);
  DCHECK(current_request_id_.empty());
  current_request_id_ = generate_request_id_callback_.Run();
  req.set_request_id_str(current_request_id_);

  for (const PublicKeyCredentialDescriptor& descriptor : request.allow_list) {
    req.add_allowed_credential_id(
        std::string(descriptor.id.begin(), descriptor.id.end()));
  }

  chromeos::U2FClient::Get()->GetAssertion(
      req, base::BindOnce(&ChromeOSAuthenticator::OnGetAssertionResponse,
                          weak_factory_.GetWeakPtr(), std::move(request),
                          std::move(callback)));
}

void ChromeOSAuthenticator::OnGetAssertionResponse(
    CtapGetAssertionRequest request,
    GetAssertionCallback callback,
    absl::optional<u2f::GetAssertionResponse> response) {
  if (!response) {
    FIDO_LOG(ERROR) << "GetAssertion dbus call failed";
    std::move(callback).Run(CtapDeviceResponseCode::kCtap2ErrOther,
                            absl::nullopt);
    return;
  }

  FIDO_LOG(DEBUG) << "GetAssertion status: " << response->status();
  if (response->status() !=
          u2f::GetAssertionResponse_GetAssertionStatus_SUCCESS ||
      response->assertion_size() < 1) {
    std::move(callback).Run(CtapDeviceResponseCode::kCtap2ErrOperationDenied,
                            absl::nullopt);
    return;
  }

  u2f::Assertion assertion = response->assertion(0);

  absl::optional<AuthenticatorData> authenticator_data =
      AuthenticatorData::DecodeAuthenticatorData(
          base::as_bytes(base::make_span(assertion.authenticator_data())));
  if (!authenticator_data) {
    FIDO_LOG(ERROR) << "Authenticator data corrupted.";
    std::move(callback).Run(CtapDeviceResponseCode::kCtap2ErrOther,
                            absl::nullopt);
    return;
  }

  std::vector<uint8_t> signature(assertion.signature().begin(),
                                 assertion.signature().end());
  AuthenticatorGetAssertionResponse authenticator_response(
      std::move(*authenticator_data), std::move(signature));
  authenticator_response.transport_used = FidoTransportProtocol::kInternal;
  const std::string& credential_id = assertion.credential_id();
  authenticator_response.credential = PublicKeyCredentialDescriptor(
      CredentialType::kPublicKey,
      std::vector<uint8_t>(credential_id.begin(), credential_id.end()));
  std::move(callback).Run(CtapDeviceResponseCode::kSuccess,
                          std::move(authenticator_response));
}

void ChromeOSAuthenticator::HasCredentialForGetAssertionRequest(
    const CtapGetAssertionRequest& request,
    base::OnceCallback<void(bool has_credential)> callback) {
  u2f::HasCredentialsRequest req;
  req.set_rp_id(request.rp_id);
  if (request.app_id) {
    req.set_app_id(*request.app_id);
  }

  for (const PublicKeyCredentialDescriptor& descriptor : request.allow_list) {
    req.add_credential_id(
        std::string(descriptor.id.begin(), descriptor.id.end()));
  }

  chromeos::U2FClient::Get()->HasCredentials(
      req,
      base::BindOnce(
          [](base::OnceCallback<void(bool has_credential)> callback,
             absl::optional<u2f::HasCredentialsResponse> response) {
            std::move(callback).Run(
                response &&
                response->status() ==
                    u2f::HasCredentialsResponse_HasCredentialsStatus_SUCCESS &&
                response->credential_id().size() > 0);
          },
          std::move(callback)));
}

void ChromeOSAuthenticator::HasLegacyU2fCredentialForGetAssertionRequest(
    const CtapGetAssertionRequest& request,
    base::OnceCallback<void(bool has_credential)> callback) {
  u2f::HasCredentialsRequest req;
  req.set_rp_id(request.rp_id);
  if (request.app_id) {
    req.set_app_id(*request.app_id);
  }

  for (const PublicKeyCredentialDescriptor& descriptor : request.allow_list) {
    req.add_credential_id(
        std::string(descriptor.id.begin(), descriptor.id.end()));
  }

  chromeos::U2FClient::Get()->HasLegacyU2FCredentials(
      req,
      base::BindOnce(
          [](base::OnceCallback<void(bool has_credential)> callback,
             absl::optional<u2f::HasCredentialsResponse> response) {
            std::move(callback).Run(
                response &&
                response->status() ==
                    u2f::HasCredentialsResponse_HasCredentialsStatus_SUCCESS &&
                response->credential_id().size() > 0);
          },
          std::move(callback)));
}

void ChromeOSAuthenticator::Cancel() {
  if (current_request_id_.empty())
    return;

  u2f::CancelWebAuthnFlowRequest req;
  req.set_request_id_str(current_request_id_);
  chromeos::U2FClient::Get()->CancelWebAuthnFlow(
      req, base::BindOnce(&ChromeOSAuthenticator::OnCancelResponse,
                          weak_factory_.GetWeakPtr()));
}

void ChromeOSAuthenticator::OnCancelResponse(
    absl::optional<u2f::CancelWebAuthnFlowResponse> response) {
  current_request_id_.clear();

  if (!response) {
    FIDO_LOG(ERROR)
        << "CancelWebAuthnFlow dbus call had no response or timed out";
    return;
  }

  if (!response->canceled()) {
    FIDO_LOG(ERROR) << "Failed to cancel WebAuthn request";
  }
}

void ChromeOSAuthenticator::IsUVPlatformAuthenticatorAvailable(
    base::OnceCallback<void(bool is_uvpaa)> callback) {
  chromeos::U2FClient::IsU2FServiceAvailable(base::BindOnce(
      [](base::OnceCallback<void(bool is_uvpaa)> callback,
         bool is_u2f_service_available) {
        if (!is_u2f_service_available) {
          std::move(callback).Run(false);
          return;
        }

        // TODO(hcyang): Call u2fd here when u2fd is able to decide whether one
        // of the user verification methods exists. Currently WebAuthn
        // supports password authentication so every device with u2fd should
        // be able to perform user verification.
        std::move(callback).Run(true);
      },
      std::move(callback)));
}

void ChromeOSAuthenticator::IsPowerButtonModeEnabled(
    base::OnceCallback<void(bool is_enabled)> callback) {
  chromeos::U2FClient::Get()->IsU2FEnabled(
      u2f::IsUvpaaRequest(),
      base::BindOnce(
          [](base::OnceCallback<void(bool is_enabled)> callback,
             absl::optional<u2f::IsUvpaaResponse> response) {
            std::move(callback).Run(response && response->available());
          },
          std::move(callback)));
}

bool ChromeOSAuthenticator::IsInPairingMode() const {
  return false;
}

bool ChromeOSAuthenticator::IsPaired() const {
  return false;
}

bool ChromeOSAuthenticator::RequiresBlePairingPin() const {
  return false;
}

base::WeakPtr<FidoAuthenticator> ChromeOSAuthenticator::GetWeakPtr() {
  return weak_factory_.GetWeakPtr();
}

}  // namespace device
