// Copyright 2019 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "content/common/font_list.h"

#include "base/bind.h"
#include "base/i18n/rtl.h"
#include "base/strings/string_piece.h"
#include "base/strings/string_util.h"
#include "base/task/sequenced_task_runner.h"
#include "base/task/task_runner_util.h"
#include "base/test/task_environment.h"
#include "base/values.h"
#include "build/build_config.h"
#include "testing/gtest/include/gtest/gtest.h"

namespace {

#if !BUILDFLAG(IS_ANDROID) && !BUILDFLAG(IS_FUCHSIA)
bool HasFontWithName(const base::ListValue& list,
                     base::StringPiece expected_font_id,
                     base::StringPiece expected_display_name) {
  for (const auto& font : list.GetListDeprecated()) {
    const auto& font_names = font.GetListDeprecated();
    std::string font_id = font_names[0].GetString();
    std::string display_name = font_names[1].GetString();
    if (font_id == expected_font_id && display_name == expected_display_name)
      return true;
  }

  return false;
}
#endif  // !BUILDFLAG(IS_ANDROID) && !BUILDFLAG(IS_FUCHSIA)

}  // namespace

#if !BUILDFLAG(IS_ANDROID) && !BUILDFLAG(IS_FUCHSIA)
// GetFontList is not implemented on Android and Fuchsia.
TEST(FontList, GetFontList) {
  base::test::TaskEnvironment task_environment;

  content::GetFontListTaskRunner()->PostTask(
      FROM_HERE, base::BindOnce([] {
        std::unique_ptr<base::ListValue> fonts =
            content::GetFontList_SlowBlocking();
        ASSERT_TRUE(fonts);

#if BUILDFLAG(IS_WIN)
        EXPECT_TRUE(HasFontWithName(*fonts, "MS Gothic", "MS Gothic"));
        EXPECT_TRUE(HasFontWithName(*fonts, "Segoe UI", "Segoe UI"));
        EXPECT_TRUE(HasFontWithName(*fonts, "Verdana", "Verdana"));
#elif BUILDFLAG(IS_LINUX) || BUILDFLAG(IS_CHROMEOS)
        EXPECT_TRUE(HasFontWithName(*fonts, "Arimo", "Arimo"));
#else
        EXPECT_TRUE(HasFontWithName(*fonts, "Arial", "Arial"));
#endif
      }));
  task_environment.RunUntilIdle();
}
#endif  // !BUILDFLAG(IS_ANDROID) && !BUILDFLAG(IS_FUCHSIA)

#if BUILDFLAG(IS_WIN)
TEST(FontList, GetFontListLocalized) {
  base::i18n::SetICUDefaultLocale("ja-JP");
  std::unique_ptr<base::ListValue> ja_fonts =
      content::GetFontList_SlowBlocking();
  ASSERT_TRUE(ja_fonts);
  EXPECT_TRUE(HasFontWithName(*ja_fonts, "MS Gothic", "ＭＳ ゴシック"));

  base::i18n::SetICUDefaultLocale("ko-KR");
  std::unique_ptr<base::ListValue> ko_fonts =
      content::GetFontList_SlowBlocking();
  ASSERT_TRUE(ko_fonts);
  EXPECT_TRUE(HasFontWithName(*ko_fonts, "Malgun Gothic", "맑은 고딕"));
}
#endif  // BUILDFLAG(IS_WIN)

#if BUILDFLAG(IS_MAC)
// On some macOS versions, CTFontManager returns LastResort and/or hidden fonts.
// Ensure that someone (CTFontManager or our FontList code) filters these fonts
// on all OS versions that we support.
TEST(FontList, GetFontListDoesNotIncludeHiddenFonts) {
  std::unique_ptr<base::ListValue> fonts = content::GetFontList_SlowBlocking();

  for (const auto& font : fonts->GetListDeprecated()) {
    const auto& font_names = font.GetListDeprecated();
    const std::string& font_id = font_names[0].GetString();

    // The checks are inspired by Gecko's gfxMacPlatformFontList::AddFamily.
    EXPECT_FALSE(base::LowerCaseEqualsASCII(font_id, "lastresort"))
        << font_id << " seems to be LastResort, which should be filtered";
    EXPECT_FALSE(font_id[0] == '.')
        << font_id << " seems like a hidden font, which should be filtered";
  }
}
#endif  // BUILDFLAG(IS_MAC)
