// Copyright (c) 2021 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "content/browser/ppapi_plugin_sandboxed_process_launcher_delegate.h"

#include "build/build_config.h"
#include "content/public/common/content_switches.h"
#include "sandbox/policy/mojom/sandbox.mojom.h"

#if BUILDFLAG(IS_WIN)
#include "base/win/windows_version.h"
#include "sandbox/policy/win/sandbox_win.h"
#include "sandbox/win/src/process_mitigations.h"
#include "sandbox/win/src/sandbox_policy.h"
#include "ui/display/win/dpi.h"
#include "ui/gfx/font_render_params.h"
#endif

namespace content {
#if BUILDFLAG(IS_WIN)
bool PpapiPluginSandboxedProcessLauncherDelegate::PreSpawnTarget(
    sandbox::TargetPolicy* policy) {
  // The Pepper process is as locked-down as a renderer except that it can
  // create the server side of Chrome pipes.
  sandbox::ResultCode result;
  result = policy->AddRule(sandbox::TargetPolicy::SUBSYS_NAMED_PIPES,
                           sandbox::TargetPolicy::NAMEDPIPES_ALLOW_ANY,
                           L"\\\\.\\pipe\\chrome.*");
  if (result != sandbox::SBOX_ALL_OK)
    return false;

#if !defined(NACL_WIN64)
  // We don't support PPAPI win32k lockdown prior to Windows 10.
  if (base::win::GetVersion() >= base::win::Version::WIN10) {
    result = sandbox::policy::SandboxWin::AddWin32kLockdownPolicy(policy);
    if (result != sandbox::SBOX_ALL_OK)
      return false;
  }
#endif  // !defined(NACL_WIN64)

  // No plugins can generate executable code.
  sandbox::MitigationFlags flags = policy->GetDelayedProcessMitigations();
  flags |= sandbox::MITIGATION_DYNAMIC_CODE_DISABLE;
  if (sandbox::SBOX_ALL_OK != policy->SetDelayedProcessMitigations(flags))
    return false;

  return true;
}
#endif  // BUILDFLAG(IS_WIN)

#if BUILDFLAG(USE_ZYGOTE_HANDLE)
ZygoteHandle PpapiPluginSandboxedProcessLauncherDelegate::GetZygote() {
  const base::CommandLine& browser_command_line =
      *base::CommandLine::ForCurrentProcess();
  base::CommandLine::StringType plugin_launcher =
      browser_command_line.GetSwitchValueNative(switches::kPpapiPluginLauncher);
  if (!plugin_launcher.empty())
    return nullptr;
  return GetGenericZygote();
}
#endif  // BUILDFLAG(USE_ZYGOTE_HANDLE)

sandbox::mojom::Sandbox
PpapiPluginSandboxedProcessLauncherDelegate::GetSandboxType() {
  return sandbox::mojom::Sandbox::kPpapi;
}

#if BUILDFLAG(IS_MAC)
bool PpapiPluginSandboxedProcessLauncherDelegate::DisclaimResponsibility() {
  return true;
}
bool PpapiPluginSandboxedProcessLauncherDelegate::
    EnableCpuSecurityMitigations() {
  return true;
}
#endif

}  // namespace content
