// Copyright (c) 2012 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#import <Cocoa/Cocoa.h>

#import "content/browser/accessibility/browser_accessibility_mac.h"

#include "base/debug/stack_trace.h"
#include "base/threading/thread_task_runner_handle.h"
#include "base/time/time.h"
#import "content/browser/accessibility/browser_accessibility_cocoa.h"
#include "content/browser/accessibility/browser_accessibility_manager_mac.h"

namespace content {

// Static.
std::unique_ptr<BrowserAccessibility> BrowserAccessibility::Create(
    BrowserAccessibilityManager* manager,
    ui::AXNode* node) {
  return std::unique_ptr<BrowserAccessibilityMac>(
      new BrowserAccessibilityMac(manager, node));
}

BrowserAccessibilityMac::BrowserAccessibilityMac(
    BrowserAccessibilityManager* manager,
    ui::AXNode* node)
    : BrowserAccessibility(manager, node) {}

BrowserAccessibilityMac::~BrowserAccessibilityMac() {
  if (platform_node_) {
    platform_node_->Destroy();  // `Destroy()` also deletes the object.
    platform_node_ = nullptr;
  }
}

BrowserAccessibilityCocoa* BrowserAccessibilityMac::GetNativeWrapper() const {
  return platform_node_ ? static_cast<BrowserAccessibilityCocoa*>(
                              platform_node_->GetNativeWrapper())
                        : nullptr;
}

void BrowserAccessibilityMac::OnDataChanged() {
  BrowserAccessibility::OnDataChanged();

  if (GetNativeWrapper()) {
    [GetNativeWrapper() childrenChanged];
    return;
  }

  CreatePlatformNodes();
}

// Replace a native object and refocus if it had focus.
// This will force VoiceOver to re-announce it, and refresh Braille output.
void BrowserAccessibilityMac::ReplaceNativeObject() {
  // Since our native wrapper is owned by a platform node, in order to replace
  // the wrapper, a platform node should always be present. In other words, we
  // could have never called this method without a platform node having been
  // created.
  if (!platform_node_) {
    NOTREACHED() << "No platform node exists, so there should not be any "
                    "native wrapper to replace.";
    return;
  }

  // We need to keep the old native wrapper alive until we set up the new one
  // because we need to retrieve some information from the old wrapper in order
  // to add it to the new one, e.g. its list of children.
  base::scoped_nsobject<AXPlatformNodeCocoa> old_native_obj(
      platform_node_->ReleaseNativeWrapper());

  // We should have never called this method if a native wrapper has not been
  // created, but keep a null check just in case.
  if (!old_native_obj) {
    NOTREACHED() << "No native wrapper exists, so there is nothing to replace.";
    return;
  }

  // Replace child in parent.
  BrowserAccessibility* parent = PlatformGetParent();
  if (!parent)
    return;

  // Re-create native wrapper and also take ownership of that wrapper in
  // `platform_node_` relinquishing the ownership of the old wrapper.
  BrowserAccessibilityCocoa* new_native_obj = CreateNativeWrapper();

  // Rebuild children to pick up a newly created cocoa object.
  [parent->GetNativeViewAccessible() childrenChanged];

  // If focused, fire a focus notification on the new native object.
  if (manager_->GetFocus() == this) {
    NSAccessibilityPostNotification(
        new_native_obj, NSAccessibilityFocusedUIElementChangedNotification);
  }

  // Postpone the old native wrapper destruction. It will be destroyed after
  // a delay so that VO is securely on the new focus first (otherwise the focus
  // event will not be announced).
  // We use 1000ms; however, this magic number isn't necessary to avoid
  // use-after-free or anything scary like that. The worst case scenario if this
  // gets destroyed too early is that VoiceOver announces the wrong thing once.
  base::ThreadTaskRunnerHandle::Get()->PostDelayedTask(
      FROM_HERE,
      base::BindOnce(
          [](base::scoped_nsobject<AXPlatformNodeCocoa> destroyed) {
            if (destroyed && [destroyed instanceActive]) {
              // Follow destruction pattern from NativeReleaseReference().
              [destroyed detach];
            }
          },
          std::move(old_native_obj)),
      base::Milliseconds(1000));
}

uint32_t BrowserAccessibilityMac::PlatformChildCount() const {
  uint32_t child_count = BrowserAccessibility::PlatformChildCount();

  // If this is a table, include the extra fake nodes generated by
  // AXTableInfo, for the column nodes and the table header container, all of
  // which are only important on macOS.
  const std::vector<ui::AXNode*>* extra_mac_nodes = node()->GetExtraMacNodes();
  if (!extra_mac_nodes)
    return child_count;

  return child_count + extra_mac_nodes->size();
}

BrowserAccessibility* BrowserAccessibilityMac::PlatformGetChild(
    uint32_t child_index) const {
  if (child_index < BrowserAccessibility::PlatformChildCount())
    return BrowserAccessibility::PlatformGetChild(child_index);

  if (child_index >= PlatformChildCount())
    return nullptr;

  // If this is a table, include the extra fake nodes generated by
  // AXTableInfo, for the column nodes and the table header container, all of
  // which are only important on macOS.
  const std::vector<ui::AXNode*>* extra_mac_nodes = node()->GetExtraMacNodes();
  if (!extra_mac_nodes || extra_mac_nodes->empty())
    return nullptr;

  child_index -= BrowserAccessibility::PlatformChildCount();
  if (child_index < extra_mac_nodes->size())
    return manager_->GetFromAXNode((*extra_mac_nodes)[child_index]);

  return nullptr;
}

BrowserAccessibility* BrowserAccessibilityMac::PlatformGetFirstChild() const {
  return PlatformGetChild(0);
}

BrowserAccessibility* BrowserAccessibilityMac::PlatformGetLastChild() const {
  const std::vector<ui::AXNode*>* extra_mac_nodes = node()->GetExtraMacNodes();
  if (extra_mac_nodes && !extra_mac_nodes->empty())
    return manager_->GetFromAXNode(extra_mac_nodes->back());
  return BrowserAccessibility::PlatformGetLastChild();
}

BrowserAccessibility* BrowserAccessibilityMac::PlatformGetNextSibling() const {
  BrowserAccessibility* parent = PlatformGetParent();
  if (parent) {
    uint32_t next_child_index = node()->GetUnignoredIndexInParent() + 1;
    if (next_child_index >= parent->InternalChildCount() &&
        next_child_index < parent->PlatformChildCount()) {
      // get the extra_mac_node
      return parent->PlatformGetChild(next_child_index);
    } else if (next_child_index >= parent->PlatformChildCount()) {
      return nullptr;
    }
  }
  return BrowserAccessibility::PlatformGetNextSibling();
}

BrowserAccessibility* BrowserAccessibilityMac::PlatformGetPreviousSibling()
    const {
  BrowserAccessibility* parent = PlatformGetParent();
  if (parent) {
    uint32_t previous_child_index = node()->GetUnignoredIndexInParent() - 1;
    if (previous_child_index >= parent->InternalChildCount() &&
        previous_child_index < parent->PlatformChildCount()) {
      // get the extra_mac_node
      return parent->PlatformGetChild(previous_child_index);
    } else if (previous_child_index < 0) {
      return nullptr;
    }
  }
  return BrowserAccessibility::PlatformGetPreviousSibling();
}

gfx::NativeViewAccessible BrowserAccessibilityMac::GetNativeViewAccessible() {
  return GetNativeWrapper();
}

ui::AXPlatformNode* BrowserAccessibilityMac::GetAXPlatformNode() const {
  return platform_node_;
}

void BrowserAccessibilityMac::CreatePlatformNodes() {
  DCHECK(!platform_node_);
  platform_node_ =
      static_cast<ui::AXPlatformNodeMac*>(ui::AXPlatformNode::Create(this));
  CreateNativeWrapper();
}

BrowserAccessibilityCocoa* BrowserAccessibilityMac::CreateNativeWrapper() {
  DCHECK(platform_node_);

  BrowserAccessibilityCocoa* node_cocoa =
      [[BrowserAccessibilityCocoa alloc] initWithObject:this
                                       withPlatformNode:platform_node_];

  // `AXPlatformNodeMac` takes ownership of the Cocoa object here.
  platform_node_->SetNativeWrapper(node_cocoa);
  return node_cocoa;
}

}  // namespace content
