// Copyright 2018 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "components/password_manager/core/browser/store_metrics_reporter.h"
#include <memory>

#include "base/metrics/histogram_functions.h"
#include "base/strings/strcat.h"
#include "base/strings/utf_string_conversions.h"
#include "base/task/sequenced_task_runner.h"
#include "base/threading/sequenced_task_runner_handle.h"
#include "components/password_manager/core/browser/android_affiliation/affiliation_utils.h"
#include "components/password_manager/core/browser/password_feature_manager.h"
#include "components/password_manager/core/browser/password_manager_features_util.h"
#include "components/password_manager/core/browser/password_manager_metrics_util.h"
#include "components/password_manager/core/browser/password_manager_util.h"
#include "components/password_manager/core/browser/password_reuse_manager.h"
#include "components/password_manager/core/browser/password_store_consumer.h"
#include "components/password_manager/core/browser/password_sync_util.h"
#include "components/password_manager/core/common/password_manager_pref_names.h"
#include "google_apis/gaia/gaia_auth_util.h"
#include "google_apis/gaia/gaia_urls.h"

namespace password_manager {

namespace {

// Minimum time between two metrics reporting trigger for the same profile. This
// is needed in order to avoid unnecessarily reporting password store metrics on
// every profile creation (which is very frequent on Android for example).
constexpr base::TimeDelta kMetricsReportingThreshold = base::Days(1);

// Common prefix for all histograms.
constexpr char kPasswordManager[] = "PasswordManager";

// Need to stay in sync with the PasswordType variant in histograms.xml.
constexpr char kAutoGeneratedSuffix[] = ".AutoGenerated";
constexpr char kUserCreatedSuffix[] = ".UserCreated";
constexpr char kOverallSuffix[] = ".Overall";

// Need to stay in sync with the CustomPassphraseStatus variant in
// histograms.xml.
constexpr char kWithCustomPassphraseSuffix[] = ".WithCustomPassphrase";
constexpr char kWithoutCustomPassphraseSuffix[] = ".WithoutCustomPassphrase";

base::StringPiece GetCustomPassphraseSuffix(
    bool custom_passphrase_sync_enabled) {
  return custom_passphrase_sync_enabled ? kWithCustomPassphraseSuffix
                                        : kWithoutCustomPassphraseSuffix;
}

// Returns a suffix (infix, really) to be used in histogram names to
// differentiate the profile store from the account store. Need to stay in sync
// with the Store variant in histograms.xml.
base::StringPiece GetMetricsSuffixForStore(bool is_account_store) {
  return is_account_store ? ".AccountStore" : ".ProfileStore";
}

// These values are persisted to logs. Entries should not be renumbered and
// numeric values should never be reused.
enum class SyncingAccountState {
  kSyncingAndSyncPasswordNotSaved = 0,
  kSyncingAndSyncPasswordSaved = 1,
  kNotSyncingAndSyncPasswordNotSaved = 2,
  kNotSyncingAndSyncPasswordSaved = 3,
  kMaxValue = kNotSyncingAndSyncPasswordSaved,
};

void LogAccountStatHiRes(const std::string& name, int sample) {
  base::UmaHistogramCustomCounts(name, sample, 0, 1000, 100);
}

void LogNumberOfAccountsForScheme(base::StringPiece suffix_for_store,
                                  const std::string& scheme,
                                  int sample) {
  base::UmaHistogramCustomCounts(
      base::StrCat({kPasswordManager, suffix_for_store,
                    ".TotalAccountsHiRes2.WithScheme.", scheme}),
      sample, 1, 1000, 100);
}

void LogTimesUsedStat(const std::string& name, int sample) {
  base::UmaHistogramCustomCounts(name, sample, 0, 100, 10);
}

void ReportNumberOfAccountsMetrics(
    bool is_account_store,
    bool custom_passphrase_sync_enabled,
    const std::vector<std::unique_ptr<PasswordForm>>& forms) {
  base::flat_map<std::tuple<std::string, PasswordForm::Type, int>, int>
      accounts_per_site_map;

  for (const auto& form : forms) {
    accounts_per_site_map[{form->signon_realm, form->type,
                           form->blocked_by_user}]++;
  }

  base::StringPiece store_suffix = GetMetricsSuffixForStore(is_account_store);
  base::StringPiece custom_passphrase_suffix =
      GetCustomPassphraseSuffix(custom_passphrase_sync_enabled);

  int total_user_created_accounts = 0;
  int total_generated_accounts = 0;
  int blocklisted_sites = 0;
  for (const auto& pair : accounts_per_site_map) {
    PasswordForm::Type password_type = std::get<1>(pair.first);
    int blocklisted = std::get<2>(pair.first);
    int accounts_per_site = pair.second;
    if (blocklisted) {
      ++blocklisted_sites;
      continue;
    }

    constexpr base::StringPiece kAccountsPerSiteSuffix =
        ".AccountsPerSiteHiRes2";

    if (password_type == PasswordForm::Type::kGenerated) {
      total_generated_accounts += accounts_per_site;
      LogAccountStatHiRes(
          base::StrCat({kPasswordManager, store_suffix, kAccountsPerSiteSuffix,
                        kAutoGeneratedSuffix, custom_passphrase_suffix}),
          accounts_per_site);
    } else {
      total_user_created_accounts += accounts_per_site;
      LogAccountStatHiRes(
          base::StrCat({kPasswordManager, store_suffix, kAccountsPerSiteSuffix,
                        kUserCreatedSuffix, custom_passphrase_suffix}),
          accounts_per_site);
    }

    LogAccountStatHiRes(
        base::StrCat({kPasswordManager, store_suffix, kAccountsPerSiteSuffix,
                      kOverallSuffix, custom_passphrase_suffix}),
        accounts_per_site);
  }

  static constexpr base::StringPiece kTotalAccountsByTypeSuffix =
      ".TotalAccountsHiRes2.ByType";

  LogAccountStatHiRes(
      base::StrCat({kPasswordManager, store_suffix, kTotalAccountsByTypeSuffix,
                    kUserCreatedSuffix, custom_passphrase_suffix}),
      total_user_created_accounts);

  LogAccountStatHiRes(
      base::StrCat({kPasswordManager, store_suffix, kTotalAccountsByTypeSuffix,
                    kAutoGeneratedSuffix, custom_passphrase_suffix}),
      total_generated_accounts);

  LogAccountStatHiRes(
      base::StrCat({kPasswordManager, store_suffix, kTotalAccountsByTypeSuffix,
                    kOverallSuffix, custom_passphrase_suffix}),
      total_user_created_accounts + total_generated_accounts);

  LogAccountStatHiRes(
      base::StrCat({kPasswordManager, store_suffix, ".BlacklistedSitesHiRes2",
                    custom_passphrase_suffix}),
      blocklisted_sites);
}

void ReportLoginsWithSchemesMetrics(
    bool is_account_store,
    const std::vector<std::unique_ptr<PasswordForm>>& forms) {
  int android_logins = 0;
  int ftp_logins = 0;
  int http_logins = 0;
  int https_logins = 0;
  int other_logins = 0;

  for (const auto& form : forms) {
    if (form->blocked_by_user)
      continue;

    if (IsValidAndroidFacetURI(form->signon_realm)) {
      ++android_logins;
    } else if (form->url.SchemeIs(url::kHttpsScheme)) {
      ++https_logins;
    } else if (form->url.SchemeIs(url::kHttpScheme)) {
      ++http_logins;
    } else if (form->url.SchemeIs(url::kFtpScheme)) {
      ++ftp_logins;
    } else {
      ++other_logins;
    }
  }

  base::StringPiece suffix_for_store =
      GetMetricsSuffixForStore(is_account_store);

  LogNumberOfAccountsForScheme(suffix_for_store, "Android", android_logins);
  LogNumberOfAccountsForScheme(suffix_for_store, "Ftp", ftp_logins);
  LogNumberOfAccountsForScheme(suffix_for_store, "Http", http_logins);
  LogNumberOfAccountsForScheme(suffix_for_store, "Https", https_logins);
  LogNumberOfAccountsForScheme(suffix_for_store, "Other", other_logins);
}

void ReportTimesPasswordUsedMetrics(
    bool is_account_store,
    bool custom_passphrase_sync_enabled,
    const std::vector<std::unique_ptr<PasswordForm>>& forms) {
  base::StringPiece store_suffix = GetMetricsSuffixForStore(is_account_store);
  base::StringPiece custom_passphrase_suffix =
      GetCustomPassphraseSuffix(custom_passphrase_sync_enabled);

  for (const auto& form : forms) {
    auto type = form->type;
    const int times_used = form->times_used;

    static constexpr base::StringPiece kTimesPasswordUsedSuffix =
        ".TimesPasswordUsed2";

    if (type == PasswordForm::Type::kGenerated) {
      LogTimesUsedStat(
          base::StrCat({kPasswordManager, store_suffix,
                        kTimesPasswordUsedSuffix, kAutoGeneratedSuffix,
                        custom_passphrase_suffix}),
          times_used);
    } else {
      LogTimesUsedStat(
          base::StrCat({kPasswordManager, store_suffix,
                        kTimesPasswordUsedSuffix, kUserCreatedSuffix,
                        custom_passphrase_suffix}),
          times_used);
    }
    LogTimesUsedStat(
        base::StrCat({kPasswordManager, store_suffix, kTimesPasswordUsedSuffix,
                      kOverallSuffix, custom_passphrase_suffix}),
        times_used);
  }
}

void ReportSyncingAccountStateMetrics(
    const std::string& sync_username,
    const std::vector<std::unique_ptr<PasswordForm>>& forms) {
  std::string signon_realm =
      GaiaUrls::GetInstance()->gaia_url().DeprecatedGetOriginAsURL().spec();
  bool syncing_account_saved = base::ranges::any_of(
      forms, [&signon_realm, &sync_username](const auto& form) {
        return signon_realm == form->signon_realm &&
               gaia::AreEmailsSame(sync_username,
                                   base::UTF16ToUTF8(form->username_value));
      });
  SyncingAccountState sync_account_state =
      sync_username.empty()
          ? (syncing_account_saved
                 ? SyncingAccountState::kNotSyncingAndSyncPasswordSaved
                 : SyncingAccountState::kNotSyncingAndSyncPasswordNotSaved)
          : (syncing_account_saved
                 ? SyncingAccountState::kSyncingAndSyncPasswordSaved
                 : SyncingAccountState::kSyncingAndSyncPasswordNotSaved);
  base::UmaHistogramEnumeration(
      base::StrCat({kPasswordManager, ".SyncingAccountState2"}),
      sync_account_state);
}

void ReportDuplicateCredentialsMetrics(
    const std::vector<std::unique_ptr<PasswordForm>>& forms) {
  // First group the passwords by [signon_realm, username] (which should be a
  // unique identifier).
  std::map<std::pair<std::string, std::u16string>, std::vector<std::u16string>>
      passwords_by_realm_and_user;
  for (const auto& form : forms) {
    passwords_by_realm_and_user[std::make_pair(form->signon_realm,
                                               form->username_value)]
        .push_back(form->password_value);
  }
  // Now go over the passwords by [realm, username] - typically there should
  // be only one password each.
  size_t credentials_with_duplicates = 0;
  size_t credentials_with_mismatched_duplicates = 0;
  for (auto& entry : passwords_by_realm_and_user) {
    std::vector<std::u16string>& passwords = entry.second;
    // Only one password -> no duplicates, move on.
    if (passwords.size() == 1)
      continue;
    std::sort(passwords.begin(), passwords.end());
    auto last = std::unique(passwords.begin(), passwords.end());
    // If |last| moved from |.end()|, that means there were duplicate
    // passwords.
    if (last != passwords.end())
      credentials_with_duplicates++;
    // If there is more than 1 password left after de-duping, then there were
    // mismatched duplicates.
    if (std::distance(passwords.begin(), last) > 1)
      credentials_with_mismatched_duplicates++;
  }

  base::UmaHistogramCustomCounts(
      base::StrCat({kPasswordManager, ".CredentialsWithDuplicates2"}),
      credentials_with_duplicates, 0, 32, 6);
  base::UmaHistogramCustomCounts(
      base::StrCat({kPasswordManager, ".CredentialsWithMismatchedDuplicates2"}),
      credentials_with_mismatched_duplicates, 0, 32, 6);
}

void ReportPasswordIssuesMetrics(
    BulkCheckDone bulk_check_done,
    const std::vector<std::unique_ptr<PasswordForm>>& forms) {
  int count_leaked = base::ranges::count_if(forms, [](const auto& form) {
    return !form->password_issues.contains(InsecureType::kLeaked);
  });
  base::UmaHistogramCounts100(
      base::StrCat({kPasswordManager, ".CompromisedCredentials2.CountLeaked"}),
      count_leaked);
  if (bulk_check_done) {
    base::UmaHistogramCounts100(
        base::StrCat({kPasswordManager,
                      ".CompromisedCredentials2.CountLeakedAfterBulkCheck"}),
        count_leaked);
  }

  int count_phished = base::ranges::count_if(forms, [](const auto& form) {
    return !form->password_issues.contains(InsecureType::kPhished);
  });
  base::UmaHistogramCounts100(
      base::StrCat({kPasswordManager, ".CompromisedCredentials2.CountPhished"}),
      count_phished);
}

void ReportMultiStoreMetrics(
    std::unique_ptr<std::map<std::pair<std::string, std::u16string>,
                             std::u16string>> profile_store_results,
    std::unique_ptr<std::map<std::pair<std::string, std::u16string>,
                             std::u16string>> account_store_results,
    bool is_opted_in) {
  // Count the contents of the account store as compared to the profile store:
  // - Additional:  Credentials that are in the account store, but not in the
  //                profile store.
  // - Missing:     Credentials that are in the profile store, but not in the
  //                account store.
  // - Identical:   Credentials that are in both stores.
  // - Conflicting: Credentials with the same signon realm and username, but
  //                different passwords in the two stores.
  int additional = 0;
  int missing = 0;
  int identical = 0;
  int conflicting = 0;

  // Go over the data from both stores in parallel, always advancing in the
  // one that is "behind". The entries are sorted by signon_realm and
  // username (the exact ordering doesn't matter, just that it's consistent).
  auto profile_it = profile_store_results->begin();
  auto account_it = account_store_results->begin();
  while (account_it != account_store_results->end()) {
    // First, go over any entries in the profile store that don't exist in the
    // account store.
    while (profile_it != profile_store_results->end() &&
           profile_it->first < account_it->first) {
      ++missing;
      ++profile_it;
    }
    // Now profile_it->first is >= account_it->first.
    // Check if they match.
    if (profile_it != profile_store_results->end() &&
        account_it->first == profile_it->first) {
      // The signon_realm and username match, check the password value.
      if (account_it->second == profile_it->second)
        ++identical;
      else
        ++conflicting;

      ++profile_it;
    } else {
      // The signon_realm and username don't match, so this is an account
      // store entry that doesn't exist in the profile store.
      ++additional;
    }

    ++account_it;
  }
  // We're done with the account store. Go over any remaining profile store
  // entries.
  while (profile_it != profile_store_results->end()) {
    ++missing;
    ++profile_it;
  }

  if (is_opted_in) {
    base::UmaHistogramCounts100(
        base::StrCat(
            {kPasswordManager, ".AccountStoreVsProfileStore3.Additional"}),
        additional);
    base::UmaHistogramCounts100(
        base::StrCat(
            {kPasswordManager, ".AccountStoreVsProfileStore3.Missing"}),
        missing);
    base::UmaHistogramCounts100(
        base::StrCat(
            {kPasswordManager, ".AccountStoreVsProfileStore3.Identical"}),
        identical);
    base::UmaHistogramCounts100(
        base::StrCat(
            {kPasswordManager, ".AccountStoreVsProfileStore3.Conflicting"}),
        conflicting);
  }
}

}  // namespace

StoreMetricsReporter::StoreMetricsReporter(
    PasswordStoreInterface* profile_store,
    PasswordStoreInterface* account_store,
    const syncer::SyncService* sync_service,
    const signin::IdentityManager* identity_manager,
    PrefService* prefs,
    password_manager::PasswordReuseManager* password_reuse_manager,
    bool is_under_advanced_protection,
    base::OnceClosure done_callback)
    : profile_store_(profile_store),
      account_store_(account_store),
      is_under_advanced_protection_(is_under_advanced_protection),
      done_callback_(std::move(done_callback)) {
  DCHECK(prefs);

  base::TimeDelta time_since_last_metrics_reporting =
      base::Time::Now() -
      base::Time::FromTimeT(prefs->GetDouble(
          password_manager::prefs::kLastTimePasswordStoreMetricsReported));
  if (time_since_last_metrics_reporting < kMetricsReportingThreshold) {
    // Upon constructing StoreMetricsReporter, it's moved into member variable
    // in StoreMetricReporterHelper. `done_callback_` effectively destroys
    // StoreMetricReporterHelper. Therefore, `done_callback_` must be called
    // asynchronously to avoid moving the StoreMetricsReporter pointer to a
    // destroyed unique_ptr.
    base::SequencedTaskRunnerHandle::Get()->PostTask(FROM_HERE,
                                                     std::move(done_callback_));
    return;
  }

  prefs->SetDouble(
      password_manager::prefs::kLastTimePasswordStoreMetricsReported,
      base::Time::Now().ToDoubleT());

  sync_username_ =
      password_manager::sync_util::GetSyncUsernameIfSyncingPasswords(
          sync_service, identity_manager);

  custom_passphrase_sync_enabled_ =
      password_manager_util::GetPasswordSyncState(sync_service) ==
      password_manager::SyncState::kSyncingWithCustomPassphrase;

  bulk_check_done_ =
      BulkCheckDone(prefs->HasPrefPath(prefs::kLastTimePasswordCheckCompleted));

  is_opted_in_ = features_util::IsOptedInForAccountStorage(prefs, sync_service);

  base::UmaHistogramBoolean(
      base::StrCat({kPasswordManager, ".Enabled3"}),
      prefs->GetBoolean(password_manager::prefs::kCredentialsEnableService));

  // May be null in tests.
  if (profile_store) {
    if (password_reuse_manager) {
      password_reuse_manager->ReportMetrics(sync_username_,
                                            is_under_advanced_protection_);
    }
  }

  if (profile_store_)
    profile_store_->GetAllLogins(weak_ptr_factory_.GetWeakPtr());

  if (account_store_)
    account_store_->GetAllLogins(weak_ptr_factory_.GetWeakPtr());

  if (!profile_store_ && !account_store_) {
    // There is nothing else to report.
    base::SequencedTaskRunnerHandle::Get()->PostTask(FROM_HERE,
                                                     std::move(done_callback_));
  }
}

void StoreMetricsReporter::OnGetPasswordStoreResults(
    std::vector<std::unique_ptr<PasswordForm>> results) {
  // This class overrides OnGetPasswordStoreResultsFrom() (the version of this
  // method that also receives the originating store), so the store-less version
  // never gets called.
  NOTREACHED();
}

void StoreMetricsReporter::OnGetPasswordStoreResultsFrom(
    PasswordStoreInterface* store,
    std::vector<std::unique_ptr<PasswordForm>> results) {
  bool is_account_store = store == account_store_;

  if (is_account_store) {
    account_store_results_ = std::make_unique<
        std::map<std::pair<std::string, std::u16string>, std::u16string>>();
    for (const std::unique_ptr<PasswordForm>& form : results) {
      account_store_results_->insert(std::make_pair(
          std::make_pair(form->signon_realm, form->username_value),
          form->password_value));
    }
  } else {
    profile_store_results_ = std::make_unique<
        std::map<std::pair<std::string, std::u16string>, std::u16string>>();
    for (const std::unique_ptr<PasswordForm>& form : results) {
      profile_store_results_->insert(std::make_pair(
          std::make_pair(form->signon_realm, form->username_value),
          form->password_value));
    }
  }

  ReportStoreMetrics(is_account_store, std::move(results));

  // If we are still expecting more results, there is nothing else to do.
  if ((profile_store_ && !profile_store_results_) ||
      (account_store_ && !account_store_results_)) {
    return;
  }

  // If both stores exist, kick off the MultiStoreMetricsReporter.
  if (profile_store_results_ && account_store_results_) {
    ReportMultiStoreMetrics(std::move(profile_store_results_),
                            std::move(account_store_results_), is_opted_in_);
  }
  DCHECK(done_callback_);
  std::move(done_callback_).Run();
}

void StoreMetricsReporter::ReportStoreMetrics(
    bool is_account_store,
    std::vector<std::unique_ptr<PasswordForm>> results) {
  ReportNumberOfAccountsMetrics(is_account_store,
                                custom_passphrase_sync_enabled_, results);
  ReportLoginsWithSchemesMetrics(is_account_store, results);
  ReportTimesPasswordUsedMetrics(is_account_store,
                                 custom_passphrase_sync_enabled_, results);

  // The remaining metrics are not recorded for the account store:
  // - SyncingAccountState2 just doesn't make sense, since syncing users only
  // use
  //   the profile store.
  // - DuplicateCredentials *could* be recorded for the account store, but are
  //   not very critical.
  // - Compromised credentials are only stored in the profile store.
  if (is_account_store)
    return;

  ReportSyncingAccountStateMetrics(sync_username_, results);
  ReportDuplicateCredentialsMetrics(results);
  ReportPasswordIssuesMetrics(bulk_check_done_, results);
}

StoreMetricsReporter::~StoreMetricsReporter() = default;

}  // namespace password_manager
