// Copyright 2021 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "components/password_manager/core/browser/password_store_proxy_backend.h"
#include <memory>
#include <utility>
#include <vector>

#include "base/barrier_callback.h"
#include "base/barrier_closure.h"
#include "base/callback.h"
#include "base/callback_helpers.h"
#include "base/functional/identity.h"
#include "base/memory/scoped_refptr.h"
#include "base/metrics/histogram_functions.h"
#include "base/ranges/algorithm.h"
#include "base/strings/strcat.h"
#include "components/password_manager/core/browser/field_info_table.h"
#include "components/password_manager/core/common/password_manager_features.h"
#include "components/password_manager/core/common/password_manager_pref_names.h"
#include "components/prefs/pref_service.h"
#include "components/sync/model/proxy_model_type_controller_delegate.h"

namespace password_manager {

namespace {

bool ShouldExecuteModifyOperationsOnShadowBackend(PrefService* prefs,
                                                  bool is_syncing) {
  // TODO(crbug.com/1306001): Reenable or clean up for local-only users.
  return false;
}

bool ShouldExecuteReadOperationsOnShadowBackend(PrefService* prefs,
                                                bool is_syncing) {
  if (ShouldExecuteModifyOperationsOnShadowBackend(prefs, is_syncing)) {
    // Read operations are always allowed whenever modifications are allowed.
    // i.e. necessary migrations have happened and appropriate flags are set.
    return true;
  }

  if (!is_syncing)
    return false;

  if (!base::FeatureList::IsEnabled(features::kUnifiedPasswordManagerAndroid))
    return false;

  features::UpmExperimentVariation variation =
      features::kUpmExperimentVariationParam.Get();
  switch (variation) {
    case features::UpmExperimentVariation::kEnableForSyncingUsers:
    case features::UpmExperimentVariation::kEnableOnlyBackendForSyncingUsers:
    case features::UpmExperimentVariation::kEnableForAllUsers:
      return false;
    case features::UpmExperimentVariation::kShadowSyncingUsers:
      return true;
  }
  NOTREACHED() << "Define explicitly whether shadow traffic is recorded!";
  return false;
}

bool ShouldExecuteDeletionsOnShadowBackend(PrefService* prefs,
                                           bool is_syncing) {
  if (ShouldExecuteModifyOperationsOnShadowBackend(prefs, is_syncing))
    return true;

  if (!is_syncing)
    return false;

  if (!base::FeatureList::IsEnabled(features::kUnifiedPasswordManagerAndroid))
    return false;

  features::UpmExperimentVariation variation =
      features::kUpmExperimentVariationParam.Get();
  switch (variation) {
    case features::UpmExperimentVariation::kEnableForSyncingUsers:
    case features::UpmExperimentVariation::kEnableOnlyBackendForSyncingUsers:
      return true;
    case features::UpmExperimentVariation::kEnableForAllUsers:
      // All passwords are in the remote storage. There should not be a
      // shadow backend anymore.
      return false;
    case features::UpmExperimentVariation::kShadowSyncingUsers:
      return false;
  }
  NOTREACHED()
      << "Define explicitly whether deletions on both backends are required!";
  return false;
}

// This helper is used to determine main *and* shadow backends. Technically,
// some "Enable" groups don't require shadow traffic but they use it for safe
// deletions.
bool UsesAndroidBackendAsMainBackend(bool is_syncing) {
  if (!is_syncing)
    return false;

  if (!base::FeatureList::IsEnabled(features::kUnifiedPasswordManagerAndroid))
    return false;

  features::UpmExperimentVariation variation =
      features::kUpmExperimentVariationParam.Get();
  switch (variation) {
    case features::UpmExperimentVariation::kEnableForSyncingUsers:
    case features::UpmExperimentVariation::kEnableOnlyBackendForSyncingUsers:
    case features::UpmExperimentVariation::kEnableForAllUsers:
      return true;
    case features::UpmExperimentVariation::kShadowSyncingUsers:
      return false;
  }
  NOTREACHED() << "Define explicitly whether Android is the main backend!";
  return false;
}

bool IsBuiltInBackendSyncEnabled() {
  DCHECK(
      base::FeatureList::IsEnabled(features::kUnifiedPasswordManagerAndroid));

  features::UpmExperimentVariation variation =
      features::kUpmExperimentVariationParam.Get();
  switch (variation) {
    case features::UpmExperimentVariation::kEnableForSyncingUsers:
    case features::UpmExperimentVariation::kShadowSyncingUsers:
    case features::UpmExperimentVariation::kEnableOnlyBackendForSyncingUsers:
      return true;
    case features::UpmExperimentVariation::kEnableForAllUsers:
      return false;
  }
  NOTREACHED() << "Define which backend handles sync change callbacks!";
  return false;
}

void CallOnSyncEnabledOrDisabledForEnabledBackend(
    bool originates_from_android,
    base::RepeatingClosure sync_enabled_or_disabled_cb) {
  if (IsBuiltInBackendSyncEnabled()) {
    if (!originates_from_android) {
      sync_enabled_or_disabled_cb.Run();
    }
    return;
  }
  sync_enabled_or_disabled_cb.Run();
}

using MethodName = base::StrongAlias<struct MethodNameTag, std::string>;

struct LoginsResultImpl {
  using ResultType = LoginsResult;
  using ElementsType = LoginsResult;

  static LoginsResult* GetElements(LoginsResult& logins) { return &logins; }

  static std::unique_ptr<PasswordForm> Clone(
      const std::unique_ptr<PasswordForm>& login) {
    return std::make_unique<PasswordForm>(*login);
  }

  static bool IsLess(const std::unique_ptr<PasswordForm>& lhs,
                     const std::unique_ptr<PasswordForm>& rhs) {
    return PasswordFormUniqueKey(*lhs) < PasswordFormUniqueKey(*rhs);
  }

  static bool HaveInconsistentPasswords(
      const std::unique_ptr<PasswordForm>& lhs,
      const std::unique_ptr<PasswordForm>& rhs) {
    return lhs->password_value != rhs->password_value;
  }
};

struct LoginsResultOrErrorImpl {
  using ResultType = LoginsResultOrError;
  using ElementsType = LoginsResult;

  static LoginsResult* GetElements(LoginsResultOrError& logins_or_error) {
    return absl::holds_alternative<PasswordStoreBackendError>(logins_or_error)
               ? nullptr
               : &absl::get<LoginsResult>(logins_or_error);
  }

  static std::unique_ptr<PasswordForm> Clone(
      const std::unique_ptr<PasswordForm>& login) {
    return std::make_unique<PasswordForm>(*login);
  }

  static bool IsLess(const std::unique_ptr<PasswordForm>& lhs,
                     const std::unique_ptr<PasswordForm>& rhs) {
    return PasswordFormUniqueKey(*lhs) < PasswordFormUniqueKey(*rhs);
  }

  static bool HaveInconsistentPasswords(
      const std::unique_ptr<PasswordForm>& lhs,
      const std::unique_ptr<PasswordForm>& rhs) {
    return lhs->password_value != rhs->password_value;
  }
};

struct PasswordStoreChangeListImpl {
  using ResultType = absl::optional<PasswordStoreChangeList>;
  using ElementsType = PasswordStoreChangeList;

  static PasswordStoreChangeList* GetElements(
      absl::optional<PasswordStoreChangeList>& changelist) {
    return changelist.has_value() ? &changelist.value() : nullptr;
  }

  static PasswordStoreChange Clone(const PasswordStoreChange& change) {
    return change;
  }

  static bool IsLess(const PasswordStoreChange& lhs,
                     const PasswordStoreChange& rhs) {
    return std::forward_as_tuple(PasswordFormUniqueKey(lhs.form()),
                                 lhs.type()) <
           std::forward_as_tuple(PasswordFormUniqueKey(rhs.form()), rhs.type());
  }

  static bool HaveInconsistentPasswords(const PasswordStoreChange& lhs,
                                        const PasswordStoreChange& rhs) {
    // We never consider PasswordStoreChange having inconsistent passwords.
    return false;
  }
};

void InvokeCallbackWithCombinedStatus(base::OnceCallback<void(bool)> completion,
                                      std::vector<bool> statuses) {
  std::move(completion).Run(base::ranges::all_of(statuses, base::identity()));
}

// Records the difference metrics between |main_result| and |backend_result|
// when returned by |method_name|. |main_result| and |backend_result| must be
// vectors of type T. |is_less| can be used to compare two objects of type T.
// |is_inconsistent| can be used to compute if the two objects have inconsistent
// password values.
template <typename T, typename IsLess, typename IsInconsistent>
void RecordMetrics(const MethodName& method_name,
                   const std::vector<T>& main_result,
                   const std::vector<T>& backend_result,
                   IsLess is_less,
                   IsInconsistent is_inconsistent) {
  // Comparison is done by creating two sets that contain pointers to the
  // objects stored in |main_result| and |backend_result|. Using the passed
  // comparison methods, we compute and report metrics regarding the difference
  // between both result vectors.
  auto is_less_ptr = [is_less](const T* lhs, const T* rhs) {
    return is_less(*lhs, *rhs);
  };

  auto address_of = [](const T& object) { return &object; };

  auto main_elements =
      base::MakeFlatSet<const T*>(main_result, is_less_ptr, address_of);
  auto shadow_elements =
      base::MakeFlatSet<const T*>(backend_result, is_less_ptr, address_of);

  auto common_elements = [&] {
    std::vector<const T*> vec;
    vec.reserve(main_elements.size());
    base::ranges::set_intersection(main_elements, shadow_elements,
                                   std::back_inserter(vec), is_less_ptr);
    return base::flat_set<const T*, decltype(is_less_ptr)>(std::move(vec),
                                                           is_less_ptr);
  }();

  // The cardinalities from which we compute the metrics.
  size_t main_minus_shadow = main_elements.size() - common_elements.size();
  size_t shadow_minus_main = shadow_elements.size() - common_elements.size();
  size_t diff = main_minus_shadow + shadow_minus_main;
  size_t total = diff + common_elements.size();
  size_t inconsistent = base::ranges::count_if(common_elements, [&](auto* f) {
    auto lhs = main_elements.find(f);
    auto rhs = shadow_elements.find(f);
    DCHECK(lhs != main_elements.end());
    DCHECK(rhs != shadow_elements.end());
    return (*is_inconsistent)(**lhs, **rhs);
  });

  // Emits a pair of absolute and relative metrics.
  auto Emit = [&method_name](base::StringPiece metric_infix, size_t nominator,
                             size_t denominator) {
    std::string prefix =
        base::StrCat({"PasswordManager.PasswordStoreProxyBackend.",
                      method_name.value(), ".", metric_infix, "."});
    base::UmaHistogramCounts1M(prefix + "Abs", nominator);
    if (denominator != 0) {
      size_t ceiling_of_percentage =
          (nominator * 100 + denominator - 1) / denominator;
      base::UmaHistogramPercentage(prefix + "Rel", ceiling_of_percentage);
    }
  };
  Emit("Diff", diff, total);
  Emit("MainMinusShadow", main_minus_shadow, total);
  Emit("ShadowMinusMain", shadow_minus_main, total);
  Emit("InconsistentPasswords", inconsistent, common_elements.size());
}

// Records the metrics of a pair of MethodName calls to the main and
// the shadow backends once both calls are finished. MethodName() is expected to
// return an std::vector<ApiMethodImpl::ResultType>. ApiMethodImpl classes need
// to provide 4 methods:
// - GetElements(): returns the elements to be compared
// - Clone(): Returns a copy of an element, used to cache the main results.
// - IsLess(): to compare elements.
// - HaveInconsistentPasswords(): Whether elements have inconsistent passwords
//
// The class is ref-counted because it is equally owned by the two parallel
// method calls : it must outlive the first returning one and shall  be
// destroyed after the second one returns.
template <typename ApiMethodImpl>
class ShadowTrafficMetricsRecorder
    : public base::RefCounted<ShadowTrafficMetricsRecorder<ApiMethodImpl>> {
 public:
  explicit ShadowTrafficMetricsRecorder(MethodName method_name)
      : method_name_(std::move(method_name)) {}

  // Returns the unchanged |result| so it can be passed to the main handler.
  typename ApiMethodImpl::ResultType RecordMainResult(
      typename ApiMethodImpl::ResultType result) {
    if (auto* elements = ApiMethodImpl::GetElements(result)) {
      if (!first_result_) {
        first_result_ =
            absl::make_optional<typename ApiMethodImpl::ElementsType>();
        first_result_->reserve(elements->size());
        for (const auto& e : *elements)
          first_result_->push_back(ApiMethodImpl::Clone(e));
      } else {
        RecordMetrics(method_name_, /*main_result=*/*elements,
                      /*shadow_result=*/*first_result_, &ApiMethodImpl::IsLess,
                      &ApiMethodImpl::HaveInconsistentPasswords);
      }
    }

    return result;
  }

  void RecordShadowResult(typename ApiMethodImpl::ResultType result) {
    if (auto* elements = ApiMethodImpl::GetElements(result)) {
      if (!first_result_) {
        first_result_ = std::move(*elements);
      } else {
        RecordMetrics(method_name_,
                      /*main_result=*/*first_result_,
                      /*shadow_result=*/*elements, &ApiMethodImpl::IsLess,
                      &ApiMethodImpl::HaveInconsistentPasswords);
      }
    }
  }

 private:
  friend class base::RefCounted<ShadowTrafficMetricsRecorder<ApiMethodImpl>>;
  ~ShadowTrafficMetricsRecorder() = default;

  // Stores the result of the backend that returns first.
  absl::optional<typename ApiMethodImpl::ElementsType> first_result_;
  const MethodName method_name_;
};

}  // namespace

PasswordStoreProxyBackend::PasswordStoreProxyBackend(
    PasswordStoreBackend* built_in_backend,
    PasswordStoreBackend* android_backend,
    PrefService* prefs,
    SyncDelegate* sync_delegate)
    : built_in_backend_(built_in_backend),
      android_backend_(android_backend),
      prefs_(prefs),
      sync_delegate_(sync_delegate) {}

PasswordStoreProxyBackend::~PasswordStoreProxyBackend() = default;

void PasswordStoreProxyBackend::InitBackend(
    RemoteChangesReceived remote_form_changes_received,
    base::RepeatingClosure sync_enabled_or_disabled_cb,
    base::OnceCallback<void(bool)> completion) {
  base::RepeatingCallback<void(bool)> pending_initialization_calls =
      base::BarrierCallback<bool>(
          /*num_callbacks=*/2, base::BindOnce(&InvokeCallbackWithCombinedStatus,
                                              std::move(completion)));

  // Both backends need to be initialized, so using the helpers for main/shadow
  // backend is unnecessary and won't work since the sync status may not be
  // available yet.
  built_in_backend_->InitBackend(
      base::BindRepeating(
          &PasswordStoreProxyBackend::OnRemoteFormChangesReceived,
          weak_ptr_factory_.GetWeakPtr(),
          CallbackOriginatesFromAndroidBackend(false),
          remote_form_changes_received),
      base::BindRepeating(&CallOnSyncEnabledOrDisabledForEnabledBackend,
                          /*originates_from_android=*/false,
                          sync_enabled_or_disabled_cb),
      base::BindOnce(pending_initialization_calls));

  android_backend_->InitBackend(
      base::BindRepeating(
          &PasswordStoreProxyBackend::OnRemoteFormChangesReceived,
          weak_ptr_factory_.GetWeakPtr(),
          CallbackOriginatesFromAndroidBackend(true),
          std::move(remote_form_changes_received)),
      base::BindRepeating(&CallOnSyncEnabledOrDisabledForEnabledBackend,
                          /*originates_from_android=*/true,
                          std::move(sync_enabled_or_disabled_cb)),
      base::BindOnce(pending_initialization_calls));
}

void PasswordStoreProxyBackend::Shutdown(base::OnceClosure shutdown_completed) {
  base::RepeatingClosure pending_shutdown_calls = base::BarrierClosure(
      /*num_closures=*/2, std::move(shutdown_completed));
  android_backend_->Shutdown(pending_shutdown_calls);
  built_in_backend_->Shutdown(pending_shutdown_calls);
}

void PasswordStoreProxyBackend::GetAllLoginsAsync(LoginsOrErrorReply callback) {
  auto handler = base::MakeRefCounted<
      ShadowTrafficMetricsRecorder<LoginsResultOrErrorImpl>>(
      MethodName("GetAllLoginsAsync"));
  main_backend()->GetAllLoginsAsync(
      base::BindOnce(&ShadowTrafficMetricsRecorder<
                         LoginsResultOrErrorImpl>::RecordMainResult,
                     handler)
          .Then(std::move(callback)));

  if (ShouldExecuteReadOperationsOnShadowBackend(
          prefs_, sync_delegate_->IsSyncingPasswordsEnabled())) {
    shadow_backend()->GetAllLoginsAsync(
        base::BindOnce(&ShadowTrafficMetricsRecorder<
                           LoginsResultOrErrorImpl>::RecordShadowResult,
                       handler));
  }
}

void PasswordStoreProxyBackend::GetAutofillableLoginsAsync(
    LoginsOrErrorReply callback) {
  auto handler = base::MakeRefCounted<
      ShadowTrafficMetricsRecorder<LoginsResultOrErrorImpl>>(
      MethodName("GetAutofillableLoginsAsync"));
  main_backend()->GetAutofillableLoginsAsync(
      base::BindOnce(&ShadowTrafficMetricsRecorder<
                         LoginsResultOrErrorImpl>::RecordMainResult,
                     handler)
          .Then(std::move(callback)));

  if (ShouldExecuteReadOperationsOnShadowBackend(
          prefs_, sync_delegate_->IsSyncingPasswordsEnabled())) {
    shadow_backend()->GetAutofillableLoginsAsync(
        base::BindOnce(&ShadowTrafficMetricsRecorder<
                           LoginsResultOrErrorImpl>::RecordShadowResult,
                       handler));
  }
}

void PasswordStoreProxyBackend::GetAllLoginsForAccountAsync(
    absl::optional<std::string> account,
    LoginsOrErrorReply callback) {
  NOTREACHED();
}

void PasswordStoreProxyBackend::FillMatchingLoginsAsync(
    LoginsReply callback,
    bool include_psl,
    const std::vector<PasswordFormDigest>& forms) {
  auto handler =
      base::MakeRefCounted<ShadowTrafficMetricsRecorder<LoginsResultImpl>>(
          MethodName("FillMatchingLoginsAsync"));
  main_backend()->FillMatchingLoginsAsync(
      base::BindOnce(
          &ShadowTrafficMetricsRecorder<LoginsResultImpl>::RecordMainResult,
          handler)
          .Then(std::move(callback)),
      include_psl, forms);

  if (ShouldExecuteReadOperationsOnShadowBackend(
          prefs_, sync_delegate_->IsSyncingPasswordsEnabled())) {
    shadow_backend()->FillMatchingLoginsAsync(
        base::BindOnce(
            &ShadowTrafficMetricsRecorder<LoginsResultImpl>::RecordShadowResult,
            handler),
        include_psl, forms);
  }
}

void PasswordStoreProxyBackend::AddLoginAsync(
    const PasswordForm& form,
    PasswordStoreChangeListReply callback) {
  auto handler = base::MakeRefCounted<
      ShadowTrafficMetricsRecorder<PasswordStoreChangeListImpl>>(
      MethodName("AddLoginAsync"));

  main_backend()->AddLoginAsync(
      form, base::BindOnce(&ShadowTrafficMetricsRecorder<
                               PasswordStoreChangeListImpl>::RecordMainResult,
                           handler)
                .Then(std::move(callback)));
  if (ShouldExecuteModifyOperationsOnShadowBackend(
          prefs_, sync_delegate_->IsSyncingPasswordsEnabled())) {
    shadow_backend()->AddLoginAsync(
        form,
        base::BindOnce(&ShadowTrafficMetricsRecorder<
                           PasswordStoreChangeListImpl>::RecordShadowResult,
                       handler));
  }
}

void PasswordStoreProxyBackend::UpdateLoginAsync(
    const PasswordForm& form,
    PasswordStoreChangeListReply callback) {
  auto handler = base::MakeRefCounted<
      ShadowTrafficMetricsRecorder<PasswordStoreChangeListImpl>>(
      MethodName("UpdateLoginAsync"));

  main_backend()->UpdateLoginAsync(
      form, base::BindOnce(&ShadowTrafficMetricsRecorder<
                               PasswordStoreChangeListImpl>::RecordMainResult,
                           handler)
                .Then(std::move(callback)));
  if (ShouldExecuteModifyOperationsOnShadowBackend(
          prefs_, sync_delegate_->IsSyncingPasswordsEnabled())) {
    shadow_backend()->UpdateLoginAsync(
        form,
        base::BindOnce(&ShadowTrafficMetricsRecorder<
                           PasswordStoreChangeListImpl>::RecordShadowResult,
                       handler));
  }
}

void PasswordStoreProxyBackend::RemoveLoginAsync(
    const PasswordForm& form,
    PasswordStoreChangeListReply callback) {
  auto handler = base::MakeRefCounted<
      ShadowTrafficMetricsRecorder<PasswordStoreChangeListImpl>>(
      MethodName("RemoveLoginAsync"));

  main_backend()->RemoveLoginAsync(
      form, base::BindOnce(&ShadowTrafficMetricsRecorder<
                               PasswordStoreChangeListImpl>::RecordMainResult,
                           handler)
                .Then(std::move(callback)));
  if (ShouldExecuteDeletionsOnShadowBackend(
          prefs_, sync_delegate_->IsSyncingPasswordsEnabled())) {
    shadow_backend()->RemoveLoginAsync(
        form,
        base::BindOnce(&ShadowTrafficMetricsRecorder<
                           PasswordStoreChangeListImpl>::RecordShadowResult,
                       handler));
  }
}

void PasswordStoreProxyBackend::RemoveLoginsByURLAndTimeAsync(
    const base::RepeatingCallback<bool(const GURL&)>& url_filter,
    base::Time delete_begin,
    base::Time delete_end,
    base::OnceCallback<void(bool)> sync_completion,
    PasswordStoreChangeListReply callback) {
  auto handler = base::MakeRefCounted<
      ShadowTrafficMetricsRecorder<PasswordStoreChangeListImpl>>(
      MethodName("RemoveLoginsByURLAndTimeAsync"));

  main_backend()->RemoveLoginsByURLAndTimeAsync(
      url_filter, delete_begin, delete_end, std::move(sync_completion),
      base::BindOnce(&ShadowTrafficMetricsRecorder<
                         PasswordStoreChangeListImpl>::RecordMainResult,
                     handler)
          .Then(std::move(callback)));
  if (ShouldExecuteDeletionsOnShadowBackend(
          prefs_, sync_delegate_->IsSyncingPasswordsEnabled())) {
    shadow_backend()->RemoveLoginsByURLAndTimeAsync(
        url_filter, std::move(delete_begin), std::move(delete_end),
        base::OnceCallback<void(bool)>(),
        base::BindOnce(&ShadowTrafficMetricsRecorder<
                           PasswordStoreChangeListImpl>::RecordShadowResult,
                       handler));
  }
}

void PasswordStoreProxyBackend::RemoveLoginsCreatedBetweenAsync(
    base::Time delete_begin,
    base::Time delete_end,
    PasswordStoreChangeListReply callback) {
  auto handler = base::MakeRefCounted<
      ShadowTrafficMetricsRecorder<PasswordStoreChangeListImpl>>(
      MethodName("RemoveLoginsCreatedBetweenAsync"));

  main_backend()->RemoveLoginsCreatedBetweenAsync(
      delete_begin, delete_end,
      base::BindOnce(&ShadowTrafficMetricsRecorder<
                         PasswordStoreChangeListImpl>::RecordMainResult,
                     handler)
          .Then(std::move(callback)));
  if (ShouldExecuteDeletionsOnShadowBackend(
          prefs_, sync_delegate_->IsSyncingPasswordsEnabled())) {
    shadow_backend()->RemoveLoginsCreatedBetweenAsync(
        std::move(delete_begin), std::move(delete_end),
        base::BindOnce(&ShadowTrafficMetricsRecorder<
                           PasswordStoreChangeListImpl>::RecordShadowResult,
                       handler));
  }
}

void PasswordStoreProxyBackend::DisableAutoSignInForOriginsAsync(
    const base::RepeatingCallback<bool(const GURL&)>& origin_filter,
    base::OnceClosure completion) {
  main_backend()->DisableAutoSignInForOriginsAsync(origin_filter,
                                                   std::move(completion));
  if (ShouldExecuteModifyOperationsOnShadowBackend(
          prefs_, sync_delegate_->IsSyncingPasswordsEnabled())) {
    shadow_backend()->DisableAutoSignInForOriginsAsync(
        origin_filter,
        /*completion=*/base::DoNothing());
  }
}

SmartBubbleStatsStore* PasswordStoreProxyBackend::GetSmartBubbleStatsStore() {
  return main_backend()->GetSmartBubbleStatsStore();
}

FieldInfoStore* PasswordStoreProxyBackend::GetFieldInfoStore() {
  return main_backend()->GetFieldInfoStore();
}

std::unique_ptr<syncer::ProxyModelTypeControllerDelegate>
PasswordStoreProxyBackend::CreateSyncControllerDelegate() {
  switch (features::kUpmExperimentVariationParam.Get()) {
    case features::UpmExperimentVariation::kEnableForSyncingUsers:
    case features::UpmExperimentVariation::kEnableOnlyBackendForSyncingUsers:
    case features::UpmExperimentVariation::kShadowSyncingUsers:
      DCHECK(!base::FeatureList::IsEnabled(
          features::kUnifiedPasswordManagerSyncUsingAndroidBackendOnly))
          << "Without support for local passwords, use legacy sync controller";
      return built_in_backend_->CreateSyncControllerDelegate();
    case features::UpmExperimentVariation::kEnableForAllUsers:
      return base::FeatureList::IsEnabled(
                 features::kUnifiedPasswordManagerSyncUsingAndroidBackendOnly)
                 ? android_backend_->CreateSyncControllerDelegate()
                 : built_in_backend_->CreateSyncControllerDelegate();
  }
  NOTREACHED() << "Define which backend creates the sync delegate.";
  return nullptr;
}

void PasswordStoreProxyBackend::ClearAllLocalPasswords() {
  NOTIMPLEMENTED();
}

void PasswordStoreProxyBackend::OnSyncServiceInitialized(
    syncer::SyncService* sync_service) {
  android_backend_->OnSyncServiceInitialized(sync_service);
}

PasswordStoreBackend* PasswordStoreProxyBackend::main_backend() {
  return UsesAndroidBackendAsMainBackend(
             sync_delegate_->IsSyncingPasswordsEnabled())
             ? android_backend_
             : built_in_backend_;
}

PasswordStoreBackend* PasswordStoreProxyBackend::shadow_backend() {
  return UsesAndroidBackendAsMainBackend(
             sync_delegate_->IsSyncingPasswordsEnabled())
             ? built_in_backend_
             : android_backend_;
}

void PasswordStoreProxyBackend::OnRemoteFormChangesReceived(
    CallbackOriginatesFromAndroidBackend originates_from_android,
    RemoteChangesReceived remote_form_changes_received,
    absl::optional<PasswordStoreChangeList> changes) {
  // `remote_form_changes_received` is used to inform observers about changes in
  // the backend. This check guarantees observers are informed only about
  // changes in the main backend.
  if (originates_from_android.value() ==
      UsesAndroidBackendAsMainBackend(
          sync_delegate_->IsSyncingPasswordsEnabled())) {
    remote_form_changes_received.Run(std::move(changes));
  }
}

}  // namespace password_manager
