// Copyright 2015 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "components/password_manager/core/browser/password_manager_util.h"

#include <algorithm>
#include <memory>
#include <string>
#include <tuple>
#include <utility>

#include "base/bind.h"
#include "base/containers/contains.h"
#include "base/containers/cxx20_erase.h"
#include "base/feature_list.h"
#include "base/metrics/histogram_functions.h"
#include "base/ranges/algorithm.h"
#include "base/strings/strcat.h"
#include "base/time/time.h"
#include "build/branding_buildflags.h"
#include "build/build_config.h"
#include "components/autofill/core/browser/autofill_client.h"
#include "components/autofill/core/browser/logging/log_manager.h"
#include "components/autofill/core/browser/ui/popup_item_ids.h"
#include "components/autofill/core/common/password_generation_util.h"
#include "components/password_manager/core/browser/android_affiliation/affiliation_utils.h"
#include "components/password_manager/core/browser/credentials_cleaner.h"
#include "components/password_manager/core/browser/credentials_cleaner_runner.h"
#include "components/password_manager/core/browser/http_credentials_cleaner.h"
#include "components/password_manager/core/browser/old_google_credentials_cleaner.h"
#include "components/password_manager/core/browser/password_bubble_experiment.h"
#include "components/password_manager/core/browser/password_feature_manager.h"
#include "components/password_manager/core/browser/password_form.h"
#include "components/password_manager/core/browser/password_generation_frame_helper.h"
#include "components/password_manager/core/browser/password_manager.h"
#include "components/password_manager/core/browser/password_manager_client.h"
#include "components/password_manager/core/browser/password_manager_driver.h"
#include "components/password_manager/core/browser/password_manager_features_util.h"
#include "components/password_manager/core/browser/password_manager_metrics_util.h"
#include "components/password_manager/core/browser/password_store_consumer.h"
#include "components/password_manager/core/common/password_manager_features.h"
#include "components/password_manager/core/common/password_manager_pref_names.h"
#include "components/prefs/pref_service.h"
#include "components/signin/public/base/signin_metrics.h"
#include "components/signin/public/identity_manager/identity_manager.h"
#include "components/sync/driver/sync_service.h"
#include "components/sync/driver/sync_user_settings.h"
#include "url/url_util.h"

using autofill::password_generation::PasswordGenerationType;
using password_manager::PasswordForm;

namespace password_manager_util {
namespace {

std::tuple<int, base::Time, int> GetPriorityProperties(
    const PasswordForm* form) {
  return std::make_tuple(-static_cast<int>(GetMatchType(*form)),
                         form->date_last_used,
                         static_cast<int>(form->in_store));
}

// Consider the following properties:
// 1. Match strength for the original form (Exact > Web Affiliations > PSL).
// 2. Last time used. Most recent is better.
// 3. Account vs. profile store. Account is better.
bool IsBetterMatch(const PasswordForm* lhs, const PasswordForm* rhs) {
  return GetPriorityProperties(lhs) > GetPriorityProperties(rhs);
}

}  // namespace

bool IsSavingPasswordsEnabled(const PrefService* pref_service,
                              const syncer::SyncService* sync_service) {
  DCHECK(pref_service);
  const PrefService::Preference* save_passwords_pref =
      pref_service->FindPreference(
          password_manager::prefs::kCredentialsEnableService);
  DCHECK(save_passwords_pref);
#if BUILDFLAG(IS_ANDROID)
  if (!password_bubble_experiment::HasChosenToSyncPasswords(sync_service)) {
    return save_passwords_pref->GetValue()->GetBool();
  }

  if (!password_manager::features::UsesUnifiedPasswordManagerUi()) {
    return save_passwords_pref->GetValue()->GetBool();
  }

  if (save_passwords_pref->IsManaged()) {
    return save_passwords_pref->GetValue()->GetBool();
  }

  return pref_service->GetBoolean(
      password_manager::prefs::kOfferToSavePasswordsEnabledGMS);
#else
  return save_passwords_pref->GetValue()->GetBool();
#endif
}

bool IsAutoSignInEnabled(const PrefService* pref_service,
                         const syncer::SyncService* sync_service) {
  DCHECK(pref_service);
  const PrefService::Preference* auto_sign_in_pref =
      pref_service->FindPreference(
          password_manager::prefs::kCredentialsEnableAutosignin);
  DCHECK(auto_sign_in_pref);
#if BUILDFLAG(IS_ANDROID)
  if (!password_bubble_experiment::HasChosenToSyncPasswords(sync_service)) {
    return auto_sign_in_pref->GetValue()->GetBool();
  }

  if (!password_manager::features::UsesUnifiedPasswordManagerUi()) {
    return auto_sign_in_pref->GetValue()->GetBool();
  }

  if (auto_sign_in_pref->IsManaged()) {
    return auto_sign_in_pref->GetValue()->GetBool();
  }

  return pref_service->GetBoolean(
      password_manager::prefs::kAutoSignInEnabledGMS);
#else
  return auto_sign_in_pref->GetValue()->GetBool();
#endif
}

// Update |credential| to reflect usage.
void UpdateMetadataForUsage(PasswordForm* credential) {
  ++credential->times_used;

  // Remove alternate usernames. At this point we assume that we have found
  // the right username.
  credential->all_possible_usernames.clear();
}

password_manager::SyncState GetPasswordSyncState(
    const syncer::SyncService* sync_service) {
  if (!sync_service ||
      !sync_service->GetActiveDataTypes().Has(syncer::PASSWORDS)) {
    return password_manager::SyncState::kNotSyncing;
  }

  if (sync_service->IsSyncFeatureActive()) {
    return sync_service->GetUserSettings()->IsUsingExplicitPassphrase()
               ? password_manager::SyncState::kSyncingWithCustomPassphrase
               : password_manager::SyncState::kSyncingNormalEncryption;
  }

  DCHECK(base::FeatureList::IsEnabled(
      password_manager::features::kEnablePasswordsAccountStorage));
  // Account passwords are enabled only for users with normal encryption at
  // the moment. Data types won't become active for non-sync users with custom
  // passphrase.
  return password_manager::SyncState::kAccountPasswordsActiveNormalEncryption;
}

void TrimUsernameOnlyCredentials(
    std::vector<std::unique_ptr<PasswordForm>>* android_credentials) {
  // Remove username-only credentials which are not federated.
  base::EraseIf(*android_credentials,
                [](const std::unique_ptr<PasswordForm>& form) {
                  return form->scheme == PasswordForm::Scheme::kUsernameOnly &&
                         form->federation_origin.opaque();
                });

  // Set "skip_zero_click" on federated credentials.
  std::for_each(android_credentials->begin(), android_credentials->end(),
                [](const std::unique_ptr<PasswordForm>& form) {
                  if (form->scheme == PasswordForm::Scheme::kUsernameOnly)
                    form->skip_zero_click = true;
                });
}

bool IsLoggingActive(const password_manager::PasswordManagerClient* client) {
  const autofill::LogManager* log_manager = client->GetLogManager();
  return log_manager && log_manager->IsLoggingActive();
}

bool ManualPasswordGenerationEnabled(
    password_manager::PasswordManagerDriver* driver) {
  password_manager::PasswordGenerationFrameHelper* password_generation_manager =
      driver ? driver->GetPasswordGenerationHelper() : nullptr;
  if (!password_generation_manager ||
      !password_generation_manager->IsGenerationEnabled(false /*logging*/)) {
    return false;
  }

  LogPasswordGenerationEvent(
      autofill::password_generation::PASSWORD_GENERATION_CONTEXT_MENU_SHOWN);
  return true;
}

bool ShowAllSavedPasswordsContextMenuEnabled(
    password_manager::PasswordManagerDriver* driver) {
  password_manager::PasswordManager* password_manager =
      driver ? driver->GetPasswordManager() : nullptr;
  if (!password_manager)
    return false;

  password_manager::PasswordManagerClient* client = password_manager->client();
  if (!client ||
      !client->IsFillingFallbackEnabled(driver->GetLastCommittedURL()))
    return false;

  return true;
}

void UserTriggeredManualGenerationFromContextMenu(
    password_manager::PasswordManagerClient* password_manager_client,
    autofill::AutofillClient* autofill_client) {
  if (autofill_client) {
    autofill_client->HideAutofillPopup(
        autofill::PopupHidingReason::kOverlappingWithPasswordGenerationPopup);
  }
  if (!password_manager_client->GetPasswordFeatureManager()
           ->ShouldShowAccountStorageOptIn()) {
    password_manager_client->GeneratePassword(PasswordGenerationType::kManual);
    LogPasswordGenerationEvent(autofill::password_generation::
                                   PASSWORD_GENERATION_CONTEXT_MENU_PRESSED);
    return;
  }
  // The client ensures the callback won't be run if it is destroyed, so
  // base::Unretained is safe.
  password_manager_client->TriggerReauthForPrimaryAccount(
      signin_metrics::ReauthAccessPoint::kGeneratePasswordContextMenu,
      base::BindOnce(
          [](password_manager::PasswordManagerClient* client,
             password_manager::PasswordManagerClient::ReauthSucceeded
                 succeeded) {
            if (succeeded) {
              client->GeneratePassword(PasswordGenerationType::kManual);
              LogPasswordGenerationEvent(
                  autofill::password_generation::
                      PASSWORD_GENERATION_CONTEXT_MENU_PRESSED);
            }
          },
          base::Unretained(password_manager_client)));
}

// TODO(http://crbug.com/890318): Add unitests to check cleaners are correctly
// created.
void RemoveUselessCredentials(
    password_manager::CredentialsCleanerRunner* cleaning_tasks_runner,
    scoped_refptr<password_manager::PasswordStoreInterface> store,
    PrefService* prefs,
    base::TimeDelta delay,
    base::RepeatingCallback<network::mojom::NetworkContext*()>
        network_context_getter) {
  DCHECK(cleaning_tasks_runner);

#if !BUILDFLAG(IS_IOS)
  // Can be null for some unittests.
  if (!network_context_getter.is_null()) {
    cleaning_tasks_runner->MaybeAddCleaningTask(
        std::make_unique<password_manager::HttpCredentialCleaner>(
            store, network_context_getter, prefs));
  }
#endif  // !BUILDFLAG(IS_IOS)

  // TODO(crbug.com/450621): Remove this when enough number of clients switch
  // to the new version of Chrome.
  cleaning_tasks_runner->MaybeAddCleaningTask(
      std::make_unique<password_manager::OldGoogleCredentialCleaner>(store,
                                                                     prefs));

  if (cleaning_tasks_runner->HasPendingTasks()) {
    // The runner will delete itself once the clearing tasks are done, thus we
    // are releasing ownership here.
    base::SequencedTaskRunnerHandle::Get()->PostDelayedTask(
        FROM_HERE,
        base::BindOnce(
            &password_manager::CredentialsCleanerRunner::StartCleaning,
            cleaning_tasks_runner->GetWeakPtr()),
        delay);
  }
}

base::StringPiece GetSignonRealmWithProtocolExcluded(const PasswordForm& form) {
  base::StringPiece signon_realm = form.signon_realm;

  // Find the web origin (with protocol excluded) in the signon_realm.
  const size_t after_protocol = signon_realm.find(form.url.host_piece());

  // Keep the string starting with position |after_protocol|.
  return signon_realm.substr(std::min(after_protocol, signon_realm.size()));
}

GetLoginMatchType GetMatchType(const password_manager::PasswordForm& form) {
  if (password_manager::IsValidAndroidFacetURI(form.signon_realm)) {
    DCHECK(form.is_affiliation_based_match);
    DCHECK(!form.is_public_suffix_match);
    return GetLoginMatchType::kExact;
  }
  if (form.is_affiliation_based_match)
    return GetLoginMatchType::kAffiliated;

  return form.is_public_suffix_match ? GetLoginMatchType::kPSL
                                     : GetLoginMatchType::kExact;
}

void FindBestMatches(
    const std::vector<const PasswordForm*>& non_federated_matches,
    PasswordForm::Scheme scheme,
    std::vector<const PasswordForm*>* non_federated_same_scheme,
    std::vector<const PasswordForm*>* best_matches,
    const PasswordForm** preferred_match) {
  DCHECK(base::ranges::none_of(non_federated_matches,
                               &PasswordForm::blocked_by_user));
  DCHECK(non_federated_same_scheme);
  DCHECK(best_matches);
  DCHECK(preferred_match);

  *preferred_match = nullptr;
  best_matches->clear();
  non_federated_same_scheme->clear();

  for (auto* match : non_federated_matches) {
    if (match->scheme == scheme)
      non_federated_same_scheme->push_back(match);
  }

  if (non_federated_same_scheme->empty())
    return;

  std::sort(non_federated_same_scheme->begin(),
            non_federated_same_scheme->end(), IsBetterMatch);

  // Map from usernames to the best matching password forms.
  std::map<std::u16string, std::vector<const PasswordForm*>>
      matches_per_username;
  for (const PasswordForm* match : *non_federated_same_scheme) {
    auto it = matches_per_username.find(match->username_value);
    // The first match for |username_value| in the sorted array is best
    // match.
    if (it == matches_per_username.end()) {
      matches_per_username[match->username_value] = {match};
      best_matches->push_back(match);
    } else {
      // Insert another credential only if the store is different as well as the
      // password value.
      if (base::Contains(it->second, match->in_store,
                         [](const auto* form) { return form->in_store; })) {
        continue;
      };
      if (base::Contains(
              it->second, match->password_value,
              [](const auto* form) { return form->password_value; })) {
        continue;
      };
      best_matches->push_back(match);
      it->second.push_back(match);
    }
  }

  *preferred_match = *non_federated_same_scheme->begin();
}

const PasswordForm* FindFormByUsername(
    const std::vector<const PasswordForm*>& forms,
    const std::u16string& username_value) {
  for (const PasswordForm* form : forms) {
    if (form->username_value == username_value)
      return form;
  }
  return nullptr;
}

const PasswordForm* GetMatchForUpdating(
    const PasswordForm& submitted_form,
    const std::vector<const PasswordForm*>& credentials,
    bool username_updated_in_bubble) {
  // This is the case for the credential management API. It should not depend on
  // form managers. Once that's the case, this should be turned into a DCHECK.
  // TODO(crbug/947030): turn it into a DCHECK.
  if (!submitted_form.federation_origin.opaque())
    return nullptr;

  // Try to return form with matching |username_value|.
  const PasswordForm* username_match =
      FindFormByUsername(credentials, submitted_form.username_value);
  if (username_match) {
    if (GetMatchType(*username_match) != GetLoginMatchType::kPSL)
      return username_match;

    const auto& password_to_save = submitted_form.new_password_value.empty()
                                       ? submitted_form.password_value
                                       : submitted_form.new_password_value;
    // Normally, the copy of the PSL matched credentials, adapted for the
    // current domain, is saved automatically without asking the user, because
    // the copy likely represents the same account, i.e., the one for which
    // the user already agreed to store a password.
    //
    // However, if the user changes the suggested password, it might indicate
    // that the autofilled credentials and |submitted_password_form|
    // actually correspond to two different accounts (see
    // http://crbug.com/385619).
    return password_to_save == username_match->password_value ? username_match
                                                              : nullptr;
  }

  // Next attempt is to find a match by password value. It should not be tried
  // when the username was actually detected.
  if (submitted_form.type == PasswordForm::Type::kApi ||
      !submitted_form.username_value.empty()) {
    return nullptr;
  }

  for (const PasswordForm* stored_match : credentials) {
    if (stored_match->password_value == submitted_form.password_value)
      return stored_match;
  }

  // If the user manually changed the username value: consider this at this
  // point of the heuristic a new credential (didn't match other
  // passwords/usernames).
  if (username_updated_in_bubble)
    return nullptr;

  // Last try. The submitted form had no username but a password. Assume that
  // it's an existing credential.
  return credentials.empty() ? nullptr : credentials.front();
}

PasswordForm MakeNormalizedBlocklistedForm(
    password_manager::PasswordFormDigest digest) {
  PasswordForm result;
  result.blocked_by_user = true;
  result.scheme = std::move(digest.scheme);
  result.signon_realm = std::move(digest.signon_realm);
  // In case |digest| corresponds to an Android credential copy the origin as
  // is, otherwise clear out the path by calling GetOrigin().
  if (password_manager::FacetURI::FromPotentiallyInvalidSpec(digest.url.spec())
          .IsValidAndroidFacetURI()) {
    result.url = std::move(digest.url);
  } else {
    // GetOrigin() will return an empty GURL if the origin is not valid or
    // standard. DCHECK that this will not happen.
    DCHECK(digest.url.is_valid());
    DCHECK(digest.url.IsStandard());
    result.url = digest.url.DeprecatedGetOriginAsURL();
  }
  return result;
}

bool CanUseBiometricAuth(device_reauth::BiometricAuthenticator* authenticator,
                         device_reauth::BiometricAuthRequester requester) {
  return authenticator &&
         authenticator->CanAuthenticate(requester) ==
             device_reauth::BiometricsAvailability::kAvailable &&
         base::FeatureList::IsEnabled(
             password_manager::features::kBiometricTouchToFill);
}

GURL StripAuthAndParams(const GURL& gurl) {
  GURL::Replacements rep;
  rep.ClearUsername();
  rep.ClearPassword();
  rep.ClearQuery();
  rep.ClearRef();
  return gurl.ReplaceComponents(rep);
}

GURL ConstructGURLWithScheme(const std::string& url) {
  GURL gurl = GURL(url);
  if (!gurl.has_scheme()) {
    GURL https_url(
        base::StrCat({url::kHttpsScheme, url::kStandardSchemeSeparator, url}));
    if (url::HostIsIPAddress(https_url.host())) {
      GURL::Replacements replacements;
      replacements.SetSchemeStr(url::kHttpScheme);
      return https_url.ReplaceComponents(replacements);
    }
    return https_url;
  }
  return gurl;
}

bool IsValidPasswordURL(const GURL& url) {
  return url.is_valid() && url.SchemeIsHTTPOrHTTPS();
}

std::string GetSignonRealm(const GURL& url) {
  GURL::Replacements rep;
  rep.ClearUsername();
  rep.ClearPassword();
  rep.ClearQuery();
  rep.ClearRef();
  rep.SetPathStr("");
  return url.ReplaceComponents(rep).spec();
}

bool UsesPasswordManagerGoogleBranding(bool is_syncing) {
#if BUILDFLAG(GOOGLE_CHROME_BRANDING)
  return true;
#else
  return is_syncing;
#endif  // GOOGLE_CHROME_BRANDING
}

}  // namespace password_manager_util
