/*
 * ATRAC3+ compatible decoder
 *
 * Copyright (c) 2010-2013 Maxim Poliakovski
 *
 * This file is part of FFmpeg.
 *
 * FFmpeg is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * FFmpeg is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with FFmpeg; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 */

#ifndef AVCODEC_ATRAC3PLUS_DATA_H
#define AVCODEC_ATRAC3PLUS_DATA_H

#include <stddef.h>
#include <stdint.h>

/** VLC tables for wordlen */
static const uint8_t atrac3p_wl_cbs[][12] = {
    {  1,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0 },
    {  1,  0,  4,  0,  0,  0,  0,  0,  0,  0,  0,  0 },
    {  1,  0,  2,  3,  2,  0,  0,  0,  0,  0,  0,  0 },
    {  1,  0,  2,  3,  2,  0,  0,  0,  0,  0,  0,  0 },
};

/** VLC tables for code table indexes */
static const uint8_t atrac3p_ct_cbs[][12] = {
    {  1,  1,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0 },
    {  0,  1,  5,  2,  0,  0,  0,  0,  0,  0,  0,  0 },
    {  0,  1,  5,  2,  0,  0,  0,  0,  0,  0,  0,  0 },
    {  1,  0,  1,  6,  0,  0,  0,  0,  0,  0,  0,  0 },
};
/* Symbols for wordlen interleaved with symbols for code table */
static const uint8_t atrac3p_wl_ct_xlats[] = {
    /* wordlen table 1 - 3 entries */
    0,  1,  7,
    /* code table 1    - 4 entries */
    0,  1,  2,  3,
    /* wordlen table 2 - 5 entries */
    0,  1,  2,  6,  7,
    /* code table 2    - 8 entries */
    0,  1,  2,  3,  4,  5,  6,  7,
    /* wordlen table 3 - 8 entries */
    0,  1,  7,  2,  5,  6,  3,  4,
    /* code table 3    - 8 entries */
    0,  1,  2,  3,  6,  7,  4,  5,
    /* wordlen table 4 - 8 entries */
    0,  1,  7,  2,  3,  6,  4,  5,
    /* code table 4    - 8 entries */
    0,  1,  2,  3,  4,  5,  6,  7,
};

/** VLC tables for scale factor indexes */
static const uint8_t atrac3p_sf_cbs[][12] = {
    {  0,  1,  4,  2,  0,  0,  0,  7, 50,  0,  0,  0 },
    {  0,  1,  4,  2,  0,  0,  0,  7, 50,  0,  0,  0 },
    {  1,  0,  2,  0,  4,  0,  2,  1, 54,  0,  0,  0 },
    {  0,  1,  4,  0,  4,  0,  3,  0, 52,  0,  0,  0 },
    {  0,  1,  4,  2,  0,  8,  0,  0,  0,  0,  0,  0 },
    {  0,  1,  4,  2,  0,  8,  0,  0,  0,  0,  0,  0 },
    {  1,  0,  2,  2,  2,  0,  8,  0,  0,  0,  0,  0 },
    {  0,  1,  4,  2,  2,  2,  4,  0,  0,  0,  0,  0 },
};

static const uint8_t atrac3p_sf_xlats[] = {
    /* Scale factor index 1 - 64 entries */
     0,  1, 61, 62, 63,  2, 60,  3,  4,  5,  6, 57, 58, 59,  7,  8,
     9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 21, 22, 23, 24,
    25, 26, 27, 28, 29, 30, 31, 32, 33, 34, 35, 36, 37, 38, 39, 40,
    41, 42, 43, 44, 45, 46, 47, 48, 49, 50, 51, 52, 53, 54, 55, 56,
    /* Scale factor index 2 - 64 entries */
     0,  1,  2, 62, 63,  3, 61,  4,  5,  6, 57, 58, 59, 60,  7,  8,
     9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 21, 22, 23, 24,
    25, 26, 27, 28, 29, 30, 31, 32, 33, 34, 35, 36, 37, 38, 39, 40,
    41, 42, 43, 44, 45, 46, 47, 48, 49, 50, 51, 52, 53, 54, 55, 56,
    /* Scale factor index 3 - 64 entries */
     0,  1, 63,  2,  3, 61, 62,  4, 60, 59,  5,  6,  7,  8,  9, 10,
    11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25, 26,
    27, 28, 29, 30, 31, 32, 33, 34, 35, 36, 37, 38, 39, 40, 41, 42,
    43, 44, 45, 46, 47, 48, 49, 50, 51, 52, 53, 54, 55, 56, 57, 58,
    /* Scale factor index 4 - 64 entries */
     0,  1,  2, 62, 63,  3,  4, 60, 61,  5, 58, 59,  6,  7,  8,  9,
    10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25,
    26, 27, 28, 29, 30, 31, 32, 33, 34, 35, 36, 37, 38, 39, 40, 41,
    42, 43, 44, 45, 46, 47, 48, 49, 50, 51, 52, 53, 54, 55, 56, 57,
    /* Scale factor index 5 - 15 entries */
     0,  1, 13, 14, 15,  2, 12,  3,  4,  5,  6,  7,  9, 10, 11,
    /* Scale factor index 6 - 15 entries */
     0,  1,  2, 14, 15,  3, 13,  4,  5,  6,  7,  9, 10, 11, 12,
    /* Scale factor index 7 - 15 entries */
     0,  1, 15,  2, 14,  3, 13,  4,  5,  6,  7,  9, 10, 11, 12,
    /* Scale factor index 8 - 15 entries */
     0,  1,  2, 14, 15,  3, 13,  4, 12,  5, 11,  6,  7,  9, 10,
};

static const uint8_t atrac3p_ct_huff1[4][2] = {
    { 0, 1 }, { 1, 2 }, { 2, 3 }, { 3, 3 },
};

static const uint8_t atrac3p_ct_huff2[8][2] = {
    { 0, 2 }, { 1, 3 }, { 2, 3 }, { 3, 3 }, { 4, 3 }, { 5, 3 }, { 6, 4 },
    { 7, 4 },
};

static const uint8_t atrac3p_ct_huff3[8][2] = {
    { 0, 2 }, { 1, 3 }, { 2, 3 }, { 3, 3 }, { 6, 3 }, { 7, 3 }, { 4, 4 },
    { 5, 4 },
};

static const uint8_t atrac3p_ct_huff4[8][2] = {
    { 0, 1 }, { 1, 3 }, { 2, 4 }, { 3, 4 }, { 4, 4 }, { 5, 4 }, { 6, 4 },
    { 7, 4 },
};

/* weights for quantized word lengths */
static const int8_t atrac3p_wl_weights[6][32] = {
    { 5, 5, 4, 4, 3, 3, 2, 2, 1, 1, 0, 0, 0, 0, 0, 0,
      0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
    { 5, 5, 5, 4, 4, 4, 3, 3, 3, 2, 2, 2, 1, 1, 1, 0,
      0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
    { 6, 5, 5, 5, 4, 4, 4, 4, 3, 3, 3, 3, 2, 2, 1, 1,
      1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 0, 0, 0, 0 },
    { 5, 5, 4, 4, 3, 3, 2, 2, 1, 1, 0, 0, 0, 0, 0, 0,
      0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
    { 5, 5, 5, 4, 4, 4, 3, 3, 3, 2, 2, 2, 1, 1, 1, 0,
      0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 },
    { 6, 5, 5, 5, 5, 5, 5, 5, 3, 3, 3, 3, 2, 2, 1, 1,
      0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 }
};

/* weights for quantized scale factors
 * sf_weights[i] = i / (tab_idx + 1)
 * where tab_idx = [1,2] */
static const int8_t atrac3p_sf_weights[2][32] = {
    { 0, 0, 1, 1,  2,  2,  3,  3,  4,  4,  5,  5,  6,  6,  7,  7,
      8, 8, 9, 9, 10, 10, 11, 11, 12, 12, 13, 13, 14, 14, 15, 15 },
    { 0, 0, 0, 1, 1, 1, 2, 2, 2, 3, 3, 3, 4, 4,  4,  5,
      5, 5, 6, 6, 6, 7, 7, 7, 8, 8, 8, 9, 9, 9, 10, 10 }
};

/** Ungroup table for word length segments.
 *  Numbers in this table tell which coeff belongs to which segment. */
static const uint8_t atrac3p_qu_num_to_seg[32] = {
    0, 0, 0, 1, 1, 1, 2, 2, 2, 3, 3, 3, 4, 4, 4, 5,
    5, 5, 6, 6, 6, 7, 7, 7, 8, 8, 8, 9, 9, 9, 9, 9
};

/** Map quant unit number to subband number */
static const uint8_t atrac3p_qu_to_subband[32] = {
    0, 0, 0, 0, 0, 0, 0, 0, 1, 1,  1,  1,  2,  2,  2, 2,
    3, 3, 4, 4, 5, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15
};

/** Map subband number to number of power compensation groups */
static const int atrac3p_subband_to_num_powgrps[16] = {
    1, 2, 2, 3, 3, 3, 4, 4, 4, 4, 4, 5, 5, 5, 5, 5
};

/** 3D base shape tables. The values are grouped together as follows:
 *  [num_start_values = 8][num_shape_tables = 16][num_seg_coeffs = 9]
 *  For each of the 8 start values there are 16 different shapes each
 *  9 coefficients long. */
static const int8_t atrac3p_wl_shapes[8][16][9] = {
    { {  0,  0,  0,  0,  0,  0,  0, -2, -1 },
      {  0,  0,  0,  0,  0,  0,  0, -5, -1 },
      {  0,  0,  0, -7,  0,  0,  0,  0,  0 },
      {  0,  0,  0,  0,  0, -7,  0,  0,  0 },
      {  0,  0,  0,  0,  0,  0, -5,  0,  0 },
      {  0,  0,  0,  0, -5,  0,  0,  0,  0 },
      { -7, -7,  0,  0,  0,  0,  0,  0,  0 },
      {  0, -7,  0,  0,  0,  0,  0,  0,  0 },
      { -2, -2, -5,  0,  0,  0,  0,  0,  0 },
      {  0,  0,  0, -2, -5,  0,  0,  0,  0 },
      {  0,  0,  0,  0,  0, -2, -5,  0,  0 },
      {  0,  0,  0, -5,  0,  0,  0,  0,  0 },
      {  0, -2, -7, -2,  0,  0,  0,  0,  0 },
      {  0,  0,  0,  0, -2, -5,  0,  0,  0 },
      {  0,  0,  0, -5, -5,  0,  0,  0,  0 },
      {  0,  0,  0, -5, -2,  0,  0,  0,  0 } },
    { { -1, -5, -3, -2, -1, -1,  0,  0,  0 },
      { -2, -5, -3, -3, -2, -1, -1,  0,  0 },
      {  0, -1, -1, -1,  0,  0,  0,  0,  0 },
      { -1, -3,  0,  0,  0,  0,  0,  0,  0 },
      { -1, -2,  0,  0,  0,  0,  0,  0,  0 },
      { -1, -3, -1,  0,  0,  0,  0,  1,  1 },
      { -1, -5, -3, -3, -2, -1,  0,  0,  0 },
      { -1, -1, -4, -2, -2, -1, -1,  0,  0 },
      { -1, -1, -3, -2, -3, -1, -1, -1,  0 },
      { -1, -4, -2, -3, -1,  0,  0,  0,  0 },
      {  0, -1, -2, -2, -1, -1,  0,  0,  0 },
      {  0, -2, -1,  0,  0,  0,  0,  0,  0 },
      { -1, -1,  0,  0,  0,  0,  0,  0,  0 },
      { -1, -1, -3, -2, -2, -1, -1, -1,  0 },
      {  0,  0,  0,  0,  0,  0,  0,  0,  0 },
      {  0, -1, -3, -2, -2, -1, -1, -1,  0 }, },
    { { -1, -2,  0,  1,  1,  1,  1,  1,  1 },
      {  0, -1,  1,  1,  1,  1,  1,  1,  1 },
      {  0, -2,  1,  1,  1,  1,  1,  1,  1 },
      {  0, -2,  0,  1,  1,  1,  1,  1,  1 },
      { -1, -1,  0,  1,  1,  1,  1,  1,  1 },
      {  0,  0, -1,  0,  1,  1,  1,  1,  1 },
      { -1, -1,  1,  1,  1,  1,  1,  1,  1 },
      {  0,  0, -1,  1,  1,  1,  1,  1,  1 },
      {  0, -1,  0,  1,  1,  1,  1,  1,  1 },
      { -1, -1, -1,  1,  1,  1,  1,  1,  1 },
      {  0,  0,  0,  0,  1,  1,  1,  1,  1 },
      {  0,  0,  0,  1,  1,  1,  1,  1,  1 },
      {  0, -1, -1,  1,  1,  1,  1,  1,  1 },
      {  0,  1,  0,  1,  1,  1,  1,  1,  1 },
      {  0, -3, -2,  1,  1,  1,  1,  2,  2 },
      { -3, -5, -3,  2,  2,  2,  2,  2,  2 }, },
    { { -1, -2,  0,  2,  2,  2,  2,  2,  2 },
      { -1, -2,  0,  1,  2,  2,  2,  2,  2 },
      {  0, -2,  0,  2,  2,  2,  2,  2,  2 },
      { -1,  0,  1,  2,  2,  2,  2,  2,  2 },
      {  0,  0,  1,  2,  2,  2,  2,  2,  2 },
      {  0, -2,  0,  1,  2,  2,  2,  2,  2 },
      {  0, -1,  1,  2,  2,  2,  2,  2,  2 },
      { -1, -1,  0,  2,  2,  2,  2,  2,  2 },
      { -1, -1,  0,  1,  2,  2,  2,  2,  2 },
      { -1, -2, -1,  2,  2,  2,  2,  2,  2 },
      {  0, -1,  0,  2,  2,  2,  2,  2,  2 },
      {  1,  1,  0,  1,  2,  2,  2,  2,  2 },
      {  0,  1,  2,  2,  2,  2,  2,  2,  2 },
      {  1,  0,  0,  1,  2,  2,  2,  2,  2 },
      {  0,  0,  0,  1,  2,  2,  2,  2,  2 },
      { -1, -1, -1,  1,  2,  2,  2,  2,  2 }, },
    { {  0,  1,  2,  3,  3,  3,  3,  3,  3 },
      {  1,  1,  2,  3,  3,  3,  3,  3,  3 },
      { -1,  0,  1,  2,  3,  3,  3,  3,  3 },
      {  0,  0,  2,  3,  3,  3,  3,  3,  3 },
      { -1,  0,  1,  3,  3,  3,  3,  3,  3 },
      {  0,  0,  1,  3,  3,  3,  3,  3,  3 },
      {  1,  2,  3,  3,  3,  3,  3,  3,  3 },
      {  1,  2,  2,  3,  3,  3,  3,  3,  3 },
      {  0,  1,  1,  3,  3,  3,  3,  3,  3 },
      {  0,  0,  1,  2,  3,  3,  3,  3,  3 },
      { -1,  1,  2,  3,  3,  3,  3,  3,  3 },
      { -1,  0,  2,  3,  3,  3,  3,  3,  3 },
      {  2,  2,  3,  3,  3,  3,  3,  3,  3 },
      {  1,  1,  3,  3,  3,  3,  3,  3,  3 },
      {  0,  2,  3,  3,  3,  3,  3,  3,  3 },
      {  0,  1,  1,  2,  3,  3,  3,  3,  3 }, },
    { {  0,  1,  2,  3,  4,  4,  4,  4,  4 },
      {  1,  2,  3,  4,  4,  4,  4,  4,  4 },
      {  0,  0,  2,  3,  4,  4,  4,  4,  4 },
      {  1,  1,  2,  4,  4,  4,  4,  4,  4 },
      {  0,  1,  2,  4,  4,  4,  4,  4,  4 },
      { -1,  0,  1,  3,  4,  4,  4,  4,  4 },
      {  0,  0,  1,  3,  4,  4,  4,  4,  4 },
      {  1,  1,  2,  3,  4,  4,  4,  4,  4 },
      {  0,  1,  1,  3,  4,  4,  4,  4,  4 },
      {  2,  2,  3,  4,  4,  4,  4,  4,  4 },
      {  1,  1,  3,  4,  4,  4,  4,  4,  4 },
      {  1,  2,  2,  4,  4,  4,  4,  4,  4 },
      { -1,  0,  2,  3,  4,  4,  4,  4,  4 },
      {  0,  1,  3,  4,  4,  4,  4,  4,  4 },
      {  1,  2,  2,  3,  4,  4,  4,  4,  4 },
      {  0,  2,  3,  4,  4,  4,  4,  4,  4 }, },
    { {  1,  2,  3,  4,  5,  5,  5,  5,  5 },
      {  0,  1,  2,  3,  4,  5,  5,  5,  5 },
      {  0,  1,  2,  3,  5,  5,  5,  5,  5 },
      {  1,  1,  3,  4,  5,  5,  5,  5,  5 },
      {  1,  1,  2,  4,  5,  5,  5,  5,  5 },
      {  1,  2,  2,  4,  5,  5,  5,  5,  5 },
      {  1,  1,  2,  3,  5,  5,  5,  5,  5 },
      {  2,  2,  3,  4,  5,  5,  5,  5,  5 },
      {  0,  1,  2,  4,  5,  5,  5,  5,  5 },
      {  2,  2,  3,  5,  5,  5,  5,  5,  5 },
      {  1,  2,  3,  5,  5,  5,  5,  5,  5 },
      {  0,  1,  3,  4,  5,  5,  5,  5,  5 },
      {  1,  2,  2,  3,  5,  5,  5,  5,  5 },
      {  2,  3,  4,  5,  5,  5,  5,  5,  5 },
      {  0,  2,  3,  4,  5,  5,  5,  5,  5 },
      {  1,  1,  1,  3,  4,  5,  5,  5,  5 }, },
    { {  1,  2,  3,  4,  5,  5,  5,  6,  6 },
      {  1,  2,  3,  4,  5,  6,  6,  6,  6 },
      {  2,  3,  4,  5,  6,  6,  6,  6,  6 },
      {  1,  2,  3,  4,  6,  6,  6,  6,  6 },
      {  2,  2,  3,  4,  5,  5,  5,  6,  6 },
      {  1,  2,  3,  4,  5,  5,  6,  6,  6 },
      {  2,  2,  3,  4,  6,  6,  6,  6,  6 },
      {  2,  2,  3,  4,  5,  6,  6,  6,  6 },
      {  2,  2,  4,  5,  6,  6,  6,  6,  6 },
      {  2,  2,  3,  5,  6,  6,  6,  6,  6 },
      {  1,  2,  3,  5,  6,  6,  6,  6,  6 },
      {  2,  3,  3,  5,  6,  6,  6,  6,  6 },
      {  1,  2,  4,  5,  6,  6,  6,  6,  6 },
      {  2,  2,  3,  4,  5,  5,  6,  6,  6 },
      {  2,  3,  3,  4,  6,  6,  6,  6,  6 },
      {  1,  3,  4,  5,  6,  6,  6,  6,  6 } }
};

/** 2D base shape tables for scale factor coding.
 *  The values are grouped together as follows:
 *  [num_shape_tables = 64][num_seg_coeffs = 9] */
static const int8_t atrac3p_sf_shapes[64][9] = {
    { -3, -2, -1,  0,  3,  5,  6,  8, 40 },
    { -3, -2,  0,  1,  7,  9, 11, 13, 20 },
    { -1,  0,  0,  1,  6,  8, 10, 13, 41 },
    {  0,  0,  0,  2,  5,  5,  6,  8, 14 },
    {  0,  0,  0,  2,  6,  7,  8, 11, 47 },
    {  0,  0,  1,  2,  5,  7,  8, 10, 32 },
    {  0,  0,  1,  3,  8, 10, 12, 14, 47 },
    {  0,  0,  2,  4,  9, 10, 12, 14, 40 },
    {  0,  0,  3,  5,  9, 10, 12, 14, 22 },
    {  0,  1,  3,  5, 10, 14, 18, 22, 31 },
    {  0,  2,  5,  6, 10, 10, 10, 12, 46 },
    {  0,  2,  5,  7, 12, 14, 15, 18, 44 },
    {  1,  1,  4,  5,  7,  7,  8,  9, 15 },
    {  1,  2,  2,  2,  4,  5,  7,  9, 26 },
    {  1,  2,  2,  3,  6,  7,  7,  8, 47 },
    {  1,  2,  2,  3,  6,  8, 10, 13, 22 },
    {  1,  3,  4,  7, 13, 17, 21, 24, 41 },
    {  1,  4,  0,  4, 10, 12, 13, 14, 17 },
    {  2,  3,  3,  3,  6,  8, 10, 13, 48 },
    {  2,  3,  3,  4,  9, 12, 14, 17, 47 },
    {  2,  3,  3,  5, 10, 12, 14, 17, 25 },
    {  2,  3,  5,  7,  8,  9,  9,  9, 13 },
    {  2,  3,  5,  9, 16, 21, 25, 28, 33 },
    {  2,  4,  5,  8, 12, 14, 17, 19, 26 },
    {  2,  4,  6,  8, 12, 13, 13, 15, 20 },
    {  2,  4,  7, 12, 20, 26, 30, 32, 35 },
    {  3,  3,  5,  6, 12, 14, 16, 19, 34 },
    {  3,  4,  4,  5,  7,  9, 10, 11, 48 },
    {  3,  4,  5,  6,  8,  9, 10, 11, 16 },
    {  3,  5,  5,  5,  7,  9, 10, 13, 35 },
    {  3,  5,  5,  7, 10, 12, 13, 15, 49 },
    {  3,  5,  7,  7,  8,  7,  9, 12, 21 },
    {  3,  5,  7,  8, 12, 14, 15, 15, 24 },
    {  3,  5,  7, 10, 16, 21, 24, 27, 44 },
    {  3,  5,  8, 14, 21, 26, 28, 29, 42 },
    {  3,  6, 10, 13, 18, 19, 20, 22, 27 },
    {  3,  6, 11, 16, 24, 27, 28, 29, 31 },
    {  4,  5,  4,  3,  4,  6,  8, 11, 18 },
    {  4,  6,  5,  6,  9, 10, 12, 14, 20 },
    {  4,  6,  7,  6,  6,  6,  7,  8, 46 },
    {  4,  6,  7,  9, 13, 16, 18, 20, 48 },
    {  4,  6,  7,  9, 14, 17, 20, 23, 31 },
    {  4,  6,  9, 11, 14, 15, 15, 17, 21 },
    {  4,  8, 13, 20, 27, 32, 35, 36, 38 },
    {  5,  6,  6,  4,  5,  6,  7,  6,  6 },
    {  5,  7,  7,  8,  9,  9, 10, 12, 49 },
    {  5,  8,  9,  9, 10, 11, 12, 13, 42 },
    {  5,  8, 10, 12, 15, 16, 17, 19, 42 },
    {  5,  8, 12, 17, 26, 31, 32, 33, 44 },
    {  5,  9, 13, 16, 20, 22, 23, 23, 35 },
    {  6,  8,  8,  7,  6,  5,  6,  8, 15 },
    {  6,  8,  8,  8,  9, 10, 12, 16, 24 },
    {  6,  8,  8,  9, 10, 10, 11, 11, 13 },
    {  6,  8, 10, 13, 19, 21, 24, 26, 32 },
    {  6,  9, 10, 11, 13, 13, 14, 16, 49 },
    {  7,  9,  9, 10, 13, 14, 16, 19, 27 },
    {  7, 10, 12, 13, 16, 16, 17, 17, 27 },
    {  7, 10, 12, 14, 17, 19, 20, 22, 48 },
    {  8,  9, 10,  9, 10, 11, 11, 11, 19 },
    {  8, 11, 12, 12, 13, 13, 13, 13, 17 },
    {  8, 11, 13, 14, 16, 17, 19, 20, 27 },
    {  8, 12, 17, 22, 26, 28, 29, 30, 33 },
    { 10, 14, 16, 19, 21, 22, 22, 24, 28 },
    { 10, 15, 17, 18, 21, 22, 23, 25, 43 }
};

static const uint8_t atrac3p_ct_restricted_to_full[2][7][4] = {
    { { 0, 5, 4, 1 },
      { 0, 1, 2, 3 },
      { 3, 0, 4, 2 },
      { 4, 0, 1, 2 },
      { 1, 0, 4, 3 },
      { 3, 0, 2, 1 },
      { 0, 3, 1, 2 } },
    { { 4, 0, 1, 2 },
      { 0, 3, 2, 1 },
      { 0, 1, 2, 3 },
      { 0, 1, 2, 4 },
      { 0, 1, 2, 3 },
      { 1, 4, 2, 0 },
      { 0, 1, 2, 3 } }
};

/** Tables for spectrum coding */

/* If the first entry of a subtable is negative, it means
 * that another VLC is to be reused. */
static int8_t atrac3p_spectra_cbs[][12] = {
    {  1,  0,  0,  1,  7,  0, 19,  5, 13, 21,  6,  8 },
    {  0,  1,  0,  4, 11,  0,  1, 29,  6, 20,  7,  2 },
    {  0,  0,  1,  8,  0, 13, 18,  7,  2,  0,  0,  0 },
    {  0,  2,  4,  0,  0,  0,  0,  0,  0,  0,  0,  0 },
    {  0,  0,  1,  3,  5,  8, 12, 23, 72, 68, 31,  2 },
    {  0,  1,  3,  2,  6,  4,  0,  0,  0,  0,  0,  0 },
    {  0,  1,  2,  2,  2,  6, 14, 21, 13,  2,  0,  0 },
    {  1,  0,  0,  0,  8,  1, 18,  9, 22, 10, 12,  0 },
    {  0,  0,  0,  0, 16, 11, 32, 19,  1,  2,  0,  0 },
    {  1,  0,  0,  4,  2,  2,  9, 15, 12,  4,  0,  0 },
    {  0,  1,  0,  4,  3,  5, 16, 28, 34, 26,  4,  0 },
    {  0,  0,  0,  0,  9, 12, 16, 44, 98, 42,  4,  0 },
    {  0,  1,  1,  2,  2,  5,  7, 21, 54, 85, 62, 16 },
    {  0,  0,  3,  2,  5,  7, 17, 23,  6,  0,  0,  0 },
    {  1,  0,  0,  2,  6,  0,  7, 21, 15, 17,  8,  4 },
    {  0,  1,  4,  0,  4,  3,  8,  3,  2,  0,  0,  0 },
    {  0,  0,  5,  0,  4,  6, 10, 16,  8,  0,  0,  0 },
    {  0,  3,  1,  2,  0,  0,  0,  0,  0,  0,  0,  0 },
    {  0,  1,  5,  2,  0,  0,  0,  0,  0,  0,  0,  0 },
    {  0,  0,  0,  3,  4, 12, 15, 34, 83, 75, 30,  0 },
    {  0,  0,  0,  3, 14, 10, 20, 16,  0,  0,  0,  0 },
    {  1,  0,  3,  1,  0,  4,  0,  0,  0,  0,  0,  0 },
    {  1,  0,  0,  2,  2,  6, 12, 18, 19, 15,  6,  0 },
    {  0,  0,  1,  1, 13,  1, 14, 28, 33, 81, 32, 52 },
    {  1,  1,  1,  1,  2,  0,  0,  0,  0,  0,  0,  0 },
    {  0,  1,  0,  2,  3,  6, 19,  9, 75, 110, 0,  0 },
    {  0,  0,  1,  3,  5,  5, 13, 27, 69, 96, 35,  2 },
    {  0,  0,  0,  7,  6,  8, 22, 20,  0,  0,  0,  0 },
    {  1,  0,  0,  6,  2,  0,  0,  0, 19,  9, 24, 20 },
    {  0,  0,  1,  2, 13,  1, 31, 13, 16,  4,  0,  0 },
    {  0,  2,  3,  2,  0,  0,  0,  0,  0,  0,  0,  0 },
    {  0,  0,  0,  5,  4, 12, 17, 47, 24, 12,  0,  0 },
    {  0,  2,  2,  4,  0,  0,  0,  0,  0,  0,  0,  0 },
    {  0,  0,  0,  0,  1, 16, 31, 36, 172, 0,  0,  0 },
    {  0,  0,  0,  5, 12,  9, 12, 15, 10,  0,  0,  0 },
    {  0,  1,  0,  6,  2,  6, 18,  4, 26,  6, 12,  0 },
    {  1,  0,  2,  2,  0,  4,  3,  8,  3,  2,  0,  0 },
    {  0,  2,  3,  2,  0,  0,  0,  0,  0,  0,  0,  0 },
    {  0,  0,  0,  4,  7, 12, 19, 21, 58,  0,  0,  0 },
    {  1,  1,  1,  0,  3,  2,  0,  0,  0,  0,  0,  0 },
    {  0,  0,  1,  8,  6,  8,  8,  0,  0,  0,  0,  0 },
    {  0,  0,  1,  5,  7,  8, 16, 22,  4,  0,  0,  0 },
    {  0,  1,  0,  8,  0,  1, 16, 10, 29, 12,  4,  0 },
    {  0,  0,  5,  2,  2,  9,  5,  2,  0,  0,  0,  0 },
    {  0,  0,  1,  1, 10,  4, 16, 29, 46, 75, 74,  0 },
    {  0,  0,  0,  1,  7, 12, 36, 63,  2,  0,  0,  0 },
    {  0,  3,  0,  1,  3,  4,  4,  0,  0,  0,  0,  0 },
    {  0,  1,  2,  2,  6,  8,  6,  3,  1,  2,  0,  0 },
    {  0,  1,  1,  3,  4,  6, 13, 25, 10,  0,  0,  0 },
    {  1,  1,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0 },
    {  0,  1,  0,  4,  8,  3,  8, 24, 17, 12,  4,  0 },
    {  0,  1,  1,  5,  2,  8,  7, 13,  8,  4,  0,  0 },
    {  1,  0,  0,  4,  0,  4,  5,  9, 30, 45, 21,  2 },
    {  0,  1,  3,  3,  4,  4,  0,  0,  0,  0,  0,  0 },
    {  0,  0,  1,  3,  4,  8, 10, 36, 60, 78, 48,  8 },
    { -6 },
    {  1,  0,  0,  2,  6,  0, 11, 13, 12, 24,  4,  8 },
    {  1,  0,  0,  4,  0,  8,  4,  9, 19, 13, 13, 10 },
    {  1,  0,  0,  4,  0,  5, 12, 13, 14,  0,  0,  0 },
    {  0,  1,  0,  4,  4,  5,  9, 30, 45, 21,  2,  0 },
    {  0,  0,  1,  4,  4,  4, 12, 30, 73, 75, 22,  0 },
    { -5 },
    {  0,  0,  3,  2,  4,  8, 23, 13, 10,  0,  0,  0 },
    { -14 },
    {  0,  0,  1,  3, 12,  0, 30,  9, 18,  8,  0,  0 },
    { -9 },
    {  1,  0,  0,  4,  0,  3,  5, 16, 28, 34, 26,  4 },
    { -11 },
    {  0,  0,  0,  4,  4,  9, 13, 37, 76, 72, 39,  2 },
    { -6 },
    { -28 },
    { -22 },
    { -2 },
    { -31 },
    { -60 },
    {  0,  0,  2,  2,  4,  5, 11, 26, 67, 78, 51, 10 },
    { -6 },
    { -35 },
    {  0,  1,  0,  4,  6,  7, 10, 22, 11, 16,  4,  0 },
    {  0,  1,  0,  0,  4, 11,  8, 28, 92, 97, 13,  2 },
    { -59 },
    {  1,  0,  0,  0,  4,  6,  6, 14, 42, 63, 59, 30 },
    { -75 },
    {  1,  0,  2,  0,  2,  2,  6, 17, 14, 13,  6,  0 },
    {  1,  0,  0,  1,  7,  0, 20,  4, 10, 24,  2, 12 },
    {  1,  0,  1,  3,  2,  3,  7,  4,  4,  0,  0,  0 },
    {  0,  1,  1,  3,  4,  9, 15, 12,  4,  0,  0,  0 },
    { -66 },
    { -32 },
    { -12 },
    {  0,  1,  1,  3,  4,  6, 14, 22, 12,  0,  0,  0 },
    { -42 },
    {  1,  0,  1,  3,  2,  3,  7,  4,  4,  0,  0,  0 },
    {  1,  1,  1,  0,  4,  0,  0,  0,  0,  0,  0,  0 },
    { -17 },
    { -39 },
    {  1,  0,  2,  0,  2,  4, 11,  9,  2,  0,  0,  0 },
    { -62 },
    { -28 },
    {  1,  0,  0,  2,  3,  5, 12, 14, 18, 15,  9,  2 },
    {  1,  1,  1,  0,  4,  0,  0,  0,  0,  0,  0,  0 },
    {  1,  1,  1,  1,  2,  0,  0,  0,  0,  0,  0,  0 },
    {  1,  0,  0,  2,  2,  2,  6, 12, 34, 92, 54, 20 },
    {  1,  0,  0,  3,  0,  3,  3, 10, 40, 85, 61, 50 },
    {  0,  1,  2,  1,  4,  7, 10, 26, 12,  0,  0,  0 },
    {  1,  0,  0,  1,  7,  0, 19,  5, 13, 23,  0, 12 },
    { -78 },
    {  1,  0,  0,  0,  1,  4,  9, 4, 103, 110, 24, 0 },
    {  1,  0,  2,  2,  2,  4,  0,  0,  0,  0,  0,  0 },
    {  0,  1,  4,  0,  0,  0,  8, 11, 24, 53, 64, 60 },
    { -47 },
    {  0,  1,  0,  4,  6, 10, 12,  7, 15,  4,  4,  0 },
};

static const uint8_t atrac3p_spectra_xlats[] = {
    /* Table set A, code table 0, wordlen 1 - 81 entries */
    0x00, 0x03, 0x40, 0xC0, 0x10, 0x30, 0x04, 0x0C, 0x01, 0x50, 0xD0, 0x70,
    0xF0, 0xC4, 0x14, 0x34, 0x4C, 0x1C, 0x3C, 0x41, 0xC1, 0x31, 0x05, 0x0D,
    0xC3, 0x13, 0x07, 0x0F, 0x44, 0xCC, 0x11, 0x43, 0x33, 0x54, 0x74, 0xDC,
    0xFC, 0x71, 0x15, 0x4D, 0xCD, 0x1D, 0xD3, 0xC7, 0x37, 0x3F, 0xD4, 0xF4,
    0x5C, 0x7C, 0x51, 0xD1, 0xF1, 0x45, 0xC5, 0x35, 0xDD, 0x3D, 0x53, 0x73,
    0xF3, 0x47, 0x17, 0x77, 0x4F, 0xCF, 0x1F, 0x55, 0xF5, 0x7D, 0xD7, 0x5F,
    0xFF, 0xD5, 0x75, 0x5D, 0xFD, 0x57, 0xF7, 0xDF, 0x7F,
    /* Table set A, code table 0, wordlen 2 - 81 entries */
    0x00, 0x40, 0x10, 0x04, 0x01, 0x50, 0x44, 0x14, 0x54, 0x41, 0x11, 0x51,
    0x05, 0x45, 0x15, 0x55, 0x90, 0x80, 0x20, 0x60, 0x84, 0x94, 0x24, 0x64,
    0x08, 0x48, 0x18, 0x58, 0x81, 0x91, 0x21, 0x85, 0x95, 0x65, 0x09, 0x49,
    0x19, 0x59, 0x02, 0x42, 0x12, 0x52, 0x06, 0x46, 0x16, 0x56, 0x88, 0x61,
    0x25, 0x29, 0x69, 0x5A, 0xA0, 0xA4, 0x98, 0x28, 0x68, 0xA1, 0xA5, 0x89,
    0x99, 0xA9, 0x82, 0x92, 0x22, 0x62, 0x96, 0x26, 0x66, 0x0A, 0x4A, 0x1A,
    0xA8, 0x86, 0xA6, 0x8A, 0x9A, 0x2A, 0x6A, 0xA2, 0xAA,
    /* Table set A, code table 0, wordlen 3 - 49 entries */
    0x00, 0x08, 0x38, 0x01, 0x09, 0x39, 0x07, 0x0F, 0x3F, 0x10, 0x30, 0x11,
    0x31, 0x02, 0x0A, 0x3A, 0x05, 0x06, 0x0E, 0x3E, 0x17, 0x37, 0x18, 0x28,
    0x19, 0x29, 0x2A, 0x32, 0x03, 0x0B, 0x33, 0x3B, 0x0D, 0x15, 0x3D, 0x16,
    0x1E, 0x36, 0x1F, 0x2F, 0x12, 0x1A, 0x13, 0x2B, 0x1D, 0x35, 0x2E, 0x1B,
    0x2D,
    /* Table set A, code table 0, wordlen 4 - 6 entries */
    0x01, 0x02, 0x00, 0x03, 0x04, 0x05,
    /* Table set A, code table 0, wordlen 5 - 225 entries */
    0x00, 0x10, 0xF0, 0x01, 0x11, 0xF1, 0x0F, 0x1F, 0xFF, 0x20, 0xE0, 0xE1,
    0x02, 0xF2, 0x0E, 0x1E, 0x2F, 0x30, 0xD0, 0x21, 0x12, 0x22, 0xE2, 0x03,
    0x0D, 0x2E, 0xEE, 0xFE, 0xEF, 0x40, 0xC0, 0x31, 0xC1, 0xD1, 0x32, 0xD2,
    0x13, 0x23, 0xE3, 0xF3, 0x04, 0xF4, 0x0C, 0x1C, 0x1D, 0x2D, 0xED, 0xFD,
    0x3E, 0xDE, 0x3F, 0xDF, 0x50, 0x60, 0x70, 0x90, 0xA0, 0xB0, 0x41, 0x51,
    0x61, 0x71, 0x91, 0xA1, 0xB1, 0x42, 0x62, 0x92, 0xA2, 0xC2, 0x33, 0xC3,
    0xD3, 0x14, 0x24, 0x34, 0xD4, 0xE4, 0x05, 0x15, 0xF5, 0x06, 0x16, 0x26,
    0xE6, 0xF6, 0x07, 0x17, 0xE7, 0xF7, 0x09, 0x19, 0x29, 0xF9, 0x0A, 0x1A,
    0x2A, 0xEA, 0xFA, 0x0B, 0x1B, 0xFB, 0x2C, 0x3C, 0xDC, 0xEC, 0xFC, 0x3D,
    0x4D, 0xCD, 0xDD, 0x4E, 0x6E, 0x7E, 0xAE, 0xCE, 0x4F, 0x5F, 0x6F, 0x7F,
    0x9F, 0xAF, 0xBF, 0xCF, 0x52, 0x72, 0xB2, 0x43, 0x53, 0x63, 0x73, 0x93,
    0xA3, 0xB3, 0x44, 0x64, 0x74, 0x94, 0xA4, 0xB4, 0xC4, 0x25, 0x35, 0xA5,
    0xC5, 0xD5, 0xE5, 0x36, 0x46, 0xB6, 0xC6, 0xD6, 0x27, 0x37, 0x47, 0xB7,
    0xC7, 0xD7, 0x39, 0x49, 0x59, 0xC9, 0xD9, 0xE9, 0x3A, 0x4A, 0x5A, 0xCA,
    0xDA, 0x2B, 0x3B, 0x4B, 0x6B, 0x7B, 0xDB, 0xEB, 0x4C, 0x5C, 0x6C, 0x7C,
    0x9C, 0xAC, 0xCC, 0x5D, 0x6D, 0x7D, 0x9D, 0xAD, 0xBD, 0x5E, 0x9E, 0xBE,
    0x54, 0x45, 0x55, 0x65, 0x75, 0x95, 0xB5, 0x56, 0x66, 0x76, 0x96, 0xA6,
    0x57, 0x67, 0x97, 0xA7, 0x69, 0x79, 0xA9, 0xB9, 0x6A, 0x7A, 0x9A, 0xAA,
    0xBA, 0x5B, 0x9B, 0xAB, 0xBB, 0xCB, 0xBC, 0x77, 0x99,
    /* Table set A, code table 0, wordlen 6 - 16 entries */
    0x01, 0x00, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0D, 0x0E,
    0x0A, 0x0B, 0x0C, 0x0F,
    /* Table set A, code table 0, wordlen 7 - 63 entries */
    0x00, 0x01, 0x3F, 0x02, 0x3E, 0x03, 0x3D, 0x04, 0x05, 0x06, 0x3A, 0x3B,
    0x3C, 0x07, 0x08, 0x09, 0x0A, 0x1A, 0x1B, 0x1C, 0x24, 0x25, 0x26, 0x36,
    0x37, 0x38, 0x39, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F, 0x10, 0x19, 0x1D, 0x1E,
    0x1F, 0x21, 0x22, 0x23, 0x27, 0x2F, 0x30, 0x31, 0x32, 0x33, 0x34, 0x35,
    0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x29, 0x2A, 0x2B, 0x2C, 0x2D,
    0x2E, 0x18, 0x28,
    /* Table set A, code table 1, wordlen 1 - 81 entries */
    0x00, 0x40, 0xC0, 0x10, 0x30, 0x04, 0x0C, 0x01, 0x03, 0xD0, 0x50, 0x70,
    0xF0, 0xC4, 0x34, 0x4C, 0xCC, 0x1C, 0x41, 0xC1, 0x31, 0x05, 0x0D, 0x43,
    0xC3, 0x13, 0x07, 0x0F, 0x44, 0x14, 0x74, 0xDC, 0x3C, 0x11, 0x1D, 0x33,
    0x37, 0x54, 0xD4, 0xF4, 0x5C, 0x7C, 0xFC, 0xD1, 0x71, 0xF1, 0x15, 0x35,
    0x4D, 0xCD, 0xDD, 0x3D, 0xD3, 0x73, 0x47, 0xC7, 0x17, 0x77, 0x3F, 0x51,
    0x45, 0xC5, 0x55, 0x53, 0xF3, 0x4F, 0xCF, 0x1F, 0xFF, 0xD5, 0x75, 0xF5,
    0x5D, 0x7D, 0xFD, 0x57, 0xD7, 0xF7, 0x5F, 0xDF, 0x7F,
    /* Table set A, code table 1, wordlen 2 - 81 entries */
    0x00, 0x40, 0x10, 0x50, 0x04, 0x44, 0x14, 0x54, 0x01, 0x41, 0x11, 0x51,
    0x05, 0x45, 0x15, 0x55, 0x90, 0x94, 0x58, 0x91, 0x95, 0x19, 0x59, 0x06,
    0x46, 0x16, 0x56, 0x80, 0x60, 0x84, 0x24, 0x64, 0xA4, 0x08, 0x48, 0x18,
    0x68, 0x81, 0x21, 0x61, 0xA1, 0x85, 0x25, 0x65, 0xA5, 0x09, 0x49, 0x99,
    0x69, 0xA9, 0x02, 0x42, 0x12, 0x52, 0x96, 0x26, 0x66, 0x1A, 0x5A, 0x20,
    0xA0, 0x88, 0x98, 0x28, 0xA8, 0x89, 0x29, 0x82, 0x92, 0x22, 0x62, 0x86,
    0xA6, 0x0A, 0x4A, 0x9A, 0x6A, 0xAA, 0xA2, 0x8A, 0x2A,
    /* Table set A, code table 1, wordlen 3 - 49 entries */
    0x00, 0x08, 0x38, 0x01, 0x07, 0x39, 0x0F, 0x09, 0x3F, 0x10, 0x30, 0x31,
    0x02, 0x3A, 0x06, 0x0E, 0x3E, 0x17, 0x18, 0x28, 0x11, 0x29, 0x0A, 0x32,
    0x03, 0x0B, 0x3B, 0x05, 0x0D, 0x3D, 0x16, 0x1F, 0x37, 0x19, 0x12, 0x1A,
    0x2A, 0x13, 0x33, 0x15, 0x35, 0x1E, 0x2E, 0x36, 0x2F, 0x1B, 0x2B, 0x1D,
    0x2D,
    /* Table set A, code table 1, wordlen 4 - 121 entries */
    0x00, 0x10, 0xF0, 0x01, 0x0F, 0xF1, 0x1F, 0xFF, 0x20, 0xE0, 0x11, 0x02,
    0x0E, 0x30, 0x50, 0xB0, 0xD0, 0x21, 0xE1, 0x12, 0xF2, 0x03, 0x05, 0x0B,
    0x0D, 0x1E, 0xFE, 0x2F, 0xEF, 0x40, 0xC0, 0x31, 0x51, 0xB1, 0xC1, 0xD1,
    0x22, 0x52, 0xE2, 0x13, 0xF3, 0x04, 0x15, 0xF5, 0x1B, 0xEB, 0xFB, 0x0C,
    0x1D, 0xFD, 0x2E, 0x5E, 0xEE, 0x3F, 0x5F, 0xBF, 0xDF, 0x41, 0x32, 0x42,
    0xB2, 0xD2, 0x23, 0x53, 0xB3, 0xE3, 0x14, 0x24, 0xE4, 0xF4, 0x25, 0x35,
    0xD5, 0xE5, 0x2B, 0x3B, 0xDB, 0x1C, 0x2C, 0xBC, 0xEC, 0xFC, 0x2D, 0xBD,
    0xED, 0x3E, 0x4E, 0xBE, 0xDE, 0x4F, 0xCF, 0xC2, 0x33, 0x43, 0xC3, 0xD3,
    0x34, 0x44, 0x54, 0xB4, 0xD4, 0x45, 0x55, 0xC5, 0x4B, 0xCB, 0x3C, 0x4C,
    0x5C, 0xCC, 0xDC, 0x3D, 0x4D, 0x5D, 0xCD, 0xDD, 0xCE, 0xC4, 0xB5, 0x5B,
    0xBB,
    /* Table set A, code table 1, wordlen 5 - 225 entries */
    0x00, 0x10, 0xF0, 0x01, 0x11, 0xF1, 0x0F, 0x1F, 0xFF, 0x20, 0xE0, 0x21,
    0xE1, 0x02, 0x12, 0xF2, 0x0E, 0x1E, 0xFE, 0x2F, 0xEF, 0x30, 0xD0, 0x31,
    0xD1, 0x22, 0xE2, 0x03, 0x13, 0xF3, 0x0D, 0x1D, 0xFD, 0x2E, 0xEE, 0x3F,
    0xDF, 0x40, 0x60, 0x70, 0x90, 0xA0, 0xC0, 0x41, 0xC1, 0x32, 0x42, 0xC2,
    0xD2, 0x23, 0x33, 0xD3, 0xE3, 0x04, 0x14, 0x24, 0xE4, 0xF4, 0x06, 0x16,
    0xF6, 0x07, 0x09, 0x0A, 0x1A, 0xFA, 0x0C, 0x1C, 0x2C, 0xEC, 0xFC, 0x2D,
    0x3D, 0xDD, 0xED, 0x3E, 0x4E, 0xCE, 0xDE, 0x4F, 0xCF, 0x50, 0xB0, 0x51,
    0x61, 0x71, 0x91, 0xA1, 0xB1, 0x52, 0x62, 0x72, 0x92, 0xA2, 0xB2, 0x43,
    0x53, 0x63, 0x73, 0x93, 0xA3, 0xC3, 0x34, 0x44, 0x64, 0xA4, 0xC4, 0xD4,
    0x05, 0x15, 0x25, 0x35, 0xD5, 0xE5, 0xF5, 0x26, 0x36, 0x46, 0xC6, 0xD6,
    0xE6, 0x17, 0x27, 0x37, 0xC7, 0xD7, 0xE7, 0xF7, 0x19, 0x29, 0x39, 0xC9,
    0xD9, 0xE9, 0xF9, 0x2A, 0x3A, 0x4A, 0x5A, 0xCA, 0xDA, 0xEA, 0x0B, 0x1B,
    0x2B, 0x3B, 0xCB, 0xDB, 0xEB, 0xFB, 0x3C, 0x4C, 0x6C, 0x7C, 0x9C, 0xAC,
    0xBC, 0xCC, 0xDC, 0x4D, 0x5D, 0x6D, 0x7D, 0x9D, 0xAD, 0xBD, 0xCD, 0x5E,
    0x6E, 0x7E, 0x9E, 0xAE, 0xBE, 0x5F, 0x6F, 0x7F, 0x9F, 0xAF, 0xBF, 0xB3,
    0x54, 0x74, 0x94, 0xB4, 0x45, 0x55, 0x65, 0x75, 0x95, 0xA5, 0xB5, 0xC5,
    0x56, 0x66, 0x76, 0x96, 0xA6, 0xB6, 0x47, 0x57, 0x67, 0xA7, 0xB7, 0x49,
    0x59, 0x69, 0xA9, 0xB9, 0x6A, 0x7A, 0x9A, 0xAA, 0xBA, 0x4B, 0x5B, 0x6B,
    0x7B, 0x9B, 0xAB, 0xBB, 0x5C, 0x77, 0x97, 0x79, 0x99,
    /* Table set A, code table 1, wordlen 6 - 256 entries */
    0x00, 0x01, 0x10, 0x11, 0x21, 0x12, 0x20, 0x31, 0x02, 0x22, 0x13, 0x30,
    0x41, 0x32, 0x03, 0x23, 0x14, 0x24, 0x40, 0x51, 0x61, 0xD1, 0xE1, 0x42,
    0x52, 0xD2, 0x33, 0x43, 0xD3, 0x04, 0x34, 0x05, 0x15, 0x25, 0x16, 0x1D,
    0x2D, 0x1E, 0x2E, 0x50, 0x60, 0xD0, 0xE0, 0xF0, 0x71, 0x81, 0xF1, 0x62,
    0x72, 0xE2, 0xF2, 0x53, 0x63, 0xE3, 0xF3, 0x44, 0x54, 0xD4, 0xE4, 0xF4,
    0x35, 0x45, 0x55, 0xD5, 0xE5, 0xF5, 0x06, 0x26, 0x36, 0xD6, 0x07, 0x17,
    0x27, 0x37, 0xD7, 0x18, 0x28, 0x1C, 0x0D, 0x3D, 0x4D, 0x5D, 0x6D, 0x8D,
    0x0E, 0x3E, 0x4E, 0x5E, 0x0F, 0x1F, 0x2F, 0x3F, 0x5F, 0x70, 0x80, 0x90,
    0xC0, 0x91, 0xA1, 0xB1, 0xC1, 0x82, 0x92, 0xA2, 0xC2, 0x73, 0x83, 0x93,
    0xA3, 0xC3, 0x64, 0x74, 0x84, 0x94, 0xA4, 0xC4, 0x65, 0x75, 0x85, 0x46,
    0x56, 0x66, 0xC6, 0xE6, 0xF6, 0x47, 0x57, 0xE7, 0xF7, 0x08, 0x38, 0x48,
    0x58, 0x68, 0xD8, 0xE8, 0xF8, 0x09, 0x19, 0x29, 0x39, 0x59, 0xD9, 0xE9,
    0xF9, 0x1A, 0x2A, 0x3A, 0xDA, 0xEA, 0xFA, 0x1B, 0x2B, 0xDB, 0xEB, 0xFB,
    0x0C, 0x2C, 0x3C, 0xDC, 0xEC, 0x7D, 0x9D, 0xAD, 0xBD, 0xCD, 0x6E, 0x7E,
    0x8E, 0x9E, 0xAE, 0xBE, 0xCE, 0x4F, 0x6F, 0x7F, 0x8F, 0xAF, 0xA0, 0xB2,
    0xB3, 0xB4, 0x95, 0xA5, 0xB5, 0xC5, 0x76, 0x86, 0x96, 0xA6, 0xB6, 0x67,
    0x77, 0x87, 0x97, 0xC7, 0x78, 0x88, 0x98, 0xC8, 0x49, 0x69, 0x79, 0x89,
    0x99, 0xC9, 0x0A, 0x4A, 0x5A, 0x6A, 0x7A, 0xCA, 0x0B, 0x3B, 0x4B, 0x5B,
    0x6B, 0xCB, 0x4C, 0x5C, 0x6C, 0x7C, 0x8C, 0x9C, 0xAC, 0xBC, 0xCC, 0xFC,
    0xDD, 0xED, 0xFD, 0xDE, 0xEE, 0xFE, 0x9F, 0xBF, 0xCF, 0xDF, 0xEF, 0xFF,
    0xB0, 0xA7, 0xB7, 0xA8, 0xB8, 0xA9, 0xB9, 0x8A, 0x9A, 0xAA, 0xBA, 0x7B,
    0x8B, 0x9B, 0xAB, 0xBB,
    /* Table set A, code table 1, wordlen 7 - 63 entries */
    0x00, 0x01, 0x3F, 0x02, 0x3E, 0x03, 0x04, 0x3B, 0x3C, 0x3D, 0x05, 0x06,
    0x07, 0x08, 0x38, 0x39, 0x3A, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x1A,
    0x1B, 0x24, 0x25, 0x26, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x0F, 0x10,
    0x11, 0x12, 0x13, 0x14, 0x15, 0x19, 0x1C, 0x1D, 0x1E, 0x1F, 0x21, 0x22,
    0x23, 0x27, 0x2B, 0x2C, 0x2D, 0x2E, 0x2F, 0x30, 0x31, 0x16, 0x17, 0x18,
    0x28, 0x29, 0x2A,
    /* Table set A, code table 2, wordlen 1 - 81 entries */
    0x00, 0x40, 0xC0, 0x10, 0x30, 0x04, 0x0C, 0x01, 0x03, 0xD0, 0x70, 0x34,
    0x1C, 0x0D, 0x13, 0x07, 0x50, 0xF0, 0x44, 0xC4, 0x14, 0x74, 0x4C, 0xCC,
    0xDC, 0x3C, 0x41, 0xC1, 0x11, 0x31, 0x05, 0x1D, 0x43, 0xC3, 0x33, 0x37,
    0x0F, 0x54, 0xF4, 0xFC, 0xD1, 0x71, 0x15, 0x4D, 0xCD, 0xDD, 0xD3, 0x73,
    0x47, 0xC7, 0x77, 0x3F, 0xD4, 0x5C, 0x7C, 0x51, 0xF1, 0x45, 0xC5, 0x55,
    0x35, 0x3D, 0x53, 0xF3, 0x17, 0x4F, 0xCF, 0x1F, 0xFF, 0x75, 0xF5, 0x5D,
    0x7D, 0xD7, 0xF7, 0x5F, 0xDF, 0xD5, 0xFD, 0x57, 0x7F,
    /* Table set A, code table 2, wordlen 2 - 25 entries */
    0x00, 0x08, 0x38, 0x01, 0x07, 0x09, 0x39, 0x0F, 0x3F, 0x10, 0x02, 0x06,
    0x30, 0x11, 0x31, 0x0A, 0x3A, 0x0E, 0x17, 0x37, 0x32, 0x16, 0x3E, 0x12,
    0x36,
    /* Table set A, code table 2, wordlen 3 - 49 entries */
    0x00, 0x08, 0x38, 0x01, 0x07, 0x09, 0x39, 0x0F, 0x3F, 0x10, 0x30, 0x02,
    0x3A, 0x06, 0x0E, 0x18, 0x28, 0x11, 0x31, 0x0A, 0x03, 0x05, 0x3E, 0x17,
    0x37, 0x19, 0x29, 0x12, 0x2A, 0x32, 0x0B, 0x33, 0x3B, 0x0D, 0x15, 0x3D,
    0x16, 0x1E, 0x36, 0x1F, 0x2F, 0x1A, 0x13, 0x1B, 0x2B, 0x1D, 0x2D, 0x35,
    0x2E,
    /* Table set A, code table 2, wordlen 4 - 6 entries */
    0x00, 0x01, 0x02, 0x03, 0x04, 0x05,
    /* Table set A, code table 2, wordlen 5 - 8 entries */
    0x01, 0x00, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
    /* Table set A, code table 2, wordlen 6 - 256 entries */
    0x00, 0x01, 0x11, 0x10, 0x21, 0x12, 0x22, 0x20, 0x30, 0x31, 0x41, 0x02,
    0x32, 0x03, 0x13, 0x23, 0x33, 0x14, 0x24, 0x40, 0x51, 0x61, 0x42, 0x52,
    0x43, 0x53, 0x04, 0x34, 0x44, 0x15, 0x25, 0x35, 0x16, 0x26, 0x50, 0x60,
    0x71, 0x81, 0xD1, 0x62, 0x72, 0x82, 0xD2, 0x63, 0x73, 0xD3, 0x54, 0x64,
    0x05, 0x45, 0x55, 0x65, 0x06, 0x36, 0x46, 0x56, 0x17, 0x27, 0x37, 0x47,
    0x18, 0x28, 0x38, 0x19, 0x1D, 0x2D, 0x3D, 0x1E, 0x70, 0x80, 0x90, 0xD0,
    0xE0, 0x91, 0xA1, 0xB1, 0xC1, 0xE1, 0xF1, 0x92, 0xA2, 0xC2, 0xE2, 0xF2,
    0x83, 0x93, 0xA3, 0xC3, 0xE3, 0xF3, 0x74, 0x84, 0x94, 0xA4, 0xC4, 0xD4,
    0xE4, 0xF4, 0x75, 0x85, 0x95, 0xD5, 0xE5, 0x66, 0x76, 0x86, 0xD6, 0xE6,
    0x07, 0x57, 0x67, 0x77, 0xD7, 0x08, 0x48, 0x58, 0x68, 0xD8, 0x09, 0x29,
    0x39, 0x49, 0x59, 0x69, 0x1A, 0x2A, 0x3A, 0x4A, 0x1B, 0x2B, 0x1C, 0x2C,
    0x3C, 0x4C, 0x0D, 0x4D, 0x5D, 0x6D, 0x7D, 0x8D, 0x0E, 0x2E, 0x3E, 0x4E,
    0x5E, 0x6E, 0x1F, 0x2F, 0x3F, 0x4F, 0x5F, 0xA0, 0xB0, 0xC0, 0xF0, 0xB2,
    0xB3, 0xB4, 0xA5, 0xB5, 0xC5, 0xF5, 0x96, 0xA6, 0xB6, 0xC6, 0xF6, 0x87,
    0x97, 0xA7, 0xB7, 0xC7, 0xE7, 0xF7, 0x78, 0x88, 0x98, 0xA8, 0xC8, 0xE8,
    0xF8, 0x79, 0x89, 0x99, 0xC9, 0xD9, 0xE9, 0xF9, 0x0A, 0x5A, 0x6A, 0x7A,
    0x8A, 0xDA, 0xEA, 0xFA, 0x0B, 0x3B, 0x4B, 0x5B, 0x6B, 0x7B, 0x8B, 0xDB,
    0x0C, 0x5C, 0x6C, 0x7C, 0x8C, 0x9C, 0xDC, 0x9D, 0xAD, 0xBD, 0xCD, 0x7E,
    0x8E, 0x9E, 0xAE, 0xBE, 0x0F, 0x6F, 0x7F, 0x8F, 0x9F, 0xAF, 0xB8, 0xA9,
    0xB9, 0x9A, 0xAA, 0xBA, 0xCA, 0x9B, 0xAB, 0xBB, 0xCB, 0xEB, 0xFB, 0xAC,
    0xBC, 0xCC, 0xEC, 0xFC, 0xDD, 0xED, 0xFD, 0xCE, 0xDE, 0xEE, 0xFE, 0xBF,
    0xCF, 0xDF, 0xEF, 0xFF,
    /* Table set A, code table 2, wordlen 7 - 63 entries */
    0x00, 0x02, 0x03, 0x01, 0x05, 0x06, 0x07, 0x09, 0x36, 0x37, 0x38, 0x3A,
    0x3B, 0x3C, 0x3D, 0x3E, 0x3F, 0x04, 0x08, 0x0A, 0x0B, 0x0C, 0x0E, 0x31,
    0x34, 0x35, 0x39, 0x0D, 0x0F, 0x10, 0x11, 0x12, 0x13, 0x16, 0x17, 0x19,
    0x1A, 0x1E, 0x27, 0x2B, 0x2C, 0x2D, 0x2E, 0x2F, 0x30, 0x32, 0x33, 0x14,
    0x15, 0x18, 0x1B, 0x1C, 0x1D, 0x1F, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26,
    0x28, 0x29, 0x2A,
    /* Table set A, code table 3, wordlen 1 - 9 entries */
    0x00, 0x04, 0x0C, 0x01, 0x03, 0x05, 0x0D, 0x07, 0x0F,
    /* Table set A, code table 3, wordlen 2 - 81 entries */
    0x00, 0x40, 0x01, 0x10, 0x04, 0x80, 0x50, 0x20, 0x14, 0x05, 0x02, 0x90,
    0x60, 0x44, 0x54, 0x24, 0x08, 0x18, 0x41, 0x11, 0x15, 0x09, 0x06, 0xA0,
    0x84, 0x94, 0x64, 0xA4, 0x48, 0x58, 0x28, 0x51, 0x21, 0x45, 0x55, 0x25,
    0x19, 0x12, 0x16, 0x0A, 0x1A, 0x68, 0xA8, 0x81, 0x91, 0x61, 0xA1, 0x85,
    0x95, 0x65, 0xA5, 0x49, 0x59, 0x29, 0x69, 0x42, 0x52, 0x46, 0x56, 0x2A,
    0x88, 0x98, 0x89, 0x99, 0xA9, 0x82, 0x92, 0x22, 0x62, 0x86, 0x26, 0x66,
    0x4A, 0x5A, 0x6A, 0xA2, 0x96, 0xA6, 0x8A, 0x9A, 0xAA,
    /* Table set A, code table 3, wordlen 3 - 256 entries */
    0x00, 0x10, 0x40, 0x50, 0x04, 0x44, 0x14, 0x54, 0x01, 0x41, 0x11, 0x51,
    0x05, 0x45, 0x15, 0x55, 0x90, 0x20, 0x94, 0x64, 0x18, 0x21, 0x95, 0x19,
    0x69, 0x02, 0x52, 0x06, 0x46, 0x16, 0x80, 0x60, 0x84, 0xD4, 0x24, 0x08,
    0x48, 0x58, 0x68, 0x81, 0x91, 0x61, 0x85, 0x25, 0x65, 0xA5, 0x09, 0x49,
    0x59, 0x29, 0x42, 0x12, 0x56, 0x96, 0xA6, 0x0A, 0x17, 0x1B, 0xD0, 0xC4,
    0x74, 0xF4, 0x88, 0xC8, 0x28, 0xA1, 0x71, 0xC5, 0xD5, 0x75, 0x99, 0xB9,
    0x4D, 0x1D, 0x2D, 0x6D, 0x22, 0x62, 0x66, 0x4A, 0x1A, 0x9A, 0x6A, 0x8E,
    0x5E, 0x43, 0x23, 0x07, 0x47, 0x57, 0x6B, 0xC0, 0xA0, 0xE0, 0x70, 0xB0,
    0xA4, 0xE4, 0x34, 0xB4, 0x98, 0xD8, 0xA8, 0x38, 0x78, 0x0C, 0x4C, 0x1C,
    0x5C, 0x9C, 0x6C, 0x7C, 0xC1, 0xD1, 0xE1, 0x31, 0xE5, 0x35, 0xB5, 0xF5,
    0x89, 0xA9, 0x79, 0xF9, 0x0D, 0xCD, 0x9D, 0xDD, 0xAD, 0x3D, 0x7D, 0x82,
    0xC2, 0x92, 0xD2, 0xE2, 0x72, 0xF2, 0x86, 0xD6, 0xE6, 0x76, 0xB6, 0x8A,
    0x5A, 0xDA, 0xEA, 0xFA, 0x4E, 0x1E, 0x9E, 0xEE, 0x03, 0x13, 0x53, 0x97,
    0xB7, 0x0B, 0x4B, 0x8B, 0x5B, 0x9B, 0xEB, 0x7B, 0x0F, 0x4F, 0x1F, 0x5F,
    0x9F, 0x2F, 0x3F, 0xBF, 0xE8, 0xB8, 0xF8, 0x8C, 0x2C, 0x3C, 0xFC, 0xB1,
    0xC9, 0xD9, 0xE9, 0x39, 0x5D, 0xED, 0xBD, 0xA2, 0x32, 0x26, 0x36, 0x2A,
    0xAA, 0xBA, 0x0E, 0x2E, 0x6E, 0x83, 0xC3, 0x93, 0x63, 0xB3, 0xA7, 0x37,
    0x30, 0xF0, 0xCC, 0xDC, 0xAC, 0xEC, 0xBC, 0xF1, 0x8D, 0xFD, 0xB2, 0xC6,
    0xF6, 0xCA, 0x3A, 0x7A, 0xCE, 0xDE, 0xAE, 0x3E, 0x7E, 0xBE, 0xFE, 0xD3,
    0xA3, 0xE3, 0x33, 0x73, 0xF3, 0x87, 0xC7, 0xD7, 0x27, 0x67, 0xE7, 0x77,
    0xF7, 0xCB, 0xDB, 0x2B, 0xAB, 0x3B, 0xBB, 0xFB, 0x8F, 0xCF, 0xDF, 0x6F,
    0xAF, 0xEF, 0x7F, 0xFF,
    /* Table set A, code table 3, wordlen 4 - 6 entries */
    0x01, 0x00, 0x02, 0x03, 0x04, 0x05,
    /* Table set A, code table 3, wordlen 5 - 225 entries */
    0x00, 0xF0, 0x0F, 0x10, 0x01, 0xFF, 0x20, 0xE0, 0x11, 0xF1, 0x0E, 0x1F,
    0x30, 0x40, 0xD0, 0x21, 0xE1, 0x02, 0x12, 0x22, 0xE2, 0xF2, 0x03, 0x13,
    0x1E, 0x2E, 0x3E, 0xEE, 0xFE, 0x2F, 0xEF, 0xD2, 0x43, 0xF3, 0x04, 0x0D,
    0x2D, 0x3D, 0x3F, 0xDF, 0x50, 0x60, 0x70, 0x90, 0xB0, 0x31, 0x41, 0x91,
    0xA1, 0xC1, 0xD1, 0x42, 0xA2, 0xC2, 0x23, 0x33, 0xE3, 0x24, 0x34, 0xB4,
    0xD4, 0xF4, 0x05, 0x15, 0x45, 0xE5, 0x16, 0x36, 0x56, 0xA6, 0xC6, 0xD6,
    0xF6, 0x57, 0xC7, 0xF7, 0x09, 0x29, 0x49, 0x59, 0x69, 0xF9, 0x0A, 0x2A,
    0x3A, 0x4A, 0xDA, 0xEA, 0xFA, 0x0B, 0x2B, 0xAB, 0xEB, 0xFB, 0x0C, 0x1C,
    0x2C, 0x3C, 0x4C, 0x5C, 0xCC, 0xDC, 0xFC, 0x1D, 0x4D, 0x6D, 0xBD, 0xCD,
    0xED, 0xFD, 0x4E, 0x6E, 0xCE, 0xDE, 0x7F, 0xA0, 0xC0, 0x51, 0x61, 0x71,
    0xB1, 0x32, 0x52, 0x62, 0x72, 0x92, 0xB2, 0x53, 0x63, 0x73, 0x93, 0xA3,
    0xB3, 0xC3, 0xD3, 0x14, 0x44, 0x54, 0x64, 0x74, 0x94, 0xA4, 0xC4, 0xE4,
    0x25, 0x35, 0x55, 0x65, 0x75, 0x95, 0xA5, 0xB5, 0xC5, 0xD5, 0xF5, 0x06,
    0x26, 0x46, 0x66, 0x76, 0x96, 0xB6, 0xE6, 0x07, 0x17, 0x27, 0x37, 0x47,
    0x67, 0x77, 0x97, 0xA7, 0xB7, 0xD7, 0xE7, 0x19, 0x39, 0x79, 0x99, 0xA9,
    0xB9, 0xC9, 0xD9, 0xE9, 0x1A, 0x5A, 0x6A, 0x7A, 0x9A, 0xAA, 0xBA, 0xCA,
    0x1B, 0x3B, 0x4B, 0x5B, 0x6B, 0x7B, 0x9B, 0xBB, 0xCB, 0xDB, 0x6C, 0x7C,
    0x9C, 0xAC, 0xBC, 0xEC, 0x5D, 0x7D, 0x9D, 0xAD, 0xDD, 0x5E, 0x7E, 0x9E,
    0xAE, 0xBE, 0x4F, 0x5F, 0x6F, 0x9F, 0xAF, 0xBF, 0xCF,
    /* Table set A, code table 3, wordlen 6 - 256 entries */
    0x00, 0x10, 0x01, 0x11, 0x20, 0x21, 0x02, 0x12, 0x22, 0x31, 0x41, 0x32,
    0x13, 0x23, 0x30, 0x40, 0x51, 0x42, 0x03, 0x33, 0x43, 0x04, 0x14, 0x24,
    0x34, 0x15, 0x25, 0x50, 0x61, 0x71, 0xD1, 0x52, 0x62, 0x72, 0xD2, 0x53,
    0x63, 0xD3, 0x44, 0x54, 0x64, 0x05, 0x35, 0x45, 0x55, 0x16, 0x26, 0x36,
    0x46, 0x17, 0x27, 0x1D, 0x2D, 0x3D, 0x60, 0x70, 0xD0, 0x81, 0x91, 0xA1,
    0xC1, 0xE1, 0xF1, 0x82, 0x92, 0xC2, 0xE2, 0xF2, 0x73, 0x83, 0xE3, 0xF3,
    0x74, 0x84, 0xC4, 0xD4, 0xE4, 0xF4, 0x65, 0x75, 0x85, 0xD5, 0xE5, 0x06,
    0x56, 0x66, 0xD6, 0xE6, 0x07, 0x37, 0x47, 0x57, 0x67, 0xD7, 0xE7, 0x18,
    0x28, 0x38, 0x48, 0x58, 0xD8, 0x19, 0x29, 0x2A, 0x1C, 0x2C, 0x0D, 0x4D,
    0x5D, 0x6D, 0x7D, 0x8D, 0x9D, 0x1E, 0x2E, 0x3E, 0x4E, 0x5E, 0x6E, 0x7E,
    0x1F, 0x2F, 0x3F, 0x80, 0x90, 0xA0, 0xC0, 0xE0, 0xF0, 0xB1, 0xA2, 0xB2,
    0x93, 0xA3, 0xB3, 0xC3, 0x94, 0xA4, 0xB4, 0x95, 0xA5, 0xB5, 0xC5, 0xF5,
    0x76, 0x86, 0x96, 0xA6, 0xC6, 0xF6, 0x77, 0x87, 0x97, 0xA7, 0xC7, 0xF7,
    0x08, 0x68, 0x78, 0x88, 0x98, 0xC8, 0xE8, 0xF8, 0x09, 0x39, 0x49, 0x59,
    0x69, 0x79, 0x89, 0xD9, 0xE9, 0xF9, 0x0A, 0x1A, 0x3A, 0x4A, 0x5A, 0x6A,
    0xDA, 0xEA, 0xFA, 0x1B, 0x2B, 0x3B, 0x4B, 0x5B, 0xDB, 0xEB, 0xFB, 0x0C,
    0x3C, 0x4C, 0x5C, 0x6C, 0x7C, 0x8C, 0x9C, 0xDC, 0xEC, 0xAD, 0xBD, 0xCD,
    0xDD, 0xED, 0x0E, 0x8E, 0x9E, 0xAE, 0xBE, 0x0F, 0x4F, 0x5F, 0x6F, 0x7F,
    0x8F, 0x9F, 0xAF, 0xB0, 0xB6, 0xB7, 0xA8, 0xB8, 0x99, 0xA9, 0xB9, 0xC9,
    0x7A, 0x8A, 0x9A, 0xAA, 0xBA, 0xCA, 0x0B, 0x6B, 0x7B, 0x8B, 0x9B, 0xCB,
    0xAC, 0xBC, 0xCC, 0xFC, 0xFD, 0xCE, 0xDE, 0xEE, 0xFE, 0xBF, 0xCF, 0xDF,
    0xEF, 0xFF, 0xAB, 0xBB,
    /* Table set A, code table 3, wordlen 7 - 63 entries */
    0x00, 0x01, 0x02, 0x03, 0x3D, 0x3E, 0x3F, 0x04, 0x05, 0x06, 0x3A, 0x3B,
    0x3C, 0x07, 0x08, 0x09, 0x0A, 0x36, 0x37, 0x38, 0x39, 0x0B, 0x0C, 0x0D,
    0x0E, 0x0F, 0x10, 0x19, 0x1A, 0x1B, 0x1C, 0x1D, 0x1E, 0x23, 0x24, 0x25,
    0x26, 0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x11, 0x12, 0x13, 0x14, 0x15,
    0x16, 0x17, 0x18, 0x1F, 0x21, 0x22, 0x27, 0x28, 0x29, 0x2A, 0x2B, 0x2C,
    0x2D, 0x2E, 0x2F,
    /* Table set A, code table 4, wordlen 1 - 81 entries */
    0x00, 0x40, 0xC0, 0x10, 0x30, 0x04, 0x0C, 0x01, 0x03, 0x50, 0xD0, 0x70,
    0xF0, 0xC4, 0x34, 0x4C, 0xCC, 0x1C, 0x41, 0xC1, 0x31, 0x05, 0x0D, 0x43,
    0xC3, 0x13, 0x07, 0x0F, 0x44, 0x14, 0x74, 0xDC, 0x3C, 0x11, 0x1D, 0x33,
    0x37, 0x54, 0xD4, 0xF4, 0x5C, 0x7C, 0xFC, 0xD1, 0x71, 0xF1, 0xC5, 0x15,
    0x35, 0x4D, 0xCD, 0xDD, 0x3D, 0xD3, 0x73, 0x47, 0xC7, 0x17, 0x77, 0x1F,
    0x3F, 0x51, 0x45, 0x55, 0xD5, 0x75, 0xF5, 0x5D, 0x7D, 0xFD, 0x53, 0xF3,
    0x57, 0xD7, 0xF7, 0x4F, 0xCF, 0x5F, 0xDF, 0x7F, 0xFF,
    /* Table set A, code table 4, wordlen 2 - 81 entries */
    0x00, 0x40, 0x01, 0x10, 0x50, 0x04, 0x44, 0x14, 0x54, 0x41, 0x11, 0x51,
    0x05, 0x45, 0x15, 0x55, 0x59, 0x80, 0x90, 0x20, 0x60, 0x84, 0x94, 0x24,
    0x64, 0x08, 0x48, 0x18, 0x58, 0x81, 0x91, 0x21, 0x61, 0x85, 0x95, 0x25,
    0x65, 0x09, 0x49, 0x19, 0x02, 0x42, 0x12, 0x52, 0x06, 0x46, 0x16, 0x56,
    0xA0, 0xA4, 0x68, 0xA1, 0xA5, 0x99, 0x29, 0x69, 0x96, 0x66, 0x4A, 0x1A,
    0x5A, 0x88, 0x98, 0x28, 0x89, 0xA9, 0x82, 0x92, 0x22, 0x62, 0x86, 0x26,
    0xA6, 0x0A, 0x9A, 0x2A, 0x6A, 0xA8, 0xA2, 0x8A, 0xAA,
    /* Table set A, code table 4, wordlen 3 - 7 entries */
    0x00, 0x07, 0x01, 0x02, 0x06, 0x03, 0x05,
    /* Table set A, code table 4, wordlen 4 - 121 entries */
    0x00, 0x10, 0xF0, 0x01, 0x0F, 0x11, 0xF1, 0x1F, 0xFF, 0x20, 0xE0, 0x21,
    0xE1, 0x02, 0x12, 0xF2, 0x0E, 0x1E, 0xFE, 0x2F, 0xEF, 0x30, 0x50, 0xD0,
    0xD1, 0x22, 0xE2, 0x03, 0x13, 0xF3, 0x0D, 0x1D, 0x2D, 0xFD, 0x2E, 0xEE,
    0x3F, 0xDF, 0x40, 0xB0, 0xC0, 0x31, 0x41, 0x51, 0xB1, 0xC1, 0x32, 0xB2,
    0xC2, 0xD2, 0x23, 0xB3, 0xD3, 0xE3, 0x04, 0x14, 0xE4, 0xF4, 0x05, 0x15,
    0xD5, 0xE5, 0xF5, 0x0B, 0x1B, 0x2B, 0x3B, 0xEB, 0xFB, 0x0C, 0x1C, 0x2C,
    0xFC, 0x3D, 0x5D, 0xED, 0x3E, 0x4E, 0x5E, 0xBE, 0xDE, 0x4F, 0x5F, 0xBF,
    0xCF, 0x42, 0x52, 0x33, 0x53, 0xC3, 0x24, 0xB4, 0xD4, 0x25, 0x35, 0xC5,
    0x4B, 0xCB, 0xDB, 0x3C, 0x4C, 0x5C, 0xDC, 0xEC, 0x4D, 0xBD, 0xCD, 0xDD,
    0xCE, 0x43, 0x34, 0x44, 0x54, 0xC4, 0x45, 0x55, 0xB5, 0x5B, 0xBB, 0xBC,
    0xCC,
    /* Table set A, code table 4, wordlen 5 - 8 entries */
    0x01, 0x02, 0x00, 0x03, 0x04, 0x05, 0x06, 0x07,
    /* Table set A, code table 4, wordlen 6 - 256 entries */
    0x02, 0x00, 0x30, 0x21, 0x31, 0x41, 0x61, 0x12, 0x22, 0x42, 0x62, 0x43,
    0x53, 0x24, 0x45, 0x26, 0x27, 0x10, 0x40, 0xB0, 0x01, 0x11, 0x81, 0x32,
    0x52, 0x72, 0x92, 0x03, 0x13, 0x33, 0x63, 0x14, 0x34, 0x54, 0x64, 0x74,
    0x05, 0x15, 0x25, 0x35, 0x55, 0x65, 0x06, 0x46, 0x56, 0x57, 0x67, 0x88,
    0x20, 0x51, 0x91, 0xD1, 0xF2, 0x23, 0x83, 0x93, 0x04, 0x44, 0x84, 0x94,
    0x75, 0x85, 0xC5, 0x36, 0x66, 0x96, 0xB6, 0x07, 0x37, 0x97, 0x08, 0x28,
    0x38, 0x48, 0x68, 0x09, 0x69, 0x79, 0x0A, 0x2A, 0x1B, 0x9B, 0x2C, 0x4D,
    0x50, 0x60, 0x70, 0x80, 0x90, 0xA0, 0xC0, 0xD0, 0xE0, 0xF0, 0x71, 0xA1,
    0xB1, 0xC1, 0xE1, 0xF1, 0x82, 0xA2, 0xB2, 0xC2, 0xD2, 0xE2, 0x73, 0xA3,
    0xB3, 0xC3, 0xD3, 0xE3, 0xF3, 0xA4, 0xB4, 0xC4, 0xD4, 0xE4, 0xF4, 0x95,
    0xA5, 0xB5, 0xD5, 0xE5, 0xF5, 0x16, 0x76, 0x86, 0xA6, 0xC6, 0xD6, 0xE6,
    0xF6, 0x17, 0x47, 0x77, 0x87, 0xA7, 0xB7, 0xC7, 0xD7, 0xE7, 0xF7, 0x18,
    0x58, 0x78, 0x98, 0xA8, 0xB8, 0xC8, 0xD8, 0xE8, 0xF8, 0x19, 0x29, 0x39,
    0x49, 0x59, 0x89, 0x99, 0xA9, 0xB9, 0xC9, 0xD9, 0xE9, 0xF9, 0x1A, 0x3A,
    0x4A, 0x5A, 0x6A, 0x7A, 0x8A, 0x9A, 0xAA, 0xBA, 0xCA, 0xDA, 0xEA, 0xFA,
    0x0B, 0x2B, 0x3B, 0x4B, 0x5B, 0x6B, 0x7B, 0x8B, 0xAB, 0xBB, 0xCB, 0xDB,
    0xEB, 0xFB, 0x0C, 0x1C, 0x3C, 0x4C, 0x5C, 0x6C, 0x7C, 0x8C, 0x9C, 0xAC,
    0xBC, 0xCC, 0xDC, 0xEC, 0xFC, 0x0D, 0x1D, 0x2D, 0x3D, 0x5D, 0x6D, 0x7D,
    0x8D, 0x9D, 0xAD, 0xBD, 0xCD, 0xDD, 0xED, 0xFD, 0x0E, 0x1E, 0x2E, 0x3E,
    0x4E, 0x5E, 0x6E, 0x7E, 0x8E, 0x9E, 0xAE, 0xBE, 0xCE, 0xDE, 0xEE, 0xFE,
    0x0F, 0x1F, 0x2F, 0x3F, 0x4F, 0x5F, 0x6F, 0x7F, 0x8F, 0x9F, 0xAF, 0xBF,
    0xCF, 0xDF, 0xEF, 0xFF,
    /* Table set A, code table 4, wordlen 7 - 63 entries */
    0x00, 0x01, 0x02, 0x3E, 0x3F, 0x03, 0x04, 0x05, 0x06, 0x08, 0x36, 0x38,
    0x39, 0x3A, 0x3B, 0x3C, 0x3D, 0x07, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E,
    0x35, 0x37, 0x0F, 0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x24, 0x25, 0x27,
    0x2A, 0x34, 0x16, 0x19, 0x1C, 0x23, 0x26, 0x28, 0x29, 0x2B, 0x2D, 0x2E,
    0x2F, 0x30, 0x31, 0x32, 0x33, 0x17, 0x18, 0x1A, 0x1B, 0x1D, 0x1E, 0x1F,
    0x21, 0x22, 0x2C,
    /* Table set A, code table 5, wordlen 1 - 81 entries */
    0x00, 0x40, 0xC0, 0x30, 0x04, 0x01, 0x03, 0x10, 0x0C, 0xD0, 0x70, 0x34,
    0x1C, 0x0D, 0x07, 0x50, 0xF0, 0x44, 0xC4, 0x14, 0x4C, 0xCC, 0x3C, 0x41,
    0xC1, 0x11, 0x31, 0x05, 0x43, 0xC3, 0x13, 0x33, 0x0F, 0x74, 0xDC, 0x1D,
    0x37, 0x54, 0xD4, 0xF4, 0x5C, 0x7C, 0xFC, 0xD1, 0x71, 0xF1, 0xC5, 0x15,
    0x35, 0x4D, 0xCD, 0xDD, 0x3D, 0x53, 0xD3, 0x73, 0x47, 0xC7, 0x17, 0x77,
    0x4F, 0x1F, 0x3F, 0x51, 0x45, 0x55, 0xF3, 0xCF, 0xFF, 0xD5, 0x75, 0xF5,
    0x5D, 0x7D, 0xFD, 0x57, 0xD7, 0xF7, 0x5F, 0xDF, 0x7F,
    /* Table set A, code table 5, wordlen 2 - 25 entries */
    0x00, 0x08, 0x38, 0x01, 0x07, 0x09, 0x39, 0x0F, 0x3F, 0x10, 0x02, 0x06,
    0x30, 0x11, 0x31, 0x0A, 0x3A, 0x0E, 0x17, 0x37, 0x32, 0x16, 0x3E, 0x12,
    0x36,
    /* Table set A, code table 5, wordlen 3 - 7 entries */
    0x00, 0x01, 0x02, 0x06, 0x07, 0x03, 0x05,
    /* Table set A, code table 5, wordlen 4 - 121 entries */
    0x00, 0x01, 0x0F, 0x1F, 0x10, 0xE0, 0xF0, 0x11, 0xF1, 0x2F, 0xFF, 0x20,
    0x21, 0xE1, 0x02, 0x12, 0xF2, 0x03, 0xF3, 0x0E, 0x2E, 0xFE, 0x3F, 0x30,
    0x40, 0xD0, 0xC1, 0xD1, 0x22, 0xC2, 0x33, 0xE3, 0x0C, 0xCC, 0x0D, 0x1D,
    0x2D, 0xFD, 0x1E, 0x3E, 0x5E, 0xEF, 0xC0, 0x52, 0xB2, 0xD2, 0x43, 0xC3,
    0xD3, 0x24, 0x45, 0xF5, 0x4B, 0x5B, 0xFB, 0x1C, 0x3D, 0xBD, 0xDD, 0xEE,
    0xBF, 0xCF, 0xDF, 0x50, 0xB0, 0x31, 0x41, 0x51, 0xB1, 0x32, 0x42, 0xE2,
    0x13, 0x23, 0x53, 0xB3, 0x04, 0x14, 0x34, 0x44, 0x54, 0xB4, 0xC4, 0xD4,
    0xE4, 0xF4, 0x05, 0x15, 0x25, 0x35, 0x55, 0xB5, 0xC5, 0xD5, 0xE5, 0x0B,
    0x1B, 0x2B, 0x3B, 0xBB, 0xCB, 0xDB, 0xEB, 0x2C, 0x3C, 0x4C, 0x5C, 0xBC,
    0xDC, 0xEC, 0xFC, 0x4D, 0x5D, 0xCD, 0xED, 0x4E, 0xBE, 0xCE, 0xDE, 0x4F,
    0x5F,
    /* Table set A, code table 5, wordlen 5 - 8 entries */
    0x00, 0x01, 0x02, 0x03, 0x06, 0x07, 0x04, 0x05,
    /* Table set A, code table 5, wordlen 6 - 31 entries */
    0x04, 0x00, 0x01, 0x02, 0x03, 0x1C, 0x1D, 0x1E, 0x1F, 0x05, 0x06, 0x07,
    0x18, 0x19, 0x1B, 0x08, 0x09, 0x0E, 0x13, 0x15, 0x16, 0x17, 0x1A, 0x0A,
    0x0B, 0x0C, 0x0D, 0x0F, 0x11, 0x12, 0x14,
    /* Table set A, code table 5, wordlen 7 - 63 entries */
    0x00, 0x01, 0x02, 0x3D, 0x3E, 0x3F, 0x03, 0x04, 0x05, 0x06, 0x3A, 0x3B,
    0x3C, 0x07, 0x08, 0x09, 0x0A, 0x36, 0x37, 0x38, 0x39, 0x0B, 0x0C, 0x0D,
    0x0E, 0x0F, 0x1A, 0x1B, 0x1C, 0x24, 0x25, 0x26, 0x31, 0x32, 0x33, 0x34,
    0x35, 0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x17, 0x18, 0x19, 0x1D, 0x1E,
    0x1F, 0x21, 0x22, 0x23, 0x27, 0x2B, 0x2C, 0x2D, 0x2E, 0x2F, 0x30, 0x16,
    0x28, 0x29, 0x2A,
    /* Table set A, code table 6, wordlen 1 - 81 entries */
    0x00, 0x40, 0xC0, 0x10, 0x30, 0x04, 0x0C, 0x01, 0x03, 0x70, 0x50, 0xD0,
    0xF0, 0x44, 0xC4, 0x14, 0x34, 0x4C, 0x1C, 0x3C, 0x31, 0x05, 0x0D, 0x13,
    0x07, 0x0F, 0x74, 0xCC, 0xDC, 0xFC, 0x41, 0xC1, 0x11, 0x43, 0xC3, 0x33,
    0x54, 0xD4, 0xF4, 0x5C, 0x7C, 0x51, 0xD1, 0x71, 0xF1, 0x45, 0xC5, 0x15,
    0x35, 0x4D, 0xCD, 0x1D, 0x3D, 0x53, 0xD3, 0x73, 0xF3, 0x47, 0xC7, 0x17,
    0x37, 0x4F, 0xCF, 0x1F, 0x3F, 0x55, 0xD5, 0x75, 0xF5, 0x5D, 0xDD, 0xFD,
    0x57, 0xD7, 0x77, 0xF7, 0xFF, 0x7D, 0x5F, 0xDF, 0x7F,
    /* Table set A, code table 6, wordlen 2 - 25 entries */
    0x00, 0x08, 0x38, 0x01, 0x07, 0x39, 0x0F, 0x09, 0x3F, 0x10, 0x30, 0x31,
    0x02, 0x3A, 0x06, 0x0E, 0x17, 0x37, 0x11, 0x0A, 0x32, 0x16, 0x3E, 0x12,
    0x36,
    /* Table set A, code table 6, wordlen 3 - 256 entries */
    0x00, 0x40, 0x10, 0x50, 0x04, 0x44, 0x14, 0x01, 0x41, 0x05, 0x45, 0x55,
    0x54, 0x11, 0x51, 0x15, 0x80, 0x90, 0x60, 0x24, 0x64, 0xA4, 0x48, 0x61,
    0x95, 0x25, 0xA5, 0x02, 0x42, 0x52, 0x16, 0x56, 0x20, 0x84, 0x94, 0x18,
    0x58, 0x81, 0x91, 0x85, 0x65, 0x09, 0x49, 0x19, 0x59, 0x99, 0x29, 0x69,
    0x79, 0x5D, 0x12, 0x62, 0x06, 0x46, 0x86, 0x66, 0x1A, 0x5A, 0x6A, 0x47,
    0x17, 0xC0, 0xA0, 0xE0, 0xC4, 0xD4, 0x74, 0x08, 0x78, 0x0C, 0x4C, 0x1C,
    0x5C, 0xD1, 0x21, 0xE1, 0x71, 0xC5, 0xE5, 0x75, 0xB5, 0x89, 0xBD, 0x92,
    0x22, 0x96, 0xA6, 0x36, 0x0A, 0x4A, 0x8A, 0x9A, 0x2A, 0x7A, 0xDE, 0x6E,
    0x43, 0x13, 0x53, 0x23, 0x07, 0x77, 0x4B, 0x1B, 0x9B, 0x6B, 0x2F, 0xD0,
    0x30, 0x70, 0xE4, 0x34, 0xF4, 0xC8, 0x98, 0x28, 0x68, 0xA8, 0xE8, 0x38,
    0xB8, 0xF8, 0x9C, 0x2C, 0x6C, 0x7C, 0xA1, 0xB1, 0xD5, 0x35, 0xC9, 0xD9,
    0xA9, 0xE9, 0x39, 0xB9, 0xF9, 0xCD, 0x1D, 0x2D, 0xAD, 0x7D, 0xC2, 0xD2,
    0xA2, 0xB2, 0xF2, 0xC6, 0x26, 0x76, 0xB6, 0xDA, 0xAA, 0xEA, 0x3A, 0xFA,
    0x0E, 0x4E, 0x2E, 0x7E, 0xBE, 0xFE, 0x03, 0x83, 0x63, 0xA3, 0xB3, 0x87,
    0x57, 0x97, 0xD7, 0x27, 0x0B, 0x8B, 0x5B, 0x2B, 0xAB, 0xCF, 0x1F, 0x9F,
    0x7F, 0xBF, 0xB0, 0xF0, 0xB4, 0x88, 0xD8, 0x8C, 0xCC, 0xDC, 0xAC, 0xEC,
    0x3C, 0xBC, 0xFC, 0xC1, 0x31, 0xF1, 0xF5, 0x0D, 0x4D, 0x8D, 0x9D, 0xDD,
    0x6D, 0xED, 0x3D, 0xFD, 0x82, 0xE2, 0x32, 0x72, 0xD6, 0xE6, 0xF6, 0xCA,
    0xBA, 0x8E, 0xCE, 0x1E, 0x5E, 0x9E, 0xAE, 0xEE, 0x3E, 0xC3, 0x93, 0xD3,
    0xE3, 0x33, 0x73, 0xF3, 0xC7, 0x67, 0xA7, 0xE7, 0x37, 0xB7, 0xF7, 0xCB,
    0xDB, 0xEB, 0x3B, 0x7B, 0xBB, 0xFB, 0x0F, 0x4F, 0x8F, 0x5F, 0xDF, 0x6F,
    0xAF, 0xEF, 0x3F, 0xFF,
    /* Table set A, code table 6, wordlen 4 - 121 entries */
    0x00, 0x10, 0x20, 0xE0, 0xF0, 0x02, 0x0E, 0xEF, 0x30, 0x01, 0x11, 0x21,
    0x31, 0xF1, 0x12, 0xF2, 0x1E, 0xEE, 0xDF, 0xFF, 0x40, 0xC0, 0xD0, 0xD1,
    0xE1, 0x22, 0x32, 0x42, 0xD2, 0xE2, 0x03, 0x13, 0x23, 0xB3, 0xC3, 0xE3,
    0xF3, 0xE4, 0x05, 0xF5, 0x2B, 0x0C, 0xFC, 0x1D, 0x2D, 0xBD, 0xDD, 0xFD,
    0x2E, 0x4E, 0xDE, 0xFE, 0x0F, 0x1F, 0x2F, 0x3F, 0x50, 0xB0, 0x41, 0x51,
    0xB1, 0xC1, 0x52, 0xB2, 0xC2, 0x33, 0x43, 0x53, 0xD3, 0x04, 0x14, 0x24,
    0x34, 0x44, 0x54, 0xB4, 0xC4, 0xD4, 0xF4, 0x15, 0x25, 0x35, 0x45, 0x55,
    0xB5, 0xC5, 0xD5, 0xE5, 0x0B, 0x1B, 0x3B, 0x4B, 0x5B, 0xBB, 0xCB, 0xDB,
    0xEB, 0xFB, 0x1C, 0x2C, 0x3C, 0x4C, 0x5C, 0xBC, 0xCC, 0xDC, 0xEC, 0x0D,
    0x3D, 0x4D, 0x5D, 0xCD, 0xED, 0x3E, 0x5E, 0xBE, 0xCE, 0x4F, 0xCF, 0x5F,
    0xBF,
    /* Table set A, code table 6, wordlen 5 - 15 entries */
    0x00, 0x01, 0x0F, 0x0E, 0x02, 0x03, 0x0D, 0x04, 0x06, 0x0A, 0x0C, 0x05,
    0x07, 0x09, 0x0B,
    /* Table set A, code table 6, wordlen 6 - 31 entries */
    0x00, 0x01, 0x1F, 0x02, 0x1E, 0x03, 0x04, 0x0F, 0x11, 0x1C, 0x1D, 0x05,
    0x06, 0x07, 0x08, 0x18, 0x19, 0x1A, 0x1B, 0x09, 0x0A, 0x0B, 0x15, 0x16,
    0x17, 0x0C, 0x13, 0x14, 0x0D, 0x0E, 0x12,
    /* Table set A, code table 6, wordlen 7 - 63 entries */
    0x00, 0x01, 0x02, 0x3E, 0x3F, 0x03, 0x04, 0x3C, 0x3D, 0x05, 0x06, 0x07,
    0x39, 0x3A, 0x3B, 0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x1A, 0x26, 0x34,
    0x35, 0x36, 0x37, 0x38, 0x0E, 0x0F, 0x10, 0x11, 0x12, 0x13, 0x19, 0x1B,
    0x1C, 0x1D, 0x1E, 0x1F, 0x21, 0x22, 0x23, 0x24, 0x25, 0x27, 0x2D, 0x2E,
    0x2F, 0x30, 0x31, 0x32, 0x33, 0x14, 0x15, 0x16, 0x17, 0x18, 0x28, 0x29,
    0x2A, 0x2B, 0x2C,
    /* Table set A, code table 7, wordlen 1 - 4 entries */
    0x00, 0x01, 0x02, 0x03,
    /* Table set A, code table 7, wordlen 2 - 81 entries */
    0x00, 0x40, 0x10, 0x04, 0x01, 0x50, 0x44, 0x14, 0x54, 0x41, 0x11, 0x05,
    0x15, 0x51, 0x45, 0x55, 0x80, 0x90, 0x20, 0x64, 0x08, 0x19, 0x02, 0x06,
    0x60, 0x84, 0x94, 0x24, 0x48, 0x18, 0x58, 0x81, 0x91, 0x21, 0x61, 0x85,
    0x95, 0x25, 0x65, 0x09, 0x49, 0x59, 0x42, 0x12, 0x52, 0x46, 0x16, 0x56,
    0xA0, 0xA4, 0x98, 0x28, 0x68, 0xA1, 0xA5, 0x99, 0x29, 0x69, 0x96, 0x26,
    0x66, 0x0A, 0x4A, 0x1A, 0x5A, 0x88, 0xA8, 0x89, 0xA9, 0x82, 0x92, 0x22,
    0x62, 0x86, 0xA6, 0x2A, 0x6A, 0xA2, 0x8A, 0x9A, 0xAA,
    /* Table set A, code table 7, wordlen 3 - 49 entries */
    0x00, 0x08, 0x38, 0x01, 0x39, 0x07, 0x0F, 0x09, 0x3F, 0x10, 0x30, 0x31,
    0x02, 0x3A, 0x06, 0x0E, 0x17, 0x11, 0x0A, 0x32, 0x0D, 0x16, 0x3E, 0x37,
    0x18, 0x28, 0x19, 0x29, 0x12, 0x2A, 0x03, 0x3B, 0x05, 0x15, 0x1E, 0x1F,
    0x2F, 0x1A, 0x0B, 0x2B, 0x33, 0x35, 0x3D, 0x2E, 0x36, 0x13, 0x1B, 0x1D,
    0x2D,
    /* Table set A, code table 7, wordlen 4 - 121 entries */
    0x00, 0x10, 0xF0, 0x01, 0x0F, 0x11, 0xF1, 0x1F, 0xFF, 0x20, 0xE0, 0x02,
    0xF2, 0x0E, 0x21, 0xE1, 0x12, 0xE2, 0x1E, 0x2E, 0xFE, 0x2F, 0xEF, 0x30,
    0x50, 0xB0, 0xC0, 0xD0, 0x31, 0xB1, 0xD1, 0x22, 0xD2, 0x03, 0x13, 0xE3,
    0xF3, 0xF4, 0x05, 0xE5, 0xF5, 0x0B, 0x1B, 0x0C, 0x0D, 0x1D, 0x2D, 0xFD,
    0x3E, 0xEE, 0x3F, 0x5F, 0xDF, 0x40, 0x41, 0x51, 0xC1, 0x32, 0x42, 0x52,
    0xB2, 0xC2, 0x23, 0x33, 0xB3, 0xC3, 0xD3, 0x04, 0x14, 0x24, 0xD4, 0xE4,
    0x15, 0x25, 0xC5, 0xD5, 0x2B, 0x3B, 0xEB, 0xFB, 0x1C, 0x2C, 0x3C, 0x5C,
    0xEC, 0xFC, 0x3D, 0x5D, 0xDD, 0xED, 0x4E, 0x5E, 0xBE, 0xCE, 0xDE, 0x4F,
    0xBF, 0xCF, 0x43, 0x53, 0x34, 0x54, 0xB4, 0xC4, 0x35, 0x45, 0x55, 0xB5,
    0x4B, 0x5B, 0xCB, 0xDB, 0x4C, 0xBC, 0xCC, 0xDC, 0x4D, 0xBD, 0xCD, 0x44,
    0xBB,
    /* Table set A, code table 7, wordlen 5 - 15 entries */
    0x00, 0x01, 0x0E, 0x0F, 0x02, 0x03, 0x0D, 0x04, 0x06, 0x0A, 0x0C, 0x05,
    0x07, 0x09, 0x0B,
    /* Table set A, code table 7, wordlen 6 - 256 entries */
    0x00, 0x10, 0x01, 0x11, 0x20, 0x21, 0x02, 0x12, 0x30, 0x31, 0x41, 0x22,
    0x03, 0x13, 0x23, 0x14, 0x40, 0x51, 0x61, 0x32, 0x42, 0x33, 0x04, 0x24,
    0x15, 0x16, 0x50, 0x60, 0xD0, 0x71, 0x81, 0xD1, 0xE1, 0xF1, 0x52, 0x62,
    0x72, 0xD2, 0x43, 0x53, 0x63, 0xD3, 0x34, 0x44, 0x54, 0x05, 0x25, 0x35,
    0x45, 0x06, 0x26, 0x36, 0x17, 0x27, 0x18, 0x0D, 0x1D, 0x2D, 0x3D, 0x1E,
    0x2E, 0x1F, 0x70, 0x80, 0xE0, 0xF0, 0x91, 0xA1, 0xC1, 0x82, 0x92, 0xC2,
    0xE2, 0xF2, 0x73, 0x83, 0x93, 0xE3, 0xF3, 0x64, 0x74, 0x84, 0xD4, 0xE4,
    0xF4, 0x55, 0x65, 0xD5, 0xE5, 0xF5, 0x46, 0x56, 0x66, 0xD6, 0x07, 0x37,
    0x47, 0x57, 0x08, 0x28, 0x38, 0x48, 0x19, 0x29, 0x39, 0x1A, 0x2A, 0x1B,
    0x1C, 0x2C, 0x3C, 0x4D, 0x5D, 0x6D, 0x0E, 0x3E, 0x4E, 0x5E, 0x0F, 0x2F,
    0x3F, 0x4F, 0x90, 0xA0, 0xB0, 0xC0, 0xB1, 0xA2, 0xB2, 0xA3, 0xB3, 0xC3,
    0x94, 0xA4, 0xB4, 0xC4, 0x75, 0x85, 0x95, 0xA5, 0xC5, 0x76, 0x86, 0x96,
    0xE6, 0xF6, 0x67, 0x77, 0x87, 0xD7, 0xE7, 0xF7, 0x58, 0x68, 0x78, 0x88,
    0xD8, 0xE8, 0xF8, 0x09, 0x49, 0x59, 0x69, 0xD9, 0xE9, 0xF9, 0x0A, 0x3A,
    0x4A, 0x5A, 0xDA, 0xEA, 0x0B, 0x2B, 0x3B, 0x4B, 0xDB, 0x0C, 0x4C, 0x5C,
    0x6C, 0xDC, 0x7D, 0x8D, 0x9D, 0xAD, 0xBD, 0xCD, 0x6E, 0x7E, 0x8E, 0x9E,
    0xAE, 0xBE, 0x5F, 0x6F, 0x7F, 0x8F, 0x9F, 0xAF, 0xB5, 0xA6, 0xB6, 0xC6,
    0x97, 0xA7, 0xC7, 0x98, 0xA8, 0xB8, 0xC8, 0x79, 0x89, 0x99, 0xA9, 0xB9,
    0xC9, 0x6A, 0x7A, 0x8A, 0x9A, 0xAA, 0xCA, 0xFA, 0x5B, 0x6B, 0x7B, 0x8B,
    0xCB, 0xEB, 0xFB, 0x7C, 0x8C, 0x9C, 0xAC, 0xBC, 0xEC, 0xFC, 0xDD, 0xED,
    0xFD, 0xCE, 0xDE, 0xEE, 0xFE, 0xBF, 0xCF, 0xDF, 0xB7, 0xBA, 0x9B, 0xAB,
    0xBB, 0xCC, 0xEF, 0xFF,
    /* Table set B, code table 0, wordlen 1 - 81 entries */
    0x00, 0x01, 0x03, 0x40, 0xC0, 0x10, 0x30, 0x04, 0x0C, 0x50, 0xD0, 0x70,
    0xF0, 0x34, 0x1C, 0x05, 0x0D, 0x13, 0x07, 0x0F, 0x44, 0xC4, 0x14, 0x4C,
    0xCC, 0x3C, 0x41, 0xC1, 0x11, 0x31, 0x43, 0xC3, 0x33, 0x54, 0x74, 0xDC,
    0xFC, 0x71, 0x15, 0x4D, 0x1D, 0xD3, 0xC7, 0x37, 0x3F, 0xD4, 0xF4, 0x5C,
    0x7C, 0x51, 0xD1, 0xF1, 0x45, 0xC5, 0x55, 0x35, 0xCD, 0xDD, 0x3D, 0x53,
    0x73, 0xF3, 0x47, 0x17, 0x77, 0x4F, 0xCF, 0x1F, 0xFF, 0xF5, 0x7D, 0xD7,
    0x5F, 0xD5, 0x75, 0x5D, 0xFD, 0x57, 0xF7, 0xDF, 0x7F,
    /* Table set B, code table 0, wordlen 2 - 81 entries */
    0x00, 0x40, 0x10, 0x04, 0x01, 0x50, 0x44, 0x14, 0x54, 0x41, 0x11, 0x05,
    0x15, 0x80, 0x51, 0x45, 0x55, 0x90, 0x20, 0x60, 0x24, 0x08, 0x18, 0x09,
    0x02, 0x06, 0x84, 0x94, 0x64, 0x48, 0x58, 0x81, 0x91, 0x21, 0x61, 0x95,
    0x25, 0x65, 0x19, 0x59, 0x42, 0x12, 0x46, 0x16, 0x56, 0xA0, 0xA4, 0x28,
    0x68, 0x85, 0xA5, 0x49, 0x29, 0x69, 0x52, 0x0A, 0x1A, 0x5A, 0x88, 0x98,
    0xA1, 0x89, 0x99, 0xA9, 0x22, 0x62, 0x96, 0x26, 0x66, 0x4A, 0x6A, 0xA8,
    0x82, 0x92, 0xA2, 0x86, 0xA6, 0x8A, 0x9A, 0x2A, 0xAA,
    /* Table set B, code table 0, wordlen 3 - 49 entries */
    0x00, 0x08, 0x38, 0x01, 0x07, 0x30, 0x09, 0x39, 0x0F, 0x3F, 0x10, 0x18,
    0x28, 0x31, 0x02, 0x3A, 0x03, 0x05, 0x06, 0x0E, 0x17, 0x37, 0x11, 0x19,
    0x29, 0x0A, 0x32, 0x0B, 0x3B, 0x0D, 0x15, 0x3D, 0x3E, 0x1F, 0x2F, 0x12,
    0x1A, 0x2A, 0x13, 0x1B, 0x2B, 0x33, 0x1D, 0x2D, 0x35, 0x16, 0x1E, 0x2E,
    0x36,
    /* Table set B, code table 0, wordlen 4 - 121 entries */
    0x00, 0x10, 0xF0, 0x01, 0x0F, 0x11, 0xF1, 0x1F, 0xFF, 0x20, 0xE0, 0x02,
    0xF2, 0x0E, 0x21, 0xE1, 0x12, 0xE2, 0x1E, 0x2E, 0xFE, 0x2F, 0xEF, 0x30,
    0x50, 0xB0, 0xC0, 0xD0, 0x31, 0xB1, 0xD1, 0x22, 0xD2, 0x03, 0x13, 0xE3,
    0xF3, 0xF4, 0x05, 0xE5, 0xF5, 0x0B, 0x1B, 0x0C, 0x0D, 0x1D, 0x2D, 0xFD,
    0x3E, 0xEE, 0x3F, 0x5F, 0xDF, 0x40, 0x41, 0x51, 0xC1, 0x32, 0x42, 0x52,
    0xB2, 0xC2, 0x23, 0x33, 0xB3, 0xC3, 0xD3, 0x04, 0x14, 0x24, 0xD4, 0xE4,
    0x15, 0x25, 0xC5, 0xD5, 0x2B, 0x3B, 0xEB, 0xFB, 0x1C, 0x2C, 0x3C, 0x5C,
    0xEC, 0xFC, 0x3D, 0x5D, 0xDD, 0xED, 0x4E, 0x5E, 0xBE, 0xCE, 0xDE, 0x4F,
    0xBF, 0xCF, 0x43, 0x53, 0x34, 0x54, 0xB4, 0xC4, 0x35, 0x45, 0x55, 0xB5,
    0x4B, 0x5B, 0xCB, 0xDB, 0x4C, 0xBC, 0xCC, 0xDC, 0x4D, 0xBD, 0xCD, 0x44,
    0xBB,
    /* Table set B, code table 0, wordlen 5 - 225 entries */
    0x00, 0x10, 0xF0, 0x01, 0x0F, 0x11, 0xF1, 0x1F, 0xFF, 0x20, 0xE0, 0x02,
    0x0E, 0x30, 0xD0, 0x21, 0xE1, 0x12, 0xF2, 0x03, 0x0D, 0x1E, 0xFE, 0x2F,
    0xEF, 0x40, 0x60, 0x70, 0x90, 0xA0, 0xC0, 0x31, 0xD1, 0x22, 0x32, 0xD2,
    0xE2, 0x13, 0x23, 0xE3, 0xF3, 0x04, 0x06, 0x07, 0x09, 0x0A, 0x0C, 0x1D,
    0x2D, 0xFD, 0x2E, 0x3E, 0xEE, 0x3F, 0xDF, 0x50, 0xB0, 0x41, 0x51, 0x61,
    0x71, 0x91, 0xA1, 0xB1, 0xC1, 0x42, 0x62, 0x72, 0x92, 0xA2, 0xC2, 0x33,
    0x93, 0xA3, 0xD3, 0x14, 0x24, 0xE4, 0xF4, 0x05, 0x15, 0xF5, 0x16, 0x26,
    0xD6, 0xE6, 0xF6, 0x17, 0x27, 0xD7, 0xE7, 0xF7, 0x19, 0x29, 0x39, 0xE9,
    0xF9, 0x1A, 0x2A, 0xEA, 0xFA, 0x0B, 0x1B, 0xFB, 0x1C, 0x2C, 0xEC, 0xFC,
    0x3D, 0x7D, 0x9D, 0xDD, 0xED, 0x4E, 0x6E, 0x7E, 0x9E, 0xAE, 0xCE, 0xDE,
    0x4F, 0x5F, 0x6F, 0x7F, 0x9F, 0xAF, 0xBF, 0xCF, 0x52, 0xB2, 0x43, 0x53,
    0x63, 0x73, 0xB3, 0xC3, 0x34, 0x44, 0x64, 0x74, 0x94, 0xA4, 0xB4, 0xC4,
    0xD4, 0x25, 0x35, 0x65, 0x75, 0x95, 0xA5, 0xD5, 0xE5, 0x36, 0x46, 0x56,
    0x66, 0xA6, 0xB6, 0xC6, 0x37, 0x47, 0x57, 0xB7, 0xC7, 0x49, 0x59, 0x69,
    0xB9, 0xC9, 0xD9, 0x3A, 0x4A, 0x5A, 0x6A, 0xAA, 0xBA, 0xCA, 0xDA, 0x2B,
    0x3B, 0x6B, 0x7B, 0x9B, 0xAB, 0xDB, 0xEB, 0x3C, 0x4C, 0x6C, 0x7C, 0x9C,
    0xAC, 0xCC, 0xDC, 0x4D, 0x5D, 0x6D, 0xAD, 0xBD, 0xCD, 0x5E, 0xBE, 0x54,
    0x45, 0x55, 0xB5, 0xC5, 0x76, 0x96, 0x67, 0x77, 0x97, 0xA7, 0x79, 0x99,
    0xA9, 0x7A, 0x9A, 0x4B, 0x5B, 0xBB, 0xCB, 0x5C, 0xBC,
    /* Table set B, code table 0, wordlen 7 - 63 entries */
    0x00, 0x01, 0x3F, 0x02, 0x3E, 0x03, 0x04, 0x3C, 0x3D, 0x05, 0x06, 0x07,
    0x08, 0x38, 0x39, 0x3A, 0x3B, 0x09, 0x0A, 0x0B, 0x0C, 0x19, 0x1A, 0x1B,
    0x1C, 0x1D, 0x1E, 0x1F, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27, 0x33,
    0x34, 0x35, 0x36, 0x37, 0x0D, 0x0E, 0x0F, 0x10, 0x11, 0x12, 0x13, 0x2D,
    0x2E, 0x2F, 0x30, 0x31, 0x32, 0x14, 0x15, 0x16, 0x17, 0x18, 0x28, 0x29,
    0x2A, 0x2B, 0x2C,
    /* Table set B, code table 1, wordlen 2 - 81 entries */
    0x00, 0x40, 0x04, 0x01, 0x10, 0x50, 0x44, 0x14, 0x54, 0x41, 0x11, 0x51,
    0x05, 0x45, 0x15, 0x55, 0x80, 0x90, 0x20, 0x60, 0x84, 0x94, 0x24, 0x64,
    0x08, 0x48, 0x18, 0x81, 0x91, 0x61, 0x85, 0x95, 0x25, 0x65, 0x09, 0x49,
    0x19, 0x59, 0x02, 0x42, 0x12, 0x52, 0x06, 0x46, 0x16, 0x56, 0xA4, 0x58,
    0x68, 0x21, 0xA5, 0x29, 0x69, 0x1A, 0x5A, 0xA0, 0x88, 0x98, 0x28, 0xA1,
    0x89, 0x99, 0xA9, 0x92, 0x22, 0x62, 0x86, 0x96, 0x26, 0x66, 0x0A, 0x4A,
    0x6A, 0xA8, 0x82, 0xA2, 0xA6, 0x8A, 0x9A, 0x2A, 0xAA,
    /* Table set B, code table 1, wordlen 4 - 121 entries */
    0x00, 0x10, 0xF0, 0x01, 0x0F, 0xF1, 0x1F, 0xFF, 0x20, 0xE0, 0x11, 0x02,
    0x0E, 0x30, 0x50, 0xB0, 0xD0, 0x21, 0xE1, 0x12, 0xF2, 0x03, 0x05, 0x0B,
    0x0D, 0x1E, 0xFE, 0x2F, 0xEF, 0x40, 0xC0, 0x31, 0x51, 0xB1, 0xC1, 0xD1,
    0x22, 0x52, 0xE2, 0x13, 0xF3, 0x04, 0x15, 0xF5, 0x1B, 0xEB, 0xFB, 0x0C,
    0x1D, 0xFD, 0x2E, 0x5E, 0xEE, 0x3F, 0x5F, 0xBF, 0xDF, 0x41, 0x32, 0x42,
    0xB2, 0xD2, 0x23, 0x53, 0xB3, 0xE3, 0x14, 0x24, 0xE4, 0xF4, 0x25, 0x35,
    0xD5, 0xE5, 0x2B, 0x3B, 0xDB, 0x1C, 0x2C, 0xBC, 0xEC, 0xFC, 0x2D, 0xBD,
    0xED, 0x3E, 0x4E, 0xBE, 0xDE, 0x4F, 0xCF, 0xC2, 0x33, 0x43, 0xC3, 0xD3,
    0x34, 0x44, 0x54, 0xB4, 0xD4, 0x45, 0x55, 0xC5, 0x4B, 0xCB, 0x3C, 0x4C,
    0x5C, 0xCC, 0xDC, 0x3D, 0x4D, 0x5D, 0xCD, 0xDD, 0xCE, 0xC4, 0xB5, 0x5B,
    0xBB,
    /* Table set B, code table 1, wordlen 6 - 256 entries */
    0x00, 0x10, 0x01, 0x11, 0x20, 0x21, 0x02, 0x12, 0x30, 0x31, 0x41, 0x22,
    0x32, 0x03, 0x13, 0x23, 0x14, 0x40, 0x51, 0x61, 0x42, 0x52, 0x33, 0x43,
    0x04, 0x24, 0x34, 0x15, 0x25, 0x16, 0x50, 0x60, 0x70, 0x71, 0x81, 0xD1,
    0xE1, 0x62, 0x72, 0x82, 0xD2, 0x53, 0x63, 0x73, 0xD3, 0x44, 0x54, 0x05,
    0x35, 0x45, 0x55, 0x06, 0x26, 0x36, 0x07, 0x17, 0x27, 0x37, 0x18, 0x28,
    0x19, 0x1D, 0x2D, 0x3D, 0x1E, 0x2E, 0x1F, 0x80, 0x90, 0xD0, 0xE0, 0xF0,
    0x91, 0xA1, 0xB1, 0xC1, 0xF1, 0x92, 0xA2, 0xB2, 0xC2, 0xE2, 0xF2, 0x83,
    0x93, 0xA3, 0xC3, 0xE3, 0xF3, 0x64, 0x74, 0x84, 0x94, 0xD4, 0xE4, 0xF4,
    0x65, 0x75, 0x85, 0xD5, 0xE5, 0x46, 0x56, 0x66, 0x76, 0xD6, 0xE6, 0x47,
    0x57, 0x67, 0xD7, 0x08, 0x38, 0x48, 0x58, 0x09, 0x29, 0x39, 0x49, 0x0A,
    0x1A, 0x2A, 0x3A, 0x1B, 0x2B, 0x0C, 0x1C, 0x2C, 0x3C, 0x0D, 0x4D, 0x5D,
    0x6D, 0x7D, 0x0E, 0x3E, 0x4E, 0x5E, 0x6E, 0x0F, 0x2F, 0x3F, 0x4F, 0xA0,
    0xB0, 0xC0, 0xB3, 0xA4, 0xB4, 0xC4, 0x95, 0xA5, 0xB5, 0xC5, 0xF5, 0x86,
    0x96, 0xA6, 0xB6, 0xC6, 0xF6, 0x77, 0x87, 0x97, 0xA7, 0xC7, 0xE7, 0xF7,
    0x68, 0x78, 0x88, 0x98, 0xD8, 0xE8, 0xF8, 0x59, 0x69, 0x79, 0x89, 0xD9,
    0xE9, 0xF9, 0x4A, 0x5A, 0x6A, 0x7A, 0xDA, 0xEA, 0x0B, 0x3B, 0x4B, 0x5B,
    0xDB, 0xEB, 0x4C, 0x5C, 0x6C, 0x7C, 0x8C, 0xDC, 0x8D, 0x9D, 0xAD, 0xBD,
    0xCD, 0x7E, 0x8E, 0x9E, 0xAE, 0xBE, 0x5F, 0x6F, 0x7F, 0x8F, 0x9F, 0xB7,
    0xA8, 0xB8, 0xC8, 0x99, 0xA9, 0xB9, 0xC9, 0x8A, 0x9A, 0xAA, 0xBA, 0xCA,
    0xFA, 0x6B, 0x7B, 0x8B, 0x9B, 0xAB, 0xBB, 0xCB, 0xFB, 0x9C, 0xAC, 0xBC,
    0xCC, 0xEC, 0xFC, 0xDD, 0xED, 0xFD, 0xCE, 0xDE, 0xEE, 0xFE, 0xAF, 0xBF,
    0xCF, 0xDF, 0xEF, 0xFF,
    /* Table set B, code table 2, wordlen 6 - 256 entries */
    0x00, 0x01, 0x10, 0x11, 0x20, 0x21, 0x02, 0x12, 0x30, 0x31, 0x22, 0x03,
    0x13, 0x40, 0x41, 0x51, 0x32, 0x42, 0x23, 0x33, 0x04, 0x14, 0x24, 0x15,
    0x50, 0x61, 0x71, 0xD1, 0xE1, 0x52, 0x62, 0xD2, 0x43, 0x53, 0xD3, 0x34,
    0x44, 0x05, 0x25, 0x35, 0x06, 0x16, 0x26, 0x17, 0x18, 0x1D, 0x2D, 0x3D,
    0x1E, 0x2E, 0x60, 0x70, 0x80, 0xD0, 0xE0, 0xF0, 0x81, 0x91, 0xA1, 0xC1,
    0xF1, 0x72, 0x82, 0x92, 0xC2, 0xE2, 0xF2, 0x63, 0x73, 0xE3, 0xF3, 0x54,
    0x64, 0x74, 0xD4, 0xE4, 0xF4, 0x45, 0x55, 0x65, 0xD5, 0xE5, 0xF5, 0x36,
    0x46, 0x56, 0xD6, 0xE6, 0x07, 0x27, 0x37, 0x47, 0xD7, 0x08, 0x28, 0x38,
    0x19, 0x29, 0x1A, 0x1B, 0x1C, 0x2C, 0x0D, 0x4D, 0x5D, 0x6D, 0x7D, 0x0E,
    0x3E, 0x4E, 0x5E, 0x6E, 0x0F, 0x1F, 0x2F, 0x3F, 0x4F, 0x90, 0xA0, 0xC0,
    0xB1, 0xA2, 0xB2, 0x83, 0x93, 0xA3, 0xB3, 0xC3, 0x84, 0x94, 0xA4, 0xC4,
    0x75, 0x85, 0x95, 0xC5, 0x66, 0x76, 0x86, 0x96, 0xC6, 0xF6, 0x57, 0x67,
    0x77, 0xE7, 0xF7, 0x48, 0x58, 0x68, 0x78, 0xD8, 0xE8, 0xF8, 0x09, 0x39,
    0x49, 0x59, 0xD9, 0xE9, 0xF9, 0x0A, 0x2A, 0x3A, 0x4A, 0xDA, 0xEA, 0x0B,
    0x2B, 0x3B, 0xDB, 0xEB, 0x0C, 0x3C, 0x4C, 0x5C, 0x6C, 0xDC, 0x8D, 0x9D,
    0xAD, 0xBD, 0xCD, 0x7E, 0x8E, 0x9E, 0xAE, 0xBE, 0xCE, 0x5F, 0x6F, 0x7F,
    0x8F, 0x9F, 0xAF, 0xB0, 0xB4, 0xA5, 0xB5, 0xA6, 0xB6, 0x87, 0x97, 0xA7,
    0xB7, 0xC7, 0x88, 0x98, 0xA8, 0xC8, 0x69, 0x79, 0x89, 0x99, 0xA9, 0xC9,
    0x5A, 0x6A, 0x7A, 0x9A, 0xCA, 0xFA, 0x4B, 0x5B, 0x6B, 0x7B, 0xCB, 0xFB,
    0x7C, 0x8C, 0x9C, 0xAC, 0xBC, 0xCC, 0xEC, 0xFC, 0xDD, 0xED, 0xFD, 0xDE,
    0xEE, 0xFE, 0xBF, 0xCF, 0xDF, 0xEF, 0xB8, 0xB9, 0x8A, 0xAA, 0xBA, 0x8B,
    0x9B, 0xAB, 0xBB, 0xFF,
    /* Table set B, code table 3, wordlen 2 - 81 entries */
    0x00, 0x40, 0x10, 0x04, 0x01, 0x50, 0x44, 0x14, 0x41, 0x11, 0x05, 0x80,
    0x54, 0x51, 0x45, 0x15, 0x55, 0x02, 0x90, 0x20, 0x60, 0x84, 0x24, 0x08,
    0x18, 0x09, 0x12, 0x06, 0xA0, 0x94, 0x64, 0x48, 0x58, 0x81, 0x91, 0x21,
    0x61, 0x85, 0x95, 0x25, 0x65, 0x49, 0x19, 0x59, 0x42, 0x52, 0x46, 0x16,
    0x56, 0x0A, 0xA4, 0x28, 0x68, 0xA1, 0xA5, 0x29, 0x69, 0x26, 0x4A, 0x1A,
    0x5A, 0x88, 0x98, 0xA8, 0x89, 0x99, 0xA9, 0x82, 0x92, 0x22, 0x62, 0x86,
    0x96, 0x66, 0x9A, 0x2A, 0x6A, 0xA2, 0xA6, 0x8A, 0xAA,
    /* Table set B, code table 3, wordlen 3 - 256 entries */
    0x00, 0x40, 0x10, 0x04, 0x01, 0x50, 0x44, 0x14, 0x54, 0x41, 0x11, 0x51,
    0x05, 0x45, 0x15, 0x55, 0x20, 0x95, 0x65, 0x49, 0x59, 0x52, 0x46, 0x16,
    0x80, 0x90, 0x60, 0x84, 0x94, 0x24, 0x64, 0xA4, 0x08, 0x48, 0x18, 0x58,
    0x81, 0x91, 0x21, 0x61, 0x85, 0x25, 0x09, 0x19, 0x69, 0x02, 0x42, 0x12,
    0x06, 0x56, 0x5A, 0x57, 0xD0, 0x74, 0x68, 0x5C, 0xC1, 0xD5, 0xA5, 0xE5,
    0x75, 0xB5, 0xF5, 0x99, 0xD9, 0xA9, 0xE9, 0x79, 0xB9, 0xF9, 0x1D, 0x5D,
    0x9D, 0xDD, 0x6D, 0xAD, 0xED, 0x7D, 0xBD, 0xFD, 0x82, 0x92, 0xD2, 0x62,
    0x96, 0xD6, 0x26, 0x66, 0xA6, 0xE6, 0x76, 0xB6, 0xF6, 0x0A, 0x4A, 0x1A,
    0x9A, 0xDA, 0x2A, 0x6A, 0xAA, 0xEA, 0x7A, 0xBA, 0xFA, 0x5E, 0x9E, 0xDE,
    0x6E, 0xAE, 0xEE, 0x7E, 0xBE, 0xFE, 0x03, 0x13, 0x53, 0x17, 0x97, 0xD7,
    0x67, 0xA7, 0xE7, 0x77, 0xB7, 0xF7, 0x5B, 0x9B, 0xDB, 0x6B, 0xAB, 0xEB,
    0x7B, 0xBB, 0xFB, 0x5F, 0x9F, 0xDF, 0x6F, 0xAF, 0xEF, 0x7F, 0xBF, 0xFF,
    0xC0, 0xA0, 0xE0, 0x30, 0xC4, 0xD4, 0xE4, 0x34, 0xB4, 0xF4, 0x88, 0xC8,
    0x98, 0xD8, 0x28, 0xA8, 0xE8, 0x38, 0x78, 0xB8, 0xF8, 0x0C, 0x4C, 0x1C,
    0x9C, 0xDC, 0x6C, 0xAC, 0xEC, 0x7C, 0xBC, 0xFC, 0xD1, 0xA1, 0xE1, 0x31,
    0x71, 0xB1, 0xF1, 0xC5, 0x35, 0x89, 0xC9, 0x29, 0x39, 0x0D, 0x4D, 0x8D,
    0xCD, 0x2D, 0x3D, 0x22, 0xA2, 0xE2, 0x72, 0xB2, 0xF2, 0x86, 0xC6, 0x36,
    0x8A, 0xCA, 0x3A, 0x0E, 0x4E, 0x8E, 0xCE, 0x1E, 0x2E, 0x3E, 0x43, 0x83,
    0x93, 0xD3, 0x23, 0x63, 0xA3, 0xE3, 0x73, 0xB3, 0xF3, 0x07, 0x47, 0x87,
    0xC7, 0x27, 0x37, 0x4B, 0x8B, 0xCB, 0x1B, 0x2B, 0x3B, 0x4F, 0x8F, 0xCF,
    0x1F, 0x70, 0xB0, 0xF0, 0x8C, 0xCC, 0x2C, 0x3C, 0xC2, 0x32, 0xC3, 0x0F,
    0x2F, 0x3F, 0x33, 0x0B,
    /* Table set B, code table 3, wordlen 5 - 225 entries */
    0x00, 0x10, 0xF0, 0x01, 0x0F, 0x11, 0xF1, 0x02, 0x0E, 0x1F, 0xFF, 0x20,
    0xE0, 0x21, 0xF2, 0xFE, 0xEF, 0x30, 0xD0, 0xE1, 0x12, 0x22, 0xE2, 0x03,
    0x0D, 0x1D, 0x1E, 0x2E, 0xEE, 0x2F, 0xDF, 0x40, 0x60, 0x70, 0x90, 0xA0,
    0xB0, 0xC0, 0x31, 0x71, 0x91, 0xC1, 0xD1, 0x32, 0xD2, 0x13, 0xE3, 0xF3,
    0x04, 0x05, 0x06, 0x07, 0x17, 0xF7, 0x09, 0x19, 0x0A, 0x1A, 0xFA, 0x0C,
    0x1C, 0x2D, 0xED, 0xFD, 0x3E, 0x7E, 0xDE, 0x3F, 0x6F, 0x7F, 0x9F, 0xAF,
    0xCF, 0x50, 0x41, 0x51, 0x61, 0xA1, 0xB1, 0x62, 0x72, 0x92, 0xA2, 0xC2,
    0x23, 0x33, 0x63, 0x73, 0x93, 0xA3, 0xD3, 0x14, 0x24, 0x34, 0xD4, 0xE4,
    0xF4, 0x15, 0xF5, 0x16, 0x26, 0xD6, 0xE6, 0xF6, 0x27, 0x37, 0x47, 0xE7,
    0x29, 0x39, 0xC9, 0xD9, 0xE9, 0xF9, 0x2A, 0xEA, 0x0B, 0x1B, 0xFB, 0x2C,
    0x7C, 0xEC, 0xFC, 0x3D, 0x4D, 0x6D, 0x7D, 0xDD, 0x4E, 0x5E, 0x6E, 0x9E,
    0xAE, 0xCE, 0x4F, 0x5F, 0x42, 0x52, 0xB2, 0x43, 0xB3, 0xC3, 0x44, 0x64,
    0x74, 0x94, 0xA4, 0x25, 0x35, 0x65, 0x75, 0x95, 0xA5, 0xE5, 0x36, 0x46,
    0x66, 0x76, 0x96, 0xA6, 0xB6, 0xC6, 0x57, 0xA7, 0xB7, 0xC7, 0xD7, 0x59,
    0xA9, 0xB9, 0x3A, 0x4A, 0x6A, 0xCA, 0xDA, 0x2B, 0x3B, 0x6B, 0x9B, 0xAB,
    0xDB, 0xEB, 0x3C, 0x6C, 0x9C, 0xAC, 0xCC, 0xDC, 0x5D, 0x9D, 0xAD, 0xBD,
    0xCD, 0xBE, 0xBF, 0x53, 0x54, 0xB4, 0xC4, 0x45, 0x55, 0xB5, 0xC5, 0xD5,
    0x56, 0x67, 0x77, 0x97, 0x49, 0x69, 0x79, 0x99, 0x5A, 0x7A, 0x9A, 0xAA,
    0xBA, 0x4B, 0x5B, 0x7B, 0xBB, 0xCB, 0x4C, 0x5C, 0xBC,
    /* Table set B, code table 3, wordlen 7 - 63 entries */
    0x00, 0x01, 0x3F, 0x02, 0x3E, 0x03, 0x3D, 0x04, 0x05, 0x06, 0x3A, 0x3B,
    0x3C, 0x07, 0x08, 0x09, 0x0A, 0x19, 0x1A, 0x1B, 0x1C, 0x1D, 0x22, 0x23,
    0x25, 0x26, 0x27, 0x37, 0x38, 0x39, 0x0B, 0x0D, 0x0E, 0x0F, 0x1E, 0x1F,
    0x21, 0x24, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x0C, 0x10, 0x11, 0x12,
    0x13, 0x15, 0x29, 0x2B, 0x2C, 0x2D, 0x2E, 0x2F, 0x30, 0x14, 0x16, 0x17,
    0x18, 0x28, 0x2A,
    /* Table set B, code table 4, wordlen 1 - 81 entries */
    0x00, 0x01, 0x40, 0xC0, 0x10, 0x30, 0x04, 0x0C, 0x03, 0x50, 0xD0, 0x70,
    0xF0, 0xC4, 0x14, 0x34, 0x4C, 0x1C, 0x3C, 0xC1, 0x11, 0x31, 0x05, 0x0D,
    0xC3, 0x13, 0x33, 0x07, 0x0F, 0x44, 0xCC, 0x41, 0x43, 0x54, 0x74, 0xDC,
    0xFC, 0x71, 0x15, 0x4D, 0x1D, 0x37, 0x3F, 0xD4, 0xF4, 0x5C, 0x7C, 0x51,
    0xD1, 0xF1, 0x45, 0xC5, 0x35, 0xCD, 0xDD, 0x3D, 0x53, 0xD3, 0x73, 0xF3,
    0x47, 0xC7, 0x17, 0x77, 0x4F, 0xCF, 0x1F, 0x55, 0xFF, 0xD5, 0x75, 0xF5,
    0x5D, 0x7D, 0xFD, 0x57, 0xD7, 0xF7, 0x5F, 0xDF, 0x7F,
    /* Table set B, code table 4, wordlen 2 - 25 entries */
    0x00, 0x07, 0x08, 0x38, 0x01, 0x39, 0x0F, 0x10, 0x09, 0x3F, 0x30, 0x31,
    0x02, 0x3A, 0x06, 0x0E, 0x17, 0x11, 0x0A, 0x3E, 0x37, 0x12, 0x32, 0x16,
    0x36,
    /* Table set B, code table 4, wordlen 3 - 49 entries */
    0x00, 0x07, 0x08, 0x38, 0x01, 0x09, 0x39, 0x0F, 0x3F, 0x10, 0x30, 0x31,
    0x02, 0x3A, 0x06, 0x0E, 0x17, 0x37, 0x18, 0x28, 0x11, 0x19, 0x29, 0x0A,
    0x03, 0x0B, 0x3B, 0x05, 0x0D, 0x3D, 0x3E, 0x1F, 0x2F, 0x12, 0x1A, 0x2A,
    0x32, 0x13, 0x33, 0x15, 0x35, 0x16, 0x1E, 0x2E, 0x36, 0x1B, 0x2B, 0x1D,
    0x2D,
    /* Table set B, code table 4, wordlen 7 - 63 entries */
    0x00, 0x01, 0x02, 0x3E, 0x3F, 0x03, 0x04, 0x3C, 0x3D, 0x05, 0x06, 0x07,
    0x39, 0x3A, 0x3B, 0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x1A, 0x1B, 0x25, 0x26,
    0x34, 0x35, 0x36, 0x37, 0x38, 0x0D, 0x0E, 0x0F, 0x10, 0x11, 0x12, 0x19,
    0x1C, 0x1D, 0x1E, 0x1F, 0x21, 0x22, 0x23, 0x24, 0x27, 0x2E, 0x2F, 0x30,
    0x31, 0x32, 0x33, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x28, 0x29, 0x2A,
    0x2B, 0x2C, 0x2D,
    /* Table set B, code table 5, wordlen 2 - 25 entries */
    0x00, 0x01, 0x08, 0x38, 0x07, 0x39, 0x0F, 0x30, 0x09, 0x3F, 0x10, 0x31,
    0x02, 0x3A, 0x06, 0x0E, 0x17, 0x11, 0x0A, 0x3E, 0x37, 0x12, 0x32, 0x16,
    0x36,
    /* Table set B, code table 5, wordlen 3 - 7 entries */
    0x00, 0x07, 0x01, 0x02, 0x03, 0x05, 0x06,
    /* Table set B, code table 5, wordlen 6 - 31 entries */
    0x00, 0x01, 0x1F, 0x02, 0x1E, 0x03, 0x04, 0x0D, 0x1D, 0x05, 0x06, 0x07,
    0x0E, 0x0F, 0x11, 0x12, 0x13, 0x1A, 0x1B, 0x1C, 0x08, 0x09, 0x0C, 0x14,
    0x15, 0x16, 0x17, 0x18, 0x19, 0x0A, 0x0B,
    /* Table set B, code table 6, wordlen 2 - 81 entries */
    0x00, 0x40, 0x01, 0x10, 0x04, 0x02, 0x80, 0x50, 0x90, 0x05, 0x06, 0x20,
    0x60, 0x44, 0x14, 0x54, 0x24, 0x08, 0x18, 0x41, 0x11, 0x15, 0x09, 0xA0,
    0x84, 0x94, 0x64, 0xA4, 0x28, 0x51, 0x45, 0x55, 0x19, 0x12, 0x16, 0x0A,
    0x1A, 0x48, 0x58, 0x68, 0x81, 0x91, 0x21, 0x61, 0x85, 0x95, 0x25, 0x65,
    0x49, 0x59, 0x29, 0x69, 0x42, 0x46, 0x56, 0x88, 0x98, 0xA8, 0xA1, 0xA5,
    0x99, 0xA9, 0x52, 0x22, 0x26, 0x66, 0x4A, 0x5A, 0x2A, 0x6A, 0x89, 0x82,
    0x92, 0x62, 0x86, 0x96, 0xA6, 0x8A, 0xAA, 0xA2, 0x9A,
    /* Table set B, code table 6, wordlen 3 - 7 entries */
    0x00, 0x01, 0x07, 0x02, 0x03, 0x05, 0x06,
    /* Table set B, code table 6, wordlen 4 - 6 entries */
    0x01, 0x00, 0x02, 0x05, 0x03, 0x04,
    /* Table set B, code table 6, wordlen 5 - 225 entries */
    0x00, 0xF0, 0x01, 0x10, 0x0F, 0x11, 0xF1, 0x20, 0xE0, 0x02, 0x0E, 0x1F,
    0xFF, 0xD0, 0x21, 0xE1, 0x12, 0xF2, 0x07, 0x0A, 0x0D, 0x1E, 0xFE, 0x2F,
    0xEF, 0x30, 0x70, 0x90, 0xA0, 0xC0, 0x71, 0x91, 0xC1, 0xD1, 0x32, 0x92,
    0xE2, 0x03, 0x13, 0x63, 0x04, 0x06, 0xE6, 0xE7, 0xF7, 0x09, 0x19, 0x39,
    0xFA, 0x0C, 0x1C, 0xDD, 0xED, 0xFD, 0x2E, 0x7E, 0x9E, 0x3F, 0x9F, 0x40,
    0x50, 0x60, 0xB0, 0x31, 0x41, 0x61, 0xA1, 0xB1, 0x22, 0x42, 0x72, 0xA2,
    0xB2, 0xC2, 0xD2, 0x23, 0x33, 0x73, 0xA3, 0xC3, 0xD3, 0xE3, 0xF3, 0x14,
    0x24, 0x34, 0x44, 0x74, 0xD4, 0xE4, 0x05, 0x25, 0x45, 0x65, 0x95, 0xA5,
    0x16, 0x26, 0x46, 0x76, 0xA6, 0xB6, 0xC6, 0xD6, 0xF6, 0x17, 0x27, 0x37,
    0x47, 0x67, 0xA7, 0xD7, 0x29, 0x69, 0xB9, 0xD9, 0xE9, 0xF9, 0x1A, 0x2A,
    0x3A, 0x9A, 0xCA, 0xDA, 0xEA, 0x0B, 0x1B, 0x3B, 0x6B, 0xEB, 0xFB, 0x2C,
    0x6C, 0xEC, 0xFC, 0x1D, 0x2D, 0x4D, 0x6D, 0x9D, 0xAD, 0x3E, 0x4E, 0x6E,
    0xAE, 0xCE, 0xEE, 0x4F, 0x5F, 0x6F, 0xDF, 0x51, 0x52, 0x62, 0x43, 0x93,
    0xB3, 0x54, 0x94, 0xA4, 0xF4, 0x15, 0x75, 0xB5, 0xE5, 0xF5, 0x36, 0x56,
    0x66, 0x96, 0x57, 0x77, 0x49, 0x59, 0xA9, 0xC9, 0x4A, 0x5A, 0x6A, 0x7A,
    0xAA, 0xBA, 0x2B, 0x4B, 0x7B, 0x9B, 0xAB, 0xDB, 0x3C, 0x4C, 0x7C, 0x9C,
    0xAC, 0xBC, 0xCC, 0x3D, 0x5D, 0x7D, 0xBD, 0xCD, 0x5E, 0xBE, 0xDE, 0xBF,
    0xCF, 0x53, 0x64, 0xB4, 0xC4, 0x35, 0x55, 0xC5, 0xD5, 0x97, 0xB7, 0xC7,
    0x79, 0x99, 0x5B, 0xBB, 0xCB, 0x5C, 0xDC, 0x7F, 0xAF,
    /* Table set B, code table 6, wordlen 6 - 256 entries */
    0x00, 0x10, 0x01, 0x11, 0x21, 0x02, 0x12, 0x20, 0x22, 0x13, 0x30, 0x31,
    0x41, 0xD1, 0xE1, 0x32, 0x52, 0x03, 0x23, 0x2D, 0x40, 0x50, 0x60, 0x80,
    0xD0, 0xE0, 0x51, 0x61, 0xF1, 0x42, 0x62, 0xD2, 0xE2, 0xF2, 0x33, 0x43,
    0xC3, 0xD3, 0xE3, 0x04, 0x14, 0xD4, 0xF4, 0x25, 0x35, 0x16, 0x17, 0xF7,
    0xD8, 0x1C, 0x3C, 0x0D, 0x1D, 0x3D, 0x5D, 0x0E, 0x1E, 0x2E, 0x7E, 0x2F,
    0xC0, 0xF0, 0x71, 0x81, 0x91, 0xC1, 0x72, 0x82, 0x92, 0xB2, 0xC2, 0x53,
    0x63, 0x73, 0x93, 0xA3, 0xF3, 0x24, 0x44, 0x64, 0x84, 0xA4, 0xB4, 0x05,
    0x15, 0x95, 0xD5, 0x06, 0x26, 0x36, 0x46, 0x96, 0xD6, 0xE6, 0xF6, 0x07,
    0x27, 0x37, 0xD7, 0xE7, 0x08, 0x18, 0x28, 0x38, 0xE8, 0xF8, 0x09, 0x19,
    0x29, 0xE9, 0xF9, 0x0A, 0x1A, 0xCA, 0xDA, 0xEA, 0x0B, 0x1B, 0xDB, 0xEB,
    0xFB, 0x2C, 0x4C, 0x5C, 0x7C, 0x8C, 0x4D, 0x6D, 0x8D, 0x9D, 0xFD, 0x3E,
    0x5E, 0x6E, 0x8E, 0x9E, 0xEE, 0x0F, 0x1F, 0x3F, 0x4F, 0x5F, 0x6F, 0x7F,
    0xCF, 0x70, 0xA1, 0xA2, 0x83, 0xB3, 0x34, 0x74, 0xC4, 0xE4, 0x55, 0x65,
    0x85, 0xA5, 0xC5, 0xE5, 0xF5, 0x56, 0x66, 0x76, 0x86, 0xA6, 0xC6, 0x57,
    0x67, 0x77, 0x97, 0xA7, 0x48, 0x88, 0x98, 0x49, 0x59, 0x79, 0x99, 0x3A,
    0x4A, 0x8A, 0xBA, 0xFA, 0x2B, 0x7B, 0x0C, 0xAC, 0xBC, 0xCC, 0xEC, 0x7D,
    0xAD, 0xBD, 0xDD, 0x4E, 0xBE, 0xCE, 0xFE, 0x8F, 0x9F, 0xAF, 0xBF, 0xDF,
    0xEF, 0xFF, 0x90, 0xA0, 0xB0, 0xB1, 0x54, 0x94, 0x45, 0x75, 0xB5, 0xB6,
    0x47, 0x87, 0xB7, 0xC7, 0x58, 0x68, 0x78, 0xA8, 0xB8, 0xC8, 0x39, 0x69,
    0x89, 0xA9, 0xB9, 0xC9, 0xD9, 0x2A, 0x5A, 0x6A, 0x7A, 0x9A, 0xAA, 0x3B,
    0x4B, 0x5B, 0x6B, 0x8B, 0x9B, 0xAB, 0xBB, 0xCB, 0x6C, 0x9C, 0xDC, 0xFC,
    0xCD, 0xED, 0xAE, 0xDE,
    /* Table set B, code table 6, wordlen 7 - 63 entries */
    0x00, 0x01, 0x3F, 0x3E, 0x02, 0x03, 0x3C, 0x3D, 0x04, 0x05, 0x06, 0x07,
    0x39, 0x3A, 0x3B, 0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x34, 0x35, 0x36, 0x37,
    0x38, 0x0D, 0x0E, 0x0F, 0x10, 0x11, 0x12, 0x19, 0x1A, 0x1B, 0x1C, 0x1D,
    0x1E, 0x1F, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27, 0x2E, 0x2F, 0x30,
    0x31, 0x32, 0x33, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x28, 0x29, 0x2A,
    0x2B, 0x2C, 0x2D,
    /* Table set B, code table 7, wordlen 1 - 81 entries */
    0x00, 0x03, 0x40, 0xC0, 0x10, 0x30, 0x04, 0x0C, 0x01, 0x50, 0xD0, 0x70,
    0xF0, 0xC4, 0x14, 0x34, 0x4C, 0x1C, 0x3C, 0xC1, 0x11, 0x31, 0x05, 0x0D,
    0x13, 0x33, 0x07, 0x0F, 0x44, 0xCC, 0x41, 0x43, 0xC3, 0x54, 0x74, 0xDC,
    0xFC, 0xF1, 0xC5, 0x15, 0x1D, 0x53, 0xC7, 0x37, 0x4F, 0x3F, 0xD4, 0xF4,
    0x5C, 0x7C, 0x51, 0xD1, 0x71, 0x45, 0x55, 0x35, 0x4D, 0xCD, 0xDD, 0x3D,
    0xD3, 0x73, 0xF3, 0x47, 0x17, 0x77, 0xCF, 0x1F, 0xFF, 0xD5, 0x75, 0xF5,
    0x5D, 0x7D, 0xFD, 0x57, 0xD7, 0xF7, 0x5F, 0xDF, 0x7F,
    /* Table set B, code table 7, wordlen 3 - 256 entries */
    0x00, 0x40, 0x10, 0x04, 0x01, 0x05, 0x50, 0x14, 0x54, 0x41, 0x11, 0x51,
    0x45, 0x15, 0x55, 0x44, 0x95, 0x6A, 0x03, 0x80, 0xC0, 0x90, 0xD0, 0x94,
    0xD4, 0x24, 0x64, 0x58, 0x91, 0xA1, 0x85, 0xD5, 0x25, 0x65, 0xA5, 0xE5,
    0x75, 0xB5, 0xF5, 0x19, 0x59, 0x99, 0xD9, 0x69, 0xA9, 0xE9, 0x79, 0xB9,
    0xF9, 0x4D, 0x5D, 0x9D, 0xDD, 0x6D, 0xAD, 0xED, 0x7D, 0xBD, 0xFD, 0x02,
    0x42, 0x52, 0x06, 0x46, 0x16, 0x56, 0x96, 0xD6, 0x26, 0x66, 0xA6, 0xE6,
    0x76, 0xB6, 0xF6, 0x1A, 0x5A, 0x9A, 0xDA, 0xAA, 0xEA, 0x7A, 0xBA, 0xFA,
    0x5E, 0x9E, 0xDE, 0x6E, 0xAE, 0xEE, 0x7E, 0xBE, 0xFE, 0x07, 0x47, 0x57,
    0x97, 0xD7, 0x67, 0xA7, 0xE7, 0x77, 0xB7, 0xF7, 0x5B, 0x9B, 0xDB, 0x6B,
    0xAB, 0xEB, 0x7B, 0xBB, 0xFB, 0x5F, 0x9F, 0xDF, 0x6F, 0xAF, 0xEF, 0x7F,
    0xBF, 0xFF, 0x20, 0x60, 0x70, 0xB0, 0xF0, 0x84, 0xC4, 0xA4, 0xE4, 0x74,
    0xB4, 0xF4, 0x08, 0x88, 0x18, 0x98, 0xD8, 0x68, 0xA8, 0xE8, 0x78, 0xB8,
    0xF8, 0x1C, 0x5C, 0x9C, 0xDC, 0x6C, 0xAC, 0xEC, 0x3C, 0x7C, 0xBC, 0xFC,
    0x81, 0xD1, 0x21, 0x61, 0xE1, 0x71, 0xB1, 0xF1, 0xC5, 0x35, 0x09, 0x49,
    0x89, 0xC9, 0x29, 0x39, 0x0D, 0x8D, 0xCD, 0x1D, 0x2D, 0x3D, 0x92, 0xD2,
    0x22, 0x62, 0xA2, 0xE2, 0x72, 0xB2, 0xF2, 0x86, 0xC6, 0x36, 0x0A, 0x4A,
    0x8A, 0xCA, 0x2A, 0x3A, 0x0E, 0x4E, 0x8E, 0xCE, 0x1E, 0x2E, 0x3E, 0x13,
    0x53, 0x93, 0xD3, 0x63, 0xA3, 0xE3, 0x73, 0xB3, 0xF3, 0x87, 0xC7, 0x17,
    0x27, 0x37, 0x0B, 0x4B, 0x8B, 0xCB, 0x1B, 0x2B, 0x3B, 0x0F, 0x4F, 0x8F,
    0xCF, 0x1F, 0x2F, 0x3F, 0xA0, 0xE0, 0x30, 0x34, 0x48, 0xC8, 0x28, 0x38,
    0x0C, 0x4C, 0x8C, 0xCC, 0x2C, 0xC1, 0x31, 0x82, 0xC2, 0x12, 0x32, 0x43,
    0x83, 0xC3, 0x23, 0x33,
    /* Table set B, code table 7, wordlen 4 - 11 entries */
    0x00, 0x01, 0x0F, 0x02, 0x0E, 0x05, 0x0D, 0x03, 0x04, 0x0B, 0x0C,
    /* Table set B, code table 7, wordlen 5 - 225 entries */
    0x00, 0x10, 0xF0, 0x01, 0x0F, 0x20, 0xE0, 0x11, 0xF1, 0x02, 0x0E, 0x1F,
    0xFF, 0xD0, 0x21, 0xE1, 0x12, 0xF2, 0x03, 0x0D, 0x1E, 0xFE, 0x2F, 0xEF,
    0x30, 0x40, 0x60, 0x70, 0x90, 0xA0, 0xC0, 0x31, 0xD1, 0x22, 0xE2, 0x13,
    0xF3, 0x04, 0x06, 0x07, 0x09, 0x0C, 0x1D, 0xFD, 0x2E, 0xEE, 0x3F, 0xDF,
    0x50, 0xB0, 0x41, 0x61, 0x71, 0x91, 0xA1, 0xC1, 0x32, 0x62, 0x72, 0x92,
    0xA2, 0xD2, 0x23, 0xD3, 0xE3, 0x14, 0xF4, 0x05, 0x16, 0x26, 0xE6, 0xF6,
    0x17, 0x27, 0xE7, 0xF7, 0x19, 0x29, 0xF9, 0x0A, 0x1A, 0x2A, 0xFA, 0x0B,
    0x1C, 0x2C, 0xFC, 0x2D, 0x3D, 0xED, 0x3E, 0x4E, 0x7E, 0x9E, 0xDE, 0x4F,
    0x6F, 0x7F, 0x9F, 0xAF, 0xCF, 0x51, 0xB1, 0x42, 0x52, 0xB2, 0xC2, 0x33,
    0x63, 0x73, 0x93, 0xA3, 0xB3, 0xC3, 0x24, 0x34, 0x74, 0xA4, 0xD4, 0xE4,
    0x15, 0x25, 0x65, 0x95, 0xE5, 0xF5, 0x36, 0xD6, 0x37, 0x47, 0xC7, 0xD7,
    0x39, 0x59, 0xB9, 0xC9, 0xD9, 0xE9, 0x3A, 0x6A, 0xDA, 0xEA, 0x1B, 0x2B,
    0x9B, 0xAB, 0xEB, 0xFB, 0x6C, 0x7C, 0x9C, 0xAC, 0xEC, 0x4D, 0x6D, 0x7D,
    0x9D, 0xAD, 0xBD, 0xDD, 0x5E, 0x6E, 0xAE, 0xCE, 0x5F, 0x43, 0x53, 0x44,
    0x54, 0x64, 0x94, 0xB4, 0xC4, 0x35, 0x45, 0x55, 0x75, 0xA5, 0xB5, 0xC5,
    0xD5, 0x46, 0x56, 0x66, 0x76, 0x96, 0xA6, 0xB6, 0xC6, 0x57, 0x67, 0x77,
    0x97, 0xA7, 0xB7, 0x49, 0x69, 0x79, 0x99, 0xA9, 0x4A, 0x5A, 0x7A, 0x9A,
    0xAA, 0xBA, 0xCA, 0x3B, 0x4B, 0x5B, 0x6B, 0x7B, 0xBB, 0xCB, 0xDB, 0x3C,
    0x4C, 0x5C, 0xBC, 0xCC, 0xDC, 0x5D, 0xCD, 0xBE, 0xBF,
    /* Table set B, code table 7, wordlen 7 - 63 entries */
    0x00, 0x01, 0x02, 0x3E, 0x3F, 0x03, 0x04, 0x05, 0x3B, 0x3C, 0x3D, 0x06,
    0x07, 0x08, 0x09, 0x0A, 0x36, 0x37, 0x38, 0x39, 0x3A, 0x0B, 0x0C, 0x0D,
    0x0E, 0x0F, 0x10, 0x2F, 0x31, 0x32, 0x33, 0x34, 0x35, 0x11, 0x12, 0x13,
    0x14, 0x2D, 0x2E, 0x30, 0x15, 0x16, 0x17, 0x18, 0x19, 0x1A, 0x1B, 0x25,
    0x26, 0x27, 0x28, 0x29, 0x2A, 0x2B, 0x2C, 0x1C, 0x1D, 0x1E, 0x23, 0x1F,
    0x21, 0x22, 0x24,
    /* Table set B, code table 7, wordlen 7 - 8 entries */
    0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
    /* Table set B, code table 7, wordlen 7 - 8 entries */
    0x00, 0x01, 0x07, 0x02, 0x06, 0x03, 0x04, 0x05,
    /* Table set B, code table 7, wordlen 7 - 16 entries */
    0x07, 0x05, 0x08, 0x06, 0x09, 0x04, 0x0A, 0x0B, 0x00, 0x01, 0x02, 0x03,
    0x0C, 0x0D, 0x0E, 0x0F,
    /* Table set B, code table 7, wordlen 7 - 15 entries */
    0x0F, 0x0E, 0x01, 0x0D, 0x02, 0x03, 0x0C, 0x04, 0x05, 0x06, 0x07, 0x08,
    0x09, 0x0A, 0x0B,
    /* Table set B, code table 7, wordlen 7 - 16 entries */
    0x00, 0x01, 0x0E, 0x0F, 0x02, 0x0D, 0x03, 0x0C, 0x04, 0x05, 0x06, 0x07,
    0x08, 0x09, 0x0A, 0x0B,
    /* Table set B, code table 7, wordlen 7 - 16 entries */
    0x00, 0x01, 0x0F, 0x0E, 0x02, 0x0D, 0x03, 0x0C, 0x04, 0x05, 0x06, 0x07,
    0x08, 0x09, 0x0A, 0x0B,
    /* Table set B, code table 7, wordlen 7 - 32 entries */
    0x00, 0x01, 0x1F, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x1A, 0x1B, 0x1C,
    0x1D, 0x1E, 0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F, 0x10, 0x11,
    0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19,
    /* Table set B, code table 7, wordlen 7 - 31 entries */
    0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0A, 0x0B, 0x0C,
    0x0D, 0x0E, 0x0F, 0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18,
    0x19, 0x1A, 0x1B, 0x1C, 0x1D, 0x1E, 0x1F,
    /* Table set B, code table 7, wordlen 7 - 32 entries */
    0x00, 0x1C, 0x1D, 0x01, 0x02, 0x03, 0x04, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F,
    0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19, 0x1A, 0x1B,
    0x1E, 0x1F, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0A,
    /* Table set B, code table 7, wordlen 7 - 31 entries */
    0x02, 0x03, 0x04, 0x05, 0x06, 0x01, 0x07, 0x08, 0x09, 0x0A, 0x0B, 0x0C,
    0x0D, 0x0E, 0x0F, 0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18,
    0x19, 0x1A, 0x1B, 0x1C, 0x1D, 0x1E, 0x1F,
    /* Table set B, code table 7, wordlen 7 - 32 entries */
    0x00, 0x01, 0x02, 0x1F, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x1D, 0x1E,
    0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F, 0x10, 0x11, 0x12, 0x13, 0x14,
    0x15, 0x16, 0x17, 0x18, 0x19, 0x1A, 0x1B, 0x1C,
    /* Table set B, code table 7, wordlen 7 - 16 entries */
    0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0A, 0x0B,
    0x0C, 0x0D, 0x0E, 0x0F,
    /* Table set B, code table 7, wordlen 7 - 8 entries */
    0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
    /* Table set B, code table 7, wordlen 7 - 8 entries */
    0x00, 0x01, 0x07, 0x02, 0x03, 0x04, 0x05, 0x06,
    /* Table set B, code table 7, wordlen 7 - 32 entries */
    0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F, 0x10, 0x11, 0x05, 0x06,
    0x07, 0x12, 0x13, 0x14, 0x15, 0x16, 0x04, 0x17, 0x18, 0x19, 0x1A, 0x1B,
    0x00, 0x01, 0x02, 0x03, 0x1C, 0x1D, 0x1E, 0x1F,
    /* Table set B, code table 7, wordlen 7 - 32 entries */
    0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19, 0x1A, 0x1B, 0x1C, 0x0E,
    0x0F, 0x10, 0x11, 0x1D, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x1E, 0x04, 0x05,
    0x06, 0x07, 0x08, 0x1F, 0x00, 0x01, 0x02, 0x03,
    /* Table set B, code table 7, wordlen 7 - 32 entries */
    0x00, 0x01, 0x02, 0x1F, 0x03, 0x1D, 0x1E, 0x04, 0x05, 0x06, 0x1B, 0x1C,
    0x07, 0x18, 0x19, 0x1A, 0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F,
    0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
    /* Table set B, code table 7, wordlen 7 - 256 entries */
    0x00, 0x01, 0xFF, 0x02, 0xFE, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0xFB,
    0xFC, 0xFD, 0x09, 0x0A, 0x0B, 0x0C, 0xF6, 0xF7, 0xF8, 0xF9, 0xFA, 0x0D,
    0x0E, 0x0F, 0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19,
    0x1A, 0x1B, 0x1C, 0x1D, 0x1E, 0x1F, 0x20, 0x21, 0x22, 0x23, 0x24, 0x25,
    0x26, 0xF3, 0xF4, 0xF5, 0x27, 0x28, 0x29, 0x2A, 0x2B, 0x2C, 0x2D, 0x2E,
    0x2F, 0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x3A,
    0x3B, 0x3C, 0x3D, 0x3E, 0x3F, 0x40, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46,
    0x47, 0x48, 0x49, 0x4A, 0x4B, 0x4C, 0x4D, 0x4E, 0x4F, 0x50, 0x51, 0x52,
    0x53, 0x54, 0x55, 0x56, 0x57, 0x58, 0x59, 0x5A, 0x5B, 0x5C, 0x5D, 0xC2,
    0xC3, 0xC4, 0xC5, 0xC6, 0xC7, 0xC8, 0xC9, 0xCA, 0xCB, 0xCC, 0xCD, 0xCE,
    0xCF, 0xD0, 0xD1, 0xD2, 0xD3, 0xD4, 0xD5, 0xD6, 0xD7, 0xD8, 0xD9, 0xDA,
    0xDB, 0xDC, 0xDD, 0xDE, 0xDF, 0xE0, 0xE1, 0xE2, 0xE3, 0xE4, 0xE5, 0xE6,
    0xE7, 0xE8, 0xE9, 0xEA, 0xEB, 0xEC, 0xED, 0xEE, 0xEF, 0xF0, 0xF1, 0xF2,
    0x5E, 0x5F, 0x60, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0x68, 0x69,
    0x6A, 0x6B, 0x6C, 0x6D, 0x6E, 0x6F, 0x70, 0x71, 0x72, 0x73, 0x74, 0x75,
    0x76, 0x77, 0x78, 0x79, 0x7A, 0x7B, 0x7C, 0x7D, 0x7E, 0x7F, 0x80, 0x81,
    0x82, 0x83, 0x84, 0x85, 0x86, 0x87, 0x88, 0x89, 0x8A, 0x8B, 0x8C, 0x8D,
    0x8E, 0x8F, 0x90, 0x91, 0x92, 0x93, 0x94, 0x95, 0x96, 0x97, 0x98, 0x99,
    0x9A, 0x9B, 0x9C, 0x9D, 0x9E, 0x9F, 0xA0, 0xA1, 0xA2, 0xA3, 0xA4, 0xA5,
    0xA6, 0xA7, 0xA8, 0xA9, 0xAA, 0xAB, 0xAC, 0xAD, 0xAE, 0xAF, 0xB0, 0xB1,
    0xB2, 0xB3, 0xB4, 0xB5, 0xB6, 0xB7, 0xB8, 0xB9, 0xBA, 0xBB, 0xBC, 0xBD,
    0xBE, 0xBF, 0xC0, 0xC1,
};

/** Tables for spectrum coding. */
typedef struct Atrac3pSpecCodeTab {
    uint8_t group_size;  ///< number of coefficients grouped together
    uint8_t num_coeffs;  ///< 1 - map index to a single value, > 1 - map index to a vector of values
    uint8_t bits;        ///< number of bits a single coefficient occupy
    uint8_t is_signed;   ///< 1 - values in that table are signed ones, otherwise - absolute ones
} Atrac3pSpecCodeTab;

static const Atrac3pSpecCodeTab atrac3p_spectra_tabs[112] = {
    /* table set = A */
    /* code table = 0 */
    { 1, 4, 2, 1 }, // wordlen = 1
    { 1, 4, 2, 0 }, // wordlen = 2
    { 1, 2, 3, 1 }, // wordlen = 3
    { 1, 1, 3, 0 }, // wordlen = 4
    { 1, 2, 4, 1 }, // wordlen = 5
    { 1, 1, 4, 0 }, // wordlen = 6
    { 1, 1, 6, 1 }, // wordlen = 7

    /* code table = 1 */
    { 4, 4, 2, 1 }, // wordlen = 1
    { 1, 4, 2, 0 }, // wordlen = 2
    { 1, 2, 3, 1 }, // wordlen = 3
    { 1, 2, 4, 1 }, // wordlen = 4
    { 1, 2, 4, 1 }, // wordlen = 5
    { 1, 2, 4, 0 }, // wordlen = 6
    { 1, 1, 6, 1 }, // wordlen = 7

    /* code table = 2 */
    { 1, 4, 2, 1 }, // wordlen = 1
    { 1, 2, 3, 1 }, // wordlen = 2
    { 1, 2, 3, 1 }, // wordlen = 3
    { 1, 1, 3, 0 }, // wordlen = 4
    { 1, 1, 3, 0 }, // wordlen = 5
    { 1, 2, 4, 0 }, // wordlen = 6
    { 1, 1, 6, 1 }, // wordlen = 7

    /* code table = 3 */
    { 1, 2, 2, 1 }, // wordlen = 1
    { 1, 4, 2, 0 }, // wordlen = 2
    { 1, 4, 2, 0 }, // wordlen = 3
    { 1, 1, 3, 0 }, // wordlen = 4
    { 1, 2, 4, 1 }, // wordlen = 5
    { 1, 2, 4, 0 }, // wordlen = 6
    { 1, 1, 6, 1 }, // wordlen = 7

    /* code table = 4 */
    { 1, 4, 2, 1 }, // wordlen = 1
    { 1, 4, 2, 0 }, // wordlen = 2
    { 1, 1, 3, 1 }, // wordlen = 3
    { 1, 2, 4, 1 }, // wordlen = 4
    { 1, 1, 3, 0 }, // wordlen = 5
    { 1, 2, 4, 0 }, // wordlen = 6
    { 1, 1, 6, 1 }, // wordlen = 7

    /* code table = 5 */
    { 1, 4, 2, 1 }, // wordlen = 1
    { 1, 2, 3, 1 }, // wordlen = 2
    { 1, 1, 3, 1 }, // wordlen = 3
    { 1, 2, 4, 1 }, // wordlen = 4
    { 1, 1, 3, 0 }, // wordlen = 5
    { 1, 1, 5, 1 }, // wordlen = 6
    { 1, 1, 6, 1 }, // wordlen = 7

    /* code table = 6 */
    { 2, 4, 2, 1 }, // wordlen = 1
    { 1, 2, 3, 1 }, // wordlen = 2
    { 1, 4, 2, 0 }, // wordlen = 3
    { 1, 2, 4, 1 }, // wordlen = 4
    { 1, 1, 4, 1 }, // wordlen = 5
    { 1, 1, 5, 1 }, // wordlen = 6
    { 1, 1, 6, 1 }, // wordlen = 7

    /* code table = 7 */
    { 1, 2, 1, 0 }, // wordlen = 1
    { 2, 4, 2, 0 }, // wordlen = 2
    { 1, 2, 3, 1 }, // wordlen = 3
    { 1, 2, 4, 1 }, // wordlen = 4
    { 1, 1, 4, 1 }, // wordlen = 5
    { 2, 2, 4, 0 }, // wordlen = 6
    { 4, 1, 6, 1 }, // wordlen = 7

    /* table set = B */
    /* code table = 0 */
    { 4, 4, 2, 1 }, // wordlen = 1
    { 1, 4, 2, 0 }, // wordlen = 2
    { 4, 2, 3, 1 }, // wordlen = 3
    { 1, 2, 4, 1 }, // wordlen = 4
    { 1, 2, 4, 1 }, // wordlen = 5
    { 1, 1, 4, 0 }, // wordlen = 6
    { 1, 1, 6, 1 }, // wordlen = 7

    /* code table = 1 */
    { 1, 4, 2, 1 }, // wordlen = 1
    { 1, 4, 2, 0 }, // wordlen = 2
    { 1, 2, 3, 1 }, // wordlen = 3
    { 1, 2, 4, 1 }, // wordlen = 4
    { 1, 2, 4, 1 }, // wordlen = 5
    { 1, 2, 4, 0 }, // wordlen = 6
    { 1, 1, 6, 1 }, // wordlen = 7

    /* code table = 2 */
    { 4, 4, 2, 1 }, // wordlen = 1
    { 4, 4, 2, 0 }, // wordlen = 2
    { 1, 2, 3, 1 }, // wordlen = 3
    { 1, 2, 4, 1 }, // wordlen = 4
    { 2, 2, 4, 1 }, // wordlen = 5
    { 2, 2, 4, 0 }, // wordlen = 6
    { 4, 1, 6, 1 }, // wordlen = 7

    /* code table = 3 */
    { 1, 4, 2, 1 }, // wordlen = 1
    { 1, 4, 2, 0 }, // wordlen = 2
    { 1, 4, 2, 0 }, // wordlen = 3
    { 2, 2, 4, 1 }, // wordlen = 4
    { 1, 2, 4, 1 }, // wordlen = 5
    { 1, 2, 4, 0 }, // wordlen = 6
    { 1, 1, 6, 1 }, // wordlen = 7

    /* code table = 4 */
    { 1, 4, 2, 1 }, // wordlen = 1
    { 4, 2, 3, 1 }, // wordlen = 2
    { 1, 2, 3, 1 }, // wordlen = 3
    { 4, 2, 4, 1 }, // wordlen = 4
    { 1, 1, 3, 0 }, // wordlen = 5
    { 1, 2, 4, 0 }, // wordlen = 6
    { 1, 1, 6, 1 }, // wordlen = 7

    /* code table = 5 */
    { 2, 4, 2, 1 }, // wordlen = 1
    { 1, 2, 3, 1 }, // wordlen = 2
    { 4, 1, 3, 1 }, // wordlen = 3
    { 1, 1, 3, 0 }, // wordlen = 4
    { 1, 1, 3, 0 }, // wordlen = 5
    { 1, 1, 5, 1 }, // wordlen = 6
    { 2, 1, 6, 1 }, // wordlen = 7

    /* code table = 6 */
    { 1, 4, 2, 1 }, // wordlen = 1
    { 1, 4, 2, 0 }, // wordlen = 2
    { 1, 1, 3, 1 }, // wordlen = 3
    { 1, 1, 3, 0 }, // wordlen = 4
    { 4, 2, 4, 1 }, // wordlen = 5
    { 1, 2, 4, 0 }, // wordlen = 6
    { 1, 1, 6, 1 }, // wordlen = 7

    /* code table = 7 */
    { 1, 4, 2, 1 }, // wordlen = 1
    { 4, 4, 2, 0 }, // wordlen = 2
    { 4, 4, 2, 0 }, // wordlen = 3
    { 1, 1, 4, 1 }, // wordlen = 4
    { 1, 2, 4, 1 }, // wordlen = 5
    { 1, 1, 5, 1 }, // wordlen = 6
    { 1, 1, 6, 1 }, // wordlen = 7
};

/* Huffman tables for gain control data. */
static const uint8_t atrac3p_gain_cbs[][12] = {
    {  1,  1,  1,  1,  1,  1,  2,  0,  0,  0,  0,  0 },
    {  1,  1,  1,  1,  1,  1,  2,  0,  0,  0,  0,  0 },
    {  1,  0,  2,  2,  1,  2,  8,  0,  0,  0,  0,  0 },
    {  1,  1,  1,  1,  1,  0,  2,  0,  8,  0,  0,  0 },
    {  1,  0,  3,  1,  1,  0,  2,  0,  8,  0,  0,  0 },
    {  1,  1,  1,  1,  1,  0,  1,  2,  8,  0,  0,  0 },
    {  0,  1,  0,  2, 11, 18,  0,  0,  0,  0,  0,  0 },
    {  0,  1,  2,  4,  4,  4,  0, 16,  0,  0,  0,  0 },
    {  0,  0,  0,  3, 23,  6,  0,  0,  0,  0,  0,  0 },
    {  0,  0,  5,  3,  2,  3,  2, 16,  0,  0,  0,  0 },
    {  1,  0,  0,  3,  2,  6, 20,  0,  0,  0,  0,  0 },
};

static const uint8_t atrac3p_gain_xlats[] = {
    /* Number of gain control points 1 - 8 entries */
     0,  1,  2,  3,  4,  5,  6,  7,
    /* Number of gain control points 2 - 8 entries */
     0,  1,  7,  2,  6,  3,  4,  5,
    /* Gain compensation level 1 - 16 entries */
     7,  5,  8,  6,  9,  4, 10, 11,  0,  1,  2,  3, 12, 13, 14, 15,
    /* Gain compensation level 2 - 15 entries */
    15, 14,  1, 13,  2,  3, 12,  4,  5,  6,  7,  8,  9, 10, 11,
    /* Gain compensation level 3 - 16 entries */
     0,  1, 14, 15,  2, 13,  3, 12,  4,  5,  6,  7,  8,  9, 10, 11,
    /* Gain compensation level 4 - 16 entries */
     0,  1, 15, 14,  2, 13,  3, 12,  4,  5,  6,  7,  8,  9, 10, 11,
    /* Gain compensation location 3 - 32 entries */
     0,  1, 31,  2,  3,  4,  5,  6,  7, 26, 27, 28, 29, 30,  8,  9, 10, 11, 12,
    13, 14, 15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25,
    /* Gain compensation location 1 - 31 entries */
     1,  2,  3,  4,  5,  6,  7,  8,  9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19,
    20, 21, 22, 23, 24, 25, 26, 27, 28, 29, 30, 31,
    /* Gain compensation location 4 - 32 entries */
     0, 28, 29,  1,  2,  3,  4, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 21, 22,
    23, 24, 25, 26, 27, 30, 31,  5,  6,  7,  8,  9, 10,
    /* Gain compensation location 2 - 31 entries */
     2,  3,  4,  5,  6,  1,  7,  8,  9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19,
    20, 21, 22, 23, 24, 25, 26, 27, 28, 29, 30, 31,
    /* Gain compensation location 5 - 32 entries */
     0,  1,  2, 31,  3,  4,  5,  6,  7,  8, 29, 30,  9, 10, 11, 12, 13, 14, 15,
    16, 17, 18, 19, 20, 21, 22, 23, 24, 25, 26, 27, 28,
};

/* Huffman tables for GHA waves data. */
static const uint8_t atrac3p_tone_cbs[][12] = {
    {  1,  0,  1,  2,  4,  8,  0,  0,  0,  0,  0,  0 },
    {  1,  1,  1,  1,  1,  1,  2,  0,  0,  0,  0,  0 },
    {  1,  1,  1,  1,  0,  4,  0,  0,  0,  0,  0,  0 },
    {  0,  0,  0, 10,  8,  6,  0,  8,  0,  0,  0,  0 },
    {  0,  0,  0, 11,  5,  6,  6,  4,  0,  0,  0,  0 },
    {  0,  1,  3,  3,  1,  4,  4, 16,  0,  0,  0,  0 },
    {  1,  0,  0,  2,  2,  0,  9,  9, 29,104, 100, 0 },
};

static const uint8_t atrac3p_tone_xlats[] = {
    /* Number of tone bands - 16 entries */
      0,   1,   2,   3,   4,   5,   6,   7,   8,   9,  10,  11,  12,  13,  14,
     15,
    /* Number of tones 1 - 8 entries */
      0,   1,   2,   3,   4,   5,   6,   7,
    /* Number of tones 2 - 8 entries */
      0,   1,   7,   2,   3,   4,   5,   6,
    /* Amplitude information 1 - 32 entries */
      8,   9,  10,  11,  12,  13,  14,  15,  16,  17,   5,   6,   7,  18,  19,
     20,  21,  22,   4,  23,  24,  25,  26,  27,   0,   1,   2,   3,  28,  29,
     30,  31,
    /* Amplitude information 2 - 32 entries */
     18,  19,  20,  21,  22,  23,  24,  25,  26,  27,  28,  14,  15,  16,  17,
     29,   9,  10,  11,  12,  13,  30,   4,   5,   6,   7,   8,  31,   0,   1,
      2,   3,
    /* Amplitude information 3 - 32 entries */
      0,   1,   2,  31,   3,  29,  30,   4,   5,   6,  27,  28,   7,  24,  25,
     26,   8,   9,  10,  11,  12,  13,  14,  15,  16,  17,  18,  19,  20,  21,
     22,  23,
    /* Frequencies - 256 entries */
      0,   1, 255,   2, 254,   3,   4,   5,   6,   7,   8, 251, 252, 253,   9,
     10,  11,  12, 246, 247, 248, 249, 250,  13,  14,  15,  16,  17,  18,  19,
     20,  21,  22,  23,  24,  25,  26,  27,  28,  29,  30,  31,  32,  33,  34,
     35,  36,  37,  38, 243, 244, 245,  39,  40,  41,  42,  43,  44,  45,  46,
     47,  48,  49,  50,  51,  52,  53,  54,  55,  56,  57,  58,  59,  60,  61,
     62,  63,  64,  65,  66,  67,  68,  69,  70,  71,  72,  73,  74,  75,  76,
     77,  78,  79,  80,  81,  82,  83,  84,  85,  86,  87,  88,  89,  90,  91,
     92,  93, 194, 195, 196, 197, 198, 199, 200, 201, 202, 203, 204, 205, 206,
    207, 208, 209, 210, 211, 212, 213, 214, 215, 216, 217, 218, 219, 220, 221,
    222, 223, 224, 225, 226, 227, 228, 229, 230, 231, 232, 233, 234, 235, 236,
    237, 238, 239, 240, 241, 242,  94,  95,  96,  97,  98,  99, 100, 101, 102,
    103, 104, 105, 106, 107, 108, 109, 110, 111, 112, 113, 114, 115, 116, 117,
    118, 119, 120, 121, 122, 123, 124, 125, 126, 127, 128, 129, 130, 131, 132,
    133, 134, 135, 136, 137, 138, 139, 140, 141, 142, 143, 144, 145, 146, 147,
    148, 149, 150, 151, 152, 153, 154, 155, 156, 157, 158, 159, 160, 161, 162,
    163, 164, 165, 166, 167, 168, 169, 170, 171, 172, 173, 174, 175, 176, 177,
    178, 179, 180, 181, 182, 183, 184, 185, 186, 187, 188, 189, 190, 191, 192,
    193,
};

#endif /* AVCODEC_ATRAC3PLUS_DATA_H */
