// Copyright 2017 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

import * as Common from '../common/common.js';
import * as i18n from '../i18n/i18n.js';
import * as Platform from '../platform/platform.js';

import {FrameAssociated} from './FrameAssociated.js';  // eslint-disable-line no-unused-vars
import {Events as TargetManagerEvents, Target, TargetManager} from './SDKModel.js';  // eslint-disable-line no-unused-vars
import {SourceMap, TextSourceMap} from './SourceMap.js';  // eslint-disable-line no-unused-vars

export const UIStrings = {
  /**
  *@description Error message when failing to load a source map text
  *@example {An error occurred} PH1
  */
  devtoolsFailedToLoadSourcemapS: 'DevTools failed to load SourceMap: {PH1}',
};
const str_ = i18n.i18n.registerUIStrings('sdk/SourceMapManager.js', UIStrings);
const i18nString = i18n.i18n.getLocalizedString.bind(undefined, str_);

/**
 * @template {!FrameAssociated} T
 */
export class SourceMapManager extends Common.ObjectWrapper.ObjectWrapper {
  /**
   * @param {!Target} target
   */
  constructor(target) {
    super();

    this._target = target;
    /** @type {boolean} */
    this._isEnabled = true;

    /** @type {!Map<!T, string>} */
    this._relativeSourceURL = new Map();
    /** @type {!Map<!T, string>} */
    this._relativeSourceMapURL = new Map();
    /** @type {!Map<!T, string>} */
    this._resolvedSourceMapId = new Map();

    /** @type {!Map<string, !SourceMap>} */
    this._sourceMapById = new Map();
    /** @type {!Platform.MapUtilities.Multimap<string, !T>} */
    this._sourceMapIdToLoadingClients = new Platform.MapUtilities.Multimap();
    /** @type {!Platform.MapUtilities.Multimap<string, !T>} */
    this._sourceMapIdToClients = new Platform.MapUtilities.Multimap();

    TargetManager.instance().addEventListener(TargetManagerEvents.InspectedURLChanged, this._inspectedURLChanged, this);
  }

  /**
   * @param {boolean} isEnabled
   */
  setEnabled(isEnabled) {
    if (isEnabled === this._isEnabled) {
      return;
    }
    this._isEnabled = isEnabled;
    // We need this copy, because `this._resolvedSourceMapId` is getting modified
    // in the loop body and trying to iterate over it at the same time leads to
    // an infinite loop.
    const clients = [...this._resolvedSourceMapId.keys()];
    for (const client of clients) {
      const relativeSourceURL = this._relativeSourceURL.get(client);
      const relativeSourceMapURL = this._relativeSourceMapURL.get(client);
      this.detachSourceMap(client);
      this.attachSourceMap(client, relativeSourceURL, relativeSourceMapURL);
    }
  }

  /**
   * @param {!Common.EventTarget.EventTargetEvent} event
   */
  _inspectedURLChanged(event) {
    if (event.data !== this._target) {
      return;
    }

    // We need this copy, because `this._resolvedSourceMapId` is getting modified
    // in the loop body and trying to iterate over it at the same time leads to
    // an infinite loop.
    const prevSourceMapIds = new Map(this._resolvedSourceMapId);
    for (const [client, prevSourceMapId] of prevSourceMapIds) {
      const relativeSourceURL = this._relativeSourceURL.get(client);
      const relativeSourceMapURL = this._relativeSourceMapURL.get(client);
      if (relativeSourceURL === undefined || relativeSourceMapURL === undefined) {
        continue;
      }
      const resolvedUrls = this._resolveRelativeURLs(relativeSourceURL, relativeSourceMapURL);
      if (resolvedUrls !== null && prevSourceMapId !== resolvedUrls.sourceMapId) {
        this.detachSourceMap(client);
        this.attachSourceMap(client, relativeSourceURL, relativeSourceMapURL);
      }
    }
  }

  /**
   * @param {!T} client
   * @return {?SourceMap}
   */
  sourceMapForClient(client) {
    const sourceMapId = this._resolvedSourceMapId.get(client);
    if (!sourceMapId) {
      return null;
    }
    return this._sourceMapById.get(sourceMapId) || null;
  }

  /**
   * @param {!SourceMap} sourceMap
   * @return {!Array<!T>}
   */
  clientsForSourceMap(sourceMap) {
    const sourceMapId = this._getSourceMapId(sourceMap.compiledURL(), sourceMap.url());
    if (this._sourceMapIdToClients.has(sourceMapId)) {
      return [...this._sourceMapIdToClients.get(sourceMapId)];
    }
    return [...this._sourceMapIdToLoadingClients.get(sourceMapId)];
  }

  /**
   * @param {string} sourceURL
   * @param {string} sourceMapURL
   */
  _getSourceMapId(sourceURL, sourceMapURL) {
    return `${sourceURL}:${sourceMapURL}`;
  }

  /**
   * @param {string} sourceURL
   * @param {string} sourceMapURL
   * @return {?{sourceURL: string, sourceMapURL: string, sourceMapId: string}}
   */
  _resolveRelativeURLs(sourceURL, sourceMapURL) {
    // |sourceURL| can be a random string, but is generally an absolute path.
    // Complete it to inspected page url for relative links.
    const resolvedSourceURL = Common.ParsedURL.ParsedURL.completeURL(this._target.inspectedURL(), sourceURL);
    if (!resolvedSourceURL) {
      return null;
    }
    const resolvedSourceMapURL = Common.ParsedURL.ParsedURL.completeURL(resolvedSourceURL, sourceMapURL);
    if (!resolvedSourceMapURL) {
      return null;
    }
    return {
      sourceURL: resolvedSourceURL,
      sourceMapURL: resolvedSourceMapURL,
      sourceMapId: this._getSourceMapId(resolvedSourceURL, resolvedSourceMapURL)
    };
  }

  /**
   * @param {!T} client
   * @param {string|undefined} relativeSourceURL
   * @param {string|undefined} relativeSourceMapURL
   */
  attachSourceMap(client, relativeSourceURL, relativeSourceMapURL) {
    // TODO(chromium:1011811): Strengthen the type to obsolte the undefined check once sdk/ is fully typescriptified.
    if (relativeSourceURL === undefined || !relativeSourceMapURL) {
      return;
    }
    console.assert(!this._resolvedSourceMapId.has(client), 'SourceMap is already attached to client');
    const resolvedURLs = this._resolveRelativeURLs(relativeSourceURL, relativeSourceMapURL);
    if (!resolvedURLs) {
      return;
    }
    this._relativeSourceURL.set(client, relativeSourceURL);
    this._relativeSourceMapURL.set(client, relativeSourceMapURL);

    const {sourceURL, sourceMapURL, sourceMapId} = resolvedURLs;
    this._resolvedSourceMapId.set(client, sourceMapId);

    if (!this._isEnabled) {
      return;
    }

    this.dispatchEventToListeners(Events.SourceMapWillAttach, client);

    if (this._sourceMapById.has(sourceMapId)) {
      attach.call(this, sourceMapId, client);
      return;
    }
    if (!this._sourceMapIdToLoadingClients.has(sourceMapId)) {
      TextSourceMap.load(sourceMapURL, sourceURL, client.createPageResourceLoadInitiator())
          .catch(error => {
            Common.Console.Console.instance().warn(
                i18nString(UIStrings.devtoolsFailedToLoadSourcemapS, {PH1: error.message}));
            return null;
          })
          .then(onSourceMap.bind(this, sourceMapId));
    }
    this._sourceMapIdToLoadingClients.set(sourceMapId, client);

    /**
     * @param {string} sourceMapId
     * @param {?SourceMap} sourceMap
     * @this {SourceMapManager<T>}
     */
    function onSourceMap(sourceMapId, sourceMap) {
      this._sourceMapLoadedForTest();
      const clients = this._sourceMapIdToLoadingClients.get(sourceMapId);
      this._sourceMapIdToLoadingClients.deleteAll(sourceMapId);
      if (!clients.size) {
        return;
      }
      if (!sourceMap) {
        for (const client of clients) {
          this.dispatchEventToListeners(Events.SourceMapFailedToAttach, client);
        }
        return;
      }
      this._sourceMapById.set(sourceMapId, sourceMap);
      for (const client of clients) {
        attach.call(this, sourceMapId, client);
      }
    }

    /**
     * @param {string} sourceMapId
     * @param {!T} client
     * @this {SourceMapManager<T>}
     */
    function attach(sourceMapId, client) {
      this._sourceMapIdToClients.set(sourceMapId, client);
      const sourceMap = this._sourceMapById.get(sourceMapId);
      this.dispatchEventToListeners(Events.SourceMapAttached, {client: client, sourceMap: sourceMap});
    }
  }

  /**
   * @param {!T} client
   */
  detachSourceMap(client) {
    const sourceMapId = this._resolvedSourceMapId.get(client);
    this._relativeSourceURL.delete(client);
    this._relativeSourceMapURL.delete(client);
    this._resolvedSourceMapId.delete(client);

    if (!sourceMapId) {
      return;
    }
    if (!this._sourceMapIdToClients.hasValue(sourceMapId, client)) {
      if (this._sourceMapIdToLoadingClients.delete(sourceMapId, client)) {
        this.dispatchEventToListeners(Events.SourceMapFailedToAttach, client);
      }
      return;
    }
    this._sourceMapIdToClients.delete(sourceMapId, client);
    const sourceMap = this._sourceMapById.get(sourceMapId);
    if (!sourceMap) {
      return;
    }
    if (!this._sourceMapIdToClients.has(sourceMapId)) {
      this._sourceMapById.delete(sourceMapId);
    }
    this.dispatchEventToListeners(Events.SourceMapDetached, {client: client, sourceMap: sourceMap});
  }

  _sourceMapLoadedForTest() {
  }

  dispose() {
    TargetManager.instance().removeEventListener(
        TargetManagerEvents.InspectedURLChanged, this._inspectedURLChanged, this);
  }
}

export const Events = {
  SourceMapWillAttach: Symbol('SourceMapWillAttach'),
  SourceMapFailedToAttach: Symbol('SourceMapFailedToAttach'),
  SourceMapAttached: Symbol('SourceMapAttached'),
  SourceMapDetached: Symbol('SourceMapDetached'),
  SourceMapChanged: Symbol('SourceMapChanged')
};
