// Copyright 2017 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

import * as Common from '../common/common.js';
import * as Root from '../root/root.js';

import {Attributes, Cookie} from './Cookie.js';  // eslint-disable-line no-unused-vars
import {Resource} from './Resource.js';          // eslint-disable-line no-unused-vars
import {ResourceTreeModel} from './ResourceTreeModel.js';
import {Capability, SDKModel, Target} from './SDKModel.js';  // eslint-disable-line no-unused-vars

export class CookieModel extends SDKModel {
  /**
   * @param {!Target} target
   */
  constructor(target) {
    super(target);

    /** Array<!Cookie> */
    this._blockedCookies = new Map();
    this._cookieToBlockedReasons = new Map();
  }

  /**
   * @param {!Cookie} cookie
   * @param {?Array<!BlockedReason>} blockedReasons
   */
  addBlockedCookie(cookie, blockedReasons) {
    const key = cookie.key();
    const previousCookie = this._blockedCookies.get(key);
    this._blockedCookies.set(key, cookie);
    this._cookieToBlockedReasons.set(cookie, blockedReasons);
    if (previousCookie) {
      this._cookieToBlockedReasons.delete(key);
    }
  }

  getCookieToBlockedReasonsMap() {
    return this._cookieToBlockedReasons;
  }

  /**
   * @param {!Array<string>} urls
   * @return {!Promise<!Array<!Cookie>>}
   */
  async getCookies(urls) {
    const response = await this.target().networkAgent().invoke_getCookies({urls});
    if (response.getError()) {
      return [];
    }
    const normalCookies = response.cookies.map(Cookie.fromProtocolCookie);
    return normalCookies.concat(Array.from(this._blockedCookies.values()));
  }

  /**
   * @param {!Cookie} cookie
   * @return {!Promise<void>}
   */
  async deleteCookie(cookie) {
    await this.deleteCookies([cookie]);
  }

  /**
   * @param {string=} domain
   * @param {string=} securityOrigin
   * @return {!Promise<void>}
   */
  async clear(domain, securityOrigin) {
    const cookies = await this.getCookiesForDomain(domain || null);
    if (securityOrigin) {
      const cookiesToDelete = cookies.filter(cookie => {
        return cookie.matchesSecurityOrigin(securityOrigin);
      });
      await this.deleteCookies(cookiesToDelete);
    } else {
      await this.deleteCookies(cookies);
    }
  }

  /**
   * @param {!Cookie} cookie
   * @return {!Promise<boolean>}
   */
  async saveCookie(cookie) {
    let domain = cookie.domain();
    if (!domain.startsWith('.')) {
      domain = '';
    }
    let expires = undefined;
    if (cookie.expires()) {
      expires = Math.floor(Date.parse(`${cookie.expires()}`) / 1000);
    }
    const enabled = Root.Runtime.experiments.isEnabled('experimentalCookieFeatures');
    /** @param {Protocol.Network.CookieSourceScheme} scheme */
    const preserveUnset = scheme => scheme === Protocol.Network.CookieSourceScheme.Unset ? scheme : undefined;
    const protocolCookie = {
      name: cookie.name(),
      value: cookie.value(),
      url: cookie.url() || undefined,
      domain,
      path: cookie.path(),
      secure: cookie.secure(),
      httpOnly: cookie.httpOnly(),
      sameSite: cookie.sameSite(),
      expires,
      priority: cookie.priority(),
      sameParty: cookie.sameParty(),
      sourceScheme: enabled ? cookie.sourceScheme() : preserveUnset(cookie.sourceScheme()),
      sourcePort: enabled ? cookie.sourcePort() : undefined,
    };
    const response = await this.target().networkAgent().invoke_setCookie(protocolCookie);
    const error = response.getError();
    if (error || !response.success) {
      return false;
    }
    return response.success;
  }

  /**
   * Returns cookies needed by current page's frames whose security origins are |domain|.
   * @param {?string} domain
   * @return {!Promise<!Array<!Cookie>>}
   */
  getCookiesForDomain(domain) {
    const resourceURLs = [];
    /**
     * @param {!Resource} resource
     * @return {boolean}
     */
    function populateResourceURLs(resource) {
      const documentURL = Common.ParsedURL.ParsedURL.fromString(resource.documentURL);
      if (documentURL && (!domain || documentURL.securityOrigin() === domain)) {
        resourceURLs.push(resource.url);
      }
      return false;
    }
    const resourceTreeModel = this.target().model(ResourceTreeModel);
    if (resourceTreeModel) {
      // In case the current frame was unreachable, add it's cookies
      // because they might help to debug why the frame was unreachable.
      if (resourceTreeModel.mainFrame && resourceTreeModel.mainFrame.unreachableUrl()) {
        resourceURLs.push(resourceTreeModel.mainFrame.unreachableUrl());
      }

      resourceTreeModel.forAllResources(populateResourceURLs);
    }
    return this.getCookies(resourceURLs);
  }

  /**
   * @param {!Array<!Cookie>} cookies
   * @return {!Promise<void>}
   */
  async deleteCookies(cookies) {
    const networkAgent = this.target().networkAgent();
    this._blockedCookies.clear();
    this._cookieToBlockedReasons.clear();
    await Promise.all(cookies.map(
        cookie => networkAgent.invoke_deleteCookies(
            {name: cookie.name(), url: undefined, domain: cookie.domain(), path: cookie.path()})));
  }
}

SDKModel.register(CookieModel, Capability.Network, false);

/** @typedef {!{uiString: string, attribute: ?Attributes}} */
// @ts-ignore typedef
export let BlockedReason;
