// Copyright 2020 The Dawn Authors
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

#include "TextureFormatUtils.h"

namespace utils {
    bool TextureFormatSupportsStorageTexture(wgpu::TextureFormat format) {
        switch (format) {
            case wgpu::TextureFormat::R32Uint:
            case wgpu::TextureFormat::R32Sint:
            case wgpu::TextureFormat::R32Float:
            case wgpu::TextureFormat::RGBA8Unorm:
            case wgpu::TextureFormat::RGBA8Snorm:
            case wgpu::TextureFormat::RGBA8Uint:
            case wgpu::TextureFormat::RGBA8Sint:
            case wgpu::TextureFormat::RG32Uint:
            case wgpu::TextureFormat::RG32Sint:
            case wgpu::TextureFormat::RG32Float:
            case wgpu::TextureFormat::RGBA16Uint:
            case wgpu::TextureFormat::RGBA16Sint:
            case wgpu::TextureFormat::RGBA16Float:
            case wgpu::TextureFormat::RGBA32Uint:
            case wgpu::TextureFormat::RGBA32Sint:
            case wgpu::TextureFormat::RGBA32Float:
                return true;

            default:
                return false;
        }
    }

    uint32_t GetTexelBlockSizeInBytes(wgpu::TextureFormat textureFormat) {
        switch (textureFormat) {
            case wgpu::TextureFormat::R8Unorm:
            case wgpu::TextureFormat::R8Snorm:
            case wgpu::TextureFormat::R8Uint:
            case wgpu::TextureFormat::R8Sint:
                return 1u;

            case wgpu::TextureFormat::R16Uint:
            case wgpu::TextureFormat::R16Sint:
            case wgpu::TextureFormat::R16Float:
            case wgpu::TextureFormat::RG8Unorm:
            case wgpu::TextureFormat::RG8Snorm:
            case wgpu::TextureFormat::RG8Uint:
            case wgpu::TextureFormat::RG8Sint:
                return 2u;

            case wgpu::TextureFormat::R32Float:
            case wgpu::TextureFormat::R32Uint:
            case wgpu::TextureFormat::R32Sint:
            case wgpu::TextureFormat::RG16Uint:
            case wgpu::TextureFormat::RG16Sint:
            case wgpu::TextureFormat::RG16Float:
            case wgpu::TextureFormat::RGBA8Unorm:
            case wgpu::TextureFormat::RGBA8UnormSrgb:
            case wgpu::TextureFormat::RGBA8Snorm:
            case wgpu::TextureFormat::RGBA8Uint:
            case wgpu::TextureFormat::RGBA8Sint:
            case wgpu::TextureFormat::BGRA8Unorm:
            case wgpu::TextureFormat::BGRA8UnormSrgb:
            case wgpu::TextureFormat::RGB10A2Unorm:
            case wgpu::TextureFormat::RG11B10Ufloat:
            case wgpu::TextureFormat::RGB9E5Ufloat:
            case wgpu::TextureFormat::Depth32Float:
                return 4u;

            case wgpu::TextureFormat::RG32Float:
            case wgpu::TextureFormat::RG32Uint:
            case wgpu::TextureFormat::RG32Sint:
            case wgpu::TextureFormat::RGBA16Uint:
            case wgpu::TextureFormat::RGBA16Sint:
            case wgpu::TextureFormat::RGBA16Float:
                return 8u;

            case wgpu::TextureFormat::RGBA32Float:
            case wgpu::TextureFormat::RGBA32Uint:
            case wgpu::TextureFormat::RGBA32Sint:
                return 16u;

            case wgpu::TextureFormat::BC1RGBAUnorm:
            case wgpu::TextureFormat::BC1RGBAUnormSrgb:
            case wgpu::TextureFormat::BC4RUnorm:
            case wgpu::TextureFormat::BC4RSnorm:
                return 8u;

            case wgpu::TextureFormat::BC2RGBAUnorm:
            case wgpu::TextureFormat::BC2RGBAUnormSrgb:
            case wgpu::TextureFormat::BC3RGBAUnorm:
            case wgpu::TextureFormat::BC3RGBAUnormSrgb:
            case wgpu::TextureFormat::BC5RGUnorm:
            case wgpu::TextureFormat::BC5RGSnorm:
            case wgpu::TextureFormat::BC6HRGBUfloat:
            case wgpu::TextureFormat::BC6HRGBFloat:
            case wgpu::TextureFormat::BC7RGBAUnorm:
            case wgpu::TextureFormat::BC7RGBAUnormSrgb:
                return 16u;

            case wgpu::TextureFormat::Depth24Plus:
            case wgpu::TextureFormat::Depth24PlusStencil8:

            // Block size of a multi-planar format depends on aspect.
            case wgpu::TextureFormat::R8BG8Biplanar420Unorm:

            case wgpu::TextureFormat::Stencil8:
            case wgpu::TextureFormat::Undefined:
                UNREACHABLE();
        }
    }

    uint32_t GetTextureFormatBlockWidth(wgpu::TextureFormat textureFormat) {
        switch (textureFormat) {
            case wgpu::TextureFormat::R8Unorm:
            case wgpu::TextureFormat::R8Snorm:
            case wgpu::TextureFormat::R8Uint:
            case wgpu::TextureFormat::R8Sint:
            case wgpu::TextureFormat::R16Uint:
            case wgpu::TextureFormat::R16Sint:
            case wgpu::TextureFormat::R16Float:
            case wgpu::TextureFormat::RG8Unorm:
            case wgpu::TextureFormat::RG8Snorm:
            case wgpu::TextureFormat::RG8Uint:
            case wgpu::TextureFormat::RG8Sint:
            case wgpu::TextureFormat::R32Float:
            case wgpu::TextureFormat::R32Uint:
            case wgpu::TextureFormat::R32Sint:
            case wgpu::TextureFormat::RG16Uint:
            case wgpu::TextureFormat::RG16Sint:
            case wgpu::TextureFormat::RG16Float:
            case wgpu::TextureFormat::RGBA8Unorm:
            case wgpu::TextureFormat::RGBA8UnormSrgb:
            case wgpu::TextureFormat::RGBA8Snorm:
            case wgpu::TextureFormat::RGBA8Uint:
            case wgpu::TextureFormat::RGBA8Sint:
            case wgpu::TextureFormat::BGRA8Unorm:
            case wgpu::TextureFormat::BGRA8UnormSrgb:
            case wgpu::TextureFormat::RGB10A2Unorm:
            case wgpu::TextureFormat::RG11B10Ufloat:
            case wgpu::TextureFormat::RGB9E5Ufloat:
            case wgpu::TextureFormat::RG32Float:
            case wgpu::TextureFormat::RG32Uint:
            case wgpu::TextureFormat::RG32Sint:
            case wgpu::TextureFormat::RGBA16Uint:
            case wgpu::TextureFormat::RGBA16Sint:
            case wgpu::TextureFormat::RGBA16Float:
            case wgpu::TextureFormat::RGBA32Float:
            case wgpu::TextureFormat::RGBA32Uint:
            case wgpu::TextureFormat::RGBA32Sint:
            case wgpu::TextureFormat::Depth32Float:
            case wgpu::TextureFormat::Depth24Plus:
            case wgpu::TextureFormat::Depth24PlusStencil8:
                return 1u;

            case wgpu::TextureFormat::BC1RGBAUnorm:
            case wgpu::TextureFormat::BC1RGBAUnormSrgb:
            case wgpu::TextureFormat::BC4RUnorm:
            case wgpu::TextureFormat::BC4RSnorm:
            case wgpu::TextureFormat::BC2RGBAUnorm:
            case wgpu::TextureFormat::BC2RGBAUnormSrgb:
            case wgpu::TextureFormat::BC3RGBAUnorm:
            case wgpu::TextureFormat::BC3RGBAUnormSrgb:
            case wgpu::TextureFormat::BC5RGUnorm:
            case wgpu::TextureFormat::BC5RGSnorm:
            case wgpu::TextureFormat::BC6HRGBUfloat:
            case wgpu::TextureFormat::BC6HRGBFloat:
            case wgpu::TextureFormat::BC7RGBAUnorm:
            case wgpu::TextureFormat::BC7RGBAUnormSrgb:
                return 4u;

            // Block size of a multi-planar format depends on aspect.
            case wgpu::TextureFormat::R8BG8Biplanar420Unorm:

            case wgpu::TextureFormat::Stencil8:
            case wgpu::TextureFormat::Undefined:
                UNREACHABLE();
        }
    }

    uint32_t GetTextureFormatBlockHeight(wgpu::TextureFormat textureFormat) {
        switch (textureFormat) {
            case wgpu::TextureFormat::R8Unorm:
            case wgpu::TextureFormat::R8Snorm:
            case wgpu::TextureFormat::R8Uint:
            case wgpu::TextureFormat::R8Sint:
            case wgpu::TextureFormat::R16Uint:
            case wgpu::TextureFormat::R16Sint:
            case wgpu::TextureFormat::R16Float:
            case wgpu::TextureFormat::RG8Unorm:
            case wgpu::TextureFormat::RG8Snorm:
            case wgpu::TextureFormat::RG8Uint:
            case wgpu::TextureFormat::RG8Sint:
            case wgpu::TextureFormat::R32Float:
            case wgpu::TextureFormat::R32Uint:
            case wgpu::TextureFormat::R32Sint:
            case wgpu::TextureFormat::RG16Uint:
            case wgpu::TextureFormat::RG16Sint:
            case wgpu::TextureFormat::RG16Float:
            case wgpu::TextureFormat::RGBA8Unorm:
            case wgpu::TextureFormat::RGBA8UnormSrgb:
            case wgpu::TextureFormat::RGBA8Snorm:
            case wgpu::TextureFormat::RGBA8Uint:
            case wgpu::TextureFormat::RGBA8Sint:
            case wgpu::TextureFormat::BGRA8Unorm:
            case wgpu::TextureFormat::BGRA8UnormSrgb:
            case wgpu::TextureFormat::RGB10A2Unorm:
            case wgpu::TextureFormat::RG11B10Ufloat:
            case wgpu::TextureFormat::RGB9E5Ufloat:
            case wgpu::TextureFormat::RG32Float:
            case wgpu::TextureFormat::RG32Uint:
            case wgpu::TextureFormat::RG32Sint:
            case wgpu::TextureFormat::RGBA16Uint:
            case wgpu::TextureFormat::RGBA16Sint:
            case wgpu::TextureFormat::RGBA16Float:
            case wgpu::TextureFormat::RGBA32Float:
            case wgpu::TextureFormat::RGBA32Uint:
            case wgpu::TextureFormat::RGBA32Sint:
            case wgpu::TextureFormat::Depth32Float:
            case wgpu::TextureFormat::Depth24Plus:
            case wgpu::TextureFormat::Depth24PlusStencil8:
                return 1u;

            case wgpu::TextureFormat::BC1RGBAUnorm:
            case wgpu::TextureFormat::BC1RGBAUnormSrgb:
            case wgpu::TextureFormat::BC4RUnorm:
            case wgpu::TextureFormat::BC4RSnorm:
            case wgpu::TextureFormat::BC2RGBAUnorm:
            case wgpu::TextureFormat::BC2RGBAUnormSrgb:
            case wgpu::TextureFormat::BC3RGBAUnorm:
            case wgpu::TextureFormat::BC3RGBAUnormSrgb:
            case wgpu::TextureFormat::BC5RGUnorm:
            case wgpu::TextureFormat::BC5RGSnorm:
            case wgpu::TextureFormat::BC6HRGBUfloat:
            case wgpu::TextureFormat::BC6HRGBFloat:
            case wgpu::TextureFormat::BC7RGBAUnorm:
            case wgpu::TextureFormat::BC7RGBAUnormSrgb:
                return 4u;

            // Block size of a multi-planar format depends on aspect.
            case wgpu::TextureFormat::R8BG8Biplanar420Unorm:

            case wgpu::TextureFormat::Stencil8:
            case wgpu::TextureFormat::Undefined:
                UNREACHABLE();
        }
    }

    const char* GetWGSLColorTextureComponentType(wgpu::TextureFormat textureFormat) {
        switch (textureFormat) {
            case wgpu::TextureFormat::R8Unorm:
            case wgpu::TextureFormat::R8Snorm:
            case wgpu::TextureFormat::R16Float:
            case wgpu::TextureFormat::RG8Unorm:
            case wgpu::TextureFormat::RG8Snorm:
            case wgpu::TextureFormat::R32Float:
            case wgpu::TextureFormat::RG16Float:
            case wgpu::TextureFormat::RGBA8Unorm:
            case wgpu::TextureFormat::RGBA8Snorm:
            case wgpu::TextureFormat::RGB10A2Unorm:
            case wgpu::TextureFormat::RG11B10Ufloat:
            case wgpu::TextureFormat::RGB9E5Ufloat:
            case wgpu::TextureFormat::RG32Float:
            case wgpu::TextureFormat::RGBA16Float:
            case wgpu::TextureFormat::RGBA32Float:
            case wgpu::TextureFormat::BGRA8Unorm:
            case wgpu::TextureFormat::BGRA8UnormSrgb:
            case wgpu::TextureFormat::RGBA8UnormSrgb:
                return "f32";

            case wgpu::TextureFormat::R8Uint:
            case wgpu::TextureFormat::R16Uint:
            case wgpu::TextureFormat::RG8Uint:
            case wgpu::TextureFormat::R32Uint:
            case wgpu::TextureFormat::RG16Uint:
            case wgpu::TextureFormat::RGBA8Uint:
            case wgpu::TextureFormat::RG32Uint:
            case wgpu::TextureFormat::RGBA16Uint:
            case wgpu::TextureFormat::RGBA32Uint:
                return "u32";

            case wgpu::TextureFormat::R8Sint:
            case wgpu::TextureFormat::R16Sint:
            case wgpu::TextureFormat::RG8Sint:
            case wgpu::TextureFormat::R32Sint:
            case wgpu::TextureFormat::RG16Sint:
            case wgpu::TextureFormat::RGBA8Sint:
            case wgpu::TextureFormat::RG32Sint:
            case wgpu::TextureFormat::RGBA16Sint:
            case wgpu::TextureFormat::RGBA32Sint:
                return "i32";

            default:
                UNREACHABLE();
        }
    }

    const char* GetWGSLImageFormatQualifier(wgpu::TextureFormat textureFormat) {
        switch (textureFormat) {
            case wgpu::TextureFormat::RGBA8Unorm:
                return "rgba8unorm";
            case wgpu::TextureFormat::RGBA8Snorm:
                return "rgba8snorm";
            case wgpu::TextureFormat::RGBA8Uint:
                return "rgba8uint";
            case wgpu::TextureFormat::RGBA8Sint:
                return "rgba8sint";
            case wgpu::TextureFormat::RGBA16Uint:
                return "rgba16uint";
            case wgpu::TextureFormat::RGBA16Sint:
                return "rgba16sint";
            case wgpu::TextureFormat::RGBA16Float:
                return "rgba16float";
            case wgpu::TextureFormat::R32Uint:
                return "r32uint";
            case wgpu::TextureFormat::R32Sint:
                return "r32sint";
            case wgpu::TextureFormat::R32Float:
                return "r32float";
            case wgpu::TextureFormat::RG32Uint:
                return "rg32uint";
            case wgpu::TextureFormat::RG32Sint:
                return "rg32sint";
            case wgpu::TextureFormat::RG32Float:
                return "rg32float";
            case wgpu::TextureFormat::RGBA32Uint:
                return "rgba32uint";
            case wgpu::TextureFormat::RGBA32Sint:
                return "rgba32sint";
            case wgpu::TextureFormat::RGBA32Float:
                return "rgba32float";

            // The below do not currently exist in the WGSL spec, but are used
            // for tests that expect compilation failure.
            case wgpu::TextureFormat::R8Unorm:
                return "r8unorm";
            case wgpu::TextureFormat::R8Snorm:
                return "r8snorm";
            case wgpu::TextureFormat::R8Uint:
                return "r8uint";
            case wgpu::TextureFormat::R8Sint:
                return "r8sint";
            case wgpu::TextureFormat::R16Uint:
                return "r16uint";
            case wgpu::TextureFormat::R16Sint:
                return "r16sint";
            case wgpu::TextureFormat::R16Float:
                return "r16float";
            case wgpu::TextureFormat::RG8Unorm:
                return "rg8unorm";
            case wgpu::TextureFormat::RG8Snorm:
                return "rg8snorm";
            case wgpu::TextureFormat::RG8Uint:
                return "rg8uint";
            case wgpu::TextureFormat::RG8Sint:
                return "rg8sint";
            case wgpu::TextureFormat::RG16Uint:
                return "rg16uint";
            case wgpu::TextureFormat::RG16Sint:
                return "rg16sint";
            case wgpu::TextureFormat::RG16Float:
                return "rg16float";
            case wgpu::TextureFormat::RGB10A2Unorm:
                return "rgb10a2unorm";
            case wgpu::TextureFormat::RG11B10Ufloat:
                return "rg11b10ufloat";

            default:
                UNREACHABLE();
        }
    }

}  // namespace utils
