// Copyright 2015 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "third_party/blink/renderer/modules/storage/dom_window_storage.h"

#include "base/feature_list.h"
#include "base/memory/scoped_refptr.h"
#include "services/network/public/mojom/web_sandbox_flags.mojom-blink.h"
#include "third_party/blink/public/common/features.h"
#include "third_party/blink/renderer/core/frame/local_dom_window.h"
#include "third_party/blink/renderer/core/frame/local_frame.h"
#include "third_party/blink/renderer/core/frame/settings.h"
#include "third_party/blink/renderer/core/page/page.h"
#include "third_party/blink/renderer/modules/storage/storage_area.h"
#include "third_party/blink/renderer/modules/storage/storage_controller.h"
#include "third_party/blink/renderer/modules/storage/storage_namespace.h"
#include "third_party/blink/renderer/platform/instrumentation/use_counter.h"
#include "third_party/blink/renderer/platform/weborigin/security_origin.h"

namespace blink {

DOMWindowStorage::DOMWindowStorage(LocalDOMWindow& window)
    : Supplement<LocalDOMWindow>(window) {}

void DOMWindowStorage::Trace(Visitor* visitor) const {
  visitor->Trace(session_storage_);
  visitor->Trace(local_storage_);
  Supplement<LocalDOMWindow>::Trace(visitor);
}

// static
const char DOMWindowStorage::kSupplementName[] = "DOMWindowStorage";

// static
DOMWindowStorage& DOMWindowStorage::From(LocalDOMWindow& window) {
  DOMWindowStorage* supplement =
      Supplement<LocalDOMWindow>::From<DOMWindowStorage>(window);
  if (!supplement) {
    supplement = MakeGarbageCollected<DOMWindowStorage>(window);
    ProvideTo(window, supplement);
  }
  return *supplement;
}

// static
StorageArea* DOMWindowStorage::sessionStorage(LocalDOMWindow& window,
                                              ExceptionState& exception_state) {
  return From(window).sessionStorage(exception_state);
}

// static
StorageArea* DOMWindowStorage::localStorage(LocalDOMWindow& window,
                                            ExceptionState& exception_state) {
  return From(window).localStorage(exception_state);
}

StorageArea* DOMWindowStorage::sessionStorage(
    ExceptionState& exception_state) const {
  LocalDOMWindow* window = GetSupplementable();
  if (!window->GetFrame())
    return nullptr;

  String access_denied_message = "Access is denied for this document.";
  if (!window->GetSecurityOrigin()->CanAccessSessionStorage()) {
    if (window->IsSandboxed(network::mojom::blink::WebSandboxFlags::kOrigin))
      exception_state.ThrowSecurityError(
          "The document is sandboxed and lacks the 'allow-same-origin' flag.");
    else if (window->Url().ProtocolIs("data"))
      exception_state.ThrowSecurityError(
          "Storage is disabled inside 'data:' URLs.");
    else
      exception_state.ThrowSecurityError(access_denied_message);
    return nullptr;
  }

  if (window->GetSecurityOrigin()->IsLocal()) {
    UseCounter::Count(window, WebFeature::kFileAccessedSessionStorage);
  }

  if (session_storage_) {
    if (!session_storage_->CanAccessStorage()) {
      exception_state.ThrowSecurityError(access_denied_message);
      return nullptr;
    }
    return session_storage_;
  }

  StorageNamespace* storage_namespace =
      StorageNamespace::From(window->GetFrame()->GetPage());
  if (!storage_namespace)
    return nullptr;
  auto storage_area =
      storage_namespace->GetCachedArea(window->GetSecurityOrigin());
  session_storage_ =
      StorageArea::Create(window, std::move(storage_area),
                          StorageArea::StorageType::kSessionStorage);

  if (!session_storage_->CanAccessStorage()) {
    exception_state.ThrowSecurityError(access_denied_message);
    return nullptr;
  }
  return session_storage_;
}

StorageArea* DOMWindowStorage::localStorage(
    ExceptionState& exception_state) const {
  LocalDOMWindow* window = GetSupplementable();
  if (!window->GetFrame())
    return nullptr;

  String access_denied_message = "Access is denied for this document.";
  if (!window->GetSecurityOrigin()->CanAccessLocalStorage()) {
    if (window->IsSandboxed(network::mojom::blink::WebSandboxFlags::kOrigin))
      exception_state.ThrowSecurityError(
          "The document is sandboxed and lacks the 'allow-same-origin' flag.");
    else if (window->Url().ProtocolIs("data"))
      exception_state.ThrowSecurityError(
          "Storage is disabled inside 'data:' URLs.");
    else
      exception_state.ThrowSecurityError(access_denied_message);
    return nullptr;
  }

  if (window->GetSecurityOrigin()->IsLocal()) {
    UseCounter::Count(window, WebFeature::kFileAccessedLocalStorage);
  }

  if (local_storage_) {
    if (!local_storage_->CanAccessStorage()) {
      exception_state.ThrowSecurityError(access_denied_message);
      return nullptr;
    }
    return local_storage_;
  }
  if (!window->GetFrame()->GetSettings()->GetLocalStorageEnabled())
    return nullptr;
  auto storage_area = StorageController::GetInstance()->GetLocalStorageArea(
      window->GetSecurityOrigin());
  local_storage_ = StorageArea::Create(window, std::move(storage_area),
                                       StorageArea::StorageType::kLocalStorage);

  if (!local_storage_->CanAccessStorage()) {
    exception_state.ThrowSecurityError(access_denied_message);
    return nullptr;
  }
  return local_storage_;
}

}  // namespace blink
