// Copyright 2015 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef THIRD_PARTY_BLINK_RENDERER_PLATFORM_LOADER_FETCH_CACHED_METADATA_HANDLER_H_
#define THIRD_PARTY_BLINK_RENDERER_PLATFORM_LOADER_FETCH_CACHED_METADATA_HANDLER_H_

#include <stdint.h>
#include "third_party/blink/public/mojom/fetch/fetch_api_request.mojom-blink.h"
#include "third_party/blink/public/mojom/loader/code_cache.mojom-blink-forward.h"
#include "third_party/blink/renderer/platform/heap/handle.h"
#include "third_party/blink/renderer/platform/platform_export.h"
#include "third_party/blink/renderer/platform/weborigin/security_origin.h"
#include "third_party/blink/renderer/platform/wtf/allocator/allocator.h"
#include "third_party/blink/renderer/platform/wtf/forward.h"

namespace blink {

class CachedMetadata;
class ResourceResponse;
class WebProcessMemoryDump;

// A callback for sending the serialized data of cached metadata to the
// persistent storage.
// TODO(pasko): rename this class to CachedMetadataPersister.
class PLATFORM_EXPORT CachedMetadataSender {
  USING_FAST_MALLOC(CachedMetadataSender);

 public:
  static std::unique_ptr<CachedMetadataSender> Create(
      const ResourceResponse&,
      blink::mojom::CodeCacheType,
      scoped_refptr<const SecurityOrigin> requestor_origin);

  virtual ~CachedMetadataSender() = default;
  virtual void Send(const uint8_t*, size_t) = 0;

  // IsServedFromCacheStorage is used to alter caching strategy to be more
  // aggressive. See V8CodeCache::GetCompileOptions() for an example.
  virtual bool IsServedFromCacheStorage() = 0;
};

// Returns whether we should use isolated code cache for a particular response.
PLATFORM_EXPORT bool ShouldUseIsolatedCodeCache(
    mojom::blink::RequestContextType,
    const ResourceResponse&);

// Handler class for caching operations.
class CachedMetadataHandler : public GarbageCollected<CachedMetadataHandler> {
 public:
  enum ClearCacheType {
    // Clears the in-memory cache, but doesn't update persistent storage. The
    // old cached metadata is considered invalid.
    kClearLocally,

    // Discards the in-memory cache for memory reduction, preventing any further
    // uses or updates. The cached metadata will no longer be available, but
    // should not be considered invalid.
    kDiscardLocally,

    // Clears the metadata in both memory and persistent storage via
    // blink::Platform::CacheMetadata.
    kClearPersistentStorage
  };

  // Enum for marking serialized cached metadatas so that the deserializers
  // do not conflict.
  enum CachedMetadataType : uint32_t {
    kSingleEntry,    // the metadata is a single CachedMetadata entry
    kSourceKeyedMap  // the metadata is multiple CachedMetadata entries keyed by
                     // a source string.
  };

  virtual ~CachedMetadataHandler() = default;
  virtual void Trace(Visitor* visitor) const {}

  // Reset existing metadata. Subclasses can ignore setting new metadata after
  // clearing with |kDiscardLocally| to save memory.
  virtual void ClearCachedMetadata(ClearCacheType) = 0;

  // Returns the encoding to which the cache is specific.
  virtual String Encoding() const = 0;

  virtual bool IsServedFromCacheStorage() const = 0;

  // Dump cache size kept in memory.
  virtual void OnMemoryDump(WebProcessMemoryDump* pmd,
                            const String& dump_prefix) const = 0;

  virtual size_t GetCodeCacheSize() const = 0;

 protected:
  CachedMetadataHandler() = default;
};

// A CachedMetadataHandler which stores one piece of metadata.
class SingleCachedMetadataHandler : public CachedMetadataHandler {
 public:
  // Caches the given metadata in association with this resource and suggests
  // that the platform persist it. The dataTypeID is a pseudo-randomly chosen
  // identifier that is used to distinguish data generated by the caller.
  virtual void SetCachedMetadata(uint32_t data_type_id,
                                 const uint8_t*,
                                 size_t) = 0;

  // Permanently disable persisting CachedMetadata in the platform only when it
  // is set.
  void DisableSendToPlatformForTesting() {
    disable_send_to_platform_for_testing_ = true;
  }

  // Returns cached metadata of the given type associated with this resource.
  // This cached metadata can be pruned at any time.
  virtual scoped_refptr<CachedMetadata> GetCachedMetadata(
      uint32_t data_type_id) const = 0;

 protected:
  SingleCachedMetadataHandler() = default;
  bool disable_send_to_platform_for_testing_ = false;
};

}  // namespace blink

#endif  // THIRD_PARTY_BLINK_RENDERER_PLATFORM_LOADER_FETCH_CACHED_METADATA_HANDLER_H_
