// Copyright 2015 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef THIRD_PARTY_BLINK_RENDERER_MODULES_QUOTA_STORAGE_MANAGER_H_
#define THIRD_PARTY_BLINK_RENDERER_MODULES_QUOTA_STORAGE_MANAGER_H_

#include "mojo/public/cpp/bindings/remote.h"
#include "third_party/blink/public/mojom/permissions/permission.mojom-blink.h"
#include "third_party/blink/public/mojom/quota/quota_manager_host.mojom-blink.h"
#include "third_party/blink/renderer/core/dom/events/event_target.h"
#include "third_party/blink/renderer/core/execution_context/execution_context.h"
#include "third_party/blink/renderer/platform/bindings/script_wrappable.h"
#include "third_party/blink/renderer/platform/heap/heap.h"
#include "third_party/blink/renderer/platform/mojo/heap_mojo_receiver.h"
#include "third_party/blink/renderer/platform/mojo/heap_mojo_remote.h"
#include "third_party/blink/renderer/platform/mojo/heap_mojo_wrapper_mode.h"

namespace blink {

class ExecutionContext;
class ScriptPromise;
class ScriptPromiseResolver;
class ScriptState;

class StorageManager final : public EventTargetWithInlineData,
                             public ExecutionContextClient,
                             public mojom::blink::QuotaChangeListener {
  DEFINE_WRAPPERTYPEINFO();

 public:
  explicit StorageManager(ExecutionContext*);
  ~StorageManager() override;

  ScriptPromise persisted(ScriptState*);
  ScriptPromise persist(ScriptState*);

  ScriptPromise estimate(ScriptState*);

  void Trace(Visitor* visitor) const override;

  // EventTargetWithInlineData
  DEFINE_ATTRIBUTE_EVENT_LISTENER(quotachange, kQuotachange)
  const AtomicString& InterfaceName() const override;
  ExecutionContext* GetExecutionContext() const override;

  // network::mojom::blink::QuotaChangeListener
  void OnQuotaChange() override;

 protected:
  // EventTarget overrides.
  void AddedEventListener(const AtomicString& event_type,
                          RegisteredEventListener&) final;
  void RemovedEventListener(const AtomicString& event_type,
                            const RegisteredEventListener&) final;

 private:
  mojom::blink::PermissionService* GetPermissionService(ExecutionContext*);

  void PermissionServiceConnectionError();
  void PermissionRequestComplete(ScriptPromiseResolver*,
                                 mojom::blink::PermissionStatus);

  // Called when a quota change event listener is added.
  void StartObserving();

  // Called when all the change event listeners have been removed.
  void StopObserving();

  // Binds the interface (if not already bound) with the given interface
  // provider, and returns it,
  mojom::blink::QuotaManagerHost* GetQuotaHost(ExecutionContext*);

  HeapMojoRemote<mojom::blink::PermissionService,
                 HeapMojoWrapperMode::kWithoutContextObserver>
      permission_service_;
  HeapMojoRemote<mojom::blink::QuotaManagerHost,
                 HeapMojoWrapperMode::kWithoutContextObserver>
      quota_host_;

  HeapMojoReceiver<mojom::blink::QuotaChangeListener, StorageManager>
      change_listener_receiver_;
};

}  // namespace blink

#endif  // THIRD_PARTY_BLINK_RENDERER_MODULES_QUOTA_STORAGE_MANAGER_H_
