/*
 * Copyright (C) 2004, 2005, 2006, 2008 Nikolas Zimmermann <zimmermann@kde.org>
 * Copyright (C) 2004, 2005 Rob Buis <buis@kde.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; see the file COPYING.LIB.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include "third_party/blink/renderer/core/svg/svg_point_list.h"

#include "third_party/blink/renderer/core/svg/animation/smil_animation_effect_parameters.h"
#include "third_party/blink/renderer/core/svg/svg_parser_utilities.h"
#include "third_party/blink/renderer/platform/geometry/float_point.h"
#include "third_party/blink/renderer/platform/heap/heap.h"
#include "third_party/blink/renderer/platform/wtf/text/character_visitor.h"
#include "third_party/blink/renderer/platform/wtf/text/wtf_string.h"

namespace blink {

SVGPointList::SVGPointList() = default;

SVGPointList::~SVGPointList() = default;

template <typename CharType>
SVGParsingError SVGPointList::Parse(const CharType* ptr, const CharType* end) {
  if (!SkipOptionalSVGSpaces(ptr, end))
    return SVGParseStatus::kNoError;

  const CharType* list_start = ptr;
  for (;;) {
    float x = 0;
    float y = 0;
    if (!ParseNumber(ptr, end, x) ||
        !ParseNumber(ptr, end, y, kDisallowWhitespace))
      return SVGParsingError(SVGParseStatus::kExpectedNumber, ptr - list_start);

    Append(MakeGarbageCollected<SVGPoint>(FloatPoint(x, y)));

    if (!SkipOptionalSVGSpaces(ptr, end))
      break;

    if (*ptr == ',') {
      ++ptr;
      SkipOptionalSVGSpaces(ptr, end);

      // ',' requires the list to be continued
      continue;
    }
  }
  return SVGParseStatus::kNoError;
}

SVGParsingError SVGPointList::SetValueAsString(const String& value) {
  Clear();

  if (value.IsEmpty())
    return SVGParseStatus::kNoError;

  return WTF::VisitCharacters(value, [&](const auto* chars, unsigned length) {
    return Parse(chars, chars + length);
  });
}

void SVGPointList::Add(const SVGPropertyBase* other,
                       const SVGElement* context_element) {
  auto* other_list = To<SVGPointList>(other);

  if (length() != other_list->length())
    return;

  for (uint32_t i = 0; i < length(); ++i)
    at(i)->SetValue(at(i)->Value() + other_list->at(i)->Value());
}

void SVGPointList::CalculateAnimatedValue(
    const SMILAnimationEffectParameters& parameters,
    float percentage,
    unsigned repeat_count,
    const SVGPropertyBase* from_value,
    const SVGPropertyBase* to_value,
    const SVGPropertyBase* to_at_end_of_duration_value,
    const SVGElement* context_element) {
  auto* from_list = To<SVGPointList>(from_value);
  auto* to_list = To<SVGPointList>(to_value);

  if (!AdjustFromToListValues(from_list, to_list, percentage))
    return;

  auto* to_at_end_of_duration_list =
      To<SVGPointList>(to_at_end_of_duration_value);

  uint32_t from_point_list_size = from_list->length();
  uint32_t to_point_list_size = to_list->length();
  uint32_t to_at_end_of_duration_list_size =
      to_at_end_of_duration_list->length();

  for (uint32_t i = 0; i < to_point_list_size; ++i) {
    FloatPoint effective_from;
    if (from_point_list_size)
      effective_from = from_list->at(i)->Value();
    FloatPoint effective_to = to_list->at(i)->Value();
    FloatPoint effective_to_at_end;
    if (i < to_at_end_of_duration_list_size)
      effective_to_at_end = to_at_end_of_duration_list->at(i)->Value();

    FloatPoint result(
        ComputeAnimatedNumber(parameters, percentage, repeat_count,
                              effective_from.X(), effective_to.X(),
                              effective_to_at_end.X()),
        ComputeAnimatedNumber(parameters, percentage, repeat_count,
                              effective_from.Y(), effective_to.Y(),
                              effective_to_at_end.Y()));
    if (parameters.is_additive)
      result += at(i)->Value();

    at(i)->SetValue(result);
  }
}

float SVGPointList::CalculateDistance(const SVGPropertyBase* to,
                                      const SVGElement*) const {
  // FIXME: Distance calculation is not possible for SVGPointList right now. We
  // need the distance for every single value.
  return -1;
}

}  // namespace blink
