// Copyright 2018 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "third_party/blink/renderer/core/css/css_property_value_set.h"
#include "testing/gtest/include/gtest/gtest.h"
#include "third_party/blink/renderer/core/css/parser/css_parser.h"
#include "third_party/blink/renderer/core/css/parser/css_parser_context.h"
#include "third_party/blink/renderer/core/css/style_rule.h"
#include "third_party/blink/renderer/core/css/style_sheet_contents.h"
#include "third_party/blink/renderer/core/testing/page_test_base.h"
#include "third_party/blink/renderer/platform/heap/heap.h"

namespace blink {

class CSSPropertyValueSetTest : public PageTestBase {
 public:
  StyleRule* RuleAt(StyleSheetContents* sheet, wtf_size_t index) {
    return To<StyleRule>(sheet->ChildRules()[index].Get());
  }
};

TEST_F(CSSPropertyValueSetTest, MergeAndOverrideOnConflictCustomProperty) {
  auto* context = MakeGarbageCollected<CSSParserContext>(GetDocument());
  auto* style_sheet = MakeGarbageCollected<StyleSheetContents>(context);

  String sheet_text = R"CSS(
    #first {
      color: red;
      --x:foo;
      --y:foo;
    }
    #second {
      color: green;
      --x:bar;
      --y:bar;
    }
  )CSS";

  CSSParser::ParseSheet(context, style_sheet, sheet_text,
                        CSSDeferPropertyParsing::kNo);
  StyleRule* rule0 = RuleAt(style_sheet, 0);
  StyleRule* rule1 = RuleAt(style_sheet, 1);
  MutableCSSPropertyValueSet& set0 = rule0->MutableProperties();
  MutableCSSPropertyValueSet& set1 = rule1->MutableProperties();

  EXPECT_EQ(3u, set0.PropertyCount());
  EXPECT_EQ("red", set0.GetPropertyValue(CSSPropertyID::kColor));
  EXPECT_EQ("foo", set0.GetPropertyValue(AtomicString("--x")));
  EXPECT_EQ("foo", set0.GetPropertyValue(AtomicString("--y")));
  EXPECT_EQ(3u, set1.PropertyCount());
  EXPECT_EQ("green", set1.GetPropertyValue(CSSPropertyID::kColor));
  EXPECT_EQ("bar", set1.GetPropertyValue(AtomicString("--x")));
  EXPECT_EQ("bar", set1.GetPropertyValue(AtomicString("--y")));

  set0.MergeAndOverrideOnConflict(&set1);

  EXPECT_EQ(3u, set0.PropertyCount());
  EXPECT_EQ("green", set0.GetPropertyValue(CSSPropertyID::kColor));
  EXPECT_EQ("bar", set0.GetPropertyValue(AtomicString("--x")));
  EXPECT_EQ("bar", set0.GetPropertyValue(AtomicString("--y")));
  EXPECT_EQ(3u, set1.PropertyCount());
  EXPECT_EQ("green", set1.GetPropertyValue(CSSPropertyID::kColor));
  EXPECT_EQ("bar", set1.GetPropertyValue(AtomicString("--x")));
  EXPECT_EQ("bar", set1.GetPropertyValue(AtomicString("--y")));
}

}  // namespace blink
