// Copyright 2022 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef CONTENT_BROWSER_ATTRIBUTION_REPORTING_ATTRIBUTION_FILTER_DATA_H_
#define CONTENT_BROWSER_ATTRIBUTION_REPORTING_ATTRIBUTION_FILTER_DATA_H_

#include <stddef.h>

#include <string>
#include <vector>

#include "base/containers/flat_map.h"
#include "base/types/expected.h"
#include "content/browser/attribution_reporting/attribution_reporting.mojom-forward.h"
#include "content/browser/attribution_reporting/attribution_source_type.h"
#include "content/common/content_export.h"
#include "third_party/abseil-cpp/absl/types/optional.h"

namespace base {
class Value;
}  // namespace base

namespace content {

// Supports persistence to disk via serializaton to/from proto.
class CONTENT_EXPORT AttributionFilterData {
 public:
  using FilterValues = base::flat_map<std::string, std::vector<std::string>>;

  // Deserializes `string`, if valid. Returns `absl::nullopt` if not.
  //
  // If deserialization succeeded, the returned filter data will also contain a
  // "source_type" key corresponding to `source_type`, representing the
  // autogenerated field of the same name.
  static absl::optional<AttributionFilterData> DeserializeSourceFilterData(
      const std::string& string,
      AttributionSourceType source_type);

  // Source filter data is not allowed to contain a `source_type` filter.
  static absl::optional<AttributionFilterData> FromSourceFilterValues(
      FilterValues&& filter_values);

  // Trigger filter data is allowed to contain a `source_type` filter.
  static absl::optional<AttributionFilterData> FromTriggerFilterValues(
      FilterValues&& filter_values);

  static base::expected<AttributionFilterData,
                        attribution_reporting::mojom::SourceRegistrationError>
  FromSourceJSON(base::Value* input_value);

  // Returns filter data that matches only the given source type.
  static AttributionFilterData ForSourceType(AttributionSourceType source_type);

  // Creates without validation.
  static AttributionFilterData CreateForTesting(FilterValues filter_values);

  AttributionFilterData();

  ~AttributionFilterData();

  AttributionFilterData(const AttributionFilterData&);
  AttributionFilterData(AttributionFilterData&&);

  AttributionFilterData& operator=(const AttributionFilterData&);
  AttributionFilterData& operator=(AttributionFilterData&&);

  const FilterValues& filter_values() const { return filter_values_; }

  std::string Serialize() const;

 private:
  static absl::optional<AttributionFilterData> FromFilterValues(
      FilterValues&& filter_values,
      size_t extra_filters_allowed);

  explicit AttributionFilterData(FilterValues filter_values);

  FilterValues filter_values_;
};

}  // namespace content

#endif  // CONTENT_BROWSER_ATTRIBUTION_REPORTING_ATTRIBUTION_FILTER_DATA_H_
