// Copyright 2022 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "components/services/app_service/public/cpp/capability_access_update.h"

#include "components/services/app_service/public/mojom/types.mojom.h"
#include "testing/gtest/include/gtest/gtest.h"
#include "third_party/abseil-cpp/absl/types/optional.h"

namespace {
const char app_id[] = "abcdefgh";
}  // namespace

class CapabilityAccessUpdateTest : public testing::Test {
 protected:
  absl::optional<bool> expect_camera_;
  bool expect_camera_changed_;

  absl::optional<bool> expect_microphone_;
  bool expect_microphone_changed_;

  AccountId account_id_ = AccountId::FromUserEmail("test@gmail.com");

  void ExpectNoChange() {
    expect_camera_changed_ = false;
    expect_microphone_changed_ = false;
  }

  void CheckExpects(const apps::CapabilityAccessUpdate& u) {
    EXPECT_EQ(expect_camera_, u.Camera());
    EXPECT_EQ(expect_camera_changed_, u.CameraChanged());

    EXPECT_EQ(expect_microphone_, u.Microphone());
    EXPECT_EQ(expect_microphone_changed_, u.MicrophoneChanged());

    EXPECT_EQ(account_id_, u.AccountId());
  }

  void TestCapabilityAccessUpdate(apps::CapabilityAccess* state,
                                  apps::CapabilityAccess* delta) {
    apps::CapabilityAccessUpdate u(state, delta, account_id_);

    EXPECT_EQ(app_id, u.AppId());
    EXPECT_EQ(state == nullptr, u.StateIsNull());

    ExpectNoChange();
    CheckExpects(u);

    // IsAccessingCamera tests.
    if (state) {
      state->camera = false;
      expect_camera_ = false;
      expect_camera_changed_ = false;
      CheckExpects(u);
    }

    if (delta) {
      delta->camera = true;
      expect_camera_ = true;
      expect_camera_changed_ = true;
      CheckExpects(u);
    }

    if (state) {
      apps::CapabilityAccessUpdate::Merge(state, delta);
      ExpectNoChange();
      CheckExpects(u);
    }

    // IsAccessingMicrophone tests.
    if (state) {
      state->microphone = false;
      expect_microphone_ = false;
      expect_microphone_changed_ = false;
      CheckExpects(u);
    }

    if (delta) {
      delta->microphone = true;
      expect_microphone_ = true;
      expect_microphone_changed_ = true;
      CheckExpects(u);
    }

    if (state) {
      apps::CapabilityAccessUpdate::Merge(state, delta);
      ExpectNoChange();
      CheckExpects(u);
    }
  }
};

TEST_F(CapabilityAccessUpdateTest, StateIsNonNull) {
  apps::CapabilityAccessPtr state =
      std::make_unique<apps::CapabilityAccess>(app_id);
  TestCapabilityAccessUpdate(state.get(), nullptr);
}

TEST_F(CapabilityAccessUpdateTest, DeltaIsNonNull) {
  apps::CapabilityAccessPtr delta =
      std::make_unique<apps::CapabilityAccess>(app_id);
  TestCapabilityAccessUpdate(nullptr, delta.get());
}

TEST_F(CapabilityAccessUpdateTest, BothAreNonNull) {
  apps::CapabilityAccessPtr state =
      std::make_unique<apps::CapabilityAccess>(app_id);
  apps::CapabilityAccessPtr delta =
      std::make_unique<apps::CapabilityAccess>(app_id);
  TestCapabilityAccessUpdate(state.get(), delta.get());
}

TEST_F(CapabilityAccessUpdateTest, ConvertEmptyCapabilityAccesses) {
  std::vector<apps::mojom::CapabilityAccessPtr> src;
  EXPECT_EQ(src,
            apps::ConvertCapabilityAccessesToMojomCapabilityAccesses(
                apps::ConvertMojomCapabilityAccessesToCapabilityAccesses(src)));
}

TEST_F(CapabilityAccessUpdateTest, Convert) {
  apps::mojom::CapabilityAccessPtr access1 =
      apps::mojom::CapabilityAccess::New();
  access1->app_id = "a";

  apps::mojom::CapabilityAccessPtr access2 =
      apps::mojom::CapabilityAccess::New();
  access2->app_id = "b";
  access2->camera = apps::mojom::OptionalBool::kTrue;

  apps::mojom::CapabilityAccessPtr access3 =
      apps::mojom::CapabilityAccess::New();
  access3->app_id = "c";
  access3->camera = apps::mojom::OptionalBool::kFalse;
  access3->microphone = apps::mojom::OptionalBool::kTrue;

  std::vector<apps::mojom::CapabilityAccessPtr> src;
  src.push_back(std::move(access1));
  src.push_back(std::move(access2));
  src.push_back(std::move(access3));
  EXPECT_EQ(src,
            apps::ConvertCapabilityAccessesToMojomCapabilityAccesses(
                apps::ConvertMojomCapabilityAccessesToCapabilityAccesses(src)));
}
