// Copyright 2021 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "components/blocked_content/android/popup_blocked_message_delegate.h"

#include "components/blocked_content/android/popup_blocked_helper.h"
#include "components/content_settings/core/browser/host_content_settings_map.h"
#include "components/content_settings/core/common/content_settings.h"
#include "components/content_settings/core/common/content_settings_types.h"
#include "components/messages/android/message_dispatcher_bridge.h"
#include "components/prefs/pref_service.h"
#include "components/resources/android/theme_resources.h"
#include "components/strings/grit/components_strings.h"
#include "ui/base/l10n/l10n_util.h"

namespace blocked_content {

bool PopupBlockedMessageDelegate::ShowMessage(
    int num_popups,
    HostContentSettingsMap* settings_map,
    base::RepeatingCallback<int(int)> resource_id_mapper,
    base::OnceClosure on_show_popups_callback) {
  if (message_ != nullptr) {  // update title only
    message_->SetTitle(l10n_util::GetPluralStringFUTF16(
        IDS_POPUPS_BLOCKED_INFOBAR_TEXT, num_popups));
    return false;
  }

  on_show_popups_callback_ = std::move(on_show_popups_callback);
  url_ = web_contents_->GetLastCommittedURL();
  // Unretained is safe because |this| will always outlive |message_| which owns
  // the callback.
  message_ = std::make_unique<messages::MessageWrapper>(
      base::BindOnce(&PopupBlockedMessageDelegate::HandleClick,
                     base::Unretained(this)),
      base::BindOnce(&PopupBlockedMessageDelegate::HandleDismissCallback,
                     base::Unretained(this)));

  message_->SetTitle(l10n_util::GetPluralStringFUTF16(
      IDS_POPUPS_BLOCKED_INFOBAR_TEXT, num_popups));

  map_ = settings_map;
  allow_settings_changes_ = !PopupSettingManagedByPolicy(map_, url_);

  // Don't allow the user to configure the setting in the UI if the setting
  // is managed by policy.
  int button_text_id = allow_settings_changes_ ? IDS_SHOW_CONTENT : IDS_OK;
  message_->SetPrimaryButtonText(l10n_util::GetStringUTF16(button_text_id));
  message_->SetIconResourceId(
      resource_id_mapper.Run(IDR_ANDROID_INFOBAR_BLOCKED_POPUPS));

  messages::MessageDispatcherBridge::Get()->EnqueueMessage(
      message_.get(), web_contents_, messages::MessageScopeType::NAVIGATION);

  return true;
}

PopupBlockedMessageDelegate::~PopupBlockedMessageDelegate() {
  if (message_ != nullptr) {
    messages::MessageDispatcherBridge::Get()->DismissMessage(
        message_.get(), web_contents_, messages::DismissReason::UNKNOWN);
  }
}

PopupBlockedMessageDelegate::PopupBlockedMessageDelegate(
    content::WebContents* web_contents)
    : web_contents_(web_contents) {}

void PopupBlockedMessageDelegate::HandleDismissCallback(
    messages::DismissReason dismiss_reason) {
  message_.reset();
  map_ = nullptr;
}

void PopupBlockedMessageDelegate::HandleClick() {
  if (!allow_settings_changes_) {
    return;
  }

  // Create exceptions.
  map_->SetNarrowestContentSetting(url_, url_, ContentSettingsType::POPUPS,
                                   CONTENT_SETTING_ALLOW);

  // Launch popups.
  ShowBlockedPopups(web_contents_);

  if (on_show_popups_callback_)
    std::move(on_show_popups_callback_).Run();
}

WEB_CONTENTS_USER_DATA_KEY_IMPL(PopupBlockedMessageDelegate)

}  // namespace blocked_content
