;;; net-utils.el --- network functions  -*- lexical-binding: t; -*-

;; Copyright (C) 1998-2022 Free Software Foundation, Inc.

;; Author: Peter Breton <pbreton@cs.umb.edu>
;; Created: Sun Mar 16 1997
;; Keywords: network comm

;; This file is part of GNU Emacs.

;; GNU Emacs is free software: you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation, either version 3 of the License, or
;; (at your option) any later version.

;; GNU Emacs is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with GNU Emacs.  If not, see <https://www.gnu.org/licenses/>.

;;; Commentary:

;;
;; There are three main areas of functionality:
;;
;; * Wrap common network utility programs (ping, traceroute, netstat,
;; nslookup, arp, route). Note that these wrappers are of the diagnostic
;; functions of these programs only.
;;
;; * Implement some very basic protocols in Emacs Lisp (finger and whois)
;;
;; * Support connections to HOST/PORT, generally for debugging and the like.
;; In other words, for doing much the same thing as "telnet HOST PORT", and
;; then typing commands.

;;; Code:

;; On some systems, programs like ifconfig are not in normal user
;; path, but rather in /sbin, /usr/sbin, etc (but non-root users can
;; still use them for queries).  Actually the trend these
;; days is for /sbin to be a symlink to /usr/sbin, but we still need to
;; search both for older systems.

(require 'subr-x)
(require 'cl-lib)

(defun net-utils--executable-find-sbin (command)
  "Return absolute name of COMMAND if found in an sbin directory."
  (let ((exec-path '("/sbin" "/usr/sbin" "/usr/local/sbin")))
    (executable-find command)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Customization Variables
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defgroup net-utils nil
  "Network utility functions."
  :prefix "net-utils-"
  :group 'comm
  :version "20.3")

(defcustom traceroute-program
  (if (eq system-type 'windows-nt)
      "tracert"
    "traceroute")
  "Program to trace network hops to a destination."
  :type  'string)

(defcustom traceroute-program-options nil
  "Options for the traceroute program."
  :type  '(repeat string))

(defcustom ping-program "ping"
  "Program to send network test packets to a host."
  :type  'string)

;; On GNU/Linux and Irix, the system's ping program seems to send packets
;; indefinitely unless told otherwise
(defcustom ping-program-options
  (and (eq system-type 'gnu/linux)
       (list "-c" "4"))
  "Options for the ping program.
These options can be used to limit how many ICMP packets are emitted."
  :type  '(repeat string))

(defcustom ifconfig-program
  (cond ((eq system-type 'windows-nt) "ipconfig")
        ((executable-find "ifconfig") "ifconfig")
        ((net-utils--executable-find-sbin "ifconfig"))
        ((net-utils--executable-find-sbin "ip"))
        (t "ip"))
  "Program to print network configuration information."
  :version "25.1"                       ; add ip
  :type  'string)

(defcustom ifconfig-program-options
  (cond ((string-match "ipconfig\\'" ifconfig-program) '("/all"))
        ((string-match "ifconfig\\'" ifconfig-program) '("-a"))
        ((string-match "ip\\'" ifconfig-program) '("addr")))
  "Options for the ifconfig program."
  :version "25.1"
  :set-after '(ifconfig-program)
  :type  '(repeat string))

(defcustom iwconfig-program
  (cond ((executable-find "iwconfig") "iwconfig")
        ((net-utils--executable-find-sbin "iw") "iw")
        (t "iw"))
  "Program to print wireless network configuration information."
  :type 'string
  :version "26.1")

(defcustom iwconfig-program-options
  (cond ((string-match-p "iw\\'" iwconfig-program) (list "dev"))
        (t nil))
 "Options for the iwconfig program."
 :type '(repeat string)
 :version "26.1")

(defcustom netstat-program
  (cond ((executable-find "netstat") "netstat")
        ((net-utils--executable-find-sbin "ss"))
        (t "ss"))
  "Program to print network statistics."
  :type  'string
  :version "26.1")

(defcustom netstat-program-options
  (list "-a")
  "Options for the netstat program."
  :type  '(repeat string))

(defcustom arp-program (or (net-utils--executable-find-sbin "arp") "arp")
  "Program to print IP to address translation tables."
  :type  'string)

(defcustom arp-program-options
  (list "-a")
  "Options for the arp program."
  :type  '(repeat string))

(defcustom route-program
  (cond ((eq system-type 'windows-nt) "route")
        ((executable-find "netstat") "netstat")
        ((net-utils--executable-find-sbin "netstat"))
        ((executable-find "ip") "ip")
        ((net-utils--executable-find-sbin "ip"))
        (t "ip"))
  "Program to print routing tables."
  :type  'string
  :version "26.1")

(defcustom route-program-options
  (cond ((eq system-type 'windows-nt) (list "print"))
        ((string-match-p "netstat\\'" route-program) (list "-r"))
        (t (list "route")))
  "Options for the route program."
  :type  '(repeat string)
  :version "26.1")

(defcustom nslookup-program "nslookup"
  "Program to interactively query DNS information."
  :type  'string)

(defcustom nslookup-program-options nil
  "Options for the nslookup program."
  :type  '(repeat string))

(defcustom nslookup-prompt-regexp "^> "
  "Regexp to match the nslookup prompt.

This variable is only used if the variable
`comint-use-prompt-regexp' is non-nil."
  :type  'regexp)

(defcustom dig-program "dig"
  "Program to query DNS information."
  :type  'string)

(defcustom dig-program-options nil
  "Options for the dig program."
  :type '(repeat string)
  :version "26.1")

(defcustom ftp-program "ftp"
  "Program to run to do FTP transfers."
  :type  'string)

(defcustom ftp-program-options nil
  "Options for the ftp program."
  :type  '(repeat string))

(defcustom ftp-prompt-regexp "^ftp>"
  "Regexp which matches the FTP program's prompt.

This variable is only used if the variable
`comint-use-prompt-regexp' is non-nil."
  :type  'regexp)

(defcustom smbclient-program "smbclient"
  "Smbclient program."
  :type  'string)

(defcustom smbclient-program-options nil
  "Options for the smbclient program."
  :type  '(repeat string))

(defcustom smbclient-prompt-regexp "^smb: >"
  "Regexp which matches the smbclient program's prompt.

This variable is only used if the variable
`comint-use-prompt-regexp' is non-nil."
  :type  'regexp)

(defcustom dns-lookup-program "host"
  "Program to interactively query DNS information."
  :type  'string)

(defcustom dns-lookup-program-options nil
  "Options for the dns-lookup program."
  :type  '(repeat string))

;; Internal variables
(defvar network-connection-service nil)
(defvar network-connection-host    nil)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Nslookup goodies
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defvar nslookup-font-lock-keywords
  (list
   (list "^[A-Za-z0-9 _]+:" 0 'font-lock-type-face)
   (list "\\<\\(SOA\\|NS\\|MX\\|A\\|CNAME\\)\\>"
         1 'font-lock-keyword-face)
   ;; Dotted quads
   (list
    (mapconcat #'identity
               (make-list 4 "[0-9]+")
               "\\.")
    0 'font-lock-variable-name-face)
   ;; Host names
   (list
    (let ((host-expression "[-A-Za-z0-9]+"))
      (concat
       (mapconcat #'identity
                  (make-list 2 host-expression)
                  "\\.")
       "\\(\\." host-expression "\\)*"))
    0 'font-lock-variable-name-face))
  "Expressions to font-lock for nslookup.")

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; General network utilities mode
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defvar net-utils-font-lock-keywords
  (list
   ;; Dotted quads
   (list
    (mapconcat #'identity (make-list 4 "[0-9]+") "\\.")
    0 'font-lock-variable-name-face)
   ;; Simple rfc4291 addresses
   (list (concat
	  "\\( \\([[:xdigit:]]+\\(:\\|::\\)\\)+[[:xdigit:]]+\\)"
	  "\\|"
	  "\\(::[[:xdigit:]]+\\)")
    0 'font-lock-variable-name-face)
   ;; Host names
   (list
    (let ((host-expression "[-A-Za-z0-9]+"))
      (concat
       (mapconcat #'identity (make-list 2 host-expression) "\\.")
       "\\(\\." host-expression "\\)*"))
    0 'font-lock-variable-name-face))
  "Expressions to font-lock for general network utilities.")

(define-derived-mode net-utils-mode special-mode "NetworkUtil"
  "Major mode for interacting with an external network utility."
  (setq-local font-lock-defaults
              '((net-utils-font-lock-keywords)))
  (setq-local revert-buffer-function #'net-utils--revert-function))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Utility functions
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;; Simplified versions of some at-point functions from ffap.el.
;; It's not worth loading all of ffap just for these.
(defun net-utils-machine-at-point ()
  (let ((pt (point)))
    (buffer-substring-no-properties
     (save-excursion
       (skip-chars-backward "-a-zA-Z0-9.")
       (point))
     (save-excursion
       (skip-chars-forward "-a-zA-Z0-9.")
       (skip-chars-backward "." pt)
       (point)))))

(defun net-utils-url-at-point ()
  (let ((pt (point)))
    (buffer-substring-no-properties
     (save-excursion
       (skip-chars-backward "--:=&?$+@-Z_a-z~#,%")
       (skip-chars-forward "^A-Za-z0-9" pt)
       (point))
     (save-excursion
       (skip-chars-forward "--:=&?$+@-Z_a-z~#,%")
       (skip-chars-backward ":;.,!?" pt)
       (point)))))

(defun net-utils-remove-ctrl-m-filter (process output-string)
  "Remove trailing control Ms."
  (with-current-buffer (process-buffer process)
    (save-excursion
      (let ((inhibit-read-only t)
            (filtered-string output-string))
        (while (string-match "\r" filtered-string)
          (setq filtered-string
                (replace-match "" nil nil filtered-string)))
        ;; Insert the text, moving the process-marker.
        (goto-char (process-mark process))
        (insert filtered-string)
        (set-marker (process-mark process) (point))))))

(declare-function w32-get-console-output-codepage "w32proc.c" ())

(defun net-utils-run-program (name header program args)
  "Run a network information program."
  (let ((buf (get-buffer-create (concat "*" name "*")))
	(coding-system-for-read
	 ;; MS-Windows versions of network utilities output text
	 ;; encoded in the console (a.k.a. "OEM") codepage, which is
	 ;; different from the default system (a.k.a. "ANSI")
	 ;; codepage.
	 (if (eq system-type 'windows-nt)
	     (intern (format "cp%d" (w32-get-console-output-codepage)))
	   coding-system-for-read)))
    (set-buffer buf)
    (erase-buffer)
    (insert header "\n")
    (set-process-filter
     (apply #'start-process name buf program args)
     #'net-utils-remove-ctrl-m-filter)
    (display-buffer buf)
    buf))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; General network utilities (diagnostic)
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;; Todo: This data could be saved in a bookmark.
(defvar net-utils--revert-cmd nil)

(defun net-utils-run-simple (buffer program-name args &optional nodisplay)
  "Run a network utility for diagnostic output only."
  (with-current-buffer (if (stringp buffer) (get-buffer-create buffer) buffer)
    (let ((proc (get-buffer-process (current-buffer))))
      (when proc
        (set-process-filter proc nil)
        (delete-process proc)))
    (let ((inhibit-read-only t))
      (erase-buffer))
    (net-utils-mode)
    (setq-local net-utils--revert-cmd
                `(net-utils-run-simple ,(current-buffer)
                                       ,program-name ,args nodisplay))
    (let ((coding-system-for-read
	   ;; MS-Windows versions of network utilities output text
	   ;; encoded in the console (a.k.a. "OEM") codepage, which is
	   ;; different from the default system (a.k.a. "ANSI")
	   ;; codepage.
	   (if (eq system-type 'windows-nt)
	       (intern (format "cp%d" (w32-get-console-output-codepage)))
	     coding-system-for-read)))
      (set-process-filter
       (apply #'start-process program-name
              (current-buffer) program-name args)
       #'net-utils-remove-ctrl-m-filter))
    (unless nodisplay (display-buffer (current-buffer)))))

(defun net-utils--revert-function (&optional _ignore-auto _noconfirm)
  (message "Reverting `%s'..." (buffer-name))
  (apply (car net-utils--revert-cmd) (cdr net-utils--revert-cmd))
  (let ((proc (get-buffer-process (current-buffer))))
    (when proc
      (set-process-sentinel
       proc
       (lambda (process event)
         (when (string= event "finished\n")
           (message "Reverting `%s' done" (process-buffer process))))))))

;;;###autoload
(defun ifconfig ()
  "Run `ifconfig-program' and display diagnostic output."
  (interactive)
  (net-utils-run-simple
   (format "*%s*" ifconfig-program)
   ifconfig-program
   ifconfig-program-options))

(defalias 'ipconfig #'ifconfig)

;;;###autoload
(defun iwconfig ()
  "Run `iwconfig-program' and display diagnostic output."
  (interactive)
  (net-utils-run-simple
   (format "*%s*" iwconfig-program)
   iwconfig-program
   iwconfig-program-options))

;;;###autoload
(defun netstat ()
  "Run `netstat-program' and display diagnostic output."
  (interactive)
  (net-utils-run-simple
   (format "*%s*" netstat-program)
   netstat-program
   netstat-program-options))

;;;###autoload
(defun arp ()
  "Run `arp-program' and display diagnostic output."
  (interactive)
  (net-utils-run-simple
   (format "*%s*" arp-program)
   arp-program
   arp-program-options))

;;;###autoload
(defun                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                              