# Geiger Engineering Smart Drive Interface simulation by Bea Wolf (D-ECHO) based on A320 canvas avionics by Joshua Davidson
# Reference: 	https://www.geigerengineering.de/_Resources/Persistent/5081e479d20ad2b8fa95bcb2088f75b275ee6190/Handbuch_Manual_E_Drive%20komplett_V1_999_DE_EN.pdf

var SDI_display = nil;
var SDI_main = nil;

var volts		=	props.globals.getNode("/systems/electrical/outputs/sdi", 1);
var base		=	props.globals.getNode("/instrumentation/sdi/");

var elapsed_sec = props.globals.getNode("/sim/time/elapsed-sec", 1);

var power_controller_release = props.globals.initNode("/fdm/jsbsim/fcs/mc300/enable-throttle", 0, "BOOL");

var instrument_path = "Aircraft/Birdy/Models/Instruments/SDI/";

var rpm 	= props.globals.getNode("/engines/engine-rpm");
var eng_volts 	= props.globals.getNode("/systems/electrical/engine-bus/volts", 1);
var batt_ah 	= props.globals.getNode("/systems/electrical/battery[0]/amp-hours", 1);
var eng_amps 	= props.globals.getNode("/systems/electrical/engine-bus/amps", 1);
var power	= props.globals.getNode("/fdm/jsbsim/propulsion/engine/power-hp", 1);
var batt_temp	= props.globals.getNode("/systems/electrical/battery[0]/temperature-degc", 1);
var motor_temp	= props.globals.initNode("/engines/engine[0]/engine-temp-degc", 15.0, "DOUBLE");
var ctrl_temp	= props.globals.initNode("/engines/engine[0]/mc300-temp-degc", 15.0, "DOUBLE");
var throttle	= props.globals.getNode("/controls/engines/engine/throttle", 1);
var pwm		= props.globals.getNode("/fdm/jsbsim/fcs/throttle-cmd-norm", 1);
var hobbs_min	= props.globals.initNode("/engines/engine[0]/hobbs-min", 0.0, "DOUBLE");
var press_alt	= props.globals.getNode("/instrumentation/altimeter[1]/pressure-alt-ft");
var vario	= props.globals.getNode("/instrumentation/vertical-speed-indicator/indicated-speed-mps", 1);
var clock_ind	= props.globals.getNode("/instrumentation/clock/indicated-string", 1);
var clock_lcl	= props.globals.getNode("/instrumentation/clock/local-short-string", 1);
var date = [
	props.globals.getNode("/sim/time/utc/day", 1),
	props.globals.getNode("/sim/time/utc/month", 1),
	props.globals.getNode("/sim/time/utc/year", 1),
];

var error_audio_signal = base.initNode("error-audio-signal", 0, "BOOL");

var state = 0;	# 0 = off; 1 = starting; 2 = running
var old_state = -1;

var lower_page = 0;	# 	0 = amps
			#	1 = RPM
			#	2 = power
			#	3 = battery temp
			#	4 = motor temperature
			#	5 = output stage temperature controller
			#	6 = throttle
			#	7 = inverter setpoint
			#	8 = minutes of operation
			#	9 = pressure altitude
			#	10 = vario
			#	11 = time of day
			#	12 = date

var error = 0;		#	0 = no error
			#	1 - 9 errors

var canvas_SDI = {
	new: func(canvas_group, file) {
		var m = { parents: [canvas_SDI] };
		
		canvas.parsesvg(canvas_group, file, {'font-mapper': func( family, weight) { return "lcd.txf" } });
		
		var svg_keys = m.getKeys();
		
		foreach(var key; svg_keys) {
			m[key] = canvas_group.getElementById(key);
		}
		
		return m;
	},
	getKeys: func() {
		return ["line1", "line2"];
	},
	update: func() {
		if( state != old_state ){
			if( state == 0 ){
				me["line1"].hide();
				me["line2"].hide();
			} else {
				me["line1"].show();
				me["line2"].show();
			}
			old_state = state;
		}
		if( state == 1 ){
			me["line1"].setText( " SDI" );
			me["line2"].setText("STARTING" );
			error_audio_signal.setBoolValue( 0 );
		} elsif( state == 2 ){
			# Check for errors
			if( eng_volts.getDoubleValue() < 40.0 ){
				error = 1;
			}
			if( ctrl_temp.getDoubleValue() > 85.0 ){
				error = 2;
			}
			if( motor_temp.getDoubleValue() > 100.0 ){
				error = 3;
			}
			if( batt_temp.getDoubleValue() > 60.0 ){ # TODO or collective battery error or cell undervoltage or communication batt - mc300 error
				error = 4;
			}
			# TODO Amps Limit error (5)
			# TODO cut off error (6)
			# TODO external error (7)
			# TODO sd card error (9)
			
			if( error == 0 ){
				error_audio_signal.setBoolValue( 0 );
				me["line1"].setText( sprintf("%02d", math.round( eng_volts.getDoubleValue() ) ) ~"V "~ sprintf("%02d", math.round( batt_ah.getDoubleValue() ) ) ~"Ah" );
				var line2 = "";
				if( lower_page == 0 ){
					line2 = "I=  "~ sprintf("%03d", math.round( eng_amps.getDoubleValue() ) ) ~"A";
				}elsif( lower_page == 1 ){
					line2 = "N= "~ sprintf("%04d", math.round( rpm.getDoubleValue() ) ) ~"U";
				}elsif( lower_page == 2 ){
					line2 = "P= "~ sprintf("%5d", math.round( power.getDoubleValue() * 745.7 ) ) ~"W";
				}elsif( lower_page == 3 ){
					line2 = "TA= "~ sprintf("%3d", math.round( batt_temp.getDoubleValue() ) ) ~"C";
				}elsif( lower_page == 4 ){
					line2 = "TM= "~ sprintf("%3d", math.round( motor_temp.getDoubleValue() ) ) ~ "C";
				}elsif( lower_page == 5 ){
					line2 = "TS= "~ sprintf("%3d", math.round( ctrl_temp.getDoubleValue() )  )~ "C";
				}elsif( lower_page == 6 ){
					line2 = "Th= "~ sprintf("%3d", math.round( throttle.getDoubleValue() * 100 ) ) ~"%";
				}elsif( lower_page == 7 ){
					line2 = "PWM="~ sprintf("%3d", math.round( pwm.getDoubleValue() * 100 ) ) ~"%";
				}elsif( lower_page == 8 ){
					line2 = "D="~ sprintf("%5d", math.round( hobbs_min.getDoubleValue() ) ) ~"m";
				}elsif( lower_page == 9 ){
					line2 = "A="~ sprintf("%5d", math.round( press_alt.getDoubleValue() * FT2M ) ) ~"m";
				}elsif( lower_page == 10 ){
					line2 = "V="~ sprintf("%+2.1f", vario.getDoubleValue() )~"ms";
				}elsif( lower_page == 11 ){
					line2 = clock_lcl.getValue() ~ right( clock_ind.getValue(), 3);
				}elsif( lower_page == 12 ){
					line2 = sprintf("%02d", date[0].getIntValue() ) ~"."~ sprintf("%02d", date[1].getIntValue() ) ~"."~ sprintf("%02d", math.mod( date[2].getIntValue() , 100 ) );
				}else{
					screen.log.write( "Smart Drive Interface: Unknown second line index, reset to 0" );
					line2 = 0;
				}
				me["line2"].setText( line2 );
			}else{
				error_audio_signal.setBoolValue( 1 );
				me["line1"].setText( "Error   " );
				if( error == 1 ){
					me["line2"].setText( "U<Umin  " );
				}elsif( error == 2 ){
					me["line2"].setText( "T.Contr." );
				}elsif( error == 3 ){
					me["line2"].setText( "T.Motor " );
				}elsif( error == 4 ){
					me["line2"].setText( "Battery " );
				}elsif( error == 5 ){
					me["line2"].setText( "I_Limit " );
				}elsif( error == 6 ){
					me["line2"].setText( "Cut OFF " );
				}elsif( error == 7 ){
					me["line2"].setText( "external" );
				}elsif( error == 8 ){
					me["line2"].setText( "Poti min" );
				}elsif( error == 9 ){
					me["line2"].setText( "SD-Card " );
				}else{
					screen.log.write( "Smart Drive Interface: Unknown Error Code, reset to 0 (no error)" );
					error = 0;
				}
			}
			
		}
	},
};

var sdi_update = maketimer(0.2, func {
	SDI_main.update();
});
sdi_update.simulatedTime = 1;

setlistener("sim/signals/fdm-initialized", func {
	SDI_display = canvas.new({
		"name": "SDI",
		"size": [512, 128],
		"view": [512, 128],
		"mipmapping": 0
	});
	SDI_display.addPlacement({"node": "sdi.screen"});
	var groupMain = SDI_display.createGroup();

	SDI_main = canvas_SDI.new(groupMain, instrument_path~"sdi_screen.svg");

	sdi_update.start();
});

var check_state = func () {
	if ( state == 0 ){
		if ( volts.getValue() >= 4.0 ) {
			state = 1;
			init();
			settimer( func() { state = 2; }, 2);
		}
	} else {
		if ( volts.getValue() < 4.0 ){
			state = 0;
		}
	}
}

var init = func () {
	
}

setlistener(volts, func{
	check_state();
});


var ctrl_time = nil;
var ctrl_prev = 0;

var controller_button = func( a ){
	if( a ){
		if( getprop("/fdm/jsbsim/fcs/mc300/throttle-at-zero") ){
			ctrl_time = elapsed_sec.getDoubleValue();
		} else {
			ctrl_time = nil;
			error = 8;
		}
	} elsif( ctrl_prev ) {
		if( ctrl_time != nil and ( elapsed_sec.getDoubleValue() - ctrl_time ) > 1.5 ){
			power_controller_release.setBoolValue( 1 );
		} else {
			power_controller_release.setBoolValue( 0 );
		}
		ctrl_time = nil;
		error = 0;
	}
	ctrl_prev = a;
}

setlistener("/fdm/jsbsim/fcs/mc300/throttle-at-zero", func( i ){
	if( !i.getBoolValue() and ctrl_prev != 0 ){
		ctrl_time = nil;
	}
	if( i.getBoolValue() and error == 8 ){
		error = 0;
	}
});

var scroll_button = func( a ){
	if( a and volts.getDoubleValue() > 4 ){
		lower_page += 1;
		if( lower_page > 12 ) lower_page = 0;
	}
}

var last_hobbs = 0.0;

var hobbs_meter = func {
	if( rpm.getDoubleValue() > 10 ){
		hobbs_min.setDoubleValue( hobbs_min.getDoubleValue() + ( elapsed_sec.getDoubleValue() - last_hobbs ) / 60 );
	}
	last_hobbs = elapsed_sec.getDoubleValue();
}

var hobbs_meter_update = maketimer( 10.0, hobbs_meter );
hobbs_meter_update.simulatedTime = 1;
hobbs_meter_update.start();
	
