/*
 * Decompiled with CFR 0.152.
 */
package org.apache.nifi.registry.bundle.extract.nar;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.FilterInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.StandardCharsets;
import java.time.LocalDateTime;
import java.time.ZoneOffset;
import java.time.format.DateTimeFormatter;
import java.time.format.DateTimeParseException;
import java.util.Date;
import java.util.jar.Attributes;
import java.util.jar.JarEntry;
import java.util.jar.JarInputStream;
import java.util.jar.Manifest;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.apache.nifi.extension.manifest.ExtensionManifest;
import org.apache.nifi.extension.manifest.parser.jaxb.JAXBExtensionManifestParser;
import org.apache.nifi.registry.bundle.extract.BundleException;
import org.apache.nifi.registry.bundle.extract.BundleExtractor;
import org.apache.nifi.registry.bundle.extract.nar.NarManifestEntry;
import org.apache.nifi.registry.bundle.model.BundleDetails;
import org.apache.nifi.registry.bundle.model.BundleIdentifier;
import org.apache.nifi.registry.extension.bundle.BuildInfo;

public class NarBundleExtractor
implements BundleExtractor {
    private static final String EXTENSION_DESCRIPTOR_ENTRY = "META-INF/docs/extension-manifest.xml";
    private static final Pattern ADDITIONAL_DETAILS_ENTRY_PATTERN = Pattern.compile("META-INF\\/docs\\/additional-details\\/(.+)\\/additionalDetails.md");
    private static final String BUILT_TIMESTAMP_FORMAT = "yyyy-MM-dd'T'HH:mm:ss'Z'";
    private static final DateTimeFormatter DATE_TIME_FORMATTER = DateTimeFormatter.ofPattern("yyyy-MM-dd'T'HH:mm:ss'Z'");
    static String NA = "N/A";

    @Override
    public BundleDetails extract(InputStream inputStream) throws IOException {
        try (JarInputStream jarInputStream = new JarInputStream(inputStream);){
            Manifest manifest = jarInputStream.getManifest();
            if (manifest == null) {
                throw new BundleException("NAR bundles must contain a valid MANIFEST");
            }
            Attributes attributes = manifest.getMainAttributes();
            BundleIdentifier bundleIdentifier = this.getBundleCoordinate(attributes);
            BundleIdentifier dependencyCoordinate = this.getDependencyBundleCoordinate(attributes);
            BuildInfo buildInfo = this.getBuildInfo(attributes);
            BundleDetails.Builder builder = new BundleDetails.Builder().coordinate(bundleIdentifier).addDependencyCoordinate(dependencyCoordinate).buildInfo(buildInfo);
            this.parseExtensionDocs(jarInputStream, builder);
            BundleDetails bundleDetails = builder.build();
            return bundleDetails;
        }
    }

    private BundleIdentifier getBundleCoordinate(Attributes attributes) {
        try {
            String groupId = attributes.getValue(NarManifestEntry.NAR_GROUP.getManifestName());
            String artifactId = attributes.getValue(NarManifestEntry.NAR_ID.getManifestName());
            String version = attributes.getValue(NarManifestEntry.NAR_VERSION.getManifestName());
            return new BundleIdentifier(groupId, artifactId, version);
        }
        catch (Exception e) {
            throw new BundleException("Unable to obtain bundle coordinate due to: " + e.getMessage(), e);
        }
    }

    private BundleIdentifier getDependencyBundleCoordinate(Attributes attributes) {
        try {
            String dependencyGroupId = attributes.getValue(NarManifestEntry.NAR_DEPENDENCY_GROUP.getManifestName());
            String dependencyArtifactId = attributes.getValue(NarManifestEntry.NAR_DEPENDENCY_ID.getManifestName());
            String dependencyVersion = attributes.getValue(NarManifestEntry.NAR_DEPENDENCY_VERSION.getManifestName());
            BundleIdentifier dependencyCoordinate = dependencyArtifactId != null ? new BundleIdentifier(dependencyGroupId, dependencyArtifactId, dependencyVersion) : null;
            return dependencyCoordinate;
        }
        catch (Exception e) {
            throw new BundleException("Unable to obtain bundle coordinate for dependency due to: " + e.getMessage(), e);
        }
    }

    private BuildInfo getBuildInfo(Attributes attributes) {
        String buildBranch = attributes.getValue(NarManifestEntry.BUILD_BRANCH.getManifestName());
        String buildTag = attributes.getValue(NarManifestEntry.BUILD_TAG.getManifestName());
        String buildRevision = attributes.getValue(NarManifestEntry.BUILD_REVISION.getManifestName());
        String buildTimestamp = attributes.getValue(NarManifestEntry.BUILD_TIMESTAMP.getManifestName());
        String buildJdk = attributes.getValue(NarManifestEntry.BUILD_JDK.getManifestName());
        String builtBy = attributes.getValue(NarManifestEntry.BUILT_BY.getManifestName());
        try {
            Date buildDate = Date.from(LocalDateTime.parse(buildTimestamp, DATE_TIME_FORMATTER).toInstant(ZoneOffset.UTC));
            BuildInfo buildInfo = new BuildInfo();
            buildInfo.setBuildTool(this.isBlank(buildJdk) ? NA : buildJdk);
            buildInfo.setBuildBranch(this.isBlank(buildBranch) ? NA : buildBranch);
            buildInfo.setBuildTag(this.isBlank(buildTag) ? NA : buildTag);
            buildInfo.setBuildRevision(this.isBlank(buildRevision) ? NA : buildRevision);
            buildInfo.setBuilt(buildDate.getTime());
            buildInfo.setBuiltBy(this.isBlank(builtBy) ? NA : builtBy);
            buildInfo.setBuildFlags(NA);
            return buildInfo;
        }
        catch (DateTimeParseException e) {
            throw new BundleException("Unable to parse " + NarManifestEntry.BUILD_TIMESTAMP.getManifestName(), e);
        }
        catch (Exception e) {
            throw new BundleException("Unable to create build info for bundle due to: " + e.getMessage(), e);
        }
    }

    public boolean isBlank(String value) {
        return value == null || value.trim().isEmpty();
    }

    private void parseExtensionDocs(JarInputStream jarInputStream, BundleDetails.Builder builder) throws IOException {
        JarEntry jarEntry;
        boolean foundExtensionDocs = false;
        while ((jarEntry = jarInputStream.getNextJarEntry()) != null) {
            String jarEntryName = jarEntry.getName();
            if (EXTENSION_DESCRIPTOR_ENTRY.equals(jarEntryName)) {
                try {
                    byte[] rawDocsContent = this.toByteArray(jarInputStream);
                    JAXBExtensionManifestParser docsParser = new JAXBExtensionManifestParser();
                    NonCloseableInputStream inputStream = new NonCloseableInputStream(new ByteArrayInputStream(rawDocsContent));
                    ExtensionManifest extensionManifest = docsParser.parse((InputStream)inputStream);
                    builder.addExtensions(extensionManifest.getExtensions());
                    builder.systemApiVersion(extensionManifest.getSystemApiVersion());
                    foundExtensionDocs = true;
                    continue;
                }
                catch (Exception e) {
                    throw new BundleException("Unable to obtain extension info for bundle due to: " + e.getMessage(), e);
                }
            }
            Matcher matcher = ADDITIONAL_DETAILS_ENTRY_PATTERN.matcher(jarEntryName);
            if (!matcher.matches()) continue;
            String extensionName = matcher.group(1);
            String additionalDetailsContent = new String(this.toByteArray(jarInputStream), StandardCharsets.UTF_8);
            builder.addAdditionalDetails(extensionName, additionalDetailsContent);
        }
        if (!foundExtensionDocs) {
            builder.systemApiVersion(NA);
        }
    }

    private byte[] toByteArray(InputStream input) throws IOException {
        int nRead;
        ByteArrayOutputStream buffer = new ByteArrayOutputStream();
        byte[] data = new byte[16384];
        while ((nRead = input.read(data, 0, data.length)) != -1) {
            buffer.write(data, 0, nRead);
        }
        return buffer.toByteArray();
    }

    private static class NonCloseableInputStream
    extends FilterInputStream {
        private final InputStream toWrap;

        public NonCloseableInputStream(InputStream toWrap) {
            super(toWrap);
            this.toWrap = toWrap;
        }

        @Override
        public int read() throws IOException {
            return this.toWrap.read();
        }

        @Override
        public int read(byte[] b) throws IOException {
            return this.toWrap.read(b);
        }

        @Override
        public int read(byte[] b, int off, int len) throws IOException {
            return this.toWrap.read(b, off, len);
        }

        @Override
        public void close() {
        }
    }
}

