import { ElementRef as viewEngine_ElementRef } from '../linker/element_ref';
import { TemplateRef as viewEngine_TemplateRef } from '../linker/template_ref';
import { ViewContainerRef as viewEngine_ViewContainerRef } from '../linker/view_container_ref';
import { Type } from '../type';
import { DirectiveDef } from './definition_interfaces';
import { LContainer, LElement, LNodeInjector } from './interfaces';
/**
 * Registers this directive as present in its node's injector by flipping the directive's
 * corresponding bit in the injector's bloom filter.
 *
 * @param injector The node injector in which the directive should be registered
 * @param type The directive to register
 */
export declare function bloomAdd(injector: LNodeInjector, type: Type<any>): void;
/**
 * Creates (or gets an existing) injector for a given element or container.
 *
 * @param node for which an injector should be retrieved / created.
 * @returns Node injector
 */
export declare function getOrCreateNodeInjectorForNode(node: LElement | LContainer): LNodeInjector;
/** Injection flags for DI. */
export declare const enum InjectFlags {
    /** Dependency is not required. Null will be injected if there is no provider for the dependency.
       */
    Optional = 1,
    /** When resolving a dependency, include the node that is requesting injection. */
    CheckSelf = 2,
    /** When resolving a dependency, include ancestors of the node requesting injection. */
    CheckParent = 4,
    /** Default injection options: required, checks both self and ancestors. */
    Default = 6,
}
/**
 * Makes a directive public to the DI system by adding it to an injector's bloom filter.
 *
 * @param di The node injector in which a directive will be added
 * @param def The definition of the directive to be made public
 */
export declare function diPublicInInjector(di: LNodeInjector, def: DirectiveDef<any>): void;
/**
 * Searches for an instance of the given directive type up the injector tree and returns
 * that instance if found.
 *
 * Specifically, it gets the bloom filter bit associated with the directive (see bloomHashBit),
 * checks that bit against the bloom filter structure to identify an injector that might have
 * the directive (see bloomFindPossibleInjector), then searches the directives on that injector
 * for a match.
 *
 * If not found, it will propagate up to the next parent injector until the token
 * is found or the top is reached.
 *
 * @param di Node injector where the search should start
 * @param token The directive type to search for
 * @param flags Injection flags (e.g. CheckParent)
 * @returns The instance found
 */
export declare function getOrCreateInjectable<T>(di: LNodeInjector, token: Type<T>, flags?: InjectFlags): T;
/**
 * Finds the closest injector that might have a certain directive.
 *
 * Each directive corresponds to a bit in an injector's bloom filter. Given the bloom bit to
 * check and a starting injector, this function traverses up injectors until it finds an
 * injector that contains a 1 for that bit in its bloom filter. A 1 indicates that the
 * injector may have that directive. It only *may* have the directive because directives begin
 * to share bloom filter bits after the BLOOM_SIZE is reached, and it could correspond to a
 * different directive sharing the bit.
 *
 * Note: We can skip checking further injectors up the tree if an injector's cbf structure
 * has a 0 for that bloom bit. Since cbf contains the merged value of all the parent
 * injectors, a 0 in the bloom bit indicates that the parents definitely do not contain
 * the directive and do not need to be checked.
 *
 * @param injector The starting node injector to check
 * @param  bloomBit The bit to check in each injector's bloom filter
 * @returns An injector that might have the directive
 */
export declare function bloomFindPossibleInjector(startInjector: LNodeInjector, bloomBit: number): LNodeInjector | null;
/**
 * Creates an ElementRef for a given node injector and stores it on the injector.
 * Or, if the ElementRef already exists, retrieves the existing ElementRef.
 *
 * @param di The node injector where we should store a created ElementRef
 * @returns The ElementRef instance to use
 */
export declare function getOrCreateElementRef(di: LNodeInjector): viewEngine_ElementRef;
/**
 * Creates a TemplateRef and stores it on the injector. Or, if the TemplateRef already
 * exists, retrieves the existing TemplateRef.
 *
 * @param di The node injector where we should store a created TemplateRef
 * @returns The TemplateRef instance to use
 */
export declare function getOrCreateTemplateRef<T>(di: LNodeInjector): viewEngine_TemplateRef<T>;
/**
 * Creates a ViewContainerRef and stores it on the injector. Or, if the ViewContainerRef
 * already exists, retrieves the existing ViewContainerRef.
 *
 * @returns The ViewContainerRef instance to use
 */
export declare function getOrCreateContainerRef(di: LNodeInjector): viewEngine_ViewContainerRef;
