"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/**
 * @license
 * Copyright Google Inc. All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
const core_1 = require("@angular-devkit/core");
const fs_1 = require("fs");
/**
 * Read a file and returns its content. This supports different file encoding.
 */
function readFile(fileName) {
    if (!fs_1.existsSync(fileName)) {
        throw new core_1.FileDoesNotExistException(fileName);
    }
    const buffer = fs_1.readFileSync(fileName);
    let len = buffer.length;
    if (len >= 2 && buffer[0] === 0xFE && buffer[1] === 0xFF) {
        // Big endian UTF-16 byte order mark detected. Since big endian is not supported by node.js,
        // flip all byte pairs and treat as little endian.
        len &= ~1;
        for (let i = 0; i < len; i += 2) {
            const temp = buffer[i];
            buffer[i] = buffer[i + 1];
            buffer[i + 1] = temp;
        }
        return buffer.toString('utf16le', 2);
    }
    if (len >= 2 && buffer[0] === 0xFF && buffer[1] === 0xFE) {
        // Little endian UTF-16 byte order mark detected
        return buffer.toString('utf16le', 2);
    }
    if (len >= 3 && buffer[0] === 0xEF && buffer[1] === 0xBB && buffer[2] === 0xBF) {
        // UTF-8 byte order mark detected
        return buffer.toString('utf8', 3);
    }
    // Default is UTF-8 with no byte order mark
    return buffer.toString('utf8');
}
exports.readFile = readFile;
function readJsonFile(path) {
    return core_1.parseJson(readFile(path), core_1.JsonParseMode.Loose);
}
exports.readJsonFile = readJsonFile;
//# sourceMappingURL=data:application/json;base64,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