"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/**
 * @license
 * Copyright Google Inc. All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
const core_1 = require("@angular-devkit/core");
const of_1 = require("rxjs/observable/of");
const throw_1 = require("rxjs/observable/throw");
const last_1 = require("rxjs/operators/last");
const mergeMap_1 = require("rxjs/operators/mergeMap");
const tap_1 = require("rxjs/operators/tap");
const interface_1 = require("../tree/interface");
function _getTypeOfResult(value) {
    if (value === undefined) {
        return 'undefined';
    }
    else if (value === null) {
        return 'null';
    }
    else if (typeof value == 'function') {
        return `Function()`;
    }
    else if (typeof value != 'object') {
        return `${typeof value}(${JSON.stringify(value)})`;
    }
    else {
        if (Object.getPrototypeOf(value) == Object) {
            return `Object(${JSON.stringify(value)})`;
        }
        else if (value.constructor) {
            return `Instance of class ${value.constructor.name}`;
        }
        else {
            return 'Unknown Object';
        }
    }
}
/**
 * When a rule or source returns an invalid value.
 */
class InvalidRuleResultException extends core_1.BaseException {
    constructor(value) {
        super(`Invalid rule result: ${_getTypeOfResult(value)}.`);
    }
}
exports.InvalidRuleResultException = InvalidRuleResultException;
class InvalidSourceResultException extends core_1.BaseException {
    constructor(value) {
        super(`Invalid source result: ${_getTypeOfResult(value)}.`);
    }
}
exports.InvalidSourceResultException = InvalidSourceResultException;
function callSource(source, context) {
    const result = source(context);
    if (result === undefined) {
        return throw_1._throw(new InvalidSourceResultException(result));
    }
    else if (interface_1.TreeSymbol in result) {
        return of_1.of(result);
    }
    else if (Symbol.observable in result) {
        // Only return the last Tree, and make sure it's a Tree.
        return result.pipe(last_1.last(), tap_1.tap(inner => {
            if (!(interface_1.TreeSymbol in inner)) {
                throw new InvalidSourceResultException(inner);
            }
        }));
    }
    else {
        return throw_1._throw(new InvalidSourceResultException(result));
    }
}
exports.callSource = callSource;
function callRule(rule, input, context) {
    return input.pipe(mergeMap_1.mergeMap(inputTree => {
        const result = rule(inputTree, context);
        if (result === undefined) {
            return of_1.of(inputTree);
        }
        else if (interface_1.TreeSymbol in result) {
            return of_1.of(result);
        }
        else if (Symbol.observable in result) {
            const obs = result;
            // Only return the last Tree, and make sure it's a Tree.
            return obs.pipe(last_1.last(), tap_1.tap(inner => {
                if (!(interface_1.TreeSymbol in inner)) {
                    throw new InvalidRuleResultException(inner);
                }
            }));
        }
        else if (result === undefined) {
            return of_1.of(inputTree);
        }
        else {
            return throw_1._throw(new InvalidRuleResultException(result));
        }
    }));
}
exports.callRule = callRule;
//# sourceMappingURL=data:application/json;base64,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