/* test.c
 *
 * Copyright (C) 2006-2022 wolfSSL Inc.
 *
 * This file is part of wolfSSL.
 *
 * wolfSSL is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * wolfSSL is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1335, USA
 */

#ifdef HAVE_CONFIG_H
    #include <config.h>
#endif

#ifndef WOLFSSL_USER_SETTINGS
    #include <wolfssl/options.h>
#endif
#include <wolfssl/wolfcrypt/settings.h>
#include <wolfssl/version.h>
#include <wolfssl/wolfcrypt/wc_port.h>

#ifndef NO_CRYPT_TEST

#if defined(HAVE_STACK_SIZE) && !defined(HAVE_WOLFCRYPT_TEST_OPTIONS)
    #define HAVE_WOLFCRYPT_TEST_OPTIONS
#endif

#if defined(HAVE_WOLFCRYPT_TEST_OPTIONS)
    #include <wolfssl/ssl.h>
    #define err_sys err_sys_remap /* remap err_sys */
    #include <wolfssl/test.h>
    #undef err_sys
#endif

#if defined(WC_ECC_NONBLOCK) && defined(WOLFSSL_PUBLIC_MP) && \
    defined(HAVE_ECC_SIGN) && defined(HAVE_ECC_VERIFY)
    #include <stdint.h>
#endif

#if defined(HAVE_STACK_SIZE_VERBOSE)
#ifdef WOLFSSL_TEST_MAX_RELATIVE_STACK_BYTES
    static ssize_t max_relative_stack = WOLFSSL_TEST_MAX_RELATIVE_STACK_BYTES;
#else
    static ssize_t max_relative_stack = -1;
#endif
#else
    #define STACK_SIZE_CHECKPOINT_WITH_MAX_CHECK(max, ...) (__VA_ARGS__, 0)
    #define STACK_SIZE_INIT()
#endif

#ifdef WOLFSSL_TRACK_MEMORY_VERBOSE
#ifdef WOLFSSL_TEST_MAX_RELATIVE_HEAP_ALLOCS
    static ssize_t max_relative_heap_allocs = WOLFSSL_TEST_MAX_RELATIVE_HEAP_ALLOCS;
#else
    static ssize_t max_relative_heap_allocs = -1;
#endif
#ifdef WOLFSSL_TEST_MAX_RELATIVE_HEAP_BYTES
    static ssize_t max_relative_heap_bytes = WOLFSSL_TEST_MAX_RELATIVE_HEAP_BYTES;
#else
    static ssize_t max_relative_heap_bytes = -1;
#endif
#define PRINT_HEAP_CHECKPOINT() {                                           \
    const ssize_t _rha = wolfCrypt_heap_peakAllocs_checkpoint() - heap_baselineAllocs; \
    const ssize_t _rhb = wolfCrypt_heap_peakBytes_checkpoint() - heap_baselineBytes;   \
    printf("    relative heap peak usage: %ld alloc%s, %ld bytes\n",         \
           (long int)_rha,                                                   \
           _rha == 1 ? "" : "s",                                             \
           (long int)_rhb);                                                  \
    if ((max_relative_heap_allocs > 0) && (_rha > max_relative_heap_allocs)) \
        return err_sys("heap allocs exceed designated max.", -1);            \
    if ((max_relative_heap_bytes > 0) && (_rhb > max_relative_heap_bytes))   \
        return err_sys("heap bytes exceed designated max.", -1);             \
    heap_baselineAllocs = wolfCrypt_heap_peakAllocs_checkpoint();            \
    heap_baselineBytes = wolfCrypt_heap_peakBytes_checkpoint();              \
    }
#else
#define PRINT_HEAP_CHECKPOINT()
#endif

#ifdef USE_FLAT_TEST_H
    #ifdef HAVE_CONFIG_H
        #include "test_paths.h"
    #endif
    #include "test.h"
#else
    #ifdef HAVE_CONFIG_H
        #include "wolfcrypt/test/test_paths.h"
    #endif
    #include "wolfcrypt/test/test.h"
#endif

/* printf mappings */
#if defined(FREESCALE_MQX) || defined(FREESCALE_KSDK_MQX)
    #include <mqx.h>
    #include <stdlib.h>
    /* see wc_port.h for fio.h and nio.h includes */
#elif defined(FREESCALE_KSDK_BM)
    #include "fsl_debug_console.h"
    #undef printf
    #define printf PRINTF
#elif defined(WOLFSSL_APACHE_MYNEWT)
    #include <assert.h>
    #include <string.h>
    #include "sysinit/sysinit.h"
    #include "os/os.h"
    #ifdef ARCH_sim
    #include "mcu/mcu_sim.h"
    #endif
    #include "os/os_time.h"
#elif defined(WOLFSSL_ESPIDF)
    #include <time.h>
    #include <sys/time.h>
#elif defined(WOLFSSL_ZEPHYR)
    #include <stdio.h>

    #define printf printk
#elif defined(MICRIUM)
    #include <os.h>
    #if (OS_VERSION < 50000)
        #include <bsp_ser.h>
        void BSP_Ser_Printf (CPU_CHAR* format, ...);
        #undef printf
        #define printf BSP_Ser_Printf
    #else
        #include <stdio.h>
    #endif
#elif defined(WOLFSSL_PB)
    #include <stdarg.h>
    int wolfssl_pb_print(const char*, ...);
    #undef printf
    #define printf wolfssl_pb_print
#elif defined(WOLFSSL_TELIT_M2MB)
    #include "wolfssl/wolfcrypt/wc_port.h" /* for m2mb headers */
    #include "m2m_log.h" /* for M2M_LOG_INFO - not standard API */
    /* remap printf */
    #undef printf
    #define printf M2M_LOG_INFO
    /* OS requires occasional sleep() */
    #ifndef TEST_SLEEP_MS
        #define TEST_SLEEP_MS 50
    #endif
    #define TEST_SLEEP() m2mb_os_taskSleep(M2MB_OS_MS2TICKS(TEST_SLEEP_MS))
    /* don't use file system for these tests, since ./certs dir isn't loaded */
    #undef  NO_FILESYSTEM
    #define NO_FILESYSTEM
#elif defined(THREADX) && !defined(WOLFSSL_WICED) && \
      !defined(THREADX_NO_DC_PRINTF)
    #ifndef NETOS
        /* since just testing, use THREADX log printf instead (NETOS prototypes
         * this elsewhere) */
        int dc_log_printf(char*, ...);
    #endif
    #undef printf
    #define printf dc_log_printf
#elif defined(ANDROID)
    #ifdef XMALLOC_USER
        #include <stdlib.h>  /* we're using malloc / free direct here */
    #endif
    #ifndef STRING_USER
        #include <stdio.h>
    #endif
    #include <android/log.h>

    #ifdef ANDROID_V454 /* See fips/android/wolfCrypt_v454_android */
        #ifndef NO_FILESYSTEM
            #define NO_FILESYSTEM /* Turn off tests that want to call SaveDerAndPem() */
        #endif
    #else
        #define printf(...)       \
              __android_log_print(ANDROID_LOG_DEBUG, "[WOLFCRYPT]", __VA_ARGS__)
        #define fprintf(fp, ...)  \
              __android_log_print(ANDROID_LOG_DEBUG, "[WOLFCRYPT]", __VA_ARGS__)
    #endif
#elif defined(WOLFSSL_DEOS)
    #include <printx.h>
    #undef printf
    #define printf printx
#else
    #ifdef XMALLOC_USER
        #include <stdlib.h>  /* we're using malloc / free direct here */
    #endif
    #if !defined(STRING_USER) && !defined(WOLFSSL_LINUXKM)
        #include <stdio.h>
    #endif

    #if defined(WOLFSSL_LINUXKM) && !defined(WOLFSSL_LINUXKM_VERBOSE_DEBUG)
        #undef printf
        #define printf(...) ({})
    #endif

    /* enable way for customer to override test/bench printf */
    #ifdef XPRINTF
        #undef  printf
        #define printf XPRINTF
    #elif !defined(printf)
        /* arrange for printf() to flush after every message -- this assures
         * redirected output (to a log file) records progress right up to the
         * moment of a crash/abort(); otherwise anything queued in stdout would
         * be lost.
         */
        #define printf(...) ( printf(__VA_ARGS__), fflush(stdout) )
    #endif
#endif

#include <wolfssl/wolfcrypt/memory.h>
#include <wolfssl/wolfcrypt/wc_port.h>
#include <wolfssl/wolfcrypt/logging.h>
#include <wolfssl/wolfcrypt/types.h>
#include <wolfssl/wolfcrypt/asn.h>
#include <wolfssl/wolfcrypt/md2.h>
#include <wolfssl/wolfcrypt/md5.h>
#include <wolfssl/wolfcrypt/md4.h>
#include <wolfssl/wolfcrypt/sha.h>
#include <wolfssl/wolfcrypt/sha256.h>
#include <wolfssl/wolfcrypt/sha512.h>
#include <wolfssl/wolfcrypt/rc2.h>
#include <wolfssl/wolfcrypt/arc4.h>
#if defined(WC_NO_RNG)
    #include <wolfssl/wolfcrypt/integer.h>
#else
    #include <wolfssl/wolfcrypt/random.h>
#endif
#include <wolfssl/wolfcrypt/coding.h>
#include <wolfssl/wolfcrypt/signature.h>
#include <wolfssl/wolfcrypt/rsa.h>
#include <wolfssl/wolfcrypt/des3.h>
#include <wolfssl/wolfcrypt/aes.h>
#include <wolfssl/wolfcrypt/wc_encrypt.h>
#include <wolfssl/wolfcrypt/cmac.h>
#include <wolfssl/wolfcrypt/siphash.h>
#include <wolfssl/wolfcrypt/poly1305.h>
#include <wolfssl/wolfcrypt/camellia.h>
#include <wolfssl/wolfcrypt/hmac.h>
#include <wolfssl/wolfcrypt/kdf.h>
#include <wolfssl/wolfcrypt/dh.h>
#include <wolfssl/wolfcrypt/dsa.h>
#include <wolfssl/wolfcrypt/srp.h>
#include <wolfssl/wolfcrypt/chacha.h>
#include <wolfssl/wolfcrypt/chacha20_poly1305.h>
#include <wolfssl/wolfcrypt/pwdbased.h>
#include <wolfssl/wolfcrypt/ripemd.h>
#include <wolfssl/wolfcrypt/error-crypt.h>
#ifdef HAVE_ECC
    #include <wolfssl/wolfcrypt/ecc.h>
#endif
#ifdef HAVE_CURVE25519
    #include <wolfssl/wolfcrypt/curve25519.h>
#endif
#ifdef HAVE_ED25519
    #include <wolfssl/wolfcrypt/ed25519.h>
#endif
#ifdef HAVE_CURVE448
    #include <wolfssl/wolfcrypt/curve448.h>
#endif
#ifdef HAVE_ED448
    #include <wolfssl/wolfcrypt/ed448.h>
#endif
#ifdef WOLFCRYPT_HAVE_ECCSI
    #include <wolfssl/wolfcrypt/eccsi.h>
#endif
#ifdef WOLFCRYPT_HAVE_SAKKE
    #include <wolfssl/wolfcrypt/sakke.h>
#endif
#if defined(HAVE_BLAKE2) || defined(HAVE_BLAKE2S)
    #include <wolfssl/wolfcrypt/blake2.h>
#endif
#ifdef WOLFSSL_SHA3
    #include <wolfssl/wolfcrypt/sha3.h>
#endif
#ifdef HAVE_LIBZ
    #include <wolfssl/wolfcrypt/compress.h>
#endif
#ifdef HAVE_PKCS7
    #include <wolfssl/wolfcrypt/pkcs7.h>
#endif
#ifdef HAVE_FIPS
    #include <wolfssl/wolfcrypt/fips_test.h>
#endif
#ifdef HAVE_SELFTEST
    #include <wolfssl/wolfcrypt/selftest.h>
#endif
#ifdef WOLFSSL_ASYNC_CRYPT
    #include <wolfssl/wolfcrypt/async.h>
#endif
#if defined(OPENSSL_EXTRA) || defined(DEBUG_WOLFSSL_VERBOSE)
    #include <wolfssl/wolfcrypt/logging.h>
#endif
#ifdef WOLFSSL_CAAM
    #include <wolfssl/wolfcrypt/port/caam/wolfcaam.h>
#endif
#ifdef WOLF_CRYPTO_CB
    #include <wolfssl/wolfcrypt/cryptocb.h>
    #ifdef HAVE_INTEL_QA_SYNC
        #include <wolfssl/wolfcrypt/port/intel/quickassist_sync.h>
    #endif
    #ifdef HAVE_CAVIUM_OCTEON_SYNC
        #include <wolfssl/wolfcrypt/port/cavium/cavium_octeon_sync.h>
    #endif
#endif

#ifdef _MSC_VER
    /* 4996 warning to use MS extensions e.g., strcpy_s instead of strncpy */
    #pragma warning(disable: 4996)
#endif

#ifdef OPENSSL_EXTRA
  #ifndef WOLFCRYPT_ONLY
    #include <wolfssl/openssl/evp.h>
    #include <wolfssl/openssl/hmac.h>
  #endif
    #include <wolfssl/openssl/rand.h>
    #include <wolfssl/openssl/aes.h>
    #include <wolfssl/openssl/des.h>
#endif

#if defined(NO_FILESYSTEM) || defined(WC_NO_RNG)
    #if !defined(USE_CERT_BUFFERS_1024) && !defined(USE_CERT_BUFFERS_2048) && \
        !defined(USE_CERT_BUFFERS_3072) && !defined(USE_CERT_BUFFERS_4096)
        #define USE_CERT_BUFFERS_2048
    #endif
    #if !defined(USE_CERT_BUFFERS_256)
        #define USE_CERT_BUFFERS_256
    #endif
#endif

#if defined(WOLFSSL_CERT_GEN) && (defined(HAVE_ECC384) || defined(HAVE_ALL_CURVES))
    #define ENABLE_ECC384_CERT_GEN_TEST
#endif

#include <wolfssl/certs_test.h>

#ifdef DEVKITPRO
    #include <wiiuse/wpad.h>
#endif

#ifdef WOLFSSL_STATIC_MEMORY
    static WOLFSSL_HEAP_HINT* HEAP_HINT;
#else
    #define HEAP_HINT NULL
#endif /* WOLFSSL_STATIC_MEMORY */

/* these cases do not have intermediate hashing support */
#if (defined(WOLFSSL_AFALG_XILINX_SHA3) && !defined(WOLFSSL_AFALG_HASH_KEEP)) \
        && !defined(WOLFSSL_XILINX_CRYPT)
    #define NO_INTM_HASH_TEST
#endif

#if defined(WOLFSSL_RENESAS_TSIP) || defined(WOLFSSL_RENESAS_SCEPROTECT) || \
    defined(WOLFSSL_SECO_CAAM)
    #define HASH_SIZE_LIMIT
#endif

#if defined(WOLFSSL_CERT_GEN) && (!defined(NO_RSA) || defined(HAVE_ECC)) || \
  (defined(WOLFSSL_TEST_CERT) && (defined(HAVE_ED25519) || defined(HAVE_ED448)))
static void initDefaultName(void);
#endif

/* for async devices */
#ifdef WOLFSSL_CAAM_DEVID
static int devId = WOLFSSL_CAAM_DEVID;
#else
static int devId = INVALID_DEVID;
#endif

#ifdef HAVE_WNR
    const char* wnrConfigFile = "wnr-example.conf";
#endif

#define TEST_STRING    "Everyone gets Friday off."
#define TEST_STRING_SZ 25

typedef struct testVector {
    const char*  input;
    const char*  output;
    size_t inLen;
    size_t outLen;
} testVector;

#ifndef WOLFSSL_TEST_SUBROUTINE
#define WOLFSSL_TEST_SUBROUTINE
#endif

PRAGMA_GCC("GCC diagnostic ignored \"-Wunused-function\"")

WOLFSSL_TEST_SUBROUTINE int  error_test(void);
WOLFSSL_TEST_SUBROUTINE int  base64_test(void);
WOLFSSL_TEST_SUBROUTINE int  base16_test(void);
WOLFSSL_TEST_SUBROUTINE int  asn_test(void);
WOLFSSL_TEST_SUBROUTINE int  md2_test(void);
WOLFSSL_TEST_SUBROUTINE int  md5_test(void);
WOLFSSL_TEST_SUBROUTINE int  md4_test(void);
WOLFSSL_TEST_SUBROUTINE int  sha_test(void);
WOLFSSL_TEST_SUBROUTINE int  sha224_test(void);
WOLFSSL_TEST_SUBROUTINE int  sha256_test(void);
WOLFSSL_TEST_SUBROUTINE int  sha512_test(void);
WOLFSSL_TEST_SUBROUTINE int  sha384_test(void);
WOLFSSL_TEST_SUBROUTINE int  sha3_test(void);
WOLFSSL_TEST_SUBROUTINE int  shake256_test(void);
WOLFSSL_TEST_SUBROUTINE int  hash_test(void);
WOLFSSL_TEST_SUBROUTINE int  hmac_md5_test(void);
WOLFSSL_TEST_SUBROUTINE int  hmac_sha_test(void);
WOLFSSL_TEST_SUBROUTINE int  hmac_sha224_test(void);
WOLFSSL_TEST_SUBROUTINE int  hmac_sha256_test(void);
WOLFSSL_TEST_SUBROUTINE int  hmac_sha384_test(void);
WOLFSSL_TEST_SUBROUTINE int  hmac_sha512_test(void);
WOLFSSL_TEST_SUBROUTINE int  hmac_sha3_test(void);
#if defined(HAVE_HKDF) && !defined(NO_HMAC)
/* hkdf_test has issue with WOLFSSL_TEST_SUBROUTINE set on Xilinx with afalg */
static int  hkdf_test(void);
#endif
WOLFSSL_TEST_SUBROUTINE int  sshkdf_test(void);
#ifdef WOLFSSL_TLS13
WOLFSSL_TEST_SUBROUTINE int  tls13_kdf_test(void);
#endif
WOLFSSL_TEST_SUBROUTINE int  x963kdf_test(void);
WOLFSSL_TEST_SUBROUTINE int  arc4_test(void);
#ifdef WC_RC2
WOLFSSL_TEST_SUBROUTINE int  rc2_test(void);
#endif
WOLFSSL_TEST_SUBROUTINE int  chacha_test(void);
WOLFSSL_TEST_SUBROUTINE int  XChaCha_test(void);
WOLFSSL_TEST_SUBROUTINE int  chacha20_poly1305_aead_test(void);
WOLFSSL_TEST_SUBROUTINE int  XChaCha20Poly1305_test(void);
WOLFSSL_TEST_SUBROUTINE int  des_test(void);
WOLFSSL_TEST_SUBROUTINE int  des3_test(void);
WOLFSSL_TEST_SUBROUTINE int  aes_test(void);
WOLFSSL_TEST_SUBROUTINE int  aes192_test(void);
WOLFSSL_TEST_SUBROUTINE int  aes256_test(void);
WOLFSSL_TEST_SUBROUTINE int  aesofb_test(void);
WOLFSSL_TEST_SUBROUTINE int  cmac_test(void);
#if defined(WOLFSSL_SIPHASH)
WOLFSSL_TEST_SUBROUTINE int  siphash_test(void);
#endif
WOLFSSL_TEST_SUBROUTINE int  poly1305_test(void);
WOLFSSL_TEST_SUBROUTINE int  aesgcm_test(void);
WOLFSSL_TEST_SUBROUTINE int  aesgcm_default_test(void);
WOLFSSL_TEST_SUBROUTINE int  gmac_test(void);
WOLFSSL_TEST_SUBROUTINE int  aesccm_test(void);
WOLFSSL_TEST_SUBROUTINE int  aeskeywrap_test(void);
WOLFSSL_TEST_SUBROUTINE int  camellia_test(void);
#ifdef WC_RSA_NO_PADDING
WOLFSSL_TEST_SUBROUTINE int  rsa_no_pad_test(void);
#endif
WOLFSSL_TEST_SUBROUTINE int  rsa_test(void);
WOLFSSL_TEST_SUBROUTINE int  dh_test(void);
WOLFSSL_TEST_SUBROUTINE int  dsa_test(void);
WOLFSSL_TEST_SUBROUTINE int  srp_test(void);
#ifndef WC_NO_RNG
WOLFSSL_TEST_SUBROUTINE int  random_test(void);
#endif /* WC_NO_RNG */
WOLFSSL_TEST_SUBROUTINE int  pwdbased_test(void);
WOLFSSL_TEST_SUBROUTINE int  ripemd_test(void);
#if defined(OPENSSL_EXTRA) && !defined(WOLFCRYPT_ONLY)
WOLFSSL_TEST_SUBROUTINE int  openssl_test(void);   /* test mini api */

WOLFSSL_TEST_SUBROUTINE int  openssl_pkey0_test(void);
WOLFSSL_TEST_SUBROUTINE int  openssl_pkey1_test(void);
WOLFSSL_TEST_SUBROUTINE int  openSSL_evpMD_test(void);
WOLFSSL_TEST_SUBROUTINE int  openssl_evpSig_test(void);
#endif

WOLFSSL_TEST_SUBROUTINE int pbkdf1_test(void);
WOLFSSL_TEST_SUBROUTINE int pkcs12_test(void);
WOLFSSL_TEST_SUBROUTINE int pbkdf2_test(void);
WOLFSSL_TEST_SUBROUTINE int scrypt_test(void);
#ifdef HAVE_ECC
    WOLFSSL_TEST_SUBROUTINE int  ecc_test(void);
    #if defined(HAVE_ECC_ENCRYPT) && defined(HAVE_AES_CBC) && \
        (defined(WOLFSSL_AES_128) || defined(WOLFSSL_AES_256))
        WOLFSSL_TEST_SUBROUTINE int  ecc_encrypt_test(void);
    #endif
    #if defined(USE_CERT_BUFFERS_256) && !defined(WOLFSSL_ATECC508A) && \
        !defined(WOLFSSL_ATECC608A) && !defined(NO_ECC256) && \
        defined(HAVE_ECC_VERIFY) && defined(HAVE_ECC_SIGN) && \
        !defined(WOLF_CRYPTO_CB_ONLY_ECC)
        /* skip for ATECC508/608A, cannot import private key buffers */
        WOLFSSL_TEST_SUBROUTINE int ecc_test_buffers(void);
    #endif
#endif
#ifdef HAVE_CURVE25519
    WOLFSSL_TEST_SUBROUTINE int  curve25519_test(void);
#endif
#ifdef HAVE_ED25519
    WOLFSSL_TEST_SUBROUTINE int  ed25519_test(void);
#endif
#ifdef HAVE_CURVE448
    WOLFSSL_TEST_SUBROUTINE int  curve448_test(void);
#endif
#ifdef HAVE_ED448
    WOLFSSL_TEST_SUBROUTINE int  ed448_test(void);
#endif
#ifdef WOLFCRYPT_HAVE_ECCSI
    WOLFSSL_TEST_SUBROUTINE int  eccsi_test(void);
#endif
#ifdef WOLFCRYPT_HAVE_SAKKE
    WOLFSSL_TEST_SUBROUTINE int  sakke_test(void);
#endif
#ifdef HAVE_BLAKE2
    WOLFSSL_TEST_SUBROUTINE int  blake2b_test(void);
#endif
#ifdef HAVE_BLAKE2S
    WOLFSSL_TEST_SUBROUTINE int  blake2s_test(void);
#endif
#ifdef HAVE_LIBZ
    WOLFSSL_TEST_SUBROUTINE int compress_test(void);
#endif
#ifdef HAVE_PKCS7
    #ifndef NO_PKCS7_ENCRYPTED_DATA
        WOLFSSL_TEST_SUBROUTINE int pkcs7encrypted_test(void);
    #endif
    #if defined(HAVE_LIBZ) && !defined(NO_PKCS7_COMPRESSED_DATA)
        WOLFSSL_TEST_SUBROUTINE int pkcs7compressed_test(void);
    #endif
    WOLFSSL_TEST_SUBROUTINE int pkcs7signed_test(void);
    WOLFSSL_TEST_SUBROUTINE int pkcs7enveloped_test(void);
    #if defined(HAVE_AESGCM) || defined(HAVE_AESCCM)
        WOLFSSL_TEST_SUBROUTINE int pkcs7authenveloped_test(void);
    #endif
    #if !defined(NO_AES) && defined(HAVE_AES_CBC)
        WOLFSSL_TEST_SUBROUTINE int pkcs7callback_test(byte* cert, word32 certSz, byte* key,
                word32 keySz);
    #endif
#endif
#if !defined(NO_ASN_TIME) && !defined(NO_RSA) && defined(WOLFSSL_TEST_CERT) && \
    !defined(NO_FILESYSTEM)
WOLFSSL_TEST_SUBROUTINE int cert_test(void);
#endif
#if defined(WOLFSSL_CERT_EXT) && defined(WOLFSSL_TEST_CERT) && \
   !defined(NO_FILESYSTEM) && defined(WOLFSSL_CERT_GEN)
WOLFSSL_TEST_SUBROUTINE int  certext_test(void);
#endif
#if defined(WOLFSSL_CERT_GEN_CACHE) && defined(WOLFSSL_TEST_CERT) && \
    defined(WOLFSSL_CERT_EXT) && defined(WOLFSSL_CERT_GEN)
WOLFSSL_TEST_SUBROUTINE int decodedCertCache_test(void);
#endif
WOLFSSL_TEST_SUBROUTINE int memory_test(void);
#ifdef HAVE_VALGRIND
WOLFSSL_TEST_SUBROUTINE int mp_test(void);
#endif
#if defined(WOLFSSL_PUBLIC_MP) && defined(WOLFSSL_KEY_GEN)
WOLFSSL_TEST_SUBROUTINE int prime_test(void);
#endif
#if defined(ASN_BER_TO_DER) && \
    (defined(WOLFSSL_TEST_CERT) || defined(OPENSSL_EXTRA) || \
     defined(OPENSSL_EXTRA_X509_SMALL))
WOLFSSL_TEST_SUBROUTINE int berder_test(void);
#endif
WOLFSSL_TEST_SUBROUTINE int logging_test(void);
#if !defined(NO_ASN) && !defined(NO_ASN_TIME)
WOLFSSL_TEST_SUBROUTINE int time_test(void);
#endif
WOLFSSL_TEST_SUBROUTINE int mutex_test(void);
#if defined(USE_WOLFSSL_MEMORY) && !defined(FREERTOS)
WOLFSSL_TEST_SUBROUTINE int memcb_test(void);
#endif
#ifdef WOLFSSL_IMX6_CAAM_BLOB
WOLFSSL_TEST_SUBROUTINE int blob_test(void);
#endif

#ifdef WOLF_CRYPTO_CB
WOLFSSL_TEST_SUBROUTINE int cryptocb_test(void);
#endif
#ifdef WOLFSSL_CERT_PIV
WOLFSSL_TEST_SUBROUTINE int certpiv_test(void);
#endif
#ifdef WOLFSSL_AES_SIV
WOLFSSL_TEST_SUBROUTINE int aes_siv_test(void);
#endif

/* General big buffer size for many tests. */
#define FOURK_BUF 4096


#define ERROR_OUT(err, eLabel) do { ret = (err); goto eLabel; } while (0)

#ifdef HAVE_STACK_SIZE
static THREAD_RETURN err_sys(const char* msg, int es)
#else
static int err_sys(const char* msg, int es)
#endif
{
    (void)msg;
    (void)es;

#ifdef WOLFSSL_LINUXKM
    lkm_printf("%s error = %d\n", msg, es);
    EXIT_TEST(es);
#else
    printf("%s error = %d\n", msg, es);
    EXIT_TEST(-1);
#endif
}

#ifndef HAVE_WOLFCRYPT_TEST_OPTIONS
/* func_args from test.h, so don't have to pull in other stuff */
typedef struct func_args {
    int    argc;
    char** argv;
    int    return_code;
} func_args;
#endif /* !HAVE_WOLFCRYPT_TEST_OPTIONS */

#if defined(HAVE_FIPS) && !defined(WOLFSSL_LINUXKM)
static void myFipsCb(int ok, int err, const char* hash)
{
    printf("in my Fips callback, ok = %d, err = %d\n", ok, err);
    printf("message = %s\n", wc_GetErrorString(err));
    printf("hash = %s\n", hash);

    if (err == IN_CORE_FIPS_E) {
        printf("In core integrity hash check failure, copy above hash\n");
        printf("into verifyCore[] in fips_test.c and rebuild\n");
    }
}
#endif /* HAVE_FIPS && !WOLFSSL_LINUXKM */

#ifdef WOLFSSL_STATIC_MEMORY
    #ifdef BENCH_EMBEDDED
        static byte gTestMemory[14000];
    #elif defined(WOLFSSL_CERT_EXT)
        static byte gTestMemory[140000];
    #elif (defined(WOLFSSL_SP_MATH_ALL) || defined(USE_FAST_MATH)) && \
          !defined(ALT_ECC_SIZE)
        static byte gTestMemory[160000];
    #else
        static byte gTestMemory[80000];
    #endif
#endif

#ifdef WOLFSSL_PB
static int wolfssl_pb_print(const char* msg, ...)
{
    int ret;
    va_list args;
    char tmpBuf[80];

    va_start(args, msg);
    ret = vsprint(tmpBuf, msg, args);
    va_end(args);

    fnDumpStringToSystemLog(tmpBuf);

    return ret;
}
#endif /* WOLFSSL_PB */

/* optional macro to add sleep between tests */
#ifndef TEST_SLEEP
#define TEST_SLEEP()
#else
    #define TEST_PASS test_pass
    #include <stdarg.h> /* for var args */
    static WC_INLINE void test_pass(const char* fmt, ...)
    {
        va_list args;
        va_start(args, fmt);
        STACK_SIZE_CHECKPOINT_WITH_MAX_CHECK(max_relative_stack, vprintf(fmt, args));
        va_end(args);
        PRINT_HEAP_CHECKPOINT();
        TEST_SLEEP();
        ASSERT_RESTORED_VECTOR_REGISTERS(exit(1););
    }
#endif

/* set test pass output to printf if not overriden */
#ifndef TEST_PASS
    /* redirect to printf */
    #define TEST_PASS(...) {                                    \
        if (STACK_SIZE_CHECKPOINT_WITH_MAX_CHECK                \
            (max_relative_stack, printf(__VA_ARGS__)) < 0) {    \
            return err_sys("post-test check failed", -1);       \
        }                                                       \
        PRINT_HEAP_CHECKPOINT();                                \
        ASSERT_RESTORED_VECTOR_REGISTERS(exit(1););             \
    }
#endif

#ifdef HAVE_STACK_SIZE
THREAD_RETURN WOLFSSL_THREAD wolfcrypt_test(void* args)
#else
int wolfcrypt_test(void* args)
#endif
{
    int ret;
#ifdef WOLFSSL_TRACK_MEMORY_VERBOSE
    long heap_baselineAllocs, heap_baselineBytes;
#endif
    STACK_SIZE_INIT();

#ifdef WOLFSSL_TRACK_MEMORY_VERBOSE
    (void)wolfCrypt_heap_peakAllocs_checkpoint();
    heap_baselineAllocs = wolfCrypt_heap_peakAllocs_checkpoint();
    (void)wolfCrypt_heap_peakBytes_checkpoint();
    heap_baselineBytes = wolfCrypt_heap_peakBytes_checkpoint();
#endif

    printf("------------------------------------------------------------------------------\n");
    printf(" wolfSSL version %s\n", LIBWOLFSSL_VERSION_STRING);
    printf("------------------------------------------------------------------------------\n");

    if (args) {
#ifdef HAVE_WOLFCRYPT_TEST_OPTIONS
        int ch;
#endif
        ((func_args*)args)->return_code = -1; /* error state */
#ifdef HAVE_WOLFCRYPT_TEST_OPTIONS
        while ((ch = mygetopt(((func_args*)args)->argc, ((func_args*)args)->argv, "s:m:a:h")) != -1) {
            switch(ch) {
            case 's':
#ifdef HAVE_STACK_SIZE_VERBOSE
                max_relative_stack = (ssize_t)atoi(myoptarg);
                break;
#else
                return err_sys("-s (max relative stack bytes) requires HAVE_STACK_SIZE_VERBOSE (--enable-stacksize=verbose).", -1);
#endif
            case 'm':
#ifdef WOLFSSL_TRACK_MEMORY_VERBOSE
                max_relative_heap_bytes = (ssize_t)atoi(myoptarg);
                break;
#else
                return err_sys("-m (max relative heap memory bytes) requires WOLFSSL_TRACK_MEMORY_VERBOSE (--enable-trackmemory=verbose).", -1);
#endif
            case 'a':
#ifdef WOLFSSL_TRACK_MEMORY_VERBOSE
                max_relative_heap_allocs = (ssize_t)atoi(myoptarg);
                break;
#else
                return err_sys("-a (max relative heap allocs) requires WOLFSSL_TRACK_MEMORY_VERBOSE (--enable-trackmemory=verbose).", -1);
#endif
            case 'h':
                return err_sys("\
options: [-s max_relative_stack_bytes] [-m max_relative_heap_memory_bytes]\n\
         [-a max_relative_heap_allocs] [-h]\n", 0);
            default:
                return err_sys("unknown test option.  try -h.", -1);
            }
        }
#endif
    }

#ifdef WOLFSSL_STATIC_MEMORY
    if (wc_LoadStaticMemory(&HEAP_HINT, gTestMemory, sizeof(gTestMemory),
                                                WOLFMEM_GENERAL, 1) != 0) {
        printf("unable to load static memory.\n");
        return(EXIT_FAILURE);
    }
#endif

#if defined(DEBUG_WOLFSSL) && !defined(HAVE_VALGRIND)
    wolfSSL_Debugging_ON();
#endif

#if defined(OPENSSL_EXTRA) || defined(DEBUG_WOLFSSL_VERBOSE)
    wc_SetLoggingHeap(HEAP_HINT);
#endif

#if defined(HAVE_FIPS) && !defined(WOLFSSL_LINUXKM)
    wolfCrypt_SetCb_fips(myFipsCb);
#endif

#if !defined(NO_BIG_INT)
    if (CheckCtcSettings() != 1) {
        printf("Sizeof mismatch (build) %x != (run) %lx\n",
               CTC_SETTINGS, (unsigned long)CheckRunTimeSettings());
        return err_sys("Build vs runtime math mismatch\n", -1000);
    }

#if defined(USE_FAST_MATH) && \
        (!defined(NO_RSA) || !defined(NO_DH) || defined(HAVE_ECC))
    if (CheckFastMathSettings() != 1)
        return err_sys("Build vs runtime fastmath FP_MAX_BITS mismatch\n",
                       -1001);
#endif /* USE_FAST_MATH */
#endif /* !NO_BIG_INT */

#if defined(WOLFSSL_CERT_GEN) && (!defined(NO_RSA) || defined(HAVE_ECC)) || \
  (defined(WOLFSSL_TEST_CERT) && (defined(HAVE_ED25519) || defined(HAVE_ED448)))
    initDefaultName();
#endif

#ifdef WOLFSSL_ASYNC_CRYPT
    ret = wolfAsync_DevOpen(&devId);
    if (ret < 0) {
        printf("Async device open failed\nRunning without async\n");
    }
#else
    (void)devId;
#endif /* WOLFSSL_ASYNC_CRYPT */

#ifdef WOLF_CRYPTO_CB
#ifdef HAVE_INTEL_QA_SYNC
    devId = wc_CryptoCb_InitIntelQa();
    if (INVALID_DEVID == devId) {
        printf("Couldn't init the Intel QA\n");
    }
#endif
#ifdef HAVE_CAVIUM_OCTEON_SYNC
    devId = wc_CryptoCb_InitOcteon();
    if (INVALID_DEVID == devId) {
        printf("Couldn't init the Cavium Octeon\n");
    }
#endif
#endif

#ifdef HAVE_SELFTEST
    if ( (ret = wolfCrypt_SelfTest()) != 0)
        return err_sys("CAVP selftest failed!\n", ret);
    else
        TEST_PASS("CAVP selftest passed!\n");
#endif

    if ( (ret = error_test()) != 0)
        return err_sys("error    test failed!\n", ret);
    else
        TEST_PASS("error    test passed!\n");

    if ( (ret = memory_test()) != 0)
        return err_sys("MEMORY   test failed!\n", ret);
    else
        TEST_PASS("MEMORY   test passed!\n");

#ifndef NO_CODING
    if ( (ret = base64_test()) != 0)
        return err_sys("base64   test failed!\n", ret);
    else
        TEST_PASS("base64   test passed!\n");
#ifdef WOLFSSL_BASE16
    if ( (ret = base16_test()) != 0)
        return err_sys("base16   test failed!\n", ret);
    else
        TEST_PASS("base16   test passed!\n");
#endif
#endif /* !NO_CODING */

#ifndef NO_ASN
    if ( (ret = asn_test()) != 0)
        return err_sys("asn      test failed!\n", ret);
    else
        TEST_PASS("asn      test passed!\n");
#endif

#ifndef WC_NO_RNG
    if ( (ret = random_test()) != 0)
        return err_sys("RANDOM   test failed!\n", ret);
    else
        TEST_PASS("RANDOM   test passed!\n");
#endif /* WC_NO_RNG */

#ifndef NO_MD5
    if ( (ret = md5_test()) != 0)
        return err_sys("MD5      test failed!\n", ret);
    else
        TEST_PASS("MD5      test passed!\n");
#endif

#ifdef WOLFSSL_MD2
    if ( (ret = md2_test()) != 0)
        return err_sys("MD2      test failed!\n", ret);
    else
        TEST_PASS("MD2      test passed!\n");
#endif

#ifndef NO_MD4
    if ( (ret = md4_test()) != 0)
        return err_sys("MD4      test failed!\n", ret);
    else
        TEST_PASS("MD4      test passed!\n");
#endif

#ifndef NO_SHA
    if ( (ret = sha_test()) != 0)
        return err_sys("SHA      test failed!\n", ret);
    else
        TEST_PASS("SHA      test passed!\n");
#endif

#ifdef WOLFSSL_SHA224
    if ( (ret = sha224_test()) != 0)
        return err_sys("SHA-224  test failed!\n", ret);
    else
        TEST_PASS("SHA-224  test passed!\n");
#endif

#ifndef NO_SHA256
    if ( (ret = sha256_test()) != 0)
        return err_sys("SHA-256  test failed!\n", ret);
    else
        TEST_PASS("SHA-256  test passed!\n");
#endif

#ifdef WOLFSSL_SHA384
    if ( (ret = sha384_test()) != 0)
        return err_sys("SHA-384  test failed!\n", ret);
    else
        TEST_PASS("SHA-384  test passed!\n");
#endif

#ifdef WOLFSSL_SHA512
    if ( (ret = sha512_test()) != 0)
        return err_sys("SHA-512  test failed!\n", ret);
    else
        TEST_PASS("SHA-512  test passed!\n");
#endif

#ifdef WOLFSSL_SHA3
    if ( (ret = sha3_test()) != 0)
        return err_sys("SHA-3    test failed!\n", ret);
    else
        TEST_PASS("SHA-3    test passed!\n");
#endif

#ifdef WOLFSSL_SHAKE256
    if ( (ret = shake256_test()) != 0)
        return err_sys("SHAKE256 test failed!\n", ret);
    else
        TEST_PASS("SHAKE256 test passed!\n");
#endif

#ifndef NO_HASH_WRAPPER
    if ( (ret = hash_test()) != 0)
        return err_sys("Hash     test failed!\n", ret);
    else
        TEST_PASS("Hash     test passed!\n");
#endif

#ifdef WOLFSSL_RIPEMD
    if ( (ret = ripemd_test()) != 0)
        return err_sys("RIPEMD   test failed!\n", ret);
    else
        TEST_PASS("RIPEMD   test passed!\n");
#endif

#ifdef HAVE_BLAKE2
    if ( (ret = blake2b_test()) != 0)
        return err_sys("BLAKE2b  test failed!\n", ret);
    else
        TEST_PASS("BLAKE2b  test passed!\n");
#endif
#ifdef HAVE_BLAKE2S
    if ( (ret = blake2s_test()) != 0)
        return err_sys("BLAKE2s  test failed!\n", ret);
    else
        TEST_PASS("BLAKE2s  test passed!\n");
#endif

#ifndef NO_HMAC
    #if !defined(NO_MD5) && !(defined(HAVE_FIPS) && defined(HAVE_FIPS_VERSION) \
                              && (HAVE_FIPS_VERSION >= 5))
        if ( (ret = hmac_md5_test()) != 0)
            return err_sys("HMAC-MD5 test failed!\n", ret);
        else
            TEST_PASS("HMAC-MD5 test passed!\n");
    #endif

    #ifndef NO_SHA
    if ( (ret = hmac_sha_test()) != 0)
        return err_sys("HMAC-SHA test failed!\n", ret);
    else
        TEST_PASS("HMAC-SHA test passed!\n");
    #endif

    #ifdef WOLFSSL_SHA224
        if ( (ret = hmac_sha224_test()) != 0)
            return err_sys("HMAC-SHA224 test failed!\n", ret);
        else
            TEST_PASS("HMAC-SHA224 test passed!\n");
    #endif

    #ifndef NO_SHA256
        if ( (ret = hmac_sha256_test()) != 0)
            return err_sys("HMAC-SHA256 test failed!\n", ret);
        else
            TEST_PASS("HMAC-SHA256 test passed!\n");
    #endif

    #ifdef WOLFSSL_SHA384
        if ( (ret = hmac_sha384_test()) != 0)
            return err_sys("HMAC-SHA384 test failed!\n", ret);
        else
            TEST_PASS("HMAC-SHA384 test passed!\n");
    #endif

    #ifdef WOLFSSL_SHA512
        if ( (ret = hmac_sha512_test()) != 0)
            return err_sys("HMAC-SHA512 test failed!\n", ret);
        else
            TEST_PASS("HMAC-SHA512 test passed!\n");
    #endif

    #if !defined(NO_HMAC) && defined(WOLFSSL_SHA3) && \
      !defined(WOLFSSL_NOSHA3_224) && !defined(WOLFSSL_NOSHA3_256) && \
      !defined(WOLFSSL_NOSHA3_384) && !defined(WOLFSSL_NOSHA3_512)
        if ( (ret = hmac_sha3_test()) != 0)
            return err_sys("HMAC-SHA3   test failed!\n", ret);
        else
            TEST_PASS("HMAC-SHA3   test passed!\n");
    #endif

    #if defined(HAVE_HKDF) && !defined(NO_HMAC)
        PRIVATE_KEY_UNLOCK();
        if ( (ret = hkdf_test()) != 0)
            return err_sys("HMAC-KDF    test failed!\n", ret);
        else
            TEST_PASS("HMAC-KDF    test passed!\n");
        PRIVATE_KEY_LOCK();
    #endif
#endif /* !NO_HMAC */

#ifdef WOLFSSL_WOLFSSH
    PRIVATE_KEY_UNLOCK();
    if ( (ret = sshkdf_test()) != 0)
        return err_sys("SSH-KDF     test failed!\n", ret);
    else
        TEST_PASS("SSH-KDF     test passed!\n");
    PRIVATE_KEY_LOCK();
#endif /* WOLFSSL_WOLFSSH */

#ifdef WOLFSSL_TLS13
    PRIVATE_KEY_UNLOCK();
    if ( (ret = tls13_kdf_test()) != 0)
        return err_sys("TLSv1.3 KDF test failed!\n", ret);
    else
        TEST_PASS("TLSv1.3 KDF test passed!\n");
    PRIVATE_KEY_LOCK();
#endif /* WOLFSSL_TLS13 */

#if defined(HAVE_X963_KDF) && defined(HAVE_ECC)
    if ( (ret = x963kdf_test()) != 0)
        return err_sys("X963-KDF    test failed!\n", ret);
    else
        TEST_PASS("X963-KDF    test passed!\n");
#endif

#if defined(HAVE_AESGCM) && defined(WOLFSSL_AES_128) && \
   !defined(WOLFSSL_AFALG_XILINX_AES) && !defined(WOLFSSL_XILINX_CRYPT)
    if ( (ret = gmac_test()) != 0)
        return err_sys("GMAC     test failed!\n", ret);
    else
        TEST_PASS("GMAC     test passed!\n");
#endif

#ifdef WC_RC2
    if ( (ret = rc2_test()) != 0)
        return err_sys("RC2      test failed!\n", ret);
    else
        TEST_PASS("RC2      test passed!\n");
#endif

#ifndef NO_RC4
    if ( (ret = arc4_test()) != 0)
        return err_sys("ARC4     test failed!\n", ret);
    else
        TEST_PASS("ARC4     test passed!\n");
#endif

#ifdef HAVE_CHACHA
    if ( (ret = chacha_test()) != 0)
        return err_sys("Chacha   test failed!\n", ret);
    else
        TEST_PASS("Chacha   test passed!\n");
#endif

#ifdef HAVE_XCHACHA
    if ( (ret = XChaCha_test()) != 0)
        return err_sys("XChacha  test failed!\n", ret);
    else
        TEST_PASS("XChacha  test passed!\n");
#endif

#ifdef HAVE_POLY1305
    if ( (ret = poly1305_test()) != 0)
        return err_sys("POLY1305 test failed!\n", ret);
    else
        TEST_PASS("POLY1305 test passed!\n");
#endif

#if defined(HAVE_CHACHA) && defined(HAVE_POLY1305)
    if ( (ret = chacha20_poly1305_aead_test()) != 0)
        return err_sys("ChaCha20-Poly1305 AEAD test failed!\n", ret);
    else
        TEST_PASS("ChaCha20-Poly1305 AEAD test passed!\n");
#endif

#if defined(HAVE_XCHACHA) && defined(HAVE_POLY1305)
    if ( (ret = XChaCha20Poly1305_test()) != 0)
        return err_sys("XChaCha20-Poly1305 AEAD test failed!\n", ret);
    else
        TEST_PASS("XChaCha20-Poly1305 AEAD test passed!\n");
#endif

#ifndef NO_DES3
    if ( (ret = des_test()) != 0)
        return err_sys("DES      test failed!\n", ret);
    else
        TEST_PASS("DES      test passed!\n");
#endif

#ifndef NO_DES3
    if ( (ret = des3_test()) != 0)
        return err_sys("DES3     test failed!\n", ret);
    else
        TEST_PASS("DES3     test passed!\n");
#endif

#ifndef NO_AES
    if ( (ret = aes_test()) != 0)
        return err_sys("AES      test failed!\n", ret);
    else
        TEST_PASS("AES      test passed!\n");

#ifdef WOLFSSL_AES_192
    if ( (ret = aes192_test()) != 0)
        return err_sys("AES192   test failed!\n", ret);
    else
        TEST_PASS("AES192   test passed!\n");
#endif

#ifdef WOLFSSL_AES_256
    if ( (ret = aes256_test()) != 0)
        return err_sys("AES256   test failed!\n", ret);
    else
        TEST_PASS("AES256   test passed!\n");
#endif

#ifdef WOLFSSL_AES_OFB
    if ( (ret = aesofb_test()) != 0)
        return err_sys("AES-OFB  test failed!\n", ret);
    else
        TEST_PASS("AESOFB   test passed!\n");
#endif

#ifdef HAVE_AESGCM
    #if !defined(WOLFSSL_AFALG) && !defined(WOLFSSL_DEVCRYPTO)
    if ( (ret = aesgcm_test()) != 0)
        return err_sys("AES-GCM  test failed!\n", ret);
    #endif
    #if !defined(WOLFSSL_AFALG_XILINX_AES) && !defined(WOLFSSL_XILINX_CRYPT) && \
        !defined(WOLFSSL_KCAPI_AES) && !(defined(WOLF_CRYPTO_CB) && \
            (defined(HAVE_INTEL_QA_SYNC) || defined(HAVE_CAVIUM_OCTEON_SYNC)))
    if ((ret = aesgcm_default_test()) != 0) {
        return err_sys("AES-GCM  test failed!\n", ret);
    }
    #endif
    if (ret == 0) {
        TEST_PASS("AES-GCM  test passed!\n");
    }
#endif

#if defined(HAVE_AESCCM) && defined(WOLFSSL_AES_128)
    if ( (ret = aesccm_test()) != 0)
        return err_sys("AES-CCM  test failed!\n", ret);
    else
        TEST_PASS("AES-CCM  test passed!\n");
#endif
#ifdef HAVE_AES_KEYWRAP
    if ( (ret = aeskeywrap_test()) != 0)
        return err_sys("AES Key Wrap test failed!\n", ret);
    else
        TEST_PASS("AES Key Wrap test passed!\n");
#endif
#ifdef WOLFSSL_AES_SIV
    if ( (ret = aes_siv_test()) != 0)
        return err_sys("AES-SIV  test failed!\n", ret);
    else
        TEST_PASS("AES-SIV  test passed!\n");
#endif
#endif

#ifdef HAVE_CAMELLIA
    if ( (ret = camellia_test()) != 0)
        return err_sys("CAMELLIA test failed!\n", ret);
    else
        TEST_PASS("CAMELLIA test passed!\n");
#endif

#if !defined(NO_RSA)
    #ifdef WC_RSA_NO_PADDING
    if ( (ret = rsa_no_pad_test()) != 0)
        return err_sys("RSA NOPAD test failed!\n", ret);
    else
        TEST_PASS("RSA NOPAD test passed!\n");
    #endif
    if ( (ret = rsa_test()) != 0)
        return err_sys("RSA      test failed!\n", ret);
    else
        TEST_PASS("RSA      test passed!\n");
#endif

#ifndef NO_DH
    PRIVATE_KEY_UNLOCK();
    if ( (ret = dh_test()) != 0)
        return err_sys("DH       test failed!\n", ret);
    else
        TEST_PASS("DH       test passed!\n");
    PRIVATE_KEY_LOCK();
#endif

#ifndef NO_DSA
    if ( (ret = dsa_test()) != 0)
        return err_sys("DSA      test failed!\n", ret);
    else
        TEST_PASS("DSA      test passed!\n");
#endif

#ifdef WOLFCRYPT_HAVE_SRP
    if ( (ret = srp_test()) != 0)
        return err_sys("SRP      test failed!\n", ret);
    else
        TEST_PASS("SRP      test passed!\n");
#endif

#ifndef NO_PWDBASED
    if ( (ret = pwdbased_test()) != 0)
        return err_sys("PWDBASED test failed!\n", ret);
    else
        TEST_PASS("PWDBASED test passed!\n");
#endif

#if defined(OPENSSL_EXTRA) && !defined(WOLFCRYPT_ONLY)
    if ( (ret = openssl_test()) != 0)
        return err_sys("OPENSSL  test failed!\n", ret);
    else
        TEST_PASS("OPENSSL  test passed!\n");

    if ( (ret = openSSL_evpMD_test()) != 0)
        return err_sys("OPENSSL (EVP MD) test failed!\n", ret);
    else
        TEST_PASS("OPENSSL (EVP MD) passed!\n");

    if ( (ret = openssl_pkey0_test()) != 0)
        return err_sys("OPENSSL (PKEY0) test failed!\n", ret);
    else
        TEST_PASS("OPENSSL (PKEY0) passed!\n");

    if ( (ret = openssl_pkey1_test()) != 0)
        return err_sys("OPENSSL (PKEY1) test failed!\n", ret);
    else
        TEST_PASS("OPENSSL (PKEY1) passed!\n");

    #if !defined(WOLF_CRYPTO_CB_ONLY_RSA) && !defined(WOLF_CRYPTO_CB_ONLY_ECC)
    if ( (ret = openssl_evpSig_test()) != 0)
        return err_sys("OPENSSL (EVP Sign/Verify) test failed!\n", ret);
    else
        TEST_PASS("OPENSSL (EVP Sign/Verify) passed!\n");
    #endif
#endif

#if defined(HAVE_ECC)
    PRIVATE_KEY_UNLOCK();
    if ( (ret = ecc_test()) != 0)
        return err_sys("ECC      test failed!\n", ret);
    else
        TEST_PASS("ECC      test passed!\n");
    PRIVATE_KEY_LOCK();
    #if defined(HAVE_ECC_ENCRYPT) && defined(HAVE_AES_CBC) && \
        (defined(WOLFSSL_AES_128) || defined(WOLFSSL_AES_256))
        if ( (ret = ecc_encrypt_test()) != 0)
            return err_sys("ECC Enc  test failed!\n", ret);
        else
            TEST_PASS("ECC Enc  test passed!\n");
    #endif
    #if defined(USE_CERT_BUFFERS_256) && !defined(WOLFSSL_ATECC508A) && \
        !defined(WOLFSSL_ATECC608A) && !defined(NO_ECC256) && \
        defined(HAVE_ECC_VERIFY) && defined(HAVE_ECC_SIGN) && \
        !defined(WOLF_CRYPTO_CB_ONLY_ECC)
        /* skip for ATECC508/608A, cannot import private key buffers */
        if ( (ret = ecc_test_buffers()) != 0)
            return err_sys("ECC buffer test failed!\n", ret);
        else
            TEST_PASS("ECC buffer test passed!\n");
    #endif
#endif

#if !defined(NO_ASN_TIME) && !defined(NO_RSA) && defined(WOLFSSL_TEST_CERT) && \
    !defined(NO_FILESYSTEM)
    if ( (ret = cert_test()) != 0)
        return err_sys("CERT     test failed!\n", ret);
    else
        TEST_PASS("CERT     test passed!\n");
#endif

#if defined(WOLFSSL_CERT_EXT) && defined(WOLFSSL_TEST_CERT) && \
   !defined(NO_FILESYSTEM) && !defined(NO_RSA) && defined(WOLFSSL_GEN_CERT)
    if ( (ret = certext_test()) != 0)
        return err_sys("CERT EXT test failed!\n", ret);
    else
        TEST_PASS("CERT EXT test passed!\n");
#endif

#if defined(WOLFSSL_CERT_GEN_CACHE) && defined(WOLFSSL_TEST_CERT) && \
    defined(WOLFSSL_CERT_EXT) && defined(WOLFSSL_CERT_GEN)
    if ( (ret = decodedCertCache_test()) != 0)
        return err_sys("DECODED CERT CACHE test failed!\n", ret);
    else
        TEST_PASS("DECODED CERT CACHE test passed!\n");
#endif

#ifdef HAVE_CURVE25519
    if ( (ret = curve25519_test()) != 0)
        return err_sys("CURVE25519 test failed!\n", ret);
    else
        TEST_PASS("CURVE25519 test passed!\n");
#endif

#ifdef HAVE_ED25519
    if ( (ret = ed25519_test()) != 0)
        return err_sys("ED25519  test failed!\n", ret);
    else
        TEST_PASS("ED25519  test passed!\n");
#endif

#ifdef HAVE_CURVE448
    if ( (ret = curve448_test()) != 0)
        return err_sys("CURVE448 test failed!\n", ret);
    else
        TEST_PASS("CURVE448 test passed!\n");
#endif

#ifdef HAVE_ED448
    if ( (ret = ed448_test()) != 0)
        return err_sys("ED448    test failed!\n", ret);
    else
        TEST_PASS("ED448    test passed!\n");
#endif

#ifdef WOLFCRYPT_HAVE_ECCSI
    if ( (ret = eccsi_test()) != 0)
        return err_sys("ECCSI    test failed!\n", ret);
    else
        TEST_PASS("ECCSI    test passed!\n");
#endif
#ifdef WOLFCRYPT_HAVE_SAKKE
    if ( (ret = sakke_test()) != 0)
        return err_sys("SAKKE    test failed!\n", ret);
    else
        TEST_PASS("SAKKE    test passed!\n");
#endif

#if defined(WOLFSSL_CMAC) && !defined(NO_AES)
    if ( (ret = cmac_test()) != 0)
        return err_sys("CMAC     test failed!\n", ret);
    else
        TEST_PASS("CMAC     test passed!\n");
#endif

#if defined(WOLFSSL_SIPHASH)
    if ( (ret = siphash_test()) != 0)
        return err_sys("SipHash  test failed!\n", ret);
    else
        TEST_PASS("SipHash  test passed!\n");
#endif

#ifdef HAVE_LIBZ
    if ( (ret = compress_test()) != 0)
        return err_sys("COMPRESS test failed!\n", ret);
    else
        TEST_PASS("COMPRESS test passed!\n");
#endif

#ifdef HAVE_PKCS7
    #ifndef NO_PKCS7_ENCRYPTED_DATA
        if ( (ret = pkcs7encrypted_test()) != 0)
            return err_sys("PKCS7encrypted  test failed!\n", ret);
        else
            TEST_PASS("PKCS7encrypted  test passed!\n");
    #endif
    #if defined(HAVE_LIBZ) && !defined(NO_PKCS7_COMPRESSED_DATA)
        if ( (ret = pkcs7compressed_test()) != 0)
            return err_sys("PKCS7compressed test failed!\n", ret);
        else
            TEST_PASS("PKCS7compressed test passed!\n");
    #endif
    if ( (ret = pkcs7signed_test()) != 0)
        return err_sys("PKCS7signed     test failed!\n", ret);
    else
        TEST_PASS("PKCS7signed     test passed!\n");

    if ( (ret = pkcs7enveloped_test()) != 0)
        return err_sys("PKCS7enveloped  test failed!\n", ret);
    else
        TEST_PASS("PKCS7enveloped  test passed!\n");

    #if defined(HAVE_AESGCM) || defined(HAVE_AESCCM)
        if ( (ret = pkcs7authenveloped_test()) != 0)
            return err_sys("PKCS7authenveloped  test failed!\n", ret);
        else
            TEST_PASS("PKCS7authenveloped  test passed!\n");
    #endif
#endif

#ifdef HAVE_VALGRIND
    if ( (ret = mp_test()) != 0)
        return err_sys("mp       test failed!\n", ret);
    else
        TEST_PASS("mp       test passed!\n");
#endif

#if defined(WOLFSSL_PUBLIC_MP) && defined(WOLFSSL_KEY_GEN)
    if ( (ret = prime_test()) != 0)
        return err_sys("prime    test failed!\n", ret);
    else
        TEST_PASS("prime    test passed!\n");
#endif

#if defined(ASN_BER_TO_DER) && \
    (defined(WOLFSSL_TEST_CERT) || defined(OPENSSL_EXTRA) || \
     defined(OPENSSL_EXTRA_X509_SMALL))
    if ( (ret = berder_test()) != 0)
        return err_sys("ber-der  test failed!\n", ret);
    else
        TEST_PASS("ber-der  test passed!\n");
#endif

    if ( (ret = logging_test()) != 0)
        return err_sys("logging  test failed!\n", ret);
    else
        TEST_PASS("logging  test passed!\n");

#if !defined(NO_ASN) && !defined(NO_ASN_TIME)
    if ( (ret = time_test()) != 0)
        return err_sys("time test failed!\n", ret);
    else
        TEST_PASS("time test passed!\n");
#endif

    if ( (ret = mutex_test()) != 0)
        return err_sys("mutex    test failed!\n", ret);
    else
        TEST_PASS("mutex    test passed!\n");

#if defined(USE_WOLFSSL_MEMORY) && !defined(FREERTOS)
    if ( (ret = memcb_test()) != 0)
        return err_sys("memcb    test failed!\n", ret);
    else
        TEST_PASS("memcb    test passed!\n");
#endif

#ifdef WOLFSSL_IMX6_CAAM_BLOB
    if ( (ret = blob_test()) != 0)
        return err_sys("blob     test failed!\n", ret);
    else
        TEST_PASS("blob     test passed!\n");
#endif

#if defined(WOLF_CRYPTO_CB) && \
    !(defined(HAVE_INTEL_QAT_SYNC) || defined(HAVE_CAVIUM_OCTEON_SYNC) || \
      defined(WOLFSSL_QNX_CAAM))
    if ( (ret = cryptocb_test()) != 0)
        return err_sys("crypto callback test failed!\n", ret);
    else
        TEST_PASS("crypto callback test passed!\n");
#endif

#ifdef WOLFSSL_CERT_PIV
    if ( (ret = certpiv_test()) != 0)
        return err_sys("cert piv test failed!\n", ret);
    else
        TEST_PASS("cert piv test passed!\n");
#endif

#ifdef WOLF_CRYPTO_CB
#ifdef HAVE_INTEL_QA_SYNC
    wc_CryptoCb_CleanupIntelQa(&devId);
#endif
#ifdef HAVE_CAVIUM_OCTEON_SYNC
    wc_CryptoCb_CleanupOcteon(&devId);
#endif
#endif

#ifdef WOLFSSL_ASYNC_CRYPT
    wolfAsync_DevClose(&devId);
#endif

    /* cleanup the thread if fixed point cache is enabled and have thread local */
#if defined(HAVE_THREAD_LS) && defined(HAVE_ECC) && defined(FP_ECC)
    wc_ecc_fp_free();
#endif

    if (args)
        ((func_args*)args)->return_code = ret;

    TEST_PASS("Test complete\n");

    EXIT_TEST(ret);
}

#ifndef NO_MAIN_DRIVER

    /* so overall tests can pull in test function */
#ifdef WOLFSSL_ESPIDF
    void app_main( )
#else

#ifdef HAVE_WOLFCRYPT_TEST_OPTIONS
    int myoptind = 0;
    char* myoptarg = NULL;
#endif

    int main(int argc, char** argv)
#endif
    {
        int ret;
        func_args args;
#ifdef WOLFSSL_ESPIDF
        /* set dummy wallclock time. */
        struct timeval utctime;
        struct timezone tz;
        utctime.tv_sec = 1521725159; /* dummy time: 2018-03-22T13:25:59+00:00 */
        utctime.tv_usec = 0;
        tz.tz_minuteswest = 0;
        tz.tz_dsttime = 0;
        settimeofday(&utctime, &tz);
#endif
#ifdef WOLFSSL_APACHE_MYNEWT
        #ifdef ARCH_sim
        mcu_sim_parse_args(argc, argv);
        #endif
        sysinit();

        /* set dummy wallclock time. */
        struct os_timeval utctime;
        struct os_timezone tz;
        utctime.tv_sec = 1521725159; /* dummy time: 2018-03-22T13:25:59+00:00 */
        utctime.tv_usec = 0;
        tz.tz_minuteswest = 0;
        tz.tz_dsttime = 0;
        os_settimeofday(&utctime, &tz);
#endif
#ifdef WOLFSSL_ZEPHYR
        /* set dummy wallclock time. */
        struct timespec utctime;
        utctime.tv_sec = 1521725159; /* dummy time: 2018-03-22T13:25:59+00:00 */
        utctime.tv_nsec = 0;
        clock_settime(CLOCK_REALTIME, &utctime);
#endif
#ifdef DEVKITPRO
        void *framebuffer;
        GXRModeObj *rmode = NULL;

        VIDEO_Init();
        WPAD_Init();

        rmode = VIDEO_GetPreferredMode(NULL);
#pragma GCC diagnostic ignored "-Wbad-function-cast"
        framebuffer = MEM_K0_TO_K1(SYS_AllocateFramebuffer(rmode));
#pragma GCC diagnostic pop
        console_init(framebuffer,20,20,rmode->fbWidth,rmode->xfbHeight,rmode->fbWidth*VI_DISPLAY_PIX_SZ);

        VIDEO_Configure(rmode);
        VIDEO_SetNextFramebuffer(framebuffer);
        VIDEO_SetBlack(FALSE);
        VIDEO_Flush();
        VIDEO_WaitVSync();
        if(rmode->viTVMode&VI_NON_INTERLACE) VIDEO_WaitVSync();
#endif

#ifdef HAVE_WNR
        if (wc_InitNetRandom(wnrConfigFile, NULL, 5000) != 0) {
            err_sys("Whitewood netRandom global config failed", -1001);
            return -1002;
        }
#endif
#ifndef WOLFSSL_ESPIDF
        args.argc = argc;
        args.argv = argv;
#endif
        if ((ret = wolfCrypt_Init()) != 0) {
            printf("wolfCrypt_Init failed %d\n", ret);
            err_sys("Error with wolfCrypt_Init!\n", -1003);
        }

#ifdef WC_RNG_SEED_CB
    wc_SetSeed_Cb(wc_GenerateSeed);
#endif

    #ifdef HAVE_STACK_SIZE
        StackSizeCheck(&args, wolfcrypt_test);
    #else
        wolfcrypt_test(&args);
    #endif

        if ((ret = wolfCrypt_Cleanup()) != 0) {
            printf("wolfCrypt_Cleanup failed %d\n", ret);
            err_sys("Error with wolfCrypt_Cleanup!\n", -1004);
        }

#ifdef HAVE_WNR
        if (wc_FreeNetRandom() < 0)
            err_sys("Failed to free netRandom context", -1005);
#endif /* HAVE_WNR */
#ifdef DOLPHIN_EMULATOR
        /* Returning from main panics the emulator. Just hang
         * and let the user force quit the emulator window. */
        printf("args.return_code: %d\n", args.return_code);
        printf("Testing complete. You may close the window now\n");
        while (1);
#endif
#ifndef WOLFSSL_ESPIDF
        printf("Exiting main with return code: %d\n", args.return_code);
        return args.return_code;
#endif
    }

#endif /* NO_MAIN_DRIVER */

/* helper to save DER, convert to PEM and save PEM */
#if !defined(NO_ASN) && (defined(HAVE_ECC) || !defined(NO_DSA) || \
(!defined(NO_RSA) && (defined(WOLFSSL_KEY_GEN) || defined(WOLFSSL_CERT_GEN)))) \
     && !defined(WOLF_CRYPTO_CB_ONLY_ECC)

#if !defined(NO_FILESYSTEM) && !defined(NO_WRITE_TEMP_FILES)
#define SaveDerAndPem(d, dSz, fD, fP, pT, eB) _SaveDerAndPem(d, dSz, fD, fP, pT, eB)
#else
#define SaveDerAndPem(d, dSz, fD, fP, pT, eB) _SaveDerAndPem(d, dSz, NULL, NULL, pT, eB)
#endif

static int _SaveDerAndPem(const byte* der, int derSz,
    const char* fileDer, const char* filePem, int pemType, int errBase)
{
#if !defined(NO_FILESYSTEM) && !defined(NO_WRITE_TEMP_FILES)
    int ret;
    XFILE derFile;

    derFile = XFOPEN(fileDer, "wb");
    if (!derFile) {
        return errBase + 0;
    }
    ret = (int)XFWRITE(der, 1, derSz, derFile);
    XFCLOSE(derFile);
    if (ret != derSz) {
        return errBase + 1;
    }
#endif

#ifdef WOLFSSL_DER_TO_PEM
    if (filePem) {
    #if !defined(NO_FILESYSTEM) && !defined(NO_WRITE_TEMP_FILES)
        XFILE pemFile;
    #endif
        byte* pem;
        int pemSz;

        /* calculate PEM size */
        pemSz = wc_DerToPem(der, derSz, NULL, 0, pemType);
        if (pemSz < 0) {
            return pemSz;
        }
        pem = (byte*)XMALLOC(pemSz, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
        if (pem == NULL) {
            return MEMORY_E;
        }
        /* Convert to PEM */
        pemSz = wc_DerToPem(der, derSz, pem, pemSz, pemType);
        if (pemSz < 0) {
            XFREE(pem, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
            return errBase + 2;
        }
    #if !defined(NO_FILESYSTEM) && !defined(NO_WRITE_TEMP_FILES)
        pemFile = XFOPEN(filePem, "wb");
        if (!pemFile) {
            XFREE(pem, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
            return errBase + 3;
        }
        ret = (int)XFWRITE(pem, 1, pemSz, pemFile);
        XFCLOSE(pemFile);
        if (ret != pemSz) {
            XFREE(pem, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
            return errBase + 4;
        }
    #endif
        XFREE(pem, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    }
#endif /* WOLFSSL_DER_TO_PEM */

    /* suppress unused variable warnings */
    (void)der;
    (void)derSz;
    (void)filePem;
    (void)fileDer;
    (void)pemType;
    (void)errBase;

    return 0;
}
#endif /* WOLFSSL_KEY_GEN || WOLFSSL_CERT_GEN */

WOLFSSL_TEST_SUBROUTINE int error_test(void)
{
    const char* errStr;
    char        out[WOLFSSL_MAX_ERROR_SZ];
    const char* unknownStr = wc_GetErrorString(0);

#ifdef NO_ERROR_STRINGS
    /* Ensure a valid error code's string matches an invalid code's.
     * The string is that error strings are not available.
     */
    errStr = wc_GetErrorString(OPEN_RAN_E);
    wc_ErrorString(OPEN_RAN_E, out);
    if (XSTRCMP(errStr, unknownStr) != 0)
        return -1100;
    if (XSTRCMP(out, unknownStr) != 0)
        return -1101;
#else
    int i;
    int j = 0;
    /* Values that are not or no longer error codes. */
    int missing[] = { -122, -123, -124,       -127, -128, -129, -159,
                      -163, -164, -165, -166, -167, -168, -169, -233,
                      0 };

    /* Check that all errors have a string and it's the same through the two
     * APIs. Check that the values that are not errors map to the unknown
     * string.
     */
    for (i = MAX_CODE_E-1; i >= WC_LAST_E; i--) {
        errStr = wc_GetErrorString(i);
        wc_ErrorString(i, out);

        if (i != missing[j]) {
            if (XSTRCMP(errStr, unknownStr) == 0)
                return -1102;
            if (XSTRCMP(out, unknownStr) == 0)
                return -1103;
            if (XSTRCMP(errStr, out) != 0)
                return -1104;
            if (XSTRLEN(errStr) >= WOLFSSL_MAX_ERROR_SZ)
                return -1105;
        }
        else {
            j++;
            if (XSTRCMP(errStr, unknownStr) != 0)
                return -1106;
            if (XSTRCMP(out, unknownStr) != 0)
                return -1107;
        }
    }

    /* Check if the next possible value has been given a string. */
    errStr = wc_GetErrorString(i);
    wc_ErrorString(i, out);
    if (XSTRCMP(errStr, unknownStr) != 0)
        return -1108;
    if (XSTRCMP(out, unknownStr) != 0)
        return -1109;
#endif

    return 0;
}

#ifndef NO_CODING

WOLFSSL_TEST_SUBROUTINE int base64_test(void)
{
    int        ret;
    WOLFSSL_SMALL_STACK_STATIC const byte good[] = "A+Gd\0\0\0";
    WOLFSSL_SMALL_STACK_STATIC const byte goodEnd[] = "A+Gd \r\n";
    WOLFSSL_SMALL_STACK_STATIC const byte good_spaces[] = " A + G d \0";
    byte       out[128];
    word32     outLen;
#ifdef WOLFSSL_BASE64_ENCODE
    byte       data[3];
    word32     dataLen;
    byte       longData[79] = { 0 };
    WOLFSSL_SMALL_STACK_STATIC const byte symbols[] = "+/A=";
#endif
    WOLFSSL_SMALL_STACK_STATIC const byte badSmall[] = "AAA!Gdj=";
    WOLFSSL_SMALL_STACK_STATIC const byte badLarge[] = "AAA~Gdj=";
    WOLFSSL_SMALL_STACK_STATIC const byte badEOL[] = "A+Gd!AA";
    WOLFSSL_SMALL_STACK_STATIC const byte badPadding[] = "AA=A";
    WOLFSSL_SMALL_STACK_STATIC const byte badChar[] = ",-.:;<=>?@[\\]^_`";
    byte goodChar[] =
        "ABCDEFGHIJKLMNOPQRSTUVWXYZ"
        "abcdefghijklmnopqrstuvwxyz"
        "0123456789+/;";
    byte charTest[] = "A+Gd\0\0\0";
    int        i;

    /* Good Base64 encodings. */
    outLen = sizeof(out);
    ret = Base64_Decode(good, sizeof(good), out, &outLen);
    if (ret != 0)
        return -1200;
    outLen = sizeof(out);
    ret = Base64_Decode(goodEnd, sizeof(goodEnd), out, &outLen);
    if (ret != 0)
        return -1201;
    outLen = sizeof(goodChar);
    ret = Base64_Decode(goodChar, sizeof(goodChar), goodChar, &outLen);
    if (ret != 0)
        return -1235;
    if (outLen != 64 / 4 * 3)
        return -1236;
    outLen = sizeof(out);
    ret = Base64_Decode(good_spaces, sizeof(good_spaces), out, &outLen);
    if (ret != 0)
        return -1201;

    /* Bad parameters. */
    outLen = 1;
    ret = Base64_Decode(good, sizeof(good), out, &outLen);
    if (ret != BAD_FUNC_ARG)
        return -1202;

    outLen = sizeof(out);
    ret = Base64_Decode(badEOL, sizeof(badEOL), out, &outLen);
    if (ret != ASN_INPUT_E)
        return -1203;
    outLen = sizeof(out);
    ret = Base64_Decode(badPadding, sizeof(badPadding), out, &outLen);
    if (ret != ASN_INPUT_E)
        return -1203;
    /* Bad character at each offset 0-3. */
    for (i = 0; i < 4; i++) {
        outLen = sizeof(out);
        ret = Base64_Decode(badSmall + i, 4, out, &outLen);
        if (ret != ASN_INPUT_E)
            return -1204 - i;
        ret = Base64_Decode(badLarge + i, 4, out, &outLen);
        if (ret != ASN_INPUT_E)
            return -1214 - i;
    }
    /* Invalid character less than 0x2b */
    for (i = 1; i < 0x2b; i++) {
        outLen = sizeof(out);
        charTest[0] = (byte)i;
        ret = Base64_Decode(charTest, sizeof(charTest), out, &outLen);
        if (ret != ASN_INPUT_E)
            return -1240 - i;
    }
    /* Bad characters in range 0x2b - 0x7a. */
    for (i = 0; i < (int)sizeof(badChar) - 1; i++) {
        outLen = sizeof(out);
        charTest[0] = badChar[i];
        ret = Base64_Decode(charTest, sizeof(charTest), out, &outLen);
        if (ret != ASN_INPUT_E)
            return -1270 - i;
    }
    /* Invalid character greater than 0x7a */
    for (i = 0x7b; i < 0x100; i++) {
        outLen = sizeof(out);
        charTest[0] = (byte)i;
        ret = Base64_Decode(charTest, sizeof(charTest), out, &outLen);
        if (ret != ASN_INPUT_E)
            return -1290 - i;
    }


#ifdef WOLFSSL_BASE64_ENCODE
    /* Decode and encode all symbols - non-alphanumeric. */
    dataLen = sizeof(data);
    ret = Base64_Decode(symbols, sizeof(symbols), data, &dataLen);
    if (ret != 0)
        return -1224;
    outLen = sizeof(out);
    ret = Base64_Encode(data, dataLen, NULL, &outLen);
    if (ret != LENGTH_ONLY_E)
        return -1225;
    outLen = sizeof(out);
    ret = Base64_Encode(data, dataLen, out, &outLen);
    if (ret != 0)
        return -1226;
    outLen = 7;
    ret = Base64_EncodeEsc(data, dataLen, out, &outLen);
    if (ret != BUFFER_E)
        return -1227;
    outLen = sizeof(out);
    ret = Base64_EncodeEsc(data, dataLen, NULL, &outLen);
    if (ret != LENGTH_ONLY_E)
        return -1228;
    outLen = sizeof(out);
    ret = Base64_EncodeEsc(data, dataLen, out, &outLen);
    if (ret != 0)
        return -1229;
    outLen = sizeof(out);
    ret = Base64_Encode_NoNl(data, dataLen, out, &outLen);
    if (ret != 0)
        return -1230;

    /* Data that results in an encoding longer than one line. */
    outLen = sizeof(out);
    dataLen = sizeof(longData);
    ret = Base64_Encode(longData, dataLen, out, &outLen);
    if (ret != 0)
        return -1231;
    outLen = sizeof(out);
    ret = Base64_EncodeEsc(longData, dataLen, out, &outLen);
    if (ret != 0)
        return -1232;
    outLen = sizeof(out);
    ret = Base64_Encode_NoNl(longData, dataLen, out, &outLen);
    if (ret != 0)
        return -1233;
#endif

    return 0;
}

#ifdef WOLFSSL_BASE16
WOLFSSL_TEST_SUBROUTINE int base16_test(void)
{
    int ret;
    WOLFSSL_SMALL_STACK_STATIC const byte testData[] = "SomeDataToEncode\n";
    WOLFSSL_SMALL_STACK_STATIC const byte encodedTestData[] = "536F6D6544617461546F456E636F64650A00";
    byte   encoded[40];
    word32 encodedLen;
    byte   plain[40];
    word32 len;

    /* length returned includes null termination */
    encodedLen = sizeof(encoded);
    ret = Base16_Encode(testData, sizeof(testData), encoded, &encodedLen);
    if (ret != 0)
        return -1300;

    len = (word32)XSTRLEN((char*)encoded);
    if (len != encodedLen - 1)
        return -1301;

    len = sizeof(plain);
    ret = Base16_Decode(encoded, encodedLen - 1, plain, &len);
    if (ret != 0)
        return -1302;

    if (len != sizeof(testData) || XMEMCMP(testData, plain, len) != 0)
        return -1303;

    if (encodedLen != sizeof(encodedTestData) ||
        XMEMCMP(encoded, encodedTestData, encodedLen) != 0) {
        return -1304;
    }

    return 0;
}
#endif /* WOLFSSL_BASE16 */
#endif /* !NO_CODING */

#ifndef NO_ASN
WOLFSSL_TEST_SUBROUTINE int asn_test(void)
{
    int ret;
    /* ASN1 encoded date buffer */
    WOLFSSL_SMALL_STACK_STATIC const byte dateBuf[] = {0x17, 0x0d, 0x31, 0x36, 0x30, 0x38, 0x31, 0x31,
                            0x32, 0x30, 0x30, 0x37, 0x33, 0x37, 0x5a};
    byte format;
    int length;
    const byte* datePart;
#ifndef NO_ASN_TIME
    struct tm timearg;
    time_t now;
#endif

    ret = wc_GetDateInfo(dateBuf, (int)sizeof(dateBuf), &datePart, &format,
                         &length);
    if (ret != 0)
        return -1400;

#ifndef NO_ASN_TIME
    /* Parameter Validation tests. */
    if (wc_GetTime(NULL, sizeof(now)) != BAD_FUNC_ARG)
        return -1401;
    if (wc_GetTime(&now, 0) != BUFFER_E)
        return -1402;

    now = 0;
    if (wc_GetTime(&now, sizeof(now)) != 0) {
        return -1403;
    }
    if (now == 0) {
        printf("RTC/Time not set!\n");
        return -1404;
    }

    ret = wc_GetDateAsCalendarTime(datePart, length, format, &timearg);
    if (ret != 0)
        return -1405;
#endif /* !NO_ASN_TIME */

    return 0;
}
#endif /* !NO_ASN */

#ifdef WOLFSSL_MD2
WOLFSSL_TEST_SUBROUTINE int md2_test(void)
{
    int ret = 0;
    Md2  md2;
    byte hash[MD2_DIGEST_SIZE];

    testVector a, b, c, d, e, f, g;
    testVector test_md2[7];
    int times = sizeof(test_md2) / sizeof(testVector), i;

    a.input  = "";
    a.output = "\x83\x50\xe5\xa3\xe2\x4c\x15\x3d\xf2\x27\x5c\x9f\x80\x69"
               "\x27\x73";
    a.inLen  = XSTRLEN(a.input);
    a.outLen = MD2_DIGEST_SIZE;

    b.input  = "a";
    b.output = "\x32\xec\x01\xec\x4a\x6d\xac\x72\xc0\xab\x96\xfb\x34\xc0"
               "\xb5\xd1";
    b.inLen  = XSTRLEN(b.input);
    b.outLen = MD2_DIGEST_SIZE;

    c.input  = "abc";
    c.output = "\xda\x85\x3b\x0d\x3f\x88\xd9\x9b\x30\x28\x3a\x69\xe6\xde"
               "\xd6\xbb";
    c.inLen  = XSTRLEN(c.input);
    c.outLen = MD2_DIGEST_SIZE;

    d.input  = "message digest";
    d.output = "\xab\x4f\x49\x6b\xfb\x2a\x53\x0b\x21\x9f\xf3\x30\x31\xfe"
               "\x06\xb0";
    d.inLen  = XSTRLEN(d.input);
    d.outLen = MD2_DIGEST_SIZE;

    e.input  = "abcdefghijklmnopqrstuvwxyz";
    e.output = "\x4e\x8d\xdf\xf3\x65\x02\x92\xab\x5a\x41\x08\xc3\xaa\x47"
               "\x94\x0b";
    e.inLen  = XSTRLEN(e.input);
    e.outLen = MD2_DIGEST_SIZE;

    f.input  = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz012345"
               "6789";
    f.output = "\xda\x33\xde\xf2\xa4\x2d\xf1\x39\x75\x35\x28\x46\xc3\x03"
               "\x38\xcd";
    f.inLen  = XSTRLEN(f.input);
    f.outLen = MD2_DIGEST_SIZE;

    g.input  = "1234567890123456789012345678901234567890123456789012345678"
               "9012345678901234567890";
    g.output = "\xd5\x97\x6f\x79\xd8\x3d\x3a\x0d\xc9\x80\x6c\x3c\x66\xf3"
               "\xef\xd8";
    g.inLen  = XSTRLEN(g.input);
    g.outLen = MD2_DIGEST_SIZE;

    test_md2[0] = a;
    test_md2[1] = b;
    test_md2[2] = c;
    test_md2[3] = d;
    test_md2[4] = e;
    test_md2[5] = f;
    test_md2[6] = g;

    wc_InitMd2(&md2);

    for (i = 0; i < times; ++i) {
        wc_Md2Update(&md2, (byte*)test_md2[i].input, (word32)test_md2[i].inLen);
        wc_Md2Final(&md2, hash);

        if (XMEMCMP(hash, test_md2[i].output, MD2_DIGEST_SIZE) != 0)
            return -1500 - i;
    }

    for (i = 0; i < times; ++i) {
        ret = wc_Md2Hash((byte*)test_md2[i].input, (word32)test_md2[i].inLen, hash);
         if (ret != 0) {
            return -1507 - i;
        }

        if (XMEMCMP(hash, test_md2[i].output, MD2_DIGEST_SIZE) != 0) {
            return -1507 - i;
        }

    }

    return 0;
}
#endif

#ifndef NO_MD5
WOLFSSL_TEST_SUBROUTINE int md5_test(void)
{
    int ret = 0;
    wc_Md5 md5, md5Copy;
    byte hash[WC_MD5_DIGEST_SIZE];
    byte hashcopy[WC_MD5_DIGEST_SIZE];
    testVector a, b, c, d, e, f;
    testVector test_md5[6];
    int times = sizeof(test_md5) / sizeof(testVector), i;

    a.input  = "";
    a.output = "\xd4\x1d\x8c\xd9\x8f\x00\xb2\x04\xe9\x80\x09\x98\xec\xf8\x42"
               "\x7e";
    a.inLen  = XSTRLEN(a.input);
    a.outLen = WC_MD5_DIGEST_SIZE;

    b.input  = "abc";
    b.output = "\x90\x01\x50\x98\x3c\xd2\x4f\xb0\xd6\x96\x3f\x7d\x28\xe1\x7f"
               "\x72";
    b.inLen  = XSTRLEN(b.input);
    b.outLen = WC_MD5_DIGEST_SIZE;

    c.input  = "message digest";
    c.output = "\xf9\x6b\x69\x7d\x7c\xb7\x93\x8d\x52\x5a\x2f\x31\xaa\xf1\x61"
               "\xd0";
    c.inLen  = XSTRLEN(c.input);
    c.outLen = WC_MD5_DIGEST_SIZE;

    d.input  = "abcdefghijklmnopqrstuvwxyz";
    d.output = "\xc3\xfc\xd3\xd7\x61\x92\xe4\x00\x7d\xfb\x49\x6c\xca\x67\xe1"
               "\x3b";
    d.inLen  = XSTRLEN(d.input);
    d.outLen = WC_MD5_DIGEST_SIZE;

    e.input  = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz012345"
               "6789";
    e.output = "\xd1\x74\xab\x98\xd2\x77\xd9\xf5\xa5\x61\x1c\x2c\x9f\x41\x9d"
               "\x9f";
    e.inLen  = XSTRLEN(e.input);
    e.outLen = WC_MD5_DIGEST_SIZE;

    f.input  = "1234567890123456789012345678901234567890123456789012345678"
               "9012345678901234567890";
    f.output = "\x57\xed\xf4\xa2\x2b\xe3\xc9\x55\xac\x49\xda\x2e\x21\x07\xb6"
               "\x7a";
    f.inLen  = XSTRLEN(f.input);
    f.outLen = WC_MD5_DIGEST_SIZE;

    test_md5[0] = a;
    test_md5[1] = b;
    test_md5[2] = c;
    test_md5[3] = d;
    test_md5[4] = e;
    test_md5[5] = f;

    ret = wc_InitMd5_ex(&md5, HEAP_HINT, devId);
    if (ret != 0)
        return -1600;
    ret = wc_InitMd5_ex(&md5Copy, HEAP_HINT, devId);
    if (ret != 0) {
        wc_Md5Free(&md5);
        return -1601;
    }

    for (i = 0; i < times; ++i) {
        ret = wc_Md5Update(&md5, (byte*)test_md5[i].input,
            (word32)test_md5[i].inLen);
        if (ret != 0)
            ERROR_OUT(-1602 - i, exit);

        ret = wc_Md5GetHash(&md5, hashcopy);
        if (ret != 0)
            ERROR_OUT(-1603 - i, exit);

        ret = wc_Md5Copy(&md5, &md5Copy);
        if (ret != 0)
            ERROR_OUT(-1604 - i, exit);

        ret = wc_Md5Final(&md5, hash);
        if (ret != 0)
            ERROR_OUT(-1605 - i, exit);

        wc_Md5Free(&md5Copy);

        if (XMEMCMP(hash, test_md5[i].output, WC_MD5_DIGEST_SIZE) != 0)
            ERROR_OUT(-1606 - i, exit);

        if (XMEMCMP(hash, hashcopy, WC_MD5_DIGEST_SIZE) != 0)
            ERROR_OUT(-1607 - i, exit);
    }

    /* BEGIN LARGE HASH TEST */ {
    byte large_input[1024];
    const char* large_digest =
        "\x44\xd0\x88\xce\xf1\x36\xd1\x78\xe9\xc8\xba\x84\xc3\xfd\xf6\xca";

    for (i = 0; i < (int)sizeof(large_input); i++) {
        large_input[i] = (byte)(i & 0xFF);
    }
    times = 100;
#ifdef WOLFSSL_PIC32MZ_HASH
    wc_Md5SizeSet(&md5, times * sizeof(large_input));
#endif
    for (i = 0; i < times; ++i) {
        ret = wc_Md5Update(&md5, (byte*)large_input,
            (word32)sizeof(large_input));
        if (ret != 0)
            ERROR_OUT(-1608, exit);
    }
    ret = wc_Md5Final(&md5, hash);
    if (ret != 0)
        ERROR_OUT(-1609, exit);
    if (XMEMCMP(hash, large_digest, WC_MD5_DIGEST_SIZE) != 0)
        ERROR_OUT(-1610, exit);
    } /* END LARGE HASH TEST */

exit:

    wc_Md5Free(&md5);
    wc_Md5Free(&md5Copy);

    return ret;
}
#endif /* NO_MD5 */


#ifndef NO_MD4

WOLFSSL_TEST_SUBROUTINE int md4_test(void)
{
    Md4  md4;
    byte hash[MD4_DIGEST_SIZE];

    testVector a, b, c, d, e, f, g;
    testVector test_md4[7];
    int times = sizeof(test_md4) / sizeof(testVector), i;

    a.input  = "";
    a.output = "\x31\xd6\xcf\xe0\xd1\x6a\xe9\x31\xb7\x3c\x59\xd7\xe0\xc0\x89"
               "\xc0";
    a.inLen  = XSTRLEN(a.input);
    a.outLen = MD4_DIGEST_SIZE;

    b.input  = "a";
    b.output = "\xbd\xe5\x2c\xb3\x1d\xe3\x3e\x46\x24\x5e\x05\xfb\xdb\xd6\xfb"
               "\x24";
    b.inLen  = XSTRLEN(b.input);
    b.outLen = MD4_DIGEST_SIZE;

    c.input  = "abc";
    c.output = "\xa4\x48\x01\x7a\xaf\x21\xd8\x52\x5f\xc1\x0a\xe8\x7a\xa6\x72"
               "\x9d";
    c.inLen  = XSTRLEN(c.input);
    c.outLen = MD4_DIGEST_SIZE;

    d.input  = "message digest";
    d.output = "\xd9\x13\x0a\x81\x64\x54\x9f\xe8\x18\x87\x48\x06\xe1\xc7\x01"
               "\x4b";
    d.inLen  = XSTRLEN(d.input);
    d.outLen = MD4_DIGEST_SIZE;

    e.input  = "abcdefghijklmnopqrstuvwxyz";
    e.output = "\xd7\x9e\x1c\x30\x8a\xa5\xbb\xcd\xee\xa8\xed\x63\xdf\x41\x2d"
               "\xa9";
    e.inLen  = XSTRLEN(e.input);
    e.outLen = MD4_DIGEST_SIZE;

    f.input  = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz012345"
               "6789";
    f.output = "\x04\x3f\x85\x82\xf2\x41\xdb\x35\x1c\xe6\x27\xe1\x53\xe7\xf0"
               "\xe4";
    f.inLen  = XSTRLEN(f.input);
    f.outLen = MD4_DIGEST_SIZE;

    g.input  = "1234567890123456789012345678901234567890123456789012345678"
               "9012345678901234567890";
    g.output = "\xe3\x3b\x4d\xdc\x9c\x38\xf2\x19\x9c\x3e\x7b\x16\x4f\xcc\x05"
               "\x36";
    g.inLen  = XSTRLEN(g.input);
    g.outLen = MD4_DIGEST_SIZE;

    test_md4[0] = a;
    test_md4[1] = b;
    test_md4[2] = c;
    test_md4[3] = d;
    test_md4[4] = e;
    test_md4[5] = f;
    test_md4[6] = g;

    wc_InitMd4(&md4);

    for (i = 0; i < times; ++i) {
        wc_Md4Update(&md4, (byte*)test_md4[i].input, (word32)test_md4[i].inLen);
        wc_Md4Final(&md4, hash);

        if (XMEMCMP(hash, test_md4[i].output, MD4_DIGEST_SIZE) != 0)
            return -1700 - i;
    }

    return 0;
}

#endif /* NO_MD4 */

#ifndef NO_SHA

WOLFSSL_TEST_SUBROUTINE int sha_test(void)
{
    int ret = 0;
    wc_Sha sha, shaCopy;
    byte hash[WC_SHA_DIGEST_SIZE];
    byte hashcopy[WC_SHA_DIGEST_SIZE];
    testVector a, b, c, d, e;
    testVector test_sha[5];
    int times = sizeof(test_sha) / sizeof(struct testVector), i;

    a.input  = "";
    a.output = "\xda\x39\xa3\xee\x5e\x6b\x4b\x0d\x32\x55\xbf\xef\x95\x60\x18"
               "\x90\xaf\xd8\x07\x09";
    a.inLen  = XSTRLEN(a.input);
    a.outLen = WC_SHA_DIGEST_SIZE;

    b.input  = "abc";
    b.output = "\xA9\x99\x3E\x36\x47\x06\x81\x6A\xBA\x3E\x25\x71\x78\x50\xC2"
               "\x6C\x9C\xD0\xD8\x9D";
    b.inLen  = XSTRLEN(b.input);
    b.outLen = WC_SHA_DIGEST_SIZE;

    c.input  = "abcdbcdecdefdefgefghfghighijhijkijkljklmklmnlmnomnopnopq";
    c.output = "\x84\x98\x3E\x44\x1C\x3B\xD2\x6E\xBA\xAE\x4A\xA1\xF9\x51\x29"
               "\xE5\xE5\x46\x70\xF1";
    c.inLen  = XSTRLEN(c.input);
    c.outLen = WC_SHA_DIGEST_SIZE;

    d.input  = "aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa"
               "aaaaaa";
    d.output = "\x00\x98\xBA\x82\x4B\x5C\x16\x42\x7B\xD7\xA1\x12\x2A\x5A\x44"
               "\x2A\x25\xEC\x64\x4D";
    d.inLen  = XSTRLEN(d.input);
    d.outLen = WC_SHA_DIGEST_SIZE;

    e.input  = "aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa"
               "aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa"
               "aaaaaaaaaa";
    e.output = "\xAD\x5B\x3F\xDB\xCB\x52\x67\x78\xC2\x83\x9D\x2F\x15\x1E\xA7"
               "\x53\x99\x5E\x26\xA0";
    e.inLen  = XSTRLEN(e.input);
    e.outLen = WC_SHA_DIGEST_SIZE;

    test_sha[0] = a;
    test_sha[1] = b;
    test_sha[2] = c;
    test_sha[3] = d;
    test_sha[4] = e;

    ret = wc_InitSha_ex(&sha, HEAP_HINT, devId);
    if (ret != 0)
        return -1800;
    ret = wc_InitSha_ex(&shaCopy, HEAP_HINT, devId);
    if (ret != 0) {
        wc_ShaFree(&sha);
        return -1801;
    }

    for (i = 0; i < times; ++i) {
        ret = wc_ShaUpdate(&sha, (byte*)test_sha[i].input,
            (word32)test_sha[i].inLen);
        if (ret != 0)
            ERROR_OUT(-1802 - i, exit);
        ret = wc_ShaGetHash(&sha, hashcopy);
        if (ret != 0)
            ERROR_OUT(-1803 - i, exit);
        ret = wc_ShaCopy(&sha, &shaCopy);
        if (ret != 0)
            ERROR_OUT(-1804 - i, exit);
        ret = wc_ShaFinal(&sha, hash);
        if (ret != 0)
            ERROR_OUT(-1805 - i, exit);
        wc_ShaFree(&shaCopy);

        if (XMEMCMP(hash, test_sha[i].output, WC_SHA_DIGEST_SIZE) != 0)
            ERROR_OUT(-1806 - i, exit);
        if (XMEMCMP(hash, hashcopy, WC_SHA_DIGEST_SIZE) != 0)
            ERROR_OUT(-1807 - i, exit);
    }

    /* BEGIN LARGE HASH TEST */ {
    byte large_input[1024];
#if defined(WOLFSSL_RENESAS_TSIP) || defined(WOLFSSL_RENESAS_SCEPROTECT)
    const char* large_digest =
            "\x1d\x6a\x5a\xf6\xe5\x7c\x86\xce\x7f\x7c\xaf\xd5\xdb\x08\xcd\x59"
            "\x15\x8c\x6d\xb6";
#else
    const char* large_digest =
           "\x8b\x77\x02\x48\x39\xe8\xdb\xd3\x9a\xf4\x05\x24\x66\x12\x2d\x9e"
           "\xc5\xd9\x0a\xac";
#endif
    for (i = 0; i < (int)sizeof(large_input); i++) {
        large_input[i] = (byte)(i & 0xFF);
    }
#if defined(WOLFSSL_RENESAS_TSIP) || defined(WOLFSSL_RENESAS_SCEPROTECT)
    times = 20;
#else
    times = 100;
#endif
#ifdef WOLFSSL_PIC32MZ_HASH
    wc_ShaSizeSet(&sha, times * sizeof(large_input));
#endif
    for (i = 0; i < times; ++i) {
        ret = wc_ShaUpdate(&sha, (byte*)large_input,
            (word32)sizeof(large_input));
        if (ret != 0)
            ERROR_OUT(-1808, exit);
    }
    ret = wc_ShaFinal(&sha, hash);
    if (ret != 0)
        ERROR_OUT(-1809, exit);
    if (XMEMCMP(hash, large_digest, WC_SHA_DIGEST_SIZE) != 0)
        ERROR_OUT(-1810, exit);
    } /* END LARGE HASH TEST */

exit:

    wc_ShaFree(&sha);
    wc_ShaFree(&shaCopy);

    return ret;
}

#endif /* NO_SHA */

#ifdef WOLFSSL_RIPEMD
WOLFSSL_TEST_SUBROUTINE int ripemd_test(void)
{
    RipeMd  ripemd;
    int ret;
    byte hash[RIPEMD_DIGEST_SIZE];

    testVector a, b, c, d;
    testVector test_ripemd[4];
    int times = sizeof(test_ripemd) / sizeof(struct testVector), i;

    a.input  = "abc";
    a.output = "\x8e\xb2\x08\xf7\xe0\x5d\x98\x7a\x9b\x04\x4a\x8e\x98\xc6"
               "\xb0\x87\xf1\x5a\x0b\xfc";
    a.inLen  = XSTRLEN(a.input);
    a.outLen = RIPEMD_DIGEST_SIZE;

    b.input  = "message digest";
    b.output = "\x5d\x06\x89\xef\x49\xd2\xfa\xe5\x72\xb8\x81\xb1\x23\xa8"
               "\x5f\xfa\x21\x59\x5f\x36";
    b.inLen  = XSTRLEN(b.input);
    b.outLen = RIPEMD_DIGEST_SIZE;

    c.input  = "abcdbcdecdefdefgefghfghighijhijkijkljklmklmnlmnomnopnopq";
    c.output = "\x12\xa0\x53\x38\x4a\x9c\x0c\x88\xe4\x05\xa0\x6c\x27\xdc"
               "\xf4\x9a\xda\x62\xeb\x2b";
    c.inLen  = XSTRLEN(c.input);
    c.outLen = RIPEMD_DIGEST_SIZE;

    d.input  = "12345678901234567890123456789012345678901234567890123456"
               "789012345678901234567890";
    d.output = "\x9b\x75\x2e\x45\x57\x3d\x4b\x39\xf4\xdb\xd3\x32\x3c\xab"
               "\x82\xbf\x63\x32\x6b\xfb";
    d.inLen  = XSTRLEN(d.input);
    d.outLen = RIPEMD_DIGEST_SIZE;

    test_ripemd[0] = a;
    test_ripemd[1] = b;
    test_ripemd[2] = c;
    test_ripemd[3] = d;

    ret = wc_InitRipeMd(&ripemd);
    if (ret != 0) {
        return -1900;
    }

    for (i = 0; i < times; ++i) {
        ret = wc_RipeMdUpdate(&ripemd, (byte*)test_ripemd[i].input,
                              (word32)test_ripemd[i].inLen);
        if (ret != 0) {
            return -1901 - i;
        }

        ret = wc_RipeMdFinal(&ripemd, hash);
        if (ret != 0) {
            return -1911 - i;
        }

        if (XMEMCMP(hash, test_ripemd[i].output, RIPEMD_DIGEST_SIZE) != 0)
            return -1921 - i;
    }

    return 0;
}
#endif /* WOLFSSL_RIPEMD */


#ifdef HAVE_BLAKE2


#define BLAKE2B_TESTS 3

static const byte blake2b_vec[BLAKE2B_TESTS][BLAKE2B_OUTBYTES] =
{
  {
    0x78, 0x6A, 0x02, 0xF7, 0x42, 0x01, 0x59, 0x03,
    0xC6, 0xC6, 0xFD, 0x85, 0x25, 0x52, 0xD2, 0x72,
    0x91, 0x2F, 0x47, 0x40, 0xE1, 0x58, 0x47, 0x61,
    0x8A, 0x86, 0xE2, 0x17, 0xF7, 0x1F, 0x54, 0x19,
    0xD2, 0x5E, 0x10, 0x31, 0xAF, 0xEE, 0x58, 0x53,
    0x13, 0x89, 0x64, 0x44, 0x93, 0x4E, 0xB0, 0x4B,
    0x90, 0x3A, 0x68, 0x5B, 0x14, 0x48, 0xB7, 0x55,
    0xD5, 0x6F, 0x70, 0x1A, 0xFE, 0x9B, 0xE2, 0xCE
  },
  {
    0x2F, 0xA3, 0xF6, 0x86, 0xDF, 0x87, 0x69, 0x95,
    0x16, 0x7E, 0x7C, 0x2E, 0x5D, 0x74, 0xC4, 0xC7,
    0xB6, 0xE4, 0x8F, 0x80, 0x68, 0xFE, 0x0E, 0x44,
    0x20, 0x83, 0x44, 0xD4, 0x80, 0xF7, 0x90, 0x4C,
    0x36, 0x96, 0x3E, 0x44, 0x11, 0x5F, 0xE3, 0xEB,
    0x2A, 0x3A, 0xC8, 0x69, 0x4C, 0x28, 0xBC, 0xB4,
    0xF5, 0xA0, 0xF3, 0x27, 0x6F, 0x2E, 0x79, 0x48,
    0x7D, 0x82, 0x19, 0x05, 0x7A, 0x50, 0x6E, 0x4B
  },
  {
    0x1C, 0x08, 0x79, 0x8D, 0xC6, 0x41, 0xAB, 0xA9,
    0xDE, 0xE4, 0x35, 0xE2, 0x25, 0x19, 0xA4, 0x72,
    0x9A, 0x09, 0xB2, 0xBF, 0xE0, 0xFF, 0x00, 0xEF,
    0x2D, 0xCD, 0x8E, 0xD6, 0xF8, 0xA0, 0x7D, 0x15,
    0xEA, 0xF4, 0xAE, 0xE5, 0x2B, 0xBF, 0x18, 0xAB,
    0x56, 0x08, 0xA6, 0x19, 0x0F, 0x70, 0xB9, 0x04,
    0x86, 0xC8, 0xA7, 0xD4, 0x87, 0x37, 0x10, 0xB1,
    0x11, 0x5D, 0x3D, 0xEB, 0xBB, 0x43, 0x27, 0xB5
  }
};



WOLFSSL_TEST_SUBROUTINE int blake2b_test(void)
{
    Blake2b b2b;
    byte    digest[64];
    byte    input[64];
    int     i, ret;

    for (i = 0; i < (int)sizeof(input); i++)
        input[i] = (byte)i;

    for (i = 0; i < BLAKE2B_TESTS; i++) {
        ret = wc_InitBlake2b(&b2b, 64);
        if (ret != 0)
            return -2000 - i;

        ret = wc_Blake2bUpdate(&b2b, input, i);
        if (ret != 0)
            return -2010 - 1;

        ret = wc_Blake2bFinal(&b2b, digest, 64);
        if (ret != 0)
            return -2020 - i;

        if (XMEMCMP(digest, blake2b_vec[i], 64) != 0) {
            return -2030 - i;
        }
    }

    return 0;
}
#endif /* HAVE_BLAKE2 */

#ifdef HAVE_BLAKE2S


#define BLAKE2S_TESTS 3

static const byte blake2s_vec[BLAKE2S_TESTS][BLAKE2S_OUTBYTES] =
{
  {
    0x69, 0x21, 0x7a, 0x30, 0x79, 0x90, 0x80, 0x94,
    0xe1, 0x11, 0x21, 0xd0, 0x42, 0x35, 0x4a, 0x7c,
    0x1f, 0x55, 0xb6, 0x48, 0x2c, 0xa1, 0xa5, 0x1e,
    0x1b, 0x25, 0x0d, 0xfd, 0x1e, 0xd0, 0xee, 0xf9,
  },
  {
    0xe3, 0x4d, 0x74, 0xdb, 0xaf, 0x4f, 0xf4, 0xc6,
    0xab, 0xd8, 0x71, 0xcc, 0x22, 0x04, 0x51, 0xd2,
    0xea, 0x26, 0x48, 0x84, 0x6c, 0x77, 0x57, 0xfb,
    0xaa, 0xc8, 0x2f, 0xe5, 0x1a, 0xd6, 0x4b, 0xea,
  },
  {
    0xdd, 0xad, 0x9a, 0xb1, 0x5d, 0xac, 0x45, 0x49,
    0xba, 0x42, 0xf4, 0x9d, 0x26, 0x24, 0x96, 0xbe,
    0xf6, 0xc0, 0xba, 0xe1, 0xdd, 0x34, 0x2a, 0x88,
    0x08, 0xf8, 0xea, 0x26, 0x7c, 0x6e, 0x21, 0x0c,
  }
};



WOLFSSL_TEST_SUBROUTINE int blake2s_test(void)
{
    Blake2s b2s;
    byte    digest[32];
    byte    input[64];
    int     i, ret;

    for (i = 0; i < (int)sizeof(input); i++)
        input[i] = (byte)i;

    for (i = 0; i < BLAKE2S_TESTS; i++) {
        ret = wc_InitBlake2s(&b2s, 32);
        if (ret != 0)
            return -2100 - i;

        ret = wc_Blake2sUpdate(&b2s, input, i);
        if (ret != 0)
            return -2110 - 1;

        ret = wc_Blake2sFinal(&b2s, digest, 32);
        if (ret != 0)
            return -2120 - i;

        if (XMEMCMP(digest, blake2s_vec[i], 32) != 0) {
            return -2130 - i;
        }
    }

    return 0;
}
#endif /* HAVE_BLAKE2S */


#ifdef WOLFSSL_SHA224
WOLFSSL_TEST_SUBROUTINE int sha224_test(void)
{
    wc_Sha224 sha, shaCopy;
    byte      hash[WC_SHA224_DIGEST_SIZE];
    byte      hashcopy[WC_SHA224_DIGEST_SIZE];
    int       ret = 0;

    testVector a, b, c;
    testVector test_sha[3];
    int times = sizeof(test_sha) / sizeof(struct testVector), i;

    a.input  = "";
    a.output = "\xd1\x4a\x02\x8c\x2a\x3a\x2b\xc9\x47\x61\x02\xbb\x28\x82\x34"
               "\xc4\x15\xa2\xb0\x1f\x82\x8e\xa6\x2a\xc5\xb3\xe4\x2f";
    a.inLen  = XSTRLEN(a.input);
    a.outLen = WC_SHA224_DIGEST_SIZE;

    b.input  = "abc";
    b.output = "\x23\x09\x7d\x22\x34\x05\xd8\x22\x86\x42\xa4\x77\xbd\xa2\x55"
               "\xb3\x2a\xad\xbc\xe4\xbd\xa0\xb3\xf7\xe3\x6c\x9d\xa7";
    b.inLen  = XSTRLEN(b.input);
    b.outLen = WC_SHA224_DIGEST_SIZE;

    c.input  = "abcdbcdecdefdefgefghfghighijhijkijkljklmklmnlmnomnopnopq";
    c.output = "\x75\x38\x8b\x16\x51\x27\x76\xcc\x5d\xba\x5d\xa1\xfd\x89\x01"
               "\x50\xb0\xc6\x45\x5c\xb4\xf5\x8b\x19\x52\x52\x25\x25";
    c.inLen  = XSTRLEN(c.input);
    c.outLen = WC_SHA224_DIGEST_SIZE;

    test_sha[0] = a;
    test_sha[1] = b;
    test_sha[2] = c;

    ret = wc_InitSha224_ex(&sha, HEAP_HINT, devId);
    if (ret != 0)
        return -2200;
    ret = wc_InitSha224_ex(&shaCopy, HEAP_HINT, devId);
    if (ret != 0) {
        wc_Sha224Free(&sha);
        return -2201;
    }

    for (i = 0; i < times; ++i) {
        ret = wc_Sha224Update(&sha, (byte*)test_sha[i].input,
            (word32)test_sha[i].inLen);
        if (ret != 0)
            ERROR_OUT(-2202 - i, exit);
        ret = wc_Sha224GetHash(&sha, hashcopy);
        if (ret != 0)
            ERROR_OUT(-2203 - i, exit);
        ret = wc_Sha224Copy(&sha, &shaCopy);
        if (ret != 0)
            ERROR_OUT(-2204 - i, exit);
        ret = wc_Sha224Final(&sha, hash);
        if (ret != 0)
            ERROR_OUT(-2205 - i, exit);
        wc_Sha224Free(&shaCopy);

        if (XMEMCMP(hash, test_sha[i].output, WC_SHA224_DIGEST_SIZE) != 0)
            ERROR_OUT(-2206 - i, exit);
        if (XMEMCMP(hash, hashcopy, WC_SHA224_DIGEST_SIZE) != 0)
            ERROR_OUT(-2207 - i, exit);
    }

exit:
    wc_Sha224Free(&sha);
    wc_Sha224Free(&shaCopy);

    return ret;
}
#endif


#ifndef NO_SHA256
WOLFSSL_TEST_SUBROUTINE int sha256_test(void)
{
    wc_Sha256 sha, shaCopy;
    byte      hash[WC_SHA256_DIGEST_SIZE];
    byte      hashcopy[WC_SHA256_DIGEST_SIZE];
    int       ret = 0;

    testVector a, b, c;
    testVector test_sha[3];
    int times = sizeof(test_sha) / sizeof(struct testVector), i;

    a.input  = "";
    a.output = "\xe3\xb0\xc4\x42\x98\xfc\x1c\x14\x9a\xfb\xf4\xc8\x99\x6f\xb9"
               "\x24\x27\xae\x41\xe4\x64\x9b\x93\x4c\xa4\x95\x99\x1b\x78\x52"
               "\xb8\x55";
    a.inLen  = XSTRLEN(a.input);
    a.outLen = WC_SHA256_DIGEST_SIZE;

    b.input  = "abc";
    b.output = "\xBA\x78\x16\xBF\x8F\x01\xCF\xEA\x41\x41\x40\xDE\x5D\xAE\x22"
               "\x23\xB0\x03\x61\xA3\x96\x17\x7A\x9C\xB4\x10\xFF\x61\xF2\x00"
               "\x15\xAD";
    b.inLen  = XSTRLEN(b.input);
    b.outLen = WC_SHA256_DIGEST_SIZE;

    c.input  = "abcdbcdecdefdefgefghfghighijhijkijkljklmklmnlmnomnopnopq";
    c.output = "\x24\x8D\x6A\x61\xD2\x06\x38\xB8\xE5\xC0\x26\x93\x0C\x3E\x60"
               "\x39\xA3\x3C\xE4\x59\x64\xFF\x21\x67\xF6\xEC\xED\xD4\x19\xDB"
               "\x06\xC1";
    c.inLen  = XSTRLEN(c.input);
    c.outLen = WC_SHA256_DIGEST_SIZE;

    test_sha[0] = a;
    test_sha[1] = b;
    test_sha[2] = c;

    ret = wc_InitSha256_ex(&sha, HEAP_HINT, devId);
    if (ret != 0)
        return -2300;
    ret = wc_InitSha256_ex(&shaCopy, HEAP_HINT, devId);
    if (ret != 0) {
        wc_Sha256Free(&sha);
        return -2301;
    }

    for (i = 0; i < times; ++i) {
        ret = wc_Sha256Update(&sha, (byte*)test_sha[i].input,
            (word32)test_sha[i].inLen);
        if (ret != 0) {
            ERROR_OUT(-2302 - i, exit);
        }
        ret = wc_Sha256GetHash(&sha, hashcopy);
        if (ret != 0)
            ERROR_OUT(-2303 - i, exit);
        ret = wc_Sha256Copy(&sha, &shaCopy);
        if (ret != 0)
            ERROR_OUT(-2304 - i, exit);
        ret = wc_Sha256Final(&sha, hash);
        if (ret != 0)
            ERROR_OUT(-2305 - i, exit);
        wc_Sha256Free(&shaCopy);

        if (XMEMCMP(hash, test_sha[i].output, WC_SHA256_DIGEST_SIZE) != 0)
            ERROR_OUT(-2306 - i, exit);
        if (XMEMCMP(hash, hashcopy, WC_SHA256_DIGEST_SIZE) != 0)
            ERROR_OUT(-2307 - i, exit);
    }

    /* BEGIN LARGE HASH TEST */ {
    byte large_input[1024];
#ifdef HASH_SIZE_LIMIT
    const char* large_digest =
            "\xa4\x75\x9e\x7a\xa2\x03\x38\x32\x88\x66\xa2\xea\x17\xea\xf8\xc7"
            "\xfe\x4e\xc6\xbb\xe3\xbb\x71\xce\xe7\xdf\x7c\x04\x61\xb3\xc2\x2f";
#else
    const char* large_digest =
           "\x27\x78\x3e\x87\x96\x3a\x4e\xfb\x68\x29\xb5\x31\xc9\xba\x57\xb4"
           "\x4f\x45\x79\x7f\x67\x70\xbd\x63\x7f\xbf\x0d\x80\x7c\xbd\xba\xe0";
#endif
    for (i = 0; i < (int)sizeof(large_input); i++) {
        large_input[i] = (byte)(i & 0xFF);
    }
#ifdef HASH_SIZE_LIMIT
    times = 20;
#else
    times = 100;
#endif
#ifdef WOLFSSL_PIC32MZ_HASH
    wc_Sha256SizeSet(&sha, times * sizeof(large_input));
#endif
    for (i = 0; i < times; ++i) {
        ret = wc_Sha256Update(&sha, (byte*)large_input,
            (word32)sizeof(large_input));
        if (ret != 0)
            ERROR_OUT(-2308, exit);
    }
    ret = wc_Sha256Final(&sha, hash);
    if (ret != 0)
        ERROR_OUT(-2309, exit);
    if (XMEMCMP(hash, large_digest, WC_SHA256_DIGEST_SIZE) != 0)
        ERROR_OUT(-2310, exit);
    } /* END LARGE HASH TEST */

exit:

    wc_Sha256Free(&sha);
    wc_Sha256Free(&shaCopy);

    return ret;
}
#endif


#ifdef WOLFSSL_SHA512
WOLFSSL_TEST_SUBROUTINE int sha512_test(void)
{
    wc_Sha512 sha, shaCopy;
    byte      hash[WC_SHA512_DIGEST_SIZE];
    byte      hashcopy[WC_SHA512_DIGEST_SIZE];
    int       ret = 0;

    testVector a, b, c;
    testVector test_sha[3];
    int times = sizeof(test_sha) / sizeof(struct testVector), i;

    a.input  = "";
    a.output = "\xcf\x83\xe1\x35\x7e\xef\xb8\xbd\xf1\x54\x28\x50\xd6\x6d\x80"
               "\x07\xd6\x20\xe4\x05\x0b\x57\x15\xdc\x83\xf4\xa9\x21\xd3\x6c"
               "\xe9\xce\x47\xd0\xd1\x3c\x5d\x85\xf2\xb0\xff\x83\x18\xd2\x87"
               "\x7e\xec\x2f\x63\xb9\x31\xbd\x47\x41\x7a\x81\xa5\x38\x32\x7a"
               "\xf9\x27\xda\x3e";
    a.inLen  = XSTRLEN(a.input);
    a.outLen = WC_SHA512_DIGEST_SIZE;

    b.input  = "abc";
    b.output = "\xdd\xaf\x35\xa1\x93\x61\x7a\xba\xcc\x41\x73\x49\xae\x20\x41"
               "\x31\x12\xe6\xfa\x4e\x89\xa9\x7e\xa2\x0a\x9e\xee\xe6\x4b\x55"
               "\xd3\x9a\x21\x92\x99\x2a\x27\x4f\xc1\xa8\x36\xba\x3c\x23\xa3"
               "\xfe\xeb\xbd\x45\x4d\x44\x23\x64\x3c\xe8\x0e\x2a\x9a\xc9\x4f"
               "\xa5\x4c\xa4\x9f";
    b.inLen  = XSTRLEN(b.input);
    b.outLen = WC_SHA512_DIGEST_SIZE;

    c.input  = "abcdefghbcdefghicdefghijdefghijkefghijklfghijklmghijklmnhi"
               "jklmnoijklmnopjklmnopqklmnopqrlmnopqrsmnopqrstnopqrstu";
    c.output = "\x8e\x95\x9b\x75\xda\xe3\x13\xda\x8c\xf4\xf7\x28\x14\xfc\x14"
               "\x3f\x8f\x77\x79\xc6\xeb\x9f\x7f\xa1\x72\x99\xae\xad\xb6\x88"
               "\x90\x18\x50\x1d\x28\x9e\x49\x00\xf7\xe4\x33\x1b\x99\xde\xc4"
               "\xb5\x43\x3a\xc7\xd3\x29\xee\xb6\xdd\x26\x54\x5e\x96\xe5\x5b"
               "\x87\x4b\xe9\x09";
    c.inLen  = XSTRLEN(c.input);
    c.outLen = WC_SHA512_DIGEST_SIZE;

    test_sha[0] = a;
    test_sha[1] = b;
    test_sha[2] = c;

    ret = wc_InitSha512_ex(&sha, HEAP_HINT, devId);
    if (ret != 0)
        return -2400;
    ret = wc_InitSha512_ex(&shaCopy, HEAP_HINT, devId);
    if (ret != 0) {
        wc_Sha512Free(&sha);
        return -2401;
    }

    for (i = 0; i < times; ++i) {
        ret = wc_Sha512Update(&sha, (byte*)test_sha[i].input,
            (word32)test_sha[i].inLen);
        if (ret != 0)
            ERROR_OUT(-2402 - i, exit);
        ret = wc_Sha512GetHash(&sha, hashcopy);
        if (ret != 0)
            ERROR_OUT(-2403 - i, exit);
        ret = wc_Sha512Copy(&sha, &shaCopy);
        if (ret != 0)
            ERROR_OUT(-2404 - i, exit);
        ret = wc_Sha512Final(&sha, hash);
        if (ret != 0)
            ERROR_OUT(-2405 - i, exit);
        wc_Sha512Free(&shaCopy);

        if (XMEMCMP(hash, test_sha[i].output, WC_SHA512_DIGEST_SIZE) != 0)
            ERROR_OUT(-2406 - i, exit);
        if (XMEMCMP(hash, hashcopy, WC_SHA512_DIGEST_SIZE) != 0)
            ERROR_OUT(-2407 - i, exit);
    }

    /* BEGIN LARGE HASH TEST */ {
    byte large_input[1024];
#ifdef HASH_SIZE_LIMIT
    const char* large_digest =
        "\x30\x9B\x96\xA6\xE9\x43\x78\x30\xA3\x71\x51\x61\xC1\xEB\xE1\xBE"
        "\xC8\xA5\xF9\x13\x5A\xD6\x6D\x9E\x46\x31\x31\x67\x8D\xE2\xC0\x0B"
        "\x2A\x1A\x03\xE1\xF3\x48\xA7\x33\xBD\x49\xF8\xFF\xF1\xC2\xC2\x95"
        "\xCB\xF0\xAF\x87\x61\x85\x58\x63\x6A\xCA\x70\x9C\x8B\x83\x3F\x5D";
#else
    const char* large_digest =
        "\x5a\x1f\x73\x90\xbd\x8c\xe4\x63\x54\xce\xa0\x9b\xef\x32\x78\x2d"
        "\x2e\xe7\x0d\x5e\x2f\x9d\x15\x1b\xdd\x2d\xde\x65\x0c\x7b\xfa\x83"
        "\x5e\x80\x02\x13\x84\xb8\x3f\xff\x71\x62\xb5\x09\x89\x63\xe1\xdc"
        "\xa5\xdc\xfc\xfa\x9d\x1a\x4d\xc0\xfa\x3a\x14\xf6\x01\x51\x90\xa4";
#endif

    for (i = 0; i < (int)sizeof(large_input); i++) {
        large_input[i] = (byte)(i & 0xFF);
    }
#ifdef HASH_SIZE_LIMIT
    times = 20;
#else
    times = 100;
#endif
    for (i = 0; i < times; ++i) {
        ret = wc_Sha512Update(&sha, (byte*)large_input,
            (word32)sizeof(large_input));
        if (ret != 0)
            ERROR_OUT(-2408, exit);
    }
    ret = wc_Sha512Final(&sha, hash);
    if (ret != 0)
        ERROR_OUT(-2409, exit);
    if (XMEMCMP(hash, large_digest, WC_SHA512_DIGEST_SIZE) != 0)
        ERROR_OUT(-2410, exit);

#ifndef NO_UNALIGNED_MEMORY_TEST
    /* Unaligned memory access test */
    for (i = 1; i < 16; i++) {
        ret = wc_Sha512Update(&sha, (byte*)large_input + i,
            (word32)sizeof(large_input) - i);
        if (ret != 0)
            ERROR_OUT(-2411, exit);
        ret = wc_Sha512Final(&sha, hash);
    }
#endif
    } /* END LARGE HASH TEST */

exit:
    wc_Sha512Free(&sha);
    wc_Sha512Free(&shaCopy);

    return ret;
}
#endif


#ifdef WOLFSSL_SHA384
WOLFSSL_TEST_SUBROUTINE int sha384_test(void)
{
    wc_Sha384 sha, shaCopy;
    byte      hash[WC_SHA384_DIGEST_SIZE];
    byte      hashcopy[WC_SHA384_DIGEST_SIZE];
    int       ret = 0;

    testVector a, b, c;
    testVector test_sha[3];
    int times = sizeof(test_sha) / sizeof(struct testVector), i;

    a.input  = "";

    a.output = "\x38\xb0\x60\xa7\x51\xac\x96\x38\x4c\xd9\x32\x7e\xb1\xb1\xe3"
               "\x6a\x21\xfd\xb7\x11\x14\xbe\x07\x43\x4c\x0c\xc7\xbf\x63\xf6"
               "\xe1\xda\x27\x4e\xde\xbf\xe7\x6f\x65\xfb\xd5\x1a\xd2\xf1\x48"
               "\x98\xb9\x5b";
    a.inLen  = XSTRLEN(a.input);
    a.outLen = WC_SHA384_DIGEST_SIZE;

    b.input  = "abc";
    b.output = "\xcb\x00\x75\x3f\x45\xa3\x5e\x8b\xb5\xa0\x3d\x69\x9a\xc6\x50"
               "\x07\x27\x2c\x32\xab\x0e\xde\xd1\x63\x1a\x8b\x60\x5a\x43\xff"
               "\x5b\xed\x80\x86\x07\x2b\xa1\xe7\xcc\x23\x58\xba\xec\xa1\x34"
               "\xc8\x25\xa7";
    b.inLen  = XSTRLEN(b.input);
    b.outLen = WC_SHA384_DIGEST_SIZE;

    c.input  = "abcdefghbcdefghicdefghijdefghijkefghijklfghijklmghijklmnhi"
               "jklmnoijklmnopjklmnopqklmnopqrlmnopqrsmnopqrstnopqrstu";
    c.output = "\x09\x33\x0c\x33\xf7\x11\x47\xe8\x3d\x19\x2f\xc7\x82\xcd\x1b"
               "\x47\x53\x11\x1b\x17\x3b\x3b\x05\xd2\x2f\xa0\x80\x86\xe3\xb0"
               "\xf7\x12\xfc\xc7\xc7\x1a\x55\x7e\x2d\xb9\x66\xc3\xe9\xfa\x91"
               "\x74\x60\x39";
    c.inLen  = XSTRLEN(c.input);
    c.outLen = WC_SHA384_DIGEST_SIZE;

    test_sha[0] = a;
    test_sha[1] = b;
    test_sha[2] = c;

    ret = wc_InitSha384_ex(&sha, HEAP_HINT, devId);
    if (ret != 0)
        return -2500;
    ret = wc_InitSha384_ex(&shaCopy, HEAP_HINT, devId);
    if (ret != 0) {
        wc_Sha384Free(&sha);
        return -2501;
    }

    for (i = 0; i < times; ++i) {
        ret = wc_Sha384Update(&sha, (byte*)test_sha[i].input,
            (word32)test_sha[i].inLen);
        if (ret != 0)
            ERROR_OUT(-2502 - i, exit);
        ret = wc_Sha384GetHash(&sha, hashcopy);
        if (ret != 0)
            ERROR_OUT(-2503 - i, exit);
        ret = wc_Sha384Copy(&sha, &shaCopy);
        if (ret != 0)
            ERROR_OUT(-2504 - i, exit);
        ret = wc_Sha384Final(&sha, hash);
        if (ret != 0)
            ERROR_OUT(-2505 - i, exit);
        wc_Sha384Free(&shaCopy);

        if (XMEMCMP(hash, test_sha[i].output, WC_SHA384_DIGEST_SIZE) != 0)
            ERROR_OUT(-2506 - i, exit);
        if (XMEMCMP(hash, hashcopy, WC_SHA384_DIGEST_SIZE) != 0)
            ERROR_OUT(-2507 - i, exit);
    }

    /* BEGIN LARGE HASH TEST */ {
    byte large_input[1024];
#ifdef HASH_SIZE_LIMIT
    const char* large_digest =
        "\xB5\xAD\x66\x6F\xD9\x58\x5E\x68\xDD\x5E\x30\xD3\x95\x72\x33\xA4"
        "\xE9\x4B\x99\x3A\xEF\xF8\xE1\xBF\x1F\x05\x32\xAA\x16\x00\x82\xEC"
        "\x15\xDA\xF2\x75\xEE\xE9\x06\xAF\x52\x8A\x5C\xEF\x72\x81\x80\xD6";
#else
    const char* large_digest =
        "\x37\x01\xdb\xff\x1e\x40\x4f\xe1\xe2\xea\x0b\x40\xbb\x3b\x39\x9a"
        "\xcc\xe8\x44\x8e\x7e\xe5\x64\xb5\x6b\x7f\x56\x64\xa7\x2b\x84\xe3"
        "\xc5\xd7\x79\x03\x25\x90\xf7\xa4\x58\xcb\x97\xa8\x8b\xb1\xa4\x81";
#endif

    for (i = 0; i < (int)sizeof(large_input); i++) {
        large_input[i] = (byte)(i & 0xFF);
    }
#ifdef HASH_SIZE_LIMIT
    times = 20;
#else
    times = 100;
#endif
    for (i = 0; i < times; ++i) {
        ret = wc_Sha384Update(&sha, (byte*)large_input,
            (word32)sizeof(large_input));
        if (ret != 0)
            ERROR_OUT(-2508, exit);
    }
    ret = wc_Sha384Final(&sha, hash);
    if (ret != 0)
        ERROR_OUT(-2509, exit);
    if (XMEMCMP(hash, large_digest, WC_SHA384_DIGEST_SIZE) != 0)
        ERROR_OUT(-2510, exit);
    } /* END LARGE HASH TEST */

exit:

    wc_Sha384Free(&sha);
    wc_Sha384Free(&shaCopy);

    return ret;
}
#endif /* WOLFSSL_SHA384 */

#ifdef WOLFSSL_SHA3
#ifndef WOLFSSL_NOSHA3_224
static int sha3_224_test(void)
{
    wc_Sha3  sha;
    byte  hash[WC_SHA3_224_DIGEST_SIZE];
    byte  hashcopy[WC_SHA3_224_DIGEST_SIZE];

    testVector a, b, c;
    testVector test_sha[3];
    int ret = 0;
    int times = sizeof(test_sha) / sizeof(struct testVector), i;

    a.input  = "";
    a.output = "\x6b\x4e\x03\x42\x36\x67\xdb\xb7\x3b\x6e\x15\x45\x4f\x0e\xb1"
               "\xab\xd4\x59\x7f\x9a\x1b\x07\x8e\x3f\x5b\x5a\x6b\xc7";
    a.inLen  = XSTRLEN(a.input);
    a.outLen = WC_SHA3_224_DIGEST_SIZE;

    b.input  = "abc";
    b.output = "\xe6\x42\x82\x4c\x3f\x8c\xf2\x4a\xd0\x92\x34\xee\x7d\x3c\x76"
               "\x6f\xc9\xa3\xa5\x16\x8d\x0c\x94\xad\x73\xb4\x6f\xdf";
    b.inLen  = XSTRLEN(b.input);
    b.outLen = WC_SHA3_224_DIGEST_SIZE;

    c.input  = "abcdbcdecdefdefgefghfghighijhijkijkljklmklmnlmnomnopnopq";
    c.output = "\x8a\x24\x10\x8b\x15\x4a\xda\x21\xc9\xfd\x55\x74\x49\x44\x79"
               "\xba\x5c\x7e\x7a\xb7\x6e\xf2\x64\xea\xd0\xfc\xce\x33";
    c.inLen  = XSTRLEN(c.input);
    c.outLen = WC_SHA3_224_DIGEST_SIZE;

    test_sha[0] = a;
    test_sha[1] = b;
    test_sha[2] = c;

    ret = wc_InitSha3_224(&sha, HEAP_HINT, devId);
    if (ret != 0)
        return -2600;

    for (i = 0; i < times; ++i) {
        ret = wc_Sha3_224_Update(&sha, (byte*)test_sha[i].input,
            (word32)test_sha[i].inLen);
        if (ret != 0)
            ERROR_OUT(-2601 - i, exit);
        ret = wc_Sha3_224_GetHash(&sha, hashcopy);
        if (ret != 0)
            ERROR_OUT(-2602 - i, exit);
        ret = wc_Sha3_224_Final(&sha, hash);
        if (ret != 0)
            ERROR_OUT(-2603 - i, exit);

        if (XMEMCMP(hash, test_sha[i].output, WC_SHA3_224_DIGEST_SIZE) != 0)
            ERROR_OUT(-2604 - i, exit);
        if (XMEMCMP(hash, hashcopy, WC_SHA3_224_DIGEST_SIZE) != 0)
            ERROR_OUT(-2605 - i, exit);
    }

    /* BEGIN LARGE HASH TEST */ {
    byte large_input[1024];
    const char* large_digest =
        "\x13\xe5\xd3\x98\x7b\x94\xda\x41\x12\xc7\x1e\x92\x3a\x19"
        "\x21\x20\x86\x6f\x24\xbf\x0a\x31\xbc\xfd\xd6\x70\x36\xf3";

    for (i = 0; i < (int)sizeof(large_input); i++) {
        large_input[i] = (byte)(i & 0xFF);
    }
    times = 100;
    for (i = 0; i < times; ++i) {
        ret = wc_Sha3_224_Update(&sha, (byte*)large_input,
            (word32)sizeof(large_input));
        if (ret != 0)
            ERROR_OUT(-2606, exit);
    }
    ret = wc_Sha3_224_Final(&sha, hash);
    if (ret != 0)
        ERROR_OUT(-2607, exit);
    if (XMEMCMP(hash, large_digest, WC_SHA3_224_DIGEST_SIZE) != 0)
        ERROR_OUT(-2608, exit);
    } /* END LARGE HASH TEST */

exit:
    wc_Sha3_224_Free(&sha);

    return ret;
}
#endif /* WOLFSSL_NOSHA3_224 */

#ifndef WOLFSSL_NOSHA3_256
static int sha3_256_test(void)
{
    wc_Sha3  sha;
    byte  hash[WC_SHA3_256_DIGEST_SIZE];
    byte  hashcopy[WC_SHA3_256_DIGEST_SIZE];

    testVector a, b, c;
    testVector test_sha[3];
    int ret = 0;
    int times = sizeof(test_sha) / sizeof(struct testVector), i;

    byte large_input[1024];
    const char* large_digest =
        "\xdc\x90\xc0\xb1\x25\xdb\x2c\x34\x81\xa3\xff\xbc\x1e\x2e\x87\xeb"
        "\x6d\x70\x85\x61\xe0\xe9\x63\x61\xff\xe5\x84\x4b\x1f\x68\x05\x15";

#if defined(WOLFSSL_HASH_FLAGS) && !defined(WOLFSSL_ASYNC_CRYPT)
    /* test vector with hash of empty string */
    const char* Keccak256EmptyOut =
        "\xc5\xd2\x46\x01\x86\xf7\x23\x3c\x92\x7e\x7d\xb2\xdc\xc7\x03\xc0"
        "\xe5\x00\xb6\x53\xca\x82\x27\x3b\x7b\xfa\xd8\x04\x5d\x85\xa4\x70";
#endif

    a.input  = "";
    a.output = "\xa7\xff\xc6\xf8\xbf\x1e\xd7\x66\x51\xc1\x47\x56\xa0\x61\xd6"
               "\x62\xf5\x80\xff\x4d\xe4\x3b\x49\xfa\x82\xd8\x0a\x4b\x80\xf8"
               "\x43\x4a";
    a.inLen  = XSTRLEN(a.input);
    a.outLen = WC_SHA3_256_DIGEST_SIZE;

    b.input  = "abc";
    b.output = "\x3a\x98\x5d\xa7\x4f\xe2\x25\xb2\x04\x5c\x17\x2d\x6b\xd3\x90"
               "\xbd\x85\x5f\x08\x6e\x3e\x9d\x52\x5b\x46\xbf\xe2\x45\x11\x43"
               "\x15\x32";
    b.inLen  = XSTRLEN(b.input);
    b.outLen = WC_SHA3_256_DIGEST_SIZE;

    c.input  = "abcdbcdecdefdefgefghfghighijhijkijkljklmklmnlmnomnopnopq";
    c.output = "\x41\xc0\xdb\xa2\xa9\xd6\x24\x08\x49\x10\x03\x76\xa8\x23\x5e"
               "\x2c\x82\xe1\xb9\x99\x8a\x99\x9e\x21\xdb\x32\xdd\x97\x49\x6d"
               "\x33\x76";
    c.inLen  = XSTRLEN(c.input);
    c.outLen = WC_SHA3_256_DIGEST_SIZE;

    test_sha[0] = a;
    test_sha[1] = b;
    test_sha[2] = c;

    ret = wc_InitSha3_256(&sha, HEAP_HINT, devId);
    if (ret != 0)
        return -2700;

    for (i = 0; i < times; ++i) {
        ret = wc_Sha3_256_Update(&sha, (byte*)test_sha[i].input,
            (word32)test_sha[i].inLen);
        if (ret != 0)
            ERROR_OUT(-2701 - i, exit);
        ret = wc_Sha3_256_GetHash(&sha, hashcopy);
        if (ret != 0)
            ERROR_OUT(-2702 - i, exit);
        ret = wc_Sha3_256_Final(&sha, hash);
        if (ret != 0)
            ERROR_OUT(-2703 - i, exit);

        if (XMEMCMP(hash, test_sha[i].output, WC_SHA3_256_DIGEST_SIZE) != 0)
            ERROR_OUT(-2704 - i, exit);
        if (XMEMCMP(hash, hashcopy, WC_SHA3_256_DIGEST_SIZE) != 0)
            ERROR_OUT(-2705 - i, exit);
    }

    /* BEGIN LARGE HASH TEST */ {
    for (i = 0; i < (int)sizeof(large_input); i++) {
        large_input[i] = (byte)(i & 0xFF);
    }
    times = 100;
    for (i = 0; i < times; ++i) {
        ret = wc_Sha3_256_Update(&sha, (byte*)large_input,
            (word32)sizeof(large_input));
        if (ret != 0)
            ERROR_OUT(-2706, exit);
    }
    ret = wc_Sha3_256_Final(&sha, hash);
    if (ret != 0)
        ERROR_OUT(-2707, exit);
    if (XMEMCMP(hash, large_digest, WC_SHA3_256_DIGEST_SIZE) != 0)
        ERROR_OUT(-2708, exit);
    } /* END LARGE HASH TEST */

    /* this is a software only variant of SHA3 not supported by external hardware devices */
#if defined(WOLFSSL_HASH_FLAGS) && !defined(WOLFSSL_ASYNC_CRYPT)
    /* Test for Keccak256 */
    ret = wc_Sha3_SetFlags(&sha, WC_HASH_SHA3_KECCAK256);
    if (ret != 0) {
        ERROR_OUT(-2709, exit);
    }
    ret = wc_Sha3_256_Update(&sha, (byte*)"", 0);
    if (ret != 0) {
        ERROR_OUT(-2710, exit);
    }
    ret = wc_Sha3_256_Final(&sha, hash);
    if (ret != 0) {
        ERROR_OUT(-2711, exit);
    }
    if (XMEMCMP(hash, Keccak256EmptyOut, WC_SHA3_256_DIGEST_SIZE) != 0) {
        ERROR_OUT(-2712, exit);
    }
#endif /* WOLFSSL_HASH_FLAGS && !WOLFSSL_ASYNC_CRYPT */

exit:
    wc_Sha3_256_Free(&sha);

    return ret;
}
#endif /* WOLFSSL_NOSHA3_256 */

#ifndef WOLFSSL_NOSHA3_384
static int sha3_384_test(void)
{
    wc_Sha3  sha;
    byte  hash[WC_SHA3_384_DIGEST_SIZE];
#ifndef NO_INTM_HASH_TEST
    byte  hashcopy[WC_SHA3_384_DIGEST_SIZE];
#endif

    testVector a, b, c;
    testVector test_sha[3];
    int ret;
    int times = sizeof(test_sha) / sizeof(struct testVector), i;

    a.input  = "";
    a.output = "\x0c\x63\xa7\x5b\x84\x5e\x4f\x7d\x01\x10\x7d\x85\x2e\x4c\x24"
               "\x85\xc5\x1a\x50\xaa\xaa\x94\xfc\x61\x99\x5e\x71\xbb\xee\x98"
               "\x3a\x2a\xc3\x71\x38\x31\x26\x4a\xdb\x47\xfb\x6b\xd1\xe0\x58"
               "\xd5\xf0\x04";
    a.inLen  = XSTRLEN(a.input);
    a.outLen = WC_SHA3_384_DIGEST_SIZE;

#if defined(WOLFSSL_AFALG_XILINX_SHA3) || defined(WOLFSSL_XILINX_CRYPT)
    /* NIST test vector with a length that is a multiple of 4 */
    b.input  = "\x7d\x80\xb1\x60\xc4\xb5\x36\xa3\xbe\xb7\x99\x80\x59\x93\x44"
               "\x04\x7c\x5f\x82\xa1\xdf\xc3\xee\xd4";
    b.output = "\x04\x1c\xc5\x86\x1b\xa3\x34\x56\x3c\x61\xd4\xef\x97\x10\xd4"
               "\x89\x6c\x31\x1c\x92\xed\xbe\x0d\x7c\xd5\x3e\x80\x3b\xf2\xf4"
               "\xeb\x60\x57\x23\x55\x70\x77\x0c\xe8\x7c\x55\x20\xd7\xec\x14"
               "\x19\x87\x22";
    b.inLen  = XSTRLEN(b.input);
    b.outLen = WC_SHA3_384_DIGEST_SIZE;
#else
    b.input  = "abc";
    b.output = "\xec\x01\x49\x82\x88\x51\x6f\xc9\x26\x45\x9f\x58\xe2\xc6\xad"
               "\x8d\xf9\xb4\x73\xcb\x0f\xc0\x8c\x25\x96\xda\x7c\xf0\xe4\x9b"
               "\xe4\xb2\x98\xd8\x8c\xea\x92\x7a\xc7\xf5\x39\xf1\xed\xf2\x28"
               "\x37\x6d\x25";
    b.inLen  = XSTRLEN(b.input);
    b.outLen = WC_SHA3_384_DIGEST_SIZE;
#endif
    c.input  = "abcdbcdecdefdefgefghfghighijhijkijkljklmklmnlmnomnopnopq";
    c.output = "\x99\x1c\x66\x57\x55\xeb\x3a\x4b\x6b\xbd\xfb\x75\xc7\x8a\x49"
               "\x2e\x8c\x56\xa2\x2c\x5c\x4d\x7e\x42\x9b\xfd\xbc\x32\xb9\xd4"
               "\xad\x5a\xa0\x4a\x1f\x07\x6e\x62\xfe\xa1\x9e\xef\x51\xac\xd0"
               "\x65\x7c\x22";
    c.inLen  = XSTRLEN(c.input);
    c.outLen = WC_SHA3_384_DIGEST_SIZE;

#ifdef WOLFSSL_XILINX_CRYPT
    test_sha[0] = b; /* hardware acc. can not handle "" string */
#else
    test_sha[0] = a;
#endif
    test_sha[1] = b;
    test_sha[2] = c;

    ret = wc_InitSha3_384(&sha, HEAP_HINT, devId);
    if (ret != 0)
        return -2800;

    for (i = 0; i < times; ++i) {
        ret = wc_Sha3_384_Update(&sha, (byte*)test_sha[i].input,
            (word32)test_sha[i].inLen);
        if (ret != 0)
            ERROR_OUT(-2801 - i, exit);
    #ifndef NO_INTM_HASH_TEST
        ret = wc_Sha3_384_GetHash(&sha, hashcopy);
        if (ret != 0)
            ERROR_OUT(-2802 - i, exit);
    #endif
        ret = wc_Sha3_384_Final(&sha, hash);
        if (ret != 0)
            ERROR_OUT(-2803 - i, exit);

        if (XMEMCMP(hash, test_sha[i].output, WC_SHA3_384_DIGEST_SIZE) != 0)
            ERROR_OUT(-2804 - i, exit);
    #ifndef NO_INTM_HASH_TEST
        if (XMEMCMP(hash, hashcopy, WC_SHA3_384_DIGEST_SIZE) != 0)
            ERROR_OUT(-2805 - i, exit);
    #endif
    }

    /* BEGIN LARGE HASH TEST */ {
    byte large_input[1024];
    const char* large_digest =
        "\x30\x44\xec\x17\xef\x47\x9f\x55\x36\x11\xd6\x3f\x8a\x31\x5a\x71"
        "\x8a\x71\xa7\x1d\x8e\x84\xe8\x6c\x24\x02\x2f\x7a\x08\x4e\xea\xd7"
        "\x42\x36\x5d\xa8\xc2\xb7\x42\xad\xec\x19\xfb\xca\xc6\x64\xb3\xa4";

    for (i = 0; i < (int)sizeof(large_input); i++) {
        large_input[i] = (byte)(i & 0xFF);
    }
    times = 100;
    for (i = 0; i < times; ++i) {
        ret = wc_Sha3_384_Update(&sha, (byte*)large_input,
            (word32)sizeof(large_input));
        if (ret != 0)
            ERROR_OUT(-2806, exit);
    }
    ret = wc_Sha3_384_Final(&sha, hash);
    if (ret != 0)
        ERROR_OUT(-2807, exit);
    if (XMEMCMP(hash, large_digest, WC_SHA3_384_DIGEST_SIZE) != 0)
        ERROR_OUT(-2808, exit);
    } /* END LARGE HASH TEST */

exit:
    wc_Sha3_384_Free(&sha);

    return ret;
}
#endif /* WOLFSSL_NOSHA3_384 */

#ifndef WOLFSSL_NOSHA3_512
static int sha3_512_test(void)
{
    wc_Sha3  sha;
    byte  hash[WC_SHA3_512_DIGEST_SIZE];
    byte  hashcopy[WC_SHA3_512_DIGEST_SIZE];

    testVector a, b, c;
    testVector test_sha[3];
    int ret;
    int times = sizeof(test_sha) / sizeof(struct testVector), i;

    a.input  = "";
    a.output = "\xa6\x9f\x73\xcc\xa2\x3a\x9a\xc5\xc8\xb5\x67\xdc\x18\x5a\x75"
               "\x6e\x97\xc9\x82\x16\x4f\xe2\x58\x59\xe0\xd1\xdc\xc1\x47\x5c"
               "\x80\xa6\x15\xb2\x12\x3a\xf1\xf5\xf9\x4c\x11\xe3\xe9\x40\x2c"
               "\x3a\xc5\x58\xf5\x00\x19\x9d\x95\xb6\xd3\xe3\x01\x75\x85\x86"
               "\x28\x1d\xcd\x26";
    a.inLen  = XSTRLEN(a.input);
    a.outLen = WC_SHA3_512_DIGEST_SIZE;

    b.input  = "abc";
    b.output = "\xb7\x51\x85\x0b\x1a\x57\x16\x8a\x56\x93\xcd\x92\x4b\x6b\x09"
               "\x6e\x08\xf6\x21\x82\x74\x44\xf7\x0d\x88\x4f\x5d\x02\x40\xd2"
               "\x71\x2e\x10\xe1\x16\xe9\x19\x2a\xf3\xc9\x1a\x7e\xc5\x76\x47"
               "\xe3\x93\x40\x57\x34\x0b\x4c\xf4\x08\xd5\xa5\x65\x92\xf8\x27"
               "\x4e\xec\x53\xf0";
    b.inLen  = XSTRLEN(b.input);
    b.outLen = WC_SHA3_512_DIGEST_SIZE;

    c.input  = "abcdbcdecdefdefgefghfghighijhijkijkljklmklmnlmnomnopnopq";
    c.output = "\x04\xa3\x71\xe8\x4e\xcf\xb5\xb8\xb7\x7c\xb4\x86\x10\xfc\xa8"
               "\x18\x2d\xd4\x57\xce\x6f\x32\x6a\x0f\xd3\xd7\xec\x2f\x1e\x91"
               "\x63\x6d\xee\x69\x1f\xbe\x0c\x98\x53\x02\xba\x1b\x0d\x8d\xc7"
               "\x8c\x08\x63\x46\xb5\x33\xb4\x9c\x03\x0d\x99\xa2\x7d\xaf\x11"
               "\x39\xd6\xe7\x5e";
    c.inLen  = XSTRLEN(c.input);
    c.outLen = WC_SHA3_512_DIGEST_SIZE;

    test_sha[0] = a;
    test_sha[1] = b;
    test_sha[2] = c;

    ret = wc_InitSha3_512(&sha, HEAP_HINT, devId);
    if (ret != 0)
        return -2900;

    for (i = 0; i < times; ++i) {
        ret = wc_Sha3_512_Update(&sha, (byte*)test_sha[i].input,
            (word32)test_sha[i].inLen);
        if (ret != 0)
            ERROR_OUT(-2901 - i, exit);
        ret = wc_Sha3_512_GetHash(&sha, hashcopy);
        if (ret != 0)
            ERROR_OUT(-2902 - i, exit);
        ret = wc_Sha3_512_Final(&sha, hash);
        if (ret != 0)
            ERROR_OUT(-2903 - i, exit);

        if (XMEMCMP(hash, test_sha[i].output, WC_SHA3_512_DIGEST_SIZE) != 0)
            ERROR_OUT(-2904 - i, exit);
        if (XMEMCMP(hash, hashcopy, WC_SHA3_512_DIGEST_SIZE) != 0)
            ERROR_OUT(-2905 - i, exit);
    }

    /* BEGIN LARGE HASH TEST */ {
    byte large_input[1024];
    const char* large_digest =
        "\x9c\x13\x26\xb6\x26\xb2\x94\x31\xbc\xf4\x34\xe9\x6f\xf2\xd6\x29"
        "\x9a\xd0\x9b\x32\x63\x2f\x18\xa7\x5f\x23\xc9\x60\xc2\x32\x0c\xbc"
        "\x57\x77\x33\xf1\x83\x81\x8a\xd3\x15\x7c\x93\xdc\x80\x9f\xed\x61"
        "\x41\xa7\x5b\xfd\x32\x0e\x38\x15\xb0\x46\x3b\x7a\x4f\xfd\x44\x88";

    for (i = 0; i < (int)sizeof(large_input); i++) {
        large_input[i] = (byte)(i & 0xFF);
    }
    times = 100;
    for (i = 0; i < times; ++i) {
        ret = wc_Sha3_512_Update(&sha, (byte*)large_input,
            (word32)sizeof(large_input));
        if (ret != 0)
            ERROR_OUT(-2906, exit);
    }
    ret = wc_Sha3_512_Final(&sha, hash);
    if (ret != 0)
        ERROR_OUT(-2907, exit);
    if (XMEMCMP(hash, large_digest, WC_SHA3_512_DIGEST_SIZE) != 0)
        ERROR_OUT(-2908, exit);
    } /* END LARGE HASH TEST */

exit:
    wc_Sha3_512_Free(&sha);

    return ret;
}
#endif /* WOLFSSL_NOSHA3_512 */

WOLFSSL_TEST_SUBROUTINE int sha3_test(void)
{
    int ret;

    (void)ret;

#ifndef WOLFSSL_NOSHA3_224
    if ((ret = sha3_224_test()) != 0)
        return ret;
#endif
#ifndef WOLFSSL_NOSHA3_256
    if ((ret = sha3_256_test()) != 0)
        return ret;
#endif
#ifndef WOLFSSL_NOSHA3_384
    if ((ret = sha3_384_test()) != 0)
        return ret;
#endif
#ifndef WOLFSSL_NOSHA3_512
    if ((ret = sha3_512_test()) != 0)
        return ret;
#endif

    return 0;
}
#endif /* WOLFSSL_SHA3 */

#ifdef WOLFSSL_SHAKE256
WOLFSSL_TEST_SUBROUTINE int shake256_test(void)
{
    wc_Shake  sha;
    byte  hash[250];

    testVector a, b, c, d, e;
    testVector test_sha[5];
    int ret = 0;
    int times = sizeof(test_sha) / sizeof(struct testVector), i;

    byte large_input[1024];
    const char* large_digest =
        "\x90\x32\x4a\xcc\xd1\xdf\xb8\x0b\x79\x1f\xb8\xc8\x5b\x54\xc8\xe7"
        "\x45\xf5\x60\x6b\x38\x26\xb2\x0a\xee\x38\x01\xf3\xd9\xfa\x96\x9f"
        "\x6a\xd7\x15\xdf\xb6\xc2\xf4\x20\x33\x44\x55\xe8\x2a\x09\x2b\x68"
        "\x2e\x18\x65\x5e\x65\x93\x28\xbc\xb1\x9e\xe2\xb1\x92\xea\x98\xac"
        "\x21\xef\x4c\xe1\xb4\xb7\xbe\x81\x5c\x1d\xd3\xb7\x17\xe5\xbb\xc5"
        "\x8c\x68\xb7\xfb\xac\x55\x8a\x9b\x4d\x91\xe4\x9f\x72\xbb\x6e\x38"
        "\xaf\x21\x7d\x21\xaa\x98\x4e\x75\xc4\xb4\x1c\x7c\x50\x45\x54\xf9"
        "\xea\x26";

    a.input  = "";
    a.output = "\x46\xb9\xdd\x2b\x0b\xa8\x8d\x13\x23\x3b\x3f\xeb\x74\x3e\xeb"
               "\x24\x3f\xcd\x52\xea\x62\xb8\x1b\x82\xb5\x0c\x27\x64\x6e\xd5"
               "\x76\x2f\xd7\x5d\xc4\xdd\xd8\xc0\xf2\x00\xcb\x05\x01\x9d\x67"
               "\xb5\x92\xf6\xfc\x82\x1c\x49\x47\x9a\xb4\x86\x40\x29\x2e\xac"
               "\xb3\xb7\xc4\xbe\x14\x1e\x96\x61\x6f\xb1\x39\x57\x69\x2c\xc7"
               "\xed\xd0\xb4\x5a\xe3\xdc\x07\x22\x3c\x8e\x92\x93\x7b\xef\x84"
               "\xbc\x0e\xab\x86\x28\x53\x34\x9e\xc7\x55\x46\xf5\x8f\xb7\xc2"
               "\x77\x5c\x38\x46\x2c\x50\x10\xd8\x46";
    a.inLen  = XSTRLEN(a.input);
    a.outLen = 114;

    b.input  = "abc";
    b.output = "\x48\x33\x66\x60\x13\x60\xa8\x77\x1c\x68\x63\x08\x0c\xc4\x11"
               "\x4d\x8d\xb4\x45\x30\xf8\xf1\xe1\xee\x4f\x94\xea\x37\xe7\x8b"
               "\x57\x39\xd5\xa1\x5b\xef\x18\x6a\x53\x86\xc7\x57\x44\xc0\x52"
               "\x7e\x1f\xaa\x9f\x87\x26\xe4\x62\xa1\x2a\x4f\xeb\x06\xbd\x88"
               "\x01\xe7\x51\xe4\x13\x85\x14\x12\x04\xf3\x29\x97\x9f\xd3\x04"
               "\x7a\x13\xc5\x65\x77\x24\xad\xa6\x4d\x24\x70\x15\x7b\x3c\xdc"
               "\x28\x86\x20\x94\x4d\x78\xdb\xcd\xdb\xd9\x12\x99\x3f\x09\x13"
               "\xf1\x64\xfb\x2c\xe9\x51\x31\xa2\xd0";
    b.inLen  = XSTRLEN(b.input);
    b.outLen = 114;

    c.input  = "abcdbcdecdefdefgefghfghighijhijkijkljklmklmnlmnomnopnopq";
    c.output = "\x4d\x8c\x2d\xd2\x43\x5a\x01\x28\xee\xfb\xb8\xc3\x6f\x6f\x87"
               "\x13\x3a\x79\x11\xe1\x8d\x97\x9e\xe1\xae\x6b\xe5\xd4\xfd\x2e"
               "\x33\x29\x40\xd8\x68\x8a\x4e\x6a\x59\xaa\x80\x60\xf1\xf9\xbc"
               "\x99\x6c\x05\xac\xa3\xc6\x96\xa8\xb6\x62\x79\xdc\x67\x2c\x74"
               "\x0b\xb2\x24\xec\x37\xa9\x2b\x65\xdb\x05\x39\xc0\x20\x34\x55"
               "\xf5\x1d\x97\xcc\xe4\xcf\xc4\x91\x27\xd7\x26\x0a\xfc\x67\x3a"
               "\xf2\x08\xba\xf1\x9b\xe2\x12\x33\xf3\xde\xbe\x78\xd0\x67\x60"
               "\xcf\xa5\x51\xee\x1e\x07\x91\x41\xd4";
    c.inLen  = XSTRLEN(c.input);
    c.outLen = 114;

    /* Taken from NIST CAVP test vectors - full rate output. */
    d.input  = "\xdc\x88\x6d\xf3\xf6\x9c\x49\x51\x3d\xe3\x62\x7e\x94\x81\xdb"
               "\x58\x71\xe8\xee\x88\xeb\x9f\x99\x61\x15\x41\x93\x0a\x8b\xc8"
               "\x85\xe0";
    d.output = "\x00\x64\x8a\xfb\xc5\xe6\x51\x64\x9d\xb1\xfd\x82\x93\x6b\x00"
               "\xdb\xbc\x12\x2f\xb4\xc8\x77\x86\x0d\x38\x5c\x49\x50\xd5\x6d"
               "\xe7\xe0\x96\xd6\x13\xd7\xa3\xf2\x7e\xd8\xf2\x63\x34\xb0\xcc"
               "\xc1\x40\x7b\x41\xdc\xcb\x23\xdf\xaa\x52\x98\x18\xd1\x12\x5c"
               "\xd5\x34\x80\x92\x52\x43\x66\xb8\x5f\xab\xb9\x7c\x6c\xd1\xe6"
               "\x06\x6f\x45\x9b\xcc\x56\x6d\xa8\x7e\xc9\xb7\xba\x36\x79\x2d"
               "\x11\x8a\xc3\x9a\x4c\xce\xf6\x19\x2b\xbf\x3a\x54\xaf\x18\xe5"
               "\x7b\x0c\x14\x61\x01\xf6\xae\xaa\x82\x2b\xc4\xb4\xc9\x70\x8b"
               "\x09\xf0\xb3\xba\xb4\x1b\xcc\xe9\x64\xd9\x99\xd1\x10\x7b\xd7"
               "\xc2";
    d.inLen  = 32;
    d.outLen = 136;
    /* Taken from NIST CAVP test vectors - more than one output block. */
    e.input  = "\x8d\x80\x01\xe2\xc0\x96\xf1\xb8\x8e\x7c\x92\x24\xa0\x86\xef"
               "\xd4\x79\x7f\xbf\x74\xa8\x03\x3a\x2d\x42\x2a\x2b\x6b\x8f\x67"
               "\x47\xe4";
    e.output = "\x2e\x97\x5f\x6a\x8a\x14\xf0\x70\x4d\x51\xb1\x36\x67\xd8\x19"
               "\x5c\x21\x9f\x71\xe6\x34\x56\x96\xc4\x9f\xa4\xb9\xd0\x8e\x92"
               "\x25\xd3\xd3\x93\x93\x42\x51\x52\xc9\x7e\x71\xdd\x24\x60\x1c"
               "\x11\xab\xcf\xa0\xf1\x2f\x53\xc6\x80\xbd\x3a\xe7\x57\xb8\x13"
               "\x4a\x9c\x10\xd4\x29\x61\x58\x69\x21\x7f\xdd\x58\x85\xc4\xdb"
               "\x17\x49\x85\x70\x3a\x6d\x6d\xe9\x4a\x66\x7e\xac\x30\x23\x44"
               "\x3a\x83\x37\xae\x1b\xc6\x01\xb7\x6d\x7d\x38\xec\x3c\x34\x46"
               "\x31\x05\xf0\xd3\x94\x9d\x78\xe5\x62\xa0\x39\xe4\x46\x95\x48"
               "\xb6\x09\x39\x5d\xe5\xa4\xfd\x43\xc4\x6c\xa9\xfd\x6e\xe2\x9a"
               "\xda\x5e\xfc\x07\xd8\x4d\x55\x32\x49\x45\x0d\xab\x4a\x49\xc4"
               "\x83\xde\xd2\x50\xc9\x33\x8f\x85\xcd\x93\x7a\xe6\x6b\xb4\x36"
               "\xf3\xb4\x02\x6e\x85\x9f\xda\x1c\xa5\x71\x43\x2f\x3b\xfc\x09"
               "\xe7\xc0\x3c\xa4\xd1\x83\xb7\x41\x11\x1c\xa0\x48\x3d\x0e\xda"
               "\xbc\x03\xfe\xb2\x3b\x17\xee\x48\xe8\x44\xba\x24\x08\xd9\xdc"
               "\xfd\x01\x39\xd2\xe8\xc7\x31\x01\x25\xae\xe8\x01\xc6\x1a\xb7"
               "\x90\x0d\x1e\xfc\x47\xc0\x78\x28\x17\x66\xf3\x61\xc5\xe6\x11"
               "\x13\x46\x23\x5e\x1d\xc3\x83\x25\x66\x6c";
    e.inLen  = 32;
    e.outLen = 250;

    test_sha[0] = a;
    test_sha[1] = b;
    test_sha[2] = c;
    test_sha[3] = d;
    test_sha[4] = e;

    ret = wc_InitShake256(&sha, HEAP_HINT, devId);
    if (ret != 0)
        return -3100;

    for (i = 0; i < times; ++i) {
        ret = wc_Shake256_Update(&sha, (byte*)test_sha[i].input,
            (word32)test_sha[i].inLen);
        if (ret != 0)
            ERROR_OUT(-3101 - i, exit);
        ret = wc_Shake256_Final(&sha, hash, (word32)test_sha[i].outLen);
        if (ret != 0)
            ERROR_OUT(-3102 - i, exit);

        if (XMEMCMP(hash, test_sha[i].output, test_sha[i].outLen) != 0)
            ERROR_OUT(-3103 - i, exit);
    }

    /* BEGIN LARGE HASH TEST */ {
    for (i = 0; i < (int)sizeof(large_input); i++) {
        large_input[i] = (byte)(i & 0xFF);
    }
    times = 100;
    for (i = 0; i < times; ++i) {
        ret = wc_Shake256_Update(&sha, (byte*)large_input,
            (word32)sizeof(large_input));
        if (ret != 0)
            ERROR_OUT(-3104, exit);
    }
    ret = wc_Shake256_Final(&sha, hash, (word32)sizeof(hash));
    if (ret != 0)
        ERROR_OUT(-3105, exit);
    if (XMEMCMP(hash, large_digest, 114) != 0)
        ERROR_OUT(-3106, exit);
    } /* END LARGE HASH TEST */

exit:
    wc_Shake256_Free(&sha);

    return ret;
}
#endif

#ifndef NO_HASH_WRAPPER
WOLFSSL_TEST_SUBROUTINE int hash_test(void)
{
    wc_HashAlg       hash;
    int              ret, exp_ret;
    int              i, j;
    int              digestSz;
    byte             data[] = "0123456789abcdef0123456789abcdef0123456";
    byte             out[WC_MAX_DIGEST_SIZE];
    byte             hashOut[WC_MAX_DIGEST_SIZE];
#if !defined(NO_ASN) || !defined(NO_DH) || defined(HAVE_ECC)
    enum wc_HashType hashType;
#endif
    enum wc_HashType typesGood[] = { WC_HASH_TYPE_MD5, WC_HASH_TYPE_SHA,
                                     WC_HASH_TYPE_SHA224, WC_HASH_TYPE_SHA256,
                                     WC_HASH_TYPE_SHA384, WC_HASH_TYPE_SHA512,
                                     WC_HASH_TYPE_SHA3_224,
                                     WC_HASH_TYPE_SHA3_256,
                                     WC_HASH_TYPE_SHA3_384,
                                     WC_HASH_TYPE_SHA3_512 };
    enum wc_HashType typesNoImpl[] = {
#ifdef NO_MD5
                                        WC_HASH_TYPE_MD5,
#endif
#ifdef NO_SHA
                                        WC_HASH_TYPE_SHA,
#endif
#ifndef WOLFSSL_SHA224
                                        WC_HASH_TYPE_SHA224,
#endif
#ifdef NO_SHA256
                                        WC_HASH_TYPE_SHA256,
#endif
#ifndef WOLFSSL_SHA384
                                        WC_HASH_TYPE_SHA384,
#endif
#ifndef WOLFSSL_SHA512
                                        WC_HASH_TYPE_SHA512,
#endif
#if !defined(WOLFSSL_SHA3) || defined(WOLFSSL_NOSHA3_224)
                                        WC_HASH_TYPE_SHA3_224,
#endif
#if !defined(WOLFSSL_SHA3) || defined(WOLFSSL_NOSHA3_256)
                                        WC_HASH_TYPE_SHA3_256,
#endif
#if !defined(WOLFSSL_SHA3) || defined(WOLFSSL_NOSHA3_384)
                                        WC_HASH_TYPE_SHA3_384,
#endif
#if !defined(WOLFSSL_SHA3) || defined(WOLFSSL_NOSHA3_512)
                                        WC_HASH_TYPE_SHA3_512,
#endif
                                        WC_HASH_TYPE_NONE
                                     };
    enum wc_HashType typesBad[]  = { WC_HASH_TYPE_NONE, WC_HASH_TYPE_MD5_SHA,
                                     WC_HASH_TYPE_MD2, WC_HASH_TYPE_MD4 };
    enum wc_HashType typesHashBad[] = { WC_HASH_TYPE_MD2, WC_HASH_TYPE_MD4,
                                        WC_HASH_TYPE_BLAKE2B,
                                        WC_HASH_TYPE_NONE };

    /* Parameter Validation testing. */
    ret = wc_HashInit(NULL, WC_HASH_TYPE_SHA256);
    if (ret != BAD_FUNC_ARG)
        return -3200;
    ret = wc_HashUpdate(NULL, WC_HASH_TYPE_SHA256, NULL, sizeof(data));
    if (ret != BAD_FUNC_ARG)
        return -3201;
    ret = wc_HashUpdate(&hash, WC_HASH_TYPE_SHA256, NULL, sizeof(data));
    if (ret != BAD_FUNC_ARG)
        return -3202;
    ret = wc_HashUpdate(NULL, WC_HASH_TYPE_SHA256, data, sizeof(data));
    if (ret != BAD_FUNC_ARG)
        return -3203;
    ret = wc_HashFinal(NULL, WC_HASH_TYPE_SHA256, NULL);
    if (ret != BAD_FUNC_ARG)
        return -3204;
    ret = wc_HashFinal(&hash, WC_HASH_TYPE_SHA256, NULL);
    if (ret != BAD_FUNC_ARG)
        return -3205;
    ret = wc_HashFinal(NULL, WC_HASH_TYPE_SHA256, out);
    if (ret != BAD_FUNC_ARG)
        return -3206;

    /* Try invalid hash algorithms. */
    for (i = 0; i < (int)(sizeof(typesBad)/sizeof(*typesBad)); i++) {
        ret = wc_HashInit(&hash, typesBad[i]);
        if (ret != BAD_FUNC_ARG)
            return -3207 - i;
        ret = wc_HashUpdate(&hash, typesBad[i], data, sizeof(data));
        if (ret != BAD_FUNC_ARG)
            return -3217 - i;
        ret = wc_HashFinal(&hash, typesBad[i], out);
        if (ret != BAD_FUNC_ARG)
            return -3227 - i;
        wc_HashFree(&hash, typesBad[i]);
    }

    /* Try valid hash algorithms. */
    for (i = 0, j = 0; i < (int)(sizeof(typesGood)/sizeof(*typesGood)); i++) {
        exp_ret = 0;
        if (typesGood[i] == typesNoImpl[j]) {
            /* Recognized but no implementation compiled in. */
            exp_ret = HASH_TYPE_E;
            j++;
        }
        ret = wc_HashInit(&hash, typesGood[i]);
        if (ret != exp_ret)
            return -3237 - i;
        ret = wc_HashUpdate(&hash, typesGood[i], data, sizeof(data));
        if (ret != exp_ret)
            return -3247 - i;
        ret = wc_HashFinal(&hash, typesGood[i], out);
        if (ret != exp_ret)
            return -3257 - i;
        wc_HashFree(&hash, typesGood[i]);

        digestSz = wc_HashGetDigestSize(typesGood[i]);
        if (exp_ret < 0 && digestSz != exp_ret)
            return -3267 - i;
        if (exp_ret == 0 && digestSz < 0)
            return -3277 - i;
        if (exp_ret == 0) {
            ret = wc_Hash(typesGood[i], data, sizeof(data), hashOut,
                                                                  digestSz - 1);
            if (ret != BUFFER_E)
                return -3287 - i;
        }
        ret = wc_Hash(typesGood[i], data, sizeof(data), hashOut, digestSz);
        if (ret != exp_ret)
            return -3297 - i;
        if (exp_ret == 0 && XMEMCMP(out, hashOut, digestSz) != 0)
            return -3307 -i;

        ret = wc_HashGetBlockSize(typesGood[i]);
        if (exp_ret < 0 && ret != exp_ret)
            return -3308 - i;
        if (exp_ret == 0 && ret < 0)
            return -3318 - i;

#if !defined(NO_ASN) || !defined(NO_DH) || defined(HAVE_ECC)
        ret = wc_HashGetOID(typesGood[i]);
        if (ret == BAD_FUNC_ARG ||
                (exp_ret == 0 && ret == HASH_TYPE_E) ||
                (exp_ret != 0 && ret != HASH_TYPE_E)) {
            return -3328 - i;
        }

        hashType = wc_OidGetHash(ret);
        if (exp_ret == 0 && hashType != typesGood[i])
            return -3338 - i;
#endif /* !defined(NO_ASN) || !defined(NO_DH) || defined(HAVE_ECC) */
    }

    for (i = 0; i < (int)(sizeof(typesHashBad)/sizeof(*typesHashBad)); i++) {
        ret = wc_Hash(typesHashBad[i], data, sizeof(data), out, sizeof(out));
        if (ret != BAD_FUNC_ARG && ret != BUFFER_E)
            return -3348 - i;
    }

#if !defined(NO_ASN) || !defined(NO_DH) || defined(HAVE_ECC)
    ret = wc_HashGetOID(WC_HASH_TYPE_MD2);
#ifdef WOLFSSL_MD2
    if (ret == HASH_TYPE_E || ret == BAD_FUNC_ARG)
        return -3358;
#else
    if (ret != HASH_TYPE_E)
        return -3359;
#endif
    hashType = wc_OidGetHash(646); /* Md2h */
#ifdef WOLFSSL_MD2
    if (hashType != WC_HASH_TYPE_MD2)
        return -3360;
#else
    if (hashType != WC_HASH_TYPE_NONE)
        return -3361;
#endif

    ret = wc_HashGetOID(WC_HASH_TYPE_MD5_SHA);
#ifndef NO_MD5
    if (ret == HASH_TYPE_E || ret == BAD_FUNC_ARG)
        return -3362;
#else
    if (ret != HASH_TYPE_E)
        return -3363;
#endif
    ret = wc_HashGetOID(WC_HASH_TYPE_MD4);
    if (ret != BAD_FUNC_ARG)
        return -3364;
    ret = wc_HashGetOID(WC_HASH_TYPE_NONE);
    if (ret != BAD_FUNC_ARG)
        return -3365;

    hashType = wc_OidGetHash(0);
    if (hashType != WC_HASH_TYPE_NONE)
        return -3366;
#endif /* !defined(NO_ASN) || !defined(NO_DH) || defined(HAVE_ECC) */

    ret = wc_HashGetBlockSize(WC_HASH_TYPE_MD2);
#ifdef WOLFSSL_MD2
    if (ret == HASH_TYPE_E || ret == BAD_FUNC_ARG)
        return -3367;
#else
    if (ret != HASH_TYPE_E)
        return -3368;
#endif
    ret = wc_HashGetDigestSize(WC_HASH_TYPE_MD2);
#ifdef WOLFSSL_MD2
    if (ret == HASH_TYPE_E || ret == BAD_FUNC_ARG)
        return -3369;
#else
    if (ret != HASH_TYPE_E)
        return -3370;
#endif

    ret = wc_HashGetBlockSize(WC_HASH_TYPE_MD4);
#ifndef NO_MD4
    if (ret == HASH_TYPE_E || ret == BAD_FUNC_ARG)
        return -3371;
#else
    if (ret != HASH_TYPE_E)
        return -3372;
#endif
    ret = wc_HashGetDigestSize(WC_HASH_TYPE_MD4);
#ifndef NO_MD4
    if (ret == HASH_TYPE_E || ret == BAD_FUNC_ARG)
        return -3373;
#else
    if (ret != HASH_TYPE_E)
        return -3374;
#endif
    ret = wc_HashGetBlockSize(WC_HASH_TYPE_MD5_SHA);
#if !defined(NO_MD5) && !defined(NO_SHA)
    if (ret == HASH_TYPE_E || ret == BAD_FUNC_ARG)
        return -3375;
#else
    if (ret != HASH_TYPE_E)
        return -3376;
#endif

    ret = wc_HashGetBlockSize(WC_HASH_TYPE_BLAKE2B);
#if defined(HAVE_BLAKE2) || defined(HAVE_BLAKE2S)
    if (ret == HASH_TYPE_E || ret == BAD_FUNC_ARG)
        return -3377;
#else
    if (ret != HASH_TYPE_E)
        return -3378;
#endif
    ret = wc_HashGetDigestSize(WC_HASH_TYPE_BLAKE2B);
#if defined(HAVE_BLAKE2) || defined(HAVE_BLAKE2S)
    if (ret == HASH_TYPE_E || ret == BAD_FUNC_ARG)
        return -3379;
#else
    if (ret != HASH_TYPE_E)
        return -3380;
#endif

    ret = wc_HashGetBlockSize(WC_HASH_TYPE_NONE);
    if (ret != BAD_FUNC_ARG)
        return -3381;
    ret = wc_HashGetDigestSize(WC_HASH_TYPE_NONE);
    if (ret != BAD_FUNC_ARG)
        return -3382;

#if !defined(NO_CERTS) && !defined(NO_ASN)
#if defined(WOLFSSL_MD2) && !defined(HAVE_SELFTEST) && !defined(HAVE_FIPS)
    ret = wc_GetCTC_HashOID(MD2);
    if (ret == 0)
        return -3383;
#endif
#ifndef NO_MD5
    ret = wc_GetCTC_HashOID(WC_MD5);
    if (ret == 0)
        return -3384;
#endif
#ifndef NO_SHA
    ret = wc_GetCTC_HashOID(WC_SHA);
    if (ret == 0)
        return -3385;
#endif
#ifdef WOLFSSL_SHA224
    ret = wc_GetCTC_HashOID(WC_SHA224);
    if (ret == 0)
        return -3386;
#endif
#ifndef NO_SHA256
    ret = wc_GetCTC_HashOID(WC_SHA256);
    if (ret == 0)
        return -3387;
#endif
#ifdef WOLFSSL_SHA384
    ret = wc_GetCTC_HashOID(WC_SHA384);
    if (ret == 0)
        return -3388;
#endif
#ifdef WOLFSSL_SHA512
    ret = wc_GetCTC_HashOID(WC_SHA512);
    if (ret == 0)
        return -3389;
#endif
    ret = wc_GetCTC_HashOID(-1);
    if (ret != 0)
        return -3390;
#endif

    return 0;
}
#endif /* !NO_HASH_WRAPPER */

#if !defined(NO_HMAC) && !defined(NO_MD5) && !(defined(HAVE_FIPS) && \
                                               defined(HAVE_FIPS_VERSION) && \
                                               (HAVE_FIPS_VERSION >= 5))
WOLFSSL_TEST_SUBROUTINE int hmac_md5_test(void)
{
    Hmac hmac;
    byte hash[WC_MD5_DIGEST_SIZE];

    const char* keys[]=
    {
        "\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b",
        "Jefe",
        "\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA"
    };

    testVector a, b, c;
    testVector test_hmac[3];

    int ret;
    int times = sizeof(test_hmac) / sizeof(testVector), i;

    a.input  = "Hi There";
    a.output = "\x92\x94\x72\x7a\x36\x38\xbb\x1c\x13\xf4\x8e\xf8\x15\x8b\xfc"
               "\x9d";
    a.inLen  = XSTRLEN(a.input);
    a.outLen = WC_MD5_DIGEST_SIZE;

    b.input  = "what do ya want for nothing?";
    b.output = "\x75\x0c\x78\x3e\x6a\xb0\xb5\x03\xea\xa8\x6e\x31\x0a\x5d\xb7"
               "\x38";
    b.inLen  = XSTRLEN(b.input);
    b.outLen = WC_MD5_DIGEST_SIZE;

    c.input  = "\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD"
               "\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD"
               "\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD"
               "\xDD\xDD\xDD\xDD\xDD\xDD";
    c.output = "\x56\xbe\x34\x52\x1d\x14\x4c\x88\xdb\xb8\xc7\x33\xf0\xe8\xb3"
               "\xf6";
    c.inLen  = XSTRLEN(c.input);
    c.outLen = WC_MD5_DIGEST_SIZE;

    test_hmac[0] = a;
    test_hmac[1] = b;
    test_hmac[2] = c;

    for (i = 0; i < times; ++i) {
    #if defined(HAVE_FIPS) || defined(HAVE_CAVIUM)
        if (i == 1) {
            continue; /* cavium can't handle short keys, fips not allowed */
        }
    #endif

        if (wc_HmacInit(&hmac, HEAP_HINT, devId) != 0) {
            return -3400;
        }

        ret = wc_HmacSetKey(&hmac, WC_MD5, (byte*)keys[i],
            (word32)XSTRLEN(keys[i]));
        if (ret != 0)
            return -3401;
        ret = wc_HmacUpdate(&hmac, (byte*)test_hmac[i].input,
                   (word32)test_hmac[i].inLen);
        if (ret != 0)
            return -3402;
        ret = wc_HmacFinal(&hmac, hash);
        if (ret != 0)
            return -3403;

        if (XMEMCMP(hash, test_hmac[i].output, WC_MD5_DIGEST_SIZE) != 0)
            return -3404 - i;

        wc_HmacFree(&hmac);
    }

#ifndef HAVE_FIPS
    if (wc_HmacSizeByType(WC_MD5) != WC_MD5_DIGEST_SIZE)
        return -3414;
#endif

    return 0;
}
#endif /* !NO_HMAC && !NO_MD5 && (!HAVE_FIPS || (HAVE_FIPS_VERSION < 5)) */

#if !defined(NO_HMAC) && !defined(NO_SHA)
WOLFSSL_TEST_SUBROUTINE int hmac_sha_test(void)
{
    Hmac hmac;
    byte hash[WC_SHA_DIGEST_SIZE];

    const char* keys[]=
    {
        "\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b"
                                                                "\x0b\x0b\x0b",
        "Jefe",
        "\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA"
                                                                "\xAA\xAA\xAA"
    };

    testVector a, b, c;
    testVector test_hmac[3];

    int ret;
    int times = sizeof(test_hmac) / sizeof(testVector), i;

    a.input  = "Hi There";
    a.output = "\xb6\x17\x31\x86\x55\x05\x72\x64\xe2\x8b\xc0\xb6\xfb\x37\x8c"
               "\x8e\xf1\x46\xbe\x00";
    a.inLen  = XSTRLEN(a.input);
    a.outLen = WC_SHA_DIGEST_SIZE;

    b.input  = "what do ya want for nothing?";
    b.output = "\xef\xfc\xdf\x6a\xe5\xeb\x2f\xa2\xd2\x74\x16\xd5\xf1\x84\xdf"
               "\x9c\x25\x9a\x7c\x79";
    b.inLen  = XSTRLEN(b.input);
    b.outLen = WC_SHA_DIGEST_SIZE;

    c.input  = "\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD"
               "\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD"
               "\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD"
               "\xDD\xDD\xDD\xDD\xDD\xDD";
    c.output = "\x12\x5d\x73\x42\xb9\xac\x11\xcd\x91\xa3\x9a\xf4\x8a\xa1\x7b"
               "\x4f\x63\xf1\x75\xd3";
    c.inLen  = XSTRLEN(c.input);
    c.outLen = WC_SHA_DIGEST_SIZE;

    test_hmac[0] = a;
    test_hmac[1] = b;
    test_hmac[2] = c;

    for (i = 0; i < times; ++i) {
#if defined(HAVE_FIPS) || defined(HAVE_CAVIUM)
        if (i == 1)
            continue; /* cavium can't handle short keys, fips not allowed */
#endif

        if (wc_HmacInit(&hmac, HEAP_HINT, devId) != 0)
            return -3500;

        ret = wc_HmacSetKey(&hmac, WC_SHA, (byte*)keys[i],
            (word32)XSTRLEN(keys[i]));
        if (ret != 0)
            return -3501;
        ret = wc_HmacUpdate(&hmac, (byte*)test_hmac[i].input,
                   (word32)test_hmac[i].inLen);
        if (ret != 0)
            return -3502;
        ret = wc_HmacFinal(&hmac, hash);
        if (ret != 0)
            return -3503;

        if (XMEMCMP(hash, test_hmac[i].output, WC_SHA_DIGEST_SIZE) != 0)
            return -3504 - i;

        wc_HmacFree(&hmac);
    }

#ifndef HAVE_FIPS
    if (wc_HmacSizeByType(WC_SHA) != WC_SHA_DIGEST_SIZE)
        return -3514;
#endif

    return 0;
}
#endif


#if !defined(NO_HMAC) && defined(WOLFSSL_SHA224)
WOLFSSL_TEST_SUBROUTINE int hmac_sha224_test(void)
{
    Hmac hmac;
    byte hash[WC_SHA224_DIGEST_SIZE];

    const char* keys[]=
    {
        "\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b"
                                                                "\x0b\x0b\x0b",
        "Jefe",
        "\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA"
                                                                "\xAA\xAA\xAA",
        "\x01\x02\x03\x04\x05\x06\x07\x08\x01\x02\x03\x04\x05\x06\x07\x08"
        "\x01\x02\x03\x04\x05\x06\x07\x08\x01\x02\x03\x04\x05\x06\x07\x08"
        "\x01\x02\x03\x04\x05\x06\x07\x08\x01\x02\x03\x04\x05\x06\x07\x08"
        "\x01\x02\x03\x04\x05\x06\x07\x08\x01\x02\x03\x04\x05\x06\x07\x08"
        "\x01\x02\x03\x04\x05\x06\x07\x08\x01\x02\x03\x04\x05\x06\x07\x08"
    };

    testVector a, b, c, d;
    testVector test_hmac[4];

    int ret;
    int times = sizeof(test_hmac) / sizeof(testVector), i;

    a.input  = "Hi There";
    a.output = "\x89\x6f\xb1\x12\x8a\xbb\xdf\x19\x68\x32\x10\x7c\xd4\x9d\xf3"
               "\x3f\x47\xb4\xb1\x16\x99\x12\xba\x4f\x53\x68\x4b\x22";
    a.inLen  = XSTRLEN(a.input);
    a.outLen = WC_SHA224_DIGEST_SIZE;

    b.input  = "what do ya want for nothing?";
    b.output = "\xa3\x0e\x01\x09\x8b\xc6\xdb\xbf\x45\x69\x0f\x3a\x7e\x9e\x6d"
               "\x0f\x8b\xbe\xa2\xa3\x9e\x61\x48\x00\x8f\xd0\x5e\x44";
    b.inLen  = XSTRLEN(b.input);
    b.outLen = WC_SHA224_DIGEST_SIZE;

    c.input  = "\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD"
               "\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD"
               "\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD"
               "\xDD\xDD\xDD\xDD\xDD\xDD";
    c.output = "\x7f\xb3\xcb\x35\x88\xc6\xc1\xf6\xff\xa9\x69\x4d\x7d\x6a\xd2"
               "\x64\x93\x65\xb0\xc1\xf6\x5d\x69\xd1\xec\x83\x33\xea";
    c.inLen  = XSTRLEN(c.input);
    c.outLen = WC_SHA224_DIGEST_SIZE;

    d.input  = "Big Key Input";
    d.output = "\xe7\x4e\x2b\x8a\xa9\xf0\x37\x2f\xed\xae\x70\x0c\x49\x47\xf1"
               "\x46\x54\xa7\x32\x6b\x55\x01\x87\xd2\xc8\x02\x0e\x3a";
    d.inLen  = XSTRLEN(d.input);
    d.outLen = WC_SHA224_DIGEST_SIZE;

    test_hmac[0] = a;
    test_hmac[1] = b;
    test_hmac[2] = c;
    test_hmac[3] = d;

    for (i = 0; i < times; ++i) {
#if defined(HAVE_FIPS) || defined(HAVE_CAVIUM)
        if (i == 1)
            continue; /* cavium can't handle short keys, fips not allowed */
#endif

        if (wc_HmacInit(&hmac, HEAP_HINT, devId) != 0)
            return -3600;

        ret = wc_HmacSetKey(&hmac, WC_SHA224, (byte*)keys[i],
            (word32)XSTRLEN(keys[i]));
        if (ret != 0)
            return -3601;
        ret = wc_HmacUpdate(&hmac, (byte*)test_hmac[i].input,
                   (word32)test_hmac[i].inLen);
        if (ret != 0)
            return -3602;
        ret = wc_HmacFinal(&hmac, hash);
        if (ret != 0)
            return -3603;

        if (XMEMCMP(hash, test_hmac[i].output, WC_SHA224_DIGEST_SIZE) != 0)
            return -3604 - i;

        wc_HmacFree(&hmac);
    }

#ifndef HAVE_FIPS
    if (wc_HmacSizeByType(WC_SHA224) != WC_SHA224_DIGEST_SIZE)
        return -3614;
#endif

    return 0;
}
#endif


#if !defined(NO_HMAC) && !defined(NO_SHA256)
WOLFSSL_TEST_SUBROUTINE int hmac_sha256_test(void)
{
    Hmac hmac;
    byte hash[WC_SHA256_DIGEST_SIZE];

    const char* keys[]=
    {
        "\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b"
                                                                "\x0b\x0b\x0b",
        "Jefe",
        "\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA"
                                                                "\xAA\xAA\xAA",
        "\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA"
                                                                "\xAA\xAA\xAA",
    };

    testVector a, b, c, d;
    testVector test_hmac[4];

    int ret;
    int times = sizeof(test_hmac) / sizeof(testVector), i;

    a.input  = "Hi There";
    a.output = "\xb0\x34\x4c\x61\xd8\xdb\x38\x53\x5c\xa8\xaf\xce\xaf\x0b\xf1"
               "\x2b\x88\x1d\xc2\x00\xc9\x83\x3d\xa7\x26\xe9\x37\x6c\x2e\x32"
               "\xcf\xf7";
    a.inLen  = XSTRLEN(a.input);
    a.outLen = WC_SHA256_DIGEST_SIZE;

    b.input  = "what do ya want for nothing?";
    b.output = "\x5b\xdc\xc1\x46\xbf\x60\x75\x4e\x6a\x04\x24\x26\x08\x95\x75"
               "\xc7\x5a\x00\x3f\x08\x9d\x27\x39\x83\x9d\xec\x58\xb9\x64\xec"
               "\x38\x43";
    b.inLen  = XSTRLEN(b.input);
    b.outLen = WC_SHA256_DIGEST_SIZE;

    c.input  = "\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD"
               "\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD"
               "\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD"
               "\xDD\xDD\xDD\xDD\xDD\xDD";
    c.output = "\x77\x3e\xa9\x1e\x36\x80\x0e\x46\x85\x4d\xb8\xeb\xd0\x91\x81"
               "\xa7\x29\x59\x09\x8b\x3e\xf8\xc1\x22\xd9\x63\x55\x14\xce\xd5"
               "\x65\xfe";
    c.inLen  = XSTRLEN(c.input);
    c.outLen = WC_SHA256_DIGEST_SIZE;

    d.input  = 0;
    d.output = "\x86\xe5\x4f\xd4\x48\x72\x5d\x7e\x5d\xcf\xe2\x23\x53\xc8\x28"
               "\xaf\x48\x78\x1e\xb4\x8c\xae\x81\x06\xa7\xe1\xd4\x98\x94\x9f"
               "\x3e\x46";
    d.inLen  = 0;
    d.outLen = WC_SHA256_DIGEST_SIZE;

    test_hmac[0] = a;
    test_hmac[1] = b;
    test_hmac[2] = c;
    test_hmac[3] = d;

    for (i = 0; i < times; ++i) {
#if defined(HAVE_FIPS) || defined(HAVE_CAVIUM)
        if (i == 1)
            continue; /* cavium can't handle short keys, fips not allowed */
#endif
#if defined(HAVE_INTEL_QA) || defined(HAVE_CAVIUM)
        if (i == 3)
            continue; /* QuickAssist can't handle empty HMAC */
#endif

        if (wc_HmacInit(&hmac, HEAP_HINT, devId) != 0)
            return -3700 - i;

        ret = wc_HmacSetKey(&hmac, WC_SHA256, (byte*)keys[i],
            (word32)XSTRLEN(keys[i]));
        if (ret != 0)
            return -3710 - i;
        if (test_hmac[i].input != NULL) {
            ret = wc_HmacUpdate(&hmac, (byte*)test_hmac[i].input,
                       (word32)test_hmac[i].inLen);
            if (ret != 0)
                return -3720 - i;
        }
        ret = wc_HmacFinal(&hmac, hash);
        if (ret != 0)
            return -3730 - i;

        if (XMEMCMP(hash, test_hmac[i].output, WC_SHA256_DIGEST_SIZE) != 0)
            return -3740 - i;

        wc_HmacFree(&hmac);
    }

#ifndef HAVE_FIPS
    if (wc_HmacSizeByType(WC_SHA256) != WC_SHA256_DIGEST_SIZE)
        return -3750;
    if (wc_HmacSizeByType(20) != BAD_FUNC_ARG)
        return -3751;
#endif
    if (wolfSSL_GetHmacMaxSize() != WC_MAX_DIGEST_SIZE)
        return -3752;

    return 0;
}
#endif


#if !defined(NO_HMAC) && defined(WOLFSSL_SHA384)
WOLFSSL_TEST_SUBROUTINE int hmac_sha384_test(void)
{
    Hmac hmac;
    byte hash[WC_SHA384_DIGEST_SIZE];

    const char* keys[]=
    {
        "\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b"
                                                                "\x0b\x0b\x0b",
        "Jefe",
        "\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA"
                                                                "\xAA\xAA\xAA",
        "\x01\x02\x03\x04\x05\x06\x07\x08\x01\x02\x03\x04\x05\x06\x07\x08"
        "\x01\x02\x03\x04\x05\x06\x07\x08\x01\x02\x03\x04\x05\x06\x07\x08"
        "\x01\x02\x03\x04\x05\x06\x07\x08\x01\x02\x03\x04\x05\x06\x07\x08"
        "\x01\x02\x03\x04\x05\x06\x07\x08\x01\x02\x03\x04\x05\x06\x07\x08"
        "\x01\x02\x03\x04\x05\x06\x07\x08\x01\x02\x03\x04\x05\x06\x07\x08"
        "\x01\x02\x03\x04\x05\x06\x07\x08\x01\x02\x03\x04\x05\x06\x07\x08"
        "\x01\x02\x03\x04\x05\x06\x07\x08\x01\x02\x03\x04\x05\x06\x07\x08"
        "\x01\x02\x03\x04\x05\x06\x07\x08\x01\x02\x03\x04\x05\x06\x07\x08"
        "\x01\x02\x03\x04\x05\x06\x07\x08\x01\x02\x03\x04\x05\x06\x07\x08"
    };

    testVector a, b, c, d;
    testVector test_hmac[4];

    int ret;
    int times = sizeof(test_hmac) / sizeof(testVector), i;

    a.input  = "Hi There";
    a.output = "\xaf\xd0\x39\x44\xd8\x48\x95\x62\x6b\x08\x25\xf4\xab\x46\x90"
               "\x7f\x15\xf9\xda\xdb\xe4\x10\x1e\xc6\x82\xaa\x03\x4c\x7c\xeb"
               "\xc5\x9c\xfa\xea\x9e\xa9\x07\x6e\xde\x7f\x4a\xf1\x52\xe8\xb2"
               "\xfa\x9c\xb6";
    a.inLen  = XSTRLEN(a.input);
    a.outLen = WC_SHA384_DIGEST_SIZE;

    b.input  = "what do ya want for nothing?";
    b.output = "\xaf\x45\xd2\xe3\x76\x48\x40\x31\x61\x7f\x78\xd2\xb5\x8a\x6b"
               "\x1b\x9c\x7e\xf4\x64\xf5\xa0\x1b\x47\xe4\x2e\xc3\x73\x63\x22"
               "\x44\x5e\x8e\x22\x40\xca\x5e\x69\xe2\xc7\x8b\x32\x39\xec\xfa"
               "\xb2\x16\x49";
    b.inLen  = XSTRLEN(b.input);
    b.outLen = WC_SHA384_DIGEST_SIZE;

    c.input  = "\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD"
               "\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD"
               "\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD"
               "\xDD\xDD\xDD\xDD\xDD\xDD";
    c.output = "\x88\x06\x26\x08\xd3\xe6\xad\x8a\x0a\xa2\xac\xe0\x14\xc8\xa8"
               "\x6f\x0a\xa6\x35\xd9\x47\xac\x9f\xeb\xe8\x3e\xf4\xe5\x59\x66"
               "\x14\x4b\x2a\x5a\xb3\x9d\xc1\x38\x14\xb9\x4e\x3a\xb6\xe1\x01"
               "\xa3\x4f\x27";
    c.inLen  = XSTRLEN(c.input);
    c.outLen = WC_SHA384_DIGEST_SIZE;

    d.input  = "Big Key Input";
    d.output = "\xd2\x3d\x29\x6e\xf5\x1e\x23\x23\x49\x18\xb3\xbf\x4c\x38\x7b"
               "\x31\x21\x17\xbb\x09\x73\x27\xf8\x12\x9d\xe9\xc6\x5d\xf9\x54"
               "\xd6\x38\x5a\x68\x53\x14\xee\xe0\xa6\x4f\x36\x7e\xb2\xf3\x1a"
               "\x57\x41\x69";
    d.inLen  = XSTRLEN(d.input);
    d.outLen = WC_SHA384_DIGEST_SIZE;

    test_hmac[0] = a;
    test_hmac[1] = b;
    test_hmac[2] = c;
    test_hmac[3] = d;

    for (i = 0; i < times; ++i) {
#if defined(HAVE_FIPS)
        if (i == 1)
            continue; /* fips not allowed */
#endif

        if (wc_HmacInit(&hmac, HEAP_HINT, devId) != 0)
            return -3800;

        ret = wc_HmacSetKey(&hmac, WC_SHA384, (byte*)keys[i],
            (word32)XSTRLEN(keys[i]));
        if (ret != 0)
            return -3801;
        ret = wc_HmacUpdate(&hmac, (byte*)test_hmac[i].input,
                   (word32)test_hmac[i].inLen);
        if (ret != 0)
            return -3802;
        ret = wc_HmacFinal(&hmac, hash);
        if (ret != 0)
            return -3803;

        if (XMEMCMP(hash, test_hmac[i].output, WC_SHA384_DIGEST_SIZE) != 0)
            return -3804 - i;

        wc_HmacFree(&hmac);
    }

#ifndef HAVE_FIPS
    if (wc_HmacSizeByType(WC_SHA384) != WC_SHA384_DIGEST_SIZE)
        return -3814;
#endif

    return 0;
}
#endif


#if !defined(NO_HMAC) && defined(WOLFSSL_SHA512)
WOLFSSL_TEST_SUBROUTINE int hmac_sha512_test(void)
{
    Hmac hmac;
    byte hash[WC_SHA512_DIGEST_SIZE];

    const char* keys[]=
    {
        "\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b"
                                                                "\x0b\x0b\x0b",
        "Jefe",
        "\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA"
                                                                "\xAA\xAA\xAA",
        "\x01\x02\x03\x04\x05\x06\x07\x08\x01\x02\x03\x04\x05\x06\x07\x08"
        "\x01\x02\x03\x04\x05\x06\x07\x08\x01\x02\x03\x04\x05\x06\x07\x08"
        "\x01\x02\x03\x04\x05\x06\x07\x08\x01\x02\x03\x04\x05\x06\x07\x08"
        "\x01\x02\x03\x04\x05\x06\x07\x08\x01\x02\x03\x04\x05\x06\x07\x08"
        "\x01\x02\x03\x04\x05\x06\x07\x08\x01\x02\x03\x04\x05\x06\x07\x08"
        "\x01\x02\x03\x04\x05\x06\x07\x08\x01\x02\x03\x04\x05\x06\x07\x08"
        "\x01\x02\x03\x04\x05\x06\x07\x08\x01\x02\x03\x04\x05\x06\x07\x08"
        "\x01\x02\x03\x04\x05\x06\x07\x08\x01\x02\x03\x04\x05\x06\x07\x08"
        "\x01\x02\x03\x04\x05\x06\x07\x08\x01\x02\x03\x04\x05\x06\x07\x08"
    };

    testVector a, b, c, d;
    testVector test_hmac[4];

    int ret;
    int times = sizeof(test_hmac) / sizeof(testVector), i;

    a.input  = "Hi There";
    a.output = "\x87\xaa\x7c\xde\xa5\xef\x61\x9d\x4f\xf0\xb4\x24\x1a\x1d\x6c"
               "\xb0\x23\x79\xf4\xe2\xce\x4e\xc2\x78\x7a\xd0\xb3\x05\x45\xe1"
               "\x7c\xde\xda\xa8\x33\xb7\xd6\xb8\xa7\x02\x03\x8b\x27\x4e\xae"
               "\xa3\xf4\xe4\xbe\x9d\x91\x4e\xeb\x61\xf1\x70\x2e\x69\x6c\x20"
               "\x3a\x12\x68\x54";
    a.inLen  = XSTRLEN(a.input);
    a.outLen = WC_SHA512_DIGEST_SIZE;

    b.input  = "what do ya want for nothing?";
    b.output = "\x16\x4b\x7a\x7b\xfc\xf8\x19\xe2\xe3\x95\xfb\xe7\x3b\x56\xe0"
               "\xa3\x87\xbd\x64\x22\x2e\x83\x1f\xd6\x10\x27\x0c\xd7\xea\x25"
               "\x05\x54\x97\x58\xbf\x75\xc0\x5a\x99\x4a\x6d\x03\x4f\x65\xf8"
               "\xf0\xe6\xfd\xca\xea\xb1\xa3\x4d\x4a\x6b\x4b\x63\x6e\x07\x0a"
               "\x38\xbc\xe7\x37";
    b.inLen  = XSTRLEN(b.input);
    b.outLen = WC_SHA512_DIGEST_SIZE;

    c.input  = "\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD"
               "\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD"
               "\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD"
               "\xDD\xDD\xDD\xDD\xDD\xDD";
    c.output = "\xfa\x73\xb0\x08\x9d\x56\xa2\x84\xef\xb0\xf0\x75\x6c\x89\x0b"
               "\xe9\xb1\xb5\xdb\xdd\x8e\xe8\x1a\x36\x55\xf8\x3e\x33\xb2\x27"
               "\x9d\x39\xbf\x3e\x84\x82\x79\xa7\x22\xc8\x06\xb4\x85\xa4\x7e"
               "\x67\xc8\x07\xb9\x46\xa3\x37\xbe\xe8\x94\x26\x74\x27\x88\x59"
               "\xe1\x32\x92\xfb";
    c.inLen  = XSTRLEN(c.input);
    c.outLen = WC_SHA512_DIGEST_SIZE;

    d.input  = "Big Key Input";
    d.output = "\x3f\xa9\xc9\xe1\xbd\xbb\x04\x55\x1f\xef\xcc\x92\x33\x08\xeb"
               "\xcf\xc1\x9a\x5b\x5b\xc0\x7c\x86\x84\xae\x8c\x40\xaf\xb1\x27"
               "\x87\x38\x92\x04\xa8\xed\xd7\xd7\x07\xa9\x85\xa0\xc2\xcd\x30"
               "\xc0\x56\x14\x49\xbc\x2f\x69\x15\x6a\x97\xd8\x79\x2f\xb3\x3b"
               "\x1e\x18\xfe\xfa";
    d.inLen  = XSTRLEN(d.input);
    d.outLen = WC_SHA512_DIGEST_SIZE;

    test_hmac[0] = a;
    test_hmac[1] = b;
    test_hmac[2] = c;
    test_hmac[3] = d;

    for (i = 0; i < times; ++i) {
#if defined(HAVE_FIPS)
        if (i == 1)
            continue; /* fips not allowed */
#endif

        if (wc_HmacInit(&hmac, HEAP_HINT, devId) != 0)
            return -3900;

        ret = wc_HmacSetKey(&hmac, WC_SHA512, (byte*)keys[i],
            (word32)XSTRLEN(keys[i]));
        if (ret != 0)
            return -3901;
        ret = wc_HmacUpdate(&hmac, (byte*)test_hmac[i].input,
                   (word32)test_hmac[i].inLen);
        if (ret != 0)
            return -3902;
        ret = wc_HmacFinal(&hmac, hash);
        if (ret != 0)
            return -3903;

        if (XMEMCMP(hash, test_hmac[i].output, WC_SHA512_DIGEST_SIZE) != 0)
            return -3904 - i;

        wc_HmacFree(&hmac);
    }

#ifndef HAVE_FIPS
    if (wc_HmacSizeByType(WC_SHA512) != WC_SHA512_DIGEST_SIZE)
        return -3914;
#endif

    return 0;
}
#endif


#if !defined(NO_HMAC) && defined(WOLFSSL_SHA3) && \
    !defined(WOLFSSL_NOSHA3_224) && !defined(WOLFSSL_NOSHA3_256) && \
    !defined(WOLFSSL_NOSHA3_384) && !defined(WOLFSSL_NOSHA3_512)
WOLFSSL_TEST_SUBROUTINE int hmac_sha3_test(void)
{
    Hmac hmac;
    byte hash[WC_SHA3_512_DIGEST_SIZE];

    const char* key[4] =
    {
        "Jefe",

        "\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b"
        "\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b",

        "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
        "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa",

        "\x01\x02\x03\x04\x05\x06\x07\x08\x01\x02\x03\x04\x05\x06\x07\x08"
        "\x01\x02\x03\x04\x05\x06\x07\x08\x01\x02\x03\x04\x05\x06\x07\x08"
        "\x01\x02\x03\x04\x05\x06\x07\x08\x01\x02\x03\x04\x05\x06\x07\x08"
        "\x01\x02\x03\x04\x05\x06\x07\x08\x01\x02\x03\x04\x05\x06\x07\x08"
        "\x01\x02\x03\x04\x05\x06\x07\x08\x01\x02\x03\x04\x05\x06\x07\x08"
        "\x01\x02\x03\x04\x05\x06\x07\x08\x01\x02\x03\x04\x05\x06\x07\x08"
        "\x01\x02\x03\x04\x05\x06\x07\x08\x01\x02\x03\x04\x05\x06\x07\x08"
        "\x01\x02\x03\x04\x05\x06\x07\x08\x01\x02\x03\x04\x05\x06\x07\x08"
        "\x01\x02\x03\x04\x05\x06\x07\x08\x01\x02\x03\x04\x05\x06\x07\x08"
        "\x01\x02\x03\x04\x05\x06\x07\x08\x01\x02\x03\x04\x05\x06\x07\x08"
    };

    const char* input[4] =
    {
        "what do ya want for nothing?",

        "Hi There",

        "\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd"
        "\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd"
        "\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd"
        "\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd"
        "\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd",

        "Big Key Input"
    };

    const int hashType[4] =
    {
        WC_SHA3_224, WC_SHA3_256, WC_SHA3_384, WC_SHA3_512
    };

    const int hashSz[4] =
    {
        WC_SHA3_224_DIGEST_SIZE, WC_SHA3_256_DIGEST_SIZE,
        WC_SHA3_384_DIGEST_SIZE, WC_SHA3_512_DIGEST_SIZE
    };

    const char* output[16] =
    {
        /* key = jefe, input = what do ya want for nothing? */
            /* HMAC-SHA3-224 */
            "\x7f\xdb\x8d\xd8\x8b\xd2\xf6\x0d\x1b\x79\x86\x34\xad\x38\x68\x11"
            "\xc2\xcf\xc8\x5b\xfa\xf5\xd5\x2b\xba\xce\x5e\x66",
            /* HMAC-SHA3-256 */
            "\xc7\xd4\x07\x2e\x78\x88\x77\xae\x35\x96\xbb\xb0\xda\x73\xb8\x87"
            "\xc9\x17\x1f\x93\x09\x5b\x29\x4a\xe8\x57\xfb\xe2\x64\x5e\x1b\xa5",
            /* HMAC-SHA3-384 */
            "\xf1\x10\x1f\x8c\xbf\x97\x66\xfd\x67\x64\xd2\xed\x61\x90\x3f\x21"
            "\xca\x9b\x18\xf5\x7c\xf3\xe1\xa2\x3c\xa1\x35\x08\xa9\x32\x43\xce"
            "\x48\xc0\x45\xdc\x00\x7f\x26\xa2\x1b\x3f\x5e\x0e\x9d\xf4\xc2\x0a",
            /* HMAC-SHA3-512 */
            "\x5a\x4b\xfe\xab\x61\x66\x42\x7c\x7a\x36\x47\xb7\x47\x29\x2b\x83"
            "\x84\x53\x7c\xdb\x89\xaf\xb3\xbf\x56\x65\xe4\xc5\xe7\x09\x35\x0b"
            "\x28\x7b\xae\xc9\x21\xfd\x7c\xa0\xee\x7a\x0c\x31\xd0\x22\xa9\x5e"
            "\x1f\xc9\x2b\xa9\xd7\x7d\xf8\x83\x96\x02\x75\xbe\xb4\xe6\x20\x24",

        /* key = 0b..., input = Hi There */
            /* HMAC-SHA3-224 */
            "\x3b\x16\x54\x6b\xbc\x7b\xe2\x70\x6a\x03\x1d\xca\xfd\x56\x37\x3d"
            "\x98\x84\x36\x76\x41\xd8\xc5\x9a\xf3\xc8\x60\xf7",
            /* HMAC-SHA3-256 */
            "\xba\x85\x19\x23\x10\xdf\xfa\x96\xe2\xa3\xa4\x0e\x69\x77\x43\x51"
            "\x14\x0b\xb7\x18\x5e\x12\x02\xcd\xcc\x91\x75\x89\xf9\x5e\x16\xbb",
            /* HMAC-SHA3-384 */
            "\x68\xd2\xdc\xf7\xfd\x4d\xdd\x0a\x22\x40\xc8\xa4\x37\x30\x5f\x61"
            "\xfb\x73\x34\xcf\xb5\xd0\x22\x6e\x1b\xc2\x7d\xc1\x0a\x2e\x72\x3a"
            "\x20\xd3\x70\xb4\x77\x43\x13\x0e\x26\xac\x7e\x3d\x53\x28\x86\xbd",
            /* HMAC-SHA3-512 */
            "\xeb\x3f\xbd\x4b\x2e\xaa\xb8\xf5\xc5\x04\xbd\x3a\x41\x46\x5a\xac"
            "\xec\x15\x77\x0a\x7c\xab\xac\x53\x1e\x48\x2f\x86\x0b\x5e\xc7\xba"
            "\x47\xcc\xb2\xc6\xf2\xaf\xce\x8f\x88\xd2\x2b\x6d\xc6\x13\x80\xf2"
            "\x3a\x66\x8f\xd3\x88\x8b\xb8\x05\x37\xc0\xa0\xb8\x64\x07\x68\x9e",

        /* key = aa..., output = dd... */
            /* HMAC-SHA3-224 */
            "\x67\x6c\xfc\x7d\x16\x15\x36\x38\x78\x03\x90\x69\x2b\xe1\x42\xd2"
            "\xdf\x7c\xe9\x24\xb9\x09\xc0\xc0\x8d\xbf\xdc\x1a",
            /* HMAC-SHA3-256 */
            "\x84\xec\x79\x12\x4a\x27\x10\x78\x65\xce\xdd\x8b\xd8\x2d\xa9\x96"
            "\x5e\x5e\xd8\xc3\x7b\x0a\xc9\x80\x05\xa7\xf3\x9e\xd5\x8a\x42\x07",
            /* HMAC-SHA3-384 */
            "\x27\x5c\xd0\xe6\x61\xbb\x8b\x15\x1c\x64\xd2\x88\xf1\xf7\x82\xfb"
            "\x91\xa8\xab\xd5\x68\x58\xd7\x2b\xab\xb2\xd4\x76\xf0\x45\x83\x73"
            "\xb4\x1b\x6a\xb5\xbf\x17\x4b\xec\x42\x2e\x53\xfc\x31\x35\xac\x6e",
            /* HMAC-SHA3-512 */
            "\x30\x9e\x99\xf9\xec\x07\x5e\xc6\xc6\xd4\x75\xed\xa1\x18\x06\x87"
            "\xfc\xf1\x53\x11\x95\x80\x2a\x99\xb5\x67\x74\x49\xa8\x62\x51\x82"
            "\x85\x1c\xb3\x32\xaf\xb6\xa8\x9c\x41\x13\x25\xfb\xcb\xcd\x42\xaf"
            "\xcb\x7b\x6e\x5a\xab\x7e\xa4\x2c\x66\x0f\x97\xfd\x85\x84\xbf\x03",

        /* key = big key, input = Big Key Input */
            /* HMAC-SHA3-224 */
            "\x29\xe0\x5e\x46\xc4\xa4\x5e\x46\x74\xbf\xd7\x2d\x1a\xd8\x66\xdb"
            "\x2d\x0d\x10\x4e\x2b\xfa\xad\x53\x7d\x15\x69\x8b",
            /* HMAC-SHA3-256 */
            "\xb5\x5b\x8d\x64\xb6\x9c\x21\xd0\xbf\x20\x5c\xa2\xf7\xb9\xb1\x4e"
            "\x88\x21\x61\x2c\x66\xc3\x91\xae\x6c\x95\x16\x85\x83\xe6\xf4\x9b",
            /* HMAC-SHA3-384 */
            "\xaa\x91\xb3\xa6\x2f\x56\xa1\xbe\x8c\x3e\x74\x38\xdb\x58\xd9\xd3"
            "\x34\xde\xa0\x60\x6d\x8d\x46\xe0\xec\xa9\xf6\x06\x35\x14\xe6\xed"
            "\x83\xe6\x7c\x77\x24\x6c\x11\xb5\x90\x82\xb5\x75\xda\x7b\x83\x2d",
            /* HMAC-SHA3-512 */
            "\x1c\xc3\xa9\x24\x4a\x4a\x3f\xbd\xc7\x20\x00\x16\x9b\x79\x47\x03"
            "\x78\x75\x2c\xb5\xf1\x2e\x62\x7c\xbe\xef\x4e\x8f\x0b\x11\x2b\x32"
            "\xa0\xee\xc9\xd0\x4d\x64\x64\x0b\x37\xf4\xdd\x66\xf7\x8b\xb3\xad"
            "\x52\x52\x6b\x65\x12\xde\x0d\x7c\xc0\x8b\x60\x01\x6c\x37\xd7\xa8"

    };

    int i = 0, iMax = sizeof(input) / sizeof(input[0]),
        j, jMax = sizeof(hashType) / sizeof(hashType[0]),
        ret;

#ifdef HAVE_FIPS
    /* FIPS requires a minimum length for HMAC keys, and "Jefe" is too
     * short. Skip it in FIPS builds. */
    i = 1;
#endif
    for (; i < iMax; i++) {
        for (j = 0; j < jMax; j++) {
            if (wc_HmacInit(&hmac, HEAP_HINT, devId) != 0)
                return -4000;

            ret = wc_HmacSetKey(&hmac, hashType[j], (byte*)key[i],
                                                       (word32)XSTRLEN(key[i]));
            if (ret != 0)
                return -4001;
            ret = wc_HmacUpdate(&hmac, (byte*)input[i],
                                                     (word32)XSTRLEN(input[i]));
            if (ret != 0)
                return -4002;
            ret = wc_HmacFinal(&hmac, hash);
            if (ret != 0)
                return -4003;
            if (XMEMCMP(hash, output[(i*jMax) + j], hashSz[j]) != 0)
                return -4004;

            wc_HmacFree(&hmac);

            if (i > 0)
                continue;

        #ifndef HAVE_FIPS
            ret = wc_HmacSizeByType(hashType[j]);
            if (ret != hashSz[j])
                return -4005;
        #endif
        }
    }

    return 0;
}
#endif


#ifdef WC_RC2
typedef struct rc2TestVector {
    const char* input;
    const char* output;
    const char* key;        /* Key, variable up to 128 bytes */
    const char* iv;         /* IV, 8-bytes */
    int inLen;
    int outLen;
    int keyLen;
    int effectiveKeyBits;   /* Up to 1024 bits supported */
} rc2TestVector;

static int rc2_ecb_test(void)
{
    int ret = 0;
    byte cipher[RC2_BLOCK_SIZE];
    byte plain[RC2_BLOCK_SIZE];

    rc2TestVector a, b, c, d, e, f, g, h;
    rc2TestVector test_rc2[8];

    int times = sizeof(test_rc2) / sizeof(rc2TestVector), i;

    a.input  = "\x00\x00\x00\x00\x00\x00\x00\x00";
    a.output = "\xeb\xb7\x73\xf9\x93\x27\x8e\xff";
    a.key    = "\x00\x00\x00\x00\x00\x00\x00\x00";
    a.inLen  = RC2_BLOCK_SIZE;
    a.outLen = RC2_BLOCK_SIZE;
    a.keyLen = 8;
    a.effectiveKeyBits = 63;

    b.input  = "\xff\xff\xff\xff\xff\xff\xff\xff";
    b.output = "\x27\x8b\x27\xe4\x2e\x2f\x0d\x49";
    b.key    = "\xff\xff\xff\xff\xff\xff\xff\xff";
    b.inLen  = RC2_BLOCK_SIZE;
    b.outLen = RC2_BLOCK_SIZE;
    b.keyLen = 8;
    b.effectiveKeyBits = 64;

    c.input  = "\x10\x00\x00\x00\x00\x00\x00\x01";
    c.output = "\x30\x64\x9e\xdf\x9b\xe7\xd2\xc2";
    c.key    = "\x30\x00\x00\x00\x00\x00\x00\x00";
    c.inLen  = RC2_BLOCK_SIZE;
    c.outLen = RC2_BLOCK_SIZE;
    c.keyLen = 8;
    c.effectiveKeyBits = 64;

    d.input  = "\x00\x00\x00\x00\x00\x00\x00\x00";
    d.output = "\x61\xa8\xa2\x44\xad\xac\xcc\xf0";
    d.key    = "\x88";
    d.inLen  = RC2_BLOCK_SIZE;
    d.outLen = RC2_BLOCK_SIZE;
    d.keyLen = 1;
    d.effectiveKeyBits = 64;

    e.input  = "\x00\x00\x00\x00\x00\x00\x00\x00";
    e.output = "\x6c\xcf\x43\x08\x97\x4c\x26\x7f";
    e.key    = "\x88\xbc\xa9\x0e\x90\x87\x5a";
    e.inLen  = RC2_BLOCK_SIZE;
    e.outLen = RC2_BLOCK_SIZE;
    e.keyLen = 7;
    e.effectiveKeyBits = 64;

    f.input  = "\x00\x00\x00\x00\x00\x00\x00\x00";
    f.output = "\x1a\x80\x7d\x27\x2b\xbe\x5d\xb1";
    f.key    = "\x88\xbc\xa9\x0e\x90\x87\x5a\x7f"
               "\x0f\x79\xc3\x84\x62\x7b\xaf\xb2";
    f.inLen  = RC2_BLOCK_SIZE;
    f.outLen = RC2_BLOCK_SIZE;
    f.keyLen = 16;
    f.effectiveKeyBits = 64;

    g.input  = "\x00\x00\x00\x00\x00\x00\x00\x00";
    g.output = "\x22\x69\x55\x2a\xb0\xf8\x5c\xa6";
    g.key    = "\x88\xbc\xa9\x0e\x90\x87\x5a\x7f"
               "\x0f\x79\xc3\x84\x62\x7b\xaf\xb2";
    g.inLen  = RC2_BLOCK_SIZE;
    g.outLen = RC2_BLOCK_SIZE;
    g.keyLen = 16;
    g.effectiveKeyBits = 128;

    h.input  = "\x00\x00\x00\x00\x00\x00\x00\x00";
    h.output = "\x5b\x78\xd3\xa4\x3d\xff\xf1\xf1";
    h.key    = "\x88\xbc\xa9\x0e\x90\x87\x5a\x7f"
               "\x0f\x79\xc3\x84\x62\x7b\xaf\xb2"
               "\x16\xf8\x0a\x6f\x85\x92\x05\x84"
               "\xc4\x2f\xce\xb0\xbe\x25\x5d\xaf"
               "\x1e";
    h.inLen  = RC2_BLOCK_SIZE;
    h.outLen = RC2_BLOCK_SIZE;
    h.keyLen = 33;
    h.effectiveKeyBits = 129;

    a.iv = b.iv = c.iv = d.iv = e.iv = f.iv = g.iv = h.iv = NULL;

    test_rc2[0] = a;
    test_rc2[1] = b;
    test_rc2[2] = c;
    test_rc2[3] = d;
    test_rc2[4] = e;
    test_rc2[5] = f;
    test_rc2[6] = g;
    test_rc2[7] = h;

    for (i = 0; i < times; ++i) {
        Rc2 enc;

        XMEMSET(cipher, 0, RC2_BLOCK_SIZE);
        XMEMSET(plain, 0, RC2_BLOCK_SIZE);

        ret = wc_Rc2SetKey(&enc, (byte*)test_rc2[i].key, test_rc2[i].keyLen,
                           NULL, test_rc2[i].effectiveKeyBits);
        if (ret != 0) {
            return -4100;
        }

        /* ECB encrypt */
        ret = wc_Rc2EcbEncrypt(&enc, cipher, (byte*)test_rc2[i].input,
                               (word32)test_rc2[i].outLen);
        if (ret != 0) {
            return -4101;
        }

        if (XMEMCMP(cipher, test_rc2[i].output, test_rc2[i].outLen)) {
            return -4102;
        }

        /* ECB decrypt */
        ret = wc_Rc2EcbDecrypt(&enc, plain, cipher, RC2_BLOCK_SIZE);
        if (ret != 0) {
            return -4103;
        }

        if (XMEMCMP(plain, test_rc2[i].input, RC2_BLOCK_SIZE)) {
            return -4104;
        }
    }

    return 0;
}

static int rc2_cbc_test(void)
{
    int ret = 0;
    byte cipher[128];
    byte plain[128];

    rc2TestVector a, b, c, d, e, f, g, h, i;
    rc2TestVector test_rc2[9];

    int times = sizeof(test_rc2) / sizeof(rc2TestVector), j;

    /* key length = 7, effective key bits = 63 */
    a.input  = "\x00\x00\x00\x00\x00\x00\x00\x00"
               "\x00\x00\x00\x00\x00\x00\x00\x00";
    a.output = "\xEB\xB7\x73\xF9\x93\x27\x8E\xFF"
               "\xF0\x51\x77\x8B\x65\xDB\x13\x57";
    a.key    = "\x00\x00\x00\x00\x00\x00\x00\x00";
    a.iv     = "\x00\x00\x00\x00\x00\x00\x00\x00";
    a.inLen  = RC2_BLOCK_SIZE*2;
    a.outLen = RC2_BLOCK_SIZE*2;
    a.keyLen = 8;
    a.effectiveKeyBits = 63;

    /* key length = 8, effective key bits = 64, all 0xFF */
    b.input  = "\xff\xff\xff\xff\xff\xff\xff\xff"
               "\xff\xff\xff\xff\xff\xff\xff\xff";
    b.output = "\xA3\xA1\x12\x65\x4F\x81\xC5\xCD"
               "\xB6\x94\x3E\xEA\x3E\x8B\x9D\x1F";
    b.key    = "\xff\xff\xff\xff\xff\xff\xff\xff";
    b.iv     = "\xff\xff\xff\xff\xff\xff\xff\xff";
    b.inLen  = RC2_BLOCK_SIZE*2;
    b.outLen = RC2_BLOCK_SIZE*2;
    b.keyLen = 8;
    b.effectiveKeyBits = 64;

    /* key length = 8, effective key bits = 64 */
    c.input  = "\x10\x00\x00\x00\x00\x00\x00\x01"
               "\x10\x00\x00\x00\x00\x00\x00\x01";
    c.output = "\xB5\x70\x14\xA2\x5F\x40\xE3\x6D"
               "\x81\x99\x8D\xE0\xB5\xD5\x3A\x05";
    c.key    = "\x30\x00\x00\x00\x00\x00\x00\x00";
    c.iv     = "\x30\x00\x00\x00\x00\x00\x00\x00";
    c.inLen  = RC2_BLOCK_SIZE*2;
    c.outLen = RC2_BLOCK_SIZE*2;
    c.keyLen = 8;
    c.effectiveKeyBits = 64;

    /* key length = 1, effective key bits = 64 */
    d.input  = "\x00\x00\x00\x00\x00\x00\x00\x00"
               "\x00\x00\x00\x00\x00\x00\x00\x00";
    d.output = "\x61\xA8\xA2\x44\xAD\xAC\xCC\xF0"
               "\x6D\x19\xE8\xF1\xFC\xE7\x38\x87";
    d.key    = "\x88";
    d.iv     = "\x00\x00\x00\x00\x00\x00\x00\x00";
    d.inLen  = RC2_BLOCK_SIZE*2;
    d.outLen = RC2_BLOCK_SIZE*2;
    d.keyLen = 1;
    d.effectiveKeyBits = 64;

    /* key length = 7, effective key bits = 64 */
    e.input  = "\x00\x00\x00\x00\x00\x00\x00\x00"
               "\x00\x00\x00\x00\x00\x00\x00\x00";
    e.output = "\x6C\xCF\x43\x08\x97\x4C\x26\x7F"
               "\xCC\x3C\x53\x57\x7C\xA1\xA4\x4B";
    e.key    = "\x88\xbc\xa9\x0e\x90\x87\x5a";
    e.iv     = "\x00\x00\x00\x00\x00\x00\x00\x00";
    e.inLen  = RC2_BLOCK_SIZE*2;
    e.outLen = RC2_BLOCK_SIZE*2;
    e.keyLen = 7;
    e.effectiveKeyBits = 64;

    /* key length = 16, effective key bits = 64 */
    f.input  = "\x00\x00\x00\x00\x00\x00\x00\x00"
               "\x00\x00\x00\x00\x00\x00\x00\x00";
    f.output = "\x1A\x80\x7D\x27\x2B\xBE\x5D\xB1"
               "\x64\xEF\xE1\xC3\xB8\xAD\xFB\xBA";
    f.key    = "\x88\xbc\xa9\x0e\x90\x87\x5a\x7f"
               "\x0f\x79\xc3\x84\x62\x7b\xaf\xb2";
    f.iv     = "\x00\x00\x00\x00\x00\x00\x00\x00";
    f.inLen  = RC2_BLOCK_SIZE*2;
    f.outLen = RC2_BLOCK_SIZE*2;
    f.keyLen = 16;
    f.effectiveKeyBits = 64;

    /* key length = 16, effective bits = 128 */
    g.input  = "\x00\x00\x00\x00\x00\x00\x00\x00"
               "\x00\x00\x00\x00\x00\x00\x00\x00";
    g.output = "\x22\x69\x55\x2A\xB0\xF8\x5C\xA6"
               "\x53\x6E\xFD\x2D\x89\xE1\x2A\x73";
    g.key    = "\x88\xbc\xa9\x0e\x90\x87\x5a\x7f"
               "\x0f\x79\xc3\x84\x62\x7b\xaf\xb2";
    g.iv     = "\x00\x00\x00\x00\x00\x00\x00\x00";
    g.inLen  = RC2_BLOCK_SIZE*2;
    g.outLen = RC2_BLOCK_SIZE*2;
    g.keyLen = 16;
    g.effectiveKeyBits = 128;

    /* key length = 33, effective bits = 129 */
    h.input  = "\x00\x00\x00\x00\x00\x00\x00\x00"
               "\x00\x00\x00\x00\x00\x00\x00\x00";
    h.output = "\x5B\x78\xD3\xA4\x3D\xFF\xF1\xF1"
               "\x45\x30\xA8\xD5\xC7\x7C\x46\x19";
    h.key    = "\x88\xbc\xa9\x0e\x90\x87\x5a\x7f"
               "\x0f\x79\xc3\x84\x62\x7b\xaf\xb2"
               "\x16\xf8\x0a\x6f\x85\x92\x05\x84"
               "\xc4\x2f\xce\xb0\xbe\x25\x5d\xaf"
               "\x1e";
    h.iv     = "\x00\x00\x00\x00\x00\x00\x00\x00";
    h.inLen  = RC2_BLOCK_SIZE*2;
    h.outLen = RC2_BLOCK_SIZE*2;
    h.keyLen = 33;
    h.effectiveKeyBits = 129;

    /* key length = 10, effective bits = 40 */
    i.input  = "\x11\x22\x33\x44\x55\x66\x77\x88"
               "\x99\xAA\xBB\xCC\xDD\xEE\xFF\x00"
               "\x11\x22\x33\x44\x55\x66\x77\x88"
               "\x99\xAA\xBB\xCC\xDD\xEE\xFF\x00";
    i.output = "\x71\x2D\x11\x99\xC9\xA0\x78\x4F"
               "\xCD\xF1\x1E\x3D\xFD\x21\x7E\xDB"
               "\xB2\x6E\x0D\xA4\x72\xBC\x31\x51"
               "\x48\xEF\x4E\x68\x3B\xDC\xCD\x7D";
    i.key    = "\x26\x1E\x57\x8E\xC9\x62\xBF\xB8"
               "\x3E\x96";
    i.iv     = "\x01\x02\x03\x04\x05\x06\x07\x08";
    i.inLen  = RC2_BLOCK_SIZE*4;
    i.outLen = RC2_BLOCK_SIZE*4;
    i.keyLen = 10;
    i.effectiveKeyBits = 40;

    test_rc2[0] = a;
    test_rc2[1] = b;
    test_rc2[2] = c;
    test_rc2[3] = d;
    test_rc2[4] = e;
    test_rc2[5] = f;
    test_rc2[6] = g;
    test_rc2[7] = h;
    test_rc2[8] = i;

    for (j = 0; j < times; ++j) {
        Rc2 rc2;

        XMEMSET(cipher, 0, sizeof(cipher));
        XMEMSET(plain, 0, sizeof(plain));

        ret = wc_Rc2SetKey(&rc2, (byte*)test_rc2[j].key, test_rc2[j].keyLen,
                           (byte*)test_rc2[j].iv, test_rc2[j].effectiveKeyBits);
        if (ret != 0) {
            return -4200;
        }

        ret = wc_Rc2CbcEncrypt(&rc2, cipher, (byte*)test_rc2[j].input,
                               test_rc2[j].inLen);
        if (ret != 0) {
            return -4201;
        }

        if (XMEMCMP(cipher, (byte*)test_rc2[j].output, test_rc2[j].outLen)) {
            return -4202;
        }

        /* reset IV for decrypt, since overriden by encrypt operation */
        ret = wc_Rc2SetIV(&rc2, (byte*)test_rc2[j].iv);
        if (ret != 0) {
            return -4203;
        }

        ret = wc_Rc2CbcDecrypt(&rc2, plain, cipher, test_rc2[j].outLen);
        if (ret != 0) {
            return -4204;
        }

        if (XMEMCMP(plain, (byte*)test_rc2[j].input, test_rc2[j].inLen)) {
            return -4205;
        }
    }

    return 0;
}

WOLFSSL_TEST_SUBROUTINE int rc2_test(void)
{
    int ret = 0;

    ret = rc2_ecb_test();
    if (ret != 0) {
        return ret;
    }

    return rc2_cbc_test();
}
#endif


#ifndef NO_RC4
WOLFSSL_TEST_SUBROUTINE int arc4_test(void)
{
    byte cipher[16];
    byte plain[16];

    const char* keys[] =
    {
        "\x01\x23\x45\x67\x89\xab\xcd\xef",
        "\x01\x23\x45\x67\x89\xab\xcd\xef",
        "\x00\x00\x00\x00\x00\x00\x00\x00",
        "\xef\x01\x23\x45"
    };

    testVector a, b, c, d;
    testVector test_arc4[4];

    int times = sizeof(test_arc4) / sizeof(testVector), i;

    a.input  = "\x01\x23\x45\x67\x89\xab\xcd\xef";
    a.output = "\x75\xb7\x87\x80\x99\xe0\xc5\x96";
    a.inLen  = 8;
    a.outLen = 8;

    b.input  = "\x00\x00\x00\x00\x00\x00\x00\x00";
    b.output = "\x74\x94\xc2\xe7\x10\x4b\x08\x79";
    b.inLen  = 8;
    b.outLen = 8;

    c.input  = "\x00\x00\x00\x00\x00\x00\x00\x00";
    c.output = "\xde\x18\x89\x41\xa3\x37\x5d\x3a";
    c.inLen  = 8;
    c.outLen = 8;

    d.input  = "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00";
    d.output = "\xd6\xa1\x41\xa7\xec\x3c\x38\xdf\xbd\x61";
    d.inLen  = 10;
    d.outLen = 10;

    test_arc4[0] = a;
    test_arc4[1] = b;
    test_arc4[2] = c;
    test_arc4[3] = d;

    for (i = 0; i < times; ++i) {
        Arc4 enc;
        Arc4 dec;
        int  keylen = 8;  /* XSTRLEN with key 0x00 not good */
        if (i == 3)
            keylen = 4;

        if (wc_Arc4Init(&enc, HEAP_HINT, devId) != 0)
            return -4400;
        if (wc_Arc4Init(&dec, HEAP_HINT, devId) != 0)
            return -4401;

        wc_Arc4SetKey(&enc, (byte*)keys[i], keylen);
        wc_Arc4SetKey(&dec, (byte*)keys[i], keylen);

        wc_Arc4Process(&enc, cipher, (byte*)test_arc4[i].input,
                    (word32)test_arc4[i].outLen);
        wc_Arc4Process(&dec, plain,  cipher, (word32)test_arc4[i].outLen);

        if (XMEMCMP(plain, test_arc4[i].input, test_arc4[i].outLen))
            return -4402 - i;

        if (XMEMCMP(cipher, test_arc4[i].output, test_arc4[i].outLen))
            return -4412 - i;

        wc_Arc4Free(&enc);
        wc_Arc4Free(&dec);
    }

    return 0;
}
#endif

#ifdef HAVE_CHACHA
WOLFSSL_TEST_SUBROUTINE int chacha_test(void)
{
    ChaCha enc;
    ChaCha dec;
    byte   cipher[128];
    byte   plain[128];
    byte   sliver[64];
    byte   input[] = {0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0};
    word32 keySz = 32;
    int    ret = 0;
    int    i;
    int    times = 4;

    WOLFSSL_SMALL_STACK_STATIC const byte key1[] =
    {
         0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
         0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
         0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
         0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
    };

    WOLFSSL_SMALL_STACK_STATIC const byte key2[] =
    {
         0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
         0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
         0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
         0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x01
    };

    WOLFSSL_SMALL_STACK_STATIC const byte key3[] =
    {
         0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
         0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
         0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
         0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
    };

    /* 128 bit key */
    WOLFSSL_SMALL_STACK_STATIC const byte key4[] =
    {
         0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
         0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
    };


    const byte* keys[] = {key1, key2, key3, key4};

    WOLFSSL_SMALL_STACK_STATIC const byte ivs1[] = {0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00};
    WOLFSSL_SMALL_STACK_STATIC const byte ivs2[] = {0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00};
    WOLFSSL_SMALL_STACK_STATIC const byte ivs3[] = {0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x01,0x00,0x00,0x00,0x00};
    WOLFSSL_SMALL_STACK_STATIC const byte ivs4[] = {0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00};


    const byte* ivs[] = {ivs1, ivs2, ivs3, ivs4};

#ifndef BENCH_EMBEDDED
    WOLFSSL_SMALL_STACK_STATIC const byte cipher_big_result[] = {
        0x06, 0xa6, 0x5d, 0x31, 0x21, 0x6c, 0xdb, 0x37, 0x48, 0x7c, 0x01, 0x9d,
        0x72, 0xdf, 0x0a, 0x5b, 0x64, 0x74, 0x20, 0xba, 0x9e, 0xe0, 0x26, 0x7a,
        0xbf, 0xdf, 0x83, 0x34, 0x3b, 0x4f, 0x94, 0x3f, 0x37, 0x89, 0xaf, 0x00,
        0xdf, 0x0f, 0x2e, 0x75, 0x16, 0x41, 0xf6, 0x7a, 0x86, 0x94, 0x9d, 0x32,
        0x56, 0xf0, 0x79, 0x71, 0x68, 0x6f, 0xa6, 0x6b, 0xc6, 0x59, 0x49, 0xf6,
        0x10, 0x34, 0x03, 0x03, 0x16, 0x53, 0x9a, 0x98, 0x2a, 0x46, 0xde, 0x17,
        0x06, 0x65, 0x70, 0xca, 0x0a, 0x1f, 0xab, 0x80, 0x26, 0x96, 0x3f, 0x3e,
        0x7a, 0x3c, 0xa8, 0x87, 0xbb, 0x65, 0xdd, 0x5e, 0x07, 0x7b, 0x34, 0xe0,
        0x56, 0xda, 0x32, 0x13, 0x30, 0xc9, 0x0c, 0xd7, 0xba, 0xe4, 0x1f, 0xa6,
        0x91, 0x4f, 0x72, 0x9f, 0xd9, 0x5c, 0x62, 0x7d, 0xa6, 0xc2, 0xbc, 0x87,
        0xae, 0x64, 0x11, 0x94, 0x3b, 0xbc, 0x6c, 0x23, 0xbd, 0x7d, 0x00, 0xb4,
        0x99, 0xf2, 0x68, 0xb5, 0x59, 0x70, 0x93, 0xad, 0x69, 0xd0, 0xb1, 0x28,
        0x70, 0x92, 0xeb, 0xec, 0x39, 0x80, 0x82, 0xde, 0x44, 0xe2, 0x8a, 0x26,
        0xb3, 0xe9, 0x45, 0xcf, 0x83, 0x76, 0x9f, 0x6a, 0xa0, 0x46, 0x4a, 0x3d,
        0x26, 0x56, 0xaf, 0x49, 0x41, 0x26, 0x1b, 0x6a, 0x41, 0x37, 0x65, 0x91,
        0x72, 0xc4, 0xe7, 0x3c, 0x17, 0x31, 0xae, 0x2e, 0x2b, 0x31, 0x45, 0xe4,
        0x93, 0xd3, 0x10, 0xaa, 0xc5, 0x62, 0xd5, 0x11, 0x4b, 0x57, 0x1d, 0xad,
        0x48, 0x06, 0xd0, 0x0d, 0x98, 0xa5, 0xc6, 0x5b, 0xd0, 0x9e, 0x22, 0xc0,
        0x00, 0x32, 0x5a, 0xf5, 0x1c, 0x89, 0x6d, 0x54, 0x97, 0x55, 0x6b, 0x46,
        0xc5, 0xc7, 0xc4, 0x48, 0x9c, 0xbf, 0x47, 0xdc, 0x03, 0xc4, 0x1b, 0xcb,
        0x65, 0xa6, 0x91, 0x9d, 0x6d, 0xf1, 0xb0, 0x7a, 0x4d, 0x3b, 0x03, 0x95,
        0xf4, 0x8b, 0x0b, 0xae, 0x39, 0xff, 0x3f, 0xf6, 0xc0, 0x14, 0x18, 0x8a,
        0xe5, 0x19, 0xbd, 0xc1, 0xb4, 0x05, 0x4e, 0x29, 0x2f, 0x0b, 0x33, 0x76,
        0x28, 0x16, 0xa4, 0xa6, 0x93, 0x04, 0xb5, 0x55, 0x6b, 0x89, 0x3d, 0xa5,
        0x0f, 0xd3, 0xad, 0xfa, 0xd9, 0xfd, 0x05, 0x5d, 0x48, 0x94, 0x25, 0x5a,
        0x2c, 0x9a, 0x94, 0x80, 0xb0, 0xe7, 0xcb, 0x4d, 0x77, 0xbf, 0xca, 0xd8,
        0x55, 0x48, 0xbd, 0x66, 0xb1, 0x85, 0x81, 0xb1, 0x37, 0x79, 0xab, 0x52,
        0x08, 0x14, 0x12, 0xac, 0xcd, 0x45, 0x4d, 0x53, 0x6b, 0xca, 0x96, 0xc7,
        0x3b, 0x2f, 0x73, 0xb1, 0x5a, 0x23, 0xbd, 0x65, 0xd5, 0xea, 0x17, 0xb3,
        0xdc, 0xa1, 0x17, 0x1b, 0x2d, 0xb3, 0x9c, 0xd0, 0xdb, 0x41, 0x77, 0xef,
        0x93, 0x20, 0x52, 0x3e, 0x9d, 0xf5, 0xbf, 0x33, 0xf7, 0x52, 0xc1, 0x90,
        0xa0, 0x15, 0x17, 0xce, 0xf7, 0xf7, 0xd0, 0x3a, 0x3b, 0xd1, 0x72, 0x56,
        0x31, 0x81, 0xae, 0x60, 0xab, 0x40, 0xc1, 0xd1, 0x28, 0x77, 0x53, 0xac,
        0x9f, 0x11, 0x0a, 0x88, 0x36, 0x4b, 0xda, 0x57, 0xa7, 0x28, 0x5c, 0x85,
        0xd3, 0x85, 0x9b, 0x79, 0xad, 0x05, 0x1c, 0x37, 0x14, 0x5e, 0x0d, 0xd0,
        0x23, 0x03, 0x42, 0x1d, 0x48, 0x5d, 0xc5, 0x3c, 0x5a, 0x08, 0xa9, 0x0d,
        0x6e, 0x82, 0x7c, 0x2e, 0x3c, 0x41, 0xcc, 0x96, 0x8e, 0xad, 0xee, 0x2a,
        0x61, 0x0b, 0x16, 0x0f, 0xa9, 0x24, 0x40, 0x85, 0xbc, 0x9f, 0x28, 0x8d,
        0xe6, 0x68, 0x4d, 0x8f, 0x30, 0x48, 0xd9, 0x73, 0x73, 0x6c, 0x9a, 0x7f,
        0x67, 0xf7, 0xde, 0x4c, 0x0a, 0x8b, 0xe4, 0xb3, 0x08, 0x2a, 0x52, 0xda,
        0x54, 0xee, 0xcd, 0xb5, 0x62, 0x4a, 0x26, 0x20, 0xfb, 0x40, 0xbb, 0x39,
        0x3a, 0x0f, 0x09, 0xe8, 0x00, 0xd1, 0x24, 0x97, 0x60, 0xe9, 0x83, 0x83,
        0xfe, 0x9f, 0x9c, 0x15, 0xcf, 0x69, 0x03, 0x9f, 0x03, 0xe1, 0xe8, 0x6e,
        0xbd, 0x87, 0x58, 0x68, 0xee, 0xec, 0xd8, 0x29, 0x46, 0x23, 0x49, 0x92,
        0x72, 0x95, 0x5b, 0x49, 0xca, 0xe0, 0x45, 0x59, 0xb2, 0xca, 0xf4, 0xfc,
        0xb7, 0x59, 0x37, 0x49, 0x28, 0xbc, 0xf3, 0xd7, 0x61, 0xbc, 0x4b, 0xf3,
        0xa9, 0x4b, 0x2f, 0x05, 0xa8, 0x01, 0xa5, 0xdc, 0x00, 0x6e, 0x01, 0xb6,
        0x45, 0x3c, 0xd5, 0x49, 0x7d, 0x5c, 0x25, 0xe8, 0x31, 0x87, 0xb2, 0xb9,
        0xbf, 0xb3, 0x01, 0x62, 0x0c, 0xd0, 0x48, 0x77, 0xa2, 0x34, 0x0f, 0x16,
        0x22, 0x28, 0xee, 0x54, 0x08, 0x93, 0x3b, 0xe4, 0xde, 0x7e, 0x63, 0xf7,
        0x97, 0x16, 0x5d, 0x71, 0x58, 0xc2, 0x2e, 0xf2, 0x36, 0xa6, 0x12, 0x65,
        0x94, 0x17, 0xac, 0x66, 0x23, 0x7e, 0xc6, 0x72, 0x79, 0x24, 0xce, 0x8f,
        0x55, 0x19, 0x97, 0x44, 0xfc, 0x55, 0xec, 0x85, 0x26, 0x27, 0xdb, 0x38,
        0xb1, 0x42, 0x0a, 0xdd, 0x05, 0x99, 0x28, 0xeb, 0x03, 0x6c, 0x9a, 0xe9,
        0x17, 0xf6, 0x2c, 0xb0, 0xfe, 0xe7, 0xa4, 0xa7, 0x31, 0xda, 0x4d, 0xb0,
        0x29, 0xdb, 0xdd, 0x8d, 0x12, 0x13, 0x9c, 0xb4, 0xcc, 0x83, 0x97, 0xfb,
        0x1a, 0xdc, 0x08, 0xd6, 0x30, 0x62, 0xe8, 0xeb, 0x8b, 0x61, 0xcb, 0x1d,
        0x06, 0xe3, 0xa5, 0x4d, 0x35, 0xdb, 0x59, 0xa8, 0x2d, 0x87, 0x27, 0x44,
        0x6f, 0xc0, 0x38, 0x97, 0xe4, 0x85, 0x00, 0x02, 0x09, 0xf6, 0x69, 0x3a,
        0xcf, 0x08, 0x1b, 0x21, 0xbb, 0x79, 0xb1, 0xa1, 0x34, 0x09, 0xe0, 0x80,
        0xca, 0xb0, 0x78, 0x8a, 0x11, 0x97, 0xd4, 0x07, 0xbe, 0x1b, 0x6a, 0x5d,
        0xdb, 0xd6, 0x1f, 0x76, 0x6b, 0x16, 0xf0, 0x58, 0x84, 0x5f, 0x59, 0xce,
        0x62, 0x34, 0xc3, 0xdf, 0x94, 0xb8, 0x2f, 0x84, 0x68, 0xf0, 0xb8, 0x51,
        0xd9, 0x6d, 0x8e, 0x4a, 0x1d, 0xe6, 0x5c, 0xd8, 0x86, 0x25, 0xe3, 0x24,
        0xfd, 0x21, 0x61, 0x13, 0x48, 0x3e, 0xf6, 0x7d, 0xa6, 0x71, 0x9b, 0xd2,
        0x6e, 0xe6, 0xd2, 0x08, 0x94, 0x62, 0x6c, 0x98, 0xfe, 0x2f, 0x9c, 0x88,
        0x7e, 0x78, 0x15, 0x02, 0x00, 0xf0, 0xba, 0x24, 0x91, 0xf2, 0xdc, 0x47,
        0x51, 0x4d, 0x15, 0x5e, 0x91, 0x5f, 0x57, 0x5b, 0x1d, 0x35, 0x24, 0x45,
        0x75, 0x9b, 0x88, 0x75, 0xf1, 0x2f, 0x85, 0xe7, 0x89, 0xd1, 0x01, 0xb4,
        0xc8, 0x18, 0xb7, 0x97, 0xef, 0x4b, 0x90, 0xf4, 0xbf, 0x10, 0x27, 0x3c,
        0x60, 0xff, 0xc4, 0x94, 0x20, 0x2f, 0x93, 0x4b, 0x4d, 0xe3, 0x80, 0xf7,
        0x2c, 0x71, 0xd9, 0xe3, 0x68, 0xb4, 0x77, 0x2b, 0xc7, 0x0d, 0x39, 0x92,
        0xef, 0x91, 0x0d, 0xb2, 0x11, 0x50, 0x0e, 0xe8, 0xad, 0x3b, 0xf6, 0xb5,
        0xc6, 0x14, 0x4d, 0x33, 0x53, 0xa7, 0x60, 0x15, 0xc7, 0x27, 0x51, 0xdc,
        0x54, 0x29, 0xa7, 0x0d, 0x6a, 0x7b, 0x72, 0x13, 0xad, 0x7d, 0x41, 0x19,
        0x4e, 0x42, 0x49, 0xcc, 0x42, 0xe4, 0xbd, 0x99, 0x13, 0xd9, 0x7f, 0xf3,
        0x38, 0xa4, 0xb6, 0x33, 0xed, 0x07, 0x48, 0x7e, 0x8e, 0x82, 0xfe, 0x3a,
        0x9d, 0x75, 0x93, 0xba, 0x25, 0x4e, 0x37, 0x3c, 0x0c, 0xd5, 0x69, 0xa9,
        0x2d, 0x9e, 0xfd, 0xe8, 0xbb, 0xf5, 0x0c, 0xe2, 0x86, 0xb9, 0x5e, 0x6f,
        0x28, 0xe4, 0x19, 0xb3, 0x0b, 0xa4, 0x86, 0xd7, 0x24, 0xd0, 0xb8, 0x89,
        0x7b, 0x76, 0xec, 0x05, 0x10, 0x5b, 0x68, 0xe9, 0x58, 0x66, 0xa3, 0xc5,
        0xb6, 0x63, 0x20, 0x0e, 0x0e, 0xea, 0x3d, 0x61, 0x5e, 0xda, 0x3d, 0x3c,
        0xf9, 0xfd, 0xed, 0xa9, 0xdb, 0x52, 0x94, 0x8a, 0x00, 0xca, 0x3c, 0x8d,
        0x66, 0x8f, 0xb0, 0xf0, 0x5a, 0xca, 0x3f, 0x63, 0x71, 0xbf, 0xca, 0x99,
        0x37, 0x9b, 0x75, 0x97, 0x89, 0x10, 0x6e, 0xcf, 0xf2, 0xf5, 0xe3, 0xd5,
        0x45, 0x9b, 0xad, 0x10, 0x71, 0x6c, 0x5f, 0x6f, 0x7f, 0x22, 0x77, 0x18,
        0x2f, 0xf9, 0x99, 0xc5, 0x69, 0x58, 0x03, 0x12, 0x86, 0x82, 0x3e, 0xbf,
        0xc2, 0x12, 0x35, 0x43, 0xa3, 0xd9, 0x18, 0x4f, 0x41, 0x11, 0x6b, 0xf3,
        0x67, 0xaf, 0x3d, 0x78, 0xe4, 0x22, 0x2d, 0xb3, 0x48, 0x43, 0x31, 0x1d,
        0xef, 0xa8, 0xba, 0x49, 0x8e, 0xa9, 0xa7, 0xb6, 0x18, 0x77, 0x84, 0xca,
        0xbd, 0xa2, 0x02, 0x1b, 0x6a, 0xf8, 0x5f, 0xda, 0xff, 0xcf, 0x01, 0x6a,
        0x86, 0x69, 0xa9, 0xe9, 0xcb, 0x60, 0x1e, 0x15, 0xdc, 0x8f, 0x5d, 0x39,
        0xb5, 0xce, 0x55, 0x5f, 0x47, 0x97, 0xb1, 0x19, 0x6e, 0x21, 0xd6, 0x13,
        0x39, 0xb2, 0x24, 0xe0, 0x62, 0x82, 0x9f, 0xed, 0x12, 0x81, 0xed, 0xee,
        0xab, 0xd0, 0x2f, 0x19, 0x89, 0x3f, 0x57, 0x2e, 0xc2, 0xe2, 0x67, 0xe8,
        0xae, 0x03, 0x56, 0xba, 0xd4, 0xd0, 0xa4, 0x89, 0x03, 0x06, 0x5b, 0xcc,
        0xf2, 0x22, 0xb8, 0x0e, 0x76, 0x79, 0x4a, 0x42, 0x1d, 0x37, 0x51, 0x5a,
        0xaa, 0x46, 0x6c, 0x2a, 0xdd, 0x66, 0xfe, 0xc6, 0x68, 0xc3, 0x38, 0xa2,
        0xae, 0x5b, 0x98, 0x24, 0x5d, 0x43, 0x05, 0x82, 0x38, 0x12, 0xd3, 0xd1,
        0x75, 0x2d, 0x4f, 0x61, 0xbd, 0xb9, 0x10, 0x87, 0x44, 0x2a, 0x78, 0x07,
        0xff, 0xf4, 0x0f, 0xa1, 0xf3, 0x68, 0x9f, 0xbe, 0xae, 0xa2, 0x91, 0xf0,
        0xc7, 0x55, 0x7a, 0x52, 0xd5, 0xa3, 0x8d, 0x6f, 0xe4, 0x90, 0x5c, 0xf3,
        0x5f, 0xce, 0x3d, 0x23, 0xf9, 0x8e, 0xae, 0x14, 0xfb, 0x82, 0x9a, 0xa3,
        0x04, 0x5f, 0xbf, 0xad, 0x3e, 0xf2, 0x97, 0x0a, 0x60, 0x40, 0x70, 0x19,
        0x72, 0xad, 0x66, 0xfb, 0x78, 0x1b, 0x84, 0x6c, 0x98, 0xbc, 0x8c, 0xf8,
        0x4f, 0xcb, 0xb5, 0xf6, 0xaf, 0x7a, 0xb7, 0x93, 0xef, 0x67, 0x48, 0x02,
        0x2c, 0xcb, 0xe6, 0x77, 0x0f, 0x7b, 0xc1, 0xee, 0xc5, 0xb6, 0x2d, 0x7e,
        0x62, 0xa0, 0xc0, 0xa7, 0xa5, 0x80, 0x31, 0x92, 0x50, 0xa1, 0x28, 0x22,
        0x95, 0x03, 0x17, 0xd1, 0x0f, 0xf6, 0x08, 0xe5, 0xec
    };
#define CHACHA_BIG_TEST_SIZE 1305
#ifndef WOLFSSL_SMALL_STACK
    byte   cipher_big[CHACHA_BIG_TEST_SIZE] = {0};
    byte   plain_big[CHACHA_BIG_TEST_SIZE] = {0};
    byte   input_big[CHACHA_BIG_TEST_SIZE] = {0};
#else
    byte*  cipher_big;
    byte*  plain_big;
    byte*  input_big;
#endif /* WOLFSSL_SMALL_STACK */
    int    block_size;
#endif /* BENCH_EMBEDDED */

    byte a[] = {0x76,0xb8,0xe0,0xad,0xa0,0xf1,0x3d,0x90};
    byte b[] = {0x45,0x40,0xf0,0x5a,0x9f,0x1f,0xb2,0x96};
    byte c[] = {0xde,0x9c,0xba,0x7b,0xf3,0xd6,0x9e,0xf5};
    byte d[] = {0x89,0x67,0x09,0x52,0x60,0x83,0x64,0xfd};

    byte* test_chacha[4];

    test_chacha[0] = a;
    test_chacha[1] = b;
    test_chacha[2] = c;
    test_chacha[3] = d;

#ifndef BENCH_EMBEDDED
#ifdef WOLFSSL_SMALL_STACK
    cipher_big = (byte*)XMALLOC(CHACHA_BIG_TEST_SIZE, HEAP_HINT,
                                                       DYNAMIC_TYPE_TMP_BUFFER);
    if (cipher_big == NULL) {
        return MEMORY_E;
    }
    plain_big = (byte*)XMALLOC(CHACHA_BIG_TEST_SIZE, HEAP_HINT,
                                                       DYNAMIC_TYPE_TMP_BUFFER);
    if (plain_big == NULL) {
        return MEMORY_E;
    }
    input_big = (byte*)XMALLOC(CHACHA_BIG_TEST_SIZE, HEAP_HINT,
                                                       DYNAMIC_TYPE_TMP_BUFFER);
    if (input_big == NULL) {
        return MEMORY_E;
    }
    XMEMSET(cipher_big, 0, CHACHA_BIG_TEST_SIZE);
    XMEMSET(plain_big, 0, CHACHA_BIG_TEST_SIZE);
    XMEMSET(input_big, 0, CHACHA_BIG_TEST_SIZE);
#endif /* WOLFSSL_SMALL_STACK */
#endif /* BENCH_EMBEDDED */

    for (i = 0; i < times; ++i) {
        if (i < 3) {
            keySz = 32;
        }
        else {
            keySz = 16;
        }

        XMEMCPY(plain, keys[i], keySz);
        XMEMSET(cipher, 0, 32);
        XMEMCPY(cipher + 4, ivs[i], 8);

        ret |= wc_Chacha_SetKey(&enc, keys[i], keySz);
        ret |= wc_Chacha_SetKey(&dec, keys[i], keySz);
        if (ret != 0)
            return ret;

        ret |= wc_Chacha_SetIV(&enc, cipher, 0);
        ret |= wc_Chacha_SetIV(&dec, cipher, 0);
        if (ret != 0)
            return ret;
        XMEMCPY(plain, input, 8);

        ret |= wc_Chacha_Process(&enc, cipher, plain,  (word32)8);
        ret |= wc_Chacha_Process(&dec, plain,  cipher, (word32)8);
        if (ret != 0)
            return ret;

        if (XMEMCMP(test_chacha[i], cipher, 8))
            return -4700 - i;

        if (XMEMCMP(plain, input, 8))
            return -4710 - i;
    }

    /* test of starting at a different counter
       encrypts all of the information and decrypts starting at 2nd chunk */
    XMEMSET(plain,  0, sizeof(plain));
    XMEMSET(sliver, 1, sizeof(sliver)); /* set as 1's to not match plain */
    XMEMSET(cipher, 0, sizeof(cipher));
    XMEMCPY(cipher + 4, ivs[0], 8);

    ret |= wc_Chacha_SetKey(&enc, keys[0], keySz);
    ret |= wc_Chacha_SetKey(&dec, keys[0], keySz);
    if (ret != 0)
        return ret;

    ret |= wc_Chacha_SetIV(&enc, cipher, 0);
    ret |= wc_Chacha_SetIV(&dec, cipher, 1);
    if (ret != 0)
        return ret;

    ret |= wc_Chacha_Process(&enc, cipher, plain,  sizeof(plain));
    ret |= wc_Chacha_Process(&dec, sliver,  cipher + 64, sizeof(sliver));
    if (ret != 0)
        return ret;

    if (XMEMCMP(plain + 64, sliver, 64))
        return -4720;

#ifndef BENCH_EMBEDDED
    /* test of encrypting more data */
    keySz = 32;

    ret |= wc_Chacha_SetKey(&enc, keys[0], keySz);
    ret |= wc_Chacha_SetKey(&dec, keys[0], keySz);
    if (ret != 0)
        return ret;

    ret |= wc_Chacha_SetIV(&enc, ivs[2], 0);
    ret |= wc_Chacha_SetIV(&dec, ivs[2], 0);
    if (ret != 0)
        return ret;

    ret |= wc_Chacha_Process(&enc, cipher_big, plain_big, CHACHA_BIG_TEST_SIZE);
    ret |= wc_Chacha_Process(&dec, plain_big,  cipher_big,
                                                          CHACHA_BIG_TEST_SIZE);
    if (ret != 0)
        return ret;

    if (XMEMCMP(plain_big, input_big, CHACHA_BIG_TEST_SIZE))
        return -4721;

    if (XMEMCMP(cipher_big, cipher_big_result, CHACHA_BIG_TEST_SIZE))
        return -4722;

    for (i = 0; i < 18; ++i) {
        /* this will test all paths */
        /* block sizes: 1 2 3 4 7 8 15 16 31 32 63 64 127 128 255 256 511 512 */
        block_size = (2 << (i%9)) - (i<9?1:0);
        keySz = 32;

        ret |= wc_Chacha_SetKey(&enc, keys[0], keySz);
        ret |= wc_Chacha_SetKey(&dec, keys[0], keySz);
        if (ret != 0)
            return ret;

        ret |= wc_Chacha_SetIV(&enc, ivs[2], 0);
        ret |= wc_Chacha_SetIV(&dec, ivs[2], 0);
        if (ret != 0)
            return ret;

        ret |= wc_Chacha_Process(&enc, cipher_big, plain_big,  block_size);
        ret |= wc_Chacha_Process(&dec, plain_big,  cipher_big, block_size);
        if (ret != 0)
            return ret;

        if (XMEMCMP(plain_big, input_big, block_size))
            return -4723-i;

        if (XMEMCMP(cipher_big, cipher_big_result, block_size))
            return -4724-i;
    }

    /* Streaming test */
    for (i = 1; i <= (int)CHACHA_CHUNK_BYTES + 1; i++) {
        int j, rem;

        ret = wc_Chacha_SetKey(&enc, keys[0], keySz);
        if (ret != 0)
            return -4725;
        ret = wc_Chacha_SetKey(&dec, keys[0], keySz);
        if (ret != 0)
            return -4726;

        ret = wc_Chacha_SetIV(&enc, ivs[2], 0);
        if (ret != 0)
            return -4727;
        ret = wc_Chacha_SetIV(&dec, ivs[2], 0);
        if (ret != 0)
            return -4728;

        for (j = 0; j < CHACHA_BIG_TEST_SIZE - i; j+= i) {
            ret = wc_Chacha_Process(&enc, cipher_big + j, plain_big + j, i);
            if (ret != 0)
                return -4729;
            ret = wc_Chacha_Process(&dec, plain_big + j, cipher_big + j, i);
            if (ret != 0)
                return -4730;
        }

        rem = CHACHA_BIG_TEST_SIZE - j;
        ret = wc_Chacha_Process(&enc, cipher_big + j, plain_big + j, rem);
        if (ret != 0)
            return -4731;
        ret = wc_Chacha_Process(&dec, plain_big + j, cipher_big + j, rem);
        if (ret != 0)
            return -4732;

        if (XMEMCMP(plain_big, input_big, CHACHA_BIG_TEST_SIZE))
            return -4733;

        if (XMEMCMP(cipher_big, cipher_big_result, CHACHA_BIG_TEST_SIZE))
            return -4734;
    }

#ifdef WOLFSSL_SMALL_STACK
    XFREE(cipher_big, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    XFREE(plain_big, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    XFREE(input_big, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
#endif /* WOLFSSL_SMALL_STACK */
#endif /* BENCH_EMBEDDED */

    return 0;
}
#endif /* HAVE_CHACHA */


#ifdef HAVE_POLY1305
WOLFSSL_TEST_SUBROUTINE int poly1305_test(void)
{
    int      ret = 0;
    int      i;
    byte     tag[16];
    Poly1305 enc;

    WOLFSSL_SMALL_STACK_STATIC const byte msg1[] =
    {
        0x43,0x72,0x79,0x70,0x74,0x6f,0x67,0x72,
        0x61,0x70,0x68,0x69,0x63,0x20,0x46,0x6f,
        0x72,0x75,0x6d,0x20,0x52,0x65,0x73,0x65,
        0x61,0x72,0x63,0x68,0x20,0x47,0x72,0x6f,
        0x75,0x70
    };

    WOLFSSL_SMALL_STACK_STATIC const byte msg2[] =
    {
        0x48,0x65,0x6c,0x6c,0x6f,0x20,0x77,0x6f,0x72,
        0x6c,0x64,0x21
    };

    WOLFSSL_SMALL_STACK_STATIC const byte msg3[] =
    {
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
    };

    WOLFSSL_SMALL_STACK_STATIC const byte msg4[] =
    {
        0xd3,0x1a,0x8d,0x34,0x64,0x8e,0x60,0xdb,
        0x7b,0x86,0xaf,0xbc,0x53,0xef,0x7e,0xc2,
        0xa4,0xad,0xed,0x51,0x29,0x6e,0x08,0xfe,
        0xa9,0xe2,0xb5,0xa7,0x36,0xee,0x62,0xd6,
        0x3d,0xbe,0xa4,0x5e,0x8c,0xa9,0x67,0x12,
        0x82,0xfa,0xfb,0x69,0xda,0x92,0x72,0x8b,
        0x1a,0x71,0xde,0x0a,0x9e,0x06,0x0b,0x29,
        0x05,0xd6,0xa5,0xb6,0x7e,0xcd,0x3b,0x36,
        0x92,0xdd,0xbd,0x7f,0x2d,0x77,0x8b,0x8c,
        0x98,0x03,0xae,0xe3,0x28,0x09,0x1b,0x58,
        0xfa,0xb3,0x24,0xe4,0xfa,0xd6,0x75,0x94,
        0x55,0x85,0x80,0x8b,0x48,0x31,0xd7,0xbc,
        0x3f,0xf4,0xde,0xf0,0x8e,0x4b,0x7a,0x9d,
        0xe5,0x76,0xd2,0x65,0x86,0xce,0xc6,0x4b,
        0x61,0x16
    };

    WOLFSSL_SMALL_STACK_STATIC const byte msg5[] =
    {
        0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
        0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
    };

    WOLFSSL_SMALL_STACK_STATIC const byte msg6[] =
    {
        0xd3,0x1a,0x8d,0x34,0x64,0x8e,0x60,0xdb,
        0x7b,0x86,0xaf,0xbc,0x53,0xef,0x7e,0xc2,
        0xa4,0xad,0xed,0x51,0x29,0x6e,0x08,0xfe,
        0xa9,0xe2,0xb5,0xa7,0x36,0xee,0x62,0xd6,
        0x3d,0xbe,0xa4,0x5e,0x8c,0xa9,0x67,0x12,
        0x82,0xfa,0xfb,0x69,0xda,0x92,0x72,0x8b,
        0xfa,0xb3,0x24,0xe4,0xfa,0xd6,0x75,0x94,
        0x1a,0x71,0xde,0x0a,0x9e,0x06,0x0b,0x29,
        0xa9,0xe2,0xb5,0xa7,0x36,0xee,0x62,0xd6,
        0x3d,0xbe,0xa4,0x5e,0x8c,0xa9,0x67,0x12,
        0xfa,0xb3,0x24,0xe4,0xfa,0xd6,0x75,0x94,
        0x05,0xd6,0xa5,0xb6,0x7e,0xcd,0x3b,0x36,
        0x92,0xdd,0xbd,0x7f,0x2d,0x77,0x8b,0x8c,
        0x7b,0x86,0xaf,0xbc,0x53,0xef,0x7e,0xc2,
        0x98,0x03,0xae,0xe3,0x28,0x09,0x1b,0x58,
        0xfa,0xb3,0x24,0xe4,0xfa,0xd6,0x75,0x94,
        0x55,0x85,0x80,0x8b,0x48,0x31,0xd7,0xbc,
        0x3f,0xf4,0xde,0xf0,0x8e,0x4b,0x7a,0x9d,
        0xe5,0x76,0xd2,0x65,0x86,0xce,0xc6,0x4b,
        0x61,0x16
    };

    byte additional[] =
    {
        0x50,0x51,0x52,0x53,0xc0,0xc1,0xc2,0xc3,
        0xc4,0xc5,0xc6,0xc7
    };

    WOLFSSL_SMALL_STACK_STATIC const byte correct0[] =
    {
        0x01,0x03,0x80,0x8a,0xfb,0x0d,0xb2,0xfd,
        0x4a,0xbf,0xf6,0xaf,0x41,0x49,0xf5,0x1b
    };

    WOLFSSL_SMALL_STACK_STATIC const byte correct1[] =
    {
        0xa8,0x06,0x1d,0xc1,0x30,0x51,0x36,0xc6,
        0xc2,0x2b,0x8b,0xaf,0x0c,0x01,0x27,0xa9
    };

    WOLFSSL_SMALL_STACK_STATIC const byte correct2[] =
    {
        0xa6,0xf7,0x45,0x00,0x8f,0x81,0xc9,0x16,
        0xa2,0x0d,0xcc,0x74,0xee,0xf2,0xb2,0xf0
    };

    WOLFSSL_SMALL_STACK_STATIC const byte correct3[] =
    {
        0x49,0xec,0x78,0x09,0x0e,0x48,0x1e,0xc6,
        0xc2,0x6b,0x33,0xb9,0x1c,0xcc,0x03,0x07
    };

    WOLFSSL_SMALL_STACK_STATIC const byte correct4[] =
    {
        0x1a,0xe1,0x0b,0x59,0x4f,0x09,0xe2,0x6a,
        0x7e,0x90,0x2e,0xcb,0xd0,0x60,0x06,0x91
    };

    WOLFSSL_SMALL_STACK_STATIC const byte correct5[] =
    {
        0x03,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    };

    WOLFSSL_SMALL_STACK_STATIC const byte correct6[] =
    {
        0xea,0x11,0x5c,0x4f,0xd0,0xc0,0x10,0xae,
        0xf7,0xdf,0xda,0x77,0xa2,0xe9,0xaf,0xca
    };

    WOLFSSL_SMALL_STACK_STATIC const byte key[] = {
        0x85,0xd6,0xbe,0x78,0x57,0x55,0x6d,0x33,
        0x7f,0x44,0x52,0xfe,0x42,0xd5,0x06,0xa8,
        0x01,0x03,0x80,0x8a,0xfb,0x0d,0xb2,0xfd,
        0x4a,0xbf,0xf6,0xaf,0x41,0x49,0xf5,0x1b
    };

    WOLFSSL_SMALL_STACK_STATIC const byte key2[] = {
        0x74,0x68,0x69,0x73,0x20,0x69,0x73,0x20,
        0x33,0x32,0x2d,0x62,0x79,0x74,0x65,0x20,
        0x6b,0x65,0x79,0x20,0x66,0x6f,0x72,0x20,
        0x50,0x6f,0x6c,0x79,0x31,0x33,0x30,0x35
    };

    WOLFSSL_SMALL_STACK_STATIC const byte key4[] = {
        0x7b,0xac,0x2b,0x25,0x2d,0xb4,0x47,0xaf,
        0x09,0xb6,0x7a,0x55,0xa4,0xe9,0x55,0x84,
        0x0a,0xe1,0xd6,0x73,0x10,0x75,0xd9,0xeb,
        0x2a,0x93,0x75,0x78,0x3e,0xd5,0x53,0xff
    };

    WOLFSSL_SMALL_STACK_STATIC const byte key5[] = {
        0x02,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
    };

    const byte* msgs[]  = {NULL, msg1, msg2, msg3, msg5, msg6};
    word32      szm[]   = {0, sizeof(msg1), sizeof(msg2),
                           sizeof(msg3), sizeof(msg5), sizeof(msg6)};
    const byte* keys[]  = {key, key, key2, key2, key5, key};
    const byte* tests[] = {correct0, correct1, correct2, correct3, correct5,
                           correct6};

    for (i = 0; i < 6; i++) {
        ret = wc_Poly1305SetKey(&enc, keys[i], 32);
        if (ret != 0)
            return -4800 - i;

        ret = wc_Poly1305Update(&enc, msgs[i], szm[i]);
        if (ret != 0)
            return -4810 - i;

        ret = wc_Poly1305Final(&enc, tag);
        if (ret != 0)
            return -4820 - i;

        if (XMEMCMP(tag, tests[i], sizeof(tag)))
            return -4830 - i;
    }

    /* Check TLS MAC function from 2.8.2 https://tools.ietf.org/html/rfc7539 */
    XMEMSET(tag, 0, sizeof(tag));
    ret = wc_Poly1305SetKey(&enc, key4, sizeof(key4));
    if (ret != 0)
        return -4840;

    ret = wc_Poly1305_MAC(&enc, additional, sizeof(additional),
                                   (byte*)msg4, sizeof(msg4), tag, sizeof(tag));
    if (ret != 0)
        return -4841;

    if (XMEMCMP(tag, correct4, sizeof(tag)))
        return -4842;

    /* Check fail of TLS MAC function if altering additional data */
    XMEMSET(tag, 0, sizeof(tag));
    additional[0]++;
    ret = wc_Poly1305_MAC(&enc, additional, sizeof(additional),
                                   (byte*)msg4, sizeof(msg4), tag, sizeof(tag));
    if (ret != 0)
        return -4843;

    if (XMEMCMP(tag, correct4, sizeof(tag)) == 0)
        return -4844;


    return 0;
}
#endif /* HAVE_POLY1305 */


#if defined(HAVE_CHACHA) && defined(HAVE_POLY1305)
WOLFSSL_TEST_SUBROUTINE int chacha20_poly1305_aead_test(void)
{
    /* Test #1 from Section 2.8.2 of draft-irtf-cfrg-chacha20-poly1305-10 */
    /* https://tools.ietf.org/html/draft-irtf-cfrg-chacha20-poly1305-10  */

    WOLFSSL_SMALL_STACK_STATIC const byte key1[] = {
        0x80, 0x81, 0x82, 0x83, 0x84, 0x85, 0x86, 0x87,
        0x88, 0x89, 0x8a, 0x8b, 0x8c, 0x8d, 0x8e, 0x8f,
        0x90, 0x91, 0x92, 0x93, 0x94, 0x95, 0x96, 0x97,
        0x98, 0x99, 0x9a, 0x9b, 0x9c, 0x9d, 0x9e, 0x9f
    };

    WOLFSSL_SMALL_STACK_STATIC const byte plaintext1[] = {
        0x4c, 0x61, 0x64, 0x69, 0x65, 0x73, 0x20, 0x61,
        0x6e, 0x64, 0x20, 0x47, 0x65, 0x6e, 0x74, 0x6c,
        0x65, 0x6d, 0x65, 0x6e, 0x20, 0x6f, 0x66, 0x20,
        0x74, 0x68, 0x65, 0x20, 0x63, 0x6c, 0x61, 0x73,
        0x73, 0x20, 0x6f, 0x66, 0x20, 0x27, 0x39, 0x39,
        0x3a, 0x20, 0x49, 0x66, 0x20, 0x49, 0x20, 0x63,
        0x6f, 0x75, 0x6c, 0x64, 0x20, 0x6f, 0x66, 0x66,
        0x65, 0x72, 0x20, 0x79, 0x6f, 0x75, 0x20, 0x6f,
        0x6e, 0x6c, 0x79, 0x20, 0x6f, 0x6e, 0x65, 0x20,
        0x74, 0x69, 0x70, 0x20, 0x66, 0x6f, 0x72, 0x20,
        0x74, 0x68, 0x65, 0x20, 0x66, 0x75, 0x74, 0x75,
        0x72, 0x65, 0x2c, 0x20, 0x73, 0x75, 0x6e, 0x73,
        0x63, 0x72, 0x65, 0x65, 0x6e, 0x20, 0x77, 0x6f,
        0x75, 0x6c, 0x64, 0x20, 0x62, 0x65, 0x20, 0x69,
        0x74, 0x2e
    };

    WOLFSSL_SMALL_STACK_STATIC const byte iv1[] = {
        0x07, 0x00, 0x00, 0x00, 0x40, 0x41, 0x42, 0x43,
        0x44, 0x45, 0x46, 0x47
    };

    WOLFSSL_SMALL_STACK_STATIC const byte aad1[] = { /* additional data */
        0x50, 0x51, 0x52, 0x53, 0xc0, 0xc1, 0xc2, 0xc3,
        0xc4, 0xc5, 0xc6, 0xc7
    };

    WOLFSSL_SMALL_STACK_STATIC const byte cipher1[] = { /* expected output from operation */
        0xd3, 0x1a, 0x8d, 0x34, 0x64, 0x8e, 0x60, 0xdb,
        0x7b, 0x86, 0xaf, 0xbc, 0x53, 0xef, 0x7e, 0xc2,
        0xa4, 0xad, 0xed, 0x51, 0x29, 0x6e, 0x08, 0xfe,
        0xa9, 0xe2, 0xb5, 0xa7, 0x36, 0xee, 0x62, 0xd6,
        0x3d, 0xbe, 0xa4, 0x5e, 0x8c, 0xa9, 0x67, 0x12,
        0x82, 0xfa, 0xfb, 0x69, 0xda, 0x92, 0x72, 0x8b,
        0x1a, 0x71, 0xde, 0x0a, 0x9e, 0x06, 0x0b, 0x29,
        0x05, 0xd6, 0xa5, 0xb6, 0x7e, 0xcd, 0x3b, 0x36,
        0x92, 0xdd, 0xbd, 0x7f, 0x2d, 0x77, 0x8b, 0x8c,
        0x98, 0x03, 0xae, 0xe3, 0x28, 0x09, 0x1b, 0x58,
        0xfa, 0xb3, 0x24, 0xe4, 0xfa, 0xd6, 0x75, 0x94,
        0x55, 0x85, 0x80, 0x8b, 0x48, 0x31, 0xd7, 0xbc,
        0x3f, 0xf4, 0xde, 0xf0, 0x8e, 0x4b, 0x7a, 0x9d,
        0xe5, 0x76, 0xd2, 0x65, 0x86, 0xce, 0xc6, 0x4b,
        0x61, 0x16
    };

    WOLFSSL_SMALL_STACK_STATIC const byte authTag1[] = { /* expected output from operation */
        0x1a, 0xe1, 0x0b, 0x59, 0x4f, 0x09, 0xe2, 0x6a,
        0x7e, 0x90, 0x2e, 0xcb, 0xd0, 0x60, 0x06, 0x91
    };

    /* Test #2 from Appendix A.2 in draft-irtf-cfrg-chacha20-poly1305-10 */
    /* https://tools.ietf.org/html/draft-irtf-cfrg-chacha20-poly1305-10  */

    WOLFSSL_SMALL_STACK_STATIC const byte key2[] = {
        0x1c, 0x92, 0x40, 0xa5, 0xeb, 0x55, 0xd3, 0x8a,
        0xf3, 0x33, 0x88, 0x86, 0x04, 0xf6, 0xb5, 0xf0,
        0x47, 0x39, 0x17, 0xc1, 0x40, 0x2b, 0x80, 0x09,
        0x9d, 0xca, 0x5c, 0xbc, 0x20, 0x70, 0x75, 0xc0
    };

    WOLFSSL_SMALL_STACK_STATIC const byte plaintext2[] = {
        0x49, 0x6e, 0x74, 0x65, 0x72, 0x6e, 0x65, 0x74,
        0x2d, 0x44, 0x72, 0x61, 0x66, 0x74, 0x73, 0x20,
        0x61, 0x72, 0x65, 0x20, 0x64, 0x72, 0x61, 0x66,
        0x74, 0x20, 0x64, 0x6f, 0x63, 0x75, 0x6d, 0x65,
        0x6e, 0x74, 0x73, 0x20, 0x76, 0x61, 0x6c, 0x69,
        0x64, 0x20, 0x66, 0x6f, 0x72, 0x20, 0x61, 0x20,
        0x6d, 0x61, 0x78, 0x69, 0x6d, 0x75, 0x6d, 0x20,
        0x6f, 0x66, 0x20, 0x73, 0x69, 0x78, 0x20, 0x6d,
        0x6f, 0x6e, 0x74, 0x68, 0x73, 0x20, 0x61, 0x6e,
        0x64, 0x20, 0x6d, 0x61, 0x79, 0x20, 0x62, 0x65,
        0x20, 0x75, 0x70, 0x64, 0x61, 0x74, 0x65, 0x64,
        0x2c, 0x20, 0x72, 0x65, 0x70, 0x6c, 0x61, 0x63,
        0x65, 0x64, 0x2c, 0x20, 0x6f, 0x72, 0x20, 0x6f,
        0x62, 0x73, 0x6f, 0x6c, 0x65, 0x74, 0x65, 0x64,
        0x20, 0x62, 0x79, 0x20, 0x6f, 0x74, 0x68, 0x65,
        0x72, 0x20, 0x64, 0x6f, 0x63, 0x75, 0x6d, 0x65,
        0x6e, 0x74, 0x73, 0x20, 0x61, 0x74, 0x20, 0x61,
        0x6e, 0x79, 0x20, 0x74, 0x69, 0x6d, 0x65, 0x2e,
        0x20, 0x49, 0x74, 0x20, 0x69, 0x73, 0x20, 0x69,
        0x6e, 0x61, 0x70, 0x70, 0x72, 0x6f, 0x70, 0x72,
        0x69, 0x61, 0x74, 0x65, 0x20, 0x74, 0x6f, 0x20,
        0x75, 0x73, 0x65, 0x20, 0x49, 0x6e, 0x74, 0x65,
        0x72, 0x6e, 0x65, 0x74, 0x2d, 0x44, 0x72, 0x61,
        0x66, 0x74, 0x73, 0x20, 0x61, 0x73, 0x20, 0x72,
        0x65, 0x66, 0x65, 0x72, 0x65, 0x6e, 0x63, 0x65,
        0x20, 0x6d, 0x61, 0x74, 0x65, 0x72, 0x69, 0x61,
        0x6c, 0x20, 0x6f, 0x72, 0x20, 0x74, 0x6f, 0x20,
        0x63, 0x69, 0x74, 0x65, 0x20, 0x74, 0x68, 0x65,
        0x6d, 0x20, 0x6f, 0x74, 0x68, 0x65, 0x72, 0x20,
        0x74, 0x68, 0x61, 0x6e, 0x20, 0x61, 0x73, 0x20,
        0x2f, 0xe2, 0x80, 0x9c, 0x77, 0x6f, 0x72, 0x6b,
        0x20, 0x69, 0x6e, 0x20, 0x70, 0x72, 0x6f, 0x67,
        0x72, 0x65, 0x73, 0x73, 0x2e, 0x2f, 0xe2, 0x80,
        0x9d
    };

    WOLFSSL_SMALL_STACK_STATIC const byte iv2[] = {
        0x00, 0x00, 0x00, 0x00, 0x01, 0x02, 0x03, 0x04,
        0x05, 0x06, 0x07, 0x08
    };

    WOLFSSL_SMALL_STACK_STATIC const byte aad2[] = { /* additional data */
        0xf3, 0x33, 0x88, 0x86, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x4e, 0x91
    };

    WOLFSSL_SMALL_STACK_STATIC const byte cipher2[] = { /* expected output from operation */
        0x64, 0xa0, 0x86, 0x15, 0x75, 0x86, 0x1a, 0xf4,
        0x60, 0xf0, 0x62, 0xc7, 0x9b, 0xe6, 0x43, 0xbd,
        0x5e, 0x80, 0x5c, 0xfd, 0x34, 0x5c, 0xf3, 0x89,
        0xf1, 0x08, 0x67, 0x0a, 0xc7, 0x6c, 0x8c, 0xb2,
        0x4c, 0x6c, 0xfc, 0x18, 0x75, 0x5d, 0x43, 0xee,
        0xa0, 0x9e, 0xe9, 0x4e, 0x38, 0x2d, 0x26, 0xb0,
        0xbd, 0xb7, 0xb7, 0x3c, 0x32, 0x1b, 0x01, 0x00,
        0xd4, 0xf0, 0x3b, 0x7f, 0x35, 0x58, 0x94, 0xcf,
        0x33, 0x2f, 0x83, 0x0e, 0x71, 0x0b, 0x97, 0xce,
        0x98, 0xc8, 0xa8, 0x4a, 0xbd, 0x0b, 0x94, 0x81,
        0x14, 0xad, 0x17, 0x6e, 0x00, 0x8d, 0x33, 0xbd,
        0x60, 0xf9, 0x82, 0xb1, 0xff, 0x37, 0xc8, 0x55,
        0x97, 0x97, 0xa0, 0x6e, 0xf4, 0xf0, 0xef, 0x61,
        0xc1, 0x86, 0x32, 0x4e, 0x2b, 0x35, 0x06, 0x38,
        0x36, 0x06, 0x90, 0x7b, 0x6a, 0x7c, 0x02, 0xb0,
        0xf9, 0xf6, 0x15, 0x7b, 0x53, 0xc8, 0x67, 0xe4,
        0xb9, 0x16, 0x6c, 0x76, 0x7b, 0x80, 0x4d, 0x46,
        0xa5, 0x9b, 0x52, 0x16, 0xcd, 0xe7, 0xa4, 0xe9,
        0x90, 0x40, 0xc5, 0xa4, 0x04, 0x33, 0x22, 0x5e,
        0xe2, 0x82, 0xa1, 0xb0, 0xa0, 0x6c, 0x52, 0x3e,
        0xaf, 0x45, 0x34, 0xd7, 0xf8, 0x3f, 0xa1, 0x15,
        0x5b, 0x00, 0x47, 0x71, 0x8c, 0xbc, 0x54, 0x6a,
        0x0d, 0x07, 0x2b, 0x04, 0xb3, 0x56, 0x4e, 0xea,
        0x1b, 0x42, 0x22, 0x73, 0xf5, 0x48, 0x27, 0x1a,
        0x0b, 0xb2, 0x31, 0x60, 0x53, 0xfa, 0x76, 0x99,
        0x19, 0x55, 0xeb, 0xd6, 0x31, 0x59, 0x43, 0x4e,
        0xce, 0xbb, 0x4e, 0x46, 0x6d, 0xae, 0x5a, 0x10,
        0x73, 0xa6, 0x72, 0x76, 0x27, 0x09, 0x7a, 0x10,
        0x49, 0xe6, 0x17, 0xd9, 0x1d, 0x36, 0x10, 0x94,
        0xfa, 0x68, 0xf0, 0xff, 0x77, 0x98, 0x71, 0x30,
        0x30, 0x5b, 0xea, 0xba, 0x2e, 0xda, 0x04, 0xdf,
        0x99, 0x7b, 0x71, 0x4d, 0x6c, 0x6f, 0x2c, 0x29,
        0xa6, 0xad, 0x5c, 0xb4, 0x02, 0x2b, 0x02, 0x70,
        0x9b
    };

    WOLFSSL_SMALL_STACK_STATIC const byte authTag2[] = { /* expected output from operation */
        0xee, 0xad, 0x9d, 0x67, 0x89, 0x0c, 0xbb, 0x22,
        0x39, 0x23, 0x36, 0xfe, 0xa1, 0x85, 0x1f, 0x38
    };

    byte generatedCiphertext[265]; /* max plaintext2/cipher2 */
    byte generatedPlaintext[265];  /* max plaintext2/cipher2 */
    byte generatedAuthTag[CHACHA20_POLY1305_AEAD_AUTHTAG_SIZE];
    int err;

    ChaChaPoly_Aead aead;

#if !defined(USE_INTEL_CHACHA_SPEEDUP) && !defined(WOLFSSL_ARMASM)
    #define TEST_SMALL_CHACHA_CHUNKS 32
#else
    #define TEST_SMALL_CHACHA_CHUNKS 64
#endif
    #ifdef TEST_SMALL_CHACHA_CHUNKS
    word32 testLen;
    #endif

    XMEMSET(generatedCiphertext, 0, sizeof(generatedCiphertext));
    XMEMSET(generatedAuthTag, 0, sizeof(generatedAuthTag));
    XMEMSET(generatedPlaintext, 0, sizeof(generatedPlaintext));

    /* Parameter Validation testing */
    /* Encrypt */
    err = wc_ChaCha20Poly1305_Encrypt(NULL, iv1, aad1, sizeof(aad1), plaintext1,
            sizeof(plaintext1), generatedCiphertext, generatedAuthTag);
    if (err != BAD_FUNC_ARG)
        return -4900;
    err = wc_ChaCha20Poly1305_Encrypt(key1, NULL, aad1, sizeof(aad1),
            plaintext1, sizeof(plaintext1), generatedCiphertext,
            generatedAuthTag);
    if (err != BAD_FUNC_ARG)
        return -4901;
    err = wc_ChaCha20Poly1305_Encrypt(key1, iv1, aad1, sizeof(aad1), NULL,
            sizeof(plaintext1), generatedCiphertext, generatedAuthTag);
    if (err != BAD_FUNC_ARG)
        return -4902;
    err = wc_ChaCha20Poly1305_Encrypt(key1, iv1, aad1, sizeof(aad1), plaintext1,
            sizeof(plaintext1), NULL, generatedAuthTag);
    if (err != BAD_FUNC_ARG)
        return -4903;
    err = wc_ChaCha20Poly1305_Encrypt(key1, iv1, aad1, sizeof(aad1), plaintext1,
            sizeof(plaintext1), generatedCiphertext, NULL);
    if (err != BAD_FUNC_ARG)
        return -4904;
    err = wc_ChaCha20Poly1305_Encrypt(key1, iv1, aad1, sizeof(aad1), NULL,
            sizeof(plaintext1), generatedCiphertext, generatedAuthTag);
    if (err != BAD_FUNC_ARG)
        return -4905;
    /* Decrypt */
    err = wc_ChaCha20Poly1305_Decrypt(NULL, iv2, aad2, sizeof(aad2), cipher2,
            sizeof(cipher2), authTag2, generatedPlaintext);
    if (err != BAD_FUNC_ARG)
        return -4906;
    err = wc_ChaCha20Poly1305_Decrypt(key2, NULL, aad2, sizeof(aad2), cipher2,
            sizeof(cipher2), authTag2, generatedPlaintext);
    if (err != BAD_FUNC_ARG)
        return -4907;
    err = wc_ChaCha20Poly1305_Decrypt(key2, iv2, aad2, sizeof(aad2), NULL,
            sizeof(cipher2), authTag2, generatedPlaintext);
    if (err != BAD_FUNC_ARG)
        return -4908;
    err = wc_ChaCha20Poly1305_Decrypt(key2, iv2, aad2, sizeof(aad2), cipher2,
            sizeof(cipher2), NULL, generatedPlaintext);
    if (err != BAD_FUNC_ARG)
        return -4909;
    err = wc_ChaCha20Poly1305_Decrypt(key2, iv2, aad2, sizeof(aad2), cipher2,
            sizeof(cipher2), authTag2, NULL);
    if (err != BAD_FUNC_ARG)
        return -4910;
    err = wc_ChaCha20Poly1305_Decrypt(key2, iv2, aad2, sizeof(aad2), NULL,
            sizeof(cipher2), authTag2, generatedPlaintext);
    if (err != BAD_FUNC_ARG)
        return -4911;


    /* Test #1 */
    err = wc_ChaCha20Poly1305_Encrypt(key1, iv1,
                                       aad1, sizeof(aad1),
                                       plaintext1, sizeof(plaintext1),
                                       generatedCiphertext, generatedAuthTag);
    if (err) {
        return err;
    }

    /* -- Check the ciphertext and authtag */
    if (XMEMCMP(generatedCiphertext, cipher1, sizeof(cipher1))) {
        return -4912;
    }
    if (XMEMCMP(generatedAuthTag, authTag1, sizeof(authTag1))) {
        return -4913;
    }

    /* -- Verify decryption works */
    err = wc_ChaCha20Poly1305_Decrypt(key1, iv1,
                                       aad1, sizeof(aad1),
                                       cipher1, sizeof(cipher1),
                                       authTag1, generatedPlaintext);
    if (err) {
        return err;
    }
    if (XMEMCMP(generatedPlaintext, plaintext1, sizeof(plaintext1))) {
        return -4914;
    }


    XMEMSET(generatedCiphertext, 0, sizeof(generatedCiphertext));
    XMEMSET(generatedAuthTag, 0, sizeof(generatedAuthTag));
    XMEMSET(generatedPlaintext, 0, sizeof(generatedPlaintext));

    /* Test #2 */
    err = wc_ChaCha20Poly1305_Encrypt(key2, iv2,
                                       aad2, sizeof(aad2),
                                       plaintext2, sizeof(plaintext2),
                                       generatedCiphertext, generatedAuthTag);
    if (err) {
        return err;
    }

    /* -- Check the ciphertext and authtag */
    if (XMEMCMP(generatedCiphertext, cipher2, sizeof(cipher2))) {
        return -4915;
    }
    if (XMEMCMP(generatedAuthTag, authTag2, sizeof(authTag2))) {
        return -4916;
    }

    /* -- Verify decryption works */
    err = wc_ChaCha20Poly1305_Decrypt(key2, iv2,
                                      aad2, sizeof(aad2),
                                      cipher2, sizeof(cipher2),
                                      authTag2, generatedPlaintext);
    if (err) {
        return err;
    }

    if (XMEMCMP(generatedPlaintext, plaintext2, sizeof(plaintext2))) {
        return -4917;
    }


    /* AEAD init/update/final - bad argument tests */
    err = wc_ChaCha20Poly1305_Init(NULL, key1, iv1,
        CHACHA20_POLY1305_AEAD_DECRYPT);
    if (err != BAD_FUNC_ARG)
        return -4918;
    err = wc_ChaCha20Poly1305_Init(&aead, NULL, iv1,
        CHACHA20_POLY1305_AEAD_DECRYPT);
    if (err != BAD_FUNC_ARG)
        return -4919;
    err = wc_ChaCha20Poly1305_Init(&aead, key1, NULL,
        CHACHA20_POLY1305_AEAD_DECRYPT);
    if (err != BAD_FUNC_ARG)
        return -4920;
    err = wc_ChaCha20Poly1305_UpdateAad(NULL, aad1, sizeof(aad1));
    if (err != BAD_FUNC_ARG)
        return -4921;
    err = wc_ChaCha20Poly1305_UpdateAad(&aead, NULL, sizeof(aad1));
    if (err != BAD_FUNC_ARG)
        return -4922;
    err = wc_ChaCha20Poly1305_UpdateData(NULL, generatedPlaintext,
        generatedPlaintext, sizeof(plaintext1));
    if (err != BAD_FUNC_ARG)
        return -4923;
    err = wc_ChaCha20Poly1305_UpdateData(&aead, generatedPlaintext, NULL,
        sizeof(plaintext1));
    if (err != BAD_FUNC_ARG)
        return -4924;
    err = wc_ChaCha20Poly1305_UpdateData(&aead, NULL, generatedPlaintext,
        sizeof(plaintext1));
    if (err != BAD_FUNC_ARG)
        return -4925;
    err = wc_ChaCha20Poly1305_Final(NULL, generatedAuthTag);
    if (err != BAD_FUNC_ARG)
        return -4926;
    err = wc_ChaCha20Poly1305_Final(&aead, NULL);
    if (err != BAD_FUNC_ARG)
        return -4927;

    /* AEAD init/update/final - bad state tests */
    /* clear struct - make valgrind happy to resolve
        "Conditional jump or move depends on uninitialised value(s)".
        The enum is "int" size and aead.state is "byte" */
    /* The wc_ChaCha20Poly1305_Init function does this normally */
    XMEMSET(&aead, 0, sizeof(aead));
    aead.state = CHACHA20_POLY1305_STATE_INIT;
    err = wc_ChaCha20Poly1305_UpdateAad(&aead, aad1, sizeof(aad1));
    if (err != BAD_STATE_E)
        return -4928;
    aead.state = CHACHA20_POLY1305_STATE_DATA;
    err = wc_ChaCha20Poly1305_UpdateAad(&aead, aad1, sizeof(aad1));
    if (err != BAD_STATE_E)
        return -4929;
    aead.state = CHACHA20_POLY1305_STATE_INIT;
    err = wc_ChaCha20Poly1305_UpdateData(&aead, generatedPlaintext,
        generatedPlaintext, sizeof(plaintext1));
    if (err != BAD_STATE_E)
        return -4930;
    aead.state = CHACHA20_POLY1305_STATE_INIT;
    err = wc_ChaCha20Poly1305_Final(&aead, generatedAuthTag);
    if (err != BAD_STATE_E)
        return -4931;
    aead.state = CHACHA20_POLY1305_STATE_READY;
    err = wc_ChaCha20Poly1305_Final(&aead, generatedAuthTag);
    if (err != BAD_STATE_E)
        return -4932;

    XMEMSET(generatedCiphertext, 0, sizeof(generatedCiphertext));
    XMEMSET(generatedAuthTag, 0, sizeof(generatedAuthTag));
    XMEMSET(generatedPlaintext, 0, sizeof(generatedPlaintext));

    /* Test 1 - Encrypt */
    err = wc_ChaCha20Poly1305_Init(&aead, key1, iv1,
        CHACHA20_POLY1305_AEAD_ENCRYPT);
    if (err != 0)
        return -4933;
    err = wc_ChaCha20Poly1305_UpdateAad(&aead, aad1, sizeof(aad1));
    if (err != 0)
        return -4934;
#ifdef TEST_SMALL_CHACHA_CHUNKS
    /* test doing data in smaller chunks */
    for (testLen=0; testLen<sizeof(plaintext1); ) {
        word32 dataLen = sizeof(plaintext1) - testLen;
        if (dataLen > TEST_SMALL_CHACHA_CHUNKS)
            dataLen = TEST_SMALL_CHACHA_CHUNKS;
        err = wc_ChaCha20Poly1305_UpdateData(&aead, &plaintext1[testLen],
            &generatedCiphertext[testLen], dataLen);
        if (err != 0)
            return -4935;
        testLen += dataLen;
    }
#else
    err = wc_ChaCha20Poly1305_UpdateData(&aead, plaintext1,
        generatedCiphertext, sizeof(plaintext1));
#endif
    err = wc_ChaCha20Poly1305_Final(&aead, generatedAuthTag);
    if (err != 0)
        return -4936;
    err = wc_ChaCha20Poly1305_CheckTag(generatedAuthTag, authTag1);
    if (err != 0)
        return -4937;
    if (XMEMCMP(generatedCiphertext, cipher1, sizeof(cipher1))) {
        return -4938;
    }

    /* Test 1 - Decrypt */
    err = wc_ChaCha20Poly1305_Init(&aead, key1, iv1,
        CHACHA20_POLY1305_AEAD_DECRYPT);
    if (err != 0)
        return -4939;
    err = wc_ChaCha20Poly1305_UpdateAad(&aead, aad1, sizeof(aad1));
    if (err != 0)
        return -4940;
#ifdef TEST_SMALL_CHACHA_CHUNKS
    /* test doing data in smaller chunks */
    for (testLen=0; testLen<sizeof(plaintext1); ) {
        word32 dataLen = sizeof(plaintext1) - testLen;
        if (dataLen > TEST_SMALL_CHACHA_CHUNKS)
            dataLen = TEST_SMALL_CHACHA_CHUNKS;
        err = wc_ChaCha20Poly1305_UpdateData(&aead,
            &generatedCiphertext[testLen], &generatedPlaintext[testLen],
            dataLen);
        if (err != 0)
            return -4941;
        testLen += dataLen;
    }
#else
    err = wc_ChaCha20Poly1305_UpdateData(&aead, generatedCiphertext,
        generatedPlaintext, sizeof(cipher1));
#endif
    err = wc_ChaCha20Poly1305_Final(&aead, generatedAuthTag);
    if (err != 0)
        return -4942;
    err = wc_ChaCha20Poly1305_CheckTag(generatedAuthTag, authTag1);
    if (err != 0)
        return -4943;
    if (XMEMCMP(generatedPlaintext, plaintext1, sizeof(plaintext1))) {
        return -4944;
    }

    XMEMSET(generatedCiphertext, 0, sizeof(generatedCiphertext));
    XMEMSET(generatedAuthTag, 0, sizeof(generatedAuthTag));
    XMEMSET(generatedPlaintext, 0, sizeof(generatedPlaintext));

    /* Test 2 - Encrypt */
    err = wc_ChaCha20Poly1305_Init(&aead, key2, iv2,
        CHACHA20_POLY1305_AEAD_ENCRYPT);
    if (err != 0)
        return -4945;
    err = wc_ChaCha20Poly1305_UpdateAad(&aead, aad2, sizeof(aad2));
    if (err != 0)
        return -4946;
#ifdef TEST_SMALL_CHACHA_CHUNKS
    /* test doing data in smaller chunks */
    for (testLen=0; testLen<sizeof(plaintext2); ) {
        word32 dataLen = sizeof(plaintext2) - testLen;
        if (dataLen > TEST_SMALL_CHACHA_CHUNKS)
            dataLen = TEST_SMALL_CHACHA_CHUNKS;
        err = wc_ChaCha20Poly1305_UpdateData(&aead, &plaintext2[testLen],
            &generatedCiphertext[testLen], dataLen);
        if (err != 0)
            return -4947;
        testLen += dataLen;
    }
#else
    err = wc_ChaCha20Poly1305_UpdateData(&aead, plaintext2, generatedCiphertext,
        sizeof(plaintext2));
#endif
    err = wc_ChaCha20Poly1305_Final(&aead, generatedAuthTag);
    if (err != 0)
        return -4948;
    err = wc_ChaCha20Poly1305_CheckTag(generatedAuthTag, authTag2);
    if (err != 0)
        return -4949;
    if (XMEMCMP(generatedCiphertext, cipher2, sizeof(cipher2))) {
        return -4950;
    }

    /* Test 2 - Decrypt */
    err = wc_ChaCha20Poly1305_Init(&aead, key2, iv2,
        CHACHA20_POLY1305_AEAD_DECRYPT);
    if (err != 0)
        return -4951;
    err = wc_ChaCha20Poly1305_UpdateAad(&aead, aad2, sizeof(aad2));
    if (err != 0)
        return -4952;
#ifdef TEST_SMALL_CHACHA_CHUNKS
    /* test doing data in smaller chunks */
    for (testLen=0; testLen<sizeof(plaintext2); ) {
        word32 dataLen = sizeof(plaintext2) - testLen;
        if (dataLen > TEST_SMALL_CHACHA_CHUNKS)
            dataLen = TEST_SMALL_CHACHA_CHUNKS;
        err = wc_ChaCha20Poly1305_UpdateData(&aead,
            &generatedCiphertext[testLen], &generatedPlaintext[testLen],
            dataLen);
        if (err != 0)
            return -4953;
        testLen += dataLen;
    }
#else
    err = wc_ChaCha20Poly1305_UpdateData(&aead, generatedCiphertext,
        generatedPlaintext, sizeof(cipher2));
#endif
    err = wc_ChaCha20Poly1305_Final(&aead, generatedAuthTag);
    if (err != 0)
        return -4954;
    err = wc_ChaCha20Poly1305_CheckTag(generatedAuthTag, authTag2);
    if (err != 0)
        return -4955;
    if (XMEMCMP(generatedPlaintext, plaintext2, sizeof(plaintext2))) {
        return -4956;
    }

    return err;
}
#endif /* HAVE_CHACHA && HAVE_POLY1305 */


#ifndef NO_DES3
WOLFSSL_TEST_SUBROUTINE int des_test(void)
{
    WOLFSSL_SMALL_STACK_STATIC const byte vector[] = { /* "now is the time for all " w/o trailing 0 */
        0x6e,0x6f,0x77,0x20,0x69,0x73,0x20,0x74,
        0x68,0x65,0x20,0x74,0x69,0x6d,0x65,0x20,
        0x66,0x6f,0x72,0x20,0x61,0x6c,0x6c,0x20
    };

    byte plain[24];
    byte cipher[24];

    Des enc;
    Des dec;

    WOLFSSL_SMALL_STACK_STATIC const byte key[] =
    {
        0x01,0x23,0x45,0x67,0x89,0xab,0xcd,0xef
    };

    WOLFSSL_SMALL_STACK_STATIC const byte iv[] =
    {
        0x12,0x34,0x56,0x78,0x90,0xab,0xcd,0xef
    };

    WOLFSSL_SMALL_STACK_STATIC const byte verify[] =
    {
        0x8b,0x7c,0x52,0xb0,0x01,0x2b,0x6c,0xb8,
        0x4f,0x0f,0xeb,0xf3,0xfb,0x5f,0x86,0x73,
        0x15,0x85,0xb3,0x22,0x4b,0x86,0x2b,0x4b
    };

    int ret;

    ret = wc_Des_SetKey(&enc, key, iv, DES_ENCRYPTION);
    if (ret != 0)
        return -5000;

    ret = wc_Des_CbcEncrypt(&enc, cipher, vector, sizeof(vector));
    if (ret != 0)
        return -5001;

    ret = wc_Des_SetKey(&dec, key, iv, DES_DECRYPTION);
    if (ret != 0)
        return -5002;

    ret = wc_Des_CbcDecrypt(&dec, plain, cipher, sizeof(cipher));
    if (ret != 0)
        return -5003;

    if (XMEMCMP(plain, vector, sizeof(plain)))
        return -5004;

    if (XMEMCMP(cipher, verify, sizeof(cipher)))
        return -5005;

    ret = wc_Des_CbcEncryptWithKey(cipher, vector, sizeof(vector), key, iv);
    if (ret != 0)
        return -5006;

#if defined(WOLFSSL_ENCRYPTED_KEYS) && !defined(NO_SHA)
    {
        EncryptedInfo info;
        XMEMSET(&info, 0, sizeof(EncryptedInfo));
        XMEMCPY(info.iv, iv, sizeof(iv));
        info.ivSz = sizeof(iv);
        info.keySz = sizeof(key);
        info.cipherType = WC_CIPHER_DES;

        ret = wc_BufferKeyEncrypt(&info, cipher, sizeof(cipher), key,
                sizeof(key), WC_HASH_TYPE_SHA);
        if (ret != 0)
            return -5007;

        /* Test invalid info ptr */
        ret = wc_BufferKeyEncrypt(NULL, cipher, sizeof(cipher), key,
                sizeof(key), WC_HASH_TYPE_SHA);
        if (ret != BAD_FUNC_ARG)
            return -5008;

    #ifndef NO_PWDBASED
        /* Test invalid hash type - only applies to wc_PBKDF1 call */
        ret = wc_BufferKeyEncrypt(&info, cipher, sizeof(cipher), key,
                sizeof(key), WC_HASH_TYPE_NONE);
        if (ret == 0)
            return -5009;
    #endif /* !NO_PWDBASED */
    }
#endif

    return 0;
}
#endif /* !NO_DES3 */


#ifndef NO_DES3
WOLFSSL_TEST_SUBROUTINE int des3_test(void)
{
    WOLFSSL_SMALL_STACK_STATIC const byte vector[] = { /* "Now is the time for all " w/o trailing 0 */
        0x4e,0x6f,0x77,0x20,0x69,0x73,0x20,0x74,
        0x68,0x65,0x20,0x74,0x69,0x6d,0x65,0x20,
        0x66,0x6f,0x72,0x20,0x61,0x6c,0x6c,0x20
    };

    byte plain[24];
    byte cipher[24];

    Des3 enc;
    Des3 dec;

    WOLFSSL_SMALL_STACK_STATIC const byte key3[] =
    {
        0x01,0x23,0x45,0x67,0x89,0xab,0xcd,0xef,
        0xfe,0xde,0xba,0x98,0x76,0x54,0x32,0x10,
        0x89,0xab,0xcd,0xef,0x01,0x23,0x45,0x67
    };
    WOLFSSL_SMALL_STACK_STATIC const byte iv3[] =
    {
        0x12,0x34,0x56,0x78,0x90,0xab,0xcd,0xef,
        0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,
        0x11,0x21,0x31,0x41,0x51,0x61,0x71,0x81

    };

    WOLFSSL_SMALL_STACK_STATIC const byte verify3[] =
    {
        0x43,0xa0,0x29,0x7e,0xd1,0x84,0xf8,0x0e,
        0x89,0x64,0x84,0x32,0x12,0xd5,0x08,0x98,
        0x18,0x94,0x15,0x74,0x87,0x12,0x7d,0xb0
    };

    int ret;


    if (wc_Des3Init(&enc, HEAP_HINT, devId) != 0)
        return -5100;
    if (wc_Des3Init(&dec, HEAP_HINT, devId) != 0)
        return -5101;

    ret = wc_Des3_SetKey(&enc, key3, iv3, DES_ENCRYPTION);
    if (ret != 0)
        return -5102;
    ret = wc_Des3_SetKey(&dec, key3, iv3, DES_DECRYPTION);
    if (ret != 0)
        return -5103;
    ret = wc_Des3_CbcEncrypt(&enc, cipher, vector, sizeof(vector));
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &enc.asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    if (ret != 0)
        return -5104;
    ret = wc_Des3_CbcDecrypt(&dec, plain, cipher, sizeof(cipher));
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &dec.asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    if (ret != 0)
        return -5105;

    if (XMEMCMP(plain, vector, sizeof(plain)))
        return -5106;

    if (XMEMCMP(cipher, verify3, sizeof(cipher)))
        return -5107;

#if defined(OPENSSL_EXTRA) && !defined(WOLFCRYPT_ONLY)
    /* test the same vectors with using compatibility layer */
    {
        DES_key_schedule ks1;
        DES_key_schedule ks2;
        DES_key_schedule ks3;
        DES_cblock iv4;

        XMEMCPY(ks1, key3, sizeof(DES_key_schedule));
        XMEMCPY(ks2, key3 + 8, sizeof(DES_key_schedule));
        XMEMCPY(ks3, key3 + 16, sizeof(DES_key_schedule));
        XMEMCPY(iv4, iv3, sizeof(DES_cblock));

        XMEMSET(plain, 0, sizeof(plain));
        XMEMSET(cipher, 0, sizeof(cipher));

        DES_ede3_cbc_encrypt(vector, cipher, sizeof(vector), &ks1, &ks2, &ks3,
                &iv4, DES_ENCRYPT);
        DES_ede3_cbc_encrypt(cipher, plain, sizeof(cipher), &ks1, &ks2, &ks3,
                &iv4, DES_DECRYPT);

        if (XMEMCMP(plain, vector, sizeof(plain)))
            return -5108;

        if (XMEMCMP(cipher, verify3, sizeof(cipher)))
            return -5109;
    }
#endif /* OPENSSL_EXTRA */

    wc_Des3Free(&enc);
    wc_Des3Free(&dec);

#if defined(WOLFSSL_ENCRYPTED_KEYS) && !defined(NO_SHA)
    {
        EncryptedInfo info;
        XMEMSET(&info, 0, sizeof(EncryptedInfo));
        XMEMCPY(info.iv, iv3, sizeof(iv3));
        info.ivSz = sizeof(iv3);
        info.keySz = sizeof(key3);
        info.cipherType = WC_CIPHER_DES3;

        ret = wc_BufferKeyEncrypt(&info, cipher, sizeof(cipher), key3,
                sizeof(key3), WC_HASH_TYPE_SHA);
        if (ret != 0)
            return -5110;
    }
#endif

    return 0;
}
#endif /* NO_DES3 */


#ifndef NO_AES

#if defined(WOLFSSL_AES_OFB) || defined(WOLFSSL_AES_CFB) || \
    defined(WOLFSSL_AES_XTS)
#if defined(OPENSSL_EXTRA) && !defined(WOLFCRYPT_ONLY) \
    && !defined(HAVE_SELFTEST) && !defined(HAVE_FIPS)
/* pass in the function, key, iv, plain text and expected and this function
 * tests that the encryption and decryption is successful */
static int EVP_test(const WOLFSSL_EVP_CIPHER* type, const byte* key,
        const byte* iv, const byte* plain, int plainSz,
        const byte* expected, int expectedSz)
{
#ifdef WOLFSSL_SMALL_STACK
    EVP_CIPHER_CTX *ctx = NULL;
#else
    EVP_CIPHER_CTX ctx[1];
#endif
    int idx, ret = 0, cipherSz;
    byte* cipher;

#ifdef WOLFSSL_SMALL_STACK
    if ((ctx = wolfSSL_EVP_CIPHER_CTX_new()) == NULL)
        return MEMORY_E;
#endif

    cipher = (byte*)XMALLOC(plainSz, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    if (cipher == NULL) {
        ret = -5120;
        goto EVP_TEST_END;
    }

    /* test encrypt */
    EVP_CIPHER_CTX_init(ctx);
    if (EVP_CipherInit(ctx, type, key, iv, 1) == 0) {
        ret = -5121;
        goto EVP_TEST_END;
    }

    if (EVP_CipherUpdate(ctx, cipher, &idx, plain, expectedSz) == 0) {
        ret = -5122;
        goto EVP_TEST_END;
    }

    cipherSz = idx;
    if (EVP_CipherFinal(ctx, cipher + cipherSz, &idx) == 0) {
        ret = -5123;
        goto EVP_TEST_END;
    }
    cipherSz += idx;

    if (XMEMCMP(cipher, expected, plainSz)) {
        ret = -5124;
        goto EVP_TEST_END;
    }

    /* test decrypt */
    EVP_CIPHER_CTX_init(ctx);
    if (EVP_CipherInit(ctx, type, key, iv, 0) == 0) {
        ret = -5125;
        goto EVP_TEST_END;
    }

    if (EVP_CipherUpdate(ctx, cipher, &idx, cipher, expectedSz) == 0) {
        ret = -5126;
        goto EVP_TEST_END;
    }

    cipherSz = idx;
    if (EVP_CipherFinal(ctx, cipher + cipherSz, &idx) == 0) {
        ret = -5127;
        goto EVP_TEST_END;
    }
    cipherSz += idx;

    if ((expectedSz != cipherSz) || XMEMCMP(plain, cipher, plainSz)) {
        ret = -5128;
        goto EVP_TEST_END;
    }

EVP_TEST_END:
    if (cipher)
        XFREE(cipher, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    (void)cipherSz;
#ifdef WOLFSSL_SMALL_STACK
    wolfSSL_EVP_CIPHER_CTX_free(ctx);
#endif

    return ret;
}
#endif /* OPENSSL_EXTRA */
#endif /* WOLFSSL_AES_OFB || WOLFSSL_AES_CFB */

#ifdef WOLFSSL_AES_OFB
    /* test vector from https://csrc.nist.gov/Projects/cryptographic-algorithm-validation-program/Block-Ciphers */
    WOLFSSL_TEST_SUBROUTINE int aesofb_test(void)
    {
    #ifdef WOLFSSL_AES_256
        WOLFSSL_SMALL_STACK_STATIC const byte key1[] =
        {
            0xc4,0xc7,0xfa,0xd6,0x53,0x5c,0xb8,0x71,
            0x4a,0x5c,0x40,0x77,0x9a,0x8b,0xa1,0xd2,
            0x53,0x3e,0x23,0xb4,0xb2,0x58,0x73,0x2a,
            0x5b,0x78,0x01,0xf4,0xe3,0x71,0xa7,0x94
        };
        WOLFSSL_SMALL_STACK_STATIC const byte iv1[] =
        {
            0x5e,0xb9,0x33,0x13,0xb8,0x71,0xff,0x16,
            0xb9,0x8a,0x9b,0xcb,0x43,0x33,0x0d,0x6f
        };
        WOLFSSL_SMALL_STACK_STATIC const byte plain1[] =
        {
            0x6d,0x0b,0xb0,0x79,0x63,0x84,0x71,0xe9,
            0x39,0xd4,0x53,0x14,0x86,0xc1,0x4c,0x25,
            0x9a,0xee,0xc6,0xf3,0xc0,0x0d,0xfd,0xd6,
            0xc0,0x50,0xa8,0xba,0xa8,0x20,0xdb,0x71,
            0xcc,0x12,0x2c,0x4e,0x0c,0x17,0x15,0xef,
            0x55,0xf3,0x99,0x5a,0x6b,0xf0,0x2a,0x4c
        };
        WOLFSSL_SMALL_STACK_STATIC const byte cipher1[] =
        {
            0x0f,0x54,0x61,0x71,0x59,0xd0,0x3f,0xfc,
            0x1b,0xfa,0xfb,0x60,0x29,0x30,0xd7,0x00,
            0xf4,0xa4,0xa8,0xe6,0xdd,0x93,0x94,0x46,
            0x64,0xd2,0x19,0xc4,0xc5,0x4d,0xde,0x1b,
            0x04,0x53,0xe1,0x73,0xf5,0x18,0x74,0xae,
            0xfd,0x64,0xa2,0xe1,0xe2,0x76,0x13,0xb0
        };
    #endif /* WOLFSSL_AES_256 */


    #ifdef WOLFSSL_AES_128
        WOLFSSL_SMALL_STACK_STATIC const byte key2[] =
        {
            0x10,0xa5,0x88,0x69,0xd7,0x4b,0xe5,0xa3,
            0x74,0xcf,0x86,0x7c,0xfb,0x47,0x38,0x59
        };
        WOLFSSL_SMALL_STACK_STATIC const byte iv2[] =
        {
            0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
            0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
        };
        WOLFSSL_SMALL_STACK_STATIC const byte plain2[] =
        {
            0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
            0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
        };
        WOLFSSL_SMALL_STACK_STATIC const byte cipher2[] =
        {
            0x6d,0x25,0x1e,0x69,0x44,0xb0,0x51,0xe0,
            0x4e,0xaa,0x6f,0xb4,0xdb,0xf7,0x84,0x65
        };
    #endif /* WOLFSSL_AES_128 */


    #ifdef WOLFSSL_AES_192
        WOLFSSL_SMALL_STACK_STATIC const byte key3[] = {
            0xd0,0x77,0xa0,0x3b,0xd8,0xa3,0x89,0x73,
            0x92,0x8c,0xca,0xfe,0x4a,0x9d,0x2f,0x45,
            0x51,0x30,0xbd,0x0a,0xf5,0xae,0x46,0xa9
        };
        WOLFSSL_SMALL_STACK_STATIC const byte iv3[] =
        {
            0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
            0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
        };
        WOLFSSL_SMALL_STACK_STATIC const byte cipher3[] =
        {
            0xab,0xc7,0x86,0xfb,0x1e,0xdb,0x50,0x45,
            0x80,0xc4,0xd8,0x82,0xef,0x29,0xa0,0xc7
        };
        WOLFSSL_SMALL_STACK_STATIC const byte plain3[] =
        {
            0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
            0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
        };
    #endif /* WOLFSSL_AES_192 */

#ifdef WOLFSSL_SMALL_STACK
        Aes *enc = NULL;
#else
        Aes enc[1];
#endif
        byte cipher[AES_BLOCK_SIZE * 4];
    #ifdef HAVE_AES_DECRYPT
#ifdef WOLFSSL_SMALL_STACK
        Aes *dec = NULL;
#else
        Aes dec[1];
#endif
        byte plain [AES_BLOCK_SIZE * 4];
    #endif
        int  ret = 0;

#ifdef WOLFSSL_SMALL_STACK
    if ((enc = (Aes *)XMALLOC(sizeof *enc, HEAP_HINT, DYNAMIC_TYPE_AES)) == NULL)
        ERROR_OUT(-1, out);
#ifdef HAVE_AES_DECRYPT
    if ((dec = (Aes *)XMALLOC(sizeof *dec, HEAP_HINT, DYNAMIC_TYPE_AES)) == NULL)
        ERROR_OUT(-1, out);
#endif
#endif

        XMEMSET(enc, 0, sizeof *enc);
    #ifdef HAVE_AES_DECRYPT
        XMEMSET(dec, 0, sizeof *dec);
    #endif

#ifdef WOLFSSL_AES_128
        /* 128 key size test */
    #if defined(OPENSSL_EXTRA) && !defined(WOLFCRYPT_ONLY) \
        && !defined(HAVE_SELFTEST) && !defined(HAVE_FIPS)
        ret = EVP_test(EVP_aes_128_ofb(), key2, iv2, plain2, sizeof(plain2),
                cipher2, sizeof(cipher2));
        if (ret != 0) {
            goto out;
        }
    #endif

        ret = wc_AesSetKey(enc, key2, sizeof(key2), iv2, AES_ENCRYPTION);
        if (ret != 0)
            ERROR_OUT(-5129, out);
    #ifdef HAVE_AES_DECRYPT
        /* decrypt uses AES_ENCRYPTION */
        ret = wc_AesSetKey(dec, key2, sizeof(key2), iv2, AES_ENCRYPTION);
        if (ret != 0)
            ERROR_OUT(-5130, out);
    #endif

        XMEMSET(cipher, 0, sizeof(cipher));
        ret = wc_AesOfbEncrypt(enc, cipher, plain2, AES_BLOCK_SIZE);
        if (ret != 0)
            ERROR_OUT(-5131, out);

        if (XMEMCMP(cipher, cipher2, AES_BLOCK_SIZE))
            ERROR_OUT(-5132, out);

    #ifdef HAVE_AES_DECRYPT
        ret = wc_AesOfbDecrypt(dec, plain, cipher2, AES_BLOCK_SIZE);
        if (ret != 0)
            ERROR_OUT(-5133, out);

        if (XMEMCMP(plain, plain2, AES_BLOCK_SIZE))
            ERROR_OUT(-5134, out);
    #endif /* HAVE_AES_DECRYPT */
#endif /* WOLFSSL_AES_128 */

#ifdef WOLFSSL_AES_192
        /* 192 key size test */
    #if defined(OPENSSL_EXTRA) && !defined(WOLFCRYPT_ONLY) \
        && !defined(HAVE_SELFTEST) && !defined(HAVE_FIPS)
        ret = EVP_test(EVP_aes_192_ofb(), key3, iv3, plain3, sizeof(plain3),
                cipher3, sizeof(cipher3));
        if (ret != 0) {
            goto out;
        }
    #endif

        ret = wc_AesSetKey(enc, key3, sizeof(key3), iv3, AES_ENCRYPTION);
        if (ret != 0)
            ERROR_OUT(-5135, out);
    #ifdef HAVE_AES_DECRYPT
        /* decrypt uses AES_ENCRYPTION */
        ret = wc_AesSetKey(dec, key3, sizeof(key3), iv3, AES_ENCRYPTION);
        if (ret != 0)
            ERROR_OUT(-5136, out);
    #endif

        XMEMSET(cipher, 0, sizeof(cipher));
        ret = wc_AesOfbEncrypt(enc, cipher, plain3, AES_BLOCK_SIZE);
        if (ret != 0)
            ERROR_OUT(-5137, out);

        if (XMEMCMP(cipher, cipher3, AES_BLOCK_SIZE))
            ERROR_OUT(-5138, out);

    #ifdef HAVE_AES_DECRYPT
        ret = wc_AesOfbDecrypt(dec, plain, cipher3, AES_BLOCK_SIZE);
        if (ret != 0)
            ERROR_OUT(-5139, out);

        if (XMEMCMP(plain, plain3, AES_BLOCK_SIZE))
            ERROR_OUT(-5140, out);
    #endif /* HAVE_AES_DECRYPT */
#endif /* WOLFSSL_AES_192 */

#ifdef WOLFSSL_AES_256
        /* 256 key size test */
    #if defined(OPENSSL_EXTRA) && !defined(WOLFCRYPT_ONLY) \
        && !defined(HAVE_SELFTEST) && !defined(HAVE_FIPS)
        ret = EVP_test(EVP_aes_256_ofb(), key1, iv1, plain1, sizeof(plain1),
                cipher1, sizeof(cipher1));
        if (ret != 0) {
            goto out;
        }
    #endif

        ret = wc_AesSetKey(enc, key1, sizeof(key1), iv1, AES_ENCRYPTION);
        if (ret != 0)
            ERROR_OUT(-5141, out);
    #ifdef HAVE_AES_DECRYPT
        /* decrypt uses AES_ENCRYPTION */
        ret = wc_AesSetKey(dec, key1, sizeof(key1), iv1, AES_ENCRYPTION);
        if (ret != 0)
            ERROR_OUT(-5142, out);
    #endif

        XMEMSET(cipher, 0, sizeof(cipher));
        ret = wc_AesOfbEncrypt(enc, cipher, plain1, AES_BLOCK_SIZE);
        if (ret != 0)
            ERROR_OUT(-5143, out);

        if (XMEMCMP(cipher, cipher1, AES_BLOCK_SIZE))
            ERROR_OUT(-5144, out);

        ret = wc_AesOfbEncrypt(enc, cipher + AES_BLOCK_SIZE,
                plain1 + AES_BLOCK_SIZE, AES_BLOCK_SIZE);
        if (ret != 0)
            ERROR_OUT(-5145, out);

        if (XMEMCMP(cipher + AES_BLOCK_SIZE, cipher1 + AES_BLOCK_SIZE,
                    AES_BLOCK_SIZE))
            ERROR_OUT(-5146, out);

    #ifdef HAVE_AES_DECRYPT
        ret = wc_AesOfbDecrypt(dec, plain, cipher1, AES_BLOCK_SIZE);
        if (ret != 0)
            ERROR_OUT(-5147, out);

        if (XMEMCMP(plain, plain1, AES_BLOCK_SIZE))
            ERROR_OUT(-5148, out);

        ret = wc_AesOfbDecrypt(dec, plain + AES_BLOCK_SIZE,
                cipher1 + AES_BLOCK_SIZE, AES_BLOCK_SIZE);
        if (ret != 0)
            ERROR_OUT(-5149, out);

        if (XMEMCMP(plain + AES_BLOCK_SIZE, plain1 + AES_BLOCK_SIZE,
                    AES_BLOCK_SIZE))
            ERROR_OUT(-5150, out);
    #endif /* HAVE_AES_DECRYPT */

        /* multiple blocks at once */
        ret = wc_AesSetKey(enc, key1, sizeof(key1), iv1, AES_ENCRYPTION);
        if (ret != 0)
            ERROR_OUT(-5151, out);
    #ifdef HAVE_AES_DECRYPT
        /* decrypt uses AES_ENCRYPTION */
        ret = wc_AesSetKey(dec, key1, sizeof(key1), iv1, AES_ENCRYPTION);
        if (ret != 0)
            ERROR_OUT(-5152, out);
    #endif

        XMEMSET(cipher, 0, sizeof(cipher));
        ret = wc_AesOfbEncrypt(enc, cipher, plain1, AES_BLOCK_SIZE * 3);
        if (ret != 0)
            ERROR_OUT(-5153, out);

        if (XMEMCMP(cipher, cipher1, AES_BLOCK_SIZE * 3))
            ERROR_OUT(-5154, out);

    #ifdef HAVE_AES_DECRYPT
        ret = wc_AesOfbDecrypt(dec, plain, cipher1, AES_BLOCK_SIZE * 3);
        if (ret != 0)
            ERROR_OUT(-5155, out);

        if (XMEMCMP(plain, plain1, AES_BLOCK_SIZE * 3))
            ERROR_OUT(-5156, out);
    #endif /* HAVE_AES_DECRYPT */

        /* inline decrypt/encrypt*/
        ret = wc_AesSetKey(enc, key1, sizeof(key1), iv1, AES_ENCRYPTION);
        if (ret != 0)
            ERROR_OUT(-5157, out);
    #ifdef HAVE_AES_DECRYPT
        /* decrypt uses AES_ENCRYPTION */
        ret = wc_AesSetKey(dec, key1, sizeof(key1), iv1, AES_ENCRYPTION);
        if (ret != 0)
            ERROR_OUT(-5158, out);
    #endif

        XMEMCPY(cipher, plain1, AES_BLOCK_SIZE * 2);
        ret = wc_AesOfbEncrypt(enc, cipher, cipher, AES_BLOCK_SIZE * 2);
        if (ret != 0)
            ERROR_OUT(-5159, out);

        if (XMEMCMP(cipher, cipher1, AES_BLOCK_SIZE * 2))
            ERROR_OUT(-5160, out);

    #ifdef HAVE_AES_DECRYPT
        ret = wc_AesOfbDecrypt(dec, cipher, cipher, AES_BLOCK_SIZE * 2);
        if (ret != 0)
            ERROR_OUT(-5161, out);

        if (XMEMCMP(cipher, plain1, AES_BLOCK_SIZE * 2))
            ERROR_OUT(-5162, out);
    #endif /* HAVE_AES_DECRYPT */

        /* 256 key size test leftover support */
        ret = wc_AesSetKey(enc, key1, sizeof(key1), iv1, AES_ENCRYPTION);
        if (ret != 0)
            ERROR_OUT(-5163, out);
    #ifdef HAVE_AES_DECRYPT
        /* decrypt uses AES_ENCRYPTION */
        ret = wc_AesSetKey(dec, key1, sizeof(key1), iv1, AES_ENCRYPTION);
        if (ret != 0)
            ERROR_OUT(-5164, out);
    #endif

        XMEMSET(cipher, 0, sizeof(cipher));
        ret = wc_AesOfbEncrypt(enc, cipher, plain1, 3);
        if (ret != 0)
            ERROR_OUT(-5165, out);

        if (XMEMCMP(cipher, cipher1, 3))
            ERROR_OUT(-5166, out);

        ret = wc_AesOfbEncrypt(enc, cipher + 3, plain1 + 3, AES_BLOCK_SIZE);
        if (ret != 0)
            ERROR_OUT(-5167, out);

        if (XMEMCMP(cipher + 3, cipher1 + 3, AES_BLOCK_SIZE))
            ERROR_OUT(-5168, out);

    #ifdef HAVE_AES_DECRYPT
        ret = wc_AesOfbDecrypt(dec, plain, cipher1, 6);
        if (ret != 0)
            ERROR_OUT(-5169, out);

        if (XMEMCMP(plain, plain1, 6))
            ERROR_OUT(-5170, out);

        ret = wc_AesOfbDecrypt(dec, plain + 6, cipher1 + 6, AES_BLOCK_SIZE);
        if (ret != 0)
            ERROR_OUT(-5171, out);

        if (XMEMCMP(plain + 6, plain1 + 6, AES_BLOCK_SIZE))
            ERROR_OUT(-5172, out);
    #endif /* HAVE_AES_DECRYPT */

  out:

        wc_AesFree(enc);
        wc_AesFree(dec);
#ifdef WOLFSSL_SMALL_STACK
    if (enc)
        XFREE(enc, HEAP_HINT, DYNAMIC_TYPE_AES);
#ifdef HAVE_AES_DECRYPT
    if (dec)
        XFREE(dec, HEAP_HINT, DYNAMIC_TYPE_AES);
#endif
#endif

#endif /* WOLFSSL_AES_256 */

        return ret;
    }
#endif /* WOLFSSL_AES_OFB */

#if defined(WOLFSSL_AES_CFB)
    /* Test cases from NIST SP 800-38A, Recommendation for Block Cipher Modes of Operation Methods an*/
    static int aescfb_test(void)
    {
#ifdef WOLFSSL_SMALL_STACK
        Aes *enc = NULL;
#else
        Aes enc[1];
#endif
        int enc_inited = 0;
        byte cipher[AES_BLOCK_SIZE * 4];
    #ifdef HAVE_AES_DECRYPT
#ifdef WOLFSSL_SMALL_STACK
        Aes *dec = NULL;
#else
        Aes dec[1];
#endif
        int dec_inited = 0;
        byte plain [AES_BLOCK_SIZE * 4];
    #endif
        int  ret = 0;

        WOLFSSL_SMALL_STACK_STATIC const byte iv[] = {
            0x00,0x01,0x02,0x03,0x04,0x05,0x06,0x07,
            0x08,0x09,0x0a,0x0b,0x0c,0x0d,0x0e,0x0f
        };

#ifdef WOLFSSL_AES_128
        WOLFSSL_SMALL_STACK_STATIC const byte key1[] =
        {
            0x2b,0x7e,0x15,0x16,0x28,0xae,0xd2,0xa6,
            0xab,0xf7,0x15,0x88,0x09,0xcf,0x4f,0x3c
        };

        WOLFSSL_SMALL_STACK_STATIC const byte cipher1[] =
        {
            0x3b,0x3f,0xd9,0x2e,0xb7,0x2d,0xad,0x20,
            0x33,0x34,0x49,0xf8,0xe8,0x3c,0xfb,0x4a,
            0xc8,0xa6,0x45,0x37,0xa0,0xb3,0xa9,0x3f,
            0xcd,0xe3,0xcd,0xad,0x9f,0x1c,0xe5,0x8b,
            0x26,0x75,0x1f,0x67,0xa3,0xcb,0xb1,0x40,
            0xb1,0x80,0x8c,0xf1,0x87,0xa4,0xf4,0xdf
        };

        WOLFSSL_SMALL_STACK_STATIC const byte msg1[] =
        {
            0x6b,0xc1,0xbe,0xe2,0x2e,0x40,0x9f,0x96,
            0xe9,0x3d,0x7e,0x11,0x73,0x93,0x17,0x2a,
            0xae,0x2d,0x8a,0x57,0x1e,0x03,0xac,0x9c,
            0x9e,0xb7,0x6f,0xac,0x45,0xaf,0x8e,0x51,
            0x30,0xc8,0x1c,0x46,0xa3,0x5c,0xe4,0x11,
            0xe5,0xfb,0xc1,0x19,0x1a,0x0a,0x52,0xef
        };
#endif /* WOLFSSL_AES_128 */

#ifdef WOLFSSL_AES_192
        /* 192 size key test */
        WOLFSSL_SMALL_STACK_STATIC const byte key2[] =
        {
            0x8e,0x73,0xb0,0xf7,0xda,0x0e,0x64,0x52,
            0xc8,0x10,0xf3,0x2b,0x80,0x90,0x79,0xe5,
            0x62,0xf8,0xea,0xd2,0x52,0x2c,0x6b,0x7b
        };

        WOLFSSL_SMALL_STACK_STATIC const byte cipher2[] =
        {
            0xcd,0xc8,0x0d,0x6f,0xdd,0xf1,0x8c,0xab,
            0x34,0xc2,0x59,0x09,0xc9,0x9a,0x41,0x74,
            0x67,0xce,0x7f,0x7f,0x81,0x17,0x36,0x21,
            0x96,0x1a,0x2b,0x70,0x17,0x1d,0x3d,0x7a,
            0x2e,0x1e,0x8a,0x1d,0xd5,0x9b,0x88,0xb1,
            0xc8,0xe6,0x0f,0xed,0x1e,0xfa,0xc4,0xc9,
            0xc0,0x5f,0x9f,0x9c,0xa9,0x83,0x4f,0xa0,
            0x42,0xae,0x8f,0xba,0x58,0x4b,0x09,0xff
        };

        WOLFSSL_SMALL_STACK_STATIC const byte msg2[] =
        {
            0x6b,0xc1,0xbe,0xe2,0x2e,0x40,0x9f,0x96,
            0xe9,0x3d,0x7e,0x11,0x73,0x93,0x17,0x2a,
            0xae,0x2d,0x8a,0x57,0x1e,0x03,0xac,0x9c,
            0x9e,0xb7,0x6f,0xac,0x45,0xaf,0x8e,0x51,
            0x30,0xc8,0x1c,0x46,0xa3,0x5c,0xe4,0x11,
            0xe5,0xfb,0xc1,0x19,0x1a,0x0a,0x52,0xef,
            0xf6,0x9f,0x24,0x45,0xdf,0x4f,0x9b,0x17,
            0xad,0x2b,0x41,0x7b,0xe6,0x6c,0x37,0x10
        };
#endif /* WOLFSSL_AES_192 */

#ifdef WOLFSSL_AES_256
        /* 256 size key simple test */
        WOLFSSL_SMALL_STACK_STATIC const byte key3[] =
        {
            0x60,0x3d,0xeb,0x10,0x15,0xca,0x71,0xbe,
            0x2b,0x73,0xae,0xf0,0x85,0x7d,0x77,0x81,
            0x1f,0x35,0x2c,0x07,0x3b,0x61,0x08,0xd7,
            0x2d,0x98,0x10,0xa3,0x09,0x14,0xdf,0xf4
        };

        WOLFSSL_SMALL_STACK_STATIC const byte cipher3[] =
        {
            0xdc,0x7e,0x84,0xbf,0xda,0x79,0x16,0x4b,
            0x7e,0xcd,0x84,0x86,0x98,0x5d,0x38,0x60,
            0x39,0xff,0xed,0x14,0x3b,0x28,0xb1,0xc8,
            0x32,0x11,0x3c,0x63,0x31,0xe5,0x40,0x7b,
            0xdf,0x10,0x13,0x24,0x15,0xe5,0x4b,0x92,
            0xa1,0x3e,0xd0,0xa8,0x26,0x7a,0xe2,0xf9,
            0x75,0xa3,0x85,0x74,0x1a,0xb9,0xce,0xf8,
            0x20,0x31,0x62,0x3d,0x55,0xb1,0xe4,0x71
        };

        WOLFSSL_SMALL_STACK_STATIC const byte msg3[] =
        {
            0x6b,0xc1,0xbe,0xe2,0x2e,0x40,0x9f,0x96,
            0xe9,0x3d,0x7e,0x11,0x73,0x93,0x17,0x2a,
            0xae,0x2d,0x8a,0x57,0x1e,0x03,0xac,0x9c,
            0x9e,0xb7,0x6f,0xac,0x45,0xaf,0x8e,0x51,
            0x30,0xc8,0x1c,0x46,0xa3,0x5c,0xe4,0x11,
            0xe5,0xfb,0xc1,0x19,0x1a,0x0a,0x52,0xef,
            0xf6,0x9f,0x24,0x45,0xdf,0x4f,0x9b,0x17,
            0xad,0x2b,0x41,0x7b,0xe6,0x6c,0x37,0x10
        };
#endif /* WOLFSSL_AES_256 */

#ifdef WOLFSSL_SMALL_STACK
    if ((enc = (Aes *)XMALLOC(sizeof *enc, HEAP_HINT, DYNAMIC_TYPE_AES)) == NULL)
        ERROR_OUT(-1, out);
#ifdef HAVE_AES_DECRYPT
    if ((dec = (Aes *)XMALLOC(sizeof *dec, HEAP_HINT, DYNAMIC_TYPE_AES)) == NULL)
        ERROR_OUT(-1, out);
#endif
#endif

    if (wc_AesInit(enc, HEAP_HINT, devId) != 0)
        ERROR_OUT(-5173, out);
    else
        enc_inited = 1;
#ifdef HAVE_AES_DECRYPT
    if (wc_AesInit(dec, HEAP_HINT, devId) != 0)
        ERROR_OUT(-5174, out);
    else
        dec_inited = 1;
#endif

#ifdef WOLFSSL_AES_128
        /* 128 key tests */
    #if defined(OPENSSL_EXTRA) && !defined(WOLFCRYPT_ONLY) \
        && !defined(HAVE_SELFTEST) && !defined(HAVE_FIPS)
        ret = EVP_test(EVP_aes_128_cfb128(), key1, iv, msg1, sizeof(msg1),
                cipher1, sizeof(cipher1));
        if (ret != 0) {
            return ret;
        }
    #endif

        ret = wc_AesSetKey(enc, key1, AES_BLOCK_SIZE, iv, AES_ENCRYPTION);
        if (ret != 0)
            ERROR_OUT(-5175, out);
    #ifdef HAVE_AES_DECRYPT
        /* decrypt uses AES_ENCRYPTION */
        ret = wc_AesSetKey(dec, key1, AES_BLOCK_SIZE, iv, AES_ENCRYPTION);
        if (ret != 0)
            ERROR_OUT(-5176, out);
    #endif

        XMEMSET(cipher, 0, sizeof(cipher));
        ret = wc_AesCfbEncrypt(enc, cipher, msg1, AES_BLOCK_SIZE * 2);
        if (ret != 0)
            ERROR_OUT(-5177, out);

        if (XMEMCMP(cipher, cipher1, AES_BLOCK_SIZE * 2))
            ERROR_OUT(-5178, out);

        /* test restarting encryption process */
        ret = wc_AesCfbEncrypt(enc, cipher + (AES_BLOCK_SIZE * 2),
                msg1 + (AES_BLOCK_SIZE * 2), AES_BLOCK_SIZE);
        if (ret != 0)
            ERROR_OUT(-5179, out);

        if (XMEMCMP(cipher + (AES_BLOCK_SIZE * 2),
                    cipher1 + (AES_BLOCK_SIZE * 2), AES_BLOCK_SIZE))
            ERROR_OUT(-5180, out);

    #ifdef HAVE_AES_DECRYPT
        ret = wc_AesCfbDecrypt(dec, plain, cipher, AES_BLOCK_SIZE * 3);
        if (ret != 0)
            ERROR_OUT(-5181, out);

        if (XMEMCMP(plain, msg1, AES_BLOCK_SIZE * 3))
            ERROR_OUT(-5182, out);
    #endif /* HAVE_AES_DECRYPT */
#endif /* WOLFSSL_AES_128 */

#ifdef WOLFSSL_AES_192
        /* 192 key size test */
    #if defined(OPENSSL_EXTRA) && !defined(WOLFCRYPT_ONLY) \
        && !defined(HAVE_SELFTEST) && !defined(HAVE_FIPS)
        ret = EVP_test(EVP_aes_192_cfb128(), key2, iv, msg2, sizeof(msg2),
                cipher2, sizeof(cipher2));
        if (ret != 0) {
            return ret;
        }
    #endif

        ret = wc_AesSetKey(enc, key2, sizeof(key2), iv, AES_ENCRYPTION);
        if (ret != 0)
            ERROR_OUT(-5183, out);
    #ifdef HAVE_AES_DECRYPT
        /* decrypt uses AES_ENCRYPTION */
        ret = wc_AesSetKey(dec, key2, sizeof(key2), iv, AES_ENCRYPTION);
        if (ret != 0)
            ERROR_OUT(-5184, out);
    #endif

        XMEMSET(cipher, 0, sizeof(cipher));
        ret = wc_AesCfbEncrypt(enc, cipher, msg2, AES_BLOCK_SIZE * 4);
        if (ret != 0)
            ERROR_OUT(-5185, out);

        if (XMEMCMP(cipher, cipher2, AES_BLOCK_SIZE * 4))
            ERROR_OUT(-5186, out);

    #ifdef HAVE_AES_DECRYPT
        ret = wc_AesCfbDecrypt(dec, plain, cipher, AES_BLOCK_SIZE * 4);
        if (ret != 0)
            ERROR_OUT(-5187, out);

        if (XMEMCMP(plain, msg2, AES_BLOCK_SIZE * 4))
            ERROR_OUT(-5188, out);
    #endif /* HAVE_AES_DECRYPT */
#endif /* WOLFSSL_AES_192 */

#ifdef WOLFSSL_AES_256
        /* 256 key size test */
    #if defined(OPENSSL_EXTRA) && !defined(WOLFCRYPT_ONLY) \
        && !defined(HAVE_SELFTEST) && !defined(HAVE_FIPS)
        ret = EVP_test(EVP_aes_256_cfb128(), key3, iv, msg3, sizeof(msg3),
                cipher3, sizeof(cipher3));
        if (ret != 0) {
            return ret;
        }
    #endif
        ret = wc_AesSetKey(enc, key3, sizeof(key3), iv, AES_ENCRYPTION);
        if (ret != 0)
            ERROR_OUT(-5189, out);
    #ifdef HAVE_AES_DECRYPT
        /* decrypt uses AES_ENCRYPTION */
        ret = wc_AesSetKey(dec, key3, sizeof(key3), iv, AES_ENCRYPTION);
        if (ret != 0)
            ERROR_OUT(-5190, out);
    #endif

        /* test with data left overs, magic lengths are checking near edges */
        XMEMSET(cipher, 0, sizeof(cipher));
        ret = wc_AesCfbEncrypt(enc, cipher, msg3, 4);
        if (ret != 0)
            ERROR_OUT(-5191, out);

        if (XMEMCMP(cipher, cipher3, 4))
            ERROR_OUT(-5192, out);

        ret = wc_AesCfbEncrypt(enc, cipher + 4, msg3 + 4, 27);
        if (ret != 0)
            ERROR_OUT(-5193, out);

        if (XMEMCMP(cipher + 4, cipher3 + 4, 27))
            ERROR_OUT(-5194, out);

        ret = wc_AesCfbEncrypt(enc, cipher + 31, msg3 + 31,
                (AES_BLOCK_SIZE * 4) - 31);
        if (ret != 0)
            ERROR_OUT(-5195, out);

        if (XMEMCMP(cipher, cipher3, AES_BLOCK_SIZE * 4))
            ERROR_OUT(-5196, out);

    #ifdef HAVE_AES_DECRYPT
        ret = wc_AesCfbDecrypt(dec, plain, cipher, 4);
        if (ret != 0)
            ERROR_OUT(-5197, out);

        if (XMEMCMP(plain, msg3, 4))
            ERROR_OUT(-5198, out);

        ret = wc_AesCfbDecrypt(dec, plain + 4, cipher + 4, 4);
        if (ret != 0)
            ERROR_OUT(-5199, out);

        ret = wc_AesCfbDecrypt(dec, plain + 8, cipher + 8, 23);
        if (ret != 0)
            ERROR_OUT(-5200, out);

        if (XMEMCMP(plain + 4, msg3 + 4, 27))
            ERROR_OUT(-5201, out);

        ret = wc_AesCfbDecrypt(dec, plain + 31, cipher + 31,
                (AES_BLOCK_SIZE * 4) - 31);
        if (ret != 0)
            ERROR_OUT(-5202, out);

        if (XMEMCMP(plain, msg3, AES_BLOCK_SIZE * 4))
            ERROR_OUT(-5203, out);
    #endif /* HAVE_AES_DECRYPT */
#endif /* WOLFSSL_AES_256 */

  out:

    if (enc_inited)
        wc_AesFree(enc);
    if (dec_inited)
        wc_AesFree(dec);

#ifdef WOLFSSL_SMALL_STACK
    if (enc)
        XFREE(enc, HEAP_HINT, DYNAMIC_TYPE_AES);
#ifdef HAVE_AES_DECRYPT
    if (dec)
        XFREE(dec, HEAP_HINT, DYNAMIC_TYPE_AES);
#endif
#endif

        return ret;
    }

#if !defined(HAVE_SELFTEST) && !defined(HAVE_FIPS)
    static int aescfb1_test(void)
    {
#ifdef WOLFSSL_SMALL_STACK
        Aes *enc = NULL;
#else
        Aes enc[1];
#endif
        int enc_inited = 0;
        byte cipher[AES_BLOCK_SIZE];
    #ifdef HAVE_AES_DECRYPT
#ifdef WOLFSSL_SMALL_STACK
        Aes *dec = NULL;
#else
        Aes dec[1];
#endif
        int dec_inited = 0;
        byte plain [AES_BLOCK_SIZE];
    #endif
        int  ret = 0;

#ifdef WOLFSSL_AES_128
        WOLFSSL_SMALL_STACK_STATIC const byte iv[] = {
            0x4d,0xbb,0xdc,0xaa,0x59,0xf3,0x63,0xc9,
            0x2a,0x3b,0x98,0x43,0xad,0x20,0xe2,0xb7
        };

        WOLFSSL_SMALL_STACK_STATIC const byte key1[] =
        {
            0xcd,0xef,0x9d,0x06,0x61,0xba,0xe4,0x73,
            0x8d,0x1a,0x58,0xa2,0xa6,0x22,0x8b,0x66
        };

        WOLFSSL_SMALL_STACK_STATIC const byte cipher1[] =
        {
            0x00
        };

        WOLFSSL_SMALL_STACK_STATIC const byte msg1[] =
        {
            0xC0
        };
#endif /* WOLFSSL_AES_128 */
#ifdef WOLFSSL_AES_192
        WOLFSSL_SMALL_STACK_STATIC const byte iv2[] = {
            0x57,0xc6,0x89,0x7c,0x99,0x52,0x28,0x13,
            0xbf,0x67,0x9c,0xe1,0x13,0x70,0xaf,0x5e
        };

        WOLFSSL_SMALL_STACK_STATIC const byte key2[] =
        {
            0xba,0xa1,0x58,0xa1,0x6b,0x50,0x4a,0x10,
            0x8e,0xd4,0x33,0x2e,0xe7,0xf2,0x9b,0xf6,
            0xd1,0xac,0x46,0xa8,0xde,0x5a,0xfe,0x7a
        };

        WOLFSSL_SMALL_STACK_STATIC const byte cipher2[] =
        {
            0x30
        };

        WOLFSSL_SMALL_STACK_STATIC const byte msg2[] =
        {
            0x80
        };
#endif /* WOLFSSL_AES_192 */
#ifdef WOLFSSL_AES_256
        WOLFSSL_SMALL_STACK_STATIC const byte iv3[] = {
            0x63,0x2e,0x9f,0x83,0x1f,0xa3,0x80,0x5e,
            0x52,0x02,0xbc,0xe0,0x6d,0x04,0xf9,0xa0
        };

        WOLFSSL_SMALL_STACK_STATIC const byte key3[] =
        {
            0xf6,0xfa,0xe4,0xf1,0x5d,0x91,0xfc,0x50,
            0x88,0x78,0x4f,0x84,0xa5,0x37,0x12,0x7e,
            0x32,0x63,0x55,0x9c,0x62,0x73,0x88,0x20,
            0xc2,0xcf,0x3d,0xe1,0x1c,0x2a,0x30,0x40
        };

        WOLFSSL_SMALL_STACK_STATIC const byte cipher3[] =
        {
            0xF7, 0x00
        };

        WOLFSSL_SMALL_STACK_STATIC const byte msg3[] =
        {
            0x41, 0xC0
        };
#endif /* WOLFSSL_AES_256 */

#ifdef WOLFSSL_SMALL_STACK
    if ((enc = (Aes *)XMALLOC(sizeof *enc, HEAP_HINT, DYNAMIC_TYPE_AES)) == NULL)
        ERROR_OUT(-1, out);
#ifdef HAVE_AES_DECRYPT
    if ((dec = (Aes *)XMALLOC(sizeof *dec, HEAP_HINT, DYNAMIC_TYPE_AES)) == NULL)
        ERROR_OUT(-1, out);
#endif
#endif

    if (wc_AesInit(enc, HEAP_HINT, devId) != 0)
        ERROR_OUT(-5204, out);
    else
        enc_inited = 1;
#ifdef HAVE_AES_DECRYPT
    if (wc_AesInit(dec, HEAP_HINT, devId) != 0)
        ERROR_OUT(-5205, out);
    else
        dec_inited = 1;
#endif

#ifdef WOLFSSL_AES_128
        /* 128 key tests */
        ret = wc_AesSetKey(enc, key1, AES_BLOCK_SIZE, iv, AES_ENCRYPTION);
        if (ret != 0)
            ERROR_OUT(-5206, out);
    #ifdef HAVE_AES_DECRYPT
        /* decrypt uses AES_ENCRYPTION */
        ret = wc_AesSetKey(dec, key1, AES_BLOCK_SIZE, iv, AES_ENCRYPTION);
        if (ret != 0)
            ERROR_OUT(-5207, out);
    #endif

        XMEMSET(cipher, 0, sizeof(cipher));
        ret = wc_AesCfb1Encrypt(enc, cipher, msg1, 2);
        if (ret != 0)
            ERROR_OUT(-5208, out);

        if (cipher[0] != cipher1[0])
            ERROR_OUT(-5209, out);

    #ifdef HAVE_AES_DECRYPT
        ret = wc_AesCfb1Decrypt(dec, plain, cipher, 2);
        if (ret != 0)
            ERROR_OUT(-5210, out);

        if (plain[0] != msg1[0])
            ERROR_OUT(-5211, out);
    #endif /* HAVE_AES_DECRYPT */

    #ifdef OPENSSL_EXTRA
        ret = wc_AesSetKey(enc, key1, AES_BLOCK_SIZE, iv, AES_ENCRYPTION);
        if (ret != 0)
            ERROR_OUT(-5212, out);

        XMEMSET(cipher, 0, sizeof(cipher));
        ret = wc_AesCfb1Encrypt(enc, cipher, msg1,
                sizeof(msg1) * WOLFSSL_BIT_SIZE);
        if (ret != 0)
            ERROR_OUT(-5213, out);

    #ifndef WOLFCRYPT_ONLY
        ret = EVP_test(EVP_aes_128_cfb1(), key1, iv, msg1, sizeof(msg1),
                cipher, sizeof(msg1));
        if (ret != 0) {
            goto out;
        }
    #endif
    #endif
#endif /* WOLFSSL_AES_128 */
#ifdef WOLFSSL_AES_192
        /* 192 key tests */
        ret = wc_AesSetKey(enc, key2, sizeof(key2), iv2, AES_ENCRYPTION);
        if (ret != 0)
            ERROR_OUT(-5214, out);

        XMEMSET(cipher, 0, sizeof(cipher));
        ret = wc_AesCfb1Encrypt(enc, cipher, msg2, 4);
        if (ret != 0)
            ERROR_OUT(-5215, out);
        if (XMEMCMP(cipher, cipher2, sizeof(cipher2)) != 0)
            ERROR_OUT(-5216, out);

    #ifdef OPENSSL_EXTRA
        ret = wc_AesSetKey(enc, key2, sizeof(key2), iv2, AES_ENCRYPTION);
        if (ret != 0)
            ERROR_OUT(-5217, out);

        XMEMSET(cipher, 0, sizeof(cipher));
        ret = wc_AesCfb1Encrypt(enc, cipher, msg2,
                sizeof(msg2) * WOLFSSL_BIT_SIZE);
        if (ret != 0)
            ERROR_OUT(-5218, out);

        #ifndef WOLFCRYPT_ONLY
        ret = EVP_test(EVP_aes_192_cfb1(), key2, iv2, msg2, sizeof(msg2),
                cipher, sizeof(msg2));
        if (ret != 0) {
            goto out;
        }
        #endif
    #endif
#endif /* WOLFSSL_AES_192 */

#ifdef WOLFSSL_AES_256
        /* 256 key tests */
        ret = wc_AesSetKey(enc, key3, sizeof(key3), iv3, AES_ENCRYPTION);
        if (ret != 0)
            ERROR_OUT(-5219, out);

        XMEMSET(cipher, 0, sizeof(cipher));
        ret = wc_AesCfb1Encrypt(enc, cipher, msg3, 10);
        if (ret != 0)
            ERROR_OUT(-5220, out);
        if (XMEMCMP(cipher, cipher3, sizeof(cipher3)) != 0)
            ERROR_OUT(-5221, out);

    #ifdef OPENSSL_EXTRA
        ret = wc_AesSetKey(enc, key3, sizeof(key3), iv3, AES_ENCRYPTION);
        if (ret != 0)
            ERROR_OUT(-5222, out);

        XMEMSET(cipher, 0, sizeof(cipher));
        ret = wc_AesCfb1Encrypt(enc, cipher, msg3,
                sizeof(msg3) * WOLFSSL_BIT_SIZE);
        if (ret != 0)
            ERROR_OUT(-5223, out);

        #ifndef WOLFCRYPT_ONLY
        ret = EVP_test(EVP_aes_256_cfb1(), key3, iv3, msg3, sizeof(msg3),
                cipher, sizeof(msg3));
        if (ret != 0) {
            goto out;
        }
        #endif
    #endif

  out:

    if (enc_inited)
        wc_AesFree(enc);
#ifdef HAVE_AES_DECRYPT
    if (dec_inited)
        wc_AesFree(dec);
#endif

#ifdef WOLFSSL_SMALL_STACK
    if (enc)
        XFREE(enc, HEAP_HINT, DYNAMIC_TYPE_AES);
#ifdef HAVE_AES_DECRYPT
    if (dec)
        XFREE(dec, HEAP_HINT, DYNAMIC_TYPE_AES);
#endif
#endif

#endif /* WOLFSSL_AES_256 */
        return ret;
    }

    static int aescfb8_test(void)
    {
#ifdef WOLFSSL_SMALL_STACK
        Aes *enc = NULL;
#else
        Aes enc[1];
#endif
        int enc_inited = 0;
        byte cipher[AES_BLOCK_SIZE];
    #ifdef HAVE_AES_DECRYPT
#ifdef WOLFSSL_SMALL_STACK
        Aes *dec = NULL;
#else
        Aes dec[1];
#endif
        int dec_inited = 0;
        byte plain [AES_BLOCK_SIZE];
    #endif
        int  ret = 0;

#ifdef WOLFSSL_AES_128
        WOLFSSL_SMALL_STACK_STATIC const byte iv[] = {
            0xf4,0x75,0xc6,0x49,0x91,0xb2,0x0e,0xae,
            0xe1,0x83,0xa2,0x26,0x29,0xe2,0x1e,0x22
        };

        WOLFSSL_SMALL_STACK_STATIC const byte key1[] =
        {
            0xc8,0xfe,0x9b,0xf7,0x7b,0x93,0x0f,0x46,
            0xd2,0x07,0x8b,0x8c,0x0e,0x65,0x7c,0xd4
        };

        WOLFSSL_SMALL_STACK_STATIC const byte cipher1[] =
        {
            0xd2,0x76,0x91
        };

        WOLFSSL_SMALL_STACK_STATIC const byte msg1[] =
        {
            0xc9,0x06,0x35
        };
#endif /* WOLFSSL_AES_128 */
#ifdef WOLFSSL_AES_192
        WOLFSSL_SMALL_STACK_STATIC const byte iv2[] = {
            0x0a,0x02,0x84,0x6b,0x62,0xab,0xb6,0x93,
            0xef,0x31,0xd7,0x54,0x84,0x2e,0xed,0x29
        };

        WOLFSSL_SMALL_STACK_STATIC const byte key2[] =
        {
            0xba,0xf0,0x8b,0x76,0x31,0x7a,0x65,0xc5,
            0xf0,0x7a,0xe6,0xf5,0x7e,0xb0,0xe6,0x54,
            0x88,0x65,0x93,0x24,0xd2,0x97,0x09,0xe3
        };

        WOLFSSL_SMALL_STACK_STATIC const byte cipher2[] =
        {
            0x72,0x9c,0x0b,0x6d,0xeb,0x75,0xfa,0x6e,
            0xb5,0xe8
        };

        WOLFSSL_SMALL_STACK_STATIC const byte msg2[] =
        {
            0x98,0x95,0x93,0x24,0x02,0x39,0x3d,0xc3,
            0x3a,0x60
        };
#endif
#ifdef WOLFSSL_AES_256
        WOLFSSL_SMALL_STACK_STATIC const byte iv3[] = {
            0x33,0x8c,0x55,0x2f,0xf1,0xec,0xa1,0x44,
            0x08,0xe0,0x5d,0x8c,0xf9,0xf3,0xb3,0x1b
        };

        WOLFSSL_SMALL_STACK_STATIC const byte key3[] =
        {
            0x06,0x48,0x74,0x09,0x2f,0x7a,0x13,0xcc,
            0x44,0x62,0x24,0x7a,0xd4,0x23,0xd0,0xe9,
            0x6e,0xdf,0x42,0xe8,0xb6,0x7a,0x5a,0x23,
            0xb7,0xa0,0xa6,0x47,0x7b,0x09,0x8e,0x66
        };

        WOLFSSL_SMALL_STACK_STATIC const byte cipher3[] =
        {
            0x1c,0xff,0x95
        };

        WOLFSSL_SMALL_STACK_STATIC const byte msg3[] =
        {
            0xb9,0x74,0xfa
        };
#endif

#ifdef WOLFSSL_SMALL_STACK
    if ((enc = (Aes *)XMALLOC(sizeof *enc, HEAP_HINT, DYNAMIC_TYPE_AES)) == NULL)
        ERROR_OUT(-5238, out);
#ifdef HAVE_AES_DECRYPT
    if ((dec = (Aes *)XMALLOC(sizeof *dec, HEAP_HINT, DYNAMIC_TYPE_AES)) == NULL)
        ERROR_OUT(-5239, out);
#endif
#endif

    if (wc_AesInit(enc, HEAP_HINT, devId) != 0)
        ERROR_OUT(-5224, out);
    else
        enc_inited = 1;
#ifdef HAVE_AES_DECRYPT
    if (wc_AesInit(dec, HEAP_HINT, devId) != 0)
        ERROR_OUT(-5225, out);
    else
        dec_inited = 1;
#endif

#ifdef WOLFSSL_AES_128
        /* 128 key tests */
    #if defined(OPENSSL_EXTRA) && !defined(WOLFCRYPT_ONLY)
        ret = EVP_test(EVP_aes_128_cfb8(), key1, iv, msg1, sizeof(msg1),
                cipher1, sizeof(cipher1));
        if (ret != 0) {
            return ret;
        }
    #endif
        ret = wc_AesSetKey(enc, key1, AES_BLOCK_SIZE, iv, AES_ENCRYPTION);
        if (ret != 0)
            ERROR_OUT(-5226, out);
    #ifdef HAVE_AES_DECRYPT
        /* decrypt uses AES_ENCRYPTION */
        ret = wc_AesSetKey(dec, key1, AES_BLOCK_SIZE, iv, AES_ENCRYPTION);
        if (ret != 0)
            ERROR_OUT(-5227, out);
    #endif

        XMEMSET(cipher, 0, sizeof(cipher));
        ret = wc_AesCfb8Encrypt(enc, cipher, msg1, sizeof(msg1));
        if (ret != 0)
            ERROR_OUT(-5228, out);

        if (XMEMCMP(cipher, cipher1, sizeof(cipher1)) != 0)
            ERROR_OUT(-5229, out);

    #ifdef HAVE_AES_DECRYPT
        ret = wc_AesCfb8Decrypt(dec, plain, cipher, sizeof(msg1));
        if (ret != 0)
            ERROR_OUT(-5230, out);

        if (XMEMCMP(plain, msg1, sizeof(msg1)) != 0)
            ERROR_OUT(-5231, out);
    #endif /* HAVE_AES_DECRYPT */
#endif /* WOLFSSL_AES_128 */
#ifdef WOLFSSL_AES_192
        /* 192 key tests */
        ret = wc_AesSetKey(enc, key2, sizeof(key2), iv2, AES_ENCRYPTION);
        if (ret != 0)
            ERROR_OUT(-5232, out);

        XMEMSET(cipher, 0, sizeof(cipher));
        ret = wc_AesCfb8Encrypt(enc, cipher, msg2, sizeof(msg2));
        if (ret != 0)
            ERROR_OUT(-5233, out);
        if (XMEMCMP(cipher, cipher2, sizeof(msg2)) != 0)
            ERROR_OUT(-5234, out);
#if defined(OPENSSL_EXTRA) && !defined(WOLFCRYPT_ONLY)
        ret = EVP_test(EVP_aes_192_cfb8(), key2, iv2, msg2, sizeof(msg2),
                cipher2, sizeof(msg2));
        if (ret != 0) {
            return ret;
        }
#endif

#endif /* WOLFSSL_AES_192 */

#ifdef WOLFSSL_AES_256
        /* 256 key tests */
        ret = wc_AesSetKey(enc, key3, sizeof(key3), iv3, AES_ENCRYPTION);
        if (ret != 0)
            ERROR_OUT(-5235, out);

        XMEMSET(cipher, 0, sizeof(cipher));
        ret = wc_AesCfb8Encrypt(enc, cipher, msg3, sizeof(msg3));
        if (ret != 0)
            ERROR_OUT(-5236, out);
        if (XMEMCMP(cipher, cipher3, sizeof(cipher3)) != 0)
            ERROR_OUT(-5237, out);

    #if defined(OPENSSL_EXTRA) && !defined(WOLFCRYPT_ONLY)
        ret = EVP_test(EVP_aes_256_cfb8(), key3, iv3, msg3, sizeof(msg3),
                cipher3, sizeof(msg3));
        if (ret != 0) {
            goto out;
        }
    #endif

      out:

        if (enc_inited)
            wc_AesFree(enc);
#ifdef HAVE_AES_DECRYPT
        if (dec_inited)
            wc_AesFree(dec);
#endif

#ifdef WOLFSSL_SMALL_STACK
    if (enc)
        XFREE(enc, HEAP_HINT, DYNAMIC_TYPE_AES);
#ifdef HAVE_AES_DECRYPT
    if (dec)
        XFREE(dec, HEAP_HINT, DYNAMIC_TYPE_AES);
#endif
#endif

#endif /* WOLFSSL_AES_256 */

        return ret;
    }
#endif /* !HAVE_SELFTEST && !HAVE_FIPS */
#endif /* WOLFSSL_AES_CFB */


static int aes_key_size_test(void)
{
    int    ret;
#ifdef WOLFSSL_SMALL_STACK
    Aes    *aes;
#else
    Aes    aes[1];
#endif
    byte   key16[] = { 0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,
                       0x38, 0x39, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66 };
#ifndef WOLFSSL_CRYPTOCELL
    byte   key24[] = { 0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,
                       0x38, 0x39, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66,
                       0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37 };
#endif
    byte   key32[] = { 0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,
                       0x38, 0x39, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66,
                       0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,
                       0x38, 0x39, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66 };
    byte   iv[]    = "1234567890abcdef";
#ifndef HAVE_FIPS
    word32 keySize;
#endif

#ifdef WOLFSSL_SMALL_STACK
    if ((aes = (Aes *)XMALLOC(sizeof *aes, HEAP_HINT, DYNAMIC_TYPE_AES)) == NULL)
        return -5315;
#endif

#if !defined(HAVE_FIPS) || \
    defined(HAVE_FIPS_VERSION) && (HAVE_FIPS_VERSION >= 2)
    /* w/ FIPS v1 (cert 2425) wc_AesInit just returns 0 always as it's not
     * supported with that FIPS version */
    ret = wc_AesInit(NULL, HEAP_HINT, devId);
    if (ret != BAD_FUNC_ARG)
        ERROR_OUT(-5300, out);
#endif

    ret = wc_AesInit(aes, HEAP_HINT, devId);
    /* 0 check OK for FIPSv1 */
    if (ret != 0)
        ERROR_OUT(-5301, out);

#ifndef HAVE_FIPS
    /* Parameter Validation testing. */
    ret = wc_AesGetKeySize(NULL, NULL);
    if (ret != BAD_FUNC_ARG)
        ERROR_OUT(-5302, out);
    ret = wc_AesGetKeySize(aes, NULL);
    if (ret != BAD_FUNC_ARG)
        ERROR_OUT(-5303, out);
    ret = wc_AesGetKeySize(NULL, &keySize);
    if (ret != BAD_FUNC_ARG)
        ERROR_OUT(-5304, out);
    /* Crashes in FIPS */
    ret = wc_AesSetKey(NULL, key16, sizeof(key16), iv, AES_ENCRYPTION);
    if (ret != BAD_FUNC_ARG)
        ERROR_OUT(-5305, out);
#endif
    /* NULL IV indicates to use all zeros IV. */
    ret = wc_AesSetKey(aes, key16, sizeof(key16), NULL, AES_ENCRYPTION);
#ifdef WOLFSSL_AES_128
    if (ret != 0)
#else
    if (ret != BAD_FUNC_ARG)
#endif
        ERROR_OUT(-5306, out);
    ret = wc_AesSetKey(aes, key32, sizeof(key32) - 1, iv, AES_ENCRYPTION);
    if (ret != BAD_FUNC_ARG)
        ERROR_OUT(-5307, out);
/* CryptoCell handles rounds internally */
#if !defined(HAVE_FIPS) && !defined(WOLFSSL_CRYPTOCELL)
    /* PSA don't use aes->rounds */
#if !defined(WOLFSSL_HAVE_PSA) || defined(WOLFSSL_PSA_NO_AES)
    /* Force invalid rounds */
    aes->rounds = 16;
    ret = wc_AesGetKeySize(aes, &keySize);
    if (ret != BAD_FUNC_ARG)
        ERROR_OUT(-5308, out);
#endif
#endif

    ret = wc_AesSetKey(aes, key16, sizeof(key16), iv, AES_ENCRYPTION);
#ifdef WOLFSSL_AES_128
    if (ret != 0)
#else
    if (ret != BAD_FUNC_ARG)
#endif
        ERROR_OUT(-5309, out);
#if !defined(HAVE_FIPS) && defined(WOLFSSL_AES_128)
    ret = wc_AesGetKeySize(aes, &keySize);
    if (ret != 0 || keySize != sizeof(key16))
        ERROR_OUT(-5310, out);
#endif
#ifndef WOLFSSL_CRYPTOCELL
/* Cryptocell only supports AES-128 key size */
    ret = wc_AesSetKey(aes, key24, sizeof(key24), iv, AES_ENCRYPTION);
#ifdef WOLFSSL_AES_192
    if (ret != 0)
#else
    if (ret != BAD_FUNC_ARG)
#endif
        ERROR_OUT(-5311, out);
#if !defined(HAVE_FIPS) && defined(WOLFSSL_AES_192)
    ret = wc_AesGetKeySize(aes, &keySize);
    if (ret != 0 || keySize != sizeof(key24))
        ERROR_OUT(-5312, out);
#endif

    ret = wc_AesSetKey(aes, key32, sizeof(key32), iv, AES_ENCRYPTION);
#ifdef WOLFSSL_AES_256
    if (ret != 0)
#else
    if (ret != BAD_FUNC_ARG)
#endif
        ERROR_OUT(-5313, out);
#if !defined(HAVE_FIPS) && defined(WOLFSSL_AES_256)
    ret = wc_AesGetKeySize(aes, &keySize);
    if (ret != 0 || keySize != sizeof(key32))
        ERROR_OUT(-5314, out);
#endif
#endif /* !WOLFSSL_CRYPTOCELL */
    ret = 0; /* success */
  out:

    wc_AesFree(aes);
#ifdef WOLFSSL_SMALL_STACK
    XFREE(aes, HEAP_HINT, DYNAMIC_TYPE_AES);
#endif

    return ret;
}

#if defined(WOLFSSL_AES_XTS)
/* test vectors from http://csrc.nist.gov/groups/STM/cavp/block-cipher-modes.html */
#ifdef WOLFSSL_AES_128
static int aes_xts_128_test(void)
{
#ifdef WOLFSSL_SMALL_STACK
    XtsAes *aes = NULL;
#else
    XtsAes aes[1];
#endif
    int aes_inited = 0;
    int ret = 0;
    unsigned char buf[AES_BLOCK_SIZE * 2];
    unsigned char cipher[AES_BLOCK_SIZE * 2];

    /* 128 key tests */
    WOLFSSL_SMALL_STACK_STATIC unsigned char k1[] = {
        0xa1, 0xb9, 0x0c, 0xba, 0x3f, 0x06, 0xac, 0x35,
        0x3b, 0x2c, 0x34, 0x38, 0x76, 0x08, 0x17, 0x62,
        0x09, 0x09, 0x23, 0x02, 0x6e, 0x91, 0x77, 0x18,
        0x15, 0xf2, 0x9d, 0xab, 0x01, 0x93, 0x2f, 0x2f
    };

    WOLFSSL_SMALL_STACK_STATIC unsigned char i1[] = {
        0x4f, 0xae, 0xf7, 0x11, 0x7c, 0xda, 0x59, 0xc6,
        0x6e, 0x4b, 0x92, 0x01, 0x3e, 0x76, 0x8a, 0xd5
    };

    WOLFSSL_SMALL_STACK_STATIC unsigned char p1[] = {
        0xeb, 0xab, 0xce, 0x95, 0xb1, 0x4d, 0x3c, 0x8d,
        0x6f, 0xb3, 0x50, 0x39, 0x07, 0x90, 0x31, 0x1c
    };

    /* plain text test of partial block is not from NIST test vector list */
    WOLFSSL_SMALL_STACK_STATIC unsigned char pp[] = {
        0xeb, 0xab, 0xce, 0x95, 0xb1, 0x4d, 0x3c, 0x8d,
        0x6f, 0xb3, 0x50, 0x39, 0x07, 0x90, 0x31, 0x1c,
        0x6e, 0x4b, 0x92, 0x01, 0x3e, 0x76, 0x8a, 0xd5
    };

    WOLFSSL_SMALL_STACK_STATIC unsigned char c1[] = {
        0x77, 0x8a, 0xe8, 0xb4, 0x3c, 0xb9, 0x8d, 0x5a,
        0x82, 0x50, 0x81, 0xd5, 0xbe, 0x47, 0x1c, 0x63
    };

    WOLFSSL_SMALL_STACK_STATIC unsigned char k2[] = {
        0x39, 0x25, 0x79, 0x05, 0xdf, 0xcc, 0x77, 0x76,
        0x6c, 0x87, 0x0a, 0x80, 0x6a, 0x60, 0xe3, 0xc0,
        0x93, 0xd1, 0x2a, 0xcf, 0xcb, 0x51, 0x42, 0xfa,
        0x09, 0x69, 0x89, 0x62, 0x5b, 0x60, 0xdb, 0x16
    };

    WOLFSSL_SMALL_STACK_STATIC unsigned char i2[] = {
        0x5c, 0xf7, 0x9d, 0xb6, 0xc5, 0xcd, 0x99, 0x1a,
        0x1c, 0x78, 0x81, 0x42, 0x24, 0x95, 0x1e, 0x84
    };

    WOLFSSL_SMALL_STACK_STATIC unsigned char p2[] = {
        0xbd, 0xc5, 0x46, 0x8f, 0xbc, 0x8d, 0x50, 0xa1,
        0x0d, 0x1c, 0x85, 0x7f, 0x79, 0x1c, 0x5c, 0xba,
        0xb3, 0x81, 0x0d, 0x0d, 0x73, 0xcf, 0x8f, 0x20,
        0x46, 0xb1, 0xd1, 0x9e, 0x7d, 0x5d, 0x8a, 0x56
    };

    WOLFSSL_SMALL_STACK_STATIC unsigned char c2[] = {
        0xd6, 0xbe, 0x04, 0x6d, 0x41, 0xf2, 0x3b, 0x5e,
        0xd7, 0x0b, 0x6b, 0x3d, 0x5c, 0x8e, 0x66, 0x23,
        0x2b, 0xe6, 0xb8, 0x07, 0xd4, 0xdc, 0xc6, 0x0e,
        0xff, 0x8d, 0xbc, 0x1d, 0x9f, 0x7f, 0xc8, 0x22
    };

#ifdef WOLFSSL_SMALL_STACK
    if ((aes = (XtsAes *)XMALLOC(sizeof *aes, HEAP_HINT, DYNAMIC_TYPE_AES)) == NULL)
        ERROR_OUT(-5417, out);
#endif

#if defined(OPENSSL_EXTRA) && !defined(WOLFCRYPT_ONLY) \
    && !defined(HAVE_SELFTEST) && !defined(HAVE_FIPS)
    ret = EVP_test(EVP_aes_128_xts(), k2, i2, p2, sizeof(p2), c2, sizeof(c2));
    if (ret != 0) {
        printf("EVP_aes_128_xts failed!\n");
        goto out;
    }
#endif

    XMEMSET(buf, 0, sizeof(buf));
    if (wc_AesXtsSetKey(aes, k2, sizeof(k2), AES_ENCRYPTION,
            HEAP_HINT, devId) != 0)
        ERROR_OUT(-5400, out);
    else
        aes_inited = 1;

    ret = wc_AesXtsEncrypt(aes, buf, p2, sizeof(p2), i2, sizeof(i2));
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &aes->aes.asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    if (ret != 0)
        ERROR_OUT(-5401, out);
    if (XMEMCMP(c2, buf, sizeof(c2)))
        ERROR_OUT(-5402, out);

    XMEMSET(buf, 0, sizeof(buf));
    wc_AesXtsFree(aes);

    if (wc_AesXtsSetKey(aes, k1, sizeof(k1), AES_ENCRYPTION,
            HEAP_HINT, devId) != 0)
        ERROR_OUT(-5403, out);
    ret = wc_AesXtsEncrypt(aes, buf, p1, sizeof(p1), i1, sizeof(i1));
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &aes->aes.asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    if (ret != 0)
        ERROR_OUT(-5404, out);
    if (XMEMCMP(c1, buf, AES_BLOCK_SIZE))
        ERROR_OUT(-5405, out);

    /* partial block encryption test */
    XMEMSET(cipher, 0, sizeof(cipher));
    ret = wc_AesXtsEncrypt(aes, cipher, pp, sizeof(pp), i1, sizeof(i1));
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &aes->aes.asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    if (ret != 0)
        ERROR_OUT(-5406, out);
    wc_AesXtsFree(aes);

    /* partial block decrypt test */
    XMEMSET(buf, 0, sizeof(buf));
    if (wc_AesXtsSetKey(aes, k1, sizeof(k1), AES_DECRYPTION,
            HEAP_HINT, devId) != 0)
        ERROR_OUT(-5407, out);
    ret = wc_AesXtsDecrypt(aes, buf, cipher, sizeof(pp), i1, sizeof(i1));
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &aes->aes.asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    if (ret != 0)
        ERROR_OUT(-5408, out);
    if (XMEMCMP(pp, buf, sizeof(pp)))
        ERROR_OUT(-5409, out);

    /* NIST decrypt test vector */
    XMEMSET(buf, 0, sizeof(buf));
    ret = wc_AesXtsDecrypt(aes, buf, c1, sizeof(c1), i1, sizeof(i1));
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &aes->aes.asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    if (ret != 0)
        ERROR_OUT(-5410, out);
    if (XMEMCMP(p1, buf, AES_BLOCK_SIZE))
        ERROR_OUT(-5411, out);

    /* fail case with decrypting using wrong key */
    XMEMSET(buf, 0, sizeof(buf));
    ret = wc_AesXtsDecrypt(aes, buf, c2, sizeof(c2), i2, sizeof(i2));
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &aes->aes.asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    if (ret != 0)
        ERROR_OUT(-5412, out);
    if (XMEMCMP(p2, buf, sizeof(p2)) == 0) /* fail case with wrong key */
        ERROR_OUT(-5413, out);

    wc_AesXtsFree(aes);

    /* set correct key and retest */
    XMEMSET(buf, 0, sizeof(buf));
    if (wc_AesXtsSetKey(aes, k2, sizeof(k2), AES_DECRYPTION,
            HEAP_HINT, devId) != 0)
        ERROR_OUT(-5414, out);
    ret = wc_AesXtsDecrypt(aes, buf, c2, sizeof(c2), i2, sizeof(i2));
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &aes->aes.asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    if (ret != 0)
        ERROR_OUT(-5415, out);
    if (XMEMCMP(p2, buf, sizeof(p2)))
        ERROR_OUT(-5416, out);

  out:

    if (aes_inited)
        wc_AesXtsFree(aes);

#ifdef WOLFSSL_SMALL_STACK
    if (aes)
        XFREE(aes, HEAP_HINT, DYNAMIC_TYPE_AES);
#endif

    return ret;
}
#endif /* WOLFSSL_AES_128 */


#ifdef WOLFSSL_AES_256
static int aes_xts_256_test(void)
{
#ifdef WOLFSSL_SMALL_STACK
    XtsAes *aes = NULL;
#else
    XtsAes aes[1];
#endif
    int aes_inited = 0;
    int ret = 0;
    unsigned char buf[AES_BLOCK_SIZE * 3];
    unsigned char cipher[AES_BLOCK_SIZE * 3];

    /* 256 key tests */
    WOLFSSL_SMALL_STACK_STATIC unsigned char k1[] = {
        0x1e, 0xa6, 0x61, 0xc5, 0x8d, 0x94, 0x3a, 0x0e,
        0x48, 0x01, 0xe4, 0x2f, 0x4b, 0x09, 0x47, 0x14,
        0x9e, 0x7f, 0x9f, 0x8e, 0x3e, 0x68, 0xd0, 0xc7,
        0x50, 0x52, 0x10, 0xbd, 0x31, 0x1a, 0x0e, 0x7c,
        0xd6, 0xe1, 0x3f, 0xfd, 0xf2, 0x41, 0x8d, 0x8d,
        0x19, 0x11, 0xc0, 0x04, 0xcd, 0xa5, 0x8d, 0xa3,
        0xd6, 0x19, 0xb7, 0xe2, 0xb9, 0x14, 0x1e, 0x58,
        0x31, 0x8e, 0xea, 0x39, 0x2c, 0xf4, 0x1b, 0x08
    };

    WOLFSSL_SMALL_STACK_STATIC unsigned char i1[] = {
        0xad, 0xf8, 0xd9, 0x26, 0x27, 0x46, 0x4a, 0xd2,
        0xf0, 0x42, 0x8e, 0x84, 0xa9, 0xf8, 0x75, 0x64
    };

    WOLFSSL_SMALL_STACK_STATIC unsigned char p1[] = {
        0x2e, 0xed, 0xea, 0x52, 0xcd, 0x82, 0x15, 0xe1,
        0xac, 0xc6, 0x47, 0xe8, 0x10, 0xbb, 0xc3, 0x64,
        0x2e, 0x87, 0x28, 0x7f, 0x8d, 0x2e, 0x57, 0xe3,
        0x6c, 0x0a, 0x24, 0xfb, 0xc1, 0x2a, 0x20, 0x2e
    };

    /* plain text test of partial block is not from NIST test vector list */
    WOLFSSL_SMALL_STACK_STATIC unsigned char pp[] = {
        0xeb, 0xab, 0xce, 0x95, 0xb1, 0x4d, 0x3c, 0x8d,
        0x6f, 0xb3, 0x50, 0x39, 0x07, 0x90, 0x31, 0x1c,
        0x6e, 0x4b, 0x92, 0x01, 0x3e, 0x76, 0x8a, 0xd5
    };

    WOLFSSL_SMALL_STACK_STATIC unsigned char c1[] = {
        0xcb, 0xaa, 0xd0, 0xe2, 0xf6, 0xce, 0xa3, 0xf5,
        0x0b, 0x37, 0xf9, 0x34, 0xd4, 0x6a, 0x9b, 0x13,
        0x0b, 0x9d, 0x54, 0xf0, 0x7e, 0x34, 0xf3, 0x6a,
        0xf7, 0x93, 0xe8, 0x6f, 0x73, 0xc6, 0xd7, 0xdb
    };

    WOLFSSL_SMALL_STACK_STATIC unsigned char k2[] = {
        0xad, 0x50, 0x4b, 0x85, 0xd7, 0x51, 0xbf, 0xba,
        0x69, 0x13, 0xb4, 0xcc, 0x79, 0xb6, 0x5a, 0x62,
        0xf7, 0xf3, 0x9d, 0x36, 0x0f, 0x35, 0xb5, 0xec,
        0x4a, 0x7e, 0x95, 0xbd, 0x9b, 0xa5, 0xf2, 0xec,
        0xc1, 0xd7, 0x7e, 0xa3, 0xc3, 0x74, 0xbd, 0x4b,
        0x13, 0x1b, 0x07, 0x83, 0x87, 0xdd, 0x55, 0x5a,
        0xb5, 0xb0, 0xc7, 0xe5, 0x2d, 0xb5, 0x06, 0x12,
        0xd2, 0xb5, 0x3a, 0xcb, 0x47, 0x8a, 0x53, 0xb4
    };

    WOLFSSL_SMALL_STACK_STATIC unsigned char i2[] = {
        0xe6, 0x42, 0x19, 0xed, 0xe0, 0xe1, 0xc2, 0xa0,
        0x0e, 0xf5, 0x58, 0x6a, 0xc4, 0x9b, 0xeb, 0x6f
    };

    WOLFSSL_SMALL_STACK_STATIC unsigned char p2[] = {
        0x24, 0xcb, 0x76, 0x22, 0x55, 0xb5, 0xa8, 0x00,
        0xf4, 0x6e, 0x80, 0x60, 0x56, 0x9e, 0x05, 0x53,
        0xbc, 0xfe, 0x86, 0x55, 0x3b, 0xca, 0xd5, 0x89,
        0xc7, 0x54, 0x1a, 0x73, 0xac, 0xc3, 0x9a, 0xbd,
        0x53, 0xc4, 0x07, 0x76, 0xd8, 0xe8, 0x22, 0x61,
        0x9e, 0xa9, 0xad, 0x77, 0xa0, 0x13, 0x4c, 0xfc
    };

    WOLFSSL_SMALL_STACK_STATIC unsigned char c2[] = {
        0xa3, 0xc6, 0xf3, 0xf3, 0x82, 0x79, 0x5b, 0x10,
        0x87, 0xd7, 0x02, 0x50, 0xdb, 0x2c, 0xd3, 0xb1,
        0xa1, 0x62, 0xa8, 0xb6, 0xdc, 0x12, 0x60, 0x61,
        0xc1, 0x0a, 0x84, 0xa5, 0x85, 0x3f, 0x3a, 0x89,
        0xe6, 0x6c, 0xdb, 0xb7, 0x9a, 0xb4, 0x28, 0x9b,
        0xc3, 0xea, 0xd8, 0x10, 0xe9, 0xc0, 0xaf, 0x92
    };

#ifdef WOLFSSL_SMALL_STACK
    if ((aes = (XtsAes *)XMALLOC(sizeof *aes, HEAP_HINT, DYNAMIC_TYPE_AES)) == NULL)
        ERROR_OUT(-5515, out);
#endif

#if defined(OPENSSL_EXTRA) && !defined(WOLFCRYPT_ONLY) \
    && !defined(HAVE_SELFTEST) && !defined(HAVE_FIPS)
    ret = EVP_test(EVP_aes_256_xts(), k2, i2, p2, sizeof(p2), c2, sizeof(c2));
    if (ret != 0) {
        printf("EVP_aes_256_xts failed\n");
        goto out;
    }
#endif

    XMEMSET(buf, 0, sizeof(buf));
    if (wc_AesXtsSetKey(aes, k2, sizeof(k2), AES_ENCRYPTION,
            HEAP_HINT, devId) != 0)
        ERROR_OUT(-5500, out);
    else
        aes_inited = 1;
    ret = wc_AesXtsEncrypt(aes, buf, p2, sizeof(p2), i2, sizeof(i2));
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &aes->aes.asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    if (ret != 0)
        ERROR_OUT(-5501, out);
    if (XMEMCMP(c2, buf, sizeof(c2)))
        ERROR_OUT(-5502, out);
    wc_AesXtsFree(aes);

    XMEMSET(buf, 0, sizeof(buf));
    if (wc_AesXtsSetKey(aes, k1, sizeof(k1), AES_ENCRYPTION,
            HEAP_HINT, devId) != 0)
        ERROR_OUT(-5503, out);
    ret = wc_AesXtsEncrypt(aes, buf, p1, sizeof(p1), i1, sizeof(i1));
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &aes->aes.asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    if (ret != 0)
        ERROR_OUT(-5504, out);
    if (XMEMCMP(c1, buf, AES_BLOCK_SIZE))
        ERROR_OUT(-5505, out);

    /* partial block encryption test */
    XMEMSET(cipher, 0, sizeof(cipher));
    ret = wc_AesXtsEncrypt(aes, cipher, pp, sizeof(pp), i1, sizeof(i1));
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &aes->aes.asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    if (ret != 0)
        ERROR_OUT(-5506, out);
    wc_AesXtsFree(aes);

    /* partial block decrypt test */
    XMEMSET(buf, 0, sizeof(buf));
    if (wc_AesXtsSetKey(aes, k1, sizeof(k1), AES_DECRYPTION,
            HEAP_HINT, devId) != 0)
        ERROR_OUT(-5507, out);
    ret = wc_AesXtsDecrypt(aes, buf, cipher, sizeof(pp), i1, sizeof(i1));
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &aes->aes.asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    if (ret != 0)
        ERROR_OUT(-5508, out);
    if (XMEMCMP(pp, buf, sizeof(pp)))
        ERROR_OUT(-5509, out);

    /* NIST decrypt test vector */
    XMEMSET(buf, 0, sizeof(buf));
    ret = wc_AesXtsDecrypt(aes, buf, c1, sizeof(c1), i1, sizeof(i1));
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &aes->aes.asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    if (ret != 0)
        ERROR_OUT(-5510, out);
    if (XMEMCMP(p1, buf, AES_BLOCK_SIZE))
        ERROR_OUT(-5511, out);
    wc_AesXtsFree(aes);

    XMEMSET(buf, 0, sizeof(buf));
    if (wc_AesXtsSetKey(aes, k2, sizeof(k2), AES_DECRYPTION,
            HEAP_HINT, devId) != 0)
        ERROR_OUT(-5512, out);
    ret = wc_AesXtsDecrypt(aes, buf, c2, sizeof(c2), i2, sizeof(i2));
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &aes->aes.asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    if (ret != 0)
        ERROR_OUT(-5513, out);
    if (XMEMCMP(p2, buf, sizeof(p2)))
        ERROR_OUT(-5514, out);

  out:

    if (aes_inited)
        wc_AesXtsFree(aes);

#ifdef WOLFSSL_SMALL_STACK
    if (aes)
        XFREE(aes, HEAP_HINT, DYNAMIC_TYPE_AES);
#endif

    return ret;
}
#endif /* WOLFSSL_AES_256 */


#if defined(WOLFSSL_AES_128) && defined(WOLFSSL_AES_256)
/* both 128 and 256 bit key test */
static int aes_xts_sector_test(void)
{
#ifdef WOLFSSL_SMALL_STACK
    XtsAes *aes = NULL;
#else
    XtsAes aes[1];
#endif
    int aes_inited = 0;
    int ret = 0;
    unsigned char buf[AES_BLOCK_SIZE * 2];

    /* 128 key tests */
    WOLFSSL_SMALL_STACK_STATIC unsigned char k1[] = {
        0xa3, 0xe4, 0x0d, 0x5b, 0xd4, 0xb6, 0xbb, 0xed,
        0xb2, 0xd1, 0x8c, 0x70, 0x0a, 0xd2, 0xdb, 0x22,
        0x10, 0xc8, 0x11, 0x90, 0x64, 0x6d, 0x67, 0x3c,
        0xbc, 0xa5, 0x3f, 0x13, 0x3e, 0xab, 0x37, 0x3c
    };

    WOLFSSL_SMALL_STACK_STATIC unsigned char p1[] = {
        0x20, 0xe0, 0x71, 0x94, 0x05, 0x99, 0x3f, 0x09,
        0xa6, 0x6a, 0xe5, 0xbb, 0x50, 0x0e, 0x56, 0x2c
    };

    WOLFSSL_SMALL_STACK_STATIC unsigned char c1[] = {
        0x74, 0x62, 0x35, 0x51, 0x21, 0x02, 0x16, 0xac,
        0x92, 0x6b, 0x96, 0x50, 0xb6, 0xd3, 0xfa, 0x52
    };
    word64 s1 = 141;

    /* 256 key tests */
    WOLFSSL_SMALL_STACK_STATIC unsigned char k2[] = {
        0xef, 0x01, 0x0c, 0xa1, 0xa3, 0x66, 0x3e, 0x32,
        0x53, 0x43, 0x49, 0xbc, 0x0b, 0xae, 0x62, 0x23,
        0x2a, 0x15, 0x73, 0x34, 0x85, 0x68, 0xfb, 0x9e,
        0xf4, 0x17, 0x68, 0xa7, 0x67, 0x4f, 0x50, 0x7a,
        0x72, 0x7f, 0x98, 0x75, 0x53, 0x97, 0xd0, 0xe0,
        0xaa, 0x32, 0xf8, 0x30, 0x33, 0x8c, 0xc7, 0xa9,
        0x26, 0xc7, 0x73, 0xf0, 0x9e, 0x57, 0xb3, 0x57,
        0xcd, 0x15, 0x6a, 0xfb, 0xca, 0x46, 0xe1, 0xa0
    };

    WOLFSSL_SMALL_STACK_STATIC unsigned char p2[] = {
        0xed, 0x98, 0xe0, 0x17, 0x70, 0xa8, 0x53, 0xb4,
        0x9d, 0xb9, 0xe6, 0xaa, 0xf8, 0x8f, 0x0a, 0x41,
        0xb9, 0xb5, 0x6e, 0x91, 0xa5, 0xa2, 0xb1, 0x1d,
        0x40, 0x52, 0x92, 0x54, 0xf5, 0x52, 0x3e, 0x75
    };

    WOLFSSL_SMALL_STACK_STATIC unsigned char c2[] = {
        0xca, 0x20, 0xc5, 0x5e, 0x8d, 0xc1, 0x49, 0x68,
        0x7d, 0x25, 0x41, 0xde, 0x39, 0xc3, 0xdf, 0x63,
        0x00, 0xbb, 0x5a, 0x16, 0x3c, 0x10, 0xce, 0xd3,
        0x66, 0x6b, 0x13, 0x57, 0xdb, 0x8b, 0xd3, 0x9d
    };
    word64 s2 = 187;

#ifdef WOLFSSL_SMALL_STACK
    if ((aes = (XtsAes *)XMALLOC(sizeof *aes, HEAP_HINT, DYNAMIC_TYPE_AES)) == NULL)
        ERROR_OUT(-5612, out);
#endif

    XMEMSET(buf, 0, sizeof(buf));
    if (wc_AesXtsSetKey(aes, k1, sizeof(k1), AES_ENCRYPTION,
            HEAP_HINT, devId) != 0)
        ERROR_OUT(-5600, out);
    else
        aes_inited = 1;
    ret = wc_AesXtsEncryptSector(aes, buf, p1, sizeof(p1), s1);
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &aes->aes.asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    if (ret != 0)
        ERROR_OUT(-5601, out);
    if (XMEMCMP(c1, buf, AES_BLOCK_SIZE))
        ERROR_OUT(-5602, out);
    wc_AesXtsFree(aes);

    /* decrypt test */
    XMEMSET(buf, 0, sizeof(buf));
    if (wc_AesXtsSetKey(aes, k1, sizeof(k1), AES_DECRYPTION,
            HEAP_HINT, devId) != 0)
        ERROR_OUT(-5603, out);
    ret = wc_AesXtsDecryptSector(aes, buf, c1, sizeof(c1), s1);
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &aes->aes.asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    if (ret != 0)
        ERROR_OUT(-5604, out);
    if (XMEMCMP(p1, buf, AES_BLOCK_SIZE))
        ERROR_OUT(-5605, out);
    wc_AesXtsFree(aes);

    /* 256 bit key tests */
    XMEMSET(buf, 0, sizeof(buf));
    if (wc_AesXtsSetKey(aes, k2, sizeof(k2), AES_ENCRYPTION,
            HEAP_HINT, devId) != 0)
        ERROR_OUT(-5606, out);
    ret = wc_AesXtsEncryptSector(aes, buf, p2, sizeof(p2), s2);
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &aes->aes.asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    if (ret != 0)
        ERROR_OUT(-5607, out);
    if (XMEMCMP(c2, buf, sizeof(c2)))
        ERROR_OUT(-5608, out);
    wc_AesXtsFree(aes);

    /* decrypt test */
    XMEMSET(buf, 0, sizeof(buf));
    if (wc_AesXtsSetKey(aes, k2, sizeof(k2), AES_DECRYPTION,
            HEAP_HINT, devId) != 0)
        ERROR_OUT(-5609, out);
    ret = wc_AesXtsDecryptSector(aes, buf, c2, sizeof(c2), s2);
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &aes->aes.asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    if (ret != 0)
        ERROR_OUT(-5610, out);
    if (XMEMCMP(p2, buf, sizeof(p2)))
        ERROR_OUT(-5611, out);

  out:

    if (aes_inited)
        wc_AesXtsFree(aes);

#ifdef WOLFSSL_SMALL_STACK
    if (aes)
        XFREE(aes, HEAP_HINT, DYNAMIC_TYPE_AES);
#endif

    return ret;
}
#endif /* WOLFSSL_AES_128 && WOLFSSL_AES_256 */


#ifdef WOLFSSL_AES_128
/* testing of bad arguments */
static int aes_xts_args_test(void)
{
#ifdef WOLFSSL_SMALL_STACK
    XtsAes *aes = NULL;
#else
    XtsAes aes[1];
#endif
    int aes_inited = 0;
    int ret;
    unsigned char buf[AES_BLOCK_SIZE * 2];

    /* 128 key tests */
    WOLFSSL_SMALL_STACK_STATIC unsigned char k1[] = {
        0xa3, 0xe4, 0x0d, 0x5b, 0xd4, 0xb6, 0xbb, 0xed,
        0xb2, 0xd1, 0x8c, 0x70, 0x0a, 0xd2, 0xdb, 0x22,
        0x10, 0xc8, 0x11, 0x90, 0x64, 0x6d, 0x67, 0x3c,
        0xbc, 0xa5, 0x3f, 0x13, 0x3e, 0xab, 0x37, 0x3c
    };

    WOLFSSL_SMALL_STACK_STATIC unsigned char p1[] = {
        0x20, 0xe0, 0x71, 0x94, 0x05, 0x99, 0x3f, 0x09,
        0xa6, 0x6a, 0xe5, 0xbb, 0x50, 0x0e, 0x56, 0x2c
    };

    WOLFSSL_SMALL_STACK_STATIC unsigned char c1[] = {
        0x74, 0x62, 0x35, 0x51, 0x21, 0x02, 0x16, 0xac,
        0x92, 0x6b, 0x96, 0x50, 0xb6, 0xd3, 0xfa, 0x52
    };
    word64 s1 = 141;

#ifdef WOLFSSL_SMALL_STACK
    if ((aes = (XtsAes *)XMALLOC(sizeof *aes, HEAP_HINT, DYNAMIC_TYPE_AES)) == NULL)
        ERROR_OUT(-5708, out);
#endif

    if (wc_AesXtsSetKey(NULL, k1, sizeof(k1), AES_ENCRYPTION,
            HEAP_HINT, devId) == 0)
        ERROR_OUT(-5700, out);
    if (wc_AesXtsSetKey(aes, NULL, sizeof(k1), AES_ENCRYPTION,
            HEAP_HINT, devId) == 0)
        ERROR_OUT(-5701, out);

    /* encryption operations */
    if (wc_AesXtsSetKey(aes, k1, sizeof(k1), AES_ENCRYPTION,
            HEAP_HINT, devId) != 0)
        ERROR_OUT(-5702, out);
    else
        aes_inited = 1;
    ret = wc_AesXtsEncryptSector(NULL, buf, p1, sizeof(p1), s1);
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &aes->aes.asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    if (ret == 0)
        ERROR_OUT(-5703, out);

    ret = wc_AesXtsEncryptSector(aes, NULL, p1, sizeof(p1), s1);
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &aes->aes.asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    if (ret == 0)
        ERROR_OUT(-5704, out);
    wc_AesXtsFree(aes);

    /* decryption operations */
    if (wc_AesXtsSetKey(aes, k1, sizeof(k1), AES_DECRYPTION,
            HEAP_HINT, devId) != 0)
        ERROR_OUT(-5705, out);
    ret = wc_AesXtsDecryptSector(NULL, buf, c1, sizeof(c1), s1);
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &aes->aes.asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    if (ret == 0)
        ERROR_OUT(-5706, out);

    ret = wc_AesXtsDecryptSector(aes, NULL, c1, sizeof(c1), s1);
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &aes->aes.asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    if (ret == 0)
        ERROR_OUT(-5707, out);

    ret = 0;

  out:

    if (aes_inited)
        wc_AesXtsFree(aes);

#ifdef WOLFSSL_SMALL_STACK
    if (aes)
        XFREE(aes, HEAP_HINT, DYNAMIC_TYPE_AES);
#endif

    return ret;
}
#endif /* WOLFSSL_AES_128 */
#endif /* WOLFSSL_AES_XTS */

#if defined(HAVE_AES_CBC) && defined(WOLFSSL_AES_128)
static int aes_cbc_test(void)
{
    byte cipher[AES_BLOCK_SIZE];
    byte plain[AES_BLOCK_SIZE];
    int  ret;
    WOLFSSL_SMALL_STACK_STATIC const byte msg[] = { /* "Now is the time for all " w/o trailing 0 */
        0x6e,0x6f,0x77,0x20,0x69,0x73,0x20,0x74,
        0x68,0x65,0x20,0x74,0x69,0x6d,0x65,0x20,
        0x66,0x6f,0x72,0x20,0x61,0x6c,0x6c,0x20
    };
    byte key[] = "0123456789abcdef   ";  /* align */
    byte iv[]  = "1234567890abcdef   ";  /* align */

    XMEMSET(cipher, 0, AES_BLOCK_SIZE);
    XMEMSET(plain, 0, AES_BLOCK_SIZE);

    /* Parameter Validation testing. */
    ret = wc_AesCbcEncryptWithKey(cipher, msg, AES_BLOCK_SIZE, key, 17, NULL);
    if (ret != BAD_FUNC_ARG)
        return -5800;
#ifdef HAVE_AES_DECRYPT
    ret = wc_AesCbcDecryptWithKey(plain, cipher, AES_BLOCK_SIZE, key, 17, NULL);
    if (ret != BAD_FUNC_ARG)
        return -5801;
#endif

    ret = wc_AesCbcEncryptWithKey(cipher, msg, AES_BLOCK_SIZE, key,
                                  AES_BLOCK_SIZE, iv);
    if (ret != 0)
        return -5802;
#ifdef HAVE_AES_DECRYPT
    ret = wc_AesCbcDecryptWithKey(plain, cipher, AES_BLOCK_SIZE, key,
                                  AES_BLOCK_SIZE, iv);
    if (ret != 0)
        return -5803;
    if (XMEMCMP(plain, msg, AES_BLOCK_SIZE) != 0)
        return -5804;
#endif /* HAVE_AES_DECRYPT */

    (void)plain;
    return 0;
}
#endif

#if defined(HAVE_AES_ECB) && !defined(HAVE_FIPS) && !defined(HAVE_SELFTEST)
static int aesecb_test(void)
{
#ifdef WOLFSSL_SMALL_STACK
    Aes *enc = (Aes *)XMALLOC(sizeof *enc, HEAP_HINT, DYNAMIC_TYPE_AES);
#else
    Aes enc[1];
#endif
    byte cipher[AES_BLOCK_SIZE * 4];
#ifdef HAVE_AES_DECRYPT
#ifdef WOLFSSL_SMALL_STACK
    Aes *dec = (Aes *)XMALLOC(sizeof *dec, HEAP_HINT, DYNAMIC_TYPE_AES);
#else
    Aes dec[1];
#endif
    byte plain [AES_BLOCK_SIZE * 4];
#endif /* HAVE_AES_DECRYPT */
    int  ret = 0;

#if defined(WOLFSSL_AES_256)
    {
        WOLFSSL_SMALL_STACK_STATIC const byte niPlain[] =
        {
            0x6b,0xc1,0xbe,0xe2,0x2e,0x40,0x9f,0x96,
            0xe9,0x3d,0x7e,0x11,0x73,0x93,0x17,0x2a
        };

        WOLFSSL_SMALL_STACK_STATIC const byte niCipher[] =
        {
            0xf3,0xee,0xd1,0xbd,0xb5,0xd2,0xa0,0x3c,
            0x06,0x4b,0x5a,0x7e,0x3d,0xb1,0x81,0xf8
        };

        WOLFSSL_SMALL_STACK_STATIC const byte niKey[] =
        {
            0x60,0x3d,0xeb,0x10,0x15,0xca,0x71,0xbe,
            0x2b,0x73,0xae,0xf0,0x85,0x7d,0x77,0x81,
            0x1f,0x35,0x2c,0x07,0x3b,0x61,0x08,0xd7,
            0x2d,0x98,0x10,0xa3,0x09,0x14,0xdf,0xf4
        };

        if (wc_AesInit(enc, HEAP_HINT, devId) != 0)
            ERROR_OUT(-5900, out);
    #if defined(HAVE_AES_DECRYPT)
        if (wc_AesInit(dec, HEAP_HINT, devId) != 0)
            ERROR_OUT(-5901, out);
    #endif

        XMEMSET(cipher, 0, AES_BLOCK_SIZE);
        ret = wc_AesSetKey(enc, niKey, sizeof(niKey), cipher, AES_ENCRYPTION);
        if (ret != 0)
            ERROR_OUT(-5923, out);
        if (wc_AesEcbEncrypt(enc, cipher, niPlain, AES_BLOCK_SIZE) != 0)
            ERROR_OUT(-5924, out);
        if (XMEMCMP(cipher, niCipher, AES_BLOCK_SIZE) != 0)
            ERROR_OUT(-5925, out);

        XMEMSET(plain, 0, AES_BLOCK_SIZE);
        ret = wc_AesSetKey(dec, niKey, sizeof(niKey), plain, AES_DECRYPTION);
        if (ret != 0)
            ERROR_OUT(-5926, out);
        if (wc_AesEcbDecrypt(dec, plain, niCipher, AES_BLOCK_SIZE) != 0)
            ERROR_OUT(-5927, out);
        wc_AesEcbDecrypt(dec, plain, niCipher, AES_BLOCK_SIZE);
        if (XMEMCMP(plain, niPlain, AES_BLOCK_SIZE) != 0)
            ERROR_OUT(-5928, out);
    }

    wc_AesFree(enc);
#ifdef HAVE_AES_DECRYPT
    wc_AesFree(dec);
#endif

  out:
#ifdef WOLFSSL_SMALL_STACK
    if (enc)
        XFREE(enc, HEAP_HINT, DYNAMIC_TYPE_AES);
#ifdef HAVE_AES_DECRYPT
    if (dec)
        XFREE(dec, HEAP_HINT, DYNAMIC_TYPE_AES);
#endif
#endif
#endif /* WOLFSSL_AES_256 */

    return ret;
}
#endif /* HAVE_AES_ECB */

#ifdef WOLFSSL_AES_COUNTER
static int aesctr_test(Aes* enc, Aes* dec, byte* cipher, byte* plain)
{
    int ret;

    /* test vectors from "Recommendation for Block Cipher Modes of
     * Operation" NIST Special Publication 800-38A */

    WOLFSSL_SMALL_STACK_STATIC const byte ctrIv[] =
    {
        0xf0,0xf1,0xf2,0xf3,0xf4,0xf5,0xf6,0xf7,
        0xf8,0xf9,0xfa,0xfb,0xfc,0xfd,0xfe,0xff
    };

    WOLFSSL_SMALL_STACK_STATIC const byte ctrPlain[] =
    {
        0x6b,0xc1,0xbe,0xe2,0x2e,0x40,0x9f,0x96,
        0xe9,0x3d,0x7e,0x11,0x73,0x93,0x17,0x2a,
        0xae,0x2d,0x8a,0x57,0x1e,0x03,0xac,0x9c,
        0x9e,0xb7,0x6f,0xac,0x45,0xaf,0x8e,0x51,
        0x30,0xc8,0x1c,0x46,0xa3,0x5c,0xe4,0x11,
        0xe5,0xfb,0xc1,0x19,0x1a,0x0a,0x52,0xef,
        0xf6,0x9f,0x24,0x45,0xdf,0x4f,0x9b,0x17,
        0xad,0x2b,0x41,0x7b,0xe6,0x6c,0x37,0x10
    };

#ifdef WOLFSSL_ARMASM
    WOLFSSL_SMALL_STACK_STATIC const byte ctrIvWrap32[] =
    {
        0xff,0xff,0xff,0xff,0x0f,0xff,0xff,0xff,
        0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff
    };
    WOLFSSL_SMALL_STACK_STATIC const byte ctrIvWrap32_2[] =
    {
        0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
        0xff,0xff,0xff,0xff,0x0f,0xff,0xff,0xfe
    };
    WOLFSSL_SMALL_STACK_STATIC const byte ctrIvWrap64[] =
    {
        0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
        0x0f,0xff,0xff,0xff,0xff,0xff,0xff,0xff
    };
    WOLFSSL_SMALL_STACK_STATIC const byte ctrIvWrap64_2[] =
    {
        0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xf0,
        0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xfe
    };
    WOLFSSL_SMALL_STACK_STATIC const byte ctrIvWrap96[] =
    {
        0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
        0xff,0xff,0xff,0xff,0x0f,0xff,0xff,0xff
    };
    WOLFSSL_SMALL_STACK_STATIC const byte ctrIvWrap96_2[] =
    {
        0xff,0xff,0xff,0xf0,0xff,0xff,0xff,0xff,
        0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xfe
    };
#endif
    WOLFSSL_SMALL_STACK_STATIC const byte ctrIvWrap128[] =
    {
        0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
        0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff
    };
#ifdef WOLFSSL_ARMASM
    WOLFSSL_SMALL_STACK_STATIC const byte ctrIvWrap128_2[] =
    {
        0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
        0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xfe
    };
#endif

#ifdef WOLFSSL_AES_128
    WOLFSSL_SMALL_STACK_STATIC const byte oddCipher[] =
    {
        0xb9,0xd7,0xcb,0x08,0xb0,0xe1,0x7b,0xa0,
        0xc2
    };

    WOLFSSL_SMALL_STACK_STATIC const byte ctr128Key[] =
    {
        0x2b,0x7e,0x15,0x16,0x28,0xae,0xd2,0xa6,
        0xab,0xf7,0x15,0x88,0x09,0xcf,0x4f,0x3c
    };

    WOLFSSL_SMALL_STACK_STATIC const byte ctr128Cipher[] =
    {
        0x87,0x4d,0x61,0x91,0xb6,0x20,0xe3,0x26,
        0x1b,0xef,0x68,0x64,0x99,0x0d,0xb6,0xce,
        0x98,0x06,0xf6,0x6b,0x79,0x70,0xfd,0xff,
        0x86,0x17,0x18,0x7b,0xb9,0xff,0xfd,0xff,
        0x5a,0xe4,0xdf,0x3e,0xdb,0xd5,0xd3,0x5e,
        0x5b,0x4f,0x09,0x02,0x0d,0xb0,0x3e,0xab,
        0x1e,0x03,0x1d,0xda,0x2f,0xbe,0x03,0xd1,
        0x79,0x21,0x70,0xa0,0xf3,0x00,0x9c,0xee
    };

#ifdef WOLFSSL_ARMASM
    WOLFSSL_SMALL_STACK_STATIC const byte ctr128Wrap32Cipher[] =
    {
        0xb3,0x8b,0x58,0xbc,0xce,0xf4,0x71,0x78,
        0xf6,0x7c,0xdb,0xb4,0x27,0x2b,0x0a,0xbf,
        0x7e,0xad,0xea,0x5c,0xd1
    };
    WOLFSSL_SMALL_STACK_STATIC const byte ctr128Wrap32CipherLong[] =
    {
        0xb3,0x8b,0x58,0xbc,0xce,0xf4,0x71,0x78,
        0xf6,0x7c,0xdb,0xb4,0x27,0x2b,0x0a,0xbf,
        0x7e,0xad,0xea,0x5c,0xd1,0xb7,0x98,0xf0,
        0x22,0x20,0xfe,0x67,0xb0,0x02,0x23,0x50
    };
    WOLFSSL_SMALL_STACK_STATIC const byte ctr128Wrap32_2CipherLong[] =
    {
        0x6e,0xa1,0x27,0x4d,0xea,0x20,0x5f,0x39,
        0x68,0xc8,0xb6,0x78,0xde,0xfc,0x53,0x5c,
        0x90,0xc8,0xf6,0xc6,0xfa,0xe0,0x7b,0x09,
        0x7c,0xf8,0x9c,0x6a,0x5a,0xa5,0x17,0x7f,
        0x03,0x92,0x5f,0x4e,0x85,0xea,0x26,0xc9,
        0x5a,0xc2,0x74,0xe2,0xbf,0xe4,0x1b,0xd4
    };

    WOLFSSL_SMALL_STACK_STATIC const byte ctr128Wrap64Cipher[] =
    {
        0xdd,0x17,0x10,0x7c,0x45,0x04,0xac,0x43,
        0xef,0xa8,0xcc,0x32,0x34,0x87,0x88,0xd7,
        0xae,0x74,0x94,0x72,0x8e
    };
    WOLFSSL_SMALL_STACK_STATIC const byte ctr128Wrap64CipherLong[] =
    {
        0xdd,0x17,0x10,0x7c,0x45,0x04,0xac,0x43,
        0xef,0xa8,0xcc,0x32,0x34,0x87,0x88,0xd7,
        0xae,0x74,0x94,0x72,0x8e,0xd0,0x71,0xc0,
        0x89,0x8a,0xa1,0xb0,0x29,0xa0,0x10,0x9e
    };
    WOLFSSL_SMALL_STACK_STATIC const byte ctr128Wrap64_2CipherLong[] =
    {
        0x3f,0xe7,0xd5,0xf3,0xfa,0x09,0xfe,0x40,
        0xa6,0xa1,0x32,0x8b,0x57,0x12,0xb9,0xfa,
        0xf2,0x2d,0xe4,0x3c,0x66,0x1d,0x0a,0x8e,
        0x46,0xf8,0x2e,0x33,0xce,0x8d,0x4e,0x3b,
        0x17,0x67,0x9e,0x9f,0x76,0x9e,0xc2,0x99,
        0xd5,0xd4,0x71,0xed,0xb4,0x33,0xb2,0xcd
    };

    WOLFSSL_SMALL_STACK_STATIC const byte ctr128Wrap96Cipher[] =
    {
        0x55,0x24,0xc2,0x73,0xca,0xa3,0x48,0x03,
        0x0b,0x72,0x8d,0xd7,0x6c,0x99,0x8e,0x04,
        0x9d,0x77,0xc9,0x5f,0x38
    };
    WOLFSSL_SMALL_STACK_STATIC const byte ctr128Wrap96CipherLong[] =
    {
        0x55,0x24,0xc2,0x73,0xca,0xa3,0x48,0x03,
        0x0b,0x72,0x8d,0xd7,0x6c,0x99,0x8e,0x04,
        0x9d,0x77,0xc9,0x5f,0x38,0xb5,0x6e,0x44,
        0x21,0x8e,0xda,0x57,0xe0,0x41,0xc7,0x6a
    };
    WOLFSSL_SMALL_STACK_STATIC const byte ctr128Wrap96_2CipherLong[] =
    {
        0xc8,0x81,0x1a,0xbe,0xc7,0x5b,0x93,0x6f,
        0xe6,0x52,0xe4,0xb1,0x2d,0x1c,0x39,0xbc,
        0xeb,0x82,0x27,0x0a,0x7e,0xa5,0x0e,0x2d,
        0x32,0xda,0xbe,0x10,0x7a,0x10,0xcc,0xd3,
        0x6f,0xc6,0x83,0x28,0x05,0x57,0x8a,0x24,
        0x44,0x76,0x17,0x81,0xb9,0x5c,0x94,0x81
    };
#endif

    WOLFSSL_SMALL_STACK_STATIC const byte ctr128Wrap128Cipher[] =
    {
        0xe1,0x33,0x38,0xe3,0x6c,0xb7,0x19,0x62,
        0xe0,0x0d,0x02,0x0b,0x4c,0xed,0xbd,0x86,
        0xd3,0xda,0xe1,0x5b,0x04
    };
    WOLFSSL_SMALL_STACK_STATIC const byte ctr128Wrap128CipherLong[] =
    {
        0xe1,0x33,0x38,0xe3,0x6c,0xb7,0x19,0x62,
        0xe0,0x0d,0x02,0x0b,0x4c,0xed,0xbd,0x86,
        0xd3,0xda,0xe1,0x5b,0x04,0xbb,0x35,0x2f,
        0xa0,0xf5,0x9f,0xeb,0xfc,0xb4,0xda,0x3e
    };
#ifdef WOLFSSL_ARMASM
    WOLFSSL_SMALL_STACK_STATIC const byte ctr128Wrap128_2CipherLong[] =
    {
        0xba,0x76,0xaa,0x54,0xd5,0xb5,0x60,0x67,
        0xc1,0xa7,0x90,0x3b,0x3f,0xdd,0xfa,0x89,
        0x24,0xdf,0x0c,0x56,0x5c,0xf4,0x2a,0x68,
        0x97,0x87,0x13,0xb6,0x7a,0xd1,0x24,0xfd,
        0x4d,0x3f,0x77,0x4a,0xb9,0xe4,0x7d,0xa2,
        0xdb,0xb9,0x31,0x5e,0xa3,0x11,0x06,0x80
    };
#endif
#endif /* WOLFSSL_AES_128 */

#ifdef WOLFSSL_AES_192
    WOLFSSL_SMALL_STACK_STATIC const byte ctr192Key[] =
    {
        0x8e,0x73,0xb0,0xf7,0xda,0x0e,0x64,0x52,
        0xc8,0x10,0xf3,0x2b,0x80,0x90,0x79,0xe5,
        0x62,0xf8,0xea,0xd2,0x52,0x2c,0x6b,0x7b
    };

    WOLFSSL_SMALL_STACK_STATIC const byte ctr192Cipher[] =
    {
        0x1a,0xbc,0x93,0x24,0x17,0x52,0x1c,0xa2,
        0x4f,0x2b,0x04,0x59,0xfe,0x7e,0x6e,0x0b,
        0x09,0x03,0x39,0xec,0x0a,0xa6,0xfa,0xef,
        0xd5,0xcc,0xc2,0xc6,0xf4,0xce,0x8e,0x94,
        0x1e,0x36,0xb2,0x6b,0xd1,0xeb,0xc6,0x70,
        0xd1,0xbd,0x1d,0x66,0x56,0x20,0xab,0xf7,
        0x4f,0x78,0xa7,0xf6,0xd2,0x98,0x09,0x58,
        0x5a,0x97,0xda,0xec,0x58,0xc6,0xb0,0x50
    };

#ifdef WOLFSSL_ARMASM
    WOLFSSL_SMALL_STACK_STATIC const byte ctr192Wrap32Cipher[] =
    {
        0x28,0xaa,0xfa,0x90,0x72,0x74,0x86,0xaf,
        0x72,0x73,0x35,0x17,0x70,0x4e,0x7d,0xca,
        0x0c,0x33,0x97,0x06,0xc0
    };
    WOLFSSL_SMALL_STACK_STATIC const byte ctr192Wrap32CipherLong[] =
    {
        0x28,0xaa,0xfa,0x90,0x72,0x74,0x86,0xaf,
        0x72,0x73,0x35,0x17,0x70,0x4e,0x7d,0xca,
        0x0c,0x33,0x97,0x06,0xc0,0xbe,0x83,0x87,
        0xdd,0xd3,0xff,0xd8,0xe4,0x6a,0x5b,0x84
    };
    WOLFSSL_SMALL_STACK_STATIC const byte ctr192Wrap32_2CipherLong[] =
    {
        0xf5,0x00,0xa2,0x91,0x54,0xa3,0x76,0xa2,
        0xdd,0xad,0x16,0x89,0xe5,0xf0,0x1d,0x40,
        0x84,0xcd,0x74,0x84,0xcb,0x8b,0x9e,0x29,
        0xa9,0x8a,0x12,0x65,0xa0,0x79,0x5e,0xce,
        0xd9,0x50,0x65,0x21,0x86,0xb0,0x85,0x0d,
        0x98,0x2d,0x9a,0x5a,0x11,0xbe,0xa5,0x7f
    };

    WOLFSSL_SMALL_STACK_STATIC const byte ctr192Wrap64Cipher[] =
    {
        0xfe,0x39,0x27,0x97,0xac,0xe5,0xb8,0x74,
        0xb9,0x8c,0xbf,0x58,0x71,0xa4,0x80,0x33,
        0x3d,0xf7,0xb4,0xfd,0x8c
    };
    WOLFSSL_SMALL_STACK_STATIC const byte ctr192Wrap64CipherLong[] =
    {
        0xfe,0x39,0x27,0x97,0xac,0xe5,0xb8,0x74,
        0xb9,0x8c,0xbf,0x58,0x71,0xa4,0x80,0x33,
        0x3d,0xf7,0xb4,0xfd,0x8c,0x55,0x47,0x10,
        0xd5,0x91,0x35,0xbe,0xd8,0x0d,0xa5,0x9e
    };
    WOLFSSL_SMALL_STACK_STATIC const byte ctr192Wrap64_2CipherLong[] =
    {
        0x59,0xf1,0xed,0x70,0x62,0x42,0xa8,0x06,
        0x07,0x36,0xe1,0xc5,0x04,0x79,0xc3,0x9b,
        0xd1,0x14,0x5c,0xcc,0x6f,0x81,0x5f,0x2f,
        0xa0,0xde,0xcf,0x61,0x55,0x18,0x7a,0xac,
        0xb0,0x59,0x37,0x90,0x53,0xb3,0x00,0x88,
        0xb4,0x49,0x90,0x7b,0x96,0xcd,0xcc,0xc3
    };

    WOLFSSL_SMALL_STACK_STATIC const byte ctr192Wrap96Cipher[] =
    {
        0x41,0x21,0x40,0x31,0xfb,0xc8,0xad,0x23,
        0xde,0x00,0x03,0xd8,0x96,0x45,0xc7,0xb5,
        0x47,0xb5,0xf3,0x30,0x3b
    };
    WOLFSSL_SMALL_STACK_STATIC const byte ctr192Wrap96CipherLong[] =
    {
        0x41,0x21,0x40,0x31,0xfb,0xc8,0xad,0x23,
        0xde,0x00,0x03,0xd8,0x96,0x45,0xc7,0xb5,
        0x47,0xb5,0xf3,0x30,0x3b,0xef,0xcd,0x80,
        0xe3,0x61,0x34,0xef,0x4e,0x1b,0x79,0xc1
    };
    WOLFSSL_SMALL_STACK_STATIC const byte ctr192Wrap96_2CipherLong[] =
    {
        0x3c,0xb2,0xff,0xc0,0x24,0xe1,0xf5,0xc4,
        0x0f,0xd1,0x0a,0x1b,0xbe,0x1f,0x23,0xa1,
        0x8e,0xbf,0x2b,0x96,0xb6,0x37,0xc8,0x25,
        0x06,0x90,0xe2,0xca,0x71,0x24,0x52,0x95,
        0xaa,0x8c,0x80,0xdf,0xb7,0xd7,0x30,0xb0,
        0xcc,0x06,0x4f,0x28,0xa2,0x74,0x27,0xf8
    };
#endif

    WOLFSSL_SMALL_STACK_STATIC const byte ctr192Wrap128Cipher[] =
    {
        0x5c,0xc3,0x8f,0xab,0x30,0xb6,0xac,0x67,
        0xdc,0xc2,0x1e,0x7b,0x01,0x2e,0xcf,0x98,
        0x8c,0x68,0xa7,0xd9,0x57
    };
    WOLFSSL_SMALL_STACK_STATIC const byte ctr192Wrap128CipherLong[] =
    {
        0x5c,0xc3,0x8f,0xab,0x30,0xb6,0xac,0x67,
        0xdc,0xc2,0x1e,0x7b,0x01,0x2e,0xcf,0x98,
        0x8c,0x68,0xa7,0xd9,0x57,0xab,0x09,0x0f,
        0x01,0xc4,0x4e,0x62,0xaf,0xc2,0xdf,0x1a
    };
#ifdef WOLFSSL_ARMASM
    WOLFSSL_SMALL_STACK_STATIC const byte ctr192Wrap128_2CipherLong[] =
    {
        0x88,0x0a,0x26,0x4e,0xa8,0x26,0x21,0xe0,
        0xfc,0xbc,0x63,0xdc,0xd9,0x60,0x52,0xb2,
        0x99,0x2f,0xbb,0x1e,0x00,0xf5,0x9f,0x6d,
        0xab,0x48,0x0f,0xc6,0x37,0x12,0x56,0xe3,
        0x12,0x8d,0x31,0xc8,0xea,0xf4,0x41,0x82,
        0x7a,0x88,0xe0,0xd7,0xf0,0x67,0x03,0xa4
    };
#endif
#endif
#ifdef WOLFSSL_AES_256
    WOLFSSL_SMALL_STACK_STATIC const byte ctr256Key[] =
    {
        0x60,0x3d,0xeb,0x10,0x15,0xca,0x71,0xbe,
        0x2b,0x73,0xae,0xf0,0x85,0x7d,0x77,0x81,
        0x1f,0x35,0x2c,0x07,0x3b,0x61,0x08,0xd7,
        0x2d,0x98,0x10,0xa3,0x09,0x14,0xdf,0xf4
    };

    WOLFSSL_SMALL_STACK_STATIC const byte ctr256Cipher[] =
    {
        0x60,0x1e,0xc3,0x13,0x77,0x57,0x89,0xa5,
        0xb7,0xa7,0xf5,0x04,0xbb,0xf3,0xd2,0x28,
        0xf4,0x43,0xe3,0xca,0x4d,0x62,0xb5,0x9a,
        0xca,0x84,0xe9,0x90,0xca,0xca,0xf5,0xc5,
        0x2b,0x09,0x30,0xda,0xa2,0x3d,0xe9,0x4c,
        0xe8,0x70,0x17,0xba,0x2d,0x84,0x98,0x8d,
        0xdf,0xc9,0xc5,0x8d,0xb6,0x7a,0xad,0xa6,
        0x13,0xc2,0xdd,0x08,0x45,0x79,0x41,0xa6
    };

#ifdef WOLFSSL_ARMASM
    WOLFSSL_SMALL_STACK_STATIC const byte ctr256Wrap32Cipher[] =
    {
        0xb0,0xa8,0xc0,0x65,0x85,0x20,0x0d,0x5c,
        0x25,0xcf,0xe7,0x58,0x63,0xc8,0xd4,0xea,
        0xa2,0x13,0x47,0x74,0xda
    };
    WOLFSSL_SMALL_STACK_STATIC const byte ctr256Wrap32CipherLong[] =
    {
        0xb0,0xa8,0xc0,0x65,0x85,0x20,0x0d,0x5c,
        0x25,0xcf,0xe7,0x58,0x63,0xc8,0xd4,0xea,
        0xa2,0x13,0x47,0x74,0xda,0x89,0x77,0x40,
        0x28,0x9c,0xe8,0x19,0x26,0x32,0xd8,0x1f
    };
    WOLFSSL_SMALL_STACK_STATIC const byte ctr256Wrap32_2CipherLong[] =
    {
        0xf6,0xd9,0x22,0xc6,0x80,0x29,0xaf,0x14,
        0x54,0x6c,0x0a,0xce,0x42,0xea,0x3c,0xa1,
        0x7c,0xeb,0x36,0x0d,0x8e,0xd7,0x8c,0x59,
        0xa8,0x09,0x9f,0x9e,0xba,0x5b,0x95,0xfa,
        0x26,0x8c,0x37,0x59,0xf8,0xae,0x8e,0xaa,
        0x4d,0xe4,0x1c,0xfe,0x51,0xc7,0xb7,0xcc
    };

    WOLFSSL_SMALL_STACK_STATIC const byte ctr256Wrap64Cipher[] =
    {
        0x74,0x1a,0x52,0x41,0x76,0xb4,0x11,0x8f,
        0xfd,0x57,0x31,0xfd,0x3d,0x76,0x8f,0x07,
        0xd4,0x94,0x4c,0xcd,0x4d
    };
    WOLFSSL_SMALL_STACK_STATIC const byte ctr256Wrap64CipherLong[] =
    {
        0x74,0x1a,0x52,0x41,0x76,0xb4,0x11,0x8f,
        0xfd,0x57,0x31,0xfd,0x3d,0x76,0x8f,0x07,
        0xd4,0x94,0x4c,0xcd,0x4d,0x47,0x5a,0x92,
        0x26,0x49,0x81,0x7a,0xda,0x36,0x27,0x01
    };
    WOLFSSL_SMALL_STACK_STATIC const byte ctr256Wrap64_2CipherLong[] =
    {
        0xf7,0x9c,0xbf,0xf6,0xa2,0xaa,0x8a,0x0a,
        0x63,0x8a,0x20,0x2f,0x12,0xf1,0x8e,0x49,
        0x30,0xc0,0x8d,0x5c,0x5f,0x8b,0xbc,0x16,
        0xdd,0x71,0xee,0x13,0x14,0x7b,0xe1,0x25,
        0xcb,0x87,0x8a,0xc6,0xdc,0x1d,0x54,0x7a,
        0xe1,0xe4,0x6f,0x0d,0x95,0x1b,0xd1,0x8b
    };

    WOLFSSL_SMALL_STACK_STATIC const byte ctr256Wrap96Cipher[] =
    {
        0xb9,0x07,0x02,0xb8,0xbe,0x94,0xbf,0x53,
        0xdf,0x83,0x8e,0x23,0x8c,0x67,0x0c,0x81,
        0xb8,0x69,0xa1,0x48,0x45
    };
    WOLFSSL_SMALL_STACK_STATIC const byte ctr256Wrap96CipherLong[] =
    {
        0xb9,0x07,0x02,0xb8,0xbe,0x94,0xbf,0x53,
        0xdf,0x83,0x8e,0x23,0x8c,0x67,0x0c,0x81,
        0xb8,0x69,0xa1,0x48,0x45,0xf1,0xc6,0x27,
        0x36,0xa8,0xb2,0x4b,0x0e,0x62,0x6b,0x72
    };
    WOLFSSL_SMALL_STACK_STATIC const byte ctr256Wrap96_2CipherLong[] =
    {
        0xd5,0x56,0x73,0xaa,0xb8,0xe4,0x06,0xf6,
        0x83,0x45,0x3a,0xb4,0xb9,0x63,0xec,0xad,
        0x73,0xc5,0xab,0x78,0xb1,0x21,0xab,0xef,
        0x69,0x15,0xb7,0x0c,0xe9,0xb4,0x3a,0xe7,
        0xbc,0xc4,0x22,0xbd,0x93,0xba,0x52,0xe0,
        0x91,0x2f,0x5e,0x8d,0x6d,0x59,0xf7,0xc2
    };
#endif

    WOLFSSL_SMALL_STACK_STATIC const byte ctr256Wrap128Cipher[] =
    {
        0x50,0xfd,0x97,0xc3,0xe6,0x1a,0xbb,0x48,
        0x73,0xfb,0x78,0xdf,0x1e,0x8e,0x77,0xe6,
        0x4b,0x45,0x7c,0xd6,0x8a
    };
    WOLFSSL_SMALL_STACK_STATIC const byte ctr256Wrap128CipherLong[] =
    {
        0x50,0xfd,0x97,0xc3,0xe6,0x1a,0xbb,0x48,
        0x73,0xfb,0x78,0xdf,0x1e,0x8e,0x77,0xe6,
        0x4b,0x45,0x7c,0xd6,0x8a,0xcc,0xda,0x4a,
        0x89,0xfa,0x23,0x6c,0x06,0xbf,0x26,0x05
    };
#ifdef WOLFSSL_ARMASM
    WOLFSSL_SMALL_STACK_STATIC const byte ctr256Wrap128_2CipherLong[] =
    {
        0x24,0x5c,0x09,0xa0,0x3b,0x1a,0x5a,0x94,
        0x2b,0x93,0x56,0x13,0x48,0xa0,0x21,0xce,
        0x95,0x11,0xa3,0x76,0xd6,0x59,0x88,0x42,
        0x04,0x71,0x69,0x62,0x28,0xb2,0xee,0x9d,
        0xd5,0xa0,0xea,0xc7,0x37,0x93,0x92,0xc7,
        0xf2,0xb6,0x8d,0xd9,0x59,0x1a,0xfa,0xbb
    };
#endif
#endif
    int i;
    struct {
        const byte* key;
        int         keySz;
        const byte* iv;
        const byte* plain;
        int         len;
        const byte* cipher;
    } testVec[] = {
#ifdef WOLFSSL_AES_128
        { ctr128Key, (int)sizeof(ctr128Key), ctrIv,
          ctrPlain, (int)sizeof(ctrPlain), ctr128Cipher },
        /* let's try with just 9 bytes, non block size test */
        { ctr128Key, (int)sizeof(ctr128Key), ctrIv,
          ctrPlain, (int)sizeof(oddCipher), ctr128Cipher },
        /* and an additional 9 bytes to reuse tmp left buffer */
        { NULL, 0, NULL, ctrPlain, (int)sizeof(oddCipher), oddCipher },
        /* Counter wrapping */
        { ctr128Key, (int)sizeof(ctr128Key), ctrIvWrap128,
          ctrPlain, (int)sizeof(ctr128Wrap128Cipher), ctr128Wrap128Cipher },
        { ctr128Key, (int)sizeof(ctr128Key), ctrIvWrap128,
          ctrPlain, (int)sizeof(ctr128Wrap128CipherLong),
          ctr128Wrap128CipherLong },
    #ifdef WOLFSSL_ARMASM
        { ctr128Key, (int)sizeof(ctr128Key), ctrIvWrap128_2,
          ctrPlain, (int)sizeof(ctr128Wrap128_2CipherLong),
          ctr128Wrap128_2CipherLong },
        { ctr128Key, (int)sizeof(ctr128Key), ctrIvWrap96,
          ctrPlain, (int)sizeof(ctr128Wrap96Cipher), ctr128Wrap96Cipher },
        { ctr128Key, (int)sizeof(ctr128Key), ctrIvWrap96,
          ctrPlain, (int)sizeof(ctr128Wrap96CipherLong),
          ctr128Wrap96CipherLong },
        { ctr128Key, (int)sizeof(ctr128Key), ctrIvWrap96_2,
          ctrPlain, (int)sizeof(ctr128Wrap96_2CipherLong),
          ctr128Wrap96_2CipherLong },
        { ctr128Key, (int)sizeof(ctr128Key), ctrIvWrap64,
          ctrPlain, (int)sizeof(ctr128Wrap64Cipher), ctr128Wrap64Cipher },
        { ctr128Key, (int)sizeof(ctr128Key), ctrIvWrap64,
          ctrPlain, (int)sizeof(ctr128Wrap64CipherLong),
          ctr128Wrap64CipherLong },
        { ctr128Key, (int)sizeof(ctr128Key), ctrIvWrap64_2,
          ctrPlain, (int)sizeof(ctr128Wrap64_2CipherLong),
          ctr128Wrap64_2CipherLong },
        { ctr128Key, (int)sizeof(ctr128Key), ctrIvWrap32,
          ctrPlain, (int)sizeof(ctr128Wrap32Cipher), ctr128Wrap32Cipher },
        { ctr128Key, (int)sizeof(ctr128Key), ctrIvWrap32,
          ctrPlain, (int)sizeof(ctr128Wrap32CipherLong),
          ctr128Wrap32CipherLong },
        { ctr128Key, (int)sizeof(ctr128Key), ctrIvWrap32_2,
          ctrPlain, (int)sizeof(ctr128Wrap32_2CipherLong),
          ctr128Wrap32_2CipherLong },
    #endif
#endif
#ifdef WOLFSSL_AES_192
        { ctr192Key, (int)sizeof(ctr192Key), ctrIv,
          ctrPlain, (int)sizeof(ctrPlain), ctr192Cipher },
        /* let's try with just 9 bytes, non block size test */
        { ctr192Key, (int)sizeof(ctr192Key), ctrIv,
          ctrPlain, (int)sizeof(oddCipher), ctr192Cipher },
        /* Counter wrapping */
        { ctr192Key, (int)sizeof(ctr192Key), ctrIvWrap128,
          ctrPlain, (int)sizeof(ctr192Wrap128Cipher), ctr192Wrap128Cipher },
        { ctr192Key, (int)sizeof(ctr192Key), ctrIvWrap128,
          ctrPlain, (int)sizeof(ctr192Wrap128CipherLong),
          ctr192Wrap128CipherLong },
    #ifdef WOLFSSL_ARMASM
        { ctr192Key, (int)sizeof(ctr192Key), ctrIvWrap128_2,
          ctrPlain, (int)sizeof(ctr192Wrap128_2CipherLong),
          ctr192Wrap128_2CipherLong },
        { ctr192Key, (int)sizeof(ctr192Key), ctrIvWrap96,
          ctrPlain, (int)sizeof(ctr192Wrap96Cipher), ctr192Wrap96Cipher },
        { ctr192Key, (int)sizeof(ctr192Key), ctrIvWrap96,
          ctrPlain, (int)sizeof(ctr192Wrap96CipherLong),
          ctr192Wrap96CipherLong },
        { ctr192Key, (int)sizeof(ctr192Key), ctrIvWrap96_2,
          ctrPlain, (int)sizeof(ctr192Wrap96_2CipherLong),
          ctr192Wrap96_2CipherLong },
        { ctr192Key, (int)sizeof(ctr192Key), ctrIvWrap64,
          ctrPlain, (int)sizeof(ctr192Wrap64Cipher), ctr192Wrap64Cipher },
        { ctr192Key, (int)sizeof(ctr192Key), ctrIvWrap64,
          ctrPlain, (int)sizeof(ctr192Wrap64CipherLong),
          ctr192Wrap64CipherLong },
        { ctr192Key, (int)sizeof(ctr192Key), ctrIvWrap64_2,
          ctrPlain, (int)sizeof(ctr192Wrap64_2CipherLong),
          ctr192Wrap64_2CipherLong },
        { ctr192Key, (int)sizeof(ctr192Key), ctrIvWrap32,
          ctrPlain, (int)sizeof(ctr192Wrap32Cipher), ctr192Wrap32Cipher },
        { ctr192Key, (int)sizeof(ctr192Key), ctrIvWrap32,
          ctrPlain, (int)sizeof(ctr192Wrap32CipherLong),
          ctr192Wrap32CipherLong },
        { ctr192Key, (int)sizeof(ctr192Key), ctrIvWrap32_2,
          ctrPlain, (int)sizeof(ctr192Wrap32_2CipherLong),
          ctr192Wrap32_2CipherLong },
    #endif
#endif
#ifdef WOLFSSL_AES_256
        { ctr256Key, (int)sizeof(ctr256Key), ctrIv,
          ctrPlain, (int)sizeof(ctrPlain), ctr256Cipher },
        /* let's try with just 9 bytes, non block size test */
        { ctr256Key, (int)sizeof(ctr256Key), ctrIv,
          ctrPlain, (int)sizeof(oddCipher), ctr256Cipher },
        /* Counter wrapping */
        { ctr256Key, (int)sizeof(ctr256Key), ctrIvWrap128,
          ctrPlain, (int)sizeof(ctr256Wrap128Cipher), ctr256Wrap128Cipher },
        { ctr256Key, (int)sizeof(ctr256Key), ctrIvWrap128,
          ctrPlain, (int)sizeof(ctr256Wrap128CipherLong),
          ctr256Wrap128CipherLong },
    #ifdef WOLFSSL_ARMASM
        { ctr256Key, (int)sizeof(ctr256Key), ctrIvWrap128_2,
          ctrPlain, (int)sizeof(ctr256Wrap128_2CipherLong),
          ctr256Wrap128_2CipherLong },
        { ctr256Key, (int)sizeof(ctr256Key), ctrIvWrap96,
          ctrPlain, (int)sizeof(ctr256Wrap96Cipher), ctr256Wrap96Cipher },
        { ctr256Key, (int)sizeof(ctr256Key), ctrIvWrap96,
          ctrPlain, (int)sizeof(ctr256Wrap96CipherLong),
          ctr256Wrap96CipherLong },
        { ctr256Key, (int)sizeof(ctr256Key), ctrIvWrap96_2,
          ctrPlain, (int)sizeof(ctr256Wrap96_2CipherLong),
          ctr256Wrap96_2CipherLong },
        { ctr256Key, (int)sizeof(ctr256Key), ctrIvWrap64,
          ctrPlain, (int)sizeof(ctr256Wrap64Cipher), ctr256Wrap64Cipher },
        { ctr256Key, (int)sizeof(ctr256Key), ctrIvWrap64,
          ctrPlain, (int)sizeof(ctr256Wrap64CipherLong),
          ctr256Wrap64CipherLong },
        { ctr256Key, (int)sizeof(ctr256Key), ctrIvWrap64_2,
          ctrPlain, (int)sizeof(ctr256Wrap64_2CipherLong),
          ctr256Wrap64_2CipherLong },
        { ctr256Key, (int)sizeof(ctr256Key), ctrIvWrap32,
          ctrPlain, (int)sizeof(ctr256Wrap32Cipher), ctr256Wrap32Cipher },
        { ctr256Key, (int)sizeof(ctr256Key), ctrIvWrap32,
          ctrPlain, (int)sizeof(ctr256Wrap32CipherLong),
          ctr256Wrap32CipherLong },
        { ctr256Key, (int)sizeof(ctr256Key), ctrIvWrap32_2,
          ctrPlain, (int)sizeof(ctr256Wrap32_2CipherLong),
          ctr256Wrap32_2CipherLong },
    #endif
#endif
    };
    #define AES_CTR_TEST_LEN (int)(sizeof(testVec) / sizeof(*testVec))

    for (i = 0; i < AES_CTR_TEST_LEN; i++) {
        if (testVec[i].key != NULL) {
            ret = wc_AesSetKeyDirect(enc, testVec[i].key, testVec[i].keySz,
                testVec[i].iv, AES_ENCRYPTION);
            if (ret != 0) {
                ERROR_OUT(-5930 - i * 10, out);
            }
            /* Ctr only uses encrypt, even on key setup */
            ret = wc_AesSetKeyDirect(dec, testVec[i].key, testVec[i].keySz,
                testVec[i].iv, AES_ENCRYPTION);
            if (ret != 0) {
                ERROR_OUT(-5931 - i * 10, out);
            }
        }

        ret = wc_AesCtrEncrypt(enc, cipher, testVec[i].plain, testVec[i].len);
        if (ret != 0) {
            ERROR_OUT(-5932 - i * 10, out);
        }
        ret = wc_AesCtrEncrypt(dec, plain, cipher, testVec[i].len);
        if (ret != 0) {
            ERROR_OUT(-5933 - i * 10, out);
        }

        if (XMEMCMP(plain, ctrPlain, testVec[i].len)) {
            ERROR_OUT(-5934 - i * 10, out);
        }
        if (XMEMCMP(cipher, testVec[i].cipher, testVec[i].len)) {
            ERROR_OUT(-5935 - i * 10, out);
        }
    }

out:
    return ret;
}
#endif /* WOLFSSL_AES_COUNTER */


WOLFSSL_TEST_SUBROUTINE int aes_test(void)
{
#if defined(HAVE_AES_CBC) || defined(WOLFSSL_AES_COUNTER) || defined(WOLFSSL_AES_DIRECT)
#ifdef WOLFSSL_SMALL_STACK
    Aes *enc = (Aes *)XMALLOC(sizeof *enc, HEAP_HINT, DYNAMIC_TYPE_AES);
#else
    Aes enc[1];
#endif
    byte cipher[AES_BLOCK_SIZE * 4];
#ifdef HAVE_AES_DECRYPT
#ifdef WOLFSSL_SMALL_STACK
    Aes *dec = (Aes *)XMALLOC(sizeof *dec, HEAP_HINT, DYNAMIC_TYPE_AES);
#else
    Aes dec[1];
#endif
    byte plain [AES_BLOCK_SIZE * 4];
#endif /* HAVE_AES_DECRYPT */
#endif /* HAVE_AES_CBC || WOLFSSL_AES_COUNTER || WOLFSSL_AES_DIRECT */
    int  ret = 0;

#ifdef HAVE_AES_CBC
#ifdef WOLFSSL_AES_128
    WOLFSSL_SMALL_STACK_STATIC const byte msg[] = { /* "Now is the time for all " w/o trailing 0 */
        0x6e,0x6f,0x77,0x20,0x69,0x73,0x20,0x74,
        0x68,0x65,0x20,0x74,0x69,0x6d,0x65,0x20,
        0x66,0x6f,0x72,0x20,0x61,0x6c,0x6c,0x20
    };

    WOLFSSL_SMALL_STACK_STATIC const byte verify[] =
    {
        0x95,0x94,0x92,0x57,0x5f,0x42,0x81,0x53,
        0x2c,0xcc,0x9d,0x46,0x77,0xa2,0x33,0xcb
    };

    WOLFSSL_SMALL_STACK_STATIC const byte key[] = "0123456789abcdef   ";  /* align */
    WOLFSSL_SMALL_STACK_STATIC const byte iv[]  = "1234567890abcdef   ";  /* align */

#ifdef WOLFSSL_SMALL_STACK
#if defined(HAVE_AES_CBC) || defined(WOLFSSL_AES_COUNTER) || defined(WOLFSSL_AES_DIRECT)
    if (enc == NULL)
        ERROR_OUT(-5990, out);
#endif
#if defined(HAVE_AES_DECRYPT) || defined(WOLFSSL_AES_COUNTER) || defined(WOLFSSL_AES_DIRECT)
    if (dec == NULL)
        ERROR_OUT(-5991, out);
#endif
#endif

    if (wc_AesInit(enc, HEAP_HINT, devId) != 0)
        ERROR_OUT(-5900, out); /* note this error code is used programmatically in cleanup. */
#if defined(HAVE_AES_DECRYPT) || defined(WOLFSSL_AES_COUNTER)
    if (wc_AesInit(dec, HEAP_HINT, devId) != 0)
        ERROR_OUT(-5901, out); /* note this error code is used programmatically in cleanup. */
#endif

    ret = wc_AesSetKey(enc, key, AES_BLOCK_SIZE, iv, AES_ENCRYPTION);
    if (ret != 0)
        ERROR_OUT(-5902, out);
#if defined(HAVE_AES_DECRYPT) || defined(WOLFSSL_AES_COUNTER)
    ret = wc_AesSetKey(dec, key, AES_BLOCK_SIZE, iv, AES_DECRYPTION);
    if (ret != 0)
        ERROR_OUT(-5903, out);
#endif

    XMEMSET(cipher, 0, AES_BLOCK_SIZE * 4);
    ret = wc_AesCbcEncrypt(enc, cipher, msg, AES_BLOCK_SIZE);
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &enc->asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    if (ret != 0)
        ERROR_OUT(-5904, out);
#ifdef HAVE_AES_DECRYPT
    XMEMSET(plain, 0, AES_BLOCK_SIZE * 4);
    ret = wc_AesCbcDecrypt(dec, plain, cipher, AES_BLOCK_SIZE);
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &dec->asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    if (ret != 0)
        ERROR_OUT(-5905, out);

    if (XMEMCMP(plain, msg, AES_BLOCK_SIZE))
        ERROR_OUT(-5906, out);
#endif /* HAVE_AES_DECRYPT */
    if (XMEMCMP(cipher, verify, AES_BLOCK_SIZE))
        ERROR_OUT(-5907, out);
#endif /* WOLFSSL_AES_128 */

#if defined(WOLFSSL_AESNI) && defined(HAVE_AES_DECRYPT)
    {
        WOLFSSL_SMALL_STACK_STATIC const byte bigMsg[] = {
            /* "All work and no play makes Jack a dull boy. " */
            0x41,0x6c,0x6c,0x20,0x77,0x6f,0x72,0x6b,
            0x20,0x61,0x6e,0x64,0x20,0x6e,0x6f,0x20,
            0x70,0x6c,0x61,0x79,0x20,0x6d,0x61,0x6b,
            0x65,0x73,0x20,0x4a,0x61,0x63,0x6b,0x20,
            0x61,0x20,0x64,0x75,0x6c,0x6c,0x20,0x62,
            0x6f,0x79,0x2e,0x20,0x41,0x6c,0x6c,0x20,
            0x77,0x6f,0x72,0x6b,0x20,0x61,0x6e,0x64,
            0x20,0x6e,0x6f,0x20,0x70,0x6c,0x61,0x79,
            0x20,0x6d,0x61,0x6b,0x65,0x73,0x20,0x4a,
            0x61,0x63,0x6b,0x20,0x61,0x20,0x64,0x75,
            0x6c,0x6c,0x20,0x62,0x6f,0x79,0x2e,0x20,
            0x41,0x6c,0x6c,0x20,0x77,0x6f,0x72,0x6b,
            0x20,0x61,0x6e,0x64,0x20,0x6e,0x6f,0x20,
            0x70,0x6c,0x61,0x79,0x20,0x6d,0x61,0x6b,
            0x65,0x73,0x20,0x4a,0x61,0x63,0x6b,0x20,
            0x61,0x20,0x64,0x75,0x6c,0x6c,0x20,0x62,
            0x6f,0x79,0x2e,0x20,0x41,0x6c,0x6c,0x20,
            0x77,0x6f,0x72,0x6b,0x20,0x61,0x6e,0x64,
            0x20,0x6e,0x6f,0x20,0x70,0x6c,0x61,0x79,
            0x20,0x6d,0x61,0x6b,0x65,0x73,0x20,0x4a,
            0x61,0x63,0x6b,0x20,0x61,0x20,0x64,0x75,
            0x6c,0x6c,0x20,0x62,0x6f,0x79,0x2e,0x20,
            0x41,0x6c,0x6c,0x20,0x77,0x6f,0x72,0x6b,
            0x20,0x61,0x6e,0x64,0x20,0x6e,0x6f,0x20,
            0x70,0x6c,0x61,0x79,0x20,0x6d,0x61,0x6b,
            0x65,0x73,0x20,0x4a,0x61,0x63,0x6b,0x20,
            0x61,0x20,0x64,0x75,0x6c,0x6c,0x20,0x62,
            0x6f,0x79,0x2e,0x20,0x41,0x6c,0x6c,0x20,
            0x77,0x6f,0x72,0x6b,0x20,0x61,0x6e,0x64,
            0x20,0x6e,0x6f,0x20,0x70,0x6c,0x61,0x79,
            0x20,0x6d,0x61,0x6b,0x65,0x73,0x20,0x4a,
            0x61,0x63,0x6b,0x20,0x61,0x20,0x64,0x75,
            0x6c,0x6c,0x20,0x62,0x6f,0x79,0x2e,0x20,
            0x41,0x6c,0x6c,0x20,0x77,0x6f,0x72,0x6b,
            0x20,0x61,0x6e,0x64,0x20,0x6e,0x6f,0x20,
            0x70,0x6c,0x61,0x79,0x20,0x6d,0x61,0x6b,
            0x65,0x73,0x20,0x4a,0x61,0x63,0x6b,0x20,
            0x61,0x20,0x64,0x75,0x6c,0x6c,0x20,0x62,
            0x6f,0x79,0x2e,0x20,0x41,0x6c,0x6c,0x20,
            0x77,0x6f,0x72,0x6b,0x20,0x61,0x6e,0x64,
            0x20,0x6e,0x6f,0x20,0x70,0x6c,0x61,0x79,
            0x20,0x6d,0x61,0x6b,0x65,0x73,0x20,0x4a,
            0x61,0x63,0x6b,0x20,0x61,0x20,0x64,0x75,
            0x6c,0x6c,0x20,0x62,0x6f,0x79,0x2e,0x20,
            0x41,0x6c,0x6c,0x20,0x77,0x6f,0x72,0x6b,
            0x20,0x61,0x6e,0x64,0x20,0x6e,0x6f,0x20,
            0x70,0x6c,0x61,0x79,0x20,0x6d,0x61,0x6b,
            0x65,0x73,0x20,0x4a,0x61,0x63,0x6b,0x20
        };
        WOLFSSL_SMALL_STACK_STATIC const byte bigKey[] = "0123456789abcdeffedcba9876543210";
        word32 keySz, msgSz;
#ifdef WOLFSSL_SMALL_STACK
        byte *bigCipher = (byte *)XMALLOC(sizeof(bigMsg), HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
        byte *bigPlain = (byte *)XMALLOC(sizeof(bigMsg), HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);

        if ((bigCipher == NULL) ||
            (bigPlain == NULL)) {
            if (bigCipher != NULL)
                XFREE(bigCipher, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
            ERROR_OUT(-5992, out);
        }
#else
        byte bigCipher[sizeof(bigMsg)];
        byte bigPlain[sizeof(bigMsg)];
#endif

        /* Iterate from one AES_BLOCK_SIZE of bigMsg through the whole
         * message by AES_BLOCK_SIZE for each size of AES key. */
        for (keySz = 16; keySz <= 32; keySz += 8) {
            for (msgSz = AES_BLOCK_SIZE;
                 msgSz <= sizeof(bigMsg);
                 msgSz += AES_BLOCK_SIZE) {

                XMEMSET(bigCipher, 0, sizeof(bigMsg));
                XMEMSET(bigPlain, 0, sizeof(bigMsg));
                ret = wc_AesSetKey(enc, bigKey, keySz, iv, AES_ENCRYPTION);
                if (ret != 0) {
                    ret = -5908;
                    break;
                }
                ret = wc_AesSetKey(dec, bigKey, keySz, iv, AES_DECRYPTION);
                if (ret != 0) {
                    ret = -5909;
                    break;
                }

                ret = wc_AesCbcEncrypt(enc, bigCipher, bigMsg, msgSz);
            #if defined(WOLFSSL_ASYNC_CRYPT)
                ret = wc_AsyncWait(ret, &enc->asyncDev, WC_ASYNC_FLAG_NONE);
            #endif
                if (ret != 0) {
                    ret = -5910;
                    break;
                }

                ret = wc_AesCbcDecrypt(dec, bigPlain, bigCipher, msgSz);
            #if defined(WOLFSSL_ASYNC_CRYPT)
                ret = wc_AsyncWait(ret, &dec->asyncDev, WC_ASYNC_FLAG_NONE);
            #endif
                if (ret != 0) {
                    ret = -5911;
                    break;
                }

                if (XMEMCMP(bigPlain, bigMsg, msgSz)) {
                    ret = -5912;
                    break;
                }
            }
            if (ret != 0)
                break;
        }

#ifdef WOLFSSL_SMALL_STACK
        XFREE(bigCipher, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
        XFREE(bigPlain, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
#endif

        if (ret != 0)
            goto out;
    }
#endif /* WOLFSSL_AESNI && HAVE_AES_DECRYPT */

    /* Test of AES IV state with encrypt/decrypt */
#ifdef WOLFSSL_AES_128
    {
        /* Test Vector from "NIST Special Publication 800-38A, 2001 Edition"
         * https://nvlpubs.nist.gov/nistpubs/legacy/sp/nistspecialpublication800-38a.pdf
         */
        WOLFSSL_SMALL_STACK_STATIC const byte msg2[] =
        {
            0x6b, 0xc1, 0xbe, 0xe2, 0x2e, 0x40, 0x9f, 0x96,
            0xe9, 0x3d, 0x7e, 0x11, 0x73, 0x93, 0x17, 0x2a,
            0xae, 0x2d, 0x8a, 0x57, 0x1e, 0x03, 0xac, 0x9c,
            0x9e, 0xb7, 0x6f, 0xac, 0x45, 0xaf, 0x8e, 0x51
        };

        WOLFSSL_SMALL_STACK_STATIC const byte verify2[] =
        {
            0x76, 0x49, 0xab, 0xac, 0x81, 0x19, 0xb2, 0x46,
            0xce, 0xe9, 0x8e, 0x9b, 0x12, 0xe9, 0x19, 0x7d,
            0x50, 0x86, 0xcb, 0x9b, 0x50, 0x72, 0x19, 0xee,
            0x95, 0xdb, 0x11, 0x3a, 0x91, 0x76, 0x78, 0xb2
        };
        WOLFSSL_SMALL_STACK_STATIC const byte key2[] = {
            0x2b, 0x7e, 0x15, 0x16, 0x28, 0xae, 0xd2, 0xa6,
            0xab, 0xf7, 0x15, 0x88, 0x09, 0xcf, 0x4f, 0x3c
        };
        WOLFSSL_SMALL_STACK_STATIC const byte iv2[]  = {
            0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
            0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f
        };


        ret = wc_AesSetKey(enc, key2, sizeof(key2), iv2, AES_ENCRYPTION);
        if (ret != 0)
            ERROR_OUT(-5913, out);
        XMEMSET(cipher, 0, AES_BLOCK_SIZE * 2);
        ret = wc_AesCbcEncrypt(enc, cipher, msg2, AES_BLOCK_SIZE);
    #if defined(WOLFSSL_ASYNC_CRYPT)
        ret = wc_AsyncWait(ret, &enc->asyncDev, WC_ASYNC_FLAG_NONE);
    #endif
        if (ret != 0)
            ERROR_OUT(-5914, out);
        if (XMEMCMP(cipher, verify2, AES_BLOCK_SIZE))
            ERROR_OUT(-5915, out);

        ret = wc_AesCbcEncrypt(enc, cipher + AES_BLOCK_SIZE,
                msg2 + AES_BLOCK_SIZE, AES_BLOCK_SIZE);
    #if defined(WOLFSSL_ASYNC_CRYPT)
        ret = wc_AsyncWait(ret, &enc->asyncDev, WC_ASYNC_FLAG_NONE);
    #endif
        if (ret != 0)
            ERROR_OUT(-5916, out);
        if (XMEMCMP(cipher + AES_BLOCK_SIZE, verify2 + AES_BLOCK_SIZE,
                    AES_BLOCK_SIZE))
            ERROR_OUT(-5917, out);

        #if defined(HAVE_AES_DECRYPT)
        ret = wc_AesSetKey(dec, key2, sizeof(key2), iv2, AES_DECRYPTION);
        if (ret != 0)
            ERROR_OUT(-5918, out);
        XMEMSET(plain, 0, AES_BLOCK_SIZE * 2);
        ret = wc_AesCbcDecrypt(dec, plain, verify2, AES_BLOCK_SIZE);
    #if defined(WOLFSSL_ASYNC_CRYPT)
        ret = wc_AsyncWait(ret, &dec->asyncDev, WC_ASYNC_FLAG_NONE);
    #endif
        if (ret != 0)
            ERROR_OUT(-5919, out);
        if (XMEMCMP(plain, msg2, AES_BLOCK_SIZE))
            ERROR_OUT(-5920, out);

        ret = wc_AesCbcDecrypt(dec, plain + AES_BLOCK_SIZE,
                verify2 + AES_BLOCK_SIZE, AES_BLOCK_SIZE);
    #if defined(WOLFSSL_ASYNC_CRYPT)
        ret = wc_AsyncWait(ret, &dec->asyncDev, WC_ASYNC_FLAG_NONE);
    #endif
        if (ret != 0)
            ERROR_OUT(-5921, out);
        if (XMEMCMP(plain + AES_BLOCK_SIZE, msg2 + AES_BLOCK_SIZE,
                    AES_BLOCK_SIZE))
            ERROR_OUT(-5922, out);

        #endif /* HAVE_AES_DECRYPT */
    }
#endif /* WOLFSSL_AES_128 */
#endif /* HAVE_AES_CBC */

#ifdef WOLFSSL_AES_COUNTER
    ret = aesctr_test(enc, dec, cipher, plain);
    if (ret != 0)
        return ret;
#endif

#if defined(WOLFSSL_AES_DIRECT) && defined(WOLFSSL_AES_256)
    {
        WOLFSSL_SMALL_STACK_STATIC const byte niPlain[] =
        {
            0x6b,0xc1,0xbe,0xe2,0x2e,0x40,0x9f,0x96,
            0xe9,0x3d,0x7e,0x11,0x73,0x93,0x17,0x2a
        };

        WOLFSSL_SMALL_STACK_STATIC const byte niCipher[] =
        {
            0xf3,0xee,0xd1,0xbd,0xb5,0xd2,0xa0,0x3c,
            0x06,0x4b,0x5a,0x7e,0x3d,0xb1,0x81,0xf8
        };

        WOLFSSL_SMALL_STACK_STATIC const byte niKey[] =
        {
            0x60,0x3d,0xeb,0x10,0x15,0xca,0x71,0xbe,
            0x2b,0x73,0xae,0xf0,0x85,0x7d,0x77,0x81,
            0x1f,0x35,0x2c,0x07,0x3b,0x61,0x08,0xd7,
            0x2d,0x98,0x10,0xa3,0x09,0x14,0xdf,0xf4
        };

        XMEMSET(cipher, 0, AES_BLOCK_SIZE);
        ret = wc_AesSetKey(enc, niKey, sizeof(niKey), cipher, AES_ENCRYPTION);
        if (ret != 0)
            ERROR_OUT(-5994, out);
#if !defined(HAVE_SELFTEST) && \
    (defined(WOLFSSL_LINUXKM) || \
     !defined(HAVE_FIPS) || \
     (defined(FIPS_VERSION_GE) && FIPS_VERSION_GE(5,3)))
        if (wc_AesEncryptDirect(enc, cipher, niPlain) != 0)
            ERROR_OUT(-5995, out);
#else
        wc_AesEncryptDirect(enc, cipher, niPlain);
#endif
        if (XMEMCMP(cipher, niCipher, AES_BLOCK_SIZE) != 0)
            ERROR_OUT(-5996, out);

        XMEMSET(plain, 0, AES_BLOCK_SIZE);
        ret = wc_AesSetKey(dec, niKey, sizeof(niKey), plain, AES_DECRYPTION);
        if (ret != 0)
            ERROR_OUT(-5997, out);
#if !defined(HAVE_SELFTEST) && \
    (defined(WOLFSSL_LINUXKM) || \
     !defined(HAVE_FIPS) || \
     (defined(FIPS_VERSION_GE) && FIPS_VERSION_GE(5,3)))
        if (wc_AesDecryptDirect(dec, plain, niCipher) != 0)
            ERROR_OUT(-5998, out);
#else
        wc_AesDecryptDirect(dec, plain, niCipher);
#endif
        if (XMEMCMP(plain, niPlain, AES_BLOCK_SIZE) != 0)
            ERROR_OUT(-5999, out);
    }
#endif /* WOLFSSL_AES_DIRECT && WOLFSSL_AES_256 */

    ret = aes_key_size_test();
    if (ret != 0)
        goto out;

#if defined(HAVE_AES_CBC) && defined(WOLFSSL_AES_128)
    ret = aes_cbc_test();
    if (ret != 0)
        goto out;
#endif

#if defined(WOLFSSL_AES_XTS)
    #ifdef WOLFSSL_AES_128
    ret = aes_xts_128_test();
    if (ret != 0)
        goto out;
    #endif
    #ifdef WOLFSSL_AES_256
    ret = aes_xts_256_test();
    if (ret != 0)
        goto out;
    #endif
    #if defined(WOLFSSL_AES_128) && defined(WOLFSSL_AES_256)
    ret = aes_xts_sector_test();
    if (ret != 0)
        goto out;
    #endif
    #ifdef WOLFSSL_AES_128
    ret = aes_xts_args_test();
    if (ret != 0)
        goto out;
    #endif
#endif

#if defined(WOLFSSL_AES_CFB)
    ret = aescfb_test();
    if (ret != 0)
        goto out;
#if !defined(HAVE_SELFTEST) && !defined(HAVE_FIPS)
    ret = aescfb1_test();
    if (ret != 0)
        goto out;

    ret = aescfb8_test();
    if (ret != 0)
        goto out;
#endif
#endif

#if defined(HAVE_AES_ECB) && !defined(HAVE_FIPS) && !defined(HAVE_SELFTEST)
    ret = aesecb_test();
    if (ret != 0)
        goto out;
#endif

  out:

#if defined(HAVE_AES_CBC) || defined(WOLFSSL_AES_COUNTER) || defined(WOLFSSL_AES_DIRECT)
#ifdef WOLFSSL_SMALL_STACK
    if (enc) {
        if (ret != -5900) /* note this must match ERRROR_OUT() code
                           * for wc_AesInit(enc, ...) failure above.
                           */
            wc_AesFree(enc);
        XFREE(enc, HEAP_HINT, DYNAMIC_TYPE_AES);
    }
#else
    if (ret != -5900)
        wc_AesFree(enc);
#endif
    (void)cipher;
#ifdef HAVE_AES_DECRYPT
#ifdef WOLFSSL_SMALL_STACK
    if (dec) {
        if ((ret != -5900) && (ret != -5901))
            /* note these codes must match the ERRROR_OUT() codes for
             * wc_AesInit() failures above.
             */
            wc_AesFree(dec);
        XFREE(dec, HEAP_HINT, DYNAMIC_TYPE_AES);
    }
#else
    if ((ret != -5900) && (ret != -5901))
        wc_AesFree(dec);
#endif
    (void)plain;
#endif /* HAVE_AES_DECRYPT */
#endif /* HAVE_AES_CBC || WOLFSSL_AES_COUNTER || WOLFSSL_AES_DIRECT */

    return ret;
}

#ifdef WOLFSSL_AES_192
WOLFSSL_TEST_SUBROUTINE int aes192_test(void)
{
#ifdef HAVE_AES_CBC
#ifdef WOLFSSL_SMALL_STACK
    Aes *enc = NULL;
#else
    Aes enc[1];
#endif
    byte cipher[AES_BLOCK_SIZE];
#ifdef HAVE_AES_DECRYPT
#ifdef WOLFSSL_SMALL_STACK
    Aes *dec = NULL;
#else
    Aes dec[1];
#endif
    byte plain[AES_BLOCK_SIZE];
#endif
#endif /* HAVE_AES_CBC */
    int  ret = 0;

#ifdef HAVE_AES_CBC
    /* Test vectors from NIST Special Publication 800-38A, 2001 Edition
     * Appendix F.2.3  */

    WOLFSSL_SMALL_STACK_STATIC const byte msg[] = {
        0x6b,0xc1,0xbe,0xe2,0x2e,0x40,0x9f,0x96,
        0xe9,0x3d,0x7e,0x11,0x73,0x93,0x17,0x2a
    };

    WOLFSSL_SMALL_STACK_STATIC const byte verify[] =
    {
        0x4f,0x02,0x1d,0xb2,0x43,0xbc,0x63,0x3d,
        0x71,0x78,0x18,0x3a,0x9f,0xa0,0x71,0xe8
    };

    WOLFSSL_SMALL_STACK_STATIC byte key[] = {
        0x8e,0x73,0xb0,0xf7,0xda,0x0e,0x64,0x52,
        0xc8,0x10,0xf3,0x2b,0x80,0x90,0x79,0xe5,
        0x62,0xf8,0xea,0xd2,0x52,0x2c,0x6b,0x7b
    };
    WOLFSSL_SMALL_STACK_STATIC byte iv[]  = {
        0x00,0x01,0x02,0x03,0x04,0x05,0x06,0x07,
        0x08,0x09,0x0A,0x0B,0x0C,0x0D,0x0E,0x0F
    };

#ifdef WOLFSSL_SMALL_STACK
    if ((enc = (Aes *)XMALLOC(sizeof *enc, HEAP_HINT, DYNAMIC_TYPE_AES)) == NULL)
        ERROR_OUT(-6008, out);
#ifdef HAVE_AES_DECRYPT
    if ((dec = (Aes *)XMALLOC(sizeof *dec, HEAP_HINT, DYNAMIC_TYPE_AES)) == NULL)
        ERROR_OUT(-6009, out);
#endif
#endif

    if (wc_AesInit(enc, HEAP_HINT, devId) != 0)
        ERROR_OUT(-6000, out);
#ifdef HAVE_AES_DECRYPT
    if (wc_AesInit(dec, HEAP_HINT, devId) != 0)
        ERROR_OUT(-6001, out);
#endif

    ret = wc_AesSetKey(enc, key, (int) sizeof(key), iv, AES_ENCRYPTION);
    if (ret != 0)
        ERROR_OUT(-6002, out);
#ifdef HAVE_AES_DECRYPT
    ret = wc_AesSetKey(dec, key, (int) sizeof(key), iv, AES_DECRYPTION);
    if (ret != 0)
        ERROR_OUT(-6003, out);
#endif

    XMEMSET(cipher, 0, AES_BLOCK_SIZE);
    ret = wc_AesCbcEncrypt(enc, cipher, msg, (int) sizeof(msg));
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &enc->asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    if (ret != 0)
        ERROR_OUT(-6004, out);
#ifdef HAVE_AES_DECRYPT
    XMEMSET(plain, 0, AES_BLOCK_SIZE);
    ret = wc_AesCbcDecrypt(dec, plain, cipher, (int) sizeof(cipher));
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &dec->asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    if (ret != 0)
        ERROR_OUT(-6005, out);
    if (XMEMCMP(plain, msg, (int) sizeof(plain))) {
        ERROR_OUT(-6006, out);
    }
#endif

    if (XMEMCMP(cipher, verify, (int) sizeof(cipher)))
        ERROR_OUT(-6007, out);

    wc_AesFree(enc);
#ifdef HAVE_AES_DECRYPT
    wc_AesFree(dec);
#endif

  out:
#ifdef WOLFSSL_SMALL_STACK
    if (enc)
        XFREE(enc, HEAP_HINT, DYNAMIC_TYPE_AES);
#ifdef HAVE_AES_DECRYPT
    if (dec)
        XFREE(dec, HEAP_HINT, DYNAMIC_TYPE_AES);
#endif
#endif
#endif /* HAVE_AES_CBC */

    return ret;
}
#endif /* WOLFSSL_AES_192 */

#ifdef WOLFSSL_AES_256
WOLFSSL_TEST_SUBROUTINE int aes256_test(void)
{
#ifdef HAVE_AES_CBC
#ifdef WOLFSSL_SMALL_STACK
    Aes *enc = NULL;
#else
    Aes enc[1];
#endif
    byte cipher[AES_BLOCK_SIZE];
#ifdef HAVE_AES_DECRYPT
#ifdef WOLFSSL_SMALL_STACK
    Aes *dec = NULL;
#else
    Aes dec[1];
#endif
    byte plain[AES_BLOCK_SIZE];
#endif
#endif /* HAVE_AES_CBC */
    int  ret = 0;

#ifdef HAVE_AES_CBC
    /* Test vectors from NIST Special Publication 800-38A, 2001 Edition,
     * Appendix F.2.5  */
    WOLFSSL_SMALL_STACK_STATIC const byte msg[] = {
        0x6b,0xc1,0xbe,0xe2,0x2e,0x40,0x9f,0x96,
        0xe9,0x3d,0x7e,0x11,0x73,0x93,0x17,0x2a
    };

    WOLFSSL_SMALL_STACK_STATIC const byte verify[] =
    {
        0xf5,0x8c,0x4c,0x04,0xd6,0xe5,0xf1,0xba,
        0x77,0x9e,0xab,0xfb,0x5f,0x7b,0xfb,0xd6
    };

    WOLFSSL_SMALL_STACK_STATIC byte key[] = {
        0x60,0x3d,0xeb,0x10,0x15,0xca,0x71,0xbe,
        0x2b,0x73,0xae,0xf0,0x85,0x7d,0x77,0x81,
        0x1f,0x35,0x2c,0x07,0x3b,0x61,0x08,0xd7,
        0x2d,0x98,0x10,0xa3,0x09,0x14,0xdf,0xf4
    };
    WOLFSSL_SMALL_STACK_STATIC byte iv[]  = {
        0x00,0x01,0x02,0x03,0x04,0x05,0x06,0x07,
        0x08,0x09,0x0A,0x0B,0x0C,0x0D,0x0E,0x0F
    };

#ifdef WOLFSSL_SMALL_STACK
    if ((enc = (Aes *)XMALLOC(sizeof *enc, HEAP_HINT, DYNAMIC_TYPE_AES)) == NULL)
        ERROR_OUT(-6108, out);
#ifdef HAVE_AES_DECRYPT
    if ((dec = (Aes *)XMALLOC(sizeof *dec, HEAP_HINT, DYNAMIC_TYPE_AES)) == NULL)
        ERROR_OUT(-6109, out);
#endif
#endif

    if (wc_AesInit(enc, HEAP_HINT, devId) != 0)
        ERROR_OUT(-6100, out);
#ifdef HAVE_AES_DECRYPT
    if (wc_AesInit(dec, HEAP_HINT, devId) != 0)
        ERROR_OUT(-6101, out);
#endif

    ret = wc_AesSetKey(enc, key, (int) sizeof(key), iv, AES_ENCRYPTION);
    if (ret != 0)
        ERROR_OUT(-6102, out);
#ifdef HAVE_AES_DECRYPT
    ret = wc_AesSetKey(dec, key, (int) sizeof(key), iv, AES_DECRYPTION);
    if (ret != 0)
        ERROR_OUT(-6103, out);
#endif

    XMEMSET(cipher, 0, AES_BLOCK_SIZE);
    ret = wc_AesCbcEncrypt(enc, cipher, msg, (int) sizeof(msg));
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &enc->asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    if (ret != 0)
        ERROR_OUT(-6104, out);
#ifdef HAVE_AES_DECRYPT
    XMEMSET(plain, 0, AES_BLOCK_SIZE);
    ret = wc_AesCbcDecrypt(dec, plain, cipher, (int) sizeof(cipher));
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &dec->asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    if (ret != 0)
        ERROR_OUT(-6105, out);
    if (XMEMCMP(plain, msg, (int) sizeof(plain))) {
        ERROR_OUT(-6106, out);
    }
#endif

    if (XMEMCMP(cipher, verify, (int) sizeof(cipher)))
        ERROR_OUT(-6107, out);

    wc_AesFree(enc);
#ifdef HAVE_AES_DECRYPT
    wc_AesFree(dec);
#endif

  out:
#ifdef WOLFSSL_SMALL_STACK
    if (enc)
        XFREE(enc, HEAP_HINT, DYNAMIC_TYPE_AES);
#ifdef HAVE_AES_DECRYPT
    if (dec)
        XFREE(dec, HEAP_HINT, DYNAMIC_TYPE_AES);
#endif
#endif
#endif /* HAVE_AES_CBC */

    return ret;
}
#endif /* WOLFSSL_AES_256 */


#ifdef HAVE_AESGCM

#ifdef WOLFSSL_AES_128
static int aesgcm_default_test_helper(byte* key, int keySz, byte* iv, int ivSz,
                byte* plain, int plainSz, byte* cipher, int cipherSz,
                byte* aad, int aadSz, byte* tag, int tagSz)
{
    int ret, enc_inited = 0, dec_inited = 0;
#ifdef WOLFSSL_SMALL_STACK
    Aes *enc = NULL;
    Aes *dec = NULL;
#else
    Aes enc[1];
    Aes dec[1];
#endif

    byte resultT[AES_BLOCK_SIZE];
    byte resultP[AES_BLOCK_SIZE * 3];
    byte resultC[AES_BLOCK_SIZE * 3];
    int  result;

#ifdef WOLFSSL_SMALL_STACK
    if ((enc = (Aes *)XMALLOC(sizeof *enc, HEAP_HINT, DYNAMIC_TYPE_AES)) == NULL)
        ERROR_OUT(-6118, out);
    if ((dec = (Aes *)XMALLOC(sizeof *dec, HEAP_HINT, DYNAMIC_TYPE_AES)) == NULL)
        ERROR_OUT(-6119, out);
#endif

    XMEMSET(resultT, 0, sizeof(resultT));
    XMEMSET(resultC, 0, sizeof(resultC));
    XMEMSET(resultP, 0, sizeof(resultP));

    if (wc_AesInit(enc, HEAP_HINT, devId) != 0)
        ERROR_OUT(-6110, out);
    else
        enc_inited = 1;
    if (wc_AesInit(dec, HEAP_HINT, devId) != 0)
        ERROR_OUT(-6111, out);
    else
        dec_inited = 1;

    result = wc_AesGcmSetKey(enc, key, keySz);
    if (result != 0)
        ERROR_OUT(-6112, out);

    /* AES-GCM encrypt and decrypt both use AES encrypt internally */
    result = wc_AesGcmEncrypt(enc, resultC, plain, plainSz, iv, ivSz,
                                        resultT, tagSz, aad, aadSz);

#if defined(WOLFSSL_ASYNC_CRYPT)
    result = wc_AsyncWait(result, &enc->asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    if (result != 0)
        ERROR_OUT(-6113, out);
    if (cipher != NULL) {
        if (XMEMCMP(cipher, resultC, cipherSz))
            ERROR_OUT(-6114, out);
    }
    if (XMEMCMP(tag, resultT, tagSz))
        ERROR_OUT(-6115, out);


#ifdef HAVE_AES_DECRYPT
    result = wc_AesGcmSetKey(dec, key, keySz);
    if (result != 0)
        ERROR_OUT(-6116, out);

    result = wc_AesGcmDecrypt(dec, resultP, resultC, cipherSz,
                      iv, ivSz, resultT, tagSz, aad, aadSz);
#if defined(WOLFSSL_ASYNC_CRYPT)
    result = wc_AsyncWait(result, &dec->asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    if (result != 0)
        ERROR_OUT(-6117, out);
    if (plain != NULL) {
        if (XMEMCMP(plain, resultP, plainSz))
            ERROR_OUT(-6118, out);
    }

#endif /* HAVE_AES_DECRYPT */

    ret = 0;

  out:

    if (enc_inited)
        wc_AesFree(enc);
    if (dec_inited)
        wc_AesFree(dec);

#ifdef WOLFSSL_SMALL_STACK
    if (enc)
        XFREE(enc, HEAP_HINT, DYNAMIC_TYPE_AES);
    if (dec)
        XFREE(dec, HEAP_HINT, DYNAMIC_TYPE_AES);
#endif

    return ret;
}
#endif


/* tests that only use 12 byte IV and 16 or less byte AAD
 * test vectors are from NIST SP 800-38D
 * https://csrc.nist.gov/Projects/Cryptographic-Algorithm-Validation-Program/CAVP-TESTING-BLOCK-CIPHER-MODES*/
WOLFSSL_TEST_SUBROUTINE int aesgcm_default_test(void)
{
#ifdef WOLFSSL_AES_128
    byte key1[] = {
        0x29, 0x8e, 0xfa, 0x1c, 0xcf, 0x29, 0xcf, 0x62,
        0xae, 0x68, 0x24, 0xbf, 0xc1, 0x95, 0x57, 0xfc
    };

    byte iv1[] = {
        0x6f, 0x58, 0xa9, 0x3f, 0xe1, 0xd2, 0x07, 0xfa,
        0xe4, 0xed, 0x2f, 0x6d
    };

    ALIGN64 byte plain1[] = {
        0xcc, 0x38, 0xbc, 0xcd, 0x6b, 0xc5, 0x36, 0xad,
        0x91, 0x9b, 0x13, 0x95, 0xf5, 0xd6, 0x38, 0x01,
        0xf9, 0x9f, 0x80, 0x68, 0xd6, 0x5c, 0xa5, 0xac,
        0x63, 0x87, 0x2d, 0xaf, 0x16, 0xb9, 0x39, 0x01
    };

    byte aad1[] = {
        0x02, 0x1f, 0xaf, 0xd2, 0x38, 0x46, 0x39, 0x73,
        0xff, 0xe8, 0x02, 0x56, 0xe5, 0xb1, 0xc6, 0xb1
    };

    ALIGN64 byte cipher1[] = {
        0xdf, 0xce, 0x4e, 0x9c, 0xd2, 0x91, 0x10, 0x3d,
        0x7f, 0xe4, 0xe6, 0x33, 0x51, 0xd9, 0xe7, 0x9d,
        0x3d, 0xfd, 0x39, 0x1e, 0x32, 0x67, 0x10, 0x46,
        0x58, 0x21, 0x2d, 0xa9, 0x65, 0x21, 0xb7, 0xdb
    };

    byte tag1[] = {
        0x54, 0x24, 0x65, 0xef, 0x59, 0x93, 0x16, 0xf7,
        0x3a, 0x7a, 0x56, 0x05, 0x09, 0xa2, 0xd9, 0xf2
    };

    byte key2[] = {
        0x01, 0x6d, 0xbb, 0x38, 0xda, 0xa7, 0x6d, 0xfe,
        0x7d, 0xa3, 0x84, 0xeb, 0xf1, 0x24, 0x03, 0x64
    };

    byte iv2[] = {
        0x07, 0x93, 0xef, 0x3a, 0xda, 0x78, 0x2f, 0x78,
        0xc9, 0x8a, 0xff, 0xe3
    };

    ALIGN64 byte plain2[] = {
        0x4b, 0x34, 0xa9, 0xec, 0x57, 0x63, 0x52, 0x4b,
        0x19, 0x1d, 0x56, 0x16, 0xc5, 0x47, 0xf6, 0xb7
    };

    ALIGN64 byte cipher2[] = {
        0x60, 0x9a, 0xa3, 0xf4, 0x54, 0x1b, 0xc0, 0xfe,
        0x99, 0x31, 0xda, 0xad, 0x2e, 0xe1, 0x5d, 0x0c
    };

    byte tag2[] = {
        0x33, 0xaf, 0xec, 0x59, 0xc4, 0x5b, 0xaf, 0x68,
        0x9a, 0x5e, 0x1b, 0x13, 0xae, 0x42, 0x36, 0x19
    };

    byte key3[] = {
        0xb0, 0x1e, 0x45, 0xcc, 0x30, 0x88, 0xaa, 0xba,
        0x9f, 0xa4, 0x3d, 0x81, 0xd4, 0x81, 0x82, 0x3f
    };

    byte iv3[] = {
        0x5a, 0x2c, 0x4a, 0x66, 0x46, 0x87, 0x13, 0x45,
        0x6a, 0x4b, 0xd5, 0xe1
    };

    byte tag3[] = {
        0x01, 0x42, 0x80, 0xf9, 0x44, 0xf5, 0x3c, 0x68,
        0x11, 0x64, 0xb2, 0xff
    };

    int ret;
    ret = aesgcm_default_test_helper(key1, sizeof(key1), iv1, sizeof(iv1),
            plain1, sizeof(plain1), cipher1, sizeof(cipher1),
            aad1, sizeof(aad1), tag1, sizeof(tag1));
    if (ret != 0) {
        return ret;
    }
    ret = aesgcm_default_test_helper(key2, sizeof(key2), iv2, sizeof(iv2),
            plain2, sizeof(plain2), cipher2, sizeof(cipher2),
            NULL, 0, tag2, sizeof(tag2));
    if (ret != 0) {
        return ret;
    }
    ret = aesgcm_default_test_helper(key3, sizeof(key3), iv3, sizeof(iv3),
            NULL, 0, NULL, 0,
            NULL, 0, tag3, sizeof(tag3));
    if (ret != 0) {
        return ret;
    }
#endif

    return 0;
}

WOLFSSL_TEST_SUBROUTINE int aesgcm_test(void)
{
#ifdef WOLFSSL_SMALL_STACK
    Aes *enc = NULL;
    Aes *dec = NULL;
#else
    Aes enc[1];
    Aes dec[1];
#endif

    /*
     * This is Test Case 16 from the document Galois/
     * Counter Mode of Operation (GCM) by McGrew and
     * Viega.
     */
    WOLFSSL_SMALL_STACK_STATIC const byte p[] =
    {
        0xd9, 0x31, 0x32, 0x25, 0xf8, 0x84, 0x06, 0xe5,
        0xa5, 0x59, 0x09, 0xc5, 0xaf, 0xf5, 0x26, 0x9a,
        0x86, 0xa7, 0xa9, 0x53, 0x15, 0x34, 0xf7, 0xda,
        0x2e, 0x4c, 0x30, 0x3d, 0x8a, 0x31, 0x8a, 0x72,
        0x1c, 0x3c, 0x0c, 0x95, 0x95, 0x68, 0x09, 0x53,
        0x2f, 0xcf, 0x0e, 0x24, 0x49, 0xa6, 0xb5, 0x25,
        0xb1, 0x6a, 0xed, 0xf5, 0xaa, 0x0d, 0xe6, 0x57,
        0xba, 0x63, 0x7b, 0x39
    };

#if defined(WOLFSSL_AES_256) || defined(WOLFSSL_AES_192)
    WOLFSSL_SMALL_STACK_STATIC const byte a[] =
    {
        0xfe, 0xed, 0xfa, 0xce, 0xde, 0xad, 0xbe, 0xef,
        0xfe, 0xed, 0xfa, 0xce, 0xde, 0xad, 0xbe, 0xef,
        0xab, 0xad, 0xda, 0xd2
    };
#endif

#ifdef WOLFSSL_AES_256
    WOLFSSL_SMALL_STACK_STATIC const byte k1[] =
    {
        0xfe, 0xff, 0xe9, 0x92, 0x86, 0x65, 0x73, 0x1c,
        0x6d, 0x6a, 0x8f, 0x94, 0x67, 0x30, 0x83, 0x08,
        0xfe, 0xff, 0xe9, 0x92, 0x86, 0x65, 0x73, 0x1c,
        0x6d, 0x6a, 0x8f, 0x94, 0x67, 0x30, 0x83, 0x08
    };

    WOLFSSL_SMALL_STACK_STATIC const byte iv1[] =
    {
        0xca, 0xfe, 0xba, 0xbe, 0xfa, 0xce, 0xdb, 0xad,
        0xde, 0xca, 0xf8, 0x88
    };
#endif /* WOLFSSL_AES_256 */

#if defined(WOLFSSL_AES_256) || defined(WOLFSSL_AES_192)
    WOLFSSL_SMALL_STACK_STATIC const byte c1[] =
    {
        0x52, 0x2d, 0xc1, 0xf0, 0x99, 0x56, 0x7d, 0x07,
        0xf4, 0x7f, 0x37, 0xa3, 0x2a, 0x84, 0x42, 0x7d,
        0x64, 0x3a, 0x8c, 0xdc, 0xbf, 0xe5, 0xc0, 0xc9,
        0x75, 0x98, 0xa2, 0xbd, 0x25, 0x55, 0xd1, 0xaa,
        0x8c, 0xb0, 0x8e, 0x48, 0x59, 0x0d, 0xbb, 0x3d,
        0xa7, 0xb0, 0x8b, 0x10, 0x56, 0x82, 0x88, 0x38,
        0xc5, 0xf6, 0x1e, 0x63, 0x93, 0xba, 0x7a, 0x0a,
        0xbc, 0xc9, 0xf6, 0x62
    };
#endif /* WOLFSSL_AES_256 || WOLFSSL_AES_192 */

    WOLFSSL_SMALL_STACK_STATIC const byte t1[] =
    {
        0x76, 0xfc, 0x6e, 0xce, 0x0f, 0x4e, 0x17, 0x68,
        0xcd, 0xdf, 0x88, 0x53, 0xbb, 0x2d, 0x55, 0x1b
    };

    /* FIPS, QAT and PIC32MZ HW Crypto only support 12-byte IV */
#if !defined(HAVE_FIPS) && \
        !defined(WOLFSSL_PIC32MZ_CRYPT) && \
        !defined(FREESCALE_LTC) && !defined(FREESCALE_MMCAU) && \
        !defined(WOLFSSL_XILINX_CRYPT) && !defined(WOLFSSL_AFALG_XILINX_AES) && \
        !defined(WOLFSSL_SILABS_SE_ACCEL) && !defined(WOLFSSL_KCAPI_AES) && \
        !(defined(WOLF_CRYPTO_CB) && \
            (defined(HAVE_INTEL_QA_SYNC) || defined(HAVE_CAVIUM_OCTEON_SYNC)))

    #define ENABLE_NON_12BYTE_IV_TEST
#ifdef WOLFSSL_AES_192
    /* Test Case 12, uses same plaintext and AAD data. */
    WOLFSSL_SMALL_STACK_STATIC const byte k2[] =
    {
        0xfe, 0xff, 0xe9, 0x92, 0x86, 0x65, 0x73, 0x1c,
        0x6d, 0x6a, 0x8f, 0x94, 0x67, 0x30, 0x83, 0x08,
        0xfe, 0xff, 0xe9, 0x92, 0x86, 0x65, 0x73, 0x1c
    };

    WOLFSSL_SMALL_STACK_STATIC const byte iv2[] =
    {
        0x93, 0x13, 0x22, 0x5d, 0xf8, 0x84, 0x06, 0xe5,
        0x55, 0x90, 0x9c, 0x5a, 0xff, 0x52, 0x69, 0xaa,
        0x6a, 0x7a, 0x95, 0x38, 0x53, 0x4f, 0x7d, 0xa1,
        0xe4, 0xc3, 0x03, 0xd2, 0xa3, 0x18, 0xa7, 0x28,
        0xc3, 0xc0, 0xc9, 0x51, 0x56, 0x80, 0x95, 0x39,
        0xfc, 0xf0, 0xe2, 0x42, 0x9a, 0x6b, 0x52, 0x54,
        0x16, 0xae, 0xdb, 0xf5, 0xa0, 0xde, 0x6a, 0x57,
        0xa6, 0x37, 0xb3, 0x9b
    };

    WOLFSSL_SMALL_STACK_STATIC const byte c2[] =
    {
        0xd2, 0x7e, 0x88, 0x68, 0x1c, 0xe3, 0x24, 0x3c,
        0x48, 0x30, 0x16, 0x5a, 0x8f, 0xdc, 0xf9, 0xff,
        0x1d, 0xe9, 0xa1, 0xd8, 0xe6, 0xb4, 0x47, 0xef,
        0x6e, 0xf7, 0xb7, 0x98, 0x28, 0x66, 0x6e, 0x45,
        0x81, 0xe7, 0x90, 0x12, 0xaf, 0x34, 0xdd, 0xd9,
        0xe2, 0xf0, 0x37, 0x58, 0x9b, 0x29, 0x2d, 0xb3,
        0xe6, 0x7c, 0x03, 0x67, 0x45, 0xfa, 0x22, 0xe7,
        0xe9, 0xb7, 0x37, 0x3b
    };

    WOLFSSL_SMALL_STACK_STATIC const byte t2[] =
    {
        0xdc, 0xf5, 0x66, 0xff, 0x29, 0x1c, 0x25, 0xbb,
        0xb8, 0x56, 0x8f, 0xc3, 0xd3, 0x76, 0xa6, 0xd9
    };
#endif /* WOLFSSL_AES_192 */
#ifdef WOLFSSL_AES_128
    /* The following is an interesting test case from the example
     * FIPS test vectors for AES-GCM. IVlen = 1 byte */
    WOLFSSL_SMALL_STACK_STATIC const byte p3[] =
    {
        0x57, 0xce, 0x45, 0x1f, 0xa5, 0xe2, 0x35, 0xa5,
        0x8e, 0x1a, 0xa2, 0x3b, 0x77, 0xcb, 0xaf, 0xe2
    };

    WOLFSSL_SMALL_STACK_STATIC const byte k3[] =
    {
        0xbb, 0x01, 0xd7, 0x03, 0x81, 0x1c, 0x10, 0x1a,
        0x35, 0xe0, 0xff, 0xd2, 0x91, 0xba, 0xf2, 0x4b
    };

    WOLFSSL_SMALL_STACK_STATIC const byte iv3[] =
    {
        0xca
    };

    WOLFSSL_SMALL_STACK_STATIC const byte c3[] =
    {
        0x6b, 0x5f, 0xb3, 0x9d, 0xc1, 0xc5, 0x7a, 0x4f,
        0xf3, 0x51, 0x4d, 0xc2, 0xd5, 0xf0, 0xd0, 0x07
    };

    WOLFSSL_SMALL_STACK_STATIC const byte a3[] =
    {
        0x40, 0xfc, 0xdc, 0xd7, 0x4a, 0xd7, 0x8b, 0xf1,
        0x3e, 0x7c, 0x60, 0x55, 0x50, 0x51, 0xdd, 0x54
    };

    WOLFSSL_SMALL_STACK_STATIC const byte t3[] =
    {
        0x06, 0x90, 0xed, 0x01, 0x34, 0xdd, 0xc6, 0x95,
        0x31, 0x2e, 0x2a, 0xf9, 0x57, 0x7a, 0x1e, 0xa6
    };
#endif /* WOLFSSL_AES_128 */
#ifdef WOLFSSL_AES_256
    int ivlen;
#endif
#endif

    byte resultT[sizeof(t1)];
    byte resultP[sizeof(p) + AES_BLOCK_SIZE];
    byte resultC[sizeof(p) + AES_BLOCK_SIZE];
    int  result = 0;
    int ret;
#ifdef WOLFSSL_AES_256
    int  alen;
    #if !defined(WOLFSSL_AFALG_XILINX_AES) && !defined(WOLFSSL_XILINX_CRYPT)
    int  plen;
    #endif
#endif

#if !defined(BENCH_EMBEDDED) && !defined(HAVE_CAVIUM)
    #if !defined(BENCH_AESGCM_LARGE)
        #define BENCH_AESGCM_LARGE 1024
    #endif
    byte *large_input = (byte *)XMALLOC(BENCH_AESGCM_LARGE, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    byte *large_output = (byte *)XMALLOC(BENCH_AESGCM_LARGE + AES_BLOCK_SIZE, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    byte *large_outdec = (byte *)XMALLOC(BENCH_AESGCM_LARGE, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);

    if ((! large_input) || (! large_output) || (! large_outdec))
        ERROR_OUT(MEMORY_E, out);

    XMEMSET(large_input, 0, BENCH_AESGCM_LARGE);
    XMEMSET(large_output, 0, BENCH_AESGCM_LARGE + AES_BLOCK_SIZE);
    XMEMSET(large_outdec, 0, BENCH_AESGCM_LARGE);
#endif

#ifdef WOLFSSL_SMALL_STACK
    if ((enc = (Aes *)XMALLOC(sizeof *enc, HEAP_HINT, DYNAMIC_TYPE_AES)) == NULL)
        ERROR_OUT(-6342, out);
    if ((dec = (Aes *)XMALLOC(sizeof *dec, HEAP_HINT, DYNAMIC_TYPE_AES)) == NULL)
        ERROR_OUT(-6343, out);
#endif

    (void)result;

    XMEMSET(resultT, 0, sizeof(resultT));
    XMEMSET(resultC, 0, sizeof(resultC));
    XMEMSET(resultP, 0, sizeof(resultP));

    if (wc_AesInit(enc, HEAP_HINT, devId) != 0) {
        ERROR_OUT(-6300, out);
    }
    if (wc_AesInit(dec, HEAP_HINT, devId) != 0) {
        ERROR_OUT(-6301, out);
    }

#ifdef WOLFSSL_AES_256
    result = wc_AesGcmSetKey(enc, k1, sizeof(k1));
    if (result != 0)
        ERROR_OUT(-6302, out);

    /* AES-GCM encrypt and decrypt both use AES encrypt internally */
    result = wc_AesGcmEncrypt(enc, resultC, p, sizeof(p), iv1, sizeof(iv1),
                                        resultT, sizeof(resultT), a, sizeof(a));
#if defined(WOLFSSL_ASYNC_CRYPT)
    result = wc_AsyncWait(result, &enc->asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    if (result != 0)
        ERROR_OUT(-6303, out);
    if (XMEMCMP(c1, resultC, sizeof(c1)))
        ERROR_OUT(-6304, out);
    if (XMEMCMP(t1, resultT, sizeof(resultT)))
        ERROR_OUT(-6305, out);

#ifdef HAVE_AES_DECRYPT
    result = wc_AesGcmSetKey(dec, k1, sizeof(k1));
    if (result != 0)
        ERROR_OUT(-6306, out);

    result = wc_AesGcmDecrypt(dec, resultP, resultC, sizeof(c1),
                      iv1, sizeof(iv1), resultT, sizeof(resultT), a, sizeof(a));
#if defined(WOLFSSL_ASYNC_CRYPT)
    result = wc_AsyncWait(result, &dec->asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    if (result != 0)
        ERROR_OUT(-6307, out);
    if (XMEMCMP(p, resultP, sizeof(p)))
        ERROR_OUT(-6308, out);
#endif /* HAVE_AES_DECRYPT */

    /* Large buffer test */
#ifdef BENCH_AESGCM_LARGE
    /* setup test buffer */
    for (alen=0; alen<BENCH_AESGCM_LARGE; alen++)
        large_input[alen] = (byte)alen;

    /* AES-GCM encrypt and decrypt both use AES encrypt internally */
    result = wc_AesGcmEncrypt(enc, large_output, large_input,
                              BENCH_AESGCM_LARGE, iv1, sizeof(iv1),
                              resultT, sizeof(resultT), a, sizeof(a));
#if defined(WOLFSSL_ASYNC_CRYPT)
    result = wc_AsyncWait(result, &enc->asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    if (result != 0)
        ERROR_OUT(-6309, out);

#ifdef HAVE_AES_DECRYPT
    result = wc_AesGcmDecrypt(dec, large_outdec, large_output,
                              BENCH_AESGCM_LARGE, iv1, sizeof(iv1), resultT,
                              sizeof(resultT), a, sizeof(a));
#if defined(WOLFSSL_ASYNC_CRYPT)
    result = wc_AsyncWait(result, &dec->asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    if (result != 0)
        ERROR_OUT(-6310, out);
    if (XMEMCMP(large_input, large_outdec, BENCH_AESGCM_LARGE))
        ERROR_OUT(-6311, out);
#endif /* HAVE_AES_DECRYPT */
#endif /* BENCH_AESGCM_LARGE */
#if defined(ENABLE_NON_12BYTE_IV_TEST) && defined(WOLFSSL_AES_256)
    /* Variable IV length test */
    for (ivlen=1; ivlen<(int)sizeof(k1); ivlen++) {
         /* AES-GCM encrypt and decrypt both use AES encrypt internally */
         result = wc_AesGcmEncrypt(enc, resultC, p, sizeof(p), k1,
                         (word32)ivlen, resultT, sizeof(resultT), a, sizeof(a));
#if defined(WOLFSSL_ASYNC_CRYPT)
        result = wc_AsyncWait(result, &enc->asyncDev, WC_ASYNC_FLAG_NONE);
#endif
        if (result != 0)
            ERROR_OUT(-6312, out);
#ifdef HAVE_AES_DECRYPT
        result = wc_AesGcmDecrypt(dec, resultP, resultC, sizeof(c1), k1,
                         (word32)ivlen, resultT, sizeof(resultT), a, sizeof(a));
#if defined(WOLFSSL_ASYNC_CRYPT)
        result = wc_AsyncWait(result, &dec->asyncDev, WC_ASYNC_FLAG_NONE);
#endif
        if (result != 0)
            ERROR_OUT(-6313, out);
#endif /* HAVE_AES_DECRYPT */
    }
#endif

#if !(defined(WOLF_CRYPTO_CB) && defined(HAVE_INTEL_QA_SYNC))
    /* Variable authenticated data length test */
    for (alen=0; alen<(int)sizeof(p); alen++) {
         /* AES-GCM encrypt and decrypt both use AES encrypt internally */
         result = wc_AesGcmEncrypt(enc, resultC, p, sizeof(p), iv1,
                        sizeof(iv1), resultT, sizeof(resultT), p, (word32)alen);
#if defined(WOLFSSL_ASYNC_CRYPT)
        result = wc_AsyncWait(result, &enc->asyncDev, WC_ASYNC_FLAG_NONE);
#endif
        if (result != 0)
            ERROR_OUT(-6314, out);
#ifdef HAVE_AES_DECRYPT
        result = wc_AesGcmDecrypt(dec, resultP, resultC, sizeof(c1), iv1,
                        sizeof(iv1), resultT, sizeof(resultT), p, (word32)alen);
#if defined(WOLFSSL_ASYNC_CRYPT)
        result = wc_AsyncWait(result, &dec->asyncDev, WC_ASYNC_FLAG_NONE);
#endif
        if (result != 0)
            ERROR_OUT(-6315, out);
#endif /* HAVE_AES_DECRYPT */
    }
#endif

#if !defined(WOLFSSL_AFALG_XILINX_AES) && !defined(WOLFSSL_XILINX_CRYPT)
#ifdef BENCH_AESGCM_LARGE
    /* Variable plain text length test */
    for (plen=1; plen<BENCH_AESGCM_LARGE; plen++) {
        /* AES-GCM encrypt and decrypt both use AES encrypt internally */
        result = wc_AesGcmEncrypt(enc, large_output, large_input,
                                  plen, iv1, sizeof(iv1), resultT,
                                  sizeof(resultT), a, sizeof(a));
#if defined(WOLFSSL_ASYNC_CRYPT)
        result = wc_AsyncWait(result, &enc->asyncDev, WC_ASYNC_FLAG_NONE);
#endif
        if (result != 0)
            ERROR_OUT(-6316, out);

#ifdef HAVE_AES_DECRYPT
        result = wc_AesGcmDecrypt(dec, large_outdec, large_output,
                                  plen, iv1, sizeof(iv1), resultT,
                                  sizeof(resultT), a, sizeof(a));
#if defined(WOLFSSL_ASYNC_CRYPT)
        result = wc_AsyncWait(result, &dec->asyncDev, WC_ASYNC_FLAG_NONE);
#endif
        if (result != 0)
            ERROR_OUT(-6317, out);
#endif /* HAVE_AES_DECRYPT */
    }
#else /* BENCH_AESGCM_LARGE */
    /* Variable plain text length test */
    for (plen=1; plen<(int)sizeof(p); plen++) {
         /* AES-GCM encrypt and decrypt both use AES encrypt internally */
         result = wc_AesGcmEncrypt(enc, resultC, p, (word32)plen, iv1,
                           sizeof(iv1), resultT, sizeof(resultT), a, sizeof(a));
#if defined(WOLFSSL_ASYNC_CRYPT)
        result = wc_AsyncWait(result, &enc->asyncDev, WC_ASYNC_FLAG_NONE);
#endif
        if (result != 0)
            ERROR_OUT(-6318, out);
#ifdef HAVE_AES_DECRYPT
        result = wc_AesGcmDecrypt(dec, resultP, resultC, (word32)plen, iv1,
                           sizeof(iv1), resultT, sizeof(resultT), a, sizeof(a));
#if defined(WOLFSSL_ASYNC_CRYPT)
        result = wc_AsyncWait(result, &dec->asyncDev, WC_ASYNC_FLAG_NONE);
#endif
        if (result != 0)
            ERROR_OUT(-6319, out);
#endif /* HAVE_AES_DECRYPT */
    }
#endif /* BENCH_AESGCM_LARGE */
#endif
#endif /* WOLFSSL_AES_256 */

    /* test with IV != 12 bytes */
#ifdef ENABLE_NON_12BYTE_IV_TEST
    XMEMSET(resultT, 0, sizeof(resultT));
    XMEMSET(resultC, 0, sizeof(resultC));
    XMEMSET(resultP, 0, sizeof(resultP));

#ifdef WOLFSSL_AES_192
    wc_AesGcmSetKey(enc, k2, sizeof(k2));
    /* AES-GCM encrypt and decrypt both use AES encrypt internally */
    result = wc_AesGcmEncrypt(enc, resultC, p, sizeof(p), iv2, sizeof(iv2),
                                        resultT, sizeof(resultT), a, sizeof(a));
#if defined(WOLFSSL_ASYNC_CRYPT)
    result = wc_AsyncWait(result, &enc->asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    if (result != 0)
        ERROR_OUT(-6320, out);
    if (XMEMCMP(c2, resultC, sizeof(c2)))
        ERROR_OUT(-6321, out);
    if (XMEMCMP(t2, resultT, sizeof(resultT)))
        ERROR_OUT(-6322, out);

#ifdef HAVE_AES_DECRYPT
    result = wc_AesGcmDecrypt(enc, resultP, resultC, sizeof(c1),
                      iv2, sizeof(iv2), resultT, sizeof(resultT), a, sizeof(a));
#if defined(WOLFSSL_ASYNC_CRYPT)
    result = wc_AsyncWait(result, &enc->asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    if (result != 0)
        ERROR_OUT(-6323, out);
    if (XMEMCMP(p, resultP, sizeof(p)))
        ERROR_OUT(-6324, out);
#endif /* HAVE_AES_DECRYPT */

    XMEMSET(resultT, 0, sizeof(resultT));
    XMEMSET(resultC, 0, sizeof(resultC));
    XMEMSET(resultP, 0, sizeof(resultP));
#endif /* WOLFSSL_AES_192 */
#ifdef WOLFSSL_AES_128
    wc_AesGcmSetKey(enc, k3, sizeof(k3));
    /* AES-GCM encrypt and decrypt both use AES encrypt internally */
    result = wc_AesGcmEncrypt(enc, resultC, p3, sizeof(p3), iv3, sizeof(iv3),
                                        resultT, sizeof(t3), a3, sizeof(a3));
#if defined(WOLFSSL_ASYNC_CRYPT)
    result = wc_AsyncWait(result, &enc->asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    if (result != 0)
        ERROR_OUT(-6325, out);
    if (XMEMCMP(c3, resultC, sizeof(c3)))
        ERROR_OUT(-6326, out);
    if (XMEMCMP(t3, resultT, sizeof(t3)))
        ERROR_OUT(-6327, out);

#ifdef HAVE_AES_DECRYPT
    result = wc_AesGcmDecrypt(enc, resultP, resultC, sizeof(c3),
                      iv3, sizeof(iv3), resultT, sizeof(t3), a3, sizeof(a3));
#if defined(WOLFSSL_ASYNC_CRYPT)
    result = wc_AsyncWait(result, &enc->asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    if (result != 0)
        ERROR_OUT(-6328, out);
    if (XMEMCMP(p3, resultP, sizeof(p3)))
        ERROR_OUT(-6329, out);
#endif /* HAVE_AES_DECRYPT */
#endif /* WOLFSSL_AES_128 */
#endif /* ENABLE_NON_12BYTE_IV_TEST */

#if defined(WOLFSSL_AES_256) && !defined(WOLFSSL_AFALG_XILINX_AES) && \
    !defined(WOLFSSL_XILINX_CRYPT) && \
    !(defined(WOLF_CRYPTO_CB) && \
        defined(HAVE_INTEL_QA_SYNC) || defined(HAVE_CAVIUM_OCTEON_SYNC))
    XMEMSET(resultT, 0, sizeof(resultT));
    XMEMSET(resultC, 0, sizeof(resultC));
    XMEMSET(resultP, 0, sizeof(resultP));

    wc_AesGcmSetKey(enc, k1, sizeof(k1));
    /* AES-GCM encrypt and decrypt both use AES encrypt internally */
    result = wc_AesGcmEncrypt(enc, resultC, p, sizeof(p), iv1, sizeof(iv1),
                                resultT + 1, sizeof(resultT) - 1, a, sizeof(a));
#if defined(WOLFSSL_ASYNC_CRYPT)
    result = wc_AsyncWait(result, &enc->asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    if (result != 0)
        ERROR_OUT(-6330, out);
    if (XMEMCMP(c1, resultC, sizeof(c1)))
        ERROR_OUT(-6331, out);
    if (XMEMCMP(t1, resultT + 1, sizeof(resultT) - 1))
        ERROR_OUT(-6332, out);

#ifdef HAVE_AES_DECRYPT
    result = wc_AesGcmDecrypt(enc, resultP, resultC, sizeof(p),
              iv1, sizeof(iv1), resultT + 1, sizeof(resultT) - 1, a, sizeof(a));
#if defined(WOLFSSL_ASYNC_CRYPT)
    result = wc_AsyncWait(result, &enc->asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    if (result != 0)
        ERROR_OUT(-6333, out);
    if (XMEMCMP(p, resultP, sizeof(p)))
        ERROR_OUT(-6334, out);
#endif /* HAVE_AES_DECRYPT */
#endif /* WOLFSSL_AES_256 */

#if !defined(HAVE_FIPS) || \
    (defined(HAVE_FIPS_VERSION) && (HAVE_FIPS_VERSION >= 2))
    /* Test encrypt with internally generated IV */
#if defined(WOLFSSL_AES_256) && !(defined(WC_NO_RNG) || defined(HAVE_SELFTEST)) \
    && !(defined(WOLF_CRYPTO_CB) && defined(HAVE_CAVIUM_OCTEON_SYNC))
    {
        WC_RNG rng;
        byte randIV[12];

        result = wc_InitRng_ex(&rng, HEAP_HINT, devId);
        if (result != 0)
            ERROR_OUT(-6335, out);

        XMEMSET(randIV, 0, sizeof(randIV));
        XMEMSET(resultT, 0, sizeof(resultT));
        XMEMSET(resultC, 0, sizeof(resultC));
        XMEMSET(resultP, 0, sizeof(resultP));

        wc_AesGcmSetKey(enc, k1, sizeof(k1));
        result = wc_AesGcmSetIV(enc, sizeof(randIV), NULL, 0, &rng);
        if (result != 0)
            ERROR_OUT(-6336, out);

        result = wc_AesGcmEncrypt_ex(enc,
                        resultC, p, sizeof(p),
                        randIV, sizeof(randIV),
                        resultT, sizeof(resultT),
                        a, sizeof(a));
    #if defined(WOLFSSL_ASYNC_CRYPT)
        result = wc_AsyncWait(result, &enc->asyncDev, WC_ASYNC_FLAG_NONE);
    #endif
        if (result != 0)
            ERROR_OUT(-6337, out);

        /* Check the IV has been set. */
        {
            word32 i, ivSum = 0;

            for (i = 0; i < sizeof(randIV); i++)
                ivSum += randIV[i];
            if (ivSum == 0)
                ERROR_OUT(-6338, out);
        }

#ifdef HAVE_AES_DECRYPT
        wc_AesGcmSetKey(dec, k1, sizeof(k1));
        result = wc_AesGcmSetIV(dec, sizeof(randIV), NULL, 0, &rng);
        if (result != 0)
            ERROR_OUT(-6339, out);

        result = wc_AesGcmDecrypt(dec,
                          resultP, resultC, sizeof(c1),
                          randIV, sizeof(randIV),
                          resultT, sizeof(resultT),
                          a, sizeof(a));
#if defined(WOLFSSL_ASYNC_CRYPT)
        result = wc_AsyncWait(result, &dec->asyncDev, WC_ASYNC_FLAG_NONE);
#endif
        if (result != 0)
            ERROR_OUT(-6340, out);
        if (XMEMCMP(p, resultP, sizeof(p)))
            ERROR_OUT(-6341, out);
#endif /* HAVE_AES_DECRYPT */

        wc_FreeRng(&rng);
    }
#endif /* WOLFSSL_AES_256 && !(WC_NO_RNG || HAVE_SELFTEST) */
#endif /* HAVE_FIPS_VERSION >= 2 */

#if !defined(WOLFSSL_AFALG_XILINX_AES) && !defined(WOLFSSL_XILINX_CRYPT)
#ifdef WOLFSSL_AES_256
#ifdef WOLFSSL_AESGCM_STREAM
    result = wc_AesGcmEncryptInit(enc, k1, sizeof(k1), iv1, sizeof(iv1));
    if (result != 0)
        ERROR_OUT(-6360, out);
    result = wc_AesGcmEncryptUpdate(enc, resultC, p, sizeof(p), a, sizeof(a));
    if (result != 0)
        ERROR_OUT(-6361, out);
    result = wc_AesGcmEncryptFinal(enc, resultT, sizeof(resultT));
    if (result != 0)
        ERROR_OUT(-6362, out);
    if (XMEMCMP(resultC, c1, sizeof(c1)) != 0)
        ERROR_OUT(-6363, out);
    if (XMEMCMP(resultT, t1, sizeof(t1)) != 0)
        ERROR_OUT(-6364, out);

#ifdef HAVE_AES_DECRYPT
    result = wc_AesGcmDecryptInit(enc, k1, sizeof(k1), iv1, sizeof(iv1));
    if (result != 0)
        ERROR_OUT(-6370, out);
    result = wc_AesGcmDecryptUpdate(enc, resultP, c1, sizeof(c1), a, sizeof(a));
    if (result != 0)
        ERROR_OUT(-6371, out);
    result = wc_AesGcmDecryptFinal(enc, t1, sizeof(t1));
    if (result != 0)
        ERROR_OUT(-6372, out);
    if (XMEMCMP(resultP, p, sizeof(p)) != 0)
        ERROR_OUT(-6373, out);
#endif

    /* alen is the size to pass in with each update. */
    for (alen = 1; alen < AES_BLOCK_SIZE + 1; alen++) {
        result = wc_AesGcmEncryptInit(enc, k1, sizeof(k1), iv1, sizeof(iv1));
        if (result != 0)
            ERROR_OUT(-6380, out);

        /* plen is the offset into AAD to update with. */
        for (plen = 0; plen < (int)sizeof(a); plen += alen)  {
            int len = sizeof(a) - plen;
            if (len > alen) len = alen;
            result = wc_AesGcmEncryptUpdate(enc, NULL, NULL, 0, a + plen, len);
            if (result != 0)
                ERROR_OUT(-6381, out);
        }
        /* plen is the offset into plaintext to update with. */
        for (plen = 0; plen < (int)sizeof(p); plen += alen)  {
            int len = sizeof(p) - plen;
            if (len > alen) len = alen;
            result = wc_AesGcmEncryptUpdate(enc, resultC + plen, p + plen, len,
                NULL, 0);
            if (result != 0)
                ERROR_OUT(-6382, out);
        }
        result = wc_AesGcmEncryptFinal(enc, resultT, sizeof(resultT));
        if (result != 0)
            ERROR_OUT(-6383, out);
        if (XMEMCMP(resultC, c1, sizeof(c1)) != 0)
            ERROR_OUT(-6384, out);
        if (XMEMCMP(resultT, t1, sizeof(t1)) != 0)
            ERROR_OUT(-6385, out);
    }

#ifdef HAVE_AES_DECRYPT
    for (alen = 1; alen < AES_BLOCK_SIZE + 1; alen++) {
        result = wc_AesGcmDecryptInit(enc, k1, sizeof(k1), iv1, sizeof(iv1));
        if (result != 0)
            ERROR_OUT(-6390, out);

        /* plen is the offset into AAD to update with. */
        for (plen = 0; plen < (int)sizeof(a); plen += alen)  {
            int len = sizeof(a) - plen;
            if (len > alen) len = alen;
            result = wc_AesGcmDecryptUpdate(enc, NULL, NULL, 0, a + plen, len);
            if (result != 0)
                ERROR_OUT(-6391, out);
        }
        /* plen is the offset into cipher text to update with. */
        for (plen = 0; plen < (int)sizeof(c1); plen += alen)  {
            int len = sizeof(c1) - plen;
            if (len > alen) len = alen;
            result = wc_AesGcmDecryptUpdate(enc, resultP + plen, c1 + plen, len,
                NULL, 0);
            if (result != 0)
                ERROR_OUT(-6392, out);
        }
        result = wc_AesGcmDecryptFinal(enc, t1, sizeof(t1));
        if (result != 0)
            ERROR_OUT(-6393, out);
        if (XMEMCMP(resultP, p, sizeof(p)) != 0)
            ERROR_OUT(-6394, out);
    }
#endif /* HAVE_AES_DECRYPT */
#endif /* WOLFSSL_AESGCM_STREAM */
#endif /* WOLFSSL_AES_256 */
#endif /* !WOLFSSL_AFALG_XILINX_AES && !WOLFSSL_XILINX_CRYPT */

    wc_AesFree(enc);
    wc_AesFree(dec);

    ret = 0;

  out:

#if !defined(BENCH_EMBEDDED) && !defined(HAVE_CAVIUM)
    if (large_input)
        XFREE(large_input, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    if (large_output)
        XFREE(large_output, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    if (large_outdec)
        XFREE(large_outdec, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
#endif


#ifdef WOLFSSL_SMALL_STACK
    if (enc)
        XFREE(enc, HEAP_HINT, DYNAMIC_TYPE_AES);
    if (dec)
        XFREE(dec, HEAP_HINT, DYNAMIC_TYPE_AES);
#endif

    return ret;
}

#ifdef WOLFSSL_AES_128
WOLFSSL_TEST_SUBROUTINE int gmac_test(void)
{
    int ret;
#ifdef WOLFSSL_SMALL_STACK
    Gmac *gmac;
#else
    Gmac gmac[1];
#endif

    WOLFSSL_SMALL_STACK_STATIC const byte k1[] =
    {
        0x89, 0xc9, 0x49, 0xe9, 0xc8, 0x04, 0xaf, 0x01,
        0x4d, 0x56, 0x04, 0xb3, 0x94, 0x59, 0xf2, 0xc8
    };
    WOLFSSL_SMALL_STACK_STATIC const byte iv1[] =
    {
        0xd1, 0xb1, 0x04, 0xc8, 0x15, 0xbf, 0x1e, 0x94,
        0xe2, 0x8c, 0x8f, 0x16
    };
    WOLFSSL_SMALL_STACK_STATIC const byte a1[] =
    {
       0x82, 0xad, 0xcd, 0x63, 0x8d, 0x3f, 0xa9, 0xd9,
       0xf3, 0xe8, 0x41, 0x00, 0xd6, 0x1e, 0x07, 0x77
    };
    WOLFSSL_SMALL_STACK_STATIC const byte t1[] =
    {
        0x88, 0xdb, 0x9d, 0x62, 0x17, 0x2e, 0xd0, 0x43,
        0xaa, 0x10, 0xf1, 0x6d, 0x22, 0x7d, 0xc4, 0x1b
    };

#if (!defined(HAVE_FIPS) ||                                             \
     (defined(HAVE_FIPS_VERSION) && (HAVE_FIPS_VERSION >= 2)))

    /* FIPS builds only allow 16-byte auth tags. */
    /* This sample uses a 15-byte auth tag. */
    WOLFSSL_SMALL_STACK_STATIC const byte k2[] =
    {
        0x40, 0xf7, 0xec, 0xb2, 0x52, 0x6d, 0xaa, 0xd4,
        0x74, 0x25, 0x1d, 0xf4, 0x88, 0x9e, 0xf6, 0x5b
    };
    WOLFSSL_SMALL_STACK_STATIC const byte iv2[] =
    {
        0xee, 0x9c, 0x6e, 0x06, 0x15, 0x45, 0x45, 0x03,
        0x1a, 0x60, 0x24, 0xa7
    };
    WOLFSSL_SMALL_STACK_STATIC const byte a2[] =
    {
        0x94, 0x81, 0x2c, 0x87, 0x07, 0x4e, 0x15, 0x18,
        0x34, 0xb8, 0x35, 0xaf, 0x1c, 0xa5, 0x7e, 0x56
    };
    WOLFSSL_SMALL_STACK_STATIC const byte t2[] =
    {
        0xc6, 0x81, 0x79, 0x8e, 0x3d, 0xda, 0xb0, 0x9f,
        0x8d, 0x83, 0xb0, 0xbb, 0x14, 0xb6, 0x91
    };
#endif

    byte tag[16];

#ifdef WOLFSSL_SMALL_STACK
    if ((gmac = (Gmac *)XMALLOC(sizeof *gmac, HEAP_HINT, DYNAMIC_TYPE_AES)) == NULL)
        return -6409;
#endif

    XMEMSET(gmac, 0, sizeof *gmac); /* clear context */
    (void)wc_AesInit(&gmac->aes, HEAP_HINT, INVALID_DEVID); /* Make sure devId updated */
    XMEMSET(tag, 0, sizeof(tag));
    wc_GmacSetKey(gmac, k1, sizeof(k1));
    wc_GmacUpdate(gmac, iv1, sizeof(iv1), a1, sizeof(a1), tag, sizeof(t1));
    if (XMEMCMP(t1, tag, sizeof(t1)) != 0)
        ERROR_OUT(-6400, out);

#if (!defined(HAVE_FIPS) ||                                  \
     (defined(HAVE_FIPS_VERSION) && (HAVE_FIPS_VERSION >= 2)) )

    XMEMSET(tag, 0, sizeof(tag));
    wc_GmacSetKey(gmac, k2, sizeof(k2));
    wc_GmacUpdate(gmac, iv2, sizeof(iv2), a2, sizeof(a2), tag, sizeof(t2));
    if (XMEMCMP(t2, tag, sizeof(t2)) != 0)
        ERROR_OUT(-6401, out);

#if !defined(WC_NO_RNG) && !defined(HAVE_SELFTEST) && !defined(NO_AES_DECRYPT)
    {
        WOLFSSL_SMALL_STACK_STATIC const byte badT[] =
        {
            0xde, 0xad, 0xbe, 0xef, 0x17, 0x2e, 0xd0, 0x43,
            0xaa, 0x10, 0xf1, 0x6d, 0x22, 0x7d, 0xc4, 0x1b
        };

        WC_RNG rng;
        byte iv[12];

        #ifndef HAVE_FIPS
            if (wc_InitRng_ex(&rng, HEAP_HINT, devId) != 0)
                ERROR_OUT(-6402, out);
        #else
            if (wc_InitRng(&rng) != 0)
                ERROR_OUT(-6403, out);
        #endif

        if (wc_GmacVerify(k1, sizeof(k1), iv1, sizeof(iv1), a1, sizeof(a1),
                    t1, sizeof(t1)) != 0)
            ERROR_OUT(-6404, out);
        if (wc_GmacVerify(k1, sizeof(k1), iv1, sizeof(iv1), a1, sizeof(a1),
                    badT, sizeof(badT)) != AES_GCM_AUTH_E)
            ERROR_OUT(-6405, out);
        if (wc_GmacVerify(k2, sizeof(k2), iv2, sizeof(iv2), a2, sizeof(a2),
                    t2, sizeof(t2)) != 0)
            ERROR_OUT(-6406, out);

        XMEMSET(tag, 0, sizeof(tag));
        XMEMSET(iv, 0, sizeof(iv));
        if (wc_Gmac(k1, sizeof(k1), iv, sizeof(iv), a1, sizeof(a1),
                    tag, sizeof(tag), &rng) != 0)
            ERROR_OUT(-6407, out);
        if (wc_GmacVerify(k1, sizeof(k1), iv, sizeof(iv), a1, sizeof(a1),
                    tag, sizeof(tag)) != 0)
            ERROR_OUT(-6408, out);
        wc_FreeRng(&rng);
    }
#endif /* !WC_NO_RNG && !HAVE_SELFTEST && !NO_AES_DECRYPT */
#endif /* HAVE_FIPS */

    ret = 0;

  out:
    wc_AesFree(&gmac->aes);
#ifdef WOLFSSL_SMALL_STACK
    XFREE(gmac, HEAP_HINT, DYNAMIC_TYPE_AES);
#endif

    return ret;
}
#endif /* WOLFSSL_AES_128 */
#endif /* HAVE_AESGCM */

#if defined(HAVE_AESCCM) && defined(WOLFSSL_AES_128)
WOLFSSL_TEST_SUBROUTINE int aesccm_test(void)
{
    int ret;
#ifdef WOLFSSL_SMALL_STACK
    Aes *enc;
#else
    Aes enc[1];
#endif

    /* key */
    WOLFSSL_SMALL_STACK_STATIC const byte k[] =
    {
        0xc0, 0xc1, 0xc2, 0xc3, 0xc4, 0xc5, 0xc6, 0xc7,
        0xc8, 0xc9, 0xca, 0xcb, 0xcc, 0xcd, 0xce, 0xcf
    };

    /* nonce */
    WOLFSSL_SMALL_STACK_STATIC const byte iv[] =
    {
        0x00, 0x00, 0x00, 0x03, 0x02, 0x01, 0x00, 0xa0,
        0xa1, 0xa2, 0xa3, 0xa4, 0xa5
    };

    /* plaintext */
    WOLFSSL_SMALL_STACK_STATIC const byte p[] =
    {
        0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
        0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
        0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e
    };
    /* plaintext - long */
    WOLFSSL_SMALL_STACK_STATIC const byte pl[] =
    {
        0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
        0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
        0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f,
        0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27,
        0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f,
        0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,
        0x38, 0x39, 0x3a, 0x3b, 0x3c, 0x3d, 0x3e, 0x3f,
        0x40, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47,
        0x48, 0x49, 0x4a, 0x4b, 0x4c, 0x4d, 0x4e, 0x4f,
        0x50
    };


    WOLFSSL_SMALL_STACK_STATIC const byte a[] =
    {
        0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07
    };

    /* ciphertext */
    WOLFSSL_SMALL_STACK_STATIC const byte c[] =
    {
        0x58, 0x8c, 0x97, 0x9a, 0x61, 0xc6, 0x63, 0xd2,
        0xf0, 0x66, 0xd0, 0xc2, 0xc0, 0xf9, 0x89, 0x80,
        0x6d, 0x5f, 0x6b, 0x61, 0xda, 0xc3, 0x84
    };
    /* tag - authentication */
    WOLFSSL_SMALL_STACK_STATIC const byte t[] =
    {
        0x17, 0xe8, 0xd1, 0x2c, 0xfd, 0xf9, 0x26, 0xe0
    };
    /* ciphertext - long */
    WOLFSSL_SMALL_STACK_STATIC const byte cl[] =
    {
        0x58, 0x8c, 0x97, 0x9a, 0x61, 0xc6, 0x63, 0xd2,
        0xf0, 0x66, 0xd0, 0xc2, 0xc0, 0xf9, 0x89, 0x80,
        0x6d, 0x5f, 0x6b, 0x61, 0xda, 0xc3, 0x84, 0xe0,
        0x44, 0x2d, 0xbe, 0x25, 0xfa, 0x48, 0x2b, 0xa8,
        0x36, 0x0b, 0xbf, 0x01, 0xc0, 0x12, 0x45, 0xa4,
        0x82, 0x9f, 0x20, 0x6c, 0xc3, 0xd6, 0xae, 0x5b,
        0x54, 0x8d, 0xd0, 0xb1, 0x69, 0x2c, 0xec, 0x5e,
        0x95, 0xa5, 0x6b, 0x48, 0xc3, 0xc6, 0xc8, 0x9e,
        0xc7, 0x92, 0x98, 0x9d, 0x26, 0x7d, 0x2a, 0x10,
        0x0b
    };
    /* tag - authentication - long */
    WOLFSSL_SMALL_STACK_STATIC const byte tl[] =
    {
        0x89, 0xd8, 0xd2, 0x02, 0xc5, 0xcf, 0xae, 0xf4
    };

    /* tag - authentication - empty plaintext */
    WOLFSSL_SMALL_STACK_STATIC const byte t_empty[] =
    {
        0xe4, 0x28, 0x8a, 0xc3, 0x78, 0x00, 0x0f, 0xf5
    };

    byte t2[sizeof(t)];
    byte p2[sizeof(p)];
    byte c2[sizeof(c)];
    byte iv2[sizeof(iv)];
    byte pl2[sizeof(pl)];
    byte cl2[sizeof(cl)];
    byte tl2[sizeof(tl)];
    byte t_empty2[sizeof(t_empty)];

    int result;

#ifdef WOLFSSL_SMALL_STACK
    if ((enc = (Aes *)XMALLOC(sizeof *enc, HEAP_HINT, DYNAMIC_TYPE_AES)) == NULL)
        return -6521;
#endif

    XMEMSET(enc, 0, sizeof *enc); /* clear context */
    XMEMSET(t2, 0, sizeof(t2));
    XMEMSET(c2, 0, sizeof(c2));
    XMEMSET(p2, 0, sizeof(p2));

    result = wc_AesInit(enc, HEAP_HINT, devId);
    if (result != 0)
        ERROR_OUT(-6499, out);

    result = wc_AesCcmSetKey(enc, k, sizeof(k));
    if (result != 0)
        ERROR_OUT(-6500, out);

    /* AES-CCM encrypt and decrypt both use AES encrypt internally */
    result = wc_AesCcmEncrypt(enc, c2, p, sizeof(c2), iv, sizeof(iv),
                                                 t2, sizeof(t2), a, sizeof(a));
    if (result != 0)
        ERROR_OUT(-6501, out);
    if (XMEMCMP(c, c2, sizeof(c2)))
        ERROR_OUT(-6502, out);
    if (XMEMCMP(t, t2, sizeof(t2)))
        ERROR_OUT(-6503, out);

    result = wc_AesCcmDecrypt(enc, p2, c2, sizeof(p2), iv, sizeof(iv),
                                                 t2, sizeof(t2), a, sizeof(a));
    if (result != 0)
        ERROR_OUT(-6504, out);
    if (XMEMCMP(p, p2, sizeof(p2)))
        ERROR_OUT(-6505, out);

    /* Test the authentication failure */
    t2[0]++; /* Corrupt the authentication tag. */
    result = wc_AesCcmDecrypt(enc, p2, c, sizeof(p2), iv, sizeof(iv),
                                                 t2, sizeof(t2), a, sizeof(a));
    if (result == 0)
        ERROR_OUT(-6506, out);

    /* Clear c2 to compare against p2. p2 should be set to zero in case of
     * authentication fail. */
    XMEMSET(c2, 0, sizeof(c2));
    if (XMEMCMP(p2, c2, sizeof(p2)))
        ERROR_OUT(-6507, out);
    wc_AesFree(enc);

    XMEMSET(enc, 0, sizeof(Aes)); /* clear context */
    XMEMSET(t2, 0, sizeof(t2));
    XMEMSET(c2, 0, sizeof(c2));
    XMEMSET(p2, 0, sizeof(p2));
    XMEMSET(iv2, 0, sizeof(iv2));

#ifndef HAVE_SELFTEST
    /* selftest build does not have wc_AesCcmSetNonce() or
     * wc_AesCcmEncrypt_ex() */
    if (wc_AesCcmSetKey(enc, k, sizeof(k)) != 0)
        ERROR_OUT(-6508, out);

    if (wc_AesCcmSetNonce(enc, iv, sizeof(iv)) != 0)
        ERROR_OUT(-6509, out);
    if (wc_AesCcmEncrypt_ex(enc, c2, p, sizeof(c2), iv2, sizeof(iv2),
                            t2, sizeof(t2), a, sizeof(a)) != 0)
        ERROR_OUT(-6510, out);
    if (XMEMCMP(iv, iv2, sizeof(iv2)))
        ERROR_OUT(-6511, out);
    if (XMEMCMP(c, c2, sizeof(c2)))
        ERROR_OUT(-6512, out);
    if (XMEMCMP(t, t2, sizeof(t2)))
        ERROR_OUT(-6513, out);
#endif

#if !defined(HAVE_FIPS) && !defined(HAVE_SELFTEST)
    /* test fail on invalid IV sizes */
    result = wc_AesCcmSetKey(enc, k, sizeof(k));
    if (result != 0)
        ERROR_OUT(-6514, out);

    /* AES-CCM encrypt and decrypt both use AES encrypt internally */
    result = wc_AesCcmEncrypt(enc, c2, p, sizeof(c2), iv, sizeof(iv),
            t2, 1, a, sizeof(a));
    if (result == 0) {
        ERROR_OUT(-6515, out);
    }
#endif

    /* AES-CCM encrypt and decrypt both use AES encrypt internally */
    result = wc_AesCcmEncrypt(enc, cl2, pl, sizeof(cl2), iv, sizeof(iv),
                                                tl2, sizeof(tl2), a, sizeof(a));
    if (result != 0)
        ERROR_OUT(-6516, out);
    if (XMEMCMP(cl, cl2, sizeof(cl2)))
        ERROR_OUT(-6517, out);
    if (XMEMCMP(tl, tl2, sizeof(tl2)))
        ERROR_OUT(-6518, out);

    result = wc_AesCcmDecrypt(enc, pl2, cl2, sizeof(pl2), iv, sizeof(iv),
                                                tl2, sizeof(tl2), a, sizeof(a));
    if (result != 0)
        ERROR_OUT(-6519, out);
    if (XMEMCMP(pl, pl2, sizeof(pl2)))
        ERROR_OUT(-6520, out);

    /* test empty message as null input or output with nonzero inSz. */
    result = wc_AesCcmEncrypt(enc, pl2 /* out */, NULL /* in */, 1 /* inSz */,
                              iv, sizeof(iv), t_empty2, sizeof(t_empty2),
                              a, sizeof(a));
    if (result != BAD_FUNC_ARG)
        ERROR_OUT(-6527, out);
    result = wc_AesCcmEncrypt(enc, NULL /* out */, (const byte *)"" /* in */, 1 /* inSz */,
                              iv, sizeof(iv), t_empty2, sizeof(t_empty2),
                              a, sizeof(a));
    if (result != BAD_FUNC_ARG)
        ERROR_OUT(-6528, out);
    result = wc_AesCcmDecrypt(enc, pl2, NULL /* in */, 1 /* inSz */,
                              iv, sizeof(iv), t_empty2, sizeof(t_empty2), a,
                              sizeof(a));
    if (result != BAD_FUNC_ARG)
        ERROR_OUT(-6529, out);
    result = wc_AesCcmDecrypt(enc, NULL /* out */, (const byte *)"" /* in */, 1 /* inSz */,
                              iv, sizeof(iv), t_empty2, sizeof(t_empty2), a,
                              sizeof(a));
    if (result != BAD_FUNC_ARG)
        ERROR_OUT(-6530, out);

    /* test empty message as null input and output with zero inSz --
     * must either succeed, or fail early with BAD_FUNC_ARG.
     */
    result = wc_AesCcmEncrypt(enc, NULL /* out */, NULL /* in */, 0 /* inSz */,
                              iv, sizeof(iv), t_empty2, sizeof(t_empty2),
                              a, sizeof(a));
    if (result != BAD_FUNC_ARG) {
        if (result != 0)
            ERROR_OUT(-6521, out);
        if (XMEMCMP(t_empty, t_empty2, sizeof(t_empty2)))
            ERROR_OUT(-6522, out);

        result = wc_AesCcmDecrypt(enc, NULL /* out */, NULL /* in */,
                                  0 /* inSz */, iv, sizeof(iv), t_empty2,
                                  sizeof(t_empty2), a, sizeof(a));
        if (result != 0)
            ERROR_OUT(-6523, out);
    }

    /* test empty message as zero-length string -- must work. */
    result = wc_AesCcmEncrypt(enc, pl2, (const byte *)"", 0 /* inSz */, iv,
                              sizeof(iv), t_empty2, sizeof(t_empty2), a,
                              sizeof(a));
    if (result != 0)
        ERROR_OUT(-6524, out);
    if (XMEMCMP(t_empty, t_empty2, sizeof(t_empty2)))
        ERROR_OUT(-6525, out);

    result = wc_AesCcmDecrypt(enc, pl2, (const byte *)"", 0 /* inSz */,
                              iv, sizeof(iv), t_empty2, sizeof(t_empty2), a,
                              sizeof(a));
    if (result != 0)
        ERROR_OUT(-6526, out);

    wc_AesFree(enc);

    ret = 0;

  out:
#ifdef WOLFSSL_SMALL_STACK
    XFREE(enc, HEAP_HINT, DYNAMIC_TYPE_AES);
#endif

    return ret;
}
#endif /* HAVE_AESCCM WOLFSSL_AES_128 */


#ifdef HAVE_AES_KEYWRAP

#define MAX_KEYWRAP_TEST_OUTLEN 40
#define MAX_KEYWRAP_TEST_PLAINLEN 32

typedef struct keywrapVector {
    const byte* kek;
    const byte* data;
    const byte* verify;
    word32 kekLen;
    word32 dataLen;
    word32 verifyLen;
} keywrapVector;

WOLFSSL_TEST_SUBROUTINE int aeskeywrap_test(void)
{
    int wrapSz, plainSz, testSz, i;

    /* test vectors from RFC 3394 (kek, data, verify) */

#ifdef WOLFSSL_AES_128
    /* Wrap 128 bits of Key Data with a 128-bit KEK */
    WOLFSSL_SMALL_STACK_STATIC const byte k1[] = {
        0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
        0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F
    };

    WOLFSSL_SMALL_STACK_STATIC const byte d1[] = {
        0x00, 0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77,
        0x88, 0x99, 0xAA, 0xBB, 0xCC, 0xDD, 0xEE, 0xFF
    };

    WOLFSSL_SMALL_STACK_STATIC const byte v1[] = {
        0x1F, 0xA6, 0x8B, 0x0A, 0x81, 0x12, 0xB4, 0x47,
        0xAE, 0xF3, 0x4B, 0xD8, 0xFB, 0x5A, 0x7B, 0x82,
        0x9D, 0x3E, 0x86, 0x23, 0x71, 0xD2, 0xCF, 0xE5
    };
#endif /* WOLFSSL_AES_128 */

#ifdef WOLFSSL_AES_192
    /* Wrap 128 bits of Key Data with a 192-bit KEK */
    WOLFSSL_SMALL_STACK_STATIC const byte k2[] = {
        0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
        0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F,
        0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17
    };

    WOLFSSL_SMALL_STACK_STATIC const byte d2[] = {
        0x00, 0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77,
        0x88, 0x99, 0xAA, 0xBB, 0xCC, 0xDD, 0xEE, 0xFF
    };

    WOLFSSL_SMALL_STACK_STATIC const byte v2[] = {
        0x96, 0x77, 0x8B, 0x25, 0xAE, 0x6C, 0xA4, 0x35,
        0xF9, 0x2B, 0x5B, 0x97, 0xC0, 0x50, 0xAE, 0xD2,
        0x46, 0x8A, 0xB8, 0xA1, 0x7A, 0xD8, 0x4E, 0x5D
    };
#endif

#ifdef WOLFSSL_AES_256
    /* Wrap 128 bits of Key Data with a 256-bit KEK */
    WOLFSSL_SMALL_STACK_STATIC const byte k3[] = {
        0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
        0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F,
        0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
        0x18, 0x19, 0x1A, 0x1B, 0x1C, 0x1D, 0x1E, 0x1F
    };

    WOLFSSL_SMALL_STACK_STATIC const byte d3[] = {
        0x00, 0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77,
        0x88, 0x99, 0xAA, 0xBB, 0xCC, 0xDD, 0xEE, 0xFF
    };

    WOLFSSL_SMALL_STACK_STATIC const byte v3[] = {
        0x64, 0xE8, 0xC3, 0xF9, 0xCE, 0x0F, 0x5B, 0xA2,
        0x63, 0xE9, 0x77, 0x79, 0x05, 0x81, 0x8A, 0x2A,
        0x93, 0xC8, 0x19, 0x1E, 0x7D, 0x6E, 0x8A, 0xE7
    };
#endif

#ifdef WOLFSSL_AES_192
    /* Wrap 192 bits of Key Data with a 192-bit KEK */
    WOLFSSL_SMALL_STACK_STATIC const byte k4[] = {
        0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
        0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F,
        0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17
    };

    WOLFSSL_SMALL_STACK_STATIC const byte d4[] = {
        0x00, 0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77,
        0x88, 0x99, 0xAA, 0xBB, 0xCC, 0xDD, 0xEE, 0xFF,
        0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07
    };

    WOLFSSL_SMALL_STACK_STATIC const byte v4[] = {
        0x03, 0x1D, 0x33, 0x26, 0x4E, 0x15, 0xD3, 0x32,
        0x68, 0xF2, 0x4E, 0xC2, 0x60, 0x74, 0x3E, 0xDC,
        0xE1, 0xC6, 0xC7, 0xDD, 0xEE, 0x72, 0x5A, 0x93,
        0x6B, 0xA8, 0x14, 0x91, 0x5C, 0x67, 0x62, 0xD2
    };
#endif

#ifdef WOLFSSL_AES_256
    /* Wrap 192 bits of Key Data with a 256-bit KEK */
    WOLFSSL_SMALL_STACK_STATIC const byte k5[] = {
        0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
        0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F,
        0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
        0x18, 0x19, 0x1A, 0x1B, 0x1C, 0x1D, 0x1E, 0x1F
    };

    WOLFSSL_SMALL_STACK_STATIC const byte d5[] = {
        0x00, 0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77,
        0x88, 0x99, 0xAA, 0xBB, 0xCC, 0xDD, 0xEE, 0xFF,
        0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07
    };

    WOLFSSL_SMALL_STACK_STATIC const byte v5[] = {
        0xA8, 0xF9, 0xBC, 0x16, 0x12, 0xC6, 0x8B, 0x3F,
        0xF6, 0xE6, 0xF4, 0xFB, 0xE3, 0x0E, 0x71, 0xE4,
        0x76, 0x9C, 0x8B, 0x80, 0xA3, 0x2C, 0xB8, 0x95,
        0x8C, 0xD5, 0xD1, 0x7D, 0x6B, 0x25, 0x4D, 0xA1
    };

    /* Wrap 256 bits of Key Data with a 256-bit KEK */
    WOLFSSL_SMALL_STACK_STATIC const byte k6[] = {
        0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
        0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F,
        0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
        0x18, 0x19, 0x1A, 0x1B, 0x1C, 0x1D, 0x1E, 0x1F
    };

    WOLFSSL_SMALL_STACK_STATIC const byte d6[] = {
        0x00, 0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77,
        0x88, 0x99, 0xAA, 0xBB, 0xCC, 0xDD, 0xEE, 0xFF,
        0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
        0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F
    };

    WOLFSSL_SMALL_STACK_STATIC const byte v6[] = {
        0x28, 0xC9, 0xF4, 0x04, 0xC4, 0xB8, 0x10, 0xF4,
        0xCB, 0xCC, 0xB3, 0x5C, 0xFB, 0x87, 0xF8, 0x26,
        0x3F, 0x57, 0x86, 0xE2, 0xD8, 0x0E, 0xD3, 0x26,
        0xCB, 0xC7, 0xF0, 0xE7, 0x1A, 0x99, 0xF4, 0x3B,
        0xFB, 0x98, 0x8B, 0x9B, 0x7A, 0x02, 0xDD, 0x21
    };
#endif /* WOLFSSL_AES_256 */

    byte output[MAX_KEYWRAP_TEST_OUTLEN];
    byte plain [MAX_KEYWRAP_TEST_PLAINLEN];

    const keywrapVector test_wrap[] =
    {
    #ifdef WOLFSSL_AES_128
        {k1, d1, v1, sizeof(k1), sizeof(d1), sizeof(v1)},
    #endif
    #ifdef WOLFSSL_AES_192
        {k2, d2, v2, sizeof(k2), sizeof(d2), sizeof(v2)},
    #endif
    #ifdef WOLFSSL_AES_256
        {k3, d3, v3, sizeof(k3), sizeof(d3), sizeof(v3)},
    #endif
    #ifdef WOLFSSL_AES_192
        {k4, d4, v4, sizeof(k4), sizeof(d4), sizeof(v4)},
    #endif
    #ifdef WOLFSSL_AES_256
        {k5, d5, v5, sizeof(k5), sizeof(d5), sizeof(v5)},
        {k6, d6, v6, sizeof(k6), sizeof(d6), sizeof(v6)}
    #endif
    };
    testSz = sizeof(test_wrap) / sizeof(keywrapVector);

    XMEMSET(output, 0, sizeof(output));
    XMEMSET(plain,  0, sizeof(plain));

    for (i = 0; i < testSz; i++) {

        wrapSz = wc_AesKeyWrap(test_wrap[i].kek, test_wrap[i].kekLen,
                               test_wrap[i].data, test_wrap[i].dataLen,
                               output, sizeof(output), NULL);

        if ( (wrapSz < 0) || (wrapSz != (int)test_wrap[i].verifyLen) )
            return -6600;

        if (XMEMCMP(output, test_wrap[i].verify, test_wrap[i].verifyLen) != 0)
            return -6601;

        plainSz = wc_AesKeyUnWrap((byte*)test_wrap[i].kek, test_wrap[i].kekLen,
                                  output, wrapSz,
                                  plain, sizeof(plain), NULL);

        if ( (plainSz < 0) || (plainSz != (int)test_wrap[i].dataLen) )
            return -6602;

        if (XMEMCMP(plain, test_wrap[i].data, test_wrap[i].dataLen) != 0)
            return -6603 - i;
    }

    return 0;
}
#endif /* HAVE_AES_KEYWRAP */


#endif /* NO_AES */


#ifdef HAVE_CAMELLIA

enum {
    CAM_ECB_ENC, CAM_ECB_DEC, CAM_CBC_ENC, CAM_CBC_DEC
};

typedef struct {
    int type;
    const byte* plaintext;
    const byte* iv;
    const byte* ciphertext;
    const byte* key;
    word32 keySz;
    int errorCode;
} test_vector_t;

WOLFSSL_TEST_SUBROUTINE int camellia_test(void)
{
    /* Camellia ECB Test Plaintext */
    WOLFSSL_SMALL_STACK_STATIC const byte pte[] =
    {
        0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef,
        0xfe, 0xdc, 0xba, 0x98, 0x76, 0x54, 0x32, 0x10
    };

    /* Camellia ECB Test Initialization Vector */
    WOLFSSL_SMALL_STACK_STATIC const byte ive[] = {0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0};

    /* Test 1: Camellia ECB 128-bit key */
    WOLFSSL_SMALL_STACK_STATIC const byte k1[] =
    {
        0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef,
        0xfe, 0xdc, 0xba, 0x98, 0x76, 0x54, 0x32, 0x10
    };
    WOLFSSL_SMALL_STACK_STATIC const byte c1[] =
    {
        0x67, 0x67, 0x31, 0x38, 0x54, 0x96, 0x69, 0x73,
        0x08, 0x57, 0x06, 0x56, 0x48, 0xea, 0xbe, 0x43
    };

    /* Test 2: Camellia ECB 192-bit key */
    WOLFSSL_SMALL_STACK_STATIC const byte k2[] =
    {
        0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef,
        0xfe, 0xdc, 0xba, 0x98, 0x76, 0x54, 0x32, 0x10,
        0x00, 0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77
    };
    WOLFSSL_SMALL_STACK_STATIC const byte c2[] =
    {
        0xb4, 0x99, 0x34, 0x01, 0xb3, 0xe9, 0x96, 0xf8,
        0x4e, 0xe5, 0xce, 0xe7, 0xd7, 0x9b, 0x09, 0xb9
    };

    /* Test 3: Camellia ECB 256-bit key */
    WOLFSSL_SMALL_STACK_STATIC const byte k3[] =
    {
        0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef,
        0xfe, 0xdc, 0xba, 0x98, 0x76, 0x54, 0x32, 0x10,
        0x00, 0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77,
        0x88, 0x99, 0xaa, 0xbb, 0xcc, 0xdd, 0xee, 0xff
    };
    WOLFSSL_SMALL_STACK_STATIC const byte c3[] =
    {
        0x9a, 0xcc, 0x23, 0x7d, 0xff, 0x16, 0xd7, 0x6c,
        0x20, 0xef, 0x7c, 0x91, 0x9e, 0x3a, 0x75, 0x09
    };

    /* Camellia CBC Test Plaintext */
    WOLFSSL_SMALL_STACK_STATIC const byte ptc[] =
    {
        0x6B, 0xC1, 0xBE, 0xE2, 0x2E, 0x40, 0x9F, 0x96,
        0xE9, 0x3D, 0x7E, 0x11, 0x73, 0x93, 0x17, 0x2A
    };

    /* Camellia CBC Test Initialization Vector */
    WOLFSSL_SMALL_STACK_STATIC const byte ivc[] =
    {
        0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
        0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F
    };

    /* Test 4: Camellia-CBC 128-bit key */
    WOLFSSL_SMALL_STACK_STATIC const byte k4[] =
    {
        0x2B, 0x7E, 0x15, 0x16, 0x28, 0xAE, 0xD2, 0xA6,
        0xAB, 0xF7, 0x15, 0x88, 0x09, 0xCF, 0x4F, 0x3C
    };
    WOLFSSL_SMALL_STACK_STATIC const byte c4[] =
    {
        0x16, 0x07, 0xCF, 0x49, 0x4B, 0x36, 0xBB, 0xF0,
        0x0D, 0xAE, 0xB0, 0xB5, 0x03, 0xC8, 0x31, 0xAB
    };

    /* Test 5: Camellia-CBC 192-bit key */
    WOLFSSL_SMALL_STACK_STATIC const byte k5[] =
    {
        0x8E, 0x73, 0xB0, 0xF7, 0xDA, 0x0E, 0x64, 0x52,
        0xC8, 0x10, 0xF3, 0x2B, 0x80, 0x90, 0x79, 0xE5,
        0x62, 0xF8, 0xEA, 0xD2, 0x52, 0x2C, 0x6B, 0x7B
    };
    WOLFSSL_SMALL_STACK_STATIC const byte c5[] =
    {
        0x2A, 0x48, 0x30, 0xAB, 0x5A, 0xC4, 0xA1, 0xA2,
        0x40, 0x59, 0x55, 0xFD, 0x21, 0x95, 0xCF, 0x93
    };

    /* Test 6: CBC 256-bit key */
    WOLFSSL_SMALL_STACK_STATIC const byte k6[] =
    {
        0x60, 0x3D, 0xEB, 0x10, 0x15, 0xCA, 0x71, 0xBE,
        0x2B, 0x73, 0xAE, 0xF0, 0x85, 0x7D, 0x77, 0x81,
        0x1F, 0x35, 0x2C, 0x07, 0x3B, 0x61, 0x08, 0xD7,
        0x2D, 0x98, 0x10, 0xA3, 0x09, 0x14, 0xDF, 0xF4
    };
    WOLFSSL_SMALL_STACK_STATIC const byte c6[] =
    {
        0xE6, 0xCF, 0xA3, 0x5F, 0xC0, 0x2B, 0x13, 0x4A,
        0x4D, 0x2C, 0x0B, 0x67, 0x37, 0xAC, 0x3E, 0xDA
    };

    byte out[CAMELLIA_BLOCK_SIZE];
    Camellia cam;
    int i, testsSz, ret;
    WOLFSSL_SMALL_STACK_STATIC const test_vector_t testVectors[] =
    {
        {CAM_ECB_ENC, pte, ive, c1, k1, sizeof(k1), -114},
        {CAM_ECB_ENC, pte, ive, c2, k2, sizeof(k2), -115},
        {CAM_ECB_ENC, pte, ive, c3, k3, sizeof(k3), -116},
        {CAM_ECB_DEC, pte, ive, c1, k1, sizeof(k1), -117},
        {CAM_ECB_DEC, pte, ive, c2, k2, sizeof(k2), -118},
        {CAM_ECB_DEC, pte, ive, c3, k3, sizeof(k3), -119},
        {CAM_CBC_ENC, ptc, ivc, c4, k4, sizeof(k4), -120},
        {CAM_CBC_ENC, ptc, ivc, c5, k5, sizeof(k5), -121},
        {CAM_CBC_ENC, ptc, ivc, c6, k6, sizeof(k6), -122},
        {CAM_CBC_DEC, ptc, ivc, c4, k4, sizeof(k4), -123},
        {CAM_CBC_DEC, ptc, ivc, c5, k5, sizeof(k5), -124},
        {CAM_CBC_DEC, ptc, ivc, c6, k6, sizeof(k6), -125}
    };

    testsSz = sizeof(testVectors)/sizeof(test_vector_t);
    for (i = 0; i < testsSz; i++) {
        if (wc_CamelliaSetKey(&cam, testVectors[i].key, testVectors[i].keySz,
                                                        testVectors[i].iv) != 0)
            return testVectors[i].errorCode;

        switch (testVectors[i].type) {
            case CAM_ECB_ENC:
                ret = wc_CamelliaEncryptDirect(&cam, out,
                                                testVectors[i].plaintext);
                if (ret != 0 || XMEMCMP(out, testVectors[i].ciphertext,
                                                        CAMELLIA_BLOCK_SIZE))
                    return testVectors[i].errorCode;
                break;
            case CAM_ECB_DEC:
                ret = wc_CamelliaDecryptDirect(&cam, out,
                                                    testVectors[i].ciphertext);
                if (ret != 0 || XMEMCMP(out, testVectors[i].plaintext,
                                                        CAMELLIA_BLOCK_SIZE))
                    return testVectors[i].errorCode;
                break;
            case CAM_CBC_ENC:
                ret = wc_CamelliaCbcEncrypt(&cam, out, testVectors[i].plaintext,
                                                           CAMELLIA_BLOCK_SIZE);
                if (ret != 0 || XMEMCMP(out, testVectors[i].ciphertext,
                                                        CAMELLIA_BLOCK_SIZE))
                    return testVectors[i].errorCode;
                break;
            case CAM_CBC_DEC:
                ret = wc_CamelliaCbcDecrypt(&cam, out,
                                testVectors[i].ciphertext, CAMELLIA_BLOCK_SIZE);
                if (ret != 0 || XMEMCMP(out, testVectors[i].plaintext,
                                                           CAMELLIA_BLOCK_SIZE))
                    return testVectors[i].errorCode;
                break;
            default:
                break;
        }
    }

    /* Setting the IV and checking it was actually set. */
    ret = wc_CamelliaSetIV(&cam, ivc);
    if (ret != 0 || XMEMCMP(cam.reg, ivc, CAMELLIA_BLOCK_SIZE))
        return -6700;

    /* Setting the IV to NULL should be same as all zeros IV */
    if (wc_CamelliaSetIV(&cam, NULL) != 0 ||
                                    XMEMCMP(cam.reg, ive, CAMELLIA_BLOCK_SIZE))
        return -6701;

    /* First parameter should never be null */
    if (wc_CamelliaSetIV(NULL, NULL) == 0)
        return -6702;

    /* First parameter should never be null, check it fails */
    if (wc_CamelliaSetKey(NULL, k1, sizeof(k1), NULL) == 0)
        return -6703;

    /* Key should have a size of 16, 24, or 32 */
    if (wc_CamelliaSetKey(&cam, k1, 0, NULL) == 0)
        return -6704;

    return 0;
}
#endif /* HAVE_CAMELLIA */

#ifdef HAVE_XCHACHA
WOLFSSL_TEST_SUBROUTINE int XChaCha_test(void) {
    int ret = -6830;

    WOLFSSL_SMALL_STACK_STATIC const byte Plaintext[] = {
        0x54, 0x68, 0x65, 0x20, 0x64, 0x68, 0x6f, 0x6c, 0x65, 0x20, 0x28, 0x70, 0x72, 0x6f, 0x6e, 0x6f, /* The dhole (prono */
        0x75, 0x6e, 0x63, 0x65, 0x64, 0x20, 0x22, 0x64, 0x6f, 0x6c, 0x65, 0x22, 0x29, 0x20, 0x69, 0x73, /* unced "dole") is */
        0x20, 0x61, 0x6c, 0x73, 0x6f, 0x20, 0x6b, 0x6e, 0x6f, 0x77, 0x6e, 0x20, 0x61, 0x73, 0x20, 0x74, /*  also known as t */
        0x68, 0x65, 0x20, 0x41, 0x73, 0x69, 0x61, 0x74, 0x69, 0x63, 0x20, 0x77, 0x69, 0x6c, 0x64, 0x20, /* he Asiatic wild  */
        0x64, 0x6f, 0x67, 0x2c, 0x20, 0x72, 0x65, 0x64, 0x20, 0x64, 0x6f, 0x67, 0x2c, 0x20, 0x61, 0x6e, /* dog, red dog, an */
        0x64, 0x20, 0x77, 0x68, 0x69, 0x73, 0x74, 0x6c, 0x69, 0x6e, 0x67, 0x20, 0x64, 0x6f, 0x67, 0x2e, /* d whistling dog. */
        0x20, 0x49, 0x74, 0x20, 0x69, 0x73, 0x20, 0x61, 0x62, 0x6f, 0x75, 0x74, 0x20, 0x74, 0x68, 0x65, /*  It is about the */
        0x20, 0x73, 0x69, 0x7a, 0x65, 0x20, 0x6f, 0x66, 0x20, 0x61, 0x20, 0x47, 0x65, 0x72, 0x6d, 0x61, /*  size of a Germa */
        0x6e, 0x20, 0x73, 0x68, 0x65, 0x70, 0x68, 0x65, 0x72, 0x64, 0x20, 0x62, 0x75, 0x74, 0x20, 0x6c, /* n shepherd but l */
        0x6f, 0x6f, 0x6b, 0x73, 0x20, 0x6d, 0x6f, 0x72, 0x65, 0x20, 0x6c, 0x69, 0x6b, 0x65, 0x20, 0x61, /* ooks more like a */
        0x20, 0x6c, 0x6f, 0x6e, 0x67, 0x2d, 0x6c, 0x65, 0x67, 0x67, 0x65, 0x64, 0x20, 0x66, 0x6f, 0x78, /*  long-legged fox */
        0x2e, 0x20, 0x54, 0x68, 0x69, 0x73, 0x20, 0x68, 0x69, 0x67, 0x68, 0x6c, 0x79, 0x20, 0x65, 0x6c, /* . This highly el */
        0x75, 0x73, 0x69, 0x76, 0x65, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x73, 0x6b, 0x69, 0x6c, 0x6c, 0x65, /* usive and skille */
        0x64, 0x20, 0x6a, 0x75, 0x6d, 0x70, 0x65, 0x72, 0x20, 0x69, 0x73, 0x20, 0x63, 0x6c, 0x61, 0x73, /* d jumper is clas */
        0x73, 0x69, 0x66, 0x69, 0x65, 0x64, 0x20, 0x77, 0x69, 0x74, 0x68, 0x20, 0x77, 0x6f, 0x6c, 0x76, /* sified with wolv */
        0x65, 0x73, 0x2c, 0x20, 0x63, 0x6f, 0x79, 0x6f, 0x74, 0x65, 0x73, 0x2c, 0x20, 0x6a, 0x61, 0x63, /* es, coyotes, jac */
        0x6b, 0x61, 0x6c, 0x73, 0x2c, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x66, 0x6f, 0x78, 0x65, 0x73, 0x20, /* kals, and foxes  */
        0x69, 0x6e, 0x20, 0x74, 0x68, 0x65, 0x20, 0x74, 0x61, 0x78, 0x6f, 0x6e, 0x6f, 0x6d, 0x69, 0x63, /* in the taxonomic */
        0x20, 0x66, 0x61, 0x6d, 0x69, 0x6c, 0x79, 0x20, 0x43, 0x61, 0x6e, 0x69, 0x64, 0x61, 0x65, 0x2e  /*  family Canidae. */
    };

    WOLFSSL_SMALL_STACK_STATIC const byte Key[] = {
        0x80, 0x81, 0x82, 0x83, 0x84, 0x85, 0x86, 0x87, 0x88, 0x89, 0x8a, 0x8b, 0x8c, 0x8d, 0x8e, 0x8f,
        0x90, 0x91, 0x92, 0x93, 0x94, 0x95, 0x96, 0x97, 0x98, 0x99, 0x9a, 0x9b, 0x9c, 0x9d, 0x9e, 0x9f
    };

    WOLFSSL_SMALL_STACK_STATIC const byte IV[] = {
        0x40, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47, 0x48, 0x49, 0x4a, 0x4b, 0x4c, 0x4d, 0x4e, 0x4f, /* @ABCDEFGHIJKLMNO */
        0x50, 0x51, 0x52, 0x53, 0x54, 0x55, 0x56, 0x58 };                                               /* PQRSTUVW */

    WOLFSSL_SMALL_STACK_STATIC const byte Ciphertext[] = {
        0x45, 0x59, 0xab, 0xba, 0x4e, 0x48, 0xc1, 0x61, 0x02, 0xe8, 0xbb, 0x2c, 0x05, 0xe6, 0x94, 0x7f,
        0x50, 0xa7, 0x86, 0xde, 0x16, 0x2f, 0x9b, 0x0b, 0x7e, 0x59, 0x2a, 0x9b, 0x53, 0xd0, 0xd4, 0xe9,
        0x8d, 0x8d, 0x64, 0x10, 0xd5, 0x40, 0xa1, 0xa6, 0x37, 0x5b, 0x26, 0xd8, 0x0d, 0xac, 0xe4, 0xfa,
        0xb5, 0x23, 0x84, 0xc7, 0x31, 0xac, 0xbf, 0x16, 0xa5, 0x92, 0x3c, 0x0c, 0x48, 0xd3, 0x57, 0x5d,
        0x4d, 0x0d, 0x2c, 0x67, 0x3b, 0x66, 0x6f, 0xaa, 0x73, 0x10, 0x61, 0x27, 0x77, 0x01, 0x09, 0x3a,
        0x6b, 0xf7, 0xa1, 0x58, 0xa8, 0x86, 0x42, 0x92, 0xa4, 0x1c, 0x48, 0xe3, 0xa9, 0xb4, 0xc0, 0xda,
        0xec, 0xe0, 0xf8, 0xd9, 0x8d, 0x0d, 0x7e, 0x05, 0xb3, 0x7a, 0x30, 0x7b, 0xbb, 0x66, 0x33, 0x31,
        0x64, 0xec, 0x9e, 0x1b, 0x24, 0xea, 0x0d, 0x6c, 0x3f, 0xfd, 0xdc, 0xec, 0x4f, 0x68, 0xe7, 0x44,
        0x30, 0x56, 0x19, 0x3a, 0x03, 0xc8, 0x10, 0xe1, 0x13, 0x44, 0xca, 0x06, 0xd8, 0xed, 0x8a, 0x2b,
        0xfb, 0x1e, 0x8d, 0x48, 0xcf, 0xa6, 0xbc, 0x0e, 0xb4, 0xe2, 0x46, 0x4b, 0x74, 0x81, 0x42, 0x40,
        0x7c, 0x9f, 0x43, 0x1a, 0xee, 0x76, 0x99, 0x60, 0xe1, 0x5b, 0xa8, 0xb9, 0x68, 0x90, 0x46, 0x6e,
        0xf2, 0x45, 0x75, 0x99, 0x85, 0x23, 0x85, 0xc6, 0x61, 0xf7, 0x52, 0xce, 0x20, 0xf9, 0xda, 0x0c,
        0x09, 0xab, 0x6b, 0x19, 0xdf, 0x74, 0xe7, 0x6a, 0x95, 0x96, 0x74, 0x46, 0xf8, 0xd0, 0xfd, 0x41,
        0x5e, 0x7b, 0xee, 0x2a, 0x12, 0xa1, 0x14, 0xc2, 0x0e, 0xb5, 0x29, 0x2a, 0xe7, 0xa3, 0x49, 0xae,
        0x57, 0x78, 0x20, 0xd5, 0x52, 0x0a, 0x1f, 0x3f, 0xb6, 0x2a, 0x17, 0xce, 0x6a, 0x7e, 0x68, 0xfa,
        0x7c, 0x79, 0x11, 0x1d, 0x88, 0x60, 0x92, 0x0b, 0xc0, 0x48, 0xef, 0x43, 0xfe, 0x84, 0x48, 0x6c,
        0xcb, 0x87, 0xc2, 0x5f, 0x0a, 0xe0, 0x45, 0xf0, 0xcc, 0xe1, 0xe7, 0x98, 0x9a, 0x9a, 0xa2, 0x20,
        0xa2, 0x8b, 0xdd, 0x48, 0x27, 0xe7, 0x51, 0xa2, 0x4a, 0x6d, 0x5c, 0x62, 0xd7, 0x90, 0xa6, 0x63,
        0x93, 0xb9, 0x31, 0x11, 0xc1, 0xa5, 0x5d, 0xd7, 0x42, 0x1a, 0x10, 0x18, 0x49, 0x74, 0xc7, 0xc5
    };

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
    struct ChaCha *chacha = (struct ChaCha *)XMALLOC(sizeof *chacha, HEAP_HINT, DYNAMIC_TYPE_CIPHER);
    byte *buf1 = (byte *)XMALLOC(sizeof Plaintext, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    byte *buf2 = (byte *)XMALLOC(sizeof Plaintext, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);

    if ((chacha == NULL) || (buf1 == NULL) || (buf2 == NULL))
        ERROR_OUT(MEMORY_E, out);
#else
    struct ChaCha chacha[1];
    byte buf1[sizeof Plaintext];
    byte buf2[sizeof Plaintext];
#endif

    ret = wc_XChacha_SetKey(chacha, Key, sizeof Key, IV, sizeof IV, 0);
    if (ret < 0)
        ERROR_OUT(-6831, out);

    ret = wc_Chacha_Process(chacha, buf1, Plaintext, sizeof Plaintext);
    if (ret < 0)
        ERROR_OUT(-6832, out);

    if (XMEMCMP(buf1, Ciphertext, sizeof Plaintext))
        ERROR_OUT(-6833, out);

    ret = wc_XChacha_SetKey(chacha, Key, sizeof Key, IV, sizeof IV, 0);
    if (ret < 0)
        ERROR_OUT(-6834, out);

    ret = wc_Chacha_Process(chacha, buf2, buf1, sizeof Plaintext);
    if (ret < 0)
        ERROR_OUT(-6835, out);

    if (XMEMCMP(buf2, Plaintext, sizeof Plaintext))
        ERROR_OUT(-6836, out);

  out:

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
    if (chacha)
        XFREE(chacha, HEAP_HINT, DYNAMIC_TYPE_CIPHER);
    if (buf1)
        XFREE(buf1, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    if (buf2)
        XFREE(buf2, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
#endif

    return ret;
}
#endif /* HAVE_XCHACHA */

#if defined(HAVE_XCHACHA) && defined(HAVE_POLY1305)
WOLFSSL_TEST_SUBROUTINE int XChaCha20Poly1305_test(void) {
    int ret;

    WOLFSSL_SMALL_STACK_STATIC const byte Plaintext[] = {
        0x4c, 0x61, 0x64, 0x69, 0x65, 0x73, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x47, 0x65, 0x6e, 0x74, 0x6c, /* Ladies and Gentl */
        0x65, 0x6d, 0x65, 0x6e, 0x20, 0x6f, 0x66, 0x20, 0x74, 0x68, 0x65, 0x20, 0x63, 0x6c, 0x61, 0x73, /* emen of the clas */
        0x73, 0x20, 0x6f, 0x66, 0x20, 0x27, 0x39, 0x39, 0x3a, 0x20, 0x49, 0x66, 0x20, 0x49, 0x20, 0x63, /* s of '99: If I c */
        0x6f, 0x75, 0x6c, 0x64, 0x20, 0x6f, 0x66, 0x66, 0x65, 0x72, 0x20, 0x79, 0x6f, 0x75, 0x20, 0x6f, /* ould offer you o */
        0x6e, 0x6c, 0x79, 0x20, 0x6f, 0x6e, 0x65, 0x20, 0x74, 0x69, 0x70, 0x20, 0x66, 0x6f, 0x72, 0x20, /* nly one tip for */
        0x74, 0x68, 0x65, 0x20, 0x66, 0x75, 0x74, 0x75, 0x72, 0x65, 0x2c, 0x20, 0x73, 0x75, 0x6e, 0x73, /* the future, suns */
        0x63, 0x72, 0x65, 0x65, 0x6e, 0x20, 0x77, 0x6f, 0x75, 0x6c, 0x64, 0x20, 0x62, 0x65, 0x20, 0x69, /* creen would be i */
        0x74, 0x2e };                                                                                   /* t. */

    WOLFSSL_SMALL_STACK_STATIC const byte AAD[] = { 0x50, 0x51, 0x52, 0x53, 0xc0, 0xc1, 0xc2, 0xc3, 0xc4, 0xc5, 0xc6, 0xc7 }; /* PQRS........ */

    WOLFSSL_SMALL_STACK_STATIC const byte Key[] = {
        0x80, 0x81, 0x82, 0x83, 0x84, 0x85, 0x86, 0x87, 0x88, 0x89, 0x8a, 0x8b, 0x8c, 0x8d, 0x8e, 0x8f,
        0x90, 0x91, 0x92, 0x93, 0x94, 0x95, 0x96, 0x97, 0x98, 0x99, 0x9a, 0x9b, 0x9c, 0x9d, 0x9e, 0x9f
    };

    WOLFSSL_SMALL_STACK_STATIC const byte IV[] = {
        0x40, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47, 0x48, 0x49, 0x4a, 0x4b, 0x4c, 0x4d, 0x4e, 0x4f, /* @ABCDEFGHIJKLMNO */
        0x50, 0x51, 0x52, 0x53, 0x54, 0x55, 0x56, 0x57 };                                               /* PQRSTUVW */

    WOLFSSL_SMALL_STACK_STATIC const byte Ciphertext[] = {
      0xbd, 0x6d, 0x17, 0x9d, 0x3e, 0x83, 0xd4, 0x3b, 0x95, 0x76, 0x57, 0x94, 0x93, 0xc0, 0xe9, 0x39,
      0x57, 0x2a, 0x17, 0x00, 0x25, 0x2b, 0xfa, 0xcc, 0xbe, 0xd2, 0x90, 0x2c, 0x21, 0x39, 0x6c, 0xbb,
      0x73, 0x1c, 0x7f, 0x1b, 0x0b, 0x4a, 0xa6, 0x44, 0x0b, 0xf3, 0xa8, 0x2f, 0x4e, 0xda, 0x7e, 0x39,
      0xae, 0x64, 0xc6, 0x70, 0x8c, 0x54, 0xc2, 0x16, 0xcb, 0x96, 0xb7, 0x2e, 0x12, 0x13, 0xb4, 0x52,
      0x2f, 0x8c, 0x9b, 0xa4, 0x0d, 0xb5, 0xd9, 0x45, 0xb1, 0x1b, 0x69, 0xb9, 0x82, 0xc1, 0xbb, 0x9e,
      0x3f, 0x3f, 0xac, 0x2b, 0xc3, 0x69, 0x48, 0x8f, 0x76, 0xb2, 0x38, 0x35, 0x65, 0xd3, 0xff, 0xf9,
      0x21, 0xf9, 0x66, 0x4c, 0x97, 0x63, 0x7d, 0xa9, 0x76, 0x88, 0x12, 0xf6, 0x15, 0xc6, 0x8b, 0x13,
      0xb5, 0x2e };

    WOLFSSL_SMALL_STACK_STATIC const byte Tag[] = {
      0xc0, 0x87, 0x59, 0x24, 0xc1, 0xc7, 0x98, 0x79, 0x47, 0xde, 0xaf, 0xd8, 0x78, 0x0a, 0xcf, 0x49
    };

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
    byte *buf1 = (byte *)XMALLOC(sizeof Ciphertext + sizeof Tag, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    byte *buf2 = (byte *)XMALLOC(sizeof Plaintext, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);

    if ((buf1 == NULL) || (buf2 == NULL))
        ERROR_OUT(-6480, out);
#else
    byte buf1[sizeof Ciphertext + sizeof Tag];
    byte buf2[sizeof Plaintext];
#endif

    ret = wc_XChaCha20Poly1305_Encrypt(buf1, sizeof Ciphertext + sizeof Tag,
                                       Plaintext, sizeof Plaintext,
                                       AAD, sizeof AAD,
                                       IV, sizeof IV,
                                       Key, sizeof Key);

    if (ret < 0)
        ERROR_OUT(-6841, out);

    if (XMEMCMP(buf1, Ciphertext, sizeof Ciphertext))
        ERROR_OUT(-6842, out);

    if (XMEMCMP(buf1 + sizeof Ciphertext, Tag, CHACHA20_POLY1305_AEAD_AUTHTAG_SIZE))
        ERROR_OUT(-6843, out);

    ret = wc_XChaCha20Poly1305_Decrypt(buf2, sizeof Plaintext,
                                 buf1, sizeof Ciphertext + sizeof Tag,
                                       AAD, sizeof AAD,
                                       IV, sizeof IV,
                                       Key, sizeof Key);

    if (ret < 0)
        ERROR_OUT(-6844, out);

    if (XMEMCMP(buf2, Plaintext, sizeof Plaintext))
        ERROR_OUT(-6845, out);

  out:

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
    if (buf1 != NULL)
        XFREE(buf1, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    if (buf2 != NULL)
        XFREE(buf2, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
#endif

    return ret;
}
#endif /* defined(HAVE_XCHACHA) && defined(HAVE_POLY1305) */

#ifndef WC_NO_RNG
static int _rng_test(WC_RNG* rng, int errorOffset)
{
    byte block[32];
    int ret, i;

    XMEMSET(block, 0, sizeof(block));

    ret = wc_RNG_GenerateBlock(rng, block, sizeof(block));
    if (ret != 0) {
        ret = -6850;
        goto exit;
    }

    /* Check for 0's */
    for (i=0; i<(int)sizeof(block); i++) {
        if (block[i] == 0) {
            ret++;
        }
    }
    /* All zeros count check */
    if (ret >= (int)sizeof(block)) {
        ret = -6851;
        goto exit;
    }

    ret = wc_RNG_GenerateByte(rng, block);
    if (ret != 0) {
        ret = -6852;
        goto exit;
    }

    /* Parameter validation testing. */
    ret = wc_RNG_GenerateBlock(NULL, block, sizeof(block));
    if (ret != BAD_FUNC_ARG) {
        ret = -6853;
        goto exit;
    }
    ret = wc_RNG_GenerateBlock(rng, NULL, sizeof(block));
    if (ret != BAD_FUNC_ARG) {
        ret = -6854;
        goto exit;
    }

    ret = wc_RNG_GenerateByte(NULL, block);
    if (ret != BAD_FUNC_ARG) {
        ret = -6855;
        goto exit;
    }
    ret = wc_RNG_GenerateByte(rng, NULL);
    if (ret != BAD_FUNC_ARG) {
        ret = -6856;
        goto exit;
    }

    ret = 0;

exit:
    if (ret != 0)
        ret += errorOffset;

    return ret;
}


static int random_rng_test(void)
{
    WC_RNG localRng;
    WC_RNG* rng;
    int ret;

    rng = &localRng;
    /* Test stack based RNG. */
#ifndef HAVE_FIPS
    ret = wc_InitRng_ex(rng, HEAP_HINT, devId);
#else
    ret = wc_InitRng(rng);
#endif
    if (ret != 0) return -6900;

    ret = _rng_test(rng, -6300);

    /* Make sure and free RNG */
    wc_FreeRng(rng);

    if (ret != 0) return ret;

#if !defined(HAVE_FIPS) && !defined(HAVE_SELFTEST) && !defined(WOLFSSL_NO_MALLOC)
    {
        byte nonce[8] = { 0 };
        /* Test dynamic RNG. */
        rng = wc_rng_new(nonce, (word32)sizeof(nonce), HEAP_HINT);
        if (rng == NULL) return -6901;

        ret = _rng_test(rng, -6310);

        wc_rng_free(rng);
    }
#endif

    return ret;
}

#if defined(HAVE_HASHDRBG) && !defined(CUSTOM_RAND_GENERATE_BLOCK)

#ifdef WC_RNG_SEED_CB
static int seed_cb(OS_Seed* os, byte* output, word32 sz)
{
    word32 i;
    (void)os;
    /* Known answer test. Set the seed to the same value every time. */
    for (i = 0; i < sz; i++)
        output[i] = (byte)i;
    return 0;
}

static int rng_seed_test(void)
{
#ifndef HAVE_FIPS
    WOLFSSL_SMALL_STACK_STATIC const byte check[] =
    {
        0x83, 0x46, 0x65, 0x2f, 0x5c, 0x44, 0x16, 0x5f,
        0xb3, 0x89, 0x26, 0xde, 0x0b, 0x6b, 0xa2, 0x06,
        0x7e, 0xa7, 0x9a, 0x55, 0x22, 0x01, 0xb0, 0x22,
        0xf4, 0x7e, 0xa2, 0x66, 0xc4, 0x08, 0x6f, 0xba
    };
#else
    /* FIPS uses a longer seed, so different check value. */
    WOLFSSL_SMALL_STACK_STATIC const byte check[] =
    {
        0xaf, 0x31, 0xcc, 0xef, 0xa9, 0x29, 0x4c, 0x24,
        0xbd, 0xa5, 0xa3, 0x52, 0x69, 0xf3, 0xb9, 0xb2,
        0x1e, 0xd4, 0x52, 0x3b, 0x9a, 0x96, 0x06, 0x20,
        0xc0, 0x5f, 0x44, 0x06, 0x1f, 0x80, 0xdf, 0xe0
    };
#endif
    byte output[WC_SHA256_DIGEST_SIZE];
    WC_RNG rng;
    int ret;

    ret = wc_SetSeed_Cb(seed_cb);
    if (ret != 0) {
        ret = -7007;
        goto exit;
    }
    ret = wc_InitRng(&rng);
    if (ret != 0) {
        ret = -7008;
        goto exit;
    }
    ret = wc_RNG_GenerateBlock(&rng, output, sizeof(output));
    if (ret != 0) {
        ret = -7009;
        goto exit;
    }
    ret = XMEMCMP(output, check, sizeof(output));
    if (ret != 0) {
        ret = -7010;
        goto exit;
    }
    ret = wc_FreeRng(&rng);
    if (ret != 0) {
        ret = -7011;
        goto exit;
    }
    ret = wc_SetSeed_Cb(wc_GenerateSeed);
    if (ret != 0) {
        ret = -7012;
    }

exit:
    return ret;
}
#endif


WOLFSSL_TEST_SUBROUTINE int random_test(void)
{
    WOLFSSL_SMALL_STACK_STATIC const byte test1Entropy[] =
    {
        0xa6, 0x5a, 0xd0, 0xf3, 0x45, 0xdb, 0x4e, 0x0e, 0xff, 0xe8, 0x75, 0xc3,
        0xa2, 0xe7, 0x1f, 0x42, 0xc7, 0x12, 0x9d, 0x62, 0x0f, 0xf5, 0xc1, 0x19,
        0xa9, 0xef, 0x55, 0xf0, 0x51, 0x85, 0xe0, 0xfb, 0x85, 0x81, 0xf9, 0x31,
        0x75, 0x17, 0x27, 0x6e, 0x06, 0xe9, 0x60, 0x7d, 0xdb, 0xcb, 0xcc, 0x2e
    };
    WOLFSSL_SMALL_STACK_STATIC const byte test1Output[] =
    {
        0xd3, 0xe1, 0x60, 0xc3, 0x5b, 0x99, 0xf3, 0x40, 0xb2, 0x62, 0x82, 0x64,
        0xd1, 0x75, 0x10, 0x60, 0xe0, 0x04, 0x5d, 0xa3, 0x83, 0xff, 0x57, 0xa5,
        0x7d, 0x73, 0xa6, 0x73, 0xd2, 0xb8, 0xd8, 0x0d, 0xaa, 0xf6, 0xa6, 0xc3,
        0x5a, 0x91, 0xbb, 0x45, 0x79, 0xd7, 0x3f, 0xd0, 0xc8, 0xfe, 0xd1, 0x11,
        0xb0, 0x39, 0x13, 0x06, 0x82, 0x8a, 0xdf, 0xed, 0x52, 0x8f, 0x01, 0x81,
        0x21, 0xb3, 0xfe, 0xbd, 0xc3, 0x43, 0xe7, 0x97, 0xb8, 0x7d, 0xbb, 0x63,
        0xdb, 0x13, 0x33, 0xde, 0xd9, 0xd1, 0xec, 0xe1, 0x77, 0xcf, 0xa6, 0xb7,
        0x1f, 0xe8, 0xab, 0x1d, 0xa4, 0x66, 0x24, 0xed, 0x64, 0x15, 0xe5, 0x1c,
        0xcd, 0xe2, 0xc7, 0xca, 0x86, 0xe2, 0x83, 0x99, 0x0e, 0xea, 0xeb, 0x91,
        0x12, 0x04, 0x15, 0x52, 0x8b, 0x22, 0x95, 0x91, 0x02, 0x81, 0xb0, 0x2d,
        0xd4, 0x31, 0xf4, 0xc9, 0xf7, 0x04, 0x27, 0xdf
    };
    WOLFSSL_SMALL_STACK_STATIC const byte test2EntropyA[] =
    {
        0x63, 0x36, 0x33, 0x77, 0xe4, 0x1e, 0x86, 0x46, 0x8d, 0xeb, 0x0a, 0xb4,
        0xa8, 0xed, 0x68, 0x3f, 0x6a, 0x13, 0x4e, 0x47, 0xe0, 0x14, 0xc7, 0x00,
        0x45, 0x4e, 0x81, 0xe9, 0x53, 0x58, 0xa5, 0x69, 0x80, 0x8a, 0xa3, 0x8f,
        0x2a, 0x72, 0xa6, 0x23, 0x59, 0x91, 0x5a, 0x9f, 0x8a, 0x04, 0xca, 0x68
    };
    WOLFSSL_SMALL_STACK_STATIC const byte test2EntropyB[] =
    {
        0xe6, 0x2b, 0x8a, 0x8e, 0xe8, 0xf1, 0x41, 0xb6, 0x98, 0x05, 0x66, 0xe3,
        0xbf, 0xe3, 0xc0, 0x49, 0x03, 0xda, 0xd4, 0xac, 0x2c, 0xdf, 0x9f, 0x22,
        0x80, 0x01, 0x0a, 0x67, 0x39, 0xbc, 0x83, 0xd3
    };
    WOLFSSL_SMALL_STACK_STATIC const byte test2Output[] =
    {
        0x04, 0xee, 0xc6, 0x3b, 0xb2, 0x31, 0xdf, 0x2c, 0x63, 0x0a, 0x1a, 0xfb,
        0xe7, 0x24, 0x94, 0x9d, 0x00, 0x5a, 0x58, 0x78, 0x51, 0xe1, 0xaa, 0x79,
        0x5e, 0x47, 0x73, 0x47, 0xc8, 0xb0, 0x56, 0x62, 0x1c, 0x18, 0xbd, 0xdc,
        0xdd, 0x8d, 0x99, 0xfc, 0x5f, 0xc2, 0xb9, 0x20, 0x53, 0xd8, 0xcf, 0xac,
        0xfb, 0x0b, 0xb8, 0x83, 0x12, 0x05, 0xfa, 0xd1, 0xdd, 0xd6, 0xc0, 0x71,
        0x31, 0x8a, 0x60, 0x18, 0xf0, 0x3b, 0x73, 0xf5, 0xed, 0xe4, 0xd4, 0xd0,
        0x71, 0xf9, 0xde, 0x03, 0xfd, 0x7a, 0xea, 0x10, 0x5d, 0x92, 0x99, 0xb8,
        0xaf, 0x99, 0xaa, 0x07, 0x5b, 0xdb, 0x4d, 0xb9, 0xaa, 0x28, 0xc1, 0x8d,
        0x17, 0x4b, 0x56, 0xee, 0x2a, 0x01, 0x4d, 0x09, 0x88, 0x96, 0xff, 0x22,
        0x82, 0xc9, 0x55, 0xa8, 0x19, 0x69, 0xe0, 0x69, 0xfa, 0x8c, 0xe0, 0x07,
        0xa1, 0x80, 0x18, 0x3a, 0x07, 0xdf, 0xae, 0x17
    };

    byte output[WC_SHA256_DIGEST_SIZE * 4];
    int ret;

    ret = wc_RNG_HealthTest(0, test1Entropy, sizeof(test1Entropy), NULL, 0,
                            output, sizeof(output));
    if (ret != 0)
        return -7000;

    if (XMEMCMP(test1Output, output, sizeof(output)) != 0)
        return -7001;

    ret = wc_RNG_HealthTest(1, test2EntropyA, sizeof(test2EntropyA),
                            test2EntropyB, sizeof(test2EntropyB),
                            output, sizeof(output));
    if (ret != 0)
        return -7002;

    if (XMEMCMP(test2Output, output, sizeof(output)) != 0)
        return -7003;

    /* Basic RNG generate block test */
    if ((ret = random_rng_test()) != 0)
        return ret;

    /* Test the seed check function. */
#if !(defined(HAVE_FIPS) || defined(HAVE_SELFTEST)) || \
    (defined(HAVE_FIPS_VERSION) && (HAVE_FIPS_VERSION >= 2))
    {
        word32 i, outputSz;

        /* Repeat the same byte over and over. Should fail. */
        outputSz = sizeof(output);
        XMEMSET(output, 1, outputSz);
        ret = wc_RNG_TestSeed(output, outputSz);
        if (ret == 0)
            return -7004;

        /* Every byte of the entropy scratch is different,
         * entropy is a single byte that shouldn't match. */
        outputSz = (sizeof(output) / 2) + 1;
        for (i = 0; i < outputSz; i++)
            output[i] = (byte)i;
        ret = wc_RNG_TestSeed(output, outputSz);
        if (ret != 0)
            return -7005;

        outputSz = sizeof(output);
        for (i = 0; i < outputSz; i++)
            output[i] = (byte)i;
        ret = wc_RNG_TestSeed(output, outputSz);
        if (ret != 0)
            return -7006;
    }
#endif

    /* Test the seed callback. */
#ifdef WC_RNG_SEED_CB
    if ((ret = rng_seed_test()) != 0)
        return ret;
#endif

    return 0;
}

#else

WOLFSSL_TEST_SUBROUTINE int random_test(void)
{
    /* Basic RNG generate block test */
    return random_rng_test();
}

#endif /* HAVE_HASHDRBG && !CUSTOM_RAND_GENERATE_BLOCK */
#endif /* WC_NO_RNG */

#ifndef MEM_TEST_SZ
    #define MEM_TEST_SZ 1024
#endif

#if defined(WOLFSSL_STATIC_MEMORY) || !defined(WOLFSSL_NO_MALLOC)
static int simple_mem_test(int sz)
{
    int ret = 0;
    byte* b;
    int i;

    b = (byte*)XMALLOC(sz, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    if (b == NULL) {
        return -7110;
    }
    /* utilize memory */
    for (i = 0; i < sz; i++) {
        b[i] = (byte)i;
    }
    /* read back and verify */
    for (i = 0; i < sz; i++) {
        if (b[i] != (byte)i) {
            ret = -7111;
            break;
        }
    }
    XFREE(b, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    return ret;
}
#endif

WOLFSSL_TEST_SUBROUTINE int memory_test(void)
{
    int ret = 0;
#if defined(COMPLEX_MEM_TEST) || defined(WOLFSSL_STATIC_MEMORY)
    int i;
#endif
#ifdef WOLFSSL_STATIC_MEMORY
    word32 size[] = { WOLFMEM_BUCKETS };
    word32 dist[] = { WOLFMEM_DIST };
    byte buffer[30000]; /* make large enough to involve many bucket sizes */
    int pad = -(int)((wc_ptr_t)buffer) & (WOLFSSL_STATIC_ALIGN - 1);
              /* pad to account for if head of buffer is not at set memory
               * alignment when tests are ran */
#endif

#ifdef WOLFSSL_STATIC_MEMORY
    /* check macro settings */
    if (sizeof(size)/sizeof(word32) != WOLFMEM_MAX_BUCKETS) {
        return -7200;
    }

    if (sizeof(dist)/sizeof(word32) != WOLFMEM_MAX_BUCKETS) {
        return -7201;
    }

    for (i = 0; i < WOLFMEM_MAX_BUCKETS; i++) {
        if ((size[i] % WOLFSSL_STATIC_ALIGN) != 0) {
            /* each element in array should be divisible by alignment size */
            return -7202;
        }
    }

    for (i = 1; i < WOLFMEM_MAX_BUCKETS; i++) {
        if (size[i - 1] >= size[i]) {
            return -7203; /* sizes should be in increasing order  */
        }
    }

    /* check that padding size returned is possible */
    if (wolfSSL_MemoryPaddingSz() < WOLFSSL_STATIC_ALIGN) {
        return -7204; /* no room for wc_Memory struct */
    }

    if (wolfSSL_MemoryPaddingSz() < 0) {
        return -7205;
    }

    if (wolfSSL_MemoryPaddingSz() % WOLFSSL_STATIC_ALIGN != 0) {
        return -7206; /* not aligned! */
    }

    /* check function to return optimum buffer size (rounded down) */
    ret = wolfSSL_StaticBufferSz(buffer, sizeof(buffer), WOLFMEM_GENERAL);
    if ((ret - pad) % WOLFSSL_STATIC_ALIGN != 0) {
        return -7207; /* not aligned! */
    }

    if (ret < 0) {
        return -7208;
    }

    if ((unsigned int)ret > sizeof(buffer)) {
        return -7209; /* did not round down as expected */
    }

    if (ret != wolfSSL_StaticBufferSz(buffer, ret, WOLFMEM_GENERAL)) {
        return -7210; /* return value changed when using suggested value */
    }

    ret = wolfSSL_MemoryPaddingSz();
    ret += pad; /* add space that is going to be needed if buffer not aligned */
    if (wolfSSL_StaticBufferSz(buffer, size[0] + ret + 1, WOLFMEM_GENERAL) !=
            (ret + (int)size[0])) {
        return -7211; /* did not round down to nearest bucket value */
    }

    ret = wolfSSL_StaticBufferSz(buffer, sizeof(buffer), WOLFMEM_IO_POOL);
    if ((ret - pad) < 0) {
        return -7212;
    }

    if (((ret - pad) % (WOLFMEM_IO_SZ + wolfSSL_MemoryPaddingSz())) != 0) {
        return -7213; /* not even chunks of memory for IO size */
    }

    if (((ret - pad) % WOLFSSL_STATIC_ALIGN) != 0) {
        return -7214; /* memory not aligned */
    }

    /* check for passing bad or unknown arguments to functions */
    if (wolfSSL_StaticBufferSz(NULL, 1, WOLFMEM_GENERAL) > 0) {
        return -7215;
    }

    if (wolfSSL_StaticBufferSz(buffer, 1, WOLFMEM_GENERAL) != 0) {
        return -7216; /* should round to 0 since struct + bucket will not fit */
    }

    (void)dist; /* avoid static analysis warning of variable not used */
#endif

#if defined(WOLFSSL_STATIC_MEMORY) || !defined(WOLFSSL_NO_MALLOC)
    /* simple test */
    ret = simple_mem_test(MEM_TEST_SZ);
    if (ret != 0)
        return ret;
#endif

#ifdef COMPLEX_MEM_TEST
    /* test various size blocks */
    for (i = 1; i < MEM_TEST_SZ; i*=2) {
        ret = simple_mem_test(i);
        if (ret != 0)
            return ret;
    }
#endif

#if !defined(USE_FAST_MATH) && !defined(WOLFSSL_NO_MALLOC)
    /* realloc test */
    {
        byte *c = NULL;
        byte *b = (byte*)XMALLOC(MEM_TEST_SZ, HEAP_HINT,
                                 DYNAMIC_TYPE_TMP_BUFFER);
        if (b) {
            c = (byte*)XREALLOC(b, MEM_TEST_SZ+sizeof(word32), HEAP_HINT,
                                DYNAMIC_TYPE_TMP_BUFFER);
            if (c)
                b = c;
        }
        if (b)
            XFREE(b, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
        if ((b == NULL) || (c == NULL)) {
            return -7217;
        }
    }
#endif

    return ret;
}

#ifndef NO_FILESYSTEM

/* Cert Paths */
#ifdef FREESCALE_MQX
    #define CERT_PREFIX "a:\\"
    #define CERT_PATH_SEP "\\"
#elif defined(WOLFSSL_uTKERNEL2)
    #define CERT_PREFIX "/uda/"
    #define CERT_PATH_SEP "/"
#elif defined(_WIN32_WCE)
    #define CERT_PREFIX "\\windows\\"
    #define CERT_PATH_SEP "\\"
#endif

#ifndef CERT_PREFIX
    #define CERT_PREFIX "./"
#endif
#ifndef CERT_PATH_SEP
    #define CERT_PATH_SEP "/"
#endif
#ifndef CERT_WRITE_TEMP_DIR
    #define CERT_WRITE_TEMP_DIR CERT_PREFIX
#endif

#define CERT_ROOT CERT_PREFIX "certs" CERT_PATH_SEP

/* Generated Test Certs */
#if !defined(USE_CERT_BUFFERS_1024) && !defined(USE_CERT_BUFFERS_2048) && \
    !defined(USE_CERT_BUFFERS_3072) && !defined(USE_CERT_BUFFERS_4096)
    #if !defined(NO_RSA) && !defined(NO_ASN)
        static const char* clientKey  = CERT_ROOT "client-key.der";
        static const char* clientCert = CERT_ROOT "client-cert.der";
        #ifdef WOLFSSL_CERT_EXT
            static const char* clientKeyPub  = CERT_ROOT "client-keyPub.der";
        #endif
    #endif /* !NO_RSA && !NO_ASN */
#endif
#if !defined(USE_CERT_BUFFERS_1024) && !defined(USE_CERT_BUFFERS_2048)
    #if !defined(NO_RSA) && !defined(NO_ASN)
        #if defined(WOLFSSL_CERT_GEN) || defined(HAVE_PKCS7)
            static const char* rsaCaKeyFile  = CERT_ROOT "ca-key.der";
            #ifdef WOLFSSL_CERT_GEN
            static const char* rsaCaCertFile = CERT_ROOT "ca-cert.pem";
            #endif
            #if defined(WOLFSSL_ALT_NAMES) || defined(HAVE_PKCS7)
            static const char* rsaCaCertDerFile = CERT_ROOT "ca-cert.der";
            #endif
            #ifdef HAVE_PKCS7
                static const char* rsaServerCertDerFile =
                                               CERT_ROOT "server-cert.der";
                static const char* rsaServerKeyDerFile =
                                               CERT_ROOT "server-key.der";
            #endif
        #endif
    #endif /* !NO_RSA && !NO_ASN */
#endif /* !USE_CERT_BUFFER_* */
#if !defined(USE_CERT_BUFFERS_1024) && !defined(USE_CERT_BUFFERS_2048) && \
    !defined(USE_CERT_BUFFERS_3072) && !defined(USE_CERT_BUFFERS_4096)
    #if !defined(NO_ASN) && !defined(NO_DH)
        static const char* dhParamsFile = CERT_ROOT "dh2048.der";
    #endif
#endif
#if !defined(NO_ASN) && !defined(NO_DH)
    #if defined(WOLFSSL_DH_EXTRA) && (!defined(HAVE_FIPS) ||                \
        (defined(HAVE_FIPS_VERSION) && (HAVE_FIPS_VERSION > 2)))
        static const char* dhKeyFile = CERT_ROOT "statickeys/dh-ffdhe2048.der";
        static const char* dhKeyPubFile = CERT_ROOT "statickeys/dh-ffdhe2048-pub.der";
    #endif
#endif
#if !defined(USE_CERT_BUFFERS_1024) && !defined(USE_CERT_BUFFERS_2048)
    #ifndef NO_DSA
        static const char* dsaKey = CERT_ROOT "dsa2048.der";
    #endif
#endif /* !USE_CERT_BUFFER_* */
#if !defined(USE_CERT_BUFFERS_256) && !defined(NO_ECC256)
    #ifdef HAVE_ECC
        /* cert files to be used in rsa cert gen test, check if RSA enabled */
        #ifdef HAVE_ECC_KEY_IMPORT
            static const char* eccKeyDerFile = CERT_ROOT "ecc-key.der";
        #endif
#endif
#if !defined(USE_CERT_BUFFERS_256) && !defined(NO_ASN)
        #if defined(HAVE_ECC) && defined(WOLFSSL_CERT_GEN)
            #ifndef NO_RSA
                static const char* eccKeyPubFileDer = CERT_ROOT "ecc-keyPub.der";
            #endif
            static const char* eccCaKeyFile  = CERT_ROOT "ca-ecc-key.der";
            static const char* eccCaCertFile = CERT_ROOT "ca-ecc-cert.pem";
            #ifdef ENABLE_ECC384_CERT_GEN_TEST
                static const char* eccCaKey384File =
                                               CERT_ROOT "ca-ecc384-key.der";
                static const char* eccCaCert384File =
                                               CERT_ROOT "ca-ecc384-cert.pem";
            #endif
        #endif
        #if defined(HAVE_PKCS7) && defined(HAVE_ECC)
            static const char* eccClientKey  = CERT_ROOT "ecc-client-key.der";
            static const char* eccClientCert = CERT_ROOT "client-ecc-cert.der";
        #endif
    #endif /* HAVE_ECC */
    #ifdef HAVE_ED25519
        #ifdef WOLFSSL_TEST_CERT
            static const char* serverEd25519Cert =
                                         CERT_ROOT "ed25519/server-ed25519.der";
            static const char* caEd25519Cert     =
                                             CERT_ROOT "ed25519/ca-ed25519.der";
        #endif
    #endif
    #ifdef HAVE_ED448
        #ifdef WOLFSSL_TEST_CERT
            static const char* serverEd448Cert =
                                             CERT_ROOT "ed448/server-ed448.der";
            static const char* caEd448Cert     = CERT_ROOT "ed448/ca-ed448.der";
        #endif
    #endif
#endif /* !USE_CERT_BUFFER_* */

#if !defined(NO_ASN_TIME) && !defined(NO_RSA) && defined(WOLFSSL_TEST_CERT) && \
    !defined(NO_FILESYSTEM)
    static const char* certExtNc  =
            CERT_ROOT "test" CERT_PATH_SEP "cert-ext-nc.der";
    static const char* certExtIa  =
            CERT_ROOT "test" CERT_PATH_SEP "cert-ext-ia.der";
    static const char* certExtNct =
            CERT_ROOT "test" CERT_PATH_SEP "cert-ext-nct.der";
#endif

#ifndef NO_WRITE_TEMP_FILES
#ifdef HAVE_ECC
    #ifdef WOLFSSL_CERT_GEN
         static const char* certEccPemFile =   CERT_WRITE_TEMP_DIR "certecc.pem";
         static const char* certEccDerFile = CERT_WRITE_TEMP_DIR "certecc.der";
    #endif
    #if defined(WOLFSSL_CERT_GEN) && !defined(NO_RSA)
        static const char* certEccRsaPemFile = CERT_WRITE_TEMP_DIR "certeccrsa.pem";
        static const char* certEccRsaDerFile = CERT_WRITE_TEMP_DIR "certeccrsa.der";
    #endif
    #if defined(HAVE_ECC_KEY_EXPORT) && !defined(WC_NO_RNG) && \
        !defined(WOLF_CRYPTO_CB_ONLY_ECC)
        static const char* eccCaKeyPemFile  = CERT_WRITE_TEMP_DIR "ecc-key.pem";
        static const char* eccPubKeyDerFile = CERT_WRITE_TEMP_DIR "ecc-public-key.der";
        static const char* eccCaKeyTempFile = CERT_WRITE_TEMP_DIR "ecc-key.der";
    #if defined(HAVE_PKCS8) && !defined(WC_NO_RNG) && \
       !defined(WOLF_CRYPTO_CB_ONLY_ECC)
        static const char* eccPkcs8KeyDerFile = CERT_WRITE_TEMP_DIR "ecc-key-pkcs8.der";
    #endif
    #endif /* HAVE_ECC_KEY_EXPORT */
#endif /* HAVE_ECC */

#ifndef NO_RSA
    #ifdef WOLFSSL_CERT_GEN
        static const char* otherCertDerFile = CERT_WRITE_TEMP_DIR "othercert.der";
        static const char* certDerFile = CERT_WRITE_TEMP_DIR "cert.der";
        static const char* otherCertPemFile = CERT_WRITE_TEMP_DIR "othercert.pem";
        static const char* certPemFile = CERT_WRITE_TEMP_DIR "cert.pem";
        #if defined(WOLFSSL_CERT_REQ) && !defined(WOLFSSL_NO_MALLOC)
            static const char* certReqDerFile = CERT_WRITE_TEMP_DIR "certreq.der";
            static const char* certReqPemFile = CERT_WRITE_TEMP_DIR "certreq.pem";
        #endif
    #endif
#endif /* !NO_RSA */

#if !defined(NO_RSA) || !defined(NO_DSA)
    #ifdef WOLFSSL_KEY_GEN
        static const char* keyDerFile = CERT_WRITE_TEMP_DIR "key.der";
        static const char* keyPemFile = CERT_WRITE_TEMP_DIR "key.pem";
    #endif
#endif

#endif /* !NO_WRITE_TEMP_FILES */
#endif /* !NO_FILESYSTEM */


#if defined(WOLFSSL_CERT_GEN) && (!defined(NO_RSA) || defined(HAVE_ECC)) || \
  (defined(WOLFSSL_TEST_CERT) && (defined(HAVE_ED25519) || defined(HAVE_ED448)))
static CertName certDefaultName;
static void initDefaultName(void)
{
#if defined(WOLFSSL_MULTI_ATTRIB) && defined(WOLFSSL_TEST_CERT)
    NameAttrib* n;
#endif

    XMEMCPY(certDefaultName.country, "US", sizeof("US"));
    certDefaultName.countryEnc = CTC_PRINTABLE;
    XMEMCPY(certDefaultName.state, "Oregon", sizeof("Oregon"));
    certDefaultName.stateEnc = CTC_UTF8;
    XMEMCPY(certDefaultName.street, "Main St", sizeof("Main St"));
    certDefaultName.streetEnc = CTC_UTF8;
    XMEMCPY(certDefaultName.locality, "Portland", sizeof("Portland"));
    certDefaultName.localityEnc = CTC_UTF8;
    XMEMCPY(certDefaultName.sur, "Test", sizeof("Test"));
    certDefaultName.surEnc = CTC_UTF8;
    XMEMCPY(certDefaultName.org, "wolfSSL", sizeof("wolfSSL"));
    certDefaultName.orgEnc = CTC_UTF8;
    XMEMCPY(certDefaultName.unit, "Development", sizeof("Development"));
    certDefaultName.unitEnc = CTC_UTF8;
    XMEMCPY(certDefaultName.commonName, "www.wolfssl.com", sizeof("www.wolfssl.com"));
    certDefaultName.commonNameEnc = CTC_UTF8;
    XMEMCPY(certDefaultName.serialDev, "wolfSSL12345", sizeof("wolfSSL12345"));
    certDefaultName.serialDevEnc = CTC_PRINTABLE;
    XMEMCPY(certDefaultName.postalCode, "12-456", sizeof("12-456"));
    certDefaultName.postalCodeEnc = CTC_PRINTABLE;
#ifdef WOLFSSL_CERT_EXT
    XMEMCPY(certDefaultName.busCat, "Private Organization", sizeof("Private Organization"));
    certDefaultName.busCatEnc = CTC_UTF8;
    XMEMCPY(certDefaultName.joiSt, "US", sizeof("US"));
    certDefaultName.joiStEnc = CTC_PRINTABLE;
    XMEMCPY(certDefaultName.joiC, "Oregon", sizeof("Oregon"));
    certDefaultName.joiCEnc = CTC_PRINTABLE;
#endif
    XMEMCPY(certDefaultName.email, "info@wolfssl.com", sizeof("info@wolfssl.com"));
    XMEMCPY(certDefaultName.userId, "TestUserID", sizeof("TestUserID"));
    certDefaultName.userIdEnc = CTC_PRINTABLE;

#if defined(WOLFSSL_MULTI_ATTRIB) && defined(WOLFSSL_TEST_CERT)
    /* test having additional OUs and setting DC */
    n = &certDefaultName.name[0];
    n->id   = ASN_ORGUNIT_NAME;
    n->type = CTC_UTF8;
    n->sz   = sizeof("Development-2");
    XMEMCPY(n->value, "Development-2", sizeof("Development-2"));

    #if CTC_MAX_ATTRIB > 3
    n = &certDefaultName.name[1];
    n->id   = ASN_DOMAIN_COMPONENT;
    n->type = CTC_UTF8;
    n->sz   = sizeof("com");
    XMEMCPY(n->value, "com", sizeof("com"));

    n = &certDefaultName.name[2];
    n->id   = ASN_DOMAIN_COMPONENT;
    n->type = CTC_UTF8;
    n->sz   = sizeof("wolfssl");
    XMEMCPY(n->value, "wolfssl", sizeof("wolfssl"));
    #endif
#endif /* WOLFSSL_MULTI_ATTRIB && WOLFSSL_TEST_CERT */

#ifdef WOLFSSL_CUSTOM_OID
    /* TODO: Add test case for custom OID's */
#endif
}

#ifdef WOLFSSL_CERT_EXT
    #if ((defined(HAVE_ED25519) || defined(HAVE_ED448)) && \
                                defined(WOLFSSL_TEST_CERT)) || defined(HAVE_ECC)
    WOLFSSL_SMALL_STACK_STATIC const char certKeyUsage[] =
        "digitalSignature,nonRepudiation";
    #endif
    #if defined(WOLFSSL_CERT_REQ) && !defined(NO_RSA)
        WOLFSSL_SMALL_STACK_STATIC const char certKeyUsage2[] =
        "digitalSignature,nonRepudiation,keyEncipherment,keyAgreement";
    #endif
#endif /* WOLFSSL_CERT_EXT */
#endif /* WOLFSSL_CERT_GEN */

#ifndef NO_RSA

#if !defined(NO_ASN_TIME) && !defined(NO_RSA) && defined(WOLFSSL_TEST_CERT) && \
    !defined(NO_FILESYSTEM)
static byte minSerial[] = { 0x02, 0x01, 0x01 };
static byte minName[] = { 0x30, 0x00 };
static byte nameBad[] = {
    0x30, 0x08,
          0x31, 0x06,
                0x30, 0x04,
                      0x06, 0x02,
                            0x55, 0x04,
};
static byte minDates[] = {
    0x30, 0x1e,
          0x17, 0x0d,
                0x31, 0x38, 0x30, 0x34, 0x31, 0x33, 0x31, 0x35,
                0x32, 0x33, 0x31, 0x30, 0x5a,
          0x17, 0x0d,
                0x32, 0x31, 0x30, 0x31, 0x30, 0x37, 0x31, 0x35,
                0x32, 0x33, 0x31, 0x30, 0x5a
};
static byte minPubKey[] = {
    0x30, 0x1c,
          0x30, 0x0d,
                0x06, 0x09,
                      0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01,
                      0x01,
                0x05, 0x00,
          0x03, 0x0b,
                0x00, 0x30, 0x08,
                            0x02, 0x01,
                                  0x03,
                            0x02, 0x03,
                                  0x01, 0x00, 0x01
};
static byte minSigAlg[] = {
    0x30, 0x0d,
          0x06, 0x09,
                0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01,
                0x0b,
          0x05, 0x00
};
static byte minSig[] = {
    0x03, 0x01,
          0x00
};

static int add_seq(byte* certData, int offset, byte* data, byte length)
{
    XMEMMOVE(certData + offset + 2, data, length);
    certData[offset++] = 0x30;
    certData[offset++] = length;
    return offset + length;
}
static int add_data(byte* certData, int offset, byte* data, byte length)
{
    XMEMCPY(certData + offset, data, length);
    return offset + length;
}

static int cert_asn1_test(void)
{
    int ret;
    int len[3];
    DecodedCert cert;
    byte certData[114];
    byte* badCert = NULL;

    len[2] = add_data(certData, 0, minSerial, (byte)sizeof(minSerial));
    len[2] = add_data(certData, len[2], minSigAlg, (byte)sizeof(minSigAlg));
    len[2] = add_data(certData, len[2], minName, (byte)sizeof(minName));
    len[2] = add_data(certData, len[2], minDates, (byte)sizeof(minDates));
    len[2] = add_data(certData, len[2], minName, (byte)sizeof(minName));
    len[2] = add_data(certData, len[2], minPubKey, (byte)sizeof(minPubKey));
    len[1] = add_seq(certData, 0, certData, len[2]);
    len[1] = add_data(certData, len[1], minSigAlg, (byte)sizeof(minSigAlg));
    len[1] = add_data(certData, len[1], minSig, (byte)sizeof(minSig));
    len[0] = add_seq(certData, 0, certData, len[1]);

    /* Minimal good certificate */
    InitDecodedCert(&cert, certData, len[0], 0);
    ret = ParseCert(&cert, CERT_TYPE, NO_VERIFY, NULL);
    FreeDecodedCert(&cert);
    if (ret != 0) {
        ERROR_OUT(-7300, done);
    }

    /* Bad issuer name */
    len[2] = add_data(certData, 0, minSerial, (byte)sizeof(minSerial));
    len[2] = add_data(certData, len[2], minSigAlg, (byte)sizeof(minSigAlg));
    len[2] = add_data(certData, len[2], nameBad, (byte)sizeof(nameBad));
    len[2] = add_data(certData, len[2], minDates, (byte)sizeof(minDates));
    len[2] = add_data(certData, len[2], minName, (byte)sizeof(minName));
    len[2] = add_data(certData, len[2], minPubKey, (byte)sizeof(minPubKey));
    len[1] = add_seq(certData, 0, certData, len[2]);
    len[1] = add_data(certData, len[1], minSigAlg, (byte)sizeof(minSigAlg));
    len[1] = add_data(certData, len[1], minSig, (byte)sizeof(minSig));
    len[0] = add_seq(certData, 0, certData, len[1]);
    /* Put data into allocated buffer to allow access error checking. */
    badCert = (byte*)XMALLOC(len[0], HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    XMEMCPY(badCert, certData, len[0]);
    InitDecodedCert(&cert, badCert, len[0], 0);
    ret = ParseCert(&cert, CERT_TYPE, NO_VERIFY, NULL);
    FreeDecodedCert(&cert);
    if (ret != ASN_PARSE_E) {
        ERROR_OUT(-7301, done);
    }
    XFREE(badCert, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    badCert = NULL;
    ret = 0;

done:
    if (badCert != NULL)
        XFREE(badCert, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    return ret;
}

WOLFSSL_TEST_SUBROUTINE int cert_test(void)
{
#if !defined(NO_FILESYSTEM)
    DecodedCert cert;
    byte*       tmp;
    size_t      bytes;
    XFILE       file;
    int         ret;

    tmp = (byte*)XMALLOC(FOURK_BUF, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    if (tmp == NULL)
        return -7400;

    /* Certificate with Name Constraints extension. */
    file = XFOPEN(certExtNc, "rb");
    if (!file) {
        ERROR_OUT(-7401, done);
    }
    bytes = XFREAD(tmp, 1, FOURK_BUF, file);
    XFCLOSE(file);
    InitDecodedCert(&cert, tmp, (word32)bytes, 0);
    ret = ParseCert(&cert, CERT_TYPE, NO_VERIFY, NULL);
    if (ret != 0) {
        ERROR_OUT(-7402, done);
    }
    FreeDecodedCert(&cert);

    /* Certificate with Inhibit Any Policy extension. */
    file = XFOPEN(certExtIa, "rb");
    if (!file) {
        ERROR_OUT(-7403, done);
    }
    bytes = XFREAD(tmp, 1, FOURK_BUF, file);
    XFCLOSE(file);
    InitDecodedCert(&cert, tmp, (word32)bytes, 0);
    ret = ParseCert(&cert, CERT_TYPE, NO_VERIFY, NULL);
    if (ret != 0) {
        ERROR_OUT(-7404, done);
    }
    FreeDecodedCert(&cert);

    /* Certificate with Netscape Certificate Type extension. */
    file = XFOPEN(certExtNct, "rb");
    if (!file) {
        ERROR_OUT(-7405, done);
    }
    bytes = XFREAD(tmp, 1, FOURK_BUF, file);
    XFCLOSE(file);
    InitDecodedCert(&cert, tmp, (word32)bytes, 0);
    ret = ParseCert(&cert, CERT_TYPE, NO_VERIFY, NULL);
#ifndef IGNORE_NETSCAPE_CERT_TYPE
    if (ret != 0) {
        ERROR_OUT(-7406, done);
    }
#else
    if (ret != ASN_CRIT_EXT_E) {
        ERROR_OUT(-7407, done);
    }
    ret = 0;
#endif

done:
    FreeDecodedCert(&cert);
    XFREE(tmp, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
#endif /* !NO_FILESYSTEM */

    if (ret == 0)
        ret = cert_asn1_test();

    return ret;
}
#endif /* WOLFSSL_TEST_CERT */

#if defined(WOLFSSL_CERT_EXT) && defined(WOLFSSL_TEST_CERT) && \
   !defined(NO_FILESYSTEM) && defined(WOLFSSL_CERT_GEN)
WOLFSSL_TEST_SUBROUTINE int certext_test(void)
{
    DecodedCert cert;
    byte*       tmp;
    size_t      bytes;
    XFILE       file;
    int         ret;

    /* created from rsa_test : othercert.der */
    byte skid_rsa[]   = "\x33\xD8\x45\x66\xD7\x68\x87\x18\x7E\x54"
                        "\x0D\x70\x27\x91\xC7\x26\xD7\x85\x65\xC0";

    /* created from rsa_test : othercert.der */
    byte akid_rsa[] = "\x27\x8E\x67\x11\x74\xC3\x26\x1D\x3F\xED"
                      "\x33\x63\xB3\xA4\xD8\x1D\x30\xE5\xE8\xD5";

#ifdef HAVE_ECC
   /* created from ecc_test_cert_gen : certecc.der */
#ifdef ENABLE_ECC384_CERT_GEN_TEST
    /* Authority key id from ./certs/ca-ecc384-cert.pem */
    byte akid_ecc[] = "\xAB\xE0\xC3\x26\x4C\x18\xD4\x72\xBB\xD2"
                      "\x84\x8C\x9C\x0A\x05\x92\x80\x12\x53\x52";
#else
    /* Authority key id from ./certs/ca-ecc-cert.pem */
    byte akid_ecc[] = "\x56\x8E\x9A\xC3\xF0\x42\xDE\x18\xB9\x45"
                      "\x55\x6E\xF9\x93\xCF\xEA\xC3\xF3\xA5\x21";
#endif
#endif /* HAVE_ECC */

    /* created from rsa_test : cert.der */
    byte kid_ca[] = "\x33\xD8\x45\x66\xD7\x68\x87\x18\x7E\x54"
                    "\x0D\x70\x27\x91\xC7\x26\xD7\x85\x65\xC0";

    tmp = (byte*)XMALLOC(FOURK_BUF, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    if (tmp == NULL)
        return -7500;

    /* load othercert.der (Cert signed by an authority) */
    file = XFOPEN(otherCertDerFile, "rb");
    if (!file) {
        XFREE(tmp, HEAP_HINT ,DYNAMIC_TYPE_TMP_BUFFER);
        return -7501;
    }

    bytes = XFREAD(tmp, 1, FOURK_BUF, file);
    XFCLOSE(file);

    InitDecodedCert(&cert, tmp, (word32)bytes, 0);

    ret = ParseCert(&cert, CERT_TYPE, NO_VERIFY, 0);
    if (ret != 0)
        return -7502;

    /* check the SKID from a RSA certificate */
    if (XMEMCMP(skid_rsa, cert.extSubjKeyId, sizeof(cert.extSubjKeyId)))
        return -7503;

    /* check the AKID from an RSA certificate */
    if (XMEMCMP(akid_rsa, cert.extAuthKeyId, sizeof(cert.extAuthKeyId)))
        return -7504;

    /* check the Key Usage from an RSA certificate */
    if (!cert.extKeyUsageSet)
        return -7505;

    if (cert.extKeyUsage != (KEYUSE_KEY_ENCIPHER|KEYUSE_KEY_AGREE))
        return -7506;

    /* check the CA Basic Constraints from an RSA certificate */
    if (cert.isCA)
        return -7507;

#ifndef WOLFSSL_SEP /* test only if not using SEP policies */
    /* check the Certificate Policies Id */
    if (cert.extCertPoliciesNb != 1)
        return -7508;

    if (strncmp(cert.extCertPolicies[0], "2.16.840.1.101.3.4.1.42", 23))
        return -7509;
#endif

    FreeDecodedCert(&cert);

#ifdef HAVE_ECC
    /* load certecc.der (Cert signed by our ECC CA test in ecc_test_cert_gen) */
    file = XFOPEN(certEccDerFile, "rb");
    if (!file) {
        XFREE(tmp, HEAP_HINT ,DYNAMIC_TYPE_TMP_BUFFER);
        return -7510;
    }

    bytes = XFREAD(tmp, 1, FOURK_BUF, file);
    XFCLOSE(file);

    InitDecodedCert(&cert, tmp, (word32)bytes, 0);

    ret = ParseCert(&cert, CERT_TYPE, NO_VERIFY, 0);
    if (ret != 0)
        return -7511;

    /* check the SKID from a ECC certificate - generated dynamically */

    /* check the AKID from an ECC certificate */
    if (XMEMCMP(akid_ecc, cert.extAuthKeyId, sizeof(cert.extAuthKeyId)))
        return -7512;

    /* check the Key Usage from an ECC certificate */
    if (!cert.extKeyUsageSet)
        return -7513;

    if (cert.extKeyUsage != (KEYUSE_DIGITAL_SIG|KEYUSE_CONTENT_COMMIT))
        return -7514;

    /* check the CA Basic Constraints from an ECC certificate */
    if (cert.isCA)
        return -7515;

#ifndef WOLFSSL_SEP /* test only if not using SEP policies */
    /* check the Certificate Policies Id */
    if (cert.extCertPoliciesNb != 2)
        return -7516;

    if (strncmp(cert.extCertPolicies[0], "2.4.589440.587.101.2.1.9632587.1", 32))
        return -7517;

    if (strncmp(cert.extCertPolicies[1], "1.2.13025.489.1.113549", 22))
        return -7518;
#endif

    FreeDecodedCert(&cert);
#endif /* HAVE_ECC */

    /* load cert.der (self signed certificate) */
    file = XFOPEN(certDerFile, "rb");
    if (!file) {
        XFREE(tmp, HEAP_HINT ,DYNAMIC_TYPE_TMP_BUFFER);
        return -7519;
    }

    bytes = XFREAD(tmp, 1, FOURK_BUF, file);
    XFCLOSE(file);

    InitDecodedCert(&cert, tmp, (word32)bytes, 0);

    ret = ParseCert(&cert, CERT_TYPE, NO_VERIFY, 0);
    if (ret != 0)
        return -7520;

    /* check the SKID from a CA certificate */
    if (XMEMCMP(kid_ca, cert.extSubjKeyId, sizeof(cert.extSubjKeyId)))
        return -7521;

    /* check the AKID from an CA certificate */
    if (XMEMCMP(kid_ca, cert.extAuthKeyId, sizeof(cert.extAuthKeyId)))
        return -7522;

    /* check the Key Usage from CA certificate */
    if (!cert.extKeyUsageSet)
        return -7523;

    if (cert.extKeyUsage != (KEYUSE_KEY_CERT_SIGN|KEYUSE_CRL_SIGN))
        return -7524;

    /* check the CA Basic Constraints CA certificate */
    if (!cert.isCA)
        return -7525;

#ifndef WOLFSSL_SEP /* test only if not using SEP policies */
    /* check the Certificate Policies Id */
    if (cert.extCertPoliciesNb != 2)
        return -7526;

    if (strncmp(cert.extCertPolicies[0], "2.16.840.1.101.3.4.1.42", 23))
        return -7527;

    if (strncmp(cert.extCertPolicies[1], "1.2.840.113549.1.9.16.6.5", 25))
        return -7528;
#endif

    FreeDecodedCert(&cert);
    XFREE(tmp, HEAP_HINT ,DYNAMIC_TYPE_TMP_BUFFER);

    return 0;
}
#endif /* WOLFSSL_CERT_EXT && WOLFSSL_TEST_CERT &&
          !NO_FILESYSTEM && WOLFSSL_CERT_GEN */

#if defined(WOLFSSL_CERT_GEN_CACHE) && defined(WOLFSSL_TEST_CERT) && \
    defined(WOLFSSL_CERT_EXT) && defined(WOLFSSL_CERT_GEN)
WOLFSSL_TEST_SUBROUTINE int decodedCertCache_test(void)
{
    int ret = 0;
    Cert cert;
    FILE* file;
    byte* der;
    word32 derSz;

    derSz = FOURK_BUF;
    der = (byte *)XMALLOC(FOURK_BUF, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    if (der == NULL)
        ret = -7600;

    if (ret == 0) {
        /* load cert.der */
        file = XFOPEN(certDerFile, "rb");
        if (file != NULL) {
            derSz = (word32)XFREAD(der, 1, FOURK_BUF, file);
            XFCLOSE(file);
        }
        else
            ret = -7601;
    }

    if (ret == 0) {
        if (wc_InitCert_ex(&cert, HEAP_HINT, devId)) {
            ret = -7602;
        }
    }

    if (ret == 0) {
        ret = wc_SetSubjectBuffer(&cert, der, derSz);
    }

    if (ret == 0) {
        if(wc_SetSubjectBuffer(NULL, der, derSz) != BAD_FUNC_ARG)
            ret = -7603;
    }

    if (ret == 0) {
        if (wc_SetSubjectRaw(&cert, der, derSz) != 0)
            ret = -7604;
    }

    if (ret == 0) {
        if(wc_SetSubjectRaw(NULL, der, derSz) != BAD_FUNC_ARG)
            ret = -7605;
    }

    if (ret == 0) {
        if(wc_SetIssuerBuffer(&cert, der, derSz) != 0)
            ret = -7606;
    }

    if (ret == 0) {
        if(wc_SetIssuerBuffer(NULL, der, derSz) != BAD_FUNC_ARG)
            ret = -7607;
    }

    if (ret == 0) {
        if(wc_SetIssuerRaw(&cert, der, derSz) != 0)
            ret = -7608;
    }

    if (ret == 0) {
        if(wc_SetIssuerRaw(NULL, der, derSz) != BAD_FUNC_ARG)
            ret = -7609;
    }

#ifdef WOLFSSL_ALT_NAMES
    if (ret == 0) {
        if(wc_SetAltNamesBuffer(&cert, der, derSz) != 0)
            ret = -7610;
    }

    if (ret == 0) {
        if(wc_SetAltNamesBuffer(NULL, der, derSz) != BAD_FUNC_ARG)
            ret = -7611;
    }

    if (ret == 0) {
        if(wc_SetDatesBuffer(&cert, der, derSz) != 0)
            ret = -7612;
    }

    if (ret == 0) {
        if(wc_SetDatesBuffer(NULL, der, derSz) != BAD_FUNC_ARG)
            ret = -7613;
    }
#endif

    if (ret == 0) {
        if(wc_SetAuthKeyIdFromCert(&cert, der, derSz) != 0)
            ret = -7614;
    }

    if (ret == 0) {
        if(wc_SetAuthKeyIdFromCert(NULL, der, derSz) != BAD_FUNC_ARG)
            ret = -7615;
    }

    wc_SetCert_Free(&cert);
    if (ret == 0) {
        if(cert.decodedCert != NULL)
            ret = -7616;
    }

    XFREE(der, HEAP_HINT ,DYNAMIC_TYPE_TMP_BUFFER);

    return ret;
}
#endif /* defined(WOLFSSL_CERT_GEN_CACHE) && defined(WOLFSSL_TEST_CERT) &&
          defined(WOLFSSL_CERT_EXT) && defined(WOLFSSL_CERT_GEN) */

#define RSA_TEST_BYTES 512 /* up to 4096-bit key */

#if !defined(NO_ASN) && !defined(WOLFSSL_RSA_PUBLIC_ONLY) && \
                                               !defined(WOLFSSL_RSA_VERIFY_ONLY)
static int rsa_flatten_test(RsaKey* key)
{
    int    ret;
    byte   e[RSA_TEST_BYTES];
    byte   n[RSA_TEST_BYTES];
    word32 eSz = sizeof(e);
    word32 nSz = sizeof(n);

    /* Parameter Validation testing. */
    ret = wc_RsaFlattenPublicKey(NULL, e, &eSz, n, &nSz);
#ifdef HAVE_USER_RSA
    /* Implementation using IPP Libraries returns:
     *     -101 = USER_CRYPTO_ERROR
     */
    if (ret == 0)
#else
    if (ret != BAD_FUNC_ARG)
#endif
        return -7620;
    ret = wc_RsaFlattenPublicKey(key, NULL, &eSz, n, &nSz);
#ifdef HAVE_USER_RSA
    /* Implementation using IPP Libraries returns:
     *     -101 = USER_CRYPTO_ERROR
     */
    if (ret == 0)
#else
    if (ret != BAD_FUNC_ARG)
#endif
        return -7621;
    ret = wc_RsaFlattenPublicKey(key, e, NULL, n, &nSz);
#ifdef HAVE_USER_RSA
    /* Implementation using IPP Libraries returns:
     *     -101 = USER_CRYPTO_ERROR
     */
    if (ret == 0)
#else
    if (ret != BAD_FUNC_ARG)
#endif
        return -7622;
    ret = wc_RsaFlattenPublicKey(key, e, &eSz, NULL, &nSz);
#ifdef HAVE_USER_RSA
    /* Implementation using IPP Libraries returns:
     *     -101 = USER_CRYPTO_ERROR
     */
    if (ret == 0)
#else
    if (ret != BAD_FUNC_ARG)
#endif
        return -7623;
    ret = wc_RsaFlattenPublicKey(key, e, &eSz, n, NULL);
#ifdef HAVE_USER_RSA
    /* Implementation using IPP Libraries returns:
     *     -101 = USER_CRYPTO_ERROR
     */
    if (ret == 0)
#else
    if (ret != BAD_FUNC_ARG)
#endif
        return -7624;
    ret = wc_RsaFlattenPublicKey(key, e, &eSz, n, &nSz);
    if (ret != 0)
        return -7625;
    eSz = 0;
    ret = wc_RsaFlattenPublicKey(key, e, &eSz, n, &nSz);
#ifdef HAVE_USER_RSA
    /* Implementation using IPP Libraries returns:
     *     -101 = USER_CRYPTO_ERROR
     */
    if (ret == 0)
#elif defined(HAVE_FIPS) && \
      (!defined(HAVE_FIPS_VERSION) || (HAVE_FIPS_VERSION < 2))
    if (ret != 0)
#else
    if (ret != RSA_BUFFER_E)
#endif
        return -7626;
    eSz = sizeof(e);
    nSz = 0;
    ret = wc_RsaFlattenPublicKey(key, e, &eSz, n, &nSz);
#ifdef HAVE_USER_RSA
    /* Implementation using IPP Libraries returns:
     *     -101 = USER_CRYPTO_ERROR
     */
    if (ret == 0)
#else
    if (ret != RSA_BUFFER_E)
#endif
        return -7627;

    return 0;
}
#endif /* NO_ASN */

#if !defined(HAVE_FIPS) && !defined(HAVE_USER_RSA) && !defined(NO_ASN) \
    && !defined(WOLFSSL_RSA_VERIFY_ONLY)
static int rsa_export_key_test(RsaKey* key)
{
    int ret;
    byte e[3];
    word32 eSz = sizeof(e);
    byte n[RSA_TEST_BYTES];
    word32 nSz = sizeof(n);
    byte d[RSA_TEST_BYTES];
    word32 dSz = sizeof(d);
    byte p[RSA_TEST_BYTES/2];
    word32 pSz = sizeof(p);
    byte q[RSA_TEST_BYTES/2];
    word32 qSz = sizeof(q);
    word32 zero = 0;

    ret = wc_RsaExportKey(NULL, e, &eSz, n, &nSz, d, &dSz, p, &pSz, q, &qSz);
    if (ret != BAD_FUNC_ARG)
        return -7630;
    ret = wc_RsaExportKey(key, NULL, &eSz, n, &nSz, d, &dSz, p, &pSz, q, &qSz);
    if (ret != BAD_FUNC_ARG)
        return -7631;
    ret = wc_RsaExportKey(key, e, NULL, n, &nSz, d, &dSz, p, &pSz, q, &qSz);
    if (ret != BAD_FUNC_ARG)
        return -7632;
    ret = wc_RsaExportKey(key, e, &eSz, NULL, &nSz, d, &dSz, p, &pSz, q, &qSz);
    if (ret != BAD_FUNC_ARG)
        return -7633;
    ret = wc_RsaExportKey(key, e, &eSz, n, NULL, d, &dSz, p, &pSz, q, &qSz);
    if (ret != BAD_FUNC_ARG)
        return -7634;
    ret = wc_RsaExportKey(key, e, &eSz, n, &nSz, NULL, &dSz, p, &pSz, q, &qSz);
    if (ret != BAD_FUNC_ARG)
        return -7635;
    ret = wc_RsaExportKey(key, e, &eSz, n, &nSz, d, NULL, p, &pSz, q, &qSz);
    if (ret != BAD_FUNC_ARG)
        return -7636;
    ret = wc_RsaExportKey(key, e, &eSz, n, &nSz, d, &dSz, NULL, &pSz, q, &qSz);
    if (ret != BAD_FUNC_ARG)
        return -7637;
    ret = wc_RsaExportKey(key, e, &eSz, n, &nSz, d, &dSz, p, NULL, q, &qSz);
    if (ret != BAD_FUNC_ARG)
        return -7638;
    ret = wc_RsaExportKey(key, e, &eSz, n, &nSz, d, &dSz, p, &pSz, NULL, &qSz);
    if (ret != BAD_FUNC_ARG)
        return -7639;
    ret = wc_RsaExportKey(key, e, &eSz, n, &nSz, d, &dSz, p, &pSz, q, NULL);
    if (ret != BAD_FUNC_ARG)
        return -7640;

    ret = wc_RsaExportKey(key, e, &zero, n, &nSz, d, &dSz, p, &pSz, q, &qSz);
    if (ret != RSA_BUFFER_E)
        return -7641;
    ret = wc_RsaExportKey(key, e, &eSz, n, &zero, d, &dSz, p, &pSz, q, &qSz);
    if (ret != RSA_BUFFER_E)
        return -7642;
#ifndef WOLFSSL_RSA_PUBLIC_ONLY
    ret = wc_RsaExportKey(key, e, &eSz, n, &nSz, d, &zero, p, &pSz, q, &qSz);
    if (ret != RSA_BUFFER_E)
        return -7643;
    ret = wc_RsaExportKey(key, e, &eSz, n, &nSz, d, &dSz, p, &zero, q, &qSz);
    if (ret != RSA_BUFFER_E)
        return -7644;
    ret = wc_RsaExportKey(key, e, &eSz, n, &nSz, d, &dSz, p, &pSz, q, &zero);
    if (ret != RSA_BUFFER_E)
        return -7645;
#endif /* WOLFSSL_RSA_PUBLIC_ONLY */

    ret = wc_RsaExportKey(key, e, &eSz, n, &nSz, d, &dSz, p, &pSz, q, &qSz);
    if (ret != 0)
        return -7646;

    return 0;
}
#endif /* !HAVE_FIPS && !USER_RSA && !NO_ASN */

#ifndef NO_SIG_WRAPPER
static int rsa_sig_test(RsaKey* key, word32 keyLen, int modLen, WC_RNG* rng)
{
    int ret;
    word32 sigSz;
    WOLFSSL_SMALL_STACK_STATIC const byte in[] = TEST_STRING;
    WOLFSSL_SMALL_STACK_STATIC const byte hash[] = {
        0xf2, 0x02, 0x95, 0x65, 0xcb, 0xf6, 0x2a, 0x59,
        0x39, 0x2c, 0x05, 0xff, 0x0e, 0x29, 0xaf, 0xfe,
        0x47, 0x33, 0x8c, 0x99, 0x8d, 0x58, 0x64, 0x83,
        0xa6, 0x58, 0x0a, 0x33, 0x0b, 0x84, 0x5f, 0x5f
    };
    WOLFSSL_SMALL_STACK_STATIC const byte hashEnc[] = {
        0x30, 0x31, 0x30, 0x0d, 0x06, 0x09, 0x60, 0x86,
        0x48, 0x01, 0x65, 0x03, 0x04, 0x02, 0x01, 0x05,
        0x00, 0x04, 0x20,

        0xf2, 0x02, 0x95, 0x65, 0xcb, 0xf6, 0x2a, 0x59,
        0x39, 0x2c, 0x05, 0xff, 0x0e, 0x29, 0xaf, 0xfe,
        0x47, 0x33, 0x8c, 0x99, 0x8d, 0x58, 0x64, 0x83,
        0xa6, 0x58, 0x0a, 0x33, 0x0b, 0x84, 0x5f, 0x5f
    };
    word32 inLen = (word32)XSTRLEN((char*)in);
    byte   out[RSA_TEST_BYTES];

    /* Parameter Validation testing. */
    ret = wc_SignatureGetSize(WC_SIGNATURE_TYPE_NONE, key, keyLen);
    if (ret != BAD_FUNC_ARG)
        return -7650;
    ret = wc_SignatureGetSize(WC_SIGNATURE_TYPE_RSA, key, 0);
    if (ret != BAD_FUNC_ARG)
        return -7651;

    sigSz = (word32)modLen;
    ret = wc_SignatureGenerate(WC_HASH_TYPE_SHA256, WC_SIGNATURE_TYPE_RSA, NULL,
                               inLen, out, &sigSz, key, keyLen, rng);
    if (ret != BAD_FUNC_ARG)
        return -7652;
    ret = wc_SignatureGenerate(WC_HASH_TYPE_SHA256, WC_SIGNATURE_TYPE_RSA, in,
                               0, out, &sigSz, key, keyLen, rng);
    if (ret != BAD_FUNC_ARG)
        return -7653;
    ret = wc_SignatureGenerate(WC_HASH_TYPE_SHA256, WC_SIGNATURE_TYPE_RSA, in,
                               inLen, NULL, &sigSz, key, keyLen, rng);
    if (ret != BAD_FUNC_ARG)
        return -7654;
    ret = wc_SignatureGenerate(WC_HASH_TYPE_SHA256, WC_SIGNATURE_TYPE_RSA, in,
                               inLen, out, NULL, key, keyLen, rng);
    if (ret != BAD_FUNC_ARG)
        return -7655;
    ret = wc_SignatureGenerate(WC_HASH_TYPE_SHA256, WC_SIGNATURE_TYPE_RSA, in,
                               inLen, out, &sigSz, NULL, keyLen, rng);
    if (ret != BAD_FUNC_ARG)
        return -7656;
    ret = wc_SignatureGenerate(WC_HASH_TYPE_SHA256, WC_SIGNATURE_TYPE_RSA, in,
                               inLen, out, &sigSz, key, 0, rng);
    if (ret != BAD_FUNC_ARG)
        return -7657;
    ret = wc_SignatureGenerate(WC_HASH_TYPE_SHA256, WC_SIGNATURE_TYPE_RSA, in,
                               inLen, out, &sigSz, key, keyLen, NULL);
#ifdef HAVE_USER_RSA
    /* Implementation using IPP Libraries returns:
     *     -101 = USER_CRYPTO_ERROR
     */
    if (ret == 0)
#elif defined(WOLFSSL_AFALG_XILINX_RSA) || defined(WOLFSSL_XILINX_CRYPT)
    /* blinding / rng handled with hardware acceleration */
    if (ret != 0)
#elif defined(WOLFSSL_ASYNC_CRYPT) || defined(WOLF_CRYPTO_CB)
    /* async may not require RNG */
    #if defined(WOLF_CRYPTO_CB_ONLY_RSA)
    if (ret != NO_VALID_DEVID)
    #else
    if (ret != 0 && ret != MISSING_RNG_E)
    #endif
#elif defined(HAVE_FIPS) || !defined(WC_RSA_BLINDING)
    /* FIPS140 implementation does not do blinding */
    if (ret != 0)
#elif defined(WOLFSSL_RSA_PUBLIC_ONLY) || defined(WOLFSSL_RSA_VERIFY_ONLY)
    if (ret != SIG_TYPE_E)
#elif defined(WOLFSSL_CRYPTOCELL)
    /* RNG is handled with the cryptocell */
    if (ret != 0)
#else
    if (ret != MISSING_RNG_E)
#endif
        return -7658;
    sigSz = 0;
    ret = wc_SignatureGenerate(WC_HASH_TYPE_SHA256, WC_SIGNATURE_TYPE_RSA, in,
                               inLen, out, &sigSz, key, keyLen, rng);
    if (ret != BAD_FUNC_ARG)
        return -7659;

    ret = wc_SignatureVerify(WC_HASH_TYPE_SHA256, WC_SIGNATURE_TYPE_RSA, NULL,
                             inLen, out, (word32)modLen, key, keyLen);
    if (ret != BAD_FUNC_ARG)
        return -7660;
    ret = wc_SignatureVerify(WC_HASH_TYPE_SHA256, WC_SIGNATURE_TYPE_RSA, in,
                             0, out, (word32)modLen, key, keyLen);
    if (ret != BAD_FUNC_ARG)
        return -7661;
    ret = wc_SignatureVerify(WC_HASH_TYPE_SHA256, WC_SIGNATURE_TYPE_RSA, in,
                             inLen, NULL, (word32)modLen, key, keyLen);
    if (ret != BAD_FUNC_ARG)
        return -7662;
    ret = wc_SignatureVerify(WC_HASH_TYPE_SHA256, WC_SIGNATURE_TYPE_RSA, in,
                             inLen, out, 0, key, keyLen);
    if (ret != BAD_FUNC_ARG)
        return -7663;
    ret = wc_SignatureVerify(WC_HASH_TYPE_SHA256, WC_SIGNATURE_TYPE_RSA, in,
                             inLen, out, (word32)modLen, NULL, keyLen);
    if (ret != BAD_FUNC_ARG)
        return -7664;
    ret = wc_SignatureVerify(WC_HASH_TYPE_SHA256, WC_SIGNATURE_TYPE_RSA, in,
                             inLen, out, (word32)modLen, key, 0);
    if (ret != BAD_FUNC_ARG)
        return -7665;

#ifndef HAVE_ECC
    ret = wc_SignatureGetSize(WC_SIGNATURE_TYPE_ECC, key, keyLen);
    if (ret != SIG_TYPE_E)
        return -7666;
#endif
#if defined(WOLF_CRYPTO_CB_ONLY_RSA)
    return 0;
#endif
    /* Use APIs. */
    ret = wc_SignatureGetSize(WC_SIGNATURE_TYPE_RSA, key, keyLen);
    if (ret != modLen)
        return -7667;
    ret = wc_SignatureGetSize(WC_SIGNATURE_TYPE_RSA_W_ENC, key, keyLen);
    if (ret != modLen)
        return -7668;

    sigSz = (word32)ret;
#if !defined(WOLFSSL_RSA_PUBLIC_ONLY) && !defined(WOLFSSL_RSA_VERIFY_ONLY)
    XMEMSET(out, 0, sizeof(out));
    ret = wc_SignatureGenerate(WC_HASH_TYPE_SHA256, WC_SIGNATURE_TYPE_RSA, in,
                               inLen, out, &sigSz, key, keyLen, rng);
    if (ret != 0)
        return -7669;

    ret = wc_SignatureVerify(WC_HASH_TYPE_SHA256, WC_SIGNATURE_TYPE_RSA, in,
                             inLen, out, (word32)modLen, key, keyLen);
    if (ret != 0)
        return -7670;

    sigSz = (word32)sizeof(out);
    ret = wc_SignatureGenerate(WC_HASH_TYPE_SHA256, WC_SIGNATURE_TYPE_RSA_W_ENC,
                               in, inLen, out, &sigSz, key, keyLen, rng);
    if (ret != 0)
        return -7671;

    ret = wc_SignatureVerify(WC_HASH_TYPE_SHA256, WC_SIGNATURE_TYPE_RSA_W_ENC,
                             in, inLen, out, (word32)modLen, key, keyLen);
    if (ret != 0)
        return -7672;

    /* Wrong signature type. */
    ret = wc_SignatureVerify(WC_HASH_TYPE_SHA256, WC_SIGNATURE_TYPE_RSA, in,
                             inLen, out, (word32)modLen, key, keyLen);
    if (ret == 0)
        return -7673;

    /* check hash functions */
    sigSz = (word32)sizeof(out);
    ret = wc_SignatureGenerateHash(WC_HASH_TYPE_SHA256, WC_SIGNATURE_TYPE_RSA,
        hash, (int)sizeof(hash), out, &sigSz, key, keyLen, rng);
    if (ret != 0)
        return -7674;

    ret = wc_SignatureVerifyHash(WC_HASH_TYPE_SHA256, WC_SIGNATURE_TYPE_RSA,
        hash, (int)sizeof(hash), out, (word32)modLen, key, keyLen);
    if (ret != 0)
        return -7675;

    sigSz = (word32)sizeof(out);
    ret = wc_SignatureGenerateHash(WC_HASH_TYPE_SHA256, WC_SIGNATURE_TYPE_RSA_W_ENC,
        hashEnc, (int)sizeof(hashEnc), out, &sigSz, key, keyLen, rng);
    if (ret != 0)
        return -7676;

    ret = wc_SignatureVerifyHash(WC_HASH_TYPE_SHA256, WC_SIGNATURE_TYPE_RSA_W_ENC,
        hashEnc, (int)sizeof(hashEnc), out, (word32)modLen, key, keyLen);
    if (ret != 0)
        return -7677;
#else
    (void)hash;
    (void)hashEnc;
#endif /* !WOLFSSL_RSA_PUBLIC_ONLY && !WOLFSSL_RSA_VERIFY_ONLY */

    return 0;
}
#endif /* !NO_SIG_WRAPPER */

#ifdef WC_RSA_NONBLOCK
static int rsa_nb_test(RsaKey* key, const byte* in, word32 inLen, byte* out,
    word32 outSz, byte* plain, word32 plainSz, WC_RNG* rng)
{
    int ret = 0, count;
    int signSz = 0;
    RsaNb nb;
    byte* inlinePlain = NULL;

    /* Enable non-blocking RSA mode - provide context */
    ret = wc_RsaSetNonBlock(key, &nb);
    if (ret != 0)
        return ret;

#ifdef WC_RSA_NONBLOCK_TIME
    /* Enable time based RSA blocking. 8 microseconds max (3.1GHz) */
    ret = wc_RsaSetNonBlockTime(key, 8, 3100);
    if (ret != 0)
        return ret;
#endif

    count = 0;
    do {
        ret = wc_RsaSSL_Sign(in, inLen, out, outSz, key, rng);
        count++; /* track number of would blocks */
        if (ret == FP_WOULDBLOCK) {
            /* do "other" work here */
        }
    } while (ret == FP_WOULDBLOCK);
    if (ret < 0) {
        return ret;
    }
#ifdef DEBUG_WOLFSSL
    printf("RSA non-block sign: %d times\n", count);
#endif
    signSz = ret;

    /* Test non-blocking verify */
    XMEMSET(plain, 0, plainSz);
    count = 0;
    do {
        ret = wc_RsaSSL_Verify(out, (word32)signSz, plain, plainSz, key);
        count++; /* track number of would blocks */
        if (ret == FP_WOULDBLOCK) {
            /* do "other" work here */
        }
    } while (ret == FP_WOULDBLOCK);
    if (ret < 0) {
        return ret;
    }
#ifdef DEBUG_WOLFSSL
    printf("RSA non-block verify: %d times\n", count);
#endif

    if (signSz == ret && XMEMCMP(plain, in, (size_t)ret)) {
        return SIG_VERIFY_E;
    }

    /* Test inline non-blocking verify */
    count = 0;
    do {
        ret = wc_RsaSSL_VerifyInline(out, (word32)signSz, &inlinePlain, key);
        count++; /* track number of would blocks */
        if (ret == FP_WOULDBLOCK) {
            /* do "other" work here */
        }
    } while (ret == FP_WOULDBLOCK);
    if (ret < 0) {
        return ret;
    }
#ifdef DEBUG_WOLFSSL
    printf("RSA non-block inline verify: %d times\n", count);
#endif

    if (signSz == ret && XMEMCMP(inlinePlain, in, (size_t)ret)) {
        return SIG_VERIFY_E;
    }

    /* Disabling non-block RSA mode */
    ret = wc_RsaSetNonBlock(key, NULL);

    (void)count;

    return 0;
}
#endif

#if !defined(HAVE_USER_RSA) && !defined(NO_ASN)
static int rsa_decode_test(RsaKey* keyPub)
{
    int        ret;
    word32     inSz;
    word32     inOutIdx;
    WOLFSSL_SMALL_STACK_STATIC const byte n[2] = { 0x00, 0x23 };
    WOLFSSL_SMALL_STACK_STATIC const byte e[2] = { 0x00, 0x03 };
    WOLFSSL_SMALL_STACK_STATIC const byte good[] = { 0x30, 0x06, 0x02, 0x01, 0x23, 0x02, 0x1,
           0x03 };
    WOLFSSL_SMALL_STACK_STATIC const byte goodAlgId[] = {
            0x30, 0x18, 0x30, 0x16,
            0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x01,
            0x03, 0x09, 0x00, 0x30, 0x06, 0x02, 0x01, 0x23, 0x02, 0x1, 0x03 };
    WOLFSSL_SMALL_STACK_STATIC const byte goodAlgIdNull[] = {
            0x30, 0x1a, 0x30, 0x18,
            0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x01,
            0x05, 0x00, 0x03, 0x09, 0x00, 0x30, 0x06, 0x02, 0x01, 0x23,
            0x02, 0x1, 0x03 };
    WOLFSSL_SMALL_STACK_STATIC const byte badAlgIdNull[] = {
            0x30, 0x1b, 0x30, 0x19,
            0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x01,
            0x05, 0x01, 0x00, 0x03, 0x09, 0x00, 0x30, 0x06, 0x02, 0x01, 0x23,
            0x02, 0x1, 0x03 };
    WOLFSSL_SMALL_STACK_STATIC const byte badNotBitString[] = {
            0x30, 0x18, 0x30, 0x16,
            0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x01,
            0x04, 0x09, 0x00, 0x30, 0x06, 0x02, 0x01, 0x23, 0x02, 0x1, 0x03 };
    WOLFSSL_SMALL_STACK_STATIC const byte badBitStringLen[] = {
            0x30, 0x18, 0x30, 0x16,
            0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x01,
            0x03, 0x0a, 0x00, 0x30, 0x06, 0x02, 0x01, 0x23, 0x02, 0x1, 0x03 };
    WOLFSSL_SMALL_STACK_STATIC const byte badNoSeq[] = {
            0x30, 0x16, 0x30, 0x14,
            0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x01,
            0x07, 0x00, 0x02, 0x01, 0x23, 0x02, 0x1, 0x03 };
    WOLFSSL_SMALL_STACK_STATIC const byte badNoObj[] = {
            0x30, 0x0f, 0x30, 0x0d, 0x05, 0x00, 0x03, 0x09, 0x00, 0x30, 0x06,
            0x02, 0x01, 0x23, 0x02, 0x1, 0x03 };
    WOLFSSL_SMALL_STACK_STATIC const byte badIntN[] = {
            0x30, 0x06, 0x02, 0x05, 0x23, 0x02, 0x1, 0x03 };
    WOLFSSL_SMALL_STACK_STATIC const byte badNotIntE[] = {
            0x30, 0x06, 0x02, 0x01, 0x23, 0x04, 0x1, 0x03 };
    WOLFSSL_SMALL_STACK_STATIC const byte badLength[] = {
            0x30, 0x04, 0x02, 0x01, 0x23, 0x02, 0x1, 0x03 };
    WOLFSSL_SMALL_STACK_STATIC const byte badBitStrNoZero[] = {
            0x30, 0x17, 0x30, 0x15,
            0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x01,
            0x03, 0x08, 0x30, 0x06, 0x02, 0x01, 0x23, 0x02, 0x1, 0x03 };

    ret = wc_InitRsaKey(keyPub, NULL);
    if (ret != 0)
        return -7690;

    /* Parameter Validation testing. */
    ret = wc_RsaPublicKeyDecodeRaw(NULL, sizeof(n), e, sizeof(e), keyPub);
    if (ret != BAD_FUNC_ARG) {
        ret = -7691;
        goto done;
    }
    ret = wc_RsaPublicKeyDecodeRaw(n, sizeof(n), NULL, sizeof(e), keyPub);
    if (ret != BAD_FUNC_ARG) {
        ret = -7692;
        goto done;
    }
    ret = wc_RsaPublicKeyDecodeRaw(n, sizeof(n), e, sizeof(e), NULL);
    if (ret != BAD_FUNC_ARG) {
        ret = -7693;
        goto done;
    }
    ret = wc_RsaPublicKeyDecodeRaw(n, (word32)-1, e, sizeof(e), keyPub);
#if !defined(WOLFSSL_SP_MATH) & !defined(WOLFSSL_SP_MATH_ALL)
    if (ret != 0) {
#else
    if (ret != ASN_GETINT_E) {
#endif
        ret = -7694;
        goto done;
    }
    wc_FreeRsaKey(keyPub);
    ret = wc_InitRsaKey(keyPub, NULL);
    if (ret != 0)
        return -7695;
    ret = wc_RsaPublicKeyDecodeRaw(n, sizeof(n), e, (word32)-1, keyPub);
#if !defined(WOLFSSL_SP_MATH) & !defined(WOLFSSL_SP_MATH_ALL)
    if (ret != 0) {
#else
    if (ret != ASN_GETINT_E) {
#endif
        ret = -7696;
        goto done;
    }
    wc_FreeRsaKey(keyPub);
    ret = wc_InitRsaKey(keyPub, NULL);
    if (ret != 0)
        return -7697;

    /* Use API. */
    ret = wc_RsaPublicKeyDecodeRaw(n, sizeof(n), e, sizeof(e), keyPub);
    if (ret != 0) {
        ret = -7698;
        goto done;
    }
    wc_FreeRsaKey(keyPub);
    ret = wc_InitRsaKey(keyPub, NULL);
    if (ret != 0)
        return -7699;

    /* Parameter Validation testing. */
    inSz = sizeof(good);
    ret = wc_RsaPublicKeyDecode(NULL, &inOutIdx, keyPub, inSz);
    if (ret != BAD_FUNC_ARG) {
        ret = -7700;
        goto done;
    }
    ret = wc_RsaPublicKeyDecode(good, NULL, keyPub, inSz);
    if (ret != BAD_FUNC_ARG) {
        ret = -7701;
        goto done;
    }
    ret = wc_RsaPublicKeyDecode(good, &inOutIdx, NULL, inSz);
    if (ret != BAD_FUNC_ARG) {
        ret = -7702;
        goto done;
    }

    /* Use good data and offset to bad data. */
    inOutIdx = 2;
    inSz = sizeof(good) - inOutIdx;
    ret = wc_RsaPublicKeyDecode(good, &inOutIdx, keyPub, inSz);
    if (ret != ASN_PARSE_E) {
        ret = -7703;
        goto done;
    }
    inOutIdx = 2;
    inSz = sizeof(goodAlgId) - inOutIdx;
    ret = wc_RsaPublicKeyDecode(goodAlgId, &inOutIdx, keyPub, inSz);
    if (ret != ASN_PARSE_E) {
        ret = -7704;
        goto done;
    }
    inOutIdx = 2;
    inSz = sizeof(goodAlgId);
    ret = wc_RsaPublicKeyDecode(goodAlgId, &inOutIdx, keyPub, inSz);
#ifndef WOLFSSL_NO_DECODE_EXTRA
    if (ret != ASN_PARSE_E)
#else
    if (ret != ASN_RSA_KEY_E)
#endif
    {
        ret = -7705;
        goto done;
    }
    /* Try different bad data. */
    inSz = sizeof(badAlgIdNull);
    inOutIdx = 0;
    ret = wc_RsaPublicKeyDecode(badAlgIdNull, &inOutIdx, keyPub, inSz);
    if (ret != ASN_EXPECT_0_E) {
        ret = -7706;
        goto done;
    }
    inSz = sizeof(badNotBitString);
    inOutIdx = 0;
    ret = wc_RsaPublicKeyDecode(badNotBitString, &inOutIdx, keyPub, inSz);
    if (ret != ASN_BITSTR_E) {
        ret = -7707;
        goto done;
    }
    inSz = sizeof(badBitStringLen);
    inOutIdx = 0;
    ret = wc_RsaPublicKeyDecode(badBitStringLen, &inOutIdx, keyPub, inSz);
    if (ret != ASN_PARSE_E) {
        ret = -7708;
        goto done;
    }
    inSz = sizeof(badNoSeq);
    inOutIdx = 0;
    ret = wc_RsaPublicKeyDecode(badNoSeq, &inOutIdx, keyPub, inSz);
    if (ret != ASN_PARSE_E) {
        ret = -7709;
        goto done;
    }
    inSz = sizeof(badNoObj);
    inOutIdx = 0;
    ret = wc_RsaPublicKeyDecode(badNoObj, &inOutIdx, keyPub, inSz);
    if (ret != ASN_PARSE_E && ret != ASN_OBJECT_ID_E) {
        ret = -7710;
        goto done;
    }
    inSz = sizeof(badIntN);
    inOutIdx = 0;
    ret = wc_RsaPublicKeyDecode(badIntN, &inOutIdx, keyPub, inSz);
    if (ret != ASN_RSA_KEY_E && ret != ASN_PARSE_E) {
        ret = -7711;
        goto done;
    }
    inSz = sizeof(badNotIntE);
    inOutIdx = 0;
    ret = wc_RsaPublicKeyDecode(badNotIntE, &inOutIdx, keyPub, inSz);
    if (ret != ASN_RSA_KEY_E && ret != ASN_PARSE_E) {
        ret = -7712;
        goto done;
    }
    /* TODO: Shouldn't pass as the sequence length is too small. */
    inSz = sizeof(badLength);
    inOutIdx = 0;
    ret = wc_RsaPublicKeyDecode(badLength, &inOutIdx, keyPub, inSz);
#ifndef WOLFSSL_ASN_TEMPLATE
    if (ret != 0)
#else
    if (ret != ASN_PARSE_E)
#endif
    {
        ret = -7713;
        goto done;
    }
    /* TODO: Shouldn't ignore object id's data. */
    wc_FreeRsaKey(keyPub);
    ret = wc_InitRsaKey(keyPub, NULL);
    if (ret != 0)
        return -7714;

    inSz = sizeof(badBitStrNoZero);
    inOutIdx = 0;
    ret = wc_RsaPublicKeyDecode(badBitStrNoZero, &inOutIdx, keyPub, inSz);
    if (ret != ASN_EXPECT_0_E && ret != ASN_PARSE_E) {
        ret = -7715;
        goto done;
    }
    wc_FreeRsaKey(keyPub);
    ret = wc_InitRsaKey(keyPub, NULL);
    if (ret != 0)
        return -7716;

    /* Valid data cases. */
    inSz = sizeof(good);
    inOutIdx = 0;
    ret = wc_RsaPublicKeyDecode(good, &inOutIdx, keyPub, inSz);
    if (ret != 0) {
        ret = -7717;
        goto done;
    }
    if (inOutIdx != inSz) {
        ret = -7718;
        goto done;
    }
    wc_FreeRsaKey(keyPub);
    ret = wc_InitRsaKey(keyPub, NULL);
    if (ret != 0)
        return -7719;

    inSz = sizeof(goodAlgId);
    inOutIdx = 0;
    ret = wc_RsaPublicKeyDecode(goodAlgId, &inOutIdx, keyPub, inSz);
    if (ret != 0) {
        ret = -7720;
        goto done;
    }
    if (inOutIdx != inSz) {
        ret = -7721;
        goto done;
    }
    wc_FreeRsaKey(keyPub);
    ret = wc_InitRsaKey(keyPub, NULL);
    if (ret != 0)
        return -7722;

    inSz = sizeof(goodAlgIdNull);
    inOutIdx = 0;
    ret = wc_RsaPublicKeyDecode(goodAlgIdNull, &inOutIdx, keyPub, inSz);
    if (ret != 0) {
        ret = -7723;
        goto done;
    }
    if (inOutIdx != inSz) {
        ret = -7724;
        goto done;
    }

done:
    wc_FreeRsaKey(keyPub);
    return ret;
}
#endif

#if defined(WC_RSA_PSS) && !defined(HAVE_FIPS_VERSION) /* not supported with FIPSv1 */
/* Need to create known good signatures to test with this. */
#if !defined(WOLFSSL_RSA_VERIFY_ONLY) && !defined(WOLFSSL_RSA_PUBLIC_ONLY) && \
!defined(WOLF_CRYPTO_CB_ONLY_RSA)
static int rsa_pss_test(WC_RNG* rng, RsaKey* key)
{
    byte             digest[WC_MAX_DIGEST_SIZE];
    int              ret     = 0;
    const char       inStr[] = TEST_STRING;
    word32           inLen   = (word32)TEST_STRING_SZ;
    word32           outSz;
    word32           plainSz;
    word32           digestSz;
    int              i, j;
#ifdef RSA_PSS_TEST_WRONG_PARAMS
    int              k, l;
#endif
    int              len;
    byte*            plain;
    int              mgf[]   = {
#ifndef NO_SHA
                                 WC_MGF1SHA1,
#endif
#ifdef WOLFSSL_SHA224
                                 WC_MGF1SHA224,
#endif
                                 WC_MGF1SHA256,
#ifdef WOLFSSL_SHA384
                                 WC_MGF1SHA384,
#endif
#ifdef WOLFSSL_SHA512
                                 WC_MGF1SHA512
#endif
                               };
    enum wc_HashType hash[]  = {
#ifndef NO_SHA
                                 WC_HASH_TYPE_SHA,
#endif
#ifdef WOLFSSL_SHA224
                                 WC_HASH_TYPE_SHA224,
#endif
                                 WC_HASH_TYPE_SHA256,
#ifdef WOLFSSL_SHA384
                                 WC_HASH_TYPE_SHA384,
#endif
#ifdef WOLFSSL_SHA512
                                 WC_HASH_TYPE_SHA512,
#endif
                               };

    WC_DECLARE_VAR(in, byte, RSA_TEST_BYTES, HEAP_HINT);
    WC_DECLARE_VAR(out, byte, RSA_TEST_BYTES, HEAP_HINT);
    WC_DECLARE_VAR(sig, byte, RSA_TEST_BYTES, HEAP_HINT);

#ifdef WC_DECLARE_VAR_IS_HEAP_ALLOC
    if (in == NULL || out == NULL || sig == NULL)
        ERROR_OUT(MEMORY_E, exit_rsa_pss);
#endif
    XMEMCPY(in, inStr, inLen);

    /* Test all combinations of hash and MGF. */
    for (j = 0; j < (int)(sizeof(hash)/sizeof(*hash)); j++) {
        /* Calculate hash of message. */
        ret = wc_Hash(hash[j], in, inLen, digest, sizeof(digest));
        if (ret != 0)
            ERROR_OUT(-7730, exit_rsa_pss);
        digestSz = wc_HashGetDigestSize(hash[j]);

        for (i = 0; i < (int)(sizeof(mgf)/sizeof(*mgf)); i++) {
            outSz = RSA_TEST_BYTES;
            do {
            #if defined(WOLFSSL_ASYNC_CRYPT)
                ret = wc_AsyncWait(ret, &key->asyncDev,
                    WC_ASYNC_FLAG_CALL_AGAIN);
            #endif
                if (ret >= 0) {
                    ret = wc_RsaPSS_Sign_ex(digest, digestSz, out, outSz,
                        hash[j], mgf[i], -1, key, rng);
                }
            } while (ret == WC_PENDING_E);
            if (ret <= 0)
                ERROR_OUT(-7731, exit_rsa_pss);
            outSz = ret;

            XMEMCPY(sig, out, outSz);
            plain = NULL;
            TEST_SLEEP();

            do {
            #if defined(WOLFSSL_ASYNC_CRYPT)
                ret = wc_AsyncWait(ret, &key->asyncDev,
                    WC_ASYNC_FLAG_CALL_AGAIN);
            #endif
                if (ret >= 0) {
                    ret = wc_RsaPSS_VerifyInline_ex(sig, outSz, &plain, hash[j],
                        mgf[i], -1, key);
                }
            } while (ret == WC_PENDING_E);
            if (ret <= 0)
                ERROR_OUT(-7732, exit_rsa_pss);
            plainSz = ret;
            TEST_SLEEP();

#if defined(HAVE_SELFTEST) && \
    (!defined(HAVE_SELFTEST_VERSION) || (HAVE_SELFTEST_VERSION < 2))
            ret = wc_RsaPSS_CheckPadding_ex(digest, digestSz, plain, plainSz,
                                         hash[j], -1);
#elif defined(HAVE_SELFTEST) && (HAVE_SELFTEST_VERSION == 2)
            ret = wc_RsaPSS_CheckPadding_ex(digest, digestSz, plain, plainSz,
                                         hash[j], -1, 0);
#else
            ret = wc_RsaPSS_CheckPadding_ex2(digest, digestSz, plain, plainSz,
                              hash[j], -1, wc_RsaEncryptSize(key)*8, HEAP_HINT);
#endif
            if (ret != 0)
                ERROR_OUT(-7733, exit_rsa_pss);

#ifdef RSA_PSS_TEST_WRONG_PARAMS
            for (k = 0; k < (int)(sizeof(mgf)/sizeof(*mgf)); k++) {
                for (l = 0; l < (int)(sizeof(hash)/sizeof(*hash)); l++) {
                    if (i == k && j == l)
                        continue;

                    XMEMCPY(sig, out, outSz);

                    do {
                    #if defined(WOLFSSL_ASYNC_CRYPT)
                        ret = wc_AsyncWait(ret, &key->asyncDev,
                            WC_ASYNC_FLAG_CALL_AGAIN);
                    #endif
                        if (ret >= 0) {
                            ret = wc_RsaPSS_VerifyInline_ex(sig, outSz,
                                (byte**)&plain, hash[l], mgf[k], -1, key);
                        }
                    } while (ret == WC_PENDING_E);
                    if (ret >= 0)
                        ERROR_OUT(-7734, exit_rsa_pss);
                }
            }
#endif
        }
    }

    /* Test that a salt length of zero works. */
    digestSz = wc_HashGetDigestSize(hash[0]);
    outSz = RSA_TEST_BYTES;
    do {
    #if defined(WOLFSSL_ASYNC_CRYPT)
        ret = wc_AsyncWait(ret, &key->asyncDev,
            WC_ASYNC_FLAG_CALL_AGAIN);
    #endif
        if (ret >= 0) {
            ret = wc_RsaPSS_Sign_ex(digest, digestSz, out, outSz, hash[0],
                mgf[0], 0, key, rng);
        }
    } while (ret == WC_PENDING_E);
    if (ret <= 0)
        ERROR_OUT(-7735, exit_rsa_pss);
    outSz = ret;
    TEST_SLEEP();

    do {
    #if defined(WOLFSSL_ASYNC_CRYPT)
        ret = wc_AsyncWait(ret, &key->asyncDev,
            WC_ASYNC_FLAG_CALL_AGAIN);
    #endif
        if (ret >= 0) {
            ret = wc_RsaPSS_Verify_ex(out, outSz, sig, outSz, hash[0], mgf[0],
                0, key);
        }
    } while (ret == WC_PENDING_E);
    if (ret <= 0)
        ERROR_OUT(-7736, exit_rsa_pss);
    plainSz = ret;
    TEST_SLEEP();

    do {
    #if defined(WOLFSSL_ASYNC_CRYPT)
        ret = wc_AsyncWait(ret, &key->asyncDev,
            WC_ASYNC_FLAG_CALL_AGAIN);
    #endif
        if (ret >= 0) {
#if defined(HAVE_SELFTEST) && \
    (!defined(HAVE_SELFTEST_VERSION) || (HAVE_SELFTEST_VERSION < 2))
            ret = wc_RsaPSS_CheckPadding_ex(digest, digestSz, sig, plainSz,
                                         hash[0], 0);
#elif defined(HAVE_SELFTEST) && (HAVE_SELFTEST_VERSION == 2)
            ret = wc_RsaPSS_CheckPadding_ex(digest, digestSz, sig, plainSz,
                                         hash[0], 0, 0);
#else
            ret = wc_RsaPSS_CheckPadding_ex2(digest, digestSz, sig, plainSz,
                hash[0], 0, 0, HEAP_HINT);
#endif
        }
    } while (ret == WC_PENDING_E);
    if (ret != 0)
        ERROR_OUT(-7737, exit_rsa_pss);

    XMEMCPY(sig, out, outSz);
    plain = NULL;
    do {
    #if defined(WOLFSSL_ASYNC_CRYPT)
        ret = wc_AsyncWait(ret, &key->asyncDev,
            WC_ASYNC_FLAG_CALL_AGAIN);
    #endif
        if (ret >= 0) {
            ret = wc_RsaPSS_VerifyInline_ex(sig, outSz, &plain, hash[0], mgf[0],
                0, key);
        }
    } while (ret == WC_PENDING_E);
    if (ret <= 0)
        ERROR_OUT(-7738, exit_rsa_pss);
    plainSz = ret;
    TEST_SLEEP();

#if defined(HAVE_SELFTEST) && \
    (!defined(HAVE_SELFTEST_VERSION) || (HAVE_SELFTEST_VERSION < 2))
            ret = wc_RsaPSS_CheckPadding_ex(digest, digestSz, plain, plainSz,
                                         hash[0], 0);
#elif defined(HAVE_SELFTEST) && (HAVE_SELFTEST_VERSION == 2)
            ret = wc_RsaPSS_CheckPadding_ex(digest, digestSz, plain, plainSz,
                                         hash[0], 0, 0);
#else
    ret = wc_RsaPSS_CheckPadding_ex2(digest, digestSz, plain, plainSz, hash[0],
                                    0, 0, HEAP_HINT);
#endif
    if (ret != 0)
        ERROR_OUT(-7739, exit_rsa_pss);

    /* Test bad salt lengths in various APIs. */
    digestSz = wc_HashGetDigestSize(hash[0]);
    outSz = RSA_TEST_BYTES;
#ifndef WOLFSSL_PSS_SALT_LEN_DISCOVER
    len = -2;
#else
    len = -3;
#endif
    do {
    #if defined(WOLFSSL_ASYNC_CRYPT)
        ret = wc_AsyncWait(ret, &key->asyncDev,
            WC_ASYNC_FLAG_CALL_AGAIN);
    #endif
        if (ret >= 0) {
            ret = wc_RsaPSS_Sign_ex(digest, digestSz, out, outSz, hash[0],
                mgf[0], len, key, rng);
        }
    } while (ret == WC_PENDING_E);
    if (ret != PSS_SALTLEN_E)
        ERROR_OUT(-7740, exit_rsa_pss);

    do {
    #if defined(WOLFSSL_ASYNC_CRYPT)
        ret = wc_AsyncWait(ret, &key->asyncDev,
            WC_ASYNC_FLAG_CALL_AGAIN);
    #endif
        if (ret >= 0) {
            ret = wc_RsaPSS_Sign_ex(digest, digestSz, out, outSz, hash[0],
                mgf[0], digestSz + 1, key, rng);
        }
    } while (ret == WC_PENDING_E);
    if (ret != PSS_SALTLEN_E)
        ERROR_OUT(-7741, exit_rsa_pss);
    TEST_SLEEP();

    do {
    #if defined(WOLFSSL_ASYNC_CRYPT)
        ret = wc_AsyncWait(ret, &key->asyncDev,
            WC_ASYNC_FLAG_CALL_AGAIN);
    #endif
        if (ret >= 0) {
            ret = wc_RsaPSS_VerifyInline_ex(sig, outSz, &plain, hash[0],
                mgf[0], -2, key);
        }
    } while (ret == WC_PENDING_E);
    if (ret != PSS_SALTLEN_E)
        ERROR_OUT(-7742, exit_rsa_pss);
    TEST_SLEEP();

    do {
    #if defined(WOLFSSL_ASYNC_CRYPT)
        ret = wc_AsyncWait(ret, &key->asyncDev,
            WC_ASYNC_FLAG_CALL_AGAIN);
    #endif
        if (ret >= 0) {
            ret = wc_RsaPSS_VerifyInline_ex(sig, outSz, &plain, hash[0], mgf[0],
                digestSz + 1, key);
        }
    } while (ret == WC_PENDING_E);
    if (ret != PSS_SALTLEN_E)
        ERROR_OUT(-7743, exit_rsa_pss);
    TEST_SLEEP();

#ifndef WOLFSSL_PSS_SALT_LEN_DISCOVER
    len = -2;
#else
    len = -3;
#endif
#if defined(HAVE_SELFTEST) && \
    (!defined(HAVE_SELFTEST_VERSION) || (HAVE_SELFTEST_VERSION < 2))
            ret = wc_RsaPSS_CheckPadding_ex(digest, digestSz, plain, plainSz,
                                         hash[0], len);
#elif defined(HAVE_SELFTEST) && (HAVE_SELFTEST_VERSION == 2)
            ret = wc_RsaPSS_CheckPadding_ex(digest, digestSz, plain, plainSz,
                                         hash[0], len, 0);
#else
    ret = wc_RsaPSS_CheckPadding_ex2(digest, digestSz, plain, plainSz, hash[0],
                                    len, 0, HEAP_HINT);
#endif
    if (ret != PSS_SALTLEN_E)
        ERROR_OUT(-7744, exit_rsa_pss);
#ifndef WOLFSSL_PSS_LONG_SALT
    len = digestSz + 1;
#else
    len = plainSz - digestSz - 1;
#endif
#if defined(HAVE_SELFTEST) && \
    (!defined(HAVE_SELFTEST_VERSION) || (HAVE_SELFTEST_VERSION < 2))
            ret = wc_RsaPSS_CheckPadding_ex(digest, digestSz, plain, plainSz,
                                         hash[0], len);
#elif defined(HAVE_SELFTEST) && (HAVE_SELFTEST_VERSION == 2)
            ret = wc_RsaPSS_CheckPadding_ex(digest, digestSz, plain, plainSz,
                                         hash[0], len, 0);
#else
    ret = wc_RsaPSS_CheckPadding_ex2(digest, digestSz, plain, plainSz, hash[0],
                                    len, 0, HEAP_HINT);
#endif
    if (ret != PSS_SALTLEN_E)
        ERROR_OUT(-7745, exit_rsa_pss);

    ret = 0;
exit_rsa_pss:
    WC_FREE_VAR(sig, HEAP_HINT);
    WC_FREE_VAR(in, HEAP_HINT);
    WC_FREE_VAR(out, HEAP_HINT);

    return ret;
}
#endif /* !WOLFSSL_RSA_VERIFY_ONLY && !WOLFSSL_RSA_PUBLIC_ONLY */
#endif

#ifdef WC_RSA_NO_PADDING
WOLFSSL_TEST_SUBROUTINE int rsa_no_pad_test(void)
{
    WC_RNG rng;
    byte*  tmp = NULL;
    size_t bytes;
    int    ret;
    word32 inLen   = 0;
    word32 idx     = 0;
    word32 outSz   = RSA_TEST_BYTES;
    word32 plainSz = RSA_TEST_BYTES;
#if !defined(USE_CERT_BUFFERS_1024) && !defined(USE_CERT_BUFFERS_2048) && \
    !defined(USE_CERT_BUFFERS_3072) && !defined(USE_CERT_BUFFERS_4096) && \
    !defined(NO_FILESYSTEM)
    XFILE  file;
#endif
    WC_DECLARE_VAR(key, RsaKey, 1, HEAP_HINT);
    WC_DECLARE_VAR(out, byte, RSA_TEST_BYTES, HEAP_HINT);
    WC_DECLARE_VAR(plain, byte, RSA_TEST_BYTES, HEAP_HINT);

#ifdef WC_DECLARE_VAR_IS_HEAP_ALLOC
    if (key == NULL || out == NULL || plain == NULL)
        ERROR_OUT(MEMORY_E, exit_rsa_nopadding);
#endif

    /* initialize stack structures */
    XMEMSET(&rng, 0, sizeof(rng));
    XMEMSET(key, 0, sizeof(RsaKey));
#ifdef USE_CERT_BUFFERS_1024
    bytes = (size_t)sizeof_client_key_der_1024;
    if (bytes < (size_t)sizeof_client_cert_der_1024)
        bytes = (size_t)sizeof_client_cert_der_1024;
#elif defined(USE_CERT_BUFFERS_2048)
    bytes = (size_t)sizeof_client_key_der_2048;
    if (bytes < (size_t)sizeof_client_cert_der_2048)
        bytes = (size_t)sizeof_client_cert_der_2048;
#else
    bytes = FOURK_BUF;
#endif

    tmp = (byte*)XMALLOC(bytes, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    if (tmp == NULL
    #ifdef WOLFSSL_ASYNC_CRYPT
        || out == NULL || plain == NULL
    #endif
    ) {
        ERROR_OUT(-7800, exit_rsa_nopadding);
    }

#ifdef USE_CERT_BUFFERS_1024
    XMEMCPY(tmp, client_key_der_1024, (size_t)sizeof_client_key_der_1024);
#elif defined(USE_CERT_BUFFERS_2048)
    XMEMCPY(tmp, client_key_der_2048, (size_t)sizeof_client_key_der_2048);
#elif defined(USE_CERT_BUFFERS_3072)
    XMEMCPY(tmp, client_key_der_3072, (size_t)sizeof_client_key_der_3072);
#elif defined(USE_CERT_BUFFERS_4096)
    XMEMCPY(tmp, client_key_der_4096, (size_t)sizeof_client_key_der_4096);
#elif !defined(NO_FILESYSTEM)
    file = XFOPEN(clientKey, "rb");
    if (!file) {
        err_sys("can't open clientKey, Please run from wolfSSL home dir", -40);
        ERROR_OUT(-7801, exit_rsa_nopadding);
    }

    bytes = XFREAD(tmp, 1, FOURK_BUF, file);
    XFCLOSE(file);
#else
    /* No key to use. */
    ERROR_OUT(-7802, exit_rsa_nopadding);
#endif /* USE_CERT_BUFFERS */

    ret = wc_InitRsaKey_ex(key, HEAP_HINT, devId);
    if (ret != 0) {
        ERROR_OUT(-7803, exit_rsa_nopadding);
    }
    ret = wc_RsaPrivateKeyDecode(tmp, &idx, key, (word32)bytes);
    if (ret != 0) {
        ERROR_OUT(-7804, exit_rsa_nopadding);
    }

    /* after loading in key use tmp as the test buffer */

#ifndef HAVE_FIPS
    ret = wc_InitRng_ex(&rng, HEAP_HINT, devId);
#else
    ret = wc_InitRng(&rng);
#endif
    if (ret != 0) {
        ERROR_OUT(-7805, exit_rsa_nopadding);
    }

#ifndef WOLFSSL_RSA_VERIFY_ONLY
    inLen = wc_RsaEncryptSize(key);
    outSz   = inLen;
    plainSz = inLen;
    XMEMSET(tmp, 7, inLen);
    do {
    #if defined(WOLFSSL_ASYNC_CRYPT)
        ret = wc_AsyncWait(ret, &key->asyncDev, WC_ASYNC_FLAG_CALL_AGAIN);
    #endif
        if (ret >= 0) {
            ret = wc_RsaDirect(tmp, inLen, out, &outSz, key,
                    RSA_PRIVATE_ENCRYPT, &rng);
        }
    } while (ret == WC_PENDING_E);
    if (ret <= 0) {
        ERROR_OUT(-7806, exit_rsa_nopadding);
    }

    /* encrypted result should not be the same as input */
    if (XMEMCMP(out, tmp, inLen) == 0) {
        ERROR_OUT(-7807, exit_rsa_nopadding);
    }
    TEST_SLEEP();

    /* decrypt with public key and compare result */
    do {
    #if defined(WOLFSSL_ASYNC_CRYPT)
        ret = wc_AsyncWait(ret, &key->asyncDev, WC_ASYNC_FLAG_CALL_AGAIN);
    #endif
        if (ret >= 0) {
            ret = wc_RsaDirect(out, outSz, plain, &plainSz, key,
                    RSA_PUBLIC_DECRYPT, &rng);
        }
    } while (ret == WC_PENDING_E);
    if (ret <= 0) {
        ERROR_OUT(-7808, exit_rsa_nopadding);
    }

    if (XMEMCMP(plain, tmp, inLen) != 0) {
        ERROR_OUT(-7809, exit_rsa_nopadding);
    }
    TEST_SLEEP();
#endif

#ifdef WC_RSA_BLINDING
    ret = wc_RsaSetRNG(NULL, &rng);
    if (ret != BAD_FUNC_ARG) {
        ERROR_OUT(-7810, exit_rsa_nopadding);
    }

    ret = wc_RsaSetRNG(key, &rng);
    if (ret < 0) {
        ERROR_OUT(-7811, exit_rsa_nopadding);
    }
#endif

    /* test encrypt and decrypt using WC_RSA_NO_PAD */
#ifndef WOLFSSL_RSA_VERIFY_ONLY
    do {
    #if defined(WOLFSSL_ASYNC_CRYPT)
        ret = wc_AsyncWait(ret, &key->asyncDev, WC_ASYNC_FLAG_CALL_AGAIN);
    #endif
        if (ret >= 0) {
            ret = wc_RsaPublicEncrypt_ex(tmp, inLen, out, (int)outSz, key, &rng,
                WC_RSA_NO_PAD, WC_HASH_TYPE_NONE, WC_MGF1NONE, NULL, 0);
        }
    } while (ret == WC_PENDING_E);
    if (ret < 0) {
        ERROR_OUT(-7812, exit_rsa_nopadding);
    }
    TEST_SLEEP();
#endif /* WOLFSSL_RSA_VERIFY_ONLY */

#ifndef WOLFSSL_RSA_PUBLIC_ONLY
    do {
    #if defined(WOLFSSL_ASYNC_CRYPT)
        ret = wc_AsyncWait(ret, &key->asyncDev, WC_ASYNC_FLAG_CALL_AGAIN);
    #endif
        if (ret >= 0) {
            ret = wc_RsaPrivateDecrypt_ex(out, outSz, plain, (int)plainSz, key,
                WC_RSA_NO_PAD, WC_HASH_TYPE_NONE, WC_MGF1NONE, NULL, 0);
        }
    } while (ret == WC_PENDING_E);
    if (ret < 0) {
        ERROR_OUT(-7813, exit_rsa_nopadding);
    }

    if (XMEMCMP(plain, tmp, inLen) != 0) {
        ERROR_OUT(-7814, exit_rsa_nopadding);
    }
    TEST_SLEEP();
#endif /* WOLFSSL_RSA_PUBLIC_ONLY */

    /* test some bad arguments */
    ret = wc_RsaDirect(out, outSz, plain, &plainSz, key, -1,
            &rng);
    if (ret != BAD_FUNC_ARG) {
        ERROR_OUT(-7815, exit_rsa_nopadding);
    }

    ret = wc_RsaDirect(out, outSz, plain, &plainSz, NULL, RSA_PUBLIC_DECRYPT,
            &rng);
    if (ret != BAD_FUNC_ARG) {
        ERROR_OUT(-7816, exit_rsa_nopadding);
    }

    ret = wc_RsaDirect(out, outSz, NULL, &plainSz, key, RSA_PUBLIC_DECRYPT,
            &rng);
    if (ret != LENGTH_ONLY_E || plainSz != inLen) {
        ERROR_OUT(-7817, exit_rsa_nopadding);
    }

    ret = wc_RsaDirect(out, outSz - 10, plain, &plainSz, key,
            RSA_PUBLIC_DECRYPT, &rng);
    if (ret != BAD_FUNC_ARG) {
        ERROR_OUT(-7818, exit_rsa_nopadding);
    }

    /* if making it to this point of code without hitting an ERROR_OUT then
     * all tests have passed */
    ret = 0;

exit_rsa_nopadding:
    wc_FreeRsaKey(key);
    XFREE(tmp, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    WC_FREE_VAR(key, HEAP_HINT);
    WC_FREE_VAR(out, HEAP_HINT);
    WC_FREE_VAR(plain, HEAP_HINT);
    wc_FreeRng(&rng);

    return ret;
}
#endif /* WC_RSA_NO_PADDING */

#if defined(WOLFSSL_HAVE_SP_RSA) && defined(USE_FAST_MATH)
static int rsa_even_mod_test(WC_RNG* rng, RsaKey* key)
{
    byte*  tmp = NULL;
    size_t bytes;
    int    ret;
    word32 inLen   = 0;
#ifndef NO_ASN
    word32 idx     = 0;
#endif
    word32 outSz   = RSA_TEST_BYTES;
#ifndef WOLFSSL_RSA_PUBLIC_ONLY
    word32 plainSz = RSA_TEST_BYTES;
#endif
#if !defined(USE_CERT_BUFFERS_2048) && !defined(USE_CERT_BUFFERS_3072) && \
    !defined(USE_CERT_BUFFERS_4096) && !defined(NO_FILESYSTEM)
    XFILE  file;
#endif
    WC_DECLARE_VAR(out, byte, RSA_TEST_BYTES, HEAP_HINT);
#ifndef WOLFSSL_RSA_PUBLIC_ONLY
    WC_DECLARE_VAR(plain, byte, RSA_TEST_BYTES, HEAP_HINT);
#endif
#ifdef WC_DECLARE_VAR_IS_HEAP_ALLOC
    if (out == NULL
    #ifndef WOLFSSL_RSA_PUBLIC_ONLY
        || plain == NULL
    #endif
    ) {
        ERROR_OUT(MEMORY_E, exit_rsa_even_mod);
    }
#endif

#if defined(USE_CERT_BUFFERS_2048)
    bytes = (size_t)sizeof_client_key_der_2048;
    if (bytes < (size_t)sizeof_client_cert_der_2048)
        bytes = (size_t)sizeof_client_cert_der_2048;
#else
    bytes = FOURK_BUF;
#endif

    tmp = (byte*)XMALLOC(bytes, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    if (tmp == NULL
    #ifdef WOLFSSL_ASYNC_CRYPT
        || out == NULL || plain == NULL
    #endif
    ) {
        ERROR_OUT(-7800, exit_rsa_even_mod);
    }

#if defined(USE_CERT_BUFFERS_2048)
    XMEMCPY(tmp, client_key_der_2048, (size_t)sizeof_client_key_der_2048);
#elif defined(USE_CERT_BUFFERS_3072)
    XMEMCPY(tmp, client_key_der_3072, (size_t)sizeof_client_key_der_3072);
#elif defined(USE_CERT_BUFFERS_4096)
    XMEMCPY(tmp, client_key_der_4096, (size_t)sizeof_client_key_der_4096);
#elif !defined(NO_FILESYSTEM)
    file = XFOPEN(clientKey, "rb");
    if (!file) {
        err_sys("can't open ./certs/client-key.der, "
                "Please run from wolfSSL home dir", -40);
        ERROR_OUT(-7801, exit_rsa_even_mod);
    }

    bytes = XFREAD(tmp, 1, FOURK_BUF, file);
    XFCLOSE(file);
#else
    /* No key to use. */
    ERROR_OUT(-7802, exit_rsa_even_mod);
#endif /* USE_CERT_BUFFERS */

#ifndef NO_ASN
    ret = wc_RsaPrivateKeyDecode(tmp, &idx, key, (word32)bytes);
    if (ret != 0) {
        ERROR_OUT(-7804, exit_rsa_even_mod);
    }
#else
    #ifdef USE_CERT_BUFFERS_2048
        ret = mp_read_unsigned_bin(&key->n, &tmp[12], 256);
        if (ret != 0) {
            ERROR_OUT(-7804, exit_rsa_even_mod);
        }
        ret = mp_set_int(&key->e, WC_RSA_EXPONENT);
        if (ret != 0) {
            ERROR_OUT(-7804, exit_rsa_even_mod);
        }
#ifndef NO_SIG_WRAPPER
        modLen = 2048;
#endif
    #else
        #error Not supported yet!
    #endif
#endif

    key->n.dp[0] &= (mp_digit)-2;
    if (ret != 0) {
        ERROR_OUT(-7804, exit_rsa_even_mod);
    }

    /* after loading in key use tmp as the test buffer */
#if !(defined(HAVE_FIPS_VERSION) && (HAVE_FIPS_VERSION == 2) && \
    (defined(WOLFSSL_SP_ARM64_ASM) || defined(WOLFSSL_SP_ARM32_ASM)))
    /* The ARM64_ASM code that was FIPS validated did not return these expected
     * failure codes. These tests cases were added after the assembly was
     * in-lined in the module and validated, these tests will be available in
     * the 140-3 module */
#if !defined(WOLFSSL_RSA_VERIFY_ONLY) && !defined(WOLFSSL_RSA_PUBLIC_ONLY)
    inLen = 32;
    outSz   = wc_RsaEncryptSize(key);
    XMEMSET(tmp, 7, plainSz);
    ret = wc_RsaSSL_Sign(tmp, inLen, out, outSz, key, rng);
    if (ret != MP_VAL && ret != MP_EXPTMOD_E && ret != MP_INVMOD_E) {
        ERROR_OUT(-7806, exit_rsa_even_mod);
    }

    ret = wc_RsaSSL_Verify(out, outSz, tmp, inLen, key);
    if (ret != MP_VAL && ret != MP_EXPTMOD_E) {
        ERROR_OUT(-7808, exit_rsa_even_mod);
    }
#endif

#ifdef WC_RSA_BLINDING
    ret = wc_RsaSetRNG(key, rng);
    if (ret < 0) {
        ERROR_OUT(-7811, exit_rsa_even_mod);
    }
#endif

    /* test encrypt and decrypt using WC_RSA_NO_PAD */
#if !defined(WOLFSSL_RSA_VERIFY_ONLY) && !defined(WOLFSSL_RSA_PUBLIC_ONLY)
    ret = wc_RsaPublicEncrypt(tmp, inLen, out, (int)outSz, key, rng);
    if (ret != MP_VAL && ret != MP_EXPTMOD_E) {
        ERROR_OUT(-7812, exit_rsa_even_mod);
    }
#endif /* WOLFSSL_RSA_VERIFY_ONLY */

#ifndef WOLFSSL_RSA_PUBLIC_ONLY
    ret = wc_RsaPrivateDecrypt(out, outSz, plain, (int)plainSz, key);
    if (ret != MP_VAL && ret != MP_EXPTMOD_E && ret != MP_INVMOD_E) {
        ERROR_OUT(-7813, exit_rsa_even_mod);
    }
#endif /* WOLFSSL_RSA_PUBLIC_ONLY */
#endif /* !(HAVE_FIPS_VERSION == 2 && WOLFSSL_SP_ARMxx_ASM) */
    /* if making it to this point of code without hitting an ERROR_OUT then
     * all tests have passed */
    ret = 0;

exit_rsa_even_mod:
    XFREE(tmp, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    WC_FREE_VAR(out, HEAP_HINT);
#ifndef WOLFSSL_RSA_PUBLIC_ONLY
    WC_FREE_VAR(plain, HEAP_HINT);
#endif

    (void)out;
    (void)outSz;
#ifndef WOLFSSL_RSA_PUBLIC_ONLY
    (void)plain;
    (void)plainSz;
#endif
    (void)inLen;
    (void)rng;

    return ret;
}
#endif /* WOLFSSL_HAVE_SP_RSA */

#ifdef WOLFSSL_CERT_GEN
static int rsa_certgen_test(RsaKey* key, RsaKey* keypub, WC_RNG* rng, byte* tmp)
{
#ifdef WOLFSSL_SMALL_STACK
    RsaKey      *caKey = (RsaKey *)XMALLOC(sizeof *caKey, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
#ifdef WOLFSSL_TEST_CERT
    DecodedCert *decode = (DecodedCert *)XMALLOC(sizeof *decode, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
#endif
#else
    RsaKey      caKey[1];
#ifdef WOLFSSL_TEST_CERT
    DecodedCert decode[1];
#endif
#endif
    byte*       der = NULL;
    int         ret;
    Cert*       myCert = NULL;
    int         certSz;
    size_t      bytes3;
    word32      idx3 = 0;
#if !defined(USE_CERT_BUFFERS_1024) && !defined(USE_CERT_BUFFERS_2048)
    XFILE       file3;
#endif
#if defined(WOLFSSL_ALT_NAMES) && !defined(NO_ASN_TIME)
    struct tm beforeTime;
    struct tm afterTime;
#endif
    const byte  mySerial[8] = {1,2,3,4,5,6,7,8};

    (void)keypub;

#ifdef WOLFSSL_SMALL_STACK
    if (caKey == NULL)
        ERROR_OUT(MEMORY_E, exit_rsa);
#ifdef WOLFSSL_TEST_CERT
    if (decode == NULL)
        ERROR_OUT(MEMORY_E, exit_rsa);
#endif

#endif

    XMEMSET(caKey, 0, sizeof *caKey);

    der = (byte*)XMALLOC(FOURK_BUF, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    if (der == NULL) {
        ERROR_OUT(-7820, exit_rsa);
    }
    myCert = (Cert*)XMALLOC(sizeof(Cert), HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    if (myCert == NULL) {
        ERROR_OUT(-7821, exit_rsa);
    }

    /* self signed */
    if (wc_InitCert_ex(myCert, HEAP_HINT, devId)) {
        ERROR_OUT(-7822, exit_rsa);
    }

    XMEMCPY(&myCert->subject, &certDefaultName, sizeof(CertName));
    XMEMCPY(myCert->serial, mySerial, sizeof(mySerial));
    myCert->serialSz = (int)sizeof(mySerial);
    myCert->isCA    = 1;
#ifndef NO_SHA256
    myCert->sigType = CTC_SHA256wRSA;
#else
    myCert->sigType = CTC_SHAwRSA;
#endif

#ifdef WOLFSSL_CERT_EXT
    /* add Policies */
    XSTRNCPY(myCert->certPolicies[0], "2.16.840.1.101.3.4.1.42",
            CTC_MAX_CERTPOL_SZ);
    XSTRNCPY(myCert->certPolicies[1], "1.2.840.113549.1.9.16.6.5",
            CTC_MAX_CERTPOL_SZ);
    myCert->certPoliciesNb = 2;

    /* add SKID from the Public Key */
    if (wc_SetSubjectKeyIdFromPublicKey(myCert, keypub, NULL) != 0) {
        ERROR_OUT(-7823, exit_rsa);
    }

     /* add AKID from the Public Key */
     if (wc_SetAuthKeyIdFromPublicKey(myCert, keypub, NULL) != 0) {
        ERROR_OUT(-7824, exit_rsa);
    }

    /* add Key Usage */
    if (wc_SetKeyUsage(myCert,"cRLSign,keyCertSign") != 0) {
        ERROR_OUT(-7825, exit_rsa);
    }
#ifdef WOLFSSL_EKU_OID
    {
        const char unique[] = "2.16.840.1.111111.100.1.10.1";
        if (wc_SetExtKeyUsageOID(myCert, unique, sizeof(unique), 0,
                    HEAP_HINT) != 0) {
            ERROR_OUT(-7826, exit_rsa);
        }
    }
#endif /* WOLFSSL_EKU_OID */
#endif /* WOLFSSL_CERT_EXT */

    ret = 0;
    do {
#if defined(WOLFSSL_ASYNC_CRYPT)
        ret = wc_AsyncWait(ret, &key->asyncDev, WC_ASYNC_FLAG_CALL_AGAIN);
#endif
        if (ret >= 0) {
            ret = wc_MakeSelfCert(myCert, der, FOURK_BUF, key, rng);
        }
    } while (ret == WC_PENDING_E);
    if (ret < 0) {
        ERROR_OUT(-7827, exit_rsa);
    }
    certSz = ret;

#ifdef WOLFSSL_TEST_CERT
    InitDecodedCert(decode, der, certSz, HEAP_HINT);
    ret = ParseCert(decode, CERT_TYPE, NO_VERIFY, 0);
    if (ret != 0) {
        FreeDecodedCert(decode);
        ERROR_OUT(-7828, exit_rsa);
    }
    FreeDecodedCert(decode);
#endif

    ret = SaveDerAndPem(der, certSz, certDerFile, certPemFile,
        CERT_TYPE, -5578);
    if (ret != 0) {
        goto exit_rsa;
    }

    /* Setup Certificate */
    if (wc_InitCert_ex(myCert, HEAP_HINT, devId)) {
        ERROR_OUT(-7829, exit_rsa);
    }

#ifdef WOLFSSL_ALT_NAMES
        /* Get CA Cert for testing */
    #ifdef USE_CERT_BUFFERS_1024
        XMEMCPY(tmp, ca_cert_der_1024, sizeof_ca_cert_der_1024);
        bytes3 = sizeof_ca_cert_der_1024;
    #elif defined(USE_CERT_BUFFERS_2048)
        XMEMCPY(tmp, ca_cert_der_2048, sizeof_ca_cert_der_2048);
        bytes3 = sizeof_ca_cert_der_2048;
    #else
        file3 = XFOPEN(rsaCaCertDerFile, "rb");
        if (!file3) {
            ERROR_OUT(-7830, exit_rsa);
        }
        bytes3 = XFREAD(tmp, 1, FOURK_BUF, file3);
        XFCLOSE(file3);
    #endif /* USE_CERT_BUFFERS */

    #if !defined(NO_FILESYSTEM) && !defined(USE_CERT_BUFFERS_1024) && \
        !defined(USE_CERT_BUFFERS_2048) && !defined(NO_ASN)
        ret = wc_SetAltNames(myCert, rsaCaCertFile);
        if (ret != 0) {
            ERROR_OUT(-7831, exit_rsa);
        }
    #endif
        /* get alt names from der */
        ret = wc_SetAltNamesBuffer(myCert, tmp, (int)bytes3);
        if (ret != 0) {
            ERROR_OUT(-7832, exit_rsa);
        }

        /* get dates from der */
        ret = wc_SetDatesBuffer(myCert, tmp, (int)bytes3);
        if (ret != 0) {
            ERROR_OUT(-7833, exit_rsa);
        }

    #ifndef NO_ASN_TIME
        ret = wc_GetCertDates(myCert, &beforeTime, &afterTime);
        if (ret < 0) {
            ERROR_OUT(-7834, exit_rsa);
        }
    #endif
#endif /* WOLFSSL_ALT_NAMES */

    /* Get CA Key */
#ifdef USE_CERT_BUFFERS_1024
    XMEMCPY(tmp, ca_key_der_1024, sizeof_ca_key_der_1024);
    bytes3 = sizeof_ca_key_der_1024;
#elif defined(USE_CERT_BUFFERS_2048)
    XMEMCPY(tmp, ca_key_der_2048, sizeof_ca_key_der_2048);
    bytes3 = sizeof_ca_key_der_2048;
#else
    file3 = XFOPEN(rsaCaKeyFile, "rb");
    if (!file3) {
        ERROR_OUT(-7835, exit_rsa);
    }

    bytes3 = XFREAD(tmp, 1, FOURK_BUF, file3);
    XFCLOSE(file3);
#endif /* USE_CERT_BUFFERS */

    ret = wc_InitRsaKey(caKey, HEAP_HINT);
    if (ret != 0) {
        ERROR_OUT(-7836, exit_rsa);
    }
    ret = wc_RsaPrivateKeyDecode(tmp, &idx3, caKey, (word32)bytes3);
    if (ret != 0) {
        ERROR_OUT(-7837, exit_rsa);
    }

#ifndef NO_SHA256
    myCert->sigType = CTC_SHA256wRSA;
#else
    myCert->sigType = CTC_SHAwRSA;
#endif

    XMEMCPY(&myCert->subject, &certDefaultName, sizeof(CertName));

#ifdef WOLFSSL_CERT_EXT
    /* add Policies */
    XSTRNCPY(myCert->certPolicies[0], "2.16.840.1.101.3.4.1.42",
            CTC_MAX_CERTPOL_SZ);
    myCert->certPoliciesNb =1;

    /* add SKID from the Public Key */
    if (wc_SetSubjectKeyIdFromPublicKey(myCert, key, NULL) != 0) {
        ERROR_OUT(-7838, exit_rsa);
    }

    /* add AKID from the CA certificate */
#if defined(USE_CERT_BUFFERS_2048)
    ret = wc_SetAuthKeyIdFromCert(myCert, ca_cert_der_2048,
                                        sizeof_ca_cert_der_2048);
#elif defined(USE_CERT_BUFFERS_1024)
    ret = wc_SetAuthKeyIdFromCert(myCert, ca_cert_der_1024,
                                        sizeof_ca_cert_der_1024);
#else
    ret = wc_SetAuthKeyId(myCert, rsaCaCertFile);
#endif
    if (ret != 0) {
        ERROR_OUT(-7839, exit_rsa);
    }

    /* add Key Usage */
    if (wc_SetKeyUsage(myCert,"keyEncipherment,keyAgreement") != 0) {
        ERROR_OUT(-7840, exit_rsa);
    }
#endif /* WOLFSSL_CERT_EXT */

#if defined(USE_CERT_BUFFERS_2048)
    ret = wc_SetIssuerBuffer(myCert, ca_cert_der_2048,
                                      sizeof_ca_cert_der_2048);
#elif defined(USE_CERT_BUFFERS_1024)
    ret = wc_SetIssuerBuffer(myCert, ca_cert_der_1024,
                                      sizeof_ca_cert_der_1024);
#else
    ret = wc_SetIssuer(myCert, rsaCaCertFile);
#endif
    if (ret < 0) {
        ERROR_OUT(-7841, exit_rsa);
    }

    certSz = wc_MakeCert(myCert, der, FOURK_BUF, key, NULL, rng);
    if (certSz < 0) {
        ERROR_OUT(-7842, exit_rsa);
    }

    ret = 0;
    do {
    #if defined(WOLFSSL_ASYNC_CRYPT)
        ret = wc_AsyncWait(ret, &caKey->asyncDev, WC_ASYNC_FLAG_CALL_AGAIN);
    #endif
        if (ret >= 0) {
            ret = wc_SignCert(myCert->bodySz, myCert->sigType, der, FOURK_BUF,
                      caKey, NULL, rng);
        }
    } while (ret == WC_PENDING_E);
    if (ret < 0) {
        ERROR_OUT(-7843, exit_rsa);
    }
    certSz = ret;

#ifdef WOLFSSL_TEST_CERT
    InitDecodedCert(decode, der, certSz, HEAP_HINT);
    ret = ParseCert(decode, CERT_TYPE, NO_VERIFY, 0);
    if (ret != 0) {
        FreeDecodedCert(decode);
        ERROR_OUT(-7844, exit_rsa);
    }
    FreeDecodedCert(decode);
#endif

    ret = SaveDerAndPem(der, certSz, otherCertDerFile, otherCertPemFile,
        CERT_TYPE, -5598);
    if (ret != 0) {
        goto exit_rsa;
    }

exit_rsa:

#ifdef WOLFSSL_SMALL_STACK
    if (caKey != NULL) {
        wc_FreeRsaKey(caKey);
        XFREE(caKey, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    }
#ifdef WOLFSSL_TEST_CERT
    if (decode != NULL)
        XFREE(decode, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
#endif
#else
    wc_FreeRsaKey(caKey);
#endif

    XFREE(myCert, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    XFREE(der, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);

    return ret;
}
#endif

#if !defined(NO_RSA) && defined(HAVE_ECC) && defined(WOLFSSL_CERT_GEN)
/* Make Cert / Sign example for ECC cert and RSA CA */
static int rsa_ecc_certgen_test(WC_RNG* rng, byte* tmp)
{
#ifdef WOLFSSL_SMALL_STACK
    RsaKey      *caKey = (RsaKey *)XMALLOC(sizeof *caKey, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    ecc_key     *caEccKey = (ecc_key *)XMALLOC(sizeof *caEccKey, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    ecc_key     *caEccKeyPub = (ecc_key *)XMALLOC(sizeof *caEccKeyPub, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
#ifdef WOLFSSL_TEST_CERT
    DecodedCert *decode = (DecodedCert *)XMALLOC(sizeof *decode, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
#endif
#else
    RsaKey      caKey[1];
    ecc_key     caEccKey[1];
    ecc_key     caEccKeyPub[1];
#ifdef WOLFSSL_TEST_CERT
    DecodedCert decode[1];
#endif
#endif
    byte*       der = NULL;
    Cert*       myCert = NULL;
    int         certSz;
    size_t      bytes3;
    word32      idx3 = 0;
#if (!defined(USE_CERT_BUFFERS_1024) && !defined(USE_CERT_BUFFERS_2048)) \
    || !defined(USE_CERT_BUFFERS_256)
    XFILE       file3;
#endif
    int ret;

#ifdef WOLFSSL_SMALL_STACK
    if ((caKey == NULL) || (caEccKey == NULL) || (caEccKeyPub == NULL)
#ifdef WOLFSSL_TEST_CERT
        || (decode == NULL)
#endif
        )
        ERROR_OUT(MEMORY_E, exit_rsa);
#endif

    XMEMSET(caKey, 0, sizeof *caKey);
    XMEMSET(caEccKey, 0, sizeof *caEccKey);
    XMEMSET(caEccKeyPub, 0, sizeof *caEccKeyPub);

    der = (byte*)XMALLOC(FOURK_BUF, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    if (der == NULL) {
        ERROR_OUT(-7850, exit_rsa);
    }
    myCert = (Cert*)XMALLOC(sizeof(Cert), HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    if (myCert == NULL) {
        ERROR_OUT(-7851, exit_rsa);
    }

    /* Get CA Key */
#ifdef USE_CERT_BUFFERS_1024
    XMEMCPY(tmp, ca_key_der_1024, sizeof_ca_key_der_1024);
    bytes3 = sizeof_ca_key_der_1024;
#elif defined(USE_CERT_BUFFERS_2048)
    XMEMCPY(tmp, ca_key_der_2048, sizeof_ca_key_der_2048);
    bytes3 = sizeof_ca_key_der_2048;
#else
    file3 = XFOPEN(rsaCaKeyFile, "rb");
    if (!file3) {
        ERROR_OUT(-7852, exit_rsa);
    }

    bytes3 = XFREAD(tmp, 1, FOURK_BUF, file3);
    XFCLOSE(file3);
#endif /* USE_CERT_BUFFERS */

    ret = wc_InitRsaKey(caKey, HEAP_HINT);
    if (ret != 0) {
        ERROR_OUT(-7853, exit_rsa);
    }
    ret = wc_RsaPrivateKeyDecode(tmp, &idx3, caKey, (word32)bytes3);
    if (ret != 0) {
        ERROR_OUT(-7854, exit_rsa);
    }

    /* Get Cert Key */
#ifdef USE_CERT_BUFFERS_256
    XMEMCPY(tmp, ecc_key_pub_der_256, sizeof_ecc_key_pub_der_256);
    bytes3 = sizeof_ecc_key_pub_der_256;
#else
    file3 = XFOPEN(eccKeyPubFileDer, "rb");
    if (!file3) {
        ERROR_OUT(-7855, exit_rsa);
    }

    bytes3 = XFREAD(tmp, 1, FOURK_BUF, file3);
    XFCLOSE(file3);
#endif

    ret = wc_ecc_init_ex(caEccKeyPub, HEAP_HINT, devId);
    if (ret != 0) {
        ERROR_OUT(-7856, exit_rsa);
    }

    idx3 = 0;
    ret = wc_EccPublicKeyDecode(tmp, &idx3, caEccKeyPub, (word32)bytes3);
    if (ret != 0) {
        ERROR_OUT(-7857, exit_rsa);
    }

    /* Setup Certificate */
    if (wc_InitCert_ex(myCert, HEAP_HINT, devId)) {
        ERROR_OUT(-7858, exit_rsa);
    }

#ifndef NO_SHA256
    myCert->sigType = CTC_SHA256wRSA;
#else
    myCert->sigType = CTC_SHAwRSA;
#endif

    XMEMCPY(&myCert->subject, &certDefaultName, sizeof(CertName));

#ifdef WOLFSSL_CERT_EXT
    /* add Policies */
    XSTRNCPY(myCert->certPolicies[0], "2.4.589440.587.101.2.1.9632587.1",
            CTC_MAX_CERTPOL_SZ);
    XSTRNCPY(myCert->certPolicies[1], "1.2.13025.489.1.113549",
            CTC_MAX_CERTPOL_SZ);
    myCert->certPoliciesNb = 2;

    /* add SKID from the Public Key */
    if (wc_SetSubjectKeyIdFromPublicKey(myCert, NULL, caEccKeyPub) != 0) {
        ERROR_OUT(-7859, exit_rsa);
    }

    /* add AKID from the CA certificate */
#if defined(USE_CERT_BUFFERS_2048)
    ret = wc_SetAuthKeyIdFromCert(myCert, ca_cert_der_2048,
                                           sizeof_ca_cert_der_2048);
#elif defined(USE_CERT_BUFFERS_1024)
    ret = wc_SetAuthKeyIdFromCert(myCert, ca_cert_der_1024,
                                           sizeof_ca_cert_der_1024);
#else
    ret = wc_SetAuthKeyId(myCert, rsaCaCertFile);
#endif
    if (ret != 0) {
        ERROR_OUT(-7860, exit_rsa);
    }

    /* add Key Usage */
    if (wc_SetKeyUsage(myCert, certKeyUsage) != 0) {
        ERROR_OUT(-7861, exit_rsa);
    }
#endif /* WOLFSSL_CERT_EXT */

#if defined(USE_CERT_BUFFERS_2048)
    ret = wc_SetIssuerBuffer(myCert, ca_cert_der_2048,
                                      sizeof_ca_cert_der_2048);
#elif defined(USE_CERT_BUFFERS_1024)
    ret = wc_SetIssuerBuffer(myCert, ca_cert_der_1024,
                                      sizeof_ca_cert_der_1024);
#else
    ret = wc_SetIssuer(myCert, rsaCaCertFile);
#endif
    if (ret < 0) {
        ERROR_OUT(-7862, exit_rsa);
    }

    certSz = wc_MakeCert(myCert, der, FOURK_BUF, NULL, caEccKeyPub, rng);
    if (certSz < 0) {
        ERROR_OUT(-7863, exit_rsa);
    }

    ret = 0;
    do {
    #if defined(WOLFSSL_ASYNC_CRYPT)
        ret = wc_AsyncWait(ret, &caEccKey->asyncDev, WC_ASYNC_FLAG_CALL_AGAIN);
    #endif
        if (ret >= 0) {
            ret = wc_SignCert(myCert->bodySz, myCert->sigType, der,
                              FOURK_BUF, caKey, NULL, rng);
        }
    } while (ret == WC_PENDING_E);
    if (ret < 0) {
        ERROR_OUT(-7864, exit_rsa);
    }
    certSz = ret;

#ifdef WOLFSSL_TEST_CERT
    InitDecodedCert(decode, der, certSz, 0);
    ret = ParseCert(decode, CERT_TYPE, NO_VERIFY, 0);
    if (ret != 0) {
        FreeDecodedCert(decode);
        ERROR_OUT(-7865, exit_rsa);

    }
    FreeDecodedCert(decode);
#endif

    ret = SaveDerAndPem(der, certSz, certEccRsaDerFile, certEccRsaPemFile,
        CERT_TYPE, -5616);
    if (ret != 0) {
        goto exit_rsa;
    }

exit_rsa:

#ifdef WOLFSSL_SMALL_STACK
    if (caKey != NULL) {
        wc_FreeRsaKey(caKey);
        XFREE(caKey, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    }
    if (caEccKey != NULL) {
        wc_ecc_free(caEccKey);
        XFREE(caEccKey, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    }
    if (caEccKeyPub != NULL) {
        wc_ecc_free(caEccKeyPub);
        XFREE(caEccKeyPub, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    }
#ifdef WOLFSSL_TEST_CERT
    if (decode != NULL)
        XFREE(decode, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
#endif
#else
    wc_FreeRsaKey(caKey);
    wc_ecc_free(caEccKey);
    wc_ecc_free(caEccKeyPub);
#endif

    XFREE(myCert, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    myCert = NULL;
    XFREE(der, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    der = NULL;

    if (ret >= 0)
        ret = 0;
    return ret;
}
#endif /* !NO_RSA && HAVE_ECC && WOLFSSL_CERT_GEN */

#ifdef WOLFSSL_KEY_GEN
static int rsa_keygen_test(WC_RNG* rng)
{
#ifdef WOLFSSL_SMALL_STACK
    RsaKey *genKey = (RsaKey *)XMALLOC(sizeof *genKey, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
#else
    RsaKey genKey[1];
#endif
    int    ret;
    byte*  der = NULL;
#ifndef WOLFSSL_CRYPTOCELL
    word32 idx = 0;
#endif
    int    derSz = 0;
#if !defined(WOLFSSL_SP_MATH) && !defined(HAVE_FIPS)
    int    keySz = 1024;
#else
    int    keySz = 2048;
#endif

#ifdef WOLFSSL_SMALL_STACK
    if (! genKey)
        ERROR_OUT(MEMORY_E, exit_rsa);
#endif

    XMEMSET(genKey, 0, sizeof *genKey);

    ret = wc_InitRsaKey_ex(genKey, HEAP_HINT, devId);
    if (ret != 0) {
        ERROR_OUT(-7870, exit_rsa);
    }

#ifdef HAVE_FIPS
    for (;;) {
#endif
        ret = wc_MakeRsaKey(genKey, keySz, WC_RSA_EXPONENT, rng);
#if defined(WOLFSSL_ASYNC_CRYPT)
        ret = wc_AsyncWait(ret, &genKey->asyncDev, WC_ASYNC_FLAG_NONE);
#endif
#ifdef HAVE_FIPS
        if (ret == PRIME_GEN_E)
            continue;
        break;
    }
#endif

    if (ret != 0) {
        ERROR_OUT(-7871, exit_rsa);
    }
    TEST_SLEEP();

#ifdef WOLFSSL_RSA_KEY_CHECK
    ret = wc_CheckRsaKey(genKey);
    if (ret != 0) {
        ERROR_OUT(-7872, exit_rsa);
    }
#endif
    der = (byte*)XMALLOC(FOURK_BUF, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    if (der == NULL) {
        ERROR_OUT(-7873, exit_rsa);
    }

    derSz = wc_RsaKeyToDer(genKey, der, FOURK_BUF);
    if (derSz < 0) {
        ERROR_OUT(-7874, exit_rsa);
    }

    ret = SaveDerAndPem(der, derSz, keyDerFile, keyPemFile,
        PRIVATEKEY_TYPE, -5555);
    if (ret != 0) {
        goto exit_rsa;
    }

    wc_FreeRsaKey(genKey);
    ret = wc_InitRsaKey(genKey, HEAP_HINT);
    if (ret != 0) {
        ERROR_OUT(-7875, exit_rsa);
    }

#ifndef WOLFSSL_CRYPTOCELL
    idx = 0;
    /* The private key part of the key gen pairs from cryptocell can't be exported */
    ret = wc_RsaPrivateKeyDecode(der, &idx, genKey, derSz);
    if (ret != 0) {
        ERROR_OUT(-7876, exit_rsa);
    }
#endif /* WOLFSSL_CRYPTOCELL */

exit_rsa:

#ifdef WOLFSSL_SMALL_STACK
    if (genKey) {
        wc_FreeRsaKey(genKey);
        XFREE(genKey, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    }
#else
    wc_FreeRsaKey(genKey);
#endif

    if (der != NULL) {
        XFREE(der, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
        der = NULL;
    }

    return ret;
}
#endif

#ifndef WOLFSSL_RSA_VERIFY_ONLY
#if !defined(WC_NO_RSA_OAEP) && !defined(WC_NO_RNG) && \
    !defined(HAVE_FAST_RSA) && !defined(HAVE_USER_RSA) && \
     (!defined(HAVE_FIPS) || \
      (defined(HAVE_FIPS_VERSION) && (HAVE_FIPS_VERSION >= 2))) \
      && !defined(WOLF_CRYPTO_CB_ONLY_RSA)
static int rsa_oaep_padding_test(RsaKey* key, WC_RNG* rng)
{
    int ret = 0;
    word32 idx = 0;
    const char inStr[] = TEST_STRING;
    const word32 inLen = (word32)TEST_STRING_SZ;
    const word32 outSz   = RSA_TEST_BYTES;
    const word32 plainSz = RSA_TEST_BYTES;
    byte*  res = NULL;

    WC_DECLARE_VAR(in, byte, TEST_STRING_SZ, HEAP_HINT);
    WC_DECLARE_VAR(out, byte, RSA_TEST_BYTES, HEAP_HINT);
    WC_DECLARE_VAR(plain, byte, RSA_TEST_BYTES, HEAP_HINT);

#ifdef WC_DECLARE_VAR_IS_HEAP_ALLOC
    if (in == NULL || out == NULL || plain == NULL)
        ERROR_OUT(MEMORY_E, exit_rsa);
#endif

    XMEMCPY(in, inStr, inLen);

#ifndef NO_SHA
    do {
#if defined(WOLFSSL_ASYNC_CRYPT)
        ret = wc_AsyncWait(ret, &key->asyncDev, WC_ASYNC_FLAG_CALL_AGAIN);
#endif

        if (ret >= 0) {
            ret = wc_RsaPublicEncrypt_ex(in, inLen, out, outSz, key, rng,
                       WC_RSA_OAEP_PAD, WC_HASH_TYPE_SHA, WC_MGF1SHA1, NULL, 0);
        }
    } while (ret == WC_PENDING_E);
    if (ret < 0) {
        ERROR_OUT(-7918, exit_rsa);
    }
    TEST_SLEEP();

#ifndef WOLFSSL_RSA_PUBLIC_ONLY
    idx = (word32)ret;
    do {
#if defined(WOLFSSL_ASYNC_CRYPT)
        ret = wc_AsyncWait(ret, &key->asyncDev, WC_ASYNC_FLAG_CALL_AGAIN);
#endif
        if (ret >= 0) {
            ret = wc_RsaPrivateDecrypt_ex(out, idx, plain, plainSz, key,
                       WC_RSA_OAEP_PAD, WC_HASH_TYPE_SHA, WC_MGF1SHA1, NULL, 0);
        }
    } while (ret == WC_PENDING_E);
    if (ret < 0) {
        ERROR_OUT(-7919, exit_rsa);
    }

    if (XMEMCMP(plain, in, inLen)) {
        ERROR_OUT(-7920, exit_rsa);
    }
    TEST_SLEEP();
    #endif /* NO_SHA */
#endif

    #ifndef NO_SHA256
    XMEMSET(plain, 0, plainSz);
    do {
#if defined(WOLFSSL_ASYNC_CRYPT)
        ret = wc_AsyncWait(ret, &key->asyncDev, WC_ASYNC_FLAG_CALL_AGAIN);
#endif
        if (ret >= 0) {
            ret = wc_RsaPublicEncrypt_ex(in, inLen, out, outSz, key, rng,
                  WC_RSA_OAEP_PAD, WC_HASH_TYPE_SHA256, WC_MGF1SHA256, NULL, 0);
        }
    } while (ret == WC_PENDING_E);
    if (ret < 0) {
        ERROR_OUT(-7921, exit_rsa);
    }
    TEST_SLEEP();

    idx = (word32)ret;
#ifndef WOLFSSL_RSA_PUBLIC_ONLY
    do {
#if defined(WOLFSSL_ASYNC_CRYPT)
        ret = wc_AsyncWait(ret, &key->asyncDev, WC_ASYNC_FLAG_CALL_AGAIN);
#endif
        if (ret >= 0) {
            ret = wc_RsaPrivateDecrypt_ex(out, idx, plain, plainSz, key,
                  WC_RSA_OAEP_PAD, WC_HASH_TYPE_SHA256, WC_MGF1SHA256, NULL, 0);
        }
    } while (ret == WC_PENDING_E);
    if (ret < 0) {
        ERROR_OUT(-7922, exit_rsa);
    }

    if (XMEMCMP(plain, in, inLen)) {
        ERROR_OUT(-7923, exit_rsa);
    }
    TEST_SLEEP();
#endif /* WOLFSSL_RSA_PUBLIC_ONLY */

#ifndef WOLFSSL_RSA_PUBLIC_ONLY
    do {
#if defined(WOLFSSL_ASYNC_CRYPT)
        ret = wc_AsyncWait(ret, &key->asyncDev, WC_ASYNC_FLAG_CALL_AGAIN);
#endif
        if (ret >= 0) {
            ret = wc_RsaPrivateDecryptInline_ex(out, idx, &res, key,
                WC_RSA_OAEP_PAD, WC_HASH_TYPE_SHA256, WC_MGF1SHA256, NULL, 0);
        }
    } while (ret == WC_PENDING_E);
    if (ret < 0) {
        ERROR_OUT(-7924, exit_rsa);
    }
    if (ret != (int)inLen) {
        ERROR_OUT(-7925, exit_rsa);
    }
    if (XMEMCMP(res, in, inLen)) {
        ERROR_OUT(-7926, exit_rsa);
    }
    TEST_SLEEP();
#endif /* WOLFSSL_RSA_PUBLIC_ONLY */

    /* check fails if not using the same optional label */
    XMEMSET(plain, 0, plainSz);
    do {
#if defined(WOLFSSL_ASYNC_CRYPT)
        ret = wc_AsyncWait(ret, &key->asyncDev, WC_ASYNC_FLAG_CALL_AGAIN);
#endif
        if (ret >= 0) {
            ret = wc_RsaPublicEncrypt_ex(in, inLen, out, outSz, key, rng,
                  WC_RSA_OAEP_PAD, WC_HASH_TYPE_SHA256, WC_MGF1SHA256, NULL, 0);
        }
    } while (ret == WC_PENDING_E);
    if (ret < 0) {
        ERROR_OUT(-7927, exit_rsa);
    }
    TEST_SLEEP();

/* TODO: investigate why Cavium Nitrox doesn't detect decrypt error here */
#if !defined(HAVE_CAVIUM) && !defined(WOLFSSL_RSA_PUBLIC_ONLY) && \
    !defined(WOLFSSL_CRYPTOCELL)
/* label is unused in cryptocell so it won't detect decrypt error due to label */
    idx = (word32)ret;
    do {
#if defined(WOLFSSL_ASYNC_CRYPT)
        ret = wc_AsyncWait(ret, &key->asyncDev, WC_ASYNC_FLAG_CALL_AGAIN);
#endif
        if (ret >= 0) {
            ret = wc_RsaPrivateDecrypt_ex(out, idx, plain, plainSz, key,
               WC_RSA_OAEP_PAD, WC_HASH_TYPE_SHA256, WC_MGF1SHA256, in, inLen);
        }
    } while (ret == WC_PENDING_E);
    if (ret > 0) { /* in this case decrypt should fail */
        ERROR_OUT(-7928, exit_rsa);
    }
    ret = 0;
    TEST_SLEEP();
#endif /* !HAVE_CAVIUM */

    /* check using optional label with encrypt/decrypt */
    XMEMSET(plain, 0, plainSz);
    do {
#if defined(WOLFSSL_ASYNC_CRYPT)
        ret = wc_AsyncWait(ret, &key->asyncDev, WC_ASYNC_FLAG_CALL_AGAIN);
#endif
        if (ret >= 0) {
            ret = wc_RsaPublicEncrypt_ex(in, inLen, out, outSz, key, rng,
               WC_RSA_OAEP_PAD, WC_HASH_TYPE_SHA256, WC_MGF1SHA256, in, inLen);
        }
    } while (ret == WC_PENDING_E);
    if (ret < 0) {
        ERROR_OUT(-7929, exit_rsa);
    }
    TEST_SLEEP();

    idx = (word32)ret;
#ifndef WOLFSSL_RSA_PUBLIC_ONLY
    do {
#if defined(WOLFSSL_ASYNC_CRYPT)
        ret = wc_AsyncWait(ret, &key->asyncDev, WC_ASYNC_FLAG_CALL_AGAIN);
#endif
        if (ret >= 0) {
            ret = wc_RsaPrivateDecrypt_ex(out, idx, plain, plainSz, key,
               WC_RSA_OAEP_PAD, WC_HASH_TYPE_SHA256, WC_MGF1SHA256, in, inLen);
        }
    } while (ret == WC_PENDING_E);
    if (ret < 0) {
        ERROR_OUT(-7930, exit_rsa);
    }

    if (XMEMCMP(plain, in, inLen)) {
        ERROR_OUT(-7931, exit_rsa);
    }
    TEST_SLEEP();
#endif /* WOLFSSL_RSA_PUBLIC_ONLY */

    #ifndef NO_SHA
        /* check fail using mismatch hash algorithms */
        XMEMSET(plain, 0, plainSz);
        do {
    #if defined(WOLFSSL_ASYNC_CRYPT)
            ret = wc_AsyncWait(ret, &key->asyncDev, WC_ASYNC_FLAG_CALL_AGAIN);
    #endif
            if (ret >= 0) {
                ret = wc_RsaPublicEncrypt_ex(in, inLen, out, outSz, key, rng,
                    WC_RSA_OAEP_PAD, WC_HASH_TYPE_SHA, WC_MGF1SHA1, in, inLen);
            }
        } while (ret == WC_PENDING_E);
        if (ret < 0) {
            ERROR_OUT(-7932, exit_rsa);
        }
        TEST_SLEEP();

/* TODO: investigate why Cavium Nitrox doesn't detect decrypt error here */
#if !defined(HAVE_CAVIUM) && !defined(WOLFSSL_RSA_PUBLIC_ONLY) && \
    !defined(WOLFSSL_CRYPTOCELL)
        idx = (word32)ret;
        do {
    #if defined(WOLFSSL_ASYNC_CRYPT)
            ret = wc_AsyncWait(ret, &key->asyncDev, WC_ASYNC_FLAG_CALL_AGAIN);
    #endif
            if (ret >= 0) {
                ret = wc_RsaPrivateDecrypt_ex(out, idx, plain, plainSz, key,
                    WC_RSA_OAEP_PAD, WC_HASH_TYPE_SHA256, WC_MGF1SHA256,
                    in, inLen);
            }
        } while (ret == WC_PENDING_E);
        if (ret > 0) { /* should fail */
            ERROR_OUT(-7933, exit_rsa);
        }
        ret = 0;
        TEST_SLEEP();
    #endif /* !HAVE_CAVIUM */
    #endif /* NO_SHA */
#endif /* NO_SHA256 */

#ifdef WOLFSSL_SHA512
    /* Check valid RSA key size is used while using hash length of SHA512
       If key size is less than (hash length * 2) + 2 then is invalid use
       and test, since OAEP padding requires this.
       BAD_FUNC_ARG is returned when this case is not met */
    if (wc_RsaEncryptSize(key) > ((int)WC_SHA512_DIGEST_SIZE * 2) + 2) {
        XMEMSET(plain, 0, plainSz);
        do {
    #if defined(WOLFSSL_ASYNC_CRYPT)
            ret = wc_AsyncWait(ret, &key->asyncDev, WC_ASYNC_FLAG_CALL_AGAIN);
    #endif
            if (ret >= 0) {
                ret = wc_RsaPublicEncrypt_ex(in, inLen, out, outSz, key, rng,
                  WC_RSA_OAEP_PAD, WC_HASH_TYPE_SHA512, WC_MGF1SHA512, NULL, 0);
            }
        } while (ret == WC_PENDING_E);
        if (ret < 0) {
            ERROR_OUT(-7934, exit_rsa);
        }
        TEST_SLEEP();

        idx = ret;
#ifndef WOLFSSL_RSA_PUBLIC_ONLY
        do {
    #if defined(WOLFSSL_ASYNC_CRYPT)
            ret = wc_AsyncWait(ret, &key->asyncDev, WC_ASYNC_FLAG_CALL_AGAIN);
    #endif
            if (ret >= 0) {
                ret = wc_RsaPrivateDecrypt_ex(out, idx, plain, plainSz, key,
                  WC_RSA_OAEP_PAD, WC_HASH_TYPE_SHA512, WC_MGF1SHA512, NULL, 0);
            }
        } while (ret == WC_PENDING_E);
        if (ret < 0) {
            ERROR_OUT(-7935, exit_rsa);
        }

        if (XMEMCMP(plain, in, inLen)) {
            ERROR_OUT(-7936, exit_rsa);
        }
        TEST_SLEEP();
#endif /* WOLFSSL_RSA_PUBLIC_ONLY */
    }
#endif /* WOLFSSL_SHA512 */

    /* check using pkcsv15 padding with _ex API */
    XMEMSET(plain, 0, plainSz);
    do {
#if defined(WOLFSSL_ASYNC_CRYPT)
        ret = wc_AsyncWait(ret, &key->asyncDev, WC_ASYNC_FLAG_CALL_AGAIN);
#endif
        if (ret >= 0) {
            ret = wc_RsaPublicEncrypt_ex(in, inLen, out, outSz, key, rng,
                  WC_RSA_PKCSV15_PAD, WC_HASH_TYPE_NONE, 0, NULL, 0);
        }
    } while (ret == WC_PENDING_E);
    if (ret < 0) {
        ERROR_OUT(-7937, exit_rsa);
    }
    TEST_SLEEP();

    idx = (word32)ret;
#ifndef WOLFSSL_RSA_PUBLIC_ONLY
    do {
#if defined(WOLFSSL_ASYNC_CRYPT)
        ret = wc_AsyncWait(ret, &key->asyncDev, WC_ASYNC_FLAG_CALL_AGAIN);
#endif
        if (ret >= 0) {
            ret = wc_RsaPrivateDecrypt_ex(out, idx, plain, plainSz, key,
                  WC_RSA_PKCSV15_PAD, WC_HASH_TYPE_NONE, 0, NULL, 0);
        }
    } while (ret == WC_PENDING_E);
    if (ret < 0) {
        ERROR_OUT(-7938, exit_rsa);
    }

    if (XMEMCMP(plain, in, inLen)) {
        ERROR_OUT(-7939, exit_rsa);
    }
    TEST_SLEEP();
#endif /* WOLFSSL_RSA_PUBLIC_ONLY */

exit_rsa:
    WC_FREE_VAR(in, HEAP_HINT);
    WC_FREE_VAR(out, HEAP_HINT);
    WC_FREE_VAR(plain, HEAP_HINT);

    (void)idx;
    (void)inStr;
    (void)res;

    if (ret >= 0)
        ret = 0;

    return ret;

}
#endif
#endif

WOLFSSL_TEST_SUBROUTINE int rsa_test(void)
{
    int    ret;
    size_t bytes;
    WC_RNG rng;
#ifdef WOLFSSL_SMALL_STACK
    byte*  tmp = NULL;
    byte*  der = NULL;
    RsaKey *key = (RsaKey *)XMALLOC(sizeof *key, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
#else
    RsaKey key[1];
    byte tmp[FOURK_BUF];
#endif
#if defined(WOLFSSL_CERT_EXT) || defined(WOLFSSL_CERT_GEN)
#ifdef WOLFSSL_SMALL_STACK
    RsaKey *keypub = (RsaKey *)XMALLOC(sizeof *keypub, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
#else
    RsaKey keypub[1];
#endif
#endif
    word32 idx = 0;
    const char inStr[] = TEST_STRING;
    const word32 inLen = (word32)TEST_STRING_SZ;
    const word32 outSz   = RSA_TEST_BYTES;
    const word32 plainSz = RSA_TEST_BYTES;
    byte*  res = NULL;
#ifndef NO_SIG_WRAPPER
    int modLen;
#endif
#if !defined(USE_CERT_BUFFERS_1024) && !defined(USE_CERT_BUFFERS_2048) && \
    !defined(USE_CERT_BUFFERS_3072) && !defined(USE_CERT_BUFFERS_4096) && \
    !defined(NO_FILESYSTEM)
    XFILE   file;
#ifdef WOLFSSL_TEST_CERT
    XFILE   file2;
#endif
#endif
#ifdef WOLFSSL_TEST_CERT
#ifdef WOLFSSL_SMALL_STACK
    DecodedCert *cert = (DecodedCert *)XMALLOC(sizeof *cert, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
#else
    DecodedCert cert[1];
#endif
#ifndef NO_ASN_TIME
    struct tm   timearg;
    const byte* date;
    byte        dateFormat;
    int         dateLength;
#endif
#endif

    WC_DECLARE_VAR(in, byte, TEST_STRING_SZ, HEAP_HINT);
    WC_DECLARE_VAR(out, byte, RSA_TEST_BYTES, HEAP_HINT);
    WC_DECLARE_VAR(plain, byte, RSA_TEST_BYTES, HEAP_HINT);

#ifdef WC_DECLARE_VAR_IS_HEAP_ALLOC
    if (in == NULL || out == NULL || plain == NULL)
        ERROR_OUT(MEMORY_E, exit_rsa);
#endif

    XMEMCPY(in, inStr, inLen);

#ifdef WOLFSSL_SMALL_STACK
    if (key == NULL)
        ERROR_OUT(MEMORY_E, exit_rsa);
#if defined(WOLFSSL_CERT_EXT) || defined(WOLFSSL_CERT_GEN)
    if (keypub == NULL)
        ERROR_OUT(MEMORY_E, exit_rsa);
#endif
#ifdef WOLFSSL_TEST_CERT
    if (cert == NULL)
        ERROR_OUT(MEMORY_E, exit_rsa);
#endif
#endif /* WOLFSSL_SMALL_STACK */

    /* initialize stack structures */
    XMEMSET(&rng, 0, sizeof(rng));
    XMEMSET(key, 0, sizeof *key);
#if defined(WOLFSSL_CERT_EXT) || defined(WOLFSSL_CERT_GEN)
    XMEMSET(keypub, 0, sizeof *keypub);
#endif

#if !defined(HAVE_USER_RSA) && !defined(NO_ASN)
    ret = rsa_decode_test(key);
    if (ret != 0)
        ERROR_OUT(ret, exit_rsa);
#endif

#ifdef USE_CERT_BUFFERS_1024
    bytes = (size_t)sizeof_client_key_der_1024;
    if (bytes < (size_t)sizeof_client_cert_der_1024)
        bytes = (size_t)sizeof_client_cert_der_1024;
#elif defined(USE_CERT_BUFFERS_2048)
    bytes = (size_t)sizeof_client_key_der_2048;
    if (bytes < (size_t)sizeof_client_cert_der_2048)
        bytes = (size_t)sizeof_client_cert_der_2048;
#elif defined(USE_CERT_BUFFERS_3072)
    bytes = (size_t)sizeof_client_key_der_3072;
    if (bytes < (size_t)sizeof_client_cert_der_3072)
        bytes = (size_t)sizeof_client_cert_der_3072;
#elif defined(USE_CERT_BUFFERS_4096)
    bytes = (size_t)sizeof_client_key_der_4096;
    if (bytes < (size_t)sizeof_client_cert_der_4096)
        bytes = (size_t)sizeof_client_cert_der_4096;
#else
    bytes = FOURK_BUF;
#endif

#if defined(WOLFSSL_SMALL_STACK)
    tmp = (byte*)XMALLOC(bytes, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    if (tmp == NULL)
        ERROR_OUT(-7900, exit_rsa);
#endif

#ifdef USE_CERT_BUFFERS_1024
    XMEMCPY(tmp, client_key_der_1024, (size_t)sizeof_client_key_der_1024);
#elif defined(USE_CERT_BUFFERS_2048)
    XMEMCPY(tmp, client_key_der_2048, (size_t)sizeof_client_key_der_2048);
#elif defined(USE_CERT_BUFFERS_3072)
    XMEMCPY(tmp, client_key_der_3072, (size_t)sizeof_client_key_der_3072);
#elif defined(USE_CERT_BUFFERS_4096)
    XMEMCPY(tmp, client_key_der_4096, (size_t)sizeof_client_key_der_4096);
#elif !defined(NO_FILESYSTEM)
    file = XFOPEN(clientKey, "rb");
    if (!file) {
        err_sys("can't open ./certs/client-key.der, "
                "Please run from wolfSSL home dir", -40);
        ERROR_OUT(-7901, exit_rsa);
    }

    bytes = XFREAD(tmp, 1, FOURK_BUF, file);
    XFCLOSE(file);
#else
    /* No key to use. */
    ERROR_OUT(-7902, exit_rsa);
#endif /* USE_CERT_BUFFERS */

    ret = wc_InitRsaKey_ex(key, HEAP_HINT, devId);
    if (ret != 0) {
        ERROR_OUT(-7903, exit_rsa);
    }
#ifndef NO_ASN
    ret = wc_RsaPrivateKeyDecode(tmp, &idx, key, (word32)bytes);
    if (ret != 0) {
        ERROR_OUT(-7904, exit_rsa);
    }
#ifndef NO_SIG_WRAPPER
    modLen = wc_RsaEncryptSize(key);
#endif
#else
    #ifdef USE_CERT_BUFFERS_2048
        ret = mp_read_unsigned_bin(&key->n, &tmp[12], 256);
        if (ret != 0) {
            ERROR_OUT(-7905, exit_rsa);
        }
        ret = mp_set_int(&key->e, WC_RSA_EXPONENT);
        if (ret != 0) {
            ERROR_OUT(-7906, exit_rsa);
        }
#ifndef NO_SIG_WRAPPER
        modLen = 2048;
#endif
    #else
        #error Not supported yet!
    #endif
#endif

#ifndef WC_NO_RNG
#ifndef HAVE_FIPS
    ret = wc_InitRng_ex(&rng, HEAP_HINT, devId);
#else
    ret = wc_InitRng(&rng);
#endif
    if (ret != 0) {
        ERROR_OUT(-7907, exit_rsa);
    }
#endif

#ifndef NO_SIG_WRAPPER
    ret = rsa_sig_test(key, sizeof *key, modLen, &rng);
    if (ret != 0)
        goto exit_rsa;
#endif

#ifdef WC_RSA_NONBLOCK
    ret = rsa_nb_test(key, in, inLen, out, outSz, plain, plainSz, &rng);
    if (ret != 0)
        goto exit_rsa;
#endif

#if !defined(WOLFSSL_RSA_VERIFY_ONLY) && !defined(WOLFSSL_RSA_PUBLIC_ONLY) && \
    !defined(WC_NO_RNG) && !defined(WOLF_CRYPTO_CB_ONLY_RSA)
    do {
#if defined(WOLFSSL_ASYNC_CRYPT)
        ret = wc_AsyncWait(ret, &key->asyncDev, WC_ASYNC_FLAG_CALL_AGAIN);
#endif
        if (ret >= 0) {
            ret = wc_RsaPublicEncrypt(in, inLen, out, outSz, key, &rng);
        }
    } while (ret == WC_PENDING_E);
    if (ret < 0) {
        ERROR_OUT(-7908, exit_rsa);
    }
    TEST_SLEEP();

#ifdef WC_RSA_BLINDING
    {
        int tmpret = ret;
        ret = wc_RsaSetRNG(key, &rng);
        if (ret < 0) {
            ERROR_OUT(-7909, exit_rsa);
        }
        ret = tmpret;
    }
#endif

    idx = (word32)ret; /* save off encrypted length */
    do {
#if defined(WOLFSSL_ASYNC_CRYPT)
        ret = wc_AsyncWait(ret, &key->asyncDev, WC_ASYNC_FLAG_CALL_AGAIN);
#endif
        if (ret >= 0) {
            ret = wc_RsaPrivateDecrypt(out, idx, plain, plainSz, key);
        }
    } while (ret == WC_PENDING_E);
    if (ret < 0) {
        ERROR_OUT(-7910, exit_rsa);
    }

    if (XMEMCMP(plain, in, inLen)) {
        ERROR_OUT(-7911, exit_rsa);
    }
    TEST_SLEEP();

    do {
#if defined(WOLFSSL_ASYNC_CRYPT)
        ret = wc_AsyncWait(ret, &key->asyncDev, WC_ASYNC_FLAG_CALL_AGAIN);
#endif
        if (ret >= 0) {
            ret = wc_RsaPrivateDecryptInline(out, idx, &res, key);
        }
    } while (ret == WC_PENDING_E);
    if (ret < 0) {
        ERROR_OUT(-7912, exit_rsa);
    }
    if (ret != (int)inLen) {
        ERROR_OUT(-7913, exit_rsa);
    }
    if (XMEMCMP(res, in, inLen)) {
        ERROR_OUT(-7914, exit_rsa);
    }
    TEST_SLEEP();

    do {
#if defined(WOLFSSL_ASYNC_CRYPT)
        ret = wc_AsyncWait(ret, &key->asyncDev, WC_ASYNC_FLAG_CALL_AGAIN);
#endif
        if (ret >= 0) {
            ret = wc_RsaSSL_Sign(in, inLen, out, outSz, key, &rng);
        }
    } while (ret == WC_PENDING_E);
    if (ret < 0) {
        ERROR_OUT(-7915, exit_rsa);
    }
    TEST_SLEEP();

#elif defined(WOLFSSL_PUBLIC_MP)
    {
        static byte signature_2048[] = {
            0x07, 0x6f, 0xc9, 0x85, 0x73, 0x9e, 0x21, 0x79,
            0x47, 0xf1, 0xa3, 0xd7, 0xf4, 0x27, 0x29, 0xbe,
            0x99, 0x5d, 0xac, 0xb2, 0x10, 0x3f, 0x95, 0xda,
            0x89, 0x23, 0xb8, 0x96, 0x13, 0x57, 0x72, 0x30,
            0xa1, 0xfe, 0x5a, 0x68, 0x9c, 0x99, 0x9d, 0x1e,
            0x05, 0xa4, 0x80, 0xb0, 0xbb, 0xd9, 0xd9, 0xa1,
            0x69, 0x97, 0x74, 0xb3, 0x41, 0x21, 0x3b, 0x47,
            0xf5, 0x51, 0xb1, 0xfb, 0xc7, 0xaa, 0xcc, 0xdc,
            0xcd, 0x76, 0xa0, 0x28, 0x4d, 0x27, 0x14, 0xa4,
            0xb9, 0x41, 0x68, 0x7c, 0xb3, 0x66, 0xe6, 0x6f,
            0x40, 0x76, 0xe4, 0x12, 0xfd, 0xae, 0x29, 0xb5,
            0x63, 0x60, 0x87, 0xce, 0x49, 0x6b, 0xf3, 0x05,
            0x9a, 0x14, 0xb5, 0xcc, 0xcd, 0xf7, 0x30, 0x95,
            0xd2, 0x72, 0x52, 0x1d, 0x5b, 0x7e, 0xef, 0x4a,
            0x02, 0x96, 0x21, 0x6c, 0x55, 0xa5, 0x15, 0xb1,
            0x57, 0x63, 0x2c, 0xa3, 0x8e, 0x9d, 0x3d, 0x45,
            0xcc, 0xb8, 0xe6, 0xa1, 0xc8, 0x59, 0xcd, 0xf5,
            0xdc, 0x0a, 0x51, 0xb6, 0x9d, 0xfb, 0xf4, 0x6b,
            0xfd, 0x32, 0x71, 0x6e, 0xcf, 0xcb, 0xb3, 0xd9,
            0xe0, 0x4a, 0x77, 0x34, 0xd6, 0x61, 0xf5, 0x7c,
            0xf9, 0xa9, 0xa4, 0xb0, 0x8e, 0x3b, 0xd6, 0x04,
            0xe0, 0xde, 0x2b, 0x5b, 0x5a, 0xbf, 0xd9, 0xef,
            0x8d, 0xa3, 0xf5, 0xb1, 0x67, 0xf3, 0xb9, 0x72,
            0x0a, 0x37, 0x12, 0x35, 0x6c, 0x8e, 0x10, 0x8b,
            0x38, 0x06, 0x16, 0x4b, 0x20, 0x20, 0x13, 0x00,
            0x2e, 0x6d, 0xc2, 0x59, 0x23, 0x67, 0x4a, 0x6d,
            0xa1, 0x46, 0x8b, 0xee, 0xcf, 0x44, 0xb4, 0x3e,
            0x56, 0x75, 0x00, 0x68, 0xb5, 0x7d, 0x0f, 0x20,
            0x79, 0x5d, 0x7f, 0x12, 0x15, 0x32, 0x89, 0x61,
            0x6b, 0x29, 0xb7, 0x52, 0xf5, 0x25, 0xd8, 0x98,
            0xe8, 0x6f, 0xf9, 0x22, 0xb4, 0xbb, 0xe5, 0xff,
            0xd0, 0x92, 0x86, 0x9a, 0x88, 0xa2, 0xaf, 0x6b
        };
        ret = sizeof(signature_2048);
        XMEMCPY(out, signature_2048, ret);
    }
#endif

#if !defined(WC_NO_RNG) && !defined(WC_NO_RSA_OAEP) && \
    ((!defined(WOLFSSL_RSA_VERIFY_ONLY) && !defined(WOLFSSL_RSA_PUBLIC_ONLY)) || \
    defined(WOLFSSL_PUBLIC_MP))  && !defined(WOLF_CRYPTO_CB_ONLY_RSA)
    idx = (word32)ret;
    XMEMSET(plain, 0, plainSz);
    do {
#if defined(WOLFSSL_ASYNC_CRYPT)
        ret = wc_AsyncWait(ret, &key->asyncDev, WC_ASYNC_FLAG_CALL_AGAIN);
#endif
        if (ret >= 0) {
#ifndef WOLFSSL_RSA_VERIFY_INLINE

#if defined(WOLFSSL_CRYPTOCELL)
        /*
        Cryptocell requires the input data and signature byte array to verify.

        first argument must be the input data
        second argument must be the length of input data
        third argument must be the signature byte array or the output from
                                                                wc_RsaSSL_Sign()
        fourth argument must be the length of the signature byte array
        */

        ret = wc_RsaSSL_Verify(in, inLen, out, outSz, key);
#else
        ret = wc_RsaSSL_Verify(out, idx, plain, plainSz, key);
#endif /* WOLFSSL_CRYPTOCELL */
#else
            byte* dec = NULL;
            ret = wc_RsaSSL_VerifyInline(out, idx, &dec, key);
            if (ret > 0) {
                XMEMCPY(plain, dec, ret);
            }
#endif
        }
    } while (ret == WC_PENDING_E);
    if (ret < 0) {
        ERROR_OUT(-7916, exit_rsa);
    }

    if (XMEMCMP(plain, in, (size_t)ret)) {
        ERROR_OUT(-7917, exit_rsa);
    }
    TEST_SLEEP();
#endif

#ifndef WOLFSSL_RSA_VERIFY_ONLY
    #if !defined(WC_NO_RSA_OAEP) && !defined(WC_NO_RNG)
    #if !defined(HAVE_FAST_RSA) && !defined(HAVE_USER_RSA) && \
        (!defined(HAVE_FIPS) || \
         (defined(HAVE_FIPS_VERSION) && (HAVE_FIPS_VERSION >= 2))) \
         && !defined(WOLF_CRYPTO_CB_ONLY_RSA)
    ret = rsa_oaep_padding_test(key, &rng);
    if (ret != 0)
        return ret;

    #endif /* !HAVE_FAST_RSA && !HAVE_FIPS */
    #endif /* WC_NO_RSA_OAEP && !WC_NO_RNG */
#endif /* WOLFSSL_RSA_VERIFY_ONLY */

#if !defined(HAVE_FIPS) && !defined(HAVE_USER_RSA) && !defined(NO_ASN) \
    && !defined(WOLFSSL_RSA_VERIFY_ONLY)
    ret = rsa_export_key_test(key);
    if (ret != 0)
        return ret;
#endif

#if !defined(NO_ASN) && !defined(WOLFSSL_RSA_PUBLIC_ONLY) && \
                                               !defined(WOLFSSL_RSA_VERIFY_ONLY)
    ret = rsa_flatten_test(key);
    if (ret != 0)
        return ret;
#endif

#if !defined(NO_FILESYSTEM) && !defined(NO_RSA) && !defined(NO_ASN) && \
    !defined(USE_CERT_BUFFERS_1024) && !defined(USE_CERT_BUFFERS_2048) && \
    !defined(USE_CERT_BUFFERS_3072) && !defined(USE_CERT_BUFFERS_4096)
    (void)clientCert;
#endif

#ifdef WOLFSSL_TEST_CERT
#if defined(WOLFSSL_MDK_ARM)
    #define sizeof(s) XSTRLEN((char *)(s))
#endif

#ifdef USE_CERT_BUFFERS_1024
    XMEMCPY(tmp, client_cert_der_1024, (size_t)sizeof_client_cert_der_1024);
    bytes = (size_t)sizeof_client_cert_der_1024;
#elif defined(USE_CERT_BUFFERS_2048)
    XMEMCPY(tmp, client_cert_der_2048, (size_t)sizeof_client_cert_der_2048);
    bytes = (size_t)sizeof_client_cert_der_2048;
#elif defined(USE_CERT_BUFFERS_3072)
    XMEMCPY(tmp, client_cert_der_3072, (size_t)sizeof_client_cert_der_3072);
    bytes = (size_t)sizeof_client_cert_der_3072;
#elif defined(USE_CERT_BUFFERS_4096)
    XMEMCPY(tmp, client_cert_der_4096, (size_t)sizeof_client_cert_der_4096);
    bytes = (size_t)sizeof_client_cert_der_4096;
#elif !defined(NO_FILESYSTEM)
    file2 = XFOPEN(clientCert, "rb");
    if (!file2) {
        ERROR_OUT(-7940, exit_rsa);
    }

    bytes = XFREAD(tmp, 1, FOURK_BUF, file2);
    XFCLOSE(file2);
#else
    /* No certificate to use. */
    ERROR_OUT(-7941, exit_rsa);
#endif

#ifdef sizeof
    #undef sizeof
#endif
    InitDecodedCert(cert, tmp, (word32)bytes, NULL);

    ret = ParseCert(cert, CERT_TYPE, NO_VERIFY, NULL);
    if (ret != 0) {
        FreeDecodedCert(cert);
        ERROR_OUT(-7942, exit_rsa);
    }

#ifndef NO_ASN_TIME
    ret = wc_GetDateInfo(cert->afterDate, cert->afterDateLen, &date,
                         &dateFormat, &dateLength);
    if (ret != 0) {
        FreeDecodedCert(cert);
        ERROR_OUT(-7943, exit_rsa);
    }
    ret = wc_GetDateAsCalendarTime(date, dateLength, dateFormat, &timearg);
    if (ret != 0) {
        FreeDecodedCert(cert);
        ERROR_OUT(-7944, exit_rsa);
    }
#endif

    FreeDecodedCert(cert);
#endif /* WOLFSSL_TEST_CERT */

#ifdef WOLFSSL_CERT_EXT

#ifdef USE_CERT_BUFFERS_1024
    XMEMCPY(tmp, client_keypub_der_1024, sizeof_client_keypub_der_1024);
    bytes = sizeof_client_keypub_der_1024;
#elif defined(USE_CERT_BUFFERS_2048)
    XMEMCPY(tmp, client_keypub_der_2048, sizeof_client_keypub_der_2048);
    bytes = sizeof_client_keypub_der_2048;
#elif defined(USE_CERT_BUFFERS_3072)
    XMEMCPY(tmp, client_keypub_der_3072, sizeof_client_keypub_der_3072);
    bytes = sizeof_client_keypub_der_3072;
#elif defined(USE_CERT_BUFFERS_4096)
    XMEMCPY(tmp, client_keypub_der_4096, sizeof_client_keypub_der_4096);
    bytes = sizeof_client_keypub_der_4096;
#else
    file = XFOPEN(clientKeyPub, "rb");
    if (!file) {
        err_sys("can't open ./certs/client-keyPub.der, "
                "Please run from wolfSSL home dir", -40);
        ERROR_OUT(-7945, exit_rsa);
    }

    bytes = XFREAD(tmp, 1, FOURK_BUF, file);
    XFCLOSE(file);
#endif /* USE_CERT_BUFFERS */

    ret = wc_InitRsaKey(keypub, HEAP_HINT);
    if (ret != 0) {
        ERROR_OUT(-7946, exit_rsa);
    }
    idx = 0;

    ret = wc_RsaPublicKeyDecode(tmp, &idx, keypub, (word32)bytes);
    if (ret != 0) {
        ERROR_OUT(-7947, exit_rsa);
    }
#endif /* WOLFSSL_CERT_EXT */

#ifdef WOLFSSL_KEY_GEN
    ret = rsa_keygen_test(&rng);
    if (ret != 0)
        goto exit_rsa;
#endif

#ifdef WOLFSSL_CERT_GEN
    /* Make Cert / Sign example for RSA cert and RSA CA */
    ret = rsa_certgen_test(key, keypub, &rng, tmp);
    if (ret != 0)
        goto exit_rsa;

#if !defined(NO_RSA) && defined(HAVE_ECC)
    ret = rsa_ecc_certgen_test(&rng, tmp);
    if (ret != 0)
        goto exit_rsa;
#endif

#if defined(WOLFSSL_CERT_REQ) && !defined(WOLFSSL_NO_MALLOC)
    {
        Cert        *req;
        int         derSz;
#ifndef WOLFSSL_SMALL_STACK
        byte*  der = NULL;
#endif

        req = (Cert *)XMALLOC(sizeof *req, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
        if (! req)
            ERROR_OUT(MEMORY_E, exit_rsa);
        der = (byte*)XMALLOC(FOURK_BUF, HEAP_HINT,DYNAMIC_TYPE_TMP_BUFFER);
        if (der == NULL) {
            ERROR_OUT(-7964, exit_rsa);
        }

        if (wc_InitCert_ex(req, HEAP_HINT, devId)) {
            ERROR_OUT(-7965, exit_rsa);
        }

        req->version = 0;
        req->isCA    = 1;
        XSTRNCPY(req->challengePw, "wolf123", CTC_NAME_SIZE);
        XMEMCPY(&req->subject, &certDefaultName, sizeof(CertName));

    #ifndef NO_SHA256
        req->sigType = CTC_SHA256wRSA;
    #else
        req->sigType = CTC_SHAwRSA;
    #endif

    #ifdef WOLFSSL_CERT_EXT
        /* add SKID from the Public Key */
        if (wc_SetSubjectKeyIdFromPublicKey(req, keypub, NULL) != 0) {
            ERROR_OUT(-7966, exit_rsa);
        }

        /* add Key Usage */
        if (wc_SetKeyUsage(req, certKeyUsage2) != 0) {
            ERROR_OUT(-7967, exit_rsa);
        }

        /* add Extended Key Usage */
        if (wc_SetExtKeyUsage(req, "serverAuth,clientAuth,codeSigning,"
                            "emailProtection,timeStamping,OCSPSigning") != 0) {
            ERROR_OUT(-7968, exit_rsa);
        }
    #ifdef WOLFSSL_EKU_OID
        {
            WOLFSSL_SMALL_STACK_STATIC const char unique[] = "2.16.840.1.111111.100.1.10.1";
            if (wc_SetExtKeyUsageOID(req, unique, sizeof(unique), 0,
                        HEAP_HINT) != 0) {
                ERROR_OUT(-7969, exit_rsa);
            }
        }
    #endif /* WOLFSSL_EKU_OID */
    #endif /* WOLFSSL_CERT_EXT */

        derSz = wc_MakeCertReq(req, der, FOURK_BUF, key, NULL);
        if (derSz < 0) {
            ERROR_OUT(-7970, exit_rsa);
        }

    #ifdef WOLFSSL_CERT_EXT
        /* Try again with "any" flag set, will override all others */
        if (wc_SetExtKeyUsage(req, "any") != 0) {
            ERROR_OUT(-7971, exit_rsa);
        }
        derSz = wc_MakeCertReq(req, der, FOURK_BUF, key, NULL);
        if (derSz < 0) {
            ERROR_OUT(-7972, exit_rsa);
        }
    #endif /* WOLFSSL_CERT_EXT */

        ret = 0;
        do {
        #if defined(WOLFSSL_ASYNC_CRYPT)
            ret = wc_AsyncWait(ret, &key->asyncDev, WC_ASYNC_FLAG_CALL_AGAIN);
        #endif
            if (ret >= 0) {
                ret = wc_SignCert(req->bodySz, req->sigType, der, FOURK_BUF,
                          key, NULL, &rng);
            }
        } while (ret == WC_PENDING_E);
        if (ret < 0) {
            ERROR_OUT(-7973, exit_rsa);
        }
        derSz = ret;

        ret = SaveDerAndPem(der, derSz, certReqDerFile, certReqPemFile,
            CERTREQ_TYPE, -5650);
        if (ret != 0) {
            goto exit_rsa;
        }

        derSz = wc_MakeCertReq_ex(req, der, FOURK_BUF, RSA_TYPE, key);
        if (derSz < 0) {
            ERROR_OUT(-7974, exit_rsa);
        }

        /* Test getting the size of the buffer without providing the buffer.
         * derSz is set to the "largest buffer" we are willing to allocate. */
        derSz = wc_MakeCertReq(req, NULL, 10000, key, NULL);
        if (derSz < 0) {
            ERROR_OUT(-7975, exit_rsa);
        }

        XFREE(der, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
        XFREE(req, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
        der = NULL;
    }
#endif /* WOLFSSL_CERT_REQ */
#endif /* WOLFSSL_CERT_GEN */

#if defined(WC_RSA_PSS) && !defined(HAVE_FIPS_VERSION) /* not supported with FIPSv1 */
/* Need to create known good signatures to test with this. */
#if !defined(WOLFSSL_RSA_VERIFY_ONLY) && !defined(WOLFSSL_RSA_PUBLIC_ONLY) && \
    !defined(WOLF_CRYPTO_CB_ONLY_RSA)
    ret = rsa_pss_test(&rng, key);
    if (ret != 0)
        goto exit_rsa;
#endif
#endif

#if defined(WOLFSSL_HAVE_SP_RSA) && defined(USE_FAST_MATH)
#ifdef WOLFSSL_SMALL_STACK
    /* New key to be loaded in rsa_even_mod_test(). */
    if (key != NULL)
#endif
        wc_FreeRsaKey(key);
    /* New key to be loaded in rsa_even_mod_test(). */
    ret = rsa_even_mod_test(&rng, key);
#endif

exit_rsa:

#ifdef WOLFSSL_SMALL_STACK
    if (key != NULL) {
        wc_FreeRsaKey(key);
        XFREE(key, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    }
    #if defined(WOLFSSL_CERT_EXT) || defined(WOLFSSL_CERT_GEN)
    if (keypub != NULL) {
        wc_FreeRsaKey(keypub);
        XFREE(keypub, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    }
    #endif
    #ifdef WOLFSSL_TEST_CERT
    if (cert != NULL)
        XFREE(cert, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    #endif
     XFREE(der, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
     XFREE(tmp, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
#else
    wc_FreeRsaKey(key);
    #if defined(WOLFSSL_CERT_EXT) || defined(WOLFSSL_CERT_GEN)
    wc_FreeRsaKey(keypub);
    #endif
#endif /* WOLFSSL_SMALL_STACK */

    wc_FreeRng(&rng);

    WC_FREE_VAR(in, HEAP_HINT);
    WC_FREE_VAR(out, HEAP_HINT);
    WC_FREE_VAR(plain, HEAP_HINT);

    (void)res;
    (void)bytes;
    (void)idx;
    (void)in;
    (void)out;
    (void)plain;
    (void)idx;
    (void)inStr;
    (void)inLen;
    (void)outSz;
    (void)plainSz;

    /* ret can be greater then 0 with certgen but all negative values should
     * be returned and treated as an error */
    if (ret >= 0) {
        return 0;
    }
    else {
        return ret;
    }
}

#endif /* !NO_RSA */


#ifndef NO_DH

static int dh_fips_generate_test(WC_RNG *rng)
{
    int    ret = 0;
#ifdef WOLFSSL_SMALL_STACK
    DhKey  *key = (DhKey *)XMALLOC(sizeof *key, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
#else
    DhKey  key[1];
#endif
    WOLFSSL_SMALL_STACK_STATIC const byte p[] = {
        0xc5, 0x7c, 0xa2, 0x4f, 0x4b, 0xd6, 0x8c, 0x3c,
        0xda, 0xc7, 0xba, 0xaa, 0xea, 0x2e, 0x5c, 0x1e,
        0x18, 0xb2, 0x7b, 0x8c, 0x55, 0x65, 0x9f, 0xea,
        0xe0, 0xa1, 0x36, 0x53, 0x2b, 0x36, 0xe0, 0x4e,
        0x3e, 0x64, 0xa9, 0xe4, 0xfc, 0x8f, 0x32, 0x62,
        0x97, 0xe4, 0xbe, 0xf7, 0xc1, 0xde, 0x07, 0x5a,
        0x89, 0x28, 0xf3, 0xfe, 0x4f, 0xfe, 0x68, 0xbc,
        0xfb, 0x0a, 0x7c, 0xa4, 0xb3, 0x14, 0x48, 0x89,
        0x9f, 0xaf, 0xb8, 0x43, 0xe2, 0xa0, 0x62, 0x5c,
        0xb4, 0x88, 0x3f, 0x06, 0x50, 0x11, 0xfe, 0x65,
        0x8d, 0x49, 0xd2, 0xf5, 0x4b, 0x74, 0x79, 0xdb,
        0x06, 0x62, 0x92, 0x89, 0xed, 0xda, 0xcb, 0x87,
        0x37, 0x16, 0xd2, 0xa1, 0x7a, 0xe8, 0xde, 0x92,
        0xee, 0x3e, 0x41, 0x4a, 0x91, 0x5e, 0xed, 0xf3,
        0x6c, 0x6b, 0x7e, 0xfd, 0x15, 0x92, 0x18, 0xfc,
        0xa7, 0xac, 0x42, 0x85, 0x57, 0xe9, 0xdc, 0xda,
        0x55, 0xc9, 0x8b, 0x28, 0x9e, 0xc1, 0xc4, 0x46,
        0x4d, 0x88, 0xed, 0x62, 0x8e, 0xdb, 0x3f, 0xb9,
        0xd7, 0xc8, 0xe3, 0xcf, 0xb8, 0x34, 0x2c, 0xd2,
        0x6f, 0x28, 0x06, 0x41, 0xe3, 0x66, 0x8c, 0xfc,
        0x72, 0xff, 0x26, 0x3b, 0x6b, 0x6c, 0x6f, 0x73,
        0xde, 0xf2, 0x90, 0x29, 0xe0, 0x61, 0x32, 0xc4,
        0x12, 0x74, 0x09, 0x52, 0xec, 0xf3, 0x1b, 0xa6,
        0x45, 0x98, 0xac, 0xf9, 0x1c, 0x65, 0x8e, 0x3a,
        0x91, 0x84, 0x4b, 0x23, 0x8a, 0xb2, 0x3c, 0xc9,
        0xfa, 0xea, 0xf1, 0x38, 0xce, 0xd8, 0x05, 0xe0,
        0xfa, 0x44, 0x68, 0x1f, 0xeb, 0xd9, 0x57, 0xb8,
        0x4a, 0x97, 0x5b, 0x88, 0xc5, 0xf1, 0xbb, 0xb0,
        0x49, 0xc3, 0x91, 0x7c, 0xd3, 0x13, 0xb9, 0x47,
        0xbb, 0x91, 0x8f, 0xe5, 0x26, 0x07, 0xab, 0xa9,
        0xc5, 0xd0, 0x3d, 0x95, 0x41, 0x26, 0x92, 0x9d,
        0x13, 0x67, 0xf2, 0x7e, 0x11, 0x88, 0xdc, 0x2d
    };
    WOLFSSL_SMALL_STACK_STATIC const byte g[] = {
        0x4a, 0x1a, 0xf3, 0xa4, 0x92, 0xe9, 0xee, 0x74,
        0x6e, 0x57, 0xd5, 0x8c, 0x2c, 0x5b, 0x41, 0x41,
        0x5e, 0xd4, 0x55, 0x19, 0xdc, 0xd9, 0x32, 0x91,
        0xf7, 0xfd, 0xc2, 0x57, 0xff, 0x03, 0x14, 0xdb,
        0xf1, 0xb7, 0x60, 0x0c, 0x43, 0x59, 0x3f, 0xff,
        0xac, 0xf1, 0x80, 0x9a, 0x15, 0x6f, 0xd8, 0x6e,
        0xb7, 0x85, 0x18, 0xc8, 0xec, 0x4e, 0x59, 0x4a,
        0xe2, 0x91, 0x43, 0x4c, 0xeb, 0x95, 0xb6, 0x2e,
        0x9a, 0xea, 0x53, 0x68, 0x80, 0x64, 0x69, 0x40,
        0xf9, 0xec, 0xbd, 0x85, 0x89, 0x26, 0x97, 0x67,
        0xaf, 0xb0, 0xad, 0x00, 0x1b, 0xd4, 0xfd, 0x94,
        0xd3, 0xe9, 0x92, 0xb1, 0xb4, 0xbc, 0x5a, 0xaa,
        0x92, 0x80, 0x89, 0x3b, 0x39, 0x05, 0x6c, 0x22,
        0x26, 0xfe, 0x5a, 0x28, 0x6c, 0x37, 0x50, 0x5a,
        0x38, 0x99, 0xcf, 0xf3, 0xc1, 0x96, 0x45, 0xdc,
        0x01, 0xcb, 0x20, 0x87, 0xa5, 0x00, 0x8c, 0xf5,
        0x4d, 0xc2, 0xef, 0xb8, 0x9b, 0xd1, 0x87, 0xbe,
        0xed, 0xd5, 0x0a, 0x29, 0x15, 0x34, 0x59, 0x4c,
        0x3a, 0x05, 0x22, 0x05, 0x44, 0x4f, 0x9f, 0xc8,
        0x47, 0x12, 0x24, 0x8e, 0xa8, 0x79, 0xe4, 0x67,
        0xba, 0x4d, 0x5b, 0x75, 0x56, 0x95, 0xeb, 0xe8,
        0x8a, 0xfa, 0x8e, 0x01, 0x8c, 0x1b, 0x74, 0x63,
        0xd9, 0x2f, 0xf7, 0xd3, 0x44, 0x8f, 0xa8, 0xf5,
        0xaf, 0x6c, 0x4f, 0xdb, 0xe7, 0xc9, 0x6c, 0x71,
        0x22, 0xa3, 0x1d, 0xf1, 0x40, 0xb2, 0xe0, 0x9a,
        0xb6, 0x72, 0xc9, 0xc0, 0x13, 0x16, 0xa2, 0x4a,
        0xe1, 0x92, 0xc7, 0x54, 0x23, 0xab, 0x9d, 0xa1,
        0xa1, 0xe5, 0x0b, 0xed, 0xba, 0xe8, 0x84, 0x37,
        0xb2, 0xe7, 0xfe, 0x32, 0x8d, 0xfa, 0x1c, 0x53,
        0x77, 0x97, 0xc7, 0xf3, 0x48, 0xc9, 0xdb, 0x2d,
        0x75, 0x52, 0x9d, 0x42, 0x51, 0x78, 0x62, 0x68,
        0x05, 0x45, 0x15, 0xf8, 0xa2, 0x4e, 0xf3, 0x0b
    };
    WOLFSSL_SMALL_STACK_STATIC const byte q[] = {
        0xe0, 0x35, 0x37, 0xaf, 0xb2, 0x50, 0x91, 0x8e,
        0xf2, 0x62, 0x2b, 0xd9, 0x9f, 0x6c, 0x11, 0x75,
        0xec, 0x24, 0x5d, 0x78, 0x59, 0xe7, 0x8d, 0xb5,
        0x40, 0x52, 0xed, 0x41
    };
    WOLFSSL_SMALL_STACK_STATIC const byte q0[] = {
                                                  0x00,
        0xe0, 0x35, 0x37, 0xaf, 0xb2, 0x50, 0x91, 0x8e,
        0xf2, 0x62, 0x2b, 0xd9, 0x9f, 0x6c, 0x11, 0x75,
        0xec, 0x24, 0x5d, 0x78, 0x59, 0xe7, 0x8d, 0xb5,
        0x40, 0x52, 0xed, 0x41
    };
    byte   priv[256];
    byte   pub[256];
    word32 privSz = sizeof(priv);
    word32 pubSz = sizeof(pub);

#ifdef WOLFSSL_SMALL_STACK
    if (key == NULL)
        ERROR_OUT(MEMORY_E, exit_gen_test);
#endif

    /* Parameter Validation testing. */
    ret = wc_DhGenerateKeyPair(NULL, rng, priv, &privSz, pub, &pubSz);
    if (ret != BAD_FUNC_ARG)
        ERROR_OUT(-7980, exit_gen_test);
    ret = wc_DhGenerateKeyPair(key, NULL, priv, &privSz, pub, &pubSz);
    if (ret != BAD_FUNC_ARG)
        ERROR_OUT(-7981, exit_gen_test);
    ret = wc_DhGenerateKeyPair(key, rng, NULL, &privSz, pub, &pubSz);
    if (ret != BAD_FUNC_ARG)
        ERROR_OUT(-7982, exit_gen_test);
    ret = wc_DhGenerateKeyPair(key, rng, priv, NULL, pub, &pubSz);
    if (ret != BAD_FUNC_ARG)
        ERROR_OUT(-7983, exit_gen_test);
    ret = wc_DhGenerateKeyPair(key, rng, priv, &privSz, NULL, &pubSz);
    if (ret != BAD_FUNC_ARG)
        ERROR_OUT(-7984, exit_gen_test);
    ret = wc_DhGenerateKeyPair(key, rng, priv, &privSz, pub, NULL);
    if (ret != BAD_FUNC_ARG)
        ERROR_OUT(-7985, exit_gen_test);

    ret = wc_InitDhKey_ex(key, HEAP_HINT, devId);
    if (ret != 0)
        ERROR_OUT(-7986, exit_gen_test);

    ret = wc_DhSetKey_ex(key, p, sizeof(p), g, sizeof(g), q0, sizeof(q0));
    if (ret != 0) {
        ERROR_OUT(-7987, exit_gen_test);
    }

    wc_FreeDhKey(key);

    ret = wc_InitDhKey_ex(key, HEAP_HINT, devId);
    if (ret != 0)
        ERROR_OUT(-7988, exit_gen_test);

    ret = wc_DhSetKey_ex(key, p, sizeof(p), g, sizeof(g), q, sizeof(q));
    if (ret != 0) {
        ERROR_OUT(-7989, exit_gen_test);
    }

    /* Use API. */
    ret = wc_DhGenerateKeyPair(key, rng, priv, &privSz, pub, &pubSz);
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &key->asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    if (ret != 0) {
        ERROR_OUT(-7990, exit_gen_test);
    }

    ret = wc_DhCheckPubKey_ex(key, pub, pubSz, q0, sizeof(q0));
    if (ret != 0) {
        ERROR_OUT(-7991, exit_gen_test);
    }

    wc_FreeDhKey(key);
    ret = wc_InitDhKey_ex(key, HEAP_HINT, devId);
    if (ret != 0)
        ERROR_OUT(-7992, exit_gen_test);

    ret = wc_DhSetKey(key, p, sizeof(p), g, sizeof(g));
    if (ret != 0) {
        ERROR_OUT(-7993, exit_gen_test);
    }

    ret = wc_DhCheckPubKey_ex(key, pub, pubSz, q, sizeof(q));
    if (ret != 0) {
        ERROR_OUT(-7994, exit_gen_test);
    }

#ifndef HAVE_SELFTEST
    ret = wc_DhCheckKeyPair(key, pub, pubSz, priv, privSz);
    if (ret != 0) {
        ERROR_OUT(-7995, exit_gen_test);
    }

    /* Taint the public key so the check fails. */
    pub[0]++;
    ret = wc_DhCheckKeyPair(key, pub, pubSz, priv, privSz);
    if (ret != MP_CMP_E) {
        ERROR_OUT(-7996, exit_gen_test);
    }

#ifdef WOLFSSL_KEY_GEN
    wc_FreeDhKey(key);
    ret = wc_InitDhKey_ex(key, HEAP_HINT, devId);
    if (ret != 0)
        ERROR_OUT(-7997, exit_gen_test);

    ret = wc_DhGenerateParams(rng, 2048, key);
    if (ret != 0) {
        ERROR_OUT(-7998, exit_gen_test);
    }

    privSz = sizeof(priv);
    pubSz = sizeof(pub);

    ret = wc_DhGenerateKeyPair(key, rng, priv, &privSz, pub, &pubSz);
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &key->asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    if (ret != 0) {
        ERROR_OUT(-7999, exit_gen_test);
    }
#endif /* WOLFSSL_KEY_GEN */
#endif /* HAVE_SELFTEST */

    ret = 0;

exit_gen_test:

#ifdef WOLFSSL_SMALL_STACK
    if (key) {
        wc_FreeDhKey(key);
        XFREE(key, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    }
#else
    wc_FreeDhKey(key);
#endif

    return ret;
}

static int dh_generate_test(WC_RNG *rng)
{
    int    ret = 0;
#ifdef WOLFSSL_SMALL_STACK
    DhKey  *smallKey = NULL;
#else
    DhKey  smallKey[1];
#endif
    byte   p[2] = { 1, 7 }; /* 263 in decimal */
    byte   g[2] = { 0, 2 };
#if !defined(WOLFSSL_SP_MATH) && !defined(HAVE_FFDHE)
#ifdef WOLFSSL_DH_CONST
    /* the table for constant DH lookup will round to the lowest byte size 21 */
    byte   priv[21];
    byte   pub[21];
#else
    byte   priv[2];
    byte   pub[2];
#endif
    word32 privSz = sizeof(priv);
    word32 pubSz = sizeof(pub);
#endif
    int smallKey_inited = 0;

#ifdef WOLFSSL_SMALL_STACK
    if ((smallKey = (DhKey *)XMALLOC(sizeof(*smallKey), HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER)) == NULL)
        return -8019;
#endif

    ret = wc_InitDhKey_ex(smallKey, HEAP_HINT, devId);
    if (ret != 0)
        ERROR_OUT(-8010, exit_gen_test);
    smallKey_inited = 1;

    /* Parameter Validation testing. */
    ret = wc_InitDhKey_ex(NULL, HEAP_HINT, devId);
    if (ret != BAD_FUNC_ARG)
        return -8011;
    wc_FreeDhKey(NULL);

    ret = wc_DhSetKey(NULL, p, sizeof(p), g, sizeof(g));
    if (ret != BAD_FUNC_ARG) {
        ERROR_OUT(-8012, exit_gen_test);
    }
    ret = wc_DhSetKey(smallKey, NULL, sizeof(p), g, sizeof(g));
    if (ret != BAD_FUNC_ARG) {
        ERROR_OUT(-8013, exit_gen_test);
    }
    ret = wc_DhSetKey(smallKey, p, 0, g, sizeof(g));
    if (ret != BAD_FUNC_ARG) {
        ERROR_OUT(-8014, exit_gen_test);
    }
    ret = wc_DhSetKey(smallKey, p, sizeof(p), NULL, sizeof(g));
    if (ret != BAD_FUNC_ARG) {
        ERROR_OUT(-8015, exit_gen_test);
    }
    ret = wc_DhSetKey(smallKey, p, sizeof(p), g, 0);
    if (ret != BAD_FUNC_ARG) {
        ERROR_OUT(-8016, exit_gen_test);
    }
    ret = wc_DhSetKey(smallKey, p, sizeof(p), g, sizeof(g));
    if (ret != 0) {
        ERROR_OUT(-8017, exit_gen_test);
    }

#if !defined(WOLFSSL_SP_MATH) && !defined(HAVE_FFDHE)
    /* Use API. */
    ret = wc_DhGenerateKeyPair(smallKey, rng, priv, &privSz, pub, &pubSz);
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &smallKey->asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    if (ret != 0) {
        ret = -8018;
    }
#else
    (void)rng;
    #if defined(HAVE_FIPS) || !defined(WOLFSSL_NO_DH186)
    ret = 0;
    #endif
#endif

#if !defined(HAVE_FIPS) && defined(WOLFSSL_NO_DH186)
    {
        byte   priv[260];
        byte   pub[260];
        word32 privSz = sizeof(priv);
        word32 pubSz = sizeof(pub);

        /* test odd ball param generation with DH */
        wc_FreeDhKey(smallKey);
        ret = wc_InitDhKey_ex(smallKey, HEAP_HINT, devId);
        if (ret != 0)
            ERROR_OUT(-8019, exit_gen_test);

        ret = wc_DhGenerateParams(rng, 2056, smallKey);
        if (ret != 0) {
            ERROR_OUT(-8020, exit_gen_test);
        }

        privSz = sizeof(priv);
        pubSz = sizeof(pub);

        ret = wc_DhGenerateKeyPair(smallKey, rng, priv, &privSz, pub, &pubSz);
    #if defined(WOLFSSL_ASYNC_CRYPT)
        ret = wc_AsyncWait(ret, &smallKey->asyncDev, WC_ASYNC_FLAG_NONE);
    #endif
        if (ret != 0) {
            ERROR_OUT(-8021, exit_gen_test);
        }
    }
#endif /* !HAVE_FIPS and WOLFSSL_NO_DH186 */

exit_gen_test:
    if (smallKey_inited)
        wc_FreeDhKey(smallKey);

#ifdef WOLFSSL_SMALL_STACK
    if (smallKey != NULL)
        XFREE(smallKey, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
#endif

    return ret;
}

#if !defined(HAVE_FIPS) && !defined(HAVE_SELFTEST)
typedef struct dh_pubvalue_test {
    const byte* data;
    word32 len;
} dh_pubvalue_test;

static int dh_test_check_pubvalue(void)
{
    int ret;
    word32 i;
    WOLFSSL_SMALL_STACK_STATIC const byte prime[] = {0x01, 0x00, 0x01};
    WOLFSSL_SMALL_STACK_STATIC const byte pubValZero[] = { 0x00 };
    WOLFSSL_SMALL_STACK_STATIC const byte pubValZeroLong[] = { 0x00, 0x00, 0x00 };
    WOLFSSL_SMALL_STACK_STATIC const byte pubValOne[] = { 0x01 };
    WOLFSSL_SMALL_STACK_STATIC const byte pubValOneLong[] = { 0x00, 0x00, 0x01 };
    WOLFSSL_SMALL_STACK_STATIC const byte pubValPrimeMinusOne[] = { 0x01, 0x00, 0x00 };
    WOLFSSL_SMALL_STACK_STATIC const byte pubValPrimeLong[] = {0x00, 0x01, 0x00, 0x01};
    WOLFSSL_SMALL_STACK_STATIC const byte pubValPrimePlusOne[] = { 0x01, 0x00, 0x02 };
    WOLFSSL_SMALL_STACK_STATIC const byte pubValTooBig0[] = { 0x02, 0x00, 0x01 };
    WOLFSSL_SMALL_STACK_STATIC const byte pubValTooBig1[] = { 0x01, 0x01, 0x01 };
    WOLFSSL_SMALL_STACK_STATIC const byte pubValTooLong[] = { 0x01, 0x00, 0x00, 0x01 };
    const dh_pubvalue_test dh_pubval_fail[] = {
        { prime, sizeof(prime) },
        { pubValZero, sizeof(pubValZero) },
        { pubValZeroLong, sizeof(pubValZeroLong) },
        { pubValOne, sizeof(pubValOne) },
        { pubValOneLong, sizeof(pubValOneLong) },
        { pubValPrimeMinusOne, sizeof(pubValPrimeMinusOne) },
        { pubValPrimeLong, sizeof(pubValPrimeLong) },
        { pubValPrimePlusOne, sizeof(pubValPrimePlusOne) },
        { pubValTooBig0, sizeof(pubValTooBig0) },
        { pubValTooBig1, sizeof(pubValTooBig1) },
        { pubValTooLong, sizeof(pubValTooLong) },
    };
    WOLFSSL_SMALL_STACK_STATIC const byte pubValTwo[] = { 0x02 };
    WOLFSSL_SMALL_STACK_STATIC const byte pubValTwoLong[] = { 0x00, 0x00, 0x02 };
    WOLFSSL_SMALL_STACK_STATIC const byte pubValGood[] = { 0x12, 0x34 };
    WOLFSSL_SMALL_STACK_STATIC const byte pubValGoodLen[] = { 0x00, 0x12, 0x34 };
    WOLFSSL_SMALL_STACK_STATIC const byte pubValGoodLong[] = { 0x00, 0x00, 0x12, 0x34 };
    const dh_pubvalue_test dh_pubval_pass[] = {
        { pubValTwo, sizeof(pubValTwo) },
        { pubValTwoLong, sizeof(pubValTwoLong) },
        { pubValGood, sizeof(pubValGood) },
        { pubValGoodLen, sizeof(pubValGoodLen) },
        { pubValGoodLong, sizeof(pubValGoodLong) },
    };

    for (i = 0; i < sizeof(dh_pubval_fail) / sizeof(*dh_pubval_fail); i++) {
        ret = wc_DhCheckPubValue(prime, sizeof(prime), dh_pubval_fail[i].data,
                                                         dh_pubval_fail[i].len);
        if (ret != MP_VAL)
            return -8020 - (int)i;
    }

    for (i = 0; i < sizeof(dh_pubval_pass) / sizeof(*dh_pubval_pass); i++) {
        ret = wc_DhCheckPubValue(prime, sizeof(prime), dh_pubval_pass[i].data,
                                                         dh_pubval_pass[i].len);
        if (ret != 0)
            return -8030 - (int)i;
    }

    return 0;
}
#endif

#if defined(HAVE_FFDHE)

#if defined(HAVE_FFDHE_4096)
    #define MAX_DH_PRIV_SZ 39
    #define MAX_DH_KEY_SZ  512
#elif defined(HAVE_FFDHE_3072)
    #define MAX_DH_PRIV_SZ 34
    #define MAX_DH_KEY_SZ  384
#else
    #define MAX_DH_PRIV_SZ 29
    #define MAX_DH_KEY_SZ  256
#endif

#ifndef WC_NO_RNG

#if !(defined(HAVE_FIPS_VERSION) && (HAVE_FIPS_VERSION == 2) && \
      (defined(WOLFSSL_SP_ARM64_ASM) || defined(WOLFSSL_SP_ARM32_ASM)))

#ifdef HAVE_PUBLIC_FFDHE
static int dh_ffdhe_test(WC_RNG *rng, const DhParams* params)
#else
static int dh_ffdhe_test(WC_RNG *rng, int name)
#endif
{
    int    ret;
    word32 privSz, pubSz, privSz2, pubSz2;
#ifdef WOLFSSL_SMALL_STACK
    byte   *priv = (byte*)XMALLOC(MAX_DH_PRIV_SZ, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    byte   *pub = (byte*)XMALLOC(MAX_DH_KEY_SZ, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    byte   *priv2 = (byte*)XMALLOC(MAX_DH_PRIV_SZ, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    byte   *pub2 = (byte*)XMALLOC(MAX_DH_KEY_SZ, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    byte   *agree = (byte*)XMALLOC(MAX_DH_KEY_SZ, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    byte   *agree2 = (byte*)XMALLOC(MAX_DH_KEY_SZ, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    DhKey  *key = (DhKey*)XMALLOC(sizeof(*key), HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    DhKey  *key2 = (DhKey*)XMALLOC(sizeof(*key2), HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
#else
    byte   priv[MAX_DH_PRIV_SZ];
    byte   pub[MAX_DH_KEY_SZ];
    byte   priv2[MAX_DH_PRIV_SZ];
    byte   pub2[MAX_DH_KEY_SZ];
    byte   agree[MAX_DH_KEY_SZ];
    byte   agree2[MAX_DH_KEY_SZ];
    DhKey  key[1];
    DhKey  key2[1];
#endif
    word32 agreeSz = MAX_DH_KEY_SZ;
    word32 agreeSz2 = MAX_DH_KEY_SZ;

#ifdef WOLFSSL_SMALL_STACK
    if ((priv == NULL) ||
        (pub == NULL) ||
        (priv2 == NULL) ||
        (pub2 == NULL) ||
        (agree == NULL) ||
        (agree2 == NULL) ||
        (key == NULL) ||
        (key2 == NULL))
        ERROR_OUT(-8050, done);
#endif

    pubSz = MAX_DH_KEY_SZ;
    pubSz2 = MAX_DH_KEY_SZ;
    #ifdef HAVE_PUBLIC_FFDHE
    privSz = MAX_DH_PRIV_SZ;
    privSz2 = MAX_DH_PRIV_SZ;
    #else
    privSz = wc_DhGetNamedKeyMinSize(name);
    privSz2 = privSz;
    #endif

    XMEMSET(key, 0, sizeof(*key));
    XMEMSET(key2, 0, sizeof(*key2));

    ret = wc_InitDhKey_ex(key, HEAP_HINT, devId);
    if (ret != 0) {
        ERROR_OUT(-8051, done);
    }
    ret = wc_InitDhKey_ex(key2, HEAP_HINT, devId);
    if (ret != 0) {
        ERROR_OUT(-8052, done);
    }

#ifdef HAVE_PUBLIC_FFDHE
    ret = wc_DhSetKey(key, params->p, params->p_len, params->g, params->g_len);
#else
    ret = wc_DhSetNamedKey(key, name);
#endif
    if (ret != 0) {
        ERROR_OUT(-8053, done);
    }

#ifdef HAVE_PUBLIC_FFDHE
    ret = wc_DhSetKey(key2, params->p, params->p_len, params->g,
                                                                 params->g_len);
#else
    ret = wc_DhSetNamedKey(key2, name);
#endif
    if (ret != 0) {
        ERROR_OUT(-8054, done);
    }

    ret = wc_DhGenerateKeyPair(key, rng, priv, &privSz, pub, &pubSz);
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &key->asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    if (ret != 0) {
        ERROR_OUT(-8055, done);
    }

    ret = wc_DhGenerateKeyPair(key2, rng, priv2, &privSz2, pub2, &pubSz2);
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &key2->asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    if (ret != 0) {
        ERROR_OUT(-8056, done);
    }

    ret = wc_DhAgree(key, agree, &agreeSz, priv, privSz, pub2, pubSz2);
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &key->asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    if (ret != 0) {
        ERROR_OUT(-8057, done);
    }

    ret = wc_DhAgree(key2, agree2, &agreeSz2, priv2, privSz2, pub, pubSz);
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &key2->asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    if (ret != 0) {
        ERROR_OUT(-8058, done);
    }

    if (agreeSz != agreeSz2 || XMEMCMP(agree, agree2, agreeSz)) {
        ERROR_OUT(-8059, done);
    }

#if defined(WOLFSSL_HAVE_SP_DH) || defined(USE_FAST_MATH)
    /* Make p even */
    key->p.dp[0] &= (mp_digit)-2;
    if (ret != 0) {
        ERROR_OUT(-8058, done);
    }

    ret = wc_DhGenerateKeyPair(key, rng, priv, &privSz, pub, &pubSz);
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &key->asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    if (ret != MP_VAL && ret != MP_EXPTMOD_E) {
        ERROR_OUT(-8058, done);
    }

    ret = wc_DhAgree(key, agree, &agreeSz, priv, privSz, pub2, pubSz2);
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &key->asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    if (ret != MP_VAL && ret != MP_EXPTMOD_E && ret != ASYNC_OP_E) {
        ERROR_OUT(-8057, done);
    }

    ret = wc_DhCheckKeyPair(key, pub, pubSz, priv, privSz);
    if (ret != MP_VAL && ret != MP_EXPTMOD_E && ret != MP_CMP_E &&
            ret != ASYNC_OP_E) {
        ERROR_OUT(-8057, done);
    }

    /* Getting here means success - set ret to 0. */
    ret = 0;
#endif

done:

#if defined(WOLFSSL_SMALL_STACK) && !defined(WC_NO_RNG)
    if (priv)
        XFREE(priv, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    if (pub)
        XFREE(pub, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    if (priv2)
        XFREE(priv2, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    if (pub2)
        XFREE(pub2, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    if (agree)
        XFREE(agree, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    if (agree2)
        XFREE(agree2, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    if (key) {
        wc_FreeDhKey(key);
        XFREE(key, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    }
    if (key2) {
        wc_FreeDhKey(key2);
        XFREE(key2, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    }
#else
    wc_FreeDhKey(key);
    wc_FreeDhKey(key2);
#endif

    return ret;
}
#endif /* !(HAVE_FIPS_VERSION == 2 && WOLFSSL_SP_ARMxx_ASM) */
#endif /* !WC_NO_RNG */
#endif /* HAVE_FFDHE */

WOLFSSL_TEST_SUBROUTINE int dh_test(void)
{
    int    ret;
    word32 bytes;
    word32 idx = 0, privSz, pubSz, privSz2, pubSz2;
#ifndef WC_NO_RNG
    WC_RNG rng;
#endif
    int keyInit = 0;

#define DH_TEST_TMP_SIZE 1024
#if !defined(USE_CERT_BUFFERS_3072) && !defined(USE_CERT_BUFFERS_4096)
    #define DH_TEST_BUF_SIZE 256
#else
    #define DH_TEST_BUF_SIZE 512
#endif
#ifndef WC_NO_RNG
    word32 agreeSz = DH_TEST_BUF_SIZE;
    word32 agreeSz2 = DH_TEST_BUF_SIZE;
#endif

#ifdef WOLFSSL_SMALL_STACK
    DhKey *key = (DhKey *)XMALLOC(sizeof *key, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    DhKey *key2 = (DhKey *)XMALLOC(sizeof *key2, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    byte  *tmp = (byte *)XMALLOC(DH_TEST_TMP_SIZE, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
#else
    DhKey key[1];
    DhKey key2[1];
    byte  tmp[DH_TEST_TMP_SIZE];
#endif

#ifndef WC_NO_RNG
#ifdef WOLFSSL_SMALL_STACK
    byte *priv = (byte *)XMALLOC(DH_TEST_BUF_SIZE, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    byte *pub = (byte *)XMALLOC(DH_TEST_BUF_SIZE, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    byte *priv2 = (byte *)XMALLOC(DH_TEST_BUF_SIZE, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    byte *pub2 = (byte *)XMALLOC(DH_TEST_BUF_SIZE, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    byte *agree = (byte *)XMALLOC(DH_TEST_BUF_SIZE, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    byte *agree2 = (byte *)XMALLOC(DH_TEST_BUF_SIZE, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    if (priv == NULL || pub == NULL || priv2 == NULL || pub2 == NULL ||
            agree == NULL || agree2 == NULL) {
        ERROR_OUT(-8100, done);
    }
#else
    byte priv[DH_TEST_BUF_SIZE];
    byte pub[DH_TEST_BUF_SIZE];
    byte priv2[DH_TEST_BUF_SIZE];
    byte pub2[DH_TEST_BUF_SIZE];
    byte agree[DH_TEST_BUF_SIZE];
    byte agree2[DH_TEST_BUF_SIZE];
#endif
#endif /* !WC_NO_RNG */

#ifdef WOLFSSL_SMALL_STACK
    if (key == NULL || key2 == NULL || tmp == NULL) {
        ERROR_OUT(-8100, done);
    }
#endif

#ifdef USE_CERT_BUFFERS_1024
    XMEMCPY(tmp, dh_key_der_1024, (size_t)sizeof_dh_key_der_1024);
    bytes = (size_t)sizeof_dh_key_der_1024;
#elif defined(USE_CERT_BUFFERS_2048)
    XMEMCPY(tmp, dh_key_der_2048, (size_t)sizeof_dh_key_der_2048);
    bytes = (size_t)sizeof_dh_key_der_2048;
#elif defined(USE_CERT_BUFFERS_3072)
    XMEMCPY(tmp, dh_key_der_3072, (size_t)sizeof_dh_key_der_3072);
    bytes = (size_t)sizeof_dh_key_der_3072;
#elif defined(USE_CERT_BUFFERS_4096)
    XMEMCPY(tmp, dh_key_der_4096, (size_t)sizeof_dh_key_der_4096);
    bytes = (size_t)sizeof_dh_key_der_4096;
#elif defined(NO_ASN)
    /* don't use file, no DER parsing */
#elif !defined(NO_FILESYSTEM)
    {
        XFILE file = XFOPEN(dhParamsFile, "rb");
        if (! file)
            ERROR_OUT(-8101, done);

        bytes = (word32) XFREAD(tmp, 1, DH_TEST_TMP_SIZE, file);
        XFCLOSE(file);
    }
#else
    /* No DH key to use. */
    ERROR_OUT(-8102, done);
#endif /* USE_CERT_BUFFERS */

    (void)idx;
    (void)tmp;
    (void)bytes;

    pubSz = DH_TEST_BUF_SIZE;
    pubSz2 = DH_TEST_BUF_SIZE;
    privSz = DH_TEST_BUF_SIZE;
    privSz2 = DH_TEST_BUF_SIZE;

#ifndef WC_NO_RNG
    XMEMSET(&rng, 0, sizeof(rng));
#endif

    /* Use API for coverage. */
    ret = wc_InitDhKey(key);
    if (ret != 0) {
        ERROR_OUT(-8103, done);
    }
    wc_FreeDhKey(key);

    ret = wc_InitDhKey_ex(key, HEAP_HINT, devId);
    if (ret != 0) {
        ERROR_OUT(-8104, done);
    }
    keyInit = 1;
    ret = wc_InitDhKey_ex(key2, HEAP_HINT, devId);
    if (ret != 0) {
        ERROR_OUT(-8105, done);
    }

#ifdef NO_ASN
#ifndef WOLFSSL_SP_MATH
    ret = wc_DhSetKey(key, dh_p, sizeof(dh_p), dh_g, sizeof(dh_g));
    if (ret != 0) {
        ERROR_OUT(-8106, done);
    }

    ret = wc_DhSetKey(key2, dh_p, sizeof(dh_p), dh_g, sizeof(dh_g));
    if (ret != 0) {
        ERROR_OUT(-8107, done);
    }
#else
    ret = wc_DhSetKey(key, dh2048_p, sizeof(dh2048_p), dh2048_g,
        sizeof(dh2048_g));
    if (ret != 0) {
        ERROR_OUT(-8106, done);
    }

    ret = wc_DhSetKey(key2, dh2048_p, sizeof(dh2048_p), dh2048_g,
        sizeof(dh2048_g));
    if (ret != 0) {
        ERROR_OUT(-8107, done);
    }
#endif
#else
    ret = wc_DhKeyDecode(tmp, &idx, key, bytes);
    if (ret != 0) {
        ERROR_OUT(-8108, done);
    }

    idx = 0;
    ret = wc_DhKeyDecode(tmp, &idx, key2, bytes);
    if (ret != 0) {
        ERROR_OUT(-8109, done);
    }
#endif

#ifndef WC_NO_RNG
#ifndef HAVE_FIPS
    ret = wc_InitRng_ex(&rng, HEAP_HINT, devId);
#else
    ret = wc_InitRng(&rng);
#endif
    if (ret != 0) {
        ERROR_OUT(-8110, done);
    }

    ret = wc_DhGenerateKeyPair(key, &rng, priv, &privSz, pub, &pubSz);
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &key->asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    if (ret != 0) {
        ERROR_OUT(-8111, done);
    }

    ret = wc_DhGenerateKeyPair(key2, &rng, priv2, &privSz2, pub2, &pubSz2);
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &key2->asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    if (ret != 0) {
        ERROR_OUT(-8112, done);
    }

    ret = wc_DhAgree(key, agree, &agreeSz, priv, privSz, pub2, pubSz2);
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &key->asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    if (ret != 0) {
        ERROR_OUT(-8113, done);
    }

    ret = wc_DhAgree(key2, agree2, &agreeSz2, priv2, privSz2, pub, pubSz);
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &key2->asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    if (ret != 0) {
        ERROR_OUT(-8114, done);
    }

    if (agreeSz != agreeSz2 || XMEMCMP(agree, agree2, agreeSz)) {
        ERROR_OUT(-8115, done);
    }
#endif /* !WC_NO_RNG */

#if defined(WOLFSSL_KEY_GEN) && !defined(HAVE_FIPS) && !defined(HAVE_SELFTEST)
    if (wc_DhCheckPrivKey(NULL, NULL, 0) != BAD_FUNC_ARG)
        ERROR_OUT(-8116, done);

    if (wc_DhCheckPrivKey(key, priv, privSz) != 0)
        ERROR_OUT(-8117, done);

    if (wc_DhExportParamsRaw(NULL, NULL, NULL, NULL, NULL, NULL, NULL) != BAD_FUNC_ARG)
        ERROR_OUT(-8118, done);
    {
        word32 pSz, qSz, gSz;
        if (wc_DhExportParamsRaw(key, NULL, &pSz, NULL, &qSz, NULL, &gSz) != LENGTH_ONLY_E)
            ERROR_OUT(-8119, done);
    }
#endif

    /* Test DH key import / export */
#if defined(WOLFSSL_DH_EXTRA) && !defined(NO_FILESYSTEM) && \
        (!defined(HAVE_FIPS) || \
        (defined(HAVE_FIPS_VERSION) && (HAVE_FIPS_VERSION > 2)))
    wc_FreeDhKey(key);
    ret = wc_InitDhKey_ex(key, HEAP_HINT, devId);
    if (ret != 0) {
        ERROR_OUT(-8120, done);
    }

#ifndef NO_ASN
    {
        /* DH Private - Key Export / Import */
    #ifdef WOLFSSL_SMALL_STACK
        byte *tmp2;
    #else
        byte  tmp2[DH_TEST_TMP_SIZE];
    #endif
        XFILE file = XFOPEN(dhKeyFile, "rb");
        if (!file)
            ERROR_OUT(-8130, done);
        bytes = (word32)XFREAD(tmp, 1, DH_TEST_TMP_SIZE, file);
        XFCLOSE(file);

    #ifdef WOLFSSL_SMALL_STACK
        tmp2 = (byte*)XMALLOC(DH_TEST_TMP_SIZE, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
        if (tmp2 == NULL)
            ERROR_OUT(-8131, done);
    #endif
        idx = 0;
        XMEMSET(tmp2, 0, DH_TEST_TMP_SIZE);

        /* Import DH Private key as DER */
        ret = wc_DhKeyDecode(tmp, &idx, key, bytes);
        if (ret == 0) {
            /* Export as DER */
            idx = DH_TEST_TMP_SIZE;
            ret = wc_DhPrivKeyToDer(key, tmp2, &idx);
        }

        /* Verify export matches original */
        if (ret <= 0 || bytes != idx || XMEMCMP(tmp, tmp2, bytes) != 0) {
            ERROR_OUT(-8132, done);
        }


        /* DH Public Key - Export / Import */
        file = XFOPEN(dhKeyPubFile, "rb");
        if (!file)
            ERROR_OUT(-8133, done);
        bytes = (word32)XFREAD(tmp, 1, DH_TEST_TMP_SIZE, file);
        XFCLOSE(file);

        /* for HAVE_WOLF_BIGINT prevent leak */
        wc_FreeDhKey(key);
        (void)wc_InitDhKey_ex(key, HEAP_HINT, devId);

        idx = 0;
        XMEMSET(tmp2, 0, DH_TEST_TMP_SIZE);

        /* Import DH Public key as DER */
        ret = wc_DhKeyDecode(tmp, &idx, key, bytes);
        if (ret == 0) {
            /* Export as DER */
            idx = DH_TEST_TMP_SIZE;
            ret = wc_DhPubKeyToDer(key, tmp2, &idx);
        }

        /* Verify export matches original */
        if (ret <= 0 || bytes != idx || XMEMCMP(tmp, tmp2, bytes) != 0) {
            ERROR_OUT(-8134, done);
        }

    #ifdef WOLFSSL_SMALL_STACK
        XFREE(tmp2, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    #endif
    }
#else
    ret = wc_DhSetKey(key, dh_p, sizeof(dh_p), dh_g, sizeof(dh_g));
    if (ret != 0) {
        ERROR_OUT(-8121, done);
    }
#endif /* !NO_ASN */

    privSz = DH_TEST_BUF_SIZE;
    pubSz = DH_TEST_BUF_SIZE;
    ret = wc_DhExportKeyPair(key, priv, &privSz, pub, &pubSz);
    if (ret != 0) {
        ERROR_OUT(-8122, done);
    }
    ret = wc_DhImportKeyPair(key2, priv, privSz, pub, pubSz);
    if (ret != 0) {
        ERROR_OUT(-8125, done);
    }
#endif /* WOLFSSL_DH_EXTRA && !NO_FILESYSTEM && !FIPS <= 2 */

#ifndef WC_NO_RNG
    ret = dh_generate_test(&rng);
    if (ret != 0)
        ERROR_OUT(-8123, done);

    ret = dh_fips_generate_test(&rng);
    if (ret != 0)
        ERROR_OUT(-8124, done);
#endif /* !WC_NO_RNG */

#if !defined(HAVE_FIPS) && !defined(HAVE_SELFTEST)
    ret = dh_test_check_pubvalue();
    if (ret != 0)
        ERROR_OUT(-8125, done);
#endif

#if !(defined(HAVE_FIPS_VERSION) && (HAVE_FIPS_VERSION == 2) && \
      (defined(WOLFSSL_SP_ARM64_ASM) || defined(WOLFSSL_SP_ARM32_ASM)))
/* RNG with DH and SP_ASM code not supported in the in-lined FIPS ASM code,
 * this will be available for testing in the 140-3 module */
#ifndef WC_NO_RNG
    /* Specialized code for key gen when using FFDHE-2048, FFDHE-3072 and FFDHE-4096 */
    #ifdef HAVE_FFDHE_2048
    #ifdef HAVE_PUBLIC_FFDHE
    ret = dh_ffdhe_test(&rng, wc_Dh_ffdhe2048_Get());
    #else
    ret = dh_ffdhe_test(&rng, WC_FFDHE_2048);
    #endif
    if (ret != 0)
        ERROR_OUT(-8126, done);
    #endif
    #ifdef HAVE_FFDHE_3072
    #ifdef HAVE_PUBLIC_FFDHE
    ret = dh_ffdhe_test(&rng, wc_Dh_ffdhe3072_Get());
    #else
    ret = dh_ffdhe_test(&rng, WC_FFDHE_3072);
    #endif
    if (ret != 0)
        ERROR_OUT(-8127, done);
    #endif
    #ifdef HAVE_FFDHE_4096
    #ifdef HAVE_PUBLIC_FFDHE
    ret = dh_ffdhe_test(&rng, wc_Dh_ffdhe4096_Get());
    #else
    ret = dh_ffdhe_test(&rng, WC_FFDHE_4096);
    #endif
    if (ret != 0)
        ERROR_OUT(-8128, done);
    #endif
#endif /* !WC_NO_RNG */
#endif /* HAVE_FIPS_VERSION == 2 && !WOLFSSL_SP_ARM64_ASM */

    wc_FreeDhKey(key);
    keyInit = 0;

#if !defined(HAVE_FIPS) && !defined(HAVE_SELFTEST) && \
    !defined(WOLFSSL_OLD_PRIME_CHECK) && !defined(WC_NO_RNG)
    /* Test Check Key */
    ret = wc_DhSetCheckKey(key, dh_p, sizeof(dh_p), dh_g, sizeof(dh_g),
                           NULL, 0, 0, &rng);
    if (ret != 0)
        ERROR_OUT(-8129, done);
    keyInit = 1; /* DhSetCheckKey also initializes the key, free it */
#endif

done:

#ifndef WC_NO_RNG
    wc_FreeRng(&rng);
#endif

#ifdef WOLFSSL_SMALL_STACK
    if (key) {
        if (keyInit)
            wc_FreeDhKey(key);
        XFREE(key, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    }
    if (key2) {
        wc_FreeDhKey(key2);
        XFREE(key2, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    }
    if (tmp)
        XFREE(tmp, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    if (priv)
        XFREE(priv, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    if (pub)
        XFREE(pub, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    if (priv2)
        XFREE(priv2, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    if (pub2)
        XFREE(pub2, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    if (agree)
        XFREE(agree, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    if (agree2)
        XFREE(agree2, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
#else
    if (keyInit)
        wc_FreeDhKey(key);
    wc_FreeDhKey(key2);
#endif

    (void)privSz;
    (void)pubSz;
    (void)pubSz2;
    (void)privSz2;

    return ret;
#undef DH_TEST_BUF_SIZE
#undef DH_TEST_TMP_SIZE
}

#endif /* NO_DH */


#ifndef NO_DSA

WOLFSSL_TEST_SUBROUTINE int dsa_test(void)
{
    int    ret = 0, answer;
    word32 bytes;
    word32 idx = 0;
    WC_RNG rng;
    wc_Sha sha;
    byte   hash[WC_SHA_DIGEST_SIZE];
    byte   signature[40];
#ifdef WOLFSSL_KEY_GEN
    byte*  der = 0;
#endif
#define DSA_TEST_TMP_SIZE 1024
#ifdef WOLFSSL_SMALL_STACK
    byte   *tmp = (byte *)XMALLOC(DSA_TEST_TMP_SIZE, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    DsaKey *key = (DsaKey *)XMALLOC(sizeof *key, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
#ifdef WOLFSSL_KEY_GEN
    DsaKey *derIn = (DsaKey *)XMALLOC(sizeof *derIn, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    DsaKey *genKey = (DsaKey *)XMALLOC(sizeof *genKey, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
#endif
    if ((tmp == NULL) ||
        (key == NULL)
#ifdef WOLFSSL_KEY_GEN
        || (derIn == NULL)
        || (genKey == NULL)
#endif
        ) {
        ret = -8216;
        goto out;
    }
#else
    byte   tmp[1024];
    DsaKey key[1];
#ifdef WOLFSSL_KEY_GEN
    DsaKey derIn[1];
    DsaKey genKey[1];
#endif
#endif

#ifdef USE_CERT_BUFFERS_1024
    XMEMCPY(tmp, dsa_key_der_1024, sizeof_dsa_key_der_1024);
    bytes = sizeof_dsa_key_der_1024;
#elif defined(USE_CERT_BUFFERS_2048)
    XMEMCPY(tmp, dsa_key_der_2048, sizeof_dsa_key_der_2048);
    bytes = sizeof_dsa_key_der_2048;
#else
    {
        XFILE  file = XFOPEN(dsaKey, "rb");
        if (!file)
            ERROR_OUT(-8200, out);

        bytes = (word32) XFREAD(tmp, 1, DSA_TEST_TMP_SIZE, file);
        XFCLOSE(file);
    }
#endif /* USE_CERT_BUFFERS */

    ret = wc_InitSha_ex(&sha, HEAP_HINT, devId);
    if (ret != 0)
        ERROR_OUT(-8201, out);
    wc_ShaUpdate(&sha, tmp, bytes);
    wc_ShaFinal(&sha, hash);
    wc_ShaFree(&sha);

    ret = wc_InitDsaKey(key);
    if (ret != 0)
        ERROR_OUT(-8202, out);

    ret = wc_DsaPrivateKeyDecode(tmp, &idx, key, bytes);
    if (ret != 0)
        ERROR_OUT(-8203, out);

#ifndef HAVE_FIPS
    ret = wc_InitRng_ex(&rng, HEAP_HINT, devId);
#else
    ret = wc_InitRng(&rng);
#endif
    if (ret != 0)
        ERROR_OUT(-8204, out);

    ret = wc_DsaSign(hash, signature, key, &rng);
    if (ret != 0)
        ERROR_OUT(-8205, out);

    ret = wc_DsaVerify(hash, signature, key, &answer);
    if (ret != 0)
        ERROR_OUT(-8206, out);
    if (answer != 1)
        ERROR_OUT(-8207, out);

    wc_FreeDsaKey(key);

#ifdef WOLFSSL_KEY_GEN
    {
    int    derSz = 0;

    ret = wc_InitDsaKey(genKey);
    if (ret != 0)
        ERROR_OUT(-8208, out);

    ret = wc_MakeDsaParameters(&rng, 1024, genKey);
    if (ret != 0) {
        wc_FreeDsaKey(genKey);
        ERROR_OUT(-8209, out);
    }

    ret = wc_MakeDsaKey(&rng, genKey);
    if (ret != 0) {
        wc_FreeDsaKey(genKey);
        ERROR_OUT(-8210, out);
    }

    der = (byte*)XMALLOC(FOURK_BUF, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    if (der == NULL) {
        wc_FreeDsaKey(genKey);
        ERROR_OUT(-8211, out);
    }

    derSz = wc_DsaKeyToDer(genKey, der, FOURK_BUF);
    if (derSz < 0) {
        XFREE(der, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
        ERROR_OUT(-8212, out);
    }

    ret = SaveDerAndPem(der, derSz, keyDerFile, keyPemFile,
        DSA_PRIVATEKEY_TYPE, -5814);
    if (ret != 0) {
        XFREE(der, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
        wc_FreeDsaKey(genKey);
        goto out;
    }

    ret = wc_InitDsaKey(derIn);
    if (ret != 0) {
        XFREE(der, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
        wc_FreeDsaKey(genKey);
        ERROR_OUT(-8213, out);
    }

    idx = 0;
    ret = wc_DsaPrivateKeyDecode(der, &idx, derIn, derSz);
    if (ret != 0) {
        XFREE(der, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
        wc_FreeDsaKey(derIn);
        wc_FreeDsaKey(genKey);
        ERROR_OUT(-8214, out);
    }

    }
#endif /* WOLFSSL_KEY_GEN */

  out:

#ifdef WOLFSSL_SMALL_STACK
    if (key) {
#endif
        if (wc_InitDsaKey_h(key, NULL) != 0)
            ret = -8215;
#ifdef WOLFSSL_SMALL_STACK
    }
#endif

#ifdef WOLFSSL_KEY_GEN
    if (der)
        XFREE(der, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
#endif

#ifdef WOLFSSL_SMALL_STACK
    if (tmp)
        XFREE(tmp, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    if (key)
        XFREE(key, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
#ifdef WOLFSSL_KEY_GEN
    if (derIn) {
        wc_FreeDsaKey(derIn);
        XFREE(derIn, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    }
    if (genKey) {
        wc_FreeDsaKey(genKey);
        XFREE(genKey, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    }
#endif

#else /* !WOLFSSL_SMALL_STACK */

#ifdef WOLFSSL_KEY_GEN
    wc_FreeDsaKey(derIn);
    wc_FreeDsaKey(genKey);
#endif

#endif

    wc_FreeRng(&rng);

    return ret;
}

#endif /* NO_DSA */

#ifdef WOLFCRYPT_HAVE_SRP

static int generate_random_salt(byte *buf, word32 size)
{
    int ret = -8220;
    WC_RNG rng;

    if(NULL == buf || !size)
        return -8221;

    if (buf && size && wc_InitRng_ex(&rng, HEAP_HINT, devId) == 0) {
        ret = wc_RNG_GenerateBlock(&rng, (byte *)buf, size);

        wc_FreeRng(&rng);
    }

    return ret;
}

static int srp_test_digest(SrpType dgstType)
{
    int r;

    byte clientPubKey[192]; /* A */
    byte serverPubKey[192]; /* B */
    word32 clientPubKeySz = 192;
    word32 serverPubKeySz = 192;

    byte username[] = "user";
    word32 usernameSz = 4;

    byte password[] = "password";
    word32 passwordSz = 8;

    WOLFSSL_SMALL_STACK_STATIC const byte N[] = {
        0xfc, 0x58, 0x7a, 0x8a, 0x70, 0xfb, 0x5a, 0x9a,
        0x5d, 0x39, 0x48, 0xbf, 0x1c, 0x46, 0xd8, 0x3b,
        0x7a, 0xe9, 0x1f, 0x85, 0x36, 0x18, 0xc4, 0x35,
        0x3f, 0xf8, 0x8a, 0x8f, 0x8c, 0x10, 0x2e, 0x01,
        0x58, 0x1d, 0x41, 0xcb, 0xc4, 0x47, 0xa8, 0xaf,
        0x9a, 0x6f, 0x58, 0x14, 0xa4, 0x68, 0xf0, 0x9c,
        0xa6, 0xe7, 0xbf, 0x0d, 0xe9, 0x62, 0x0b, 0xd7,
        0x26, 0x46, 0x5b, 0x27, 0xcb, 0x4c, 0xf9, 0x7e,
        0x1e, 0x8b, 0xe6, 0xdd, 0x29, 0xb7, 0xb7, 0x15,
        0x2e, 0xcf, 0x23, 0xa6, 0x4b, 0x97, 0x9f, 0x89,
        0xd4, 0x86, 0xc4, 0x90, 0x63, 0x92, 0xf4, 0x30,
        0x26, 0x69, 0x48, 0x9d, 0x7a, 0x4f, 0xad, 0xb5,
        0x6a, 0x51, 0xad, 0xeb, 0xf9, 0x90, 0x31, 0x77,
        0x53, 0x30, 0x2a, 0x85, 0xf7, 0x11, 0x21, 0x0c,
        0xb8, 0x4b, 0x56, 0x03, 0x5e, 0xbb, 0x25, 0x33,
        0x7c, 0xd9, 0x5a, 0xd1, 0x5c, 0xb2, 0xd4, 0x53,
        0xc5, 0x16, 0x68, 0xf0, 0xdf, 0x48, 0x55, 0x3e,
        0xd4, 0x59, 0x87, 0x64, 0x59, 0xaa, 0x39, 0x01,
        0x45, 0x89, 0x9c, 0x72, 0xff, 0xdd, 0x8f, 0x6d,
        0xa0, 0x42, 0xbc, 0x6f, 0x6e, 0x62, 0x18, 0x2d,
        0x50, 0xe8, 0x18, 0x97, 0x87, 0xfc, 0xef, 0x1f,
        0xf5, 0x53, 0x68, 0xe8, 0x49, 0xd1, 0xa2, 0xe8,
        0xb9, 0x26, 0x03, 0xba, 0xb5, 0x58, 0x6f, 0x6c,
        0x8b, 0x08, 0xa1, 0x7b, 0x6f, 0x42, 0xc9, 0x53
    };

    WOLFSSL_SMALL_STACK_STATIC const byte g[] = {
        0x02
    };

    byte salt[10];

    byte verifier[192];
    word32 v_size = sizeof(verifier);

    word32 clientProofSz = SRP_MAX_DIGEST_SIZE;
    word32 serverProofSz = SRP_MAX_DIGEST_SIZE;
#ifdef WOLFSSL_SMALL_STACK
    Srp *cli = (Srp *)XMALLOC(sizeof *cli, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    Srp *srv = (Srp *)XMALLOC(sizeof *srv, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    byte *clientProof = (byte *)XMALLOC(SRP_MAX_DIGEST_SIZE, HEAP_HINT,
                                        DYNAMIC_TYPE_TMP_BUFFER); /* M1 */
    byte *serverProof = (byte *)XMALLOC(SRP_MAX_DIGEST_SIZE, HEAP_HINT,
                                        DYNAMIC_TYPE_TMP_BUFFER); /* M2 */

    if ((cli == NULL) ||
        (srv == NULL) ||
        (clientProof == NULL) ||
        (serverProof == NULL)) {
        r = -8222;
        goto out;
    }
#else
    Srp cli[1], srv[1];
    byte clientProof[SRP_MAX_DIGEST_SIZE]; /* M1 */
    byte serverProof[SRP_MAX_DIGEST_SIZE]; /* M2 */
#endif


    /* set as 0's so if second init on srv not called SrpTerm is not on
     * garbage values */
    XMEMSET(srv, 0, sizeof *srv);
    XMEMSET(cli, 0, sizeof *cli);

    /* generating random salt */

    r = generate_random_salt(salt, sizeof(salt));

    /* client knows username and password.   */
    /* server knows N, g, salt and verifier. */

    if (!r) r = wc_SrpInit_ex(cli, dgstType, SRP_CLIENT_SIDE, HEAP_HINT, devId);
    if (!r) r = wc_SrpSetUsername(cli, username, usernameSz);

    /* loading N, g and salt in advance to generate the verifier. */

    if (!r) r = wc_SrpSetParams(cli, N,    sizeof(N),
                                      g,    sizeof(g),
                                      salt, sizeof(salt));
    if (!r) r = wc_SrpSetPassword(cli, password, passwordSz);
    if (!r) r = wc_SrpGetVerifier(cli, verifier, &v_size);

    /* client sends username to server */

    if (!r) r = wc_SrpInit_ex(srv, dgstType, SRP_SERVER_SIDE, HEAP_HINT, devId);
    if (!r) r = wc_SrpSetUsername(srv, username, usernameSz);
    if (!r) r = wc_SrpSetParams(srv, N,    sizeof(N),
                                      g,    sizeof(g),
                                      salt, sizeof(salt));
    if (!r) r = wc_SrpSetVerifier(srv, verifier, v_size);
    if (!r) r = wc_SrpGetPublic(srv, serverPubKey, &serverPubKeySz);

    /* server sends N, g, salt and B to client */

    if (!r) r = wc_SrpGetPublic(cli, clientPubKey, &clientPubKeySz);
    if (!r) r = wc_SrpComputeKey(cli, clientPubKey, clientPubKeySz,
                                       serverPubKey, serverPubKeySz);
    if (!r) r = wc_SrpGetProof(cli, clientProof, &clientProofSz);

    /* client sends A and M1 to server */

    if (!r) r = wc_SrpComputeKey(srv, clientPubKey, clientPubKeySz,
                                       serverPubKey, serverPubKeySz);
    if (!r) r = wc_SrpVerifyPeersProof(srv, clientProof, clientProofSz);
    if (!r) r = wc_SrpGetProof(srv, serverProof, &serverProofSz);

    /* server sends M2 to client */

    if (!r) r = wc_SrpVerifyPeersProof(cli, serverProof, serverProofSz);

    wc_SrpTerm(cli);
    wc_SrpTerm(srv);

#ifdef WOLFSSL_SMALL_STACK
  out:

    if (cli)
        XFREE(cli, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    if (srv)
        XFREE(srv, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    if (clientProof)
        XFREE(clientProof, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    if (serverProof)
        XFREE(serverProof, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
#endif

    return r;
}

WOLFSSL_TEST_SUBROUTINE int srp_test(void)
{
    int ret;

#ifndef NO_SHA
    ret = srp_test_digest(SRP_TYPE_SHA);
    if (ret != 0)
        return ret;
#endif
#ifndef NO_SHA256
    ret = srp_test_digest(SRP_TYPE_SHA256);
    if (ret != 0)
        return ret;
#endif
#ifdef WOLFSSL_SHA384
    ret = srp_test_digest(SRP_TYPE_SHA384);
    if (ret != 0)
        return ret;
#endif
#ifdef WOLFSSL_SHA512
    ret = srp_test_digest(SRP_TYPE_SHA512);
    if (ret != 0)
        return ret;
#endif

    return ret;
}

#endif /* WOLFCRYPT_HAVE_SRP */

#if defined(OPENSSL_EXTRA) && !defined(WOLFCRYPT_ONLY)

#if !defined(NO_AES) && !defined(WOLFCRYPT_ONLY)
static int openssl_aes_test(void)
{
#ifdef HAVE_AES_CBC
#ifdef WOLFSSL_AES_128
    {
        /* EVP_CipherUpdate test */
        WOLFSSL_SMALL_STACK_STATIC const byte cbcPlain[] =
        {
            0x6b,0xc1,0xbe,0xe2,0x2e,0x40,0x9f,0x96,
            0xe9,0x3d,0x7e,0x11,0x73,0x93,0x17,0x2a,
            0xae,0x2d,0x8a,0x57,0x1e,0x03,0xac,0x9c,
            0x9e,0xb7,0x6f,0xac,0x45,0xaf,0x8e,0x51,
            0x30,0xc8,0x1c,0x46,0xa3,0x5c,0xe4,0x11,
            0xe5,0xfb,0xc1,0x19,0x1a,0x0a,0x52,0xef,
            0xf6,0x9f,0x24,0x45,0xdf,0x4f,0x9b,0x17,
            0xad,0x2b,0x41,0x7b,0xe6,0x6c,0x37,0x10
        };

        WOLFSSL_SMALL_STACK_STATIC const byte key[] =
            "0123456789abcdef   ";  /* align */
        WOLFSSL_SMALL_STACK_STATIC const byte iv[]  =
            "1234567890abcdef   ";  /* align */

        byte cipher[AES_BLOCK_SIZE * 4];
        byte plain [AES_BLOCK_SIZE * 4];
#ifdef WOLFSSL_SMALL_STACK
        EVP_CIPHER_CTX *en = wolfSSL_EVP_CIPHER_CTX_new();
        EVP_CIPHER_CTX *de = wolfSSL_EVP_CIPHER_CTX_new();
#else
        EVP_CIPHER_CTX en[1];
        EVP_CIPHER_CTX de[1];
#endif
        int outlen ;
        int total = 0;
        int i;

#ifdef WOLFSSL_SMALL_STACK
        if ((en == NULL) || (de == NULL))
            return MEMORY_E;
#endif

        EVP_CIPHER_CTX_init(en);
        if (EVP_CipherInit(en, EVP_aes_128_cbc(),
            (unsigned char*)key, (unsigned char*)iv, 1) == 0)
            return -8400;
        if (EVP_CipherUpdate(en, (byte*)cipher, &outlen,
                    (byte*)cbcPlain, 9) == 0)
            return -8401;
        if (outlen != 0)
            return -8402;
        total += outlen;

        if (EVP_CipherUpdate(en, (byte*)&cipher[total], &outlen,
                    (byte*)&cbcPlain[9]  , 9) == 0)
            return -8403;
        if (outlen != 16)
            return -8404;
        total += outlen;

        if (EVP_CipherFinal(en, (byte*)&cipher[total], &outlen) == 0)
            return -8405;
        if (outlen != 16)
            return -8406;
        total += outlen;
        if (total != 32)
            return 3408;

        total = 0;
        EVP_CIPHER_CTX_init(de);
        if (EVP_CipherInit(de, EVP_aes_128_cbc(),
            (unsigned char*)key, (unsigned char*)iv, 0) == 0)
            return -8407;

        if (EVP_CipherUpdate(de, (byte*)plain, &outlen, (byte*)cipher, 6) == 0)
            return -8408;
        if (outlen != 0)
            return -8409;
        total += outlen;

        if (EVP_CipherUpdate(de, (byte*)&plain[total], &outlen,
                    (byte*)&cipher[6], 12) == 0)
            return -8410;
        if (outlen != 0)
        total += outlen;

        if (EVP_CipherUpdate(de, (byte*)&plain[total], &outlen,
                    (byte*)&cipher[6+12], 14) == 0)
            return -8411;
        if (outlen != 16)
            return -8412;
        total += outlen;

        if (EVP_CipherFinal(de, (byte*)&plain[total], &outlen) == 0)
            return -8413;
        if (outlen != 2)
            return -8414;
        total += outlen;

        if (total != 18)
            return 3427;

        if (XMEMCMP(plain, cbcPlain, 18))
            return -8415;

        /* test with encrypting/decrypting more than 16 bytes at once */
        total = 0;
        EVP_CIPHER_CTX_init(en);
        if (EVP_CipherInit(en, EVP_aes_128_cbc(),
            (unsigned char*)key, (unsigned char*)iv, 1) == 0)
            return -8416;
        if (EVP_CipherUpdate(en, (byte*)cipher, &outlen,
                    (byte*)cbcPlain, 17) == 0)
            return -8417;
        if (outlen != 16)
            return -8418;
        total += outlen;

        if (EVP_CipherUpdate(en, (byte*)&cipher[total], &outlen,
                    (byte*)&cbcPlain[17]  , 1) == 0)
            return -8419;
        if (outlen != 0)
            return -8420;
        total += outlen;

        if (EVP_CipherFinal(en, (byte*)&cipher[total], &outlen) == 0)
            return -8421;
        if (outlen != 16)
            return -8422;
        total += outlen;
        if (total != 32)
            return -8423;

        total = 0;
        EVP_CIPHER_CTX_init(de);
        if (EVP_CipherInit(de, EVP_aes_128_cbc(),
            (unsigned char*)key, (unsigned char*)iv, 0) == 0)
            return -8424;

        if (EVP_CipherUpdate(de, (byte*)plain, &outlen, (byte*)cipher, 17) == 0)
            return -8425;
        if (outlen != 16)
            return -8426;
        total += outlen;

        /* final call on non block size should fail */
        if (EVP_CipherFinal(de, (byte*)&plain[total], &outlen) != 0)
            return -8427;

        if (EVP_CipherUpdate(de, (byte*)&plain[total], &outlen,
                    (byte*)&cipher[17], 1) == 0)
            return -8428;
        if (outlen != 0)
        total += outlen;

        if (EVP_CipherUpdate(de, (byte*)&plain[total], &outlen,
                    (byte*)&cipher[17+1], 14) == 0)
            return -8429;
        if (outlen != 0)
            return -8430;
        total += outlen;

        if (EVP_CipherFinal(de, (byte*)&plain[total], &outlen) == 0)
            return -8431;
        if (outlen != 2)
            return -8432;
        total += outlen;

        if (total != 18)
            return -8433;

        if (XMEMCMP(plain, cbcPlain, 18))
            return -8434;

        /* test byte by byte decrypt */
        for (i = 0; i < AES_BLOCK_SIZE * 3; i++) {
            plain[i] = i;
        }

        total = 0;
        EVP_CIPHER_CTX_init(en);
        if (EVP_CipherInit(en, EVP_aes_128_cbc(),
            (unsigned char*)key, (unsigned char*)iv, 1) == 0)
            return -8435;
        if (EVP_CipherUpdate(en, (byte*)cipher, &outlen,
                    (byte*)plain, AES_BLOCK_SIZE * 3) == 0)
            return -8436;
        if (outlen != AES_BLOCK_SIZE * 3)
            return -8437;
        total += outlen;

        if (EVP_CipherFinal(en, (byte*)&cipher[total], &outlen) == 0)
            return -8438;
        if (outlen != AES_BLOCK_SIZE)
            return -8439;
        total += outlen;
        if (total != sizeof(plain))
            return -8440;

        total = 0;
        EVP_CIPHER_CTX_init(de);
        if (EVP_CipherInit(de, EVP_aes_128_cbc(),
            (unsigned char*)key, (unsigned char*)iv, 0) == 0)
            return -8441;

        for (i = 0; i < AES_BLOCK_SIZE * 4; i++) {
            if (EVP_CipherUpdate(de, (byte*)plain + total, &outlen,
                        (byte*)cipher + i, 1) == 0)
                return -8442;

            if (outlen > 0) {
                int j;

                total += outlen;
                for (j = 0; j < total; j++) {
                    if (plain[j] != j) {
                        return -8443;
                    }
                }
            }
        }

        if (EVP_CipherFinal(de, (byte*)&plain[total], &outlen) == 0)
            return -8444;
        total += outlen;
        if (total != AES_BLOCK_SIZE * 3) {
            return -8445;
        }
        for (i = 0; i < AES_BLOCK_SIZE * 3; i++) {
            if (plain[i] != i) {
                return -8446;
            }
        }

#ifdef WOLFSSL_SMALL_STACK
        wolfSSL_EVP_CIPHER_CTX_free(en);
        wolfSSL_EVP_CIPHER_CTX_free(de);
#endif
    }

    /* set buffers to be exact size to catch potential over read/write */
    {
        /* EVP_CipherUpdate test */
        WOLFSSL_SMALL_STACK_STATIC const byte cbcPlain[] =
        {
            0x6b,0xc1,0xbe,0xe2,0x2e,0x40,0x9f,0x96,
            0xe9,0x3d,0x7e,0x11,0x73,0x93,0x17,0x2a,
            0xae,0x2d,0x8a,0x57,0x1e,0x03,0xac,0x9c,
            0x9e,0xb7,0x6f,0xac,0x45,0xaf,0x8e,0x51,
            0x30,0xc8,0x1c,0x46,0xa3,0x5c,0xe4,0x11,
            0xe5,0xfb,0xc1,0x19,0x1a,0x0a,0x52,0xef,
            0xf6,0x9f,0x24,0x45,0xdf,0x4f,0x9b,0x17,
            0xad,0x2b,0x41,0x7b,0xe6,0x6c,0x37,0x10
        };

        WOLFSSL_SMALL_STACK_STATIC const byte key[] =
            "0123456789abcdef   ";  /* align */
        WOLFSSL_SMALL_STACK_STATIC const byte iv[]  =
            "1234567890abcdef   ";  /* align */

        #define EVP_TEST_BUF_SZ 18
        #define EVP_TEST_BUF_PAD 32
        byte cipher[EVP_TEST_BUF_SZ];
        byte plain [EVP_TEST_BUF_SZ];
        byte padded[EVP_TEST_BUF_PAD];
#ifdef WOLFSSL_SMALL_STACK
        EVP_CIPHER_CTX *en = wolfSSL_EVP_CIPHER_CTX_new();
        EVP_CIPHER_CTX *de = wolfSSL_EVP_CIPHER_CTX_new();
#else
        EVP_CIPHER_CTX en[1];
        EVP_CIPHER_CTX de[1];
#endif
        int outlen ;
        int total = 0;

#ifdef WOLFSSL_SMALL_STACK
        if ((en == NULL) || (de == NULL))
            return MEMORY_E;
#endif

        EVP_CIPHER_CTX_init(en);
        if (EVP_CipherInit(en, EVP_aes_128_cbc(),
            (unsigned char*)key, (unsigned char*)iv, 1) == 0)
            return -8447;
        if (EVP_CIPHER_CTX_set_padding(en, 0) != 1)
            return -8448;
        if (EVP_CipherUpdate(en, (byte*)cipher, &outlen,
                    (byte*)cbcPlain, EVP_TEST_BUF_SZ) == 0)
            return -8449;
        if (outlen != 16)
            return -8450;
        total += outlen;

        /* should fail here */
        if (EVP_CipherFinal(en, (byte*)&cipher[total], &outlen) != 0)
            return -8451;

        /* turn padding back on and do successful encrypt */
        total = 0;
        EVP_CIPHER_CTX_init(en);
        if (EVP_CipherInit(en, EVP_aes_128_cbc(),
            (unsigned char*)key, (unsigned char*)iv, 1) == 0)
            return -8452;
        if (EVP_CIPHER_CTX_set_padding(en, 1) != 1)
            return -8453;
        if (EVP_CipherUpdate(en, (byte*)padded, &outlen,
                    (byte*)cbcPlain, EVP_TEST_BUF_SZ) == 0)
            return -8454;
        if (outlen != 16)
            return -8455;
        total += outlen;

        if (EVP_CipherFinal(en, (byte*)&padded[total], &outlen) == 0)
            return -8456;
        total += outlen;
        if (total != 32)
            return -8457;
        XMEMCPY(cipher, padded, EVP_TEST_BUF_SZ);

        /* test out of bounds read on buffers w/o padding during decryption */
        total = 0;
        EVP_CIPHER_CTX_init(de);
        if (EVP_CipherInit(de, EVP_aes_128_cbc(),
            (unsigned char*)key, (unsigned char*)iv, 0) == 0)
            return -8458;

        if (EVP_CIPHER_CTX_set_padding(de, 0) != 1)
            return -8459;
        if (EVP_CipherUpdate(de, (byte*)plain, &outlen, (byte*)cipher,
                    EVP_TEST_BUF_SZ) == 0)
            return -8460;
        if (outlen != 16)
            return -8461;
        total += outlen;

        /* should fail since not using padding */
        if (EVP_CipherFinal(de, (byte*)&plain[total], &outlen) != 0)
            return -8462;

        total = 0;
        EVP_CIPHER_CTX_init(de);
        if (EVP_CipherInit(de, EVP_aes_128_cbc(),
            (unsigned char*)key, (unsigned char*)iv, 0) == 0)
            return -8463;
        if (EVP_CIPHER_CTX_set_padding(de, 1) != 1)
            return -8464;
        if (EVP_CipherUpdate(de, (byte*)padded, &outlen, (byte*)padded,
                    EVP_TEST_BUF_PAD) == 0)
            return -8465;
        if (outlen != 16)
            return -8466;
        total += outlen;

        if (EVP_CipherFinal(de, (byte*)&padded[total], &outlen) == 0)
            return -8467;
        if (XMEMCMP(padded, cbcPlain, EVP_TEST_BUF_SZ))
            return -8468;

#ifdef WOLFSSL_SMALL_STACK
        wolfSSL_EVP_CIPHER_CTX_free(en);
        wolfSSL_EVP_CIPHER_CTX_free(de);
#endif
    }

    {  /* evp_cipher test: EVP_aes_128_cbc */
#ifdef WOLFSSL_SMALL_STACK
        EVP_CIPHER_CTX *ctx = wolfSSL_EVP_CIPHER_CTX_new();
#else
        EVP_CIPHER_CTX ctx[1];
#endif

        WOLFSSL_SMALL_STACK_STATIC const byte msg[] = { /* "Now is the time for all " w/o trailing 0 */
            0x6e,0x6f,0x77,0x20,0x69,0x73,0x20,0x74,
            0x68,0x65,0x20,0x74,0x69,0x6d,0x65,0x20,
            0x66,0x6f,0x72,0x20,0x61,0x6c,0x6c,0x20
        };

        WOLFSSL_SMALL_STACK_STATIC const byte verify[] =
        {
            0x95,0x94,0x92,0x57,0x5f,0x42,0x81,0x53,
            0x2c,0xcc,0x9d,0x46,0x77,0xa2,0x33,0xcb
        };

        WOLFSSL_SMALL_STACK_STATIC const byte key[] =
            "0123456789abcdef   ";  /* align */
        WOLFSSL_SMALL_STACK_STATIC const byte iv[]  =
            "1234567890abcdef   ";  /* align */

        byte cipher[AES_BLOCK_SIZE * 4];
        byte plain [AES_BLOCK_SIZE * 4];

#ifdef WOLFSSL_SMALL_STACK
        if (ctx == NULL)
            return MEMORY_E;
#endif

        EVP_CIPHER_CTX_init(ctx);
        if (EVP_CipherInit(ctx, EVP_aes_128_cbc(), key, iv, 1) == 0)
            return -8469;

        if (EVP_Cipher(ctx, cipher, (byte*)msg, 16) != 16)
            return -8470;

        if (XMEMCMP(cipher, verify, AES_BLOCK_SIZE))
            return -8471;

        EVP_CIPHER_CTX_init(ctx);
        if (EVP_CipherInit(ctx, EVP_aes_128_cbc(), key, iv, 0) == 0)
            return -8472;

        if (EVP_Cipher(ctx, plain, cipher, 16) != 16)
            return -8473;

        if (XMEMCMP(plain, msg, AES_BLOCK_SIZE))
            return -8474;

#ifdef WOLFSSL_SMALL_STACK
        wolfSSL_EVP_CIPHER_CTX_free(ctx);
#endif
    }  /* end evp_cipher test: EVP_aes_128_cbc*/
#endif /* WOLFSSL_AES_128 */
#endif /* HAVE_AES_CBC */

#if defined(HAVE_AES_ECB) && defined(WOLFSSL_AES_256)
    {  /* evp_cipher test: EVP_aes_256_ecb*/
#ifdef WOLFSSL_SMALL_STACK
        EVP_CIPHER_CTX *ctx = wolfSSL_EVP_CIPHER_CTX_new();
#else
        EVP_CIPHER_CTX ctx[1];
#endif
        WOLFSSL_SMALL_STACK_STATIC const byte msg[] =
        {
          0x6b,0xc1,0xbe,0xe2,0x2e,0x40,0x9f,0x96,
          0xe9,0x3d,0x7e,0x11,0x73,0x93,0x17,0x2a
        };

        WOLFSSL_SMALL_STACK_STATIC const byte verify[] =
        {
            0xf3,0xee,0xd1,0xbd,0xb5,0xd2,0xa0,0x3c,
            0x06,0x4b,0x5a,0x7e,0x3d,0xb1,0x81,0xf8
        };

        WOLFSSL_SMALL_STACK_STATIC const byte key[] =
        {
          0x60,0x3d,0xeb,0x10,0x15,0xca,0x71,0xbe,
          0x2b,0x73,0xae,0xf0,0x85,0x7d,0x77,0x81,
          0x1f,0x35,0x2c,0x07,0x3b,0x61,0x08,0xd7,
          0x2d,0x98,0x10,0xa3,0x09,0x14,0xdf,0xf4
        };


        byte cipher[AES_BLOCK_SIZE * 4];
        byte plain [AES_BLOCK_SIZE * 4];

#ifdef WOLFSSL_SMALL_STACK
        if (ctx == NULL)
            return MEMORY_E;
#endif

        EVP_CIPHER_CTX_init(ctx);
        if (EVP_CipherInit(ctx, EVP_aes_256_ecb(), (unsigned char*)key, NULL, 1) == 0)
            return -8475;

        if (EVP_Cipher(ctx, cipher, (byte*)msg, 16) != 16)
            return -8476;

        if (XMEMCMP(cipher, verify, AES_BLOCK_SIZE))
            return -8477;

        EVP_CIPHER_CTX_init(ctx);
        if (EVP_CipherInit(ctx, EVP_aes_256_ecb(), (unsigned char*)key, NULL, 0) == 0)
            return -8478;

        if (EVP_Cipher(ctx, plain, cipher, 16) != 16)
            return -8479;

        if (XMEMCMP(plain, msg, AES_BLOCK_SIZE))
            return -8480;

#ifdef WOLFSSL_SMALL_STACK
        wolfSSL_EVP_CIPHER_CTX_free(ctx);
#endif
    }  /* end evp_cipher test */
#endif /* HAVE_AES_ECB && WOLFSSL_AES_256 */

#if defined(WOLFSSL_AES_DIRECT) && defined(WOLFSSL_AES_256)
    /* enable HAVE_AES_DECRYPT for AES_encrypt/decrypt */
    {
        /* Test: AES_encrypt/decrypt/set Key */
#ifdef WOLFSSL_SMALL_STACK
        AES_KEY *enc = (AES_KEY *)XMALLOC(sizeof *enc, HEAP_HINT, DYNAMIC_TYPE_AES);
        #ifdef HAVE_AES_DECRYPT
        AES_KEY *dec = (AES_KEY *)XMALLOC(sizeof *dec, HEAP_HINT, DYNAMIC_TYPE_AES);
        #endif
#else
        AES_KEY enc[1];
        #ifdef HAVE_AES_DECRYPT
        AES_KEY dec[1];
        #endif
#endif

        WOLFSSL_SMALL_STACK_STATIC const byte msg[] =
        {
            0x6b,0xc1,0xbe,0xe2,0x2e,0x40,0x9f,0x96,
            0xe9,0x3d,0x7e,0x11,0x73,0x93,0x17,0x2a
        };

        WOLFSSL_SMALL_STACK_STATIC const byte verify[] =
        {
            0xf3,0xee,0xd1,0xbd,0xb5,0xd2,0xa0,0x3c,
            0x06,0x4b,0x5a,0x7e,0x3d,0xb1,0x81,0xf8
        };

        WOLFSSL_SMALL_STACK_STATIC const byte key[] =
        {
            0x60,0x3d,0xeb,0x10,0x15,0xca,0x71,0xbe,
            0x2b,0x73,0xae,0xf0,0x85,0x7d,0x77,0x81,
            0x1f,0x35,0x2c,0x07,0x3b,0x61,0x08,0xd7,
            0x2d,0x98,0x10,0xa3,0x09,0x14,0xdf,0xf4
        };

        byte plain[sizeof(msg)];
        byte cipher[sizeof(msg)];

#ifdef WOLFSSL_SMALL_STACK
        if (enc == NULL)
            return MEMORY_E;
        #ifdef HAVE_AES_DECRYPT
        if (dec == NULL)
            return MEMORY_E;
        #endif
#endif

        AES_set_encrypt_key(key, sizeof(key)*8, enc);
        AES_set_decrypt_key(key,  sizeof(key)*8, dec);

        AES_encrypt(msg, cipher, enc);

        #ifdef HAVE_AES_DECRYPT
        AES_decrypt(cipher, plain, dec);
        if (XMEMCMP(plain, msg, AES_BLOCK_SIZE))
            return -8481;
        #endif /* HAVE_AES_DECRYPT */

        if (XMEMCMP(cipher, verify, AES_BLOCK_SIZE))
            return -8482;

#ifdef WOLFSSL_SMALL_STACK
        XFREE(enc, HEAP_HINT, DYNAMIC_TYPE_AES);
        #ifdef HAVE_AES_DECRYPT
        XFREE(dec, HEAP_HINT, DYNAMIC_TYPE_AES);
        #endif
#endif
    }
#endif /* WOLFSSL_AES_DIRECT && WOLFSSL_AES_256 */

/* EVP_Cipher with EVP_aes_xxx_ctr() */
#ifdef WOLFSSL_AES_COUNTER
    {
        byte plainBuff [64];
        byte cipherBuff[64];

#ifdef WOLFSSL_AES_128
        WOLFSSL_SMALL_STACK_STATIC const byte ctrKey[] =
        {
            0x2b,0x7e,0x15,0x16,0x28,0xae,0xd2,0xa6,
            0xab,0xf7,0x15,0x88,0x09,0xcf,0x4f,0x3c
        };

        WOLFSSL_SMALL_STACK_STATIC const byte ctrIv[] =
        {
            0xf0,0xf1,0xf2,0xf3,0xf4,0xf5,0xf6,0xf7,
            0xf8,0xf9,0xfa,0xfb,0xfc,0xfd,0xfe,0xff
        };


        WOLFSSL_SMALL_STACK_STATIC const byte ctrPlain[] =
        {
            0x6b,0xc1,0xbe,0xe2,0x2e,0x40,0x9f,0x96,
            0xe9,0x3d,0x7e,0x11,0x73,0x93,0x17,0x2a,
            0xae,0x2d,0x8a,0x57,0x1e,0x03,0xac,0x9c,
            0x9e,0xb7,0x6f,0xac,0x45,0xaf,0x8e,0x51,
            0x30,0xc8,0x1c,0x46,0xa3,0x5c,0xe4,0x11,
            0xe5,0xfb,0xc1,0x19,0x1a,0x0a,0x52,0xef,
            0xf6,0x9f,0x24,0x45,0xdf,0x4f,0x9b,0x17,
            0xad,0x2b,0x41,0x7b,0xe6,0x6c,0x37,0x10
        };

        WOLFSSL_SMALL_STACK_STATIC const byte ctrCipher[] =
        {
            0x87,0x4d,0x61,0x91,0xb6,0x20,0xe3,0x26,
            0x1b,0xef,0x68,0x64,0x99,0x0d,0xb6,0xce,
            0x98,0x06,0xf6,0x6b,0x79,0x70,0xfd,0xff,
            0x86,0x17,0x18,0x7b,0xb9,0xff,0xfd,0xff,
            0x5a,0xe4,0xdf,0x3e,0xdb,0xd5,0xd3,0x5e,
            0x5b,0x4f,0x09,0x02,0x0d,0xb0,0x3e,0xab,
            0x1e,0x03,0x1d,0xda,0x2f,0xbe,0x03,0xd1,
            0x79,0x21,0x70,0xa0,0xf3,0x00,0x9c,0xee
        };

        WOLFSSL_SMALL_STACK_STATIC const byte oddCipher[] =
        {
            0xb9,0xd7,0xcb,0x08,0xb0,0xe1,0x7b,0xa0,
            0xc2
        };
#endif

        /* test vector from "Recommendation for Block Cipher Modes of Operation"
         * NIST Special Publication 800-38A */
#ifdef WOLFSSL_AES_192
        WOLFSSL_SMALL_STACK_STATIC const byte ctr192Key[] =
        {
            0x8e,0x73,0xb0,0xf7,0xda,0x0e,0x64,0x52,
            0xc8,0x10,0xf3,0x2b,0x80,0x90,0x79,0xe5,
            0x62,0xf8,0xea,0xd2,0x52,0x2c,0x6b,0x7b
        };

        WOLFSSL_SMALL_STACK_STATIC const byte ctr192Iv[] =
        {
            0xf0,0xf1,0xf2,0xf3,0xf4,0xf5,0xf6,0xf7,
            0xf8,0xf9,0xfa,0xfb,0xfc,0xfd,0xfe,0xff
        };


        WOLFSSL_SMALL_STACK_STATIC const byte ctr192Plain[] =
        {
            0x6b,0xc1,0xbe,0xe2,0x2e,0x40,0x9f,0x96,
            0xe9,0x3d,0x7e,0x11,0x73,0x93,0x17,0x2a
        };

        WOLFSSL_SMALL_STACK_STATIC const byte ctr192Cipher[] =
        {
            0x1a,0xbc,0x93,0x24,0x17,0x52,0x1c,0xa2,
            0x4f,0x2b,0x04,0x59,0xfe,0x7e,0x6e,0x0b
        };
#endif /* WOLFSSL_AES_192 */

#ifdef WOLFSSL_AES_256
        /* test vector from "Recommendation for Block Cipher Modes of Operation"
         * NIST Special Publication 800-38A */
        WOLFSSL_SMALL_STACK_STATIC const byte ctr256Key[] =
        {
            0x60,0x3d,0xeb,0x10,0x15,0xca,0x71,0xbe,
            0x2b,0x73,0xae,0xf0,0x85,0x7d,0x77,0x81,
            0x1f,0x35,0x2c,0x07,0x3b,0x61,0x08,0xd7,
            0x2d,0x98,0x10,0xa3,0x09,0x14,0xdf,0xf4
        };

        WOLFSSL_SMALL_STACK_STATIC const byte ctr256Iv[] =
        {
            0xf0,0xf1,0xf2,0xf3,0xf4,0xf5,0xf6,0xf7,
            0xf8,0xf9,0xfa,0xfb,0xfc,0xfd,0xfe,0xff
        };


        WOLFSSL_SMALL_STACK_STATIC const byte ctr256Plain[] =
        {
            0x6b,0xc1,0xbe,0xe2,0x2e,0x40,0x9f,0x96,
            0xe9,0x3d,0x7e,0x11,0x73,0x93,0x17,0x2a
        };

        WOLFSSL_SMALL_STACK_STATIC const byte ctr256Cipher[] =
        {
            0x60,0x1e,0xc3,0x13,0x77,0x57,0x89,0xa5,
            0xb7,0xa7,0xf5,0x04,0xbb,0xf3,0xd2,0x28
        };
#endif /* WOLFSSL_AES_256 */

#ifdef WOLFSSL_SMALL_STACK
        EVP_CIPHER_CTX *en = wolfSSL_EVP_CIPHER_CTX_new();
        EVP_CIPHER_CTX *de = wolfSSL_EVP_CIPHER_CTX_new();
#else
        EVP_CIPHER_CTX en[1];
        EVP_CIPHER_CTX de[1];
#endif
#ifdef WOLFSSL_AES_128

#ifndef WOLFSSL_SMALL_STACK
        EVP_CIPHER_CTX *p_en;
        EVP_CIPHER_CTX *p_de;
#endif

#ifdef WOLFSSL_SMALL_STACK
        if ((en == NULL) || (de == NULL))
            return MEMORY_E;
#endif

        EVP_CIPHER_CTX_init(en);
        if (EVP_CipherInit(en, EVP_aes_128_ctr(),
                (unsigned char*)ctrKey, (unsigned char*)ctrIv, 0) == 0)
            return -8483;
        if (EVP_Cipher(en, (byte*)cipherBuff, (byte*)ctrPlain,
                AES_BLOCK_SIZE*4) != AES_BLOCK_SIZE*4)
            return -8484;
        EVP_CIPHER_CTX_init(de);
        if (EVP_CipherInit(de, EVP_aes_128_ctr(),
                (unsigned char*)ctrKey, (unsigned char*)ctrIv, 0) == 0)
            return -8485;

        if (EVP_Cipher(de, (byte*)plainBuff, (byte*)cipherBuff,
                AES_BLOCK_SIZE*4) != AES_BLOCK_SIZE*4)
            return -8486;

        if (XMEMCMP(cipherBuff, ctrCipher, AES_BLOCK_SIZE*4))
            return -8487;
        if (XMEMCMP(plainBuff, ctrPlain, AES_BLOCK_SIZE*4))
            return -8488;

#ifndef WOLFSSL_SMALL_STACK
        p_en = wolfSSL_EVP_CIPHER_CTX_new();
        if (p_en == NULL)
            return -8489;
        p_de = wolfSSL_EVP_CIPHER_CTX_new();
        if (p_de == NULL)
            return -8490;

        if (EVP_CipherInit(p_en, EVP_aes_128_ctr(),
                (unsigned char*)ctrKey, (unsigned char*)ctrIv, 0) == 0)
            return -8491;
        if (EVP_Cipher(p_en, (byte*)cipherBuff, (byte*)ctrPlain,
                AES_BLOCK_SIZE*4) != AES_BLOCK_SIZE*4)
            return -8492;
        if (EVP_CipherInit(p_de, EVP_aes_128_ctr(),
                (unsigned char*)ctrKey, (unsigned char*)ctrIv, 0) == 0)
            return -8493;

        if (EVP_Cipher(p_de, (byte*)plainBuff, (byte*)cipherBuff,
                AES_BLOCK_SIZE*4) != AES_BLOCK_SIZE*4)
            return -8494;

        wolfSSL_EVP_CIPHER_CTX_free(p_en);
        wolfSSL_EVP_CIPHER_CTX_free(p_de);
#endif /* !WOLFSSL_SMALL_STACK */

        if (XMEMCMP(cipherBuff, ctrCipher, AES_BLOCK_SIZE*4))
            return -8495;
        if (XMEMCMP(plainBuff, ctrPlain, AES_BLOCK_SIZE*4))
            return -8496;

        EVP_CIPHER_CTX_init(en);
        if (EVP_CipherInit(en, EVP_aes_128_ctr(),
            (unsigned char*)ctrKey, (unsigned char*)ctrIv, 0) == 0)
            return -8497;
        if (EVP_Cipher(en, (byte*)cipherBuff, (byte*)ctrPlain, 9) != 9)
            return -8498;

        EVP_CIPHER_CTX_init(de);
        if (EVP_CipherInit(de, EVP_aes_128_ctr(),
            (unsigned char*)ctrKey, (unsigned char*)ctrIv, 0) == 0)
            return -8499;

        if (EVP_Cipher(de, (byte*)plainBuff, (byte*)cipherBuff, 9) != 9)
            return -8500;

        if (XMEMCMP(plainBuff, ctrPlain, 9))
            return -8501;
        if (XMEMCMP(cipherBuff, ctrCipher, 9))
            return -8502;

        if (EVP_Cipher(en, (byte*)cipherBuff, (byte*)ctrPlain, 9) != 9)
            return -8503;
        if (EVP_Cipher(de, (byte*)plainBuff, (byte*)cipherBuff, 9) != 9)
            return -8504;

        if (XMEMCMP(plainBuff, ctrPlain, 9))
            return -8505;
        if (XMEMCMP(cipherBuff, oddCipher, 9))
            return -8506;
#endif /* WOLFSSL_AES_128 */

#ifdef WOLFSSL_AES_192
        EVP_CIPHER_CTX_init(en);
        if (EVP_CipherInit(en, EVP_aes_192_ctr(),
                (unsigned char*)ctr192Key, (unsigned char*)ctr192Iv, 0) == 0)
            return -8507;
        if (EVP_Cipher(en, (byte*)cipherBuff, (byte*)ctr192Plain,
                AES_BLOCK_SIZE) != AES_BLOCK_SIZE)
            return -8508;
        EVP_CIPHER_CTX_init(de);
        if (EVP_CipherInit(de, EVP_aes_192_ctr(),
            (unsigned char*)ctr192Key, (unsigned char*)ctr192Iv, 0) == 0)
            return -8509;

        XMEMSET(plainBuff, 0, sizeof(plainBuff));
        if (EVP_Cipher(de, (byte*)plainBuff, (byte*)cipherBuff,
                AES_BLOCK_SIZE) != AES_BLOCK_SIZE)
            return -8510;

        if (XMEMCMP(plainBuff, ctr192Plain, sizeof(ctr192Plain)))
            return -8511;
        if (XMEMCMP(ctr192Cipher, cipherBuff, sizeof(ctr192Cipher)))
            return -8512;
#endif /* WOLFSSL_AES_192 */

#ifdef WOLFSSL_AES_256
        EVP_CIPHER_CTX_init(en);
        if (EVP_CipherInit(en, EVP_aes_256_ctr(),
            (unsigned char*)ctr256Key, (unsigned char*)ctr256Iv, 0) == 0)
            return -8513;
        if (EVP_Cipher(en, (byte*)cipherBuff, (byte*)ctr256Plain,
                AES_BLOCK_SIZE) != AES_BLOCK_SIZE)
            return -8514;
        EVP_CIPHER_CTX_init(de);
        if (EVP_CipherInit(de, EVP_aes_256_ctr(),
            (unsigned char*)ctr256Key, (unsigned char*)ctr256Iv, 0) == 0)
            return -8515;

        XMEMSET(plainBuff, 0, sizeof(plainBuff));
        if (EVP_Cipher(de, (byte*)plainBuff, (byte*)cipherBuff,
                AES_BLOCK_SIZE) != AES_BLOCK_SIZE)
            return -8516;

        if (XMEMCMP(plainBuff, ctr256Plain, sizeof(ctr256Plain)))
            return -8517;
        if (XMEMCMP(ctr256Cipher, cipherBuff, sizeof(ctr256Cipher)))
            return -8518;

#ifdef WOLFSSL_SMALL_STACK
        wolfSSL_EVP_CIPHER_CTX_free(en);
        wolfSSL_EVP_CIPHER_CTX_free(de);
#endif

#endif /* WOLFSSL_AES_256 */
    }
#endif /* HAVE_AES_COUNTER */

#if defined(WOLFSSL_AES_CFB) && defined(WOLFSSL_AES_128)
    {
#ifdef WOLFSSL_SMALL_STACK
        AES_KEY *enc = (AES_KEY *)XMALLOC(sizeof *enc, HEAP_HINT, DYNAMIC_TYPE_AES);
        AES_KEY *dec = (AES_KEY *)XMALLOC(sizeof *dec, HEAP_HINT, DYNAMIC_TYPE_AES);
#else
        AES_KEY enc[1];
        AES_KEY dec[1];
#endif

        WOLFSSL_SMALL_STACK_STATIC const byte setIv[] = {
            0x00,0x01,0x02,0x03,0x04,0x05,0x06,0x07,
            0x08,0x09,0x0a,0x0b,0x0c,0x0d,0x0e,0x0f
        };

        WOLFSSL_SMALL_STACK_STATIC const byte key[] =
        {
            0x2b,0x7e,0x15,0x16,0x28,0xae,0xd2,0xa6,
            0xab,0xf7,0x15,0x88,0x09,0xcf,0x4f,0x3c
        };

        WOLFSSL_SMALL_STACK_STATIC const byte cipher1[] =
        {
            0x3b,0x3f,0xd9,0x2e,0xb7,0x2d,0xad,0x20,
            0x33,0x34,0x49,0xf8,0xe8,0x3c,0xfb,0x4a,
            0xc8,0xa6,0x45,0x37,0xa0,0xb3,0xa9,0x3f,
            0xcd,0xe3,0xcd,0xad,0x9f,0x1c,0xe5,0x8b
        };

        WOLFSSL_SMALL_STACK_STATIC const byte msg[] =
        {
            0x6b,0xc1,0xbe,0xe2,0x2e,0x40,0x9f,0x96,
            0xe9,0x3d,0x7e,0x11,0x73,0x93,0x17,0x2a,
            0xae,0x2d,0x8a,0x57,0x1e,0x03,0xac,0x9c,
            0x9e,0xb7,0x6f,0xac,0x45,0xaf,0x8e,0x51
        };

        byte cipher[AES_BLOCK_SIZE * 2];
        byte iv[AES_BLOCK_SIZE]; /* iv buffer is updeated by API */
        int  num = 0;

#ifdef WOLFSSL_SMALL_STACK
        if ((enc == NULL) || (dec == NULL))
            return MEMORY_E;
#endif

        XMEMCPY(iv, setIv, sizeof(setIv));
        wolfSSL_AES_set_encrypt_key(key, sizeof(key) * 8, enc);
        wolfSSL_AES_set_encrypt_key(key, sizeof(key) * 8, dec);

        wolfSSL_AES_cfb128_encrypt(msg, cipher, AES_BLOCK_SIZE - 1, enc, iv,
                &num, AES_ENCRYPT);

        if (XMEMCMP(cipher, cipher1, AES_BLOCK_SIZE - 1))
            return -8519;

        if (num != 15) /* should have used 15 of the 16 bytes */
            return -8520;

        wolfSSL_AES_cfb128_encrypt(msg + AES_BLOCK_SIZE - 1,
                cipher + AES_BLOCK_SIZE - 1, AES_BLOCK_SIZE + 1, enc, iv,
                &num, AES_ENCRYPT);

        if (XMEMCMP(cipher, cipher1, AES_BLOCK_SIZE * 2))
            return -8521;

        if (num != 0)
            return -8522;

#ifdef WOLFSSL_SMALL_STACK
        XFREE(enc, HEAP_HINT, DYNAMIC_TYPE_AES);
        XFREE(dec, HEAP_HINT, DYNAMIC_TYPE_AES);
#endif
    }
#endif /* WOLFSSL_AES_CFB && WOLFSSL_AES_128 */
    return 0;
}


#endif /* !defined(NO_AES) && !defined(WOLFCRYPT_ONLY) */

WOLFSSL_TEST_SUBROUTINE int openssl_test(void)
{
    int ret;
    EVP_MD_CTX md_ctx;
    testVector a, b, c, d, e, f;
    byte       hash[WC_SHA256_DIGEST_SIZE*2];  /* max size */

    a.inLen = 0;
    b.inLen = c.inLen = d.inLen = e.inLen = f.inLen = a.inLen;

    (void)a;
    (void)b;
    (void)c;
    (void)d;
    (void)e;
    (void)f;

    /* test malloc / free , 10 is an arbitrary amount of memory chosen */
    {
        byte* p;

        p = (byte*)CRYPTO_malloc(10, "", 0);

        if (p == NULL) {
            return -8600;
        }
        XMEMSET(p, 0, 10);
        CRYPTO_free(p, "", 0);
    }

#ifndef NO_MD5
    a.input  = "1234567890123456789012345678901234567890123456789012345678"
               "9012345678901234567890";
    a.output = "\x57\xed\xf4\xa2\x2b\xe3\xc9\x55\xac\x49\xda\x2e\x21\x07\xb6"
               "\x7a";
    a.inLen  = XSTRLEN(a.input);
    a.outLen = WC_MD5_DIGEST_SIZE;

    EVP_MD_CTX_init(&md_ctx);
    ret = EVP_DigestInit(&md_ctx, EVP_md5());
    if (ret == WOLFSSL_SUCCESS) {
        ret = EVP_DigestUpdate(&md_ctx, a.input, (unsigned long)a.inLen);
    }
    if (ret == WOLFSSL_SUCCESS) {
        ret = EVP_DigestFinal(&md_ctx, hash, 0);
    }
    EVP_MD_CTX_cleanup(&md_ctx);
    if (ret != WOLFSSL_SUCCESS)
        return -18601;
    if (XMEMCMP(hash, a.output, WC_MD5_DIGEST_SIZE) != 0)
        return -8601;
#endif /* NO_MD5 */

#ifndef NO_SHA
    b.input  = "aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa"
               "aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa"
               "aaaaaaaaaa";
    b.output = "\xAD\x5B\x3F\xDB\xCB\x52\x67\x78\xC2\x83\x9D\x2F\x15\x1E\xA7"
               "\x53\x99\x5E\x26\xA0";
    b.inLen  = XSTRLEN(b.input);
    b.outLen = WC_SHA_DIGEST_SIZE;

    EVP_MD_CTX_init(&md_ctx);
    ret = EVP_DigestInit(&md_ctx, EVP_sha1());
    if (ret == WOLFSSL_SUCCESS) {
        ret = EVP_DigestUpdate(&md_ctx, b.input, (unsigned long)b.inLen);
        if (ret == WOLFSSL_SUCCESS)
            ret = EVP_DigestFinal(&md_ctx, hash, 0);
    }
    EVP_MD_CTX_cleanup(&md_ctx);
    if (ret != WOLFSSL_SUCCESS)
        return -18602;
    if (XMEMCMP(hash, b.output, WC_SHA_DIGEST_SIZE) != 0)
        return -8602;
#endif /* NO_SHA */

#ifdef WOLFSSL_SHA224
    e.input  = "abcdefghbcdefghicdefghijdefghijkefghijklfghijklmghijklmnhi"
               "jklmnoijklmnopjklmnopqklmnopqrlmnopqrsmnopqrstnopqrstu";
    e.output = "\xc9\x7c\xa9\xa5\x59\x85\x0c\xe9\x7a\x04\xa9\x6d\xef\x6d\x99"
               "\xa9\xe0\xe0\xe2\xab\x14\xe6\xb8\xdf\x26\x5f\xc0\xb3";
    e.inLen  = XSTRLEN(e.input);
    e.outLen = WC_SHA224_DIGEST_SIZE;

    EVP_MD_CTX_init(&md_ctx);
    ret = EVP_DigestInit(&md_ctx, EVP_sha224());
    if (ret == WOLFSSL_SUCCESS) {
        ret = EVP_DigestUpdate(&md_ctx, e.input, (unsigned long)e.inLen);
        if (ret == WOLFSSL_SUCCESS)
            ret = EVP_DigestFinal(&md_ctx, hash, 0);
    }
    EVP_MD_CTX_cleanup(&md_ctx);
    if (ret != WOLFSSL_SUCCESS ||
            XMEMCMP(hash, e.output, WC_SHA224_DIGEST_SIZE) != 0) {
        return -8603;
    }
#endif /* WOLFSSL_SHA224 */

#ifndef NO_SHA256
    d.input  = "abcdbcdecdefdefgefghfghighijhijkijkljklmklmnlmnomnopnopq";
    d.output = "\x24\x8D\x6A\x61\xD2\x06\x38\xB8\xE5\xC0\x26\x93\x0C\x3E\x60"
               "\x39\xA3\x3C\xE4\x59\x64\xFF\x21\x67\xF6\xEC\xED\xD4\x19\xDB"
               "\x06\xC1";
    d.inLen  = XSTRLEN(d.input);
    d.outLen = WC_SHA256_DIGEST_SIZE;

    EVP_MD_CTX_init(&md_ctx);
    ret = EVP_DigestInit(&md_ctx, EVP_sha256());
    if (ret == WOLFSSL_SUCCESS) {
        ret = EVP_DigestUpdate(&md_ctx, d.input, (unsigned long)d.inLen);
        if (ret == WOLFSSL_SUCCESS)
            ret = EVP_DigestFinal(&md_ctx, hash, 0);
    }
    EVP_MD_CTX_cleanup(&md_ctx);
    if (ret != WOLFSSL_SUCCESS ||
            XMEMCMP(hash, d.output, WC_SHA256_DIGEST_SIZE) != 0) {
        return -8604;
    }
#endif /* !NO_SHA256 */

#ifdef WOLFSSL_SHA384
    e.input  = "abcdefghbcdefghicdefghijdefghijkefghijklfghijklmghijklmnhi"
               "jklmnoijklmnopjklmnopqklmnopqrlmnopqrsmnopqrstnopqrstu";
    e.output = "\x09\x33\x0c\x33\xf7\x11\x47\xe8\x3d\x19\x2f\xc7\x82\xcd\x1b"
               "\x47\x53\x11\x1b\x17\x3b\x3b\x05\xd2\x2f\xa0\x80\x86\xe3\xb0"
               "\xf7\x12\xfc\xc7\xc7\x1a\x55\x7e\x2d\xb9\x66\xc3\xe9\xfa\x91"
               "\x74\x60\x39";
    e.inLen  = XSTRLEN(e.input);
    e.outLen = WC_SHA384_DIGEST_SIZE;

    EVP_MD_CTX_init(&md_ctx);
    ret = EVP_DigestInit(&md_ctx, EVP_sha384());
    if (ret == WOLFSSL_SUCCESS) {
        ret = EVP_DigestUpdate(&md_ctx, e.input, (unsigned long)e.inLen);
        if (ret == WOLFSSL_SUCCESS)
            ret = EVP_DigestFinal(&md_ctx, hash, 0);
    }
    EVP_MD_CTX_cleanup(&md_ctx);
    if (ret != WOLFSSL_SUCCESS ||
            XMEMCMP(hash, e.output, WC_SHA384_DIGEST_SIZE) != 0) {
        return -8605;
    }
#endif /* WOLFSSL_SHA384 */

#ifdef WOLFSSL_SHA512
    f.input  = "abcdefghbcdefghicdefghijdefghijkefghijklfghijklmghijklmnhi"
               "jklmnoijklmnopjklmnopqklmnopqrlmnopqrsmnopqrstnopqrstu";
    f.output = "\x8e\x95\x9b\x75\xda\xe3\x13\xda\x8c\xf4\xf7\x28\x14\xfc\x14"
               "\x3f\x8f\x77\x79\xc6\xeb\x9f\x7f\xa1\x72\x99\xae\xad\xb6\x88"
               "\x90\x18\x50\x1d\x28\x9e\x49\x00\xf7\xe4\x33\x1b\x99\xde\xc4"
               "\xb5\x43\x3a\xc7\xd3\x29\xee\xb6\xdd\x26\x54\x5e\x96\xe5\x5b"
               "\x87\x4b\xe9\x09";
    f.inLen  = XSTRLEN(f.input);
    f.outLen = WC_SHA512_DIGEST_SIZE;

    EVP_MD_CTX_init(&md_ctx);
    ret = EVP_DigestInit(&md_ctx, EVP_sha512());
    if (ret == WOLFSSL_SUCCESS) {
        ret = EVP_DigestUpdate(&md_ctx, f.input, (unsigned long)f.inLen);
        if (ret == WOLFSSL_SUCCESS)
            ret = EVP_DigestFinal(&md_ctx, hash, 0);
    }
    EVP_MD_CTX_cleanup(&md_ctx);
    if (ret != WOLFSSL_SUCCESS ||
            XMEMCMP(hash, f.output, WC_SHA512_DIGEST_SIZE) != 0) {
        return -8606;
    }
#endif /* WOLFSSL_SHA512 */

#if !defined(HAVE_FIPS) && !defined(HAVE_SELFTEST)
#if defined(WOLFSSL_SHA512) && !defined(WOLFSSL_NOSHA512_224)
    f.input  = "abcdefghbcdefghicdefghijdefghijkefghijklfghijklmghijklmnhi"
               "jklmnoijklmnopjklmnopqklmnopqrlmnopqrsmnopqrstnopqrstu";
    f.output = "\x23\xfe\xc5\xbb\x94\xd6\x0b\x23\x30\x81\x92\x64\x0b\x0c\x45"
               "\x33\x35\xd6\x64\x73\x4f\xe4\x0e\x72\x68\x67\x4a\xf9";
    f.inLen  = XSTRLEN(f.input);
    f.outLen = WC_SHA512_224_DIGEST_SIZE;

    EVP_MD_CTX_init(&md_ctx);
    ret = EVP_DigestInit(&md_ctx, EVP_sha512_224());
    if (ret == WOLFSSL_SUCCESS) {
        ret = EVP_DigestUpdate(&md_ctx, f.input, (unsigned long)f.inLen);
        if (ret == WOLFSSL_SUCCESS)
            ret = EVP_DigestFinal(&md_ctx, hash, 0);
    }
    EVP_MD_CTX_cleanup(&md_ctx);
    if (ret != WOLFSSL_SUCCESS ||
            XMEMCMP(hash, f.output, WC_SHA512_224_DIGEST_SIZE) != 0) {
        return -8722;
    }
#endif /* WOLFSSL_SHA512 && !WOLFSSL_NOSHA512_224 */
#endif /* !HAVE_FIPS && !HAVE_SELFTEST */

#if !defined(HAVE_FIPS) && !defined(HAVE_SELFTEST)
#if defined(WOLFSSL_SHA512) && !defined(WOLFSSL_NOSHA512_256)
    f.input  = "abcdefghbcdefghicdefghijdefghijkefghijklfghijklmghijklmnhi"
               "jklmnoijklmnopjklmnopqklmnopqrlmnopqrsmnopqrstnopqrstu";
    f.output = "\x39\x28\xe1\x84\xfb\x86\x90\xf8\x40\xda\x39\x88\x12\x1d\x31"
               "\xbe\x65\xcb\x9d\x3e\xf8\x3e\xe6\x14\x6f\xea\xc8\x61\xe1\x9b"
               "\x56\x3a";
    f.inLen  = XSTRLEN(f.input);
    f.outLen = WC_SHA512_256_DIGEST_SIZE;

    EVP_MD_CTX_init(&md_ctx);
    ret = EVP_DigestInit(&md_ctx, EVP_sha512_256());
    if (ret == WOLFSSL_SUCCESS) {
        ret = EVP_DigestUpdate(&md_ctx, f.input, (unsigned long)f.inLen);
        if (ret == WOLFSSL_SUCCESS)
            ret = EVP_DigestFinal(&md_ctx, hash, 0);
    }
    EVP_MD_CTX_cleanup(&md_ctx);
    if (ret != WOLFSSL_SUCCESS ||
            XMEMCMP(hash, f.output, WC_SHA512_256_DIGEST_SIZE) != 0) {
        return -8723;
    }
#endif /* WOLFSSL_SHA512 && !WOLFSSL_NOSHA512_224 */
#endif /* !HAVE_FIPS && !HAVE_SELFTEST */

#ifdef WOLFSSL_SHA3
#ifndef WOLFSSL_NOSHA3_224
    e.input  = "abcdefghbcdefghicdefghijdefghijkefghijklfghijklmghijklmnhi"
               "jklmnoijklmnopjklmnopqklmnopqrlmnopqrsmnopqrstnopqrstu";
    e.output = "\x54\x3e\x68\x68\xe1\x66\x6c\x1a\x64\x36\x30\xdf\x77\x36\x7a"
               "\xe5\xa6\x2a\x85\x07\x0a\x51\xc1\x4c\xbf\x66\x5c\xbc";
    e.inLen  = XSTRLEN(e.input);
    e.outLen = WC_SHA3_224_DIGEST_SIZE;

    EVP_MD_CTX_init(&md_ctx);
    ret = EVP_DigestInit(&md_ctx, EVP_sha3_224());
    if (ret == WOLFSSL_SUCCESS) {
        ret = EVP_DigestUpdate(&md_ctx, e.input, (unsigned long)e.inLen);
        if (ret == WOLFSSL_SUCCESS)
            ret = EVP_DigestFinal(&md_ctx, hash, 0);
    }
    EVP_MD_CTX_cleanup(&md_ctx);
    if (ret != WOLFSSL_SUCCESS ||
            XMEMCMP(hash, e.output, WC_SHA3_224_DIGEST_SIZE) != 0) {
        return -8607;
    }
#endif /* WOLFSSL_NOSHA3_224 */

#ifndef WOLFSSL_NOSHA3_256
    d.input  = "abcdefghbcdefghicdefghijdefghijkefghijklfghijklmghijklmnhi"
               "jklmnoijklmnopjklmnopqklmnopqrlmnopqrsmnopqrstnopqrstu";
    d.output = "\x91\x6f\x60\x61\xfe\x87\x97\x41\xca\x64\x69\xb4\x39\x71\xdf"
               "\xdb\x28\xb1\xa3\x2d\xc3\x6c\xb3\x25\x4e\x81\x2b\xe2\x7a\xad"
               "\x1d\x18";
    d.inLen  = XSTRLEN(d.input);
    d.outLen = WC_SHA3_256_DIGEST_SIZE;

    EVP_MD_CTX_init(&md_ctx);
    ret = EVP_DigestInit(&md_ctx, EVP_sha3_256());
    if (ret == WOLFSSL_SUCCESS) {
        ret = EVP_DigestUpdate(&md_ctx, d.input, (unsigned long)d.inLen);
        if (ret == WOLFSSL_SUCCESS)
            ret = EVP_DigestFinal(&md_ctx, hash, 0);
    }
    EVP_MD_CTX_cleanup(&md_ctx);
    if (ret != WOLFSSL_SUCCESS ||
            XMEMCMP(hash, d.output, WC_SHA3_256_DIGEST_SIZE) != 0) {
        return -8608;
    }
#endif /* WOLFSSL_NOSHA3_256 */

    e.input  = "abcdefghbcdefghicdefghijdefghijkefghijklfghijklmghijklmnhi"
               "jklmnoijklmnopjklmnopqklmnopqrlmnopqrsmnopqrstnopqrstu";
    e.output = "\x79\x40\x7d\x3b\x59\x16\xb5\x9c\x3e\x30\xb0\x98\x22\x97\x47"
               "\x91\xc3\x13\xfb\x9e\xcc\x84\x9e\x40\x6f\x23\x59\x2d\x04\xf6"
               "\x25\xdc\x8c\x70\x9b\x98\xb4\x3b\x38\x52\xb3\x37\x21\x61\x79"
               "\xaa\x7f\xc7";
    e.inLen  = XSTRLEN(e.input);
    e.outLen = WC_SHA3_384_DIGEST_SIZE;

    EVP_MD_CTX_init(&md_ctx);
    ret = EVP_DigestInit(&md_ctx, EVP_sha3_384());
    if (ret == WOLFSSL_SUCCESS) {
        ret = EVP_DigestUpdate(&md_ctx, e.input, (unsigned long)e.inLen);
        if (ret == WOLFSSL_SUCCESS)
            ret = EVP_DigestFinal(&md_ctx, hash, 0);
    }
    EVP_MD_CTX_cleanup(&md_ctx);
    if (ret != WOLFSSL_SUCCESS ||
            XMEMCMP(hash, e.output, WC_SHA3_384_DIGEST_SIZE) != 0) {
        return -8609;
    }

#ifndef WOLFSSL_NOSHA3_512
    f.input  = "abcdefghbcdefghicdefghijdefghijkefghijklfghijklmghijklmnhi"
               "jklmnoijklmnopjklmnopqklmnopqrlmnopqrsmnopqrstnopqrstu";
    f.output = "\xaf\xeb\xb2\xef\x54\x2e\x65\x79\xc5\x0c\xad\x06\xd2\xe5\x78"
               "\xf9\xf8\xdd\x68\x81\xd7\xdc\x82\x4d\x26\x36\x0f\xee\xbf\x18"
               "\xa4\xfa\x73\xe3\x26\x11\x22\x94\x8e\xfc\xfd\x49\x2e\x74\xe8"
               "\x2e\x21\x89\xed\x0f\xb4\x40\xd1\x87\xf3\x82\x27\x0c\xb4\x55"
               "\xf2\x1d\xd1\x85";
    f.inLen  = XSTRLEN(f.input);
    f.outLen = WC_SHA3_512_DIGEST_SIZE;

    EVP_MD_CTX_init(&md_ctx);
    ret = EVP_DigestInit(&md_ctx, EVP_sha3_512());
    if (ret == WOLFSSL_SUCCESS) {
        ret = EVP_DigestUpdate(&md_ctx, f.input, (unsigned long)f.inLen);
        if (ret == WOLFSSL_SUCCESS)
            ret = EVP_DigestFinal(&md_ctx, hash, 0);
    }
    EVP_MD_CTX_cleanup(&md_ctx);
    if (ret != WOLFSSL_SUCCESS ||
            XMEMCMP(hash, f.output, WC_SHA3_512_DIGEST_SIZE) != 0) {
        return -8610;
    }
#endif /* WOLFSSL_NOSHA3_512 */
#endif /* WOLFSSL_SHA3 */

#ifndef WC_NO_RNG
    if (RAND_bytes(hash, sizeof(hash)) != WOLFSSL_SUCCESS)
        return -8611;
#endif

#ifndef NO_MD5
    c.input  = "what do ya want for nothing?";
    c.output = "\x55\x78\xe8\x48\x4b\xcc\x93\x80\x93\xec\x53\xaf\x22\xd6\x14"
               "\x76";
    c.inLen  = XSTRLEN(c.input);
    c.outLen = WC_MD5_DIGEST_SIZE;

#if defined(HAVE_FIPS_VERSION) && (HAVE_FIPS_VERSION > 2)
    /* Expect failure with MD5 + HMAC when using FIPS 140-3. */
    if (HMAC(EVP_md5(), "JefeJefeJefeJefe", 16, (byte*)c.input, (int)c.inLen,
            hash, 0) != NULL)
#else
    if (HMAC(EVP_md5(), "JefeJefeJefeJefe", 16, (byte*)c.input, (int)c.inLen,
            hash, 0) == NULL ||
        XMEMCMP(hash, c.output, WC_MD5_DIGEST_SIZE) != 0)
#endif
    {
        return -8612;
    }
#endif /* NO_MD5 */

#ifndef NO_DES3
    { /* des test */
    WOLFSSL_SMALL_STACK_STATIC const byte vector[] = { /* "now is the time for all " w/o trailing 0 */
        0x6e,0x6f,0x77,0x20,0x69,0x73,0x20,0x74,
        0x68,0x65,0x20,0x74,0x69,0x6d,0x65,0x20,
        0x66,0x6f,0x72,0x20,0x61,0x6c,0x6c,0x20
    };
    byte plain[24];
    byte cipher[24];
    const_DES_cblock key = {
        0x01,0x23,0x45,0x67,0x89,0xab,0xcd,0xef
    };
    DES_cblock iv = {
        0x12,0x34,0x56,0x78,0x90,0xab,0xcd,0xef
    };
    DES_key_schedule sched;
    WOLFSSL_SMALL_STACK_STATIC const byte verify[] = {
        0x8b,0x7c,0x52,0xb0,0x01,0x2b,0x6c,0xb8,
        0x4f,0x0f,0xeb,0xf3,0xfb,0x5f,0x86,0x73,
        0x15,0x85,0xb3,0x22,0x4b,0x86,0x2b,0x4b
    };

    DES_key_sched(&key, &sched);

    DES_cbc_encrypt(vector, cipher, sizeof(vector), &sched, &iv, DES_ENCRYPT);
    DES_cbc_encrypt(cipher, plain, sizeof(vector), &sched, &iv, DES_DECRYPT);

    if (XMEMCMP(plain, vector, sizeof(vector)) != 0)
        return -8613;

    if (XMEMCMP(cipher, verify, sizeof(verify)) != 0)
        return -8614;

        /* test changing iv */
    DES_ncbc_encrypt(vector, cipher, 8, &sched, &iv, DES_ENCRYPT);
    DES_ncbc_encrypt(vector + 8, cipher + 8, 16, &sched, &iv, DES_ENCRYPT);

    if (XMEMCMP(cipher, verify, sizeof(verify)) != 0)
        return -8615;

    }  /* end des test */
#endif /* NO_DES3 */

#if !defined(NO_AES) && !defined(WOLFCRYPT_ONLY)
    if (openssl_aes_test() != 0) {
        return -8616;
    }
#if defined(WOLFSSL_AES_128) && defined(HAVE_AES_CBC)
    {   /* evp_cipher test: EVP_aes_128_cbc */
#ifdef WOLFSSL_SMALL_STACK
        EVP_CIPHER_CTX *ctx = wolfSSL_EVP_CIPHER_CTX_new();
#else
        EVP_CIPHER_CTX ctx[1];
#endif
        int idx, cipherSz, plainSz;
        WOLFSSL_SMALL_STACK_STATIC const byte msg[] = { /* "Now is the time for all " w/o trailing 0 */
            0x6e,0x6f,0x77,0x20,0x69,0x73,0x20,0x74,
            0x68,0x65,0x20,0x74,0x69,0x6d,0x65,0x20,
            0x66,0x6f,0x72,0x20,0x61,0x6c,0x6c,0x20
        };
        WOLFSSL_SMALL_STACK_STATIC const byte verify[] = {
            0x95,0x94,0x92,0x57,0x5f,0x42,0x81,0x53,
            0x2c,0xcc,0x9d,0x46,0x77,0xa2,0x33,0xcb,
            0x3b,0x5d,0x41,0x97,0x94,0x25,0xa4,0xb4,
            0xae,0x7b,0x34,0xd0,0x3f,0x0c,0xbc,0x06
        };
        WOLFSSL_SMALL_STACK_STATIC const byte verify2[] = {
            0x95,0x94,0x92,0x57,0x5f,0x42,0x81,0x53,
            0x2c,0xcc,0x9d,0x46,0x77,0xa2,0x33,0xcb,
            0x7d,0x37,0x7b,0x0b,0x44,0xaa,0xb5,0xf0,
            0x5f,0x34,0xb4,0xde,0xb5,0xbd,0x2a,0xbb
        };
        WOLFSSL_SMALL_STACK_STATIC const byte key[] =
            "0123456789abcdef   ";  /* align */
        WOLFSSL_SMALL_STACK_STATIC const byte iv[]  =
            "1234567890abcdef   ";  /* align */
        byte cipher[AES_BLOCK_SIZE * 4];
        byte plain [AES_BLOCK_SIZE * 4];

#ifdef WOLFSSL_SMALL_STACK
        if (ctx == NULL)
            return MEMORY_E;
#endif

        cipherSz = 0;
        EVP_CIPHER_CTX_init(ctx);
        ret = EVP_CipherInit(ctx, EVP_aes_128_cbc(), key, iv, 1);
        if (ret == WOLFSSL_SUCCESS) {
            ret = EVP_CipherUpdate(ctx, cipher, &idx, (byte*)msg, sizeof(msg));
            if (ret == WOLFSSL_SUCCESS)
                cipherSz += idx;
        }
        if (ret == WOLFSSL_SUCCESS) {
            ret = EVP_CipherFinal(ctx, cipher + cipherSz, &idx);
            if (ret == WOLFSSL_SUCCESS)
                cipherSz += idx;
        }
        EVP_CIPHER_CTX_cleanup(ctx);
        if (ret != WOLFSSL_SUCCESS)
            return -8617;
        if (cipherSz != (int)sizeof(verify) || XMEMCMP(cipher, verify, cipherSz))
            return -8618;

        /* check partial decrypt (not enough padding for full block) */
        plainSz = 0;
        EVP_CIPHER_CTX_init(ctx);
        ret = EVP_CipherInit(ctx, EVP_aes_128_cbc(), key, iv, 0);
        if (ret == WOLFSSL_SUCCESS) {
            ret = EVP_CipherUpdate(ctx, plain, &idx, cipher, 1);
            if (ret == WOLFSSL_SUCCESS)
                plainSz += idx;
        }
        if (ret == WOLFSSL_SUCCESS) {
            /* this test should fail... not enough padding for full block */
            ret = EVP_CipherFinal(ctx, plain + plainSz, &idx);
            if (plainSz == 0 && ret != WOLFSSL_SUCCESS)
                ret = WOLFSSL_SUCCESS;
            else
                ret = -8619;
        }
        else
            ret = -8620;
        EVP_CIPHER_CTX_cleanup(ctx);
        if (ret != WOLFSSL_SUCCESS)
            return ret;

        plainSz = 0;
        EVP_CIPHER_CTX_init(ctx);
        ret = EVP_CipherInit(ctx, EVP_aes_128_cbc(), key, iv, 0);
        if (ret == WOLFSSL_SUCCESS) {
            ret = EVP_CipherUpdate(ctx, plain, &idx, cipher, cipherSz);
            if (ret == WOLFSSL_SUCCESS)
                plainSz += idx;
        }
        if (ret == WOLFSSL_SUCCESS) {
            ret = EVP_CipherFinal(ctx, plain + plainSz, &idx);
            if (ret == WOLFSSL_SUCCESS)
                plainSz += idx;
        }
        EVP_CIPHER_CTX_cleanup(ctx);
        if (ret != WOLFSSL_SUCCESS)
            return -8621;
        if (plainSz != (int)sizeof(msg) || XMEMCMP(plain, msg, sizeof(msg)))
            return -8622;

        cipherSz = 0;
        EVP_CIPHER_CTX_init(ctx);
        ret = EVP_CipherInit(ctx, EVP_aes_128_cbc(), key, iv, 1);
        if (ret == WOLFSSL_SUCCESS) {
            ret = EVP_CipherUpdate(ctx, cipher, &idx, msg, AES_BLOCK_SIZE);
            if (ret == WOLFSSL_SUCCESS)
                cipherSz += idx;
        }
        if (ret == WOLFSSL_SUCCESS) {
            ret = EVP_CipherFinal(ctx, cipher + cipherSz, &idx);
            if (ret == WOLFSSL_SUCCESS)
                cipherSz += idx;
        }
        EVP_CIPHER_CTX_cleanup(ctx);
        if (ret != WOLFSSL_SUCCESS)
            return -8623;
        if (cipherSz != (int)sizeof(verify2) || XMEMCMP(cipher, verify2, cipherSz))
            return -8624;

#ifdef WOLFSSL_SMALL_STACK
        wolfSSL_EVP_CIPHER_CTX_free(ctx);
#endif
    }  /* end evp_cipher test: EVP_aes_128_cbc*/
#endif /* WOLFSSL_AES_128 && HAVE_AES_CBC */

#if defined(HAVE_AES_ECB) && defined(WOLFSSL_AES_256)
    {   /* evp_cipher test: EVP_aes_256_ecb*/
#ifdef WOLFSSL_SMALL_STACK
        EVP_CIPHER_CTX *ctx = wolfSSL_EVP_CIPHER_CTX_new();
#else
        EVP_CIPHER_CTX ctx[1];
#endif
        WOLFSSL_SMALL_STACK_STATIC const byte msg[] = {
          0x6b,0xc1,0xbe,0xe2,0x2e,0x40,0x9f,0x96,
          0xe9,0x3d,0x7e,0x11,0x73,0x93,0x17,0x2a
        };
        WOLFSSL_SMALL_STACK_STATIC const byte verify[] = {
            0xf3,0xee,0xd1,0xbd,0xb5,0xd2,0xa0,0x3c,
            0x06,0x4b,0x5a,0x7e,0x3d,0xb1,0x81,0xf8
        };
        WOLFSSL_SMALL_STACK_STATIC const byte key[] = {
          0x60,0x3d,0xeb,0x10,0x15,0xca,0x71,0xbe,
          0x2b,0x73,0xae,0xf0,0x85,0x7d,0x77,0x81,
          0x1f,0x35,0x2c,0x07,0x3b,0x61,0x08,0xd7,
          0x2d,0x98,0x10,0xa3,0x09,0x14,0xdf,0xf4
        };
        byte cipher[AES_BLOCK_SIZE * 4];
        byte plain [AES_BLOCK_SIZE * 4];

#ifdef WOLFSSL_SMALL_STACK
        if (ctx == NULL)
            return MEMORY_E;
#endif

        EVP_CIPHER_CTX_init(ctx);
        ret = EVP_CipherInit(ctx, EVP_aes_256_ecb(), (unsigned char*)key, NULL, 1);
        if (ret == WOLFSSL_SUCCESS)
            ret = EVP_Cipher(ctx, cipher, (byte*)msg, 16);
        EVP_CIPHER_CTX_cleanup(ctx);
        if (ret != 16)
            return -8625;
        if (XMEMCMP(cipher, verify, AES_BLOCK_SIZE))
            return -8626;

        EVP_CIPHER_CTX_init(ctx);
        ret = EVP_CipherInit(ctx, EVP_aes_256_ecb(), (unsigned char*)key, NULL, 0);
        if (ret == WOLFSSL_SUCCESS)
            ret = EVP_Cipher(ctx, plain, cipher, 16);
        EVP_CIPHER_CTX_cleanup(ctx);
        if (ret != 16)
            return -8627;
        if (XMEMCMP(plain, msg, AES_BLOCK_SIZE))
            return -8628;

#ifdef WOLFSSL_SMALL_STACK
        wolfSSL_EVP_CIPHER_CTX_free(ctx);
#endif
    }  /* end evp_cipher test */
#endif /* HAVE_AES_ECB && WOLFSSL_AES_128 */

#define OPENSSL_TEST_ERROR (-10000)


#if defined(WOLFSSL_AES_DIRECT) && defined(WOLFSSL_AES_256)
  /* enable HAVE_AES_DECRYPT for AES_encrypt/decrypt */
{

  /* Test: AES_encrypt/decrypt/set Key */
#ifdef WOLFSSL_SMALL_STACK
  AES_KEY *enc = (AES_KEY *)XMALLOC(sizeof *enc, HEAP_HINT, DYNAMIC_TYPE_AES);
  #ifdef HAVE_AES_DECRYPT
  AES_KEY *dec = (AES_KEY *)XMALLOC(sizeof *dec, HEAP_HINT, DYNAMIC_TYPE_AES);
  #endif
#else
  AES_KEY enc[1];
  #ifdef HAVE_AES_DECRYPT
  AES_KEY dec[1];
  #endif
#endif

  WOLFSSL_SMALL_STACK_STATIC const byte msg[] =
  {
      0x6b,0xc1,0xbe,0xe2,0x2e,0x40,0x9f,0x96,
      0xe9,0x3d,0x7e,0x11,0x73,0x93,0x17,0x2a
  };

  WOLFSSL_SMALL_STACK_STATIC const byte verify[] =
  {
      0xf3,0xee,0xd1,0xbd,0xb5,0xd2,0xa0,0x3c,
      0x06,0x4b,0x5a,0x7e,0x3d,0xb1,0x81,0xf8
  };

  WOLFSSL_SMALL_STACK_STATIC const byte key[] =
  {
      0x60,0x3d,0xeb,0x10,0x15,0xca,0x71,0xbe,
      0x2b,0x73,0xae,0xf0,0x85,0x7d,0x77,0x81,
      0x1f,0x35,0x2c,0x07,0x3b,0x61,0x08,0xd7,
      0x2d,0x98,0x10,0xa3,0x09,0x14,0xdf,0xf4
  };

  byte plain[sizeof(msg)];
  byte cipher[sizeof(msg)];

  printf("openSSL extra test\n") ;

#ifdef WOLFSSL_SMALL_STACK
  if (enc == NULL)
      return MEMORY_E;
  #ifdef HAVE_AES_DECRYPT
  if (dec == NULL)
      return MEMORY_E;
  #endif
#endif

  AES_set_encrypt_key(key, sizeof(key)*8, enc);
  AES_set_decrypt_key(key,  sizeof(key)*8, dec);

  AES_encrypt(msg, cipher, enc);

#ifdef HAVE_AES_DECRYPT
  AES_decrypt(cipher, plain, dec);
  if (XMEMCMP(plain, msg, AES_BLOCK_SIZE))
      return OPENSSL_TEST_ERROR-60;
#endif /* HAVE_AES_DECRYPT */

  if (XMEMCMP(cipher, verify, AES_BLOCK_SIZE))
      return OPENSSL_TEST_ERROR-61;

#ifdef WOLFSSL_SMALL_STACK
  XFREE(enc, HEAP_HINT, DYNAMIC_TYPE_AES);
  #ifdef HAVE_AES_DECRYPT
  XFREE(dec, HEAP_HINT, DYNAMIC_TYPE_AES);
  #endif
#endif
}

#endif /* WOLFSSL_AES_DIRECT && WOLFSSL_AES_256 */

/* EVP_Cipher with EVP_aes_xxx_ctr() */
#ifdef WOLFSSL_AES_COUNTER
{
    byte plainBuff [64];
    byte cipherBuff[64];

#ifdef WOLFSSL_AES_128
    WOLFSSL_SMALL_STACK_STATIC const byte ctrKey[] =
    {
        0x2b,0x7e,0x15,0x16,0x28,0xae,0xd2,0xa6,
        0xab,0xf7,0x15,0x88,0x09,0xcf,0x4f,0x3c
    };

    WOLFSSL_SMALL_STACK_STATIC const byte ctrIv[] =
    {
        0xf0,0xf1,0xf2,0xf3,0xf4,0xf5,0xf6,0xf7,
        0xf8,0xf9,0xfa,0xfb,0xfc,0xfd,0xfe,0xff
    };

    WOLFSSL_SMALL_STACK_STATIC const byte ctrPlain[] =
    {
        0x6b,0xc1,0xbe,0xe2,0x2e,0x40,0x9f,0x96,
        0xe9,0x3d,0x7e,0x11,0x73,0x93,0x17,0x2a,
        0xae,0x2d,0x8a,0x57,0x1e,0x03,0xac,0x9c,
        0x9e,0xb7,0x6f,0xac,0x45,0xaf,0x8e,0x51,
        0x30,0xc8,0x1c,0x46,0xa3,0x5c,0xe4,0x11,
        0xe5,0xfb,0xc1,0x19,0x1a,0x0a,0x52,0xef,
        0xf6,0x9f,0x24,0x45,0xdf,0x4f,0x9b,0x17,
        0xad,0x2b,0x41,0x7b,0xe6,0x6c,0x37,0x10
    };

    WOLFSSL_SMALL_STACK_STATIC const byte ctrCipher[] =
    {
        0x87,0x4d,0x61,0x91,0xb6,0x20,0xe3,0x26,
        0x1b,0xef,0x68,0x64,0x99,0x0d,0xb6,0xce,
        0x98,0x06,0xf6,0x6b,0x79,0x70,0xfd,0xff,
        0x86,0x17,0x18,0x7b,0xb9,0xff,0xfd,0xff,
        0x5a,0xe4,0xdf,0x3e,0xdb,0xd5,0xd3,0x5e,
        0x5b,0x4f,0x09,0x02,0x0d,0xb0,0x3e,0xab,
        0x1e,0x03,0x1d,0xda,0x2f,0xbe,0x03,0xd1,
        0x79,0x21,0x70,0xa0,0xf3,0x00,0x9c,0xee
    };

    WOLFSSL_SMALL_STACK_STATIC const byte oddCipher[] =
    {
        0xb9,0xd7,0xcb,0x08,0xb0,0xe1,0x7b,0xa0,
        0xc2
    };
#endif /* WOLFSSL_AES_128 */

#ifdef WOLFSSL_AES_192
    /* test vector from "Recommendation for Block Cipher Modes of Operation"
     * NIST Special Publication 800-38A */
    WOLFSSL_SMALL_STACK_STATIC const byte ctr192Key[] =
    {
        0x8e,0x73,0xb0,0xf7,0xda,0x0e,0x64,0x52,
        0xc8,0x10,0xf3,0x2b,0x80,0x90,0x79,0xe5,
        0x62,0xf8,0xea,0xd2,0x52,0x2c,0x6b,0x7b
    };

    WOLFSSL_SMALL_STACK_STATIC const byte ctr192Iv[] =
    {
        0xf0,0xf1,0xf2,0xf3,0xf4,0xf5,0xf6,0xf7,
        0xf8,0xf9,0xfa,0xfb,0xfc,0xfd,0xfe,0xff
    };


    WOLFSSL_SMALL_STACK_STATIC const byte ctr192Plain[] =
    {
        0x6b,0xc1,0xbe,0xe2,0x2e,0x40,0x9f,0x96,
        0xe9,0x3d,0x7e,0x11,0x73,0x93,0x17,0x2a
    };

    WOLFSSL_SMALL_STACK_STATIC const byte ctr192Cipher[] =
    {
        0x1a,0xbc,0x93,0x24,0x17,0x52,0x1c,0xa2,
        0x4f,0x2b,0x04,0x59,0xfe,0x7e,0x6e,0x0b
    };
#endif /* WOLFSSL_AES_192 */

#ifdef WOLFSSL_AES_256
    /* test vector from "Recommendation for Block Cipher Modes of Operation"
     * NIST Special Publication 800-38A */
    WOLFSSL_SMALL_STACK_STATIC const byte ctr256Key[] =
    {
        0x60,0x3d,0xeb,0x10,0x15,0xca,0x71,0xbe,
        0x2b,0x73,0xae,0xf0,0x85,0x7d,0x77,0x81,
        0x1f,0x35,0x2c,0x07,0x3b,0x61,0x08,0xd7,
        0x2d,0x98,0x10,0xa3,0x09,0x14,0xdf,0xf4
    };

    WOLFSSL_SMALL_STACK_STATIC const byte ctr256Iv[] =
    {
        0xf0,0xf1,0xf2,0xf3,0xf4,0xf5,0xf6,0xf7,
        0xf8,0xf9,0xfa,0xfb,0xfc,0xfd,0xfe,0xff
    };


    WOLFSSL_SMALL_STACK_STATIC const byte ctr256Plain[] =
    {
        0x6b,0xc1,0xbe,0xe2,0x2e,0x40,0x9f,0x96,
        0xe9,0x3d,0x7e,0x11,0x73,0x93,0x17,0x2a
    };

    WOLFSSL_SMALL_STACK_STATIC const byte ctr256Cipher[] =
    {
        0x60,0x1e,0xc3,0x13,0x77,0x57,0x89,0xa5,
        0xb7,0xa7,0xf5,0x04,0xbb,0xf3,0xd2,0x28
    };
#endif /* WOLFSSL_AES_256 */

#ifdef WOLFSSL_SMALL_STACK
    EVP_CIPHER_CTX *en = wolfSSL_EVP_CIPHER_CTX_new();
    EVP_CIPHER_CTX *de = wolfSSL_EVP_CIPHER_CTX_new();
    if ((en == NULL) || (de == NULL))
        return MEMORY_E;
#else
    EVP_CIPHER_CTX en[1];
    EVP_CIPHER_CTX de[1];
#endif

#ifdef WOLFSSL_AES_128

#ifndef WOLFSSL_SMALL_STACK
    EVP_CIPHER_CTX *p_en;
    EVP_CIPHER_CTX *p_de;
#endif

    EVP_CIPHER_CTX_init(en);
    if (EVP_CipherInit(en, EVP_aes_128_ctr(),
            (unsigned char*)ctrKey, (unsigned char*)ctrIv, 0) == 0)
        return -8629;
    if (EVP_Cipher(en, (byte*)cipherBuff, (byte*)ctrPlain,
            AES_BLOCK_SIZE*4) != AES_BLOCK_SIZE*4)
        return -8630;
    EVP_CIPHER_CTX_init(de);
    if (EVP_CipherInit(de, EVP_aes_128_ctr(),
            (unsigned char*)ctrKey, (unsigned char*)ctrIv, 0) == 0)
        return -8631;

    if (EVP_Cipher(de, (byte*)plainBuff, (byte*)cipherBuff,
            AES_BLOCK_SIZE*4) != AES_BLOCK_SIZE*4)
        return -8632;

    if (XMEMCMP(cipherBuff, ctrCipher, AES_BLOCK_SIZE*4))
        return -8633;
    if (XMEMCMP(plainBuff, ctrPlain, AES_BLOCK_SIZE*4))
        return -8634;

#ifndef WOLFSSL_SMALL_STACK
    p_en = wolfSSL_EVP_CIPHER_CTX_new();
    if(p_en == NULL)return -8635;
    p_de = wolfSSL_EVP_CIPHER_CTX_new();
    if(p_de == NULL)return -8636;

    if (EVP_CipherInit(p_en, EVP_aes_128_ctr(),
            (unsigned char*)ctrKey, (unsigned char*)ctrIv, 0) == 0)
        return -8637;
    if (EVP_Cipher(p_en, (byte*)cipherBuff, (byte*)ctrPlain,
            AES_BLOCK_SIZE*4) != AES_BLOCK_SIZE*4)
        return -8638;
    if (EVP_CipherInit(p_de, EVP_aes_128_ctr(),
            (unsigned char*)ctrKey, (unsigned char*)ctrIv, 0) == 0)
        return -8639;

    if (EVP_Cipher(p_de, (byte*)plainBuff, (byte*)cipherBuff,
            AES_BLOCK_SIZE*4) != AES_BLOCK_SIZE*4)
        return -8640;

    wolfSSL_EVP_CIPHER_CTX_free(p_en);
    wolfSSL_EVP_CIPHER_CTX_free(p_de);
#endif /* !WOLFSSL_SMALL_STACK */

    if (XMEMCMP(cipherBuff, ctrCipher, AES_BLOCK_SIZE*4))
        return -8641;
    if (XMEMCMP(plainBuff, ctrPlain, AES_BLOCK_SIZE*4))
        return -8642;

    EVP_CIPHER_CTX_init(en);
    if (EVP_CipherInit(en, EVP_aes_128_ctr(),
        (unsigned char*)ctrKey, (unsigned char*)ctrIv, 0) == 0)
        return -8643;
    if (EVP_Cipher(en, (byte*)cipherBuff, (byte*)ctrPlain, 9) != 9)
        return -8644;

    EVP_CIPHER_CTX_init(de);
    if (EVP_CipherInit(de, EVP_aes_128_ctr(),
        (unsigned char*)ctrKey, (unsigned char*)ctrIv, 0) == 0)
        return -8645;

    if (EVP_Cipher(de, (byte*)plainBuff, (byte*)cipherBuff, 9) != 9)
        return -8646;

    if (XMEMCMP(plainBuff, ctrPlain, 9))
        return -8647;
    if (XMEMCMP(cipherBuff, ctrCipher, 9))
        return -8648;

    if (EVP_Cipher(en, (byte*)cipherBuff, (byte*)ctrPlain, 9) != 9)
        return -8649;
    if (EVP_Cipher(de, (byte*)plainBuff, (byte*)cipherBuff, 9) != 9)
        return -8650;

    if (XMEMCMP(plainBuff, ctrPlain, 9))
        return -8651;
    if (XMEMCMP(cipherBuff, oddCipher, 9))
        return -8652;
#endif /* WOLFSSL_AES_128 */

#ifdef WOLFSSL_AES_192
    EVP_CIPHER_CTX_init(en);
    if (EVP_CipherInit(en, EVP_aes_192_ctr(),
            (unsigned char*)ctr192Key, (unsigned char*)ctr192Iv, 0) == 0)
        return -8653;
    if (EVP_Cipher(en, (byte*)cipherBuff, (byte*)ctr192Plain,
            AES_BLOCK_SIZE) != AES_BLOCK_SIZE)
        return -8654;
    EVP_CIPHER_CTX_init(de);
    if (EVP_CipherInit(de, EVP_aes_192_ctr(),
        (unsigned char*)ctr192Key, (unsigned char*)ctr192Iv, 0) == 0)
        return -8655;

    XMEMSET(plainBuff, 0, sizeof(plainBuff));
    if (EVP_Cipher(de, (byte*)plainBuff, (byte*)cipherBuff,
            AES_BLOCK_SIZE) != AES_BLOCK_SIZE)
        return -8656;

    if (XMEMCMP(plainBuff, ctr192Plain, sizeof(ctr192Plain)))
        return -8657;
    if (XMEMCMP(ctr192Cipher, cipherBuff, sizeof(ctr192Cipher)))
        return -8658;
#endif /* WOLFSSL_AES_192 */

#ifdef WOLFSSL_AES_256
    EVP_CIPHER_CTX_init(en);
    if (EVP_CipherInit(en, EVP_aes_256_ctr(),
        (unsigned char*)ctr256Key, (unsigned char*)ctr256Iv, 0) == 0)
        return -8659;
    if (EVP_Cipher(en, (byte*)cipherBuff, (byte*)ctr256Plain,
            AES_BLOCK_SIZE) != AES_BLOCK_SIZE)
        return -8660;
    EVP_CIPHER_CTX_init(de);
    if (EVP_CipherInit(de, EVP_aes_256_ctr(),
        (unsigned char*)ctr256Key, (unsigned char*)ctr256Iv, 0) == 0)
        return -8661;

    XMEMSET(plainBuff, 0, sizeof(plainBuff));
    if (EVP_Cipher(de, (byte*)plainBuff, (byte*)cipherBuff,
            AES_BLOCK_SIZE) != AES_BLOCK_SIZE)
        return -8662;

    if (XMEMCMP(plainBuff, ctr256Plain, sizeof(ctr256Plain)))
        return -8663;
    if (XMEMCMP(ctr256Cipher, cipherBuff, sizeof(ctr256Cipher)))
        return -8664;
#endif /* WOLFSSL_AES_256 */

#ifdef WOLFSSL_SMALL_STACK
    wolfSSL_EVP_CIPHER_CTX_free(en);
    wolfSSL_EVP_CIPHER_CTX_free(de);
#endif
}
#endif /* HAVE_AES_COUNTER */

#if defined(HAVE_AES_CBC) && defined(WOLFSSL_AES_128)
{
      /* EVP_CipherUpdate test */


        WOLFSSL_SMALL_STACK_STATIC const byte cbcPlain[] =
        {
            0x6b,0xc1,0xbe,0xe2,0x2e,0x40,0x9f,0x96,
            0xe9,0x3d,0x7e,0x11,0x73,0x93,0x17,0x2a,
            0xae,0x2d,0x8a,0x57,0x1e,0x03,0xac,0x9c,
            0x9e,0xb7,0x6f,0xac,0x45,0xaf,0x8e,0x51,
            0x30,0xc8,0x1c,0x46,0xa3,0x5c,0xe4,0x11,
            0xe5,0xfb,0xc1,0x19,0x1a,0x0a,0x52,0xef,
            0xf6,0x9f,0x24,0x45,0xdf,0x4f,0x9b,0x17,
            0xad,0x2b,0x41,0x7b,0xe6,0x6c,0x37,0x10
        };

        byte key[] = "0123456789abcdef   ";  /* align */
        byte iv[]  = "1234567890abcdef   ";  /* align */

        byte cipher[AES_BLOCK_SIZE * 4];
        byte plain [AES_BLOCK_SIZE * 4];
#ifdef WOLFSSL_SMALL_STACK
        EVP_CIPHER_CTX *en = wolfSSL_EVP_CIPHER_CTX_new();
        EVP_CIPHER_CTX *de = wolfSSL_EVP_CIPHER_CTX_new();
#else
        EVP_CIPHER_CTX en[1];
        EVP_CIPHER_CTX de[1];
#endif
        int outlen ;
        int total = 0;

#ifdef WOLFSSL_SMALL_STACK
        if ((en == NULL) || (de == NULL))
            return MEMORY_E;
#endif

        EVP_CIPHER_CTX_init(en);
        if (EVP_CipherInit(en, EVP_aes_128_cbc(),
            (unsigned char*)key, (unsigned char*)iv, 1) == 0)
            return -8665;
        /* openSSL compatibility, if(inlen == 0)return 1; */
        if (EVP_CipherUpdate(en, (byte*)cipher, &outlen,
                                                    (byte*)cbcPlain, 0) != 1)
            return -8666;

        EVP_CIPHER_CTX_init(en);
        if (EVP_CipherInit(en, EVP_aes_128_cbc(),
            (unsigned char*)key, (unsigned char*)iv, 1) == 0)
            return -8667;
        if (EVP_CipherUpdate(en, (byte*)cipher, &outlen,
                                                    (byte*)cbcPlain, 9) == 0)
            return -8668;
        if(outlen != 0)
            return -8669;
        total += outlen;

        if (EVP_CipherUpdate(en, (byte*)&cipher[total], &outlen,
                                                (byte*)&cbcPlain[9]  , 9) == 0)
            return -8670;
        if(outlen != 16)
            return -8671;
        total += outlen;

        if (EVP_CipherFinal(en, (byte*)&cipher[total], &outlen) == 0)
            return -8672;
        if(outlen != 16)
            return -8673;
        total += outlen;
        if(total != 32)
            return -8674;

        total = 0;
        EVP_CIPHER_CTX_init(de);
        if (EVP_CipherInit(de, EVP_aes_128_cbc(),
            (unsigned char*)key, (unsigned char*)iv, 0) == 0)
            return -8675;

        if (EVP_CipherUpdate(de, (byte*)plain, &outlen, (byte*)cipher, 6) == 0)
            return -8676;
        if(outlen != 0)
            return -8677;
        total += outlen;

        if (EVP_CipherUpdate(de, (byte*)&plain[total], &outlen,
                                                    (byte*)&cipher[6], 12) == 0)
            return -8678;
        if(outlen != 0)
        total += outlen;

        if (EVP_CipherUpdate(de, (byte*)&plain[total], &outlen,
                                                (byte*)&cipher[6+12], 14) == 0)
            return -8679;
        if(outlen != 16)
            return -8680;
        total += outlen;

        if (EVP_CipherFinal(de, (byte*)&plain[total], &outlen) == 0)
            return -8681;
        if(outlen != 2)
            return -8682;
        total += outlen;

        if(total != 18)
            return -8683;

        if (XMEMCMP(plain, cbcPlain, 18))
            return -8684;

        total = 0;
        EVP_CIPHER_CTX_init(en);
        if (EVP_EncryptInit(en, EVP_aes_128_cbc(),
            (unsigned char*)key, (unsigned char*)iv) == 0)
            return -8685;
        if (EVP_CipherUpdate(en, (byte*)cipher, &outlen, (byte*)cbcPlain, 9) == 0)
            return -8686;
        if(outlen != 0)
            return -8687;
        total += outlen;

        if (EVP_CipherUpdate(en, (byte*)&cipher[total], &outlen, (byte*)&cbcPlain[9]  , 9) == 0)
            return -8688;
        if(outlen != 16)
            return -8689;
        total += outlen;

        if (EVP_EncryptFinal(en, (byte*)&cipher[total], &outlen) == 0)
            return -8690;
        if(outlen != 16)
            return -8691;
        total += outlen;
        if(total != 32)
            return 3438;

        total = 0;
        EVP_CIPHER_CTX_init(de);
        if (EVP_DecryptInit(de, EVP_aes_128_cbc(),
            (unsigned char*)key, (unsigned char*)iv) == 0)
            return -8692;

        if (EVP_CipherUpdate(de, (byte*)plain, &outlen, (byte*)cipher, 6) == 0)
            return -8693;
        if(outlen != 0)
            return -8694;
        total += outlen;

        if (EVP_CipherUpdate(de, (byte*)&plain[total], &outlen, (byte*)&cipher[6], 12) == 0)
            return -8695;
        if(outlen != 0)
        total += outlen;

        if (EVP_CipherUpdate(de, (byte*)&plain[total], &outlen, (byte*)&cipher[6+12], 14) == 0)
            return -8696;
        if(outlen != 16)
            return -8697;
        total += outlen;

        if (EVP_DecryptFinal(de, (byte*)&plain[total], &outlen) == 0)
            return -8698;
        if(outlen != 2)
            return -8699;
        total += outlen;

        if(total != 18)
            return 3447;

        if (XMEMCMP(plain, cbcPlain, 18))
            return -8700;

        if (EVP_CIPHER_key_length(NULL) != 0)
            return -8701;

        if (EVP_CIPHER_key_length(EVP_aes_128_cbc()) != 16)
            return -8702;

        if (EVP_CIPHER_CTX_mode(NULL) != 0)
            return -8703;

        if (EVP_CIPHER_CTX_mode(en) != (en->flags & WOLFSSL_EVP_CIPH_MODE))
            return -8704;

        EVP_CIPHER_CTX_init(en);
        if (EVP_CipherInit_ex(en, EVP_aes_128_cbc(), NULL,
            (unsigned char*)key, (unsigned char*)iv, 0) == 0)
            return -8705;

        EVP_CIPHER_CTX_init(en);
        if (EVP_EncryptInit_ex(en, EVP_aes_128_cbc(), NULL,
                (unsigned char*)key, (unsigned char*)iv) == 0)
            return -8706;

        if (wolfSSL_EVP_EncryptFinal_ex(NULL, NULL, NULL) != WOLFSSL_FAILURE)
            return -8707;

        if (wolfSSL_EVP_EncryptFinal(NULL, NULL, NULL) != WOLFSSL_FAILURE)
            return -8708;

        EVP_CIPHER_CTX_init(de);
        if (EVP_DecryptInit_ex(de, EVP_aes_128_cbc(), NULL,
                (unsigned char*)key, (unsigned char*)iv) == 0)
            return -8709;

        if (wolfSSL_EVP_DecryptFinal(NULL, NULL, NULL) != WOLFSSL_FAILURE)
            return -8710;

        if (wolfSSL_EVP_DecryptFinal_ex(NULL, NULL, NULL) != WOLFSSL_FAILURE)
            return -8711;

        if (EVP_CIPHER_CTX_block_size(NULL) != BAD_FUNC_ARG)
            return -8712;

        EVP_CIPHER_CTX_init(en);
        EVP_EncryptInit_ex(en, EVP_aes_128_cbc(), NULL,
                (unsigned char*)key, (unsigned char*)iv);
        if (EVP_CIPHER_CTX_block_size(en) != en->block_size)
            return -8713;

        if (EVP_CIPHER_block_size(NULL) != BAD_FUNC_ARG)
            return -8714;

        if (EVP_CIPHER_block_size(EVP_aes_128_cbc()) != AES_BLOCK_SIZE)
            return -8715;

        if (WOLFSSL_EVP_CIPHER_mode(NULL) != 0)
            return -8716;

        if (EVP_CIPHER_flags(EVP_aes_128_cbc()) != WOLFSSL_EVP_CIPH_CBC_MODE)
            return -8717;

        EVP_CIPHER_CTX_clear_flags(en, 0xFFFFFFFF);
        EVP_CIPHER_CTX_set_flags(en, 42);
        if (en->flags != 42)
            return -8718;

        if (EVP_CIPHER_CTX_set_padding(NULL, 0) != BAD_FUNC_ARG)
            return -8719;
        if (EVP_CIPHER_CTX_set_padding(en, 0) != WOLFSSL_SUCCESS)
            return -8720;
        if (EVP_CIPHER_CTX_set_padding(en, 1) != WOLFSSL_SUCCESS)
            return -8721;

#ifdef WOLFSSL_SMALL_STACK
        wolfSSL_EVP_CIPHER_CTX_free(en);
        wolfSSL_EVP_CIPHER_CTX_free(de);
#endif
    }
#endif /* WOLFSSL_AES_128 && HAVE_AES_CBC */
#endif /* ifndef NO_AES */
    return 0;
}

WOLFSSL_TEST_SUBROUTINE int openSSL_evpMD_test(void)
{
    int ret = 0;
#if !defined(NO_SHA256) && !defined(NO_SHA)
    WOLFSSL_EVP_MD_CTX* ctx;
    WOLFSSL_EVP_MD_CTX* ctx2;

    ctx = EVP_MD_CTX_create();
    ctx2 = EVP_MD_CTX_create();

    ret = EVP_DigestInit(ctx, EVP_sha256());
    if (ret != SSL_SUCCESS) {
        ret = -8800;
        goto openSSL_evpMD_test_done;
    }

    ret = EVP_MD_CTX_copy(ctx2, ctx);
    if (ret != SSL_SUCCESS) {
        ret = -8801;
        goto openSSL_evpMD_test_done;
    }

    if (EVP_MD_type(EVP_sha256()) != EVP_MD_CTX_type(ctx2)) {
        ret = -8802;
        goto openSSL_evpMD_test_done;
    }

    ret = EVP_DigestInit(ctx, EVP_sha1());
    if (ret != SSL_SUCCESS) {
        ret = -8803;
        goto openSSL_evpMD_test_done;
    }

    if (EVP_MD_type(EVP_sha256()) != EVP_MD_CTX_type(ctx2)) {
        ret = -8804;
        goto openSSL_evpMD_test_done;
    }

    ret = EVP_MD_CTX_copy_ex(ctx2, ctx);
    if (ret != SSL_SUCCESS) {
        ret = -8805;
        goto openSSL_evpMD_test_done;
    }

    if (EVP_MD_type(EVP_sha256()) == EVP_MD_CTX_type(ctx2)) {
        ret = -8806;
        goto openSSL_evpMD_test_done;
    }

    if (EVP_MD_type(EVP_sha1()) != EVP_MD_CTX_type(ctx2)) {
        ret = -8807;
        goto openSSL_evpMD_test_done;
    }

    if (EVP_DigestInit_ex(ctx, EVP_sha1(), NULL) != SSL_SUCCESS) {
        ret = -8808;
        goto openSSL_evpMD_test_done;
    }

    if (EVP_add_digest(NULL) != 0) {
        ret = -8809;
        goto openSSL_evpMD_test_done;
    }

    if (wolfSSL_EVP_add_cipher(NULL) != 0) {
        ret = -8810;
        goto openSSL_evpMD_test_done;
    }

    ret = 0; /* got to success state without jumping to end with a fail */

openSSL_evpMD_test_done:
    EVP_MD_CTX_destroy(ctx);
    EVP_MD_CTX_destroy(ctx2);
#endif /* NO_SHA256 */

    return ret;
}

#ifdef DEBUG_SIGN
static void show(const char *title, const char *p, unsigned int s) {
    char* i;
    printf("%s: ", title);
    for (i = p;
            i < p + s;
            printf("%c", *i), i++);
    printf("\n");
}
#else
#define show(a,b,c)
#endif

#define FOURK_BUFF 4096

#define ERR_BASE_PKEY (-5000)
WOLFSSL_TEST_SUBROUTINE int openssl_pkey0_test(void)
{
    int ret = 0;
#if !defined(NO_RSA) && !defined(HAVE_USER_RSA) && !defined(NO_SHA)
    byte*   prvTmp;
    byte*   pubTmp;
    int prvBytes;
    int pubBytes;
    RSA *prvRsa = NULL;
    RSA *pubRsa = NULL;
    EVP_PKEY *prvPkey = NULL;
    EVP_PKEY *pubPkey = NULL;
    EVP_PKEY_CTX *enc = NULL;
    EVP_PKEY_CTX *dec = NULL;

    byte   in[] = TEST_STRING;
    byte   out[256];
    size_t outlen;
    size_t keySz;
    byte   plain[256];
#if !defined(USE_CERT_BUFFERS_1024) && !defined(USE_CERT_BUFFERS_2048)
    XFILE  keyFile;
    XFILE  keypubFile;
    char cliKey[]    = "./certs/client-key.der";
    char cliKeypub[] = "./certs/client-keyPub.der";

#endif

    prvTmp = (byte*)XMALLOC(FOURK_BUFF, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    if (prvTmp == NULL)
        return ERR_BASE_PKEY-1;
    pubTmp = (byte*)XMALLOC(FOURK_BUFF, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    if (pubTmp == NULL) {
        XFREE(prvTmp, HEAP_HINT ,DYNAMIC_TYPE_TMP_BUFFER);
        return ERR_BASE_PKEY-2;
    }

#ifdef USE_CERT_BUFFERS_1024
    XMEMCPY(prvTmp, client_key_der_1024, sizeof_client_key_der_1024);
    prvBytes = sizeof_client_key_der_1024;
    XMEMCPY(pubTmp, client_keypub_der_1024, sizeof_client_keypub_der_1024);
    pubBytes = sizeof_client_keypub_der_1024;
#elif defined(USE_CERT_BUFFERS_2048)
    XMEMCPY(prvTmp, client_key_der_2048, sizeof_client_key_der_2048);
    prvBytes = sizeof_client_key_der_2048;
    XMEMCPY(pubTmp, client_keypub_der_2048, sizeof_client_keypub_der_2048);
    pubBytes = sizeof_client_keypub_der_2048;
#else
    keyFile = XFOPEN(cliKey, "rb");
    if (!keyFile) {
        XFREE(prvTmp, HEAP_HINT ,DYNAMIC_TYPE_TMP_BUFFER);
        XFREE(pubTmp, HEAP_HINT ,DYNAMIC_TYPE_TMP_BUFFER);
        err_sys("can't open ./certs/client-key.der, "
                "Please run from wolfSSL home dir", ERR_BASE_PKEY-3);
        return ERR_BASE_PKEY-3;
    }
    prvBytes = (int)XFREAD(prvTmp, 1, (int)FOURK_BUFF, keyFile);
    XFCLOSE(keyFile);
    keypubFile = XFOPEN(cliKeypub, "rb");
    if (!keypubFile) {
        XFREE(prvTmp, HEAP_HINT ,DYNAMIC_TYPE_TMP_BUFFER);
        XFREE(pubTmp, HEAP_HINT ,DYNAMIC_TYPE_TMP_BUFFER);
        err_sys("can't open ./certs/client-cert.der, "
                "Please run from wolfSSL home dir", -4);
        return ERR_BASE_PKEY-4;
    }
    pubBytes = (int)XFREAD(pubTmp, 1, (int)FOURK_BUFF, keypubFile);
    XFCLOSE(keypubFile);
#endif /* USE_CERT_BUFFERS */

        prvRsa = wolfSSL_RSA_new();
        pubRsa = wolfSSL_RSA_new();
        if((prvRsa == NULL) || (pubRsa == NULL)){
            printf("error with RSA_new\n");
            ret = ERR_BASE_PKEY-10;
            goto openssl_pkey0_test_done;
        }

        ret = wolfSSL_RSA_LoadDer_ex(prvRsa, prvTmp, prvBytes, WOLFSSL_RSA_LOAD_PRIVATE);
        if(ret != SSL_SUCCESS){
            printf("error with RSA_LoadDer_ex\n");
            ret = ERR_BASE_PKEY-11;
            goto openssl_pkey0_test_done;
        }

        ret = wolfSSL_RSA_LoadDer_ex(pubRsa, pubTmp, pubBytes, WOLFSSL_RSA_LOAD_PUBLIC);
        if(ret != SSL_SUCCESS){
            printf("error with RSA_LoadDer_ex\n");
            ret = ERR_BASE_PKEY-12;
            goto openssl_pkey0_test_done;
        }
        keySz = (size_t)RSA_size(pubRsa);

        prvPkey = wolfSSL_EVP_PKEY_new();
        pubPkey = wolfSSL_EVP_PKEY_new();
        if((prvPkey == NULL) || (pubPkey == NULL)){
            printf("error with PKEY_new\n");
            ret = ERR_BASE_PKEY-13;
            goto openssl_pkey0_test_done;
        }
        ret  = wolfSSL_EVP_PKEY_set1_RSA(prvPkey, prvRsa);
        ret += wolfSSL_EVP_PKEY_set1_RSA(pubPkey, pubRsa);
        if(ret != 2){
            printf("error with PKEY_set1_RSA\n");
            ret = ERR_BASE_PKEY-14;
            goto openssl_pkey0_test_done;
        }

        dec = EVP_PKEY_CTX_new(prvPkey, NULL);
        enc = EVP_PKEY_CTX_new(pubPkey, NULL);
        if((dec == NULL)||(enc==NULL)){
            printf("error with EVP_PKEY_CTX_new\n");
            ret = ERR_BASE_PKEY-15;
            goto openssl_pkey0_test_done;
        }

        ret = EVP_PKEY_decrypt_init(dec);
        if (ret != 1) {
            printf("error with decrypt init\n");
            ret = ERR_BASE_PKEY-16;
            goto openssl_pkey0_test_done;
        }
        ret = EVP_PKEY_encrypt_init(enc);
        if (ret != 1) {
            printf("error with encrypt init\n");
            ret = ERR_BASE_PKEY-17;
            goto openssl_pkey0_test_done;
        }
        XMEMSET(out, 0, sizeof(out));
        ret = EVP_PKEY_encrypt(enc, out, &outlen, in, sizeof(in));
        if (ret != 1) {
            printf("error encrypting msg\n");
            ret = ERR_BASE_PKEY-18;
            goto openssl_pkey0_test_done;
        }

        show("encrypted msg", out, outlen);

        XMEMSET(plain, 0, sizeof(plain));
        ret = EVP_PKEY_decrypt(dec, plain, &outlen, out, keySz);
        if (ret != 1) {
            printf("error decrypting msg\n");
            ret = ERR_BASE_PKEY-19;
            goto openssl_pkey0_test_done;
        }
        show("decrypted msg", plain, outlen);

        /* RSA_PKCS1_OAEP_PADDING test */
        ret = EVP_PKEY_decrypt_init(dec);
        if (ret != 1) {
            printf("error with decrypt init\n");
            ret = ERR_BASE_PKEY-30;
            goto openssl_pkey0_test_done;
        }
        ret = EVP_PKEY_encrypt_init(enc);
        if (ret != 1) {
            printf("error with encrypt init\n");
            ret = ERR_BASE_PKEY-31;
            goto openssl_pkey0_test_done;
        }

        if (EVP_PKEY_CTX_set_rsa_padding(dec, RSA_PKCS1_PADDING) <= 0) {
            printf("first set rsa padding error\n");
            ret = ERR_BASE_PKEY-32;
            goto openssl_pkey0_test_done;
        }

#ifndef HAVE_FIPS
        if (EVP_PKEY_CTX_set_rsa_padding(dec, RSA_PKCS1_OAEP_PADDING) <= 0){
            printf("second set rsa padding error\n");
            ret = ERR_BASE_PKEY-33;
            goto openssl_pkey0_test_done;
        }

        if (EVP_PKEY_CTX_set_rsa_padding(enc, RSA_PKCS1_OAEP_PADDING) <= 0) {
            printf("third set rsa padding error\n");
            ret = ERR_BASE_PKEY-34;
            goto openssl_pkey0_test_done;
        }
#endif

        XMEMSET(out, 0, sizeof(out));
        ret = EVP_PKEY_encrypt(enc, out, &outlen, in, sizeof(in));
        if (ret != 1) {
            printf("error encrypting msg\n");
            ret = ERR_BASE_PKEY-35;
            goto openssl_pkey0_test_done;
        }

        show("encrypted msg", out, outlen);

        XMEMSET(plain, 0, sizeof(plain));
        ret = EVP_PKEY_decrypt(dec, plain, &outlen, out, keySz);
        if (ret != 1) {
            printf("error decrypting msg\n");
            ret = ERR_BASE_PKEY-36;
            goto openssl_pkey0_test_done;
        }

        show("decrypted msg", plain, outlen);

        ret = 0; /* made it to this point without error then set success */
openssl_pkey0_test_done:

        wolfSSL_RSA_free(prvRsa);
        wolfSSL_RSA_free(pubRsa);
        EVP_PKEY_free(pubPkey);
        EVP_PKEY_free(prvPkey);
        EVP_PKEY_CTX_free(dec);
        EVP_PKEY_CTX_free(enc);
        XFREE(prvTmp, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
        XFREE(pubTmp, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
#endif /* NO_RSA */

        return ret;
}


WOLFSSL_TEST_SUBROUTINE int openssl_pkey1_test(void)
{
    int ret = 0;
#if !defined(NO_FILESYSTEM) && !defined(NO_RSA) && !defined(HAVE_USER_RSA) && \
    !defined(NO_SHA)
    EVP_PKEY_CTX* dec = NULL;
    EVP_PKEY_CTX* enc = NULL;
    EVP_PKEY* pubKey  = NULL;
    EVP_PKEY* prvKey  = NULL;
    X509* x509 = NULL;

    WOLFSSL_SMALL_STACK_STATIC const unsigned char msg[] = "sugar slapped";
    const unsigned char* clikey;
    long cliKeySz;
    size_t outlen;
    int keyLenBits = 2048;
#ifdef WOLFSSL_SMALL_STACK
    unsigned char *tmp = (unsigned char *)XMALLOC(FOURK_BUF, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    unsigned char *cipher = (unsigned char *)XMALLOC(RSA_TEST_BYTES, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    unsigned char *plain = (unsigned char *)XMALLOC(RSA_TEST_BYTES, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);

    if ((tmp == NULL) ||
        (cipher == NULL) ||
        (plain == NULL)) {
        ret = -9015;
        goto openssl_pkey1_test_done;
    }
#else
    unsigned char tmp[FOURK_BUF];
    unsigned char cipher[RSA_TEST_BYTES];
    unsigned char plain[RSA_TEST_BYTES];
#endif

#if defined(USE_CERT_BUFFERS_1024)
    XMEMCPY(tmp, client_key_der_1024, sizeof_client_key_der_1024);
    cliKeySz = (long)sizeof_client_key_der_1024;

    x509 = wolfSSL_X509_load_certificate_buffer(client_cert_der_1024,
            sizeof_client_cert_der_1024, SSL_FILETYPE_ASN1);
    keyLenBits = 1024;
#elif defined(USE_CERT_BUFFERS_2048)
    XMEMCPY(tmp, client_key_der_2048, sizeof_client_key_der_2048);
    cliKeySz = (long)sizeof_client_key_der_2048;

    x509 = wolfSSL_X509_load_certificate_buffer(client_cert_der_2048,
            sizeof_client_cert_der_2048, SSL_FILETYPE_ASN1);
#elif defined(USE_CERT_BUFFERS_3072)
    XMEMCPY(tmp, client_key_der_3072, sizeof_client_key_der_3072);
    cliKeySz = (long)sizeof_client_key_der_3072;

    x509 = wolfSSL_X509_load_certificate_buffer(client_cert_der_3072,
            sizeof_client_cert_der_3072, SSL_FILETYPE_ASN1);
    keyLenBits = 3072;
#elif defined(USE_CERT_BUFFERS_4096)
    XMEMCPY(tmp, client_key_der_4096, sizeof_client_key_der_4096);
    cliKeySz = (long)sizeof_client_key_der_4096;

    x509 = wolfSSL_X509_load_certificate_buffer(client_cert_der_4096,
            sizeof_client_cert_der_4096, SSL_FILETYPE_ASN1);
    keyLenBits = 4096;
#else
    {
        XFILE f;

        f = XFOPEN(clientKey, "rb");

        if (!f) {
            err_sys("can't open ./certs/client-key.der, "
                    "Please run from wolfSSL home dir", -41);
            ret = -9000;
            goto openssl_pkey1_test_done;
        }

        cliKeySz = (long)XFREAD(tmp, 1, FOURK_BUF, f);
        XFCLOSE(f);
    }

    /* using existing wolfSSL api to get public and private key */
    x509 = wolfSSL_X509_load_certificate_file(clientCert, SSL_FILETYPE_ASN1);
#endif /* USE_CERT_BUFFERS */
    clikey = tmp;

    if ((prvKey = EVP_PKEY_new()) == NULL) {
        ret = -9001;
        goto openssl_pkey1_test_done;
    }
    EVP_PKEY_free(prvKey);
    prvKey = NULL;

    if (x509 == NULL) {
        ret = -9002;
        goto openssl_pkey1_test_done;
    }

    pubKey = X509_get_pubkey(x509);
    if (pubKey == NULL) {
        ret = -9003;
        goto openssl_pkey1_test_done;
    }

    prvKey = d2i_PrivateKey(EVP_PKEY_RSA, NULL, &clikey, cliKeySz);
    if (prvKey == NULL) {
        ret = -9004;
        goto openssl_pkey1_test_done;
    }

    /* phase 2 API to create EVP_PKEY_CTX and encrypt/decrypt */
    if (EVP_PKEY_bits(prvKey) != keyLenBits) {
        ret = -9005;
        goto openssl_pkey1_test_done;
    }

    if (EVP_PKEY_size(prvKey) != keyLenBits/8) {
        ret = -9006;
        goto openssl_pkey1_test_done;
    }

    dec = EVP_PKEY_CTX_new(prvKey, NULL);
    enc = EVP_PKEY_CTX_new(pubKey, NULL);
    if (dec == NULL || enc == NULL) {
        ret = -9007;
        goto openssl_pkey1_test_done;
    }

    if (EVP_PKEY_decrypt_init(dec) != 1) {
        ret = -9008;
        goto openssl_pkey1_test_done;
    }

    if (EVP_PKEY_encrypt_init(enc) != 1) {
        ret = -9009;
        goto openssl_pkey1_test_done;
    }

    if (EVP_PKEY_CTX_set_rsa_padding(dec, RSA_PKCS1_PADDING) <= 0) {
        ret = -9010;
        goto openssl_pkey1_test_done;
    }

#ifndef HAVE_FIPS
    if (EVP_PKEY_CTX_set_rsa_padding(dec, RSA_PKCS1_OAEP_PADDING) <= 0){
        ret = -9011;
        goto openssl_pkey1_test_done;
    }

    if (EVP_PKEY_CTX_set_rsa_padding(enc, RSA_PKCS1_OAEP_PADDING) <= 0) {
        ret = -9012;
        goto openssl_pkey1_test_done;
    }
#endif

    XMEMSET(cipher, 0, RSA_TEST_BYTES);
    outlen = keyLenBits/8;
    if (EVP_PKEY_encrypt(enc, cipher, &outlen, msg, sizeof(msg)) < 0) {
        ret = -9013;
        goto openssl_pkey1_test_done;
    }

    XMEMSET(plain, 0, RSA_TEST_BYTES);
    if (EVP_PKEY_decrypt(dec, plain, &outlen, cipher, outlen) != 1) {
        ret = -9014;
        goto openssl_pkey1_test_done;
    }

openssl_pkey1_test_done:
    if (pubKey != NULL) {
        EVP_PKEY_free(pubKey);
    }
    if (prvKey != NULL) {
        EVP_PKEY_free(prvKey);
    }
    if (dec != NULL) {
        EVP_PKEY_CTX_free(dec);
    }
    if (enc != NULL) {
        EVP_PKEY_CTX_free(enc);
    }
    if (x509 != NULL) {
        X509_free(x509);
    }

#ifdef WOLFSSL_SMALL_STACK
    if (tmp != NULL)
        XFREE(tmp, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    if (cipher != NULL)
        XFREE(cipher, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    if (plain != NULL)
        XFREE(plain, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
#endif
#endif

    return ret;
}


#define ERR_BASE_EVPSIG (-5100)

WOLFSSL_TEST_SUBROUTINE int openssl_evpSig_test(void)
{
#if !defined(NO_RSA) && !defined(NO_SHA) && !defined(HAVE_USER_RSA)
    byte*   prvTmp;
    byte*   pubTmp;
    int prvBytes;
    int pubBytes;
    RSA *prvRsa;
    RSA *pubRsa;
    EVP_PKEY *prvPkey;
    EVP_PKEY *pubPkey;

    EVP_MD_CTX* sign;
    EVP_MD_CTX* verf;
    char msg[] = "see spot run";
    unsigned char sig[256];
    unsigned int sigSz;
    const void* pt;
    unsigned int count;
    int ret, ret1, ret2;

    #if !defined(USE_CERT_BUFFERS_1024) && !defined(USE_CERT_BUFFERS_2048)
    XFILE   keyFile;
    XFILE   keypubFile;
    char cliKey[]    = "./certs/client-key.der";
    char cliKeypub[] = "./certs/client-keyPub.der";
    #endif

    prvTmp = (byte*)XMALLOC(FOURK_BUFF, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    if (prvTmp == NULL)
        return ERR_BASE_EVPSIG-1;
    pubTmp = (byte*)XMALLOC(FOURK_BUFF, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    if (pubTmp == NULL) {
        XFREE(prvTmp, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
        return ERR_BASE_EVPSIG-2;
    }

#ifdef USE_CERT_BUFFERS_1024
    XMEMCPY(prvTmp, client_key_der_1024, sizeof_client_key_der_1024);
    prvBytes = sizeof_client_key_der_1024;
    XMEMCPY(pubTmp, client_keypub_der_1024, sizeof_client_keypub_der_1024);
    pubBytes = sizeof_client_keypub_der_1024;
#elif defined(USE_CERT_BUFFERS_2048)
    XMEMCPY(prvTmp, client_key_der_2048, sizeof_client_key_der_2048);
    prvBytes = sizeof_client_key_der_2048;
    XMEMCPY(pubTmp, client_keypub_der_2048, sizeof_client_keypub_der_2048);
    pubBytes = sizeof_client_keypub_der_2048;
#else
    keyFile = XFOPEN(cliKey, "rb");
    if (!keyFile) {
        XFREE(pubTmp, HEAP_HINT ,DYNAMIC_TYPE_TMP_BUFFER);
        XFREE(prvTmp, HEAP_HINT ,DYNAMIC_TYPE_TMP_BUFFER);
        err_sys("can't open ./certs/client-key.der, "
                "Please run from wolfSSL home dir", -40);
        return ERR_BASE_EVPSIG-3;
    }
    prvBytes = (int)XFREAD(prvTmp, 1, (int)FOURK_BUFF, keyFile);
    XFCLOSE(keyFile);
    keypubFile = XFOPEN(cliKeypub, "rb");
    if (!keypubFile) {
        XFREE(pubTmp, HEAP_HINT ,DYNAMIC_TYPE_TMP_BUFFER);
        XFREE(prvTmp, HEAP_HINT ,DYNAMIC_TYPE_TMP_BUFFER);
        err_sys("can't open ./certs/client-cert.der, "
                "Please run from wolfSSL home dir", -41);
        return ERR_BASE_EVPSIG-4;
    }
    pubBytes = (int)XFREAD(pubTmp, 1, (int)FOURK_BUFF, keypubFile);
    XFCLOSE(keypubFile);
    #endif /* USE_CERT_BUFFERS */

    prvRsa = wolfSSL_RSA_new();
    pubRsa = wolfSSL_RSA_new();
    if((prvRsa == NULL) || (pubRsa == NULL)){
        XFREE(pubTmp, HEAP_HINT ,DYNAMIC_TYPE_TMP_BUFFER);
        XFREE(prvTmp, HEAP_HINT ,DYNAMIC_TYPE_TMP_BUFFER);
        err_sys("ERROR with RSA_new", -9100);
        return ERR_BASE_EVPSIG-5;
    }

    ret1 = wolfSSL_RSA_LoadDer_ex(prvRsa, prvTmp, prvBytes, WOLFSSL_RSA_LOAD_PRIVATE);
    ret2 = wolfSSL_RSA_LoadDer_ex(pubRsa, pubTmp, pubBytes, WOLFSSL_RSA_LOAD_PUBLIC);
    if((ret1 != SSL_SUCCESS) || (ret2 != SSL_SUCCESS)){
      printf("error with RSA_LoadDer_ex\n");
      return ERR_BASE_EVPSIG-6;
    }

    prvPkey = wolfSSL_EVP_PKEY_new();
    pubPkey = wolfSSL_EVP_PKEY_new();
    if((prvPkey == NULL) || (pubPkey == NULL)){
        XFREE(pubTmp, HEAP_HINT ,DYNAMIC_TYPE_TMP_BUFFER);
        XFREE(prvTmp, HEAP_HINT ,DYNAMIC_TYPE_TMP_BUFFER);
        printf("error with KEY_new\n");
        return ERR_BASE_EVPSIG-7;
    }
    ret1  = wolfSSL_EVP_PKEY_set1_RSA(prvPkey, prvRsa);
    ret2  = wolfSSL_EVP_PKEY_set1_RSA(pubPkey, pubRsa);
    if((ret1 != 1) || (ret2 != 1)){
        XFREE(pubTmp, HEAP_HINT ,DYNAMIC_TYPE_TMP_BUFFER);
        XFREE(prvTmp, HEAP_HINT ,DYNAMIC_TYPE_TMP_BUFFER);
        printf("error with EVP_PKEY_set1_RSA\n");
        return ERR_BASE_EVPSIG-8;
    }

    /****************** sign and verify *******************/
    sign = EVP_MD_CTX_create();
    verf = EVP_MD_CTX_create();
    if((sign == NULL)||(verf == NULL)){
        printf("error with EVP_MD_CTX_create\n");
        EVP_MD_CTX_destroy(sign);
        EVP_MD_CTX_destroy(verf);
        return ERR_BASE_EVPSIG-10;
    }

    ret = EVP_SignInit(sign, EVP_sha1());
    if (ret != SSL_SUCCESS){
        printf("error with EVP_SignInit\n");
        EVP_MD_CTX_destroy(sign);
        EVP_MD_CTX_destroy(verf);
        return ERR_BASE_EVPSIG-11;
    }

    count = sizeof(msg);
    show("message = ", (char *)msg, count);

    /* sign */
    XMEMSET(sig, 0, sizeof(sig));
    pt = (const void*)msg;
    ret1 = EVP_SignUpdate(sign, pt, count);
    ret2 = EVP_SignFinal(sign, sig, &sigSz, prvPkey);
    if((ret1 != SSL_SUCCESS) || (ret2 != SSL_SUCCESS)){
        XFREE(pubTmp, HEAP_HINT ,DYNAMIC_TYPE_TMP_BUFFER);
        XFREE(prvTmp, HEAP_HINT ,DYNAMIC_TYPE_TMP_BUFFER);
        EVP_MD_CTX_destroy(sign);
        EVP_MD_CTX_destroy(verf);
        printf("error with EVP_MD_CTX_create\n");
        return ERR_BASE_EVPSIG-12;
    }
    show("signature = ", (char *)sig, sigSz);

    /* verify */
    pt = (const void*)msg;
    ret1 = EVP_VerifyInit(verf, EVP_sha1());
    ret2 = EVP_VerifyUpdate(verf, pt, count);
    if((ret1 != SSL_SUCCESS) || (ret2 != SSL_SUCCESS)){
        XFREE(pubTmp, HEAP_HINT ,DYNAMIC_TYPE_TMP_BUFFER);
        XFREE(prvTmp, HEAP_HINT ,DYNAMIC_TYPE_TMP_BUFFER);
        EVP_MD_CTX_destroy(sign);
        EVP_MD_CTX_destroy(verf);
        printf("error with EVP_Verify\n");
        return ERR_BASE_EVPSIG-13;
    }
    if (EVP_VerifyFinal(verf, sig, sigSz, pubPkey) != 1) {
        XFREE(pubTmp, HEAP_HINT ,DYNAMIC_TYPE_TMP_BUFFER);
        XFREE(prvTmp, HEAP_HINT ,DYNAMIC_TYPE_TMP_BUFFER);
        EVP_MD_CTX_destroy(sign);
        EVP_MD_CTX_destroy(verf);
        printf("error with EVP_VerifyFinal\n");
        return ERR_BASE_EVPSIG-14;
    }

    /* expect fail without update */
    EVP_VerifyInit(verf, EVP_sha1());
    if (EVP_VerifyFinal(verf, sig, sigSz, pubPkey) == 1) {
        XFREE(pubTmp, HEAP_HINT ,DYNAMIC_TYPE_TMP_BUFFER);
        XFREE(prvTmp, HEAP_HINT ,DYNAMIC_TYPE_TMP_BUFFER);
        EVP_MD_CTX_destroy(sign);
        EVP_MD_CTX_destroy(verf);
        printf("EVP_VerifyInit without update not detected\n");
        return ERR_BASE_EVPSIG-15;
    }

    XFREE(pubTmp, HEAP_HINT ,DYNAMIC_TYPE_TMP_BUFFER);
    XFREE(prvTmp, HEAP_HINT ,DYNAMIC_TYPE_TMP_BUFFER);
    EVP_MD_CTX_destroy(sign);
    EVP_MD_CTX_destroy(verf);

    wolfSSL_RSA_free(prvRsa);
    wolfSSL_RSA_free(pubRsa);
    EVP_PKEY_free(pubPkey);
    EVP_PKEY_free(prvPkey);

#endif /* NO_RSA */
    return 0;
}
#endif /* OPENSSL_EXTRA */


#ifndef NO_PWDBASED
#ifdef HAVE_SCRYPT
/* Test vectors taken from RFC 7914: scrypt PBKDF - Section 12. */
WOLFSSL_TEST_SUBROUTINE int scrypt_test(void)
{
#ifdef HAVE_FIPS
    /* RFC 7914 test vector keys are too short for FIPS. */
#else
    int   ret;
    byte  derived[64];

    WOLFSSL_SMALL_STACK_STATIC const byte verify1[] = {
        0x77, 0xd6, 0x57, 0x62, 0x38, 0x65, 0x7b, 0x20,
        0x3b, 0x19, 0xca, 0x42, 0xc1, 0x8a, 0x04, 0x97,
        0xf1, 0x6b, 0x48, 0x44, 0xe3, 0x07, 0x4a, 0xe8,
        0xdf, 0xdf, 0xfa, 0x3f, 0xed, 0xe2, 0x14, 0x42,
        0xfc, 0xd0, 0x06, 0x9d, 0xed, 0x09, 0x48, 0xf8,
        0x32, 0x6a, 0x75, 0x3a, 0x0f, 0xc8, 0x1f, 0x17,
        0xe8, 0xd3, 0xe0, 0xfb, 0x2e, 0x0d, 0x36, 0x28,
        0xcf, 0x35, 0xe2, 0x0c, 0x38, 0xd1, 0x89, 0x06
    };
    WOLFSSL_SMALL_STACK_STATIC const byte verify2[] = {
        0xfd, 0xba, 0xbe, 0x1c, 0x9d, 0x34, 0x72, 0x00,
        0x78, 0x56, 0xe7, 0x19, 0x0d, 0x01, 0xe9, 0xfe,
        0x7c, 0x6a, 0xd7, 0xcb, 0xc8, 0x23, 0x78, 0x30,
        0xe7, 0x73, 0x76, 0x63, 0x4b, 0x37, 0x31, 0x62,
        0x2e, 0xaf, 0x30, 0xd9, 0x2e, 0x22, 0xa3, 0x88,
        0x6f, 0xf1, 0x09, 0x27, 0x9d, 0x98, 0x30, 0xda,
        0xc7, 0x27, 0xaf, 0xb9, 0x4a, 0x83, 0xee, 0x6d,
        0x83, 0x60, 0xcb, 0xdf, 0xa2, 0xcc, 0x06, 0x40
    };
#if !defined(BENCH_EMBEDDED) && !defined(WOLFSSL_LINUXKM) && !defined(HAVE_INTEL_QA)
    WOLFSSL_SMALL_STACK_STATIC const byte verify3[] = {
        0x70, 0x23, 0xbd, 0xcb, 0x3a, 0xfd, 0x73, 0x48,
        0x46, 0x1c, 0x06, 0xcd, 0x81, 0xfd, 0x38, 0xeb,
        0xfd, 0xa8, 0xfb, 0xba, 0x90, 0x4f, 0x8e, 0x3e,
        0xa9, 0xb5, 0x43, 0xf6, 0x54, 0x5d, 0xa1, 0xf2,
        0xd5, 0x43, 0x29, 0x55, 0x61, 0x3f, 0x0f, 0xcf,
        0x62, 0xd4, 0x97, 0x05, 0x24, 0x2a, 0x9a, 0xf9,
        0xe6, 0x1e, 0x85, 0xdc, 0x0d, 0x65, 0x1e, 0x40,
        0xdf, 0xcf, 0x01, 0x7b, 0x45, 0x57, 0x58, 0x87
    };
#endif
#ifdef SCRYPT_TEST_ALL
    /* Test case is very slow.
     * Use for confirmation after code change or new platform.
     */
    WOLFSSL_SMALL_STACK_STATIC const byte verify4[] = {
        0x21, 0x01, 0xcb, 0x9b, 0x6a, 0x51, 0x1a, 0xae,
        0xad, 0xdb, 0xbe, 0x09, 0xcf, 0x70, 0xf8, 0x81,
        0xec, 0x56, 0x8d, 0x57, 0x4a, 0x2f, 0xfd, 0x4d,
        0xab, 0xe5, 0xee, 0x98, 0x20, 0xad, 0xaa, 0x47,
        0x8e, 0x56, 0xfd, 0x8f, 0x4b, 0xa5, 0xd0, 0x9f,
        0xfa, 0x1c, 0x6d, 0x92, 0x7c, 0x40, 0xf4, 0xc3,
        0x37, 0x30, 0x40, 0x49, 0xe8, 0xa9, 0x52, 0xfb,
        0xcb, 0xf4, 0x5c, 0x6f, 0xa7, 0x7a, 0x41, 0xa4
    };
#endif

    ret = wc_scrypt(derived, NULL, 0, NULL, 0, 4, 1, 1, sizeof(verify1));
    if (ret != 0)
        return -9200;
    if (XMEMCMP(derived, verify1, sizeof(verify1)) != 0)
        return -9201;

    ret = wc_scrypt(derived, (byte*)"password", 8, (byte*)"NaCl", 4, 10, 8, 16,
                    sizeof(verify2));
    if (ret != 0)
        return -9202;
    if (XMEMCMP(derived, verify2, sizeof(verify2)) != 0)
        return -9203;

    /* Test case with parallel overflowing */
    ret = wc_scrypt(derived, (byte*)"password", 16, (byte*)"NaCl", 16, 2, 4, 8388608,
                    sizeof(verify2));
    if (ret != BAD_FUNC_ARG)
        return -9210;

    /* Don't run these test on embedded, since they use large mallocs */
#if !defined(BENCH_EMBEDDED) && !defined(WOLFSSL_LINUXKM) && !defined(HAVE_INTEL_QA)
    ret = wc_scrypt(derived, (byte*)"pleaseletmein", 13,
                    (byte*)"SodiumChloride", 14, 14, 8, 1, sizeof(verify3));
    if (ret != 0)
        return -9204;
    if (XMEMCMP(derived, verify3, sizeof(verify3)) != 0)
        return -9205;

#ifdef SCRYPT_TEST_ALL
    ret = wc_scrypt(derived, (byte*)"pleaseletmein", 13,
                    (byte*)"SodiumChloride", 14, 20, 8, 1, sizeof(verify4));
    if (ret != 0)
        return -9206;
    if (XMEMCMP(derived, verify4, sizeof(verify4)) != 0)
        return -9207;
#endif
#endif /* !BENCH_EMBEDDED && !defined(WOLFSSL_LINUXKM) && !HAVE_INTEL_QA */

    ret = wc_scrypt_ex(derived, (byte*)"password", 8, (byte*)"NaCl", 4, 1<<10,
                       8, 16, sizeof(verify2));
    if (ret != 0)
        return -9208;
    if (XMEMCMP(derived, verify2, sizeof(verify2)) != 0)
        return -9209;

#endif /* !HAVE_FIPS */

    return 0;
}
#endif

#ifdef HAVE_PKCS12
WOLFSSL_TEST_SUBROUTINE int pkcs12_test(void)
{
    WOLFSSL_SMALL_STACK_STATIC const byte passwd[] = { 0x00, 0x73, 0x00, 0x6d, 0x00, 0x65, 0x00, 0x67,
                            0x00, 0x00 };
    WOLFSSL_SMALL_STACK_STATIC const byte salt[] =   { 0x0a, 0x58, 0xCF, 0x64, 0x53, 0x0d, 0x82, 0x3f };

    WOLFSSL_SMALL_STACK_STATIC const byte passwd2[] = { 0x00, 0x71, 0x00, 0x75, 0x00, 0x65, 0x00, 0x65,
                             0x00, 0x67, 0x00, 0x00 };
    WOLFSSL_SMALL_STACK_STATIC const byte salt2[] =   { 0x16, 0x82, 0xC0, 0xfC, 0x5b, 0x3f, 0x7e, 0xc5 };
    byte  derived[64];

    WOLFSSL_SMALL_STACK_STATIC const byte verify[] = {
        0x27, 0xE9, 0x0D, 0x7E, 0xD5, 0xA1, 0xC4, 0x11,
        0xBA, 0x87, 0x8B, 0xC0, 0x90, 0xF5, 0xCE, 0xBE,
        0x5E, 0x9D, 0x5F, 0xE3, 0xD6, 0x2B, 0x73, 0xAA
    };

    WOLFSSL_SMALL_STACK_STATIC const byte verify2[] = {
        0x90, 0x1B, 0x49, 0x70, 0xF0, 0x94, 0xF0, 0xF8,
        0x45, 0xC0, 0xF3, 0xF3, 0x13, 0x59, 0x18, 0x6A,
        0x35, 0xE3, 0x67, 0xFE, 0xD3, 0x21, 0xFD, 0x7C
    };

    int id         =  1;
    int kLen       = 24;
    int iterations =  1;
    int ret = wc_PKCS12_PBKDF(derived, passwd, sizeof(passwd), salt, 8,
                                                  iterations, kLen, WC_SHA256, id);

    if (ret < 0)
        return -9300;

    if (XMEMCMP(derived, verify, kLen) != 0)
        return -9301;

    iterations = 1000;
    ret = wc_PKCS12_PBKDF(derived, passwd2, sizeof(passwd2), salt2, 8,
                                                  iterations, kLen, WC_SHA256, id);
    if (ret < 0)
        return -9302;

    ret = wc_PKCS12_PBKDF_ex(derived, passwd2, sizeof(passwd2), salt2, 8,
                                       iterations, kLen, WC_SHA256, id, HEAP_HINT);
    if (ret < 0)
        return -9303;

    if (XMEMCMP(derived, verify2, 24) != 0)
        return -9304;

    return 0;
}
#endif /* HAVE_PKCS12 */

#if defined(HAVE_PBKDF2) && !defined(NO_SHA256) && !defined(NO_HMAC)
WOLFSSL_TEST_SUBROUTINE int pbkdf2_test(void)
{
    char passwd[] = "passwordpassword";
    WOLFSSL_SMALL_STACK_STATIC const byte salt[] = { 0x78, 0x57, 0x8E, 0x5a, 0x5d, 0x63, 0xcb, 0x06 };
    int   iterations = 2048;
    int   kLen = 24;
    byte  derived[64];

    WOLFSSL_SMALL_STACK_STATIC const byte verify[] = {
        0x43, 0x6d, 0xb5, 0xe8, 0xd0, 0xfb, 0x3f, 0x35, 0x42, 0x48, 0x39, 0xbc,
        0x2d, 0xd4, 0xf9, 0x37, 0xd4, 0x95, 0x16, 0xa7, 0x2a, 0x9a, 0x21, 0xd1
    };

    int ret = wc_PBKDF2_ex(derived, (byte*)passwd, (int)XSTRLEN(passwd), salt,
              (int)sizeof(salt), iterations, kLen, WC_SHA256, HEAP_HINT, devId);
    if (ret != 0)
        return ret;

    if (XMEMCMP(derived, verify, sizeof(verify)) != 0)
        return -9400;

    return 0;

}
#endif /* HAVE_PBKDF2 && !NO_SHA256 && !NO_HMAC */

#if defined(HAVE_PBKDF1) && !defined(NO_SHA)
WOLFSSL_TEST_SUBROUTINE int pbkdf1_test(void)
{
    char passwd[] = "password";
    WOLFSSL_SMALL_STACK_STATIC const byte salt[] = { 0x78, 0x57, 0x8E, 0x5a, 0x5d, 0x63, 0xcb, 0x06 };
    int   iterations = 1000;
    int   kLen = 16;
    byte  derived[16];

    WOLFSSL_SMALL_STACK_STATIC const byte verify[] = {
        0xDC, 0x19, 0x84, 0x7E, 0x05, 0xC6, 0x4D, 0x2F,
        0xAF, 0x10, 0xEB, 0xFB, 0x4A, 0x3D, 0x2A, 0x20
    };

    int ret = wc_PBKDF1_ex(derived, kLen, NULL, 0, (byte*)passwd,
        (int)XSTRLEN(passwd), salt, (int)sizeof(salt), iterations, WC_SHA,
        HEAP_HINT);
    if (ret != 0)
        return ret;

    if (XMEMCMP(derived, verify, sizeof(verify)) != 0)
        return -9500;

    return 0;
}
#endif /* HAVE_PBKDF2 && !NO_SHA */

WOLFSSL_TEST_SUBROUTINE int pwdbased_test(void)
{
   int ret = 0;

#if defined(HAVE_PBKDF1) && !defined(NO_SHA)
   ret = pbkdf1_test();
   if (ret != 0)
      return ret;
#endif
#if defined(HAVE_PBKDF2) && !defined(NO_SHA256) && !defined(NO_HMAC)
   ret = pbkdf2_test();
   if (ret != 0)
      return ret;
#endif
#ifdef HAVE_PKCS12
   ret = pkcs12_test();
   if (ret != 0)
      return ret;
#endif
#ifdef HAVE_SCRYPT
   ret = scrypt_test();
   if (ret != 0)
      return ret;
#endif
   return ret;
}

#endif /* NO_PWDBASED */

#if defined(HAVE_HKDF) && !defined(NO_HMAC)

/* hkdf_test has issue with WOLFSSL_TEST_SUBROUTINE set on Xilinx with afalg */
static int hkdf_test(void)
{
    int ret = 0;

#if !defined(NO_SHA) || !defined(NO_SHA256)
    int L;
    byte okm1[42];
    byte ikm1[22] = { 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b,
                      0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b,
                      0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b };
#ifndef HAVE_FIPS
    byte salt1[13] ={ 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
                      0x08, 0x09, 0x0a, 0x0b, 0x0c };
    byte info1[10] ={ 0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7,
                      0xf8, 0xf9 };
#endif
#ifndef NO_SHA
    byte res1[42] = { 0x0a, 0xc1, 0xaf, 0x70, 0x02, 0xb3, 0xd7, 0x61,
                      0xd1, 0xe5, 0x52, 0x98, 0xda, 0x9d, 0x05, 0x06,
                      0xb9, 0xae, 0x52, 0x05, 0x72, 0x20, 0xa3, 0x06,
                      0xe0, 0x7b, 0x6b, 0x87, 0xe8, 0xdf, 0x21, 0xd0,
                      0xea, 0x00, 0x03, 0x3d, 0xe0, 0x39, 0x84, 0xd3,
                      0x49, 0x18 };
#ifndef HAVE_FIPS
    byte res2[42] = { 0x08, 0x5a, 0x01, 0xea, 0x1b, 0x10, 0xf3, 0x69,
                      0x33, 0x06, 0x8b, 0x56, 0xef, 0xa5, 0xad, 0x81,
                      0xa4, 0xf1, 0x4b, 0x82, 0x2f, 0x5b, 0x09, 0x15,
                      0x68, 0xa9, 0xcd, 0xd4, 0xf1, 0x55, 0xfd, 0xa2,
                      0xc2, 0x2e, 0x42, 0x24, 0x78, 0xd3, 0x05, 0xf3,
                      0xf8, 0x96 };
#endif
#endif /* !NO_SHA */
#ifndef NO_SHA256
    byte res3[42] = { 0x8d, 0xa4, 0xe7, 0x75, 0xa5, 0x63, 0xc1, 0x8f,
                      0x71, 0x5f, 0x80, 0x2a, 0x06, 0x3c, 0x5a, 0x31,
                      0xb8, 0xa1, 0x1f, 0x5c, 0x5e, 0xe1, 0x87, 0x9e,
                      0xc3, 0x45, 0x4e, 0x5f, 0x3c, 0x73, 0x8d, 0x2d,
                      0x9d, 0x20, 0x13, 0x95, 0xfa, 0xa4, 0xb6, 0x1a,
                      0x96, 0xc8 };
#ifndef HAVE_FIPS
    byte res4[42] = { 0x3c, 0xb2, 0x5f, 0x25, 0xfa, 0xac, 0xd5, 0x7a,
                      0x90, 0x43, 0x4f, 0x64, 0xd0, 0x36, 0x2f, 0x2a,
                      0x2d, 0x2d, 0x0a, 0x90, 0xcf, 0x1a, 0x5a, 0x4c,
                      0x5d, 0xb0, 0x2d, 0x56, 0xec, 0xc4, 0xc5, 0xbf,
                      0x34, 0x00, 0x72, 0x08, 0xd5, 0xb8, 0x87, 0x18,
                      0x58, 0x65 };
#endif
#endif /* !NO_SHA256 */

    XMEMSET(okm1, 0, sizeof(okm1));
    L = (int)sizeof(okm1);

#ifndef NO_SHA
    ret = wc_HKDF(WC_SHA, ikm1, (word32)sizeof(ikm1), NULL, 0, NULL, 0,
        okm1, L);
    if (ret != 0)
        return -9700;

    if (XMEMCMP(okm1, res1, L) != 0)
        return -9701;

#ifndef HAVE_FIPS
    /* fips can't have key size under 14 bytes, salt is key too */
    L = (int)sizeof(okm1);
    ret = wc_HKDF(WC_SHA, ikm1, 11, salt1, (word32)sizeof(salt1),
        info1, (word32)sizeof(info1), okm1, L);
    if (ret != 0)
        return -9702;

    if (XMEMCMP(okm1, res2, L) != 0)
        return -9703;
#endif /* HAVE_FIPS */
#endif /* !NO_SHA */

#ifndef NO_SHA256
    ret = wc_HKDF(WC_SHA256, ikm1, (word32)sizeof(ikm1), NULL, 0, NULL, 0,
        okm1, L);
    if (ret != 0)
        return -9704;

    if (XMEMCMP(okm1, res3, L) != 0)
        return -9705;

#ifndef HAVE_FIPS
    /* fips can't have key size under 14 bytes, salt is key too */
    ret = wc_HKDF(WC_SHA256, ikm1, (word32)sizeof(ikm1),
        salt1, (word32)sizeof(salt1), info1, (word32)sizeof(info1), okm1, L);
    if (ret != 0)
        return -9706;

    if (XMEMCMP(okm1, res4, L) != 0)
        return -9707;
#endif /* HAVE_FIPS */
#endif /* !NO_SHA256 */
#endif /* !NO_SHA || !NO_SHA256 */

    return ret;
}

#endif /* HAVE_HKDF */


#ifdef WOLFSSL_WOLFSSH

typedef struct {
    byte hashId;
    byte keyId;
    const byte* k;
    word32 kSz;
    const byte* h;
    word32 hSz;
    const byte* sessionId;
    word32 sessionIdSz;
    const byte* expectedKey;
    word32 expectedKeySz;
} SshKdfTestVector;


/** Test Vector Set #3: SHA-256 **/
static const byte sshKdfTvSet3k[] = {
    0x6A, 0xC3, 0x82, 0xEA, 0xAC, 0xA0, 0x93, 0xE1,
    0x25, 0xE2, 0x5C, 0x24, 0xBE, 0xBC, 0x84, 0x64,
    0x0C, 0x11, 0x98, 0x75, 0x07, 0x34, 0x4B, 0x5C,
    0x73, 0x9C, 0xEB, 0x84, 0xA9, 0xE0, 0xB2, 0x22,
    0xB9, 0xA8, 0xB5, 0x1C, 0x83, 0x9E, 0x5E, 0xBE,
    0x49, 0xCF, 0xAD, 0xBF, 0xB3, 0x95, 0x99, 0x76,
    0x4E, 0xD5, 0x22, 0x09, 0x9D, 0xC9, 0x12, 0x75,
    0x19, 0x50, 0xDC, 0x7D, 0xC9, 0x7F, 0xBD, 0xC0,
    0x63, 0x28, 0xB6, 0x8F, 0x22, 0x78, 0x1F, 0xD3,
    0x15, 0xAF, 0x56, 0x80, 0x09, 0xA5, 0x50, 0x9E,
    0x5B, 0x87, 0xA1, 0x1B, 0xF5, 0x27, 0xC0, 0x56,
    0xDA, 0xFF, 0xD8, 0x2A, 0xB6, 0xCB, 0xC2, 0x5C,
    0xCA, 0x37, 0x14, 0x34, 0x59, 0xE7, 0xBC, 0x63,
    0xBC, 0xDE, 0x52, 0x75, 0x7A, 0xDE, 0xB7, 0xDF,
    0x01, 0xCF, 0x12, 0x17, 0x3F, 0x1F, 0xEF, 0x81,
    0x02, 0xEC, 0x5A, 0xB1, 0x42, 0xC2, 0x13, 0xDD,
    0x9D, 0x30, 0x69, 0x62, 0x78, 0xA8, 0xD8, 0xBC,
    0x32, 0xDD, 0xE9, 0x59, 0x2D, 0x28, 0xC0, 0x78,
    0xC6, 0xD9, 0x2B, 0x94, 0x7D, 0x82, 0x5A, 0xCA,
    0xAB, 0x64, 0x94, 0x84, 0x6A, 0x49, 0xDE, 0x24,
    0xB9, 0x62, 0x3F, 0x48, 0x89, 0xE8, 0xAD, 0xC3,
    0x8E, 0x8C, 0x66, 0x9E, 0xFF, 0xEF, 0x17, 0x60,
    0x40, 0xAD, 0x94, 0x5E, 0x90, 0xA7, 0xD3, 0xEE,
    0xC1, 0x5E, 0xFE, 0xEE, 0x78, 0xAE, 0x71, 0x04,
    0x3C, 0x96, 0x51, 0x11, 0x03, 0xA1, 0x6B, 0xA7,
    0xCA, 0xF0, 0xAC, 0xD0, 0x64, 0x2E, 0xFD, 0xBE,
    0x80, 0x99, 0x34, 0xFA, 0xA1, 0xA5, 0xF1, 0xBD,
    0x11, 0x04, 0x36, 0x49, 0xB2, 0x5C, 0xCD, 0x1F,
    0xEE, 0x2E, 0x38, 0x81, 0x5D, 0x4D, 0x5F, 0x5F,
    0xC6, 0xB4, 0x10, 0x29, 0x69, 0xF2, 0x1C, 0x22,
    0xAE, 0x1B, 0x0E, 0x7D, 0x36, 0x03, 0xA5, 0x56,
    0xA1, 0x32, 0x62, 0xFF, 0x62, 0x8D, 0xE2, 0x22
};
static const byte sshKdfTvSet3h[] = {
    0x7B, 0x70, 0x01, 0x18, 0x5E, 0x25, 0x6D, 0x44,
    0x93, 0x44, 0x5F, 0x39, 0xA5, 0x5F, 0xB9, 0x05,
    0xE6, 0x32, 0x1F, 0x4B, 0x5D, 0xD8, 0xBB, 0xF3,
    0x10, 0x0D, 0x51, 0xBA, 0x0B, 0xDA, 0x3D, 0x2D
};
static const byte sshKdfTvSet3sid[] = {
    0x7B, 0x70, 0x01, 0x18, 0x5E, 0x25, 0x6D, 0x44,
    0x93, 0x44, 0x5F, 0x39, 0xA5, 0x5F, 0xB9, 0x05,
    0xE6, 0x32, 0x1F, 0x4B, 0x5D, 0xD8, 0xBB, 0xF3,
    0x10, 0x0D, 0x51, 0xBA, 0x0B, 0xDA, 0x3D, 0x2D
};
static const byte sshKdfTvSet3a[] = {
    0x81, 0xF0, 0x33, 0x0E, 0xF6, 0xF0, 0x53, 0x61,
    0xB3, 0x82, 0x3B, 0xFD, 0xED, 0x6E, 0x1D, 0xE9
};
static const byte sshKdfTvSet3b[] = {
    0x3F, 0x6F, 0xD2, 0x06, 0x5E, 0xEB, 0x2B, 0x0B,
    0x1D, 0x93, 0x19, 0x5A, 0x1F, 0xED, 0x48, 0xA5
};
static const byte sshKdfTvSet3c[] = {
    0xC3, 0x54, 0x71, 0x03, 0x4E, 0x6F, 0xD6, 0x54,
    0x76, 0x13, 0x17, 0x8E, 0x23, 0x43, 0x5F, 0x21
};
static const byte sshKdfTvSet3d[] = {
    0x7E, 0x9D, 0x79, 0x03, 0x20, 0x90, 0xD9, 0x9F,
    0x98, 0xB0, 0x15, 0x63, 0x4D, 0xD9, 0xF4, 0x62
};
static const byte sshKdfTvSet3e[] = {
    0x24, 0xEE, 0x55, 0x9A, 0xD7, 0xCE, 0x71, 0x2B,
    0x68, 0x5D, 0x0B, 0x22, 0x71, 0xE4, 0x43, 0xC1,
    0x7A, 0xB1, 0xD1, 0xDC, 0xEB, 0x5A, 0x36, 0x05,
    0x69, 0xD2, 0x5D, 0x5D, 0xC2, 0x43, 0x00, 0x2F
};
static const byte sshKdfTvSet3f[] = {
    0xC3, 0x41, 0x9C, 0x2B, 0x96, 0x62, 0x35, 0x86,
    0x9D, 0x71, 0x4B, 0xA5, 0xAC, 0x48, 0xDD, 0xB7,
    0xD9, 0xE3, 0x5C, 0x8C, 0x19, 0xAA, 0xC7, 0x34,
    0x22, 0x33, 0x7A, 0x37, 0x34, 0x53, 0x60, 0x7E
};

static const SshKdfTestVector sshKdfTestVectors[] = {
    {WC_HASH_TYPE_SHA256, 'A',
        sshKdfTvSet3k, sizeof(sshKdfTvSet3k),
        sshKdfTvSet3h, sizeof(sshKdfTvSet3h),
        sshKdfTvSet3sid, sizeof(sshKdfTvSet3sid),
        sshKdfTvSet3a, sizeof(sshKdfTvSet3a)},
    {WC_HASH_TYPE_SHA256, 'B',
        sshKdfTvSet3k, sizeof(sshKdfTvSet3k),
        sshKdfTvSet3h, sizeof(sshKdfTvSet3h),
        sshKdfTvSet3sid, sizeof(sshKdfTvSet3sid),
        sshKdfTvSet3b, sizeof(sshKdfTvSet3b)},
    {WC_HASH_TYPE_SHA256, 'C',
        sshKdfTvSet3k, sizeof(sshKdfTvSet3k),
        sshKdfTvSet3h, sizeof(sshKdfTvSet3h),
        sshKdfTvSet3sid, sizeof(sshKdfTvSet3sid),
        sshKdfTvSet3c, sizeof(sshKdfTvSet3c)},
    {WC_HASH_TYPE_SHA256, 'D',
        sshKdfTvSet3k, sizeof(sshKdfTvSet3k),
        sshKdfTvSet3h, sizeof(sshKdfTvSet3h),
        sshKdfTvSet3sid, sizeof(sshKdfTvSet3sid),
        sshKdfTvSet3d, sizeof(sshKdfTvSet3d)},
    {WC_HASH_TYPE_SHA256, 'E',
        sshKdfTvSet3k, sizeof(sshKdfTvSet3k),
        sshKdfTvSet3h, sizeof(sshKdfTvSet3h),
        sshKdfTvSet3sid, sizeof(sshKdfTvSet3sid),
        sshKdfTvSet3e, sizeof(sshKdfTvSet3e)},
    {WC_HASH_TYPE_SHA256, 'F',
        sshKdfTvSet3k, sizeof(sshKdfTvSet3k),
        sshKdfTvSet3h, sizeof(sshKdfTvSet3h),
        sshKdfTvSet3sid, sizeof(sshKdfTvSet3sid),
        sshKdfTvSet3f, sizeof(sshKdfTvSet3f)},
};


int sshkdf_test(void)
{
    int result = 0;
    word32 i;
    word32 tc = sizeof(sshKdfTestVectors)/sizeof(SshKdfTestVector);
    const SshKdfTestVector* tv = NULL;
    byte cKey[32]; /* Greater of SHA256_DIGEST_SIZE and AES_BLOCK_SIZE */
    /* sId - Session ID, eKey - Expected Key, cKey - Calculated Key */

    for (i = 0, tv = sshKdfTestVectors; i < tc; i++, tv++) {
        result = wc_SSH_KDF(tv->hashId, tv->keyId,
                cKey, tv->expectedKeySz,
                tv->k, tv->kSz, tv->h, tv->hSz,
                tv->sessionId, tv->sessionIdSz);

        if (result != 0) {
            printf("KDF: Could not derive key.\n");
            result = -101;
        }
        else {
            if (memcmp(cKey, tv->expectedKey, tv->expectedKeySz) != 0) {
                printf("KDF: Calculated Key does not match Expected Key.\n");
                result = -102;
            }
        }

        if (result != 0) break;
    }

    return result;
}

#endif /* WOLFSSL_WOLFSSH */


#ifdef WOLFSSL_TLS13

#define TLSV13_PSK_DHE_SZ 40
typedef struct {
    enum wc_HashType hashAlg;
    word32 pskSz;
    word32 dheSz;
    byte psk[TLSV13_PSK_DHE_SZ];
    byte dhe[TLSV13_PSK_DHE_SZ];
    byte hashHello1[WC_MAX_DIGEST_SIZE];
    byte hashHello2[WC_MAX_DIGEST_SIZE];
    byte hashFinished1[WC_MAX_DIGEST_SIZE];
    byte hashFinished2[WC_MAX_DIGEST_SIZE];
    /* Expected */
    byte clientEarlyTrafficSecret[WC_MAX_DIGEST_SIZE];
    byte earlyExporterMasterSecret[WC_MAX_DIGEST_SIZE];
    byte clientHandshakeTrafficSecret[WC_MAX_DIGEST_SIZE];
    byte serverHandshakeTrafficSecret[WC_MAX_DIGEST_SIZE];
    byte clientApplicationTrafficSecret[WC_MAX_DIGEST_SIZE];
    byte serverApplicationTrafficSecret[WC_MAX_DIGEST_SIZE];
    byte exporterMasterSecret[WC_MAX_DIGEST_SIZE];
    byte resumptionMasterSecret[WC_MAX_DIGEST_SIZE];
} Tls13KdfTestVector;

/* The following tests come from the CAVP test vectors we used for
 * our FIPS validation. The hash values used are the components from
 * the test hashed together. hashHello1 is the hash of the
 * clientHelloRandom value of the test vector. hashHello2 is the hash
 * of the clientHelloRandom and serverHelloRandom values from the test
 * vector. hashFinished1 is clientHelloRandom, serverHelloRandom, and
 * serverFinishedRandom. hashFinished2 is clientHelloRandom,
 * serverHelloRandom, serverFinishedRandom, and clietnFinishedRandom
 * hashed together. */
static const Tls13KdfTestVector tls13KdfTestVectors[] = {
{ /* 1 */
    WC_HASH_TYPE_SHA256, 35, 35,
    { /* PSK */
        0x7b, 0xf1, 0x05, 0x31, 0x36, 0xfa, 0x03, 0xdc,
        0x31, 0x97, 0x88, 0x04, 0x9c, 0xbc, 0xee, 0xf7,
        0x8d, 0x84, 0x95, 0x26, 0xaf, 0x1d, 0x68, 0xb0,
        0x60, 0x7a, 0xcc, 0x4f, 0xc1, 0xd3, 0xa1, 0x68,
        0x7f, 0x6d, 0xbe
    },
    { /* DHE */
        0x6e, 0xa1, 0x77, 0xab, 0x2f, 0x43, 0xd2, 0x4b,
        0xe5, 0xa1, 0x09, 0xe0, 0x7a, 0xd0, 0x01, 0x35,
        0x8d, 0xf8, 0xf2, 0x5c, 0x91, 0x02, 0xb0, 0x6c,
        0x3f, 0xeb, 0xee, 0xa4, 0x42, 0x19, 0xce, 0xdc,
        0x81, 0x26, 0x40
    },
    { /* Hello 1 */
        0xd9, 0x4b, 0xe4, 0x17, 0xef, 0x58, 0x73, 0x7d,
        0x28, 0x3d, 0xf0, 0xcc, 0x05, 0x03, 0xaf, 0xac,
        0x3d, 0x92, 0x79, 0x48, 0xe8, 0x8c, 0xdb, 0xce,
        0x95, 0x82, 0x21, 0x31, 0x7b, 0x61, 0xd7, 0xc6
    },
    { /* Hello 2 */
        0xb7, 0x7f, 0x29, 0x91, 0xa4, 0x8b, 0x34, 0xdb,
        0xbd, 0xc7, 0x54, 0x1c, 0x3b, 0x86, 0xa3, 0x69,
        0xfe, 0x26, 0xe4, 0x7b, 0xac, 0x57, 0x71, 0xb3,
        0x32, 0x97, 0xed, 0xd2, 0x0e, 0x95, 0xb8, 0x63
    },
    { /* Finished 1 */
        0x65, 0xdb, 0x6d, 0x71, 0x71, 0xd0, 0xd8, 0x49,
        0xd0, 0x3c, 0x8e, 0x2b, 0x24, 0xdf, 0xc2, 0xe9,
        0xd6, 0xfd, 0xea, 0x04, 0x95, 0x7c, 0xf0, 0x7e,
        0x57, 0x74, 0x7c, 0xdd, 0xa3, 0x0b, 0x2b, 0x36
    },
    { /* Finished 2 */
        0x28, 0xf2, 0xf2, 0x79, 0xcf, 0x20, 0x52, 0x90,
        0x1d, 0x91, 0x05, 0xad, 0x44, 0x26, 0x23, 0x96,
        0x32, 0xce, 0xec, 0x61, 0xd1, 0xbf, 0x00, 0x48,
        0x4a, 0xa5, 0x60, 0xcc, 0x28, 0xb5, 0x8d, 0x98
    },
    { /* Client Early Traffic Secret */
        0x07, 0x14, 0x6a, 0x26, 0x5b, 0x6c, 0x7f, 0x4d, 0x6b, 0x47, 0x3f, 0xd5,
        0x03, 0x1d, 0xd2, 0x23, 0x3d, 0x89, 0x3e, 0xc6, 0x51, 0xd1, 0xac, 0xf8,
        0x28, 0xae, 0x4b, 0x76, 0xc8, 0x10, 0x7e, 0xdd
    },
    { /* Early Exporter Master Secret */
        0xb8, 0xd3, 0x25, 0x7e, 0x2d, 0x41, 0x7b, 0xcb, 0x5e, 0x82, 0x49, 0xf5,
        0x51, 0x3d, 0xb7, 0x59, 0x32, 0xb3, 0xdf, 0x99, 0x4e, 0x04, 0x69, 0xc6,
        0x96, 0x8e, 0xe6, 0x3d, 0x91, 0xe4, 0x81, 0x11
    },
    { /* Client Handshake Traffic Secret */
        0xd9, 0x3b, 0x54, 0xe2, 0xb0, 0xd1, 0x85, 0xf0, 0xfd, 0xf3, 0x48, 0x4a,
        0xf8, 0x0b, 0xa5, 0xdc, 0x4c, 0x37, 0xcb, 0xd4, 0x20, 0xaf, 0x60, 0xc7,
        0xd5, 0x50, 0x5d, 0x0c, 0x77, 0x3b, 0x6f, 0xd2
    },
    { /* Server Handshake Traffic Secret */
        0x4d, 0x40, 0x2b, 0xd2, 0x8c, 0x33, 0x90, 0x39, 0x67, 0x67, 0x05, 0xf7,
        0x5d, 0x37, 0x1e, 0xdc, 0x4a, 0x70, 0x6b, 0x9e, 0xf8, 0x06, 0x61, 0x89,
        0x70, 0xe1, 0x3d, 0x36, 0xad, 0x88, 0x7e, 0x5b
    },
    { /* Client Application Traffic Secret */
        0x74, 0x6e, 0xa0, 0x13, 0x18, 0x34, 0x48, 0x4d, 0x23, 0x31, 0xf1, 0xf9,
        0xee, 0x44, 0x6d, 0xad, 0xc1, 0xad, 0x92, 0x73, 0xca, 0x27, 0x16, 0x91,
        0xa2, 0x50, 0x9a, 0xfc, 0xec, 0xf0, 0x6b, 0x24
    },
    { /* Server Application Traffic Secret */
        0x89, 0x18, 0x7e, 0x34, 0x8d, 0xfc, 0x14, 0xb1, 0x4f, 0x21, 0xd8, 0x29,
        0xdb, 0x9b, 0xfb, 0x55, 0xcf, 0xa1, 0x4f, 0x95, 0xf8, 0xe0, 0xb0, 0x83,
        0xd5, 0x34, 0x9e, 0x0b, 0x83, 0x37, 0x42, 0x93
    },
    { /* Exporter Master Secret */
        0x7d, 0xc8, 0x88, 0x46, 0xd5, 0x57, 0x15, 0xb6, 0x24, 0x25, 0x92, 0x61,
        0xb1, 0x18, 0x86, 0x2a, 0x6d, 0xa5, 0x84, 0xeb, 0x59, 0xdf, 0x13, 0xbd,
        0x73, 0xaa, 0x5d, 0x65, 0xab, 0xd9, 0xb4, 0x56
    },
    { /* Resumption Master Secret */
        0x20, 0xb7, 0xd0, 0xe3, 0x82, 0x01, 0xa1, 0x04, 0xb8, 0x13, 0x29, 0xed,
        0x35, 0xe4, 0x2f, 0xbf, 0x58, 0x23, 0x7f, 0x21, 0xdb, 0x9f, 0xf8, 0xe0,
        0xe8, 0xe4, 0xab, 0xc4, 0xa1, 0x61, 0xb9, 0xbb
    }
},
{ /* 6 */
    WC_HASH_TYPE_SHA256, 0, 33,
    { 0 }, /* PSK */
    { /* DHE */
        0x7a, 0x46, 0x8c, 0x5a, 0xd1, 0x8e, 0x95, 0xba,
        0x61, 0xe6, 0x6f, 0xe6, 0x76, 0x0c, 0x20, 0x43,
        0x16, 0x82, 0x15, 0xfe, 0x54, 0xa3, 0xc7, 0xfd,
        0x3b, 0x2c, 0x88, 0xb4, 0xd3, 0x42, 0x70, 0x12,
        0x18
    },
    { /* Hello 1 */
        0x63, 0x83, 0x58, 0xab, 0x36, 0xcd, 0x0c, 0xf3,
        0x26, 0x07, 0xb5, 0x5f, 0x0b, 0x8b, 0x45, 0xd6,
        0x7d, 0x5b, 0x42, 0xdc, 0xa8, 0xaa, 0x06, 0xfb,
        0x20, 0xa5, 0xbb, 0x85, 0xdb, 0x54, 0xd8, 0x8b
    },
    { /* Hello 2 */
        0xea, 0xfe, 0x9e, 0x8e, 0xff, 0x1f, 0x6f, 0x43,
        0xf9, 0x5d, 0xfd, 0xbf, 0xe2, 0x5f, 0x02, 0x2f,
        0x6d, 0x47, 0x60, 0x9a, 0x48, 0x9a, 0x75, 0xfb,
        0xb5, 0x4a, 0xbf, 0x9c, 0x4e, 0xff, 0xbf, 0x0b
    },
    { /* Finished 1 */
        0xca, 0x25, 0xb3, 0x53, 0x8e, 0x6d, 0xc3, 0x36,
        0x17, 0x30, 0x07, 0xdf, 0x0d, 0xd7, 0x79, 0xb0,
        0x7f, 0xcb, 0xbe, 0x7a, 0xbc, 0x2d, 0x9f, 0x2d,
        0x94, 0x44, 0x94, 0xe6, 0xa4, 0xf3, 0xe8, 0x53
    },
    { /* Finished 2 */
        0x2e, 0xa6, 0x5a, 0xaf, 0xb5, 0xba, 0x9f, 0x2f,
        0x74, 0x83, 0x5d, 0xbf, 0x86, 0xa4, 0xa6, 0xf6,
        0xb9, 0x89, 0xdf, 0x17, 0xe1, 0xa8, 0x14, 0xc0,
        0xe1, 0x50, 0xfa, 0xec, 0xfa, 0xae, 0x8b, 0x7b
    },
    {
        0x20, 0x18, 0x72, 0x7c, 0xde, 0x3a, 0x85, 0x17, 0x72, 0xdc, 0xd7, 0x72,
        0xb0, 0xfc, 0x45, 0xd0, 0x62, 0xb9, 0xbb, 0x38, 0x69, 0x05, 0x7b, 0xb4,
        0x5e, 0x58, 0x5d, 0xed, 0xcd, 0x0b, 0x96, 0xd3
    },
    {
        0x68, 0x10, 0x20, 0xd1, 0x5e, 0xfc, 0x0c, 0x53, 0x85, 0xbb, 0xdb, 0x18,
        0xa8, 0x78, 0xf1, 0x2b, 0x13, 0xba, 0x64, 0x1d, 0xe7, 0x09, 0xbe, 0x13,
        0x49, 0x26, 0xf9, 0x98, 0x56, 0xf1, 0x43, 0xfb
    },
    {
        0x24, 0x35, 0x3e, 0x10, 0x6f, 0x39, 0x50, 0xd6, 0xa2, 0x12, 0x99, 0xf2,
        0xd5, 0xf5, 0x19, 0xf5, 0x84, 0xed, 0xee, 0x78, 0x2a, 0xa6, 0xfa, 0x3d,
        0x06, 0xa8, 0xa7, 0x5d, 0x97, 0x78, 0xd6, 0x58
    },
    {
        0xf4, 0x57, 0xac, 0x24, 0x7a, 0xfb, 0x7c, 0x3b, 0xb6, 0x39, 0x17, 0x14,
        0xd9, 0xd4, 0x58, 0x4d, 0x46, 0xd5, 0x1b, 0xde, 0xf7, 0x9d, 0x06, 0xee,
        0x8d, 0x1a, 0x2c, 0x25, 0x6d, 0x64, 0xde, 0x89
    },
    {
        0xb6, 0x00, 0xce, 0x63, 0xed, 0x65, 0x8b, 0x66, 0x66, 0x42, 0xc6, 0xbd,
        0x89, 0xc4, 0x71, 0x6f, 0xce, 0x28, 0xb2, 0xac, 0x97, 0x07, 0x5b, 0xea,
        0xb8, 0x1d, 0x4c, 0xeb, 0x9e, 0x71, 0x07, 0x8f
    },
    {
        0xf8, 0x92, 0xc8, 0xba, 0xe7, 0x83, 0xfe, 0x68, 0xe4, 0xd6, 0x5e, 0xcb,
        0xb3, 0xef, 0x49, 0xd0, 0xe7, 0xb1, 0xac, 0xcb, 0x39, 0x19, 0xfd, 0xa7,
        0xf7, 0xca, 0xab, 0x1e, 0x42, 0x14, 0xd8, 0xe7
    },
    {
        0x32, 0x4a, 0x1a, 0xad, 0xe2, 0xbb, 0x55, 0x8a, 0xdd, 0xe9, 0xa5, 0x2a,
        0x46, 0x5e, 0x6c, 0x83, 0x66, 0x27, 0x27, 0x94, 0xdd, 0x68, 0x59, 0xa0,
        0xbb, 0xe8, 0x31, 0x7c, 0x39, 0xd7, 0xfd, 0x6d
    },
    {
        0x58, 0xbc, 0x6c, 0x5b, 0x24, 0xad, 0x82, 0xb3, 0xcc, 0xc7, 0xd1, 0xa1,
        0xaa, 0x2b, 0x98, 0x9f, 0x2f, 0x7e, 0xa9, 0x63, 0xc2, 0x8e, 0xb6, 0x06,
        0xc2, 0x2b, 0x74, 0x4b, 0x79, 0x19, 0x7e, 0x2e
    }
},
{ /* 11 */
    WC_HASH_TYPE_SHA256, 33, 0,
    { /* PSK */
        0x3d, 0x39, 0x49, 0x36, 0x98, 0xc5, 0xfd, 0xcd,
        0xa0, 0x17, 0xbd, 0x65, 0x0a, 0xdb, 0xd4, 0x07,
        0x56, 0xa2, 0x7b, 0xb8, 0x2a, 0x7e, 0xfb, 0x26,
        0x74, 0xe1, 0xbc, 0x08, 0x4b, 0xf0, 0x30, 0x14,
        0x12
    },
    { 0 }, /* DHE */
    { /* Hello 1 */
        0xb7, 0x44, 0x74, 0x6c, 0x57, 0x1f, 0xf3, 0x84,
        0x8f, 0x63, 0xfb, 0x8c, 0x94, 0x6c, 0x16, 0x68,
        0x4b, 0xe1, 0xb5, 0xb5, 0x2a, 0x4e, 0x5f, 0xdf,
        0x4b, 0x53, 0xb2, 0x35, 0xfc, 0x30, 0xf1, 0x36
    },
    { /* Hello 2 */
        0xe6, 0x4f, 0x3a, 0x4f, 0xd7, 0xe0, 0x64, 0xd4,
        0x69, 0x50, 0xe4, 0x8b, 0xba, 0xbc, 0x47, 0x74,
        0xa7, 0x9b, 0x40, 0x91, 0x8f, 0xa8, 0x72, 0x22,
        0x97, 0xad, 0x43, 0xa7, 0x11, 0x86, 0xb5, 0x72
    },
    { /* Finished 1 */
        0x5f, 0xa6, 0x10, 0xe2, 0xa3, 0x99, 0x0b, 0x5e,
        0x57, 0xee, 0xc3, 0x3a, 0x8e, 0x04, 0xf3, 0x0e,
        0x58, 0x02, 0x09, 0xb2, 0x7e, 0x2d, 0xc6, 0xd2,
        0x08, 0xae, 0x68, 0x0a, 0x55, 0xa5, 0xda, 0x51
    },
    { /* Finished 2 */
        0xfc, 0x5b, 0xc0, 0x7e, 0x1b, 0xaa, 0xc0, 0xb4,
        0x34, 0x85, 0x49, 0x8e, 0x16, 0x31, 0x98, 0xdf,
        0x10, 0x54, 0x22, 0xda, 0x1e, 0x6b, 0x51, 0xf6,
        0x97, 0x57, 0xa0, 0x7a, 0x92, 0xe7, 0x47, 0x52
    },
    {
        0x80, 0xfa, 0x36, 0x30, 0xb8, 0x65, 0xb3, 0x2a, 0x1d, 0x68, 0x91, 0x06,
        0x98, 0xa0, 0x17, 0x8f, 0xee, 0xb7, 0x9e, 0x3d, 0xd8, 0x84, 0x99, 0x30,
        0xb9, 0xd6, 0x09, 0x25, 0x5e, 0xfb, 0x8f, 0xd3 },
    {
        0xa9, 0x89, 0x29, 0x70, 0xe4, 0x55, 0xec, 0x97, 0xfb, 0x24, 0x5b, 0xf9,
        0xf1, 0xa3, 0x19, 0x3d, 0xf1, 0x31, 0x14, 0xcd, 0x2a, 0xed, 0x21, 0xc8,
        0xb1, 0x53, 0xad, 0x11, 0x0b, 0x9e, 0x5a, 0xee },
    {
        0x72, 0xad, 0x8d, 0x7f, 0xfc, 0xb7, 0x68, 0xda, 0x27, 0x60, 0x37, 0xa3,
        0x4a, 0x63, 0xe8, 0xa5, 0xc8, 0xcd, 0x36, 0x6a, 0x77, 0x99, 0x0d, 0xa9,
        0xb1, 0x5b, 0x2f, 0x47, 0x2e, 0x22, 0xa7, 0x5e },
    {
        0x95, 0x6e, 0x85, 0x09, 0xe5, 0x04, 0x88, 0x14, 0x28, 0x8d, 0xdf, 0xe6,
        0x0d, 0x0f, 0x0d, 0x6b, 0x4e, 0x66, 0x1c, 0x03, 0xb9, 0xaa, 0x2d, 0x45,
        0x56, 0x67, 0x5c, 0x55, 0x29, 0xd6, 0x89, 0xd0 },
    {
        0xe8, 0xf2, 0x14, 0xf9, 0x9b, 0x2b, 0x9f, 0x24, 0x2b, 0x37, 0xbe, 0x86,
        0xdb, 0x23, 0x4b, 0xbe, 0x39, 0x57, 0xe8, 0xa9, 0xa5, 0xee, 0x08, 0xf2,
        0x75, 0x58, 0xdb, 0xd9, 0x51, 0xc1, 0x46, 0x02 },
    {
        0x3d, 0x19, 0xaf, 0xa3, 0x0b, 0x21, 0xf7, 0x3d, 0xe7, 0x37, 0x6e, 0x32,
        0x13, 0x48, 0x9d, 0xea, 0xe0, 0x90, 0xbf, 0x64, 0x48, 0xf7, 0x1e, 0xcc,
        0xf0, 0xbc, 0x92, 0xd7, 0x8a, 0x4a, 0xa8, 0xc1 },
    {
        0x16, 0x35, 0xb1, 0x66, 0x28, 0xa3, 0x3e, 0x19, 0xf5, 0x2d, 0x92, 0x22,
        0x95, 0x48, 0xe8, 0x34, 0x7b, 0x30, 0x50, 0xa2, 0xa0, 0xd9, 0xc2, 0x59,
        0x39, 0xf9, 0x8c, 0x69, 0xf2, 0x2a, 0xb9, 0xff },
    {
        0x32, 0x71, 0xa6, 0x87, 0x0c, 0x97, 0x42, 0x07, 0xdd, 0x5f, 0xc9, 0x44,
        0xa5, 0x7c, 0x50, 0x14, 0xfd, 0xe7, 0x5f, 0x8b, 0xd3, 0x2f, 0xdc, 0x9b,
        0xa9, 0x93, 0x22, 0x19, 0xe6, 0xf2, 0x0c, 0xd8 }
},
#ifdef WOLFSSL_SHA384
{ /* 26 */
    WC_HASH_TYPE_SHA384, 35, 35,
    { /* PSK */
        0x62, 0x83, 0x25, 0xc7, 0xcc, 0x08, 0x5e, 0x63,
        0x64, 0x56, 0xf0, 0xc6, 0x88, 0x27, 0x5a, 0x5b,
        0x68, 0x59, 0x0b, 0x14, 0x55, 0x13, 0x2e, 0xfd,
        0x8f, 0x28, 0x5b, 0x3d, 0xe3, 0xad, 0x67, 0xe4,
        0x68, 0xba, 0xf9
    },
    { /* DHE */
        0xa8, 0xb1, 0xab, 0xd8, 0xc8, 0x5b, 0x52, 0xdf,
        0x7f, 0x49, 0x10, 0xf4, 0xa1, 0x31, 0xd1, 0x91,
        0x36, 0xc1, 0x87, 0x5d, 0x42, 0x2a, 0xe7, 0x1d,
        0x2c, 0x29, 0x3d, 0x40, 0x64, 0x61, 0x63, 0x76,
        0xd8, 0x66, 0xac
    },
    { /* Hello 1 */
        0x6f, 0xc6, 0x4c, 0xe1, 0xc6, 0x68, 0x34, 0x8c,
        0x0a, 0xe1, 0xf8, 0xb8, 0x3e, 0xd4, 0xf8, 0x0b,
        0x54, 0x50, 0xe4, 0xc5, 0x4a, 0x33, 0x7d, 0xbd,
        0x90, 0xd2, 0xa2, 0xb9, 0xb7, 0x92, 0xed, 0xab,
        0x14, 0xf1, 0xe4, 0x86, 0x22, 0x67, 0xd7, 0x44,
        0x03, 0x21, 0xdc, 0x51, 0x52, 0x7f, 0x35, 0x80
    },
    { /* Hello 2 */
        0x3e, 0xcf, 0x2f, 0xc3, 0x87, 0xba, 0xc5, 0xbd,
        0x7c, 0xe8, 0x35, 0x5b, 0x95, 0x51, 0x30, 0x3b,
        0x08, 0xcc, 0x2a, 0x7d, 0xb5, 0x74, 0x7c, 0x16,
        0xb3, 0x0b, 0xe7, 0x61, 0xa3, 0x7c, 0x6c, 0xbd,
        0x39, 0x74, 0xfd, 0x1e, 0x4c, 0xff, 0xc8, 0xcc,
        0xa0, 0xef, 0x29, 0x4d, 0x94, 0xaa, 0x55, 0x6f,
    },
    { /* Finished 1 */
        0x06, 0xc1, 0x47, 0x78, 0x66, 0x53, 0x6f, 0x24,
        0x94, 0x61, 0x69, 0xec, 0xd8, 0x60, 0x31, 0x2f,
        0xbf, 0xd6, 0x8a, 0x29, 0x17, 0xff, 0xa3, 0x88,
        0x13, 0x09, 0x8c, 0x9d, 0x6c, 0x64, 0x84, 0x48,
        0x44, 0xdd, 0x2d, 0x29, 0x4d, 0xe6, 0x98, 0x2b,
        0x45, 0x3b, 0x84, 0x33, 0x79, 0xb2, 0x75, 0x68
    },
    { /* Finished 2 */
        0x28, 0x1e, 0x18, 0xf7, 0x9c, 0x32, 0xa9, 0xbf,
        0x0c, 0x24, 0x58, 0x21, 0xce, 0xbc, 0xf2, 0x44,
        0xb1, 0x18, 0xaf, 0x9d, 0xd9, 0x20, 0xf9, 0xf4,
        0xed, 0xcc, 0x53, 0x82, 0x66, 0x5c, 0x46, 0x94,
        0x8c, 0x36, 0x5e, 0xca, 0x9f, 0xd8, 0x9a, 0xd3,
        0xf0, 0xe1, 0x53, 0x71, 0xdd, 0x19, 0x1e, 0x59
    },
    {
        0xd0, 0xef, 0xa8, 0xcb, 0x5b, 0x14, 0x0f, 0x0a, 0x62, 0xba, 0x5a, 0xb1,
        0xc5, 0xb5, 0x3f, 0x11, 0xda, 0xa1, 0x0c, 0x9c, 0xb4, 0x32, 0x48, 0x4e,
        0xfa, 0x84, 0x4f, 0xe4, 0xe7, 0x91, 0x8f, 0x42, 0x3f, 0xc7, 0x4e, 0xd3,
        0x83, 0x3d, 0x7f, 0x70, 0x12, 0xee, 0x9a, 0x37, 0x01, 0xbb, 0x14, 0xd3
    },
    {
        0x48, 0x6f, 0x77, 0x1d, 0x39, 0x1b, 0xa5, 0x9a, 0x76, 0xd9, 0x1d, 0x7d,
        0xb3, 0xd9, 0xb9, 0x78, 0x35, 0x0f, 0xd0, 0xe1, 0x07, 0x1f, 0x8d, 0xe5,
        0x75, 0x00, 0xda, 0xc0, 0x19, 0x01, 0xfb, 0x08, 0x35, 0xe7, 0x18, 0x8f,
        0xf0, 0x19, 0xfb, 0x46, 0xf6, 0xa5, 0x77, 0x0e, 0x90, 0x38, 0x8b, 0x15
    },
    {
        0x80, 0x8c, 0xa7, 0x24, 0x97, 0xf9, 0xd3, 0x52, 0xb0, 0x69, 0x9d, 0x4b,
        0xa4, 0x19, 0x4a, 0xb1, 0x46, 0x53, 0x3a, 0xc8, 0xe4, 0x02, 0x69, 0xf2,
        0xe7, 0xb6, 0x1d, 0x33, 0x51, 0xcc, 0x14, 0x40, 0x4a, 0xb0, 0xe7, 0x58,
        0x84, 0xba, 0xc2, 0x14, 0x58, 0x6b, 0xb9, 0xdc, 0x50, 0x98, 0x67, 0x01
    },
    {
        0xb1, 0xa8, 0xc0, 0x06, 0xb3, 0x2e, 0xa7, 0x8a, 0x6a, 0x12, 0x88, 0x00,
        0x65, 0x88, 0x9c, 0x5d, 0x35, 0xee, 0xe5, 0x51, 0x0b, 0x62, 0xf8, 0x67,
        0xe5, 0xef, 0x15, 0x1f, 0x23, 0x02, 0x74, 0x08, 0x9c, 0xc8, 0xba, 0x27,
        0x5d, 0x32, 0x19, 0x6f, 0x6d, 0x5d, 0x72, 0x5e, 0x15, 0xde, 0x30, 0xc3
    },
    {
        0xfd, 0xce, 0xf5, 0x65, 0x45, 0x84, 0xfb, 0x8c, 0x79, 0xa4, 0x6c, 0x1b,
        0x0e, 0x1b, 0xfd, 0x26, 0xa2, 0x53, 0xf4, 0x4e, 0x00, 0x4d, 0x4b, 0x0b,
        0x24, 0x6d, 0x35, 0x35, 0xd9, 0x97, 0x70, 0xc5, 0xf4, 0xee, 0xe3, 0xba,
        0x31, 0x1e, 0x2a, 0x42, 0xcb, 0xdf, 0x40, 0xb1, 0x14, 0xb8, 0x53, 0xce
    },
    {
        0xbb, 0xb3, 0x26, 0x7c, 0x22, 0x21, 0x9b, 0x72, 0x32, 0xa1, 0x97, 0xfb,
        0x78, 0x8c, 0xbe, 0x3d, 0x71, 0x45, 0xb8, 0xf5, 0x24, 0x8f, 0x0f, 0xac,
        0x42, 0x5b, 0x81, 0xe8, 0xd0, 0x71, 0x4a, 0xcb, 0x32, 0x3f, 0x03, 0xfb,
        0xec, 0x6a, 0x1f, 0x76, 0x80, 0x65, 0x01, 0x7a, 0x3d, 0xce, 0xc4, 0xdf
    },
    {
        0x3f, 0xcf, 0x2f, 0x63, 0x94, 0x94, 0x99, 0xfd, 0x04, 0x3a, 0x89, 0x83,
        0xcf, 0x06, 0x05, 0xec, 0x20, 0x3e, 0x5f, 0x51, 0x9d, 0x6e, 0x4a, 0xc6,
        0xf1, 0x2b, 0x37, 0x17, 0x34, 0x72, 0x6e, 0x1d, 0x2a, 0xfd, 0xc7, 0x73,
        0xb5, 0x07, 0x22, 0x81, 0x32, 0x2e, 0x21, 0x85, 0xaf, 0x10, 0xb2, 0x73
    },
    {
        0x52, 0x0c, 0x3d, 0x2e, 0x2d, 0x4a, 0x11, 0xae, 0x96, 0x78, 0xe9, 0x5b,
        0xd8, 0x0f, 0x6c, 0xf4, 0xbd, 0x96, 0x13, 0x55, 0x88, 0xdd, 0xa3, 0x67,
        0x36, 0x86, 0x1e, 0x0b, 0x36, 0x41, 0xec, 0xf6, 0x04, 0xb2, 0xc4, 0x16,
        0xbc, 0x2c, 0xdb, 0x30, 0x02, 0x94, 0xd4, 0x42, 0xbf, 0x38, 0xee, 0x9d
    }
},
{ /* 36 */
    WC_HASH_TYPE_SHA384, 0, 33,
    { 0 }, /* PSK */
    { /* DHE */
        0xd3, 0x00, 0x72, 0x9a, 0xa8, 0xc5, 0xf3, 0xc4,
        0xf1, 0xa0, 0x26, 0x89, 0x65, 0x70, 0xc7, 0x0b,
        0x77, 0xbb, 0xe1, 0x4b, 0x2b, 0xa8, 0x4f, 0xa6,
        0x09, 0x4b, 0xba, 0x45, 0x36, 0x15, 0xee, 0x68,
        0xfd
    },
    { /* Hello 1 */
        0x10, 0x9d, 0x8b, 0xa2, 0x93, 0xe7, 0xd3, 0xb9,
        0xb4, 0x0f, 0xeb, 0x6a, 0xb9, 0x69, 0xcb, 0x39,
        0x16, 0x29, 0xcc, 0xd3, 0xcc, 0x1a, 0x4c, 0x1b,
        0x53, 0x7c, 0x33, 0x88, 0x06, 0xbc, 0x0a, 0x02,
        0xa0, 0xbe, 0x62, 0xc0, 0xe6, 0x5e, 0x97, 0x5b,
        0x6a, 0xa1, 0x98, 0xf3, 0xd2, 0x1e, 0xcd, 0xc5
    },
    { /* Hello 2 */
        0x74, 0xc0, 0x07, 0x2c, 0xc1, 0x63, 0xcc, 0x11,
        0xad, 0x1a, 0x55, 0x63, 0xbc, 0x20, 0x77, 0x96,
        0x30, 0x1c, 0x68, 0x45, 0x1e, 0x9b, 0xa7, 0xb4,
        0xf3, 0x04, 0x45, 0x16, 0x76, 0x55, 0xf9, 0xdf,
        0x4b, 0x2f, 0x1a, 0xdf, 0x5a, 0xb0, 0x93, 0xc9,
        0xab, 0xf5, 0x32, 0x47, 0x79, 0x9c, 0x01, 0xeb
    },
    { /* Finished 1 */
        0x27, 0x08, 0x8e, 0xa5, 0xf1, 0x30, 0xe1, 0xd6,
        0x4f, 0xa2, 0x9e, 0x3b, 0x03, 0x2d, 0x2e, 0xa3,
        0x84, 0x75, 0x51, 0x3a, 0xc3, 0xf6, 0xee, 0x2e,
        0x37, 0x0c, 0xe3, 0x28, 0x46, 0xa5, 0x2d, 0xc7,
        0xf0, 0x64, 0x78, 0x53, 0x66, 0x43, 0x02, 0xa4,
        0x7a, 0x43, 0x66, 0x4b, 0xa7, 0xcb, 0x97, 0x16
    },
    { /* Finished 2 */
        0x1d, 0x0d, 0xf8, 0xe1, 0x81, 0xa5, 0xbd, 0xa8,
        0x6f, 0x9d, 0x01, 0xa4, 0x9a, 0x92, 0xe2, 0xef,
        0x08, 0xab, 0xef, 0x3e, 0x2d, 0xd4, 0x82, 0xac,
        0x68, 0x9d, 0xe0, 0x54, 0x17, 0xde, 0x1a, 0xed,
        0x57, 0xcb, 0xd9, 0x2d, 0xc8, 0xbc, 0x93, 0xe6,
        0xa3, 0xec, 0xde, 0xee, 0xa1, 0x1c, 0x41, 0x85
    },
    {
        0x7f, 0x1f, 0xe6, 0x7b, 0xd8, 0xf5, 0x2b, 0x37, 0xbe, 0xb7, 0xd0, 0x37,
        0xce, 0x46, 0xad, 0x04, 0x2f, 0xc7, 0xdb, 0xc9, 0x9a, 0xb6, 0x00, 0x3f,
        0xc1, 0x97, 0xe9, 0x5c, 0x5e, 0x14, 0xd1, 0x38, 0x4d, 0x55, 0xe1, 0x07,
        0xb5, 0x85, 0x6d, 0xfa, 0xa7, 0x66, 0xad, 0xfa, 0xb6, 0xad, 0x29, 0x44
    },
    {
        0x4e, 0x6b, 0x20, 0x99, 0x55, 0x1b, 0x21, 0x89, 0xb6, 0x70, 0xdb, 0xe8,
        0xa7, 0x16, 0x55, 0xf2, 0x93, 0x13, 0x90, 0x7d, 0xfa, 0x62, 0x65, 0x53,
        0xa0, 0x97, 0xe9, 0xb4, 0xc0, 0xf1, 0xc9, 0x1a, 0x67, 0xdd, 0xca, 0x57,
        0xbc, 0xca, 0x39, 0xe6, 0x39, 0x6b, 0x63, 0x47, 0x25, 0x08, 0x3a, 0xd7
    },
    {
        0x35, 0x0d, 0xac, 0xd8, 0x10, 0x6a, 0x46, 0x50, 0x66, 0xae, 0x02, 0xc9,
        0xde, 0x13, 0x48, 0xce, 0x53, 0xd4, 0x92, 0x62, 0xc5, 0x65, 0x10, 0x08,
        0xc2, 0xc2, 0x82, 0xed, 0x9d, 0xc9, 0x6f, 0xa8, 0xc3, 0xc1, 0x0b, 0x7c,
        0xe1, 0x97, 0x85, 0xd6, 0x46, 0x29, 0x0e, 0x42, 0x51, 0xc1, 0x35, 0xcf
    },
    {
        0x3d, 0x5d, 0x84, 0xbd, 0x16, 0x46, 0x34, 0xb3, 0xf6, 0x31, 0x49, 0x3e,
        0x8d, 0xdc, 0xcb, 0x8c, 0x6a, 0x42, 0xf4, 0x88, 0xfc, 0x19, 0xfa, 0xa2,
        0x25, 0xc7, 0xa0, 0xa4, 0xca, 0xf0, 0xea, 0x2d, 0xe8, 0xc4, 0x02, 0x14,
        0x63, 0xfb, 0xd3, 0x7b, 0x51, 0x1c, 0xce, 0xca, 0xa3, 0xc3, 0xe4, 0xa5
    },
    {
        0x7c, 0x3a, 0x55, 0x92, 0x2e, 0xdd, 0x75, 0xdd, 0x76, 0x54, 0x4a, 0x9f,
        0xd0, 0xa2, 0x88, 0x83, 0xe9, 0x27, 0xda, 0x30, 0xe9, 0x96, 0x58, 0xc5,
        0xb7, 0x56, 0xfc, 0x4b, 0xb8, 0x5d, 0xee, 0x46, 0x70, 0x4e, 0x1b, 0x06,
        0x86, 0xaf, 0x48, 0x5c, 0x17, 0x35, 0xfa, 0x69, 0xc2, 0x4d, 0xfb, 0x09
    },
    {
        0x00, 0x0e, 0x28, 0x51, 0xc1, 0x7f, 0x41, 0x89, 0x6f, 0x9a, 0xca, 0x15,
        0xee, 0xed, 0x43, 0xca, 0x6d, 0x65, 0x6f, 0x51, 0x18, 0x6c, 0x08, 0x4b,
        0x77, 0xca, 0x75, 0xc4, 0xc3, 0xde, 0x29, 0x41, 0x8b, 0xaf, 0xa7, 0x1c,
        0x28, 0x37, 0xa0, 0xa0, 0x74, 0x8e, 0x09, 0x42, 0x7a, 0x1b, 0x68, 0xdb
    },
    {
        0x14, 0x8f, 0xab, 0x28, 0x64, 0xea, 0x45, 0x88, 0xdb, 0xc1, 0xc6, 0xa0,
        0x48, 0xdf, 0x15, 0xd0, 0x28, 0x07, 0x2d, 0x6c, 0xb8, 0x42, 0xbb, 0x60,
        0x02, 0x08, 0x9e, 0x29, 0x9b, 0x8d, 0xd6, 0x1c, 0xaf, 0xf2, 0x1a, 0xdc,
        0xf0, 0x78, 0x0b, 0x4d, 0x90, 0xa1, 0x0c, 0xb3, 0x13, 0xde, 0xca, 0x5a
    },
    {
        0x4d, 0x80, 0x7d, 0x0b, 0xb9, 0x00, 0x6f, 0x65, 0x51, 0x65, 0x23, 0xde,
        0x72, 0xdc, 0x4f, 0x04, 0xa5, 0xa2, 0x90, 0x45, 0x51, 0x9e, 0xd0, 0x3a,
        0xe4, 0xd7, 0x78, 0xa3, 0x0f, 0x2d, 0x65, 0x12, 0xad, 0xc8, 0x92, 0x30,
        0x79, 0x9d, 0x9d, 0x08, 0x7a, 0x9c, 0x9f, 0x83, 0xb1, 0xca, 0x59, 0x56
    }
},
{ /* 41 */
    WC_HASH_TYPE_SHA384, 33, 0,
    { /* PSK */
        0xa4, 0x8b, 0x1b, 0x5f, 0xd0, 0xea, 0x75, 0x62,
        0x06, 0x4d, 0x68, 0x40, 0x85, 0x20, 0x45, 0x95,
        0x4a, 0x00, 0xca, 0x05, 0xeb, 0xd4, 0x1d, 0x48,
        0x81, 0x89, 0xe8, 0x86, 0x43, 0xfa, 0x28, 0x17,
        0x12
    },
    { 0 }, /* DHE */
    { /* Hello 1 */
        0x03, 0x7c, 0x33, 0x75, 0xdc, 0xc5, 0x46, 0x3a,
        0x0d, 0x56, 0xc6, 0xfb, 0xab, 0x1e, 0x1d, 0xda,
        0x59, 0xc2, 0xb2, 0xb1, 0x7c, 0x48, 0x9b, 0x06,
        0x0a, 0x5a, 0xbb, 0xf8, 0x98, 0x53, 0x78, 0x2d,
        0xd2, 0xcc, 0x87, 0x68, 0x25, 0xdd, 0x88, 0x22,
        0xcd, 0xb7, 0x74, 0x55, 0x21, 0xf9, 0x34, 0x98
    },
    { /* Hello 2 */
        0x03, 0xb4, 0xfb, 0xcc, 0x28, 0x2c, 0xc1, 0x70,
        0x42, 0x73, 0x57, 0xac, 0xdb, 0x47, 0x71, 0xf6,
        0x2e, 0x11, 0x8a, 0x5b, 0x47, 0x2f, 0x02, 0x54,
        0x95, 0x34, 0xed, 0x5f, 0x19, 0xc1, 0x75, 0xe0,
        0x76, 0xad, 0xb0, 0x90, 0x57, 0xcd, 0xfd, 0xd7,
        0x58, 0x1f, 0x0d, 0x6b, 0x9e, 0x51, 0x3c, 0x08
    },
    { /* Finished 1 */
        0x2b, 0x50, 0xd9, 0xa7, 0x43, 0x24, 0xda, 0x2c,
        0x7a, 0xaa, 0x0e, 0x37, 0xd7, 0x6b, 0x2c, 0xab,
        0x8e, 0xb2, 0xfe, 0x31, 0x1b, 0xa8, 0x12, 0x59,
        0x5b, 0x7b, 0xdc, 0x3e, 0xa7, 0x86, 0xa5, 0x48,
        0xe4, 0x46, 0x2b, 0x4c, 0xc1, 0x66, 0x4b, 0xf3,
        0x2a, 0x99, 0x93, 0x08, 0xbc, 0x3d, 0x08, 0x76
    },
    { /* Finished 2 */
        0x7c, 0x34, 0xc8, 0x56, 0x17, 0xf1, 0x62, 0x1c,
        0x9f, 0x0b, 0xeb, 0xfd, 0x69, 0x72, 0x51, 0xc5,
        0xfa, 0x74, 0x87, 0xc9, 0xbd, 0x50, 0xe9, 0x48,
        0xa7, 0x3c, 0x94, 0x3e, 0x06, 0x7d, 0xe8, 0x8e,
        0xc1, 0xd1, 0x08, 0x1f, 0x5d, 0x48, 0x8a, 0x25,
        0xfc, 0xea, 0xe7, 0xd9, 0xd4, 0xd0, 0xf9, 0xad
    },
    {
        0x4b, 0x0b, 0xed, 0xb9, 0xc8, 0xb8, 0xa8, 0x1e, 0xb0, 0x81, 0x76, 0xd5,
        0x33, 0x22, 0x71, 0x33, 0x3a, 0x85, 0x19, 0x67, 0x7e, 0x91, 0x37, 0xf2,
        0xa6, 0x11, 0x22, 0xdf, 0x41, 0x04, 0x3d, 0xa9, 0x13, 0xb9, 0xb2, 0xb1,
        0xbb, 0xd8, 0xef, 0x23, 0x7c, 0xc2, 0xab, 0x70, 0x1b, 0x51, 0x9f, 0xc9
    },
    {
        0xeb, 0x96, 0x10, 0x8c, 0x7d, 0x92, 0xea, 0x80, 0x86, 0xb2, 0xf8, 0x27,
        0xf2, 0x9a, 0x09, 0xc1, 0x7c, 0x09, 0x43, 0xbc, 0xfe, 0xc8, 0x75, 0xe0,
        0x97, 0xe7, 0x6d, 0xd5, 0xb2, 0x3c, 0xed, 0x12, 0xb7, 0x74, 0x0e, 0xe3,
        0xb6, 0xe0, 0xba, 0xe1, 0x8d, 0x89, 0xcf, 0x4f, 0x57, 0xf6, 0x6d, 0x90
    },
    {
        0x22, 0xb0, 0x39, 0x34, 0xb6, 0x6c, 0x2d, 0x7a, 0x97, 0x1c, 0x5d, 0xcc,
        0x78, 0x84, 0x71, 0xbb, 0xc6, 0x7b, 0xb6, 0xbc, 0xcc, 0x0b, 0xf8, 0xac,
        0x8e, 0xd7, 0x20, 0xbd, 0xbe, 0x32, 0xf0, 0xd6, 0xe9, 0x69, 0x13, 0xf2,
        0x9a, 0xce, 0xfe, 0x86, 0xd3, 0xee, 0xba, 0x69, 0x51, 0xb6, 0x77, 0x56
    },
    {
        0x16, 0xfd, 0xda, 0xf3, 0x5e, 0xb9, 0xa6, 0x17, 0x24, 0xb2, 0x16, 0x9f,
        0xb6, 0x59, 0x13, 0x0f, 0x25, 0x5a, 0xf1, 0x5b, 0x5f, 0xe4, 0x54, 0x2a,
        0xa7, 0xbf, 0x29, 0xaf, 0x5a, 0x77, 0xf4, 0x4f, 0x25, 0xba, 0x94, 0xad,
        0x6b, 0x91, 0x3b, 0xe7, 0xd5, 0x73, 0x0d, 0xff, 0xaa, 0xe3, 0x72, 0x2c
    },
    {
        0x22, 0xb4, 0x94, 0xc0, 0x53, 0xd7, 0x82, 0x06, 0x38, 0x9d, 0x4a, 0xa0,
        0x3f, 0xf1, 0x5f, 0x6e, 0x23, 0x8d, 0x09, 0x62, 0xbf, 0x6f, 0x7c, 0x84,
        0xc6, 0x3e, 0x15, 0xad, 0x18, 0x37, 0x76, 0x29, 0xc7, 0xd6, 0x68, 0x0c,
        0x1e, 0xc6, 0x93, 0x31, 0xef, 0x85, 0x69, 0x30, 0x68, 0xf0, 0x1e, 0x37
    },
    {
        0x6d, 0x4d, 0x20, 0xaf, 0x47, 0xe8, 0x1b, 0xfa, 0xd0, 0xb6, 0xc8, 0x97,
        0xd1, 0x03, 0xfc, 0x9d, 0x59, 0xa0, 0x68, 0x9d, 0xe9, 0x17, 0x8b, 0xce,
        0x48, 0x2c, 0x77, 0x8a, 0x22, 0x4b, 0x5c, 0x54, 0x22, 0xa1, 0x15, 0x12,
        0xe1, 0x07, 0x8e, 0x15, 0xd8, 0x7b, 0x16, 0x65, 0x99, 0x6b, 0xcb, 0x71
    },
    {
        0x79, 0x64, 0x79, 0xdd, 0x75, 0x5c, 0x6f, 0x98, 0xac, 0x03, 0xe0, 0xcd,
        0x92, 0xba, 0x0e, 0x2d, 0xb4, 0xd1, 0x8b, 0x97, 0xd0, 0x85, 0xbb, 0x2e,
        0x4f, 0x26, 0x93, 0xf5, 0x1d, 0xf3, 0xd2, 0x43, 0x4f, 0xd2, 0x47, 0xaa,
        0x91, 0x1e, 0xf3, 0x67, 0x10, 0x18, 0x2c, 0xb9, 0x01, 0xba, 0x10, 0x9f
    },
    {
        0x79, 0xb6, 0x9c, 0xbe, 0xf1, 0x6a, 0xb0, 0x92, 0xa0, 0x29, 0x52, 0x61,
        0xf1, 0xcd, 0x3a, 0x67, 0xe1, 0x6b, 0xb8, 0x9d, 0x0d, 0x95, 0xb6, 0x03,
        0x80, 0x1f, 0xd5, 0x75, 0xb6, 0x1d, 0x79, 0x02, 0x93, 0x43, 0x77, 0xa7,
        0x9d, 0x2f, 0xc3, 0x84, 0xc6, 0x83, 0x76, 0x16, 0x06, 0x98, 0x7b, 0x79
    }
},
#endif /* WOLFSSL_SHA384 */
};

const char protocolLabel[] = "tls13 ";
const char ceTrafficLabel[] = "c e traffic";
const char eExpMasterLabel[] = "e exp master";
const char cHsTrafficLabel[] = "c hs traffic";
const char sHsTrafficLabel[] = "s hs traffic";
const char cAppTrafficLabel[] = "c ap traffic";
const char sAppTrafficLabel[] = "s ap traffic";
const char expMasterLabel[] = "exp master";
const char resMasterLabel[] = "res master";
const char derivedLabel[] = "derived";


WOLFSSL_TEST_SUBROUTINE int tls13_kdf_test(void)
{
    int ret = 0;
    word32 i;
    word32 tc = sizeof(tls13KdfTestVectors)/sizeof(Tls13KdfTestVector);
    const Tls13KdfTestVector* tv = NULL;

    for (i = 0, tv = tls13KdfTestVectors; i < tc; i++, tv++) {
        byte output[WC_MAX_DIGEST_SIZE];
        byte secret[WC_MAX_DIGEST_SIZE];
        byte salt[WC_MAX_DIGEST_SIZE];
        byte zeroes[WC_MAX_DIGEST_SIZE];
        byte hashZero[WC_MAX_DIGEST_SIZE];
        int hashAlgSz;

        XMEMSET(zeroes, 0, sizeof zeroes);

        hashAlgSz = wc_HashGetDigestSize(tv->hashAlg);
        if (hashAlgSz == BAD_FUNC_ARG) break;
        ret = wc_Hash(tv->hashAlg, NULL, 0, hashZero, hashAlgSz);
        if (ret != 0) break;

        ret = wc_Tls13_HKDF_Extract(secret, NULL, 0,
                (tv->pskSz == 0) ? zeroes : (byte*)tv->psk,
                tv->pskSz, tv->hashAlg);
        if (ret != 0) break;

        ret = wc_Tls13_HKDF_Expand_Label(output, hashAlgSz,
                secret, hashAlgSz,
                (byte*)protocolLabel, (word32)strlen(protocolLabel),
                (byte*)ceTrafficLabel, (word32)strlen(ceTrafficLabel),
                tv->hashHello1, hashAlgSz, tv->hashAlg);
        if (ret != 0) break;

        ret = XMEMCMP(tv->clientEarlyTrafficSecret, output, hashAlgSz);
        if (ret != 0) break;

        ret = wc_Tls13_HKDF_Expand_Label(output, hashAlgSz,
                secret, hashAlgSz,
                (byte*)protocolLabel, (word32)strlen(protocolLabel),
                (byte*)eExpMasterLabel, (word32)strlen(eExpMasterLabel),
                tv->hashHello1, hashAlgSz, tv->hashAlg);
        if (ret != 0) break;

        ret = XMEMCMP(tv->earlyExporterMasterSecret, output, hashAlgSz);
        if (ret != 0) break;

        ret = wc_Tls13_HKDF_Expand_Label(salt, hashAlgSz,
                secret, hashAlgSz,
                (byte*)protocolLabel, (word32)strlen(protocolLabel),
                (byte*)derivedLabel, (word32)strlen(derivedLabel),
                hashZero, hashAlgSz, tv->hashAlg);
        if (ret != 0) break;

        ret = wc_Tls13_HKDF_Extract(secret, salt, hashAlgSz,
                (tv->dheSz == 0) ? zeroes : (byte*)tv->dhe,
                tv->dheSz, tv->hashAlg);
        if (ret != 0) break;

        ret = wc_Tls13_HKDF_Expand_Label(output, hashAlgSz,
                secret, hashAlgSz,
                (byte*)protocolLabel, (word32)strlen(protocolLabel),
                (byte*)cHsTrafficLabel, (word32)strlen(cHsTrafficLabel),
                tv->hashHello2, hashAlgSz, tv->hashAlg);
        if (ret != 0) break;

        ret = XMEMCMP(tv->clientHandshakeTrafficSecret,
                output, hashAlgSz);
        if (ret != 0) break;

        ret = wc_Tls13_HKDF_Expand_Label(output, hashAlgSz,
                secret, hashAlgSz,
                (byte*)protocolLabel, (word32)strlen(protocolLabel),
                (byte*)sHsTrafficLabel, (word32)strlen(sHsTrafficLabel),
                tv->hashHello2, hashAlgSz, tv->hashAlg);
        if (ret != 0) break;

        ret = XMEMCMP(tv->serverHandshakeTrafficSecret, output, hashAlgSz);
        if (ret != 0) break;

        ret = wc_Tls13_HKDF_Expand_Label(salt, hashAlgSz,
                secret, hashAlgSz,
                (byte*)protocolLabel, (word32)strlen(protocolLabel),
                (byte*)derivedLabel, (word32)strlen(derivedLabel),
                hashZero, hashAlgSz, tv->hashAlg);
        if (ret != 0) break;

        ret = wc_Tls13_HKDF_Extract(secret, salt, hashAlgSz,
                zeroes, hashAlgSz, tv->hashAlg);
        if (ret != 0) break;

        ret = wc_Tls13_HKDF_Expand_Label(output, hashAlgSz,
                secret, hashAlgSz,
                (byte*)protocolLabel, (word32)strlen(protocolLabel),
                (byte*)cAppTrafficLabel, (word32)strlen(cAppTrafficLabel),
                tv->hashFinished1, hashAlgSz, tv->hashAlg);
        if (ret != 0) break;

        ret = XMEMCMP(tv->clientApplicationTrafficSecret, output, hashAlgSz);
        if (ret != 0) break;

        ret = wc_Tls13_HKDF_Expand_Label(output, hashAlgSz,
                secret, hashAlgSz,
                (byte*)protocolLabel, (word32)strlen(protocolLabel),
                (byte*)sAppTrafficLabel, (word32)strlen(sAppTrafficLabel),
                tv->hashFinished1, hashAlgSz, tv->hashAlg);
        if (ret != 0) break;

        ret = XMEMCMP(tv->serverApplicationTrafficSecret, output, hashAlgSz);
        if (ret != 0) break;

        ret = wc_Tls13_HKDF_Expand_Label(output, hashAlgSz,
                secret, hashAlgSz,
                (byte*)protocolLabel, (word32)strlen(protocolLabel),
                (byte*)expMasterLabel, (word32)strlen(expMasterLabel),
                tv->hashFinished1, hashAlgSz, tv->hashAlg);
        if (ret != 0) break;

        ret = XMEMCMP(tv->exporterMasterSecret, output, hashAlgSz);
        if (ret != 0) break;

        ret = wc_Tls13_HKDF_Expand_Label(output, hashAlgSz,
                secret, hashAlgSz,
                (byte*)protocolLabel, (word32)strlen(protocolLabel),
                (byte*)resMasterLabel, (word32)strlen(resMasterLabel),
                tv->hashFinished2, hashAlgSz, tv->hashAlg);
        if (ret != 0) break;

        ret = XMEMCMP(tv->resumptionMasterSecret, output, hashAlgSz);
        if (ret != 0) break;
    }

    return ret;
}

#endif /* WOLFSSL_TLS13 */


#if defined(HAVE_ECC) && defined(HAVE_X963_KDF)

WOLFSSL_TEST_SUBROUTINE int x963kdf_test(void)
{
    int ret;
    byte kek[128];

#ifndef NO_SHA
    /* SHA-1, COUNT = 0
     * shared secret length: 192
     * SharedInfo length: 0
     * key data length: 128
     */
    WOLFSSL_SMALL_STACK_STATIC const byte Z[] = {
        0x1c, 0x7d, 0x7b, 0x5f, 0x05, 0x97, 0xb0, 0x3d,
        0x06, 0xa0, 0x18, 0x46, 0x6e, 0xd1, 0xa9, 0x3e,
        0x30, 0xed, 0x4b, 0x04, 0xdc, 0x64, 0xcc, 0xdd
    };

    WOLFSSL_SMALL_STACK_STATIC const byte verify[] = {
        0xbf, 0x71, 0xdf, 0xfd, 0x8f, 0x4d, 0x99, 0x22,
        0x39, 0x36, 0xbe, 0xb4, 0x6f, 0xee, 0x8c, 0xcc
    };
#endif

#ifndef NO_SHA256
    /* SHA-256, COUNT = 3
     * shared secret length: 192
     * SharedInfo length: 0
     * key data length: 128
     */
    WOLFSSL_SMALL_STACK_STATIC const byte Z2[] = {
        0xd3, 0x8b, 0xdb, 0xe5, 0xc4, 0xfc, 0x16, 0x4c,
        0xdd, 0x96, 0x7f, 0x63, 0xc0, 0x4f, 0xe0, 0x7b,
        0x60, 0xcd, 0xe8, 0x81, 0xc2, 0x46, 0x43, 0x8c
    };

    WOLFSSL_SMALL_STACK_STATIC const byte verify2[] = {
        0x5e, 0x67, 0x4d, 0xb9, 0x71, 0xba, 0xc2, 0x0a,
        0x80, 0xba, 0xd0, 0xd4, 0x51, 0x4d, 0xc4, 0x84
    };
#endif

#ifdef WOLFSSL_SHA512
    /* SHA-512, COUNT = 0
     * shared secret length: 192
     * SharedInfo length: 0
     * key data length: 128
     */
    WOLFSSL_SMALL_STACK_STATIC const byte Z3[] = {
        0x87, 0xfc, 0x0d, 0x8c, 0x44, 0x77, 0x48, 0x5b,
        0xb5, 0x74, 0xf5, 0xfc, 0xea, 0x26, 0x4b, 0x30,
        0x88, 0x5d, 0xc8, 0xd9, 0x0a, 0xd8, 0x27, 0x82
    };

    WOLFSSL_SMALL_STACK_STATIC const byte verify3[] = {
        0x94, 0x76, 0x65, 0xfb, 0xb9, 0x15, 0x21, 0x53,
        0xef, 0x46, 0x02, 0x38, 0x50, 0x6a, 0x02, 0x45
    };

    /* SHA-512, COUNT = 0
     * shared secret length: 521
     * SharedInfo length: 128
     * key data length: 1024
     */
    WOLFSSL_SMALL_STACK_STATIC const byte Z4[] = {
        0x00, 0xaa, 0x5b, 0xb7, 0x9b, 0x33, 0xe3, 0x89,
        0xfa, 0x58, 0xce, 0xad, 0xc0, 0x47, 0x19, 0x7f,
        0x14, 0xe7, 0x37, 0x12, 0xf4, 0x52, 0xca, 0xa9,
        0xfc, 0x4c, 0x9a, 0xdb, 0x36, 0x93, 0x48, 0xb8,
        0x15, 0x07, 0x39, 0x2f, 0x1a, 0x86, 0xdd, 0xfd,
        0xb7, 0xc4, 0xff, 0x82, 0x31, 0xc4, 0xbd, 0x0f,
        0x44, 0xe4, 0x4a, 0x1b, 0x55, 0xb1, 0x40, 0x47,
        0x47, 0xa9, 0xe2, 0xe7, 0x53, 0xf5, 0x5e, 0xf0,
        0x5a, 0x2d
    };

    WOLFSSL_SMALL_STACK_STATIC const byte info4[] = {
        0xe3, 0xb5, 0xb4, 0xc1, 0xb0, 0xd5, 0xcf, 0x1d,
        0x2b, 0x3a, 0x2f, 0x99, 0x37, 0x89, 0x5d, 0x31
    };

    WOLFSSL_SMALL_STACK_STATIC const byte verify4[] = {
        0x44, 0x63, 0xf8, 0x69, 0xf3, 0xcc, 0x18, 0x76,
        0x9b, 0x52, 0x26, 0x4b, 0x01, 0x12, 0xb5, 0x85,
        0x8f, 0x7a, 0xd3, 0x2a, 0x5a, 0x2d, 0x96, 0xd8,
        0xcf, 0xfa, 0xbf, 0x7f, 0xa7, 0x33, 0x63, 0x3d,
        0x6e, 0x4d, 0xd2, 0xa5, 0x99, 0xac, 0xce, 0xb3,
        0xea, 0x54, 0xa6, 0x21, 0x7c, 0xe0, 0xb5, 0x0e,
        0xef, 0x4f, 0x6b, 0x40, 0xa5, 0xc3, 0x02, 0x50,
        0xa5, 0xa8, 0xee, 0xee, 0x20, 0x80, 0x02, 0x26,
        0x70, 0x89, 0xdb, 0xf3, 0x51, 0xf3, 0xf5, 0x02,
        0x2a, 0xa9, 0x63, 0x8b, 0xf1, 0xee, 0x41, 0x9d,
        0xea, 0x9c, 0x4f, 0xf7, 0x45, 0xa2, 0x5a, 0xc2,
        0x7b, 0xda, 0x33, 0xca, 0x08, 0xbd, 0x56, 0xdd,
        0x1a, 0x59, 0xb4, 0x10, 0x6c, 0xf2, 0xdb, 0xbc,
        0x0a, 0xb2, 0xaa, 0x8e, 0x2e, 0xfa, 0x7b, 0x17,
        0x90, 0x2d, 0x34, 0x27, 0x69, 0x51, 0xce, 0xcc,
        0xab, 0x87, 0xf9, 0x66, 0x1c, 0x3e, 0x88, 0x16
    };
#endif

#ifndef NO_SHA
    ret = wc_X963_KDF(WC_HASH_TYPE_SHA, Z, sizeof(Z), NULL, 0,
                      kek, sizeof(verify));
    if (ret != 0)
        return -9800;

    if (XMEMCMP(verify, kek, sizeof(verify)) != 0)
        return -9801;
#endif

#ifndef NO_SHA256
    ret = wc_X963_KDF(WC_HASH_TYPE_SHA256, Z2, sizeof(Z2), NULL, 0,
                      kek, sizeof(verify2));
    if (ret != 0)
        return -9802;

    if (XMEMCMP(verify2, kek, sizeof(verify2)) != 0)
        return -9803;
#endif

#ifdef WOLFSSL_SHA512
    ret = wc_X963_KDF(WC_HASH_TYPE_SHA512, Z3, sizeof(Z3), NULL, 0,
                      kek, sizeof(verify3));
    if (ret != 0)
        return -9804;

    if (XMEMCMP(verify3, kek, sizeof(verify3)) != 0)
        return -9805;

    ret = wc_X963_KDF(WC_HASH_TYPE_SHA512, Z4, sizeof(Z4), info4,
                      sizeof(info4), kek, sizeof(verify4));
    if (ret != 0)
        return -9806;

    if (XMEMCMP(verify4, kek, sizeof(verify4)) != 0)
        return -9807;
#endif

    return 0;
}

#endif /* HAVE_X963_KDF */


#ifdef HAVE_ECC

/* size to use for ECC key gen tests */
#ifndef ECC_KEYGEN_SIZE
    #ifndef NO_ECC256
        #define ECC_KEYGEN_SIZE 32
    #elif defined(HAVE_ECC384)
        #define ECC_KEYGEN_SIZE 48
    #elif defined(HAVE_ECC224)
        #define ECC_KEYGEN_SIZE 28
    #elif defined(HAVE_ECC521)
        #define ECC_KEYGEN_SIZE 66
    #else
        #error No ECC keygen size defined for test
    #endif
#endif
#ifdef BENCH_EMBEDDED
    #define ECC_SHARED_SIZE 128
#else
    #define ECC_SHARED_SIZE MAX_ECC_BYTES
#endif
#define ECC_DIGEST_SIZE     MAX_ECC_BYTES
#define ECC_SIG_SIZE        ECC_MAX_SIG_SIZE

#ifndef NO_ECC_VECTOR_TEST
    #if (defined(HAVE_ECC192) || defined(HAVE_ECC224) ||\
         !defined(NO_ECC256) || defined(HAVE_ECC384) ||\
         defined(HAVE_ECC521) || defined(HAVE_ALL_CURVES))
        #define HAVE_ECC_VECTOR_TEST
    #endif
#endif

#ifdef HAVE_ECC_VECTOR_TEST
typedef struct eccVector {
    const char* msg; /* SHA-1 Encoded Message */
    const char* Qx;
    const char* Qy;
    const char* d; /* Private Key */
    const char* R;
    const char* S;
    const char* curveName;
    word32 msgLen;
    word32 keySize;
#ifndef NO_ASN
    const byte* r;
    word32      rSz;
    const byte* s;
    word32      sSz;
#endif
} eccVector;

#if !defined(WOLF_CRYPTO_CB_ONLY_ECC)
static int ecc_test_vector_item(const eccVector* vector)
{
    int ret = 0, verify = 0;
    word32  sigSz;
#ifdef WOLFSSL_SMALL_STACK
    ecc_key *userA = (ecc_key *)XMALLOC(sizeof *userA, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
#else
    ecc_key userA[1];
#endif
    WC_DECLARE_VAR(sig, byte, ECC_SIG_SIZE, HEAP_HINT);
#if !defined(NO_ASN) && !defined(HAVE_SELFTEST)
    word32  sigRawSz, rSz = MAX_ECC_BYTES, sSz = MAX_ECC_BYTES;
    WC_DECLARE_VAR(sigRaw, byte, ECC_SIG_SIZE, HEAP_HINT);
    WC_DECLARE_VAR(r, byte, MAX_ECC_BYTES, HEAP_HINT);
    WC_DECLARE_VAR(s, byte, MAX_ECC_BYTES, HEAP_HINT);
#endif

#ifdef WC_DECLARE_VAR_IS_HEAP_ALLOC
    if (sig == NULL)
        ERROR_OUT(MEMORY_E, done);
#if !defined(NO_ASN) && !defined(HAVE_SELFTEST)
    if (sigRaw == NULL || r == NULL || s == NULL)
        ERROR_OUT(MEMORY_E, done);
#endif
#endif

#ifdef WOLFSSL_SMALL_STACK
    if (userA == NULL)
        ERROR_OUT(MEMORY_E, done);
#endif

    ret = wc_ecc_init_ex(userA, HEAP_HINT, devId);
    if (ret != 0)
        goto done;

    ret = wc_ecc_import_raw(userA, vector->Qx, vector->Qy,
                                                  vector->d, vector->curveName);
    if (ret != 0)
        goto done;

    XMEMSET(sig, 0, ECC_SIG_SIZE);
    sigSz = ECC_SIG_SIZE;
    ret = wc_ecc_rs_to_sig(vector->R, vector->S, sig, &sigSz);
    if (ret != 0)
        goto done;

#if !defined(NO_ASN) && !defined(HAVE_SELFTEST)
    XMEMSET(sigRaw, 0, ECC_SIG_SIZE);
    sigRawSz = ECC_SIG_SIZE;
    ret = wc_ecc_rs_raw_to_sig(vector->r, vector->rSz, vector->s, vector->sSz,
                                                             sigRaw, &sigRawSz);
    if (ret != 0)
        goto done;
    if (sigSz != sigRawSz || XMEMCMP(sig, sigRaw, sigSz) != 0) {
        ret = -9810;
        goto done;
    }

    ret = wc_ecc_sig_to_rs(sig, sigSz, r, &rSz, s, &sSz);
    if (ret != 0)
        goto done;
    if (rSz != vector->rSz || XMEMCMP(r, vector->r, rSz) != 0 ||
        sSz != vector->sSz || XMEMCMP(s, vector->s, sSz) != 0) {
        ret = -9811;
        goto done;
    }
#endif

    do {
    #if defined(WOLFSSL_ASYNC_CRYPT)
        ret = wc_AsyncWait(ret, &userA->asyncDev, WC_ASYNC_FLAG_CALL_AGAIN);
    #endif
        if (ret == 0)
            ret = wc_ecc_verify_hash(sig, sigSz, (byte*)vector->msg,
                                               vector->msgLen, &verify, userA);
    } while (ret == WC_PENDING_E);
    if (ret != 0)
        goto done;
    TEST_SLEEP();

    if (verify != 1)
        ret = -9812;

done:

#ifdef WOLFSSL_SMALL_STACK
    if (userA != NULL) {
        wc_ecc_free(userA);
        XFREE(userA, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    }
#else
    wc_ecc_free(userA);
#endif

#if !defined(NO_ASN) && !defined(HAVE_SELFTEST)
    WC_FREE_VAR(sigRaw, HEAP_HINT);
    WC_FREE_VAR(r, HEAP_HINT);
    WC_FREE_VAR(s, HEAP_HINT);
#endif
    WC_FREE_VAR(sig, HEAP_HINT);

    return ret;
}

static int ecc_test_vector(int keySize)
{
    int     ret;
    eccVector vec;

    XMEMSET(&vec, 0, sizeof(vec));
    vec.keySize = (word32)keySize;

    switch(keySize) {

#if defined(HAVE_ECC112) || defined(HAVE_ALL_CURVES)
    case 14:
        return 0;
#endif /* HAVE_ECC112 */
#if defined(HAVE_ECC128) || defined(HAVE_ALL_CURVES)
    case 16:
        return 0;
#endif /* HAVE_ECC128 */
#if defined(HAVE_ECC160) || defined(HAVE_ALL_CURVES)
    case 20:
        return 0;
#endif /* HAVE_ECC160 */

#if defined(HAVE_ECC192) || defined(HAVE_ALL_CURVES)
    case 24:
        /* first [P-192,SHA-1] vector from FIPS 186-3 NIST vectors */
        #if 1
            vec.msg = "\x60\x80\x79\x42\x3f\x12\x42\x1d\xe6\x16\xb7\x49\x3e\xbe\x55\x1c\xf4\xd6\x5b\x92";
            vec.msgLen = 20;
        #else
            /* This is the raw message prior to SHA-1 */
            vec.msg =
                "\xeb\xf7\x48\xd7\x48\xeb\xbc\xa7\xd2\x9f\xb4\x73\x69\x8a\x6e\x6b"
                "\x4f\xb1\x0c\x86\x5d\x4a\xf0\x24\xcc\x39\xae\x3d\xf3\x46\x4b\xa4"
                "\xf1\xd6\xd4\x0f\x32\xbf\x96\x18\xa9\x1b\xb5\x98\x6f\xa1\xa2\xaf"
                "\x04\x8a\x0e\x14\xdc\x51\xe5\x26\x7e\xb0\x5e\x12\x7d\x68\x9d\x0a"
                "\xc6\xf1\xa7\xf1\x56\xce\x06\x63\x16\xb9\x71\xcc\x7a\x11\xd0\xfd"
                "\x7a\x20\x93\xe2\x7c\xf2\xd0\x87\x27\xa4\xe6\x74\x8c\xc3\x2f\xd5"
                "\x9c\x78\x10\xc5\xb9\x01\x9d\xf2\x1c\xdc\xc0\xbc\xa4\x32\xc0\xa3"
                "\xee\xd0\x78\x53\x87\x50\x88\x77\x11\x43\x59\xce\xe4\xa0\x71\xcf";
            vec.msgLen = 128;
        #endif
        vec.Qx  = "07008ea40b08dbe76432096e80a2494c94982d2d5bcf98e6";
        vec.Qy  = "76fab681d00b414ea636ba215de26d98c41bd7f2e4d65477";
        vec.d   = "e14f37b3d1374ff8b03f41b9b3fdd2f0ebccf275d660d7f3";
        vec.R   = "6994d962bdd0d793ffddf855ec5bf2f91a9698b46258a63e";
        vec.S   = "02ba6465a234903744ab02bc8521405b73cf5fc00e1a9f41";
        vec.curveName = "SECP192R1";
    #ifndef NO_ASN
        vec.r   = (byte*)"\x69\x94\xd9\x62\xbd\xd0\xd7\x93\xff\xdd\xf8\x55"
                         "\xec\x5b\xf2\xf9\x1a\x96\x98\xb4\x62\x58\xa6\x3e";
        vec.rSz = 24;
        vec.s   = (byte*)"\x02\xba\x64\x65\xa2\x34\x90\x37\x44\xab\x02\xbc"
                         "\x85\x21\x40\x5b\x73\xcf\x5f\xc0\x0e\x1a\x9f\x41";
        vec.sSz = 24;
    #endif
        break;
#endif /* HAVE_ECC192 */

#if defined(HAVE_ECC224) || defined(HAVE_ALL_CURVES)
    case 28:
        /* first [P-224,SHA-1] vector from FIPS 186-3 NIST vectors */
        #if 1
            vec.msg = "\xb9\xa3\xb8\x6d\xb0\xba\x99\xfd\xc6\xd2\x94\x6b\xfe\xbe\x9c\xe8\x3f\x10\x74\xfc";
            vec.msgLen = 20;
        #else
            /* This is the raw message prior to SHA-1 */
            vec.msg =
                "\x36\xc8\xb2\x29\x86\x48\x7f\x67\x7c\x18\xd0\x97\x2a\x9e\x20\x47"
                "\xb3\xaf\xa5\x9e\xc1\x62\x76\x4e\xc3\x0b\x5b\x69\xe0\x63\x0f\x99"
                "\x0d\x4e\x05\xc2\x73\xb0\xe5\xa9\xd4\x28\x27\xb6\x95\xfc\x2d\x64"
                "\xd9\x13\x8b\x1c\xf4\xc1\x21\x55\x89\x4c\x42\x13\x21\xa7\xbb\x97"
                "\x0b\xdc\xe0\xfb\xf0\xd2\xae\x85\x61\xaa\xd8\x71\x7f\x2e\x46\xdf"
                "\xe3\xff\x8d\xea\xb4\xd7\x93\x23\x56\x03\x2c\x15\x13\x0d\x59\x9e"
                "\x26\xc1\x0f\x2f\xec\x96\x30\x31\xac\x69\x38\xa1\x8d\x66\x45\x38"
                "\xb9\x4d\xac\x55\x34\xef\x7b\x59\x94\x24\xd6\x9b\xe1\xf7\x1c\x20";
            vec.msgLen = 128;
        #endif
        vec.Qx  = "8a4dca35136c4b70e588e23554637ae251077d1365a6ba5db9585de7";
        vec.Qy  = "ad3dee06de0be8279d4af435d7245f14f3b4f82eb578e519ee0057b1";
        vec.d   = "97c4b796e1639dd1035b708fc00dc7ba1682cec44a1002a1a820619f";
        vec.R   = "147b33758321e722a0360a4719738af848449e2c1d08defebc1671a7";
        vec.S   = "24fc7ed7f1352ca3872aa0916191289e2e04d454935d50fe6af3ad5b";
        vec.curveName = "SECP224R1";
    #ifndef NO_ASN
        vec.r   = (byte*)"\x14\x7b\x33\x75\x83\x21\xe7\x22\xa0\x36\x0a\x47"
                         "\x19\x73\x8a\xf8\x48\x44\x9e\x2c\x1d\x08\xde\xfe"
                         "\xbc\x16\x71\xa7";
        vec.rSz = 28;
        vec.s   = (byte*)"\x24\xfc\x7e\xd7\xf1\x35\x2c\xa3\x87\x2a\xa0\x91"
                         "\x61\x91\x28\x9e\x2e\x04\xd4\x54\x93\x5d\x50\xfe"
                         "\x6a\xf3\xad\x5b";
        vec.sSz = 28;
    #endif
        break;
#endif /* HAVE_ECC224 */

#if defined(HAVE_ECC239) || defined(HAVE_ALL_CURVES)
    case 30:
        return 0;
#endif /* HAVE_ECC239 */

#if !defined(NO_ECC256) || defined(HAVE_ALL_CURVES)
    case 32:
        /* first [P-256,SHA-1] vector from FIPS 186-3 NIST vectors */
        #if 1
            vec.msg = "\xa3\xf9\x1a\xe2\x1b\xa6\xb3\x03\x98\x64\x47\x2f\x18\x41\x44\xc6\xaf\x62\xcd\x0e";
            vec.msgLen = 20;
        #else
            /* This is the raw message prior to SHA-1 */
            vec.msg =
                "\xa2\x4b\x21\x76\x2e\x6e\xdb\x15\x3c\xc1\x14\x38\xdb\x0e\x92\xcd"
                "\xf5\x2b\x86\xb0\x6c\xa9\x70\x16\x06\x27\x59\xc7\x0d\x36\xd1\x56"
                "\x2c\xc9\x63\x0d\x7f\xc7\xc7\x74\xb2\x8b\x54\xe3\x1e\xf5\x58\x72"
                "\xb2\xa6\x5d\xf1\xd7\xec\x26\xde\xbb\x33\xe7\xd9\x27\xef\xcc\xf4"
                "\x6b\x63\xde\x52\xa4\xf4\x31\xea\xca\x59\xb0\x5d\x2e\xde\xc4\x84"
                "\x5f\xff\xc0\xee\x15\x03\x94\xd6\x1f\x3d\xfe\xcb\xcd\xbf\x6f\x5a"
                "\x73\x38\xd0\xbe\x3f\x2a\x77\x34\x51\x98\x3e\xba\xeb\x48\xf6\x73"
                "\x8f\xc8\x95\xdf\x35\x7e\x1a\x48\xa6\x53\xbb\x35\x5a\x31\xa1\xb4"
            vec.msgLen = 128;
        #endif
        vec.Qx  = "fa2737fb93488d19caef11ae7faf6b7f4bcd67b286e3fc54e8a65c2b74aeccb0";
        vec.Qy  = "d4ccd6dae698208aa8c3a6f39e45510d03be09b2f124bfc067856c324f9b4d09";
        vec.d   = "be34baa8d040a3b991f9075b56ba292f755b90e4b6dc10dad36715c33cfdac25";
        vec.R   = "2b826f5d44e2d0b6de531ad96b51e8f0c56fdfead3c236892e4d84eacfc3b75c";
        vec.S   = "a2248b62c03db35a7cd63e8a120a3521a89d3d2f61ff99035a2148ae32e3a248";
    #ifndef NO_ASN
        vec.r   = (byte*)"\x2b\x82\x6f\x5d\x44\xe2\xd0\xb6\xde\x53\x1a\xd9"
                         "\x6b\x51\xe8\xf0\xc5\x6f\xdf\xea\xd3\xc2\x36\x89"
                         "\x2e\x4d\x84\xea\xcf\xc3\xb7\x5c";
        vec.rSz = 32;
        vec.s   = (byte*)"\xa2\x24\x8b\x62\xc0\x3d\xb3\x5a\x7c\xd6\x3e\x8a"
                         "\x12\x0a\x35\x21\xa8\x9d\x3d\x2f\x61\xff\x99\x03"
                         "\x5a\x21\x48\xae\x32\xe3\xa2\x48";
        vec.sSz = 32;
    #endif
        vec.curveName = "SECP256R1";
        break;
#endif /* !NO_ECC256 */

#if defined(HAVE_ECC320) || defined(HAVE_ALL_CURVES)
    case 40:
        return 0;
#endif /* HAVE_ECC320 */

#if defined(HAVE_ECC384) || defined(HAVE_ALL_CURVES)
    case 48:
        /* first [P-384,SHA-1] vector from FIPS 186-3 NIST vectors */
        #if 1
            vec.msg = "\x9b\x9f\x8c\x95\x35\xa5\xca\x26\x60\x5d\xb7\xf2\xfa\x57\x3b\xdf\xc3\x2e\xab\x8b";
            vec.msgLen = 20;
        #else
            /* This is the raw message prior to SHA-1 */
            vec.msg =
                "\xab\xe1\x0a\xce\x13\xe7\xe1\xd9\x18\x6c\x48\xf7\x88\x9d\x51\x47"
                "\x3d\x3a\x09\x61\x98\x4b\xc8\x72\xdf\x70\x8e\xcc\x3e\xd3\xb8\x16"
                "\x9d\x01\xe3\xd9\x6f\xc4\xf1\xd5\xea\x00\xa0\x36\x92\xbc\xc5\xcf"
                "\xfd\x53\x78\x7c\x88\xb9\x34\xaf\x40\x4c\x03\x9d\x32\x89\xb5\xba"
                "\xc5\xae\x7d\xb1\x49\x68\x75\xb5\xdc\x73\xc3\x09\xf9\x25\xc1\x3d"
                "\x1c\x01\xab\xda\xaf\xeb\xcd\xac\x2c\xee\x43\x39\x39\xce\x8d\x4a"
                "\x0a\x5d\x57\xbb\x70\x5f\x3b\xf6\xec\x08\x47\x95\x11\xd4\xb4\xa3"
                "\x21\x1f\x61\x64\x9a\xd6\x27\x43\x14\xbf\x0d\x43\x8a\x81\xe0\x60"
            vec.msgLen = 128;
        #endif
        vec.Qx  = "e55fee6c49d8d523f5ce7bf9c0425ce4ff650708b7de5cfb095901523979a7f042602db30854735369813b5c3f5ef868";
        vec.Qy  = "28f59cc5dc509892a988d38a8e2519de3d0c4fd0fbdb0993e38f18506c17606c5e24249246f1ce94983a5361c5be983e";
        vec.d   = "a492ce8fa90084c227e1a32f7974d39e9ff67a7e8705ec3419b35fb607582bebd461e0b1520ac76ec2dd4e9b63ebae71";
        vec.R   = "6820b8585204648aed63bdff47f6d9acebdea62944774a7d14f0e14aa0b9a5b99545b2daee6b3c74ebf606667a3f39b7";
        vec.S   = "491af1d0cccd56ddd520b233775d0bc6b40a6255cc55207d8e9356741f23c96c14714221078dbd5c17f4fdd89b32a907";
        vec.curveName = "SECP384R1";
    #ifndef NO_ASN
        vec.r   = (byte*)"\x68\x20\xb8\x58\x52\x04\x64\x8a\xed\x63\xbd\xff"
                         "\x47\xf6\xd9\xac\xeb\xde\xa6\x29\x44\x77\x4a\x7d"
                         "\x14\xf0\xe1\x4a\xa0\xb9\xa5\xb9\x95\x45\xb2\xda"
                         "\xee\x6b\x3c\x74\xeb\xf6\x06\x66\x7a\x3f\x39\xb7";
        vec.rSz = 48;
        vec.s   = (byte*)"\x49\x1a\xf1\xd0\xcc\xcd\x56\xdd\xd5\x20\xb2\x33"
                         "\x77\x5d\x0b\xc6\xb4\x0a\x62\x55\xcc\x55\x20\x7d"
                         "\x8e\x93\x56\x74\x1f\x23\xc9\x6c\x14\x71\x42\x21"
                         "\x07\x8d\xbd\x5c\x17\xf4\xfd\xd8\x9b\x32\xa9\x07";
        vec.sSz = 48;
    #endif
        break;
#endif /* HAVE_ECC384 */

#if defined(HAVE_ECC512) || defined(HAVE_ALL_CURVES)
    case 64:
        return 0;
#endif /* HAVE_ECC512 */

#if defined(HAVE_ECC521) || defined(HAVE_ALL_CURVES)
    case 66:
        /* first [P-521,SHA-1] vector from FIPS 186-3 NIST vectors */
        #if 1
            vec.msg = "\x1b\xf7\x03\x9c\xca\x23\x94\x27\x3f\x11\xa1\xd4\x8d\xcc\xb4\x46\x6f\x31\x61\xdf";
            vec.msgLen = 20;
        #else
            /* This is the raw message prior to SHA-1 */
            vec.msg =
                "\x50\x3f\x79\x39\x34\x0a\xc7\x23\xcd\x4a\x2f\x4e\x6c\xcc\x27\x33"
                "\x38\x3a\xca\x2f\xba\x90\x02\x19\x9d\x9e\x1f\x94\x8b\xe0\x41\x21"
                "\x07\xa3\xfd\xd5\x14\xd9\x0c\xd4\xf3\x7c\xc3\xac\x62\xef\x00\x3a"
                "\x2d\xb1\xd9\x65\x7a\xb7\x7f\xe7\x55\xbf\x71\xfa\x59\xe4\xd9\x6e"
                "\xa7\x2a\xe7\xbf\x9d\xe8\x7d\x79\x34\x3b\xc1\xa4\xbb\x14\x4d\x16"
                "\x28\xd1\xe9\xe9\xc8\xed\x80\x8b\x96\x2c\x54\xe5\xf9\x6d\x53\xda"
                "\x14\x7a\x96\x38\xf9\x4a\x91\x75\xd8\xed\x61\x05\x5f\x0b\xa5\x73"
                "\xa8\x2b\xb7\xe0\x18\xee\xda\xc4\xea\x7b\x36\x2e\xc8\x9c\x38\x2b"
            vec.msgLen = 128;
        #endif
        vec.Qx  = "12fbcaeffa6a51f3ee4d3d2b51c5dec6d7c726ca353fc014ea2bf7cfbb9b910d32cbfa6a00fe39b6cdb8946f22775398b2e233c0cf144d78c8a7742b5c7a3bb5d23";
        vec.Qy  = "09cdef823dd7bf9a79e8cceacd2e4527c231d0ae5967af0958e931d7ddccf2805a3e618dc3039fec9febbd33052fe4c0fee98f033106064982d88f4e03549d4a64d";
        vec.d   = "1bd56bd106118eda246155bd43b42b8e13f0a6e25dd3bb376026fab4dc92b6157bc6dfec2d15dd3d0cf2a39aa68494042af48ba9601118da82c6f2108a3a203ad74";
        vec.R   = "0bd117b4807710898f9dd7778056485777668f0e78e6ddf5b000356121eb7a220e9493c7f9a57c077947f89ac45d5acb6661bbcd17abb3faea149ba0aa3bb1521be";
        vec.S   = "019cd2c5c3f9870ecdeb9b323abdf3a98cd5e231d85c6ddc5b71ab190739f7f226e6b134ba1d5889ddeb2751dabd97911dff90c34684cdbe7bb669b6c3d22f2480c";
        vec.curveName = "SECP521R1";
    #ifndef NO_ASN
        vec.r   = (byte*)"\xbd\x11\x7b\x48\x07\x71\x08\x98\xf9\xdd\x77\x78"
                         "\x05\x64\x85\x77\x76\x68\xf0\xe7\x8e\x6d\xdf\x5b"
                         "\x00\x03\x56\x12\x1e\xb7\xa2\x20\xe9\x49\x3c\x7f"
                         "\x9a\x57\xc0\x77\x94\x7f\x89\xac\x45\xd5\xac\xb6"
                         "\x66\x1b\xbc\xd1\x7a\xbb\x3f\xae\xa1\x49\xba\x0a"
                         "\xa3\xbb\x15\x21\xbe";
        vec.rSz = 65;
        vec.s   = (byte*)"\x19\xcd\x2c\x5c\x3f\x98\x70\xec\xde\xb9\xb3\x23"
                         "\xab\xdf\x3a\x98\xcd\x5e\x23\x1d\x85\xc6\xdd\xc5"
                         "\xb7\x1a\xb1\x90\x73\x9f\x7f\x22\x6e\x6b\x13\x4b"
                         "\xa1\xd5\x88\x9d\xde\xb2\x75\x1d\xab\xd9\x79\x11"
                         "\xdf\xf9\x0c\x34\x68\x4c\xdb\xe7\xbb\x66\x9b\x6c"
                         "\x3d\x22\xf2\x48\x0c";
        vec.sSz = 65;
    #endif
        break;
#endif /* HAVE_ECC521 */
    default:
        return NOT_COMPILED_IN; /* Invalid key size / Not supported */
    }; /* Switch */

    ret = ecc_test_vector_item(&vec);
    if (ret < 0) {
        return ret;
    }

    return 0;
}
#endif /* WOLF_CRYPTO_CB_ONLY_ECC */

#if defined(HAVE_ECC_SIGN) && defined(WOLFSSL_ECDSA_DETERMINISTIC_K)
static int ecc_test_deterministic_k(WC_RNG* rng)
{
    int ret;
    ecc_key key;
    byte sig[72];
    word32 sigSz;
    unsigned char msg[] = "sample";
    unsigned char hash[32];
    const char* dIUT =
        "C9AFA9D845BA75166B5C215767B1D6934E50C3DB36E89B127B8A622B120F6721";
    const char* QIUTx =
        "60FED4BA255A9D31C961EB74C6356D68C049B8923B61FA6CE669622E60F29FB6";
    const char* QIUTy =
        "7903FE1008B8BC99A41AE9E95628BC64F2F1B20C2D7E9F5177A3C294D4462299";
    const byte expSig[] = {
        0x30, 0x46, 0x02, 0x21, 0x00, 0xEF, 0xD4, 0x8B,
        0x2A, 0xAC, 0xB6, 0xA8, 0xFD, 0x11, 0x40, 0xDD,
        0x9C, 0xD4, 0x5E, 0x81, 0xD6, 0x9D, 0x2C, 0x87,
        0x7B, 0x56, 0xAA, 0xF9, 0x91, 0xC3, 0x4D, 0x0E,
        0xA8, 0x4E, 0xAF, 0x37, 0x16, 0x02, 0x21, 0x00,
        0xF7, 0xCB, 0x1C, 0x94, 0x2D, 0x65, 0x7C, 0x41,
        0xD4, 0x36, 0xC7, 0xA1, 0xB6, 0xE2, 0x9F, 0x65,
        0xF3, 0xE9, 0x00, 0xDB, 0xB9, 0xAF, 0xF4, 0x06,
        0x4D, 0xC4, 0xAB, 0x2F, 0x84, 0x3A, 0xCD, 0xA8
    };

    ret = wc_ecc_init_ex(&key, HEAP_HINT, devId);
    if (ret != 0) {
        return ret;
    }
    ret = wc_ecc_import_raw(&key, QIUTx, QIUTy, dIUT, "SECP256R1");
    if (ret != 0) {
        goto done;
    }

    ret = wc_Hash(WC_HASH_TYPE_SHA256, msg,
            (word32)XSTRLEN((const char*)msg), hash, sizeof(hash));
    if (ret != 0) {
        goto done;
    }

    ret = wc_ecc_set_deterministic(&key, 1);
    if (ret != 0) {
        goto done;
    }

    sigSz = sizeof(sig);
    do {
    #if defined(WOLFSSL_ASYNC_CRYPT)
        ret = wc_AsyncWait(ret, &key.asyncDev, WC_ASYNC_FLAG_CALL_AGAIN);
    #endif
        if (ret == 0)
            ret = wc_ecc_sign_hash(hash, sizeof(hash), sig, &sigSz, rng, &key);
    } while (ret == WC_PENDING_E);
    if (ret != 0) {
        goto done;
    }
    TEST_SLEEP();

    if (sigSz != sizeof(expSig)) {
        ret = -9830;
        goto done;
    }
    if (XMEMCMP(sig, expSig, sigSz) != 0) {
        ret = -9831;
        goto done;
    }

    sigSz = sizeof(sig);
    do {
    #if defined(WOLFSSL_ASYNC_CRYPT)
        ret = wc_AsyncWait(ret, &key.asyncDev, WC_ASYNC_FLAG_CALL_AGAIN);
    #endif
        if (ret == 0)
            ret = wc_ecc_sign_hash(hash, sizeof(hash), sig, &sigSz, rng, &key);
    } while (ret == WC_PENDING_E);
    if (ret != 0) {
        goto done;
    }
    TEST_SLEEP();

done:
    wc_ecc_free(&key);
    return ret;
}

#ifdef WOLFSSL_PUBLIC_MP
#if defined(HAVE_ECC384)
/* KAT from RFC6979 */
static int ecc384_test_deterministic_k(WC_RNG* rng)
{
    int ret;
    ecc_key key;
    byte sig[72];
    word32 sigSz;
    unsigned char msg[] = "sample";
    unsigned char hash[32];
    const char* dIUT =
        "6B9D3DAD2E1B8C1C05B19875B6659F4DE23C3B667BF297BA9AA47740787137D8"
        "96D5724E4C70A825F872C9EA60D2EDF5";
    const char* QIUTx =
        "EC3A4E415B4E19A4568618029F427FA5DA9A8BC4AE92E02E06AAE5286B300C64"
        "DEF8F0EA9055866064A254515480BC13";
    const char* QIUTy =
        "8015D9B72D7D57244EA8EF9AC0C621896708A59367F9DFB9F54CA84B3F1C9DB1"
        "288B231C3AE0D4FE7344FD2533264720";
    const char* expRstr =
       "21B13D1E013C7FA1392D03C5F99AF8B30C570C6F98D4EA8E354B63A21D3DAA33"
       "BDE1E888E63355D92FA2B3C36D8FB2CD";
    const char* expSstr =
       "F3AA443FB107745BF4BD77CB3891674632068A10CA67E3D45DB2266FA7D1FEEB"
       "EFDC63ECCD1AC42EC0CB8668A4FA0AB0";
    mp_int r,s, expR, expS;

    mp_init_multi(&r, &s, &expR, &expS, NULL, NULL);
    ret = wc_ecc_init_ex(&key, HEAP_HINT, devId);
    if (ret != 0) {
        return ret;
    }
    ret = wc_ecc_import_raw(&key, QIUTx, QIUTy, dIUT, "SECP384R1");
    if (ret != 0) {
        goto done;
    }

    ret = wc_Hash(WC_HASH_TYPE_SHA256, msg,
            (word32)XSTRLEN((const char*)msg), hash, sizeof(hash));
    if (ret != 0) {
        goto done;
    }

    ret = wc_ecc_set_deterministic(&key, 1);
    if (ret != 0) {
        goto done;
    }

    sigSz = sizeof(sig);
    do {
    #if defined(WOLFSSL_ASYNC_CRYPT)
        ret = wc_AsyncWait(ret, &key.asyncDev, WC_ASYNC_FLAG_CALL_AGAIN);
    #endif
        if (ret == 0)
            ret = wc_ecc_sign_hash_ex(hash, sizeof(hash), rng, &key, &r, &s);
    } while (ret == WC_PENDING_E);
    if (ret != 0) {
        goto done;
    }
    TEST_SLEEP();

    mp_read_radix(&expR, expRstr, MP_RADIX_HEX);
    mp_read_radix(&expS, expSstr, MP_RADIX_HEX);
    if (mp_cmp(&r, &expR) != MP_EQ) {
        ret = -1;
    }

done:
    wc_ecc_free(&key);
    return ret;
}
#endif /* HAVE_ECC384 */

#if defined(HAVE_ECC521)
/* KAT from RFC6979 */
static int ecc521_test_deterministic_k(WC_RNG* rng)
{
    int ret;
    ecc_key key;
    byte sig[ECC_MAX_SIG_SIZE];
    word32 sigSz;
    unsigned char msg[] = "sample";
    unsigned char hash[32];

    const char* dIUT =
       "0FAD06DAA62BA3B25D2FB40133DA757205DE67F5BB0018FEE8C86E1B68C7E75C"
       "AA896EB32F1F47C70855836A6D16FCC1466F6D8FBEC67DB89EC0C08B0E996B83"
       "538";
    const char* QIUTx =
        "1894550D0785932E00EAA23B694F213F8C3121F86DC97A04E5A7167DB4E5BCD3"
        "71123D46E45DB6B5D5370A7F20FB633155D38FFA16D2BD761DCAC474B9A2F502"
        "3A4";
    const char* QIUTy =
        "0493101C962CD4D2FDDF782285E64584139C2F91B47F87FF82354D6630F746A2"
        "8A0DB25741B5B34A828008B22ACC23F924FAAFBD4D33F81EA66956DFEAA2BFDF"
        "CF5";
    const char* expRstr =
        "1511BB4D675114FE266FC4372B87682BAECC01D3CC62CF2303C92B3526012659"
        "D16876E25C7C1E57648F23B73564D67F61C6F14D527D54972810421E7D87589E"
        "1A7";
    const char* expSstr =
        "04A171143A83163D6DF460AAF61522695F207A58B95C0644D87E52AA1A347916"
        "E4F7A72930B1BC06DBE22CE3F58264AFD23704CBB63B29B931F7DE6C9D949A7E"
        "CFC";
    mp_int r,s, expR, expS;

    mp_init_multi(&r, &s, &expR, &expS, NULL, NULL);
    ret = wc_ecc_init_ex(&key, HEAP_HINT, devId);
    if (ret != 0) {
        return ret;
    }
    ret = wc_ecc_import_raw(&key, QIUTx, QIUTy, dIUT, "SECP521R1");
    if (ret != 0) {
        goto done;
    }

    ret = wc_Hash(WC_HASH_TYPE_SHA256, msg,
            (word32)XSTRLEN((const char*)msg), hash, sizeof(hash));
    if (ret != 0) {
        goto done;
    }

    ret = wc_ecc_set_deterministic(&key, 1);
    if (ret != 0) {
        goto done;
    }

    sigSz = sizeof(sig);
    do {
    #if defined(WOLFSSL_ASYNC_CRYPT)
        ret = wc_AsyncWait(ret, &key.asyncDev, WC_ASYNC_FLAG_CALL_AGAIN);
    #endif
        if (ret == 0)
            ret = wc_ecc_sign_hash_ex(hash, sizeof(hash), rng, &key, &r, &s);
    } while (ret == WC_PENDING_E);
    if (ret != 0) {
        goto done;
    }
    TEST_SLEEP();

    mp_read_radix(&expR, expRstr, MP_RADIX_HEX);
    mp_read_radix(&expS, expSstr, MP_RADIX_HEX);
    if (mp_cmp(&r, &expR) != MP_EQ) {
        ret = -1;
    }

done:
    wc_ecc_free(&key);
    return ret;
}
#endif /* HAVE_ECC521 */
#endif /* WOLFSSL_PUBLIC_MP */
#endif


#if defined(HAVE_ECC_SIGN) && defined(WOLFSSL_ECDSA_SET_K) && \
    !defined(WOLFSSL_KCAPI_ECC)
static int ecc_test_sign_vectors(WC_RNG* rng)
{
    int ret;
#ifdef WOLFSSL_SMALL_STACK
    ecc_key *key = NULL;
#else
    ecc_key key[1];
#endif
    int key_inited = 0;
    byte sig[72];
    word32 sigSz;
    WOLFSSL_SMALL_STACK_STATIC const unsigned char hash[32] = "test wolfSSL deterministic sign";
    WOLFSSL_SMALL_STACK_STATIC const char* dIUT =   "7d7dc5f71eb29ddaf80d6214632eeae03d9058af1fb6d22ed80badb62bc1a534";
    WOLFSSL_SMALL_STACK_STATIC const char* QIUTx =  "ead218590119e8876b29146ff89ca61770c4edbbf97d38ce385ed281d8a6b230";
    WOLFSSL_SMALL_STACK_STATIC const char* QIUTy =  "28af61281fd35e2fa7002523acc85a429cb06ee6648325389f59edfce1405141";
    WOLFSSL_SMALL_STACK_STATIC const byte k[1] = { 0x02 };
    WOLFSSL_SMALL_STACK_STATIC const byte expSig[71] = {
        0x30, 0x45, 0x02, 0x20, 0x7c, 0xf2, 0x7b, 0x18,
        0x8d, 0x03, 0x4f, 0x7e, 0x8a, 0x52, 0x38, 0x03,
        0x04, 0xb5, 0x1a, 0xc3, 0xc0, 0x89, 0x69, 0xe2,
        0x77, 0xf2, 0x1b, 0x35, 0xa6, 0x0b, 0x48, 0xfc,
        0x47, 0x66, 0x99, 0x78, 0x02, 0x21, 0x00, 0xa8,
        0x43, 0xa0, 0xce, 0x6c, 0x5e, 0x17, 0x8a, 0x53,
        0x4d, 0xaf, 0xd2, 0x95, 0x78, 0x9f, 0x84, 0x4f,
        0x94, 0xb8, 0x75, 0xa3, 0x19, 0xa5, 0xd4, 0xdf,
        0xe1, 0xd4, 0x5e, 0x9d, 0x97, 0xfe, 0x81
    };

#ifdef WOLFSSL_SMALL_STACK
    if ((key = (ecc_key *)XMALLOC(sizeof(*key), HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER)) == NULL)
        return MEMORY_E;
#endif

    ret = wc_ecc_init_ex(key, HEAP_HINT, devId);
    if (ret != 0) {
        goto done;
    }
    key_inited = 1;

    ret = wc_ecc_import_raw(key, QIUTx, QIUTy, dIUT, "SECP256R1");
    if (ret != 0) {
        goto done;
    }
    wc_ecc_set_flags(key, WC_ECC_FLAG_DEC_SIGN);

    ret = wc_ecc_sign_set_k(k, sizeof(k), key);
    if (ret != 0) {
        goto done;
    }

    sigSz = sizeof(sig);
    do {
    #if defined(WOLFSSL_ASYNC_CRYPT)
        ret = wc_AsyncWait(ret, &key->asyncDev, WC_ASYNC_FLAG_CALL_AGAIN);
    #endif
        if (ret == 0)
            ret = wc_ecc_sign_hash(hash, sizeof(hash), sig, &sigSz, rng, key);
    } while (ret == WC_PENDING_E);
    if (ret != 0) {
        goto done;
    }
    TEST_SLEEP();

    if (sigSz != sizeof(expSig)) {
        ret = -9830;
        goto done;
    }
    if (XMEMCMP(sig, expSig, sigSz) != 0) {
        ret = -9831;
        goto done;
    }

    sigSz = sizeof(sig);
    do {
    #if defined(WOLFSSL_ASYNC_CRYPT)
        ret = wc_AsyncWait(ret, &key->asyncDev, WC_ASYNC_FLAG_CALL_AGAIN);
    #endif
        if (ret == 0)
            ret = wc_ecc_sign_hash(hash, sizeof(hash), sig, &sigSz, rng, key);
    } while (ret == WC_PENDING_E);
    if (ret != 0) {
        goto done;
    }
    TEST_SLEEP();

done:
    if (key_inited)
        wc_ecc_free(key);
#ifdef WOLFSSL_SMALL_STACK
    XFREE(key, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
#endif

    return ret;
}
#endif

#ifdef HAVE_ECC_CDH
static int ecc_test_cdh_vectors(WC_RNG* rng)
{
    int ret;
#ifdef WOLFSSL_SMALL_STACK
    ecc_key *pub_key = (ecc_key *)XMALLOC(sizeof *pub_key, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    ecc_key *priv_key = (ecc_key *)XMALLOC(sizeof *priv_key, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
#else
    ecc_key pub_key[1], priv_key[1];
#endif
    byte    sharedA[32] = {0}, sharedB[32] = {0};
    word32  x, z;

    WOLFSSL_SMALL_STACK_STATIC const char* QCAVSx = "700c48f77f56584c5cc632ca65640db91b6bacce3a4df6b42ce7cc838833d287";
    WOLFSSL_SMALL_STACK_STATIC const char* QCAVSy = "db71e509e3fd9b060ddb20ba5c51dcc5948d46fbf640dfe0441782cab85fa4ac";
    WOLFSSL_SMALL_STACK_STATIC const char* dIUT =   "7d7dc5f71eb29ddaf80d6214632eeae03d9058af1fb6d22ed80badb62bc1a534";
    WOLFSSL_SMALL_STACK_STATIC const char* QIUTx =  "ead218590119e8876b29146ff89ca61770c4edbbf97d38ce385ed281d8a6b230";
    WOLFSSL_SMALL_STACK_STATIC const char* QIUTy =  "28af61281fd35e2fa7002523acc85a429cb06ee6648325389f59edfce1405141";
    WOLFSSL_SMALL_STACK_STATIC const char* ZIUT =   "46fc62106420ff012e54a434fbdd2d25ccc5852060561e68040dd7778997bd7b";

#ifdef WOLFSSL_SMALL_STACK
    if ((pub_key == NULL) ||
        (priv_key == NULL)) {
        ret = MEMORY_E;
        goto done;
    }
#endif

    XMEMSET(pub_key, 0, sizeof *pub_key);
    XMEMSET(priv_key, 0, sizeof *priv_key);

    /* setup private and public keys */
    ret = wc_ecc_init_ex(pub_key, HEAP_HINT, devId);
    if (ret != 0)
        goto done;
    ret = wc_ecc_init_ex(priv_key, HEAP_HINT, devId);
    if (ret != 0)
        goto done;
    wc_ecc_set_flags(pub_key, WC_ECC_FLAG_COFACTOR);
    wc_ecc_set_flags(priv_key, WC_ECC_FLAG_COFACTOR);
    ret = wc_ecc_import_raw(pub_key, QCAVSx, QCAVSy, NULL, "SECP256R1");
    if (ret != 0)
        goto done;
    ret = wc_ecc_import_raw(priv_key, QIUTx, QIUTy, dIUT, "SECP256R1");
    if (ret != 0)
        goto done;

#if defined(ECC_TIMING_RESISTANT) && (!defined(HAVE_FIPS) || \
    (!defined(HAVE_FIPS_VERSION) || (HAVE_FIPS_VERSION != 2))) && \
    !defined(HAVE_SELFTEST)
    ret = wc_ecc_set_rng(priv_key, rng);
    if (ret != 0)
        goto done;
#else
    (void)rng;
#endif

    /* compute ECC Cofactor shared secret */
    x = sizeof(sharedA);
    do {
    #if defined(WOLFSSL_ASYNC_CRYPT)
        ret = wc_AsyncWait(ret, &priv_key->asyncDev, WC_ASYNC_FLAG_CALL_AGAIN);
    #endif
        if (ret == 0)
            ret = wc_ecc_shared_secret(priv_key, pub_key, sharedA, &x);
    } while (ret == WC_PENDING_E);
    if (ret != 0) {
        goto done;
    }
    TEST_SLEEP();

    /* read in expected Z */
    z = sizeof(sharedB);
    ret = Base16_Decode((const byte*)ZIUT, (word32)XSTRLEN(ZIUT), sharedB, &z);
    if (ret != 0)
        goto done;

    /* compare results */
    if (x != z || XMEMCMP(sharedA, sharedB, x)) {
        ERROR_OUT(-9840, done);
    }

done:

#ifdef WOLFSSL_SMALL_STACK
    if (priv_key) {
        wc_ecc_free(priv_key);
        XFREE(priv_key, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    }
    if (pub_key) {
        wc_ecc_free(pub_key);
        XFREE(pub_key, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    }
#else
    wc_ecc_free(priv_key);
    wc_ecc_free(pub_key);
#endif

    return ret;
}
#endif /* HAVE_ECC_CDH */
#endif /* HAVE_ECC_VECTOR_TEST */

#ifdef HAVE_ECC_KEY_IMPORT
/* returns 0 on success */
static int ecc_test_make_pub(WC_RNG* rng)
{
#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
    ecc_key *key = (ecc_key *)XMALLOC(sizeof *key, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
#if defined(HAVE_ECC_DHE) && defined(HAVE_ECC_KEY_EXPORT)
    ecc_key *pub = (ecc_key *)XMALLOC(sizeof *pub, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
#endif
    byte *exportBuf = (byte *)XMALLOC(ECC_BUFSIZE, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    byte *tmp = (byte *)XMALLOC(ECC_BUFSIZE, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
#else
    ecc_key key[1];
#if defined(HAVE_ECC_DHE) && defined(HAVE_ECC_KEY_EXPORT) && !defined(WC_NO_RNG) && \
    !defined(WOLF_CRYPTO_CB_ONLY_ECC)
    ecc_key pub[1];
#endif
    byte exportBuf[ECC_BUFSIZE];
    byte tmp[ECC_BUFSIZE];
#endif
    const byte* msg = (const byte*)"test wolfSSL ECC public gen";
    word32 x;
    word32 tmpSz;
    int ret = 0;
    ecc_point* pubPoint = NULL;
#ifdef HAVE_ECC_VERIFY
    int verify = 0;
#endif

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
    if ((key == NULL) ||
#if defined(HAVE_ECC_DHE) && defined(HAVE_ECC_KEY_EXPORT)
        (pub == NULL) ||
#endif
        (exportBuf == NULL) ||
        (tmp == NULL))
        ERROR_OUT(MEMORY_E, done);
#endif

    (void)msg;
    (void)verify;
    (void)exportBuf;
    (void)rng;

    wc_ecc_init_ex(key, HEAP_HINT, devId);

#ifndef NO_ECC256
#ifdef USE_CERT_BUFFERS_256
    XMEMCPY(tmp, ecc_key_der_256, (size_t)sizeof_ecc_key_der_256);
    tmpSz = (size_t)sizeof_ecc_key_der_256;
#else
    {
        XFILE file = XFOPEN(eccKeyDerFile, "rb");
        if (!file) {
            ERROR_OUT(-9850, done);
        }

        tmpSz = (word32)XFREAD(tmp, 1, ECC_BUFSIZE, file);
        XFCLOSE(file);
    }
#endif /* USE_CERT_BUFFERS_256 */

    /* import private only then test with */
    ret = wc_ecc_import_private_key(tmp, tmpSz, NULL, 0, NULL);
    if (ret == 0) {
        ERROR_OUT(-9851, done);
    }

    ret = wc_ecc_import_private_key(NULL, tmpSz, NULL, 0, key);
    if (ret == 0) {
        ERROR_OUT(-9852, done);
    }

    x = 0;
    ret = wc_EccPrivateKeyDecode(tmp, &x, key, tmpSz);
    if (ret != 0) {
        ERROR_OUT(-9853, done);
    }

#ifdef HAVE_ECC_KEY_EXPORT
    x = ECC_BUFSIZE;
    ret = wc_ecc_export_private_only(key, exportBuf, &x);
    if (ret != 0) {
        ERROR_OUT(-9854, done);
    }

    /* make private only key */
    wc_ecc_free(key);
    wc_ecc_init_ex(key, HEAP_HINT, devId);
    ret = wc_ecc_import_private_key(exportBuf, x, NULL, 0, key);
    if (ret != 0) {
        ERROR_OUT(-9855, done);
    }

    x = ECC_BUFSIZE;
    ret = wc_ecc_export_x963_ex(key, exportBuf, &x, 0);
    if (ret == 0) {
        ERROR_OUT(-9856, done);
    }

#endif /* HAVE_ECC_KEY_EXPORT */

    ret = wc_ecc_make_pub(NULL, NULL);
    if (ret == 0) {
        ERROR_OUT(-9857, done);
    }
    TEST_SLEEP();

#ifndef WOLFSSL_NO_MALLOC
    pubPoint = wc_ecc_new_point_h(HEAP_HINT);
    if (pubPoint == NULL) {
        ERROR_OUT(-9858, done);
    }
#if !defined(WOLFSSL_CRYPTOCELL)
    ret = wc_ecc_make_pub(key, pubPoint);
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &key->asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    if (ret != 0) {
       ERROR_OUT(-9859, done);
    }
#endif
    TEST_SLEEP();

#ifdef HAVE_ECC_KEY_EXPORT
    /* export should still fail, is private only key */
    x = ECC_BUFSIZE;
    ret = wc_ecc_export_x963_ex(key, exportBuf, &x, 0);
    if (ret == 0) {
        ERROR_OUT(-9860, done);
    }
#endif /* HAVE_ECC_KEY_EXPORT */
#endif /* !WOLFSSL_NO_MALLOC */
#endif /* !NO_ECC256 */

    /* create a new key since above test for loading key is not supported */
#if defined(WOLFSSL_CRYPTOCELL) || defined(NO_ECC256) || \
    defined(WOLFSSL_QNX_CAAM) || defined(WOLFSSL_SE050) || \
    defined(WOLFSSL_SECO_CAAM)
    ret  = wc_ecc_make_key(rng, ECC_KEYGEN_SIZE, key);
    if (ret != 0) {
        ERROR_OUT(-9861, done);
    }
#endif

#if defined(HAVE_ECC_SIGN) && (!defined(ECC_TIMING_RESISTANT) || \
    (defined(ECC_TIMING_RESISTANT) && !defined(WC_NO_RNG))) && \
    !defined(WOLF_CRYPTO_CB_ONLY_ECC)
    tmpSz = ECC_BUFSIZE;
    ret = 0;
    do {
    #if defined(WOLFSSL_ASYNC_CRYPT)
        ret = wc_AsyncWait(ret, &key->asyncDev, WC_ASYNC_FLAG_CALL_AGAIN);
    #endif
        if (ret == 0) {
            ret = wc_ecc_sign_hash(msg, (word32)XSTRLEN((const char* )msg), tmp,
                &tmpSz, rng, key);
        }
    } while (ret == WC_PENDING_E);
    if (ret != 0) {
        ERROR_OUT(-9862, done);
    }
    TEST_SLEEP();

#ifdef HAVE_ECC_VERIFY
    /* try verify with private only key */
    ret = 0;
    do {
    #if defined(WOLFSSL_ASYNC_CRYPT)
        ret = wc_AsyncWait(ret, &key->asyncDev, WC_ASYNC_FLAG_CALL_AGAIN);
    #endif
        if (ret == 0) {
            ret = wc_ecc_verify_hash(tmp, tmpSz, msg,
                (word32)XSTRLEN((const char*)msg), &verify, key);
        }
    } while (ret == WC_PENDING_E);
    if (ret != 0) {
        ERROR_OUT(-9863, done);
    }

    if (verify != 1) {
        ERROR_OUT(-9864, done);
    }
    TEST_SLEEP();
#ifdef HAVE_ECC_KEY_EXPORT
    /* exporting the public part should now work */
    x = ECC_BUFSIZE;
    ret = wc_ecc_export_x963_ex(key, exportBuf, &x, 0);
    if (ret != 0) {
        ERROR_OUT(-9865, done);
    }
#endif /* HAVE_ECC_KEY_EXPORT */
#endif /* HAVE_ECC_VERIFY */

#endif /* HAVE_ECC_SIGN */

#if defined(HAVE_ECC_DHE) && defined(HAVE_ECC_KEY_EXPORT) && !defined(WC_NO_RNG)
    /* now test private only key with creating a shared secret */
    x = ECC_BUFSIZE;
    ret = wc_ecc_export_private_only(key, exportBuf, &x);
    if (ret != 0) {
        ERROR_OUT(-9866, done);
    }

#if !defined(WOLFSSL_QNX_CAAM) && !defined(WOLFSSL_SE050)
    /* make private only key */
    wc_ecc_free(key);
    wc_ecc_init_ex(key, HEAP_HINT, devId);
    ret = wc_ecc_import_private_key(exportBuf, x, NULL, 0, key);
    if (ret != 0) {
        ERROR_OUT(-9867, done);
    }

    /* check that public export fails with private only key */
    x = ECC_BUFSIZE;
    ret = wc_ecc_export_x963_ex(key, exportBuf, &x, 0);
    if (ret == 0) {
        ERROR_OUT(-9868, done);
    }
#endif /* WOLFSSL_QNX_CAAM */

#ifndef WOLF_CRYPTO_CB_ONLY_ECC
    /* make public key for shared secret */
    wc_ecc_init_ex(pub, HEAP_HINT, devId);
    ret = wc_ecc_make_key(rng, ECC_KEYGEN_SIZE, pub);
#ifdef HAVE_ECC_CDH
    wc_ecc_set_flags(key, WC_ECC_FLAG_COFACTOR);
#endif
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &pub->asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    if (ret != 0) {
        ERROR_OUT(-9869, done);
    }
    TEST_SLEEP();

#if defined(ECC_TIMING_RESISTANT) && (!defined(HAVE_FIPS) || \
    (!defined(HAVE_FIPS_VERSION) || (HAVE_FIPS_VERSION != 2))) && \
    !defined(HAVE_SELFTEST)
    ret = wc_ecc_set_rng(key, rng);
    if (ret != 0)
        goto done;
#endif

    x = ECC_BUFSIZE;
    do {
    #if defined(WOLFSSL_ASYNC_CRYPT)
        ret = wc_AsyncWait(ret, &key->asyncDev, WC_ASYNC_FLAG_CALL_AGAIN);
    #endif
        if (ret == 0) {
            ret = wc_ecc_shared_secret(key, pub, exportBuf, &x);
        }
    } while (ret == WC_PENDING_E);
    wc_ecc_free(pub);
    if (ret != 0) {
        ERROR_OUT(-9870, done);
    }
    TEST_SLEEP();
#endif /* HAVE_ECC_DHE && HAVE_ECC_KEY_EXPORT && !WC_NO_RNG */
#endif /* WOLF_CRYPTO_CB_ONLY_ECC */
    ret = 0;

done:

    wc_ecc_del_point_h(pubPoint, HEAP_HINT);

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
    if (key != NULL) {
        wc_ecc_free(key);
        XFREE(key, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    }
#if defined(HAVE_ECC_DHE) && defined(HAVE_ECC_KEY_EXPORT)
    if (pub != NULL)
        XFREE(pub, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
#endif
    if (exportBuf != NULL)
        XFREE(exportBuf, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    if (tmp != NULL)
        XFREE(tmp, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
#else
    wc_ecc_free(key);
#endif

    return ret;
}

#if defined(HAVE_ECC_KEY_EXPORT) && !defined(NO_ASN_CRYPT) && \
   !defined(WC_NO_RNG) && !defined(WOLF_CRYPTO_CB_ONLY_ECC)
static int ecc_test_key_decode(WC_RNG* rng, int keySize)
{
    int ret;
#ifdef WOLFSSL_SMALL_STACK
    ecc_key *eccKey = (ecc_key *)XMALLOC(sizeof *eccKey, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    byte *tmpBuf = (byte *)XMALLOC(ECC_BUFSIZE, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
#else
    ecc_key eccKey[1];
    byte tmpBuf[ECC_BUFSIZE];
#endif
    word32 tmpSz;
    word32 idx;

#ifdef WOLFSSL_SMALL_STACK
    if ((eccKey == NULL) || (tmpBuf == NULL))
        ERROR_OUT(MEMORY_E, done);
#endif

    ret = wc_ecc_init(eccKey);
    if (ret != 0) {
        goto done;
    }
    ret = wc_ecc_make_key(rng, keySize, eccKey);
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &eccKey->asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    if (ret != 0) {
        goto done;
    }

    tmpSz = ECC_BUFSIZE;
    ret = wc_EccKeyToDer(eccKey, tmpBuf, tmpSz);
    wc_ecc_free(eccKey);
    if (ret < 0) {
        goto done;
    }
    tmpSz = ret;

    ret = wc_ecc_init(eccKey);
    if (ret != 0) {
        goto done;
    }
    idx = 0;
    ret = wc_EccPrivateKeyDecode(tmpBuf, &idx, eccKey, tmpSz);
    if (ret != 0) {
        goto done;
    }
    wc_ecc_free(eccKey);

    ret = wc_ecc_init(eccKey);
    if (ret != 0) {
        goto done;
    }

    idx = 0;
    ret = wc_EccPublicKeyDecode(tmpBuf, &idx, eccKey, tmpSz);
    if (ret != 0) {
        goto done;
    }

    ret = 0;

  done:

#ifdef WOLFSSL_SMALL_STACK
    if (eccKey != NULL) {
        wc_ecc_free(eccKey);
        XFREE(eccKey, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    }
    if (tmpBuf != NULL)
        XFREE(tmpBuf, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
#else
    wc_ecc_free(eccKey);
#endif

    return ret;
}
#endif /* HAVE_ECC_KEY_EXPORT && !NO_ASN_CRYPT */
#endif /* HAVE_ECC_KEY_IMPORT */

#if defined(HAVE_ECC_KEY_EXPORT) && !defined(NO_ASN_CRYPT) && \
   !defined(WC_NO_RNG) && !defined(WOLF_CRYPTO_CB_ONLY_ECC)
static int ecc_test_key_gen(WC_RNG* rng, int keySize)
{
    int    ret = 0;
    int    derSz;
#ifdef HAVE_PKCS8
    word32 pkcs8Sz;
#endif
#ifdef WOLFSSL_SMALL_STACK
    byte *der = (byte *)XMALLOC(ECC_BUFSIZE, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    ecc_key *userA = (ecc_key *)XMALLOC(sizeof *userA, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
#else
    byte der[ECC_BUFSIZE];
    ecc_key userA[1];
#endif

#ifdef WOLFSSL_SMALL_STACK
    if ((der == NULL) || (userA == NULL))
        ERROR_OUT(MEMORY_E, done);
#endif

    ret = wc_ecc_init_ex(userA, HEAP_HINT, devId);
    if (ret != 0)
        goto done;

    ret = wc_ecc_make_key(rng, keySize, userA);
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &userA->asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    if (ret != 0)
        goto done;
    TEST_SLEEP();

    ret = wc_ecc_check_key(userA);
    if (ret != 0)
        goto done;
    TEST_SLEEP();

    derSz = wc_EccKeyToDer(userA, der, ECC_BUFSIZE);
    if (derSz < 0) {
        ERROR_OUT(derSz, done);
    }

    ret = SaveDerAndPem(der, derSz, eccCaKeyTempFile, eccCaKeyPemFile,
        ECC_PRIVATEKEY_TYPE, -8347);
    if (ret != 0) {
        goto done;
    }

    /* test export of public key */
    derSz = wc_EccPublicKeyToDer(userA, der, ECC_BUFSIZE, 1);
    if (derSz < 0) {
        ERROR_OUT(derSz, done);
    }
    if (derSz == 0) {
        ERROR_OUT(-9890, done);
    }

#ifdef HAVE_COMP_KEY
    /* test export of compressed public key */
    derSz = wc_EccPublicKeyToDer_ex(userA, der, ECC_BUFSIZE, 1, 1);
    if (derSz < 0) {
        ERROR_OUT(derSz, done);
    }
    if (derSz == 0) {
        ERROR_OUT(-9890, done);
    }
#endif

    ret = SaveDerAndPem(der, derSz, eccPubKeyDerFile, NULL, 0, -8348);
    if (ret != 0) {
        goto done;
    }

#ifdef HAVE_PKCS8
    /* test export of PKCS#8 unencrypted private key */
    pkcs8Sz = FOURK_BUF;
    derSz = wc_EccPrivateKeyToPKCS8(userA, der, &pkcs8Sz);
    if (derSz < 0) {
        ERROR_OUT(derSz, done);
    }

    if (derSz == 0) {
        ERROR_OUT(-9891, done);
    }

    ret = SaveDerAndPem(der, derSz, eccPkcs8KeyDerFile, NULL, 0, -8349);
    if (ret != 0) {
        goto done;
    }
#endif /* HAVE_PKCS8 */

done:

#ifdef WOLFSSL_SMALL_STACK
    if (der != NULL)
        XFREE(der, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    if (userA != NULL) {
        wc_ecc_free(userA);
        XFREE(userA, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    }
#else
    wc_ecc_free(userA);
#endif

    return ret;
}
#endif /* HAVE_ECC_KEY_EXPORT && !NO_ASN_CRYPT */

static int ecc_test_curve_size(WC_RNG* rng, int keySize, int testVerifyCount,
    int curve_id, const ecc_set_type* dp)
{
#if (defined(HAVE_ECC_DHE) || defined(HAVE_ECC_CDH)) && !defined(WC_NO_RNG) && \
    !defined(WOLFSSL_ATECC508A) && !defined(WOLFSSL_ATECC608A)
    WC_DECLARE_VAR(sharedA, byte, ECC_SHARED_SIZE, HEAP_HINT);
    WC_DECLARE_VAR(sharedB, byte, ECC_SHARED_SIZE, HEAP_HINT);
#endif
#ifdef HAVE_ECC_KEY_EXPORT
    #define ECC_KEY_EXPORT_BUF_SIZE (MAX_ECC_BYTES * 2 + 32)
    WC_DECLARE_VAR(exportBuf, byte, ECC_KEY_EXPORT_BUF_SIZE, HEAP_HINT);
#endif
    word32  x = 0;
#if (defined(HAVE_ECC_DHE) || defined(HAVE_ECC_CDH)) && !defined(WC_NO_RNG) && \
    !defined(WOLFSSL_ATECC508A) && !defined(WOLFSSL_ATECC608A)
    word32  y;
#endif
#if defined(HAVE_ECC_SIGN) && !defined(WOLFSSL_KCAPI_ECC)
    WC_DECLARE_VAR(sig, byte, ECC_SIG_SIZE, HEAP_HINT);
    WC_DECLARE_VAR(digest, byte, ECC_DIGEST_SIZE, HEAP_HINT);
    int     i;
#ifdef HAVE_ECC_VERIFY
    int     verify;
#endif /* HAVE_ECC_VERIFY */
#endif /* HAVE_ECC_SIGN */
    int     ret;
#ifdef WOLFSSL_SMALL_STACK
    ecc_key *userA = (ecc_key *)XMALLOC(sizeof *userA, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    ecc_key *userB = (ecc_key *)XMALLOC(sizeof *userB, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    ecc_key *pubKey = (ecc_key *)XMALLOC(sizeof *pubKey, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
#else
    ecc_key userA[1];
    ecc_key userB[1];
    ecc_key pubKey[1];
#endif
#ifndef WC_NO_RNG
    int     curveSize;
#endif

#ifdef WC_DECLARE_VAR_IS_HEAP_ALLOC
#if (defined(HAVE_ECC_DHE) || defined(HAVE_ECC_CDH)) && !defined(WC_NO_RNG) && \
    !defined(WOLFSSL_ATECC508A) && !defined(WOLFSSL_ATECC608A)
    if (sharedA == NULL || sharedB == NULL)
        ERROR_OUT(-9900, done);
#endif

#ifdef HAVE_ECC_KEY_EXPORT
    if (exportBuf == NULL)
        ERROR_OUT(-9901, done);
#endif

#if defined(HAVE_ECC_SIGN) && !defined(WOLFSSL_KCAPI_ECC)
    if (sig == NULL || digest == NULL)
        ERROR_OUT(-9902, done);
#endif
#endif /* WOLFSSL_SMALL_STACK */

    (void)testVerifyCount;
    (void)dp;
    (void)x;

#ifdef WOLFSSL_SMALL_STACK
    if ((userA == NULL) ||
        (userB == NULL) ||
        (pubKey == NULL))
        ERROR_OUT(-9903, done);
#endif

    XMEMSET(userA, 0, sizeof *userA);
    XMEMSET(userB, 0, sizeof *userB);
    XMEMSET(pubKey, 0, sizeof *pubKey);

    ret = wc_ecc_init_ex(userA, HEAP_HINT, devId);
    if (ret != 0)
        ERROR_OUT(-9904, done);
    ret = wc_ecc_init_ex(userB, HEAP_HINT, devId);
    if (ret != 0)
        ERROR_OUT(-9905, done);
    ret = wc_ecc_init_ex(pubKey, HEAP_HINT, devId);
    if (ret != 0)
        ERROR_OUT(-9906, done);

#ifdef WOLFSSL_CUSTOM_CURVES
    if (dp != NULL) {
        ret = wc_ecc_set_custom_curve(userA, dp);
        if (ret != 0)
            ERROR_OUT(-9907, done);
        ret = wc_ecc_set_custom_curve(userB, dp);
        if (ret != 0)
            ERROR_OUT(-9908, done);
    }
#endif

#ifndef WC_NO_RNG
    ret = wc_ecc_make_key_ex(rng, keySize, userA, curve_id);
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &userA->asyncDev, WC_ASYNC_FLAG_NONE);
#endif
#ifdef WOLF_CRYPTO_CB_ONLY_ECC
    if (ret == NO_VALID_DEVID) {
        ret = 0;
        goto done; /* no software case */
    }
#endif
    if (ret == ECC_CURVE_OID_E)
        goto done; /* catch case, where curve is not supported */
    if (ret != 0)
        ERROR_OUT(-9910, done);
    TEST_SLEEP();

    if (wc_ecc_get_curve_idx(curve_id) != -1) {
        curveSize = wc_ecc_get_curve_size_from_id(userA->dp->id);
        if (curveSize != userA->dp->size)
            ERROR_OUT(-9911, done);
    }

    ret = wc_ecc_check_key(userA);
    if (ret != 0)
        ERROR_OUT(-9912, done);
    TEST_SLEEP();

/* ATECC508/608 configuration may not support more than one ECDH key */
#if !defined(WOLFSSL_ATECC508A) && !defined(WOLFSSL_ATECC608A)

    ret = wc_ecc_make_key_ex(rng, keySize, userB, curve_id);
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &userB->asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    if (ret != 0)
        ERROR_OUT(-9914, done);
    TEST_SLEEP();

    /* only perform the below tests if the key size matches */
    if (dp == NULL && keySize > 0 && wc_ecc_size(userA) != keySize)
        ERROR_OUT(ECC_CURVE_OID_E, done);

#ifdef HAVE_ECC_DHE
#if defined(ECC_TIMING_RESISTANT) && (!defined(HAVE_FIPS) || \
    (!defined(HAVE_FIPS_VERSION) || (HAVE_FIPS_VERSION != 2))) && \
    !defined(HAVE_SELFTEST)
    ret = wc_ecc_set_rng(userA, rng);
    if (ret != 0)
        ERROR_OUT(-9915, done);
    ret = wc_ecc_set_rng(userB, rng);
    if (ret != 0)
        ERROR_OUT(-9916, done);
#endif

    x = ECC_SHARED_SIZE;
    do {
    #if defined(WOLFSSL_ASYNC_CRYPT)
        ret = wc_AsyncWait(ret, &userA->asyncDev, WC_ASYNC_FLAG_CALL_AGAIN);
    #endif
        if (ret == 0)
            ret = wc_ecc_shared_secret(userA, userB, sharedA, &x);
    } while (ret == WC_PENDING_E);
    if (ret != 0) {
        ERROR_OUT(-9917, done);
    }
    TEST_SLEEP();

    y = ECC_SHARED_SIZE;
    do {
    #if defined(WOLFSSL_ASYNC_CRYPT)
        ret = wc_AsyncWait(ret, &userB->asyncDev, WC_ASYNC_FLAG_CALL_AGAIN);
    #endif
        if (ret == 0)
            ret = wc_ecc_shared_secret(userB, userA, sharedB, &y);
    } while (ret == WC_PENDING_E);
    if (ret != 0)
        ERROR_OUT(-9918, done);

    if (y != x)
        ERROR_OUT(-9919, done);

    if (XMEMCMP(sharedA, sharedB, x))
        ERROR_OUT(-9920, done);
    TEST_SLEEP();
#endif /* HAVE_ECC_DHE */

#ifdef HAVE_ECC_CDH
    /* add cofactor flag */
    wc_ecc_set_flags(userA, WC_ECC_FLAG_COFACTOR);
    wc_ecc_set_flags(userB, WC_ECC_FLAG_COFACTOR);

    x = ECC_SHARED_SIZE;
    do {
    #if defined(WOLFSSL_ASYNC_CRYPT)
        ret = wc_AsyncWait(ret, &userA->asyncDev, WC_ASYNC_FLAG_CALL_AGAIN);
    #endif
        if (ret == 0)
            ret = wc_ecc_shared_secret(userA, userB, sharedA, &x);
    } while (ret == WC_PENDING_E);
    if (ret != 0)
        ERROR_OUT(-9921, done);
    TEST_SLEEP();

    y = ECC_SHARED_SIZE;
    do {
    #if defined(WOLFSSL_ASYNC_CRYPT)
        ret = wc_AsyncWait(ret, &userB->asyncDev, WC_ASYNC_FLAG_CALL_AGAIN);
    #endif
        if (ret == 0)
            ret = wc_ecc_shared_secret(userB, userA, sharedB, &y);
    } while (ret == WC_PENDING_E);
    if (ret != 0)
        ERROR_OUT(-9922, done);

    if (y != x)
        ERROR_OUT(-9923, done);

    if (XMEMCMP(sharedA, sharedB, x))
        ERROR_OUT(-9924, done);
    TEST_SLEEP();

    /* remove cofactor flag */
    wc_ecc_set_flags(userA, 0);
    wc_ecc_set_flags(userB, 0);
#endif /* HAVE_ECC_CDH */
#endif /* !WOLFSSL_ATECC508A && WOLFSSL_ATECC608A */

#ifdef HAVE_ECC_KEY_EXPORT
    x = ECC_KEY_EXPORT_BUF_SIZE;
    ret = wc_ecc_export_x963_ex(userA, exportBuf, &x, 0);
    if (ret != 0)
        ERROR_OUT(-9925, done);

#ifdef HAVE_ECC_KEY_IMPORT
    #ifdef WOLFSSL_CUSTOM_CURVES
        if (dp != NULL) {
            ret = wc_ecc_set_custom_curve(pubKey, dp);
            if (ret != 0)
                ERROR_OUT(-9926, done);
        }
    #endif
    ret = wc_ecc_import_x963_ex(exportBuf, x, pubKey, curve_id);
    if (ret != 0)
        ERROR_OUT(-9927, done);

#if !defined(WOLFSSL_ATECC508A) && !defined(WOLFSSL_ATECC608A)
#ifdef HAVE_ECC_DHE
    y = ECC_SHARED_SIZE;
    do {
    #if defined(WOLFSSL_ASYNC_CRYPT)
        ret = wc_AsyncWait(ret, &userB->asyncDev, WC_ASYNC_FLAG_CALL_AGAIN);
    #endif
        if (ret == 0)
            ret = wc_ecc_shared_secret(userB, pubKey, sharedB, &y);
    } while (ret == WC_PENDING_E);
    if (ret != 0)
        ERROR_OUT(-9928, done);

    if (XMEMCMP(sharedA, sharedB, y))
        ERROR_OUT(-9929, done);
    TEST_SLEEP();
#endif /* HAVE_ECC_DHE */

    #ifdef HAVE_COMP_KEY
        /* try compressed export / import too */
        x = ECC_KEY_EXPORT_BUF_SIZE;
        ret = wc_ecc_export_x963_ex(userA, exportBuf, &x, 1);
        if (ret != 0)
            ERROR_OUT(-9930, done);
        wc_ecc_free(pubKey);

        ret = wc_ecc_init_ex(pubKey, HEAP_HINT, devId);
        if (ret != 0)
            ERROR_OUT(-9931, done);
    #ifdef WOLFSSL_CUSTOM_CURVES
        if (dp != NULL) {
            ret = wc_ecc_set_custom_curve(pubKey, dp);
            if (ret != 0)
                ERROR_OUT(-9932, done);
        }
    #endif
        ret = wc_ecc_import_x963_ex(exportBuf, x, pubKey, curve_id);
        if (ret != 0)
            ERROR_OUT(-9933, done);

    #ifdef HAVE_ECC_DHE
        y = ECC_SHARED_SIZE;
        do {
        #if defined(WOLFSSL_ASYNC_CRYPT)
            ret = wc_AsyncWait(ret, &userB->asyncDev, WC_ASYNC_FLAG_CALL_AGAIN);
        #endif
            if (ret == 0)
                ret = wc_ecc_shared_secret(userB, pubKey, sharedB, &y);
        } while (ret == WC_PENDING_E);
        if (ret != 0)
            ERROR_OUT(-9934, done);

        if (XMEMCMP(sharedA, sharedB, y))
            ERROR_OUT(-9935, done);
        TEST_SLEEP();
    #endif /* HAVE_ECC_DHE */
    #endif /* HAVE_COMP_KEY */

#endif /* !WOLFSSL_ATECC508A && !WOLFSSL_ATECC608A */
#endif /* !WC_NO_RNG */

#endif /* HAVE_ECC_KEY_IMPORT */
#endif /* HAVE_ECC_KEY_EXPORT */

    /* For KCAPI cannot sign using generated ECDH key */
#if !defined(ECC_TIMING_RESISTANT) || (defined(ECC_TIMING_RESISTANT) && \
    !defined(WC_NO_RNG) && !defined(WOLFSSL_KCAPI_ECC))
#ifdef HAVE_ECC_SIGN
    /* ECC w/out Shamir has issue with all 0 digest */
    /* WC_BIGINT doesn't have 0 len well on hardware */
    /* Cryptocell has issues with all 0 digest */
#if defined(ECC_SHAMIR) && !defined(WOLFSSL_ASYNC_CRYPT) && \
    !defined(WOLFSSL_CRYPTOCELL)
    /* test DSA sign hash with zeros */
    for (i = 0; i < (int)ECC_DIGEST_SIZE; i++) {
        digest[i] = 0;
    }

    x = ECC_SIG_SIZE;
    do {
    #if defined(WOLFSSL_ASYNC_CRYPT)
        ret = wc_AsyncWait(ret, &userA->asyncDev, WC_ASYNC_FLAG_CALL_AGAIN);
    #endif
        if (ret == 0)
            ret = wc_ecc_sign_hash(digest, ECC_DIGEST_SIZE, sig, &x, rng,
                                                                        userA);
    } while (ret == WC_PENDING_E);
    if (ret != 0)
        ERROR_OUT(-9936, done);
    TEST_SLEEP();

#ifdef HAVE_ECC_VERIFY
    for (i=0; i<testVerifyCount; i++) {
        verify = 0;
        do {
        #if defined(WOLFSSL_ASYNC_CRYPT)
            ret = wc_AsyncWait(ret, &userA->asyncDev, WC_ASYNC_FLAG_CALL_AGAIN);
        #endif
            if (ret == 0)
                ret = wc_ecc_verify_hash(sig, x, digest, ECC_DIGEST_SIZE,
                                                               &verify, userA);
        } while (ret == WC_PENDING_E);
        if (ret != 0)
            ERROR_OUT(-9937, done);
        if (verify != 1)
            ERROR_OUT(-9938, done);
        TEST_SLEEP();
    }
#endif /* HAVE_ECC_VERIFY */
#endif /* ECC_SHAMIR && !WOLFSSL_ASYNC_CRYPT && !WOLFSSL_CRYPTOCELL */

    /* test DSA sign hash with sequence (0,1,2,3,4,...) */
    for (i = 0; i < (int)ECC_DIGEST_SIZE; i++) {
        digest[i] = (byte)i;
    }

    x = ECC_SIG_SIZE;
    do {
    #if defined(WOLFSSL_ASYNC_CRYPT)
        ret = wc_AsyncWait(ret, &userA->asyncDev, WC_ASYNC_FLAG_CALL_AGAIN);
    #endif
        if (ret == 0)
            ret = wc_ecc_sign_hash(digest, ECC_DIGEST_SIZE, sig, &x, rng,
                                                                        userA);
    } while (ret == WC_PENDING_E);
    if (ret != 0)
        ERROR_OUT(-9939, done);
    TEST_SLEEP();

#ifdef HAVE_ECC_VERIFY
    for (i=0; i<testVerifyCount; i++) {
        verify = 0;
        do {
        #if defined(WOLFSSL_ASYNC_CRYPT)
            ret = wc_AsyncWait(ret, &userA->asyncDev, WC_ASYNC_FLAG_CALL_AGAIN);
        #endif
            if (ret == 0)
                ret = wc_ecc_verify_hash(sig, x, digest, ECC_DIGEST_SIZE,
                                                               &verify, userA);
        } while (ret == WC_PENDING_E);
        if (ret != 0)
            ERROR_OUT(-9940, done);
        if (verify != 1)
            ERROR_OUT(-9941, done);
        TEST_SLEEP();
    }
#endif /* HAVE_ECC_VERIFY */
#endif /* HAVE_ECC_SIGN */
#endif /* !ECC_TIMING_RESISTANT || (ECC_TIMING_RESISTANT &&
        * !WC_NO_RNG && !WOLFSSL_KCAPI_ECC) */

#if defined(HAVE_ECC_KEY_EXPORT) && !defined(WC_NO_RNG) && \
    !defined(WOLFSSL_ATECC508) && !defined(WOLFSSL_ATECC608A) && \
    !defined(WOLFSSL_KCAPI_ECC)
    x = ECC_KEY_EXPORT_BUF_SIZE;
    ret = wc_ecc_export_private_only(userA, exportBuf, &x);
    if (ret != 0)
        ERROR_OUT(-9942, done);
#elif defined(HAVE_ECC_KEY_EXPORT)
    (void)exportBuf;
#endif /* HAVE_ECC_KEY_EXPORT */

done:

#ifdef WOLFSSL_SMALL_STACK
    if (userA != NULL) {
        wc_ecc_free(userA);
        XFREE(userA, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    }
    if (userB != NULL) {
        wc_ecc_free(userB);
        XFREE(userB, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    }
    if (pubKey != NULL) {
        wc_ecc_free(pubKey);
        XFREE(pubKey, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    }
#else
    wc_ecc_free(pubKey);
    wc_ecc_free(userB);
    wc_ecc_free(userA);
#endif

#if defined(HAVE_ECC_DHE) || defined(HAVE_ECC_CDH)
    WC_FREE_VAR(sharedA, HEAP_HINT);
    WC_FREE_VAR(sharedB, HEAP_HINT);
#endif
#ifdef HAVE_ECC_KEY_EXPORT
    WC_FREE_VAR(exportBuf, HEAP_HINT);
#endif
#ifdef HAVE_ECC_SIGN
    WC_FREE_VAR(sig, HEAP_HINT);
    WC_FREE_VAR(digest, HEAP_HINT);
#endif

    (void)keySize;
    (void)curve_id;
    (void)rng;

    return ret;
}

#undef  ECC_TEST_VERIFY_COUNT
#define ECC_TEST_VERIFY_COUNT 2
static int ecc_test_curve(WC_RNG* rng, int keySize)
{
    int ret;

    ret = ecc_test_curve_size(rng, keySize, ECC_TEST_VERIFY_COUNT,
        ECC_CURVE_DEF, NULL);
    if (ret < 0) {
        if (ret == ECC_CURVE_OID_E) {
            /* ignore error for curves not found */
            /* some curve sizes are only available with:
                HAVE_ECC_SECPR2, HAVE_ECC_SECPR3, HAVE_ECC_BRAINPOOL
                and HAVE_ECC_KOBLITZ */
        }
        else {
            printf("ecc_test_curve_size %d failed!: %d\n", keySize, ret);
            return ret;
        }
    }
#ifndef WOLF_CRYPTO_CB_ONLY_ECC

#ifdef HAVE_ECC_VECTOR_TEST
    ret = ecc_test_vector(keySize);
    if (ret < 0) {
        printf("ecc_test_vector %d failed!: %d\n", keySize, ret);
        return ret;
    }
#endif

#if defined(HAVE_ECC_KEY_IMPORT) && defined(HAVE_ECC_KEY_EXPORT) && \
    !defined(NO_ASN_CRYPT) && !defined(WC_NO_RNG)
    ret = ecc_test_key_decode(rng, keySize);
    if (ret < 0) {
        if (ret == ECC_CURVE_OID_E) {
            /* ignore error for curves not found */
        }
        else {
            printf("ecc_test_key_decode %d failed!: %d\n", keySize, ret);
            return ret;
        }
    }
#endif

#if defined(HAVE_ECC_KEY_EXPORT) && !defined(NO_ASN_CRYPT) && !defined(WC_NO_RNG)
    ret = ecc_test_key_gen(rng, keySize);
    if (ret < 0) {
        if (ret == ECC_CURVE_OID_E) {
            /* ignore error for curves not found */
        }
        else {
            printf("ecc_test_key_gen %d failed!: %d\n", keySize, ret);
            return ret;
        }
    }
#endif
#endif /* WOLF_CRYPTO_CB_ONLY_ECC */
    return 0;
}

#if (!defined(NO_ECC256) || defined(HAVE_ALL_CURVES)) && ECC_MIN_KEY_SZ <= 256
#if !defined(WOLFSSL_ATECC508A) && !defined(WOLFSSL_ATECC608A) && \
    defined(HAVE_ECC_KEY_IMPORT) && defined(HAVE_ECC_KEY_EXPORT) && \
    !defined(WOLFSSL_NO_MALLOC) && !defined(WOLF_CRYPTO_CB_ONLY_ECC)
static int ecc_point_test(void)
{
    int        ret;
    ecc_point* point;
    ecc_point* point2;
#ifdef HAVE_COMP_KEY
    ecc_point* point3;
    ecc_point* point4;
#endif
    word32     outLen;
    byte       out[65];
    byte       der[] = { 0x04, /* = Uncompressed */
                         0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08,
                         0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08,
                         0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08,
                         0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08,
                         0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08,
                         0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08,
                         0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08,
                         0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08 };
#if defined(HAVE_COMP_KEY) && (!defined(HAVE_FIPS) && !defined(HAVE_SELFTEST) || \
    (defined(HAVE_FIPS_VERSION) && (HAVE_FIPS_VERSION > 2)))
    byte       derComp0[] = { 0x02, /* = Compressed, y even */
                              0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08,
                              0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08,
                              0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08,
                              0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08 };
    byte       derComp1[] = { 0x03, /* = Compressed, y odd */
                              0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08,
                              0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08,
                              0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08,
                              0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08 };
#endif
    byte       altDer[] = { 0x04, /* = Uncompressed */
                            0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
                            0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
                            0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
                            0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
                            0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
                            0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
                            0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
                            0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07 };
    int curve_idx = wc_ecc_get_curve_idx(ECC_SECP256R1);

    /* if curve P256 is not enabled then test should not fail */
    if (curve_idx == ECC_CURVE_INVALID)
        return 0;

    outLen = sizeof(out);
    point = wc_ecc_new_point();
    if (point == NULL)
        return -10000;
    point2 = wc_ecc_new_point();
    if (point2 == NULL) {
        wc_ecc_del_point(point);
        return -10001;
    }
#ifdef HAVE_COMP_KEY
    point3 = wc_ecc_new_point();
    if (point3 == NULL) {
        wc_ecc_del_point(point2);
        wc_ecc_del_point(point);
        return -10002;
    }
    point4 = wc_ecc_new_point();
    if (point4 == NULL) {
        wc_ecc_del_point(point3);
        wc_ecc_del_point(point2);
        wc_ecc_del_point(point);
        return -10003;
    }
#endif

    /* Parameter Validation testing. */
    wc_ecc_del_point(NULL);
    ret = wc_ecc_import_point_der(NULL, sizeof(der), curve_idx, point);
    if (ret != ECC_BAD_ARG_E) {
        ret = -10004;
        goto done;
    }
    ret = wc_ecc_import_point_der(der, sizeof(der), ECC_CURVE_INVALID, point);
    if (ret != ECC_BAD_ARG_E) {
        ret = -10005;
        goto done;
    }
    ret = wc_ecc_import_point_der(der, sizeof(der), curve_idx, NULL);
    if (ret != ECC_BAD_ARG_E) {
        ret = -10006;
        goto done;
    }
    ret = wc_ecc_export_point_der(-1, point, out, &outLen);
    if (ret != ECC_BAD_ARG_E) {
        ret = -10007;
        goto done;
    }
    ret = wc_ecc_export_point_der(curve_idx, NULL, out, &outLen);
    if (ret != ECC_BAD_ARG_E) {
        ret = -10008;
        goto done;
    }
    ret = wc_ecc_export_point_der(curve_idx, point, NULL, &outLen);
    if (ret != LENGTH_ONLY_E || outLen != sizeof(out)) {
        ret = -10009;
        goto done;
    }
    ret = wc_ecc_export_point_der(curve_idx, point, out, NULL);
    if (ret != ECC_BAD_ARG_E) {
        ret = -10010;
        goto done;
    }
    outLen = 0;
    ret = wc_ecc_export_point_der(curve_idx, point, out, &outLen);
    if (ret != BUFFER_E) {
        ret = -10011;
        goto done;
    }
    ret = wc_ecc_copy_point(NULL, NULL);
    if (ret != ECC_BAD_ARG_E) {
        ret = -10012;
        goto done;
    }
    ret = wc_ecc_copy_point(NULL, point2);
    if (ret != ECC_BAD_ARG_E) {
        ret = -10013;
        goto done;
    }
    ret = wc_ecc_copy_point(point, NULL);
    if (ret != ECC_BAD_ARG_E) {
        ret = -10014;
        goto done;
    }
    ret = wc_ecc_cmp_point(NULL, NULL);
    if (ret != BAD_FUNC_ARG) {
        ret = -10015;
        goto done;
    }
    ret = wc_ecc_cmp_point(NULL, point2);
    if (ret != BAD_FUNC_ARG) {
        ret = -10016;
        goto done;
    }
    ret = wc_ecc_cmp_point(point, NULL);
    if (ret != BAD_FUNC_ARG) {
        ret = -10017;
        goto done;
    }

    /* Use API. */
    ret = wc_ecc_import_point_der(der, sizeof(der), curve_idx, point);
    if (ret != 0) {
        ret = -10018;
        goto done;
    }

    outLen = sizeof(out);
    ret = wc_ecc_export_point_der(curve_idx, point, out, &outLen);
    if (ret != 0) {
        ret = -10019;
        goto done;
    }
    if (outLen != sizeof(der)) {
        ret = -10020;
        goto done;
    }
    if (XMEMCMP(out, der, outLen) != 0) {
        ret = -10021;
        goto done;
    }

    ret = wc_ecc_copy_point(point2, point);
    if (ret != MP_OKAY) {
        ret = -10022;
        goto done;
    }
    ret = wc_ecc_cmp_point(point2, point);
    if (ret != MP_EQ) {
        ret = -10023;
        goto done;
    }

    ret = wc_ecc_import_point_der(altDer, sizeof(altDer), curve_idx, point2);
    if (ret != 0) {
        ret = -10024;
        goto done;
    }
    ret = wc_ecc_cmp_point(point2, point);
    if (ret != MP_GT) {
        ret = -10025;
        goto done;
    }

#if defined(HAVE_COMP_KEY) && (!defined(HAVE_FIPS) && !defined(HAVE_SELFTEST) || \
    (defined(HAVE_FIPS_VERSION) && (HAVE_FIPS_VERSION > 2)))
    ret = wc_ecc_import_point_der(derComp0, sizeof(derComp0)*2-1, curve_idx, point3);
    if (ret != 0) {
        ret = -10026;
        goto done;
    }

    ret = wc_ecc_import_point_der_ex(derComp0, sizeof(derComp0), curve_idx, point4, 0);
    if (ret != 0) {
        ret = -10027;
        goto done;
    }

    ret = wc_ecc_cmp_point(point3, point4);
    if (ret != MP_EQ) {
        ret = -10028;
        goto done;
    }

    ret = wc_ecc_import_point_der(derComp1, sizeof(derComp1)*2-1, curve_idx, point3);
    if (ret != 0) {
        ret = -10029;
        goto done;
    }

    ret = wc_ecc_import_point_der_ex(derComp1, sizeof(derComp1), curve_idx, point4, 0);
    if (ret != 0) {
        ret = -10030;
        goto done;
    }

    ret = wc_ecc_cmp_point(point3, point4);
    if (ret != MP_EQ) {
        ret = -10031;
        goto done;
    }
#endif

done:
#ifdef HAVE_COMP_KEY
    wc_ecc_del_point(point4);
    wc_ecc_del_point(point3);
#endif
    wc_ecc_del_point(point2);
    wc_ecc_del_point(point);

    return ret;
}
#endif /* !WOLFSSL_ATECC508A && HAVE_ECC_KEY_IMPORT && HAVE_ECC_KEY_EXPORT */

#if !defined(NO_SIG_WRAPPER) && !defined(WOLF_CRYPTO_CB_ONLY_ECC)
static int ecc_sig_test(WC_RNG* rng, ecc_key* key)
{
    int     ret;
    word32  sigSz;
    int     size;
    byte    out[ECC_MAX_SIG_SIZE];
    byte    in[] = TEST_STRING;
    WOLFSSL_SMALL_STACK_STATIC const byte hash[] = {
        0xf2, 0x02, 0x95, 0x65, 0xcb, 0xf6, 0x2a, 0x59,
        0x39, 0x2c, 0x05, 0xff, 0x0e, 0x29, 0xaf, 0xfe,
        0x47, 0x33, 0x8c, 0x99, 0x8d, 0x58, 0x64, 0x83,
        0xa6, 0x58, 0x0a, 0x33, 0x0b, 0x84, 0x5f, 0x5f
    };
    word32 inLen = (word32)XSTRLEN((char*)in);

    size = wc_ecc_sig_size(key);

    ret = wc_SignatureGetSize(WC_SIGNATURE_TYPE_ECC, key, sizeof(*key));
    if (ret != size)
        return -10040;

    sigSz = (word32)ret;
    ret = wc_SignatureGenerate(WC_HASH_TYPE_SHA256, WC_SIGNATURE_TYPE_ECC, in,
                               inLen, out, &sigSz, key, sizeof(*key), rng);
    if (ret != 0)
        return -10041;
    TEST_SLEEP();

    ret = wc_SignatureVerify(WC_HASH_TYPE_SHA256, WC_SIGNATURE_TYPE_ECC, in,
                             inLen, out, sigSz, key, sizeof(*key));
    if (ret != 0)
        return -10042;
    TEST_SLEEP();

    sigSz = (word32)sizeof(out);
    ret = wc_SignatureGenerateHash(WC_HASH_TYPE_SHA256, WC_SIGNATURE_TYPE_ECC,
        hash, (int)sizeof(hash), out, &sigSz, key, sizeof(*key), rng);
    if (ret != 0)
        return -10043;
    TEST_SLEEP();

    ret = wc_SignatureVerifyHash(WC_HASH_TYPE_SHA256, WC_SIGNATURE_TYPE_ECC,
        hash, (int)sizeof(hash), out, sigSz, key, sizeof(*key));
    if (ret != 0)
        return -10044;
    TEST_SLEEP();

    return 0;
}
#endif

#if defined(HAVE_ECC_KEY_IMPORT) && defined(HAVE_ECC_KEY_EXPORT) && \
   !defined(WOLF_CRYPTO_CB_ONLY_ECC)

static int ecc_exp_imp_test(ecc_key* key)
{
    int        ret;
    int        curve_id;
#ifdef WOLFSSL_SMALL_STACK
    ecc_key    *keyImp = (ecc_key *)XMALLOC(sizeof *keyImp, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
#else
    ecc_key    keyImp[1];
#endif
    byte       priv[32];
    word32     privLen;
    byte       pub[65*2];
    word32     pubLen, pubLenX, pubLenY;
    const char qx[] = "7a4e287890a1a47ad3457e52f2f76a83"
                      "ce46cbc947616d0cbaa82323818a793d";
    const char qy[] = "eec4084f5b29ebf29c44cce3b3059610"
                      "922f8b30ea6e8811742ac7238fe87308";
    const char d[]  = "8c14b793cb19137e323a6d2e2a870bca"
                      "2e7a493ec1153b3a95feb8a4873f8d08";

#ifdef WOLFSSL_SMALL_STACK
    if (keyImp == NULL)
        ERROR_OUT(-10050, done);
#endif

    wc_ecc_init_ex(keyImp, HEAP_HINT, devId);

    privLen = sizeof(priv);
    ret = wc_ecc_export_private_only(key, priv, &privLen);
    if (ret != 0) {
        ret = -10051;
        goto done;
    }
    pubLen = sizeof(pub);
    ret = wc_ecc_export_point_der(key->idx, &key->pubkey, pub, &pubLen);
    if (ret != 0) {
        ret = -10052;
        goto done;
    }

    ret = wc_ecc_import_private_key(priv, privLen, pub, pubLen, keyImp);
    if (ret != 0) {
        ret = -10053;
        goto done;
    }

    wc_ecc_free(keyImp);
    wc_ecc_init_ex(keyImp, HEAP_HINT, devId);

    ret = wc_ecc_import_raw_ex(keyImp, qx, qy, d, ECC_SECP256R1);
    if (ret != 0) {
        ret = -10054;
        goto done;
    }

    wc_ecc_free(keyImp);
    wc_ecc_init_ex(keyImp, HEAP_HINT, devId);

    curve_id = wc_ecc_get_curve_id(key->idx);
    if (curve_id < 0) {
        ret = -10055;
        goto done;
    }

    /* test import private only */
    ret = wc_ecc_import_private_key_ex(priv, privLen, NULL, 0, keyImp,
                                       curve_id);
    if (ret != 0) {
        ret = -10056;
        goto done;
    }

    wc_ecc_free(keyImp);
    wc_ecc_init_ex(keyImp, HEAP_HINT, devId);

    /* test export public raw */
    pubLenX = pubLenY = 32;
    ret = wc_ecc_export_public_raw(key, pub, &pubLenX, &pub[32], &pubLenY);
    if (ret != 0) {
        ret = -10057;
        goto done;
    }

#ifndef HAVE_SELFTEST
    /* test import of public */
    ret = wc_ecc_import_unsigned(keyImp, pub, &pub[32], NULL, ECC_SECP256R1);
    if (ret != 0) {
        ret = -10058;
        goto done;
    }
#endif

    wc_ecc_free(keyImp);
    wc_ecc_init_ex(keyImp, HEAP_HINT, devId);

    /* test export private and public raw */
    pubLenX = pubLenY = privLen = 32;
    ret = wc_ecc_export_private_raw(key, pub, &pubLenX, &pub[32], &pubLenY,
        priv, &privLen);
    if (ret != 0) {
        ret = -10059;
        goto done;
    }

#ifndef HAVE_SELFTEST
    /* test import of private and public */
    ret = wc_ecc_import_unsigned(keyImp, pub, &pub[32], priv, ECC_SECP256R1);
    if (ret != 0) {
        ret = -10060;
        goto done;
    }
#endif

done:

#ifdef WOLFSSL_SMALL_STACK
    if (keyImp != NULL) {
        wc_ecc_free(keyImp);
        XFREE(keyImp, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    }
#else
    wc_ecc_free(keyImp);
#endif

    return ret;
}
#endif /* HAVE_ECC_KEY_IMPORT && HAVE_ECC_KEY_EXPORT */

#if defined(HAVE_ECC_KEY_IMPORT) && !defined(WOLFSSL_VALIDATE_ECC_IMPORT) && \
    !defined(WOLFSSL_CRYPTOCELL) && !defined(WOLF_CRYPTO_CB_ONLY_ECC)
static int ecc_mulmod_test(ecc_key* key1)
{
    int ret;
#ifdef WOLFSSL_SMALL_STACK
    ecc_key    *key2 = (ecc_key *)XMALLOC(sizeof *key2, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    ecc_key    *key3 = (ecc_key *)XMALLOC(sizeof *key3, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
#else
    ecc_key    key2[1];
    ecc_key    key3[1];
#endif

#ifdef WOLFSSL_SMALL_STACK
    if ((key2 == NULL) || (key3 == NULL))
        ERROR_OUT(MEMORY_E, done);
#endif

    wc_ecc_init_ex(key2, HEAP_HINT, devId);
    wc_ecc_init_ex(key3, HEAP_HINT, devId);

    /* TODO: Use test data, test with WOLFSSL_VALIDATE_ECC_IMPORT. */
    /* Need base point (Gx,Gy) and parameter A - load them as the public and
     * private key in key2.
     */
    ret = wc_ecc_import_raw_ex(key2, key1->dp->Gx, key1->dp->Gy, key1->dp->Af,
                               ECC_SECP256R1);
    if (ret != 0)
        goto done;

    /* Need a point (Gx,Gy) and prime - load them as the public and private key
     * in key3.
     */
    ret = wc_ecc_import_raw_ex(key3, key1->dp->Gx, key1->dp->Gy,
                               key1->dp->prime, ECC_SECP256R1);
    if (ret != 0)
        goto done;

    ret = wc_ecc_mulmod(&key1->k, &key2->pubkey, &key3->pubkey, &key2->k, &key3->k,
                        1);
    if (ret != 0) {
        ret = -10070;
        goto done;
    }

done:

#ifdef WOLFSSL_SMALL_STACK
    if (key2 != NULL) {
        wc_ecc_free(key2);
        XFREE(key2, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    }
    if (key3 != NULL) {
        wc_ecc_free(key3);
        XFREE(key3, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    }
#else
    wc_ecc_free(key3);
    wc_ecc_free(key2);
#endif

    return ret;
}
#endif

#if defined(HAVE_ECC_DHE) && !defined(WC_NO_RNG) && \
    !defined(WOLF_CRYPTO_CB_ONLY_ECC)
static int ecc_ssh_test(ecc_key* key, WC_RNG* rng)
{
    int    ret;
    byte   out[128];
    word32 outLen = sizeof(out);

    /* Parameter Validation testing. */
    ret = wc_ecc_shared_secret_ssh(NULL, &key->pubkey, out, &outLen);
    if (ret != BAD_FUNC_ARG)
        return -10080;
    ret = wc_ecc_shared_secret_ssh(key, NULL, out, &outLen);
    if (ret != BAD_FUNC_ARG)
        return -10081;
    ret = wc_ecc_shared_secret_ssh(key, &key->pubkey, NULL, &outLen);
    if (ret != BAD_FUNC_ARG)
        return -10082;
    ret = wc_ecc_shared_secret_ssh(key, &key->pubkey, out, NULL);
    if (ret != BAD_FUNC_ARG)
        return -10083;

#if defined(ECC_TIMING_RESISTANT) && (!defined(HAVE_FIPS) || \
    (!defined(HAVE_FIPS_VERSION) || (HAVE_FIPS_VERSION != 2))) && \
    !defined(HAVE_SELFTEST)
    ret = wc_ecc_set_rng(key, rng);
    if (ret != 0)
        return -10084;
#else
    (void)rng;
#endif

    /* Use API. */
    ret = 0;
    do {
    #if defined(WOLFSSL_ASYNC_CRYPT)
        ret = wc_AsyncWait(ret, &key->asyncDev, WC_ASYNC_FLAG_CALL_AGAIN);
    #endif
        if (ret == 0)
            ret = wc_ecc_shared_secret_ssh(key, &key->pubkey, out, &outLen);
    } while (ret == WC_PENDING_E);
    if (ret != 0)
        return -10085;

    TEST_SLEEP();
    return 0;
}
#endif /* HAVE_ECC_DHE && !WC_NO_RNG */

static int ecc_def_curve_test(WC_RNG *rng)
{
    int     ret;
#ifdef WOLFSSL_SMALL_STACK
    ecc_key *key = (ecc_key *)XMALLOC(sizeof *key, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
#else
    ecc_key key[1];
#endif
#if (defined(HAVE_ECC_KEY_IMPORT) && defined(HAVE_ECC_KEY_EXPORT)) || \
    (defined(HAVE_ECC_KEY_IMPORT) && !defined(WOLFSSL_VALIDATE_ECC_IMPORT))
    word32 idx = 0;
#endif

#ifdef WOLFSSL_SMALL_STACK
    if (key == NULL)
        ERROR_OUT(MEMORY_E, done);
#endif

    wc_ecc_init_ex(key, HEAP_HINT, devId);

    /* Use API */
    ret = wc_ecc_set_flags(NULL, 0);
    if (ret != BAD_FUNC_ARG) {
        ret = -10090;
        goto done;
    }
    ret = wc_ecc_set_flags(key, 0);
    if (ret != 0) {
        ret = -10091;
        goto done;
    }
#ifndef WOLF_CRYPTO_CB_ONLY_ECC
#ifndef WC_NO_RNG
    ret = wc_ecc_make_key(rng, ECC_KEYGEN_SIZE, key);
    #if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &key->asyncDev, WC_ASYNC_FLAG_NONE);
    #endif
    if (ret != 0) {
        goto done;
    }

    #ifndef NO_SIG_WRAPPER
    ret = ecc_sig_test(rng, key);
    if (ret < 0)
        goto done;
    #endif
    TEST_SLEEP();

    #if defined(HAVE_ECC_DHE) && !defined(WOLFSSL_CRYPTOCELL) && \
       !defined(WOLF_CRYPTO_CB_ONLY_ECC)
    ret = ecc_ssh_test(key, rng);
    if (ret < 0)
        goto done;
    #endif

    wc_ecc_free(key);
#else
    (void)rng;
#endif /* !WC_NO_RNG */

#if (defined(HAVE_ECC_KEY_IMPORT) && defined(HAVE_ECC_KEY_EXPORT)) || \
    (defined(HAVE_ECC_KEY_IMPORT) && !defined(WOLFSSL_VALIDATE_ECC_IMPORT))
    /* Use test ECC key - ensure real private "d" exists */
    #ifdef USE_CERT_BUFFERS_256
    ret = wc_EccPrivateKeyDecode(ecc_key_der_256, &idx, key,
        sizeof_ecc_key_der_256);
    #else
    {
        XFILE file = XFOPEN(eccKeyDerFile, "rb");
        byte der[128];
        word32 derSz;
        if (!file) {
            ERROR_OUT(-10093, done);
        }
        derSz = (word32)XFREAD(der, 1, sizeof(der), file);
        XFCLOSE(file);
        ret = wc_EccPrivateKeyDecode(der, &idx, key, derSz);
    }
    #endif
    if (ret != 0) {
        goto done;
    }

#if defined(HAVE_ECC_KEY_IMPORT) && defined(HAVE_ECC_KEY_EXPORT)
    ret = ecc_exp_imp_test(key);
    if (ret < 0)
        goto done;
#endif
#if defined(HAVE_ECC_KEY_IMPORT) && !defined(WOLFSSL_VALIDATE_ECC_IMPORT) && \
    !defined(WOLFSSL_CRYPTOCELL)
    ret = ecc_mulmod_test(key);
    if (ret < 0)
        goto done;
#endif
#endif
#else
   (void)rng;
   (void)idx;
#endif /* WOLF_CRYPTO_CB_ONLY_ECC */
done:

    wc_ecc_free(key);
#ifdef WOLFSSL_SMALL_STACK
    if (key != NULL) {
        XFREE(key, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    }
#endif

    return ret;
}
#endif /* !NO_ECC256 || HAVE_ALL_CURVES */

#if defined(WOLFSSL_CERT_EXT) && \
    (!defined(NO_ECC256) || defined(HAVE_ALL_CURVES)) && ECC_MIN_KEY_SZ <= 256
static int ecc_decode_test(void)
{
    int        ret;
    word32     inSz;
    word32     inOutIdx;
#ifdef WOLFSSL_SMALL_STACK
    ecc_key    *key = (ecc_key *)XMALLOC(sizeof *key, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
#else
    ecc_key    key[1];
#endif

    /* SECP256R1 OID: 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x03, 0x01, 0x07 */

    /* This is ecc_clikeypub_der_256. */
    WOLFSSL_SMALL_STACK_STATIC const byte good[] = {
            0x30, 0x59, 0x30, 0x13, 0x06, 0x07, 0x2a, 0x86, 0x48, 0xce,
            0x3d, 0x02, 0x01, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d,
            0x03, 0x01, 0x07, 0x03, 0x42, 0x00, 0x04, 0x55, 0xbf, 0xf4,
            0x0f, 0x44, 0x50, 0x9a, 0x3d, 0xce, 0x9b, 0xb7, 0xf0, 0xc5,
            0x4d, 0xf5, 0x70, 0x7b, 0xd4, 0xec, 0x24, 0x8e, 0x19, 0x80,
            0xec, 0x5a, 0x4c, 0xa2, 0x24, 0x03, 0x62, 0x2c, 0x9b, 0xda,
            0xef, 0xa2, 0x35, 0x12, 0x43, 0x84, 0x76, 0x16, 0xc6, 0x56,
            0x95, 0x06, 0xcc, 0x01, 0xa9, 0xbd, 0xf6, 0x75, 0x1a, 0x42,
            0xf7, 0xbd, 0xa9, 0xb2, 0x36, 0x22, 0x5f, 0xc7, 0x5d, 0x7f,
            0xb4 };
    WOLFSSL_SMALL_STACK_STATIC const byte badNoObjId[] = { 0x30, 0x08, 0x30, 0x06, 0x03, 0x04,
            0x00, 0x04, 0x01, 0x01 };
    WOLFSSL_SMALL_STACK_STATIC const byte badOneObjId[] = { 0x30, 0x0a, 0x30, 0x08, 0x06, 0x00,
            0x03, 0x04, 0x00, 0x04, 0x01, 0x01 };
    WOLFSSL_SMALL_STACK_STATIC const byte badObjId1Len[] = { 0x30, 0x0c, 0x30, 0x0a, 0x06, 0x09,
            0x06, 0x00, 0x03, 0x04, 0x00, 0x04, 0x01, 0x01 };
    WOLFSSL_SMALL_STACK_STATIC const byte badObj2d1Len[] = { 0x30, 0x0c, 0x30, 0x0a, 0x06, 0x00,
            0x06, 0x07, 0x03, 0x04, 0x00, 0x04, 0x01, 0x01 };
    WOLFSSL_SMALL_STACK_STATIC const byte badNotBitStr[] = { 0x30, 0x14, 0x30, 0x0b, 0x06, 0x00,
            0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x03, 0x01, 0x07,
            0x04, 0x04, 0x00, 0x04, 0x01, 0x01 };
    WOLFSSL_SMALL_STACK_STATIC const byte badBitStrLen[] = { 0x30, 0x14, 0x30, 0x0b, 0x06, 0x00,
            0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x03, 0x01, 0x07,
            0x03, 0x05, 0x00, 0x04, 0x01, 0x01 };
    WOLFSSL_SMALL_STACK_STATIC const byte badNoBitStrZero[] = { 0x30, 0x13, 0x30, 0x0a, 0x06, 0x00,
            0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x03, 0x01, 0x07,
            0x03, 0x03, 0x04, 0x01, 0x01 };
    WOLFSSL_SMALL_STACK_STATIC const byte badPoint[] = { 0x30, 0x12, 0x30, 0x09, 0x06, 0x00,
            0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x03, 0x01, 0x07,
            0x03, 0x03, 0x00, 0x04, 0x01 };

#ifdef WOLFSSL_SMALL_STACK
    if (key == NULL)
        ERROR_OUT(MEMORY_E, done);
#endif

    XMEMSET(key, 0, sizeof *key);
    wc_ecc_init_ex(key, HEAP_HINT, devId);

    inSz = sizeof(good);
    ret = wc_EccPublicKeyDecode(NULL, &inOutIdx, key, inSz);
    if (ret != BAD_FUNC_ARG) {
        ret = -10100;
        goto done;
    }
    ret = wc_EccPublicKeyDecode(good, NULL, key, inSz);
    if (ret != BAD_FUNC_ARG) {
        ret = -10101;
        goto done;
    }
    ret = wc_EccPublicKeyDecode(good, &inOutIdx, NULL, inSz);
    if (ret != BAD_FUNC_ARG) {
        ret = -10102;
        goto done;
    }
    ret = wc_EccPublicKeyDecode(good, &inOutIdx, key, 0);
    if (ret != BAD_FUNC_ARG) {
        ret = -10103;
        goto done;
    }

    /* Change offset to produce bad input data. */
    inOutIdx = 2;
    inSz = sizeof(good) - inOutIdx;
    ret = wc_EccPublicKeyDecode(good, &inOutIdx, key, inSz);
    if (ret != ASN_PARSE_E) {
        ret = -10104;
        goto done;
    }
    inOutIdx = 4;
    inSz = sizeof(good) - inOutIdx;
    ret = wc_EccPublicKeyDecode(good, &inOutIdx, key, inSz);
    if (ret != ASN_PARSE_E) {
        ret = -10105;
        goto done;
    }
    /* Bad data. */
    inSz = sizeof(badNoObjId);
    inOutIdx = 0;
    ret = wc_EccPublicKeyDecode(badNoObjId, &inOutIdx, key, inSz);
    if (ret != ASN_OBJECT_ID_E && ret != ASN_PARSE_E) {
        ret = -10106;
        goto done;
    }
    inSz = sizeof(badOneObjId);
    inOutIdx = 0;
    ret = wc_EccPublicKeyDecode(badOneObjId, &inOutIdx, key, inSz);
    if (ret != ASN_OBJECT_ID_E && ret != ASN_PARSE_E) {
        ret = -10107;
        goto done;
    }
    inSz = sizeof(badObjId1Len);
    inOutIdx = 0;
    ret = wc_EccPublicKeyDecode(badObjId1Len, &inOutIdx, key, inSz);
    if (ret != ASN_PARSE_E) {
        ret = -10108;
        goto done;
    }
    inSz = sizeof(badObj2d1Len);
    inOutIdx = 0;
    ret = wc_EccPublicKeyDecode(badObj2d1Len, &inOutIdx, key, inSz);
    if (ret != ASN_PARSE_E) {
        ret = -10109;
        goto done;
    }
    inSz = sizeof(badNotBitStr);
    inOutIdx = 0;
    ret = wc_EccPublicKeyDecode(badNotBitStr, &inOutIdx, key, inSz);
    if (ret != ASN_BITSTR_E && ret != ASN_PARSE_E) {
        ret = -10110;
        goto done;
    }
    inSz = sizeof(badBitStrLen);
    inOutIdx = 0;
    ret = wc_EccPublicKeyDecode(badBitStrLen, &inOutIdx, key, inSz);
    if (ret != ASN_PARSE_E) {
        ret = -10111;
        goto done;
    }
    inSz = sizeof(badNoBitStrZero);
    inOutIdx = 0;
    ret = wc_EccPublicKeyDecode(badNoBitStrZero, &inOutIdx, key, inSz);
    if (ret != ASN_EXPECT_0_E && ret != ASN_PARSE_E) {
        ret = -10112;
        goto done;
    }
    inSz = sizeof(badPoint);
    inOutIdx = 0;
    ret = wc_EccPublicKeyDecode(badPoint, &inOutIdx, key, inSz);
    if (ret != ASN_ECC_KEY_E && ret != ASN_PARSE_E) {
        ret = -10113;
        goto done;
    }

    inSz = sizeof(good);
    inOutIdx = 0;
    ret = wc_EccPublicKeyDecode(good, &inOutIdx, key, inSz);
    if (ret != 0) {
        ret = -10114;
        goto done;
    }

done:

#ifdef WOLFSSL_SMALL_STACK
    if (key != NULL) {
        wc_ecc_free(key);
        XFREE(key, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    }
#else
    wc_ecc_free(key);
#endif

    return ret;
}
#endif /* WOLFSSL_CERT_EXT */

#ifdef WOLFSSL_CUSTOM_CURVES
static const byte eccKeyExplicitCurve[] = {
    0x30, 0x81, 0xf5, 0x30, 0x81, 0xae, 0x06, 0x07,
    0x2a, 0x86, 0x48, 0xce, 0x3d, 0x02, 0x01, 0x30,
    0x81, 0xa2, 0x02, 0x01, 0x01, 0x30, 0x2c, 0x06,
    0x07, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x01, 0x01,
    0x02, 0x21, 0x00, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff,
    0xff, 0xfc, 0x2f, 0x30, 0x06, 0x04, 0x01, 0x00,
    0x04, 0x01, 0x07, 0x04, 0x41, 0x04, 0x79, 0xbe,
    0x66, 0x7e, 0xf9, 0xdc, 0xbb, 0xac, 0x55, 0xa0,
    0x62, 0x95, 0xce, 0x87, 0x0b, 0x07, 0x02, 0x9b,
    0xfc, 0xdb, 0x2d, 0xce, 0x28, 0xd9, 0x59, 0xf2,
    0x81, 0x5b, 0x16, 0xf8, 0x17, 0x98, 0x48, 0x3a,
    0xda, 0x77, 0x26, 0xa3, 0xc4, 0x65, 0x5d, 0xa4,
    0xfb, 0xfc, 0x0e, 0x11, 0x08, 0xa8, 0xfd, 0x17,
    0xb4, 0x48, 0xa6, 0x85, 0x54, 0x19, 0x9c, 0x47,
    0xd0, 0x8f, 0xfb, 0x10, 0xd4, 0xb8, 0x02, 0x21,
    0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xfe, 0xba, 0xae, 0xdc, 0xe6, 0xaf, 0x48, 0xa0,
    0x3b, 0xbf, 0xd2, 0x5e, 0x8c, 0xd0, 0x36, 0x41,
    0x41, 0x02, 0x01, 0x01, 0x03, 0x42, 0x00, 0x04,
    0x3c, 0x4c, 0xc9, 0x5e, 0x2e, 0xa2, 0x3d, 0x49,
    0xcc, 0x5b, 0xff, 0x4f, 0xc9, 0x2e, 0x1d, 0x4a,
    0xc6, 0x21, 0xf6, 0xf3, 0xe6, 0x0b, 0x4f, 0xa9,
    0x9d, 0x74, 0x99, 0xdd, 0x97, 0xc7, 0x6e, 0xbe,
    0x14, 0x2b, 0x39, 0x9d, 0x63, 0xc7, 0x97, 0x0d,
    0x45, 0x25, 0x40, 0x30, 0x77, 0x05, 0x76, 0x88,
    0x38, 0x96, 0x29, 0x7d, 0x9c, 0xe1, 0x50, 0xbe,
    0xac, 0xf0, 0x1d, 0x86, 0xf4, 0x2f, 0x65, 0x0b
};

static int ecc_test_custom_curves(WC_RNG* rng)
{
    int     ret;
    word32  inOutIdx;
#ifdef WOLFSSL_SMALL_STACK
    ecc_key *key = (ecc_key *)XMALLOC(sizeof *key, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
#else
    ecc_key key[1];
#endif

    /* test use of custom curve - using BRAINPOOLP256R1 for test */
#if defined(HAVE_ECC_BRAINPOOL) && !defined(HAVE_INTEL_QA)
    #ifndef WOLFSSL_ECC_CURVE_STATIC
        WOLFSSL_SMALL_STACK_STATIC const ecc_oid_t ecc_oid_brainpoolp256r1[] = {
            0x2B,0x24,0x03,0x03,0x02,0x08,0x01,0x01,0x07
        };
        #define ecc_oid_brainpoolp256r1_sz \
              (sizeof(ecc_oid_brainpoolp256r1) / sizeof(ecc_oid_t))
    #else
        #define ecc_oid_brainpoolp256r1 { \
            0x2B,0x24,0x03,0x03,0x02,0x08,0x01,0x01,0x07 \
        }
        #define ecc_oid_brainpoolp256r1_sz 9
    #endif
    #define ecc_oid_brainpoolp256r1_sum 104

    WOLFSSL_SMALL_STACK_STATIC const ecc_set_type ecc_dp_brainpool256r1 = {
        32,                                                                 /* size/bytes */
        ECC_CURVE_CUSTOM,                                                   /* ID         */
        "BRAINPOOLP256R1",                                                  /* curve name */
        "A9FB57DBA1EEA9BC3E660A909D838D726E3BF623D52620282013481D1F6E5377", /* prime      */
        "7D5A0975FC2C3057EEF67530417AFFE7FB8055C126DC5C6CE94A4B44F330B5D9", /* A          */
        "26DC5C6CE94A4B44F330B5D9BBD77CBF958416295CF7E1CE6BCCDC18FF8C07B6", /* B          */
        "A9FB57DBA1EEA9BC3E660A909D838D718C397AA3B561A6F7901E0E82974856A7", /* order      */
        "8BD2AEB9CB7E57CB2C4B482FFC81B7AFB9DE27E1E3BD23C23A4453BD9ACE3262", /* Gx         */
        "547EF835C3DAC4FD97F8461A14611DC9C27745132DED8E545C1D54C72F046997", /* Gy         */
        ecc_oid_brainpoolp256r1,                                            /* oid/oidSz  */
        ecc_oid_brainpoolp256r1_sz,
        ecc_oid_brainpoolp256r1_sum,                                        /* oid sum    */
        1,                                                                  /* cofactor   */
    };
#endif /* HAVE_ECC_BRAINPOOL */

#ifdef WOLFSSL_SMALL_STACK
    if (! key) {
        ret = MEMORY_E;
        goto done;
    }
#endif

    XMEMSET(key, 0, sizeof *key);

#if defined(HAVE_ECC_BRAINPOOL) && !defined(HAVE_INTEL_QA)
    ret = ecc_test_curve_size(rng, 0, ECC_TEST_VERIFY_COUNT, ECC_CURVE_DEF,
        &ecc_dp_brainpool256r1);
    if (ret != 0) {
        printf("ECC test for custom curve failed! %d\n", ret);
        goto done;
    }
#endif

#if defined(HAVE_ECC_BRAINPOOL) || defined(HAVE_ECC_KOBLITZ)
    {
        int curve_id;
        #ifdef HAVE_ECC_BRAINPOOL
            curve_id = ECC_BRAINPOOLP256R1;
        #else
            curve_id = ECC_SECP256K1;
        #endif
        /* Test and demonstrate use of non-SECP curve */
        ret = ecc_test_curve_size(rng, 0, ECC_TEST_VERIFY_COUNT, curve_id, NULL);
        if (ret < 0) {
            printf("ECC test for curve_id %d failed! %d\n", curve_id, ret);
            goto done;
        }
    }
#endif

    ret = wc_ecc_init_ex(key, HEAP_HINT, devId);
    if (ret != 0) {
        ret = -10120;
        goto done;
    }

    inOutIdx = 0;
    ret = wc_EccPublicKeyDecode(eccKeyExplicitCurve, &inOutIdx, key,
                                                   sizeof(eccKeyExplicitCurve));
    if (ret != 0)
        ret = -10121;

  done:

#ifdef WOLFSSL_SMALL_STACK
    if (key) {
        wc_ecc_free(key);
        XFREE(key, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    }
#else
    wc_ecc_free(key);
#endif

    (void)rng;

    return ret;
}
#endif /* WOLFSSL_CUSTOM_CURVES */

#ifdef WOLFSSL_CERT_GEN

/* Make Cert / Sign example for ECC cert and ECC CA */
static int ecc_test_cert_gen(WC_RNG* rng)
{
    int ret;
#ifdef WOLFSSL_SMALL_STACK
    Cert        *myCert = (Cert *)XMALLOC(sizeof *myCert, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    #ifdef WOLFSSL_TEST_CERT
    DecodedCert *decode = (DecodedCert *)XMALLOC(sizeof *decode, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    #endif
    ecc_key *caEccKey = (ecc_key *)XMALLOC(sizeof *caEccKey, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    ecc_key *certPubKey = (ecc_key *)XMALLOC(sizeof *certPubKey, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
#else
    Cert        myCert[1];
    #ifdef WOLFSSL_TEST_CERT
    DecodedCert decode[1];
    #endif
    ecc_key caEccKey[1];
    ecc_key certPubKey[1];
#endif
    int         certSz;
    size_t      bytes;
    word32      idx = 0;
#ifndef USE_CERT_BUFFERS_256
    XFILE       file;
#endif
#ifdef WOLFSSL_SMALL_STACK
    byte* der = NULL;
#else
    byte der[FOURK_BUF];
#endif

#ifdef WOLFSSL_SMALL_STACK
    if ((myCert == NULL)
    #ifdef WOLFSSL_TEST_CERT
        || (decode == NULL)
    #endif
        || (caEccKey == NULL) || (certPubKey == NULL))
        ERROR_OUT(MEMORY_E, exit);
#endif

    XMEMSET(caEccKey, 0, sizeof *caEccKey);
    XMEMSET(certPubKey, 0, sizeof *certPubKey);

#ifdef WOLFSSL_SMALL_STACK
    der = (byte*)XMALLOC(FOURK_BUF, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    if (der == NULL) {
        ERROR_OUT(-10130, exit);
    }
#endif

    /* Get cert private key */
#ifdef ENABLE_ECC384_CERT_GEN_TEST
    /* Get Cert Key 384 */
#ifdef USE_CERT_BUFFERS_256
    XMEMCPY(der, ca_ecc_key_der_384, sizeof_ca_ecc_key_der_384);
    bytes = sizeof_ca_ecc_key_der_384;
#else
    file = XFOPEN(eccCaKey384File, "rb");
    if (!file) {
        ERROR_OUT(-10131, exit);
    }

    bytes = XFREAD(der, 1, FOURK_BUF, file);
    XFCLOSE(file);
    (void)eccCaKeyFile;
#endif /* USE_CERT_BUFFERS_256 */
#else
#ifdef USE_CERT_BUFFERS_256
    XMEMCPY(der, ca_ecc_key_der_256, sizeof_ca_ecc_key_der_256);
    bytes = sizeof_ca_ecc_key_der_256;
#else
    file = XFOPEN(eccCaKeyFile, "rb");
    if (!file) {
        ERROR_OUT(-10132, exit);
    }
    bytes = XFREAD(der, 1, FOURK_BUF, file);
    XFCLOSE(file);
#ifdef ENABLE_ECC384_CERT_GEN_TEST
    (void)eccCaKey384File;
#endif
#endif /* USE_CERT_BUFFERS_256 */
#endif /* ENABLE_ECC384_CERT_GEN_TEST */

    /* Get CA Key */
    ret = wc_ecc_init_ex(caEccKey, HEAP_HINT, devId);
    if (ret != 0) {
        ERROR_OUT(-10133, exit);
    }
    ret = wc_EccPrivateKeyDecode(der, &idx, caEccKey, (word32)bytes);
    if (ret != 0) {
        ERROR_OUT(-10134, exit);
    }

    /* Make a public key */
    ret = wc_ecc_init_ex(certPubKey, HEAP_HINT, devId);
    if (ret != 0) {
        ERROR_OUT(-10135, exit);
    }

    ret = wc_ecc_make_key(rng, ECC_KEYGEN_SIZE, certPubKey);
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &certPubKey->asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    if (ret != 0) {
        ERROR_OUT(-10136, exit);
    }
    TEST_SLEEP();

    /* Setup Certificate */
    if (wc_InitCert_ex(myCert, HEAP_HINT, devId)) {
        ERROR_OUT(-10137, exit);
    }

#ifndef NO_SHA256
    myCert->sigType = CTC_SHA256wECDSA;
#else
    myCert->sigType = CTC_SHAwECDSA;
#endif
    XMEMCPY(&myCert->subject, &certDefaultName, sizeof(CertName));

#ifdef WOLFSSL_CERT_EXT
    /* add Policies */
    XSTRNCPY(myCert->certPolicies[0], "2.4.589440.587.101.2.1.9632587.1",
            CTC_MAX_CERTPOL_SZ);
    XSTRNCPY(myCert->certPolicies[1], "1.2.13025.489.1.113549",
            CTC_MAX_CERTPOL_SZ);
    myCert->certPoliciesNb = 2;

    /* add SKID from the Public Key */
    if (wc_SetSubjectKeyIdFromPublicKey(myCert, NULL, certPubKey) != 0) {
        ERROR_OUT(-10138, exit);
    }

    /* add AKID from the Public Key */
    if (wc_SetAuthKeyIdFromPublicKey(myCert, NULL, caEccKey) != 0) {
        ERROR_OUT(-10139, exit);
    }

    /* add Key Usage */
    if (wc_SetKeyUsage(myCert, certKeyUsage) != 0) {
        ERROR_OUT(-10140, exit);
    }
#endif /* WOLFSSL_CERT_EXT */

#ifdef ENABLE_ECC384_CERT_GEN_TEST
#if defined(USE_CERT_BUFFERS_256)
    ret = wc_SetIssuerBuffer(myCert, ca_ecc_cert_der_384,
                                      sizeof_ca_ecc_cert_der_384);
#else
    ret = wc_SetIssuer(myCert, eccCaCert384File);
    (void)eccCaCertFile;
#endif
#else
#if defined(USE_CERT_BUFFERS_256)
    ret = wc_SetIssuerBuffer(myCert, ca_ecc_cert_der_256,
                                      sizeof_ca_ecc_cert_der_256);
#else
    ret = wc_SetIssuer(myCert, eccCaCertFile);
    #ifdef ENABLE_ECC384_CERT_GEN_TEST
    (void)eccCaCert384File;
    #endif
#endif
#endif /* ENABLE_ECC384_CERT_GEN_TEST */
    if (ret < 0) {
        ERROR_OUT(-10141, exit);
    }

    certSz = wc_MakeCert(myCert, der, FOURK_BUF, NULL, certPubKey, rng);
    if (certSz < 0) {
        ERROR_OUT(-10142, exit);
    }

    ret = 0;
    do {
    #if defined(WOLFSSL_ASYNC_CRYPT)
        ret = wc_AsyncWait(ret, &caEccKey->asyncDev, WC_ASYNC_FLAG_CALL_AGAIN);
    #endif
        if (ret >= 0) {
            ret = wc_SignCert(myCert->bodySz, myCert->sigType, der,
                              FOURK_BUF, NULL, caEccKey, rng);
        }
    } while (ret == WC_PENDING_E);
    if (ret < 0) {
        ERROR_OUT(-10143, exit);
    }
    certSz = ret;
    TEST_SLEEP();

#ifdef WOLFSSL_TEST_CERT
    InitDecodedCert(decode, der, certSz, HEAP_HINT);
    ret = ParseCert(decode, CERT_TYPE, NO_VERIFY, 0);
    if (ret != 0) {
        FreeDecodedCert(decode);
        ERROR_OUT(-10144, exit);

    }
    FreeDecodedCert(decode);
#endif

    ret = SaveDerAndPem(der, certSz, certEccDerFile, certEccPemFile,
        CERT_TYPE, -6735);
    if (ret != 0) {
        goto exit;
    }

exit:
#ifdef WOLFSSL_SMALL_STACK
    XFREE(der, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
#endif

#ifdef WOLFSSL_SMALL_STACK
    if (myCert != NULL)
        XFREE(myCert, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
#ifdef WOLFSSL_TEST_CERT
    if (decode != NULL)
        XFREE(decode, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
#endif
    if (caEccKey != NULL) {
        wc_ecc_free(caEccKey);
        XFREE(caEccKey, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    }
    if (certPubKey != NULL) {
        wc_ecc_free(certPubKey);
        XFREE(certPubKey, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    }
#else
    wc_ecc_free(certPubKey);
    wc_ecc_free(caEccKey);
#endif

    return ret;
}
#endif /* WOLFSSL_CERT_GEN */

#if !defined(HAVE_FIPS) && !defined(HAVE_SELFTEST) && \
    !defined(WOLFSSL_NO_MALLOC) && !defined(WOLF_CRYPTO_CB_ONLY_ECC)
/* Test for the wc_ecc_key_new() and wc_ecc_key_free() functions. */
static int ecc_test_allocator(WC_RNG* rng)
{
    int ret = 0;
    ecc_key* key;
#ifdef WC_NO_RNG
    word32 idx = 0;
#endif

    key = wc_ecc_key_new(HEAP_HINT);
    if (key == NULL) {
        ERROR_OUT(-10150, exit);
    }

#ifndef WC_NO_RNG
    ret = wc_ecc_make_key(rng, ECC_KEYGEN_SIZE, key);
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &key->asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    if (ret != 0) {
        ERROR_OUT(-10151, exit);
    }
#else
    /* use test ECC key */
    ret = wc_EccPrivateKeyDecode(ecc_key_der_256, &idx, key,
        (word32)sizeof_ecc_key_der_256);
    (void)rng;
#endif

exit:
    wc_ecc_key_free(key);
    return ret;
}
#endif

/* ECC Non-blocking tests for Sign and Verify */
/* Requires SP math and supports P384 or P256 */
/* ./configure --enable-ecc=nonblock --enable-sp=yes,nonblock CFLAGS="-DWOLFSSL_PUBLIC_MP" */
#if defined(WC_ECC_NONBLOCK) && defined(WOLFSSL_PUBLIC_MP) && \
    defined(HAVE_ECC_SIGN) && defined(HAVE_ECC_VERIFY) && \
    defined(WOLFSSL_HAVE_SP_ECC)
/* Test Data - Random */
static const uint8_t kMsg[] = {
    0x69, 0xbc, 0x9f, 0xce, 0x68, 0x17, 0xc2, 0x10, 0xea, 0xfc, 0x10, 0x65, 0x67, 0x52, 0xed, 0x78,
    0x6e, 0xb8, 0x83, 0x9c, 0x9a, 0xb4, 0x56, 0x0d, 0xc1, 0x0d, 0x1f, 0x78, 0x6e, 0x75, 0xd7, 0xbe,
    0x92, 0x6b, 0x12, 0xf6, 0x76, 0x60, 0x8e, 0xb1, 0xf4, 0x19, 0x0c, 0x81, 0xe7, 0x54, 0x5e, 0xbc,
    0xe0, 0xae, 0xc2, 0x7d, 0x1b, 0xc4, 0x6e, 0xec, 0xb1, 0x99, 0x6c, 0xbf, 0x0e, 0x38, 0xa8, 0x01,
    0xa6, 0x9a, 0x48, 0x12, 0xe4, 0xc9, 0x3b, 0xf0, 0x63, 0x46, 0x15, 0xb4, 0x61, 0xa8, 0x1a, 0x60,
    0x71, 0x87, 0x98, 0xd7, 0x6f, 0x98, 0x7b, 0x2d, 0xb9, 0x19, 0x1b, 0x21, 0x9c, 0x70, 0x58, 0xe8,
    0x0d, 0x0f, 0xe9, 0x2d, 0x9a, 0x9a, 0xf1, 0x55, 0xa0, 0x4c, 0xd3, 0x07, 0xbd, 0x97, 0x48, 0xec,
    0x88, 0x0a, 0xaf, 0xb3, 0x80, 0x78, 0xa4, 0x59, 0x43, 0x57, 0xd3, 0xa7, 0x01, 0x66, 0x0e, 0xfc
};

/* ECC Private Key "d" */
static const uint8_t kPrivKey[] = {
#ifdef HAVE_ECC521
    /* SECP521R1 */
    /* d */
    0x01, 0x68, 0x91, 0x33, 0x53, 0xe2, 0x90, 0x68,
    0x11, 0x8f, 0xaa, 0xa8, 0x76, 0x0c, 0xf7, 0x2a,
    0x07, 0x1b, 0x92, 0x2a, 0xa7, 0x82, 0x3d, 0xfa,
    0x83, 0xce, 0x70, 0xc8, 0xc2, 0x60, 0x82, 0xfe,
    0x18, 0x88, 0x68, 0xda, 0x6a, 0x83, 0x46, 0x78,
    0xe4, 0xe9, 0xe9, 0xcc, 0x51, 0x7f, 0xed, 0x81,
    0x02, 0x32, 0xee, 0x26, 0x87, 0xcc, 0xed, 0x63,
    0x3f, 0x39, 0x27, 0xf0, 0xd7, 0x17, 0x77, 0xa1,
    0xa4, 0x36
#elif defined(HAVE_ECC384)
    /* SECP384R1 */
    /* d */
    0xa4, 0xe5, 0x06, 0xe8, 0x06, 0x16, 0x3e, 0xab,
    0x89, 0xf8, 0x60, 0x43, 0xc0, 0x60, 0x25, 0xdb,
    0xba, 0x7b, 0xfe, 0x19, 0x35, 0x08, 0x55, 0x65,
    0x76, 0xe2, 0xdc, 0xe0, 0x01, 0x8b, 0x6b, 0x68,
    0xdf, 0xcf, 0x6f, 0x80, 0x12, 0xce, 0x79, 0x37,
    0xeb, 0x2b, 0x9c, 0x7b, 0xc4, 0x68, 0x1c, 0x74
#else
    /* SECP256R1 */
    /* d */
    0x1e, 0xe7, 0x70, 0x07, 0xd3, 0x30, 0x94, 0x39,
    0x28, 0x90, 0xdf, 0x23, 0x88, 0x2c, 0x4a, 0x34,
    0x15, 0xdb, 0x4c, 0x43, 0xcd, 0xfa, 0xe5, 0x1f,
    0x3d, 0x4c, 0x37, 0xfe, 0x59, 0x3b, 0x96, 0xd8
#endif
};

/* ECC public key Qx/Qy */
static const uint8_t kPubKey[] = {
#ifdef HAVE_ECC521
    /* SECP521R1 */
    /* Qx */
    0x01, 0x62, 0x6e, 0xf1, 0x00, 0xec, 0xd8, 0x99,
    0x58, 0x9b, 0x80, 0x6b, 0xfe, 0x2c, 0xf1, 0xb2,
    0xf0, 0xc8, 0x48, 0xdf, 0xac, 0xd2, 0x3b, 0x71,
    0x29, 0xab, 0xf0, 0x66, 0x63, 0xd8, 0x8e, 0xb5,
    0xc8, 0xc2, 0xfc, 0x99, 0x44, 0xe2, 0x45, 0xb1,
    0x5a, 0x7b, 0xb9, 0x73, 0x01, 0xda, 0x79, 0xec,
    0x9c, 0x26, 0x27, 0x34, 0x45, 0x26, 0xd5, 0x89,
    0x4b, 0x44, 0xfe, 0x69, 0x4e, 0x72, 0x14, 0xe3,
    0x8b, 0xbc,
    /* Qy */
    0x00, 0x0f, 0x09, 0xa2, 0x03, 0xc3, 0x5a, 0xdc,
    0x95, 0x82, 0xf6, 0xf9, 0xf6, 0x9c, 0xff, 0xb5,
    0x6b, 0x75, 0x95, 0x4b, 0xa4, 0x28, 0x5d, 0x9e,
    0x90, 0x04, 0xd1, 0xc0, 0x1e, 0xd5, 0xfd, 0x43,
    0x9e, 0x1e, 0x83, 0xc0, 0x11, 0x2b, 0x2b, 0x07,
    0x6d, 0xa9, 0x7a, 0x10, 0xd7, 0x67, 0xe7, 0x51,
    0x37, 0x24, 0xd8, 0xbf, 0x03, 0x0d, 0x8b, 0xb5,
    0x40, 0x5c, 0x4f, 0xd6, 0x13, 0x73, 0x42, 0xbc,
    0x91, 0xd9
#elif defined(HAVE_ECC384)
    /* SECP384R1 */
    /* Qx */
    0xea, 0xcf, 0x93, 0x4f, 0x2c, 0x09, 0xbb, 0x39,
    0x14, 0x0f, 0x56, 0x64, 0xc3, 0x40, 0xb4, 0xdf,
    0x0e, 0x63, 0xae, 0xe5, 0x71, 0x4b, 0x00, 0xcc,
    0x04, 0x97, 0xff, 0xe1, 0xe9, 0x38, 0x96, 0xbb,
    0x5f, 0x91, 0xb2, 0x6a, 0xcc, 0xb5, 0x39, 0x5f,
    0x8f, 0x70, 0x59, 0xf1, 0x01, 0xf6, 0x5a, 0x2b,
    /* Qy */
    0x01, 0x6c, 0x68, 0x0b, 0xcf, 0x55, 0x25, 0xaf,
    0x6d, 0x98, 0x48, 0x0a, 0xa8, 0x74, 0xc9, 0xa9,
    0x17, 0xa0, 0x0c, 0xc3, 0xfb, 0xd3, 0x23, 0x68,
    0xfe, 0x04, 0x3c, 0x63, 0x50, 0x88, 0x3b, 0xb9,
    0x4f, 0x7c, 0x67, 0x34, 0xf7, 0x3b, 0xa9, 0x73,
    0xe7, 0x1b, 0xc3, 0x51, 0x5e, 0x22, 0x18, 0xec
#else
    /* SECP256R1 */
    /* Qx */
    0x96, 0x93, 0x1c, 0x53, 0x0b, 0x43, 0x6c, 0x42,
    0x0c, 0x52, 0x90, 0xe4, 0xa7, 0xec, 0x98, 0xb1,
    0xaf, 0xd4, 0x14, 0x49, 0xd8, 0xc1, 0x42, 0x82,
    0x04, 0x78, 0xd1, 0x90, 0xae, 0xa0, 0x6c, 0x07,
    /* Qy */
    0xf2, 0x3a, 0xb5, 0x10, 0x32, 0x8d, 0xce, 0x9e,
    0x76, 0xa0, 0xd2, 0x8c, 0xf3, 0xfc, 0xa9, 0x94,
    0x43, 0x24, 0xe6, 0x82, 0x00, 0x40, 0xc6, 0xdb,
    0x1c, 0x2f, 0xcd, 0x38, 0x4b, 0x60, 0xdd, 0x61
#endif
};

/* ECC Curve */
#ifdef HAVE_ECC521
    /* SECP521R1 */
    #define ECC_CURVE_SZ 66
    #define ECC_CURVE_ID ECC_SECP521R1
#elif defined(HAVE_ECC384)
    /* SECP384R1 */
    #define ECC_CURVE_SZ 48
    #define ECC_CURVE_ID ECC_SECP384R1
#else
    /* SECP256R1 */
    #define ECC_CURVE_SZ 32
    #define ECC_CURVE_ID ECC_SECP256R1
#endif

/* Hash Algorithm */
#if defined(HAVE_ECC521) && defined(WOLFSSL_SHA3)
    #define HASH_DIGEST_SZ  WC_SHA3_512_DIGEST_SIZE
    #define HASH_SHA_VER    3
    #define CRYPTO_HASH_FN  crypto_sha3_512
#elif defined(HAVE_ECC521) && defined(WOLFSSL_SHA512)
    #define HASH_DIGEST_SZ  WC_SHA512_DIGEST_SIZE
    #define HASH_SHA_VER    2
    #define CRYPTO_HASH_FN  crypto_sha2_512
#elif defined(HAVE_ECC384) && defined(WOLFSSL_SHA3)
    #define HASH_DIGEST_SZ  WC_SHA3_384_DIGEST_SIZE
    #define HASH_SHA_VER    3
    #define CRYPTO_HASH_FN  crypto_sha3_384
#elif defined(HAVE_ECC384) && defined(WOLFSSL_SHA384)
    #define HASH_DIGEST_SZ  WC_SHA384_DIGEST_SIZE
    #define HASH_SHA_VER    2
    #define CRYPTO_HASH_FN  crypto_sha2_384
#elif !defined(NO_SHA256)
    #define HASH_DIGEST_SZ  WC_SHA256_DIGEST_SIZE
    #define HASH_SHA_VER    2
    #define CRYPTO_HASH_FN  crypto_sha2_256
#else
    #error test configuration not supported
#endif

#if defined(HAVE_ECC521) && defined(WOLFSSL_SHA3)
/* helper to perform hashing block by block */
static int crypto_sha3_512(const uint8_t *buf, uint32_t len, uint8_t *hash,
    uint32_t hashSz, uint32_t blkSz)
{
    int ret;
    uint32_t i = 0, chunk;
    wc_Sha3 sha3;

    /* validate arguments */
    if ((buf == NULL && len > 0) || hash == NULL ||
        hashSz < WC_SHA3_512_DIGEST_SIZE || blkSz == 0)
    {
        return BAD_FUNC_ARG;
    }

    /* Init Sha3_512 structure */
    ret = wc_InitSha3_512(&sha3, NULL, INVALID_DEVID);
    if (ret != 0) {
        return ret;
    }
    while (i < len) {
        chunk = blkSz;
        if ((chunk + i) > len)
            chunk = len - i;
        /* Perform chunked update */
        ret = wc_Sha3_512_Update(&sha3, (buf + i), chunk);
        if (ret != 0) {
            break;
        }
        i += chunk;
    }
    if (ret == 0) {
        /* Get final digest result */
        ret = wc_Sha3_512_Final(&sha3, hash);
    }
    return ret;
}
#elif defined(HAVE_ECC521) && defined(WOLFSSL_SHA512)
/* helper to perform hashing block by block */
static int crypto_sha2_512(const uint8_t *buf, uint32_t len, uint8_t *hash,
    uint32_t hashSz, uint32_t blkSz)
{
    int ret;
    uint32_t i = 0, chunk;
    wc_Sha512 sha512;

    /* validate arguments */
    if ((buf == NULL && len > 0) || hash == NULL ||
        hashSz < WC_SHA512_DIGEST_SIZE || blkSz == 0)
    {
        return BAD_FUNC_ARG;
    }

    /* Init Sha512 structure */
    ret = wc_InitSha512(&sha512);
    if (ret != 0) {
        return ret;
    }
    while (i < len) {
        chunk = blkSz;
        if ((chunk + i) > len)
            chunk = len - i;
        /* Perform chunked update */
        ret = wc_Sha512Update(&sha512, (buf + i), chunk);
        if (ret != 0) {
            break;
        }
        i += chunk;
    }
    if (ret == 0) {
        /* Get final digest result */
        ret = wc_Sha512Final(&sha512, hash);
    }
    return ret;
}
#elif defined(HAVE_ECC384) && defined(WOLFSSL_SHA3)
/* helper to perform hashing block by block */
static int crypto_sha3_384(const uint8_t *buf, uint32_t len, uint8_t *hash,
    uint32_t hashSz, uint32_t blkSz)
{
    int ret;
    uint32_t i = 0, chunk;
    wc_Sha3 sha3;

    /* validate arguments */
    if ((buf == NULL && len > 0) || hash == NULL ||
        hashSz < WC_SHA3_384_DIGEST_SIZE || blkSz == 0)
    {
        return BAD_FUNC_ARG;
    }

    /* Init Sha3_384 structure */
    ret = wc_InitSha3_384(&sha3, NULL, INVALID_DEVID);
    if (ret != 0) {
        return ret;
    }
    while (i < len) {
        chunk = blkSz;
        if ((chunk + i) > len)
            chunk = len - i;
        /* Perform chunked update */
        ret = wc_Sha3_384_Update(&sha3, (buf + i), chunk);
        if (ret != 0) {
            break;
        }
        i += chunk;
    }
    if (ret == 0) {
        /* Get final digest result */
        ret = wc_Sha3_384_Final(&sha3, hash);
    }
    return ret;
}
#elif defined(HAVE_ECC384) && defined(WOLFSSL_SHA384)
/* helper to perform hashing block by block */
static int crypto_sha2_384(const uint8_t *buf, uint32_t len, uint8_t *hash,
    uint32_t hashSz, uint32_t blkSz)
{
    int ret;
    uint32_t i = 0, chunk;
    wc_Sha384 sha384;

    /* validate arguments */
    if ((buf == NULL && len > 0) || hash == NULL ||
        hashSz < WC_SHA384_DIGEST_SIZE || blkSz == 0)
    {
        return BAD_FUNC_ARG;
    }

    /* Init Sha384 structure */
    ret = wc_InitSha384(&sha384);
    if (ret != 0) {
        return ret;
    }
    while (i < len) {
        chunk = blkSz;
        if ((chunk + i) > len)
            chunk = len - i;
        /* Perform chunked update */
        ret = wc_Sha384Update(&sha384, (buf + i), chunk);
        if (ret != 0) {
            break;
        }
        i += chunk;
    }
    if (ret == 0) {
        /* Get final digest result */
        ret = wc_Sha384Final(&sha384, hash);
    }
    return ret;
}
#elif !defined(NO_SHA256)
/* helper to perform hashing block by block */
static int crypto_sha2_256(const uint8_t *buf, uint32_t len, uint8_t *hash,
    uint32_t hashSz, uint32_t blkSz)
{
    int ret;
    uint32_t i = 0, chunk;
    wc_Sha256 sha256;

    /* validate arguments */
    if ((buf == NULL && len > 0) || hash == NULL ||
        hashSz < WC_SHA256_DIGEST_SIZE || blkSz == 0)
    {
        return BAD_FUNC_ARG;
    }

    /* Init Sha256 structure */
    ret = wc_InitSha256(&sha256);
    if (ret != 0) {
        return ret;
    }
    while (i < len) {
        chunk = blkSz;
        if ((chunk + i) > len)
            chunk = len - i;
        /* Perform chunked update */
        ret = wc_Sha256Update(&sha256, (buf + i), chunk);
        if (ret != 0) {
            break;
        }
        i += chunk;
    }
    if (ret == 0) {
        /* Get final digest result */
        ret = wc_Sha256Final(&sha256, hash);
    }
    return ret;
}
#endif

/* perform verify of signature and hash using public key */
/* key is public Qx + public Qy */
/* sig is r + s */
static int crypto_ecc_verify(const uint8_t *key, uint32_t keySz,
    const uint8_t *hash, uint32_t hashSz, const uint8_t *sig, uint32_t sigSz,
    uint32_t curveSz, int curveId)
{
    int ret, verify_res = 0, count = 0;
    mp_int r, s;
    ecc_key ecc;
    ecc_nb_ctx_t nb_ctx;

    /* validate arguments */
    if (key == NULL || hash == NULL || sig == NULL || curveSz == 0 ||
        hashSz == 0 || keySz < (curveSz*2) || sigSz < (curveSz*2))
    {
        return BAD_FUNC_ARG;
    }

    /* Setup the ECC key */
    ret = wc_ecc_init(&ecc);
    if (ret < 0) {
        return ret;
    }

    ret = wc_ecc_set_nonblock(&ecc, &nb_ctx);
    if (ret != MP_OKAY) {
        wc_ecc_free(&ecc);
        return ret;
    }

    /* Setup the signature r/s variables */
    ret = mp_init(&r);
    if (ret != MP_OKAY) {
        wc_ecc_free(&ecc);
        return ret;
    }
    ret = mp_init(&s);
    if (ret != MP_OKAY) {
        mp_clear(&r);
        wc_ecc_free(&ecc);
        return ret;
    }

    /* Import public key x/y */
    ret = wc_ecc_import_unsigned(
        &ecc,
        (byte*)key,             /* Public "x" Coordinate */
        (byte*)(key + curveSz), /* Public "y" Coordinate */
        NULL,                   /* Private "d" (optional) */
        curveId                 /* ECC Curve Id */
    );
    /* Make sure it was a public key imported */
    if (ret == 0 && ecc.type != ECC_PUBLICKEY) {
        ret = ECC_BAD_ARG_E;
    }

    /* Import signature r/s */
    if (ret == 0) {
        ret = mp_read_unsigned_bin(&r, sig,  curveSz);
    }
    if (ret == 0) {
        ret = mp_read_unsigned_bin(&s, sig + curveSz, curveSz);
    }

    /* Verify ECC Signature */
    if (ret == 0) {
        do {
            ret = wc_ecc_verify_hash_ex(
                &r, &s,       /* r/s as mp_int */
                hash, hashSz, /* computed hash digest */
                &verify_res,  /* verification result 1=success */
                &ecc
            );
            count++;

            /* This is where real-time work could be called */
        } while (ret == FP_WOULDBLOCK);
    #ifdef DEBUG_WOLFSSL
        printf("ECC non-block verify: %d times\n", count);
    #endif
    }

    /* check verify result */
    if (ret == 0 && verify_res == 0) {
        ret = SIG_VERIFY_E;
    }

    mp_clear(&r);
    mp_clear(&s);
    wc_ecc_free(&ecc);

    (void)count;

    return ret;
}

/* perform signature operation against hash using private key */
static int crypto_ecc_sign(const uint8_t *key, uint32_t keySz,
    const uint8_t *hash, uint32_t hashSz, uint8_t *sig, uint32_t* sigSz,
    uint32_t curveSz, int curveId, WC_RNG* rng)
{
    int ret, count = 0;
    mp_int r, s;
    ecc_key ecc;
    ecc_nb_ctx_t nb_ctx;

    /* validate arguments */
    if (key == NULL || hash == NULL || sig == NULL || sigSz == NULL ||
        curveSz == 0 || hashSz == 0 || keySz < curveSz || *sigSz < (curveSz*2))
    {
        return BAD_FUNC_ARG;
    }

    /* Initialize signature result */
    memset(sig, 0, curveSz*2);

    /* Setup the ECC key */
    ret = wc_ecc_init(&ecc);
    if (ret < 0) {
        return ret;
    }

    ret = wc_ecc_set_nonblock(&ecc, &nb_ctx);
    if (ret != MP_OKAY) {
        wc_ecc_free(&ecc);
        return ret;
    }

    /* Setup the signature r/s variables */
    ret = mp_init(&r);
    if (ret != MP_OKAY) {
        wc_ecc_free(&ecc);
        return ret;
    }
    ret = mp_init(&s);
    if (ret != MP_OKAY) {
        mp_clear(&r);
        wc_ecc_free(&ecc);
        return ret;
    }

    /* Import private key "k" */
    ret = wc_ecc_import_private_key_ex(
        key, keySz, /* private key "d" */
        NULL, 0,    /* public (optional) */
        &ecc,
        curveId     /* ECC Curve Id */
    );

    if (ret == 0) {
        do {
            /* Verify ECC Signature */
            ret = wc_ecc_sign_hash_ex(
                hash, hashSz, /* computed hash digest */
                rng, &ecc,    /* random and key context */
                &r, &s        /* r/s as mp_int */
            );
            count++;

            /* This is where real-time work could be called */
        } while (ret == FP_WOULDBLOCK);

    #ifdef DEBUG_WOLFSSL
        printf("ECC non-block sign: %d times\n", count);
    #endif
    }

    if (ret == 0) {
        /* export r/s */
        mp_to_unsigned_bin_len(&r, sig, curveSz);
        mp_to_unsigned_bin_len(&s, sig + curveSz, curveSz);
    }

    mp_clear(&r);
    mp_clear(&s);
    wc_ecc_free(&ecc);

    (void)count;

    return ret;
}

static int ecc_test_nonblock(WC_RNG* rng)
{
    int ret;
    uint8_t hash[HASH_DIGEST_SZ];
    uint8_t sig[ECC_CURVE_SZ*2];
    uint32_t sigSz = sizeof(sig);

    ret = CRYPTO_HASH_FN(
        kMsg, sizeof(kMsg), /* input message */
        hash, sizeof(hash), /* hash digest result */
        32                  /* configurable block / chunk size */
    );
    if (ret == 0) {
        /* Sign hash using private key */
        /* Note: result of an ECC sign varies for each call even with same
            private key and hash. This is because a new random public key is
            used for each operation. */
        ret = crypto_ecc_sign(
            kPrivKey, sizeof(kPrivKey), /* private key */
            hash, sizeof(hash),         /* computed hash digest */
            sig, &sigSz,                /* signature r/s */
            ECC_CURVE_SZ,               /* curve size in bytes */
            ECC_CURVE_ID,               /* curve id */
            rng
        );
    }

    if (ret == 0) {
        /* Verify generated signature is valid */
        ret = crypto_ecc_verify(
            kPubKey, sizeof(kPubKey),   /* public key point x/y */
            hash, sizeof(hash),         /* computed hash digest */
            sig, sigSz,                 /* signature r/s */
            ECC_CURVE_SZ,               /* curve size in bytes */
            ECC_CURVE_ID                /* curve id */
        );
    }

    return ret;
}
#endif /* WC_ECC_NONBLOCK && WOLFSSL_PUBLIC_MP && HAVE_ECC_SIGN && HAVE_ECC_VERIFY */

WOLFSSL_TEST_SUBROUTINE int ecc_test(void)
{
    int ret;
    WC_RNG rng;

#if defined(WOLFSSL_CERT_EXT) && \
    (!defined(NO_ECC256) || defined(HAVE_ALL_CURVES)) && ECC_MIN_KEY_SZ <= 256
    ret = ecc_decode_test();
    if (ret < 0)
        return ret;
#endif

#ifndef HAVE_FIPS
    ret = wc_InitRng_ex(&rng, HEAP_HINT, devId);
#else
    ret = wc_InitRng(&rng);
#endif
#ifndef WC_NO_RNG
    if (ret != 0)
        return -10300;
#else
    (void)ret;
#endif

#if (defined(HAVE_ECC112) || defined(HAVE_ALL_CURVES)) && ECC_MIN_KEY_SZ <= 112
    ret = ecc_test_curve(&rng, 14);
    if (ret < 0) {
        goto done;
    }
#endif /* HAVE_ECC112 */
#if (defined(HAVE_ECC128) || defined(HAVE_ALL_CURVES)) && ECC_MIN_KEY_SZ <= 128
    ret = ecc_test_curve(&rng, 16);
    if (ret < 0) {
        goto done;
    }
#endif /* HAVE_ECC128 */
#if (defined(HAVE_ECC160) || defined(HAVE_ALL_CURVES)) && ECC_MIN_KEY_SZ <= 160
    ret = ecc_test_curve(&rng, 20);
    if (ret < 0) {
        goto done;
    }
#endif /* HAVE_ECC160 */
#if (defined(HAVE_ECC192) || defined(HAVE_ALL_CURVES)) && ECC_MIN_KEY_SZ <= 192
    ret = ecc_test_curve(&rng, 24);
    if (ret < 0) {
        goto done;
    }
#endif /* HAVE_ECC192 */
#if (defined(HAVE_ECC224) || defined(HAVE_ALL_CURVES)) && ECC_MIN_KEY_SZ <= 224
    ret = ecc_test_curve(&rng, 28);
    if (ret < 0) {
        goto done;
    }
#endif /* HAVE_ECC224 */
#if (defined(HAVE_ECC239) || defined(HAVE_ALL_CURVES)) && ECC_MIN_KEY_SZ <= 239
    ret = ecc_test_curve(&rng, 30);
    if (ret < 0) {
        goto done;
    }
#endif /* HAVE_ECC239 */
#if (!defined(NO_ECC256) || defined(HAVE_ALL_CURVES)) && ECC_MIN_KEY_SZ <= 256
    ret = ecc_test_curve(&rng, 32);
    if (ret < 0) {
        goto done;
    }
#if !defined(WOLFSSL_ATECC508A) && !defined(WOLFSSL_ATECC608A) && \
    defined(HAVE_ECC_KEY_IMPORT) && defined(HAVE_ECC_KEY_EXPORT) && \
    !defined(WOLFSSL_NO_MALLOC) && !defined(WOLF_CRYPTO_CB_ONLY_ECC)
    ret = ecc_point_test();
    if (ret < 0) {
        goto done;
    }
#endif
    ret = ecc_def_curve_test(&rng);
    if (ret < 0) {
        goto done;
    }
#endif /* !NO_ECC256 */
#if (defined(HAVE_ECC320) || defined(HAVE_ALL_CURVES)) && ECC_MIN_KEY_SZ <= 320
    ret = ecc_test_curve(&rng, 40);
    if (ret < 0) {
        goto done;
    }
#endif /* HAVE_ECC320 */
#if (defined(HAVE_ECC384) || defined(HAVE_ALL_CURVES)) && ECC_MIN_KEY_SZ <= 384
    ret = ecc_test_curve(&rng, 48);
    if (ret < 0) {
        goto done;
    }
#endif /* HAVE_ECC384 */
#if (defined(HAVE_ECC512) || defined(HAVE_ALL_CURVES)) && ECC_MIN_KEY_SZ <= 512
    ret = ecc_test_curve(&rng, 64);
    if (ret < 0) {
        goto done;
    }
#endif /* HAVE_ECC512 */
#if (defined(HAVE_ECC521) || defined(HAVE_ALL_CURVES)) && ECC_MIN_KEY_SZ <= 521
    ret = ecc_test_curve(&rng, 66);
    if (ret < 0) {
        goto done;
    }
#endif /* HAVE_ECC521 */

#if defined(WOLFSSL_CUSTOM_CURVES)
    ret = ecc_test_custom_curves(&rng);
    if (ret != 0) {
        goto done;
    }
#endif

#if defined(HAVE_ECC_SIGN) && defined(WOLFSSL_ECDSA_DETERMINISTIC_K)
    ret = ecc_test_deterministic_k(&rng);
    if (ret != 0) {
        printf("ecc_test_deterministic_k failed! %d\n", ret);
        goto done;
    }
    #ifdef WOLFSSL_PUBLIC_MP
    #if defined(HAVE_ECC384)
    ret = ecc384_test_deterministic_k(&rng);
    if (ret != 0) {
        printf("ecc384_test_deterministic_k failed! %d\n", ret);
        goto done;
    }
    #endif
    #if defined(HAVE_ECC521)
    ret = ecc521_test_deterministic_k(&rng);
    if (ret != 0) {
        printf("ecc512_test_deterministic_k failed! %d\n", ret);
        goto done;
    }
    #endif
    #endif
#endif

#if defined(HAVE_ECC_SIGN) && defined(WOLFSSL_ECDSA_SET_K) && \
    !defined(WOLFSSL_KCAPI_ECC)
    ret = ecc_test_sign_vectors(&rng);
    if (ret != 0) {
        printf("ecc_test_sign_vectors failed! %d\n", ret);
        goto done;
    }
#endif
#ifdef HAVE_ECC_CDH
    ret = ecc_test_cdh_vectors(&rng);
    if (ret != 0) {
        printf("ecc_test_cdh_vectors failed! %d\n", ret);
        goto done;
    }
#endif
#if !defined(WOLFSSL_ATECC508A) && !defined(WOLFSSL_ATECC608A) && \
  !defined(WOLFSSL_STM32_PKA) && !defined(WOLFSSL_SILABS_SE_ACCEL) && \
  !defined(WOLF_CRYPTO_CB_ONLY_ECC)
    ret = ecc_test_make_pub(&rng);
    if (ret != 0) {
        printf("ecc_test_make_pub failed!: %d\n", ret);
        goto done;
    }
#elif defined(HAVE_ECC_KEY_IMPORT)
    (void)ecc_test_make_pub; /* for compiler warning */
#endif
#ifdef WOLFSSL_CERT_GEN
    ret = ecc_test_cert_gen(&rng);
    if (ret != 0) {
        printf("ecc_test_cert_gen failed!: %d\n", ret);
        goto done;
    }
#endif
#if !defined(HAVE_FIPS) && !defined(HAVE_SELFTEST) && !defined(WOLFSSL_NO_MALLOC) && \
    !defined(WOLF_CRYPTO_CB_ONLY_ECC)
    ret = ecc_test_allocator(&rng);
    if (ret != 0) {
        printf("ecc_test_allocator failed!: %d\n", ret);
        goto done;
    }
#endif

#if defined(WC_ECC_NONBLOCK) && defined(WOLFSSL_PUBLIC_MP) && \
    defined(HAVE_ECC_SIGN) && defined(HAVE_ECC_VERIFY)
    ret = ecc_test_nonblock(&rng);
    if (ret != 0) {
        printf("ecc_test_nonblock failed!: %d\n", ret);
        goto done;
    }
#endif

done:
    wc_FreeRng(&rng);

    return ret;
}

#if defined(HAVE_ECC_ENCRYPT) && defined(HAVE_AES_CBC) && \
    (defined(WOLFSSL_AES_128) || defined(WOLFSSL_AES_256))

/* ecc_encrypt_e2e_test() uses wc_ecc_ctx_set_algo(), which was added in
 * wolfFIPS 5.3.
 * ecc_encrypt_kat() is used only by ecc_encrypt_e2e_test().
 */
#if !defined(HAVE_FIPS) || (defined(FIPS_VERSION_GE) && FIPS_VERSION_GE(5,3))

#if (!defined(NO_ECC256)  || defined(HAVE_ALL_CURVES)) && \
    ECC_MIN_KEY_SZ <= 256 && defined(WOLFSSL_AES_128)
static int ecc_encrypt_kat(WC_RNG *rng)
{
    int ret = 0;
#ifdef WOLFSSL_ECIES_OLD
#ifdef WOLFSSL_SMALL_STACK
    ecc_key* userA = NULL;
#else
    ecc_key  userA[1];
#endif
    int      userAInit = 0;
#endif
#ifdef WOLFSSL_SMALL_STACK
    ecc_key* userB = NULL;
#else
    ecc_key  userB[1];
#endif
    int      userBInit = 0;
    ecc_key* tmpKey;
    byte     plain[48];
    word32   plainSz = sizeof(plain);

    WOLFSSL_SMALL_STACK_STATIC const byte privKey[] = {
        0x04, 0x80, 0xef, 0x1d, 0xbe, 0x02, 0x0c, 0x20,
        0x5b, 0xab, 0x80, 0x35, 0x5b, 0x2a, 0x0f, 0x6d,
        0xd3, 0xb0, 0x7f, 0x7e, 0x7f, 0x86, 0x8a, 0x49,
        0xee, 0xb4, 0xaa, 0x09, 0x2d, 0x1e, 0x1d, 0x02
    };
#if defined(WOLFSSL_ECIES_OLD) || defined(WOLFSSL_QNX_CAAM)
    WOLFSSL_SMALL_STACK_STATIC const byte pubKey[] = {
        0x04,
        /* X */
        0x50, 0xf2, 0x93, 0xa2, 0x48, 0xa9, 0xc0, 0x5a,
        0x9a, 0xa7, 0x70, 0x34, 0xb7, 0x7f, 0x4c, 0x3a,
        0xad, 0xfc, 0xd8, 0xb6, 0x76, 0x0a, 0xe3, 0xc1,
        0x87, 0x17, 0x07, 0x2d, 0x8d, 0xa3, 0x63, 0xa0,
        /* X */
        0xc1, 0x27, 0xb2, 0x97, 0x9b, 0x84, 0xe7, 0xcd,
        0x20, 0x65, 0x8d, 0x2b, 0x6a, 0x93, 0x75, 0xaa,
        0x8b, 0xe1, 0x3a, 0x7b, 0x24, 0x1a, 0xbe, 0xe8,
        0x36, 0xd2, 0xe6, 0x34, 0x8a, 0x7a, 0xb3, 0x28
    };
#endif
    WOLFSSL_SMALL_STACK_STATIC const byte enc_msg[] = {
#ifdef WOLFSSL_ECIES_OLD
        0x42, 0x70, 0xbf, 0xf9, 0xf4, 0x7e, 0x4b, 0x9b,
        0xb5, 0x4c, 0xcc, 0xc5, 0x94, 0xa7, 0xef, 0xaa,
        0xc3, 0x7c, 0x85, 0xa6, 0x51, 0x6e, 0xd3, 0xfa,
        0x56, 0xc9, 0x10, 0x4d, 0x14, 0x32, 0x61, 0xb8,
        0xbb, 0x66, 0x7a, 0xb5, 0xbc, 0x95, 0xf8, 0xca,
        0xd1, 0x2a, 0x19, 0x51, 0x44, 0xd8, 0x0e, 0x57,
        0x34, 0xed, 0x45, 0x89, 0x2e, 0x57, 0xbe, 0xd5,
        0x06, 0x22, 0xd7, 0x13, 0x0a, 0x0e, 0x40, 0x36,
        0x0d, 0x05, 0x0d, 0xb6, 0xae, 0x61, 0x37, 0x18,
        0x83, 0x90, 0x0a, 0x27, 0x95, 0x41, 0x8c, 0x45
#elif defined(WOLFSSL_ECIES_ISO18033)
        0x04, 0x50, 0xf2, 0x93, 0xa2, 0x48, 0xa9, 0xc0,
        0x5a, 0x9a, 0xa7, 0x70, 0x34, 0xb7, 0x7f, 0x4c,
        0x3a, 0xad, 0xfc, 0xd8, 0xb6, 0x76, 0x0a, 0xe3,
        0xc1, 0x87, 0x17, 0x07, 0x2d, 0x8d, 0xa3, 0x63,
        0xa0, 0xc1, 0x27, 0xb2, 0x97, 0x9b, 0x84, 0xe7,
        0xcd, 0x20, 0x65, 0x8d, 0x2b, 0x6a, 0x93, 0x75,
        0xaa, 0x8b, 0xe1, 0x3a, 0x7b, 0x24, 0x1a, 0xbe,
        0xe8, 0x36, 0xd2, 0xe6, 0x34, 0x8a, 0x7a, 0xb3,
        0x28, 0xbb, 0x9f, 0xa8, 0x2d, 0xe1, 0xf1, 0x67,
        0x45, 0x02, 0x19, 0xdc, 0xc8, 0x24, 0x8b, 0x20,
        0x02, 0xa0, 0x8f, 0x95, 0x12, 0x55, 0x51, 0xf8,
        0x03, 0xc4, 0x54, 0x13, 0x98, 0x2d, 0xf0, 0x31,
        0x51, 0x80, 0x45, 0x24, 0xcb, 0x8b, 0x48, 0xa6,
        0x8b, 0x8e, 0x97, 0x9c, 0x56, 0x4d, 0x70, 0x00,
        0x53, 0xd3, 0x47, 0x00, 0x5a, 0x23, 0x8c, 0xf9,
        0xfd, 0xd2, 0x33, 0x2c, 0x43, 0x6e, 0x9e, 0xb2,
        0xf4, 0x95, 0xd4, 0xcf, 0x30, 0xd6, 0xa2, 0xc5,
        0x35, 0x96, 0x6a, 0xd4, 0x36, 0x15, 0xa9, 0xbd,
        0x7f
#elif defined(WOLFSSL_ECIES_GEN_IV)
        /* EC P-256 point */
        0x04,
        /*  X */
        0x50, 0xf2, 0x93, 0xa2, 0x48, 0xa9, 0xc0, 0x5a,
        0x9a, 0xa7, 0x70, 0x34, 0xb7, 0x7f, 0x4c, 0x3a,
        0xad, 0xfc, 0xd8, 0xb6, 0x76, 0x0a, 0xe3, 0xc1,
        0x87, 0x17, 0x07, 0x2d, 0x8d, 0xa3, 0x63, 0xa0,
        /*  Y */
        0xc1, 0x27, 0xb2, 0x97, 0x9b, 0x84, 0xe7, 0xcd,
        0x20, 0x65, 0x8d, 0x2b, 0x6a, 0x93, 0x75, 0xaa,
        0x8b, 0xe1, 0x3a, 0x7b, 0x24, 0x1a, 0xbe, 0xe8,
        0x36, 0xd2, 0xe6, 0x34, 0x8a, 0x7a, 0xb3, 0x28,
        /* IV */
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        /* Encrypted Msg */
        0xe5, 0x17, 0xaf, 0x0d, 0x65, 0x4d, 0x3d, 0x50,
        0x96, 0x05, 0xc9, 0x63, 0x2c, 0xef, 0x1c, 0x1f,
        0x78, 0xc9, 0x90, 0x7a, 0x14, 0x00, 0xfc, 0x44,
        0x71, 0x6d, 0x57, 0x8c, 0xdf, 0x23, 0xca, 0x65,
        0xcf, 0x93, 0x06, 0xb6, 0x9a, 0xf4, 0x61, 0xbd,
        0x44, 0x1a, 0xeb, 0x52, 0x68, 0x0f, 0xd1, 0xde,
        /* HMAC */
        0x5a, 0x22, 0xc1, 0x5d, 0x99, 0x66, 0x3f, 0x24,
        0x35, 0x96, 0xac, 0xf7, 0xf6, 0x28, 0x45, 0x16,
        0x52, 0x19, 0x0d, 0xe4, 0xb2, 0xca, 0x5b, 0x28,
        0x4e, 0xbb, 0xf3, 0x98, 0x57, 0xd7, 0x3b, 0xe2
#else
        0x04, 0x50, 0xf2, 0x93, 0xa2, 0x48, 0xa9, 0xc0,
        0x5a, 0x9a, 0xa7, 0x70, 0x34, 0xb7, 0x7f, 0x4c,
        0x3a, 0xad, 0xfc, 0xd8, 0xb6, 0x76, 0x0a, 0xe3,
        0xc1, 0x87, 0x17, 0x07, 0x2d, 0x8d, 0xa3, 0x63,
        0xa0, 0xc1, 0x27, 0xb2, 0x97, 0x9b, 0x84, 0xe7,
        0xcd, 0x20, 0x65, 0x8d, 0x2b, 0x6a, 0x93, 0x75,
        0xaa, 0x8b, 0xe1, 0x3a, 0x7b, 0x24, 0x1a, 0xbe,
        0xe8, 0x36, 0xd2, 0xe6, 0x34, 0x8a, 0x7a, 0xb3,
        0x28, 0xe5, 0x17, 0xaf, 0x0d, 0x65, 0x4d, 0x3d,
        0x50, 0x96, 0x05, 0xc9, 0x63, 0x2c, 0xef, 0x1c,
        0x1f, 0x78, 0xc9, 0x90, 0x7a, 0x14, 0x00, 0xfc,
        0x44, 0x71, 0x6d, 0x57, 0x8c, 0xdf, 0x23, 0xca,
        0x65, 0xcf, 0x93, 0x06, 0xb6, 0x9a, 0xf4, 0x61,
        0xbd, 0x44, 0x1a, 0xeb, 0x52, 0x68, 0x0f, 0xd1,
        0xde, 0xc7, 0x3f, 0x6f, 0xce, 0xbe, 0x49, 0x61,
        0x48, 0x01, 0x77, 0x41, 0xd0, 0xd8, 0x5b, 0x48,
        0xca, 0x4e, 0x47, 0x3e, 0x47, 0xbf, 0x1d, 0x28,
        0x4c, 0x18, 0x1a, 0xfb, 0x96, 0x95, 0xda, 0xde,
        0x55
#endif
    };
    WOLFSSL_SMALL_STACK_STATIC const byte msg[] = {
        0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
        0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
        0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
        0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f,
        0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27,
        0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f
    };

#ifdef WOLFSSL_SMALL_STACK
    userB = (ecc_key *)XMALLOC(sizeof(*userB), HEAP_HINT,
                                                       DYNAMIC_TYPE_TMP_BUFFER);
    if (userB == NULL) {
        ret = -10451;
    }
#ifdef WOLFSSL_ECIES_OLD
    if (ret == 0) {
        userA = (ecc_key *)XMALLOC(sizeof(*userA), HEAP_HINT,
                                                       DYNAMIC_TYPE_TMP_BUFFER);
        if (userA == NULL) {
            ret = -10450;
        }
    }
#endif
#endif

    if (ret == 0) {
        ret = wc_ecc_init_ex(userB, HEAP_HINT, devId);
        if (ret != 0)
            ret = -10453;
    }
    if (ret == 0) {
        userBInit = 1;
#ifdef WOLFSSL_ECIES_OLD
        ret = wc_ecc_init_ex(userA, HEAP_HINT, devId);
        if (ret != 0)
            ret = -10452;
    }
    if (ret == 0) {
        userAInit = 1;
        tmpKey = userA;
#else
        tmpKey = NULL;
#endif
    }


    if (ret == 0) {
#ifdef WOLFSSL_QNX_CAAM
        ret = wc_ecc_import_private_key_ex(privKey, sizeof(privKey), pubKey,
            sizeof(pubKey), userB, ECC_SECP256R1);
#else
        ret = wc_ecc_import_private_key_ex(privKey, sizeof(privKey), NULL, 0,
                                                          userB, ECC_SECP256R1);
#endif
        if (ret != 0)
            ret = -10454;
    }

#ifdef WOLFSSL_ECIES_OLD
    if (ret == 0) {
        ret = wc_ecc_import_x963_ex(pubKey, sizeof(pubKey), userA,
                                                                 ECC_SECP256R1);
        if (ret != 0)
            ret = -10455;
    }
#endif

#if defined(ECC_TIMING_RESISTANT) && (!defined(HAVE_FIPS) || \
    (!defined(HAVE_FIPS_VERSION) || (HAVE_FIPS_VERSION != 2))) && \
    !defined(HAVE_SELFTEST)
    if (ret == 0) {
        ret = wc_ecc_set_rng(userB, rng);
        if (ret != 0) {
            ret = -10456;
        }
    }
#else
    (void)rng;
#endif

    if (ret == 0) {
        ret = wc_ecc_decrypt(userB, tmpKey, enc_msg, sizeof(enc_msg), plain,
                                                                &plainSz, NULL);
        if (ret != 0)
            ret = -10457;
    }
    if (ret == 0) {
        if (XMEMCMP(plain, msg, sizeof(msg)) != 0) {
            ret = -10458;
        }
    }

    if (userBInit)
        wc_ecc_free(userB);
#ifdef WOLFSSL_ECIES_OLD
    if (userAInit)
        wc_ecc_free(userA);
#endif
#ifdef WOLFSSL_SMALL_STACK
    if (userB != NULL) {
        XFREE(userB, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    }
#ifdef WOLFSSL_ECIES_OLD
    if (userA != NULL) {
        XFREE(userA, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    }
#endif
#endif

    return ret;
}
#endif

static int ecc_encrypt_e2e_test(WC_RNG* rng, ecc_key* userA, ecc_key* userB,
    byte encAlgo, byte kdfAlgo, byte macAlgo)
{
    int     ret = 0;
    byte    msg[48];
    byte    plain[48];
#ifdef WOLFSSL_ECIES_OLD
    byte    out[80];
#elif defined(WOLFSSL_ECIES_GEN_IV)
    byte    out[1 + ECC_KEYGEN_SIZE * 2 + 16 + 80];
#else
    byte    out[1 + ECC_KEYGEN_SIZE * 2 + 80];
#endif
    word32  outSz   = sizeof(out);
    word32  plainSz = sizeof(plain);
    int     i;
    ecEncCtx* cliCtx = NULL;
    ecEncCtx* srvCtx = NULL;
    byte cliSalt[EXCHANGE_SALT_SZ];
    byte srvSalt[EXCHANGE_SALT_SZ];
    const byte* tmpSalt;
    byte    msg2[48];
    byte    plain2[48];
#ifdef WOLFSSL_ECIES_OLD
    byte    out2[80];
#elif defined(WOLFSSL_ECIES_GEN_IV)
    byte    out2[1 + ECC_KEYGEN_SIZE * 2 + 16 + 80];
#else
    byte    out2[1 + ECC_KEYGEN_SIZE * 2 + 80];
#endif
    word32  outSz2   = sizeof(out2);
    word32  plainSz2 = sizeof(plain2);
#ifdef WOLFSSL_SMALL_STACK
    ecc_key *tmpKey = (ecc_key *)XMALLOC(sizeof(ecc_key), HEAP_HINT,
                                         DYNAMIC_TYPE_TMP_BUFFER);
#else
    ecc_key tmpKey[1];
#endif

#ifdef WOLFSSL_SMALL_STACK
    if (tmpKey == NULL) {
        ERROR_OUT(MEMORY_E, done);
    }
#endif
    ret = wc_ecc_init_ex(tmpKey, HEAP_HINT, devId);
    if (ret != 0)
        goto done;

    /* set message to incrementing 0,1,2,etc... */
    for (i = 0; i < (int)sizeof(msg); i++)
        msg[i] = i;

    /* encrypt msg to B */
    ret = wc_ecc_encrypt(userA, userB, msg, sizeof(msg), out, &outSz, NULL);
    if (ret != 0) {
        ret = -10405; goto done;
    }

#ifdef WOLFSSL_ECIES_OLD
    tmpKey->dp = userA->dp;
    ret = wc_ecc_copy_point(&userA->pubkey, &tmpKey->pubkey);
    if (ret != 0) {
        ret = -10413; goto done;
    }
#endif

    /* decrypt msg from A */
    ret = wc_ecc_decrypt(userB, tmpKey, out, outSz, plain, &plainSz, NULL);
    if (ret != 0) {
        ret = -10406; goto done;
    }

    if (XMEMCMP(plain, msg, sizeof(msg)) != 0) {
        ret = -10407; goto done;
    }

#ifndef WOLFSSL_ECIES_OLD
    /* A decrypts msg (response) from B */
    ret = wc_ecc_decrypt(userB, NULL, out, outSz, plain2, &plainSz2, NULL);
    if (ret != 0)
        goto done;

    if (XMEMCMP(plain, msg, sizeof(msg)) != 0) {
        ret = -10415; goto done;
    }
#endif

    /* let's verify message exchange works, A is client, B is server */
    cliCtx = wc_ecc_ctx_new(REQ_RESP_CLIENT, rng);
    srvCtx = wc_ecc_ctx_new(REQ_RESP_SERVER, rng);
    if (cliCtx == NULL || srvCtx == NULL) {
        ret = -10408; goto done;
    }

    ret = wc_ecc_ctx_set_algo(cliCtx, encAlgo, kdfAlgo, macAlgo);
    if (ret != 0)
        goto done;
    ret = wc_ecc_ctx_set_algo(srvCtx, encAlgo, kdfAlgo, macAlgo);
    if (ret != 0)
        goto done;

    /* get salt to send to peer */
    tmpSalt = wc_ecc_ctx_get_own_salt(cliCtx);
    if (tmpSalt == NULL) {
        ret = -10409; goto done;
    }
    XMEMCPY(cliSalt, tmpSalt, EXCHANGE_SALT_SZ);

    tmpSalt = wc_ecc_ctx_get_own_salt(srvCtx);
    if (tmpSalt == NULL) {
        ret = -10410; goto done;
    }
    XMEMCPY(srvSalt, tmpSalt, EXCHANGE_SALT_SZ);

    /* in actual use, we'd get the peer's salt over the transport */
    ret = wc_ecc_ctx_set_peer_salt(cliCtx, srvSalt);
    if (ret != 0)
        goto done;
    ret = wc_ecc_ctx_set_peer_salt(srvCtx, cliSalt);
    if (ret != 0)
        goto done;

    ret = wc_ecc_ctx_set_info(cliCtx, (byte*)"wolfSSL MSGE", 11);
    if (ret != 0)
        goto done;
    ret = wc_ecc_ctx_set_info(srvCtx, (byte*)"wolfSSL MSGE", 11);
    if (ret != 0)
        goto done;

    /* get encrypted msg (request) to send to B */
    outSz = sizeof(out);
    ret = wc_ecc_encrypt(userA, userB, msg, sizeof(msg), out, &outSz,cliCtx);
    if (ret != 0)
        goto done;

#ifndef WOLFSSL_ECIES_OLD
    wc_ecc_free(tmpKey);
#endif
    /* B decrypts msg (request) from A */
    plainSz = sizeof(plain);
    ret = wc_ecc_decrypt(userB, tmpKey, out, outSz, plain, &plainSz, srvCtx);
    if (ret != 0)
        goto done;

    if (XMEMCMP(plain, msg, sizeof(msg)) != 0) {
        ret = -10411; goto done;
    }

    /* msg2 (response) from B to A */
    for (i = 0; i < (int)sizeof(msg2); i++)
        msg2[i] = i + sizeof(msg2);

    /* get encrypted msg (response) to send to B */
    ret = wc_ecc_encrypt(userB, userA, msg2, sizeof(msg2), out2,
                      &outSz2, srvCtx);
    if (ret != 0)
        goto done;

#ifdef WOLFSSL_ECIES_OLD
    tmpKey->dp = userB->dp;
    ret = wc_ecc_copy_point(&userB->pubkey, &tmpKey->pubkey);
    if (ret != 0) {
        ret = -10414; goto done;
    }
#else
    wc_ecc_free(tmpKey);
#endif

    /* A decrypts msg (response) from B */
    ret = wc_ecc_decrypt(userA, tmpKey, out2, outSz2, plain2, &plainSz2,
                     cliCtx);
    if (ret != 0)
        goto done;

    if (XMEMCMP(plain2, msg2, sizeof(msg2)) != 0) {
        ret = -10412; goto done;
    }

#if defined(HAVE_COMP_KEY) && \
    (! defined(HAVE_FIPS) || (defined(FIPS_VERSION_GE) && FIPS_VERSION_GE(5,3)))
    /* Create new client and server contexts. */
    wc_ecc_ctx_free(srvCtx);
    wc_ecc_ctx_free(cliCtx);
    /* let's verify message exchange works, A is client, B is server */
    cliCtx = wc_ecc_ctx_new(REQ_RESP_CLIENT, rng);
    srvCtx = wc_ecc_ctx_new(REQ_RESP_SERVER, rng);
    if (cliCtx == NULL || srvCtx == NULL) {
        ret = -10416; goto done;
    }

    ret = wc_ecc_ctx_set_algo(cliCtx, encAlgo, kdfAlgo, macAlgo);
    if (ret != 0)
        goto done;
    ret = wc_ecc_ctx_set_algo(srvCtx, encAlgo, kdfAlgo, macAlgo);
    if (ret != 0)
        goto done;

    /* get salt to send to peer */
    tmpSalt = wc_ecc_ctx_get_own_salt(cliCtx);
    if (tmpSalt == NULL) {
        ret = -10417; goto done;
    }
    XMEMCPY(cliSalt, tmpSalt, EXCHANGE_SALT_SZ);

    tmpSalt = wc_ecc_ctx_get_own_salt(srvCtx);
    if (tmpSalt == NULL) {
        ret = -10418; goto done;
    }
    XMEMCPY(srvSalt, tmpSalt, EXCHANGE_SALT_SZ);

    /* in actual use, we'd get the peer's salt over the transport */
    ret = wc_ecc_ctx_set_peer_salt(cliCtx, srvSalt);
    if (ret != 0)
        goto done;
    ret = wc_ecc_ctx_set_peer_salt(srvCtx, cliSalt);
    if (ret != 0)
        goto done;

    ret = wc_ecc_ctx_set_info(cliCtx, (byte*)"wolfSSL MSGE", 12);
    if (ret != 0)
        goto done;
    ret = wc_ecc_ctx_set_info(srvCtx, (byte*)"wolfSSL MSGE", 12);
    if (ret != 0)
        goto done;

    /* get encrypted msg (request) to send to B - compressed public key */
    outSz = sizeof(out);
    ret = wc_ecc_encrypt_ex(userA, userB, msg, sizeof(msg), out, &outSz, cliCtx,
        1);
    if (ret != 0)
        goto done;

#ifndef WOLFSSL_ECIES_OLD
    wc_ecc_free(tmpKey);
#endif
    /* B decrypts msg (request) from A - out has a compressed public key */
    plainSz = sizeof(plain);
    ret = wc_ecc_decrypt(userB, tmpKey, out, outSz, plain, &plainSz, srvCtx);
    if (ret != 0)
        goto done;

    if (XMEMCMP(plain, msg, sizeof(msg)) != 0) {
        ret = -10419; goto done;
    }
#endif /* HAVE_COMP_KEY && (!FIPS || FIPS>=5.3) */

#if (!defined(NO_ECC256)  || defined(HAVE_ALL_CURVES)) && \
    (ECC_MIN_KEY_SZ <= 256) && defined(WOLFSSL_AES_128)
    ret = ecc_encrypt_kat(rng);
#endif

done:

    /* cleanup */
    wc_ecc_ctx_free(srvCtx);
    wc_ecc_ctx_free(cliCtx);

#ifdef WOLFSSL_SMALL_STACK
    if (tmpKey != NULL) {
        wc_ecc_free(tmpKey);
        XFREE(tmpKey, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    }
#else
    wc_ecc_free(tmpKey);
#endif

    return ret;
}

#endif /* !HAVE_FIPS || FIPS_VERSION_GE(5,3) */

WOLFSSL_TEST_SUBROUTINE int ecc_encrypt_test(void)
{
    WC_RNG  rng;
    int     ret;
#ifdef WOLFSSL_SMALL_STACK
    ecc_key *userA;
    ecc_key *userB;
#else
    ecc_key userA[1];
    ecc_key userB[1];
#endif

#ifndef HAVE_FIPS
    ret = wc_InitRng_ex(&rng, HEAP_HINT, devId);
#else
    ret = wc_InitRng(&rng);
#endif
    if (ret != 0)
        return -10400;

#ifdef WOLFSSL_SMALL_STACK
    userA = (ecc_key *)XMALLOC(sizeof *userA, HEAP_HINT,
                               DYNAMIC_TYPE_TMP_BUFFER);
    userB = (ecc_key *)XMALLOC(sizeof *userB, HEAP_HINT,
                               DYNAMIC_TYPE_TMP_BUFFER);
    if ((userA == NULL) || (userB == NULL)) {
        ERROR_OUT(MEMORY_E, done);
    }
#endif

    XMEMSET(userA, 0, sizeof *userA);
    XMEMSET(userB, 0, sizeof *userB);

    ret = wc_ecc_init_ex(userA, HEAP_HINT, devId);
    if (ret != 0)
        goto done;
    ret = wc_ecc_init_ex(userB, HEAP_HINT, devId);
    if (ret != 0)
        goto done;

    ret  = wc_ecc_make_key(&rng, ECC_KEYGEN_SIZE, userA);
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &userA->asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    if (ret != 0){
        ret = -10401; goto done;
    }

    ret = wc_ecc_make_key(&rng, ECC_KEYGEN_SIZE, userB);
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &userB->asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    if (ret != 0){
        ret = -10402; goto done;
    }

#if defined(ECC_TIMING_RESISTANT) && (!defined(HAVE_FIPS) || \
    (!defined(HAVE_FIPS_VERSION) || (HAVE_FIPS_VERSION != 2))) && \
    !defined(HAVE_SELFTEST)
    ret = wc_ecc_set_rng(userA, &rng);
    if (ret != 0) {
        ret = -10403; goto done;
    }
    ret = wc_ecc_set_rng(userB, &rng);
    if (ret != 0) {
        ret = -10404; goto done;
    }
#endif

#if !defined(HAVE_FIPS) || (defined(FIPS_VERSION_GE) && FIPS_VERSION_GE(5,3))

#if !defined(NO_AES) && defined(HAVE_AES_CBC)
#ifdef WOLFSSL_AES_128
    if (ret == 0) {
        ret = ecc_encrypt_e2e_test(&rng, userA, userB, ecAES_128_CBC,
            ecHKDF_SHA256, ecHMAC_SHA256);
        if (ret != 0) {
            printf("ECIES: AES_128_CBC, HKDF_SHA256, HMAC_SHA256\n");
        }
    }
#endif
#ifdef WOLFSSL_AES_256
    if (ret == 0) {
        ret = ecc_encrypt_e2e_test(&rng, userA, userB, ecAES_256_CBC,
            ecHKDF_SHA256, ecHMAC_SHA256);
        if (ret != 0) {
            printf("ECIES: AES_256_CBC, HKDF_SHA256, HMAC_SHA256\n");
        }
    }
#endif
#endif
#if !defined(NO_AES) && defined(WOLFSSL_AES_COUNTER)
#ifdef WOLFSSL_AES_128
    if (ret == 0) {
        ret = ecc_encrypt_e2e_test(&rng, userA, userB, ecAES_128_CTR,
            ecHKDF_SHA256, ecHMAC_SHA256);
        if (ret != 0) {
            printf("ECIES: AES_128_CTR, HKDF_SHA256, HMAC_SHA256\n");
        }
    }
#endif
#ifdef WOLFSSL_AES_256
    if (ret == 0) {
        ret = ecc_encrypt_e2e_test(&rng, userA, userB, ecAES_256_CTR,
            ecHKDF_SHA256, ecHMAC_SHA256);
        if (ret != 0) {
            printf("ECIES: AES_256_CTR, HKDF_SHA256, HMAC_SHA256\n");
        }
    }
#endif
#endif

#endif /* !HAVE_FIPS || FIPS_VERSION_GE(5,3) */

done:
#ifdef WOLFSSL_SMALL_STACK
    if (userA != NULL) {
        wc_ecc_free(userA);
        XFREE(userA, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    }
    if (userB != NULL) {
        wc_ecc_free(userB);
        XFREE(userB, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    }
#else
    wc_ecc_free(userB);
    wc_ecc_free(userA);
#endif

    wc_FreeRng(&rng);

    return ret;
}

#endif /* HAVE_ECC_ENCRYPT && HAVE_AES_CBC && WOLFSSL_AES_128 */

#if defined(USE_CERT_BUFFERS_256) && !defined(WOLFSSL_ATECC508A) && \
    !defined(WOLFSSL_ATECC608A) && !defined(NO_ECC256) && \
    defined(HAVE_ECC_VERIFY) && defined(HAVE_ECC_SIGN) && \
    !defined(WOLF_CRYPTO_CB_ONLY_ECC)
WOLFSSL_TEST_SUBROUTINE int ecc_test_buffers(void)
{
    size_t bytes;
#ifdef WOLFSSL_SMALL_STACK
    ecc_key *cliKey = (ecc_key *)XMALLOC(sizeof *cliKey, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    ecc_key *servKey = (ecc_key *)XMALLOC(sizeof *servKey, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    ecc_key *tmpKey = (ecc_key *)XMALLOC(sizeof *tmpKey, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
#else
    ecc_key cliKey[1];
    ecc_key servKey[1];
    ecc_key tmpKey[1];
#endif
    WC_RNG rng;
    word32 idx = 0;
    int    ret;
    /* pad our test message to 32 bytes so evenly divisible by AES_BLOCK_SZ */
    byte   in[] = "Everyone gets Friday off. ecc p";
    word32 inLen = (word32)XSTRLEN((char*)in);
    byte   out[256];
    byte   plain[256];
    int verify = 0;
    word32 x;

#ifdef WOLFSSL_SMALL_STACK
    if ((cliKey == NULL) || (servKey == NULL) || (tmpKey == NULL))
        ERROR_OUT(MEMORY_E, done);
#endif

    ret = wc_ecc_init_ex(cliKey, HEAP_HINT, devId);
    if (ret != 0)
        ERROR_OUT(-10420, done);
    ret = wc_ecc_init_ex(servKey, HEAP_HINT, devId);
    if (ret != 0)
        ERROR_OUT(-10421, done);
    ret = wc_ecc_init_ex(tmpKey, HEAP_HINT, devId);
    if (ret != 0)
        ERROR_OUT(-10421, done);

    bytes = (size_t)sizeof_ecc_clikey_der_256;
    /* place client key into ecc_key struct cliKey */
    ret = wc_EccPrivateKeyDecode(ecc_clikey_der_256, &idx, cliKey,
                                                                (word32)bytes);
    if (ret != 0)
        ERROR_OUT(-10422, done);

    idx = 0;
    bytes = (size_t)sizeof_ecc_key_der_256;

    /* place server key into ecc_key struct servKey */
    ret = wc_EccPrivateKeyDecode(ecc_key_der_256, &idx, servKey,
                                                                (word32)bytes);
    if (ret != 0)
        ERROR_OUT(-10423, done);

#ifndef WC_NO_RNG
#ifndef HAVE_FIPS
    ret = wc_InitRng_ex(&rng, HEAP_HINT, devId);
#else
    ret = wc_InitRng(&rng);
#endif
    if (ret != 0)
        ERROR_OUT(-10424, done);

#if defined(ECC_TIMING_RESISTANT) && (!defined(HAVE_FIPS) || \
    (!defined(HAVE_FIPS_VERSION) || (HAVE_FIPS_VERSION != 2))) && \
    !defined(HAVE_SELFTEST)
    ret = wc_ecc_set_rng(cliKey, &rng);
    if (ret != 0) {
        ERROR_OUT(-10425, done);
    }
    ret = wc_ecc_set_rng(servKey, &rng);
    if (ret != 0) {
        ERROR_OUT(-10425, done);
    }
#endif
#endif /* !WC_NO_RNG */

#if defined(HAVE_ECC_ENCRYPT) && defined(HAVE_HKDF) && \
    defined(HAVE_AES_CBC) && defined(WOLFSSL_AES_128)
    {
        word32 y;
        /* test encrypt and decrypt if they're available */
        x = sizeof(out);
        ret = wc_ecc_encrypt(cliKey, servKey, in, sizeof(in), out, &x, NULL);
        if (ret < 0)
            ERROR_OUT(-10426, done);

    #ifdef WOLFSSL_ECIES_OLD
        tmpKey->dp = cliKey->dp;
        ret = wc_ecc_copy_point(&cliKey->pubkey, &tmpKey->pubkey);
        if (ret != 0) {
            ret = -10414; goto done;
        }
    #endif

        y = sizeof(plain);
        ret = wc_ecc_decrypt(servKey, tmpKey, out, x, plain, &y, NULL);
        if (ret < 0)
            ERROR_OUT(-10427, done);

        if (XMEMCMP(plain, in, inLen))
            ERROR_OUT(-10428, done);
    }
#endif

    x = sizeof(out);
    do {
    #if defined(WOLFSSL_ASYNC_CRYPT)
        ret = wc_AsyncWait(ret, &cliKey->asyncDev, WC_ASYNC_FLAG_CALL_AGAIN);
    #endif
        if (ret == 0)
            ret = wc_ecc_sign_hash(in, inLen, out, &x, &rng, cliKey);
    } while (ret == WC_PENDING_E);
    if (ret < 0)
        ERROR_OUT(-10429, done);
    TEST_SLEEP();

    XMEMSET(plain, 0, sizeof(plain));

    do {
    #if defined(WOLFSSL_ASYNC_CRYPT)
        ret = wc_AsyncWait(ret, &cliKey->asyncDev, WC_ASYNC_FLAG_CALL_AGAIN);
    #endif
        if (ret == 0)
            ret = wc_ecc_verify_hash(out, x, in, inLen, &verify,
                cliKey);
    } while (ret == WC_PENDING_E);
    if (ret < 0)
        ERROR_OUT(-10430, done);

    if (verify != 1)
        ERROR_OUT(-10431, done);
    TEST_SLEEP();

#ifdef WOLFSSL_CERT_EXT
    idx = 0;

    bytes = sizeof_ecc_clikeypub_der_256;

    ret = wc_EccPublicKeyDecode(ecc_clikeypub_der_256, &idx, cliKey,
                                                               (word32) bytes);
    if (ret != 0)
        ERROR_OUT(-10432, done);
#endif

    ret = 0;

  done:

#ifdef WOLFSSL_SMALL_STACK
    if (cliKey != NULL) {
        wc_ecc_free(cliKey);
        XFREE(cliKey, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    }
    if (servKey != NULL) {
        wc_ecc_free(servKey);
        XFREE(servKey, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    }
    if (tmpKey != NULL) {
        wc_ecc_free(tmpKey);
        XFREE(tmpKey, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    }
#else
    wc_ecc_free(cliKey);
    wc_ecc_free(servKey);
    wc_ecc_free(tmpKey);
#endif

    wc_FreeRng(&rng);

    return ret;
}
#endif /* USE_CERT_BUFFERS_256 && !WOLFSSL_ATECCX08A && !NO_ECC256 */
#endif /* HAVE_ECC */


#ifdef HAVE_CURVE25519
#if defined(HAVE_CURVE25519_SHARED_SECRET) && \
                                             defined(HAVE_CURVE25519_KEY_IMPORT)
#ifdef CURVE25519_OVERFLOW_ALL_TESTS
#define X25519_TEST_CNT    5
#else
#define X25519_TEST_CNT    1
#endif
static int curve25519_overflow_test(void)
{
    /* secret key for party a */
    byte sa[X25519_TEST_CNT][32] = {
        {
            0x8d,0xaf,0x6e,0x7a,0xc1,0xeb,0x8d,0x30,
            0x99,0x86,0xd3,0x90,0x47,0x96,0x21,0x3c,
            0x3a,0x75,0xc0,0x7b,0x75,0x01,0x75,0xa3,
            0x81,0x4b,0xff,0x5a,0xbc,0x96,0x87,0x28
        },
#ifdef CURVE25519_OVERFLOW_ALL_TESTS
        {
            0x9d,0x63,0x5f,0xce,0xe2,0xe8,0xd7,0xfb,
            0x68,0x77,0x0e,0x44,0xd1,0xad,0x87,0x2b,
            0xf4,0x65,0x06,0xb7,0xbb,0xdb,0xbe,0x6e,
            0x02,0x43,0x24,0xc7,0x3d,0x7b,0x88,0x60
        },
        {
            0x63,0xbf,0x76,0xa9,0x73,0xa0,0x09,0xb9,
            0xcc,0xc9,0x4d,0x47,0x2d,0x14,0x0e,0x52,
            0xa3,0x84,0x55,0xb8,0x7c,0xdb,0xce,0xb1,
            0xe4,0x5b,0x8a,0xb9,0x30,0xf1,0xa4,0xa0
        },
        {
            0x63,0xbf,0x76,0xa9,0x73,0xa0,0x09,0xb9,
            0xcc,0xc9,0x4d,0x47,0x2d,0x14,0x0e,0x52,
            0xa3,0x84,0x55,0xb8,0x7c,0xdb,0xce,0xb1,
            0xe4,0x5b,0x8a,0xb9,0x30,0xf1,0xa4,0xa0
        },
        {
            0x63,0xbf,0x76,0xa9,0x73,0xa0,0x09,0xb9,
            0xcc,0xc9,0x4d,0x47,0x2d,0x14,0x0e,0x52,
            0xa3,0x84,0x55,0xb8,0x7c,0xdb,0xce,0xb1,
            0xe4,0x5b,0x8a,0xb9,0x30,0xf1,0xa4,0xa0
        }
#endif
    };

    /* public key for party b */
    byte pb[X25519_TEST_CNT][32] = {
        {
            0x7f,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
            0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
            0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
            0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xf0
        },
#ifdef CURVE25519_OVERFLOW_ALL_TESTS
        {
            /* 0xff first byte in original - invalid! */
            0x7f,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
            0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
            0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
            0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xf0
        },
        {
            0x36,0x1a,0x74,0x87,0x28,0x59,0xe0,0xb6,
            0xe4,0x2b,0x17,0x9b,0x16,0xb0,0x3b,0xf8,
            0xb8,0x9f,0x2a,0x8f,0xc5,0x33,0x68,0x4f,
            0xde,0x4d,0xd8,0x80,0x63,0xe7,0xb4,0x0a
        },
        {
            0x00,0x80,0x38,0x59,0x19,0x3a,0x66,0x12,
            0xfd,0xa1,0xec,0x1c,0x40,0x84,0x40,0xbd,
            0x64,0x10,0x8b,0x53,0x81,0x21,0x03,0x2d,
            0x7d,0x33,0xb4,0x01,0x57,0x0d,0xe1,0x89
        },
        {
            0x1d,0xf8,0xf8,0x33,0x89,0x6c,0xb7,0xba,
            0x94,0x73,0xfa,0xc2,0x36,0xac,0xbe,0x49,
            0xaf,0x85,0x3e,0x93,0x5f,0xae,0xb2,0xc0,
            0xc8,0x80,0x8f,0x4a,0xaa,0xd3,0x55,0x2b
        }
#endif
    };

    /* expected shared key */
    byte ss[X25519_TEST_CNT][32] = {
        {
            0x5c,0x4c,0x85,0x5f,0xfb,0x20,0x38,0xcc,
            0x55,0x16,0x5b,0x8a,0xa7,0xed,0x57,0x6e,
            0x35,0xaa,0x71,0x67,0x85,0x1f,0xb6,0x28,
            0x17,0x07,0x7b,0xda,0x76,0xdd,0xe0,0xb4
        },
#ifdef CURVE25519_OVERFLOW_ALL_TESTS
        {
            0x33,0xf6,0xc1,0x34,0x62,0x92,0x06,0x02,
            0x95,0xdb,0x91,0x4c,0x5d,0x52,0x54,0xc7,
            0xd2,0x5b,0x24,0xb5,0x4f,0x33,0x59,0x79,
            0x9f,0x6d,0x7e,0x4a,0x4c,0x30,0xd6,0x38
        },
        {
            0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
            0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
            0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
            0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x02
        },
        {
            0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
            0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
            0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
            0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x09
        },
        {
            0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
            0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
            0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
            0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x10
        }
#endif
    };

    int ret = 0;
    int            i;
    word32         y;
    byte           shared[32];
    curve25519_key userA;

    wc_curve25519_init_ex(&userA, HEAP_HINT, devId);

    for (i = 0; i < X25519_TEST_CNT; i++) {
        if (wc_curve25519_import_private_raw(sa[i], sizeof(sa[i]), pb[i],
                                                  sizeof(pb[i]), &userA) != 0) {
            ret = -10500 - i; break;
        }

        /* test against known test vector */
        XMEMSET(shared, 0, sizeof(shared));
        y = sizeof(shared);
        if (wc_curve25519_shared_secret(&userA, &userA, shared, &y) != 0) {
            ret = -10510 - i; break;
        }

        if (XMEMCMP(ss[i], shared, y)) {
            ret = -10520 - i; break;
        }
    }

    wc_curve25519_free(&userA);

    return ret;
}

/* Test the wc_curve25519_check_public API.
 *
 * returns 0 on success and -ve on failure.
 */
static int curve25519_check_public_test(void)
{
    /* Little-endian values that will fail */
    byte fail_le[][CURVE25519_KEYSIZE] = {
        {
            0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
            0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
            0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
            0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
        },
        {
            0x01,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
            0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
            0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
            0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
        },
        {
            0x01,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
            0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
            0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
            0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x81
        },
    };
    /* Big-endian values that will fail */
    byte fail_be[][CURVE25519_KEYSIZE] = {
        {
            0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
            0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
            0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
            0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
        },
        {
            0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
            0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
            0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
            0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x01
        },
        {
            0x81,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
            0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
            0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
            0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x01
        },
    };
    /* Good or valid public value */
    byte good[CURVE25519_KEYSIZE] = {
        0x01,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x01
    };
    int i;

    /* Parameter checks */
    /* NULL pointer */
    if (wc_curve25519_check_public(NULL, 0, EC25519_LITTLE_ENDIAN) !=
                                                                 BAD_FUNC_ARG) {
        return -10600;
    }
    if (wc_curve25519_check_public(NULL, 0, EC25519_BIG_ENDIAN) !=
                                                                 BAD_FUNC_ARG) {
        return -10601;
    }
    /* Length of 0 treated differently to other invalid lengths for TLS */
    if (wc_curve25519_check_public(good, 0, EC25519_LITTLE_ENDIAN) != BUFFER_E)
        return -10602;
    if (wc_curve25519_check_public(good, 0, EC25519_BIG_ENDIAN) != BUFFER_E)
        return -10603;

    /* Length not CURVE25519_KEYSIZE */
    for (i = 1; i < CURVE25519_KEYSIZE + 2; i++) {
        if (i == CURVE25519_KEYSIZE)
            continue;
        if (wc_curve25519_check_public(good, i, EC25519_LITTLE_ENDIAN) !=
                                                                ECC_BAD_ARG_E) {
            return -10604 - i;
        }
        if (wc_curve25519_check_public(good, i, EC25519_BIG_ENDIAN) !=
                                                                ECC_BAD_ARG_E) {
            return -10614 - i;
        }
    }

    /* Little-endian fail cases */
    for (i = 0; i < (int)(sizeof(fail_le) / sizeof(*fail_le)); i++) {
        if (wc_curve25519_check_public(fail_le[i], CURVE25519_KEYSIZE,
                                                  EC25519_LITTLE_ENDIAN) == 0) {
            return -10624 - i;
        }
    }
    /* Big-endian fail cases */
    for (i = 0; i < (int)(sizeof(fail_be) / sizeof(*fail_be)); i++) {
        if (wc_curve25519_check_public(fail_be[i], CURVE25519_KEYSIZE,
                                                     EC25519_BIG_ENDIAN) == 0) {
            return -10634 - i;
        }
    }

    /* Check a valid public value works! */
    if (wc_curve25519_check_public(good, CURVE25519_KEYSIZE,
                                                  EC25519_LITTLE_ENDIAN) != 0) {
        return -10644;
    }
    if (wc_curve25519_check_public(good, CURVE25519_KEYSIZE,
                                                     EC25519_BIG_ENDIAN) != 0) {
        return -10645;
    }

    return 0;
}

#endif /* HAVE_CURVE25519_SHARED_SECRET && HAVE_CURVE25519_KEY_IMPORT */

#if !defined(NO_ASN) && defined(HAVE_CURVE25519_KEY_EXPORT) && \
    defined(HAVE_CURVE25519_KEY_IMPORT)
static int curve255519_der_test(void)
{
    int ret = 0;
    /* certs/statickeys/x25519.der */
    const byte kCurve25519PrivDer[] = {
        0x30, 0x2E, 0x02, 0x01, 0x00, 0x30, 0x05, 0x06, 0x03, 0x2B, 0x65, 0x6E,
        0x04, 0x22, 0x04, 0x20, 0x78, 0x8E, 0x31, 0x5C, 0x33, 0xA9, 0x19, 0xC0,
        0x5E, 0x36, 0x70, 0x1B, 0xA4, 0xE8, 0xEF, 0xC1, 0x89, 0x8C, 0xB3, 0x15,
        0xC6, 0x79, 0xD3, 0xAC, 0x22, 0x00, 0xAE, 0xFA, 0xB3, 0xB7, 0x0F, 0x78
    };
    /* certs/statickeys/x25519-pub.der */
    const byte kCurve25519PubDer[] = {
        0x30, 0x2A, 0x30, 0x05, 0x06, 0x03, 0x2B, 0x65, 0x6E, 0x03, 0x21, 0x00,
        0x09, 0xBC, 0x8C, 0xC7, 0x45, 0x0D, 0xC1, 0xC2, 0x02, 0x57, 0x9A, 0x68,
        0x3A, 0xFD, 0x7A, 0xA8, 0xA5, 0x2F, 0xF0, 0x99, 0x39, 0x98, 0xEA, 0x26,
        0xA2, 0x5B, 0x38, 0xFD, 0x96, 0xDB, 0x2A, 0x26
    };
    curve25519_key key;
    byte output[128];
    word32 outputSz = 128;
    word32 idx;

    if (wc_curve25519_init_ex(&key, HEAP_HINT, devId) != 0) {
        return -10723;
    }

    /* Test decode / encode of Curve25519 private key only */
    if (ret == 0) {
        idx = 0;
        ret = wc_Curve25519PrivateKeyDecode(kCurve25519PrivDer, &idx, &key,
            (word32)sizeof(kCurve25519PrivDer));
    }
    if (ret == 0) {
        outputSz = (word32)sizeof(output);
        ret = wc_Curve25519PrivateKeyToDer(&key, output, outputSz);
        if (ret >= 0) {
            outputSz = ret;
            ret = 0;
        }
        else {
            ret = -10724;
        }
    }
    if (ret == 0 && (outputSz != (word32)sizeof(kCurve25519PrivDer) ||
                     XMEMCMP(output, kCurve25519PrivDer, outputSz) != 0)) {
        ret = -10725;
    }

    /* Test decode / encode of Curve25519 public key only */
    if (ret == 0) {
        idx = 0;
        ret = wc_Curve25519PublicKeyDecode(kCurve25519PubDer, &idx, &key,
            (word32)sizeof(kCurve25519PubDer));
    }
    if (ret == 0) {
        outputSz = (word32)sizeof(output);
        ret = wc_Curve25519PublicKeyToDer(&key, output, outputSz, 1);
        if (ret >= 0) {
            outputSz = ret;
            ret = 0;
        }
        else {
            ret = -10726;
        }
    }
    if (ret == 0 && (outputSz != (word32)sizeof(kCurve25519PubDer) ||
                     XMEMCMP(output, kCurve25519PubDer, outputSz) != 0)) {
        ret = -10727;
    }

    wc_curve25519_free(&key);

    return ret;
}
#endif /* !NO_ASN && HAVE_CURVE25519_KEY_EXPORT && HAVE_CURVE25519_KEY_IMPORT */

WOLFSSL_TEST_SUBROUTINE int curve25519_test(void)
{
    WC_RNG  rng;
    int ret;
#ifdef HAVE_CURVE25519_SHARED_SECRET
    byte    sharedA[32];
    byte    sharedB[32];
    word32  y;
#endif
#ifdef HAVE_CURVE25519_KEY_EXPORT
    byte    exportBuf[32];
#endif
    word32  x = 0;
    curve25519_key userA, userB, pubKey;

#if defined(HAVE_CURVE25519_SHARED_SECRET) && \
                                             defined(HAVE_CURVE25519_KEY_IMPORT)
    /* test vectors from
       https://tools.ietf.org/html/draft-josefsson-tls-curve25519-03
     */

    /* secret key for party a */
    byte sa[] = {
        0x5A,0xC9,0x9F,0x33,0x63,0x2E,0x5A,0x76,
        0x8D,0xE7,0xE8,0x1B,0xF8,0x54,0xC2,0x7C,
        0x46,0xE3,0xFB,0xF2,0xAB,0xBA,0xCD,0x29,
        0xEC,0x4A,0xFF,0x51,0x73,0x69,0xC6,0x60
    };

    /* public key for party a */
    byte pa[] = {
        0x05,0x7E,0x23,0xEA,0x9F,0x1C,0xBE,0x8A,
        0x27,0x16,0x8F,0x6E,0x69,0x6A,0x79,0x1D,
        0xE6,0x1D,0xD3,0xAF,0x7A,0xCD,0x4E,0xEA,
        0xCC,0x6E,0x7B,0xA5,0x14,0xFD,0xA8,0x63
    };

    /* secret key for party b */
    byte sb[] = {
        0x47,0xDC,0x3D,0x21,0x41,0x74,0x82,0x0E,
        0x11,0x54,0xB4,0x9B,0xC6,0xCD,0xB2,0xAB,
        0xD4,0x5E,0xE9,0x58,0x17,0x05,0x5D,0x25,
        0x5A,0xA3,0x58,0x31,0xB7,0x0D,0x32,0x60
    };

    /* public key for party b */
    byte pb[] = {
        0x6E,0xB8,0x9D,0xA9,0x19,0x89,0xAE,0x37,
        0xC7,0xEA,0xC7,0x61,0x8D,0x9E,0x5C,0x49,
        0x51,0xDB,0xA1,0xD7,0x3C,0x28,0x5A,0xE1,
        0xCD,0x26,0xA8,0x55,0x02,0x0E,0xEF,0x04
    };

    /* expected shared key */
    byte ss[] = {
        0x61,0x45,0x0C,0xD9,0x8E,0x36,0x01,0x6B,
        0x58,0x77,0x6A,0x89,0x7A,0x9F,0x0A,0xEF,
        0x73,0x8B,0x99,0xF0,0x94,0x68,0xB8,0xD6,
        0xB8,0x51,0x11,0x84,0xD5,0x34,0x94,0xAB
    };
#endif /* HAVE_CURVE25519_SHARED_SECRET */

    (void)x;

#ifndef HAVE_FIPS
    ret = wc_InitRng_ex(&rng, HEAP_HINT, devId);
#else
    ret = wc_InitRng(&rng);
#endif
    if (ret != 0)
        return -10700;

    wc_curve25519_init_ex(&userA, HEAP_HINT, devId);
    wc_curve25519_init_ex(&userB, HEAP_HINT, devId);
    wc_curve25519_init_ex(&pubKey, HEAP_HINT, devId);

    /* make curve25519 keys */
    if (wc_curve25519_make_key(&rng, 32, &userA) != 0)
        return -10701;

    if (wc_curve25519_make_key(&rng, 32, &userB) != 0)
        return -10702;

#ifdef HAVE_CURVE25519_SHARED_SECRET
    /* find shared secret key */
    x = sizeof(sharedA);
    if ((ret = wc_curve25519_shared_secret(&userA, &userB, sharedA, &x)) != 0) {
        printf("wc_curve25519_shared_secret 1 %d\n", ret);
        return -10703;
    }

    y = sizeof(sharedB);
    if ((ret = wc_curve25519_shared_secret(&userB, &userA, sharedB, &y)) != 0) {
        printf("wc_curve25519_shared_secret 2 %d\n", ret);
        return -10704;
    }

    /* compare shared secret keys to test they are the same */
    if (y != x)
        return -10705;

    if (XMEMCMP(sharedA, sharedB, x))
        return -10706;
#endif

#ifdef HAVE_CURVE25519_KEY_EXPORT
    /* export a public key and import it for another user */
    x = sizeof(exportBuf);
    if (wc_curve25519_export_public(&userA, exportBuf, &x) != 0)
        return -10707;

#ifdef HAVE_CURVE25519_KEY_IMPORT
    if (wc_curve25519_import_public(exportBuf, x, &pubKey) != 0)
        return -10708;
#endif
#endif

#if defined(HAVE_CURVE25519_SHARED_SECRET) && \
                                             defined(HAVE_CURVE25519_KEY_IMPORT)
    /* test shared key after importing a public key */
    XMEMSET(sharedB, 0, sizeof(sharedB));
    y = sizeof(sharedB);
    if (wc_curve25519_shared_secret(&userB, &pubKey, sharedB, &y) != 0)
        return -10709;

    if (XMEMCMP(sharedA, sharedB, y))
        return -10710;

    /* import RFC test vectors and compare shared key */
    if (wc_curve25519_import_private_raw(sa, sizeof(sa), pa, sizeof(pa), &userA)
            != 0)
        return -10711;

    if (wc_curve25519_import_private_raw(sb, sizeof(sb), pb, sizeof(pb), &userB)
            != 0)
        return -10712;

    /* test against known test vector */
    XMEMSET(sharedB, 0, sizeof(sharedB));
    y = sizeof(sharedB);
    if (wc_curve25519_shared_secret(&userA, &userB, sharedB, &y) != 0)
        return -10713;

    if (XMEMCMP(ss, sharedB, y))
        return -10714;

    /* test swapping roles of keys and generating same shared key */
    XMEMSET(sharedB, 0, sizeof(sharedB));
    y = sizeof(sharedB);
    if (wc_curve25519_shared_secret(&userB, &userA, sharedB, &y) != 0)
        return -10715;

    if (XMEMCMP(ss, sharedB, y))
        return -10716;

    /* test with 1 generated key and 1 from known test vector */
    if (wc_curve25519_import_private_raw(sa, sizeof(sa), pa, sizeof(pa), &userA)
        != 0)
        return -10717;

    wc_curve25519_free(&userB);
    wc_curve25519_init_ex(&userB, HEAP_HINT, devId);

    if (wc_curve25519_make_key(&rng, 32, &userB) != 0)
        return -10718;

    x = sizeof(sharedA);
    if (wc_curve25519_shared_secret(&userA, &userB, sharedA, &x) != 0)
        return -10719;

    y = sizeof(sharedB);
    if (wc_curve25519_shared_secret(&userB, &userA, sharedB, &y) != 0)
        return -10720;

    /* compare shared secret keys to test they are the same */
    if (y != x)
        return -10721;

    if (XMEMCMP(sharedA, sharedB, x))
        return -10722;

    ret = curve25519_overflow_test();
    if (ret != 0)
        return ret;
    ret = curve25519_check_public_test();
    if (ret != 0)
        return ret;
#endif /* HAVE_CURVE25519_SHARED_SECRET && HAVE_CURVE25519_KEY_IMPORT */

#if !defined(NO_ASN) && defined(HAVE_CURVE25519_KEY_EXPORT) && \
    defined(HAVE_CURVE25519_KEY_IMPORT)
    ret = curve255519_der_test();
    if (ret != 0)
        return ret;
#endif

    /* clean up keys when done */
    wc_curve25519_free(&pubKey);
    wc_curve25519_free(&userB);
    wc_curve25519_free(&userA);

    wc_FreeRng(&rng);

    return 0;
}
#endif /* HAVE_CURVE25519 */


#ifdef HAVE_ED25519
#ifdef WOLFSSL_TEST_CERT
static int ed25519_test_cert(void)
{
    DecodedCert  cert[2];
    DecodedCert* serverCert = NULL;
    DecodedCert* caCert = NULL;
#ifdef HAVE_ED25519_VERIFY
    ed25519_key  key;
    ed25519_key* pubKey = NULL;
    int          verify;
#endif /* HAVE_ED25519_VERIFY */
    int          ret;
    byte*        tmp;
    size_t       bytes;
    XFILE        file;

    tmp = (byte *)XMALLOC(FOURK_BUF, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    if (tmp == NULL) {
        ERROR_OUT(-10730, done);
    }

#ifdef USE_CERT_BUFFERS_256
    XMEMCPY(tmp, ca_ed25519_cert, sizeof_ca_ed25519_cert);
    bytes = sizeof_ca_ed25519_cert;
#elif !defined(NO_FILESYSTEM)
    file = XFOPEN(caEd25519Cert, "rb");
    if (file == NULL) {
        ERROR_OUT(-10731, done);
    }
    bytes = XFREAD(tmp, 1, FOURK_BUF, file);
    XFCLOSE(file);
#else
    /* No certificate to use. */
    ERROR_OUT(-10732, done);
#endif

    InitDecodedCert(&cert[0], tmp, (word32)bytes, 0);
    caCert = &cert[0];
    ret = ParseCert(caCert, CERT_TYPE, NO_VERIFY, NULL);
    if (ret != 0) {
        ERROR_OUT(-10733, done);
    }

#ifdef USE_CERT_BUFFERS_256
    XMEMCPY(tmp, server_ed25519_cert, sizeof_server_ed25519_cert);
    bytes = sizeof_server_ed25519_cert;
#elif !defined(NO_FILESYSTEM)
    file = XFOPEN(serverEd25519Cert, "rb");
    if (file == NULL) {
        ERROR_OUT(-10734, done);
    }
    bytes = XFREAD(tmp, 1, FOURK_BUF, file);
    XFCLOSE(file);
#else
    /* No certificate to use. */
    ERROR_OUT(-10735, done);
#endif

    InitDecodedCert(&cert[1], tmp, (word32)bytes, 0);
    serverCert = &cert[1];
    ret = ParseCert(serverCert, CERT_TYPE, NO_VERIFY, NULL);
    if (ret != 0) {
        ERROR_OUT(-10736, done);
    }

#ifdef HAVE_ED25519_VERIFY
    ret = wc_ed25519_init(&key);
    if (ret < 0) {
        ERROR_OUT(-10737, done);
    }
    pubKey = &key;
    ret = wc_ed25519_import_public(caCert->publicKey, caCert->pubKeySize,
                                                                        pubKey);
    if (ret < 0) {
        ERROR_OUT(-10738, done);
    }

    if (wc_ed25519_verify_msg(serverCert->signature, serverCert->sigLength,
                              serverCert->source + serverCert->certBegin,
                              serverCert->sigIndex - serverCert->certBegin,
                              &verify, pubKey) < 0 || verify != 1) {
        ERROR_OUT(-10739, done);
    }
#endif /* HAVE_ED25519_VERIFY */

done:
    if (tmp != NULL)
        XFREE(tmp, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
#ifdef HAVE_ED25519_VERIFY
    wc_ed25519_free(pubKey);
#endif /* HAVE_ED25519_VERIFY */
    if (caCert != NULL)
        FreeDecodedCert(caCert);
    if (serverCert != NULL)
        FreeDecodedCert(serverCert);

    return ret;
}

static int ed25519_test_make_cert(void)
{
    WC_RNG       rng;
    Cert         cert;
    DecodedCert  decode;
    ed25519_key  key;
    ed25519_key* privKey = NULL;
    int          ret = 0;
    byte*        tmp = NULL;

    wc_InitCert_ex(&cert, HEAP_HINT, devId);

#ifndef HAVE_FIPS
    ret = wc_InitRng_ex(&rng, HEAP_HINT, devId);
#else
    ret = wc_InitRng(&rng);
#endif
    if (ret != 0)
        return -10750;

    wc_ed25519_init(&key);
    privKey = &key;
    wc_ed25519_make_key(&rng, ED25519_KEY_SIZE, privKey);

    cert.daysValid = 365 * 2;
    cert.selfSigned = 1;
    XMEMCPY(&cert.issuer, &certDefaultName, sizeof(CertName));
    XMEMCPY(&cert.subject, &certDefaultName, sizeof(CertName));
    cert.isCA = 0;
#ifdef WOLFSSL_CERT_EXT
    ret = wc_SetKeyUsage(&cert, certKeyUsage);
    if (ret < 0) {
        ERROR_OUT(-10751, done);
    }
    ret = wc_SetSubjectKeyIdFromPublicKey_ex(&cert, ED25519_TYPE, privKey);
    if (ret < 0) {
        ERROR_OUT(-10752, done);
    }
    ret = wc_SetAuthKeyIdFromPublicKey_ex(&cert, ED25519_TYPE, privKey);
    if (ret < 0) {
        ERROR_OUT(-10753, done);
    }
#endif
    tmp = (byte *)XMALLOC(FOURK_BUF, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    if (tmp == NULL) {
        ERROR_OUT(-10754, done);
    }

    cert.sigType = CTC_ED25519;
    ret = wc_MakeCert_ex(&cert, tmp, FOURK_BUF, ED25519_TYPE, privKey, &rng);
    if (ret < 0) {
        ERROR_OUT(-10755, done);
    }
    ret = wc_SignCert_ex(cert.bodySz, cert.sigType, tmp, FOURK_BUF,
                                                   ED25519_TYPE, privKey, &rng);
    if (ret < 0) {
        ERROR_OUT(-10756, done);
    }

    InitDecodedCert(&decode, tmp, ret, HEAP_HINT);
    ret = ParseCert(&decode, CERT_TYPE, NO_VERIFY, 0);
    FreeDecodedCert(&decode);
    if (ret != 0) {
        ERROR_OUT(-10757, done);
    }

done:
    if (tmp != NULL)
        XFREE(tmp, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    wc_ed25519_free(privKey);
    wc_FreeRng(&rng);
    return ret;
}
#endif /* WOLFSSL_TEST_CERT */

#if defined(HAVE_ED25519_SIGN) && defined(HAVE_ED25519_KEY_EXPORT) && \
    defined(HAVE_ED25519_KEY_IMPORT)
static int ed25519ctx_test(void)
{
    int ret;
    byte   out[ED25519_SIG_SIZE];
    word32 outlen;
#ifdef HAVE_ED25519_VERIFY
    int    verify = 0;
#endif /* HAVE_ED25519_VERIFY */
    ed25519_key key;

    WOLFSSL_SMALL_STACK_STATIC const byte sKeyCtx[] = {
        0x03,0x05,0x33,0x4e,0x38,0x1a,0xf7,0x8f,
        0x14,0x1c,0xb6,0x66,0xf6,0x19,0x9f,0x57,
        0xbc,0x34,0x95,0x33,0x5a,0x25,0x6a,0x95,
        0xbd,0x2a,0x55,0xbf,0x54,0x66,0x63,0xf6
    };

    WOLFSSL_SMALL_STACK_STATIC const byte pKeyCtx[] = {
        0xdf,0xc9,0x42,0x5e,0x4f,0x96,0x8f,0x7f,
        0x0c,0x29,0xf0,0x25,0x9c,0xf5,0xf9,0xae,
        0xd6,0x85,0x1c,0x2b,0xb4,0xad,0x8b,0xfb,
        0x86,0x0c,0xfe,0xe0,0xab,0x24,0x82,0x92
    };

    WOLFSSL_SMALL_STACK_STATIC const byte sigCtx1[] = {
        0x55,0xa4,0xcc,0x2f,0x70,0xa5,0x4e,0x04,
        0x28,0x8c,0x5f,0x4c,0xd1,0xe4,0x5a,0x7b,
        0xb5,0x20,0xb3,0x62,0x92,0x91,0x18,0x76,
        0xca,0xda,0x73,0x23,0x19,0x8d,0xd8,0x7a,
        0x8b,0x36,0x95,0x0b,0x95,0x13,0x00,0x22,
        0x90,0x7a,0x7f,0xb7,0xc4,0xe9,0xb2,0xd5,
        0xf6,0xcc,0xa6,0x85,0xa5,0x87,0xb4,0xb2,
        0x1f,0x4b,0x88,0x8e,0x4e,0x7e,0xdb,0x0d
    };

    WOLFSSL_SMALL_STACK_STATIC const byte sigCtx2[] = {
        0xcc,0x5e,0x63,0xa2,0x7e,0x94,0xaf,0xd3,
        0x41,0x83,0x38,0xd2,0x48,0x6f,0xa9,0x2a,
        0xf9,0x91,0x7c,0x2d,0x98,0x9e,0x06,0xe5,
        0x02,0x77,0x72,0x1c,0x34,0x38,0x18,0xb4,
        0x21,0x96,0xbc,0x29,0x2e,0x68,0xf3,0x4d,
        0x85,0x9b,0xbe,0xad,0x17,0x9f,0x54,0x54,
        0x2d,0x4b,0x04,0xdc,0xfb,0xfa,0x4a,0x68,
        0x4e,0x39,0x50,0xfb,0x1c,0xcd,0x8d,0x0d
    };

    WOLFSSL_SMALL_STACK_STATIC const byte msgCtx[] = {
        0xf7,0x26,0x93,0x6d,0x19,0xc8,0x00,0x49,
        0x4e,0x3f,0xda,0xff,0x20,0xb2,0x76,0xa8
    };

    WOLFSSL_SMALL_STACK_STATIC const byte contextCtx[] = {
        0x66,0x6f,0x6f
    };

    outlen = sizeof(out);
    XMEMSET(out, 0, sizeof(out));

    ret = wc_ed25519_init_ex(&key, HEAP_HINT, devId);
    if (ret != 0)
        return 10800;

    ret = wc_ed25519_import_private_key(sKeyCtx, ED25519_KEY_SIZE, pKeyCtx,
                                      sizeof(pKeyCtx), &key);
    if (ret == 0)
        ret = wc_ed25519ctx_sign_msg(msgCtx, sizeof(msgCtx), out, &outlen, &key,
                               contextCtx, sizeof(contextCtx));
    if (ret == 0 && XMEMCMP(out, sigCtx1, 64) != 0)
        ret = -10801;

#if defined(HAVE_ED25519_VERIFY)
    /* test verify on good msg */
    if (ret == 0)
        ret = wc_ed25519ctx_verify_msg(out, outlen, msgCtx, sizeof(msgCtx),
            &verify, &key, contextCtx, sizeof(contextCtx));
    if (ret == 0 && verify != 1)
        ret = -10802;
#endif

    if (ret == 0)
        ret = wc_ed25519ctx_sign_msg(msgCtx, sizeof(msgCtx), out, &outlen, &key,
            NULL, 0);

    if (ret == 0 && XMEMCMP(out, sigCtx2, 64) != 0)
        ret = -10803;

#if defined(HAVE_ED25519_VERIFY)
    /* test verify on good msg */
    if (ret == 0)
        ret = wc_ed25519ctx_verify_msg(out, outlen, msgCtx, sizeof(msgCtx),
            &verify, &key, NULL, 0);
    if (ret == 0 && verify != 1)
        ret = -10804;
#endif

    wc_ed25519_free(&key);

    return ret;
}

static int ed25519ph_test(void)
{
    int ret = 0;
    byte   out[ED25519_SIG_SIZE];
    word32 outlen;
#ifdef HAVE_ED25519_VERIFY
    int    verify = 0;
#endif /* HAVE_ED25519_VERIFY */
    ed25519_key key;

    WOLFSSL_SMALL_STACK_STATIC const byte sKeyPh[] = {
        0x83,0x3f,0xe6,0x24,0x09,0x23,0x7b,0x9d,
        0x62,0xec,0x77,0x58,0x75,0x20,0x91,0x1e,
        0x9a,0x75,0x9c,0xec,0x1d,0x19,0x75,0x5b,
        0x7d,0xa9,0x01,0xb9,0x6d,0xca,0x3d,0x42
    };

    WOLFSSL_SMALL_STACK_STATIC const byte pKeyPh[] = {
        0xec,0x17,0x2b,0x93,0xad,0x5e,0x56,0x3b,
        0xf4,0x93,0x2c,0x70,0xe1,0x24,0x50,0x34,
        0xc3,0x54,0x67,0xef,0x2e,0xfd,0x4d,0x64,
        0xeb,0xf8,0x19,0x68,0x34,0x67,0xe2,0xbf
    };

    WOLFSSL_SMALL_STACK_STATIC const byte sigPh1[] = {
        0x98,0xa7,0x02,0x22,0xf0,0xb8,0x12,0x1a,
        0xa9,0xd3,0x0f,0x81,0x3d,0x68,0x3f,0x80,
        0x9e,0x46,0x2b,0x46,0x9c,0x7f,0xf8,0x76,
        0x39,0x49,0x9b,0xb9,0x4e,0x6d,0xae,0x41,
        0x31,0xf8,0x50,0x42,0x46,0x3c,0x2a,0x35,
        0x5a,0x20,0x03,0xd0,0x62,0xad,0xf5,0xaa,
        0xa1,0x0b,0x8c,0x61,0xe6,0x36,0x06,0x2a,
        0xaa,0xd1,0x1c,0x2a,0x26,0x08,0x34,0x06
    };

    WOLFSSL_SMALL_STACK_STATIC const byte sigPh2[] = {
        0xe0,0x39,0x70,0x2b,0x4c,0x25,0x95,0xa6,
        0xa5,0x41,0xac,0x85,0x09,0x23,0x6e,0x29,
        0x90,0x47,0x47,0x95,0x33,0x0c,0x9b,0x34,
        0xa7,0x5f,0x58,0xa6,0x60,0x12,0x9e,0x08,
        0xfd,0x73,0x69,0x43,0xfb,0x19,0x43,0xa5,
        0x57,0x20,0xb9,0xe0,0x95,0x7b,0x1e,0xd6,
        0x73,0x48,0x16,0x61,0x9f,0x13,0x88,0xf4,
        0x3f,0x73,0xe6,0xe3,0xba,0xa8,0x1c,0x0e
    };

    WOLFSSL_SMALL_STACK_STATIC const byte msgPh[] = {
        0x61,0x62,0x63
    };

    /* SHA-512 hash of msgPh */
    WOLFSSL_SMALL_STACK_STATIC const byte hashPh[] = {
        0xdd,0xaf,0x35,0xa1,0x93,0x61,0x7a,0xba,
        0xcc,0x41,0x73,0x49,0xae,0x20,0x41,0x31,
        0x12,0xe6,0xfa,0x4e,0x89,0xa9,0x7e,0xa2,
        0x0a,0x9e,0xee,0xe6,0x4b,0x55,0xd3,0x9a,
        0x21,0x92,0x99,0x2a,0x27,0x4f,0xc1,0xa8,
        0x36,0xba,0x3c,0x23,0xa3,0xfe,0xeb,0xbd,
        0x45,0x4d,0x44,0x23,0x64,0x3c,0xe8,0x0e,
        0x2a,0x9a,0xc9,0x4f,0xa5,0x4c,0xa4,0x9f
    };

    WOLFSSL_SMALL_STACK_STATIC const byte contextPh2[] = {
        0x66,0x6f,0x6f
    };

    outlen = sizeof(out);
    XMEMSET(out, 0, sizeof(out));

    ret = wc_ed25519_init_ex(&key, HEAP_HINT, devId);
    if (ret != 0)
        return -10900;

    ret = wc_ed25519_import_private_key(sKeyPh, ED25519_KEY_SIZE, pKeyPh,
                                      sizeof(pKeyPh), &key);
    if (ret == 0)
        ret = wc_ed25519ph_sign_msg(msgPh, sizeof(msgPh), out, &outlen, &key,
            NULL, 0);

    if (ret == 0 && XMEMCMP(out, sigPh1, 64) != 0)
        ret = -10901;

#if defined(HAVE_ED25519_VERIFY)
    /* test verify on good msg */
    if (ret == 0)
        ret = wc_ed25519ph_verify_msg(out, outlen, msgPh, sizeof(msgPh),
            &verify, &key, NULL, 0);
    if (ret == 0 && verify != 1)
        ret = -10902;
#endif

    if (ret == 0)
        ret = wc_ed25519ph_sign_msg(msgPh, sizeof(msgPh), out, &outlen, &key,
                              contextPh2, sizeof(contextPh2));

    if (ret == 0 && XMEMCMP(out, sigPh2, 64) != 0)
        ret = -10903;

#if defined(HAVE_ED25519_VERIFY)
    /* test verify on good msg */
    if (ret == 0)
        ret = wc_ed25519ph_verify_msg(out, outlen, msgPh, sizeof(msgPh), &verify,
                                &key, contextPh2, sizeof(contextPh2));
    if (ret == 0 && verify != 1)
        ret = -10904;
#endif

    if (ret == 0)
        ret = wc_ed25519ph_sign_hash(hashPh, sizeof(hashPh), out, &outlen, &key,
        NULL, 0);

    if (ret == 0 && XMEMCMP(out, sigPh1, 64) != 0)
        ret = -10905;

#if defined(HAVE_ED25519_VERIFY)
    if (ret == 0)
        ret = wc_ed25519ph_verify_hash(out, outlen, hashPh, sizeof(hashPh),
            &verify, &key, NULL, 0);
    if (ret == 0 && verify != 1)
        ret = -10906;
#endif

    if (ret == 0)
        ret = wc_ed25519ph_sign_hash(hashPh, sizeof(hashPh), out, &outlen, &key,
                              contextPh2, sizeof(contextPh2));
    if (ret == 0 && XMEMCMP(out, sigPh2, 64) != 0)
        ret = -10907;

#if defined(HAVE_ED25519_VERIFY)
    if (ret == 0)
        ret = wc_ed25519ph_verify_hash(out, outlen, hashPh, sizeof(hashPh), &verify,
                                &key, contextPh2, sizeof(contextPh2));
    if (ret == 0 && verify != 1)
        ret = -10908;
#endif

    wc_ed25519_free(&key);

    return ret;
}
#endif /* HAVE_ED25519_SIGN && HAVE_ED25519_KEY_EXPORT && HAVE_ED25519_KEY_IMPORT */

WOLFSSL_TEST_SUBROUTINE int ed25519_test(void)
{
    int ret;
    WC_RNG rng;
#if defined(HAVE_ED25519_SIGN) && defined(HAVE_ED25519_KEY_EXPORT) &&\
    defined(HAVE_ED25519_KEY_IMPORT)
    byte   out[ED25519_SIG_SIZE];
    byte   exportPKey[ED25519_KEY_SIZE];
    byte   exportSKey[ED25519_KEY_SIZE];
    word32 exportPSz;
    word32 exportSSz;
    int    i;
    word32 outlen;
#ifdef HAVE_ED25519_VERIFY
#ifdef WOLFSSL_ED25519_STREAMING_VERIFY
    int    j;
#endif
    int    verify;
#endif /* HAVE_ED25519_VERIFY */
#endif /* HAVE_ED25519_SIGN && HAVE_ED25519_KEY_EXPORT && HAVE_ED25519_KEY_IMPORT */
    word32 keySz, sigSz;
    ed25519_key key;
    ed25519_key key2;

#if defined(HAVE_ED25519_SIGN) && defined(HAVE_ED25519_KEY_EXPORT) && \
    defined(HAVE_ED25519_KEY_IMPORT)
    /* test vectors from
       https://tools.ietf.org/html/draft-josefsson-eddsa-ed25519-02
     */

    WOLFSSL_SMALL_STACK_STATIC const byte sKey1[] = {
        0x9d,0x61,0xb1,0x9d,0xef,0xfd,0x5a,0x60,
        0xba,0x84,0x4a,0xf4,0x92,0xec,0x2c,0xc4,
        0x44,0x49,0xc5,0x69,0x7b,0x32,0x69,0x19,
        0x70,0x3b,0xac,0x03,0x1c,0xae,0x7f,0x60
    };

    WOLFSSL_SMALL_STACK_STATIC const byte sKey2[] = {
        0x4c,0xcd,0x08,0x9b,0x28,0xff,0x96,0xda,
        0x9d,0xb6,0xc3,0x46,0xec,0x11,0x4e,0x0f,
        0x5b,0x8a,0x31,0x9f,0x35,0xab,0xa6,0x24,
        0xda,0x8c,0xf6,0xed,0x4f,0xb8,0xa6,0xfb
    };

    WOLFSSL_SMALL_STACK_STATIC const byte sKey3[] = {
        0xc5,0xaa,0x8d,0xf4,0x3f,0x9f,0x83,0x7b,
        0xed,0xb7,0x44,0x2f,0x31,0xdc,0xb7,0xb1,
        0x66,0xd3,0x85,0x35,0x07,0x6f,0x09,0x4b,
        0x85,0xce,0x3a,0x2e,0x0b,0x44,0x58,0xf7
    };

    /* uncompressed test */
    WOLFSSL_SMALL_STACK_STATIC const byte sKey4[] = {
        0x9d,0x61,0xb1,0x9d,0xef,0xfd,0x5a,0x60,
        0xba,0x84,0x4a,0xf4,0x92,0xec,0x2c,0xc4,
        0x44,0x49,0xc5,0x69,0x7b,0x32,0x69,0x19,
        0x70,0x3b,0xac,0x03,0x1c,0xae,0x7f,0x60
    };

    /* compressed prefix test */
    WOLFSSL_SMALL_STACK_STATIC const byte sKey5[] = {
        0x9d,0x61,0xb1,0x9d,0xef,0xfd,0x5a,0x60,
        0xba,0x84,0x4a,0xf4,0x92,0xec,0x2c,0xc4,
        0x44,0x49,0xc5,0x69,0x7b,0x32,0x69,0x19,
        0x70,0x3b,0xac,0x03,0x1c,0xae,0x7f,0x60
    };

    WOLFSSL_SMALL_STACK_STATIC const byte sKey6[] = {
        0xf5,0xe5,0x76,0x7c,0xf1,0x53,0x31,0x95,
        0x17,0x63,0x0f,0x22,0x68,0x76,0xb8,0x6c,
        0x81,0x60,0xcc,0x58,0x3b,0xc0,0x13,0x74,
        0x4c,0x6b,0xf2,0x55,0xf5,0xcc,0x0e,0xe5
    };

    WOLFSSL_SMALL_STACK_STATIC const byte* sKeys[] = {sKey1, sKey2, sKey3, sKey4, sKey5, sKey6};

    WOLFSSL_SMALL_STACK_STATIC const byte pKey1[] = {
        0xd7,0x5a,0x98,0x01,0x82,0xb1,0x0a,0xb7,
        0xd5,0x4b,0xfe,0xd3,0xc9,0x64,0x07,0x3a,
        0x0e,0xe1,0x72,0xf3,0xda,0xa6,0x23,0x25,
        0xaf,0x02,0x1a,0x68,0xf7,0x07,0x51,0x1a
    };

    WOLFSSL_SMALL_STACK_STATIC const byte pKey2[] = {
        0x3d,0x40,0x17,0xc3,0xe8,0x43,0x89,0x5a,
        0x92,0xb7,0x0a,0xa7,0x4d,0x1b,0x7e,0xbc,
        0x9c,0x98,0x2c,0xcf,0x2e,0xc4,0x96,0x8c,
        0xc0,0xcd,0x55,0xf1,0x2a,0xf4,0x66,0x0c
    };

    WOLFSSL_SMALL_STACK_STATIC const byte pKey3[] = {
        0xfc,0x51,0xcd,0x8e,0x62,0x18,0xa1,0xa3,
        0x8d,0xa4,0x7e,0xd0,0x02,0x30,0xf0,0x58,
        0x08,0x16,0xed,0x13,0xba,0x33,0x03,0xac,
        0x5d,0xeb,0x91,0x15,0x48,0x90,0x80,0x25
    };

    /* uncompressed test */
    WOLFSSL_SMALL_STACK_STATIC const byte pKey4[] = {
        0x04,0x55,0xd0,0xe0,0x9a,0x2b,0x9d,0x34,
        0x29,0x22,0x97,0xe0,0x8d,0x60,0xd0,0xf6,
        0x20,0xc5,0x13,0xd4,0x72,0x53,0x18,0x7c,
        0x24,0xb1,0x27,0x86,0xbd,0x77,0x76,0x45,
        0xce,0x1a,0x51,0x07,0xf7,0x68,0x1a,0x02,
        0xaf,0x25,0x23,0xa6,0xda,0xf3,0x72,0xe1,
        0x0e,0x3a,0x07,0x64,0xc9,0xd3,0xfe,0x4b,
        0xd5,0xb7,0x0a,0xb1,0x82,0x01,0x98,0x5a,
        0xd7
    };

    /* compressed prefix */
    WOLFSSL_SMALL_STACK_STATIC const byte pKey5[] = {
        0x40,0xd7,0x5a,0x98,0x01,0x82,0xb1,0x0a,0xb7,
        0xd5,0x4b,0xfe,0xd3,0xc9,0x64,0x07,0x3a,
        0x0e,0xe1,0x72,0xf3,0xda,0xa6,0x23,0x25,
        0xaf,0x02,0x1a,0x68,0xf7,0x07,0x51,0x1a
    };

    WOLFSSL_SMALL_STACK_STATIC const byte pKey6[] = {
        0x27,0x81,0x17,0xfc,0x14,0x4c,0x72,0x34,
        0x0f,0x67,0xd0,0xf2,0x31,0x6e,0x83,0x86,
        0xce,0xff,0xbf,0x2b,0x24,0x28,0xc9,0xc5,
        0x1f,0xef,0x7c,0x59,0x7f,0x1d,0x42,0x6e
    };

    WOLFSSL_SMALL_STACK_STATIC const byte* pKeys[] = {pKey1, pKey2, pKey3, pKey4, pKey5, pKey6};
    WOLFSSL_SMALL_STACK_STATIC const byte  pKeySz[] = {sizeof(pKey1), sizeof(pKey2), sizeof(pKey3),
                            sizeof(pKey4), sizeof(pKey5), sizeof(pKey6)};

    WOLFSSL_SMALL_STACK_STATIC const byte sig1[] = {
        0xe5,0x56,0x43,0x00,0xc3,0x60,0xac,0x72,
        0x90,0x86,0xe2,0xcc,0x80,0x6e,0x82,0x8a,
        0x84,0x87,0x7f,0x1e,0xb8,0xe5,0xd9,0x74,
        0xd8,0x73,0xe0,0x65,0x22,0x49,0x01,0x55,
        0x5f,0xb8,0x82,0x15,0x90,0xa3,0x3b,0xac,
        0xc6,0x1e,0x39,0x70,0x1c,0xf9,0xb4,0x6b,
        0xd2,0x5b,0xf5,0xf0,0x59,0x5b,0xbe,0x24,
        0x65,0x51,0x41,0x43,0x8e,0x7a,0x10,0x0b
    };

    WOLFSSL_SMALL_STACK_STATIC const byte sig2[] = {
        0x92,0xa0,0x09,0xa9,0xf0,0xd4,0xca,0xb8,
        0x72,0x0e,0x82,0x0b,0x5f,0x64,0x25,0x40,
        0xa2,0xb2,0x7b,0x54,0x16,0x50,0x3f,0x8f,
        0xb3,0x76,0x22,0x23,0xeb,0xdb,0x69,0xda,
        0x08,0x5a,0xc1,0xe4,0x3e,0x15,0x99,0x6e,
        0x45,0x8f,0x36,0x13,0xd0,0xf1,0x1d,0x8c,
        0x38,0x7b,0x2e,0xae,0xb4,0x30,0x2a,0xee,
        0xb0,0x0d,0x29,0x16,0x12,0xbb,0x0c,0x00
    };

    WOLFSSL_SMALL_STACK_STATIC const byte sig3[] = {
        0x62,0x91,0xd6,0x57,0xde,0xec,0x24,0x02,
        0x48,0x27,0xe6,0x9c,0x3a,0xbe,0x01,0xa3,
        0x0c,0xe5,0x48,0xa2,0x84,0x74,0x3a,0x44,
        0x5e,0x36,0x80,0xd7,0xdb,0x5a,0xc3,0xac,
        0x18,0xff,0x9b,0x53,0x8d,0x16,0xf2,0x90,
        0xae,0x67,0xf7,0x60,0x98,0x4d,0xc6,0x59,
        0x4a,0x7c,0x15,0xe9,0x71,0x6e,0xd2,0x8d,
        0xc0,0x27,0xbe,0xce,0xea,0x1e,0xc4,0x0a
    };

    /* uncompressed test */
    WOLFSSL_SMALL_STACK_STATIC const byte sig4[] = {
        0xe5,0x56,0x43,0x00,0xc3,0x60,0xac,0x72,
        0x90,0x86,0xe2,0xcc,0x80,0x6e,0x82,0x8a,
        0x84,0x87,0x7f,0x1e,0xb8,0xe5,0xd9,0x74,
        0xd8,0x73,0xe0,0x65,0x22,0x49,0x01,0x55,
        0x5f,0xb8,0x82,0x15,0x90,0xa3,0x3b,0xac,
        0xc6,0x1e,0x39,0x70,0x1c,0xf9,0xb4,0x6b,
        0xd2,0x5b,0xf5,0xf0,0x59,0x5b,0xbe,0x24,
        0x65,0x51,0x41,0x43,0x8e,0x7a,0x10,0x0b
    };

    /* compressed prefix */
    WOLFSSL_SMALL_STACK_STATIC const byte sig5[] = {
        0xe5,0x56,0x43,0x00,0xc3,0x60,0xac,0x72,
        0x90,0x86,0xe2,0xcc,0x80,0x6e,0x82,0x8a,
        0x84,0x87,0x7f,0x1e,0xb8,0xe5,0xd9,0x74,
        0xd8,0x73,0xe0,0x65,0x22,0x49,0x01,0x55,
        0x5f,0xb8,0x82,0x15,0x90,0xa3,0x3b,0xac,
        0xc6,0x1e,0x39,0x70,0x1c,0xf9,0xb4,0x6b,
        0xd2,0x5b,0xf5,0xf0,0x59,0x5b,0xbe,0x24,
        0x65,0x51,0x41,0x43,0x8e,0x7a,0x10,0x0b
    };

    WOLFSSL_SMALL_STACK_STATIC const byte sig6[] = {
        0x0a,0xab,0x4c,0x90,0x05,0x01,0xb3,0xe2,
        0x4d,0x7c,0xdf,0x46,0x63,0x32,0x6a,0x3a,
        0x87,0xdf,0x5e,0x48,0x43,0xb2,0xcb,0xdb,
        0x67,0xcb,0xf6,0xe4,0x60,0xfe,0xc3,0x50,
        0xaa,0x53,0x71,0xb1,0x50,0x8f,0x9f,0x45,
        0x28,0xec,0xea,0x23,0xc4,0x36,0xd9,0x4b,
        0x5e,0x8f,0xcd,0x4f,0x68,0x1e,0x30,0xa6,
        0xac,0x00,0xa9,0x70,0x4a,0x18,0x8a,0x03
    };

    WOLFSSL_SMALL_STACK_STATIC const byte* sigs[] = {sig1, sig2, sig3, sig4, sig5, sig6};

    WOLFSSL_SMALL_STACK_STATIC const byte msg1[]  = {0x0 };
    WOLFSSL_SMALL_STACK_STATIC const byte msg2[]  = {0x72};
    WOLFSSL_SMALL_STACK_STATIC const byte msg3[]  = {0xAF,0x82};

    /* test of a 1024 byte long message */
    WOLFSSL_SMALL_STACK_STATIC const byte msg4[]  = {
        0x08,0xb8,0xb2,0xb7,0x33,0x42,0x42,0x43,
        0x76,0x0f,0xe4,0x26,0xa4,0xb5,0x49,0x08,
        0x63,0x21,0x10,0xa6,0x6c,0x2f,0x65,0x91,
        0xea,0xbd,0x33,0x45,0xe3,0xe4,0xeb,0x98,
        0xfa,0x6e,0x26,0x4b,0xf0,0x9e,0xfe,0x12,
        0xee,0x50,0xf8,0xf5,0x4e,0x9f,0x77,0xb1,
        0xe3,0x55,0xf6,0xc5,0x05,0x44,0xe2,0x3f,
        0xb1,0x43,0x3d,0xdf,0x73,0xbe,0x84,0xd8,
        0x79,0xde,0x7c,0x00,0x46,0xdc,0x49,0x96,
        0xd9,0xe7,0x73,0xf4,0xbc,0x9e,0xfe,0x57,
        0x38,0x82,0x9a,0xdb,0x26,0xc8,0x1b,0x37,
        0xc9,0x3a,0x1b,0x27,0x0b,0x20,0x32,0x9d,
        0x65,0x86,0x75,0xfc,0x6e,0xa5,0x34,0xe0,
        0x81,0x0a,0x44,0x32,0x82,0x6b,0xf5,0x8c,
        0x94,0x1e,0xfb,0x65,0xd5,0x7a,0x33,0x8b,
        0xbd,0x2e,0x26,0x64,0x0f,0x89,0xff,0xbc,
        0x1a,0x85,0x8e,0xfc,0xb8,0x55,0x0e,0xe3,
        0xa5,0xe1,0x99,0x8b,0xd1,0x77,0xe9,0x3a,
        0x73,0x63,0xc3,0x44,0xfe,0x6b,0x19,0x9e,
        0xe5,0xd0,0x2e,0x82,0xd5,0x22,0xc4,0xfe,
        0xba,0x15,0x45,0x2f,0x80,0x28,0x8a,0x82,
        0x1a,0x57,0x91,0x16,0xec,0x6d,0xad,0x2b,
        0x3b,0x31,0x0d,0xa9,0x03,0x40,0x1a,0xa6,
        0x21,0x00,0xab,0x5d,0x1a,0x36,0x55,0x3e,
        0x06,0x20,0x3b,0x33,0x89,0x0c,0xc9,0xb8,
        0x32,0xf7,0x9e,0xf8,0x05,0x60,0xcc,0xb9,
        0xa3,0x9c,0xe7,0x67,0x96,0x7e,0xd6,0x28,
        0xc6,0xad,0x57,0x3c,0xb1,0x16,0xdb,0xef,
        0xef,0xd7,0x54,0x99,0xda,0x96,0xbd,0x68,
        0xa8,0xa9,0x7b,0x92,0x8a,0x8b,0xbc,0x10,
        0x3b,0x66,0x21,0xfc,0xde,0x2b,0xec,0xa1,
        0x23,0x1d,0x20,0x6b,0xe6,0xcd,0x9e,0xc7,
        0xaf,0xf6,0xf6,0xc9,0x4f,0xcd,0x72,0x04,
        0xed,0x34,0x55,0xc6,0x8c,0x83,0xf4,0xa4,
        0x1d,0xa4,0xaf,0x2b,0x74,0xef,0x5c,0x53,
        0xf1,0xd8,0xac,0x70,0xbd,0xcb,0x7e,0xd1,
        0x85,0xce,0x81,0xbd,0x84,0x35,0x9d,0x44,
        0x25,0x4d,0x95,0x62,0x9e,0x98,0x55,0xa9,
        0x4a,0x7c,0x19,0x58,0xd1,0xf8,0xad,0xa5,
        0xd0,0x53,0x2e,0xd8,0xa5,0xaa,0x3f,0xb2,
        0xd1,0x7b,0xa7,0x0e,0xb6,0x24,0x8e,0x59,
        0x4e,0x1a,0x22,0x97,0xac,0xbb,0xb3,0x9d,
        0x50,0x2f,0x1a,0x8c,0x6e,0xb6,0xf1,0xce,
        0x22,0xb3,0xde,0x1a,0x1f,0x40,0xcc,0x24,
        0x55,0x41,0x19,0xa8,0x31,0xa9,0xaa,0xd6,
        0x07,0x9c,0xad,0x88,0x42,0x5d,0xe6,0xbd,
        0xe1,0xa9,0x18,0x7e,0xbb,0x60,0x92,0xcf,
        0x67,0xbf,0x2b,0x13,0xfd,0x65,0xf2,0x70,
        0x88,0xd7,0x8b,0x7e,0x88,0x3c,0x87,0x59,
        0xd2,0xc4,0xf5,0xc6,0x5a,0xdb,0x75,0x53,
        0x87,0x8a,0xd5,0x75,0xf9,0xfa,0xd8,0x78,
        0xe8,0x0a,0x0c,0x9b,0xa6,0x3b,0xcb,0xcc,
        0x27,0x32,0xe6,0x94,0x85,0xbb,0xc9,0xc9,
        0x0b,0xfb,0xd6,0x24,0x81,0xd9,0x08,0x9b,
        0xec,0xcf,0x80,0xcf,0xe2,0xdf,0x16,0xa2,
        0xcf,0x65,0xbd,0x92,0xdd,0x59,0x7b,0x07,
        0x07,0xe0,0x91,0x7a,0xf4,0x8b,0xbb,0x75,
        0xfe,0xd4,0x13,0xd2,0x38,0xf5,0x55,0x5a,
        0x7a,0x56,0x9d,0x80,0xc3,0x41,0x4a,0x8d,
        0x08,0x59,0xdc,0x65,0xa4,0x61,0x28,0xba,
        0xb2,0x7a,0xf8,0x7a,0x71,0x31,0x4f,0x31,
        0x8c,0x78,0x2b,0x23,0xeb,0xfe,0x80,0x8b,
        0x82,0xb0,0xce,0x26,0x40,0x1d,0x2e,0x22,
        0xf0,0x4d,0x83,0xd1,0x25,0x5d,0xc5,0x1a,
        0xdd,0xd3,0xb7,0x5a,0x2b,0x1a,0xe0,0x78,
        0x45,0x04,0xdf,0x54,0x3a,0xf8,0x96,0x9b,
        0xe3,0xea,0x70,0x82,0xff,0x7f,0xc9,0x88,
        0x8c,0x14,0x4d,0xa2,0xaf,0x58,0x42,0x9e,
        0xc9,0x60,0x31,0xdb,0xca,0xd3,0xda,0xd9,
        0xaf,0x0d,0xcb,0xaa,0xaf,0x26,0x8c,0xb8,
        0xfc,0xff,0xea,0xd9,0x4f,0x3c,0x7c,0xa4,
        0x95,0xe0,0x56,0xa9,0xb4,0x7a,0xcd,0xb7,
        0x51,0xfb,0x73,0xe6,0x66,0xc6,0xc6,0x55,
        0xad,0xe8,0x29,0x72,0x97,0xd0,0x7a,0xd1,
        0xba,0x5e,0x43,0xf1,0xbc,0xa3,0x23,0x01,
        0x65,0x13,0x39,0xe2,0x29,0x04,0xcc,0x8c,
        0x42,0xf5,0x8c,0x30,0xc0,0x4a,0xaf,0xdb,
        0x03,0x8d,0xda,0x08,0x47,0xdd,0x98,0x8d,
        0xcd,0xa6,0xf3,0xbf,0xd1,0x5c,0x4b,0x4c,
        0x45,0x25,0x00,0x4a,0xa0,0x6e,0xef,0xf8,
        0xca,0x61,0x78,0x3a,0xac,0xec,0x57,0xfb,
        0x3d,0x1f,0x92,0xb0,0xfe,0x2f,0xd1,0xa8,
        0x5f,0x67,0x24,0x51,0x7b,0x65,0xe6,0x14,
        0xad,0x68,0x08,0xd6,0xf6,0xee,0x34,0xdf,
        0xf7,0x31,0x0f,0xdc,0x82,0xae,0xbf,0xd9,
        0x04,0xb0,0x1e,0x1d,0xc5,0x4b,0x29,0x27,
        0x09,0x4b,0x2d,0xb6,0x8d,0x6f,0x90,0x3b,
        0x68,0x40,0x1a,0xde,0xbf,0x5a,0x7e,0x08,
        0xd7,0x8f,0xf4,0xef,0x5d,0x63,0x65,0x3a,
        0x65,0x04,0x0c,0xf9,0xbf,0xd4,0xac,0xa7,
        0x98,0x4a,0x74,0xd3,0x71,0x45,0x98,0x67,
        0x80,0xfc,0x0b,0x16,0xac,0x45,0x16,0x49,
        0xde,0x61,0x88,0xa7,0xdb,0xdf,0x19,0x1f,
        0x64,0xb5,0xfc,0x5e,0x2a,0xb4,0x7b,0x57,
        0xf7,0xf7,0x27,0x6c,0xd4,0x19,0xc1,0x7a,
        0x3c,0xa8,0xe1,0xb9,0x39,0xae,0x49,0xe4,
        0x88,0xac,0xba,0x6b,0x96,0x56,0x10,0xb5,
        0x48,0x01,0x09,0xc8,0xb1,0x7b,0x80,0xe1,
        0xb7,0xb7,0x50,0xdf,0xc7,0x59,0x8d,0x5d,
        0x50,0x11,0xfd,0x2d,0xcc,0x56,0x00,0xa3,
        0x2e,0xf5,0xb5,0x2a,0x1e,0xcc,0x82,0x0e,
        0x30,0x8a,0xa3,0x42,0x72,0x1a,0xac,0x09,
        0x43,0xbf,0x66,0x86,0xb6,0x4b,0x25,0x79,
        0x37,0x65,0x04,0xcc,0xc4,0x93,0xd9,0x7e,
        0x6a,0xed,0x3f,0xb0,0xf9,0xcd,0x71,0xa4,
        0x3d,0xd4,0x97,0xf0,0x1f,0x17,0xc0,0xe2,
        0xcb,0x37,0x97,0xaa,0x2a,0x2f,0x25,0x66,
        0x56,0x16,0x8e,0x6c,0x49,0x6a,0xfc,0x5f,
        0xb9,0x32,0x46,0xf6,0xb1,0x11,0x63,0x98,
        0xa3,0x46,0xf1,0xa6,0x41,0xf3,0xb0,0x41,
        0xe9,0x89,0xf7,0x91,0x4f,0x90,0xcc,0x2c,
        0x7f,0xff,0x35,0x78,0x76,0xe5,0x06,0xb5,
        0x0d,0x33,0x4b,0xa7,0x7c,0x22,0x5b,0xc3,
        0x07,0xba,0x53,0x71,0x52,0xf3,0xf1,0x61,
        0x0e,0x4e,0xaf,0xe5,0x95,0xf6,0xd9,0xd9,
        0x0d,0x11,0xfa,0xa9,0x33,0xa1,0x5e,0xf1,
        0x36,0x95,0x46,0x86,0x8a,0x7f,0x3a,0x45,
        0xa9,0x67,0x68,0xd4,0x0f,0xd9,0xd0,0x34,
        0x12,0xc0,0x91,0xc6,0x31,0x5c,0xf4,0xfd,
        0xe7,0xcb,0x68,0x60,0x69,0x37,0x38,0x0d,
        0xb2,0xea,0xaa,0x70,0x7b,0x4c,0x41,0x85,
        0xc3,0x2e,0xdd,0xcd,0xd3,0x06,0x70,0x5e,
        0x4d,0xc1,0xff,0xc8,0x72,0xee,0xee,0x47,
        0x5a,0x64,0xdf,0xac,0x86,0xab,0xa4,0x1c,
        0x06,0x18,0x98,0x3f,0x87,0x41,0xc5,0xef,
        0x68,0xd3,0xa1,0x01,0xe8,0xa3,0xb8,0xca,
        0xc6,0x0c,0x90,0x5c,0x15,0xfc,0x91,0x08,
        0x40,0xb9,0x4c,0x00,0xa0,0xb9,0xd0
    };

    WOLFSSL_SMALL_STACK_STATIC const byte* msgs[] =   {msg1, msg2, msg3, msg1, msg1, msg4};
    WOLFSSL_SMALL_STACK_STATIC const word16 msgSz[] = {0 /*sizeof(msg1)*/,
                                   sizeof(msg2),
                                   sizeof(msg3),
                                   0 /*sizeof(msg1)*/,
                                   0 /*sizeof(msg1)*/,
                                   sizeof(msg4)
    };
#ifndef NO_ASN
    static byte privateEd25519[] = {
        0x30,0x2e,0x02,0x01,0x00,0x30,0x05,0x06,
        0x03,0x2b,0x65,0x70,0x04,0x22,0x04,0x20,
        0x9d,0x61,0xb1,0x9d,0xef,0xfd,0x5a,0x60,
        0xba,0x84,0x4a,0xf4,0x92,0xec,0x2c,0xc4,
        0x44,0x49,0xc5,0x69,0x7b,0x32,0x69,0x19,
        0x70,0x3b,0xac,0x03,0x1c,0xae,0x7f,0x60
    };
    static byte badPrivateEd25519[] = {
        0x30,0x52,0x02,0x01,0x00,0x30,0x05,0x06,
        0x03,0x2b,0x65,0x70,0x04,0x22,0x04,0x20,
        0x9d,0x61,0xb1,0x9d,0xef,0xfd,0x5a,0x60,
        0xba,0x84,0x4a,0xf4,0x92,0xec,0x2c,0xc4,
        0x44,0x49,0xc5,0x69,0x7b,0x32,0x69,0x19,
        0x70,0x3b,0xac,0x03,0x1c,0xae,0x7f,0x60,
        0xa1,0x22,0x04,0x21,0xd7,0x5a,0x98,0x01, /* octet len 0x20 -> 0x21 */
        0x82,0xb1,0x0a,0xb7,0xd5,0x4b,0xfe,0xd3,
        0xc9,0x64,0x07,0x3a,0x0e,0xe1,0x72,0xf3,
        0xda,0xa6,0x23,0x25,0xaf,0x02,0x1a,0x68,
        0xf7,0x07,0x51,0x1a,
        0x00  /* add additional bytes to make the pubkey bigger  */
    };
    static byte publicEd25519[] = {
        0x30,0x2a,0x30,0x05,0x06,0x03,0x2b,0x65,
        0x70,0x03,0x21,0x00,0xd7,0x5a,0x98,0x01,
        0x82,0xb1,0x0a,0xb7,0xd5,0x4b,0xfe,0xd3,
        0xc9,0x64,0x07,0x3a,0x0e,0xe1,0x72,0xf3,
        0xda,0xa6,0x23,0x25,0xaf,0x02,0x1a,0x68,
        0xf7,0x07,0x51,0x1a
    };

    /* size has been altered to catch if sanity check is done */
    static byte badPublicEd25519[] = {
        0x30,0x2a,0x30,0x05,0x06,0x03,0x2b,0x65,
        0x70,0x03,0x21,0x00,0xd7,0x5a,0x98,0x01,
        0x82,0xb1,0x0a,0xb7,0xd5,0x4b,0xfe,0xd3,
        0xc9,0x64,0x07,0x3a,0x0e,0xe1,0x72,0xf3,
        0xda,0xa6,0x23,0x25,0xaf,0x02,0x1a,0x68,
        0xf7,0x07,0x51,0x1a,
        0x00 /* add an additional byte to make the pubkey appear bigger */
    };
    static byte privPubEd25519[] = {
        0x30,0x50,0x02,0x01,0x00,0x30,0x05,0x06,
        0x03,0x2b,0x65,0x70,0x04,0x22,0x04,0x20,
        0x9d,0x61,0xb1,0x9d,0xef,0xfd,0x5a,0x60,
        0xba,0x84,0x4a,0xf4,0x92,0xec,0x2c,0xc4,
        0x44,0x49,0xc5,0x69,0x7b,0x32,0x69,0x19,
        0x70,0x3b,0xac,0x03,0x1c,0xae,0x7f,0x60,
        0x81,0x20,0xd7,0x5a,0x98,0x01,0x82,0xb1,
        0x0a,0xb7,0xd5,0x4b,0xfe,0xd3,0xc9,0x64,
        0x07,0x3a,0x0e,0xe1,0x72,0xf3,0xda,0xa6,
        0x23,0x25,0xaf,0x02,0x1a,0x68,0xf7,0x07,
        0x51,0x1a
    };

    word32 idx;
#endif /* NO_ASN */
#endif /* HAVE_ED25519_SIGN && HAVE_ED25519_KEY_EXPORT && HAVE_ED25519_KEY_IMPORT */
#if !defined(NO_ASN) && defined(HAVE_ED25519_SIGN)
    ed25519_key key3;
#endif

    /* create ed25519 keys */
#ifndef HAVE_FIPS
    ret = wc_InitRng_ex(&rng, HEAP_HINT, devId);
#else
    ret = wc_InitRng(&rng);
#endif
    if (ret != 0)
        return -11000;

    wc_ed25519_init_ex(&key, HEAP_HINT, devId);
    wc_ed25519_init_ex(&key2, HEAP_HINT, devId);
#if !defined(NO_ASN) && defined(HAVE_ED25519_SIGN)
    wc_ed25519_init_ex(&key3, HEAP_HINT, devId);
#endif
    wc_ed25519_make_key(&rng, ED25519_KEY_SIZE, &key);
    wc_ed25519_make_key(&rng, ED25519_KEY_SIZE, &key2);

    /* helper functions for signature and key size */
    keySz = wc_ed25519_size(&key);
    sigSz = wc_ed25519_sig_size(&key);

#if defined(HAVE_ED25519_SIGN) && defined(HAVE_ED25519_KEY_EXPORT) &&\
        defined(HAVE_ED25519_KEY_IMPORT)
    for (i = 0; i < 6; i++) {
        outlen = sizeof(out);
        XMEMSET(out, 0, sizeof(out));

        if (wc_ed25519_import_private_key(sKeys[i], ED25519_KEY_SIZE, pKeys[i],
                pKeySz[i], &key) != 0)
            return -11001 - i;

        if (wc_ed25519_sign_msg(msgs[i], msgSz[i], out, &outlen, &key) != 0)
            return -11011 - i;

        if (XMEMCMP(out, sigs[i], 64))
            return -11021 - i;

#if defined(HAVE_ED25519_VERIFY)
        /* test verify on good msg */
        if (wc_ed25519_verify_msg(out, outlen, msgs[i], msgSz[i], &verify,
                    &key) != 0 || verify != 1)
            return -11031 - i;

#ifdef WOLFSSL_ED25519_STREAMING_VERIFY
        /* test verify on good msg using streaming interface directly */
        if (wc_ed25519_verify_msg_init(out, outlen,
                                       &key, (byte)Ed25519, NULL, 0) != 0)
            return -11211 - i;
        for (j = 0; j < msgSz[i]; j += i) {
            if (wc_ed25519_verify_msg_update(msgs[i] + j, MIN(i, msgSz[i] - j), &key) != 0)
                return -11221 - i;
        }
        if (wc_ed25519_verify_msg_final(out, outlen, &verify,
                                        &key) != 0 || verify != 1)
            return -11231 - i;
#endif /* WOLFSSL_ED25519_STREAMING_VERIFY */

        /* test verify on bad msg */
        out[outlen-1] = out[outlen-1] + 1;
        if (wc_ed25519_verify_msg(out, outlen, msgs[i], msgSz[i], &verify,
                    &key) == 0 || verify == 1)
            return -11041 - i;
#endif /* HAVE_ED25519_VERIFY */

        /* test api for import/exporting keys */
        exportPSz = sizeof(exportPKey);
        exportSSz = sizeof(exportSKey);
        if (wc_ed25519_export_public(&key, exportPKey, &exportPSz) != 0)
            return -11051 - i;

        if (wc_ed25519_import_public_ex(exportPKey, exportPSz, &key2, 1) != 0)
            return -11061 - i;

        if (wc_ed25519_export_private_only(&key, exportSKey, &exportSSz) != 0)
            return -11071 - i;

        if (wc_ed25519_import_private_key(exportSKey, exportSSz,
                                          exportPKey, exportPSz, &key2) != 0)
            return -11081 - i;

        /* clear "out" buffer and test sign with imported keys */
        outlen = sizeof(out);
        XMEMSET(out, 0, sizeof(out));
        if (wc_ed25519_sign_msg(msgs[i], msgSz[i], out, &outlen, &key2) != 0)
            return -11091 - i;

#if defined(HAVE_ED25519_VERIFY)
        if (wc_ed25519_verify_msg(out, outlen, msgs[i], msgSz[i], &verify,
                                  &key2) != 0 || verify != 1)
            return -11101 - i;

        if (XMEMCMP(out, sigs[i], 64))
            return -11111 - i;
#endif /* HAVE_ED25519_VERIFY */
    }

    ret = ed25519ctx_test();
    if (ret != 0)
        return ret;

    ret = ed25519ph_test();
    if (ret != 0)
        return ret;

#ifndef NO_ASN
    /* Try ASN.1 encoded private-only key and public key. */
    idx = 0;
    if (wc_Ed25519PrivateKeyDecode(privateEd25519, &idx, &key3,
                                   sizeof(privateEd25519)) != 0)
        return -11121;

    idx = 0;
    if (wc_Ed25519PrivateKeyDecode(badPrivateEd25519, &idx, &key3,
                                   sizeof(badPrivateEd25519)) == 0)
        return -11122;

    if (wc_ed25519_sign_msg(msgs[0], msgSz[0], out, &outlen, &key3)
                != BAD_FUNC_ARG)
        return -11131;


    /* try with a buffer size that is too large */
    idx = 0;
    if (wc_Ed25519PublicKeyDecode(badPublicEd25519, &idx, &key3,
                                  sizeof(badPublicEd25519)) == 0)
        return -11140;

    idx = 0;
    if (wc_Ed25519PublicKeyDecode(publicEd25519, &idx, &key3,
                                  sizeof(publicEd25519)) != 0)
        return -11141;

    if (wc_ed25519_sign_msg(msgs[0], msgSz[0], out, &outlen, &key3) != 0)
        return -11151;

    if (XMEMCMP(out, sigs[0], 64))
        return -11161;

#if defined(HAVE_ED25519_VERIFY)
    /* test verify on good msg */
    if (wc_ed25519_verify_msg(out, outlen, msgs[0], msgSz[0], &verify, &key3)
                != 0 || verify != 1)
        return -11171;
#endif /* HAVE_ED25519_VERIFY */

    wc_ed25519_free(&key3);
    wc_ed25519_init(&key3);

    idx = 0;
    if (wc_Ed25519PrivateKeyDecode(privPubEd25519, &idx, &key3,
                                   sizeof(privPubEd25519)) != 0)
        return -11181;

    if (wc_ed25519_sign_msg(msgs[0], msgSz[0], out, &outlen, &key3) != 0)
        return -11191;

    if (XMEMCMP(out, sigs[0], 64))
        return -11201;

    wc_ed25519_free(&key3);
#endif /* NO_ASN */
#endif /* HAVE_ED25519_SIGN && HAVE_ED25519_KEY_EXPORT && HAVE_ED25519_KEY_IMPORT */

    /* clean up keys when done */
    wc_ed25519_free(&key);
    wc_ed25519_free(&key2);

#if defined(HAVE_HASHDRBG) || defined(NO_RC4)
    wc_FreeRng(&rng);
#endif

    /* hush warnings of unused keySz and sigSz */
    (void)keySz;
    (void)sigSz;

#ifdef WOLFSSL_TEST_CERT
    ret = ed25519_test_cert();
    if (ret < 0)
        return ret;
#ifdef WOLFSSL_CERT_GEN
    ret = ed25519_test_make_cert();
    if (ret < 0)
        return ret;
#endif /* WOLFSSL_CERT_GEN */
#endif /* WOLFSSL_TEST_CERT */

    return 0;
}
#endif /* HAVE_ED25519 */

#ifdef HAVE_CURVE448
#if defined(HAVE_CURVE448_SHARED_SECRET) && \
                                             defined(HAVE_CURVE448_KEY_IMPORT)
/* Test the wc_curve448_check_public API.
 *
 * returns 0 on success and -ve on failure.
 */
static int curve448_check_public_test(void)
{
    /* Little-endian values that will fail */
    byte fail_le[][CURVE448_KEY_SIZE] = {
        {
            0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
            0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
            0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
            0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
            0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
            0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
            0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
        },
        {
            0x01,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
            0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
            0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
            0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
            0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
            0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
            0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
        },
    };
    /* Big-endian values that will fail */
    byte fail_be[][CURVE448_KEY_SIZE] = {
        {
            0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
            0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
            0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
            0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
            0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
            0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
            0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
        },
        {
            0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
            0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
            0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
            0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
            0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
            0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
            0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x01
        },
    };
    /* Good or valid public value */
    byte good[CURVE448_KEY_SIZE] = {
        0x01,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x01
    };
    int i;

    /* Parameter checks */
    /* NULL pointer */
    if (wc_curve448_check_public(NULL, 0, EC448_LITTLE_ENDIAN) !=
                                                                 BAD_FUNC_ARG) {
        return -11300;
    }
    if (wc_curve448_check_public(NULL, 0, EC448_BIG_ENDIAN) != BAD_FUNC_ARG) {
        return -11301;
    }
    /* Length of 0 treated differently to other invalid lengths for TLS */
    if (wc_curve448_check_public(good, 0, EC448_LITTLE_ENDIAN) != BUFFER_E)
        return -11302;
    if (wc_curve448_check_public(good, 0, EC448_BIG_ENDIAN) != BUFFER_E)
        return -11303;

    /* Length not CURVE448_KEY_SIZE */
    for (i = 1; i < CURVE448_KEY_SIZE + 2; i++) {
        if (i == CURVE448_KEY_SIZE)
            continue;
        if (wc_curve448_check_public(good, i, EC448_LITTLE_ENDIAN) !=
                                                                ECC_BAD_ARG_E) {
            return -11304 - i;
        }
        if (wc_curve448_check_public(good, i, EC448_BIG_ENDIAN) !=
                                                                ECC_BAD_ARG_E) {
            return -11314 - i;
        }
    }

    /* Little-endian fail cases */
    for (i = 0; i < (int)(sizeof(fail_le) / sizeof(*fail_le)); i++) {
        if (wc_curve448_check_public(fail_le[i], CURVE448_KEY_SIZE,
                                                    EC448_LITTLE_ENDIAN) == 0) {
            return -11324 - i;
        }
    }
    /* Big-endian fail cases */
    for (i = 0; i < (int)(sizeof(fail_be) / sizeof(*fail_be)); i++) {
        if (wc_curve448_check_public(fail_be[i], CURVE448_KEY_SIZE,
                                                       EC448_BIG_ENDIAN) == 0) {
            return -11334 - i;
        }
    }

    /* Check a valid public value works! */
    if (wc_curve448_check_public(good, CURVE448_KEY_SIZE,
                                                    EC448_LITTLE_ENDIAN) != 0) {
        return -11344;
    }
    if (wc_curve448_check_public(good, CURVE448_KEY_SIZE,
                                                       EC448_BIG_ENDIAN) != 0) {
        return -11345;
    }

    return 0;
}

#endif /* HAVE_CURVE448_SHARED_SECRET && HAVE_CURVE448_KEY_IMPORT */

WOLFSSL_TEST_SUBROUTINE int curve448_test(void)
{
    WC_RNG  rng;
    int ret;
#ifdef HAVE_CURVE448_SHARED_SECRET
    byte    sharedA[CURVE448_KEY_SIZE];
    byte    sharedB[CURVE448_KEY_SIZE];
    word32  y;
#endif
#ifdef HAVE_CURVE448_KEY_EXPORT
    byte    exportBuf[CURVE448_KEY_SIZE];
#endif
    word32  x;
    curve448_key userA, userB, pubKey;

#if defined(HAVE_CURVE448_SHARED_SECRET) && \
                                             defined(HAVE_CURVE448_KEY_IMPORT)
    /* test vectors from
       https://www.rfc-editor.org/rfc/rfc7748.html
     */

    /* secret key for party a */
    byte sa[] = {
        0x6b, 0x72, 0x98, 0xa5, 0xc0, 0xd8, 0xc2, 0x9a,
        0x1d, 0xab, 0x27, 0xf1, 0xa6, 0x82, 0x63, 0x00,
        0x91, 0x73, 0x89, 0x44, 0x97, 0x41, 0xa9, 0x74,
        0xf5, 0xba, 0xc9, 0xd9, 0x8d, 0xc2, 0x98, 0xd4,
        0x65, 0x55, 0xbc, 0xe8, 0xba, 0xe8, 0x9e, 0xee,
        0xd4, 0x00, 0x58, 0x4b, 0xb0, 0x46, 0xcf, 0x75,
        0x57, 0x9f, 0x51, 0xd1, 0x25, 0x49, 0x8f, 0x9a,
    };

    /* public key for party a */
    byte pa[] = {
        0xa0, 0x1f, 0xc4, 0x32, 0xe5, 0x80, 0x7f, 0x17,
        0x53, 0x0d, 0x12, 0x88, 0xda, 0x12, 0x5b, 0x0c,
        0xd4, 0x53, 0xd9, 0x41, 0x72, 0x64, 0x36, 0xc8,
        0xbb, 0xd9, 0xc5, 0x22, 0x2c, 0x3d, 0xa7, 0xfa,
        0x63, 0x9c, 0xe0, 0x3d, 0xb8, 0xd2, 0x3b, 0x27,
        0x4a, 0x07, 0x21, 0xa1, 0xae, 0xd5, 0x22, 0x7d,
        0xe6, 0xe3, 0xb7, 0x31, 0xcc, 0xf7, 0x08, 0x9b,
    };

    /* secret key for party b */
    byte sb[] = {
        0x2d, 0x99, 0x73, 0x51, 0xb6, 0x10, 0x6f, 0x36,
        0xb0, 0xd1, 0x09, 0x1b, 0x92, 0x9c, 0x4c, 0x37,
        0x21, 0x3e, 0x0d, 0x2b, 0x97, 0xe8, 0x5e, 0xbb,
        0x20, 0xc1, 0x27, 0x69, 0x1d, 0x0d, 0xad, 0x8f,
        0x1d, 0x81, 0x75, 0xb0, 0x72, 0x37, 0x45, 0xe6,
        0x39, 0xa3, 0xcb, 0x70, 0x44, 0x29, 0x0b, 0x99,
        0xe0, 0xe2, 0xa0, 0xc2, 0x7a, 0x6a, 0x30, 0x1c,
    };

    /* public key for party b */
    byte pb[] = {
        0x09, 0x36, 0xf3, 0x7b, 0xc6, 0xc1, 0xbd, 0x07,
        0xae, 0x3d, 0xec, 0x7a, 0xb5, 0xdc, 0x06, 0xa7,
        0x3c, 0xa1, 0x32, 0x42, 0xfb, 0x34, 0x3e, 0xfc,
        0x72, 0xb9, 0xd8, 0x27, 0x30, 0xb4, 0x45, 0xf3,
        0xd4, 0xb0, 0xbd, 0x07, 0x71, 0x62, 0xa4, 0x6d,
        0xcf, 0xec, 0x6f, 0x9b, 0x59, 0x0b, 0xfc, 0xbc,
        0xf5, 0x20, 0xcd, 0xb0, 0x29, 0xa8, 0xb7, 0x3e,
    };

    /* expected shared key */
    byte ss[] = {
        0x9d, 0x87, 0x4a, 0x51, 0x37, 0x50, 0x9a, 0x44,
        0x9a, 0xd5, 0x85, 0x30, 0x40, 0x24, 0x1c, 0x52,
        0x36, 0x39, 0x54, 0x35, 0xc3, 0x64, 0x24, 0xfd,
        0x56, 0x0b, 0x0c, 0xb6, 0x2b, 0x28, 0x1d, 0x28,
        0x52, 0x75, 0xa7, 0x40, 0xce, 0x32, 0xa2, 0x2d,
        0xd1, 0x74, 0x0f, 0x4a, 0xa9, 0x16, 0x1c, 0xec,
        0x95, 0xcc, 0xc6, 0x1a, 0x18, 0xf4, 0xff, 0x07,
    };
#endif /* HAVE_CURVE448_SHARED_SECRET */

    (void)x;

#ifndef HAVE_FIPS
    ret = wc_InitRng_ex(&rng, HEAP_HINT, devId);
#else
    ret = wc_InitRng(&rng);
#endif
    if (ret != 0)
        return -11400;

    wc_curve448_init(&userA);
    wc_curve448_init(&userB);
    wc_curve448_init(&pubKey);

    /* make curve448 keys */
    if (wc_curve448_make_key(&rng, CURVE448_KEY_SIZE, &userA) != 0)
        return -11401;

    if (wc_curve448_make_key(&rng, CURVE448_KEY_SIZE, &userB) != 0)
        return -11402;

#ifdef HAVE_CURVE448_SHARED_SECRET
    /* find shared secret key */
    x = sizeof(sharedA);
    if (wc_curve448_shared_secret(&userA, &userB, sharedA, &x) != 0)
        return -11403;

    y = sizeof(sharedB);
    if (wc_curve448_shared_secret(&userB, &userA, sharedB, &y) != 0)
        return -11404;

    /* compare shared secret keys to test they are the same */
    if (y != x)
        return -11405;

    if (XMEMCMP(sharedA, sharedB, x))
        return -11406;
#endif

#ifdef HAVE_CURVE448_KEY_EXPORT
    /* export a public key and import it for another user */
    x = sizeof(exportBuf);
    if (wc_curve448_export_public(&userA, exportBuf, &x) != 0)
        return -11407;

#ifdef HAVE_CURVE448_KEY_IMPORT
    if (wc_curve448_import_public(exportBuf, x, &pubKey) != 0)
        return -11408;
#endif
#endif

#if defined(HAVE_CURVE448_SHARED_SECRET) && \
                                             defined(HAVE_CURVE448_KEY_IMPORT)
    /* test shared key after importing a public key */
    XMEMSET(sharedB, 0, sizeof(sharedB));
    y = sizeof(sharedB);
    if (wc_curve448_shared_secret(&userB, &pubKey, sharedB, &y) != 0)
        return -11409;

    if (XMEMCMP(sharedA, sharedB, y))
        return -11410;

    /* import RFC test vectors and compare shared key */
    if (wc_curve448_import_private_raw(sa, sizeof(sa), pa, sizeof(pa), &userA)
            != 0)
        return -11411;

    if (wc_curve448_import_private_raw(sb, sizeof(sb), pb, sizeof(pb), &userB)
            != 0)
        return -11412;

    /* test against known test vector */
    XMEMSET(sharedB, 0, sizeof(sharedB));
    y = sizeof(sharedB);
    if (wc_curve448_shared_secret(&userA, &userB, sharedB, &y) != 0)
        return -11413;

    if (XMEMCMP(ss, sharedB, y))
        return -11414;

    /* test swapping roles of keys and generating same shared key */
    XMEMSET(sharedB, 0, sizeof(sharedB));
    y = sizeof(sharedB);
    if (wc_curve448_shared_secret(&userB, &userA, sharedB, &y) != 0)
        return -11415;

    if (XMEMCMP(ss, sharedB, y))
        return -11416;

    /* test with 1 generated key and 1 from known test vector */
    if (wc_curve448_import_private_raw(sa, sizeof(sa), pa, sizeof(pa), &userA)
        != 0)
        return -11417;

    if (wc_curve448_make_key(&rng, 56, &userB) != 0)
        return -11418;

    x = sizeof(sharedA);
    if (wc_curve448_shared_secret(&userA, &userB, sharedA, &x) != 0)
        return -11419;

    y = sizeof(sharedB);
    if (wc_curve448_shared_secret(&userB, &userA, sharedB, &y) != 0)
        return -11420;

    /* compare shared secret keys to test they are the same */
    if (y != x)
        return -11421;

    if (XMEMCMP(sharedA, sharedB, x))
        return -11422;

    ret = curve448_check_public_test();
    if (ret != 0)
        return ret;
#endif /* HAVE_CURVE448_SHARED_SECRET && HAVE_CURVE448_KEY_IMPORT */

    /* clean up keys when done */
    wc_curve448_free(&pubKey);
    wc_curve448_free(&userB);
    wc_curve448_free(&userA);

    wc_FreeRng(&rng);

    return 0;
}
#endif /* HAVE_CURVE448 */

#ifdef HAVE_ED448
#ifdef WOLFSSL_TEST_CERT
static int ed448_test_cert(void)
{
    DecodedCert  cert[2];
    DecodedCert* serverCert = NULL;
    DecodedCert* caCert = NULL;
#ifdef HAVE_ED448_VERIFY
    ed448_key  key;
    ed448_key* pubKey = NULL;
    int          verify;
#endif /* HAVE_ED448_VERIFY */
    int          ret;
    byte*        tmp;
    size_t       bytes;
    XFILE        file;

    tmp = (byte *)XMALLOC(FOURK_BUF, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    if (tmp == NULL) {
        ERROR_OUT(-11430, done);
    }

#ifdef USE_CERT_BUFFERS_256
    XMEMCPY(tmp, ca_ed448_cert, sizeof_ca_ed448_cert);
    bytes = sizeof_ca_ed448_cert;
#elif !defined(NO_FILESYSTEM)
    file = XFOPEN(caEd448Cert, "rb");
    if (file == NULL) {
        ERROR_OUT(-11431, done);
    }
    bytes = XFREAD(tmp, 1, FOURK_BUF, file);
    XFCLOSE(file);
#else
    /* No certificate to use. */
    ERROR_OUT(-11432, done);
#endif

    InitDecodedCert(&cert[0], tmp, (word32)bytes, 0);
    caCert = &cert[0];
    ret = ParseCert(caCert, CERT_TYPE, NO_VERIFY, NULL);
    if (ret != 0) {
        ERROR_OUT(-11433, done);
    }

#ifdef USE_CERT_BUFFERS_256
    XMEMCPY(tmp, server_ed448_cert, sizeof_server_ed448_cert);
    bytes = sizeof_server_ed448_cert;
#elif !defined(NO_FILESYSTEM)
    file = XFOPEN(serverEd448Cert, "rb");
    if (file == NULL) {
        ERROR_OUT(-11434, done);
    }
    bytes = XFREAD(tmp, 1, FOURK_BUF, file);
    XFCLOSE(file);
#else
    /* No certificate to use. */
    ERROR_OUT(-11435, done);
#endif

    InitDecodedCert(&cert[1], tmp, (word32)bytes, 0);
    serverCert = &cert[1];
    ret = ParseCert(serverCert, CERT_TYPE, NO_VERIFY, NULL);
    if (ret != 0) {
        ERROR_OUT(-11436, done);
    }

#ifdef HAVE_ED448_VERIFY
    ret = wc_ed448_init(&key);
    if (ret < 0) {
        ERROR_OUT(-11437, done);
    }
    pubKey = &key;
    ret = wc_ed448_import_public(caCert->publicKey, caCert->pubKeySize, pubKey);
    if (ret < 0) {
        ERROR_OUT(-11438, done);
    }

    if (wc_ed448_verify_msg(serverCert->signature, serverCert->sigLength,
                            serverCert->source + serverCert->certBegin,
                            serverCert->sigIndex - serverCert->certBegin,
                            &verify, pubKey, NULL, 0) < 0 || verify != 1) {
        ERROR_OUT(-11439, done);
    }
#endif /* HAVE_ED448_VERIFY */

done:
    if (tmp != NULL)
        XFREE(tmp, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
#ifdef HAVE_ED448_VERIFY
    wc_ed448_free(pubKey);
#endif /* HAVE_ED448_VERIFY */
    if (caCert != NULL)
        FreeDecodedCert(caCert);
    if (serverCert != NULL)
        FreeDecodedCert(serverCert);

    return ret;
}

static int ed448_test_make_cert(void)
{
    WC_RNG       rng;
    Cert         cert;
    DecodedCert  decode;
    ed448_key  key;
    ed448_key* privKey = NULL;
    int          ret = 0;
    byte*        tmp = NULL;

    wc_InitCert_ex(&cert, HEAP_HINT, devId);

#ifndef HAVE_FIPS
    ret = wc_InitRng_ex(&rng, HEAP_HINT, devId);
#else
    ret = wc_InitRng(&rng);
#endif
    if (ret != 0)
        return -11450;

    wc_ed448_init(&key);
    privKey = &key;
    wc_ed448_make_key(&rng, ED448_KEY_SIZE, privKey);

    cert.daysValid = 365 * 2;
    cert.selfSigned = 1;
    XMEMCPY(&cert.issuer, &certDefaultName, sizeof(CertName));
    XMEMCPY(&cert.subject, &certDefaultName, sizeof(CertName));
    cert.isCA = 0;
#ifdef WOLFSSL_CERT_EXT
    ret = wc_SetKeyUsage(&cert, certKeyUsage);
    if (ret < 0) {
        ERROR_OUT(-11451, done);
    }
    ret = wc_SetSubjectKeyIdFromPublicKey_ex(&cert, ED448_TYPE, privKey);
    if (ret < 0) {
        ERROR_OUT(-11452, done);
    }
    ret = wc_SetAuthKeyIdFromPublicKey_ex(&cert, ED448_TYPE, privKey);
    if (ret < 0) {
        ERROR_OUT(-11453, done);
    }
#endif
    tmp = (byte *)XMALLOC(FOURK_BUF, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    if (tmp == NULL) {
        ERROR_OUT(-11454, done);
    }

    cert.sigType = CTC_ED448;
    ret = wc_MakeCert_ex(&cert, tmp, FOURK_BUF, ED448_TYPE, privKey, &rng);
    if (ret < 0) {
        ERROR_OUT(-11455, done);
    }
    ret = wc_SignCert_ex(cert.bodySz, cert.sigType, tmp, FOURK_BUF, ED448_TYPE,
                                                                 privKey, &rng);
    if (ret < 0) {
        ERROR_OUT(-11456, done);
    }

    InitDecodedCert(&decode, tmp, ret, HEAP_HINT);
    ret = ParseCert(&decode, CERT_TYPE, NO_VERIFY, 0);
    FreeDecodedCert(&decode);
    if (ret != 0) {
        ERROR_OUT(-11457, done);
    }

done:
    if (tmp != NULL)
        XFREE(tmp, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    wc_ed448_free(privKey);
    wc_FreeRng(&rng);
    return ret;
}
#endif /* WOLFSSL_TEST_CERT */

#if defined(HAVE_ED448_SIGN) && defined(HAVE_ED448_KEY_EXPORT) && \
    defined(HAVE_ED448_KEY_IMPORT)
static int ed448_ctx_test(void)
{
    byte   out[ED448_SIG_SIZE];
    word32 outlen;
#ifdef HAVE_ED448_VERIFY
    int    verify;
#endif /* HAVE_ED448_VERIFY */
    ed448_key key;

    WOLFSSL_SMALL_STACK_STATIC const byte sKeyCtx[] = {
        0xc4, 0xea, 0xb0, 0x5d, 0x35, 0x70, 0x07, 0xc6,
        0x32, 0xf3, 0xdb, 0xb4, 0x84, 0x89, 0x92, 0x4d,
        0x55, 0x2b, 0x08, 0xfe, 0x0c, 0x35, 0x3a, 0x0d,
        0x4a, 0x1f, 0x00, 0xac, 0xda, 0x2c, 0x46, 0x3a,
        0xfb, 0xea, 0x67, 0xc5, 0xe8, 0xd2, 0x87, 0x7c,
        0x5e, 0x3b, 0xc3, 0x97, 0xa6, 0x59, 0x94, 0x9e,
        0xf8, 0x02, 0x1e, 0x95, 0x4e, 0x0a, 0x12, 0x27,
        0x4e
    };

    WOLFSSL_SMALL_STACK_STATIC const byte pKeyCtx[] = {
        0x43, 0xba, 0x28, 0xf4, 0x30, 0xcd, 0xff, 0x45,
        0x6a, 0xe5, 0x31, 0x54, 0x5f, 0x7e, 0xcd, 0x0a,
        0xc8, 0x34, 0xa5, 0x5d, 0x93, 0x58, 0xc0, 0x37,
        0x2b, 0xfa, 0x0c, 0x6c, 0x67, 0x98, 0xc0, 0x86,
        0x6a, 0xea, 0x01, 0xeb, 0x00, 0x74, 0x28, 0x02,
        0xb8, 0x43, 0x8e, 0xa4, 0xcb, 0x82, 0x16, 0x9c,
        0x23, 0x51, 0x60, 0x62, 0x7b, 0x4c, 0x3a, 0x94,
        0x80
    };

    WOLFSSL_SMALL_STACK_STATIC const byte sigCtx[] = {
        0xd4, 0xf8, 0xf6, 0x13, 0x17, 0x70, 0xdd, 0x46,
        0xf4, 0x08, 0x67, 0xd6, 0xfd, 0x5d, 0x50, 0x55,
        0xde, 0x43, 0x54, 0x1f, 0x8c, 0x5e, 0x35, 0xab,
        0xbc, 0xd0, 0x01, 0xb3, 0x2a, 0x89, 0xf7, 0xd2,
        0x15, 0x1f, 0x76, 0x47, 0xf1, 0x1d, 0x8c, 0xa2,
        0xae, 0x27, 0x9f, 0xb8, 0x42, 0xd6, 0x07, 0x21,
        0x7f, 0xce, 0x6e, 0x04, 0x2f, 0x68, 0x15, 0xea,
        0x00, 0x0c, 0x85, 0x74, 0x1d, 0xe5, 0xc8, 0xda,
        0x11, 0x44, 0xa6, 0xa1, 0xab, 0xa7, 0xf9, 0x6d,
        0xe4, 0x25, 0x05, 0xd7, 0xa7, 0x29, 0x85, 0x24,
        0xfd, 0xa5, 0x38, 0xfc, 0xcb, 0xbb, 0x75, 0x4f,
        0x57, 0x8c, 0x1c, 0xad, 0x10, 0xd5, 0x4d, 0x0d,
        0x54, 0x28, 0x40, 0x7e, 0x85, 0xdc, 0xbc, 0x98,
        0xa4, 0x91, 0x55, 0xc1, 0x37, 0x64, 0xe6, 0x6c,
        0x3c, 0x00
    };

    WOLFSSL_SMALL_STACK_STATIC const byte msgCtx[] = {
        0x03
    };

    WOLFSSL_SMALL_STACK_STATIC const byte contextCtx[] = {
        0x66,0x6f,0x6f
    };

    outlen = sizeof(out);
    XMEMSET(out, 0, sizeof(out));

    if (wc_ed448_init_ex(&key, HEAP_HINT, devId) != 0)
        return -11500;

    if (wc_ed448_import_private_key(sKeyCtx, ED448_KEY_SIZE, pKeyCtx,
                                    sizeof(pKeyCtx), &key) != 0)
        return -11501;

    if (wc_ed448_sign_msg(msgCtx, sizeof(msgCtx), out, &outlen, &key,
                          contextCtx, sizeof(contextCtx)) != 0)
        return -11502;

    if (XMEMCMP(out, sigCtx, sizeof(sigCtx)))
        return -11503;

#if defined(HAVE_ED448_VERIFY)
    /* test verify on good msg */
    if (wc_ed448_verify_msg(out, outlen, msgCtx, sizeof(msgCtx), &verify, &key,
                            contextCtx, sizeof(contextCtx)) != 0 || verify != 1)
        return -11504;
#endif

    wc_ed448_free(&key);

    return 0;
}

static int ed448ph_test(void)
{
    byte   out[ED448_SIG_SIZE];
    word32 outlen;
#ifdef HAVE_ED448_VERIFY
    int    verify;
#endif /* HAVE_ED448_VERIFY */
    ed448_key key;

    WOLFSSL_SMALL_STACK_STATIC const byte sKeyPh[] = {
        0x83, 0x3f, 0xe6, 0x24, 0x09, 0x23, 0x7b, 0x9d,
        0x62, 0xec, 0x77, 0x58, 0x75, 0x20, 0x91, 0x1e,
        0x9a, 0x75, 0x9c, 0xec, 0x1d, 0x19, 0x75, 0x5b,
        0x7d, 0xa9, 0x01, 0xb9, 0x6d, 0xca, 0x3d, 0x42,
        0xef, 0x78, 0x22, 0xe0, 0xd5, 0x10, 0x41, 0x27,
        0xdc, 0x05, 0xd6, 0xdb, 0xef, 0xde, 0x69, 0xe3,
        0xab, 0x2c, 0xec, 0x7c, 0x86, 0x7c, 0x6e, 0x2c,
        0x49
    };

    WOLFSSL_SMALL_STACK_STATIC const byte pKeyPh[] = {
        0x25, 0x9b, 0x71, 0xc1, 0x9f, 0x83, 0xef, 0x77,
        0xa7, 0xab, 0xd2, 0x65, 0x24, 0xcb, 0xdb, 0x31,
        0x61, 0xb5, 0x90, 0xa4, 0x8f, 0x7d, 0x17, 0xde,
        0x3e, 0xe0, 0xba, 0x9c, 0x52, 0xbe, 0xb7, 0x43,
        0xc0, 0x94, 0x28, 0xa1, 0x31, 0xd6, 0xb1, 0xb5,
        0x73, 0x03, 0xd9, 0x0d, 0x81, 0x32, 0xc2, 0x76,
        0xd5, 0xed, 0x3d, 0x5d, 0x01, 0xc0, 0xf5, 0x38,
        0x80
    };

    WOLFSSL_SMALL_STACK_STATIC const byte sigPh1[] = {
        0x82, 0x2f, 0x69, 0x01, 0xf7, 0x48, 0x0f, 0x3d,
        0x5f, 0x56, 0x2c, 0x59, 0x29, 0x94, 0xd9, 0x69,
        0x36, 0x02, 0x87, 0x56, 0x14, 0x48, 0x32, 0x56,
        0x50, 0x56, 0x00, 0xbb, 0xc2, 0x81, 0xae, 0x38,
        0x1f, 0x54, 0xd6, 0xbc, 0xe2, 0xea, 0x91, 0x15,
        0x74, 0x93, 0x2f, 0x52, 0xa4, 0xe6, 0xca, 0xdd,
        0x78, 0x76, 0x93, 0x75, 0xec, 0x3f, 0xfd, 0x1b,
        0x80, 0x1a, 0x0d, 0x9b, 0x3f, 0x40, 0x30, 0xcd,
        0x43, 0x39, 0x64, 0xb6, 0x45, 0x7e, 0xa3, 0x94,
        0x76, 0x51, 0x12, 0x14, 0xf9, 0x74, 0x69, 0xb5,
        0x7d, 0xd3, 0x2d, 0xbc, 0x56, 0x0a, 0x9a, 0x94,
        0xd0, 0x0b, 0xff, 0x07, 0x62, 0x04, 0x64, 0xa3,
        0xad, 0x20, 0x3d, 0xf7, 0xdc, 0x7c, 0xe3, 0x60,
        0xc3, 0xcd, 0x36, 0x96, 0xd9, 0xd9, 0xfa, 0xb9,
        0x0f, 0x00
    };

    WOLFSSL_SMALL_STACK_STATIC const byte sigPh2[] = {
        0xc3, 0x22, 0x99, 0xd4, 0x6e, 0xc8, 0xff, 0x02,
        0xb5, 0x45, 0x40, 0x98, 0x28, 0x14, 0xdc, 0xe9,
        0xa0, 0x58, 0x12, 0xf8, 0x19, 0x62, 0xb6, 0x49,
        0xd5, 0x28, 0x09, 0x59, 0x16, 0xa2, 0xaa, 0x48,
        0x10, 0x65, 0xb1, 0x58, 0x04, 0x23, 0xef, 0x92,
        0x7e, 0xcf, 0x0a, 0xf5, 0x88, 0x8f, 0x90, 0xda,
        0x0f, 0x6a, 0x9a, 0x85, 0xad, 0x5d, 0xc3, 0xf2,
        0x80, 0xd9, 0x12, 0x24, 0xba, 0x99, 0x11, 0xa3,
        0x65, 0x3d, 0x00, 0xe4, 0x84, 0xe2, 0xce, 0x23,
        0x25, 0x21, 0x48, 0x1c, 0x86, 0x58, 0xdf, 0x30,
        0x4b, 0xb7, 0x74, 0x5a, 0x73, 0x51, 0x4c, 0xdb,
        0x9b, 0xf3, 0xe1, 0x57, 0x84, 0xab, 0x71, 0x28,
        0x4f, 0x8d, 0x07, 0x04, 0xa6, 0x08, 0xc5, 0x4a,
        0x6b, 0x62, 0xd9, 0x7b, 0xeb, 0x51, 0x1d, 0x13,
        0x21, 0x00
    };

    WOLFSSL_SMALL_STACK_STATIC const byte msgPh[] = {
        0x61,0x62,0x63
    };

    /* SHA-512 hash of msgPh */
    WOLFSSL_SMALL_STACK_STATIC const byte hashPh[] = {
        0x48, 0x33, 0x66, 0x60, 0x13, 0x60, 0xa8, 0x77,
        0x1c, 0x68, 0x63, 0x08, 0x0c, 0xc4, 0x11, 0x4d,
        0x8d, 0xb4, 0x45, 0x30, 0xf8, 0xf1, 0xe1, 0xee,
        0x4f, 0x94, 0xea, 0x37, 0xe7, 0x8b, 0x57, 0x39,
        0xd5, 0xa1, 0x5b, 0xef, 0x18, 0x6a, 0x53, 0x86,
        0xc7, 0x57, 0x44, 0xc0, 0x52, 0x7e, 0x1f, 0xaa,
        0x9f, 0x87, 0x26, 0xe4, 0x62, 0xa1, 0x2a, 0x4f,
        0xeb, 0x06, 0xbd, 0x88, 0x01, 0xe7, 0x51, 0xe4
    };

    WOLFSSL_SMALL_STACK_STATIC const byte contextPh2[] = {
        0x66,0x6f,0x6f
    };

    outlen = sizeof(out);
    XMEMSET(out, 0, sizeof(out));

    if (wc_ed448_init_ex(&key, HEAP_HINT, devId) != 0)
        return -11600;

    if (wc_ed448_import_private_key(sKeyPh, ED448_KEY_SIZE, pKeyPh,
                                    sizeof(pKeyPh), &key) != 0) {
        return -11601;
    }

    if (wc_ed448ph_sign_msg(msgPh, sizeof(msgPh), out, &outlen, &key, NULL,
                            0) != 0) {
        return -11602;
    }

    if (XMEMCMP(out, sigPh1, sizeof(sigPh1)))
        return -11603;

#if defined(HAVE_ED448_VERIFY)
    /* test verify on good msg */
    if (wc_ed448ph_verify_msg(out, outlen, msgPh, sizeof(msgPh), &verify, &key,
                              NULL, 0) != 0 || verify != 1) {
        return -11604;
    }
#endif

    if (wc_ed448ph_sign_msg(msgPh, sizeof(msgPh), out, &outlen, &key,
                            contextPh2, sizeof(contextPh2)) != 0) {
        return -11605;
    }

    if (XMEMCMP(out, sigPh2, sizeof(sigPh2)))
        return -11606;

#if defined(HAVE_ED448_VERIFY)
    /* test verify on good msg */
    if (wc_ed448ph_verify_msg(out, outlen, msgPh, sizeof(msgPh), &verify, &key,
                              contextPh2, sizeof(contextPh2)) != 0 ||
            verify != 1) {
        return -11607;
    }
#endif

    if (wc_ed448ph_sign_hash(hashPh, sizeof(hashPh), out, &outlen, &key, NULL,
                             0) != 0) {
        return -11608;
    }

    if (XMEMCMP(out, sigPh1, sizeof(sigPh1)))
        return -11609;

#if defined(HAVE_ED448_VERIFY)
    if (wc_ed448ph_verify_hash(out, outlen, hashPh, sizeof(hashPh), &verify,
                               &key, NULL, 0) != 0 || verify != 1) {
        return -11610;
    }
#endif

    if (wc_ed448ph_sign_hash(hashPh, sizeof(hashPh), out, &outlen, &key,
                             contextPh2, sizeof(contextPh2)) != 0) {
        return -11611;
    }

    if (XMEMCMP(out, sigPh2, sizeof(sigPh2)))
        return -11612;

#if defined(HAVE_ED448_VERIFY)
    if (wc_ed448ph_verify_hash(out, outlen, hashPh, sizeof(hashPh), &verify,
                               &key, contextPh2, sizeof(contextPh2)) != 0 ||
            verify != 1) {
        return -11613;
    }
#endif

    wc_ed448_free(&key);

    return 0;
}
#endif /* HAVE_ED448_SIGN && HAVE_ED448_KEY_EXPORT && HAVE_ED448_KEY_IMPORT */

WOLFSSL_TEST_SUBROUTINE int ed448_test(void)
{
    int ret;
    WC_RNG rng;
#if defined(HAVE_ED448_SIGN) && defined(HAVE_ED448_KEY_EXPORT) &&\
    defined(HAVE_ED448_KEY_IMPORT)
    byte   out[ED448_SIG_SIZE];
    int    i;
    word32 outlen;
#ifdef HAVE_ED448_VERIFY
#ifdef WOLFSSL_ED448_STREAMING_VERIFY
    int    j;
#endif /* WOLFSSL_ED448_STREAMING_VERIFY */
    int    verify;
#endif /* HAVE_ED448_VERIFY */
#endif /* HAVE_ED448_SIGN && HAVE_ED448_KEY_EXPORT && HAVE_ED448_KEY_IMPORT */
    word32 keySz, sigSz;
#ifdef WOLFSSL_SMALL_STACK
    ed448_key *key = NULL;
    ed448_key *key2 = NULL;
#else
    ed448_key key[1];
    ed448_key key2[1];
#endif

#if defined(HAVE_ED448_SIGN) && defined(HAVE_ED448_KEY_EXPORT) && \
    defined(HAVE_ED448_KEY_IMPORT)
    /* test vectors from
       https://tools.ietf.org/html/rfc8032
     */

    WOLFSSL_SMALL_STACK_STATIC const byte sKey1[] = {
        0x6c, 0x82, 0xa5, 0x62, 0xcb, 0x80, 0x8d, 0x10,
        0xd6, 0x32, 0xbe, 0x89, 0xc8, 0x51, 0x3e, 0xbf,
        0x6c, 0x92, 0x9f, 0x34, 0xdd, 0xfa, 0x8c, 0x9f,
        0x63, 0xc9, 0x96, 0x0e, 0xf6, 0xe3, 0x48, 0xa3,
        0x52, 0x8c, 0x8a, 0x3f, 0xcc, 0x2f, 0x04, 0x4e,
        0x39, 0xa3, 0xfc, 0x5b, 0x94, 0x49, 0x2f, 0x8f,
        0x03, 0x2e, 0x75, 0x49, 0xa2, 0x00, 0x98, 0xf9,
        0x5b
    };

    WOLFSSL_SMALL_STACK_STATIC const byte sKey2[] = {
        0xc4, 0xea, 0xb0, 0x5d, 0x35, 0x70, 0x07, 0xc6,
        0x32, 0xf3, 0xdb, 0xb4, 0x84, 0x89, 0x92, 0x4d,
        0x55, 0x2b, 0x08, 0xfe, 0x0c, 0x35, 0x3a, 0x0d,
        0x4a, 0x1f, 0x00, 0xac, 0xda, 0x2c, 0x46, 0x3a,
        0xfb, 0xea, 0x67, 0xc5, 0xe8, 0xd2, 0x87, 0x7c,
        0x5e, 0x3b, 0xc3, 0x97, 0xa6, 0x59, 0x94, 0x9e,
        0xf8, 0x02, 0x1e, 0x95, 0x4e, 0x0a, 0x12, 0x27,
        0x4e
    };

    WOLFSSL_SMALL_STACK_STATIC const byte sKey3[] = {
        0x25, 0x8c, 0xdd, 0x4a, 0xda, 0x32, 0xed, 0x9c,
        0x9f, 0xf5, 0x4e, 0x63, 0x75, 0x6a, 0xe5, 0x82,
        0xfb, 0x8f, 0xab, 0x2a, 0xc7, 0x21, 0xf2, 0xc8,
        0xe6, 0x76, 0xa7, 0x27, 0x68, 0x51, 0x3d, 0x93,
        0x9f, 0x63, 0xdd, 0xdb, 0x55, 0x60, 0x91, 0x33,
        0xf2, 0x9a, 0xdf, 0x86, 0xec, 0x99, 0x29, 0xdc,
        0xcb, 0x52, 0xc1, 0xc5, 0xfd, 0x2f, 0xf7, 0xe2,
        0x1b
    };

    /* uncompressed test */
    WOLFSSL_SMALL_STACK_STATIC const byte sKey4[] = {
        0x6c, 0x82, 0xa5, 0x62, 0xcb, 0x80, 0x8d, 0x10,
        0xd6, 0x32, 0xbe, 0x89, 0xc8, 0x51, 0x3e, 0xbf,
        0x6c, 0x92, 0x9f, 0x34, 0xdd, 0xfa, 0x8c, 0x9f,
        0x63, 0xc9, 0x96, 0x0e, 0xf6, 0xe3, 0x48, 0xa3,
        0x52, 0x8c, 0x8a, 0x3f, 0xcc, 0x2f, 0x04, 0x4e,
        0x39, 0xa3, 0xfc, 0x5b, 0x94, 0x49, 0x2f, 0x8f,
        0x03, 0x2e, 0x75, 0x49, 0xa2, 0x00, 0x98, 0xf9,
        0x5b
    };

    /* compressed prefix test */
    WOLFSSL_SMALL_STACK_STATIC const byte sKey5[] = {
        0x6c, 0x82, 0xa5, 0x62, 0xcb, 0x80, 0x8d, 0x10,
        0xd6, 0x32, 0xbe, 0x89, 0xc8, 0x51, 0x3e, 0xbf,
        0x6c, 0x92, 0x9f, 0x34, 0xdd, 0xfa, 0x8c, 0x9f,
        0x63, 0xc9, 0x96, 0x0e, 0xf6, 0xe3, 0x48, 0xa3,
        0x52, 0x8c, 0x8a, 0x3f, 0xcc, 0x2f, 0x04, 0x4e,
        0x39, 0xa3, 0xfc, 0x5b, 0x94, 0x49, 0x2f, 0x8f,
        0x03, 0x2e, 0x75, 0x49, 0xa2, 0x00, 0x98, 0xf9,
        0x5b
    };

    WOLFSSL_SMALL_STACK_STATIC const byte sKey6[] = {
        0x87, 0x2d, 0x09, 0x37, 0x80, 0xf5, 0xd3, 0x73,
        0x0d, 0xf7, 0xc2, 0x12, 0x66, 0x4b, 0x37, 0xb8,
        0xa0, 0xf2, 0x4f, 0x56, 0x81, 0x0d, 0xaa, 0x83,
        0x82, 0xcd, 0x4f, 0xa3, 0xf7, 0x76, 0x34, 0xec,
        0x44, 0xdc, 0x54, 0xf1, 0xc2, 0xed, 0x9b, 0xea,
        0x86, 0xfa, 0xfb, 0x76, 0x32, 0xd8, 0xbe, 0x19,
        0x9e, 0xa1, 0x65, 0xf5, 0xad, 0x55, 0xdd, 0x9c,
        0xe8
    };

    WOLFSSL_SMALL_STACK_STATIC const byte* sKeys[] = {sKey1, sKey2, sKey3, sKey4, sKey5, sKey6};

    WOLFSSL_SMALL_STACK_STATIC const byte pKey1[] = {
        0x5f, 0xd7, 0x44, 0x9b, 0x59, 0xb4, 0x61, 0xfd,
        0x2c, 0xe7, 0x87, 0xec, 0x61, 0x6a, 0xd4, 0x6a,
        0x1d, 0xa1, 0x34, 0x24, 0x85, 0xa7, 0x0e, 0x1f,
        0x8a, 0x0e, 0xa7, 0x5d, 0x80, 0xe9, 0x67, 0x78,
        0xed, 0xf1, 0x24, 0x76, 0x9b, 0x46, 0xc7, 0x06,
        0x1b, 0xd6, 0x78, 0x3d, 0xf1, 0xe5, 0x0f, 0x6c,
        0xd1, 0xfa, 0x1a, 0xbe, 0xaf, 0xe8, 0x25, 0x61,
        0x80
    };

    WOLFSSL_SMALL_STACK_STATIC const byte pKey2[] = {
        0x43, 0xba, 0x28, 0xf4, 0x30, 0xcd, 0xff, 0x45,
        0x6a, 0xe5, 0x31, 0x54, 0x5f, 0x7e, 0xcd, 0x0a,
        0xc8, 0x34, 0xa5, 0x5d, 0x93, 0x58, 0xc0, 0x37,
        0x2b, 0xfa, 0x0c, 0x6c, 0x67, 0x98, 0xc0, 0x86,
        0x6a, 0xea, 0x01, 0xeb, 0x00, 0x74, 0x28, 0x02,
        0xb8, 0x43, 0x8e, 0xa4, 0xcb, 0x82, 0x16, 0x9c,
        0x23, 0x51, 0x60, 0x62, 0x7b, 0x4c, 0x3a, 0x94,
        0x80
    };

    WOLFSSL_SMALL_STACK_STATIC const byte pKey3[] = {
        0x3b, 0xa1, 0x6d, 0xa0, 0xc6, 0xf2, 0xcc, 0x1f,
        0x30, 0x18, 0x77, 0x40, 0x75, 0x6f, 0x5e, 0x79,
        0x8d, 0x6b, 0xc5, 0xfc, 0x01, 0x5d, 0x7c, 0x63,
        0xcc, 0x95, 0x10, 0xee, 0x3f, 0xd4, 0x4a, 0xdc,
        0x24, 0xd8, 0xe9, 0x68, 0xb6, 0xe4, 0x6e, 0x6f,
        0x94, 0xd1, 0x9b, 0x94, 0x53, 0x61, 0x72, 0x6b,
        0xd7, 0x5e, 0x14, 0x9e, 0xf0, 0x98, 0x17, 0xf5,
        0x80
    };

    /* uncompressed test */
    WOLFSSL_SMALL_STACK_STATIC const byte pKey4[] = {
        0x5f, 0xd7, 0x44, 0x9b, 0x59, 0xb4, 0x61, 0xfd,
        0x2c, 0xe7, 0x87, 0xec, 0x61, 0x6a, 0xd4, 0x6a,
        0x1d, 0xa1, 0x34, 0x24, 0x85, 0xa7, 0x0e, 0x1f,
        0x8a, 0x0e, 0xa7, 0x5d, 0x80, 0xe9, 0x67, 0x78,
        0xed, 0xf1, 0x24, 0x76, 0x9b, 0x46, 0xc7, 0x06,
        0x1b, 0xd6, 0x78, 0x3d, 0xf1, 0xe5, 0x0f, 0x6c,
        0xd1, 0xfa, 0x1a, 0xbe, 0xaf, 0xe8, 0x25, 0x61,
        0x80
    };

    /* compressed prefix */
    WOLFSSL_SMALL_STACK_STATIC const byte pKey5[] = {
        0x5f, 0xd7, 0x44, 0x9b, 0x59, 0xb4, 0x61, 0xfd,
        0x2c, 0xe7, 0x87, 0xec, 0x61, 0x6a, 0xd4, 0x6a,
        0x1d, 0xa1, 0x34, 0x24, 0x85, 0xa7, 0x0e, 0x1f,
        0x8a, 0x0e, 0xa7, 0x5d, 0x80, 0xe9, 0x67, 0x78,
        0xed, 0xf1, 0x24, 0x76, 0x9b, 0x46, 0xc7, 0x06,
        0x1b, 0xd6, 0x78, 0x3d, 0xf1, 0xe5, 0x0f, 0x6c,
        0xd1, 0xfa, 0x1a, 0xbe, 0xaf, 0xe8, 0x25, 0x61,
        0x80
    };

    WOLFSSL_SMALL_STACK_STATIC const byte pKey6[] = {
        0xa8, 0x1b, 0x2e, 0x8a, 0x70, 0xa5, 0xac, 0x94,
        0xff, 0xdb, 0xcc, 0x9b, 0xad, 0xfc, 0x3f, 0xeb,
        0x08, 0x01, 0xf2, 0x58, 0x57, 0x8b, 0xb1, 0x14,
        0xad, 0x44, 0xec, 0xe1, 0xec, 0x0e, 0x79, 0x9d,
        0xa0, 0x8e, 0xff, 0xb8, 0x1c, 0x5d, 0x68, 0x5c,
        0x0c, 0x56, 0xf6, 0x4e, 0xec, 0xae, 0xf8, 0xcd,
        0xf1, 0x1c, 0xc3, 0x87, 0x37, 0x83, 0x8c, 0xf4,
        0x00
    };

    WOLFSSL_SMALL_STACK_STATIC const byte* pKeys[] = {pKey1, pKey2, pKey3, pKey4, pKey5, pKey6};
    WOLFSSL_SMALL_STACK_STATIC const byte  pKeySz[] = {sizeof(pKey1), sizeof(pKey2), sizeof(pKey3),
                            sizeof(pKey4), sizeof(pKey5), sizeof(pKey6)};

    WOLFSSL_SMALL_STACK_STATIC const byte sig1[] = {
        0x53, 0x3a, 0x37, 0xf6, 0xbb, 0xe4, 0x57, 0x25,
        0x1f, 0x02, 0x3c, 0x0d, 0x88, 0xf9, 0x76, 0xae,
        0x2d, 0xfb, 0x50, 0x4a, 0x84, 0x3e, 0x34, 0xd2,
        0x07, 0x4f, 0xd8, 0x23, 0xd4, 0x1a, 0x59, 0x1f,
        0x2b, 0x23, 0x3f, 0x03, 0x4f, 0x62, 0x82, 0x81,
        0xf2, 0xfd, 0x7a, 0x22, 0xdd, 0xd4, 0x7d, 0x78,
        0x28, 0xc5, 0x9b, 0xd0, 0xa2, 0x1b, 0xfd, 0x39,
        0x80, 0xff, 0x0d, 0x20, 0x28, 0xd4, 0xb1, 0x8a,
        0x9d, 0xf6, 0x3e, 0x00, 0x6c, 0x5d, 0x1c, 0x2d,
        0x34, 0x5b, 0x92, 0x5d, 0x8d, 0xc0, 0x0b, 0x41,
        0x04, 0x85, 0x2d, 0xb9, 0x9a, 0xc5, 0xc7, 0xcd,
        0xda, 0x85, 0x30, 0xa1, 0x13, 0xa0, 0xf4, 0xdb,
        0xb6, 0x11, 0x49, 0xf0, 0x5a, 0x73, 0x63, 0x26,
        0x8c, 0x71, 0xd9, 0x58, 0x08, 0xff, 0x2e, 0x65,
        0x26, 0x00
    };

    WOLFSSL_SMALL_STACK_STATIC const byte sig2[] = {
        0x26, 0xb8, 0xf9, 0x17, 0x27, 0xbd, 0x62, 0x89,
        0x7a, 0xf1, 0x5e, 0x41, 0xeb, 0x43, 0xc3, 0x77,
        0xef, 0xb9, 0xc6, 0x10, 0xd4, 0x8f, 0x23, 0x35,
        0xcb, 0x0b, 0xd0, 0x08, 0x78, 0x10, 0xf4, 0x35,
        0x25, 0x41, 0xb1, 0x43, 0xc4, 0xb9, 0x81, 0xb7,
        0xe1, 0x8f, 0x62, 0xde, 0x8c, 0xcd, 0xf6, 0x33,
        0xfc, 0x1b, 0xf0, 0x37, 0xab, 0x7c, 0xd7, 0x79,
        0x80, 0x5e, 0x0d, 0xbc, 0xc0, 0xaa, 0xe1, 0xcb,
        0xce, 0xe1, 0xaf, 0xb2, 0xe0, 0x27, 0xdf, 0x36,
        0xbc, 0x04, 0xdc, 0xec, 0xbf, 0x15, 0x43, 0x36,
        0xc1, 0x9f, 0x0a, 0xf7, 0xe0, 0xa6, 0x47, 0x29,
        0x05, 0xe7, 0x99, 0xf1, 0x95, 0x3d, 0x2a, 0x0f,
        0xf3, 0x34, 0x8a, 0xb2, 0x1a, 0xa4, 0xad, 0xaf,
        0xd1, 0xd2, 0x34, 0x44, 0x1c, 0xf8, 0x07, 0xc0,
        0x3a, 0x00
    };

    WOLFSSL_SMALL_STACK_STATIC const byte sig3[] = {
        0x7e, 0xee, 0xab, 0x7c, 0x4e, 0x50, 0xfb, 0x79,
        0x9b, 0x41, 0x8e, 0xe5, 0xe3, 0x19, 0x7f, 0xf6,
        0xbf, 0x15, 0xd4, 0x3a, 0x14, 0xc3, 0x43, 0x89,
        0xb5, 0x9d, 0xd1, 0xa7, 0xb1, 0xb8, 0x5b, 0x4a,
        0xe9, 0x04, 0x38, 0xac, 0xa6, 0x34, 0xbe, 0xa4,
        0x5e, 0x3a, 0x26, 0x95, 0xf1, 0x27, 0x0f, 0x07,
        0xfd, 0xcd, 0xf7, 0xc6, 0x2b, 0x8e, 0xfe, 0xaf,
        0x00, 0xb4, 0x5c, 0x2c, 0x96, 0xba, 0x45, 0x7e,
        0xb1, 0xa8, 0xbf, 0x07, 0x5a, 0x3d, 0xb2, 0x8e,
        0x5c, 0x24, 0xf6, 0xb9, 0x23, 0xed, 0x4a, 0xd7,
        0x47, 0xc3, 0xc9, 0xe0, 0x3c, 0x70, 0x79, 0xef,
        0xb8, 0x7c, 0xb1, 0x10, 0xd3, 0xa9, 0x98, 0x61,
        0xe7, 0x20, 0x03, 0xcb, 0xae, 0x6d, 0x6b, 0x8b,
        0x82, 0x7e, 0x4e, 0x6c, 0x14, 0x30, 0x64, 0xff,
        0x3c, 0x00
    };

    /* uncompressed test */
    WOLFSSL_SMALL_STACK_STATIC const byte sig4[] = {
        0x53, 0x3a, 0x37, 0xf6, 0xbb, 0xe4, 0x57, 0x25,
        0x1f, 0x02, 0x3c, 0x0d, 0x88, 0xf9, 0x76, 0xae,
        0x2d, 0xfb, 0x50, 0x4a, 0x84, 0x3e, 0x34, 0xd2,
        0x07, 0x4f, 0xd8, 0x23, 0xd4, 0x1a, 0x59, 0x1f,
        0x2b, 0x23, 0x3f, 0x03, 0x4f, 0x62, 0x82, 0x81,
        0xf2, 0xfd, 0x7a, 0x22, 0xdd, 0xd4, 0x7d, 0x78,
        0x28, 0xc5, 0x9b, 0xd0, 0xa2, 0x1b, 0xfd, 0x39,
        0x80, 0xff, 0x0d, 0x20, 0x28, 0xd4, 0xb1, 0x8a,
        0x9d, 0xf6, 0x3e, 0x00, 0x6c, 0x5d, 0x1c, 0x2d,
        0x34, 0x5b, 0x92, 0x5d, 0x8d, 0xc0, 0x0b, 0x41,
        0x04, 0x85, 0x2d, 0xb9, 0x9a, 0xc5, 0xc7, 0xcd,
        0xda, 0x85, 0x30, 0xa1, 0x13, 0xa0, 0xf4, 0xdb,
        0xb6, 0x11, 0x49, 0xf0, 0x5a, 0x73, 0x63, 0x26,
        0x8c, 0x71, 0xd9, 0x58, 0x08, 0xff, 0x2e, 0x65,
        0x26, 0x00
    };

    /* compressed prefix */
    WOLFSSL_SMALL_STACK_STATIC const byte sig5[] = {
        0x53, 0x3a, 0x37, 0xf6, 0xbb, 0xe4, 0x57, 0x25,
        0x1f, 0x02, 0x3c, 0x0d, 0x88, 0xf9, 0x76, 0xae,
        0x2d, 0xfb, 0x50, 0x4a, 0x84, 0x3e, 0x34, 0xd2,
        0x07, 0x4f, 0xd8, 0x23, 0xd4, 0x1a, 0x59, 0x1f,
        0x2b, 0x23, 0x3f, 0x03, 0x4f, 0x62, 0x82, 0x81,
        0xf2, 0xfd, 0x7a, 0x22, 0xdd, 0xd4, 0x7d, 0x78,
        0x28, 0xc5, 0x9b, 0xd0, 0xa2, 0x1b, 0xfd, 0x39,
        0x80, 0xff, 0x0d, 0x20, 0x28, 0xd4, 0xb1, 0x8a,
        0x9d, 0xf6, 0x3e, 0x00, 0x6c, 0x5d, 0x1c, 0x2d,
        0x34, 0x5b, 0x92, 0x5d, 0x8d, 0xc0, 0x0b, 0x41,
        0x04, 0x85, 0x2d, 0xb9, 0x9a, 0xc5, 0xc7, 0xcd,
        0xda, 0x85, 0x30, 0xa1, 0x13, 0xa0, 0xf4, 0xdb,
        0xb6, 0x11, 0x49, 0xf0, 0x5a, 0x73, 0x63, 0x26,
        0x8c, 0x71, 0xd9, 0x58, 0x08, 0xff, 0x2e, 0x65,
        0x26, 0x00
    };

    WOLFSSL_SMALL_STACK_STATIC const byte sig6[] = {
        0xe3, 0x01, 0x34, 0x5a, 0x41, 0xa3, 0x9a, 0x4d,
        0x72, 0xff, 0xf8, 0xdf, 0x69, 0xc9, 0x80, 0x75,
        0xa0, 0xcc, 0x08, 0x2b, 0x80, 0x2f, 0xc9, 0xb2,
        0xb6, 0xbc, 0x50, 0x3f, 0x92, 0x6b, 0x65, 0xbd,
        0xdf, 0x7f, 0x4c, 0x8f, 0x1c, 0xb4, 0x9f, 0x63,
        0x96, 0xaf, 0xc8, 0xa7, 0x0a, 0xbe, 0x6d, 0x8a,
        0xef, 0x0d, 0xb4, 0x78, 0xd4, 0xc6, 0xb2, 0x97,
        0x00, 0x76, 0xc6, 0xa0, 0x48, 0x4f, 0xe7, 0x6d,
        0x76, 0xb3, 0xa9, 0x76, 0x25, 0xd7, 0x9f, 0x1c,
        0xe2, 0x40, 0xe7, 0xc5, 0x76, 0x75, 0x0d, 0x29,
        0x55, 0x28, 0x28, 0x6f, 0x71, 0x9b, 0x41, 0x3d,
        0xe9, 0xad, 0xa3, 0xe8, 0xeb, 0x78, 0xed, 0x57,
        0x36, 0x03, 0xce, 0x30, 0xd8, 0xbb, 0x76, 0x17,
        0x85, 0xdc, 0x30, 0xdb, 0xc3, 0x20, 0x86, 0x9e,
        0x1a, 0x00
    };

    WOLFSSL_SMALL_STACK_STATIC const byte* sigs[] = {sig1, sig2, sig3, sig4, sig5, sig6};
    #define SIGSZ sizeof(sig1)

    PEDANTIC_EXTENSION WOLFSSL_SMALL_STACK_STATIC const byte msg1[]  = { };
    WOLFSSL_SMALL_STACK_STATIC const byte msg2[]  = { 0x03 };
    WOLFSSL_SMALL_STACK_STATIC const byte msg3[]  = { 0x64, 0xa6, 0x5f, 0x3c, 0xde, 0xdc, 0xdd,
                                  0x66, 0x81, 0x1e, 0x29, 0x15 };

    /* test of a 1023 byte long message */
    WOLFSSL_SMALL_STACK_STATIC const byte msg4[]  = {
        0x6d, 0xdf, 0x80, 0x2e, 0x1a, 0xae, 0x49, 0x86,
        0x93, 0x5f, 0x7f, 0x98, 0x1b, 0xa3, 0xf0, 0x35,
        0x1d, 0x62, 0x73, 0xc0, 0xa0, 0xc2, 0x2c, 0x9c,
        0x0e, 0x83, 0x39, 0x16, 0x8e, 0x67, 0x54, 0x12,
        0xa3, 0xde, 0xbf, 0xaf, 0x43, 0x5e, 0xd6, 0x51,
        0x55, 0x80, 0x07, 0xdb, 0x43, 0x84, 0xb6, 0x50,
        0xfc, 0xc0, 0x7e, 0x3b, 0x58, 0x6a, 0x27, 0xa4,
        0xf7, 0xa0, 0x0a, 0xc8, 0xa6, 0xfe, 0xc2, 0xcd,
        0x86, 0xae, 0x4b, 0xf1, 0x57, 0x0c, 0x41, 0xe6,
        0xa4, 0x0c, 0x93, 0x1d, 0xb2, 0x7b, 0x2f, 0xaa,
        0x15, 0xa8, 0xce, 0xdd, 0x52, 0xcf, 0xf7, 0x36,
        0x2c, 0x4e, 0x6e, 0x23, 0xda, 0xec, 0x0f, 0xbc,
        0x3a, 0x79, 0xb6, 0x80, 0x6e, 0x31, 0x6e, 0xfc,
        0xc7, 0xb6, 0x81, 0x19, 0xbf, 0x46, 0xbc, 0x76,
        0xa2, 0x60, 0x67, 0xa5, 0x3f, 0x29, 0x6d, 0xaf,
        0xdb, 0xdc, 0x11, 0xc7, 0x7f, 0x77, 0x77, 0xe9,
        0x72, 0x66, 0x0c, 0xf4, 0xb6, 0xa9, 0xb3, 0x69,
        0xa6, 0x66, 0x5f, 0x02, 0xe0, 0xcc, 0x9b, 0x6e,
        0xdf, 0xad, 0x13, 0x6b, 0x4f, 0xab, 0xe7, 0x23,
        0xd2, 0x81, 0x3d, 0xb3, 0x13, 0x6c, 0xfd, 0xe9,
        0xb6, 0xd0, 0x44, 0x32, 0x2f, 0xee, 0x29, 0x47,
        0x95, 0x2e, 0x03, 0x1b, 0x73, 0xab, 0x5c, 0x60,
        0x33, 0x49, 0xb3, 0x07, 0xbd, 0xc2, 0x7b, 0xc6,
        0xcb, 0x8b, 0x8b, 0xbd, 0x7b, 0xd3, 0x23, 0x21,
        0x9b, 0x80, 0x33, 0xa5, 0x81, 0xb5, 0x9e, 0xad,
        0xeb, 0xb0, 0x9b, 0x3c, 0x4f, 0x3d, 0x22, 0x77,
        0xd4, 0xf0, 0x34, 0x36, 0x24, 0xac, 0xc8, 0x17,
        0x80, 0x47, 0x28, 0xb2, 0x5a, 0xb7, 0x97, 0x17,
        0x2b, 0x4c, 0x5c, 0x21, 0xa2, 0x2f, 0x9c, 0x78,
        0x39, 0xd6, 0x43, 0x00, 0x23, 0x2e, 0xb6, 0x6e,
        0x53, 0xf3, 0x1c, 0x72, 0x3f, 0xa3, 0x7f, 0xe3,
        0x87, 0xc7, 0xd3, 0xe5, 0x0b, 0xdf, 0x98, 0x13,
        0xa3, 0x0e, 0x5b, 0xb1, 0x2c, 0xf4, 0xcd, 0x93,
        0x0c, 0x40, 0xcf, 0xb4, 0xe1, 0xfc, 0x62, 0x25,
        0x92, 0xa4, 0x95, 0x88, 0x79, 0x44, 0x94, 0xd5,
        0x6d, 0x24, 0xea, 0x4b, 0x40, 0xc8, 0x9f, 0xc0,
        0x59, 0x6c, 0xc9, 0xeb, 0xb9, 0x61, 0xc8, 0xcb,
        0x10, 0xad, 0xde, 0x97, 0x6a, 0x5d, 0x60, 0x2b,
        0x1c, 0x3f, 0x85, 0xb9, 0xb9, 0xa0, 0x01, 0xed,
        0x3c, 0x6a, 0x4d, 0x3b, 0x14, 0x37, 0xf5, 0x20,
        0x96, 0xcd, 0x19, 0x56, 0xd0, 0x42, 0xa5, 0x97,
        0xd5, 0x61, 0xa5, 0x96, 0xec, 0xd3, 0xd1, 0x73,
        0x5a, 0x8d, 0x57, 0x0e, 0xa0, 0xec, 0x27, 0x22,
        0x5a, 0x2c, 0x4a, 0xaf, 0xf2, 0x63, 0x06, 0xd1,
        0x52, 0x6c, 0x1a, 0xf3, 0xca, 0x6d, 0x9c, 0xf5,
        0xa2, 0xc9, 0x8f, 0x47, 0xe1, 0xc4, 0x6d, 0xb9,
        0xa3, 0x32, 0x34, 0xcf, 0xd4, 0xd8, 0x1f, 0x2c,
        0x98, 0x53, 0x8a, 0x09, 0xeb, 0xe7, 0x69, 0x98,
        0xd0, 0xd8, 0xfd, 0x25, 0x99, 0x7c, 0x7d, 0x25,
        0x5c, 0x6d, 0x66, 0xec, 0xe6, 0xfa, 0x56, 0xf1,
        0x11, 0x44, 0x95, 0x0f, 0x02, 0x77, 0x95, 0xe6,
        0x53, 0x00, 0x8f, 0x4b, 0xd7, 0xca, 0x2d, 0xee,
        0x85, 0xd8, 0xe9, 0x0f, 0x3d, 0xc3, 0x15, 0x13,
        0x0c, 0xe2, 0xa0, 0x03, 0x75, 0xa3, 0x18, 0xc7,
        0xc3, 0xd9, 0x7b, 0xe2, 0xc8, 0xce, 0x5b, 0x6d,
        0xb4, 0x1a, 0x62, 0x54, 0xff, 0x26, 0x4f, 0xa6,
        0x15, 0x5b, 0xae, 0xe3, 0xb0, 0x77, 0x3c, 0x0f,
        0x49, 0x7c, 0x57, 0x3f, 0x19, 0xbb, 0x4f, 0x42,
        0x40, 0x28, 0x1f, 0x0b, 0x1f, 0x4f, 0x7b, 0xe8,
        0x57, 0xa4, 0xe5, 0x9d, 0x41, 0x6c, 0x06, 0xb4,
        0xc5, 0x0f, 0xa0, 0x9e, 0x18, 0x10, 0xdd, 0xc6,
        0xb1, 0x46, 0x7b, 0xae, 0xac, 0x5a, 0x36, 0x68,
        0xd1, 0x1b, 0x6e, 0xca, 0xa9, 0x01, 0x44, 0x00,
        0x16, 0xf3, 0x89, 0xf8, 0x0a, 0xcc, 0x4d, 0xb9,
        0x77, 0x02, 0x5e, 0x7f, 0x59, 0x24, 0x38, 0x8c,
        0x7e, 0x34, 0x0a, 0x73, 0x2e, 0x55, 0x44, 0x40,
        0xe7, 0x65, 0x70, 0xf8, 0xdd, 0x71, 0xb7, 0xd6,
        0x40, 0xb3, 0x45, 0x0d, 0x1f, 0xd5, 0xf0, 0x41,
        0x0a, 0x18, 0xf9, 0xa3, 0x49, 0x4f, 0x70, 0x7c,
        0x71, 0x7b, 0x79, 0xb4, 0xbf, 0x75, 0xc9, 0x84,
        0x00, 0xb0, 0x96, 0xb2, 0x16, 0x53, 0xb5, 0xd2,
        0x17, 0xcf, 0x35, 0x65, 0xc9, 0x59, 0x74, 0x56,
        0xf7, 0x07, 0x03, 0x49, 0x7a, 0x07, 0x87, 0x63,
        0x82, 0x9b, 0xc0, 0x1b, 0xb1, 0xcb, 0xc8, 0xfa,
        0x04, 0xea, 0xdc, 0x9a, 0x6e, 0x3f, 0x66, 0x99,
        0x58, 0x7a, 0x9e, 0x75, 0xc9, 0x4e, 0x5b, 0xab,
        0x00, 0x36, 0xe0, 0xb2, 0xe7, 0x11, 0x39, 0x2c,
        0xff, 0x00, 0x47, 0xd0, 0xd6, 0xb0, 0x5b, 0xd2,
        0xa5, 0x88, 0xbc, 0x10, 0x97, 0x18, 0x95, 0x42,
        0x59, 0xf1, 0xd8, 0x66, 0x78, 0xa5, 0x79, 0xa3,
        0x12, 0x0f, 0x19, 0xcf, 0xb2, 0x96, 0x3f, 0x17,
        0x7a, 0xeb, 0x70, 0xf2, 0xd4, 0x84, 0x48, 0x26,
        0x26, 0x2e, 0x51, 0xb8, 0x02, 0x71, 0x27, 0x20,
        0x68, 0xef, 0x5b, 0x38, 0x56, 0xfa, 0x85, 0x35,
        0xaa, 0x2a, 0x88, 0xb2, 0xd4, 0x1f, 0x2a, 0x0e,
        0x2f, 0xda, 0x76, 0x24, 0xc2, 0x85, 0x02, 0x72,
        0xac, 0x4a, 0x2f, 0x56, 0x1f, 0x8f, 0x2f, 0x7a,
        0x31, 0x8b, 0xfd, 0x5c, 0xaf, 0x96, 0x96, 0x14,
        0x9e, 0x4a, 0xc8, 0x24, 0xad, 0x34, 0x60, 0x53,
        0x8f, 0xdc, 0x25, 0x42, 0x1b, 0xee, 0xc2, 0xcc,
        0x68, 0x18, 0x16, 0x2d, 0x06, 0xbb, 0xed, 0x0c,
        0x40, 0xa3, 0x87, 0x19, 0x23, 0x49, 0xdb, 0x67,
        0xa1, 0x18, 0xba, 0xda, 0x6c, 0xd5, 0xab, 0x01,
        0x40, 0xee, 0x27, 0x32, 0x04, 0xf6, 0x28, 0xaa,
        0xd1, 0xc1, 0x35, 0xf7, 0x70, 0x27, 0x9a, 0x65,
        0x1e, 0x24, 0xd8, 0xc1, 0x4d, 0x75, 0xa6, 0x05,
        0x9d, 0x76, 0xb9, 0x6a, 0x6f, 0xd8, 0x57, 0xde,
        0xf5, 0xe0, 0xb3, 0x54, 0xb2, 0x7a, 0xb9, 0x37,
        0xa5, 0x81, 0x5d, 0x16, 0xb5, 0xfa, 0xe4, 0x07,
        0xff, 0x18, 0x22, 0x2c, 0x6d, 0x1e, 0xd2, 0x63,
        0xbe, 0x68, 0xc9, 0x5f, 0x32, 0xd9, 0x08, 0xbd,
        0x89, 0x5c, 0xd7, 0x62, 0x07, 0xae, 0x72, 0x64,
        0x87, 0x56, 0x7f, 0x9a, 0x67, 0xda, 0xd7, 0x9a,
        0xbe, 0xc3, 0x16, 0xf6, 0x83, 0xb1, 0x7f, 0x2d,
        0x02, 0xbf, 0x07, 0xe0, 0xac, 0x8b, 0x5b, 0xc6,
        0x16, 0x2c, 0xf9, 0x46, 0x97, 0xb3, 0xc2, 0x7c,
        0xd1, 0xfe, 0xa4, 0x9b, 0x27, 0xf2, 0x3b, 0xa2,
        0x90, 0x18, 0x71, 0x96, 0x25, 0x06, 0x52, 0x0c,
        0x39, 0x2d, 0xa8, 0xb6, 0xad, 0x0d, 0x99, 0xf7,
        0x01, 0x3f, 0xbc, 0x06, 0xc2, 0xc1, 0x7a, 0x56,
        0x95, 0x00, 0xc8, 0xa7, 0x69, 0x64, 0x81, 0xc1,
        0xcd, 0x33, 0xe9, 0xb1, 0x4e, 0x40, 0xb8, 0x2e,
        0x79, 0xa5, 0xf5, 0xdb, 0x82, 0x57, 0x1b, 0xa9,
        0x7b, 0xae, 0x3a, 0xd3, 0xe0, 0x47, 0x95, 0x15,
        0xbb, 0x0e, 0x2b, 0x0f, 0x3b, 0xfc, 0xd1, 0xfd,
        0x33, 0x03, 0x4e, 0xfc, 0x62, 0x45, 0xed, 0xdd,
        0x7e, 0xe2, 0x08, 0x6d, 0xda, 0xe2, 0x60, 0x0d,
        0x8c, 0xa7, 0x3e, 0x21, 0x4e, 0x8c, 0x2b, 0x0b,
        0xdb, 0x2b, 0x04, 0x7c, 0x6a, 0x46, 0x4a, 0x56,
        0x2e, 0xd7, 0x7b, 0x73, 0xd2, 0xd8, 0x41, 0xc4,
        0xb3, 0x49, 0x73, 0x55, 0x12, 0x57, 0x71, 0x3b,
        0x75, 0x36, 0x32, 0xef, 0xba, 0x34, 0x81, 0x69,
        0xab, 0xc9, 0x0a, 0x68, 0xf4, 0x26, 0x11, 0xa4,
        0x01, 0x26, 0xd7, 0xcb, 0x21, 0xb5, 0x86, 0x95,
        0x56, 0x81, 0x86, 0xf7, 0xe5, 0x69, 0xd2, 0xff,
        0x0f, 0x9e, 0x74, 0x5d, 0x04, 0x87, 0xdd, 0x2e,
        0xb9, 0x97, 0xca, 0xfc, 0x5a, 0xbf, 0x9d, 0xd1,
        0x02, 0xe6, 0x2f, 0xf6, 0x6c, 0xba, 0x87
    };

    WOLFSSL_SMALL_STACK_STATIC const byte* msgs[] =   {msg1, msg2, msg3, msg1, msg1, msg4};
    WOLFSSL_SMALL_STACK_STATIC const word16 msgSz[] = {0 /*sizeof(msg1)*/,
                                   sizeof(msg2),
                                   sizeof(msg3),
                                   0 /*sizeof(msg1)*/,
                                   0 /*sizeof(msg1)*/,
                                   sizeof(msg4)
    };
#ifndef NO_ASN
    static const byte privateEd448[] = {
        0x30, 0x47, 0x02, 0x01, 0x00, 0x30, 0x05, 0x06,
        0x03, 0x2b, 0x65, 0x71, 0x04, 0x3b, 0x04, 0x39,
        0x6c, 0x82, 0xa5, 0x62, 0xcb, 0x80, 0x8d, 0x10,
        0xd6, 0x32, 0xbe, 0x89, 0xc8, 0x51, 0x3e, 0xbf,
        0x6c, 0x92, 0x9f, 0x34, 0xdd, 0xfa, 0x8c, 0x9f,
        0x63, 0xc9, 0x96, 0x0e, 0xf6, 0xe3, 0x48, 0xa3,
        0x52, 0x8c, 0x8a, 0x3f, 0xcc, 0x2f, 0x04, 0x4e,
        0x39, 0xa3, 0xfc, 0x5b, 0x94, 0x49, 0x2f, 0x8f,
        0x03, 0x2e, 0x75, 0x49, 0xa2, 0x00, 0x98, 0xf9,
        0x5b
    };
    static const byte publicEd448[] = {
        0x30, 0x43, 0x30, 0x05, 0x06, 0x03, 0x2b, 0x65,
        0x71, 0x03, 0x3a, 0x00, 0x5f, 0xd7, 0x44, 0x9b,
        0x59, 0xb4, 0x61, 0xfd, 0x2c, 0xe7, 0x87, 0xec,
        0x61, 0x6a, 0xd4, 0x6a, 0x1d, 0xa1, 0x34, 0x24,
        0x85, 0xa7, 0x0e, 0x1f, 0x8a, 0x0e, 0xa7, 0x5d,
        0x80, 0xe9, 0x67, 0x78, 0xed, 0xf1, 0x24, 0x76,
        0x9b, 0x46, 0xc7, 0x06, 0x1b, 0xd6, 0x78, 0x3d,
        0xf1, 0xe5, 0x0f, 0x6c, 0xd1, 0xfa, 0x1a, 0xbe,
        0xaf, 0xe8, 0x25, 0x61, 0x80
    };
    static const byte privPubEd448[] = {
        0x30, 0x81, 0x82, 0x02, 0x01, 0x00, 0x30, 0x05,
        0x06, 0x03, 0x2b, 0x65, 0x71, 0x04, 0x3b, 0x04,
        0x39, 0x6c, 0x82, 0xa5, 0x62, 0xcb, 0x80, 0x8d,
        0x10, 0xd6, 0x32, 0xbe, 0x89, 0xc8, 0x51, 0x3e,
        0xbf, 0x6c, 0x92, 0x9f, 0x34, 0xdd, 0xfa, 0x8c,
        0x9f, 0x63, 0xc9, 0x96, 0x0e, 0xf6, 0xe3, 0x48,
        0xa3, 0x52, 0x8c, 0x8a, 0x3f, 0xcc, 0x2f, 0x04,
        0x4e, 0x39, 0xa3, 0xfc, 0x5b, 0x94, 0x49, 0x2f,
        0x8f, 0x03, 0x2e, 0x75, 0x49, 0xa2, 0x00, 0x98,
        0xf9, 0x5b, 0x81, 0x39, 0x5f, 0xd7, 0x44, 0x9b,
        0x59, 0xb4, 0x61, 0xfd, 0x2c, 0xe7, 0x87, 0xec,
        0x61, 0x6a, 0xd4, 0x6a, 0x1d, 0xa1, 0x34, 0x24,
        0x85, 0xa7, 0x0e, 0x1f, 0x8a, 0x0e, 0xa7, 0x5d,
        0x80, 0xe9, 0x67, 0x78, 0xed, 0xf1, 0x24, 0x76,
        0x9b, 0x46, 0xc7, 0x06, 0x1b, 0xd6, 0x78, 0x3d,
        0xf1, 0xe5, 0x0f, 0x6c, 0xd1, 0xfa, 0x1a, 0xbe,
        0xaf, 0xe8, 0x25, 0x61, 0x80
    };

    word32 idx;
#endif /* NO_ASN */
#endif /* HAVE_ED448_SIGN && HAVE_ED448_KEY_EXPORT && HAVE_ED448_KEY_IMPORT */
#if !defined(NO_ASN) && defined(HAVE_ED448_SIGN)
#ifdef WOLFSSL_SMALL_STACK
    ed448_key *key3 = NULL;
#else
    ed448_key key3[1];
#endif
#endif

#ifdef WOLFSSL_SMALL_STACK
    key = (ed448_key *)XMALLOC(sizeof(*key), HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    key2 = (ed448_key *)XMALLOC(sizeof(*key2), HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
#if !defined(NO_ASN) && defined(HAVE_ED448_SIGN)
    key3 = (ed448_key *)XMALLOC(sizeof(*key3), HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
#endif
#endif

    /* create ed448 keys */
#ifndef HAVE_FIPS
    ret = wc_InitRng_ex(&rng, HEAP_HINT, devId);
#else
    ret = wc_InitRng(&rng);
#endif
    if (ret != 0) {
        XMEMSET(&rng, 0, sizeof(rng));
        ERROR_OUT(-11700, out);
    }

    if (wc_ed448_init(key) < 0)
        ERROR_OUT(-11903, out);
    if (wc_ed448_init(key2) < 0)
        ERROR_OUT(-11904, out);
#if !defined(NO_ASN) && defined(HAVE_ED448_SIGN)
    if (wc_ed448_init(key3) < 0)
        ERROR_OUT(-11905, out);
#endif
    if (wc_ed448_make_key(&rng, ED448_KEY_SIZE, key) < 0)
        ERROR_OUT(-11906, out);
    if (wc_ed448_make_key(&rng, ED448_KEY_SIZE, key2) < 0)
        ERROR_OUT(-11907, out);

    /* helper functions for signature and key size */
    keySz = wc_ed448_size(key);
    sigSz = wc_ed448_sig_size(key);

#if defined(HAVE_ED448_SIGN) && defined(HAVE_ED448_KEY_EXPORT) &&\
        defined(HAVE_ED448_KEY_IMPORT)
    for (i = 0; i < 6; i++) {
        outlen = sizeof(out);
        XMEMSET(out, 0, sizeof(out));

        if (wc_ed448_import_private_key(sKeys[i], ED448_KEY_SIZE, pKeys[i],
                pKeySz[i], key) != 0)
            ERROR_OUT(-11701 - i, out);

        if (wc_ed448_sign_msg(msgs[i], msgSz[i], out, &outlen, key, NULL,
                                                                      0) != 0)
            ERROR_OUT(-11711 - i, out);

        if (XMEMCMP(out, sigs[i], 114))
            ERROR_OUT(-11721 - i, out);

#if defined(HAVE_ED448_VERIFY)
        /* test verify on good msg */
        if (wc_ed448_verify_msg(out, outlen, msgs[i], msgSz[i], &verify, key,
                                                 NULL, 0) != 0 || verify != 1)
            ERROR_OUT(-11731 - i, out);

#ifdef WOLFSSL_ED448_STREAMING_VERIFY
        /* test verify on good msg using streaming interface directly */
        if (wc_ed448_verify_msg_init(out, outlen,
                                       key, (byte)Ed448, NULL, 0) != 0)
            ERROR_OUT(-11911 - i, out);
        for (j = 0; j < msgSz[i]; j += i) {
            if (wc_ed448_verify_msg_update(msgs[i] + j, MIN(i, msgSz[i] - j), key) != 0)
                ERROR_OUT(-11921 - i, out);
        }
        if (wc_ed448_verify_msg_final(out, outlen, &verify,
                                        key) != 0 || verify != 1)
            ERROR_OUT(-11931 - i, out);
#endif /* WOLFSSL_ED448_STREAMING_VERIFY */

        /* test verify on bad msg */
        out[outlen-2] = out[outlen-2] + 1;
        if (wc_ed448_verify_msg(out, outlen, msgs[i], msgSz[i], &verify, key,
                                                 NULL, 0) == 0 || verify == 1)
            ERROR_OUT(-11741 - i, out);
#endif /* HAVE_ED448_VERIFY */

        /* test api for import/exporting keys */
        {
            byte   *exportPKey = NULL;
            byte   *exportSKey = NULL;
            word32 exportPSz = ED448_KEY_SIZE;
            word32 exportSSz = ED448_KEY_SIZE;

            exportPKey = (byte *)XMALLOC(exportPSz, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
            exportSKey = (byte *)XMALLOC(exportSSz, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
            if ((exportPKey == NULL) || (exportSKey == NULL))
                ERROR_OUT(-11902, out);

            ret = 0;

            do {
                if (wc_ed448_export_public(key, exportPKey, &exportPSz) != 0) {
                    ret = -11751 - i;
                    break;
                }

                if (wc_ed448_import_public_ex(exportPKey, exportPSz, key2, 1) != 0) {
                    ret = -11761 - i;
                    break;
                }

                if (wc_ed448_export_private_only(key, exportSKey, &exportSSz) != 0) {
                    ret = -11771 - i;
                    break;
                }

                if (wc_ed448_import_private_key(exportSKey, exportSSz,
                                                exportPKey, exportPSz, key2) != 0) {
                    ret = -11781 - i;
                    break;
                }

                /* clear "out" buffer and test sign with imported keys */
                outlen = sizeof(out);
                XMEMSET(out, 0, sizeof(out));
                if (wc_ed448_sign_msg(msgs[i], msgSz[i], out, &outlen, key2, NULL,
                                      0) != 0) {
                    ret = -11791 - i;
                    break;
                }
            } while(0);

            XFREE(exportPKey, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
            XFREE(exportSKey, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);

            if (ret != 0)
                goto out;
        }

#if defined(HAVE_ED448_VERIFY)
        if (wc_ed448_verify_msg(out, outlen, msgs[i], msgSz[i], &verify, key2,
                                NULL, 0) != 0 || verify != 1)
            ERROR_OUT(-11801 - i, out);

        if (XMEMCMP(out, sigs[i], SIGSZ))
            ERROR_OUT(-11811 - i, out);
#endif /* HAVE_ED448_VERIFY */
    }

    ret = ed448_ctx_test();
    if (ret != 0)
        goto out;

    ret = ed448ph_test();
    if (ret != 0)
        goto out;

#ifndef NO_ASN
    /* Try ASN.1 encoded private-only key and public key. */
    idx = 0;
    if (wc_Ed448PrivateKeyDecode(privateEd448, &idx, key3,
                                 sizeof(privateEd448)) != 0)
        ERROR_OUT(-11821, out);

    if (wc_ed448_sign_msg(msgs[0], msgSz[0], out, &outlen, key3, NULL, 0)
                != BAD_FUNC_ARG)
        ERROR_OUT(-11831, out);

    idx = 0;
    if (wc_Ed448PublicKeyDecode(publicEd448, &idx, key3,
                                sizeof(publicEd448)) != 0)
        ERROR_OUT(-11841, out);

    if (wc_ed448_sign_msg(msgs[0], msgSz[0], out, &outlen, key3, NULL, 0) != 0)
        ERROR_OUT(-11851, out);

    if (XMEMCMP(out, sigs[0], SIGSZ))
        ERROR_OUT(-11861, out);

#if defined(HAVE_ED448_VERIFY)
    /* test verify on good msg */
    if (wc_ed448_verify_msg(out, outlen, msgs[0], msgSz[0], &verify, key3,
                NULL, 0) != 0 || verify != 1)
        ERROR_OUT(-11871, out);
#endif /* HAVE_ED448_VERIFY */

    wc_ed448_free(key3);
    if (wc_ed448_init(key3) < 0)
        ERROR_OUT(-11908, out);

    idx = 0;
    if (wc_Ed448PrivateKeyDecode(privPubEd448, &idx, key3,
                                 sizeof(privPubEd448)) != 0)
        ERROR_OUT(-11881, out);

    if (wc_ed448_sign_msg(msgs[0], msgSz[0], out, &outlen, key3, NULL, 0) != 0)
        ERROR_OUT(-11891, out);

    if (XMEMCMP(out, sigs[0], SIGSZ))
        ERROR_OUT(-11901, out);
#endif /* NO_ASN */
#endif /* HAVE_ED448_SIGN && HAVE_ED448_KEY_EXPORT && HAVE_ED448_KEY_IMPORT */

    ret = 0;

  out:

#ifdef WOLFSSL_SMALL_STACK
    if (key) {
        wc_ed448_free(key);
        XFREE(key, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    }
    if (key2) {
        wc_ed448_free(key2);
        XFREE(key2, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    }
#if !defined(NO_ASN) && defined(HAVE_ED448_SIGN)
    if (key3) {
        wc_ed448_free(key3);
        XFREE(key3, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    }
#endif
#else
    wc_ed448_free(key);
    wc_ed448_free(key2);
#if !defined(NO_ASN) && defined(HAVE_ED448_SIGN)
    wc_ed448_free(key3);
#endif
#endif

#if defined(HAVE_HASHDRBG) || defined(NO_RC4)
    wc_FreeRng(&rng);
#endif

    if (ret < 0)
        return ret;

    /* hush warnings of unused keySz and sigSz */
    (void)keySz;
    (void)sigSz;

#ifdef WOLFSSL_TEST_CERT
    ret = ed448_test_cert();
    if (ret < 0)
        return ret;
#ifdef WOLFSSL_CERT_GEN
    ret = ed448_test_make_cert();
    if (ret < 0)
        return ret;
#endif /* WOLFSSL_CERT_GEN */
#endif /* WOLFSSL_TEST_CERT */

    return 0;
}
#endif /* HAVE_ED448 */

#ifdef WOLFCRYPT_HAVE_ECCSI
static int eccsi_api_test(WC_RNG* rng, EccsiKey* key, mp_int* ssk,
                          ecc_point* pvt)
{
    int ret;
    byte id[1] = { 0x00 };
    int valid;
    word32 sz;
    byte data[256];
    byte hash[WC_MAX_DIGEST_SIZE];
    byte hashSz;
    byte sig[257];
    word32 sigSz;

    ret = wc_InitEccsiKey_ex(NULL, 32, ECC_SECP256R1, HEAP_HINT, INVALID_DEVID);
    if (ret != BAD_FUNC_ARG)
        return -10023;
    ret = wc_InitEccsiKey_ex(NULL, 32, ECC_SECP256R1, HEAP_HINT, INVALID_DEVID);
    if (ret != BAD_FUNC_ARG)
        return -10024;

    ret = wc_InitEccsiKey(NULL, NULL, INVALID_DEVID);
    if (ret != BAD_FUNC_ARG)
        return -10025;
    ret = wc_InitEccsiKey(NULL, HEAP_HINT, INVALID_DEVID);
    if (ret != BAD_FUNC_ARG)
        return -10026;

    wc_FreeEccsiKey(NULL);

    /* Create a valid key. */
    ret = wc_InitEccsiKey(key, NULL, INVALID_DEVID);
    if (ret != 0)
        return -10027;

    ret = wc_MakeEccsiKey(NULL, NULL);
    if (ret != BAD_FUNC_ARG)
        return -10028;
    ret = wc_MakeEccsiKey(key, NULL);
    if (ret != BAD_FUNC_ARG)
        return -10029;
    ret = wc_MakeEccsiKey(NULL, rng);
    if (ret != BAD_FUNC_ARG)
        return -10030;

    ret = wc_MakeEccsiPair(NULL, NULL, WC_HASH_TYPE_SHA256, NULL, 1, NULL,
            NULL);
    if (ret != BAD_FUNC_ARG)
        return -10031;
    ret = wc_MakeEccsiPair(key, rng, WC_HASH_TYPE_SHA256, id, 1, ssk, NULL);
    if (ret != BAD_FUNC_ARG)
        return -10032;
    ret = wc_MakeEccsiPair(key, rng, WC_HASH_TYPE_SHA256, id, 1, NULL, pvt);
    if (ret != BAD_FUNC_ARG)
        return -10033;
    ret = wc_MakeEccsiPair(key, rng, WC_HASH_TYPE_SHA256, NULL, 1, ssk, pvt);
    if (ret != BAD_FUNC_ARG)
        return -10034;
    ret = wc_MakeEccsiPair(key, NULL, WC_HASH_TYPE_SHA256, id, 1, ssk, pvt);
    if (ret != BAD_FUNC_ARG)
        return -10035;
    ret = wc_MakeEccsiPair(NULL, rng, WC_HASH_TYPE_SHA256, id, 1, ssk, pvt);
    if (ret != BAD_FUNC_ARG)
        return -10036;
    /* No key set */
    ret = wc_MakeEccsiPair(key, rng, WC_HASH_TYPE_SHA256, id, 1, ssk, pvt);
    if (ret != BAD_STATE_E)
        return -10037;

    ret = wc_ValidateEccsiPair(NULL, WC_HASH_TYPE_SHA256, NULL, 1, NULL, NULL,
            NULL);
    if (ret != BAD_FUNC_ARG)
        return -10038;
    ret = wc_ValidateEccsiPair(key, WC_HASH_TYPE_SHA256, id, 1, ssk, pvt,
            NULL);
    if (ret != BAD_FUNC_ARG)
        return -10039;
    ret = wc_ValidateEccsiPair(key, WC_HASH_TYPE_SHA256, id, 1, ssk, NULL,
            &valid);
    if (ret != BAD_FUNC_ARG)
        return -10040;
    ret = wc_ValidateEccsiPair(key, WC_HASH_TYPE_SHA256, id, 1, NULL, pvt,
            &valid);
    if (ret != BAD_FUNC_ARG)
        return -10041;
    ret = wc_ValidateEccsiPair(key, WC_HASH_TYPE_SHA256, NULL, 1, ssk, pvt,
            &valid);
    if (ret != BAD_FUNC_ARG)
        return -10042;
    ret = wc_ValidateEccsiPair(NULL, WC_HASH_TYPE_SHA256, id, 1, ssk, pvt,
            &valid);
    if (ret != BAD_FUNC_ARG)
        return -10043;
    /* No key set */
    ret = wc_ValidateEccsiPair(key, WC_HASH_TYPE_SHA256, id, 1, ssk, pvt,
            &valid);
    if (ret != BAD_STATE_E)
        return -10044;

    ret = wc_ValidateEccsiPvt(NULL, NULL, NULL);
    if (ret != BAD_FUNC_ARG)
        return -10045;
    ret = wc_ValidateEccsiPvt(key, NULL, NULL);
    if (ret != BAD_FUNC_ARG)
        return -10046;
    ret = wc_ValidateEccsiPvt(NULL, pvt, NULL);
    if (ret != BAD_FUNC_ARG)
        return -10047;
    ret = wc_ValidateEccsiPvt(NULL, NULL, &valid);
    if (ret != BAD_FUNC_ARG)
        return -10048;
    ret = wc_ValidateEccsiPvt(key, pvt, NULL);
    if (ret != BAD_FUNC_ARG)
        return -10049;
    ret = wc_ValidateEccsiPvt(key, NULL, &valid);
    if (ret != BAD_FUNC_ARG)
        return -10050;
    ret = wc_ValidateEccsiPvt(NULL, pvt, &valid);
    if (ret != BAD_FUNC_ARG)
        return -10051;
    ret = wc_EncodeEccsiPair(NULL, NULL, NULL, data, NULL);
    if (ret != BAD_FUNC_ARG)
        return -10052;
    ret = wc_EncodeEccsiPair(key, ssk, pvt, data, NULL);
    if (ret != BAD_FUNC_ARG)
        return -10053;
    ret = wc_EncodeEccsiPair(key, ssk, NULL, data, &sz);
    if (ret != BAD_FUNC_ARG)
        return -10054;
    ret = wc_EncodeEccsiPair(key, NULL, pvt, data, &sz);
    if (ret != BAD_FUNC_ARG)
        return -10055;
    ret = wc_EncodeEccsiPair(NULL, ssk, pvt, data, &sz);
    if (ret != BAD_FUNC_ARG)
        return -10056;
    /* No key created so no curve information. */
    ret = wc_EncodeEccsiPair(key, ssk, pvt, NULL, &sz);
    if (ret != LENGTH_ONLY_E)
        return -10057;

    ret = wc_EncodeEccsiSsk(NULL, NULL, data, NULL);
    if (ret != BAD_FUNC_ARG)
        return -10058;
    ret = wc_EncodeEccsiSsk(key, ssk, data, NULL);
    if (ret != BAD_FUNC_ARG)
        return -10059;
    ret = wc_EncodeEccsiSsk(key, NULL, data, &sz);
    if (ret != BAD_FUNC_ARG)
        return -10060;
    ret = wc_EncodeEccsiSsk(NULL, ssk, data, &sz);
    if (ret != BAD_FUNC_ARG)
        return -10061;

    ret = wc_EncodeEccsiPvt(NULL, NULL, data, NULL, 1);
    if (ret != BAD_FUNC_ARG)
        return -10058;
    ret = wc_EncodeEccsiPvt(key, pvt, data, NULL, 1);
    if (ret != BAD_FUNC_ARG)
        return -10059;
    ret = wc_EncodeEccsiPvt(key, NULL, data, &sz, 1);
    if (ret != BAD_FUNC_ARG)
        return -10060;
    ret = wc_EncodeEccsiPvt(NULL, pvt, data, &sz, 1);
    if (ret != BAD_FUNC_ARG)
        return -10061;

    ret = wc_DecodeEccsiPair(NULL, NULL, 0, NULL, NULL);
    if (ret != BAD_FUNC_ARG)
        return -10062;
    ret = wc_DecodeEccsiPair(key, data, 0, ssk, NULL);
    if (ret != BAD_FUNC_ARG)
        return -10063;
    ret = wc_DecodeEccsiPair(key, data, 0, NULL, pvt);
    if (ret != BAD_FUNC_ARG)
        return -10064;
    ret = wc_DecodeEccsiPair(key, NULL, 0, ssk, pvt);
    if (ret != BAD_FUNC_ARG)
        return -10065;
    ret = wc_DecodeEccsiPair(NULL, data, 0, ssk, pvt);
    if (ret != BAD_FUNC_ARG)
        return -10066;

    ret = wc_DecodeEccsiSsk(NULL, NULL, 0, NULL);
    if (ret != BAD_FUNC_ARG)
        return -10067;
    ret = wc_DecodeEccsiSsk(key, data, 0, NULL);
    if (ret != BAD_FUNC_ARG)
        return -10068;
    ret = wc_DecodeEccsiSsk(key, NULL, 0, ssk);
    if (ret != BAD_FUNC_ARG)
        return -10069;
    ret = wc_DecodeEccsiSsk(NULL, data, 0, ssk);
    if (ret != BAD_FUNC_ARG)
        return -10070;
    ret = wc_DecodeEccsiPvt(NULL, NULL, 0, NULL);
    if (ret != BAD_FUNC_ARG)
        return -10067;
    ret = wc_DecodeEccsiPvt(key, data, 0, NULL);
    if (ret != BAD_FUNC_ARG)
        return -10068;
    ret = wc_DecodeEccsiPvt(key, NULL, 0, pvt);
    if (ret != BAD_FUNC_ARG)
        return -10069;
    ret = wc_DecodeEccsiPvt(NULL, data, 0, pvt);
    if (ret != BAD_FUNC_ARG)
        return -10070;

    ret = wc_DecodeEccsiPvtFromSig(NULL, NULL, 0, NULL);
    if (ret != BAD_FUNC_ARG)
        return -10067;
    ret = wc_DecodeEccsiPvtFromSig(key, data, 0, NULL);
    if (ret != BAD_FUNC_ARG)
        return -10068;
    ret = wc_DecodeEccsiPvtFromSig(key, NULL, 0, pvt);
    if (ret != BAD_FUNC_ARG)
        return -10069;
    ret = wc_DecodeEccsiPvtFromSig(NULL, data, 0, pvt);
    if (ret != BAD_FUNC_ARG)
        return -10070;

    ret = wc_ExportEccsiKey(NULL, data, NULL);
    if (ret != BAD_FUNC_ARG)
        return -10071;
    ret = wc_ExportEccsiKey(key, data, NULL);
    if (ret != BAD_FUNC_ARG)
        return -10072;
    ret = wc_ExportEccsiKey(NULL, data, &sz);
    if (ret != BAD_FUNC_ARG)
        return -10073;
    /* No key to export */
    ret = wc_ExportEccsiKey(key, NULL, &sz);
    if (ret != BAD_STATE_E)
        return -10074;

    ret = wc_ImportEccsiKey(NULL, NULL, 0);
    if (ret != BAD_FUNC_ARG)
        return -10075;
    ret = wc_ImportEccsiKey(key, NULL, 0);
    if (ret != BAD_FUNC_ARG)
        return -10076;
    ret = wc_ImportEccsiKey(NULL, data, 0);
    if (ret != BAD_FUNC_ARG)
        return -10077;

    ret = wc_ExportEccsiPrivateKey(NULL, data, NULL);
    if (ret != BAD_FUNC_ARG)
        return -10071;
    ret = wc_ExportEccsiPrivateKey(key, data, NULL);
    if (ret != BAD_FUNC_ARG)
        return -10072;
    ret = wc_ExportEccsiPrivateKey(NULL, data, &sz);
    if (ret != BAD_FUNC_ARG)
        return -10073;
    /* No key to export */
    ret = wc_ExportEccsiPrivateKey(key, NULL, &sz);
    if (ret != BAD_STATE_E)
        return -10074;

    ret = wc_ImportEccsiPrivateKey(NULL, NULL, 0);
    if (ret != BAD_FUNC_ARG)
        return -10075;
    ret = wc_ImportEccsiPrivateKey(key, NULL, 0);
    if (ret != BAD_FUNC_ARG)
        return -10076;
    ret = wc_ImportEccsiPrivateKey(NULL, data, 0);
    if (ret != BAD_FUNC_ARG)
        return -10077;
    ret = wc_ExportEccsiPublicKey(NULL, data, NULL, 1);
    if (ret != BAD_FUNC_ARG)
        return -10078;
    ret = wc_ExportEccsiPublicKey(key, data, NULL, 1);
    if (ret != BAD_FUNC_ARG)
        return -10079;
    ret = wc_ExportEccsiPublicKey(NULL, data, &sz, 1);
    if (ret != BAD_FUNC_ARG)
        return -10080;
    /* No key to export */
    ret = wc_ExportEccsiPublicKey(key, data, &sz, 1);
    if (ret != BAD_STATE_E)
        return -10081;

    ret = wc_ImportEccsiPublicKey(NULL, NULL, 0, 1);
    if (ret != BAD_FUNC_ARG)
        return -10082;
    ret = wc_ImportEccsiPublicKey(key, NULL, 0, 1);
    if (ret != BAD_FUNC_ARG)
        return -10083;
    ret = wc_ImportEccsiPublicKey(NULL, data, 0, 1);
    if (ret != BAD_FUNC_ARG)
        return -10084;

    ret = wc_HashEccsiId(NULL, WC_HASH_TYPE_SHA256, NULL, 1, NULL, NULL, NULL);
    if (ret != BAD_FUNC_ARG)
        return -10085;
    ret = wc_HashEccsiId(key, WC_HASH_TYPE_SHA256, id, 1, pvt, hash, NULL);
    if (ret != BAD_FUNC_ARG)
        return -10086;
    ret = wc_HashEccsiId(key, WC_HASH_TYPE_SHA256, id, 1, pvt, NULL, &hashSz);
    if (ret != BAD_FUNC_ARG)
        return -10087;
    ret = wc_HashEccsiId(key, WC_HASH_TYPE_SHA256, id, 1, NULL, hash, &hashSz);
    if (ret != BAD_FUNC_ARG)
        return -10088;
    ret = wc_HashEccsiId(key, WC_HASH_TYPE_SHA256, NULL, 1, pvt, hash,
            &hashSz);
    if (ret != BAD_FUNC_ARG)
        return -10089;
    ret = wc_HashEccsiId(NULL, WC_HASH_TYPE_SHA256, id, 1, pvt, hash, &hashSz);
    if (ret != BAD_FUNC_ARG)
        return -10090;
    ret = wc_HashEccsiId(key, WC_HASH_TYPE_SHA256, id, 1, pvt, hash, &hashSz);
    if (ret != BAD_STATE_E)
        return -10091;

    ret = wc_SetEccsiHash(NULL, NULL, 1);
    if (ret != BAD_FUNC_ARG)
        return -10090;
    ret = wc_SetEccsiHash(key, NULL, 1);
    if (ret != BAD_FUNC_ARG)
        return -10090;
    ret = wc_SetEccsiHash(NULL, hash, 1);
    if (ret != BAD_FUNC_ARG)
        return -10090;

    ret = wc_SetEccsiPair(NULL, NULL, NULL);
    if (ret != BAD_FUNC_ARG)
        return -10090;
    ret = wc_SetEccsiPair(key, NULL, NULL);
    if (ret != BAD_FUNC_ARG)
        return -10090;
    ret = wc_SetEccsiPair(NULL, ssk, NULL);
    if (ret != BAD_FUNC_ARG)
        return -10090;
    ret = wc_SetEccsiPair(NULL, NULL, pvt);
    if (ret != BAD_FUNC_ARG)
        return -10090;
    ret = wc_SetEccsiPair(key, ssk, NULL);
    if (ret != BAD_FUNC_ARG)
        return -10090;
    ret = wc_SetEccsiPair(key, NULL, pvt);
    if (ret != BAD_FUNC_ARG)
        return -10090;
    ret = wc_SetEccsiPair(NULL, ssk, pvt);
    if (ret != BAD_FUNC_ARG)
        return -10090;
    ret = wc_SignEccsiHash(NULL, NULL, WC_HASH_TYPE_SHA256, NULL, 0, sig, NULL);
    if (ret != BAD_FUNC_ARG)
        return -10092;
    ret = wc_SignEccsiHash(key, rng, WC_HASH_TYPE_SHA256, data, 0, sig, NULL);
    if (ret != BAD_FUNC_ARG)
        return -10093;
    ret = wc_SignEccsiHash(key, rng, WC_HASH_TYPE_SHA256, NULL, 0, sig,
            &sigSz);
    if (ret != BAD_FUNC_ARG)
        return -10096;
    ret = wc_SignEccsiHash(key, NULL, WC_HASH_TYPE_SHA256, data, 0, sig,
            &sigSz);
    if (ret != BAD_FUNC_ARG)
        return -10098;
    ret = wc_SignEccsiHash(NULL, rng, WC_HASH_TYPE_SHA256, data, 0, sig,
            &sigSz);
    if (ret != BAD_FUNC_ARG)
        return -10099;
    /* Key not set. */
    ret = wc_SignEccsiHash(key, rng, WC_HASH_TYPE_SHA256, data, 0, NULL,
            &sigSz);
    if (ret != BAD_STATE_E)
        return -10100;

    ret = wc_VerifyEccsiHash(NULL, WC_HASH_TYPE_SHA256, NULL, 0, NULL, 0, NULL);
    if (ret != BAD_FUNC_ARG)
        return -10101;
    ret = wc_VerifyEccsiHash(key, WC_HASH_TYPE_SHA256, NULL, 0, NULL, 0, NULL);
    if (ret != BAD_FUNC_ARG)
        return -10101;
    ret = wc_VerifyEccsiHash(NULL, WC_HASH_TYPE_SHA256, data, 0, NULL, 0, NULL);
    if (ret != BAD_FUNC_ARG)
        return -10101;
    ret = wc_VerifyEccsiHash(NULL, WC_HASH_TYPE_SHA256, NULL, 0, sig, 0, NULL);
    if (ret != BAD_FUNC_ARG)
        return -10101;
    ret = wc_VerifyEccsiHash(NULL, WC_HASH_TYPE_SHA256, NULL, 0, NULL, 0,
            &valid);
    if (ret != BAD_FUNC_ARG)
        return -10101;
    ret = wc_VerifyEccsiHash(key, WC_HASH_TYPE_SHA256, data, 0, sig, 0, NULL);
    if (ret != BAD_FUNC_ARG)
        return -10102;
    ret = wc_VerifyEccsiHash(key, WC_HASH_TYPE_SHA256, data, 0, NULL, 0,
            &valid);
    if (ret != BAD_FUNC_ARG)
        return -10103;
    ret = wc_VerifyEccsiHash(key, WC_HASH_TYPE_SHA256, NULL, 0, sig, 0,
            &valid);
    if (ret != BAD_FUNC_ARG)
        return -10104;
    ret = wc_VerifyEccsiHash(NULL, WC_HASH_TYPE_SHA256, data, 0, sig, 0,
            &valid);
    if (ret != BAD_FUNC_ARG)
        return -10106;
    ret = wc_VerifyEccsiHash(key, WC_HASH_TYPE_SHA256, data, 0, sig, 0,
            &valid);
    if (ret != BAD_STATE_E)
        return -10106;

    ret = wc_SetEccsiPair(key, ssk, pvt);
    if (ret != 0)
        return -10107;
    /* Identity hash not set. */
    ret = wc_SignEccsiHash(key, rng, WC_HASH_TYPE_SHA256, data, 0, NULL,
            &sigSz);
    if (ret != BAD_STATE_E)
        return -10108;

    wc_FreeEccsiKey(key);

    return 0;
}

/* RFC 6507: Appendix A */
static int eccsi_kat_verify_test(EccsiKey* key, ecc_point* pvt)
{
    int ret;
    int verified;
    const byte msg[] = { 0x6D, 0x65, 0x73, 0x73, 0x61, 0x67, 0x65, 0x00 };
    word32 msgSz = sizeof(msg);
    byte hash[WC_SHA256_DIGEST_SIZE];
    byte hashSz = WC_SHA256_DIGEST_SIZE;
    static const byte id[] = {
        0x32, 0x30, 0x31, 0x31, 0x2D, 0x30, 0x32, 0x00,
        0x74, 0x65, 0x6C, 0x3A, 0x2B, 0x34, 0x34, 0x37,
        0x37, 0x30, 0x30, 0x39, 0x30, 0x30, 0x31, 0x32,
        0x33, 0x00
    };
    word32 idSz = sizeof(id);
    static const byte sig[] = {
        0x26, 0x9D, 0x4C, 0x8F, 0xDE, 0xB6, 0x6A, 0x74,
        0xE4, 0xEF, 0x8C, 0x0D, 0x5D, 0xCC, 0x59, 0x7D,
        0xDF, 0xE6, 0x02, 0x9C, 0x2A, 0xFF, 0xC4, 0x93,
        0x60, 0x08, 0xCD, 0x2C, 0xC1, 0x04, 0x5D, 0x81,
        0xE0, 0x9B, 0x52, 0x8D, 0x0E, 0xF8, 0xD6, 0xDF,
        0x1A, 0xA3, 0xEC, 0xBF, 0x80, 0x11, 0x0C, 0xFC,
        0xEC, 0x9F, 0xC6, 0x82, 0x52, 0xCE, 0xBB, 0x67,
        0x9F, 0x41, 0x34, 0x84, 0x69, 0x40, 0xCC, 0xFD,
        0x04,
        0x75, 0x8A, 0x14, 0x27, 0x79, 0xBE, 0x89, 0xE8,
        0x29, 0xE7, 0x19, 0x84, 0xCB, 0x40, 0xEF, 0x75,
        0x8C, 0xC4, 0xAD, 0x77, 0x5F, 0xC5, 0xB9, 0xA3,
        0xE1, 0xC8, 0xED, 0x52, 0xF6, 0xFA, 0x36, 0xD9,
        0xA7, 0x9D, 0x24, 0x76, 0x92, 0xF4, 0xED, 0xA3,
        0xA6, 0xBD, 0xAB, 0x77, 0xD6, 0xAA, 0x64, 0x74,
        0xA4, 0x64, 0xAE, 0x49, 0x34, 0x66, 0x3C, 0x52,
        0x65, 0xBA, 0x70, 0x18, 0xBA, 0x09, 0x1F, 0x79
    };
    word32 sigSz = sizeof(sig);
    static const byte pubData[] = {
        0x50, 0xD4, 0x67, 0x0B, 0xDE, 0x75, 0x24, 0x4F,
        0x28, 0xD2, 0x83, 0x8A, 0x0D, 0x25, 0x55, 0x8A,
        0x7A, 0x72, 0x68, 0x6D, 0x45, 0x22, 0xD4, 0xC8,
        0x27, 0x3F, 0xB6, 0x44, 0x2A, 0xEB, 0xFA, 0x93,
        0xDB, 0xDD, 0x37, 0x55, 0x1A, 0xFD, 0x26, 0x3B,
        0x5D, 0xFD, 0x61, 0x7F, 0x39, 0x60, 0xC6, 0x5A,
        0x8C, 0x29, 0x88, 0x50, 0xFF, 0x99, 0xF2, 0x03,
        0x66, 0xDC, 0xE7, 0xD4, 0x36, 0x72, 0x17, 0xF4
    };
    static const byte expHash[] = {
        0x49, 0x0f, 0x3f, 0xeb, 0xbc, 0x1c, 0x90, 0x2f,
        0x62, 0x89, 0x72, 0x3d, 0x7f, 0x8c, 0xbf, 0x79,
        0xdb, 0x88, 0x93, 0x08, 0x49, 0xd1, 0x9f, 0x38,
        0xf0, 0x29, 0x5b, 0x5c, 0x27, 0x6c, 0x14, 0xd1
    };

    ret = wc_ImportEccsiPublicKey(key, pubData, sizeof(pubData), 0);
    if (ret != 0)
        return -10108;

    ret = wc_DecodeEccsiPvtFromSig(key, sig, sigSz, pvt);
    if (ret != 0)
        return -10109;

    ret = wc_HashEccsiId(key, WC_HASH_TYPE_SHA256, id, idSz, pvt, hash,
            &hashSz);
    if (ret != 0)
        return -10112;
    if (hashSz != sizeof(expHash))
        return -10113;
    if (XMEMCMP(hash, expHash, hashSz) != 0)
        return -10114;

    ret = wc_SetEccsiHash(key, hash, hashSz);
    if (ret != 0)
        return -10112;

    ret = wc_VerifyEccsiHash(key, WC_HASH_TYPE_SHA256, msg, msgSz, sig, sigSz,
            &verified);
    if (ret != 0)
        return -10115;
    if (!verified)
        return -10116;

    return 0;
}
static int eccsi_enc_dec_pair_test(EccsiKey* priv, mp_int* ssk, ecc_point* pvt)
{
    int ret;
    byte data[32 * 3];
    word32 sz;
    ecc_point* decPvt = NULL;
#ifdef WOLFSSL_SMALL_STACK
    mp_int *decSsk = (mp_int *)XMALLOC(sizeof(*decSsk), HEAP_HINT,
                                       DYNAMIC_TYPE_TMP_BUFFER);
    if (decSsk == NULL)
        return -10173;
#else
    mp_int decSsk[1];
#endif

    ret = mp_init(decSsk);
    if (ret != 0)
        ERROR_OUT(-10117, out);

    decPvt = wc_ecc_new_point();
    if (decPvt == NULL)
        ERROR_OUT(-10118, out);

    ret = wc_EncodeEccsiPair(priv, ssk, pvt, NULL, &sz);
    if (ret != LENGTH_ONLY_E)
        ERROR_OUT(-10119, out);
    if (sz != 32 * 3)
        ERROR_OUT(-10120, out);
    ret = wc_EncodeEccsiPair(priv, ssk, pvt, data, &sz);
    if (ret != 0)
        ERROR_OUT(-10121, out);
    if (sz != 32* 3)
        ERROR_OUT(-10122, out);
    ret = wc_DecodeEccsiPair(priv, data, sz, decSsk, decPvt);
    if (ret != 0)
        ERROR_OUT(-10123, out);
    if (mp_cmp(ssk, decSsk) != MP_EQ)
        ERROR_OUT(-10124, out);
    if (wc_ecc_cmp_point(pvt, decPvt) != MP_EQ)
        ERROR_OUT(-10125, out);

    ret = wc_EncodeEccsiSsk(priv, ssk, NULL, &sz);
    if (ret != LENGTH_ONLY_E)
        ERROR_OUT(-10119, out);
    if (sz != 32)
        ERROR_OUT(-10120, out);
    ret = wc_EncodeEccsiSsk(priv, ssk, data, &sz);
    if (ret != 0)
        ERROR_OUT(-10121, out);
    if (sz != 32)
        ERROR_OUT(-10122, out);
    ret = wc_DecodeEccsiSsk(priv, data, sz, decSsk);
    if (ret != 0)
        ERROR_OUT(-10123, out);
    if (mp_cmp(ssk, decSsk) != MP_EQ)
        ERROR_OUT(-10124, out);

    ret = wc_EncodeEccsiPvt(priv, pvt, NULL, &sz, 1);
    if (ret != LENGTH_ONLY_E)
        ERROR_OUT(-10126, out);
    if (sz != 32 * 2)
        ERROR_OUT(-10127, out);
    ret = wc_EncodeEccsiPvt(priv, pvt, data, &sz, 1);
    if (ret != 0)
        ERROR_OUT(-10128, out);
    if (sz != 32 * 2)
        ERROR_OUT(-10129, out);
    ret = wc_DecodeEccsiPvt(priv, data, sz, decPvt);
    if (ret != 0)
        ERROR_OUT(-10130, out);
    if (wc_ecc_cmp_point(pvt, decPvt) != MP_EQ)
        ERROR_OUT(-10131, out);
    sz = sizeof(data);
    ret = wc_EncodeEccsiPvt(priv, pvt, data, &sz, 0);
    if (ret != 0)
        ERROR_OUT(-10128, out);
    if (sz != 32 * 2 + 1)
        ERROR_OUT(-10129, out);
    ret = wc_DecodeEccsiPvt(priv, data, sz, decPvt);
    if (ret != 0)
        ERROR_OUT(-10130, out);
    if (wc_ecc_cmp_point(pvt, decPvt) != MP_EQ)
        ERROR_OUT(-10131, out);

    wc_ecc_del_point(decPvt);

out:

    mp_free(decSsk);
#ifdef WOLFSSL_SMALL_STACK
    XFREE(decSsk, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
#endif

    return ret;
}

static int eccsi_imp_exp_key_test(EccsiKey* priv)
{
    int ret;
    byte data[32 * 3];
    byte out[32 * 3];
    word32 sz;

    ret = wc_ExportEccsiKey(priv, NULL, &sz);
    if (ret != LENGTH_ONLY_E)
        return -10132;
    if (sz != 32 * 3)
        return -10133;
    ret = wc_ExportEccsiKey(priv, data, &sz);
    if (ret != 0)
        return -10134;
    ret = wc_ImportEccsiKey(priv, data, sz);
    if (ret != 0)
        return -10135;
    ret = wc_ExportEccsiKey(priv, NULL, &sz);
    if (ret != LENGTH_ONLY_E)
        return -10132;
    if (sz != 32 * 3)
        return -10143;
    ret = wc_ExportEccsiKey(priv, out, &sz);
    if (ret != 0)
        return -10144;
    if (sz != 32 * 3)
        return -10145;
    if (XMEMCMP(data, out, sz) != 0)
        return -10146;

    ret = wc_ExportEccsiPrivateKey(priv, NULL, &sz);
    if (ret != LENGTH_ONLY_E)
        return -10156;
    if (sz != 32)
        return -10157;
    ret = wc_ExportEccsiPrivateKey(priv, data, &sz);
    if (ret != 0)
        return -10158;
    ret = wc_ImportEccsiPrivateKey(priv, data, sz);
    if (ret != 0)
        return -10159;
    ret = wc_ExportEccsiPrivateKey(priv, NULL, &sz);
    if (ret != LENGTH_ONLY_E)
        return -10152;
    if (sz != 32)
        return -10163;
    ret = wc_ExportEccsiPrivateKey(priv, out, &sz);
    if (ret != 0)
        return -10164;
    if (sz != 32)
        return -10165;
    if (XMEMCMP(data, out, sz) != 0)
        return -10166;

    return 0;
}

static int eccsi_imp_exp_pubkey_test(EccsiKey* key1, EccsiKey* key2)
{
    int ret;
    byte data[32 * 2 + 1];
    byte pubData[32 * 2 + 1];
    word32 sz;

    ret = wc_ExportEccsiPublicKey(key1, NULL, &sz, 1);
    if (ret != LENGTH_ONLY_E)
        return -10136;
    if (sz != 32 * 2)
        return -10137;

    ret = wc_ExportEccsiPublicKey(key1, data, &sz, 1);
    if (ret != 0)
        return -10138;

    ret = wc_ImportEccsiPublicKey(key2, data, sz, 1);
    if (ret != 0)
        return -10139;
    sz = sizeof(pubData);
    ret = wc_ExportEccsiPublicKey(key2, pubData, &sz, 1);
    if (ret != 0)
        return -10140;
    if (sz != 32 * 2)
        return -10141;
    if (XMEMCMP(data, pubData, sz) != 0)
        return -10142;

    sz = sizeof(pubData);
    ret = wc_ExportEccsiPublicKey(key2, pubData, &sz, 0);
    if (ret != 0)
        return -10140;
    if (sz != 32 * 2 + 1)
        return -10141;
    if (pubData[0] != 0x04)
        return -10140;
    if (XMEMCMP(pubData + 1, data, sz - 1) != 0)
        return -10142;
    ret = wc_ImportEccsiPublicKey(key2, pubData, sz, 0);
    if (ret != 0)
        return -10139;

    return 0;
}

static int eccsi_make_key_test(EccsiKey* priv, EccsiKey* pub, WC_RNG* rng,
                               mp_int* ssk, ecc_point* pvt)
{
    int ret;
    char mail[] = "test@wolfssl.com";
    byte* id = (byte*)mail;
    word32 idSz = (word32) XSTRLEN(mail);
    int valid;

    ret = wc_MakeEccsiKey(priv, rng);
    if (ret != 0)
        return -10143;

    ret = eccsi_imp_exp_key_test(priv);
    if (ret < 0)
        return ret;

    ret = eccsi_imp_exp_pubkey_test(priv, pub);
    if (ret < 0)
        return ret;

    ret = wc_MakeEccsiPair(priv, rng, WC_HASH_TYPE_SHA256, id, idSz, ssk, pvt);
    if (ret != 0)
        return -10144;

    ret = wc_ValidateEccsiPair(pub, WC_HASH_TYPE_SHA256, id, idSz, ssk, pvt,
            &valid);
    if (ret != 0)
        return -10145;
    if (!valid)
        return -10146;

    ret = eccsi_enc_dec_pair_test(priv, ssk, pvt);
    if (ret != 0)
        return ret;

    return 0;
}

static int eccsi_sign_verify_test(EccsiKey* priv, EccsiKey* pub, WC_RNG* rng,
                                  mp_int* ssk, ecc_point* pvt)
{
    int ret;
    byte hashPriv[WC_MAX_DIGEST_SIZE];
    byte hashPub[WC_MAX_DIGEST_SIZE];
    byte hashSz;
    byte sig[144];
    word32 sigSz;
    int verified, valid;
    char mail[] = "test@wolfssl.com";
    byte* id = (byte*)mail;
    word32 idSz = (word32) XSTRLEN(mail);
    byte msg[] = { 0x00 };
    word32 msgSz = sizeof(msg);

    ret = wc_HashEccsiId(priv, WC_HASH_TYPE_SHA256, id, idSz, pvt, hashPriv,
            &hashSz);
    if (ret != 0)
        return -10147;
    if (hashSz != 32)
        return -10148;
    ret = wc_HashEccsiId(priv, WC_HASH_TYPE_SHA256, id, idSz, pvt, hashPub,
            &hashSz);
    if (ret != 0)
        return -10149;
    if (hashSz != 32)
        return -10150;

    if (XMEMCMP(hashPriv, hashPub, hashSz) != 0)
        return -10151;

    ret = wc_SetEccsiHash(priv, hashPriv, hashSz);
    if (ret != 0)
        return -10149;
    ret = wc_SetEccsiPair(priv, ssk, pvt);
    if (ret != 0)
        return -10149;

    ret = wc_SignEccsiHash(priv, rng, WC_HASH_TYPE_SHA256, msg, msgSz, NULL,
            &sigSz);
    if (ret != LENGTH_ONLY_E)
        return -10152;
    if (sigSz != 129)
        return -10153;
    ret = wc_SignEccsiHash(priv, rng, WC_HASH_TYPE_SHA256, msg, msgSz, sig,
            &sigSz);
    if (ret != 0)
        return -10154;

    ret = wc_SetEccsiHash(pub, hashPub, hashSz);
    if (ret != 0)
        return -10149;

    ret = wc_VerifyEccsiHash(pub, WC_HASH_TYPE_SHA256, msg, msgSz, sig, sigSz,
             &verified);
    if (ret != 0)
        return -10155;
    if (!verified)
        return -10156;

    /* Check that changing HS results in verification failure. */
    hashPub[0] ^= 0x80;
    ret = wc_SetEccsiHash(pub, hashPub, hashSz);
    if (ret != 0)
        return -10149;
    ret = wc_VerifyEccsiHash(pub, WC_HASH_TYPE_SHA256, msg, msgSz, sig, sigSz,
            &verified);
    if (ret != 0)
        return -10157;
    if (verified)
        return -10158;
    hashPub[0] ^= 0x80;
    ret = wc_SetEccsiHash(pub, hashPub, hashSz);
    if (ret != 0)
        return -10149;

    /* Check that changing msg results in verification failure. */
    msg[0] ^= 0x80;
    ret = wc_VerifyEccsiHash(pub, WC_HASH_TYPE_SHA256, msg, msgSz, sig, sigSz,
           &verified);
    if (ret != 0)
        return -10159;
    if (verified)
        return -10160;
    msg[0] ^= 0x80;
    /* Check that changing signature results in verification failure. */
    sig[0] ^= 0x80;
    ret = wc_VerifyEccsiHash(pub, WC_HASH_TYPE_SHA256, msg, msgSz, sig, sigSz,
            &verified);
    if (ret != 0)
        return -10161;
    if (verified)
        return -10162;
    sig[0] ^= 0x80;

    /* Check that key state hasn't been invalidated. */
    ret = wc_VerifyEccsiHash(pub, WC_HASH_TYPE_SHA256, msg, msgSz, sig, sigSz,
            &verified);
    if (ret != 0)
        return -10163;
    if (!verified)
        return -10164;

    /* Check that verifying with the private key works. */
    ret = wc_VerifyEccsiHash(priv, WC_HASH_TYPE_SHA256, msg, msgSz, sig, sigSz,
            &verified);
    if (ret != 0)
        return -10165;
    if (!verified)
        return -10166;

    /* Check that the KPAK is converted from montgomery form. */
    ret = eccsi_imp_exp_key_test(priv);
    if (ret != 0)
        return ret;

    /* Check that KPAK can converted to Montgomery form again. */
    ret = wc_VerifyEccsiHash(priv, WC_HASH_TYPE_SHA256, msg, msgSz, sig, sigSz,
            &verified);
    if (ret != 0)
        return -10167;
    if (!verified)
        return -10168;

    /* Check that the KPAK is converted from montgomery form. */
    ret = wc_ValidateEccsiPair(pub, WC_HASH_TYPE_SHA256, id, idSz, ssk, pvt,
            &valid);
    if (ret != 0)
        return -10169;
    if (!valid)
        return -10170;

    /* Check that KPAK can converted to Montgomery form again. */
    ret = wc_VerifyEccsiHash(priv, WC_HASH_TYPE_SHA256, msg, msgSz, sig, sigSz,
            &verified);
    if (ret != 0)
        return -10171;
    if (!verified)
        return -10172;

    /* Check that the KPAK is converted from montgomery form. */
    ret = eccsi_imp_exp_pubkey_test(priv, pub);
    if (ret != 0)
        return ret;

    return 0;
}

int eccsi_test(void)
{
    int ret = 0;
    WC_RNG rng;
    int rng_inited = 0;
    EccsiKey* priv = NULL;
    EccsiKey* pub  = NULL;
    mp_int* ssk    = NULL;
    ecc_point* pvt = NULL;

    priv = (EccsiKey*)XMALLOC(sizeof(EccsiKey), HEAP_HINT,
            DYNAMIC_TYPE_TMP_BUFFER);
    if (priv == NULL)
        ret = -10205;
    else
        XMEMSET(priv, 0, sizeof(*priv));

    if (ret == 0) {
        pub = (EccsiKey*)XMALLOC(sizeof(EccsiKey), HEAP_HINT,
            DYNAMIC_TYPE_TMP_BUFFER);
        if (pub == NULL)
            ret = -10206;
        else
            XMEMSET(pub, 0, sizeof(*pub));
    }

    if (ret == 0) {
        ssk = (mp_int*)XMALLOC(sizeof(mp_int), HEAP_HINT,
                DYNAMIC_TYPE_TMP_BUFFER);
        if (ssk == NULL)
            ret = -10207;
        else
            XMEMSET(ssk, 0, sizeof(*ssk));
    }

    if (ret == 0) {
    #ifndef HAVE_FIPS
        ret = wc_InitRng_ex(&rng, HEAP_HINT, devId);
    #else
        ret = wc_InitRng(&rng);
    #endif
        if (ret != 0)
            ret = -10200;
        else
            rng_inited = 1;
    }

    if (ret == 0) {
        pvt = wc_ecc_new_point();
        if (pvt == NULL)
            ret = -10201;
    }

    if (ret == 0) {
        ret = mp_init(ssk);
        if (ret != 0)
            ret = -10202;
    }

    if (ret == 0) {
        ret = eccsi_api_test(&rng, priv, ssk, pvt);
    }

    if (ret == 0) {
        ret = wc_InitEccsiKey(pub, HEAP_HINT, INVALID_DEVID);
        if (ret != 0)
            ret = -10203;
    }

    if (ret == 0) {
        ret = wc_InitEccsiKey(priv, HEAP_HINT, INVALID_DEVID);
        if (ret != 0)
            ret = -10204;
    }

    if (ret == 0) {
        ret = eccsi_kat_verify_test(pub, pvt);
    }

    if (ret == 0) {
        ret = eccsi_make_key_test(priv, pub, &rng, ssk, pvt);
    }

    if (ret == 0) {
        ret = eccsi_sign_verify_test(priv, pub, &rng, ssk, pvt);
    }

    if (pvt != NULL)
        wc_ecc_del_point(pvt);
    if (rng_inited)
        wc_FreeRng(&rng);
    if (ssk != NULL) {
        mp_free(ssk);
        XFREE(ssk, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    }
    if (pub != NULL) {
        wc_FreeEccsiKey(pub);
        XFREE(pub, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    }
    if (priv != NULL) {
        wc_FreeEccsiKey(priv);
        XFREE(priv, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    }

    return ret;
}
#endif /* WOLFCRYPT_HAVE_ECCSI */

#ifdef WOLFCRYPT_HAVE_SAKKE
static int sakke_api_test(WC_RNG* rng, SakkeKey* key, ecc_point* rsk)
{
    int ret;
    byte id[1] = { 0x00 };
    int valid;
    byte data[256];
    word32 sz;
    byte auth[257];
    word16 authSz;
    byte ssv[256];
    word16 ssvSz;
    word32 len;

    ret = wc_InitSakkeKey_ex(NULL, 128, ECC_SAKKE_1, NULL, INVALID_DEVID);
    if (ret != BAD_FUNC_ARG)
        return -10205;
    ret = wc_InitSakkeKey_ex(NULL, 128, ECC_SAKKE_1, HEAP_HINT, INVALID_DEVID);
    if (ret != BAD_FUNC_ARG)
        return -10206;

    wc_FreeSakkeKey(NULL);

    XMEMSET(key, 0, sizeof(*key));
    wc_FreeSa