/*
 * Copyright 2002-2008 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.springframework.config.java.internal.model;

import static org.junit.Assert.assertEquals;

import org.junit.Before;
import org.junit.Test;

import org.springframework.config.java.model.ModelClass;


/**
 * Unit tests for {@link UsageError} base class.
 *
 * @author  Chris Beams
 */
public class UsageErrorTests {
    private UsageError error;

    /** Set up a a simple UsageError implementation on the fly. */
    @Before
    public void setUp() {
        ModelClass clazz = new ModelClass("org.foo.Bar");
        clazz.setSource("org/foo/Bar.java");

        error = new UsageError(clazz, 42) {
            @Override
            public String getDescription() { return "Description"; }
        };
    }

    /**
     * {@link UsageError#getAttributedDescription()} should return the value of
     * {@link UsageError#getDescription()} prefixed with source information.
     */
    @Test
    public void testGetAttributedDescription() {
        String expected = "org/foo/Bar.java:42: Description";
        String actual = error.getAttributedDescription();

        assertEquals(expected, actual);
    }

    /**
     * {@link UsageError#toString()} should return the value
     * of.{@link UsageError#getAttributedDescription()}
     */
    @Test
    public void testToString() {
        String expected = error.getAttributedDescription();
        String actual = error.toString();

        assertEquals(expected, actual);
    }

    /**
     * Special case: if modelClass is null, then getAttributedDescription should render without
     * source information
     */
    @Test
    public void testNullClass() {
        error = new UsageError(null, 0) {
            @Override
            public String getDescription() { return "XDesc"; }
        };

        String description = error.getDescription();

        assertEquals("XDesc", description);

        String toString = error.toString();

        assertEquals(description, toString);
    }

}
