package org.springframework.uaa.client.internal;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.prefs.BackingStoreException;

import org.springframework.uaa.client.ProxyService;
import org.springframework.uaa.client.TransmissionAwareUaaService;
import org.springframework.uaa.client.TransmissionService;
import org.springframework.uaa.client.VersionHelper;
import org.springframework.uaa.client.protobuf.UaaClient.Privacy.PrivacyLevel;
import org.springframework.uaa.client.protobuf.UaaClient.Product;
import org.springframework.uaa.client.protobuf.UaaClient.UaaEnvelope;
import org.springframework.uaa.client.util.Assert;
import org.springframework.uaa.client.util.Base64;

/**
 * Extension to the default {@link UaaServiceImpl} that handles automatic uploads of UAA usage data
 * to Amazon S3 as well as downloads of server-side configured parameters like timeouts etc.
 * 
 * @author Christian Dupuis
 * @since 1.0.1
 */
public class TransmissionAwareUaaServiceImpl extends UaaServiceImpl implements TransmissionAwareUaaService {

	private static final String LAST_DOWNLOAD_TIMESTAMP = "last_download_timestamp";
	private static final String LAST_UPLOAD_TIMESTAMP = "last_upload_timestamp";
	private static final String PING_THREAD_NAME_TEMPLATE = "Ping (%s/%s.%s.%s)";
	private static final String THREAD_NAME_TEMPLATE = "Synchronizer (%s/%s.%s.%s)";
	static final String DOWNLOAD_INTERVAL_KEY = "download_interval";
	static final String PING_INTERVAL_KEY = "ping_interval";
	static final String UPLOAD_INTERVAL_KEY = "upload_interval";
	static long THREAD_DELAY = 10000L; // 10sec delay before the transmission runs for the first time; this is important to prevent OSGi classloading issues on Equinox
	static long THREAD_SLEEPTIME = 1000L * 60L * 5L; // 5min sleep time between checks for up- and downloads
	static long UPLOAD_INTERVAL_DEFAULT = 1000L * 60L * 60L; // 1h interval for usage data uploads
	static long DOWNLOAD_INTERVAL_DEFAULT = 1000L * 60L * 60L * 24L; // 24h interval for DetectedProducts downloads
	static long PING_INTERVAL_DEFAULT = 1000L * 60L * 60L * 24L; // 24h interval for privacy level uploads
	
	private TransmissionService transmissionService;
	private Thread transmissionThread;
	private volatile boolean shouldExit = false;

	/**
	 * Default constructor to create an instance of {@link TransmissionAwareUaaServiceImpl}. This 
	 * will use a default {@link TransmissionService} and {@link ProxyService} implementations.
	 */
	public TransmissionAwareUaaServiceImpl() {
		this(new JdkUrlTransmissionServiceImpl(new BasicProxyService()));
	}

	/**
	 * Constructor allowing to set the {@link TransmissionService} to use.
	 * 
	 * @param transmissionService 
	 */
	public TransmissionAwareUaaServiceImpl(TransmissionService transmissionService) {
		Assert.notNull(transmissionService, "TransmissionService is required");
		this.transmissionService = transmissionService;
		
		// Register shutdown hook to cleanly shutdown the waiting thread
		Runtime.getRuntime().addShutdownHook(new Thread() {
			@Override
			public void run() {
				stopTransmissionThread();
			}
		});

		// Start up scheduling thread
		this.transmissionThread = new Thread(new UaaTransmissionThread(), getThreadName(THREAD_NAME_TEMPLATE));
		this.transmissionThread.setDaemon(true);
		this.transmissionThread.start();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setPrivacyLevel(PrivacyLevel privacyLevel) {
		// suppress update to privacy level in case it is not a change
		if (privacyLevel == getPrivacyLevel()) {
			return;
		}

		super.setPrivacyLevel(privacyLevel);
		
		// If the privacy level is being set to decline or undecided send message immediately
		if (privacyLevel == PrivacyLevel.DECLINE_TOU || privacyLevel == PrivacyLevel.UNDECIDED_TOU) {
			// Since this call can stall on unreliable network connections run it an async fashion
			new Thread(new Runnable() {
				public void run() {
					uploadUaaEnvelope();
				}
			}, getThreadName(PING_THREAD_NAME_TEMPLATE)).start();
		}
		
		// If the privacy level is changed to anything other then DECLINE_TOU and UNDECIDED_TOU
		// the change will be picked up by the transmission background job. So don't do anything here.
	}
	
	/**
	 * {@inheritDoc}
	 */
	public void requestTransmission() {
		// Only upload if the user accepted the terms of use. We cannot rely on the client to do this check.
		if (isUaaTermsOfUseAccepted()) {
			// Since this call can stall on unreliable network connections run it an async fashion
			new Thread(new Runnable() {
				public void run() {
					uploadUaaEnvelope();
				}
			}, getThreadName(PING_THREAD_NAME_TEMPLATE)).start();
		}
	}

	/**
	 * Download <code>uaa-client.xml</code> and present it to {@link UaaConfigurationProcessor#updateConfiguration()}.
	 * <p>
	 * Note: this method might stall in case of mis-configured network connections. Therefore be
	 * careful from the is called as it can easily block UI or other important user-related threads.
	 *   
	 * @return <code>true</code> or <code>false</code> to indicate successful update of internal configuration
	 */
	private boolean downloadAndUpdateConfiguration() {
		InputStream configuration = null;
		InputStream configurationSignature = null;
		try {
			configuration = transmissionService.download(UaaConfigurationProcessor.UAA_URL);
			configurationSignature = transmissionService.download(UaaConfigurationProcessor.SIGNATURE_URL);
			return UaaConfigurationProcessor.updateConfiguration(configuration, configurationSignature);
		}
		catch (IOException e) {}
		finally {
			storeTimestamp(LAST_DOWNLOAD_TIMESTAMP, System.currentTimeMillis());
			
			// Safely close the streams
			if (configuration != null) {
				try { configuration.close(); }
				catch (IOException e) {}
			}
			if (configurationSignature != null) {
				try { configurationSignature.close(); }
				catch (IOException e) {}
			}
		}
		return false;
	}

	/**
	 * Returns a String useful for naming threads. The given <code>template</code> will have the
	 * UAA name and version number in it. 
	 */
	private String getThreadName(String template) {
		Product uaaProduct = VersionHelper.getUaa();
		return String.format(template, uaaProduct.getName(), uaaProduct.getMajorVersion(), uaaProduct.getMinorVersion(),uaaProduct.getPatchVersion());
	}

	/**
	 * Get the interval stored under the <code>key</code>. If no value is stored <code>defaultValue</code> is returned.
	 * 
	 * @param key the key for the interval to load
	 * @param defaultValue the default to return
	 * @return the interval or <code>defaultValue</code> if no interval is stored
	 */
	private long retrieveInterval(String key, Long defaultValue) {
		try { P.sync(); }
		catch (BackingStoreException e) {}
		return P.getLong(key, defaultValue);
	}
	
	/**
	 * Get the timestamp stored under the <code>key</code>.
	 * 
	 * @param key the key for the timestamp to load
	 * @return the timestamp or 0 if no timestamp is stored
	 */
	private long retrieveTimestamp(String key) {
		try { P.sync(); }
		catch (BackingStoreException e) {}
		return P.getLong(key, 0L);
	}
	
	/**
	 * Write out a timestamp under a given <code>key</code> to the backend store.
	 * 
	 * @param key the key to store the timestamp under
	 * @param timestamp the timestamp to store
	 */
	private void storeTimestamp(String key, long timestamp) {
		try {
			P.putLong(key, timestamp);
			P.flush();
		}
		catch (BackingStoreException e) {}
	}
	
	/**
	 * Upload the given {@link UaaEnvelope} using the {@link TransmissionService}.
	 * <p>
	 * Note: this method might stall in case of mis-configured network connections. Therefore be
	 * careful from the is called as it can easily block UI or other important user-related threads.
	 * 
	 * @param envelope the {@link UaaEnvelope} to upload
	 * @return <code>true</code> or <code>false</code> to indicate successful upload
	 */
	private boolean uploadUaaEnvelope() {
		InputStream is = null;
		try {
			
			// Obtain the populated envelope
			UaaEnvelope env = createUaaEnvelope();

			// In order to safe bandwidth GZIP compress the payload if zipped payload is shorter than un-compressed
			String uncompressed = Base64.encodeBytes(env.toByteArray());
			String compressed = Base64.encodeBytes(env.toByteArray(), Base64.GZIP);
			if (compressed.length() < uncompressed.length()) {
				is = new ByteArrayInputStream(compressed.getBytes("UTF-8"));
			}
			else {
				is = new ByteArrayInputStream(compressed.getBytes("UTF-8"));
			}
			
			// Upload now
			boolean success = transmissionService.upload(is);

			// If upload was successful set last upload time to now
			if (success) {
				storeTimestamp(LAST_UPLOAD_TIMESTAMP, System.currentTimeMillis());
			}
			
			return success;
		}
		catch (IOException e) {}
		finally {
			if (is != null) { 
				try { is.close();	}
				catch (IOException e) {}
			}
		}
		return false;
	}
	
	String encodeUaaEnvelope(UaaEnvelope env) throws IOException {
		String uncompressed = Base64.encodeBytes(env.toByteArray());
		String compressed = Base64.encodeBytes(env.toByteArray(), Base64.GZIP);
		if (compressed.length() < uncompressed.length()) {
			return compressed;
		}
		else {
			return uncompressed;
		}
	}
	
	/**
	 * Stop the internal transmission job.
	 */
	void stopTransmissionThread() {
		shouldExit = true;
		if (transmissionThread != null) {
			transmissionThread.interrupt();
		}
	}

	/**
	 * {@link Runnable} implementation that coordinates uploads of {@link UaaEnvelope} messages and
	 * downloads of {@link UaaDetectedProductsImpl} information. 
	 */
	private class UaaTransmissionThread implements Runnable {

		/**
		 * {@inheritDoc}
		 */
		public void run() {

			// Wait a defined period; enables testing
			if (THREAD_DELAY > 0) {
				try { Thread.sleep(THREAD_DELAY); }
				catch (InterruptedException e) {}
			}

			while (true) {
				// Test if this thread needs exiting
				if (shouldExit) {
					return;
				}
				
				// Get the required timestamp information from the Preferences API 
				long lastUpload = retrieveTimestamp(LAST_UPLOAD_TIMESTAMP);
				long lastDownload = retrieveTimestamp(LAST_DOWNLOAD_TIMESTAMP);
				long now = System.currentTimeMillis();

				// Handle data upload
				if (!isUaaTermsOfUseAccepted()) {
					// Send daily pings to us to indicate un-approved or undecided installs
					if (lastUpload + retrieveInterval(PING_INTERVAL_KEY, PING_INTERVAL_DEFAULT) < now) {
						uploadUaaEnvelope();
					}
				}
				else {
					// Upload UAA data every hour
					if (lastUpload + retrieveInterval(UPLOAD_INTERVAL_KEY, UPLOAD_INTERVAL_DEFAULT) < now) {
						
						// Per UAA contract we should clear the backend store on each successful upload 
						if (uploadUaaEnvelope()) {
							clearIfPossible();
						}
					}
				}

				// Handle data download
				if (lastDownload + retrieveInterval(DOWNLOAD_INTERVAL_KEY, DOWNLOAD_INTERVAL_DEFAULT) < now) {
					downloadAndUpdateConfiguration();
				}

				try {
					// Wait for defined period before checking again but only if the running thread
					// hasn't been interrupted already
					if (!Thread.interrupted() && !shouldExit) { 
						Thread.sleep(THREAD_SLEEPTIME);
					}
				}
				catch (InterruptedException e) {
					// If we get an interrupt make exit to allow clean shutdown of JVM
					return;
				}
			}
		}
	}

}
