/*
 * Decompiled with CFR 0.152.
 */
package org.gradle.cache.internal.btree;

import com.google.common.collect.ImmutableSet;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import org.gradle.api.UncheckedIOException;
import org.gradle.cache.internal.btree.Block;
import org.gradle.cache.internal.btree.BlockPayload;
import org.gradle.cache.internal.btree.BlockPointer;
import org.gradle.cache.internal.btree.BlockStore;
import org.gradle.cache.internal.btree.CachingBlockStore;
import org.gradle.cache.internal.btree.CorruptedCacheException;
import org.gradle.cache.internal.btree.FileBackedBlockStore;
import org.gradle.cache.internal.btree.FreeListBlockStore;
import org.gradle.cache.internal.btree.KeyHasher;
import org.gradle.cache.internal.btree.StateCheckBlockStore;
import org.gradle.internal.io.StreamByteBuffer;
import org.gradle.internal.serialize.Decoder;
import org.gradle.internal.serialize.Encoder;
import org.gradle.internal.serialize.Serializer;
import org.gradle.internal.serialize.kryo.KryoBackedDecoder;
import org.gradle.internal.serialize.kryo.KryoBackedEncoder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class BTreePersistentIndexedCache<K, V> {
    private static final Logger LOGGER = LoggerFactory.getLogger(BTreePersistentIndexedCache.class);
    private final File cacheFile;
    private final KeyHasher<K> keyHasher;
    private final Serializer<V> serializer;
    private final short maxChildIndexEntries;
    private final int minIndexChildNodes;
    private final StateCheckBlockStore store;
    private HeaderBlock header;

    public BTreePersistentIndexedCache(File cacheFile, Serializer<K> keySerializer, Serializer<V> valueSerializer) {
        this(cacheFile, keySerializer, valueSerializer, 512, 512);
    }

    public BTreePersistentIndexedCache(File cacheFile, Serializer<K> keySerializer, Serializer<V> valueSerializer, short maxChildIndexEntries, int maxFreeListEntries) {
        this.cacheFile = cacheFile;
        this.keyHasher = new KeyHasher<K>(keySerializer);
        this.serializer = valueSerializer;
        this.maxChildIndexEntries = maxChildIndexEntries;
        this.minIndexChildNodes = maxChildIndexEntries / 2;
        CachingBlockStore cachingStore = new CachingBlockStore(new FileBackedBlockStore(cacheFile), (Collection<Class<? extends BlockPayload>>)ImmutableSet.of(IndexBlock.class, FreeListBlockStore.FreeListBlock.class));
        this.store = new StateCheckBlockStore(new FreeListBlockStore(cachingStore, maxFreeListEntries));
        try {
            this.open();
        }
        catch (Exception e) {
            throw new UncheckedIOException(String.format("Could not open %s.", this), (Throwable)e);
        }
    }

    public String toString() {
        return "cache " + this.cacheFile.getName() + " (" + this.cacheFile + ")";
    }

    private void open() throws Exception {
        LOGGER.debug("Opening {}", (Object)this);
        try {
            this.doOpen();
        }
        catch (CorruptedCacheException e) {
            this.rebuild();
        }
    }

    private void doOpen() throws Exception {
        BlockStore.Factory factory = new BlockStore.Factory(){

            @Override
            public Object create(Class<? extends BlockPayload> type) {
                if (type == HeaderBlock.class) {
                    return new HeaderBlock();
                }
                if (type == IndexBlock.class) {
                    return new IndexBlock();
                }
                if (type == DataBlock.class) {
                    return new DataBlock();
                }
                throw new UnsupportedOperationException();
            }
        };
        Runnable initAction = new Runnable(){

            @Override
            public void run() {
                BTreePersistentIndexedCache.this.header = new HeaderBlock();
                BTreePersistentIndexedCache.this.store.write(BTreePersistentIndexedCache.this.header);
                BTreePersistentIndexedCache.this.header.index.newRoot();
                BTreePersistentIndexedCache.this.store.flush();
            }
        };
        this.store.open(initAction, factory);
        this.header = this.store.readFirst(HeaderBlock.class);
    }

    public V get(K key) {
        try {
            try {
                DataBlock block = this.header.getRoot().get(key);
                if (block != null) {
                    return block.getValue();
                }
                return null;
            }
            catch (CorruptedCacheException e) {
                this.rebuild();
                return null;
            }
        }
        catch (Exception e) {
            throw new UncheckedIOException(String.format("Could not read entry '%s' from %s.", key, this), (Throwable)e);
        }
    }

    public void put(K key, V value) {
        try {
            long hashCode = this.keyHasher.getHashCode(key);
            Lookup lookup = this.header.getRoot().find(hashCode);
            DataBlock newBlock = null;
            if (lookup.entry != null) {
                DataBlock block = this.store.read(lookup.entry.dataBlock, DataBlock.class);
                DataBlockUpdateResult updateResult = block.useNewValue(value);
                if (updateResult.isFailed()) {
                    this.store.remove(block);
                    newBlock = new DataBlock(value, updateResult.getSerializedValue());
                }
            } else {
                newBlock = new DataBlock(value);
            }
            if (newBlock != null) {
                this.store.write(newBlock);
                lookup.indexBlock.put(hashCode, newBlock.getPos());
            }
            this.store.flush();
        }
        catch (Exception e) {
            throw new UncheckedIOException(String.format("Could not add entry '%s' to %s.", key, this), (Throwable)e);
        }
    }

    public void remove(K key) {
        try {
            Lookup lookup = this.header.getRoot().find(key);
            if (lookup.entry == null) {
                return;
            }
            lookup.indexBlock.remove(lookup.entry);
            DataBlock block = this.store.read(lookup.entry.dataBlock, DataBlock.class);
            this.store.remove(block);
            this.store.flush();
        }
        catch (Exception e) {
            throw new UncheckedIOException(String.format("Could not remove entry '%s' from %s.", key, this), (Throwable)e);
        }
    }

    private IndexBlock load(BlockPointer pos, IndexRoot root, IndexBlock parent, int index) {
        IndexBlock block = this.store.read(pos, IndexBlock.class);
        block.root = root;
        block.parent = parent;
        block.parentEntryIndex = index;
        return block;
    }

    public void reset() {
        this.close();
        try {
            this.open();
        }
        catch (Exception e) {
            throw new UncheckedIOException((Throwable)e);
        }
    }

    public void close() {
        LOGGER.debug("Closing {}", (Object)this);
        try {
            this.store.close();
        }
        catch (Exception e) {
            throw new UncheckedIOException((Throwable)e);
        }
    }

    public boolean isOpen() {
        return this.store.isOpen();
    }

    private void rebuild() throws Exception {
        LOGGER.warn("{} is corrupt. Discarding.", (Object)this);
        this.store.clear();
        this.close();
        this.doOpen();
    }

    public void verify() {
        try {
            this.doVerify();
        }
        catch (Exception e) {
            throw new UncheckedIOException(String.format("Some problems were found when checking the integrity of %s.", this), (Throwable)e);
        }
    }

    private void doVerify() throws Exception {
        ArrayList<BlockPayload> blocks = new ArrayList<BlockPayload>();
        HeaderBlock header = this.store.readFirst(HeaderBlock.class);
        blocks.add(header);
        this.verifyTree(header.getRoot(), "", blocks, Long.MAX_VALUE, true);
        Collections.sort(blocks, new Comparator<BlockPayload>(){

            @Override
            public int compare(BlockPayload block, BlockPayload block1) {
                return block.getPos().compareTo(block1.getPos());
            }
        });
        for (int i = 0; i < blocks.size() - 1; ++i) {
            Block b1 = ((BlockPayload)blocks.get(i)).getBlock();
            Block b2 = ((BlockPayload)blocks.get(i + 1)).getBlock();
            if (b1.getPos().getPos() + (long)b1.getSize() <= b2.getPos().getPos()) continue;
            throw new IOException(String.format("%s overlaps with %s", b1, b2));
        }
    }

    private void verifyTree(IndexBlock current, String prefix, Collection<BlockPayload> blocks, long maxValue, boolean loadData) throws Exception {
        boolean isLeaf;
        blocks.add(current);
        if (!prefix.equals("") && current.entries.size() < this.maxChildIndexEntries / 2) {
            throw new IOException(String.format("Too few entries found in %s", current));
        }
        if (current.entries.size() > this.maxChildIndexEntries) {
            throw new IOException(String.format("Too many entries found in %s", current));
        }
        boolean bl = isLeaf = current.entries.size() == 0 || ((IndexEntry)((IndexBlock)current).entries.get((int)0)).childIndexBlock.isNull();
        if (isLeaf ^ current.tailPos.isNull()) {
            throw new IOException(String.format("Mismatched leaf/tail-node in %s", current));
        }
        long min = Long.MIN_VALUE;
        for (IndexEntry entry : current.entries) {
            if (isLeaf ^ entry.childIndexBlock.isNull()) {
                throw new IOException(String.format("Mismatched leaf/non-leaf entry in %s", current));
            }
            if (entry.hashCode >= maxValue || entry.hashCode <= min) {
                throw new IOException(String.format("Out-of-order key in %s", current));
            }
            min = entry.hashCode;
            if (!entry.childIndexBlock.isNull()) {
                IndexBlock child = this.store.read(entry.childIndexBlock, IndexBlock.class);
                this.verifyTree(child, "   " + prefix, blocks, entry.hashCode, loadData);
            }
            if (!loadData) continue;
            DataBlock block = this.store.read(entry.dataBlock, DataBlock.class);
            blocks.add(block);
        }
        if (!current.tailPos.isNull()) {
            IndexBlock tail = this.store.read(current.tailPos, IndexBlock.class);
            this.verifyTree(tail, "   " + prefix, blocks, maxValue, loadData);
        }
    }

    private static class DataBlockUpdateResult {
        private static final DataBlockUpdateResult SUCCESS = new DataBlockUpdateResult(true, null);
        private final boolean success;
        private final StreamByteBuffer serializedValue;

        private DataBlockUpdateResult(boolean success, StreamByteBuffer serializedValue) {
            this.success = success;
            this.serializedValue = serializedValue;
        }

        static DataBlockUpdateResult success() {
            return SUCCESS;
        }

        static DataBlockUpdateResult failed(StreamByteBuffer serializedValue) {
            return new DataBlockUpdateResult(false, serializedValue);
        }

        public boolean isFailed() {
            return !this.success;
        }

        public StreamByteBuffer getSerializedValue() {
            return this.serializedValue;
        }
    }

    private class DataBlock
    extends BlockPayload {
        private int size;
        private StreamByteBuffer buffer;
        private V value;

        private DataBlock() {
        }

        public DataBlock(V value) throws Exception {
            this.value = value;
            this.setValue(value);
            this.size = this.buffer.totalBytesUnread();
        }

        public DataBlock(V value, StreamByteBuffer buffer) throws Exception {
            this.value = value;
            this.buffer = buffer;
            this.size = buffer.totalBytesUnread();
        }

        public void setValue(V value) throws Exception {
            this.buffer = StreamByteBuffer.createWithChunkSizeInDefaultRange((int)this.size);
            KryoBackedEncoder encoder = new KryoBackedEncoder(this.buffer.getOutputStream());
            BTreePersistentIndexedCache.this.serializer.write((Encoder)encoder, value);
            encoder.flush();
        }

        public V getValue() throws Exception {
            if (this.value == null) {
                this.value = BTreePersistentIndexedCache.this.serializer.read((Decoder)new KryoBackedDecoder(this.buffer.getInputStream()));
                this.buffer = null;
            }
            return this.value;
        }

        @Override
        protected byte getType() {
            return 51;
        }

        @Override
        protected int getSize() {
            return 8 + this.size;
        }

        @Override
        public void read(DataInputStream instr) throws Exception {
            this.size = instr.readInt();
            int bytes = instr.readInt();
            this.buffer = StreamByteBuffer.of((InputStream)instr, (int)bytes);
        }

        @Override
        public void write(DataOutputStream outstr) throws Exception {
            outstr.writeInt(this.size);
            outstr.writeInt(this.buffer.totalBytesUnread());
            this.buffer.writeTo((OutputStream)outstr);
            this.buffer = null;
        }

        public DataBlockUpdateResult useNewValue(V value) throws Exception {
            boolean ok;
            this.setValue(value);
            boolean bl = ok = this.buffer.totalBytesUnread() <= this.size;
            if (ok) {
                this.value = value;
                BTreePersistentIndexedCache.this.store.write(this);
                return DataBlockUpdateResult.success();
            }
            return DataBlockUpdateResult.failed(this.buffer);
        }
    }

    private class Lookup {
        final IndexBlock indexBlock;
        final IndexEntry entry;

        private Lookup(IndexBlock indexBlock, IndexEntry entry) {
            this.indexBlock = indexBlock;
            this.entry = entry;
        }
    }

    private static class IndexEntry
    implements Comparable<IndexEntry> {
        long hashCode;
        BlockPointer dataBlock;
        BlockPointer childIndexBlock;

        private IndexEntry() {
        }

        private IndexEntry(long hashCode) {
            this.hashCode = hashCode;
        }

        @Override
        public int compareTo(IndexEntry indexEntry) {
            if (this.hashCode > indexEntry.hashCode) {
                return 1;
            }
            if (this.hashCode < indexEntry.hashCode) {
                return -1;
            }
            return 0;
        }
    }

    private class IndexBlock
    extends BlockPayload {
        private final List<IndexEntry> entries = new ArrayList<IndexEntry>();
        private BlockPointer tailPos = BlockPointer.start();
        private IndexBlock parent;
        private int parentEntryIndex;
        private IndexRoot root;

        private IndexBlock() {
        }

        @Override
        protected byte getType() {
            return 119;
        }

        @Override
        protected int getSize() {
            return 12 + 24 * BTreePersistentIndexedCache.this.maxChildIndexEntries;
        }

        @Override
        public void read(DataInputStream instr) throws IOException {
            int count = instr.readInt();
            this.entries.clear();
            for (int i = 0; i < count; ++i) {
                IndexEntry entry = new IndexEntry();
                entry.hashCode = instr.readLong();
                entry.dataBlock = BlockPointer.pos(instr.readLong());
                entry.childIndexBlock = BlockPointer.pos(instr.readLong());
                this.entries.add(entry);
            }
            this.tailPos = BlockPointer.pos(instr.readLong());
        }

        @Override
        public void write(DataOutputStream outstr) throws IOException {
            outstr.writeInt(this.entries.size());
            for (IndexEntry entry : this.entries) {
                outstr.writeLong(entry.hashCode);
                outstr.writeLong(entry.dataBlock.getPos());
                outstr.writeLong(entry.childIndexBlock.getPos());
            }
            outstr.writeLong(this.tailPos.getPos());
        }

        public void put(long hashCode, BlockPointer pos) throws Exception {
            IndexEntry entry;
            int index = Collections.binarySearch(this.entries, new IndexEntry(hashCode));
            if (index >= 0) {
                entry = this.entries.get(index);
            } else {
                assert (this.tailPos.isNull());
                entry = new IndexEntry();
                entry.hashCode = hashCode;
                entry.childIndexBlock = BlockPointer.start();
                index = -index - 1;
                this.entries.add(index, entry);
            }
            entry.dataBlock = pos;
            BTreePersistentIndexedCache.this.store.write(this);
            this.maybeSplit();
        }

        private void maybeSplit() throws Exception {
            if (this.entries.size() > BTreePersistentIndexedCache.this.maxChildIndexEntries) {
                int splitPos = this.entries.size() / 2;
                IndexEntry splitEntry = this.entries.remove(splitPos);
                if (this.parent == null) {
                    this.parent = this.root.newRoot();
                }
                IndexBlock sibling = new IndexBlock();
                BTreePersistentIndexedCache.this.store.write(sibling);
                List<IndexEntry> siblingEntries = this.entries.subList(splitPos, this.entries.size());
                sibling.entries.addAll(siblingEntries);
                siblingEntries.clear();
                sibling.tailPos = this.tailPos;
                this.tailPos = splitEntry.childIndexBlock;
                splitEntry.childIndexBlock = BlockPointer.start();
                this.parent.add(this, splitEntry, sibling);
            }
        }

        private void add(IndexBlock left, IndexEntry entry, IndexBlock right) throws Exception {
            int index = left.parentEntryIndex;
            if (index < this.entries.size()) {
                IndexEntry parentEntry = this.entries.get(index);
                assert (parentEntry.childIndexBlock.equals(left.getPos()));
                parentEntry.childIndexBlock = right.getPos();
            } else {
                assert (index == this.entries.size() && (this.tailPos.isNull() || this.tailPos.equals(left.getPos())));
                this.tailPos = right.getPos();
            }
            this.entries.add(index, entry);
            entry.childIndexBlock = left.getPos();
            BTreePersistentIndexedCache.this.store.write(this);
            this.maybeSplit();
        }

        public DataBlock get(K key) throws Exception {
            Lookup lookup = this.find(key);
            if (lookup.entry == null) {
                return null;
            }
            return BTreePersistentIndexedCache.this.store.read(lookup.entry.dataBlock, DataBlock.class);
        }

        public Lookup find(K key) throws Exception {
            long checksum = BTreePersistentIndexedCache.this.keyHasher.getHashCode(key);
            return this.find(checksum);
        }

        private Lookup find(long hashCode) throws Exception {
            int index = Collections.binarySearch(this.entries, new IndexEntry(hashCode));
            if (index >= 0) {
                return new Lookup(this, this.entries.get(index));
            }
            BlockPointer childBlockPos = (index = -index - 1) == this.entries.size() ? this.tailPos : this.entries.get((int)index).childIndexBlock;
            if (childBlockPos.isNull()) {
                return new Lookup(this, null);
            }
            IndexBlock childBlock = BTreePersistentIndexedCache.this.load(childBlockPos, this.root, this, index);
            return childBlock.find(hashCode);
        }

        public void remove(IndexEntry entry) throws Exception {
            int index = this.entries.indexOf(entry);
            assert (index >= 0);
            this.entries.remove(index);
            BTreePersistentIndexedCache.this.store.write(this);
            if (entry.childIndexBlock.isNull()) {
                this.maybeMerge();
            } else {
                IndexBlock leafBlock = BTreePersistentIndexedCache.this.load(entry.childIndexBlock, this.root, this, index);
                leafBlock = leafBlock.findHighestLeaf();
                IndexEntry highestEntry = leafBlock.entries.remove(leafBlock.entries.size() - 1);
                highestEntry.childIndexBlock = entry.childIndexBlock;
                this.entries.add(index, highestEntry);
                BTreePersistentIndexedCache.this.store.write(leafBlock);
                leafBlock.maybeMerge();
            }
        }

        private void maybeMerge() throws Exception {
            if (this.parent == null) {
                if (this.entries.size() == 0 && !this.tailPos.isNull()) {
                    BTreePersistentIndexedCache.this.header.index.setRootPos(this.tailPos);
                    BTreePersistentIndexedCache.this.store.remove(this);
                }
                return;
            }
            if (this.entries.size() >= BTreePersistentIndexedCache.this.minIndexChildNodes) {
                return;
            }
            IndexBlock left = this.parent.getPrevious(this);
            if (left != null) {
                assert (this.entries.size() + left.entries.size() <= BTreePersistentIndexedCache.this.maxChildIndexEntries * 2);
                if (left.entries.size() > BTreePersistentIndexedCache.this.minIndexChildNodes) {
                    left.mergeFrom(this);
                    left.maybeSplit();
                    return;
                }
                left.mergeFrom(this);
                this.parent.maybeMerge();
                return;
            }
            IndexBlock right = this.parent.getNext(this);
            if (right != null) {
                assert (this.entries.size() + right.entries.size() <= BTreePersistentIndexedCache.this.maxChildIndexEntries * 2);
                if (right.entries.size() > BTreePersistentIndexedCache.this.minIndexChildNodes) {
                    this.mergeFrom(right);
                    this.maybeSplit();
                    return;
                }
                this.mergeFrom(right);
                this.parent.maybeMerge();
                return;
            }
            throw new IllegalStateException(String.format("%s does not have any siblings.", this.getBlock()));
        }

        private void mergeFrom(IndexBlock right) throws Exception {
            IndexEntry newChildEntry = this.parent.entries.remove(this.parentEntryIndex);
            if (right.getPos().equals(this.parent.tailPos)) {
                this.parent.tailPos = this.getPos();
            } else {
                IndexEntry newParentEntry = this.parent.entries.get(this.parentEntryIndex);
                assert (newParentEntry.childIndexBlock.equals(right.getPos()));
                newParentEntry.childIndexBlock = this.getPos();
            }
            this.entries.add(newChildEntry);
            this.entries.addAll(right.entries);
            newChildEntry.childIndexBlock = this.tailPos;
            this.tailPos = right.tailPos;
            BTreePersistentIndexedCache.this.store.write(this.parent);
            BTreePersistentIndexedCache.this.store.write(this);
            BTreePersistentIndexedCache.this.store.remove(right);
        }

        private IndexBlock getNext(IndexBlock indexBlock) throws Exception {
            int index = indexBlock.parentEntryIndex + 1;
            if (index > this.entries.size()) {
                return null;
            }
            if (index == this.entries.size()) {
                return BTreePersistentIndexedCache.this.load(this.tailPos, this.root, this, index);
            }
            return BTreePersistentIndexedCache.this.load(this.entries.get((int)index).childIndexBlock, this.root, this, index);
        }

        private IndexBlock getPrevious(IndexBlock indexBlock) throws Exception {
            int index = indexBlock.parentEntryIndex - 1;
            if (index < 0) {
                return null;
            }
            return BTreePersistentIndexedCache.this.load(this.entries.get((int)index).childIndexBlock, this.root, this, index);
        }

        private IndexBlock findHighestLeaf() throws Exception {
            if (this.tailPos.isNull()) {
                return this;
            }
            return BTreePersistentIndexedCache.this.load(this.tailPos, this.root, this, this.entries.size()).findHighestLeaf();
        }
    }

    private class HeaderBlock
    extends BlockPayload {
        private IndexRoot index;

        private HeaderBlock() {
            this.index = new IndexRoot(this);
        }

        @Override
        protected byte getType() {
            return 85;
        }

        @Override
        protected int getSize() {
            return 10;
        }

        @Override
        protected void read(DataInputStream instr) throws Exception {
            this.index.rootPos = BlockPointer.pos(instr.readLong());
            short actualChildIndexEntries = instr.readShort();
            if (actualChildIndexEntries != BTreePersistentIndexedCache.this.maxChildIndexEntries) {
                throw this.blockCorruptedException();
            }
        }

        @Override
        protected void write(DataOutputStream outstr) throws Exception {
            outstr.writeLong(this.index.rootPos.getPos());
            outstr.writeShort(BTreePersistentIndexedCache.this.maxChildIndexEntries);
        }

        public IndexBlock getRoot() throws Exception {
            return this.index.getRoot();
        }
    }

    private class IndexRoot {
        private BlockPointer rootPos = BlockPointer.start();
        private HeaderBlock owner;

        private IndexRoot(HeaderBlock owner) {
            this.owner = owner;
        }

        public void setRootPos(BlockPointer rootPos) {
            this.rootPos = rootPos;
            BTreePersistentIndexedCache.this.store.write(this.owner);
        }

        public IndexBlock getRoot() {
            return BTreePersistentIndexedCache.this.load(this.rootPos, this, null, 0);
        }

        public IndexBlock newRoot() {
            IndexBlock block = new IndexBlock();
            BTreePersistentIndexedCache.this.store.write(block);
            this.setRootPos(block.getPos());
            return block;
        }
    }
}

