/* ***** BEGIN LICENSE BLOCK *****
 * Version: MPL 1.1/GPL 2.0/LGPL 2.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is the reusable ccl java library
 * (http://www.kclee.com/clemens/java/ccl/).
 *
 * The Initial Developer of the Original Code is
 * Chr. Clemens Lee.
 * Portions created by Chr. Clemens Lee are Copyright (C) 2002
 * Chr. Clemens Lee. All Rights Reserved.
 *
 * Contributor(s): Chr. Clemens Lee <clemens@kclee.com>
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ***** END LICENSE BLOCK ***** */

package ccl.util.test;

import ccl.util.FileUtil;
import ccl.util.Test;
import ccl.util.Util;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;

/**
 * Tests for ccl.util.FileUtil.
 *
 * @version $Id: FileUtilTest.java,v 1.23 2003/10/18 07:53:24 clemens Exp clemens $
 * @author <a href="http://www.kclee.com/clemens/">
 *         Chr. Clemens Lee</a>
 *         &lt;<a href="mailto:clemens@kclee.com">
 *         clemens@kclee.com
 *         </a>>
 */
public class FileUtilTest extends Test 
{
    /**
     * Checks for a bug in copyDir that was reported by user J. Q.
     * that a sub dir gets copied irregularly.
     * The bug looks like this.<p>
     *
     * source: sourcedir/a/b gets copied to destdir/sourcedir/a/a/b
     * instead of destdir/sourcedir/a/b.<p>
     */
    private void _checkCopyDir()
        throws IOException
    {
        // test case: source/a/b/c   directory
        //            source/a/b/f   file
        
        _enterSubTest( "copyDir" );

        // set up test environment
        String sTestSourceDir      = FileUtil.createTempDir();
        String sTestDestinationDir = FileUtil.createTempDir();

        String sDirA  = "a";
        String sDirB  = FileUtil.concatPath( sDirA, "b" );
        String sDirC  = FileUtil.concatPath( sDirB, "c" );
        String sFileF = FileUtil.concatPath( sDirB, "f" );
        FileUtil.md( FileUtil.concatPath( sTestSourceDir, sDirA ) );
        FileUtil.md( FileUtil.concatPath( sTestSourceDir, sDirB ) );
        FileUtil.md( FileUtil.concatPath( sTestSourceDir, sDirC ) );
        FileUtil.writeFile( FileUtil.concatPath( sTestSourceDir, sFileF ) 
                            , "Test file\n" );

        FileUtil.copyDir( sTestSourceDir, sTestDestinationDir );

        String sTestDirB = FileUtil.concatPath( sTestDestinationDir
                                                , FileUtil.concatPath( FileUtil.getBaseName( sTestSourceDir )
                                                                       , sDirB ) );

        boolean bug = !FileUtil.existsDir( sTestDirB );

        bugIf( bug, "Directory '" + sTestDirB + "' should exist but it does not!" );

        String sTestDirC = FileUtil.concatPath( sTestDirB, "c" );

        bug = bug | (!FileUtil.existsDir( sTestDirC ));

        Assert( FileUtil.existsDir( sTestDirC ), "Directory '" + sTestDirC + "' should exist but it does not!" );

        String sTestFileF = FileUtil.concatPath( sTestDirB, "f" );

        bug = bug | (!FileUtil.existsDir( sTestFileF ));

        Assert( FileUtil.existsFile( sTestFileF ), "File '" + sTestFileF + "' should exist but it does not!" );

        if ( !bug )
        {
            // for debugging purposes we don't delete the destination tree
            // when an error has happened
            FileUtil.deleteRecursively( sTestDestinationDir );
        }
        FileUtil.deleteRecursively( sTestSourceDir      );

        _exitSubTest ();
    }

    /**
     * Tests method 'FileUtil.getFiles(String,String)'.
     * Especially if empty file suffixes return all files
     * in a directory.
     *
     * @exception   IOException   if anything goes wrong.
     */
    private void _checkGetFiles() 
        throws IOException
    {
        _enterSubTest( "getFiles" );

        String sTestDir = FileUtil.createTempDir();

        Assert( FileUtil.getFiles( sTestDir, "" ).size() == 0 );

        FileUtil.writeFile( FileUtil.concatPath( sTestDir, ".emacs" )
                            , ";; test file\n" );
        FileUtil.writeFile( FileUtil.concatPath( sTestDir, "ccl.jar" )
                            , "test" );

        Assert( FileUtil.getFiles( sTestDir, ".jar" ).size() == 1 );
        Assert( FileUtil.getFiles( sTestDir, ".zip" + File.pathSeparator + ".jar" ).size() == 1 );
        Assert( FileUtil.getFiles( sTestDir, ".jar" + File.pathSeparator + ".emacs" ).size() == 2 );

        int result = FileUtil.getFiles( sTestDir, "" ).size();
        Assert( result == 2,
                "There should be 2 files found in the test directory but we got " + result + "." );

        FileUtil.deleteRecursively( sTestDir );
        bugIf( FileUtil.exists( sTestDir ) );

        _exitSubTest();
    }

    /**
     * Checks that method 'readFile' throws a file not found
     * exception containing in its message the file name which
     * could not be found.
     */
    public void testReadFile()
        throws IOException
    {
        _enterSubTest( "readFile" );

        try
        {
            FileUtil.readFile( "abcdefg" );
            fail( "File 'abcdefg' should not exist" );
        }
        catch( FileNotFoundException fileNotFoundException )
        {
            Assert( fileNotFoundException.getMessage().indexOf( "abcdefg" ) != -1 );
            Assert( fileNotFoundException.toString().indexOf( "abcdefg" ) != -1 );
            Assert( fileNotFoundException.toString().indexOf( "'abcdefg'" ) != -1 );
        }

        _exitSubTest();
    }

    /**
     * Checks if method 'copy' with an offset works.
     */
    public void testCopyWithOffset()
        throws Exception
    {
        _enterSubTest( "copy with offset" );
        // get temporary file
        String source = FileUtil.getTempFileName();
        // write data into it
        FileUtil.writeFile( source, "Hey Jude\n" );
        // copy to another temporary file with offset
        String destination = FileUtil.getTempFileName();
        bugIf( source.equals( destination ) );
        FileUtil.copy( source, destination, 2 );
        // load new file and check for content
        String result = FileUtil.readFile( destination );
        Assert( result.equals( "y Jude\n" )
                , "Result: '" + result + "'" );
        
        FileUtil.rm( source );
        FileUtil.rm( destination );

        _exitSubTest();
    }

    /**
     * Tests method 'writeTempFile'.
     */
    public void testWriteTempFile()
        throws Exception
    {
        String testContent = "test content\n";

        String tempFile = FileUtil.writeTempFile( testContent );
        Assert( tempFile != null );
        String loadContent = FileUtil.readFile( tempFile );
        Assert( testContent.equals( loadContent ) );
        FileUtil.delete( tempFile );
    }

    /**
     * Default test constructor.
     */
    public FileUtilTest() 
    {
        super();
    }

    /**
     * Standard test constructor.
     */
    public FileUtilTest(Test pTest_) 
    {
        super(pTest_);
    }

    /**
     * @throws   Exception   Whatever can go wrong.
     */
    protected void _doIt() 
        throws Exception
    {
        testWriteTempFile();

        _checkCopyDir();

        String sLocalPath = FileUtil.getClassPath(this);
        
        // Nr. 1    fRnd
        String sContent = "bla bla Schwachsinn!!!\n";
        String sFileName = FileUtil.concatPath(sLocalPath, "test.txt");
        String sFileNameBak = sFileName + ".bak";
        FileUtil.delete(sFileName);
        bugIf(FileUtil.exists(sFileName));
        try 
        {
            FileUtil.writeFile(sFileName, sContent);
            bugIf(false);
        }
        catch(Exception e) 
        {
            bugIf(true);
        }
        bugIf(!FileUtil.exists(sFileName));
        // move test
        FileUtil.move(sFileName, sFileNameBak);
        bugIf(FileUtil.exists(sFileName));
        bugIf(!FileUtil.exists(sFileNameBak));
        FileUtil.delete(sFileNameBak);
        bugIf(FileUtil.exists(sFileNameBak));
        
        // Nr. 7
        try 
        {
            FileUtil.writeFile(sFileName, sContent);
            FileUtil.writeFile(sFileNameBak, sContent);
            bugIf(false);
        }
        catch(Exception e) 
        {
            bugIf(true);
        }
        bugIf(!FileUtil.exists(sFileName));
        bugIf(!FileUtil.exists(sFileNameBak));
        FileUtil.move(sFileName, sFileNameBak);
        bugIf(FileUtil.exists(sFileName));
        bugIf(!FileUtil.exists(sFileNameBak));
        FileUtil.delete(sFileNameBak);
        bugIf(FileUtil.exists(sFileNameBak));
        
        // Nr. 13
        // move and source does not exist
        try 
        {
            FileUtil.writeFile(sFileNameBak, sContent);
            bugIf(false);
        }
        catch(Exception e) 
        {
            bugIf(true);
        }
        bugIf(FileUtil.exists(sFileName));
        bugIf(!FileUtil.exists(sFileNameBak));
        boolean bError = FileUtil.move( sFileNameBak, sFileName );
        bugIf( bError, "Could not move file " + sFileNameBak + " to " + sFileName );
        bugIf( FileUtil.exists(sFileNameBak) );
        
        // Nr. 18   exists[File|Dir]
        bugIf(FileUtil.exists("_Asdf_"));
        bugIf(FileUtil.exists("h:\\eigen\\src\\java\\lib\\ccl\\util\\test\\_Asdf_"));
        bugIf( !FileUtil.exists( System.getProperty( "user.dir" ) ) );
        /*// Nr. 21
          bugIf(!FileUtil.exists(FileUtil.concatPath(sLocalPath, ".." + File.separator + "test")));
          bugIf(!FileUtil.exists(FileUtil.concatPath(sLocalPath, ".." + File.separator + "test" + File.separator)));*/
        // Nr. 21
        bugIf(FileUtil.exists("h:/eigen/src/java/lib/ccl/util/test/_Asdf_"));
        // Nr. 22   exists[File|Dir]
        bugIf(FileUtil.existsFile("_Asdf_"));
        bugIf(FileUtil.existsDir("_Asdf_"));
        bugIf(FileUtil.existsFile(sLocalPath));

        // concatPath test
        bugIf( Util.endsWith( "", File.separatorChar ) );
        String sConcatPathResult = ccl.util.FileUtil.concatPath( "", "gnudoit" );
        bugIf( !sConcatPathResult.equals( "gnudoit" ), 
               "sConcatPathResult: " + sConcatPathResult );

        // FileUtil.getPackagePath(..) test
        String sPackagePath = null;
        String sHomePath = System.getProperty( "user.home" );
        bugIf( !FileUtil.existsDir( sHomePath ) );
        bugIf( FileUtil.existsFile( sHomePath ) );
        try 
        {
            sPackagePath = FileUtil.getPackagePath
                   ( ".", sHomePath );
            bugIf( !sPackagePath.equals( sHomePath ), "sPackagePath: " +
                   sPackagePath );
        }
        catch( Exception pException ) 
        {
            bugIf( true, "pException: " + pException );
        }

        // test directory deletion
        String sTempDir = FileUtil.getTempDir();
        sTempDir = FileUtil.concatPath( sTempDir, "fileutiltest" );
        FileUtil.md( sTempDir );
        Assert( FileUtil.existsDir( sTempDir ) );
        String sTempFile = FileUtil.concatPath( sTempDir
                                                , "file" );
        FileUtil.writeFile( sTempFile, "dummytext" );
        //Util.setDebug( true );
        Assert( FileUtil.existsFile       ( sTempFile ) );
        bugIf ( FileUtil.deleteRecursively( sTempDir  ) );
        bugIf ( FileUtil.exists           ( sTempDir  ) );

        // check getResourceAsStream
        {
            String sIniContent = FileUtil.getResourceAsString
                   ( this, "clearsight.ini" );
            Assert( sIniContent.indexOf( "\nAuthor=Chr. Clemens Lee\n" ) != -1
                    , "Resource string: " + sIniContent );
        }

        _checkGetFiles();
        testReadFile();
    }

    public static void main(String[] args) 
    {
        Test pTest = (Test) (new FileUtilTest());
        pTest.setVerbose( true );
        pTest.run();
        Test.printResult(pTest);

        System.exit(0);
    }
}
