/**********************************************************************
Copyright (c) 2003 Erik Bengtson and others. All rights reserved.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

Contributors:
2003 Andy Jefferson - coding standards
    ...
**********************************************************************/
package org.datanucleus.store.mapped.expression;

import org.datanucleus.store.mapped.mapping.JavaTypeMapping;

/**
 * Representation of a Boolean literal in a Query.
 *
 * @version $Revision: 1.5 $
 **/
public class BooleanLiteral extends BooleanExpression implements Literal
{
    private final boolean value;

    /** Raw value that this literal represents. */
    Object rawValue;

    /**
     * Creates a boolean literal
     * @param qs the QueryExpression
     * @param mapping the mapping
     * @param value the binary value
     */    
    public BooleanLiteral(QueryExpression qs, JavaTypeMapping mapping, boolean value)
    {
        super(qs);
        this.mapping = mapping;
        this.value = value;
        st.append(value ? getBooleanTrueValue() : getBooleanFalseValue());
    }

    public Object getValue()
    {
        return new Boolean(value);
    }
    
    /**
     * Return the String value for TRUE in the database.  This should be
     * overriden by subclasses for special cases of booleans.
     * 
     * @return  The String value for TRUE in the database.
     */
    protected String getBooleanTrueValue()
    {
        return "TRUE";
    }

    /**
     * Return the String value for FALSE in the database.  This should be
     * overriden by subclasses for special cases of booleans.
     * 
     * @return  The String value for FALSE in the database.
     */
    protected String getBooleanFalseValue()
    {
        return "(1=0)";
    }

    public BooleanExpression and(ScalarExpression expr)
    {
        if (expr instanceof BooleanExpression)
        {
            return value ? (BooleanExpression)expr : this;
        }
        else
        {
            return super.and(expr);
        }
    }

    public BooleanExpression eor(ScalarExpression expr)
    {
        if (expr instanceof BooleanExpression)
        {
            return value ? expr.not() : (BooleanExpression)expr;
        }
        else
        {
            return super.eor(expr);
        }
    }

    public BooleanExpression ior(ScalarExpression expr)
    {
        if (expr instanceof BooleanExpression)
        {
            return value ? this : (BooleanExpression)expr;
        }
        else
        {
            return super.ior(expr);
        }
    }

    public BooleanExpression not()
    {
        return new BooleanLiteral(qs, mapping, !value);
    }

    public BooleanExpression eq(ScalarExpression expr)
    {
        assertValidTypeForParameterComparison(expr, BooleanExpression.class);

        if (expr instanceof BooleanExpression)
        {
            return value ? (BooleanExpression)expr : expr.not();
        }
        else
        {
            return super.eq(expr);
        }
    }

    public BooleanExpression noteq(ScalarExpression expr)
    {
        assertValidTypeForParameterComparison(expr, BooleanExpression.class);

        if (expr instanceof BooleanExpression)
        {
            return value ? expr.not() : (BooleanExpression)expr;
        }
        else
        {
            return super.noteq(expr);
        }
    }

    /**
     * Method to save a "raw" value that this literal represents.
     * This value differs from the literal value since that is of the same type as this literal.
     * @param val The raw value
     */
    public void setRawValue(Object val)
    {
        this.rawValue = val;
    }

    /**
     * Accessor for the "raw" value that this literal represents.
     * This value differs from the literal value since that is of the same type as this literal.
     * @return The raw value
     */
    public Object getRawValue()
    {
        return rawValue;
    }
}
