/****************************************************************************

 This file is part of the GLC-lib library.
 Copyright (C) 2005-2008 Laurent Ribon (laumaya@users.sourceforge.net)
 Version 1.2.0, packaged on September 2009.

 http://glc-lib.sourceforge.net

 GLC-lib is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.

 GLC-lib is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with GLC-lib; if not, write to the Free Software
 Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA

*****************************************************************************/

#include <QtDebug>
#include <QFile>

#include "glwidget.h"

// For VSYNC problem under Mac OS X
#if defined(Q_OS_MAC)
#include <OpenGL.h>
#endif

GLWidget::GLWidget(QWidget *p_parent)
: QGLWidget(p_parent)
, m_Light()
, m_pFactory(GLC_Factory::instance(this->context()))
, m_GlView(this)
, m_TreeId()
, m_MoverController()
{
//////////////////////////// GLC specific///////////////////////////////////////
	m_Light.setPosition(15.0, 15.0, 0.0);

	m_GlView.cameraHandle()->setDefaultUpVector(glc::Z_AXIS);
	m_GlView.cameraHandle()->setIsoView();

	QColor repColor;
	repColor.setRgbF(1.0, 0.11372, 0.11372, 0.0);
	m_MoverController= m_pFactory->createDefaultMoverController(repColor, &m_GlView);

	CreateScene();

//////////////////////////End GLC specific/////////////////////////////////////
}

GLWidget::~GLWidget()
{
	// Clear the collection
	m_World.clear();

	// delete the factory
		if (m_pFactory != NULL)
	{
		delete m_pFactory;
		m_pFactory= NULL;
	}
}

void GLWidget::initializeGL()
{
	// OpenGL initialisation
	m_GlView.initGl();

	// For VSYNC problem under Mac OS X
	#if defined(Q_OS_MAC)
	const GLint swapInterval = 1;
	CGLSetParameter(CGLGetCurrentContext(), kCGLCPSwapInterval, &swapInterval);
	#endif

		// Load background image
	m_GlView.loadBackGroundImage(":fond_exemple08.png");

	// Reframe the scene
	m_GlView.reframe(m_World.boundingBox());
	// Calculate camera depth of view
	m_GlView.setDistMinAndMax(m_World.boundingBox());

}

void GLWidget::paintGL()
{

	// Clear screen
	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

	// Load identity matrix
	glLoadIdentity();

//////////////////////////// GLC specific///////////////////////////////////////
	// define the light
	m_Light.enable();
	m_Light.glExecute();

	// define view matrix
	m_GlView.glExecuteCam();

	// Display the collection of GLC_Object
	m_World.glExecute(0, false);

	// Display UI Info (orbit circle)
	m_GlView.glExecute();
	m_MoverController.drawActiveMoverRep();

//////////////////////////End GLC specific/////////////////////////////////////

}

void GLWidget::resizeGL(int width, int height)
{
//////////////////////////// GLC specific///////////////////////////////////////
	m_GlView.setWinGLSize(width, height);	// Compute window aspect ratio
//////////////////////////End GLC specific/////////////////////////////////////

}

// Create GLC_Object to display
void GLWidget::CreateScene()
{

	// Load the christmas tree
	QFile treeObjFile(":tree.obj");
	GLC_World* pWorld= m_pFactory->createWorld(treeObjFile);

	GLC_Material* pCurrentMat= NULL;
	// Set material properties to the tree
	QList<GLC_3DViewInstance*> treeInstances= pWorld->collection()->instancesHandle();
	for (int i= 0; i < treeInstances.size(); ++i)
	{
		pCurrentMat= treeInstances[i]->geomAt(0)->firstMaterial();
		pCurrentMat->setAmbientColor(QColor::fromRgb(22, 134, 33, 255));
		pCurrentMat->setDiffuseColor(QColor::fromRgb(85, 170, 127, 255));
	}
	m_World= (*pWorld);
	delete pWorld;
	treeInstances= m_World.collection()->instancesHandle();
	for (int i= 0; i < treeInstances.size(); ++i)
	{
		m_TreeId.append(treeInstances[i]->id());
	}

	// spheres creation

	// Load a sphere
	QFile bouleObjFile(":boule.obj");
	pWorld= m_pFactory->createWorld(bouleObjFile);

	// translate the sphere
	GLC_3DViewInstance* pBouleInstance= pWorld->collection()->instancesHandle().first();
	// Assign material to the sphere
	pCurrentMat= pBouleInstance->geomAt(0)->firstMaterial();
	pCurrentMat->setAmbientColor(Qt::red);
	pCurrentMat->setDiffuseColor(QColor::fromRgbF(0.8, 0.2, 0.2, 1.0));

	pBouleInstance->translate(6.0, 0.0, 71.0);
	// Add the sphere to the collection
	m_World.mergeWithAnotherWorld((*pWorld));

	// Copy last sphere
	// translate the sphere
	GLC_Matrix4x4 MatTrans(16.0, -8.0, 7.0);
	// Add the sphere to the collection
	m_World.rootOccurence()->addChild((new GLC_StructInstance(pBouleInstance->representation().deepCopy()))->move(MatTrans));


	// Copy last sphere
	// translate the sphere
	MatTrans.setMatTranslate(6.0, -20.0, 4.0);
	// Add the sphere to the collection
	m_World.rootOccurence()->addChild((new GLC_StructInstance(pBouleInstance->representation().deepCopy()))->move(MatTrans));

	// Copy last sphere
	// translate the sphere
	MatTrans.setMatTranslate(23.0, 13.0, 6.0);
	// Add the sphere to the collection
	m_World.rootOccurence()->addChild((new GLC_StructInstance(pBouleInstance->representation().deepCopy()))->move(MatTrans));

	// Copy last sphere
	// translate the sphere
	MatTrans.setMatTranslate(-18.0, 15.0, 9.0);
	// Add the sphere to the collection
	m_World.rootOccurence()->addChild((new GLC_StructInstance(pBouleInstance->representation().deepCopy()))->move(MatTrans));

	// Copy last sphere
	// translate the sphere
	MatTrans.setMatTranslate(-16.0, -20.0, -8.0);
	// Add the sphere to the collection
	m_World.rootOccurence()->addChild((new GLC_StructInstance(pBouleInstance->representation().deepCopy()))->move(MatTrans));

	// Copy last sphere
	// translate the sphere
	MatTrans.setMatTranslate(19.0, 38.0, -18.0);
	// Add the sphere to the collection
	m_World.rootOccurence()->addChild((new GLC_StructInstance(pBouleInstance->representation().deepCopy()))->move(MatTrans));

	// Copy last sphere
	// translate the sphere
	MatTrans.setMatTranslate(26.0, -24.0, -22.0);
	// Add the sphere to the collection
	m_World.rootOccurence()->addChild((new GLC_StructInstance(pBouleInstance->representation().deepCopy()))->move(MatTrans));

	// Copy last sphere
	// translate the sphere
	MatTrans.setMatTranslate(-19.0, 0.0, -25.0);
	// Add the sphere to the collection
	m_World.rootOccurence()->addChild((new GLC_StructInstance(pBouleInstance->representation().deepCopy()))->move(MatTrans));

	// Copy last sphere
	// translate the sphere
	MatTrans.setMatTranslate(35.0, -14.0, -44.0);
	// Add the sphere to the collection
	m_World.rootOccurence()->addChild((new GLC_StructInstance(pBouleInstance->representation().deepCopy()))->move(MatTrans));

}

void GLWidget::mousePressEvent(QMouseEvent *e)
{
	if (m_MoverController.hasActiveMover()) return;
	switch (e->button())
	{
	case (Qt::RightButton):
		m_MoverController.setActiveMover(GLC_MoverController::TrackBall, e->x(), e->y());
		updateGL();
		break;
	case (Qt::LeftButton):
		select(e->x(), e->y());
		break;
	case (Qt::MidButton):
		m_MoverController.setActiveMover(GLC_MoverController::Zoom, e->x(), e->y());
		updateGL();
		break;

	default:
		break;
	}
}

void GLWidget::mouseMoveEvent(QMouseEvent * e)
{
	if (not m_MoverController.hasActiveMover()) return;
	m_MoverController.move(e->x(), e->y());
	m_GlView.setDistMinAndMax(m_World.boundingBox());
	updateGL();
}

void GLWidget::mouseReleaseEvent(QMouseEvent*)
{
	if (not m_MoverController.hasActiveMover()) return;
	m_MoverController.setNoMover();
	updateGL();
}

void GLWidget::select(const int x, const int y)
{

	setAutoBufferSwap(false);
	GLC_uint SelectionID= m_GlView.select(this, x, y);
	setAutoBufferSwap(true);

	if (SelectionID != 0)
	{
		GLC_3DViewInstance instance(*m_World.collection()->instanceHandle(SelectionID));
		if ((!instance.isEmpty()) && (not m_TreeId.contains(instance.id())))
		{	// Instance is in the collection and is not the tree
			if (instance.geomAt(0)->firstMaterial()->ambientColor() == Qt::blue)
			{
				instance.geomAt(0)->firstMaterial()->setAmbientColor(Qt::red);
				instance.geomAt(0)->firstMaterial()->setDiffuseColor(QColor::fromRgbF(0.8, 0.2, 0.2, 1.0));
			}
			else
			{
				instance.geomAt(0)->firstMaterial()->setAmbientColor(Qt::blue);
				instance.geomAt(0)->firstMaterial()->setDiffuseColor(QColor::fromRgbF(0.2, 0.2, 0.8, 1.0));
			}

		}
	}
	updateGL();
}

